(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     62097,       2157]
NotebookOptionsPosition[     51656,       1792]
NotebookOutlinePosition[     53113,       1832]
CellTagsIndexPosition[     53027,       1827]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Vectors and Matrices" :> 
          Documentation`HelpLookup["paclet:tutorial/VectorsAndMatrices"], 
          "Constructing Matrices" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ConstructingMatrices"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Vectors and Matrices\"\>", 
       2->"\<\"Constructing Matrices\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"IdentityMatrix" :> 
          Documentation`HelpLookup["paclet:ref/IdentityMatrix"], "Diagonal" :> 
          Documentation`HelpLookup["paclet:ref/Diagonal"], "Tr" :> 
          Documentation`HelpLookup["paclet:ref/Tr"], "KroneckerDelta" :> 
          Documentation`HelpLookup["paclet:ref/KroneckerDelta"], "Band" :> 
          Documentation`HelpLookup["paclet:ref/Band"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"IdentityMatrix\"\>", 2->"\<\"Diagonal\"\>", 
       3->"\<\"Tr\"\>", 4->"\<\"KroneckerDelta\"\>", 
       5->"\<\"Band\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Constructing Matrices" :> 
          Documentation`HelpLookup["paclet:guide/ConstructingMatrices"], 
          "Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Constructing Matrices\"\>", 
       2->"\<\"Matrices and Linear Algebra\"\>", 
       3->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["DiagonalMatrix", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["DiagonalMatrix",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/DiagonalMatrix"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a matrix with the elements of ",
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     " on the leading diagonal, and ",
     Cell[BoxData["0"], "InlineFormula"],
     " elsewhere. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["DiagonalMatrix",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/DiagonalMatrix"], "[", 
       RowBox[{
        StyleBox["list", "TI"], ",", 
        StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a matrix with the elements of ",
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     " on the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " diagonal."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->10792]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "For positive ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["DiagonalMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DiagonalMatrix"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", 
    StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
 " puts the elements ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 " positions above the main diagonal. ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["DiagonalMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DiagonalMatrix"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", 
    RowBox[{"-", 
     StyleBox["k", "TI"]}]}], "]"}]], "InlineFormula"],
 " puts the elements ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 " positions below. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->18190670],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["DiagonalMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DiagonalMatrix"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", 
    StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
 " fills the ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " diagonal of a square matrix with the elements from ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 ". Different values of ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 " lead to different matrix dimensions."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->210868340]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->303212492],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->681901445],

Cell["Construct a diagonal matrix:", "ExampleText",
 CellID->223157580],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->59463002],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"a", "0", "0"},
     {"0", "b", "0"},
     {"0", "0", "c"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{61, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->341690366]
}, Open  ]],

Cell["A superdiagonal matrix:", "ExampleText",
 CellID->4138244],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", "1"}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->262164887],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztVrtugzAUNTb0kbZ7qy7ZM3TOD7RTPwJFlZKhapRmYGNkZGTkExgZGRkZ
GRkZGRnp9QPsIEip4kgdYgl8bS7nXB8/rt/t/frj095vVvb8bWdv15vV9/z1
awddxEAILeFZPCKwG7DYs2waqNW3Rd8kDMOGFbRgbdd1RZswL5OZeZ7HcUxN
2iYAeN+rr1u0LMv6fwdBoGJaNJ62gI1mf+i9YjB1XTuOI3lI28upOO9JPHw8
nucBKDVxx4PTNLU04YOoIO0hPtOrLEud45Dzp/LIudbFA2LxBXU2nptWIdFl
DPLQ+Rlapbc9v+G/Xy5eIzrSObEm66hrti88x3kw1Gy/ke6rOCe1j8jkvYiV
M60CZRSzSQxDqxSL9n/cVfq9fN+f4CVzveoFZ2lVVX0dLaEj6VZtURRRFFFT
PdshrYiMryuHJEkykBMhRMGuiwc0O8jt0+4qd1CbvyBL3SUyBpVOj5zuagw6
MCRxg5MR49GIiajH7lO9UQtkfgkZR346gij/5YhtFM+ifmC9/CaKjB8KnzsA
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{57, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->235509352]
}, Open  ]],

Cell["A subdiagonal matrix:", "ExampleText",
 CellID->37778175],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"-", "1"}]}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->330982464],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"0", "0", "0"},
     {"a", "0", "0"},
     {"0", "b", "0"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{61, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->121459073]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->797378833],

Cell[TextData[{
 "The elements in ",
 Cell[BoxData[
  ButtonBox["DiagonalMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiagonalMatrix"]], "InlineFormula"],
 " are chosen to match the elements of the vector:"
}], "ExampleText",
 CellID->454727216],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"1", "/", 
    RowBox[{"Range", "[", "3", "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->462087911],

Cell["Exact number entries:", "ExampleText",
 CellID->288120164],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", "v", "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->209775268],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", "0", "0"},
     {"0", 
      FractionBox["1", "2"], "0"},
     {"0", "0", 
      FractionBox["1", "3"]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{69, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->398186906]
}, Open  ]],

Cell["Machine-number entries:", "ExampleText",
 CellID->247596],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"N", "[", "v", "]"}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->732459161],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1.`", "0.`", "0.`"},
     {"0.`", "0.5`", "0.`"},
     {"0.`", "0.`", "0.3333333333333333`"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{131, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->7047466]
}, Open  ]],

Cell["Arbitrary-precision number entries:", "ExampleText",
 CellID->562264309],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"N", "[", 
    RowBox[{"v", ",", "20"}], "]"}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->9652364],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1.`20.", "0", "0"},
     {"0", "0.5`20.", "0"},
     {"0", "0", "0.33333333333333333333333333333333333333`20."}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{495, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]//MatrixForm=",
 CellID->14882233]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->62990351],

Cell[TextData[{
 "When the vector is a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " object, ",
 Cell[BoxData[
  ButtonBox["DiagonalMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiagonalMatrix"]], "InlineFormula"],
 " will give a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " object:"
}], "ExampleText",
 CellID->627060822],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"v", " ", "=", " ", 
  RowBox[{"SparseArray", "[", 
   RowBox[{"Range", "[", "100", "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->358264379],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "100", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 100, ">"],
     Editable->False], ",", 
    RowBox[{"{", "100", "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{177, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->348576475]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DiagonalMatrix", "[", "v", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->77176116],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "100", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 100, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "100"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{209, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->396456531]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->639571560],

Cell["\<\
Express a matrix as the sum of its diagonal and off-diagonal parts:\
\>", "ExampleText",
 CellID->166912557],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"m", "=", 
   RowBox[{"Array", "[", 
    RowBox[{
     RowBox[{
      SubscriptBox["a", "##"], "&"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "3"}], "}"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19742325],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztVztPAkEQXu6Qh6gRn2gs/AP+Da38BRaEEBOuMBqloKOkpKTkJ1BSUlJS
UlJSUlKeM/vglmEOFgE1xktub2dnZuexs9/tPpaqleeXUjUol24f3ktvlaD8
cXv/+g5DfkIIEcB7VxDQD6En3yAM4XuJrY+0fMTNCjovtbD1YDBJuHuC8JIz
HjwoeBLLxdEjOfuenLXdbutZc8SKZ0zNdKEvzjmpyAJ0LjiJJbTyo16va9qX
06iYh8Nht9uVJqBBZ6h2mtC5LculTJYGgwH1rtVq2T4v5IquW4rQ+zvUonGo
LE+n01qtFsXhm1EVipplYW47R0hnnDyiWmlC576kpWw3Gg1wOjTVJOPw+v0+
5gqL0WdyiJV/GMPnqsNFTtFQolCo897I6hiPx/FZvV5jVOFHtBtsO9HOwTVG
527It2BWWW32GO2i8VrTiSVSEf0XpST6roHWh06zYk35TE0ZNJ/xWDTnuVjT
+87Wbat8TZ5yUnPofsZJLEH3tJN3WLk/ge4ZJ+/4XK3CBh4Vt6NFEdgty78f
3bNOcWC1fCfKb1Il66B8wTl6DuWvnLSL/1J6dDcoj7uFQx4cO7Z5Cyifj+VG
KN9sNpdaPyBW+XMpxXDyXxGLp3y6Q3hcje4StndwophMJmH4cyiv6NFo1Ol0
Zt5r7+BApU/2vx/l1U2g1+sxp01IsY5ul3HYd5F4lF+lpeKCWp47w9M7FdJJ
vbqe1rLR3YxnibXcCj79V0bWPaiEzVEc778erIWcSd+75m+KHHpfMnJGW10j
sPtEaCUVWK1IfAJlIkc1\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{105, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->933661938]
}, Open  ]],

Cell[BoxData[{
 RowBox[{
  RowBox[{"md", "=", 
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{"Diagonal", "[", "m", "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"mo", "=", 
   RowBox[{"m", "-", "md"}]}], ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->99672092],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{"MatrixForm", ",", 
   RowBox[{"{", 
    RowBox[{"md", ",", "mo"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->265145035],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWT1MFEEUXnZvgePOKBIBhQKjiYqaoCSiodMENcbKxMpITmMChdEgBR0l
JSUl5ZWUV1JSUlJeeQUFJeX63vwwc9/N/i+GSHLsvJ15733z5s3bb2c/tjbX
fvxsba5/b8293Wj9Xlv//mdu5dcG3QqGPM87pd/TaY/aEbXE7zSK6PqF/wcs
iz9vEeRZkJeE1mf679PNGku1i151d97utXRDao7HavJA4d1rCr39/X2l9x7s
+ClyKOTt7W0lTwh5b29PyQ2XfmhjobZ3yzXKIKbGAzHi/PxcQpV2lfWTk5NO
pyNU1NxmrSs7Q9QjIA/rKBwfH3PzppDJaK/X03az2GnkGMfilI6e8jNmzcrE
MHDG7E6Ou5gdiCfU0d3a2jLz393dTUSAVodBHqtQa1KM6na7MiwqUmMat7pb
Kx0pe+VYHgF5VMg7OzvklJs3hExpc3h4GI8ArY5mmnMaloZTawrWrn4RKf/o
6CjMEYsZuvImDAAtZo+UabfQRjQxIQQqRmGMHV55rk/XMvpxRyl+vmIPWTu4
XGZIP6bWyFlSJqghVXlot9sKsz0TU2Hz+OEV5ABOR1Ff9UTsYdRfN2cS8FGi
Ux5Vg29Jr5K6dR1svY7F4RMMuwYYHZzLS+hHH2+sGMVbXU2x8rVPm2Ngnsr5
rQRKX+zWgX1R1GoIdvz/3voC9D9J8TYO45dBbpZCw1wt0HVugI89hNHIITAj
G7Y1YH4LsZ44q+qlZzHQ38fV5h0z8fXzCmYwYGlgtz934gHU+FQpPDu7euRh
crfBDvqpO/zksZ8Vz0ji7NPYe57aXexpPe2IE+dGtqdIMZ/Z8CMn03mE/HRI
6gfyYlAUi2IxjodRxDpWJQstzpM5enyfX4o5ajJyxXIwKxfNlnUYrzSG2iyF
Se9JdyalVaTyvLJYtuRlrsVrbvyKx3PRLCsar23z4qpxvyqE955T60UpfDoG
y5H7GYUY7qtxS6W8rqZ4ucp88Spbr/adAp+3yHw/pFhFZo1nl5opu9+mWHax
Gb63Iu6Z93n0/MjWHWDR47G9NvN1o2oCGvep1zvRZU5MEd9jl5U+foynnZg3
cYzWRk3vzGdnZ1FUjEmaM4Fut3twcMDNpDOBKs8+pUxOyTU37VOETqejTmAv
n/9dnGbgqZUacvlcOAsDtU9H7EjR4qt1+1eRMud5HKm8Z4kuZpXOc9O0Rp0R
wvNzjlBNZaavtOJZXVIMsF6aKuCuFZohas91mE8jpV/Ot9frybS0vg/4tE/K
8zY+Vfcpjyb0bNQKl//iIM/rCXbfl40mrAz7iWdpSZaNDfmtgoC32+3klbCr
36TWkgVPfU9q6ngrxbu5cRndCZDdX74+aRyqmksczwDtorp+E73ya6I39Bfr
MjhE\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{235, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->18795566]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->305041818],

Cell["\<\
Verify the similarity of a matrix to the diagonal matrix of its eigenvalues:\
\>", "ExampleText",
 CellID->813739165],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"N", "[", 
   RowBox[{"HilbertMatrix", "[", "3", "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->80797988],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.`", ",", "0.5`", ",", "0.3333333333333333`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.5`", ",", "0.3333333333333333`", ",", "0.25`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.3333333333333333`", ",", "0.25`", ",", "0.2`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{451, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->571734619]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"vals", ",", "vecs"}], "}"}], "=", 
  RowBox[{"Eigensystem", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->44890258],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1.408318927123654`", ",", "0.12232706585390583`", ",", 
     "0.002687340355773502`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "0.8270449269720095`"}], ",", 
       RowBox[{"-", "0.4598639043655439`"}], ",", 
       RowBox[{"-", "0.32329843524449897`"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.5474484307206745`", ",", 
       RowBox[{"-", "0.5282902350674367`"}], ",", 
       RowBox[{"-", "0.6490066588517127`"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0.12765932974653427`", ",", 
       RowBox[{"-", "0.7137468858034134`"}], ",", "0.6886715316713732`"}], 
      "}"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{475, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->266495205]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Transpose", "[", "vecs", "]"}], ".", 
  RowBox[{"DiagonalMatrix", "[", "vals", "]"}], ".", "vecs"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1028688765],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1.0000000000000002`", ",", "0.5000000000000002`", ",", 
     "0.3333333333333334`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.5000000000000002`", ",", "0.33333333333333326`", ",", 
     "0.25000000000000006`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.33333333333333337`", ",", "0.25000000000000006`", ",", 
     "0.20000000000000004`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{451, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->41367775]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->487887351],

Cell["Define a Jordan matrix: ", "ExampleText",
 CellID->40787963],

Cell[BoxData[
 RowBox[{
  RowBox[{"jordanMatrix", "[", 
   RowBox[{"\[Lambda]_", ",", "n_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{"Table", "[", 
     RowBox[{"\[Lambda]", ",", 
      RowBox[{"{", "n", "}"}]}], "]"}], "]"}], "+", 
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{"n", "-", "1"}], "}"}]}], "]"}], ",", "1"}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->516754666],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"jordanMatrix", "[", 
   RowBox[{"\[Lambda]", ",", "3"}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->520724121],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztlqFuwzAQht14abam44PlBcN9gQ3tIaJqUgumTV1BWGBgYGEfobAwMDAw
sDCwsNA72+facb2tSzRpUhcpubPlfGf/9tl+ipaz55doOZ9Go8dF9DabT99H
D68LqKI9QsgE3vEdAZ+BJ94JY2DNr8+/dL1eM/GQsSgnSYJlKlpdCbeqqu12
y11epgAcoh2A9cBStEdqWZY2ZbVamWyf90s9BFmntaH4QdbyciDKh8MhjmMd
gapaGURGPDPCEPt+LRhpmgKGu96R7BVF4bcmgnQgYJMo1Kjr+qd9vTXU4G7f
mh8zgp7L7hFAErlUfiFCoPTAcs8Zgetvrjyl8cBq5/77/r+VpaOduTdg/eOa
/Y7inm87V+2Z7sJW8/3XmTYrvCi2maVK2cvK0izLzmilz12zFeyB+/1euMyd
nSHqK2djt9ttNhvumrsyHAV4+srT+PN8dO/Pcj/O89xxakH3MGKfuXPnKyZo
0zhb7XuB34KpldVMD8YfWAx7PbuZYlQwQtEz1K/Zv3bMk9sPsuQVoCtbUyQ7
ZM3s45YqXfH+R3ofwwvnRQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{57, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->312962934]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->426947172],

Cell[TextData[{
 "Construct an ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 "\[Times]",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " tridiagonal matrix:"
}], "ExampleText",
 CellID->22957795],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "5"}], ";"}], "\n", 
 RowBox[{"MatrixForm", "[", 
  RowBox[{
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{
     RowBox[{"Array", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["a", "#"], "&"}], ",", 
       RowBox[{"n", "-", "1"}]}], "]"}], ",", 
     RowBox[{"-", "1"}]}], "]"}], "+", 
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{"Array", "[", 
     RowBox[{
      RowBox[{
       SubscriptBox["b", "#"], "&"}], ",", "n"}], "]"}], "]"}], "+", 
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{
     RowBox[{"Array", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["c", "#"], "&"}], ",", 
       RowBox[{"n", "-", "1"}]}], "]"}], ",", "1"}], "]"}]}], "]"}]}], "Input",\

 CellLabel->"In[1]:=",
 CellID->4728434],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      SubscriptBox["b", "1"], 
      SubscriptBox["c", "1"], "0", "0", "0"},
     {
      SubscriptBox["a", "1"], 
      SubscriptBox["b", "2"], 
      SubscriptBox["c", "2"], "0", "0"},
     {"0", 
      SubscriptBox["a", "2"], 
      SubscriptBox["b", "3"], 
      SubscriptBox["c", "3"], "0"},
     {"0", "0", 
      SubscriptBox["a", "3"], 
      SubscriptBox["b", "4"], 
      SubscriptBox["c", "4"]},
     {"0", "0", "0", 
      SubscriptBox["a", "4"], 
      SubscriptBox["b", "5"]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{121, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->133538661]
}, Open  ]],

Cell[TextData[{
 "This can also be done using ",
 Cell[BoxData[
  ButtonBox["Band",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Band"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->415918509],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"SparseArray", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Band", "[", 
       RowBox[{"{", 
        RowBox[{"2", ",", "1"}], "}"}], "]"}], "->", 
      RowBox[{"Array", "[", 
       RowBox[{
        RowBox[{
         SubscriptBox["a", "#"], "&"}], ",", 
        RowBox[{"n", "-", "1"}]}], "]"}]}], ",", 
     RowBox[{
      RowBox[{"Band", "[", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}], "]"}], "->", 
      RowBox[{"Array", "[", 
       RowBox[{
        RowBox[{
         SubscriptBox["b", "#"], "&"}], ",", "n"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{"Band", "[", 
       RowBox[{"{", 
        RowBox[{"1", ",", "2"}], "}"}], "]"}], "->", 
      RowBox[{"Array", "[", 
       RowBox[{
        RowBox[{
         SubscriptBox["c", "#"], "&"}], ",", 
        RowBox[{"n", "-", "1"}]}], "]"}]}]}], "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25448404],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVmb9vGjEUx839aENL6UEypFKlsqBIXauq/0GjRsrQPwFFlRJVVas0kdgY
GRlv5E9gvJGRkZGRkZGR0bV99p3vxXe1z76KIt35B89+X79nf1Cc69HD7bcf
o4e7m9Hg8n706/bu5vfg88970uW3EEKP5Plyjkgdkxp7HjEmZfr+St4+6ezT
Fq1g9kEd9m3IeufzOe+9AlaR1CZv9Jy1J5MJ/95n/QGrbjabJEloNZB8wjKk
0/h8OvLxyNPm/SHwnqlbr9fQWxzHsoYwG8Wnjqx7PbD2l6VWdA0nzPpwOIzH
41yrL3pTuan2BrT6XEW5RjmuqdbpdEqE0aqXafVWq1VoraZHygCJTOV+26Xq
/CyCZAuRjVRUxbK92+2ajGDEIwhjxSIq7W1ZVX4O7FX1eBR6bLp81yHUxXwP
pce0MQVDsPJIRJ63W0q/IVaf9SGPZx/Yq2f5/l9bfeRrPQMxONEa/QnEHZIa
tiGD9fMSKva3YC8dH3CC+FJbbw32u0+fFXpMlq3a/2gNeuoKrAa/dl4jagWP
feDxhX4mpPYz0Nbbj264rSJ0V2ndbSSKakL3eXzd+4RMHmp5gGdfsPhCa/Sx
UlbPaiCtna75lJfvtEfL8T4Dbcjit8b5qGIwPHmCxWX9r7W8m+w3C0LUtHJ/
Zur9RkAaB1qz2Kg3o7LS6snfSn+bxX20zaisnsM2itVUdu8TUvnCwoNgg2D0
e2xyto+VwnpWdI+GUg7F79UHrdEdkIUeaENW680aYreMFmuyYbUFIWpauT8z
en7dUNlGPc1/gPJ7nY7+aoxp7D7KLmhsG706NLbx+QasVU1jGw/nuMhmszN9
rPQ1Y3TEy0v2/Ww2qxwF2QxZnM6S3zvLs8RxvN/vWRXbMVeUaZa32+1isaBV
+QYtSRJ+u9w8e032/is243K5VNyGkvDwlRhodkSnKrVkVxRulPVu6gukBawv
J7C5unx35uo8knvhxz0z1b10PR7JH4sOz0UeI6bCmpzlnp/kgytIr9czBTV5
WuU395D6bWM1VUV5yqzS/2Kh1h/MbhGK\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{117, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->329110931]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->553509347],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["IdentityMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/IdentityMatrix"]], "InlineFormula"],
 " is a special case of ",
 Cell[BoxData[
  ButtonBox["DiagonalMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiagonalMatrix"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->51040628],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"IdentityMatrix", "[", "5", "]"}], "\[Equal]", 
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"Table", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", "5", "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->583471562],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->463983632]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->426657608],

Cell["Several simple properties hold for diagonal matrices: ", "ExampleText",
 CellID->207737819],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"Range", "[", "5", "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->112880689],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Inverse",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Inverse"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["MatrixExp",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixExp"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["MatrixPower",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixPower"]], "InlineFormula"],
 " commute with ",
 Cell[BoxData[
  ButtonBox["DiagonalMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiagonalMatrix"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->12002127],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Inverse", "[", 
   RowBox[{"DiagonalMatrix", "[", "v", "]"}], "]"}], "\[Equal]", 
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"1", "/", "v"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->45601439],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->243171569]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixExp", "[", 
   RowBox[{"DiagonalMatrix", "[", "v", "]"}], "]"}], "\[Equal]", 
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"Exp", "[", "v", "]"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->89372165],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->26950582]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixPower", "[", 
   RowBox[{
    RowBox[{"DiagonalMatrix", "[", "v", "]"}], ",", "n"}], "]"}], "\[Equal]", 
  
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"Power", "[", 
    RowBox[{"v", ",", "n"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->569204146],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->30968656]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Det",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Det"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Tr",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Tr"]], "InlineFormula"],
 " have commuting relations:"
}], "ExampleText",
 CellID->286928],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Det", "[", 
   RowBox[{"DiagonalMatrix", "[", "v", "]"}], "]"}], "\[Equal]", 
  RowBox[{"Times", "@@", "v"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->40665350],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->37953274]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Tr", "[", 
   RowBox[{"DiagonalMatrix", "[", "v", "]"}], "]"}], "\[Equal]", 
  RowBox[{"Plus", "@@", "v"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->69797476],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->395442248]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->38318280],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Diagonal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Diagonal"]], "InlineFormula"],
 " of ",
 Cell[BoxData[
  ButtonBox["DiagonalMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DiagonalMatrix"]], "InlineFormula"],
 " gives the original vector:"
}], "ExampleText",
 CellID->69651662],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "100"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->107517729],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Diagonal", "[", 
   RowBox[{"DiagonalMatrix", "[", "v", "]"}], "]"}], "===", "v"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->118792847],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->97464366]
}, Open  ]],

Cell[TextData[{
 "This is true even if the vector is a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " object:"
}], "ExampleText",
 CellID->507340758],

Cell[BoxData[
 RowBox[{
  RowBox[{"s", "=", 
   RowBox[{"SparseArray", "[", "v", "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->289927538],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Diagonal", "[", 
   RowBox[{"DiagonalMatrix", "[", "s", "]"}], "]"}], "===", "s"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->17959681],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->112449460]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->71533622],

Cell["\<\
Matrices with only sub\[Hyphen] or superdiagonals are always nilpotent:\
\>", "ExampleText",
 CellID->51548694],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{
     RowBox[{"Range", "[", "4", "]"}], ",", 
     RowBox[{"-", "1"}]}], "]"}], ",", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->49370446],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmcEOgjAMhgtIRBNlcvLI0+jJh1iMCR6MRnl/3SAQUzc3GVMkNWHQ0v39
2MpmdMfL4nDi5XHP8+2VX4rj/pZvzlfhigKAkAFAugZxfRdX1dF8MnEATGQb
tf4Xe4bs1NHG+nNDPkb5es2H+3+aH+eLZFO3BqpQmExzXilUpR8/zfwpfow6
TKOHdfD4hlazIO9LM0b+qWc/5kgsuRLiGgWXTq8v3m5vQxUVt3bVp94P5Rs3
/cFdTCuj8fr9njohaqJ2yOznmczvamiIipTaC09evL4srGiWRPMnNGotd0bT
vhRUcd32JV+1jrPj7/tqGl/zSTR90/iqdbt1nWqdaL5HQ7XuOoLDmk+i0dNQ
rbuO4LDmk2j0NL+tdZsoaaaGM+6Hf9NtdNhIdVKDnmmc638amvnLPrQgeACO
OX/Z\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{528, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->268793304]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{
     RowBox[{"Range", "[", "4", "]"}], ",", "1"}], "]"}], ",", "5"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->107050031],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmcEOgjAMhgtIRBNlcvLI0+jJh1iMCR6MRnl/3SAQUzc3GVMkNWHQ0v39
2MpmdMfL4nDi5XHP8+2VX4rj/pZvzlfhigKAkAFAugZxfRdX1dF8MnEATGQb
tf4Xe4bs1NHG+nNDPkb5es2H+3+aH+eLZFO3BqpQmExzXilUpR8/zfwpfow6
TKOHdfD4hlazIO9LM0b+qWc/5kgsuRLiGgWXTq8v3m5vQxUVt3bVp94P5Rs3
/cFdTCuj8fr9njohaqJ2yOznmczvamiIipTaC09evL4srGiWRPMnNGotd0bT
vhRUcd32JV+1jrPj7/tqGl/zSTR90/iqdbt1nWqdaL5HQ7XuOoLDmk+i0dNQ
rbuO4LDmk2j0NL+tdZsoaaaGM+6Hf9NtdNhIdVKDnmmc638amvnLPrQgeACO
OX/Z\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{528, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->287165353]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->257787544],

Cell[TextData[{
 "The size of the matrix generated by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["DiagonalMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DiagonalMatrix"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", " ", 
    StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
 " equals ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["Length",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Length"], "[", 
    StyleBox["list", "TI"], "]"}], " ", "+", " ", 
   RowBox[{
    ButtonBox["Abs",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Abs"], "[", 
    StyleBox["k", "TI"], "]"}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->411244880],

Cell[BoxData[{
 RowBox[{
  RowBox[{"list", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{"1", ",", "10"}], "}"}], "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"k", "=", 
   RowBox[{"RandomInteger", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "20"}], ",", "20"}], "}"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->596419710],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Thread", "[", 
  RowBox[{
   RowBox[{"Dimensions", "[", 
    RowBox[{"DiagonalMatrix", "[", 
     RowBox[{"list", ",", "k"}], "]"}], "]"}], "\[Equal]", 
   RowBox[{
    RowBox[{"Length", "[", "list", "]"}], "+", 
    RowBox[{"Abs", "[", "k", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->192165093],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "True"}], "}"}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->95152814]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->965313707],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Band",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Band"]], "InlineFormula"],
 " can be used to construct diagonals equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["DiagonalMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DiagonalMatrix"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", " ", 
    StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->243960997],

Cell[BoxData[{
 RowBox[{
  RowBox[{"list", "=", 
   RowBox[{"Range", "[", "5", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"k", "=", 
   RowBox[{"RandomInteger", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "2"}], ",", "2"}], "}"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->192204259],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"list", ",", "k"}], "]"}], "\[Equal]", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Band", "[", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"k", ">", "0"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", 
          RowBox[{"1", "+", "k"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", "-", "k"}], ",", "1"}], "}"}]}], "]"}], "]"}], 
     "\[Rule]", "list"}], ",", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"Length", "[", "list", "]"}], "+", 
       RowBox[{"Abs", "[", "k", "]"}]}], ")"}], 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->505019813],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->293677160]
}, Open  ]],

Cell[TextData[{
 "They will be ",
 Cell[BoxData[
  ButtonBox["SameQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SameQ"]], "InlineFormula"],
 " if the vector is a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->192825226],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"s", "=", 
   RowBox[{"SparseArray", "[", "list", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"DiagonalMatrix", "[", 
   RowBox[{"s", ",", "k"}], "]"}], "===", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Band", "[", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"k", ">", "0"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", 
          RowBox[{"1", "+", "k"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", "-", "k"}], ",", "1"}], "}"}]}], "]"}], "]"}], 
     "\[Rule]", "s"}], ",", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"Length", "[", "s", "]"}], "+", 
       RowBox[{"Abs", "[", "k", "]"}]}], ")"}], 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}]}]}], "]"}]}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->30548269],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->462300011]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17552],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["IdentityMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IdentityMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Diagonal",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Diagonal"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Tr",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Tr"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["KroneckerDelta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KroneckerDelta"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Band",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Band"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->25383]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->7726],

Cell[TextData[ButtonBox["Vectors and Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/VectorsAndMatrices"]], "Tutorials",
 CellID->4237],

Cell[TextData[ButtonBox["Constructing Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ConstructingMatrices"]], "Tutorials",
 CellID->12359]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with DiagonalMatrix",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=DiagonalMatrix"]\
, None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->11429198],

Cell[TextData[ButtonBox["Constructing Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ConstructingMatrices"]], "MoreAbout",
 CellID->232151804],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->177645340],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->356009644]
}, Open  ]],

Cell["New in 1  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"DiagonalMatrix - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 37, 59.3044956}", "context" -> "System`", 
    "keywords" -> {
     "identity matrix", "matrix diagonal", "special matrices", 
      "structured matrix"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "DiagonalMatrix[list] gives a matrix with the elements of list on the \
leading diagonal, and 0 elsewhere. DiagonalMatrix[list, k] gives a matrix \
with the elements of list on the k\\[Null]^th diagonal.", 
    "synonyms" -> {"diagonal matrix"}, "title" -> "DiagonalMatrix", "type" -> 
    "Symbol", "uri" -> "ref/DiagonalMatrix"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[11142, 302, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->303212492]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 52883, 1820}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2701, 61, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3324, 88, 4198, 83, 70, "ObjectNameGrid"],
Cell[7525, 173, 1289, 40, 70, "Usage",
 CellID->10792]
}, Open  ]],
Cell[CellGroupData[{
Cell[8851, 218, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[9329, 235, 967, 34, 70, "Notes",
 CellID->18190670],
Cell[10299, 271, 806, 26, 70, "Notes",
 CellID->210868340]
}, Closed]],
Cell[CellGroupData[{
Cell[11142, 302, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->303212492],
Cell[CellGroupData[{
Cell[11527, 316, 148, 5, 70, "ExampleSection",
 CellID->681901445],
Cell[11678, 323, 71, 1, 70, "ExampleText",
 CellID->223157580],
Cell[CellGroupData[{
Cell[11774, 328, 204, 7, 28, "Input",
 CellID->59463002],
Cell[11981, 337, 757, 23, 78, "Output",
 CellID->341690366]
}, Open  ]],
Cell[12753, 363, 64, 1, 70, "ExampleText",
 CellID->4138244],
Cell[CellGroupData[{
Cell[12842, 368, 221, 8, 28, "Input",
 CellID->262164887],
Cell[13066, 378, 772, 17, 78, "Output",
 Evaluatable->False,
 CellID->235509352]
}, Open  ]],
Cell[13853, 398, 63, 1, 70, "ExampleText",
 CellID->37778175],
Cell[CellGroupData[{
Cell[13941, 403, 238, 8, 70, "Input",
 CellID->330982464],
Cell[14182, 413, 757, 23, 78, "Output",
 CellID->121459073]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14988, 442, 220, 7, 70, "ExampleSection",
 CellID->797378833],
Cell[15211, 451, 258, 8, 70, "ExampleText",
 CellID->454727216],
Cell[15472, 461, 178, 6, 70, "Input",
 CellID->462087911],
Cell[15653, 469, 64, 1, 70, "ExampleText",
 CellID->288120164],
Cell[CellGroupData[{
Cell[15742, 474, 143, 4, 70, "Input",
 CellID->209775268],
Cell[15888, 480, 807, 25, 96, "Output",
 CellID->398186906]
}, Open  ]],
Cell[16710, 508, 63, 1, 70, "ExampleText",
 CellID->247596],
Cell[CellGroupData[{
Cell[16798, 513, 172, 5, 70, "Input",
 CellID->732459161],
Cell[16973, 520, 791, 23, 78, "Output",
 CellID->7047466]
}, Open  ]],
Cell[17779, 546, 78, 1, 70, "ExampleText",
 CellID->562264309],
Cell[CellGroupData[{
Cell[17882, 551, 196, 6, 70, "Input",
 CellID->9652364],
Cell[18081, 559, 811, 23, 78, "Output",
 CellID->14882233]
}, Open  ]],
Cell[18907, 585, 124, 3, 70, "ExampleDelimiter",
 CellID->62990351],
Cell[19034, 590, 505, 18, 70, "ExampleText",
 CellID->627060822],
Cell[CellGroupData[{
Cell[19564, 612, 175, 5, 70, "Input",
 CellID->358264379],
Cell[19742, 619, 459, 16, 36, "Output",
 CellID->348576475]
}, Open  ]],
Cell[CellGroupData[{
Cell[20238, 640, 109, 3, 70, "Input",
 CellID->77176116],
Cell[20350, 645, 487, 17, 36, "Output",
 CellID->396456531]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20886, 668, 227, 7, 70, "ExampleSection",
 CellID->639571560],
Cell[21116, 677, 118, 3, 70, "ExampleText",
 CellID->166912557],
Cell[CellGroupData[{
Cell[21259, 684, 287, 10, 70, "Input",
 CellID->19742325],
Cell[21549, 696, 1037, 21, 70, "Output",
 Evaluatable->False,
 CellID->933661938]
}, Open  ]],
Cell[22601, 720, 268, 9, 70, "Input",
 CellID->99672092],
Cell[CellGroupData[{
Cell[22894, 733, 182, 6, 70, "Input",
 CellID->265145035],
Cell[23079, 741, 1618, 31, 70, "Output",
 Evaluatable->False,
 CellID->18795566]
}, Open  ]],
Cell[24712, 775, 125, 3, 70, "ExampleDelimiter",
 CellID->305041818],
Cell[24840, 780, 127, 3, 70, "ExampleText",
 CellID->813739165],
Cell[CellGroupData[{
Cell[24992, 787, 160, 5, 70, "Input",
 CellID->80797988],
Cell[25155, 794, 467, 14, 36, "Output",
 CellID->571734619]
}, Open  ]],
Cell[CellGroupData[{
Cell[25659, 813, 182, 6, 70, "Input",
 CellID->44890258],
Cell[25844, 821, 887, 26, 54, "Output",
 CellID->266495205]
}, Open  ]],
Cell[CellGroupData[{
Cell[26768, 852, 189, 5, 70, "Input",
 CellID->1028688765],
Cell[26960, 859, 590, 19, 36, "Output",
 CellID->41367775]
}, Open  ]],
Cell[27565, 881, 125, 3, 70, "ExampleDelimiter",
 CellID->487887351],
Cell[27693, 886, 66, 1, 70, "ExampleText",
 CellID->40787963],
Cell[27762, 889, 517, 17, 70, "Input",
 CellID->516754666],
Cell[CellGroupData[{
Cell[28304, 910, 173, 5, 70, "Input",
 CellID->520724121],
Cell[28480, 917, 796, 17, 70, "Output",
 Evaluatable->False,
 CellID->312962934]
}, Open  ]],
Cell[29291, 937, 125, 3, 70, "ExampleDelimiter",
 CellID->426947172],
Cell[29419, 942, 236, 9, 70, "ExampleText",
 CellID->22957795],
Cell[CellGroupData[{
Cell[29680, 955, 780, 27, 70, "Input",
 CellID->4728434],
Cell[30463, 984, 1190, 38, 108, "Output",
 CellID->133538661]
}, Open  ]],
Cell[31668, 1025, 203, 8, 70, "ExampleText",
 CellID->415918509],
Cell[CellGroupData[{
Cell[31896, 1037, 968, 32, 70, "Input",
 CellID->25448404],
Cell[32867, 1071, 1293, 25, 70, "Output",
 Evaluatable->False,
 CellID->329110931]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34209, 1102, 237, 7, 70, "ExampleSection",
 CellID->553509347],
Cell[34449, 1111, 347, 12, 70, "ExampleText",
 CellID->51040628],
Cell[CellGroupData[{
Cell[34821, 1127, 265, 8, 70, "Input",
 CellID->583471562],
Cell[35089, 1137, 161, 5, 36, "Output",
 CellID->463983632]
}, Open  ]],
Cell[35265, 1145, 125, 3, 70, "ExampleDelimiter",
 CellID->426657608],
Cell[35393, 1150, 97, 1, 70, "ExampleText",
 CellID->207737819],
Cell[35493, 1153, 143, 5, 70, "Input",
 CellID->112880689],
Cell[35639, 1160, 588, 22, 70, "ExampleText",
 CellID->12002127],
Cell[CellGroupData[{
Cell[36252, 1186, 239, 7, 70, "Input",
 CellID->45601439],
Cell[36494, 1195, 161, 5, 36, "Output",
 CellID->243171569]
}, Open  ]],
Cell[CellGroupData[{
Cell[36692, 1205, 248, 7, 70, "Input",
 CellID->89372165],
Cell[36943, 1214, 160, 5, 36, "Output",
 CellID->26950582]
}, Open  ]],
Cell[CellGroupData[{
Cell[37140, 1224, 306, 10, 70, "Input",
 CellID->569204146],
Cell[37449, 1236, 160, 5, 36, "Output",
 CellID->30968656]
}, Open  ]],
Cell[37624, 1244, 307, 12, 70, "ExampleText",
 CellID->286928],
Cell[CellGroupData[{
Cell[37956, 1260, 198, 6, 70, "Input",
 CellID->40665350],
Cell[38157, 1268, 160, 5, 36, "Output",
 CellID->37953274]
}, Open  ]],
Cell[CellGroupData[{
Cell[38354, 1278, 196, 6, 70, "Input",
 CellID->69797476],
Cell[38553, 1286, 161, 5, 36, "Output",
 CellID->395442248]
}, Open  ]],
Cell[38729, 1294, 124, 3, 70, "ExampleDelimiter",
 CellID->38318280],
Cell[38856, 1299, 343, 12, 70, "ExampleText",
 CellID->69651662],
Cell[39202, 1313, 175, 6, 70, "Input",
 CellID->107517729],
Cell[CellGroupData[{
Cell[39402, 1323, 171, 5, 70, "Input",
 CellID->118792847],
Cell[39576, 1330, 160, 5, 36, "Output",
 CellID->97464366]
}, Open  ]],
Cell[39751, 1338, 233, 8, 70, "ExampleText",
 CellID->507340758],
Cell[39987, 1348, 149, 5, 70, "Input",
 CellID->289927538],
Cell[CellGroupData[{
Cell[40161, 1357, 170, 5, 70, "Input",
 CellID->17959681],
Cell[40334, 1364, 161, 5, 36, "Output",
 CellID->112449460]
}, Open  ]],
Cell[40510, 1372, 124, 3, 70, "ExampleDelimiter",
 CellID->71533622],
Cell[40637, 1377, 121, 3, 70, "ExampleText",
 CellID->51548694],
Cell[CellGroupData[{
Cell[40783, 1384, 252, 8, 70, "Input",
 CellID->49370446],
Cell[41038, 1394, 643, 15, 70, "Output",
 Evaluatable->False,
 CellID->268793304]
}, Open  ]],
Cell[CellGroupData[{
Cell[41718, 1414, 235, 8, 70, "Input",
 CellID->107050031],
Cell[41956, 1424, 643, 15, 70, "Output",
 Evaluatable->False,
 CellID->287165353]
}, Open  ]],
Cell[42614, 1442, 125, 3, 70, "ExampleDelimiter",
 CellID->257787544],
Cell[42742, 1447, 683, 25, 70, "ExampleText",
 CellID->411244880],
Cell[43428, 1474, 444, 15, 70, "Input",
 CellID->596419710],
Cell[CellGroupData[{
Cell[43897, 1493, 336, 10, 70, "Input",
 CellID->192165093],
Cell[44236, 1505, 208, 7, 36, "Output",
 CellID->95152814]
}, Open  ]],
Cell[44459, 1515, 125, 3, 70, "ExampleDelimiter",
 CellID->965313707],
Cell[44587, 1520, 461, 16, 70, "ExampleText",
 CellID->243960997],
Cell[45051, 1538, 313, 11, 70, "Input",
 CellID->192204259],
Cell[CellGroupData[{
Cell[45389, 1553, 785, 26, 70, "Input",
 CellID->505019813],
Cell[46177, 1581, 161, 5, 36, "Output",
 CellID->293677160]
}, Open  ]],
Cell[46353, 1589, 340, 13, 70, "ExampleText",
 CellID->192825226],
Cell[CellGroupData[{
Cell[46718, 1606, 865, 29, 70, "Input",
 CellID->30548269],
Cell[47586, 1637, 161, 5, 36, "Output",
 CellID->462300011]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[47808, 1649, 311, 9, 70, "SeeAlsoSection",
 CellID->17552],
Cell[48122, 1660, 1190, 44, 70, "SeeAlso",
 CellID->25383]
}, Open  ]],
Cell[CellGroupData[{
Cell[49349, 1709, 313, 9, 70, "TutorialsSection",
 CellID->7726],
Cell[49665, 1720, 148, 3, 70, "Tutorials",
 CellID->4237],
Cell[49816, 1725, 152, 3, 70, "Tutorials",
 CellID->12359]
}, Open  ]],
Cell[CellGroupData[{
Cell[50005, 1733, 305, 8, 70, "RelatedLinksSection"],
Cell[50313, 1743, 370, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[50720, 1760, 318, 9, 70, "MoreAboutSection",
 CellID->11429198],
Cell[51041, 1771, 153, 3, 70, "MoreAbout",
 CellID->232151804],
Cell[51197, 1776, 163, 3, 70, "MoreAbout",
 CellID->177645340],
Cell[51363, 1781, 198, 4, 70, "MoreAbout",
 CellID->356009644]
}, Open  ]],
Cell[51576, 1788, 50, 0, 70, "History"],
Cell[51629, 1790, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

