(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     43660,       1630]
NotebookOptionsPosition[     35286,       1336]
NotebookOutlinePosition[     36574,       1372]
CellTagsIndexPosition[     36488,       1367]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Partitioning Data into Clusters" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/PartitioningDataIntoClusters"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Partitioning Data into Clusters\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SquaredEuclideanDistance" :> 
          Documentation`HelpLookup["paclet:ref/SquaredEuclideanDistance"], 
          "ManhattanDistance" :> 
          Documentation`HelpLookup["paclet:ref/ManhattanDistance"], 
          "ChebyshevDistance" :> 
          Documentation`HelpLookup["paclet:ref/ChebyshevDistance"], 
          "BrayCurtisDistance" :> 
          Documentation`HelpLookup["paclet:ref/BrayCurtisDistance"], 
          "CanberraDistance" :> 
          Documentation`HelpLookup["paclet:ref/CanberraDistance"], "Norm" :> 
          Documentation`HelpLookup["paclet:ref/Norm"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"SquaredEuclideanDistance\"\>", 
       2->"\<\"ManhattanDistance\"\>", 3->"\<\"ChebyshevDistance\"\>", 
       4->"\<\"BrayCurtisDistance\"\>", 5->"\<\"CanberraDistance\"\>", 
       6->"\<\"Norm\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Exploratory Data Analysis" :> 
          Documentation`HelpLookup["paclet:guide/DataClassification"], 
          "Distance and Dissimilarity Measures" :> 
          Documentation`HelpLookup[
           "paclet:guide/DistanceAndDissimilarityMeasures"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Numerical Data Handling" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumericalDataHandling"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Exploratory Data Analysis\"\>", 
       2->"\<\"Distance and Dissimilarity Measures\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       4->"\<\"New in 6.0: Numerical Data Handling\"\>", 
       5->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["EuclideanDistance", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["EuclideanDistance",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/EuclideanDistance"], "[", 
       RowBox[{
        StyleBox["u", "TI"], ",", 
        StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator] gives the Euclidean distance between vectors ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->19629]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["EuclideanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/EuclideanDistance"], "[", 
   RowBox[{
    StyleBox["u", "TI"], ",", 
    StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Norm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Norm"], "[", 
   RowBox[{
    StyleBox["u", "TI"], "-", 
    StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->214729278,
  ButtonNote->"214729278"]
}], "Notes",
 CellID->24221]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->313161395],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->145081740],

Cell["Euclidean distance between two vectors:", "ExampleText",
 CellID->235641551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->196650888],

Cell[BoxData[
 SqrtBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"c", "-", "z"}], "]"}], "2"]}]]], "Output",
 ImageSize->{250, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->111639942]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->125461358],

Cell["Euclidean distance between numeric vectors:", "ExampleText",
 CellID->319587507],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "4", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190454384],

Cell[BoxData[
 SqrtBox["14"]], "Output",
 ImageSize->{35, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->279466979]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29421],

Cell["Compute distance between any vectors of equal length:", "ExampleText",
 CellID->86273121],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EuclideanDistance", "[", 
  RowBox[{
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", "100"}], "]"}], ",", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", "100"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->488208944],

Cell[BoxData["20.487668134192884`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->157290585]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->204991544],

Cell["Compute distance between vectors of any precision:", "ExampleText",
 CellID->14044010],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EuclideanDistance", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "5", ",", "2", ",", "3", ",", "10"}], "}"}], ",", 
     "50"}], "]"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"4", ",", "15", ",", "20", ",", "5", ",", "5"}], "}"}], ",", 
     "50"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->323525768],

Cell[BoxData["21.4941852602046770392440105754817129959638184678447704189519`\
49.39794000867204"], "Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14070958]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27726],

Cell["Cluster data using Euclidean distance:", "ExampleText",
 CellID->245478961],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindClusters", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"DistanceFunction", "\[Rule]", "EuclideanDistance"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->108189764],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"5", ",", "10"}], "}"}], "}"}]}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->474254754]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->124651963],

Cell["Demonstrate the triangle inequality:", "ExampleText",
 CellID->162085005],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d1", "=", 
  RowBox[{"EuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->395493032],

Cell[BoxData[
 RowBox[{"Abs", "[", 
  RowBox[{"b", "-", "c"}], "]"}]], "Output",
 ImageSize->{66, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9635856]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d2", "=", 
  RowBox[{"EuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "c"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"d", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13424126],

Cell[BoxData[
 RowBox[{"Abs", "[", 
  RowBox[{"a", "-", "d"}], "]"}]], "Output",
 ImageSize->{66, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->537624074]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d3", "=", 
  RowBox[{"EuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"d", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->51893016],

Cell[BoxData[
 SqrtBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"b", "-", "c"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"a", "-", "d"}], "]"}], "2"]}]]], "Output",
 ImageSize->{170, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->377922225]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"d3", "<=", 
   RowBox[{"d1", "+", "d2"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->175054461],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->233061910]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28666],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["EuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EuclideanDistance"]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["Norm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Norm"]], "InlineFormula"],
 " of a difference:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->214729278],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->218507111],

Cell[BoxData[
 SqrtBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"c", "-", "z"}], "]"}], "2"]}]]], "Output",
 ImageSize->{250, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->394168908]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], "-", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21346996],

Cell[BoxData[
 SqrtBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"c", "-", "z"}], "]"}], "2"]}]]], "Output",
 ImageSize->{250, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->109356102]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->286900690],

Cell[TextData[{
 "The square of ",
 Cell[BoxData[
  ButtonBox["EuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EuclideanDistance"]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  ButtonBox["SquaredEuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SquaredEuclideanDistance"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->555367459],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"EuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}], "^", "2"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->325280469],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"c", "-", "z"}], "]"}], "2"]}]], "Output",
 ImageSize->{231, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->467583752]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaredEuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->283254267],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"c", "-", "z"}], "]"}], "2"]}]], "Output",
 ImageSize->{231, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->8804327]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->263875481],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["EuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EuclideanDistance"]], "InlineFormula"],
 " is greater than or equal to ",
 Cell[BoxData[
  ButtonBox["ChebyshevDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChebyshevDistance"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->581795014],

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", "=", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->625730726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"EuclideanDistance", "[", 
    RowBox[{"u", ",", "v"}], "]"}], "\[GreaterEqual]", 
   RowBox[{"ChebyshevDistance", "[", 
    RowBox[{"u", ",", "v"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->196694015],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->559711902]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23098092],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["CosineDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CosineDistance"]], "InlineFormula"],
 " includes a dot product scaled by Euclidean distances from the origin:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->38175833],

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", "=", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->201109570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"scale", "=", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"EuclideanDistance", "[", 
     RowBox[{"u", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "]"}], 
    RowBox[{"EuclideanDistance", "[", 
     RowBox[{"v", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "]"}]}], ")"}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->3304662],

Cell[BoxData[
 RowBox[{
  SqrtBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"Abs", "[", "a", "]"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", "b", "]"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", "c", "]"}], "2"]}]], " ", 
  SqrtBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"Abs", "[", "x", "]"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", "y", "]"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", "z", "]"}], "2"]}]]}]], "Output",
 ImageSize->{381, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->206354649]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"1", "-", 
   RowBox[{
    RowBox[{"u", ".", "v"}], "/", "scale"}]}], "\[Equal]", 
  RowBox[{"CosineDistance", "[", 
   RowBox[{"u", ",", "v"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->45975943],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->24522044]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->680253860],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["CorrelationDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CorrelationDistance"]], "InlineFormula"],
 " includes a dot product scaled by Euclidean distances from means:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->776019056],

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", "=", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->26318500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"scale", "=", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"EuclideanDistance", "[", 
     RowBox[{"u", ",", 
      RowBox[{"Mean", "[", "u", "]"}]}], "]"}], 
    RowBox[{"EuclideanDistance", "[", 
     RowBox[{"v", ",", 
      RowBox[{"Mean", "[", "v", "]"}]}], "]"}]}], ")"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->211376369],

Cell[BoxData[
 RowBox[{
  SqrtBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"Abs", "[", 
      RowBox[{"a", "+", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], "]"}], "2"], 
    "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", 
      RowBox[{"b", "+", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], "]"}], "2"], 
    "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", 
      RowBox[{
       RowBox[{
        FractionBox["1", "3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}], "+", "c"}], 
      "]"}], "2"]}]], " ", 
  SqrtBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"Abs", "[", 
      RowBox[{"x", "+", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "x"}], "-", "y", "-", "z"}], ")"}]}]}], "]"}], "2"], 
    "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", 
      RowBox[{"y", "+", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "x"}], "-", "y", "-", "z"}], ")"}]}]}], "]"}], "2"], 
    "+", 
    SuperscriptBox[
     RowBox[{"Abs", "[", 
      RowBox[{
       RowBox[{
        FractionBox["1", "3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "x"}], "-", "y", "-", "z"}], ")"}]}], "+", "z"}], 
      "]"}], "2"]}]]}]], "Output",
 ImageSize->{483, 78},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->260814563]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CorrelationDistance", "[", 
   RowBox[{"u", ",", "v"}], "]"}], "==", 
  RowBox[{"1", "-", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"u", "-", 
       RowBox[{"Mean", "[", "u", "]"}]}], ")"}], ".", 
     RowBox[{"(", 
      RowBox[{"v", "-", 
       RowBox[{"Mean", "[", "v", "]"}]}], ")"}]}], "/", 
    "scale"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->284988311],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->351253309]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->70099947],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["StandardDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/StandardDeviation"]], "InlineFormula"],
 " as a ",
 Cell[BoxData[
  ButtonBox["EuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EuclideanDistance"]], "InlineFormula"],
 " from the ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->394865487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->463545505],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"a", ",", "b", ",", "c"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->438684048]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mean", "=", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Mean", "[", "data", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"Length", "[", "data", "]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->132468290],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{264, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->333744200]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"StandardDeviation", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->917053855],

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   SqrtBox["6"]], 
  RowBox[{"(", 
   RowBox[{"\[Sqrt]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{"2", " ", "a"}], "-", "b", "-", "c"}], ")"}], " ", 
       RowBox[{"Conjugate", "[", "a", "]"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "+", 
         RowBox[{"2", " ", "b"}], "-", "c"}], ")"}], " ", 
       RowBox[{"Conjugate", "[", "b", "]"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "-", "b", "+", 
         RowBox[{"2", " ", "c"}]}], ")"}], " ", 
       RowBox[{"Conjugate", "[", "c", "]"}]}]}], ")"}]}], ")"}]}]], "Output",
 ImageSize->{563, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->12335711]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"%", "==", 
   RowBox[{
    RowBox[{"EuclideanDistance", "[", 
     RowBox[{"data", ",", "mean"}], "]"}], "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{
      RowBox[{"Length", "[", "data", "]"}], "-", "1"}], "]"}]}]}], "//", 
  "FullSimplify"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->42727618],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->280996924]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->59709880],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["EuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EuclideanDistance"]], "InlineFormula"],
 " computed from ",
 Cell[BoxData[
  ButtonBox["RootMeanSquare",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootMeanSquare"]], "InlineFormula"],
 " of a difference:"
}], "ExampleText",
 CellID->149042657],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "4", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->290459063],

Cell[BoxData[
 SqrtBox["14"]], "Output",
 ImageSize->{35, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->16936519]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RootMeanSquare", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], "-", 
    RowBox[{"{", 
     RowBox[{"2", ",", "4", ",", "6"}], "}"}]}], "]"}], " ", 
  RowBox[{"Sqrt", "[", 
   RowBox[{"Length", "[", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->159972835],

Cell[BoxData[
 SqrtBox["14"]], "Output",
 ImageSize->{35, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->178520310]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->10116],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SquaredEuclideanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SquaredEuclideanDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ManhattanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ManhattanDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChebyshevDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChebyshevDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BrayCurtisDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BrayCurtisDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CanberraDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CanberraDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Norm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Norm"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->32525]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->3309],

Cell[TextData[ButtonBox["Partitioning Data into Clusters",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/PartitioningDataIntoClusters"]], "Tutorials",
 CellID->22256]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with EuclideanDistance",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
EuclideanDistance"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->636469],

Cell[TextData[ButtonBox["Exploratory Data Analysis",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DataClassification"]], "MoreAbout",
 CellID->211720070],

Cell[TextData[ButtonBox["Distance and Dissimilarity Measures",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DistanceAndDissimilarityMeasures"]], "MoreAbout",
 CellID->285631764],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->179591581],

Cell[TextData[ButtonBox["New in 6.0: Numerical Data Handling",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60NumericalDataHandling"]], "MoreAbout",
 CellID->614621375],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->208487546]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"EuclideanDistance - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 40, 43.6212020}", "context" -> "System`", 
    "keywords" -> {"Euclidean distance", "Euclidean norm"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "EuclideanDistance[u, v] gives the Euclidean distance between vectors u \
and v.", "synonyms" -> {"euclidean distance"}, "title" -> "EuclideanDistance",
     "type" -> "Symbol", "uri" -> "ref/EuclideanDistance"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7354, 213, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->313161395]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 36345, 1360}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3252, 73, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3875, 100, 1582, 37, 70, "ObjectNameGrid"],
Cell[5460, 139, 714, 23, 70, "Usage",
 CellID->19629]
}, Open  ]],
Cell[CellGroupData[{
Cell[6211, 167, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6689, 184, 628, 24, 70, "Notes",
 CellID->24221]
}, Closed]],
Cell[CellGroupData[{
Cell[7354, 213, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->313161395],
Cell[CellGroupData[{
Cell[7739, 227, 148, 5, 70, "ExampleSection",
 CellID->145081740],
Cell[7890, 234, 82, 1, 70, "ExampleText",
 CellID->235641551],
Cell[CellGroupData[{
Cell[7997, 239, 254, 8, 28, "Input",
 CellID->196650888],
Cell[8254, 249, 452, 16, 45, "Output",
 CellID->111639942]
}, Open  ]],
Cell[8721, 268, 125, 3, 70, "ExampleDelimiter",
 CellID->125461358],
Cell[8849, 273, 86, 1, 70, "ExampleText",
 CellID->319587507],
Cell[CellGroupData[{
Cell[8960, 278, 254, 8, 70, "Input",
 CellID->190454384],
Cell[9217, 288, 170, 6, 38, "Output",
 CellID->279466979]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9436, 300, 216, 7, 70, "ExampleSection",
 CellID->29421],
Cell[9655, 309, 95, 1, 70, "ExampleText",
 CellID->86273121],
Cell[CellGroupData[{
Cell[9775, 314, 266, 8, 70, "Input",
 CellID->488208944],
Cell[10044, 324, 176, 5, 36, "Output",
 CellID->157290585]
}, Open  ]],
Cell[10235, 332, 125, 3, 70, "ExampleDelimiter",
 CellID->204991544],
Cell[10363, 337, 92, 1, 70, "ExampleText",
 CellID->14044010],
Cell[CellGroupData[{
Cell[10480, 342, 427, 14, 70, "Input",
 CellID->323525768],
Cell[10910, 358, 238, 6, 36, "Output",
 CellID->14070958]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[11197, 370, 223, 7, 70, "ExampleSection",
 CellID->27726],
Cell[11423, 379, 81, 1, 70, "ExampleText",
 CellID->245478961],
Cell[CellGroupData[{
Cell[11529, 384, 477, 16, 70, "Input",
 CellID->108189764],
Cell[12009, 402, 498, 18, 36, "Output",
 CellID->474254754]
}, Open  ]],
Cell[12522, 423, 125, 3, 70, "ExampleDelimiter",
 CellID->124651963],
Cell[12650, 428, 79, 1, 70, "ExampleText",
 CellID->162085005],
Cell[CellGroupData[{
Cell[12754, 433, 263, 9, 70, "Input",
 CellID->395493032],
Cell[13020, 444, 208, 7, 36, "Output",
 CellID->9635856]
}, Open  ]],
Cell[CellGroupData[{
Cell[13265, 456, 262, 9, 70, "Input",
 CellID->13424126],
Cell[13530, 467, 210, 7, 36, "Output",
 CellID->537624074]
}, Open  ]],
Cell[CellGroupData[{
Cell[13777, 479, 262, 9, 70, "Input",
 CellID->51893016],
Cell[14042, 490, 359, 13, 45, "Output",
 CellID->377922225]
}, Open  ]],
Cell[CellGroupData[{
Cell[14438, 508, 155, 5, 70, "Input",
 CellID->175054461],
Cell[14596, 515, 161, 5, 36, "Output",
 CellID->233061910]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14806, 526, 233, 7, 70, "ExampleSection",
 CellID->28666],
Cell[15042, 535, 394, 13, 70, "ExampleText",
 CellID->214729278],
Cell[CellGroupData[{
Cell[15461, 552, 254, 8, 70, "Input",
 CellID->218507111],
Cell[15718, 562, 452, 16, 45, "Output",
 CellID->394168908]
}, Open  ]],
Cell[CellGroupData[{
Cell[16207, 583, 240, 8, 70, "Input",
 CellID->21346996],
Cell[16450, 593, 452, 16, 45, "Output",
 CellID->109356102]
}, Open  ]],
Cell[16917, 612, 125, 3, 70, "ExampleDelimiter",
 CellID->286900690],
Cell[17045, 617, 375, 13, 70, "ExampleText",
 CellID->555367459],
Cell[CellGroupData[{
Cell[17445, 634, 282, 9, 70, "Input",
 CellID->325280469],
Cell[17730, 645, 431, 15, 36, "Output",
 CellID->467583752]
}, Open  ]],
Cell[CellGroupData[{
Cell[18198, 665, 261, 8, 70, "Input",
 CellID->283254267],
Cell[18462, 675, 429, 15, 36, "Output",
 CellID->8804327]
}, Open  ]],
Cell[18906, 693, 125, 3, 70, "ExampleDelimiter",
 CellID->263875481],
Cell[19034, 698, 415, 13, 70, "ExampleText",
 CellID->581795014],
Cell[19452, 713, 281, 10, 70, "Input",
 CellID->625730726],
Cell[CellGroupData[{
Cell[19758, 727, 281, 8, 70, "Input",
 CellID->196694015],
Cell[20042, 737, 161, 5, 36, "Output",
 CellID->559711902]
}, Open  ]],
Cell[20218, 745, 124, 3, 70, "ExampleDelimiter",
 CellID->23098092],
Cell[20345, 750, 306, 8, 70, "ExampleText",
 CellID->38175833],
Cell[20654, 760, 281, 10, 70, "Input",
 CellID->201109570],
Cell[CellGroupData[{
Cell[20960, 774, 412, 14, 70, "Input",
 CellID->3304662],
Cell[21375, 790, 627, 22, 45, "Output",
 CellID->206354649]
}, Open  ]],
Cell[CellGroupData[{
Cell[22039, 817, 239, 8, 70, "Input",
 CellID->45975943],
Cell[22281, 827, 160, 5, 36, "Output",
 CellID->24522044]
}, Open  ]],
Cell[22456, 835, 125, 3, 70, "ExampleDelimiter",
 CellID->680253860],
Cell[22584, 840, 312, 8, 70, "ExampleText",
 CellID->776019056],
Cell[22899, 850, 280, 10, 70, "Input",
 CellID->26318500],
Cell[CellGroupData[{
Cell[23204, 864, 352, 11, 70, "Input",
 CellID->211376369],
Cell[23559, 877, 1723, 64, 99, "Output",
 CellID->260814563]
}, Open  ]],
Cell[CellGroupData[{
Cell[25319, 946, 422, 15, 70, "Input",
 CellID->284988311],
Cell[25744, 963, 161, 5, 36, "Output",
 CellID->351253309]
}, Open  ]],
Cell[25920, 971, 124, 3, 70, "ExampleDelimiter",
 CellID->70099947],
Cell[26047, 976, 470, 17, 70, "ExampleText",
 CellID->394865487],
Cell[CellGroupData[{
Cell[26542, 997, 152, 5, 70, "Input",
 CellID->463545505],
Cell[26697, 1004, 213, 7, 36, "Output",
 CellID->438684048]
}, Open  ]],
Cell[CellGroupData[{
Cell[26947, 1016, 251, 8, 70, "Input",
 CellID->132468290],
Cell[27201, 1026, 538, 19, 51, "Output",
 CellID->333744200]
}, Open  ]],
Cell[CellGroupData[{
Cell[27776, 1050, 116, 3, 70, "Input",
 CellID->917053855],
Cell[27895, 1055, 863, 29, 74, "Output",
 CellID->12335711]
}, Open  ]],
Cell[CellGroupData[{
Cell[28795, 1089, 331, 11, 70, "Input",
 CellID->42727618],
Cell[29129, 1102, 161, 5, 36, "Output",
 CellID->280996924]
}, Open  ]],
Cell[29305, 1110, 124, 3, 70, "ExampleDelimiter",
 CellID->59709880],
Cell[29432, 1115, 363, 12, 70, "ExampleText",
 CellID->149042657],
Cell[CellGroupData[{
Cell[29820, 1131, 254, 8, 70, "Input",
 CellID->290459063],
Cell[30077, 1141, 169, 6, 38, "Output",
 CellID->16936519]
}, Open  ]],
Cell[CellGroupData[{
Cell[30283, 1152, 405, 13, 70, "Input",
 CellID->159972835],
Cell[30691, 1167, 170, 6, 38, "Output",
 CellID->178520310]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[30922, 1180, 311, 9, 70, "SeeAlsoSection",
 CellID->10116],
Cell[31236, 1191, 1527, 53, 70, "SeeAlso",
 CellID->32525]
}, Open  ]],
Cell[CellGroupData[{
Cell[32800, 1249, 313, 9, 70, "TutorialsSection",
 CellID->3309],
Cell[33116, 1260, 170, 3, 70, "Tutorials",
 CellID->22256]
}, Open  ]],
Cell[CellGroupData[{
Cell[33323, 1268, 305, 8, 70, "RelatedLinksSection"],
Cell[33631, 1278, 376, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[34044, 1295, 316, 9, 70, "MoreAboutSection",
 CellID->636469],
Cell[34363, 1306, 155, 3, 70, "MoreAbout",
 CellID->211720070],
Cell[34521, 1311, 179, 3, 70, "MoreAbout",
 CellID->285631764],
Cell[34703, 1316, 179, 3, 70, "MoreAbout",
 CellID->179591581],
Cell[34885, 1321, 175, 3, 70, "MoreAbout",
 CellID->614621375],
Cell[35063, 1326, 151, 3, 70, "MoreAbout",
 CellID->208487546]
}, Open  ]],
Cell[35229, 1332, 27, 0, 70, "History"],
Cell[35259, 1334, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

