(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    113457,       2645]
NotebookOptionsPosition[    105872,       2376]
NotebookOutlinePosition[    107241,       2414]
CellTagsIndexPosition[    107154,       2409]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ChiSquareDistribution" :> 
          Documentation`HelpLookup["paclet:ref/ChiSquareDistribution"], 
          "NoncentralFRatioDistribution" :> 
          Documentation`HelpLookup[
           "paclet:ref/NoncentralFRatioDistribution"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"ChiSquareDistribution\"\>", 
       2->"\<\"NoncentralFRatioDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/ContinuousStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Continuous Statistical Distributions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FRatioDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FRatioDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FRatioDistribution"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents an F-ratio distribution with ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     " numerator and ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     " denominator degrees of freedom."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->25762]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability density for value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " in an F-ratio distribution is proportional to ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{
     SuperscriptBox["x", 
      RowBox[{
       RowBox[{"n", "/", "2"}], "-", "1"}]], "(", 
     RowBox[{
      RowBox[{"x", "\[InvisibleTimes]", "n"}], "+", "m"}], ")"}], 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"(", 
       RowBox[{"n", "+", "m"}], ")"}]}], "/", "2"}]], TraditionalForm]], 
  "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", ">", "0"}], TraditionalForm]], "InlineMath"],
 ", and is zero for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "<", "0"}], TraditionalForm]], "InlineMath"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->19881,
  ButtonNote->"19881"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 "For integer ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ", the F-ratio distribution gives the distribution of the ratio of variances \
for samples from normal distributions."
}], "Notes",
 CellID->168709853],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FRatioDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FRatioDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " to be any positive real numbers."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->25743],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FRatioDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FRatioDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->6831,
  ButtonNote->"6831"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->314321293],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->81272351],

Cell["The mean and variance of an F-ratio distribution:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->6831],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"FRatioDistribution", "[", 
   RowBox[{"n", ",", "m"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23031],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1VktOwzAQTR2nDdAPKqhiWYkFO44BG1hygKhCaimIUCroLktWVZc9Qpc9
AkfoMXoMYzt24k7GqenHUuLMjOfNzIs98mM07j+/ReNBL+rej6K4P+h9du/e
R1zlVzzPm/Ln+srj34x/yWfKGJ/Nd52/A27yAimqIb59PhNlFTPlM+Wz0PtK
1napUL5Q9rei+wC1YcjaTgAqMTJQBjqZTH7lWK/XNR2EyZGuCzJZAZVoDV+Z
fA2sUmthHN9AzFfRPbLxS6K3seglMkGqozmDfpIkShsemr+SrFD74TksyeDc
hcetdR6HycCSv02/K3P69NliNXfmSHcQN26yPUpzvbdpcfu/6Cqi+XBiCc8F
Rw9QLcYazGGTLYsW546aa2GPpECnq4K9Vcsko0EOZbF5Y9Ew5JaF181ImmkC
ctq+ZxpAhvZbIF8U/ic8x5do3Kqqq66yvAFeQhZtuoN6/xxVm1Ywn88Xi0VW
gZHbh7Eq922aPJ+AXWe+vaHhH8fxbDaTLiztGHCHyBtAtQgzRCuAJ4Q4ZuOE
Yz0f4o3jmPtJOHw7Rfva+0++sHT/wD4UsrIOKmso9LKqqjY0rBmXFqT0JsNv
T+o8tyVbXFb2BzRSqCKh96B/ZnuK/UFHDHiHOANyJceiy+UykWO1WrVQtCdW
7H3iTIlrtUBK0Tqor5kf5Bz3eGVmv+uw4k4dyc/0xu5V/gC/1hRk\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{141, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->21303637]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"FRatioDistribution", "[", 
   RowBox[{"n", ",", "m"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4915],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9WM1uEzEQdvYnm0JLEdCqByQqEHCCQ1UQAoTgAvTAQ0QVIgUhotJDbrnB
qcoxj5BjHoFHyGPkmEcwttdOZr8de3cT6EpdZ+zxN+Nvd362n7oXvc/fuxdn
p93DD+fdfu/s9Ofh+x/naipuCSEu1d+TA6F+S/XL/F1KqUZ6j/U9GQwGf8w1
n89/rebj4XAozSXa6pYqCJGaZXsZJTVGdlWPiRoTNer52MpuPYfNL5TjSvQY
UHeI7NYjQI2IB+VTdZwJKxutdClbmMAs2Wtcz0DL6magGxPElVaygTdxwPot
znpAjpjTJVfDX8Ardv3fcxjw4GYdHivP+X+YTD3+++bXZc7Fns/WjbU5cvmj
HjfLdzRZzYviSr3ny2pFjo9aLPG+8OgpO8uxhj4U2fLM8twlVBczZAJz7lSY
WZ0cLWkwl13x7eascci7Hl6LlhzTEfhU/c7sgIzrT0G+XXqeGMd3WLuuTm5b
Lx/DLi3rNL3P7v59pbPXwLc9kNsV6+9Y1C3Q2q5gHiOqSv8taxW1tOx6B7EF
sULv5eoHcibJe9YISfPH1dKjWidwnQtFNQGhb+2y+apzODyM1LQmnulhChFx
bIbczng8nkwmknn+mSQd2EZPgq2RjRDDZ/C/w7R/W+J6s6pRXucsjRBpJ/yM
nKLf749GI8nEd2ZRsdpgBNK4F06/VG/eyDo5BTvf6wFbnIzZAFl4xXpxUkvr
I5zz9QZYX2We15FLfV4ttthdOTuz2WxoLnGib3H++PTSHmF+5afLK5qblsUO
4S8WiwL+dDoN4ncsvrbT9vr+Lehbh6zaDcLXo/BI2O+bmCxX5BLaI9h3j5Gx
Y6FfSH4/HwJSVaXCd/95LSu71j++/+QyU73qVKoHwtdJvpCkbrH+lbJHpZ/B
6pOiu+3irMXE2H/J+sd+pVX617y2NPlC4zML5yntefPpmImF5tWj9BXK5Cqe
T/q8NShm+FAG1/oHsM7nT+SiiRX9cx9kwnii8l2e/1SivQt6X4zeffC5Z8YH
rG7+fzPR+gshfR+G\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{141, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->210704753]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1581],

Cell["\<\
The probability density function is related to the beta function:\
\>", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->19881],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"n", ",", "m"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27132],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["m", 
    RowBox[{"m", "/", "2"}]], " ", 
   SuperscriptBox["n", 
    RowBox[{"n", "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["n", "2"]}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"m", "+", 
      RowBox[{"n", " ", "x"}]}], ")"}], 
    RowBox[{
     FractionBox["1", "2"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "m"}], "-", "n"}], ")"}]}]]}], 
  RowBox[{"Beta", "[", 
   RowBox[{
    FractionBox["n", "2"], ",", 
    FractionBox["m", "2"]}], "]"}]]], "Output",
 ImageSize->{186, 48},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->140128221]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"FRatioDistribution", "[", 
      RowBox[{"5", ",", "15"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19495],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->128169820]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->30231],

Cell["\<\
Generate a set of pseudorandom numbers that have the F-ratio distribution:\
\>", "ExampleText",
 CellID->25172],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomReal", "[", 
  RowBox[{
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"2", ",", "5"}], "]"}], ",", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27982],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.4675798720442913`", ",", "1.2426428451718272`", ",", 
   "6.4173348221127124`", ",", "2.7628865431731504`", ",", 
   "1.848744194417356`"}], "}"}]], "Output",
 ImageSize->{308, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->893117492]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10982],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"FRatioDistribution", "[", 
   RowBox[{"n", ",", "m"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7393],

Cell[BoxData[
 RowBox[{"\[Piecewise]", GridBox[{
    {
     FractionBox[
      RowBox[{"2", " ", 
       SqrtBox["2"], " ", 
       SqrtBox[
        RowBox[{
         RowBox[{"-", "4"}], "+", "m"}]], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "2"}], "+", "m", "+", 
         RowBox[{"2", " ", "n"}]}], ")"}]}], 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "6"}], "+", "m"}], ")"}], " ", 
       SqrtBox["n"], " ", 
       SqrtBox[
        RowBox[{
         RowBox[{"-", "2"}], "+", "m", "+", "n"}]]}]], 
     RowBox[{"m", ">", "6"}]},
    {"Indeterminate", 
     TagBox["True",
      "PiecewiseDefault",
      AutoDelete->False,
      DeletionWarning->True]}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxItemSize->{
    "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.84]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}]}]], "Output",
 ImageSize->{174, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->657256644]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"FRatioDistribution", "[", 
   RowBox[{"n", ",", "m"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2728],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWc1uEzEQdvYnTaClIEFVURBFPSBAos8ASHBC0EeIqqL2gIhKD7nlyo9K
jrlzCeKSOxceIfAUOeYRlrXX3ng/j+3d7pawUrw79nhmPOMZzzivemfHR+96
ZyeHvd2Xp73+8cnhh90X70/TrrDFWKvDGNvfZul3kn6JH3+66S/7xK+Qt9Fg
MPglnvl8zseC5Vg4HA4FjYS10yZOSbJYDMtHIKXvQI7yd5S+o/TN+0MJq/GM
bPYgHHqph0D1owar8QCoBpoE5qo6ioWEBVacw5KMo1ebK0RfAyyJuwa4oUZx
iRXVkCZ0cD+nuDvggFhd9G/055CKHG9ehw4JvpTRo3edl6PJ2CK/rf+imlO+
Z+P16cI6UvGjnG7yPRot+1lxpJx9SaxA6aOUlmhZaOox2UtpDWUoasvSS+su
0nExQkbQp1aFkVXBQa4G8cgR22yKG0X5s0WvRU5K0wHI5N8zGwDj+BOAvxr2
RD++SfJV5+S6lPIBzPqWNjxMb5GzeUOvZ/Ujz2EltwBue8bRAvcAPiC5PgOs
dYDRjuifPvyrFfHxJH9DSv3UQ0XspS54sN6aZzIhRx4h/JSMCEDHEJ4X5j7l
pIry2GnSdkUNqTxNZYDLVuSJbVMEn4YaomjGdIoHRL+Y5Nsu9los/Vr0Z6sZ
j8eTySQh/EDfhzmV2juqAYpOfVX3gyp50IGmOXsEIWWwnpACuYTmcl/0Uypg
qGqpnoRGjklQ1fdVv98fjUb8E6O6HsU5k2tI2YHPPxEf4yvS34ZxOp66pOSf
GLVxvC11hmcQwjhvC2A6lvGGPjP5zuDT0aK/kywLQG0W4i+ZW5i7Hk+krE6a
zWZD8bCfvAkzc+e6MvJapaEOyYPOLa8AZ5XTZRIsFouCBNPp1ClBR0pAr7Kc
BFgj/mlsrR19DmaBGvwd4E1SApTTbXdc5Q7APwDeSMyc/HqFtdIWuOvhuleB
g/ILpIn+UDXH3AT4PqEZVw6K/PecdsHcHs+JpnNV8lzQ4BsebdTNdXF9D0mt
3E5k/rOs3IhIh7J3pKabyYzL5bBISazXmhPrVa2LP327suosHPVP2+4OORe9
zpZp65oRPPPJRbFLZ+5UVl2CXpwUb5uVVZCmaRN932YtfQv0f1QKqDeHTX37
1eNP+Q16Lc+sXwXTHornI7WfvCtuwMaXW91U8OCa1jaqwFrVUf16K2S2m9BS
ljd4xTaeK6rMKvix5R8epEj5DZesSjXHFL6hdV9NFwCMeQTOr1ojlstDdhKt
KnfMxnGftFRt6cqqkF/XMx93Ax8Pib1etXYtFz8wp8XK+62dSpRWeFnFlxaf
SEe/7+M2RTqPRD/ewuPdzRE5ax96sQZ7LLD4Z/a/N2v9BfiJkKo=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{264, 47},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->241610088]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->18362],

Cell["\<\
The fourth moment of a particular F-ratio distribution is finite:\
\>", "ExampleText",
 CellID->13279],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], ",", 
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"5", ",", "9"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6809],

Cell[BoxData[
 FractionBox["216513", "625"]], "Output",
 ImageSize->{53, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->479598763]
}, Open  ]],

Cell["\<\
However, some F-ratio distributions are too heavy-tailed to have fourth \
moments:\
\>", "ExampleText",
 CellID->3490],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], ",", 
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"5", ",", "8"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->10093],

Cell[BoxData[
 RowBox[{
  RowBox[{"Integrate", "::", "\<\"idiv\"\>"}], ":", 
  " ", "\<\"Integral of \\!\\(x\\^\\(11/2\\)\\/\\((8 + \\(\\(5\\\\ \
x\\)\\))\\)\\^\\(13/2\\)\\) does not converge on \\!\\({0, \[Infinity]}\\). \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Integrate/idiv\\\", \
ButtonNote -> \\\"Integrate::idiv\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->305322866],

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   SuperscriptBox["x", "4"], ",", 
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"5", ",", "8"}], "]"}], ",", "x"}], "]"}]], "Output",
 ImageSize->{317, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->102901604]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30364],

Cell[TextData[{
 "The 0.75 quantile of an F-ratio distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "5"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "=", "20"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->23047],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"5", ",", "20"}], "]"}], ",", "0.75"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29508],

Cell[BoxData["1.4499522904903044`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->174010482]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18847],

Cell[TextData[{
 "Compute the p\[Hyphen]value for an F\[Hyphen]test with ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " degrees of freedom:"
}], "ExampleText",
 CellID->813014373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"FRatioDistribution", "[", 
     RowBox[{"n", ",", "m"}], "]"}], ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->944228315],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"BetaRegularized", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"n", " ", "x"}], 
     RowBox[{"m", "+", 
      RowBox[{"n", " ", "x"}]}]], ",", 
    FractionBox["n", "2"], ",", 
    FractionBox["m", "2"]}], "]"}]}]], "Output",
 ImageSize->{228, 29},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->96535367]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->713828730],

Cell["\<\
Plot the cumulative distribution function of an F-ratio distribution:\
\>", "ExampleText",
 CellID->8505],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"FRatioDistribution", "[", 
      RowBox[{"4", ",", "10"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{
      RowBox[{"-", "1"}], "/", "10"}], ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28184],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 153},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->246096911]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->480],

Cell[TextData[{
 "A contour plot as both ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " are varied:"
}], "ExampleText",
 CellID->9234],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"FRatioDistribution", "[", 
      RowBox[{"5", ",", "m"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"m", ",", "1", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6698],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJx1fQVYlU3z96E7DindYHeCumML2KBiYWNgd4MBFraY2JgoNiYWioBKHcU6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   "], {{{}, 
     {RGBColor[0.3364482609601082, 0.13222588030198124`, 0.5898990485215532], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNVl1o1WUcfs//8/x3nNhNsZvFcIEfUbAbITxCw111bCV4Ee1ixukDT1QQ
pBKKNN3UligGSV8E0SHDsptF4CiixeaF1CojCLZ0cQY1poiuLmz1PDzPmBc/
3t95/7/vj+c9XU+9uPOFKIQwA4pBveUQnslDOFQKYQQfPse5AdT0+QBktqUh
lPFtOejsMN+P+0dBtyB3CcYy0Dj4QdydwXlvkA3arIKfxPelSLY77YP2x3G/
EEnnIvhlx1Ek8Ac6SLkihJeyEL7Dt3+gcx30E2gvYt+XS38Rv2+AIB5eLUmm
BaqmincClCWy8WeQ/AT4qzincG52bg/B3nr4uwA+hW4FOpsi6U1Yl7ZvOoZJ
+OoDPQn+cKJargPfRC73RKrPl4i9nsnXNO7XxtJ7EL7Ow8dYpHpS/l/I1CPV
mvxX0GuAboM/Dvtpojhej1TvHte4x3W+D7GPpvL/DmS+d3zVknwyzguOqwE6
D76Fc6Ak2RpoD2QK9t35z/r70SA/9FsLinsRfL2kk/aaQX1jD1quFfkFnB9C
/hf7aFnvLdxfQazbY9lgX5bCap3Z73n3cZt7+Sbm9jSoPVHN54Jq24bcX0Gt
RhP17mHHz1pSl/XkXS/ov6B693uGa9yFTLXPLEMbj+H+AO7PxZr3ftvZgfMJ
0F2e+d3m16Knn6TKtQP8OfBnwcfgP07Vk79h7yPwpyPZoW4NNABfz6Xaia2p
cubcrkkUN2Oege4Z8EPQ/QP8u6l29y/wH6SaizxVDlOu9y7Q/qD8xp3jxkhz
zR59g7vXstU95zf63WyZZfdgo+U7vC+8H4Le4Ux3rNGU96gb/tcztkSxsJbU
GboDA1hP5jkWtP+dxg32pe75/xG2f85U21vWofxW4w9lvk4lR5kl7zZxgP1h
fbmTu81T5tlMOZddf9Z+nbGuapvU2eH7ce8PZ77hOBjDZeR7d6y9YZ27YtX6
B/R6GnR/rN3hOeC9IU7S1rf4PpFL5z2c7+f6NoxzJF/FSp7EUdZ+3jj2SC5M
XMGkRe/LiPfutndn3hh4ljOZC3c2GCeIvcS5bmNdr/cl9VxOOmfmxhy5s7TN
3aZ9zvSsZ4uytE2cGYOfL3LpzOL8HbQlFpZQt2WcZszcW2IXMZ+4chKyp3L9
brjGxBPODN+PpnGImEEsIl4QNxp+v/h+0B7zIf6xT8TCl8v6XUGu+zLN1+O4
2wn6LNa+8OR+1Y1h9EFsnImFVcS6K8aoUei9URYW8xvzo17V88z52eQdW/Y7
Q1li1DXkdz1X3MSCy54f1nza78IuY+1+7yB7Qnus+W+eKe75Jb+5xJpfY80g
83g7US7d8PNpqjx6PL/s/RHjJuPh7BO/uHfc2aO4j4kXqNMa1OtEortjvu/z
TOzxG8g5I7YTcwu/1zXjDWUYS7vjmfMbyZ2izRHPEueKPO08Db8p/A4nynVl
hwb8ZrAv7MkW96Xm3aM/1qDLdZjzjF01tvA/AvGF/o/4nV55t5quyXCyihWD
xoU+58p38yR6fgrU5v8mhePn3hCjK65Vt3GPd4XfIMq2WZ64SJnY+8TdOOga
8NuC/5Nkxsy9mWaXc0ubJ1znTv+HYu96jfmsM8/cPG0Xts+5mvVsPQ+beaG+
EkeO+38F56ziPa36/wLnpuI4W67noLF0ZX8Yb+E60Fe782cM/wNF0VMo
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtk11oj1Ecx8/2PM85z3+NcmPtRk1TLki5IXEj3Fhh5ULtSt5qXlZiW/Iy
LzMzsxcX8pYS2Rq7mpTFBcKFsqIUDcuNKG5wI/l8+7r49vy+v+/vnOec3+97
Grbua95bHUJYB3Jwrgzhfgxhhrggcb0I4QB4mYVwlsJB9Ifof9Br0G+iHQJT
6MPotZUQhtCPVIXQjz5JvBOUaNtAF7WPUwgN1L8FB9HOs2eF2r7c/7qKHjP/
cxhtCNSizQL7idtZU8det9C7iC+U1i6BfuL1rH+KHnOfbQQ+O/MZS87XV/pu
vaAtOqdYa45SP0H9XGrfgB74B/gy4mlwEb4HPCM+QX038Sv0RfB3WsN+Lew/
qVp6sJK9H1Ezh3gL39fRNdKU+0L8Nbq3S+G1Oiu4AW+CVxFn4DJ8Lfw9tdPR
/54Hv8KebeAF/ztNbgbtc3S8EP0w8QbWP4ePZe5lZ/Rs1NOfxL+jZ7eK+jpq
68EIfDN8PnEjGIe3wFupjdypJ7cHNOuaimenmZ+Ebyz9r3rNHJ6jn8rtqeM6
e7QmT+xI7pF6c4z1i+FLwITWwpcTrwAP4K3wRmoXVDx75X4k10hbQ665tCfk
hbvwDuKm0t6TZ3T23eTO5L5D0tngWZVz6tWm0mvVM/V2gNzf6v89Tu6xevuL
3GhhT8vLmtlYYU/IC+rh7cKekpc0w3HiXvAp8x3vFPaYvKUeX0ueiWahNzCa
7AHNXh6+l+wReUMe/ZjcM/VKHi24z+rSvezO/bYGSr9VvbEnyTPWbPUG5eXu
5LvI07vg7fBv1fbwYLJH5U31oCPZg/KeaqaSPSJv6A1o1tuj/62Z667fk2ej
O/8DlcqNKg==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4225127828803246, 0.28186884090594366`, 0.7108731455646595], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFVktsFlUUns7fmTtDS/9WDZUSkTZ9bGwaXVEwLkxqF4piXbhhYUAeplRJ
jJa2VqRqhQqWPpBGKD6SpoiiG6iCRkiUuPVRY2JAEVoNmCjGGGPY+H1+X9PF
zT1z5txzz/M7t3bjUx1PxlEUXcMqYE2GKHomiaJ1oA+DcaQ0iqpIl4pOY+2H
zZ/EftT0WpybAL2rJIouxvrXi308i6LdaRQ1xPp/CCuGzEHwl+ZRdFus/blU
dCP0jEBmJehHYU9lLpq84VLLxKLP495bY+m8AfoA5Hfg/D2gB0F3gW4BfQH7
2/jO4VMfVjVWJ/jXwf8S/Hp8D0FPSUG+/whbLmANgLce/zuwVoD+HvIHg+Te
wfdy7L2Qvwr+6aDvAaw680dj3bkBdDf2UFiMLePGOPWALlrmMvScgJ4qfB+L
Zdde8N+IxaPNjANjRP8bEtGMzzl8n8WagczP0BHB/tU48xHoXxP5fB7074ns
68b/p5GDOdBTBe08+ydkMvxrw/dXoAcT2fAu6L5Edn4OenciH2lD0bngXun8
/gGZIfybBv876JvF6oOd74F/KZHPp0DvSuTXm6B/SBSj5aCrg2KwDzZmqIEi
6G/A+xcyzZBZk6jmWG/MP79ZA62JaPLLYn33l4jXapmiY7cyXowj7Z8H/Qvr
OdZ9c6ATysCGZ2H/MtDTuHt/qWL2E+zZw7qPdI7nea7dZ9lb7QXl5Yx58+Y/
AJ07U+m7M1et054W7M1YJ13XrG/GsweyozhTwzrF/+NBdbEd/BTny2P1WG2s
M3eDtyWoxp6IZTvjsdG+0a9xnK3M1L+vQrYuU+/0g749U+90MhegV0WL9c28
3wz91anifD9kXssUtwrw78pVp72p+oe9w37aGdRTl0G/wtoCfRNkb8mFDd+C
vz0INzaDLs2FOayFL1wPF8F/3nqugN5nPZtjYUPAugr+SFBPM5aXHM+/wT8a
1Jfs4W73Y6vrhXWyGv+vB+WLtVw0FtWB9zXWB+7pHvfsg+AtS3V2Cfa/gnLO
uG4Nim0hKGfMVwN8qs/Vr9TVUlD9TLsPaT/zMBqUi39S2U2bmYeeoFz8lspP
+lgfZB/1VWAvD8InYgyxhjaQnjTmzKXKAePGPO8JyvXZRHVDGfbcjlS1wvxN
GR/o27hpxnRNQT0+XBBmkB4qCHvo1w3onA3yk33Q7rMxdF8JOl9lXCYOEPMG
jXvcm01Tbtj6H8PZdZnsuBd7ksvPF4nZWE2O65BteAj39KfqjW7Q2xLFkH1S
NF4xZjOOW7l7ZdazL3W/rMddL0HPHcRj3HkoU42wr9stU8zVAyOembSL59mj
nal0MzaMHW3rwr7FM22F65g1QN/4zVgRY04YZxgP5pkxIb5+aIztDeqNAWMA
bSSmsUeLmer7ZdABdjTbP9YsfRylXeC3gbctqA/Zg8wn+4D6KXufY8KZ8L7n
woZM+E5snzd2Et8OZMJN4s6SXHdQf7PjSz1lxiVicmpsYgw/gezrmWJR51lI
zGl0rBifas/gTuP2nLGXvVxmncSPx1Pp5rxh/7FnOXOOee5MOc7UOQbZmUS5
LBp7GUPi/cKcpA811l9jrCW9NwiP2FMfY9+fyR/612YZzs8pz9Am55z6OfPY
38SihzPV60IOmyyzFPxTiXLAWDM/q6yny7q24t5PU8XoM2JgJuwv9wxgXPg+
OuI30hjuuWYM4WwZ83ypyIRrzPlJyJ5L1SM1njH0mXjN+fv/7A3CTdYKe2LC
fbHwVmNs+V447jfDI0E5Zn75RpvwO22tZzd1nsb/M1ibYt3/gm3g+/Qtv1H5
ruzIFm2qdayYv02uJT5w212TnPWN8eI7s8HvqP8AF94tpw==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlF1o1mUYxp/9/3uf531tc1sf60MUP/EkHXVkO/BAWEI1sol44oFM9yF+
BUFr71rGCmum5NoWkmklhN96olNH5Gd0WmMQiJXaKCooJSiik34X1//ggvt6
7vt5nvt7QefOjh1ZCKEL1IKHKyE8BHbVhHCAg60xhC/BFozqwOlSCItTCGeQ
D2D/NLZPgXnwEezrkPuxfwLeBg5jewi0Y/sBfBvyVjAfXoW/gTwIWuBD8HeR
h8HuzGc3+e/v6LspD+FT+O/wPVnxJrajyXwlfAR5f7Kss25sXyiHcJS7CX6K
+4+jP4Y8jM3XyOuw2Y1+sYD/i8BlYrkCzmLfgs057I9iP4F8XjEgH+HsNeS+
Ih75P8RbV8F25KXgBLrjYAP6j+Bvo3srWvekcsj7P0TfVbwP8HcffAZf9vH/
ZuSLoBN9BNO8tV7xo28Bx7ifJ78tn27Ds4I3oP9Jf2P/PnIr6EW/gXwMZPb5
BHx2cixV+F7k94r8qQZ/IW/k/hh3V4N+eDXZX93XX3d5oyn3n58j/1PUSzG/
jLycmH4mlgbO9qJfUnFuJ9BPob/M2feZ+6ENfjLZ935wD/nP5NxPKiZ8nw0G
a9yjZeyfSda1Yf8t8jfJtVWPnkZ+DpuB3D4G+VPUcwX8F/6uL+J/FP4H/LGi
PxbCbyBfT86NzvTWhWRbvbkS/klyb/aBZng9/j2b2Uf5Nis6d/LxN+Tv+KMm
d39OwsdBCXlTZt+y6FrJR+Xuec6aMuewEd4EGms9c4vQ/VdyLyhezcoteCV3
DT5LzqlyqbPxZJ/lq3zohq/C34O5a6Bc/KoZyZ0T5eJeybOh+G/Df0yeffn3
jma1mF/1Sw9yM/4dwr8mzl7nr4aKd8Nc9KP8NQaWZN4Rmr0p7izLPYNfRM+s
ZrUbfIjteNl3uzQvyOfU4zXOwSO8PRb910H4i8hfJdduCAzC15Q9awvABHcv
Red6deZZupPcC5qpV8ruUfWmZlC7oLXoL+0EzdL95F7TTI1gvx80Z96R+3S/
7F7XzHYgv1T234pZtZbP8lU1b0dX5WxO5hl7E/nBinevYnwVPqviXao/Ovh7
bXIudOdScsyKVTW7hjwaHZsWe+LutujdvUz7KnrHaLdoJ05Hz6BmTztBtetN
5qphCd4VvXvUI9ptPdG10Y77t+QeVO+phh/DZ6J7QTvkf76Dt1A=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.508577304800541, 0.4315118015099061, 0.8318472426077658], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFll1olmUYx5/32fs897vAra2SiMLFFq2TND0yaSelSGTC+qJSiNTyY7k8
6CB1lUT0CYnv+44GlQVB0RoRRGngiejMWiIVDSpsM0KT3PoAzXlgvz//Czq4
uf7PdV/3dV/f93P9o4P9W/Msy/axWljLyyx7N2XZl5Usu1hkWTd4LXgQoV0V
yyyFV0NugO9e+KMwF0BPw9tQy7JP4Pchczf4HHg//M/QVYBPILcW+efBl5XW
JT3bwauQ/6DFd61psfwL4PehC1gfw/sl91npWBm4BzqQ24YGeq5q9d5m8B3o
fAvZCngRd21Afhb8GPwD4CHkDrA/Aj6JjTkyK8Ad6LicNVzNsvda7GN3bl1j
0ofMwtw2yYaHoA+z5ohPqvpbuI172mv+Vqy6cp99ABu+565Tii37JwPfB/8I
eApcR894bj1HoV8rUfA3sb85ORY/YfO/nOmDvxq6H/4keAs6xwO/DH2JdT63
bW9jyxHwg0GryDQ5+0XpuA1CV3B+L/vNqn1XDGSD8qOc6pz2dXYb9ELuvN/A
PTdGzXwYZyT/BPq+C7++TfZT+F7hZN9UaxcKn62D97AuRS0o1qqH+eTk8eQ6
Opzss3z8lHNvcEcH9p6D92PhmPQmx1exnStsn/Svg/dKcm2tR/9U7hpYDn9r
6RjdDt4EfhO8BNltfM8iMw86v3T8VDszcfYeZPpZX/E9T/nkvlUV19RU4br6
E3o13/eDq9Ay6m01OneUjllSfvk+Dn+A79nogZvhZ6Vr+Dh6fgPfBL4uOb6K
p+xWjlUzncl1oPjsih5WDHX/TGEbLqJjY839PsF+W9U1tog4P1W65qeT61Ux
7IT/XOmeG02eFzq7Dn3DpefDYegh1pPIfI7MnaV92QvuK913L0JfL5zXpei8
K7nWlW/Vk2opL+2z/FWPrQz7J6IPK9F38lf12MSP4Zp90DxQLajffi+cD+VC
NikOsrOO/oOlY9OODTvBezhzNndNKKeaJc3oi7/QsaZ0DaiXlBvl5WzkXzNK
tTIRfao6OBr4GPSb6P0fsOcP9CwBT0IbhedQo+p+U69pPr2THCvxNQs037Tf
CKx50xPzYTR6VfwrsPlK1kica8QcUS57w4aNyb2kPjof9aVaOo0t7REr1fvf
yf5qX3UoH5XDZtimvGgOdVZsd3/hu+Tvmap9noR3Jvydho4UnuvKseaG7t2O
vqeTYznL/jXRIydixkv/WMxg2T/E/jPJNr0KfS1Z11DkojPehXr0gmpwLLkO
1Vv7Auu9aiTbskVvUfp/FkyFX8vg3RbvlPzZXXWOfy48a/qCPxG2qS9PRW/e
QowXtzoWt0KXtTpGqqNfI7ZdkRflTn5fG75fihgpbnq7FkcvKO/KbU/YORN5
UVzX1xzbR9D/T/LsVT47Iqd6K+bCZtX67shRV+RM9j9buv8Va82Gj+J/YDTe
sO6oVeV7PN7errB/YeRLfdpaOnaK2w7ozmRbdWdbxFDv67E4P114Jmoeqj9m
ombq4av0/wf7lxhM
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlFtoz2EYx9//f/v/3t9fmW1kLmhOhSvZrhA3WIo5zCmn0myxWRsXLpgN
F86KZoZyLEVYUk4jUZizZLJy2nDhUOZUZC74fHvei6ee7/M87/M+50GlNSXV
SedcOZQJnfHOvU05V5Rw7i7UDH4X8BNoJng29IBH96DPkXN7wP/gC9Ffhv+I
fXGwr0ZfjOwO+sMZzhWkncuCHuktH3agfwttAudjnwffG/0BdAux/4CvS958
yedq9FNi546jG8KbYvglyH6hO4nsNfxa7L8Ef/vR74MGgnPw2YLuU4ivHeqH
/RZoJPpFvN+FrhQ6m2GyDehuQFXwQ6Em+NvQSnhMXD6+6/D5FVyFoFU87+cG
/7WR1Uy1OoV+LnwlskPwD5HNAK+NLHbZ5OFvBbJO+DJkd+C/BX+d0DHwK/D4
0J+j4JcBX4MisA/1Us+USwW4I2k5ZeO/kf8mgxeDi8CbwZnYHgH3ofa5actV
PahHtz6yWjerH/AfoXPYj0hYL8ZgX5KyngwG7+C/P9ivAk+MzYfeKudO7E57
i13xrUPXk/e7+asB2onufWS+5WOZt5gV6xvwVXSzkN3X0GLTFzwtzNd8bC7g
v2dsvHLqQv81stpoPsfy12ioNWkxx+i+q2foqqCLyPYg2wifgq7At0TGq2Yl
3nqmXimGefCj0jbLj8G3sL0J1SRtRjQ751M2e5qhSuyfI8tNWE+qwG0Bq+fi
l3vLVbI13mZCs6CZ3upt5jRrv8Hb1euU1esE9AL+iLd8NRON+i9luatn67zt
nHaji/dTwdfB3c5qoti1E9oF5TBHviKrtXa+G/5vZP2T/3pvOyd/2oFx4PHe
9lH1bIqtZ3qfRQ2mgafHtguyKcC2MNS/HHK6B+APAf8h9qfeZn9xwm5Hr7TN
im6IatkO1YV66dbURDbLmgnN0jZvtVKNlOuk2G6Rcs5OWw/Vu6ZMy3UC+oMZ
FvMwdMNDPPr/B3w6stwU809wj4CVj2Z5b2y5aqa7iP+2t1ugHe4PPyDkI/wO
PhFZrvojB5wb7HWjyvC1NLbe6kY2eNsx7ZZubgX+a7318hqyxpTJxKsmbei6
Uxa7/uwV243VbV0A7dVfYT40M8/Af4O94nkPTob4lK92LSu2t6qvbmtO2mqn
G1sR2Y3TbVMN/wPT29pz
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5864814020186534, 0.5639746101909466, 0.9100153317142022], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwtVklollcUffle/ne/BJSqCFHThQrqRo0KBedFqyBtHeqw04V1DI2ixpWi
RYpVqdVFKlgsiFrnah0jNXFIHJK0SI2YJigEg5K4sNpiUeOm53Du4v33vvfu
u+O59/uHLl37xZoshHAVK2Jl+PkOB9PBdxeHMLcQwkDsKzPxfUDXphA+txD6
gt+B9RarAvIVoNuLQkhYHdDzM/abcH4K9CTWLvCPo/bkeX/EZf6E7l0mXS3g
vwb/D/gVkH8FOjTILzpJ317Ahx9NZ+/AHwNf4/68d//boKfG9KYV/Pemu28y
6abPDYjxJlYt+CrI9fr5LMp5LLynXL3nYJ7nhJR75mYrfNiCNQz8LcieiNL5
Lc7KsLZBzzzY3wS+HDKLwK8GX8r4yRcU40d5CK+xPw5+NehSnPcHPwJ8B9av
kJ/CPGHVgZ9UEsKnufxhTdYk1eVcJtn9Qf7Td8pXJtmm3c4oOcrUgl7OZLcr
ak/+EewfhHwelV/S2TivjsLKdM8h46WN9VhLovLGswbP7XLwm3E3oEi2S7Du
sEam2OhrM+gQE1YOm+pKm4xvY1KMzG+pv+0p1p487RF7zBXr8Emuu7PQU8A6
AP4q6AcmzNFH4pXvWJtZzt9AvO1JfraBVpr8Zg0bvY60y96gP3shvyCXrmeQ
HWfK42CcleWK+z9Tzeo8P8QVnocdSe/5lnXe47Wej3dVpngai4Un5pD5JF/u
OKQP7Mc2nP8E+lXQHd8wv8+L5Sv97HaeuZoIzEwuEX63OcZptwn+3MXaAPnr
oJ+ZsD0j1578mEz4ehjkS6P7s8CE73K/7+MyT/CuM6nn/krKKfM5Bzpn54rp
Y9B6nK/E+W3QfUn12A46M5fNN9A9KVceqG+zSec10NG5Zsg+8O0F4ZI5Xu81
fQn5v7F2Y78bMvcL6vMfMuFrcZDu00n6L5Im+X8G9HJS3kjPOz8Kekaa+uc9
znqT9P0Cuiop//1x388053ph8wH4MVH3Z12GuOcZ5+FY+PDQxA+PmqGMiz3x
h/fFVtDmgmYi6/nce4C5K/e+Yu7WOc7KovBBbDB+zkv2L7E4M6rHe31ucu5d
gF9VSfUj3m855qfB7lRTj+90efZ+S9SefGtUDphP9g/rTFy98pnL+UacHnCs
Et8DHHvNsNOCVZ1pFtb4PCz12BjXBu8f9s6HxJrpW3IvKjbGNQFn403z6zzo
qFy2DHwy2Sc+Kx2j7PUO7/ce0O6keIpAuyA/MQqbC01+sLd5xpnZAx+vmHLc
bZrdzCf1Dcqlk/3PbxhjeQL505xBUbUc7vWlvk7XyVl1wedVCfT8a6rTl9C3
DKspaoZd9zk2AmeHTLG3M/Zceo4Sn7lq8RJ275js8VvY4nObfdNk8oF5+s2U
K747Y3pLP+gveeL0sWP1dxNeyfObWu06Of96fAbyG8tvKnG13+Qr/SS+LjnG
GFuX45B4We654nfmnn9raL/VfWB8T00x8r9Ck+tnLsoc5/8D+v8YFQ==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlFtoz2EYx9/tv9/7/qecJjVrLkYOF2JRasQNVkrkdInChrGtMVdTJDll
3MxElJwPM2eTszDbSEybLQrLbApDZM2Nz7fnd/Gt7/M+z/u+zzlneemCklTn
3DKQBrKDc1ngHIfnwUR4LrgBrwPjvHOVyH3w4oRzp+GnwBLuVnHm4WngMPwI
+I79V7AHXgly053bjDwC/oUPc+Cbsf+BXMh70+BTwR3ku6ALfSfYCd8FBqMb
CE7AT4KqyLk36DfBc/ChD97rzZf9oBvbLnDGWQw34UVJ51r5KxNUYDsP+Sw8
G/2WYDEqtp/Iu5FXoG9A7kXOTJrP8rUAVKEv4KwJ7sBe5Mnoy+H/sD8AbwAb
Ui2/a7HNCOZ7e8J8CcFyJZ8mkJ+r2PdX7Fx4RXyf4/iV86nop4Ch8PnojnK3
kDdecDcJfiGXIXfA80EHcinye3ge+MBbTZHFtg2MQT8KXIJfBsP0F8iFL8V+
NXf/ectlc8J6IcXbW8pnfbCYFMtI8Bq5GPkdfHzC+PBgvaSz0fj+F7mOXDwk
vppgf+jtQQnzfVKwXlMM8n1sMN8UQzu8DRx05vMafJmBzSp4P9BGbN/ifkvF
/je2f+J+Ug8ol82R1UY5PYbueLB+mYv+KvwKWOesh2fydpG3t9Wz+cilyAPg
48Ei5IVge4rlbA5389KtNkV6E930YL2sHqvh7hj0j9IsB/Xecq5cr8fmirce
UO3rnfV6d2S9qp7vDZZD5U4xNSA3xvNa4WzWWiPrReVAf13gLD3V/ryF7e14
fhSjZutlZL2tfKgX+iKrnfKrXdATWW31x0ZvM6zZVQ9q1j9GVjvl65BqEyyX
+l+zUB2sFpqJFnhr3G+7sSmCrwEZzma4GP6A94akWE5Xkr+DyLnOZvA5+mfB
7sqfxfBFwWZXPTAM+6ykzabeaIys59Xr2jGbsK0Itit0dh9+L1julBPtnnRv
s6P8lsBnJa2WqvlF5FpvtVFOa7G/GO877UDNYo+3XaeZlO8f4/8Vw1NvNVet
tROavO0I7YZy5Gr4Cu6UwWeDT7zdGe9f9fxj9E+86bYS3z5sd3jjyqF2Ube3
XamddB1+DbQ42ynajW8j2xXq+f9m0c2K
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.649758740262307, 0.665643685641856, 0.9114578601292921], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFVVtsVVUQ3fecnj2nwo/0aqRF0kqAiCSCFIgxhKiBYj/4AIwopoDxRZBH
iDxCDBUb0hqi8gNFokF5kyiPW/jwQcEfpL1oQQOt+AME0R8SIPL4KAlrZU1y
PyZ7n9kzs2fWrNmn4a0Vc5YnIYQfICmkpxBCDTZNWOdUhbAE+6egr8X6EeQ8
9OUYwlnIajg2Wwhr8hAK2A/gbARsZmN9Aet/WN+F7Ezkx5gN8HsC9juwry9o
ZfyJsPkUe4OUIM/jexzWZqztrq/BXY+Z4r8J/fuQVdj3QHcGMhp3fojYAXIP
+oU4XwpZi/1LpjPqd2O/C/II7AezEJ7GvgX675Hb2OoQFsDnNPa/QjZB31FQ
fszhFnTfwucLfP8Evx9N2NBna6r4V3HP15lqOJfIH59hKmxX4ewmvq9AxsL+
NeJSpfi068T6Z6o+1Dres73+do8zK1WMZa4nbjXQlxH/rCnuz1hPQOo8Rp33
ZQNx856WcN4FKeJ7OtYXTTFbTX1ib7qxP2nq60ysq5H/IPRHsD9q4srfsH0S
+iL027F/B0liCSOhq8t1H3MoOq/KOP8qF7eI/TOOP2PnHn9xov6RV32OI2tn
P1u8p4dwyeEoLgw6J8iHY9Adj/J/GbFnmPSbnE/ErBPnvVHfTTh/xRSTnFqQ
VOzNsWVNr5vqak+FMXt4wHk8kfODw0YT50egjknVqnELZJr3bCX0102zcRHr
BcgEnBWj8ljrHGD/p0PuQPe/qefEkf0jzuTRXVPMgmNF382wec5n7w2czzfh
14X4z1YLO/Kyq6D5IneKXuOjUXix9iFRnCAfyOtm5/Z7WFsTcWMkzutNHF6U
VPrwAL77MtVRjfODmd6A24mwIveKzkXey1nqxncjZDnq6jdhy/nsdhyY9yKP
fxjnhyDDPBfGYUxiP8Xx78Wdl2PljWl1m1uZ8mbOZ6J4QA5ciuIx8/qX/TH1
7B+s10zvWVuqOeA7tw5rlec3znlADrR4juzFl/DbYbIbj9gfm+yG+Xw3OXeI
C33HoD/3TVj3QYZ4fNbRk8l2LvCZBzmSihNbnFf74LcfMtR96EvcyL/bXvvw
qHeNvOb9Q92GszWrSu/PNmIYxY8bWAcy9YXv4ndRs8H4rIn60Y4H8axHXjuj
almfKj7v4XtW43tysBSF0zdYJ0e99x0+a5zxjcihzZT3lUS1kif8/xT9LWHN
7A97/Uum/hWdM+scA84L/2GsfRRya/S7/oD99ai+bfA7GP8U9MtM3+zRes+5
P628o5xpvl/EdQ9s95qwZh28o+SYtDkuxPJzE57jTfPP2eQssc+cr5JziLg9
DpsPcvnyXWj3d+aeiR/0KUPSVDE5O31eL/16Tb6c8bLbkLub/U1oMM0YY7KX
zIPz/ompf8zhMxNfyVXObb/Xzn/U0VRvBvE7nynvAdj/ZXp3ahFjeK534m2c
v5rLh3fznO98J2y3W6WG1LnUB9tzubj4O9bfcv0bJzgG9O2I4jE5/BBedwjk

         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlFmIj2EUxr///5vv/36WC8sQBqEsWWKkkS1R1gsXg6zZkiUNUZYka0Ky
XVgj+1a2mcGU3ZWZMYwlzFyS7UYha1F+T4+LZ+Y83znv+z/nPc85HecsKV6c
jaJoNsgDU0IUTQW1fMzB6/lTB9bC14GqhP/4D2WiqAM4iu8YKMMuJP439kn8
DeMoGsq3Q9if+NYPO49vdZzfx7fyjL+V4CsHM7h7ESjBNz6NoivEvoY/wa4B
W4kdTYJPOb+DmHPw7uBs8G/qtxpzpgb7QM65L+f8Yvz3OZOPfxtI4Q3AUXyj
4O+xY4LnYX8EP+CN4CXYn8Fl7AUC8dNBNXYV2IJvLPxazjUo95lgPeerE7+N
cozIvSn+Zdg/QQW+2tS1PCH+G/FF8N3kPhr0gPcMzmUG+IU9ODifsfhTYvO5
byX8j4D9F+zP+g0K8HeCF3P/AuI3cXZjcG++ElOG7zqYlfWbP8CuzLmWzZy/
gD2QO/Zydhrf1pDvFc43hz8Hu+Fbc45tzp1L4fv1hvgexu7VYzAqzz3rgu8l
6BO7ngT7MmgW+/17EXsH3la5gkfwPdxZmPUdXeAHg7WzChRgV4HO2JMyrrUt
MQtj11yN70tiXx14C++Lfzv+QeAw9obgt9CZ1qk1Lm3Ph6/g7ClpALsWDMMe
Htw79aA38Q/hXfFtAYXY74LvloZGYI8M7o16vp77bsEL8L+K3cvvwb3WfS2w
WwbnOi5jLXyAD4mtiVbwNqlz08ypljrVFLumdvD2qXstjTfBdxO0iX2fevkz
WDvqaVf4meBZWQ3Kg3uiXuTDu+EvC7bXgCLs/sGzoPo0Cxdz1oZmoia4J3pv
vUkFvqs5z8JscCP4Tt2lnM4k1pS0JM2eCO6xequd8ZmzX8CurPdDfbCGpB3V
fCm4BuUuDd0Nrlm1SkNjVCvIZD0jH4N7pNz1ptL+pZxnWTOwE9+zxLtEO+lI
Yk1LywOy3n33Eu8i7cBX6m3ifF6D08EzqNmTZg4E16DcNRMTU+8g7Z5SeAfs
jsFn9WaTg3eCdoF2rrQ/N3GsZmBC6pnTrGknnk+sKWlJO+x4Ys1Kq3qz25rV
/+8hzZUG7zz1VzP8IljD0q5msjJ4BjV7mql+5P4mde3a6f8AE8m9Ig==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7130360785059605, 0.7673127610927655, 0.9129003885443818], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFVm1o1lUUv8//Pv//vdvqg7n5Id1oFVHZl5UrKKkZNawniSUt7c2IKCpa
lLi+TNtqH8w9Rgsqc00TyhoU1HrxEWov9iEYj3NLK5gI4tBIh+YmQaHW7+fv
wD7c557/ueeee15+55yn/umXH2pLnHOzWB7rrcy591LnloFZC8Zu7C+B/0h0
rhVra865Z3Dei1UNfif2gVQyBcj24LwC63Lo+Tc49wB0tIM/AN7NTrqfitJ/
B/YcVgn8IchFyNSA3gD6dpwvxXcKPSeh5y7wPktkTwdkVuJsB+gnQY9gb/HS
swDyLup7KNH75K+EjsvA3wdeJeiqIB0zeedOYX0PuUZ8D2M9CPnnIPt50Pdr
iXyg/Wdh/xj413mdNZn8i/g+B5lrQK8H3WL2r87L1xrQXyS6R/sZj3aTKTI2
ieLGfavFkPuqRDF5F++uiTrf5JUbxnwf+H2Z5EqJ4jQAfn1O9tDmWZzPZfKv
jPMU67ucYt6cV9yZz/1ROaUvi/K6z5xfmcmWC9jPY70P/mHwrwXdzzyA3hl1
pw57LdYL0HEW539hvQ35o/j+Bvt26P8E8WtPFbevQHekyuNC0D8G5dhhfx56
xqD/hyCc0d9W6DmB9bNTLKcsnq+aT02gN3phg/ntwt2P6Fde56nluY/8oG++
v9tsYD7pH22jHG2nTtq7y2x+BXt/Kn4FdMSg93qhq2DxJ/buM3ou0fvMyTB4
Z/KKfRVi83eQ3LqomDKeF8H/L5Mti7Fvx3oH9Hm8+TvkG4i1TBglBoklvkH9
9eDtypT30US1wVgdAH9PqlxfjfPF+O4D/VhesaMv1VHv8a0vE+G1xwmLrGs8
7fZCxxMgDiWq7+VRNfZTotr6zal2GWtimzv1k2beRs2eNtx9GHeP4bs7yG7a
zHwzb8zFH6aXOifyspf11QTZiig9ew33jDPfbzVsrPOqFdYW/V4T5DvXFusz
t0Bm0Hz8BfoWYlXmhNNBwyp7Q9F8r0ZsYtQbR7xixLvXe8Wqw83XLfVPWc8i
fwJ3S6n0j6byn753JtpZdwcTxRVPuDdw/mZQXmshvyQVNthX2rxyznh0Wc8p
gnenl82Mx3KjPd49EfRNn5YZ/3Reb/Pdi9B9OOhsG+TLmXo567DO6ouYPW24
JS7HLEes0W+DvlnztId2dXvFhXcZ74L1t1N4azRIL2N2g1cMp71yealeIDMe
dMbeSb/ZPwvW0wes3tcbro6zFyfq8+ypG/w89hqtJzA2JcvpoJc9tIuxnLP5
1GI77zLm7IOM7T2wJR9VS12WM/L55pC9u9kLTz3W2+tt3hFTzD3zzvg3mAzn
03GbTc+y9+CNCiesEXOVNpv2eNFHUvUFxr5MnVG9m3VRmSrmSZSttJN9q9Fm
E/tZk+WrMmoWslamU81C1ttk1CxZZf4Rb/SRsSfN+NOWGZuVxAHxQPzwvNbk
eZd8ynM/YzR9mjG/PszkM/2lLp4Fq/FzyXyfXG1360w/bWGv/dr67RWweUFU
rS8Cryao3jgDe633foy3dmbKbT3OrwqqW/rCt1lrk4kwQTyUvM7I3w/Z2VQ4
Psb/LJn65Qj4J1PNYNZryeT78VM23DwKmbVBum9kj8Td19nnwXs8qNaL4O1I
1S8PgB7P9F/DR/U49rHxqHwz1x8EzR7iucF6ELHEGbstKOdT2C+kwmIV7q2I
6p/EyITNIM7wT4NwMWy5ZL/6h/+RMtXZiNcZ+X8S/5lqiLaWbdYyJ3Xp/H8f
6icG2GuOWp+5CXeXBvXUov2vCTajN9l/mEOw8VesAu6vxX4wKgd3Q3BFUJ01
Y783aH4Qb532n4o42Gh6GF/SxHOz9Srik31ks8WK+B4xv5ZEzULWzqT1A84H
zuNpww/xeykO4N/PWR00k27DfmtQz9piflEne2jRfGeP67Ye+D/HqGKO
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlV1ol2UYxp///937sc0O1OVB5XRMpKkn6dagRs4iy2b4Eek0t0noiZXF
/Drqw5pQNMU6qNXcTFBLMCgVVCi35YGwL5uWwmQQjhbpkdsIgpb+Lq4d3Lz3
9VzPcz/35/OWvf72+p35EEIzUoBcz0K4hhTlQjgfhfBaGsIscBN6K2uvom+L
Qyhh720O1sI/lISwGv4SuAK+Dvwe+F3kEfQh1l6GO8j5dnAjuDvyWie2mpF+
8FPgX+GuIhl7V+JQA/Y3gy/A/wa/E72Mte/RRyKfbQfHkW38Dl+CHOZ8HecH
4TZw58dwFUg5+idIJedfga9EL2d/s2IFN7K/CbkOX0iMxeiL4N+HfxGbOfS8
/Ie/AP8oeimyA74dfjv+tGGzDv4Ma/VwGzPHvpS1esWDDOUd41blFunI2adC
5R75Bf1KcK4akK7IOavGqQ7wG+il3HUzdU6UiyeQHvBa7m8Rj7wDt4S1s9j7
AzwBN4l0cV8V0gv3JnuG4RZGzvWmzLVXzpdx32OZYyvivt3oz3KmC3t/wu8F
vwS+DP4LvAe8EtwDHlN9wNXgi+BR8Nfc/VXi3KpGz8HNBK+LnNMV4OWpe0l3
jKWOQb7XIN+BR/CjNucYlBvVVLVUjobRe2L35gfICfYfR9Zw3xfgmez/PHYt
1NNHwH3gMmyVYGMAPJi599Tj1dhrTdybe9h/K/Edsr0fmc/eD7E/od6Er4L7
RvGAdymHsWdIs6OZKkW/FzvXp/OelW7W5kaemQPoLchS7p7Mu3eHwcsi93A/
uA/5NOceG0jso3zbC17E3v+xXwk+A26A+5e1OvBn8k+zwtpy+QZ/NPOdumsO
MX+Evj91LhRTJXqUOPfa/4I48IbIPTpPsceeLcWjWVIOFLtmqh79b/Y8o/cD
GUzss3zdhxyBG0/ci5rhCvB/sXtZ/quXB1h7PHJP12D/6cyzpzenBO5u7F4/
qZnK3IPqPfk4N3PNVCvNaD97OzPHqhlSrxXDr5p+v46hX03sm2pQkDhnil0x
rEqdA8Wunp+KnXP5qho9n7qn1cuagU2p3wS9BZr5o4l7Rr2iHgmZZ0qzpBnY
olmJ/dbpDbrP/sBab849fyt1TpVL1VizO4e1b/Oe4X9Sz5xmTTHdSd0Dqr1q
fgzb98CH8p6JJ1PXWPFpRsdj11S1VM41G3oDiqZn5BTfQvhO+Lc434b+JVIb
bHNB4jdKb5PetHNwo7HfTsVwEn0q8Sw2cuZn+J9S21L9CvA9yvz2KYczEv9j
VBv18J3YNddezczi1D2j/OuN+xH9h9S25ePD6LOn9ytH81PPoOLTP2QGdxUj
N4Jr2qG3I/O/QDGtTvxP079M/4AH3njfLQ==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7749187388813827, 0.851095633498544, 0.9085261039925843], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNl22s12MYx3//3//3VJ3D0BND9CAqYTmyEWU2DnEsTXnoEWur7Wj1Qhtt
2XhhNRVGhXSEWbxAx6bCOqE3VKTWg/KiYutppnNMpuL76XtteXHvvn7Xfd3X
fd3X9b2u6/5dOf2pca1pkiS315KkrnmcPtaIHqbxghhDNOaLP7FmepHoPt2S
5NIiSV4Wb4zGdxpTxL9MvN1Vkryi7z/zJNlWJsnQ0DE09vbV3sGSe1syM3Lr
Qs/Fmq8T/yPRLbLhQ82DJL9A+p7TaNZ3Q82yfcWfr70fVD77Xu27RvT2mu3c
U7PNjZqXanQX3RG2PiB6p/QvFt2Db+3tKL1nbJYkI1LveUb6nq28p5/WT+TW
fULyb+p7ieRGyuaB+u6lfRN0v+UaZ9Cjtel123uLdNyq8Zq+x0vua82bNMbV
zTspmVPSN0oy62TbX5pHaLwh+qj4xzVGS26j5G7SuEP8yaxjh75Xiv5etvUX
fZg41Lx3rXhzJXNR6n0dcfdcvCbpnKHvVdK7QeM88YdITw/x2yQ/WnOD5Fol
87foSaL/wE7JjNRYI/nD4k8sfGZLzb5vRY/4p3PbwR7OQX6P9l0u+VcDRxMj
RsPEv1F+fK7uGLQE/1nNPWVLp+jfJHN7YR9cI9lrC+OD2DcH9hbr3EGSO609
99VtO+d2yuffavRJuHySbBF9VGSDxnlgQPPB1DLgoZd0/FicW2sM/KyWzm2Z
/blLMk9XXtuamY9O1vjG5sel4wz2EA/ZNkdnj0Gf9PxQdyzADVggdgOIkeiB
4l+dncub1ty58VGsbwqZKdI9tTqHp6+D35kbr/j3TdkwK3e+EAdwhE/+IUal
/dQ/9I4Qf7j0NXVz3h+IXCJnrxZvkHStDDuxBXli3FQ33npp3lF3LPDdK+G/
SZXxClaJf+/KGDiVusagZ2bm+IOfRZIZqdFO3ol3RdSf7trXrXKeIdsv5I/q
fj1L42ZwZnnW5tacW+QUdairsk/f0/y+xmWZ9x3JvXdV6jtg/w1a31f6bse1
vqm0XvxCbIZErj8Y+c55LYF/7sb9wVdLrGE//mgMn2A/trI2KnVukqM3Vs55
4rSCHNFIYx17x0Q+HY4cB3dbA5Ngd3HURuJyIGLXUzp3akyTrofk162l1xZm
zn1qwM2VMQEetkmmrXAfoNZgF7g9KD+0l/YR9eBI6fxujTwHV+vqrn3YMVv0
JZnj/Ij0T5L8SzXn49S66VT8WuVauKDuGJDPAwrjBsxM0b7JpWv2odz9ABlq
z8+lfQFetgRmuOfS6BVL45v8BddduWPYKR3XV/bP3Zpv1lqb+F1hG7XiKc1/
/q8XrgmsXij5R3Xudq31Fm9t3X3qTO7ah+/3lvb1gciV2ZFfG0v3Kmol/Wps
0LMy3597tWa2EczgjwWB8y2lexU1kt4yLfoLccJW7OMed1W+C/j9JTDcJ+zs
LT3N0t9et+2HUuc09wVn4G0ePVZ3uaCyfGPhfkCcwRt1kFo3M3cvp4+TAw9G
zlHLj+XGOOeyjn96SF9D5TOow+h5p+56/E7gEBvbw86HK+MGnBB7sLU48AZO
yWtkPwv5Ceiu/A2u26Kv1cXLKtv3s/Ss19wo/jzRmzPbiA82R4+gN2aF84w6
vTR1rUbf8cw6f03dM8g1+gZ0n+gjh4Juyyy/T99f6r5PlD4ffUtCJzEDY9Tg
3alrB3VjWNRH8o/4Ux/BAP1vVPTz12vukecnrv8dUcupU2BofthGvuHD7fLB
Txr36HtZ6j6EDdh4Fi+psd8Zftib2l58BUZ5F2ADNWhZ2I/PukIe+iyWog41
RC2i/82NOvxt2EKtujBzflIT9/OWK20T56yKs9aJ/1hp+bbw5d6wk3OJIfce
ELl5MOog59KLlqeu581R36GHF35D8H5YKH98ojOyzIMcAP/06pXRr0emzmHO
GF34LUIcxsXbjbvtyP3+4O0B3nm/gvkxEUvi+EnhN8Hp8A91Eh8NlL7VhWWW
RGzxLT1gRtT55VpfUXhPe+Q8etF1Z2obyCFqOnkE70zYxhtkepxL3IgffmD9
tsz3PVb6bbcm3l28X8AVGDwdOHyr8nuXN8s00bNL5wEY7h/+p15TV3iHENsR
cccPw3+D4s0zPt49E6J/YsP+0m9f7rtRc0dl+8BUU2pcsX9A6KeuTYkeB5bG
Bp6otdiNPZxL/NjzbuH4EbsXC/cY7J1a+r+F/gyu7gmM7kqdk+TjC5J5u3Ku
0me2Rq+hdlHL8D11ZU/kC/3h89w9oj36B1g4GfWLXEgk829hjE4X/UXu/Wvr
5+66Omol9Ra5ZYHhJyX/Ve58aArMsef50rZiJ/mxOXJkhWxfXtlP2ASPmokf
e8dZ88LH8H+X7P2F+xJv8IviHU4/Bw/490fJXFK4n/Dv8nHhe/Ff+Fb8GxL/
l6MW8KZen/tt1B61nrOvKvy+JM5gnxzAVxsku7dyDPDLrvDt/Mr/frz9weA3
uXHYS3o+zY3fZsnMKRyXfoX/PaitXaI7C9fgs3Uv6vl/RX7PPA==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllW1ol2UUxu/n//yf57nnXFC6EUakMTOyrDAx6HUh5GQWWdSil831MlTY
KgOFLBLqQ2lSWtbmnG2lSPVBa0HTCl/6VprpTN20D2kfNEV0Eyra6ndxfbjg
nPu83PdzznXOM6WlfUFbKYTwGiiDL7MQXs5DmMjhiiSEbcjbwShyP/ZvsS8s
QjiC/VjJvl9xNqHkmEps48DHyD3ge2zPoA8iHweHYwgDYBy5FqYh9GJfBnaT
/y7scxWP/TFsy3U/tnfAGPIc7Eu5bx72T1LfeTP6IewvoH+BzzD6RfAutvfA
i8hz8e/BvhP8hz4GPsLWATqJbcscO7vku3bhf3fZd7bkfoPuVg0e1dvA1+Tq
AxUVvkO5VxHTgzwFezv6AmL28z3DxDcS+3vit/Zin1T2mxvwvyG6VgexP43/
8/j0YzvM2aTcd+quemJuyv1mvfVz/L9BPsDZcsWDacj7OHsQeRP2qejr0Weh
d6OPR/+bOy4gt4GXcn+D3t6burb10bJq/BOxG9GrsV+L/3z8b0RfiW0xZ0+R
qx+fSmxrEt89I/fb9IaNyF3gSvq9lrMNyHvwqUHuAoPkuoaz95GP4l9Gn0PO
pegPc8cT6C2FubYDfRl6J7iCu6vwKRN7Gvtp5FbwK7Z15J+JfwM+R9GPRPuu
A3uxXZe7FqfwebbwHco9hD4+moPiXiNn96M/V5i74pB600F8Q8k9+gG9HftJ
9D/AZGzVuWfnw8RceAP7pZI5sR39EjHTka/nfc3oH6DXUp+9nG1FP4VeT+yP
oJZcQ5w9UnYPusi1AdQFc1yxb+WupXKI2yc4q0rN8YTYO8B68VH3Y1sDSqo1
+aZiqwX3ETsqjpP738KzPj/17H6XuTaa4Qp87wUdqWdUXB7IHCtOL9F+iK6F
ev4P8VW5ufYX2IStO5pLqkkr8kjuWT2QOnca3XvdcZZ829Cnlc25IfKNcdaU
uOe/oZ9QTbhrFvYnkZsKz444ebm4Ej0bNcG5zhE/OXXO3fj+qXnB9xiow14Z
3ftOzQBv+xTsSlwj7baT+Fen3nGH8H08mvvzlA/57Wj+aIesQH41upd1oCOa
o+KmOLwF+ZVo29Xoo+QeLLwr9I3qfVM0N8SBPdgeUj3L/oaV2M6DJWVz5s7o
mqvW6vls9Nuje/lZcOwM9DdT5xhBHgargzm5CDkU3o0DqWexFM0dzeQtyLdG
11oxWwv3UL0T5zrJfxVoST0jmsXWzLOvmWzEd23mXmiH1mi3gMsS7wDtsgno
zal32g58mwvPrny0yxZn3m3aadolv0TvRu2Umci3iaOpOaF/y+bo2uof042+
mjv6Es/QGe37wrtXHNC/bWfmXaR/3EXkfYW5oR0+knmGNbuq4X5y/Zx71yvn
9Ogdqd34eurdoJ2jXaMdcQY9y72rtIP1rQejuaNvfiB3DsWqp6rlxOh/pWp6
T+5/pv6V4mRf4Z6pV+LkOfSzhXutGoiLq6JnXZz8H7p2Bmw=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8317460833246725, 0.8700460386521661, 0.8830675046068279], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmH2QlmUVxp/nfZ/7eR8Hg43GUjYTGVYxY4Ngl7U/RNDpa8a1dYiiptzF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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlVtsVkUUhefvfy7Dg9qYIFaUakK5iNTWFvwxkVolig8WSjBaE4EWDUYT
ChRRVETFFyVKolEjqCnhfksEIwmgRoEivkiMFy5pi0aCIi1ERUFF9FtZPOyc
tWbv2TNn9l4z17W1T5ldFkJYjCVYXwxhI9YKfoSBhiyE4/Dp4GbG1oLXRONf
iyG8j/9W+FvgB0m0DtybhnB7IYQ92F7wnDyETvCV2Gf4P8X2M79IzrPgHvxj
mT8XfoD4TsYGgSuwKeRfhjWSu5n5o8EjB4RwGF89ObYQuxmbA76WsavwlYip
JnYD/Evy1cK3aH34peAdxNeDX2fNZ8BPR//Lb9hf4L+xDvytzHkOfIwcldor
/hbmj2HsVfB8xioHeI/a23b4Nnx/Ev8AvBvrwPcB/DT7u5x/2An+irGF4O+w
DLydObWJcw6BH4ZPhg9mTl30GelsXsG/E/wf5zWp6H/oI189e7iE72TWu4X5
o+BHE5/Bh4pP/S/HsMfA7xDTxNp7yXEefx/5GsELmXMEvouYy4hthneBVzC2
Ej4YOwMfQ/5yvi3wNnJdwP9s4pr+Q64S/DXy1THWDz+J1SReoxfcjVUnrvm/
uecotgm+j7ld2CFyNSSOrYW/VPQcnVXOmncUfWbaWw92b+I9aq366LPSmlXE
jmC/34KHkbMN3zxifub/u4jph5/CZiXu2Zujz0RnoX/4Afx99L+vwl5IXXPV
Wj3Xkbrn1Gtjxcm9n7FfyH8CmwvfB/8JfBx7Wf2LtSgePp7cDdhU1Ta499Wz
6lVp4PnoHlTvncPeZe7VWEXBNZQ2b4PPSKxR9dZAxmYW3WNnwAfYw4/Et+Nf
ir8Oq4KX8G9jrYHws8E9Ku1vitaS7oBTzJ2G/6nEZ/4EvoPYPfAT8DX41mK7
yTeB+HbyrYzW7kj4uWjNSCvSlLRbDR9asIbVS9XEHCm4p+bnrpFqozN8G7w8
eq0K8vWCRzD/fubehy2Bvxjd29LcMuZ/Q57R0iP2eeoa6OxVc2nlBuK/Llgz
uptao7WgOyoHZ9gXwTWR1i7kro00p7MfRI7OxDUogScSX07uUmrtjct8FtKg
zv6Pi1pRDa6I1ri0rRqpdquxqWWuoXp1WOa7Uj07D3x39N2hNe6KXlNrvVlw
b43K3JvqsY/gbaz5RsEaljZn4l+cWKNb8S+NvnsPsodP4A/nzlWOPQT+ODVW
jpPE3pn5LZBGunXXp+5daU69dD376UncU0fxv5e59qrR4+RbkFsLQxhbpdoT
s6Dof14PXkS+GcF37I3qbfKtS1zDRv0b/vFl7rGq6BqrtsqxCd+S1Het3ogm
4k9Ha0U11t5uyqxN7XGPah19t6jGu8nVf/HuVA/qrLZmfst0ZtrL7NRvjfYk
LYyLfjukiRr4itRa0hsjbatH1ZvSuN7G87nvNr2R+pcnU7+N+qdrpN3cb0FT
4rNYFP0W6UwORcfIJ82p92al1oJ6UG9FZea7Wm/GcPUOMasTa6wG/+/R2tcd
Kt+jqe9CxfwPT/QDDg==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8885734277679622, 0.8889964438057882, 0.8576089052210717], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmFlsVVUUhs9tzzk9fTFGRcSogDi24C0YhLbgVGiLTwRoUQGNxReNUQFr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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlG1o1lUYxs/z7Dnn+fvFIsEvpaikZDOaQrq56TYnlX6SHqWQIFyCKCOj
TXEaGnO+LPHtU0FQZCurqZESbr59sG3NlvWpD73NpPKF6VAzU0Prd3H14YL7
Otd97v/5n3Pd98TG1c+9kg8hrAAF0JmFcBksh/TmQngohXASXg2vAV8Qb4sh
7CL3SfjDo0KYQ850cg/Ah9FHwErib1irR7sIb4SX2DOO/PmsVaN9ztodtK3g
flkIa+BriiE0Ub8fPszBTqAtI/91tEpwGL4ZfRO1psIfRysHj1CvnLUP0H8H
L6L1sHaL+C+wB+0J1r4k/he8QVwHthN3gJvkrgJt1P6Yei/w7Z/gP6P9Asbo
W9Q4TdybeW8r+nfkvw//jHgKWMreQ6w1sP8KvIP4KPoM8q/mXPsavKngb7Tw
vwPkjEPrBOeInyXnLPHz6J+Q2wUmkr+QtQ/RqkA38RL0T4m7kr+lf9xHPENn
gJfQb7P3bua7Hc752wfApILPsJ94XfTbjIffgF8HzfAp8AeodRw+k3gW2Eg8
FJ17kzcahH8NfiS/gbUrxE8nv73OfD77/43QlxbspTHwOQV7ajW13oUvyvsO
I3oteJvaVfIc2jQ8c564Av0q2ia9aZk9dom4kxq1eXt2Arkl9pTI7YYXM9dU
rSXsaY6+I93NUd0Z8V4wlPOeueTeAxsKPoO83g7/p8yeH+G9iuSvJPdP0IW+
O9kr8uQf8EPkP8r+ySDHeQIYDP7nv9GPoJcTTwM/RPecek090UZ8K9rbd/lm
t/otszflocXJNbRXZz6G1gNeJV5E/vfRPSDvqydfSn5DvZ08vkX+j+7de+TP
5GxVmf9lbHRvpMx3pR4ZnZwjTd9QL89j7eWCe1pel8flbXn+GbT98Lq8Pay3
q2ZtQfAbrmP/r5m9MMw3ZqPt1DzI29O6+/tF/4veIM/+PtbqidsKngWnMs8i
zQR9ayP4NudvqlZl8tuq5g7id9BnBffEb+x9D16T94x4E/2t6N7eh94XfUad
TT0vb7WCvpw9NoB2JrPX9abt0Z6T19QzLdEzUbOwBv4V/LWie1szSrNOb6C7
18xTbz2YPNvUY+fgjyWfRT3Rn3nmaNboDhLaU5l7WR48mHlGaDaop+Wt29Gz
Th6TFy9E97I8qd5RD6t31UMLkj0rr+oOmznr2qJnr2bSruQeVO/pzhqSZ5Rm
k+6kQrM/efbpjjdn9rC8q55ZHz3zNevlAXnno+TZJQ/9B16t2s4=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.9411759999999999, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNls1vVlUQxk/pvTeXlUaXYKIrrJa/QNwoRum7s0QLBBoliC78aNK+rSxM
tCCyMRYKW5CPmGjKR9tEobylLYlRqy0lYWEUUSqmTSiEJlUaF/r8nOeNXUzO
uedjzswzz8zcx159+8W3VqWU8oaUGjV+W6b0jWRjltKy5L7ktjY2a79T+72S
DZq3aNwiOa39AclGrT0jRdc1fqD1fZIuSVVyQnJI0i85Ljno75NeOyw55fGI
5L08pc8kuzR/V+NJycuaf1Gk9Llkq95p09otvdWXwrZrtm+j9h9ZndLXDaHj
+SL09Ehel7wh6Zbs9vcenUvS972GVp19dHX40CufnpNk2lsvPc2SO8JhUx72
fyr5Ujgd0PdRcNC4tYj5UxqbpedCQ2B1wXhtl74RzSua39D5Z4vAbVcW9rZp
fW8KP/7DTmt/S2op/G2R7ESn7n1chN5J3a1K/mkMO+82hp61WczB7bUUtuMz
a3e8PqQzg1m8t0bjQmPYOS6ZkVyRTEl+kVyX/ODx5xR4Mf6UAiOwAsvvhMlk
Gd9Nedh1RmcaMg4KpxQ2Mz9q26r297B586H50Ws8zmcxB/cRzS9Itumt/Y5l
j3nDvf0eq97rtw70Ys8Tediw2/tV86DH35ck0/b7D87Kl4reumh/8KWWhR7W
XlEcLpYRp6r9QfeTeeQHusHgWIq7Z1PM4U+f7n5SRN4ckI6PysDphSL4ATcG
nTPk0JBH8uacdbC3IJmTXJVUzBPegJunzU/2JiSXHdcxxxmub1kVOdSTRW7B
sf2y5asy4jTsN0/53lm/j//4NMq+zv6YB+8urfD1hNav5cGvcd/Dp19tx4zt
GrR/047BmPfGvT+aBe4dDZEXNXPwsu8Nm58125OcE8T6hvVeMZen/N0uu3bk
kdN1TqODfKoYwzrvpxzTVseVcXMeuUaOt3re7HXOkg/oJEduukbCD2oQOrZK
Xlr1f50YziIn4Rt5ydkuv8U+d6gT3OG7JmwfKgIP/MCfPufLDs8322bsqfMS
Tt51vaAW3NP3kuS3FPNlyXyKvWbf22SsDtrH9fb3mPfgIvW4zTUKDrbZBuaV
FevtxrzdXK3b2WydI/LrgSJiPea4Ea/77knUz9Eyco+8S6q1DZL3tdedBY93
uh+AY5dj2u11eME3tne7fpIjcP+dLPwgJ97Moq8s6a0/y9i7p3GxjD360lwZ
tRQ8qePYT/yGXFcXHVN6EOOw58vGGszJvXV59E1q8LJ7LrbfNA9Y327fZ72+
y7HmLvWRng0G9DPWHrfOCef/jHm46LsLtmHO67Pm6JDPDNoOcIer1F108uac
c3jGXIbT8Ip/Af4JwAQb8A3bwBMs+aZfkBvkBWuVeu+27eytUTw78+iXt+BL
EefgUdVx5GzVOUa/3eCeCCc6XCP+woYi+gWcajev6PN786h/xJ2YoxPdk+YD
/xF78si5+TJq3Fr7Pm//wWvAMVp0XMBtydguOI8GjEnzilz8vYz6uMY4g2OL
c3yD/wewa5v/c6Zc26iPTxeBEfhcdQxn/Oa8v3mTt7ENHcSvzzlLbaZG73A8
Dzq+530Gvt32/wD9kz5KT67/q520UHepv/CC+/Ct03eI+3XHnHX6I725/m+4
z+dG7Vu9fjMS04eLqHHEuN7/6BP06Xqfp+6csz/9tumEOYn98JZ8b3LfX/Z/
Dnk0VAY/wIY+D88q5iu+gkmLaxWYPVhEbYI/h/xO/X/2uG3ZZ4z6bd8Rf9O7
6Jc1nx1cYfeYY3o/D5uwh/Oj/l8C1xFje8W8o6d1+Bw6j+rubBnxSuYwfk/7
LH2Pf5aadRKvdY49dozbDs40+d8GvOmvE44La6O2fcJ3zth21v8FPy7FVQ==

         "],
         VertexColors->None], 
        PolygonBox[{{1631, 2338, 1214, 369}, {2239, 1555, 374, 512}, {2232, 
         2231, 159, 1024}, {2251, 2247, 378, 1093}, {1627, 1707, 602, 1144}, {
         2184, 2183, 1024, 1025}, {1493, 2169, 1005, 1051}, {1620, 2337, 1211,
          347}, {2282, 1604, 454, 1125}, {2280, 2281, 1052, 87}, {2236, 1552, 
         369, 507}, {2248, 2304, 1153, 1092}, {1609, 2334, 1204, 331}, {2349, 
         1608, 324, 1251}, {2247, 2249, 1092, 378}, {2307, 2250, 1093, 
         1155}, {1554, 2238, 507, 374}, {1497, 2170, 1007, 1052}, {2277, 1597,
          443, 1117}, {1706, 2293, 1136, 602}, {2169, 2277, 1117, 1005}, {
         2299, 1627, 1144, 1143}, {1678, 1439, 74, 442}, {2332, 1401, 87, 
         1198}, {2290, 2223, 469, 1133}, {1626, 2299, 1143, 159}, {1680, 1619,
          477, 578}, {2305, 2239, 512, 1153}, {1603, 2332, 1198, 324}, {1705, 
         2289, 1133, 601}, {1703, 1595, 442, 598}, {1402, 1703, 598, 454}, {
         1419, 2185, 1025, 174}, {2334, 1406, 101, 1204}, {2337, 1413, 130, 
         1211}, {2338, 1419, 174, 1214}, {2223, 1510, 331, 469}, {1413, 1681, 
         578, 130}, {2170, 2282, 1125, 1007}, {1607, 2349, 1251, 101}, {2294, 
         2228, 485, 1136}, {1412, 1705, 601, 477}, {2228, 1528, 347, 485}, {
         2275, 2276, 1051, 74}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1396, 2331, 1597, 2277, 2169, 1493, 2211, 2276, 2275, 1439, 
         1678, 1596, 2348, 1595, 1703, 1402, 2333, 1604, 2282, 2170, 1497, 
         2212, 2281, 2280, 1444, 1401, 2332, 1603, 1503, 1608, 2349, 1607, 
         1447, 1406, 2334, 1609, 1509, 2222, 1510, 2223, 2290, 2289, 1705, 
         1412, 2336, 1619, 1680, 1681, 1413, 2337, 1620, 1527, 2227, 1528, 
         2228, 2294, 2293, 1706, 1707, 1627, 2299, 1626, 1454, 1417, 2231, 
         2232, 2183, 2184, 2185, 1419, 2338, 1631, 1551, 2235, 1552, 2236, 
         1633, 2238, 1554, 2237, 1555, 2239, 2305, 2304, 2248, 2249, 2247, 
         2251, 2250, 2307}],
        "0.9`"],
       Annotation[#, 0.9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1388, 1909, 1910, 1907, 1480, 1908, 2146, 1906, 1479, 1905, 
         1478, 1579, 1578, 2273, 2272, 2168, 1477, 2203, 2269, 2268, 1431, 
         1674, 1577, 2270, 1576, 2271, 1694, 1391, 2209, 1487, 2208, 1486, 
         1585, 1584, 1435, 1675, 1587, 2346, 1586, 1698, 1395, 2330, 1592, 
         1677, 1594, 2347, 1593, 1701, 1702, 1602, 2279, 1601, 1443, 1400, 
         2215, 1502, 2174, 2175, 2285, 2286, 2283, 1606, 2284, 2218, 2219, 
         2217, 2221, 2220, 2288, 1704, 1410, 2335, 1618, 1679, 1451, 1411, 
         2226, 1526, 2178, 2179, 2180, 1415, 2229, 1535, 2181, 2182, 2297, 
         2298, 2296, 1625, 2101, 1542, 2230, 1543, 2030, 1630, 2105, 2358, 
         1550, 2233, 2234, 2047, 2046, 2301, 2110, 2111, 2053, 2055, 2054, 
         2303, 2243, 2244, 2242, 2246, 2245, 2306}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1422, 1457, 2187, 2186, 2241, 2240, 1421, 1456, 1634, 2302, 
         1635, 1713, 1712, 2109, 2044, 2045, 2043, 2104, 2103, 2300, 2028, 
         2029, 2027, 2100, 2099, 2295, 2013, 2014, 2012, 2097, 2096, 2291, 
         2292, 1999, 1525, 2225, 1524, 2357, 2094, 1617, 1986, 1517, 2224, 
         1516, 2356, 2092, 1613, 2091, 1614, 2287, 2177, 2176, 1508, 2216, 
         1405, 2173, 2172, 2171, 2214, 2213, 1399, 1442, 1599, 2278, 1600, 
         1700, 1699, 2084, 2085, 1935, 1492, 2210, 1491, 1583, 2329, 1390, 
         1434, 1581, 2345, 1582, 1485, 1575, 2328, 1386, 1430, 1573, 2344, 
         1574, 1475, 1903, 1476, 1904, 2145, 1387, 1692, 1693, 1564, 2327, 
         1383, 1427, 2262, 2201, 2202, 2167, 2263, 2264, 1565, 1888, 1466, 
         1745, 2069, 2070, 1566, 2265, 1567, 2266, 1467, 1889, 1468, 1890, 
         2139, 1892, 1469, 1891, 1470, 1893, 2119, 2360, 2120, 1649, 2313, 
         1650, 2121, 2122, 1651, 1459, 1652, 1458, 1557, 1568}],
        "0.7`"],
       Annotation[#, 0.7, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2113, 2059, 2060, 1864, 2058, 2057, 2112, 2051, 2052, 1860, 
         2050, 2049, 2107, 2108, 2042, 1549, 1856, 1548, 2041, 1629, 2026, 
         1541, 1845, 1540, 2025, 1624, 2011, 1534, 1834, 1533, 2010, 1622, 
         1998, 1523, 1823, 1522, 1997, 1616, 1985, 1515, 1812, 1514, 1984, 
         1611, 2089, 1612, 2090, 1803, 1802, 1507, 1972, 2153, 1792, 1791, 
         1790, 1501, 1960, 2151, 1780, 1779, 1778, 1948, 1947, 1394, 1438, 
         1676, 1591, 1741, 1742, 1743, 1697, 2081, 1924, 1925, 2207, 2206, 
         1484, 2205, 2359, 1572, 2267, 2143, 2144, 1429, 1673, 1571, 2141, 
         2142, 1691, 1474, 2257, 1562, 2134, 2135, 2136, 1426, 2065, 2066, 
         1883, 1464, 1744, 2067, 2258, 2259, 2137, 2138, 1885, 1465, 1884, 
         1887, 1886, 1382, 1689, 1690, 1563, 2068, 2261, 2260}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1739, 1740, 1688, 1381, 1881, 1882, 1880, 1463, 2255, 2256, 
         2252, 2254, 2253, 1425, 1380, 2133, 1902, 1473, 1754, 1755, 2077, 
         2078, 2075, 1570, 2076, 2355, 1483, 2204, 1921, 1923, 1922, 2080, 
         1695, 1696, 1589, 2083, 1588, 2274, 1437, 1393, 1943, 1944, 1776, 
         1777, 1441, 1398, 1956, 1957, 1788, 1789, 1446, 1404, 1968, 1969, 
         1800, 1801, 1449, 1408, 1980, 1981, 1811, 1813, 1814, 2156, 1994, 
         1521, 1822, 1824, 1825, 2158, 2007, 1532, 1833, 1835, 1836, 2160, 
         2022, 1539, 1844, 1846, 1847, 2162, 2038, 1547, 1855, 1857, 1858, 
         2164, 2048, 1553, 1859, 1861, 1862, 2165, 2056, 1556, 1863, 1865, 
         1866, 2166}],
        "0.5`"],
       Annotation[#, 0.5, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1719, 1716, 1717, 1645, 1715, 2351, 1714, 1718, 1710, 1711, 
         1640, 1709, 2343, 1708, 2106, 2039, 2342, 2040, 1852, 2037, 2036, 
         2102, 2023, 2024, 1841, 2021, 2020, 2098, 2008, 2009, 1830, 2006, 
         2005, 2095, 1995, 1996, 1819, 1993, 1992, 2093, 1982, 1983, 1808, 
         1979, 1978, 2088, 1970, 1971, 1797, 1967, 1966, 2087, 1958, 1959, 
         1785, 1955, 1954, 2086, 1945, 1946, 1773, 1496, 1942, 1590, 1934, 
         1490, 1766, 1489, 1933, 1580, 1920, 1482, 1760, 1481, 1919, 1569, 
         2072, 2074, 2073, 1753, 1752, 1472, 1901, 2131, 2132, 1424, 1672, 
         1561, 1734, 1735, 1736, 1687, 1738, 1737}],
        "0.4`"],
       Annotation[#, 0.4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1379, 1686, 1733, 1732, 1731, 1560, 1671, 1423, 1378, 1897, 
         1898, 1750, 1751, 1428, 1385, 1915, 1916, 1759, 1761, 1433, 2148, 
         2147, 1930, 1488, 1765, 1767, 1768, 2149, 1940, 1494, 1772, 1774, 
         1775, 2150, 1952, 1499, 1784, 1786, 1787, 2152, 1964, 1505, 1796, 
         1798, 1799, 2154, 1976, 1512, 1807, 1809, 1810, 2155, 1990, 1519, 
         1818, 1820, 1821, 2157, 2003, 1530, 1829, 1831, 1832, 2159, 2018, 
         1537, 1840, 1842, 1843, 2161, 2034, 1545, 1851, 1853, 1854, 2126, 
         2127, 1659, 1873, 1876, 1877, 1725, 2115, 1641, 1878, 1642, 2116, 
         1662, 1663, 2352, 1653, 2123, 2362, 2363, 2361, 1868, 1727, 2117, 
         1646, 1869, 1647, 2118, 1670}],
        "0.3`"],
       Annotation[#, 0.3, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1461, 2061, 1558, 1870, 2062, 2063, 1559, 2323, 2324, 2322, 
         1664, 1728, 1460, 2200, 2199, 2321, 2320, 1879, 2353, 2354, 2318, 
         2319, 1660, 2317, 2197, 2198, 1723, 1875, 1874, 1872, 1658, 2125, 
         1632, 2035, 1546, 1849, 1544, 2033, 1628, 2019, 1538, 1838, 1536, 
         2017, 1623, 2004, 1531, 1827, 1529, 2002, 1621, 1991, 1520, 1816, 
         1518, 1989, 1615, 1977, 1513, 1805, 1511, 1975, 1610, 1965, 1506, 
         1794, 1504, 1963, 1605, 1953, 1500, 1782, 1498, 1951, 1598, 1941, 
         1495, 1770, 1939, 1938, 2082, 1931, 1932, 1763, 1929, 1928, 2079, 
         1917, 1918, 1757, 1914, 1913, 2071, 1899, 1900, 1747, 1896, 2326, 
         1895, 2064, 1684, 2325, 1685, 1722, 1721, 1462, 1683, 1730}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1669, 2311, 1644, 2310, 1643, 1726, 2193, 2192, 2191, 1654, 
         2314, 1682, 2316, 1657, 2194, 2195, 2196, 1665, 2130, 2129, 1871, 
         1661, 2128, 1668, 1667, 1666, 1729, 1639, 1724, 1638, 2124, 1656, 
         2315, 1655, 2163, 1420, 2341, 2340, 2339, 1850, 1848, 2032, 2031, 
         1418, 1455, 1839, 1837, 2016, 2015, 1416, 1453, 1828, 1826, 2001, 
         2000, 1414, 1452, 1817, 1815, 1988, 1987, 1409, 1450, 1806, 1804, 
         1974, 1973, 1407, 1448, 1795, 1793, 1962, 1961, 1403, 1445, 1783, 
         1781, 1950, 1949, 1397, 1440, 1771, 1769, 1937, 1936, 1392, 1436, 
         1764, 1762, 1927, 1926, 1389, 1432, 1758, 1756, 1912, 1911, 1384, 
         2140, 1749, 1748, 1746, 1471, 2364, 1894, 2114, 2309, 1637, 2308, 
         1636, 1720, 2189, 2188, 2312, 1648, 2350, 1867, 2190}],
        "0.1`"],
       Annotation[#, 0.1, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  ImageMargins->0.,
  ImageSize->{225, Automatic},
  PlotRange->{{0, 5}, {1, 10}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 229},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->418979782]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18756],

Cell["The probability density function integrates to unity:", "ExampleText",
 CellID->19744],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"FRatioDistribution", "[", 
      RowBox[{"n", ",", "m"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"n", ">", "0"}], ",", 
      RowBox[{"m", ">", "0"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23565],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->282426340]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30953],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], ",", 
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"5", ",", "10"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->113093181],

Cell[BoxData[
 FractionBox["1155", "8"]], "Output",
 ImageSize->{37, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->511962800]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", "=", 
  RowBox[{"CharacteristicFunction", "[", 
   RowBox[{
    RowBox[{"FRatioDistribution", "[", 
     RowBox[{"5", ",", "10"}], "]"}], ",", "t"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->461150560],

Cell[BoxData[
 FractionBox[
  RowBox[{"45045", " ", 
   SqrtBox["\[Pi]"], " ", 
   RowBox[{"HypergeometricU", "[", 
    RowBox[{
     FractionBox["5", "2"], ",", 
     RowBox[{"-", "4"}], ",", 
     RowBox[{
      RowBox[{"-", "2"}], " ", "\[ImaginaryI]", " ", "t"}]}], "]"}]}], 
  "1024"]], "Output",
 ImageSize->{278, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->167635533]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{"cf", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "4"}], "}"}]}], "]"}], ",", 
   RowBox[{"t", "\[Rule]", "0"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->264624179],

Cell[BoxData[
 FractionBox["1155", "8"]], "Output",
 ImageSize->{37, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->11984194]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6767],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FRatioDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FRatioDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is not a positive real number:"
}], "ExampleText",
 CellID->32293],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"FRatioDistribution", "[", 
   RowBox[{
    RowBox[{"1", "+", "I"}], ",", "5"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23704],

Cell[BoxData[
 RowBox[{
  RowBox[{"FRatioDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(1 + \[ImaginaryI]\\) is expected to be \
positive.\"\>"}]], "Message", "MSG",
 CellID->44131263],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"FRatioDistribution", "[", 
   RowBox[{
    RowBox[{"1", "+", "\[ImaginaryI]"}], ",", "5"}], "]"}], "]"}]], "Output",
 ImageSize->{232, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->88684412]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11209],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"FRatioDistribution", "[", 
    RowBox[{"n", ",", "m"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"n", "\[Rule]", "2"}], ",", 
    RowBox[{"m", "\[Rule]", "I"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16279],

Cell[BoxData[
 RowBox[{
  RowBox[{"Greater", "::", "\<\"nord\"\>"}], ":", 
  " ", "\<\"Invalid comparison with \\!\\(\[ImaginaryI]\\) attempted. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/General/nord\\\", \
ButtonNote -> \\\"Greater::nord\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->866541612],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1V01SwjAUDk0KVMU/VFiyUXeewXEcXbl251grMzAOYwdZsGOpK1hyBJYc
wSNwDI5RkzRpw+OllB8zU9L3/72vTRqe/F6r2fF77cBvPHb9sNUOvhoPn12u
ogVCyJBfl3XC7yN+J69hFPGZil/W7/d/5ZjP59+png4Gg0gO8oNqi/zH5cmI
K81qSCc+O8oqZsZnxmehp0rW9jhtPKBMV2anIGvFkLXdAVk1Iop2VdYllCy8
JbBID53DojViJfQS8FK+JeBLjYypF9sCDc2ofopVz5AdW3f/z18GKtS+ew4z
EBzn4TFfnztn0rXgt+k3ZU6vPVutw405SpDm4iZ5R1mqJ4uWfM8X9XI0H7lY
wrHg2V1Ui7EGMSyyZekO546ZvnCHZECnu4I7q5adhAY5lMUWjVXDMh9ZeF2s
pJl2jEhm79vIVwEytN8Aubr0POE6Pkfr6u/kgUJ5DaKELLbpGhqNf3v/Vxv3
NR6PJ5OJuN0DiGtAfjOi0lye4UV0f4YcGFFhGI5GI3G7v6KWj+IuK27ran63
emXlDqxRVOGnCe58fH5E8bOFe4TIqU9Hy1EdwFRJdSXx42t/KUd8vuCnOrXK
4hXJZWXXq60I/DzgdwuweAoLen7J3Ql8o/LxUARR8F0pWJkQnVajdHcQ853S
i1UrjskiGs/ggTpVIMNzyPMazwnmPgMyfHPugXwSpbvn9tlf1u7f/F68otF6
bV6ALM3Um02n04Ecs9nsCsUU/2MhhT9gg6mE\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{142, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->102003967]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->9662],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChiSquareDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChiSquareDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NoncentralFRatioDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NoncentralFRatioDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->23615]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->16750],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Continuous Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ContinuousStatisticalDistributions"]], "MoreAbout",\

 CellID->920678]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FRatioDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 44, 22.9080656}", "context" -> "System`", 
    "keywords" -> {
     "F distribution", "Fisher's F distribution", "F ratio distribution"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "FRatioDistribution[n, m] represents an F-ratio distribution with n \
numerator and m denominator degrees of freedom.", 
    "synonyms" -> {"ratio distribution"}, "title" -> "FRatioDistribution", 
    "type" -> "Symbol", "uri" -> "ref/FRatioDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7951, 260, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->314321293]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 107011, 2402}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2038, 52, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2661, 79, 1583, 37, 70, "ObjectNameGrid"],
Cell[4247, 118, 752, 23, 70, "Usage",
 CellID->25762]
}, Open  ]],
Cell[CellGroupData[{
Cell[5036, 146, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5514, 163, 890, 33, 70, "Notes",
 CellID->429991538],
Cell[6407, 198, 309, 10, 70, "Notes",
 CellID->168709853],
Cell[6719, 210, 496, 17, 70, "Notes",
 CellID->25743],
Cell[7218, 229, 696, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7951, 260, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->314321293],
Cell[CellGroupData[{
Cell[8336, 274, 147, 5, 70, "ExampleSection",
 CellID->81272351],
Cell[8486, 281, 135, 2, 70, "ExampleText",
 CellID->6831],
Cell[CellGroupData[{
Cell[8646, 287, 165, 5, 28, "Input",
 CellID->23031],
Cell[8814, 294, 934, 19, 56, "Output",
 Evaluatable->False,
 CellID->21303637]
}, Open  ]],
Cell[CellGroupData[{
Cell[9785, 318, 168, 5, 28, "Input",
 CellID->4915],
Cell[9956, 325, 1265, 25, 66, "Output",
 Evaluatable->False,
 CellID->210704753]
}, Open  ]],
Cell[11236, 353, 120, 3, 70, "ExampleDelimiter",
 CellID->1581],
Cell[11359, 358, 160, 4, 70, "ExampleText",
 CellID->19881],
Cell[CellGroupData[{
Cell[11544, 366, 189, 6, 70, "Input",
 CellID->27132],
Cell[11736, 374, 746, 28, 69, "Output",
 CellID->140128221]
}, Open  ]],
Cell[CellGroupData[{
Cell[12519, 407, 310, 10, 28, "Input",
 CellID->19495],
Cell[12832, 419, 3104, 55, 167, "Output",
 Evaluatable->False,
 CellID->128169820]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[15985, 480, 216, 7, 70, "ExampleSection",
 CellID->30231],
Cell[16204, 489, 121, 3, 70, "ExampleText",
 CellID->25172],
Cell[CellGroupData[{
Cell[16350, 496, 196, 6, 70, "Input",
 CellID->27982],
Cell[16549, 504, 334, 10, 36, "Output",
 CellID->893117492]
}, Open  ]],
Cell[16898, 517, 121, 3, 70, "ExampleDelimiter",
 CellID->10982],
Cell[17022, 522, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[17146, 529, 168, 5, 70, "Input",
 CellID->7393],
Cell[17317, 536, 1374, 45, 74, "Output",
 CellID->657256644]
}, Open  ]],
Cell[CellGroupData[{
Cell[18728, 586, 168, 5, 70, "Input",
 CellID->2728],
Cell[18899, 593, 1606, 30, 70, "Output",
 Evaluatable->False,
 CellID->241610088]
}, Open  ]],
Cell[20520, 626, 121, 3, 70, "ExampleDelimiter",
 CellID->18362],
Cell[20644, 631, 112, 3, 70, "ExampleText",
 CellID->13279],
Cell[CellGroupData[{
Cell[20781, 638, 232, 7, 70, "Input",
 CellID->6809],
Cell[21016, 647, 185, 6, 51, "Output",
 CellID->479598763]
}, Open  ]],
Cell[21216, 656, 128, 4, 70, "ExampleText",
 CellID->3490],
Cell[CellGroupData[{
Cell[21369, 664, 233, 7, 70, "Input",
 CellID->10093],
Cell[21605, 673, 462, 8, 70, "Message",
 CellID->305322866],
Cell[22070, 683, 327, 10, 39, "Output",
 CellID->102901604]
}, Open  ]],
Cell[22412, 696, 121, 3, 70, "ExampleDelimiter",
 CellID->30364],
Cell[22536, 701, 297, 11, 70, "ExampleText",
 CellID->23047],
Cell[CellGroupData[{
Cell[22858, 716, 198, 6, 70, "Input",
 CellID->29508],
Cell[23059, 724, 176, 5, 36, "Output",
 CellID->174010482]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[23284, 735, 223, 7, 70, "ExampleSection",
 CellID->18847],
Cell[23510, 744, 262, 9, 70, "ExampleText",
 CellID->813014373],
Cell[CellGroupData[{
Cell[23797, 757, 219, 7, 70, "Input",
 CellID->944228315],
Cell[24019, 766, 404, 14, 50, "Output",
 CellID->96535367]
}, Open  ]],
Cell[24438, 783, 125, 3, 70, "ExampleDelimiter",
 CellID->713828730],
Cell[24566, 788, 115, 3, 70, "ExampleText",
 CellID->8505],
Cell[CellGroupData[{
Cell[24706, 795, 359, 12, 70, "Input",
 CellID->28184],
Cell[25068, 809, 2673, 48, 70, "Output",
 Evaluatable->False,
 CellID->246096911]
}, Open  ]],
Cell[27756, 860, 119, 3, 70, "ExampleDelimiter",
 CellID->480],
Cell[27878, 865, 217, 9, 70, "ExampleText",
 CellID->9234],
Cell[CellGroupData[{
Cell[28120, 878, 385, 12, 70, "Input",
 CellID->6698],
Cell[28508, 892, 69002, 1167, 70, "Output",
 CellID->418979782]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[97559, 2065, 233, 7, 70, "ExampleSection",
 CellID->18756],
Cell[97795, 2074, 92, 1, 70, "ExampleText",
 CellID->19744],
Cell[CellGroupData[{
Cell[97912, 2079, 474, 15, 70, "Input",
 CellID->23565],
Cell[98389, 2096, 158, 5, 36, "Output",
 CellID->282426340]
}, Open  ]],
Cell[98562, 2104, 121, 3, 70, "ExampleDelimiter",
 CellID->30953],
Cell[98686, 2109, 97, 2, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[98808, 2115, 238, 7, 70, "Input",
 CellID->113093181],
Cell[99049, 2124, 181, 6, 51, "Output",
 CellID->511962800]
}, Open  ]],
Cell[CellGroupData[{
Cell[99267, 2135, 240, 7, 70, "Input",
 CellID->461150560],
Cell[99510, 2144, 432, 15, 61, "Output",
 CellID->167635533]
}, Open  ]],
Cell[CellGroupData[{
Cell[99979, 2164, 262, 9, 70, "Input",
 CellID->264624179],
Cell[100244, 2175, 180, 6, 51, "Output",
 CellID->11984194]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[100473, 2187, 326, 9, 70, "ExampleSection",
 CellID->6767],
Cell[100802, 2198, 373, 13, 70, "ExampleText",
 CellID->32293],
Cell[CellGroupData[{
Cell[101200, 2215, 190, 6, 70, "Input",
 CellID->23704],
Cell[101393, 2223, 213, 5, 70, "Message",
 CellID->44131263],
Cell[101609, 2230, 294, 9, 36, "Output",
 CellID->88684412]
}, Open  ]],
Cell[101918, 2242, 121, 3, 70, "ExampleDelimiter",
 CellID->11209],
Cell[102042, 2247, 144, 4, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[102211, 2255, 300, 10, 70, "Input",
 CellID->16279],
Cell[102514, 2267, 381, 7, 70, "Message",
 CellID->866541612],
Cell[102898, 2276, 980, 20, 70, "Output",
 Evaluatable->False,
 CellID->102003967]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[103939, 2303, 310, 9, 70, "SeeAlsoSection",
 CellID->9662],
Cell[104252, 2314, 517, 17, 70, "SeeAlso",
 CellID->23615]
}, Open  ]],
Cell[CellGroupData[{
Cell[104806, 2336, 314, 9, 70, "TutorialsSection",
 CellID->16750],
Cell[105123, 2347, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[105317, 2355, 299, 8, 70, "MoreAboutSection"],
Cell[105619, 2365, 181, 4, 70, "MoreAbout",
 CellID->920678]
}, Open  ]],
Cell[105815, 2372, 27, 0, 70, "History"],
Cell[105845, 2374, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

