(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    427597,      10439]
NotebookOptionsPosition[    387923,       9071]
NotebookOutlinePosition[    390277,       9122]
CellTagsIndexPosition[    390190,       9117]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Mathematica Internet Connectivity" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/InternetConnectivity"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Mathematica Internet Connectivity\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"CountryData" :> 
          Documentation`HelpLookup["paclet:ref/CountryData"], "DateListPlot" :> 
          Documentation`HelpLookup["paclet:ref/DateListPlot"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"CountryData\"\>", 
       2->"\<\"DateListPlot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Integrated Data Sources" :> 
          Documentation`HelpLookup["paclet:guide/DataCollections"], 
          "New in 6.0: Data Handling & Data Sources" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60DataHandlingAndDataSources"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Integrated Data Sources\"\>", 
       2->"\<\"New in 6.0: Data Handling & Data Sources\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FinancialData", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FinancialData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FinancialData"], "[", "\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the last known price or value for the financial \
entity specified by ",
     Cell[BoxData["\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\""], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FinancialData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FinancialData"], "[", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", 
        StyleBox["start", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of dates and daily closing values for ",
     Cell[BoxData["\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\""], "InlineFormula"],
     " from ",
     Cell[BoxData[
      StyleBox["start", "TI"]], "InlineFormula"],
     " until the current date."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FinancialData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FinancialData"], "[", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["start", "TI"], ",", 
          StyleBox["end", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of dates and daily closing values for \
dates from ",
     Cell[BoxData[
      StyleBox["start", "TI"]], "InlineFormula"],
     " to ",
     Cell[BoxData[
      StyleBox["end", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FinancialData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FinancialData"], "[", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["start", "TI"], ",", 
          StyleBox["end", "TI"], ",", 
          StyleBox["period", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of dates and prices for the specified \
periods lying between ",
     Cell[BoxData[
      StyleBox["start", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["end", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FinancialData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FinancialData"], "[", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"prop\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the value of the specified property for the \
financial entity ",
     Cell[BoxData["\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\""], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FinancialData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FinancialData"], "[", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"prop\", \"TI\"]\)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["start", "TI"], ",", 
          StyleBox["end", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of dates and values of a property for a \
sequence of dates or periods. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 StyleBox["Note: ",
  FontSlant->"Italic"],
 Cell[BoxData[
  ButtonBox["FinancialData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FinancialData"]], "InlineFormula",
  FontSlant->"Italic"],
 StyleBox[" contains experimental features.",
  FontSlant->"Italic"]
}], "Notes",
 CellID->208171595],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FinancialData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FinancialData"]], "InlineFormula"],
 " can retrieve data on U.S. and other stocks, mutual funds and other \
financial instruments, as well as indices, and currency exchange rates."
}], "Notes",
 CellID->288717216],

Cell[TextData[{
 "U.S. stocks and other financial instruments can be specified by standard \
ticker symbols such as ",
 Cell[BoxData["\"\<GE\>\""], "InlineFormula"],
 ". "
}], "Notes",
 CellID->67228700],

Cell[TextData[{
 "Financial instruments can in general be specified by including the name of \
the exchange or listing, as in ",
 Cell[BoxData["\"\<NYSE:GE\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<F:BMW\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<OB:EGAN\>\""], "InlineFormula"],
 ", etc. Indices are specified as ",
 Cell[BoxData["\"\<^DJI\>\""], "InlineFormula"],
 " etc. The alternative form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"exchange\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"security\", \"TI\"]\)\>\""}], "}"}]], "InlineFormula"],
 " can also be used."
}], "Notes",
 CellID->182515409],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FinancialData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FinancialData"], "[", 
   RowBox[{
    StyleBox["patt", "TI"], ",", "\"\<Lookup\>\""}], "]"}]], 
  "InlineFormula"],
 " gives a list of financial instruments whose symbols or names match the \
string pattern ",
 Cell[BoxData[
  StyleBox["patt", "TI"]], "InlineFormula"],
 ". The search is by default done only for U.S. financial instruments."
}], "Notes",
 CellID->199059835],

Cell["\<\
Prices are given in the currencies in which instruments are quoted.\
\>", "Notes",
 CellID->426441168],

Cell[TextData[{
 "Currency exchange rates are specified for example as ",
 Cell[BoxData["\"\<EUR/USD\>\""], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"\"\<EUR\>\"", ",", "\"\<USD\>\""}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->90563636],

Cell[TextData[{
 "Times and dates can be given in any date format supported by ",
 Cell[BoxData[
  ButtonBox["DateList",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DateList"]], "InlineFormula"],
 ". ",
 Cell[BoxData[
  RowBox[{"{", 
   StyleBox["y", "TI"], "}"}]], "InlineFormula"],
 " specifies the beginning of a year."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 "Possible periods include: ",
 Cell[BoxData["\"\<Day\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Week\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Month\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Year\>\""], "InlineFormula"],
 ". "
}], "Notes",
 CellID->380179337],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["All",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/All"]], "InlineFormula"],
 " specifies every available date or time."
}], "Notes",
 CellID->91804797],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FinancialData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FinancialData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", 
    StyleBox["\[Ellipsis]", "TR"], ",", 
    RowBox[{"{", 
     StyleBox["date", "TI"], "}"}]}], "]"}]], "InlineFormula"],
 " gives results for a particular date."
}], "Notes",
 CellID->103182654],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FinancialData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FinancialData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", " ", "\"\<Properties\>\""}], "]"}]], 
  "InlineFormula"],
 " gives a list of all properties available for a financial entity."
}], "Notes",
 CellID->54231339],

Cell["\<\
Properties related to the most recent available prices include:\
\>", "Notes",
 CellID->143754142],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Ask\>\"", Cell[
    "the lowest offer to sell currently available ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Bid\>\"", Cell[
    "the highest offer to buy currently available ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<LatestTrade\>\"", Cell[
    "the price and time for the most recent available trade", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Price\>\"", Cell[
    "the price for the most recent available trade", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True}}}},
 CellID->121387011],

Cell[TextData[{
 "The default data sources used by ",
 Cell[BoxData[
  ButtonBox["FinancialData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FinancialData"]], "InlineFormula"],
 " typically give data delayed by 15 minutes or more. "
}], "Notes",
 CellID->321673510],

Cell["\<\
All times are given for the location of the exchange on which a financial \
instrument is traded, or New York time for global instruments such as \
currencies.\
\>", "Notes",
 CellID->67000935],

Cell["Properties related to daily prices and trading include:", "Notes",
 CellID->622463],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Change\>\"", Cell[
    "price change from the previous closing price", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Close\>\"", Cell[
    "latest known closing price", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FractionalChange\>\"", Cell[
    "fractional price change from the previous closing price", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<High\>\"", Cell[
    "highest price during the trading day", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Low\>\"", Cell[
    "lowest price during the trading day", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Open\>\"", Cell[
    "opening price for the day", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Range\>\"", Cell[
    "lowest and highest prices during the trading day", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Return\>\"", Cell[
    "daily return on a particular day, allowing dividends", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Volume\>\"", Cell[
    "number of units traded during the day", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True, True}}}},
 CellID->119473767],

Cell["\<\
When markets are closed, daily information is given for the most recent \
trading day.\
\>", "Notes",
 CellID->7509541],

Cell["\<\
When historical price and trading properties are requested, daily values are \
given by default. Properties over longer periods can also be requested.\
\>", "Notes",
 CellID->47166445],

Cell[TextData[{
 "For historical data, properties such as ",
 Cell[BoxData["\"\<High\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Low\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Close\>\""], "InlineFormula"],
 " are adjusted for stock splits, dividends and related changes."
}], "Notes",
 CellID->703660863],

Cell["Properties related to raw historical daily prices include:", "Notes",
 CellID->41178427],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<RawClose\>\"", Cell[
    "closing unadjusted price for a period", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<RawRange\>\"", Cell[
    "lowest and highest unadjusted prices during a period", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<RawHigh\>\"", Cell[
    "highest unadjusted price during a period", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<RawLow\>\"", Cell[
    "lowest unadjusted price during a period", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<RawOpen\>\"", Cell[
    "opening unadjusted price for a period", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True}}}},
 CellID->89792513],

Cell["\<\
Properties related to long-term historical prices and trends include:\
\>", "Notes",
 CellID->215444426],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Average200Day\>\"", Cell[
    "200-day moving average", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Average50Day\>\"", Cell[
    "50-day moving average", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<AverageVolume3Month\>\"", Cell[
    "3-month average of daily trading volume", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Change200Day\>\"", Cell[
    "change in price from 200-day moving average", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Change50Day\>\"", Cell[
    "change in price from 50-day moving average", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ChangeHigh52Week\>\"", Cell[
    "change in price from 52-week high", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ChangeLow52Week\>\"", Cell[
    "change in price from 52-week low", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<CumulativeFractionalChange\>\"", 
    Cell["cumulative fractional change over a given range of dates ", 
     "TableText"]},
   {Cell[" ", "TableRowIcon"], "\"\<CumulativeReturn\>\"", Cell[
    "cumulative return over a given range of dates", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FractionalChange200Day\>\"", Cell[
    "fractional change in price from 200-day moving average", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FractionalChange50Day\>\"", Cell[
    "fractional change in price from 50-day moving average", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FractionalChangeHigh52Week\>\"", 
    Cell["fractional change from 52-week high", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FractionalChangeLow52Week\>\"", Cell[
    "fractional change from 52-week low", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<High52Week\>\"", Cell[
    "highest price over preceding 52 weeks", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Low52Week\>\"", Cell[
    "lowest price over preceding 52 weeks", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Range52Week\>\"", Cell[
    "price range over preceding 52 weeks", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Volatility20Day\>\"", Cell[
    "price volatility over preceding 20 days", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Volatility50Day\>\"", Cell[
    "price volatility over preceding 50 days", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, None, None, 
     None, None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, True, 
      True, True, True, True, True, True, True}}}},
 CellID->687300919],

Cell["Name-related properties include:", "Notes",
 CellID->21048359],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<CIK\>\"", Cell[
    "CIK number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Company\>\"", Cell[
    "name of corporate entity (if applicable)", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<CUSIP\>\"", Cell[
    "CUSIP number for a U.S. financial entity", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Exchange\>\"", Cell[
    "exchange to which financial entity refers", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ISIN\>\"", Cell[
    "International Securities Identifying Number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Name\>\"", Cell[
    "English name for a financial entity", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StandardName\>\"", Cell[TextData[{
     "standard ",
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " name for a financial entity"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Symbol\>\"", Cell[
    "ticker symbol", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True}}}},
 CellID->362079982],

Cell["Company-related properties include:", "Notes",
 CellID->32994604],

Cell[BoxData[GridBox[{
   {Cell[" ", "TableRowIcon"], "\"\<Sector\>\"", Cell[
    "industry sector in which a company operates ", "TableText"]},
   {Cell[" ", "TableRowIcon"], "\"\<SICCode\>\"", Cell[
    "primary SIC code for a company", "TableText"]},
   {Cell[" ", "TableRowIcon"], "\"\<Website\>\"", Cell[
    "company website URL", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True}}}},
 CellID->595208210],

Cell["Fundamentals-related properties for companies include:", "Notes",
 CellID->536689746],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<BookValuePerShare\>\"", Cell[
    "book value per share", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FloatShares\>\"", Cell[
    "number of shares available for trade in the open market", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ForwardPERatio\>\"", Cell[
    "forward price-to-earnings ratio estimate", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<MarketCap\>\"", Cell[
    "market capitalization", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PEGRatio\>\"", Cell[
    "price-to-earnings divided by projected growth rate", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PERatio\>\"", Cell[
    "price-to-earnings ratio", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PriceTarget\>\"", Cell[
    "average of analysts' 1-year price targets", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PriceToBookRatio\>\"", Cell[
    "price divided by book value", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PriceToSalesRatio\>\"", Cell[
    "price divided by annual sales ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ShortRatio\>\"", Cell[
    "fraction of trading volume sold short ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<YearPERatioEstimate\>\"", Cell[
    "ratio of price to current-year estimated earnings", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, 
      True}}}},
 CellID->955033037],

Cell["Earnings-related properties for companies include:", "Notes",
 CellID->411153225],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<EarningsPerShare\>\"", Cell[
    "earnings for the most recent four quarters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<EBITDA\>\"", Cell[
    "standard cash flow measure", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ForwardEarnings\>\"", Cell[
    "average analysts' earnings per share for the next year", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<QuarterForwardEarnings\>\"", Cell[
    "average analysts' earnings per share for the next quarter", 
     "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<YearEarningsEstimate\>\"", Cell[
    "average analysts' earnings per share for the current year", 
     "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True}}}},
 CellID->322122191],

Cell["Dividends-related properties for companies include:", "Notes",
 CellID->400765134],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<DividendYield\>\"", Cell[
    "ratio of annual dividend to current price ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DividendPerShare\>\"", Cell[
    "annual dividend per share", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Dividend\>\"", Cell[
    "dividend during a specified period ", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True}}}},
 CellID->230349554],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FinancialData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FinancialData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"group\", \"TI\"]\)\>\"", ",", "\"\<Members\>\""}], "]"}]], 
  "InlineFormula"],
 " gives a list of the member entities of a specified group."
}], "Notes",
 CellID->112592060],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FinancialData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FinancialData"], "[", "\"\<Groups\>\"", "]"}]], 
  "InlineFormula"],
 " gives a list of available groups."
}], "Notes",
 CellID->25099614],

Cell["\<\
Groups include companies in an index, and companies on a particular exchange.\
\
\>", "Notes",
 CellID->327866574],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FinancialData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FinancialData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\"", ",", 
    StyleBox["\[Ellipsis]", "TR"], ",", "\"\<\!\(\*
StyleBox[\"form\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
 " can give data in various forms, including:"
}], "Notes",
 CellID->553509082],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Value\>\"", Cell[
    "value only", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DateValue\>\"", Cell[TextData[{
     "list of the form ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        StyleBox["date", "TI"], ",", 
        StyleBox["value", "TI"]}], "}"}]], "InlineFormula"]
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True}}}},
 CellID->17546678],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FinancialData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FinancialData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\"", ",", 
    StyleBox["\[Ellipsis]", "TR"], ",", "\"\<\!\(\*
StyleBox[\"ann\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " gives various annotations associated with a property. Typical annotations \
include:"
}], "Notes",
 CellID->630797083],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Currency\>\"", Cell[
    "currency ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Description\>\"", Cell[
    "description of the property", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<LongDescription\>\"", Cell[
    "longer textual description of the property", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Units\>\"", Cell[
    "units in which the value is given", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsName\>\"", Cell[
    "English name for the units used", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsNotation\>\"", Cell[
    "notation for the units used", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsStandardName\>\"", Cell[
    TextData[{
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " standard name for the units used"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->217110870],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FinancialData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FinancialData"]], "InlineFormula"],
 " provides gateways to external financial data sources. Its use is subject \
to any restrictions associated with those sources, and may require additional \
licensing. "
}], "Notes",
 CellID->122835266],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FinancialData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FinancialData"]], "InlineFormula"],
 " requires internet connectivity."
}], "Notes",
 CellID->94188222],

Cell[TextData[{
 "Note: ",
 Cell[BoxData[
  ButtonBox["FinancialData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FinancialData"]], "InlineFormula"],
 " is intended for informational purposes only. Wolfram Research is not \
responsible for the accuracy or timeliness of any data."
}], "Notes",
 CellID->168949472]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->463832686],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(11)", "ExampleCount"]
}], "ExampleSection",
 CellID->351172630],

Cell["Find the latest price for GE stock:", "ExampleText",
 CellID->3690073],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<GE\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->276236122],

Cell[BoxData["35.13`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->805037850]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->370889348],

Cell["Find the latest market cap for GE:", "ExampleText",
 CellID->50749667],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<MarketCap\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->67295819],

Cell[BoxData["3.533`*^11"], "Output",
 ImageSize->{75, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->120606000]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->212765019],

Cell["Plot the stock price for GE since January 1, 2000:", "ExampleText",
 CellID->300090350],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "\"\<Jan. 1, 2000\>\""}], "]"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->416856143],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{331, 227},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->679432529]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->741408050],

Cell["\<\
Plot the log of the Dow-Jones Industrial Average for all available dates:\
\>", "ExampleText",
 CellID->16278858],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{"#1", ",", 
      RowBox[{"Log", "[", 
       RowBox[{"10", ",", "#2"}], "]"}]}], "}"}], "&"}], "@@@", 
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<^DJI\>\"", ",", "All"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29514426],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXd2PHEcRH9/M7Md9+OzYARsCcUhIooBDyAeOk9x+3fl8PjtYIoRIFhEc
TqRECo5lzJ8AhAje7+bsJI5IgmTxEBmFNxSJF5CMeOUNiX9k6Orqru6p6dnZ
mekxJ/DqfLc9Ozv1q+qq6uqqmvGLW1ffeP2nW1ffvLh17PSVrctvvHnxZ8fW
374iDoX7gmDfRhAE/zoSiPepeCf/weuM+Idv+btnxe9QnBaEcihfMbyN9Hfh
NScGzzvOjNRnYYVznhN/4eJz7Lj++2X51ZPwOyQM8uzc2GBU17PPcF6h8BtB
K1cA6QYxfe44q2wMb48YiWRmhUt6jibPgijnxPG9Ljtvmgx8fdsX7/o8t2Qz
5z3fGMF0HuDtnJPyCxXoVOa8lPoMfLckr+f2It+ZI2ELXJ+AX528SdTjHvyG
7ROjLL7UeFt+lpaK63P+t+xz9MRArXj9cL07oCRwEhiUI5vDHhvzz//Xx4vi
1767kimVDHzAJdKR459f/Wy4svOjV2/iMrSHOGh7PK/kUq47GMP98MLvhaTw
p0xW0nbD27dvzzi+Py3SYxhHguKXKl7xLgX/FEJBIQhHA9CA3/7mL7i+eKMW
qr+xpDJcSdRZX2hI5StpkZ7jeLSSoFID0R2/PHFqnbsU9gyFIBReXx271ys1
rcldQeWLDa9cpr3jjPYG4bvvfN4KT+WUhUdonfKyHOOaDUQj5ZGS1invl+OP
fvcPRbnjYW7LqL3y8seK2pxidptW4LZpb268z2mPhzjhSbu045Z5BXLI40gT
UDyOBzveaBbbraTdGQ8STWv77Uu3/vTZP6UVRcKKUMnEm0OtyjlUprutjklb
ikYa19rk+j13lH5PjO8ExUtvfTqFoox1w/Mv3kARBOPaCHRekucZYgcysrFU
v/oS67nNG9ZZHCfwUx8fJgJ08oBlBfSRUpzz4gjuF8xOKmY08dob69fJuuqj
5hLN51+AGhxYUhIKMyhJv50ocT0R866ODSrihCxHLHGGDGc+c1RVviZOdSEH
rzmqLVVfaHXsvm37VSsy0Fprx/dzDBViIW1R0YWJ1VZqc+ngaDYeY5vHEcUd
4uh3N9/XGO18guYdaQgsyM54uAvcQlw4rM+FlaUHmkU+Ret+0P3wxt/Pbn4g
IsO1UTIcJOPBdqr8PWKmtchgBspWpJNdj+tjd2VpgpiiKlwOZWbCoBuVo4sF
S9YVksp2W4bw3OYHxncNEWEQXXjlYzoKWKX2x5gnQc/97q//GHStdU7rim98
Z9bfU0gGOyKUCBbSvJ2BLvT0qrI22SXpKWynxrvK1igAwVx2o0hL4jNRhfhZ
VLMLOnuQxQCApyrN4khrMc1HGThztHfJ4EDDgBO6bB044XW+OCbXbIGf2I+o
dIBIM3iydTQYCxL/4rWIZ5H3QI9zevWabYfWTuyZ1jGCteE6ZfmPbYib+inG
T+3PWmxxLX6r9Y68Uca+4Dsyb4raJzzxztmN6zzb9bQ3zAtymmg9Ur51SZ71
y1/82V5dYTLprNVxEvSYPchrhQq3Y01+qhFqAE1aF55e27WkLuXT+cH3PyJ8
whJe27qJuyWS5PfO3+BZCTkX8xh7aP+T4CJMLhpjD/o0eLgiH8W+x7ZpLUWU
vViNdRynVER9Tss0j4h0dldrGrztWp5ac1sVfbHuSC8Z2xl/0B08629//ffQ
zr4Y/nhcarK5JipaGGYjwge9YuaIlplO8BlxIZbWgV8JDjDrbBetnf0uQgj7
1XuYDL+2J1AdEuc85A2JjP0iikQI12S0k5krDMDJ3kswYiwrrIgcB3Gb0RCz
8h9rxFGoYvQgwlzeqdVrqfYLIo60kAWxzkZtT8a7l976VNUjs/iDcHWYFPAr
vUaHfLRYqsXyIsQDc1dVQ4r9GnLzzq8+F1e9fOmWmCDkRnjlDE7hJVTU0mfX
PuTQsZ7SMX+abEtd/Igp5/K634FL7CSsyFDvRXKriuUR+FUaIg5Rstq7PsC+
fVhyRcorFmgThRhMX/WMaX0tuxcvluGDTFpwfjMLymUlYty6qJhLrNncN3FM
DykJ+cWBkkh1aJFbC10opFWM7QjKzNl9LaATkQ4GG6XoHk61Fpm9H0YhZly1
DptFCADhioiEIgslh1cvfDIFWV97bbLMqliK/RlyaTIgPGLnWoMxltHwebZe
VM3ZT5/HIKZoS/y8fvEPuZnk+B6RY0j0GHxVq2LFmEDhccZYbWoKokdT2j2S
tlfN809HhLsVk9d3IZCSNI5zBfc4h2vj4Nl9OObKxAG+LpMQ9+rHU2uvoPzD
XCN02dw+79Zx5dCV3g8S3H7lYsrjWobnRMA1SNBgc17sYG3EHFPHgVnLPFY2
p/eJPCp/MkU/U9UWi7P3fFadVYjwyuVbllf7ugNVEL90/kOyZ6iV1LcEjlF3
WnKsUY4uxDUc3VNKZvXxZHPpThRMojya4POoMdl7neX6+Ky8eTe1emFroOJx
YtJA87lHw2470iUBhM/V00ou/mjKOkIJ3zIzldEjiAoOeMUQhJvr14V7Ebu8
nIY+o7huRhEIPl5wZduToWaY6KdTkWpxxOGiDh4NdywU5QjnwHew9WXromln
oviuJnRIhGel/KH5jhyzPVaMNo+V+MloV8b5UWsYIFe836KLL/F5cNYrjeWh
qsq1xoWdE1lsTM1lL/JOpHAy0NlgipmxICbnryrV2e0F7gnhNTVdiU5o2rBm
bvo4ymrjs8tY3pPSt/IsO5un35NhpV1Z5dYC1s375Moq2bNjWk2VvXbx6vjJ
ZKTz2jgbZtcF8hp6pZ73JgQjVy0AS35MCsqqTiRlq9nseNa0NNj+XMeOj6Wz
rZ/VKC5YFLAP26weS7UoumxvTY5NDr5nUbX7hbB/DevW42EyM4LZ7fCUHFM8
QHVuXaEAREflOWPKWdaTwzQU66nOh5uVCva7oHPtUAu6LEc4gkzre8YLWnsA
29oxv2jX48bDXVnj4ist7/Wqj/c0/IpUPUP6J9AVQqr2mpHKEcCMLaq/ZVHW
7BjkN2Mai5eI8TJZO40BfTbl91TmmOeKhNTUle2+M/25PGbnihKMGV5+CSxB
2EdTTqyx9PE9sxLIxQB3xuKgM6LM1NQADeAFy8E8s+l14fse3eNl+1HTdUTt
Rbl1rj53ZZnvUCHqMuSI0iCKrVnVfiJ2cruhu1hUBEicipeQmzxrOn8uj8n5
wlr2ZFDEF3lTEDmVhueYrtFuyeIAuJp3XG2J4gEjhdDBy+y+l/O0mJq+aQVY
LD59S1+MlE13QqSmK8lIGdaxkM2PXts6pTNQFbWtYVRt5TJW+DskR4MsdMwD
oN/fCuIlhrjLqKPETSTY0x4ro+fifCsvzzlakudw2zAaxaPG+tzsZ9zwFayn
9MP0PkLeVO6SIsy24rISMsQ0bximBLA+DwtRA+Bc/2mQzyCVcYOZIVMfjbW2
nxrvEjesXoqyNrVLONZTEwnnqT2O7C8Rq+hkvHtm/Rp1h12zcrVT+OO8wa+p
fadlnOI8y82WqSV1iTMCPV7ZzlXHgPoiO2Zy6Kbjh3IPKNBM3UFMay66eaEh
/3DgeAnfWCe3o6+1cYKZmY6VL9tB+5NS7VBTDaX/eYUH8RkrUD3gY21vduYH
9J9XYHgOnEviPsVdMO25KDW04BDhluyqaAi0Qlb+Tc9ahP4GxUB9qna3o8RC
V55oi87eGzIonOHZevKr8CY1we7S3JaxINQffvLjmxunrmMJAmOUnh1byv1I
InYcwqh1vRE1jnYhwlE9W8iLf2t9nM0Lv7NRSr+zad0dIRgQGAH5FJxBcfd3
GSI+Bs3pMy3m/ZuGctVVxkUNoiTenemXAt8j272pvFvRH2XQUoiVpktvlijV
dWVXNJ7dbZr5e5J9v35s6dwXdHSPnFxnhtvC/YJX4d2r/iSL38/mmNqjJjMG
uRpPe3ojV7bM/QTt0YJ1I9wDFOpYgb6yvQt+gp3jT9N1pxq3Jb+SsuvGuO/F
LJ65n+6brdGHDoPcXnfnDtDLdJwDj99onybbWbZL0b43osc09pGplOvYBXb4
yI1OYiiabDDvDPdnI5BnVvu/TAT9aGvSNRTNasDv2+S14zaoG20C7Zo+p00p
6og9kUJOAtpxq3N437Vf6nY/GcYdly/dUp+bDIbJ+PH+Jn9ojqTmvh9NOWS6
zrtf/VIPYkwn0MZE9enZ+0+ot7g6TrNI6tj50dTa06X61XfQ8mfhRDOXCeqV
8uiFrtUpr/Mz7VLF7DDuqmGhWmCxNX/Kkj/qDxB19OMws7wz2FDjmTM45mM/
CN7Eld+egoRRLOuX84XIvnfraCN0s+SfXKisDJN4k80wFT3Byp0R4k9dayoz
qHJ3mW/kncBGRpyafv5jU12SdX5r36p7N/VKzXtci2eNj6vmYfgY4r95lsXk
lWILTVDcNViHchDJjE8yks3IPaYxvMvR4GjqYb7F5oM/CYvXnv1RfsJBmVeZ
/VH7NqPmqmlnqdVZj2FnaPklqgZwSv5WY7kXjbAWgTUYHgsutSZT6NMDr9ue
dspOwNjOg4Bnb08nTyiO2qdgW/dCq9R4lnC6PtTRepmVjqjS5qDhT99PkvyM
hvdbkx9QA0/R3gxpCrY+9Fqlxu9LmG+NGj6tO6t97VGTzwmJKQcwxD7P6bpR
R9uhnsfzC7wb3J++AzXQj/Z0AuuT5GPlLLVHbSXV+p7tZWqbmukS49SwX8n0
Zt7rlbrdfY0rMz67AnKcvCca666UdhVh+mFvWLBay+JuR4d0tqPJH/1hijt1
Vx90vlfExJ8H06b2WkQ5ZPMO8efB1JfdTqe6qnJGiaR6yKuU7bxvXICg75Uq
PL2vY1mZ+ETuE3kN+YBXirbPou5uPCVzr7LOm9g+YDnN7HNjQgKjwPwvB013
uxOSDMdmNF7uOcEpUYFnKS3ahTfvpHAhlJlzectzksuiUJfUa1s3h1RJML67
T1ijoOgZenVyKdBNvjw0fSlroyTg+hSnJm8SUMareSeNC4vtO7FfynSJUY+n
8iRdkonfDh8+hu7zpXwmdpBIhHcSBWZETe8RejnTccG7YKO0SMN95ZkAlaaa
6WHr0ZNI5MKsEAvo+p47Px5K3hnAe/JiJGbuaQQ/1GG+ac4bhg05poc8qmyf
wQS2HnmjxseQ5eS9cVnegNhe+39Jyp7oJHO3mbteQbJz6az/d8Z/m787PcYV
gVqDhNkVa9z/u6wW70pmJsmY//8JBtJC99S7YN9/AJ5qXQY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{328, 223},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->118794820]
}, Open  ]],

Cell["Join points and fill:", "ExampleText",
 CellID->158291915],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"#1", ",", 
       RowBox[{"Log", "[", 
        RowBox[{"10", ",", "#2"}], "]"}]}], "}"}], "&"}], "@@@", 
    RowBox[{"FinancialData", "[", 
     RowBox[{"\"\<^DJI\>\"", ",", "All"}], "]"}]}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Bottom"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->525400874],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{331, 225},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->15421067]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->120816537],

Cell["Plot the trading volume for GE in the first quarter of 2000:", \
"ExampleText",
 CellID->10783646],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<GE\>\"", ",", "\"\<Volume\>\"", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"2000", ",", "1", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"2000", ",", "4", ",", "1"}], "}"}]}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->619929818],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{257, 160},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->213560312]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->912995407],

Cell["\<\
Find a list of NYSE stocks whose ticker symbols begin with UN:\
\>", "ExampleText",
 CellID->127449912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<NYSE:UN*\>\"", ",", "\"\<Lookup\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->885562420],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"NYSE:UN\"\>", ",", "\<\"NYSE:UNF\"\>", ",", "\<\"NYSE:UNH\"\>",
    ",", "\<\"NYSE:UNM\"\>", ",", "\<\"NYSE:UNM-PA\"\>", 
   ",", "\<\"NYSE:UNN\"\>", ",", "\<\"NYSE:UNP\"\>", ",", "\<\"NYSE:UNS\"\>", 
   ",", "\<\"NYSE:UNT\"\>"}], "}"}]], "Output",
 ImageSize->{364, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61089102]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->277215538],

Cell["Give the members of the Dow Jones Industrial Average:", "ExampleText",
 CellID->447998576],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<^DJI\>\"", ",", "\"\<Members\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->215281312],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AA\"\>", ",", "\<\"AIG\"\>", ",", "\<\"AXP\"\>", 
   ",", "\<\"BA\"\>", ",", "\<\"C\"\>", ",", "\<\"CAT\"\>", ",", "\<\"DD\"\>",
    ",", "\<\"DIS\"\>", ",", "\<\"GE\"\>", ",", "\<\"GM\"\>", 
   ",", "\<\"HD\"\>", ",", "\<\"HON\"\>", ",", "\<\"HPQ\"\>", 
   ",", "\<\"IBM\"\>", ",", "\<\"INTC\"\>", ",", "\<\"JNJ\"\>", 
   ",", "\<\"JPM\"\>", ",", "\<\"KO\"\>", ",", "\<\"MCD\"\>", 
   ",", "\<\"MMM\"\>", ",", "\<\"MO\"\>", ",", "\<\"MRK\"\>", 
   ",", "\<\"MSFT\"\>", ",", "\<\"PFE\"\>", ",", "\<\"PG\"\>", 
   ",", "\<\"T\"\>", ",", "\<\"UTX\"\>", ",", "\<\"VZ\"\>", 
   ",", "\<\"WMT\"\>", ",", "\<\"XOM\"\>"}], "}"}]], "Output",
 ImageSize->{455, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->227354048]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->658206691],

Cell["Find a list of companies in the books industry segment:", "ExampleText",
 CellID->82597117],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<Books\>\"", ",", "\"\<Members\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->411671714],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"DE:PES\"\>", ",", "\<\"DE:SL1\"\>", ",", "\<\"F:PES\"\>", 
   ",", "\<\"F:SL1\"\>", ",", "\<\"NASDAQ:CRRC\"\>", 
   ",", "\<\"NASDAQ:SCHL\"\>", ",", "\<\"NYSE:HTN\"\>", 
   ",", "\<\"NYSE:MHP\"\>", ",", "\<\"NYSE:PSO\"\>", ",", "\<\"OB:LNGMF\"\>", 
   ",", "\<\"OB:MPNC\"\>", ",", "\<\"OB:SPLIF\"\>", ",", "\<\"OB:TRDY\"\>", 
   ",", "\<\"PK:GBKF\"\>", ",", "\<\"PK:IVBK\"\>", ",", "\<\"PK:VBIB\"\>", 
   ",", "\<\"V:LMD\"\>"}], "}"}]], "Output",
 ImageSize->{556, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->766076723]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->177809840],

Cell["\<\
Find the current net asset value for the Fidelity Magellan fund:\
\>", "ExampleText",
 CellID->131409717],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<FMAGX\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->68668378],

Cell[BoxData["89.19`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->355381940]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->695728270],

Cell["\<\
Find the current exchange rate between euros and U.S. dollars:\
\>", "ExampleText",
 CellID->226997588],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<EUR/USD\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->446276343],

Cell[BoxData["1.3303`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->82861923]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->268634527],

Cell["Find the price of gold in U.S. dollars:", "ExampleText",
 CellID->11500889],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<XAU/USD\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->590840916],

Cell[BoxData["624.6`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->493099498]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(48)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Names & Identifiers",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(15)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->265569808],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell["Specifying Exchanges", "ExampleSubsubsection"],
  $Line = 0; Null]], "ExampleSubsubsection",
 CellID->722287],

Cell["Find the stock price for GE:", "ExampleText",
 CellID->2099160],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<GE\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->298533035],

Cell[BoxData["34.36`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->592195199]
}, Open  ]],

Cell["Specify the exchange explicitly:", "ExampleText",
 CellID->78002510],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<NYSE:GE\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->739276185],

Cell[BoxData["34.36`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->9046840]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"{", 
   RowBox[{"\"\<NYSE\>\"", ",", "\"\<GE\>\""}], "}"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->85141268],

Cell[BoxData["34.36`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->213366921]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->176730052],

Cell["Find the stock price for Apple:", "ExampleText",
 CellID->71027552],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<AAPL\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->464158350],

Cell[BoxData["89.59`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->320877701]
}, Open  ]],

Cell["Specify the exchange explicitly:", "ExampleText",
 CellID->328506286],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<NASDAQ:AAPL\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->108438947],

Cell[BoxData["89.59`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->595793689]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->36845349],

Cell["For non-U.S. stocks, the exchange must be specified:", "ExampleText",
 CellID->387738862],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<F:BMW\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->458258508],

Cell[BoxData["41.04`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->649785924]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<L:BP\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->226929606],

Cell[BoxData["511.5`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->43187262]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[
  "Getting Names & Identifiers", "ExampleSubsubsection"],
  $Line = 0; Null]], "ExampleSubsubsection",
 CellID->332015634],

Cell["Find the name corresponding to a ticker symbol:", "ExampleText",
 CellID->93757185],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->25987035],

Cell[BoxData["\<\"General Electric Company\"\>"], "Output",
 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->541322348]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17993225],

Cell["Find the name for a ticker symbol on a non-U.S. exchange:", \
"ExampleText",
 CellID->12201578],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<F:BMW\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->224287922],

Cell[BoxData["\<\"Bayerische Motoren Werke AG\"\>"], "Output",
 ImageSize->{193, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23598697]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->332574725],

Cell["Find back-office identifiers for stocks:", "ExampleText",
 CellID->165557687],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<CUSIP\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->288461389],

Cell[BoxData["\<\"369604103\"\>"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->941911628]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<ISIN\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->51887334],

Cell[BoxData["\<\"US3696041033\"\>"], "Output",
 ImageSize->{88, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->31938098]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->198424648],

Cell["Find the exchange on which a stock is traded:", "ExampleText",
 CellID->575434581],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Exchange\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->428178773],

Cell[BoxData["\<\"NYSE\"\>"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->97857245]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell["Looking Up Names", "ExampleSubsubsection"],
  $Line = 0; Null]], "ExampleSubsubsection",
 CellID->42165130],

Cell["Look up all instruments whose symbols begin with ABD:", "ExampleText",
 CellID->187325701],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<ABD*\>\"", ",", "\"\<Lookup\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->265754125],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"ABDAX\"\>", ",", "\<\"ABDBX\"\>", ",", "\<\"ABDIX\"\>", 
   ",", "\<\"NYSE:ABD\"\>"}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1287019696]
}, Open  ]],

Cell["Find the names corresponding to these symbols:", "ExampleText",
 CellID->537757506],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"#", ",", "\"\<Name\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<ABD*\>\"", ",", "\"\<Lookup\>\""}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->11470067],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Axp Portf Builder Conservative\"\>", 
   ",", "\<\"Allegiant Bond Fund Class B\"\>", 
   ",", "\<\"Axp Core Bond Fund Class I\"\>", 
   ",", "\<\"ACCO Brands Corporation\"\>"}], "}"}]], "Output",
 ImageSize->{429, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1484738088]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->646995844],

Cell["Use arbitrary string patterns for lookups:", "ExampleText",
 CellID->346300904],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{
   RowBox[{"\"\<A\>\"", "~~", "___", "~~", "\"\<ZZ\>\""}], ",", 
   "\"\<Lookup\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->898062766],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AMEX:MZZ\"\>", ",", "\<\"AMEX:PZZ\"\>", 
   ",", "\<\"AMEX:ZZZZ\"\>", ",", "\<\"AX:AZZ\"\>", ",", "\<\"NYSE:AZZ\"\>"}],
   "}"}]], "Output",
 ImageSize->{336, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->233066876]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell["Groups and Sectors", "ExampleSubsubsection"],
  $Line = 0; Null]], "ExampleSubsubsection",
 CellID->206196781],

Cell["Give a list of all general types of groups supported:", "ExampleText",
 CellID->53789131],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<Groups\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->37322416],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Currencies\"\>", ",", "\<\"Exchanges\"\>", 
   ",", "\<\"ExchangeTradedFunds\"\>", ",", "\<\"Futures\"\>", 
   ",", "\<\"Indices\"\>", ",", "\<\"MutualFunds\"\>", 
   ",", "\<\"Sectors\"\>"}], "}"}]], "Output",
 ImageSize->{307, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->313618589]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->135005035],

Cell["Find the total number of exchanges supported:", "ExampleText",
 CellID->16994087],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", 
  RowBox[{"FinancialData", "[", "\"\<Exchanges\>\"", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->54396352],

Cell[BoxData["85"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->148380073]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->118942676],

Cell["Find a list of companies in the books industry segment:", "ExampleText",
 CellID->284166854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<Books\>\"", ",", "\"\<Members\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->351923496],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"DE:PES\"\>", ",", "\<\"DE:SL1\"\>", ",", "\<\"F:PES\"\>", 
   ",", "\<\"F:SL1\"\>", ",", "\<\"NASDAQ:CRRC\"\>", 
   ",", "\<\"NASDAQ:SCHL\"\>", ",", "\<\"NYSE:HTN\"\>", 
   ",", "\<\"NYSE:MHP\"\>", ",", "\<\"NYSE:PSO\"\>", ",", "\<\"OB:LNGMF\"\>", 
   ",", "\<\"OB:MPNC\"\>", ",", "\<\"OB:SPLIF\"\>", ",", "\<\"OB:TRDY\"\>", 
   ",", "\<\"PK:GBKF\"\>", ",", "\<\"PK:IVBK\"\>", ",", "\<\"PK:VBIB\"\>", 
   ",", "\<\"V:LMD\"\>"}], "}"}]], "Output",
 ImageSize->{556, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->144812804]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->77961267],

Cell["Find the industry sector for Microsoft:", "ExampleText",
 CellID->336628928],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<MSFT\>\"", ",", "\"\<Sector\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->91794653],

Cell[BoxData["\<\"ComputerProgrammingDataProcessing\"\>"], "Output",
 ImageSize->{235, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->40942134]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[
  "Other Financial Instruments", "ExampleSubsubsection"],
  $Line = 0; Null]], "ExampleSubsubsection",
 CellID->313174033],

Cell[TextData[{
 "Indices have symbols that begin with ",
 Cell[BoxData["^"], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->151207433],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<^DJI\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->333743693],

Cell[BoxData["12110.41`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->261108333]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<^DJI\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->180640064],

Cell[BoxData["\<\"DOW JONES INDUSTRIAL AVERAGE I\"\>"], "Output",
 ImageSize->{214, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6871051]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->342023374],

Cell["\<\
Mutual funds are typically specified by 5-letter ticker symbols:\
\>", "ExampleText",
 CellID->384057746],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<FMAGX\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->65799697],

Cell[BoxData["89.19`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2099440]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<FMAGX\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->250026892],

Cell[BoxData["\<\"Fidelity Magellan\"\>"], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->75952996]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->67761741],

Cell["Preferred stocks:", "ExampleText",
 CellID->657420572],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<JPM-PK\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->212197307],

Cell[BoxData["24.11`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->529210167]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<JPM-PK\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->332990830],

Cell[BoxData["\<\"J.P. Morgan Chase Capital XI\"\>"], "Output",
 ImageSize->{200, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->246831888]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->228816305],

Cell["Find prices for different share classes:", "ExampleText",
 CellID->307577191],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<BRK-A\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->47970543],

Cell[BoxData["105310.`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->54388660]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<BRK-B\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->448851325],

Cell[BoxData["3532.`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->251571347]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<BRK-B\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->430589200],

Cell[BoxData["\<\"BERKSHIRE HATH HLD B\"\>"], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->408367039]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->212108301],

Cell[TextData[{
 "Exchange rates have symbols separated by ",
 Cell[BoxData["/"], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->246308691],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<EUR/USD\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190823870],

Cell[BoxData["1.3303`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->10911257]
}, Open  ]],

Cell["Exchange rates can also be requested using lists:", "ExampleText",
 CellID->942555933],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"{", 
   RowBox[{"\"\<EUR\>\"", ",", "\"\<USD\>\""}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2985750],

Cell[BoxData["1.3303`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->151882719]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->360520360],

Cell["The price of gold in U.S. dollars:", "ExampleText",
 CellID->188236723],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<XAU/USD\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->70810601],

Cell[BoxData["624.6`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->465624813]
}, Open  ]],

Cell["The price of silver:", "ExampleText",
 CellID->5065487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<XAG/USD\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->134129235],

Cell[BoxData["14.006`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->29776333]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Prices",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(10)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->72887098],

Cell["Find the current price for a stock:", "ExampleText",
 CellID->6745604],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<GE\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->324218402],

Cell[BoxData["35.82`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61347497]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Price\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->196401469],

Cell[BoxData["35.82`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->283557429]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->84142534],

Cell["Price with time stamp:", "ExampleText",
 CellID->17331403],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<LatestTrade\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->329558717],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2007", ",", "3", ",", "23", ",", "16", ",", "2", ",", "0.`"}], 
    "}"}], ",", "35.82`"}], "}"}]], "Output",
 ImageSize->{219, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->495829465]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->225106620],

Cell["Most recent closing price:", "ExampleText",
 CellID->38450493],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Close\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->49765797],

Cell[BoxData["35.81`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->245598005]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->276075381],

Cell["Low and high for the current or most recent trading day:", "ExampleText",
 CellID->124756331],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<GE\>\"", ",", "\"\<Low\>\""}], "]"}], ",", 
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<GE\>\"", ",", "\"\<High\>\""}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->185965031],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"35.6`", ",", "35.91`"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->284595998]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Range\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->326529512],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"35.6`", ",", "35.91`"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->33143547]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->392425290],

Cell["200-day average price:", "ExampleText",
 CellID->12690591],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Average200Day\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->85805936],

Cell[BoxData["35.7612`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->176543103]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->137422354],

Cell["Change from 52-week low:", "ExampleText",
 CellID->434204360],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<ChangeLow52Week\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->35467110],

Cell[BoxData["3.76`"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->127317523]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->144022931],

Cell["Fractional change from 52-week low:", "ExampleText",
 CellID->48678768],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<FractionalChangeLow52Week\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->500033543],

Cell[BoxData["0.1173`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->517476370]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13154962],

Cell["Daily cumulative fractional change since January 1, 2005:", \
"ExampleText",
 CellID->385569970],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "\"\<CumulativeFractionalChange\>\"", ",", 
    RowBox[{"{", 
     RowBox[{"2005", ",", "1", ",", "1"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->106671],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 129},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->120439119]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->260027542],

Cell["\<\
Value of the cumulative fractional change from January 1, 2005 to today:\
\>", "ExampleText",
 CellID->226668632],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Last", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "\"\<CumulativeFractionalChange\>\"", ",", 
    RowBox[{"{", 
     RowBox[{"2005", ",", "1", ",", "1"}], "}"}], ",", "\"\<Value\>\""}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->54403038],

Cell[BoxData["1.0430984274898083`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->72339279]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->18459002]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Dates & Time Series",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->47965353],

Cell["Get a current stock price:", "ExampleText",
 CellID->158686579],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<GE\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->47890169],

Cell[BoxData["34.36`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->108215363]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->180309490],

Cell["Get prices for a range of dates:", "ExampleText",
 CellID->278238793],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "15"}], "}"}]}], "}"}]}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->68140534],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "10"}], "}"}], ",", "34.12`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "11"}], "}"}], ",", "33.83`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "14"}], "}"}], ",", "34.21`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "15"}], "}"}], ",", "34.`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{321, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->538537496]
}, Open  ]],

Cell["Give the price values only:", "ExampleText",
 CellID->235460803],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "15"}], "}"}]}], "}"}], ",", 
   "\"\<Value\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->117134940],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"34.12`", ",", "33.83`", ",", "34.21`", ",", "34.`"}], 
  "}"}]], "Output",
 ImageSize->{178, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->217365398]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19845487],

Cell["Use the string forms of dates:", "ExampleText",
 CellID->32229768],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", 
   RowBox[{"{", 
    RowBox[{"\"\<March 10, 2005\>\"", ",", "\"\<March 15, 2005\>\""}], 
    "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->211214615],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "10"}], "}"}], ",", "34.12`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "11"}], "}"}], ",", "33.83`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "14"}], "}"}], ",", "34.21`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "15"}], "}"}], ",", "34.`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{321, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->4726197]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->490111333],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["All",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/All"]], "InlineFormula"],
 " gives data for all available dates:"
}], "ExampleText",
 CellID->102397045],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "All"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->566637411],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 133},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->360235877]
}, Open  ]],

Cell["Give all available data starting from a particular date:", "ExampleText",
 CellID->1034216],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", 
    RowBox[{"{", 
     RowBox[{"2000", ",", "1", ",", "1"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->182543731],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 134},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->201833203]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{"{", 
   StyleBox["y", "TI"], "}"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["y", "TI"], ",", "1", ",", "1"}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->143893669],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", 
    RowBox[{"{", "2000", "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->95991783],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 134},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->274423287]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->412064084],

Cell["\<\
Give the monthly highs for a stock price over a range of dates:\
\>", "ExampleText",
 CellID->1036457487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<High\>\"", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "2005", "}"}], ",", 
     RowBox[{"{", "2006", "}"}], ",", "\"\<Month\>\""}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->65159324],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "1", ",", "3"}], "}"}], ",", 
     "34.623301965125926`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "2", ",", "1"}], "}"}], ",", 
     "34.581889204545455`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "3", ",", "1"}], "}"}], ",", 
     "34.456705490848584`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "4", ",", "1"}], "}"}], ",", 
     "34.567790055248615`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "5", ",", "2"}], "}"}], ",", 
     "35.262143640350885`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "6", ",", "1"}], "}"}], ",", 
     "35.28689466089467`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "7", ",", "1"}], "}"}], ",", 
     "34.00655652173913`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "8", ",", "1"}], "}"}], ",", 
     "32.89981553109194`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "9", ",", "1"}], "}"}], ",", 
     "33.08054054054054`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "10", ",", "3"}], "}"}], ",", 
     "33.004423473901504`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "11", ",", "1"}], "}"}], ",", 
     "34.76309630459127`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2005", ",", "12", ",", "1"}], "}"}], ",", 
     "34.9358116975749`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{520, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->147203428]
}, Open  ]],

Cell["Give yearly highs:", "ExampleText",
 CellID->507504431],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<High\>\"", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1995", "}"}], ",", 
     RowBox[{"{", "2005", "}"}], ",", "\"\<Year\>\""}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->572331369],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1995", ",", "1", ",", "3"}], "}"}], ",", "9.60568438446709`"}],
     "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1996", ",", "1", ",", "2"}], "}"}], ",", 
     "14.261306474820143`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1997", ",", "1", ",", "2"}], "}"}], ",", 
     "20.945564548228738`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1998", ",", "1", ",", "2"}], "}"}], ",", 
     "28.96422050630437`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1999", ",", "1", ",", "4"}], "}"}], ",", 
     "44.989999999999995`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "3"}], "}"}], ",", "51.4855`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2001", ",", "1", ",", "2"}], "}"}], ",", "45.9888202247191`"}],
     "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2002", ",", "1", ",", "2"}], "}"}], ",", 
     "36.54301477832512`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2003", ",", "1", ",", "2"}], "}"}], ",", 
     "29.33334794863765`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2004", ",", "1", ",", "2"}], "}"}], ",", 
     "35.22525414660246`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{492, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->206942306]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->69506496],

Cell["Give a price for a particular day:", "ExampleText",
 CellID->402751654],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"2005", ",", "3", ",", "15"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25599404],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"2005", ",", "3", ",", "15"}], "}"}], ",", "34.`"}], "}"}], 
  "}"}]], "Output",
 ImageSize->{150, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->686459304]
}, Open  ]],

Cell["Give the value only:", "ExampleText",
 CellID->401392888],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"2005", ",", "3", ",", "15"}], "}"}], "}"}], ",", 
   "\"\<Value\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->193041198],

Cell[BoxData[
 RowBox[{"{", "34.`", "}"}]], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->34008531]
}, Open  ]],

Cell["Give the trading volume for a particular day:", "ExampleText",
 CellID->30386929],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Volume\>\"", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"2005", ",", "3", ",", "15"}], "}"}], "}"}], ",", 
   "\"\<Value\>\""}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->572532350],

Cell[BoxData[
 RowBox[{"{", "17383800", "}"}]], "Output",
 ImageSize->{79, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->242598719]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Plots & Charts",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->71471726],

Cell["Basic plot of a stock:", "ExampleText",
 CellID->204052994],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<AAPL\>\"", ",", 
    RowBox[{"{", "2000", "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->60522201],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 134},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->274667918]
}, Open  ]],

Cell["Join points and fill:", "ExampleText",
 CellID->99209185],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<AAPL\>\"", ",", 
     RowBox[{"{", "2000", "}"}]}], "]"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Bottom"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->456871292],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 134},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->12803459]
}, Open  ]],

Cell["Log plot:", "ExampleText",
 CellID->653599555],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"#1", ",", 
       RowBox[{"Log", "[", 
        RowBox[{"10", ",", "#2"}], "]"}]}], "}"}], "&"}], "@@@", 
    RowBox[{"FinancialData", "[", 
     RowBox[{"\"\<AAPL\>\"", ",", 
      RowBox[{"{", "2000", "}"}]}], "]"}]}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Bottom"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->186058442],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->125759440]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->151322591],

Cell["Plot two stock prices together:", "ExampleText",
 CellID->368987215],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<GS\>\"", ",", 
       RowBox[{"{", "2000", "}"}]}], "]"}], ",", 
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<MS\>\"", ",", 
       RowBox[{"{", "2000", "}"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Bottom"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->670556123],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 130},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->141555012]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->307643591],

Cell["Plot the channel between high and low daily prices:", "ExampleText",
 CellID->71892613],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<AAPL\>\"", ",", "\"\<Low\>\"", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2006", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2006", ",", "2"}], "}"}]}], "}"}]}], "]"}], ",", 
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<AAPL\>\"", ",", "\"\<High\>\"", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2006", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2006", ",", "2"}], "}"}]}], "}"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Filling", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"1", "\[Rule]", 
      RowBox[{"{", "2", "}"}]}], "}"}]}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->124995501],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXItzVNUZv+y9u5uERwQV0D7QVvtfKI8gNAUCISQIETEQKyiNAqlABBqq
8gwjBQrZBAgkISEkQEDeBhsSHsYiUYSKyihoFZzKOINjZ5g6k/7O+c69Z/fs
vXvvXbYdOzWzJHv3nnu+3/f+znfOMq6odNbTvysqnT2z6IHR84pemDV75vwH
Rj0/Dx/pvTRNO4h/ywdreN+Dd/zfwZ4e/I3//RB+BzBAx9+gJt5o9GOIewHx
N9NlrO5xLHtrmA9p5k8/lyd08ZeNCfJbv2K/w4w7PjD6N8eh6XTN3vJrQ16L
6aNG9PMxmxE3mzrCz2zRSCyxRN3v62OuoDKXrRxIctGfapZQJUsOXOpCu+7P
h7xJyds8DvKy5OM6S9i7ZELxsg5EWaSFIRRNwJ7Dvh7nC3qTlYeZVGmx+3d5
fDba42PRGFI6QZVru1k8y8Z1tjjt421/T0+GY0cJvd/lnWpCfdh7Um8/vDvL
QmAdYDd7vF4SzBvtYc5W/3AP+TV72ozBZhSP/txw+Dw6LrC/fXrs4rVTDmKz
NGBUL+sqlsegHeYE8eDH+//d+3S9s74bH62t6Iwbr16zEuVHPf+P3E/Ddff5
L/Da33qpZtu5Mdnb8cljWVu8a5lFgF/wu+fOnRN3H1Kuf6lc88gXhE0NfSSy
aOFRIq8FAEDTf196+PmS12fPatUCuKul5Y6ve7p4L15LF79RsaZj/txDWSOq
jxy+3FvQ1uKi4oNJoZk0sR70hj0aIVJaAMQ0fW/Lxdf3/w3ktLQPL/9D03Nz
aq5d+4Y9wOj3Sgl9nu+M69dvzXiquWh6M6SQyWmVzmuBiFJLi+dH/eOPvoaM
z5y+pvXnlDauP4Lr1FLKNClBnozSAE6pdV/HmOyab7+9nUJKFGvImnLG1Xaf
//Juk9aiBUeZ7lJMS4dNglbW8GrtHpNSfW33svI2hRLVHYSMxc4hPqn14Zzd
vv39lMkN059klsHpGaAHLoECt9J7YvxAVBmT8xsEwqRo6pHKrj+9dooRu9ek
iL9QJXw1I1aagmLXW58VPdVSNH23b4q87tEvXbxRNL2FERsYRXHTxrMb1p9R
KMoKDD+Qws2b3/3cJ80Mfg3xzZzRcrL9E21QFE1YK8SXiOaK5e1QanI0oURE
FkaO0wwSTbxAE6FFS09Al5D5pZveIzwR8r34/lfa4Fi6f1x2onn3BYWutU7Q
LbVrP/NJN82UMYJ52/Er2n2xdBFXEXT4qCj7DUfzi1HCr/zSpvUged6T03Zr
9zPaIYs2xAC7QRhKi5V2mkLdGvfTpOgXTGJeOHJEtUofr9L5h060XVHoG9H0
Azy2IBchObKbP0kKQ1Vl19gx21lqpQwGtwYmluEy4zBhFEzbHRO0Cb/hN6Pk
vGZ1h3ZPUhihYeiZTU0YJ1J8HVpFGMPRGCtWd4hoGk6AU+fzPFG4C2bPSfAR
EyfU8RxfBY3e7RMprVTx4LgxO5gXpfOIUfXSnqK8mjUzt5xuPNUvDivRGzGs
SsFqrao4SopxbHrucSGYxdQpu2bO2LN6ZTs8BNFtQFJYkfmQ/yyphj54/8aO
CeXf3bz1/e1/vV7y57cPv9uX34xGjHoH3v58yQGYQrAnxjPj5Ru2khA+QQyB
STPJWDYCv0dVQ/WNb/SoAwHGQh9+M3Kss6KZhtz+9p87C1e+d/qK1sfMvwRf
M/g1pX3TPp1shI81Cqfugo6mT2tGHAzypxFlgRyGI6ol78iDdjaiA+rXH/3d
0q9+6/pNfHTp/OdabwV9UEGvu6HXQci0MM7NX9/+fMrkRnKQfkmhR7BBcJC+
CMM+VrbNlKWm37h4tal4HWxJy1DQhzgLQP+HJccJAXuA8Ed3X8x+AqcnqgeK
ICwqkU9A9VQKEwqDKi5U53COvknxBU9CxpF8Sa3Q/Stt5/fNr2a30hW+SLKj
srYCAZZGlk7iOnzq6iq2FoOO1AhMXkQ+t3ZNB/wHY6EAeJSoFrxzSdpBsIBJ
2GmPIm13fduR5c3yvsrl7l0X5jy3n8yPuPS0jgxZNWDJnANLx66Ba4YpkvMI
mGXaZxC+BcbBJVTLVgFpSXGJWANvlVw0z97w2Vsf9Fg1BwIFwoXFVRSXbLzu
izfd1DQMf9f0CiGnyoW7n8rdtjh7FRyBYwjC81DXwU4Ry1nUCyXFGQIPhC+R
v3P8YsuMCnaL7OFwaeRsS5flbQpn0bsqURzGZnDNqRdzbtvRY2v2WlE0CEIg
h1uwINgR+XsQcQ0JCxpE0mYx0vDJJ8kYEoJpSD7JJ8rn7xerAaQoCBvc2/Ip
NZi4i6v2TwPm3JT+yP9CMFbIGDkFdxESEBh4XjRQkqCAQNyAWQeS4hJRDfWU
5DJ37A6KIWKVhUyAWMryWNCFS6lX2XdVueM0w99c+wocwCwxA37jPT4ReEiR
bH7OYQgB8dnZrSjlrIj9sCe+8ATSu+Rrw7Pbi6fWs5U2xWZIEXyxDGck5Etd
jXOOgpdaTx9czGxjT0nlpx0XBJ+kGTYHrwkNABg1ciuVc7xzZIB+xeqTxMcQ
BfcAUx+iTuW5Mc26jx9EzsVlxygwE1fAXzd+KaxF9DIQYoCIZT7dF1fkbeVP
bsHkC2bUi8ggdZFpZrzHC2pXvvomdZ4M5NGs4VvIXOy5mZBTi9usScY7ZUEM
hLHSC2bHl5paHxgvdfjWrzvNALx4BAFqWmETIhXJD8sT4vfExkNn1reyqanu
uXy468CCGnYr4Movs0fiqyCvXth4SNHbXcosNCsyzrChVcx27LmcP/fQ5AKW
Lmg8iz8wW3rxhiGTVh+YMXUJs4ZVk00gSGFhb5W0GEpc1uW/AodgU9M+U1fl
wTde20+QgI1HOXOf20mXkClkzDBTVkU46dp/TupS8qibVRrMiw1V1+7EI3SB
9GH5JbOFwqn1XH382orEGl+1GG3ltZ21JwmwrtQURkJbJEuGV8GSnfEWTW+5
dPGGHV6q5WWvRn2eKlpyXAtvf/gQ8ENENBAuBNPPeayKla6PViJ7I+/A+la+
dFCtG1X8ZJmLctcLbvvZ4qe1tTN+2YNR8VNNi4jckP8yG3ovJ4rcsGA8Izkn
N7J19CK4bvNrR+fMbm2qPo2ggLwJvxmXFVHqQ1vsRt647VQRQRx9bdFTd8kZ
vawR7NEfWtZ4YXc7GzqQU8znne2Rw6uhBm6vhmW/ibFmYNzTT+zcMK8RBh55
ZC5cByUydAXoCI5qbUXoqyJdQoMqetKdrFDVeEB1t7VK1QaZvjajqIH5Wp+4
iMP5y4A7o3iHFoAS2qkZs5gsCjYGRaqxy1pfYIHpjFKune1Rtm870b68kQ0d
rDyf4Rg77me/0sEZ2QysCIxWjyhdMHEj9LOk7LjIltTlc8b2Utmxpsb3nLHJ
lY2KzbkPPoTbBQxk0vgdZJ5UkeC9GKF24CiSYKWSPXobW+v2t61HZf16n2cs
D5h6n5a3dWVxjaiNUC6iaHRGgvFIAmyoiiS6wlxSsof04AkJ7Q60txxFLWrV
xTq1Q52RwPzhBM5IcnneZhnSL5KuzjMwcIlEdgrskcCMYEzOSDaW1BVPqWO+
5RcJRjYVruhuvyyQIJwiqNohoS4EVgqopdnQAQqSkOL1apczfkfF2m0U6xJC
QL1LZwQy5tsjkNWPisDZoy1Z0LJaIKHGtjMSGb/tkSCAQRB2SDxohVJ7XCRR
kdDzMhbfrSAhn5PdA7X/7QEJeAAnYiZYO7UJ42ailRBV5gX5DXJHLQaJjGk+
kRg0Em6DjEVSoY0WZyzUrxkxrDoOCz2PlWPrrHV2UvWGhapFmgshC4HLGcvp
U1ezRmyJbH7LAYustBJjsfxGYICpw+BpDrnDYI8BwQ4hjw1VMehKjFfzlYvn
CCy0zjDMKB2p7HLGIr3YHousfVQs3nRDqxXCQltPzljkKphjiemjIzKuWHgA
ZoQV6ZCkkCAXozYgJAgXCBpO1s87zsjClh9nvP/2Vaql5+RVIXWhdEOwzhkZ
sV9NSjzqqVy7XhHHGLSehuZZl4akj0UMljLOOGVVxev2DHgSCilEzqbidVS5
URU9cnjVjYtXqeci11oJcHs4k6deK3zQkkp4lYxWdhGYV+gwRKvKyYS84dEo
OVEew3wgbwQI1IdIC1+c/5g6S7KOVXss/vlglwp+WD1sX+CX+172+OV6k49P
X7e2E1J/dtwmUgICFOpT1NTwBtcObbrAE2s9DmdMneWPmhhVu8APhXd2fGqH
n7qgsprg2cHIeayaVjgQtdvuQEqtBvQgMoGaNtKdUctuIO08wWpW5awqmXMA
RuG2V5MC1KHoblfd+KUgr+4/2uOWOyeGEmMzvaP2GlliUFK3m1BiyYOFjx1K
Ot8gVyKEEks/BD3WvXPbxXOJxzGIqFtEiOQuuD0imDBEZ5erU4mI9q0IUff5
L8dk1zgjgpqhbLvM7bZH61BNxCDBOrwu/xXilQ55sSHqyoKQyJWJrsjGDYmH
alxckxcgiVIzXPQbZWxUkVGkkzsThAyBHfGQ3XLbhfd+yoyQIUcgAUpkMurZ
I5M7s7pSE6camdw/I2SyI2WLLGx1JPCidfYLBdWuZ0H8olq7YK/o+hEq2WlS
UVFXtnbHO/x86TExnvYi2WmcVCObNKFWzEyUZHfJHhntZsvxOzd1Th29idUG
/pDZnb0lRGdPfTp5+Pqtm88KCifbP5lW2OSMCOuQKY83Wl12HTUZihd2yw2R
d3+kb9VYvesP+fEqlFpqrKJ9A7k21ZVVceq0R7tZ6j6H7NarPStCJqMojZfd
+XtTiyxM5UW0Xy0uOx6nQcqjyH8IqijrhLzAEhhLLSr67o+122Edb7JHJONV
wOzx1GSXsYw8MLWIwtapBtrrmf3MPnaiUdUeoZKrNkIlO7n+UNn5HtW4lAel
5co1/iDb8XJ9TePlaQY3RN59j3hvyH8ZZZHkHfUC1pXs1Kk9MtlppvEoGVE4
2nGSvAYJmcw4utm5AjK4YZwMyGfV6ICiDKXZfwIZLUxk1Y4obZ2wUiMRYVOr
LJmnU40NvoT4wHakqUs2Nnu76CyryCgrqDFC1n/qTkHyyNTdbcPMiHkT6xe8
eJiQhaPrRur9PT7ZXL3K3JNqVDLSh5TOF6+gwp2dXfBUZEFk72GPVokT9NE7
rKtefTMurviPE4SGGqRs8rASJzg3vREXIJpl5W3Q2a9HbcHtijUdEGTjzndp
Q1sgmzCmRmjdDZn3eEFVgaxQSV7I1+JkJK/7MvAO8oMjIKgCAlChUqyv7Yb3
tu7roN2odcVVv53WyGobf3JzRyfPK4XN7pM4ccqtPR3vTrRdiVR2obAGOhT+
0DeKRnYWXT1vQ/zJEx9qRrtTrPIUBO8Ypa149S+AlJ+3EwslpNHS+YdgX1hl
256p1K2zrz5wqd0Y68SlK1byx7LSQ+LspZnmq5Eu1P12wtdUvO6L8x/7xJew
O2TbwYhDqp6stI56xO2MB82egNjXdztL7wen1Y1T8Rl0HnJx9iokzvtsEcGA
YcZeENn32dy+VaniotU3NQsZgMG2qHZMKL91/eadyMk8QeNmcYQHHoeKiZ2D
U/EYyhrc7XsYifAkOhmo4qLOlzwnM8gWl6yxfeCy+T843NBQ15Za7YzwQAWN
rtTXbt+W8R69iLJSA6nfIYlDImtqf0jscmUffi3rCTozTXvoWUMjcCCsv2Ag
zHrU8/S6Ep3c0HjPjxxVphXVP+Q7PjCH5b9ZUZRXs7msBV6NRSGsFr4UeWRu
9YhSLQAgvHsKzIjoWgCo1dMpASVOuX3Xy7sm6dtkrxSzCvG58ZubClfAdM0T
QSG8gYzwQmgAdGSbALAHgFwLb81ZAuRwBOAuf4adJVpY3OD6LT8F2Q/qu+B0
XZBXj49GsgovmfnZN9UgMMQICCw3e6vGvzjtmsl+kN8Dv4P7P2wdIwPjI1Z9
Jq3jHvNHzvX/q+NG/s7p//q4899ar38DpJWkXQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 134},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->85983140]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<AAPL\>\"", ",", "\"\<Low\>\"", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2006", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2006", ",", "2"}], "}"}]}], "}"}]}], "]"}], ",", 
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<AAPL\>\"", ",", "\"\<High\>\"", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2006", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2006", ",", "2"}], "}"}]}], "}"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Filling", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"1", "\[Rule]", 
      RowBox[{"{", "2", "}"}]}], "}"}]}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->5101232],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 134},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->135093639]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<AAPL\>\"", ",", "\"\<Low\>\"", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2006", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2006", ",", "2"}], "}"}]}], "}"}]}], "]"}], ",", 
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<AAPL\>\"", ",", "\"\<High\>\"", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2006", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2006", ",", "2"}], "}"}]}], "}"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Filling", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"1", "\[Rule]", 
      RowBox[{"{", "2", "}"}]}], "}"}]}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}], ",", 
   RowBox[{"FillingStyle", "\[Rule]", "Red"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->24200996],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 134},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->59400019]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->985935685]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Fundamentals",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->85828566],

Cell["Find the market cap for GE:", "ExampleText",
 CellID->451483001],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<MarketCap\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->144800147],

Cell[BoxData["3.683`*^11"], "Output",
 ImageSize->{75, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17500742]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->70698397],

Cell["Number of shares available for trade in the open market:", "ExampleText",
 CellID->9502744],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<FloatShares\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->230900282],

Cell[BoxData["10276960000"], "Output",
 ImageSize->{88, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63256558]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->138311976],

Cell["Book value per share:", "ExampleText",
 CellID->93081590],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<BookValuePerShare\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->453802474],

Cell[BoxData["10.928`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->348559992]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->281422811],

Cell["Earnings per share for the most recent four quarters:", "ExampleText",
 CellID->140895079],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<EarningsPerShare\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->116609142],

Cell[BoxData["2.004`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->499445470]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->177329415],

Cell["Price-to-earnings ratio based on trailing earnings:", "ExampleText",
 CellID->92044503],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<PERatio\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->94552650],

Cell[BoxData["17.87`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->145313565]
}, Open  ]],

Cell["Estimate of forward PE ratio:", "ExampleText",
 CellID->289858732],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<ForwardPERatio\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->328471706],

Cell[BoxData["14.38`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->455376]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->76926374],

Cell["Find the most recent dividend for GE:", "ExampleText",
 CellID->30122779],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Dividend\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->574160085],

Cell[BoxData["0.28`"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->70885543]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->143408027],

Cell["Plot all available dividends for GE:", "ExampleText",
 CellID->70384877],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "\"\<Dividend\>\"", ",", "All"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->64150158],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW81v3EQUn9jedVqyyW4L200QUI5w5Qxts9ACUugB2nKggrRCag8IVIJA
Qkj8AXAF0USFlLQSqsSpKnDgxAWkIK6c+UfMm/fs2fGb8dd67N2Frja2387H
+83vzXvzxna2tneuvfve9s71q9snz93Y/uDa9asfnjz7/g34yV8SQtyHv89H
Aq4juMK/+1EEZ/P4DBx9qCB8FPETyEv5Y4Qf4cHBA7GfUdeLf5LngaVOUpbU
XYNzAGePtU3OQ2z6rAWF6Cg57h7VBJNahNJol6rhaTUEk+Xlmq2H0j3Ky56G
1a6nSJaXxyc8BPq4feu4eetVazsRKkiZ2BQHRmsv6aXDeiHWS7NcerRJPcPu
pg3yEaDdrO0GtZGvVajf/shLj2B+OajQz2rtHupzsSIPXdPJpuNERmgeJRNf
TKKyLYrqETmvvKg9ReOs9YPHee5vSS8yYNyC8xJVT/EblOAL+2N8dVk5X7e4
zPHNuv6840tsV9dmfpatlpUMnzsHf516/psvv/gtni3zxMS8Wyph2ZV3gS3E
EKwh/Itv3IGzvFxFGewT1z+G9Q8PD2P5SSY/xeQRHJYcYrx04a54DjG9ffle
XN6dESbpJSL84/d/rrxz78LrB/A9v7X/4nhv69XvhAeS8LPljpSPXn7rh88+
/eXls7fgJ2z+rd4JtukoEaTNU7uiz0bvVxz9iSjl31ok70bpNYD74wbraRll
NX7hszEq+fQLNyXoHptL65Xs9FgBGiwPCA3pHDCmjjvVN2Kj1y38yrk9HD0f
ca8yAmmHx22tQiXD5/b+nzSJJihizh9hCMJKs6OIA946tHICszaOJkcYGj53
61mEt8aMbNnqn+Mzu8saFmgjnp7eFpqMcyxQOmHoIer55OMfG2Ec87SOPkYZ
I3zUOT5zsxFeN9JjPP/afsDsylmajssnUIaIH8+eDtMyymWyKEdNVs+iKGdF
JUf/5qXbRHeXxZlh7ugt6435S859iCJ8GP8DkslGtI7AlCce43zrxOxRajJZ
SLJ68S6ELsPWA+dopVgVZSdB+dJ4DwHiAYQMlMOoIEMzdpxl1t0ilJgPdfWY
AHnGTw/+5utfNQ65hTmnVVEeRUI/2nmQmpX9fIyWbCVg56KoYsexeXriHWBP
A0e9KG3XqVZtMk/B2OvqpLuTtN+iL0QsIyOZLlJbdWmR24MD1+SWUex9kv2N
N3f5msQzH7fs9i0j1iNDC9otHn+kEevycreWfDSyxQW+x3PLH+nkMaBY5zTs
kS7u+1xX+VxGvytYd+Xg5XkrBCLOv/NeEw1lKCpikU8FyqOh4kZtfNNkAYSL
+XqgfB0qjqLsdb8l3izRwJsr6/LyuUZTcJdk/qIo1We+o+1C3cZPqs9Xv8n+
003kRC18Xn/91c/rTpnj5W6ZKu59/uYSybQiy5xx3SkjKKfW3mR36loHrLl0
Vyxe39M6itZY2vAkn1peHxx8/yvMYYrKwyj7eQt/fmLcx6jJCy8vjMELhGWa
lZ2X5+3f2ZoUzSgDM94U4NzUQSCfCPF91CBqekeul8vnPQHbSyXPrh1GZPN+
mU/eKQuPRe2uAP+39aVdRihO8cxoMr9XG2GMtGbfH+g1yihqt+wI+o0yTfGJ
7+R7jfBLuvgOfiUqs4tGWTS3iyZsfMef5iEd2RFB7puEyqoOcdLdomzPCHMQ
t79DI7T5d9yy0IYalixmXeQPiNHid2GUnVeUwdZWhjHNGxoNYFq43Xf+eFz3
vhLl+21gaHfBGGnN9r/0e2SNjNniWd1GmX6YTc26d/d8F2UF5r6/reir61JA
Goy6+BTJyNf8HIRt7cPp+RbL7qJya6iNqWaY4xleHnMGwpbnVrnn+816m+1+
wSLHpkWLfIvFCL690lVZDqU3KW2+JSLN+h3t/9I75/JenMryYh82bS0eWqGZ
+vE+jHLe+IluygKT/7TJ+r+d+kex9C+KMV8u\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 128},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->261546325]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Computations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->871755336],

Cell["\<\
Find the daily average trading volume for GE in the first quarter of 2000:\
\>", "ExampleText",
 CellID->43443820],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "\"\<Close\>\"", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"2000", ",", "1", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2000", ",", "4", ",", "1"}], "}"}]}], "}"}], ",", 
    "\"\<Value\>\""}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->603077345],

Cell[BoxData["39.73428571428571`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->709550607]
}, Open  ]],

Cell["\<\
Find the standard deviation in the closing price for GE in the first quarter \
of 2000:\
\>", "ExampleText",
 CellID->510284768],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"StandardDeviation", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "\"\<Close\>\"", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"2000", ",", "1", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2000", ",", "4", ",", "1"}], "}"}]}], "}"}], ",", 
    "\"\<Value\>\""}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->81549659],

Cell[BoxData["2.7755212903488466`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->41555066]
}, Open  ]],

Cell["Find the skewness of the price distribution:", "ExampleText",
 CellID->299693805],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", "\"\<Close\>\"", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"2000", ",", "1", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2000", ",", "4", ",", "1"}], "}"}]}], "}"}], ",", 
    "\"\<Value\>\""}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->33503638],

Cell[BoxData["0.5303266762771258`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->932703993]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->220175310],

Cell["\<\
Find the correlation between cumulative returns for a stock and the S&P 500 \
index:\
\>", "ExampleText",
 CellID->57534863],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Correlation", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<GE\>\"", ",", "\"\<CumulativeReturn\>\"", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "2006", "}"}], ",", 
       RowBox[{"{", "2007", "}"}]}], "}"}], ",", "\"\<Value\>\""}], "]"}], 
   ",", 
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<SP500\>\"", ",", "\"\<CumulativeReturn\>\"", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "2006", "}"}], ",", 
       RowBox[{"{", "2007", "}"}]}], "}"}], ",", "\"\<Value\>\""}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->201574402],

Cell[BoxData["0.872571947193847`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->111202725]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->510092804],

Cell["\<\
Compute the covariance between the time series of two indices:\
\>", "ExampleText",
 CellID->460617968],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Covariance", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<^GSPC\>\"", ",", "\"\<Price\>\"", ",", 
     RowBox[{"{", "2004", "}"}], ",", "\"\<Value\>\""}], "]"}], ",", 
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<^DJI\>\"", ",", "\"\<Price\>\"", ",", 
     RowBox[{"{", "2004", "}"}], ",", "\"\<Value\>\""}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->294789194],

Cell[BoxData["65727.2811718179`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->531938029]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Groups & Indices",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->454059883],

Cell["Current price of the S&P 500 index:", "ExampleText",
 CellID->53133309],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<^GSPC\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10775634],

Cell[BoxData["1436.11`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->67411612]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<SP500\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->279976511],

Cell[BoxData["1436.11`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->15625594]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->831960422],

Cell["First few members of the S&P 500:", "ExampleText",
 CellID->142119902],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<SP500\>\"", ",", "\"\<Members\>\""}], "]"}], ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->54000930],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"A\"\>", ",", "\<\"AA\"\>", ",", "\<\"AAPL\"\>", 
   ",", "\<\"ABC\"\>", ",", "\<\"ABI\"\>", ",", "\<\"ABK\"\>", 
   ",", "\<\"ABT\"\>", ",", "\<\"ACE\"\>", ",", "\<\"ACS\"\>", 
   ",", "\<\"ADBE\"\>"}], "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->149416337]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->63661671],

Cell["First few members of the \"drugs\" industry sector:", "ExampleText",
 CellID->13041951],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<Drugs\>\"", ",", "\"\<Members\>\""}], "]"}], ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->560901108],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"NASDAQ:ABAX\"\>", ",", "\<\"NASDAQ:ACAD\"\>", 
   ",", "\<\"PK:ACAMY\"\>", ",", "\<\"AMEX:AKC\"\>", ",", "\<\"OB:ACCP\"\>", 
   ",", "\<\"NASDAQ:ACHN\"\>", ",", "\<\"OB:AGNM\"\>", ",", "\<\"OB:AGLV\"\>",
    ",", "\<\"OB:ACUR\"\>", ",", "\<\"TO:AHX\"\>"}], "}"}]], "Output",
 ImageSize->{389, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->101961137]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->446995658],

Cell["First few members of the Frankfurt exchange:", "ExampleText",
 CellID->33817476],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<Frankfurt\>\"", ",", "\"\<Members\>\""}], "]"}], ",", "10"}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6430592],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"F:05H\"\>", ",", "\<\"F:1001383\"\>", ",", "\<\"F:1001397\"\>",
    ",", "\<\"F:1001401\"\>", ",", "\<\"F:1001405\"\>", 
   ",", "\<\"F:1001407\"\>", ",", "\<\"F:1001412\"\>", 
   ",", "\<\"F:1001414\"\>", ",", "\<\"F:1001416\"\>", 
   ",", "\<\"F:1001417\"\>"}], "}"}]], "Output",
 ImageSize->{378, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->769824777]
}, Open  ]],

Cell["Total number of available members of the Frankfurt exchange:", \
"ExampleText",
 CellID->402144433],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<Frankfurt\>\"", ",", "\"\<Members\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->241470852],

Cell[BoxData["13462"], "Output",
 ImageSize->{42, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->268388356]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Stock Screens",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->374873301],

Cell["\<\
Find members of the S&P 100 with market caps above 400 billion:\
\>", "ExampleText",
 CellID->122635831],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<SP100\>\"", ",", "\"\<Members\>\""}], "]"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"#", ",", "\"\<MarketCap\>\""}], "]"}], ">", 
     RowBox[{"200", " ", 
      SuperscriptBox["10", "9"]}]}], "&"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->49952296],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"BAC\"\>", ",", "\<\"C\"\>", ",", "\<\"GE\"\>", 
   ",", "\<\"MSFT\"\>", ",", "\<\"PG\"\>", ",", "\<\"T\"\>", 
   ",", "\<\"XOM\"\>"}], "}"}]], "Output",
 ImageSize->{198, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->271633376]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"#", ",", "\"\<Name\>\""}], "]"}], "&"}], "/@", "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->138084224],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Bank of America Corporation\"\>", 
   ",", "\<\"Citigroup Inc.\"\>", ",", "\<\"General Electric Company\"\>", 
   ",", "\<\"Microsoft Corporation\"\>", ",", "\<\"Procter & Gamble Co.\"\>", 
   ",", "\<\"AT&T Inc.\"\>", ",", "\<\"Exxon Mobil Corporation\"\>"}], 
  "}"}]], "Output",
 ImageSize->{563, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->71661753]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3656482],

Cell["Find members of the S&P 100 with low 50-day volatilities:", \
"ExampleText",
 CellID->110245438],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<SP100\>\"", ",", "\"\<Members\>\""}], "]"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"#", ",", "\"\<Volatility50Day\>\""}], "]"}], "<", "0.1"}], 
    "&"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26927731],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"HET\"\>", ",", "\<\"JNJ\"\>"}], "}"}]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->344368164]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"#", ",", "\"\<Name\>\""}], "]"}], "&"}], "/@", "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->6883866],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Harrahs Entertainment, Inc.\"\>", 
   ",", "\<\"Johnson & Johnson\"\>"}], "}"}]], "Output",
 ImageSize->{338, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->74253831]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->98966124],

Cell["Do a typical stock screen based on PEG and PE ratios:", "ExampleText",
 CellID->12442663],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<SP100\>\"", ",", "\"\<Members\>\""}], "]"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"FinancialData", "[", 
       RowBox[{"#", ",", "\"\<PEGRatio\>\""}], "]"}], "<", "1.5"}], "&&", 
     RowBox[{
      RowBox[{"FinancialData", "[", 
       RowBox[{"#", ",", "\"\<PERatio\>\""}], "]"}], "<", "12"}]}], "&"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->251992734],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"ALL\"\>", ",", "\<\"BAC\"\>", ",", "\<\"BHI\"\>", 
   ",", "\<\"COF\"\>", ",", "\<\"COP\"\>", ",", "\<\"DOW\"\>", 
   ",", "\<\"GS\"\>", ",", "\<\"HIG\"\>", ",", "\<\"LEH\"\>", 
   ",", "\<\"MER\"\>", ",", "\<\"MS\"\>", ",", "\<\"TXN\"\>"}], 
  "}"}]], "Output",
 ImageSize->{380, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->87597851]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"#", ",", "\"\<Name\>\""}], "]"}], "&"}], "/@", "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->24965078],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Allstate Corporation\"\>", 
   ",", "\<\"Bank of America Corporation\"\>", 
   ",", "\<\"Baker Hughes Inc.\"\>", 
   ",", "\<\"Capital One Financial Corp.\"\>", ",", "\<\"ConocoPhillips\"\>", 
   ",", "\<\"Dow Chemical Co.\"\>", ",", "\<\"Goldman Sachs Group Inc.\"\>", 
   ",", "\<\"Hartford Financial Services Group Inc.\"\>", 
   ",", "\<\"Lehman Brothers Holdings Inc.\"\>", 
   ",", "\<\"Merrill Lynch & Co., Inc.\"\>", ",", "\<\"MORGAN STANLEY\"\>", 
   ",", "\<\"Texas Instruments Inc.\"\>"}], "}"}]], "Output",
 ImageSize->{499, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->451336175]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->253526391],

Cell["\<\
Find NASDAQ stocks with symbols beginning with AA, and market caps below a \
billion:\
\>", "ExampleText",
 CellID->92272593],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"FinancialData", "[", 
    RowBox[{"\"\<NASDAQ:AA*\>\"", ",", "\"\<Lookup\>\""}], "]"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"#", ",", "\"\<MarketCap\>\""}], "]"}], "<", "10*^9"}], "&"}]}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->148009114],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"NASDAQCM:AATK\"\>", ",", "\<\"NASDAQGM:AAME\"\>", 
   ",", "\<\"NASDAQGM:AANB\"\>", ",", "\<\"NASDAQGS:AACC\"\>", 
   ",", "\<\"NASDAQGS:AAON\"\>", ",", "\<\"NASDAQGS:AATI\"\>", 
   ",", "\<\"NASDAQGS:AAWW\"\>"}], "}"}]], "Output",
 ImageSize->{416, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->194350598]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Currencies & Metals",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->10369721],

Cell["Find a list of available currencies:", "ExampleText",
 CellID->12647730],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<Currencies\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->590075703],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AED\"\>", ",", "\<\"ALL\"\>", ",", "\<\"ANG\"\>", 
   ",", "\<\"ARS\"\>", ",", "\<\"AUD\"\>", ",", "\<\"AWG\"\>", 
   ",", "\<\"BBD\"\>", ",", "\<\"BDT\"\>", ",", "\<\"BGN\"\>", 
   ",", "\<\"BHD\"\>", ",", "\<\"BIF\"\>", ",", "\<\"BMD\"\>", 
   ",", "\<\"BND\"\>", ",", "\<\"BOB\"\>", ",", "\<\"BRL\"\>", 
   ",", "\<\"BSD\"\>", ",", "\<\"BTN\"\>", ",", "\<\"BWP\"\>", 
   ",", "\<\"BYR\"\>", ",", "\<\"BZD\"\>", ",", "\<\"CAD\"\>", 
   ",", "\<\"CHF\"\>", ",", "\<\"CLP\"\>", ",", "\<\"CNY\"\>", 
   ",", "\<\"COP\"\>", ",", "\<\"CRC\"\>", ",", "\<\"CUP\"\>", 
   ",", "\<\"CVE\"\>", ",", "\<\"CYP\"\>", ",", "\<\"CZK\"\>", 
   ",", "\<\"DJF\"\>", ",", "\<\"DKK\"\>", ",", "\<\"DOP\"\>", 
   ",", "\<\"DZD\"\>", ",", "\<\"ECS\"\>", ",", "\<\"EEK\"\>", 
   ",", "\<\"EGP\"\>", ",", "\<\"ERN\"\>", ",", "\<\"ETB\"\>", 
   ",", "\<\"EUR\"\>", ",", "\<\"FJD\"\>", ",", "\<\"FKP\"\>", 
   ",", "\<\"GBP\"\>", ",", "\<\"GHC\"\>", ",", "\<\"GIP\"\>", 
   ",", "\<\"GMD\"\>", ",", "\<\"GNF\"\>", ",", "\<\"GTQ\"\>", 
   ",", "\<\"GYD\"\>", ",", "\<\"HKD\"\>", ",", "\<\"HNL\"\>", 
   ",", "\<\"HRK\"\>", ",", "\<\"HTG\"\>", ",", "\<\"HUF\"\>", 
   ",", "\<\"IDR\"\>", ",", "\<\"ILS\"\>", ",", "\<\"INR\"\>", 
   ",", "\<\"IQD\"\>", ",", "\<\"IRR\"\>", ",", "\<\"ISK\"\>", 
   ",", "\<\"JMD\"\>", ",", "\<\"JOD\"\>", ",", "\<\"JPY\"\>", 
   ",", "\<\"KES\"\>", ",", "\<\"KHR\"\>", ",", "\<\"KMF\"\>", 
   ",", "\<\"KPW\"\>", ",", "\<\"KRW\"\>", ",", "\<\"KWD\"\>", 
   ",", "\<\"KYD\"\>", ",", "\<\"KZT\"\>", ",", "\<\"LAK\"\>", 
   ",", "\<\"LBP\"\>", ",", "\<\"LKR\"\>", ",", "\<\"LRD\"\>", 
   ",", "\<\"LSL\"\>", ",", "\<\"LTL\"\>", ",", "\<\"LVL\"\>", 
   ",", "\<\"LYD\"\>", ",", "\<\"MAD\"\>", ",", "\<\"MDL\"\>", 
   ",", "\<\"MGF\"\>", ",", "\<\"MKD\"\>", ",", "\<\"MMK\"\>", 
   ",", "\<\"MNT\"\>", ",", "\<\"MOP\"\>", ",", "\<\"MRO\"\>", 
   ",", "\<\"MTL\"\>", ",", "\<\"MUR\"\>", ",", "\<\"MVR\"\>", 
   ",", "\<\"MWK\"\>", ",", "\<\"MXN\"\>", ",", "\<\"MYR\"\>", 
   ",", "\<\"MZM\"\>", ",", "\<\"NAD\"\>", ",", "\<\"NGN\"\>", 
   ",", "\<\"NIO\"\>", ",", "\<\"NOK\"\>", ",", "\<\"NPR\"\>", 
   ",", "\<\"NZD\"\>", ",", "\<\"OMR\"\>", ",", "\<\"PAB\"\>", 
   ",", "\<\"PEN\"\>", ",", "\<\"PGK\"\>", ",", "\<\"PHP\"\>", 
   ",", "\<\"PKR\"\>", ",", "\<\"PLN\"\>", ",", "\<\"PYG\"\>", 
   ",", "\<\"QAR\"\>", ",", "\<\"ROL\"\>", ",", "\<\"RON\"\>", 
   ",", "\<\"RUB\"\>", ",", "\<\"RWF\"\>", ",", "\<\"SAR\"\>", 
   ",", "\<\"SBD\"\>", ",", "\<\"SCR\"\>", ",", "\<\"SDD\"\>", 
   ",", "\<\"SEK\"\>", ",", "\<\"SGD\"\>", ",", "\<\"SHP\"\>", 
   ",", "\<\"SIT\"\>", ",", "\<\"SKK\"\>", ",", "\<\"SLL\"\>", 
   ",", "\<\"SOS\"\>", ",", "\<\"SRG\"\>", ",", "\<\"STD\"\>", 
   ",", "\<\"SVC\"\>", ",", "\<\"SYP\"\>", ",", "\<\"SZL\"\>", 
   ",", "\<\"THB\"\>", ",", "\<\"TND\"\>", ",", "\<\"TOP\"\>", 
   ",", "\<\"TRY\"\>", ",", "\<\"TTD\"\>", ",", "\<\"TWD\"\>", 
   ",", "\<\"TZS\"\>", ",", "\<\"UAH\"\>", ",", "\<\"UGX\"\>", 
   ",", "\<\"USD\"\>", ",", "\<\"UYU\"\>", ",", "\<\"VEB\"\>", 
   ",", "\<\"VND\"\>", ",", "\<\"VUV\"\>", ",", "\<\"WST\"\>", 
   ",", "\<\"XAF\"\>", ",", "\<\"XAG\"\>", ",", "\<\"XAL\"\>", 
   ",", "\<\"XAU\"\>", ",", "\<\"XCD\"\>", ",", "\<\"XCP\"\>", 
   ",", "\<\"XOF\"\>", ",", "\<\"XPD\"\>", ",", "\<\"XPF\"\>", 
   ",", "\<\"XPT\"\>", ",", "\<\"YER\"\>", ",", "\<\"ZAR\"\>", 
   ",", "\<\"ZMK\"\>", ",", "\<\"ZWD\"\>", ",", "\<\"ZWN\"\>"}], 
  "}"}]], "Output",
 ImageSize->{522, 169},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->73559336]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->78454681],

Cell["Find the current number of Japanese yen per U.S. dollar:", "ExampleText",
 CellID->9684183],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<USD/JPY\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9944848],

Cell[BoxData["118.13`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->132385911]
}, Open  ]],

Cell["Give the pair of currencies in a list:", "ExampleText",
 CellID->229798417],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"{", 
   RowBox[{"\"\<USD\>\"", ",", "\"\<JPY\>\""}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->75280672],

Cell[BoxData["118.13`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->29819732]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->408339838],

Cell["\<\
Find the current price for a Troy ounce of gold in U.S. dollars:\
\>", "ExampleText",
 CellID->151252491],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<XAU/USD\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10801005],

Cell[BoxData["624.6`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->302367457]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Find 100-day moving averages of a stock price:", "ExampleText",
 CellID->522670720],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"MovingAverage", "[", 
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{
      "\"\<GE\>\"", ",", "\"\<Jan. 1, 2000\>\"", ",", "\"\<Value\>\""}], 
      "]"}], ",", "100"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"\"\<Jan. 1, 2000\>\"", ",", "Automatic", ",", "\"\<Day\>\""}], 
    "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->99878051],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{201, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->52970150]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22398025],

Cell["\<\
Find the log distribution of daily returns for the S&P 500 index:\
\>", "ExampleText",
 CellID->56114065],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Log", "[", 
   RowBox[{"1", "+", 
    RowBox[{"BinCounts", "[", 
     RowBox[{
      RowBox[{"FinancialData", "[", 
       RowBox[{
       "\"\<SP500\>\"", ",", "\"\<Return\>\"", ",", "All", ",", 
        "\"\<Value\>\""}], "]"}], ",", "0.001"}], "]"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->143688033],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxd13lsFVUUx/HLKqagZauyPyoSBMW2QLEg9hSUTZa2bFKBDsvrW9s+CgbU
gFcTiYpEECEEjI4QxNhAoEQUMeSKQRFFIDXwh5o8lyiLpCUSDAhUTE9N5jv/
NJ/+7pkzc9/c+970X1RdGm5tjBnUypj//jYfjYUNBddvjR36e6H+Q0zg6ABn
wt3ge+HecAjOhgfAA+FB8GD4QXgonAPnwnnwMHg4PALOh0fCj8AF8Ch4NPwo
PAZ+DC6EJWgLmyLksBmLHDbjkMPmceSweQI5bMYjh80E5LCZiBw2k5DDZjJy
2DyJHDZTkMNmKnLYTEMOm+nIYVMctMAWdrApQT1sYQebUtTDFnawmYF62MIO
NjNRD1vYwWYW6mELO9jMRj1sYQebOaiHLexg8xTqYQs72MxFPWxhB5sy1MMW
drB5GvWwhR1s5qEetrCDzXzUwxZ2sFmAetjCboEEv5/LUQ9b2MHGCzoEC+zB
FvZhB6dhsxD9F0pGm93PXdh4ufn+ZGHwfj3kFvU+7OA0bBYFzx9aFMwF9jDe
IvdhB6dhsxj9F6M/cg+5Re7DDuPTsFkSdAiWJcH59pBb5D5yhzy9RK/vqj6/
YfQPS+vQ4OsZtTf08w8Hx3uwDaM/7OB0WEpezx3S9uW22rdCsmZ9dGhfonWz
Q7BUyJSRL3z869A2zfZgWyFrNiRGPd/vjmb7FdK4//TNVd3vbLZDnq6QbWcy
rr0/IkP7R2TYukt/FzkdH4pI2ZANE79bqblEpFNN3uj6lzpp/4iszQv/uM3T
3902Ig91HnR562m1H5FnHnhr45t1aheRP7vODV99tYv2j0ju+K0HJ59Qm6hs
uv/mXPNuV+0flf2pZX+VrVRLVOZ16Gm3FOjvei8q31w/99qand21f1TK88e0
/+mE2o/KloJDmy8dydL+Uemw+o1ex8vv0f5RSV/rO+ulnJb3gpjsrDu645V6
dSgmDzfl7r/Ypof2j4n5vmpp4pzai0l+04UDv63rqf1j0n9A3ulJ36r9mMw+
69r36dZL+8ekvFzmb96hTsekVcPGG03lahOXk+d/aTjQVh2Ky56Tz5aMGqnn
k7jUXiyKTBuu9uKyat/ZTdMiej02Ltlrb7fcpNfvx+WH++LbvxqodnF5L/vt
3qm6lvuPS0b9laW3Vuv8mIT0mzi1IOtDnb9QQvb0qc8e/IfOtyRk7Ocz+g7p
1zL/CTm/uaZod6F+Pvb2+AkNpSs6qv2ELI/uOjw9Sz9fl5Au59yR2tLO2j8h
X58ZcOWfjmqTlHEzb+S3G3+X9k9Kkylp8pbfrf2Tkjj2yeH6g/o8ekm52ung
izdGdNT+Sfmg6IujX/bQ59NPyrGjY1Ys/1TfE11S12t77Z8MrgdTGVxPoUrZ
dehs1vHe7bR/pax6p/6zrJ9b1l+lLKsdXVi3Xd9LbSXWf2Vwf3AYn67U9XZN
958qrW+l/atkTlnenprsJt1/qrD/wbYqeD6/Krh/uarg9aRRb6p1HvQIVQf3
b4E92MJ+Ne4f509jvEkFx2em8H2UCtbnIBe4GOM9OIXxFvl62If3wg4+Bafh
RtgsDToTDsE5sMDFsAenYAuvh314L+zgU3AaboRNTdCZcAjOgQUuhj04Bdua
/5+HfwEGp6z2
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 296.}, {0., 6.90975328164481}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 123},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->121433221]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->238838549],

Cell["\<\
Plot the cumulative changes of a stock since 2000 compared to the S&P 500:\
\>", "ExampleText",
 CellID->96863732],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<BAC\>\"", ",", "\"\<CumulativeFractionalChange\>\"", ",", 
       RowBox[{"{", "2000", "}"}]}], "]"}], ",", 
     RowBox[{"FinancialData", "[", 
      RowBox[{
      "\"\<SP500\>\"", ",", "\"\<CumulativeFractionalChange\>\"", ",", 
       RowBox[{"{", "2000", "}"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Bottom"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->81022563],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->750832308]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->571789222],

Cell["\<\
Plot the cumulative changes of S&P 100 drug companies since 2000, with \
tickers as tooltips:\
\>", "ExampleText",
 CellID->57888114],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DateListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Tooltip", "[", 
      RowBox[{
       RowBox[{"FinancialData", "[", 
        RowBox[{"#", ",", "\"\<CumulativeFractionalChange\>\"", ",", 
         RowBox[{"{", "2000", "}"}]}], "]"}], ",", "#"}], "]"}], "&"}], "/@", 
    
    RowBox[{"Select", "[", 
     RowBox[{
      RowBox[{"FinancialData", "[", 
       RowBox[{"\"\<SP100\>\"", ",", "\"\<Members\>\""}], "]"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"FinancialData", "[", 
         RowBox[{"#", ",", "\"\<Sector\>\""}], "]"}], "===", 
        "\"\<Drugs\>\""}], "&"}]}], "]"}]}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->56078898],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    TagBox[
     TooltipBox[LineBox[CompressedData["
1:eJxV3Xd8i937B/CoPapR1GqJvapCbVW57S32Jq1a5SFGq3ao2RqxWzNaatSI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       "]],
      "\"ABT\""],
     Annotation[#, "ABT", "Tooltip"]& ]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], 
    TagBox[
     TooltipBox[LineBox[CompressedData["
1:eJxVnXlczN/3x4dKdoNkr5Elu7ETMW/73mRfM5Il61iyLyFbtrGmiCFrpEGy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       "]],
      "\"AMGN\""],
     Annotation[#, "AMGN", "Tooltip"]& ]}, 
   {Hue[0.14213595499957954`, 0.6, 0.6], 
    TagBox[
     TooltipBox[LineBox[CompressedData["
1:eJxV3XWcUk0XB3DsVuxW7FbsVq7dYrdit4vx2IEda2C3Yrdit2KsuSp2K9Zy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       "]],
      "\"BMY\""],
     Annotation[#, "BMY", "Tooltip"]& ]}, 
   {Hue[0.37820393249936934`, 0.6, 0.6], 
    TagBox[
     TooltipBox[LineBox[CompressedData["
1:eJxV3XdcTf8fB/AbISvXKGRFGlaurRIdM2Vc2ftKVr4oZGRdO0kuGdlXMjJv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       "]],
      "\"JNJ\""],
     Annotation[#, "JNJ", "Tooltip"]& ]}, 
   {Hue[0.6142719099991583, 0.6, 0.6], 
    TagBox[
     TooltipBox[LineBox[CompressedData["
1:eJxV3XVcU98bB/CBge1EEDBnolizsXdNFNRht8MAE2dgx2xF0KFiCxO7Z4Ct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       "]],
      "\"MEDI\""],
     Annotation[#, "MEDI", "Tooltip"]& ]}, 
   {Hue[0.8503398874989481, 0.6, 0.6], 
    TagBox[
     TooltipBox[LineBox[CompressedData["
1:eJxV3XlcjF/7B/CxZI+xV5ZGtmwZS0lZuu3KMvadKfueNbvsSjHWbGVIdhq7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       "]],
      "\"MRK\""],
     Annotation[#, "MRK", "Tooltip"]& ]}, 
   {Hue[0.08640786499873876, 0.6, 0.6], 
    TagBox[
     TooltipBox[LineBox[CompressedData["
1:eJxVnXW8Ut3Sx7FbsVuxW7Fb2fYxsX1MPHZjt2K3Yrdit2K3Yh1bsVsx2ZjY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       "]],
      "\"PFE\""],
     Annotation[#, "PFE", "Tooltip"]& ]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  AxesOrigin->NCache[{3156019200, 0.5}, {3.1560192*^9, 0.5}],
  Frame->True,
  FrameTicks->{{Automatic, Automatic}, {{{
       NCache[3155673600, 3.1556736*^9], 
       FormBox["\"2000\"", TraditionalForm]}, {
       NCache[3218832000, 3.218832*^9], 
       FormBox["\"2002\"", TraditionalForm]}, {
       NCache[3281904000, 3.281904*^9], 
       FormBox["\"2004\"", TraditionalForm]}, {
       NCache[3345062400, 3.3450624*^9], 
       FormBox["\"2006\"", TraditionalForm]}, {
       NCache[3187296000, 3.187296*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3250368000, 3.250368*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3313526400, 3.3135264*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3376598400, 3.3765984*^9], 
       FormBox["\" \"", TraditionalForm]}}, {{
       NCache[3155673600, 3.1556736*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3218832000, 3.218832*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3281904000, 3.281904*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3345062400, 3.3450624*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3187296000, 3.187296*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3250368000, 3.250368*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3313526400, 3.3135264*^9], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[3376598400, 3.3765984*^9], 
       FormBox["\" \"", TraditionalForm]}}}},
  GridLines->NCache[{{{3155673600, 
       GrayLevel[0.8]}, {3218832000, 
       GrayLevel[0.8]}, {3281904000, 
       GrayLevel[0.8]}, {3345062400, 
       GrayLevel[0.8]}}, None}, {{{3.1556736*^9, 
       GrayLevel[0.8]}, {3.218832*^9, 
       GrayLevel[0.8]}, {3.281904*^9, 
       GrayLevel[0.8]}, {3.3450624*^9, 
       GrayLevel[0.8]}}, None}],
  ImageSize->{249., Automatic},
  PlotRange->{{3.1560192*^9, 3.3835968*^9}, {0.35532456512515975`, 
   1.9605593402653283`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]},
  Ticks->None]], "Output",
 ImageSize->{253, 174},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->254856070]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->434753880],

Cell["\<\
Find the distribution of 50-day volatility values for the stocks in the S&P \
500, in 1% bins:\
\>", "ExampleText",
 CellID->775102060],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{"BinCounts", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"FinancialData", "[", 
        RowBox[{"#", ",", "\"\<Volatility50Day\>\""}], "]"}], "&"}], "/@", 
      RowBox[{"FinancialData", "[", 
       RowBox[{"\"\<SP500\>\"", ",", "\"\<Members\>\""}], "]"}]}], ",", 
     "0.01"}], "]"}], ",", 
   RowBox[{"InterpolationOrder", "\[Rule]", "0"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4537665],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyF1bsKwkAQBdDBytLCwsJCg4iIiO+3RuP7UfkJgrX4B/PJfoKKcy00N7MQ
LocsOxOy2RQvt/M1JSL31/VOG/NPPML4/L2fdpxxnHWcI0YfeWYbBccBMeqV
iNFn2XHFcZUYdWrEE8s68cCyQTy0bBJHli1i1GsTzyw7xOinG29dmHvxlr6T
ZN2v0e+QGM8xIsZ6Y2Lsg4njKTHWmxEHliEx3uOc2PabMmO9BbF9h8qMcyAi
tnNAHcuSGP06lhWxfc/qWNbENpTacsNs85lRb5tsdSw7YtRzLPtkq2M5JFsd
yzHZ6lhOxH//tb/5T0Pkego=
   "], {{{}, 
     {Hue[0.67, 0.6, 0.6], Opacity[0.2], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[{{110, 113, 111}},
         VertexColors->None], PolygonBox[{{84, 87, 86, 85}},
         VertexColors->None], PolygonBox[{{88, 95, 94, 93, 92, 91, 90, 89}},
         VertexColors->None], 
        PolygonBox[{{2, 112, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 1}},
         VertexColors->None], 
        PolygonBox[{{16, 83, 82, 81, 80, 79, 78, 77, 76, 75, 74, 73, 72, 71, 
         70, 69, 68, 67, 66, 65, 64, 63, 62, 61, 60, 59, 58, 57, 56, 55, 54, 
         53, 52, 51, 50, 49, 48, 47, 46, 45, 44, 43, 42, 41, 40, 39, 38, 37, 
         36, 35, 34, 33, 32, 31, 30, 29, 28, 27, 26, 25, 24, 23, 22, 21, 20, 
         19, 18, 17}},
         VertexColors->None]}]}, {}, {}}, {{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{2, 1, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71,
        72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 
       89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 
       105, 106, 107, 108, 109, 110, 111}]}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 56.}, {0., 36.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 120},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->201941089]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->354303428],

Cell["\<\
Make a scatter plot of PEG ratio compared to PE for S&P 100 companies:\
\>", "ExampleText",
 CellID->404604247],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"Select", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Text", "[", 
        RowBox[{"#", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"FinancialData", "[", 
            RowBox[{"#", ",", "\"\<PEGRatio\>\""}], "]"}], ",", 
           RowBox[{"FinancialData", "[", 
            RowBox[{"#", ",", "\"\<PERatio\>\""}], "]"}]}], "}"}]}], "]"}], 
       "&"}], "/@", 
      RowBox[{"FinancialData", "[", 
       RowBox[{"\"\<SP100\>\"", ",", "\"\<Members\>\""}], "]"}]}], ",", 
     RowBox[{
      RowBox[{"FreeQ", "[", 
       RowBox[{"#", ",", "_Missing"}], "]"}], "&"}]}], "]"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}], ",", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "50"}], "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->719098037],

Cell[BoxData[
 GraphicsBox[{InsetBox["\<\"AA\"\>", {1.11, 13.12}], 
   InsetBox["\<\"ABT\"\>", {1.9, 48.79}], 
   InsetBox["\<\"AEP\"\>", {3.4, 19.15}], 
   InsetBox["\<\"AES\"\>", {1.25, 39.89}], 
   InsetBox["\<\"AIG\"\>", {0.83, 12.7}], 
   InsetBox["\<\"ALL\"\>", {1.04, 7.76}], 
   InsetBox["\<\"AMGN\"\>", {0.89, 24.39}], 
   InsetBox["\<\"ATI\"\>", {0.97, 19.41}], 
   InsetBox["\<\"AVP\"\>", {2.23, 35.71}], 
   InsetBox["\<\"AXP\"\>", {1.3, 19.19}], 
   InsetBox["\<\"BA\"\>", {1.22, 32.21}], 
   InsetBox["\<\"BAC\"\>", {1.17, 11.24}], 
   InsetBox["\<\"BAX\"\>", {1.62, 24.71}], 
   InsetBox["\<\"BDK\"\>", {1.34, 12.92}], 
   InsetBox["\<\"BHI\"\>", {0.69, 9.08}], 
   InsetBox["\<\"BMY\"\>", {1.89, 34.44}], 
   InsetBox["\<\"BNI\"\>", {1.01, 15.75}], 
   InsetBox["\<\"BUD\"\>", {1.98, 20.13}], 
   InsetBox["\<\"C\"\>", {1.15, 12.04}], 
   InsetBox["\<\"CAT\"\>", {0.84, 12.86}], 
   InsetBox["\<\"CCU\"\>", {2.19, 25.84}], 
   InsetBox["\<\"CI\"\>", {1.18, 14.17}], 
   InsetBox["\<\"CL\"\>", {1.99, 27.28}], 
   InsetBox["\<\"CMCSA\"\>", {2.58, 33.54}], 
   InsetBox["\<\"COF\"\>", {0.84, 10.21}], 
   InsetBox["\<\"COP\"\>", {1.16, 7.11}], 
   InsetBox["\<\"CPB\"\>", {3.1, 20.89}], 
   InsetBox["\<\"CSC\"\>", {1.18, 26.65}], 
   InsetBox["\<\"CSCO\"\>", {1.22, 25.45}], 
   InsetBox["\<\"CVX\"\>", {2.06, 9.35}], 
   InsetBox["\<\"DD\"\>", {1.78, 15.2}], 
   InsetBox["\<\"DELL\"\>", {1.24, 17.86}], 
   InsetBox["\<\"DIS\"\>", {1.49, 17.12}], 
   InsetBox["\<\"DOW\"\>", {1.43, 11.88}], 
   InsetBox["\<\"EMC\"\>", {1.39, 25.24}], 
   InsetBox["\<\"EP\"\>", {1.84, 22.95}], 
   InsetBox["\<\"ETR\"\>", {2.3, 19.44}], 
   InsetBox["\<\"EXC\"\>", {1.66, 28.76}], 
   InsetBox["\<\"FDX\"\>", {1.1, 17.46}], 
   InsetBox["\<\"GD\"\>", {1.49, 16.97}], 
   InsetBox["\<\"GE\"\>", {1.61, 17.87}], 
   InsetBox["\<\"GOOG\"\>", {1.08, 46.47}], 
   InsetBox["\<\"GS\"\>", {0.71, 9.88}], 
   InsetBox["\<\"HAL\"\>", {0.75, 13.8}], 
   InsetBox["\<\"HD\"\>", {1.06, 13.72}], 
   InsetBox["\<\"HET\"\>", {2.12, 29.33}], 
   InsetBox["\<\"HIG\"\>", {0.87, 11.22}], 
   InsetBox["\<\"HNZ\"\>", {2.76, 20.39}], 
   InsetBox["\<\"HON\"\>", {1.35, 18.87}], 
   InsetBox["\<\"HPQ\"\>", {1.21, 17.54}], 
   InsetBox["\<\"IBM\"\>", {1.34, 15.58}], 
   InsetBox["\<\"INTC\"\>", {1.48, 22.33}], 
   InsetBox["\<\"IP\"\>", {2.85, 3.04}], 
   InsetBox["\<\"JNJ\"\>", {1.71, 16.3}], 
   InsetBox["\<\"JPM\"\>", {1.18, 12.}], 
   InsetBox["\<\"KO\"\>", {2.35, 22.32}], 
   InsetBox["\<\"LEH\"\>", {0.8, 10.54}], 
   InsetBox["\<\"LTD\"\>", {0.99, 15.6}], 
   InsetBox["\<\"MCD\"\>", {2.02, 15.71}], 
   InsetBox["\<\"MDT\"\>", {1.43, 21.39}], 
   InsetBox["\<\"MEDI\"\>", {1.03, 170.4}], 
   InsetBox["\<\"MER\"\>", {0.93, 11.2}], 
   InsetBox["\<\"MMM\"\>", {1.52, 15.37}], 
   InsetBox["\<\"MO\"\>", {2.04, 15.08}], 
   InsetBox["\<\"MRK\"\>", {2.51, 21.79}], 
   InsetBox["\<\"MS\"\>", {0.78, 10.02}], 
   InsetBox["\<\"MSFT\"\>", {1.38, 24.16}], 
   InsetBox["\<\"NSC\"\>", {0.88, 13.96}], 
   InsetBox["\<\"NSM\"\>", {1.68, 20.66}], 
   InsetBox["\<\"ORCL\"\>", {1.15, 24.69}], 
   InsetBox["\<\"PEP\"\>", {1.75, 19.17}], 
   InsetBox["\<\"PFE\"\>", {2.34, 9.7}], 
   InsetBox["\<\"PG\"\>", {1.64, 22.73}], 
   InsetBox["\<\"RF\"\>", {1.6, 13.54}], 
   InsetBox["\<\"ROK\"\>", {1.22, 12.24}], 
   InsetBox["\<\"RTN\"\>", {1.11, 18.5}], 
   InsetBox["\<\"S\"\>", {2.34, 43.51}], 
   InsetBox["\<\"SLB\"\>", {0.84, 22.92}], 
   InsetBox["\<\"SLE\"\>", {3.12, 39.84}], 
   InsetBox["\<\"SO\"\>", {3.38, 17.44}], 
   InsetBox["\<\"T\"\>", {1.61, 20.73}], 
   InsetBox["\<\"TGT\"\>", {0.99, 19.08}], 
   InsetBox["\<\"TWX\"\>", {1.39, 12.85}], 
   InsetBox["\<\"TXN\"\>", {1.33, 11.23}], 
   InsetBox["\<\"TYC\"\>", {1.21, 16.61}], 
   InsetBox["\<\"UPS\"\>", {1.43, 18.42}], 
   InsetBox["\<\"USB\"\>", {1.43, 13.81}], 
   InsetBox["\<\"UTX\"\>", {1.33, 17.81}], 
   InsetBox["\<\"VZ\"\>", {3.26, 17.93}], 
   InsetBox["\<\"WB\"\>", {1.27, 12.26}], 
   InsetBox["\<\"WFC\"\>", {1.19, 14.07}], 
   InsetBox["\<\"WMB\"\>", {1.31, 56.37}], 
   InsetBox["\<\"WMT\"\>", {1.03, 17.74}], 
   InsetBox["\<\"WY\"\>", {4.16, 42.12}], 
   InsetBox["\<\"XOM\"\>", {2.2, 11.24}], 
   InsetBox["\<\"XRX\"\>", {1.27, 14.34}]},
  AspectRatio->1,
  Frame->True,
  ImageSize->{276., Automatic},
  PlotRange->{{0, 5}, {0, 50}}]], "Output",
 ImageSize->{280, 282},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->215204516]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1028363587],

Cell["\<\
Make a ranked plot of the forward PE estimates to trailing PE values for the \
S&P 100:\
\>", "ExampleText",
 CellID->76722058],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Reverse", "[", 
   RowBox[{"Sort", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"FinancialData", "[", 
        RowBox[{"#", ",", "\"\<PERatio\>\""}], "]"}], "/", 
       RowBox[{"FinancialData", "[", 
        RowBox[{"#", ",", "\"\<ForwardPERatio\>\""}], "]"}]}], "&"}], "/@", 
     RowBox[{"FinancialData", "[", 
      RowBox[{"\"\<SP100\>\"", ",", "\"\<Members\>\""}], "]"}]}], "]"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->283057930],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJw1y2tME1YchvEyhUyQZFyaoQUmCyIIKkPBdYB9C4VawEJvtANTzDLGZQgk
oqhMaYNmLgM+IAs4zIhaUCAgdhMJ4CjTkjEJMupCdNMBEZhcBEdRLmp38fxP
cnLyy3Nen0/y5OlvcTictH/vf++BoZ5n3nd46FXt1vBOrQHn/+OJI380zLYr
yJuwTtu/fGrFjvl9XP8zrTPwKYfZF+V1OwJ+7SH7YcLilLXWSPaH/MLt+UlP
m+CNt8LLLB/W3H/NHITyIxvqLGXk7UhdHXbgfUUORmNrpq7o2ivmDzDDXaj9
zfqSOQRXzZkhcynknTBrLwg/z1tm3oWfR75YPyAmhyL+QUWbYRs5DJ31m8a4
o0vMu2FbEXPLN5M/RLfxeFWEB5mPxRNDHRftyR/hTEM6vzn2BXM4CnIxKI0m
R8AnJfUsl0eOxHiGrqea95x5D5rbLvWd4JIFEPI9KkuEi8yA8Erpw4KNzDrA
Jl6suepCXYi3+72Di0xW1oV4sJPvbN/BzInC2N3i/pFvqUdhfY1rfcjX1KNx
/qZ3VYCeejTmc+4VhmVTF8ElwVpoTKEuwm3PieAv+dRjkKt5lVfkTz0GrUvh
0x5jC6zHwpozVPwuWRcL5co5y8gAdTHKVg9Lxm9RFyO9uDlw1w3qe1FSvV+Y
0EJ9L1Z575XdO0ZdgoYrWyZK8qlL4JvVdfNQDvU4dA0OTHSkUY9D+I5Q44Kc
ejwyQmr6cvyox8NNanOV2v5mPQHnF0X3Y1eZdQkwyCYMU1bq+xChlI++fEJ9
H0Rey6lTw9SlmN0qrMsYoi7FLXdD6dFW6omQaaOPOddRT4RT593Cg+eoJ2F0
zEe3lowkNC5VD2Wdpv9J6A3MRYqe2ZSEuFqrpLaA9jJU59u5bT9Aexkys812
5jDay2Do/q5JQzbJwPvh96bNXrSXY1rW7n16+hnby+Hv0PLZHQuzTo7ST7kD
fWSTHC1RqkeRl5k5CvC6dH5FNbRX4KA1KO96Be0VKNHuaW9U016BMu2MNGYb
7ZU44xS6Rd8+z/ZKHFrUNjhVMeuUiBVY8tuOMpuUmHJQPP4+l5mjwnH77soW
Ne1VeCExqyI30l4Fl0xHff07tFfhzY72ydhwKfhGXcAc2yeDf7LA7aeLT9k+
GYezvR+31M6yfTK+6bOzeZ5k5qjRZe/qLuidYXs10iKc8/5qZdapMVvp/jDI
kdmkRr9wdLDp0TTbazD5o1yf9XqK7TXIzzIMXkt5wvYadDgqKxLnJtleg0qf
ZqOodpztP8Y6s+/q8/2/CP4B+f/YVQ==
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0.5},
  PlotRange->{{0., 100.}, {0.20596205962059624`, 2.4395052806746405`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->450381436]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell["\<\
A stock listed on different exchanges is given in different currencies:\
\>", "ExampleText",
 CellID->560639371],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<DCX\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->484418457],

Cell[BoxData["71.94`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->24159429]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<F:DCX\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->607678962],

Cell[BoxData["54.07`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->257936795]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<EUR/USD\>\"", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->135458308],

Cell[BoxData["1.3303`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->112325819]
}, Open  ]],

Cell["\<\
The ratio of prices is normally near the currency exchange rate:\
\>", "ExampleText",
 CellID->761357570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"FinancialData", "[", "\"\<DCX\>\"", "]"}], "/", 
  RowBox[{"FinancialData", "[", "\"\<F:DCX\>\"", "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->84837171],

Cell[BoxData["1.3304975032365451`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->373922157]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["\<\
Certain properties are available only when the relevant market is open:\
\>", "ExampleText",
 CellID->163621777],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<Bid\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->569611999],

Cell[BoxData[
 RowBox[{"Missing", "[", "\<\"NotAvailable\"\>", "]"}]], "Output",
 ImageSize->{151, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->72074793]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->174359581],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FinancialData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FinancialData"]], "InlineFormula"],
 " by default assumes U.S. stocks:"
}], "ExampleText",
 CellID->342256626],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<BMW\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19576045],

Cell[BoxData[
 RowBox[{
  RowBox[{"FinancialData", "::", "\<\"notent\"\>"}], ":", 
  " ", "\<\"\\!\\(\\\"BMW\\\"\\) is not a known entity in FinancialData. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/FinancialData\\\", ButtonNote \
-> \\\"FinancialData::notent\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->48956927],

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\<\"BMW\"\>", "]"}]], "Output",
 ImageSize->{130, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->510173495]
}, Open  ]],

Cell["\<\
Give an explicit exchange to find information on non-U.S. stocks:\
\>", "ExampleText",
 CellID->233017874],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<F:BMW\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->729483323],

Cell[BoxData["41.04`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->91322303]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->121223470],

Cell["\<\
Non-U.S. stocks are quoted in the standard currencies of their exchanges:\
\>", "ExampleText",
 CellID->174677099],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", "\"\<F:BMW\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->298282296],

Cell[BoxData["41.04`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->351349438]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->813057292],

Cell["Daily stock prices are not available for holidays:", "ExampleText",
 CellID->623115820],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"2000", ",", "1", ",", "1"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->899477076],

Cell[BoxData[
 RowBox[{"Missing", "[", "\<\"NotAvailable\"\>", "]"}]], "Output",
 ImageSize->{151, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->70775367]
}, Open  ]],

Cell["The monthly data shows on which days trading occurred:", "ExampleText",
 CellID->77428252],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"2000", ",", "1"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->85641797],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "3"}], "}"}], ",", "42.31`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "4"}], "}"}], ",", "40.62`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "5"}], "}"}], ",", "40.55`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "6"}], "}"}], ",", "41.09`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "7"}], "}"}], ",", "42.68`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "10"}], "}"}], ",", "42.67`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "11"}], "}"}], ",", "42.74`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "12"}], "}"}], ",", "42.88`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "13"}], "}"}], ",", "43.37`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "14"}], "}"}], ",", "42.6`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "18"}], "}"}], ",", "41.75`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "19"}], "}"}], ",", "41.95`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "20"}], "}"}], ",", "41.17`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "21"}], "}"}], ",", "40.66`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "24"}], "}"}], ",", "38.97`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "25"}], "}"}], ",", "39.07`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "26"}], "}"}], ",", "39.9`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "27"}], "}"}], ",", "39.99`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "28"}], "}"}], ",", "37.8`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2000", ",", "1", ",", "31"}], "}"}], ",", "37.8`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{478, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->55181212]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->140140145],

Cell["The 52-week high includes intraday prices:", "ExampleText",
 CellID->281263559],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FinancialData", "[", 
  RowBox[{"\"\<GE\>\"", ",", "\"\<High52Week\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->62622579],

Cell[BoxData["38.49`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->615049915]
}, Open  ]],

Cell["\<\
It is not generally equal to the maximum of daily closing prices:\
\>", "ExampleText",
 CellID->983404076],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{"FinancialData", "[", 
   RowBox[{"\"\<GE\>\"", ",", 
    RowBox[{
     RowBox[{"DateList", "[", "]"}], "-", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
      "}"}]}], ",", "\"\<Value\>\""}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->128956369],

Cell[BoxData["37.81`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->68184022]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->589267740],

Cell["\<\
Make a log rank plot of values of world currencies relative to U.S. dollars:\
\>", "ExampleText",
 CellID->559244587],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogPlot", "[", 
  RowBox[{
   RowBox[{"Reverse", "[", 
    RowBox[{"Sort", "[", 
     RowBox[{"Select", "[", 
      RowBox[{
       RowBox[{"Quiet", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"FinancialData", "[", 
           RowBox[{"{", 
            RowBox[{"#", ",", "\"\<USD\>\""}], "}"}], "]"}], "&"}], "/@", 
         RowBox[{"FinancialData", "[", "\"\<Currencies\>\"", "]"}]}], "]"}], 
       ",", "NumericQ"}], "]"}], "]"}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->462795108],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{273, 169},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->532817396]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["CountryData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CountryData"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["DateListPlot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DateListPlot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[{
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/InternetConnectivity"],
  FontSlant->"Italic"],
 ButtonBox[" Internet Connectivity",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/InternetConnectivity"]
}], "Tutorials",
 CellID->341631938]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->149354000],

Cell[TextData[ButtonBox["Integrated Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DataCollections"]], "MoreAbout",
 CellID->147604963],

Cell[TextData[ButtonBox["New in 6.0: Data Handling & Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60DataHandlingAndDataSources"]], "MoreAbout",
 CellID->549668142]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FinancialData - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 42, 46.1243380}", "context" -> "System`", 
    "keywords" -> {
     "AMEX", "bonds", "commodities", "currencies", "currency", "data feeds", 
      "datafeeds", "dividends", "earnings", "exchange rates", "finance", 
      "financial options", "funds", "futures", "high\\[Hyphen]low close", 
      "index", "market cap", "mutual funds", "NASD", "NASDAQ", "NYSE", 
      "options", "PEG ratio", "PE ratio", "price data", "securities", 
      "security", "stocks", "trading", "trading volume", "USD", "EPS"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "FinancialData[\"name\"] gives the last known price or value for the \
financial entity specified by \" name\". FinancialData[\"name\", start] gives \
a list of dates and daily closing values for \" name\" from start until the \
current date. FinancialData[\"name\", {start, end}] gives a list of dates and \
daily closing values for dates from start to end. FinancialData[\"name\", \
{start, end, period}] gives a list of dates and prices for the specified \
periods lying between start and end. FinancialData[\"name\", \" prop\"] gives \
the value of the specified property for the financial entity \" name\". \
FinancialData[\"name\", \" prop\", {start, end, ...}] gives a list of dates \
and values of a property for a sequence of dates or periods. ", 
    "synonyms" -> {"financial data"}, "title" -> "FinancialData", "type" -> 
    "Symbol", "uri" -> "ref/FinancialData"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[30557, 848, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->463832686]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 390046, 9110}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2106, 52, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2729, 79, 1578, 37, 70, "ObjectNameGrid"],
Cell[4310, 118, 3744, 111, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[8091, 234, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[8569, 251, 313, 11, 70, "Notes",
 CellID->208171595],
Cell[8885, 264, 322, 8, 70, "Notes",
 CellID->288717216],
Cell[9210, 274, 203, 6, 70, "Notes",
 CellID->67228700],
Cell[9416, 282, 646, 18, 70, "Notes",
 CellID->182515409],
Cell[10065, 302, 498, 15, 70, "Notes",
 CellID->199059835],
Cell[10566, 319, 112, 3, 70, "Notes",
 CellID->426441168],
Cell[10681, 324, 277, 9, 70, "Notes",
 CellID->90563636],
Cell[10961, 335, 361, 12, 70, "Notes",
 CellID->1067943069],
Cell[11325, 349, 302, 11, 70, "Notes",
 CellID->380179337],
Cell[11630, 362, 200, 7, 70, "Notes",
 CellID->91804797],
Cell[11833, 371, 406, 13, 70, "Notes",
 CellID->103182654],
Cell[12242, 386, 367, 11, 70, "Notes",
 CellID->54231339],
Cell[12612, 399, 108, 3, 70, "Notes",
 CellID->143754142],
Cell[12723, 404, 805, 15, 70, "2ColumnTableMod",
 CellID->121387011],
Cell[13531, 421, 271, 8, 70, "Notes",
 CellID->321673510],
Cell[13805, 431, 203, 5, 70, "Notes",
 CellID->67000935],
Cell[14011, 438, 89, 1, 70, "Notes",
 CellID->622463],
Cell[14103, 441, 1459, 27, 70, "2ColumnTableMod",
 CellID->119473767],
Cell[15565, 470, 129, 4, 70, "Notes",
 CellID->7509541],
Cell[15697, 476, 194, 4, 70, "Notes",
 CellID->47166445],
Cell[15894, 482, 320, 9, 70, "Notes",
 CellID->703660863],
Cell[16217, 493, 94, 1, 70, "Notes",
 CellID->41178427],
Cell[16314, 496, 923, 17, 70, "2ColumnTableMod",
 CellID->89792513],
Cell[17240, 515, 114, 3, 70, "Notes",
 CellID->215444426],
Cell[17357, 520, 2858, 49, 70, "2ColumnTableMod",
 CellID->687300919],
Cell[20218, 571, 68, 1, 70, "Notes",
 CellID->21048359],
Cell[20289, 574, 1349, 29, 70, "2ColumnTableMod",
 CellID->362079982],
Cell[21641, 605, 71, 1, 70, "Notes",
 CellID->32994604],
Cell[21715, 608, 603, 13, 70, "2ColumnTableMod",
 CellID->595208210],
Cell[22321, 623, 91, 1, 70, "Notes",
 CellID->536689746],
Cell[22415, 626, 1775, 34, 70, "2ColumnTableMod",
 CellID->955033037],
Cell[24193, 662, 87, 1, 70, "Notes",
 CellID->411153225],
Cell[24283, 665, 1010, 19, 70, "2ColumnTableMod",
 CellID->322122191],
Cell[25296, 686, 88, 1, 70, "Notes",
 CellID->400765134],
Cell[25387, 689, 644, 13, 70, "2ColumnTableMod",
 CellID->230349554],
Cell[26034, 704, 354, 11, 70, "Notes",
 CellID->112592060],
Cell[26391, 717, 261, 9, 70, "Notes",
 CellID->25099614],
Cell[26655, 728, 124, 4, 70, "Notes",
 CellID->327866574],
Cell[26782, 734, 460, 13, 70, "Notes",
 CellID->553509082],
Cell[27245, 749, 628, 17, 70, "2ColumnTableMod",
 CellID->17546678],
Cell[27876, 768, 525, 15, 70, "Notes",
 CellID->630797083],
Cell[28404, 785, 1228, 27, 70, "2ColumnTableMod",
 CellID->217110870],
Cell[29635, 814, 348, 9, 70, "Notes",
 CellID->122835266],
Cell[29986, 825, 212, 7, 70, "Notes",
 CellID->94188222],
Cell[30201, 834, 319, 9, 70, "Notes",
 CellID->168949472]
}, Closed]],
Cell[CellGroupData[{
Cell[30557, 848, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->463832686],
Cell[CellGroupData[{
Cell[30942, 862, 149, 5, 70, "ExampleSection",
 CellID->351172630],
Cell[31094, 869, 76, 1, 70, "ExampleText",
 CellID->3690073],
Cell[CellGroupData[{
Cell[31195, 874, 118, 3, 28, "Input",
 CellID->276236122],
Cell[31316, 879, 163, 5, 36, "Output",
 CellID->805037850]
}, Open  ]],
Cell[31494, 887, 125, 3, 70, "ExampleDelimiter",
 CellID->370889348],
Cell[31622, 892, 76, 1, 70, "ExampleText",
 CellID->50749667],
Cell[CellGroupData[{
Cell[31723, 897, 156, 4, 70, "Input",
 CellID->67295819],
Cell[31882, 903, 167, 5, 36, "Output",
 CellID->120606000]
}, Open  ]],
Cell[32064, 911, 125, 3, 70, "ExampleDelimiter",
 CellID->212765019],
Cell[32192, 916, 93, 1, 70, "ExampleText",
 CellID->300090350],
Cell[CellGroupData[{
Cell[32310, 921, 202, 6, 28, "Input",
 CellID->416856143],
Cell[32515, 929, 10574, 177, 248, "Output",
 Evaluatable->False,
 CellID->679432529]
}, Open  ]],
Cell[43104, 1109, 125, 3, 70, "ExampleDelimiter",
 CellID->741408050],
Cell[43232, 1114, 123, 3, 70, "ExampleText",
 CellID->16278858],
Cell[CellGroupData[{
Cell[43380, 1121, 346, 11, 28, "Input",
 CellID->29514426],
Cell[43729, 1134, 5507, 94, 244, "Output",
 Evaluatable->False,
 CellID->118794820]
}, Open  ]],
Cell[49251, 1231, 64, 1, 70, "ExampleText",
 CellID->158291915],
Cell[CellGroupData[{
Cell[49340, 1236, 467, 14, 47, "Input",
 CellID->525400874],
Cell[49810, 1252, 7983, 135, 246, "Output",
 Evaluatable->False,
 CellID->15421067]
}, Open  ]],
Cell[57808, 1390, 125, 3, 70, "ExampleDelimiter",
 CellID->120816537],
Cell[57936, 1395, 104, 2, 70, "ExampleText",
 CellID->10783646],
Cell[CellGroupData[{
Cell[58065, 1401, 461, 13, 47, "Input",
 CellID->619929818],
Cell[58529, 1416, 3848, 67, 181, "Output",
 Evaluatable->False,
 CellID->213560312]
}, Open  ]],
Cell[62392, 1486, 125, 3, 70, "ExampleDelimiter",
 CellID->912995407],
Cell[62520, 1491, 113, 3, 70, "ExampleText",
 CellID->127449912],
Cell[CellGroupData[{
Cell[62658, 1498, 160, 4, 70, "Input",
 CellID->885562420],
Cell[62821, 1504, 422, 10, 54, "Output",
 CellID->61089102]
}, Open  ]],
Cell[63258, 1517, 125, 3, 70, "ExampleDelimiter",
 CellID->277215538],
Cell[63386, 1522, 96, 1, 70, "ExampleText",
 CellID->447998576],
Cell[CellGroupData[{
Cell[63507, 1527, 157, 4, 70, "Input",
 CellID->215281312],
Cell[63667, 1533, 808, 16, 54, "Output",
 CellID->227354048]
}, Open  ]],
Cell[64490, 1552, 125, 3, 70, "ExampleDelimiter",
 CellID->658206691],
Cell[64618, 1557, 97, 1, 70, "ExampleText",
 CellID->82597117],
Cell[CellGroupData[{
Cell[64740, 1562, 158, 4, 70, "Input",
 CellID->411671714],
Cell[64901, 1568, 621, 13, 54, "Output",
 CellID->766076723]
}, Open  ]],
Cell[65537, 1584, 125, 3, 70, "ExampleDelimiter",
 CellID->177809840],
Cell[65665, 1589, 115, 3, 70, "ExampleText",
 CellID->131409717],
Cell[CellGroupData[{
Cell[65805, 1596, 120, 3, 70, "Input",
 CellID->68668378],
Cell[65928, 1601, 163, 5, 36, "Output",
 CellID->355381940]
}, Open  ]],
Cell[66106, 1609, 125, 3, 70, "ExampleDelimiter",
 CellID->695728270],
Cell[66234, 1614, 113, 3, 70, "ExampleText",
 CellID->226997588],
Cell[CellGroupData[{
Cell[66372, 1621, 123, 3, 70, "Input",
 CellID->446276343],
Cell[66498, 1626, 163, 5, 36, "Output",
 CellID->82861923]
}, Open  ]],
Cell[66676, 1634, 125, 3, 70, "ExampleDelimiter",
 CellID->268634527],
Cell[66804, 1639, 81, 1, 70, "ExampleText",
 CellID->11500889],
Cell[CellGroupData[{
Cell[66910, 1644, 123, 3, 70, "Input",
 CellID->590840916],
Cell[67036, 1649, 163, 5, 36, "Output",
 CellID->493099498]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[67248, 1660, 222, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[CellGroupData[{
Cell[67495, 1671, 241, 7, 70, "ExampleSubsection",
 CellID->265569808],
Cell[CellGroupData[{
Cell[67761, 1682, 148, 3, 70, "ExampleSubsubsection",
 CellID->722287],
Cell[67912, 1687, 69, 1, 70, "ExampleText",
 CellID->2099160],
Cell[CellGroupData[{
Cell[68006, 1692, 118, 3, 70, "Input",
 CellID->298533035],
Cell[68127, 1697, 163, 5, 36, "Output",
 CellID->592195199]
}, Open  ]],
Cell[68305, 1705, 74, 1, 70, "ExampleText",
 CellID->78002510],
Cell[CellGroupData[{
Cell[68404, 1710, 123, 3, 70, "Input",
 CellID->739276185],
Cell[68530, 1715, 161, 5, 36, "Output",
 CellID->9046840]
}, Open  ]],
Cell[CellGroupData[{
Cell[68728, 1725, 175, 5, 70, "Input",
 CellID->85141268],
Cell[68906, 1732, 163, 5, 36, "Output",
 CellID->213366921]
}, Open  ]],
Cell[69084, 1740, 125, 3, 70, "ExampleDelimiter",
 CellID->176730052],
Cell[69212, 1745, 73, 1, 70, "ExampleText",
 CellID->71027552],
Cell[CellGroupData[{
Cell[69310, 1750, 120, 3, 70, "Input",
 CellID->464158350],
Cell[69433, 1755, 163, 5, 36, "Output",
 CellID->320877701]
}, Open  ]],
Cell[69611, 1763, 75, 1, 70, "ExampleText",
 CellID->328506286],
Cell[CellGroupData[{
Cell[69711, 1768, 127, 3, 70, "Input",
 CellID->108438947],
Cell[69841, 1773, 163, 5, 36, "Output",
 CellID->595793689]
}, Open  ]],
Cell[70019, 1781, 124, 3, 70, "ExampleDelimiter",
 CellID->36845349],
Cell[70146, 1786, 95, 1, 70, "ExampleText",
 CellID->387738862],
Cell[CellGroupData[{
Cell[70266, 1791, 121, 3, 70, "Input",
 CellID->458258508],
Cell[70390, 1796, 163, 5, 36, "Output",
 CellID->649785924]
}, Open  ]],
Cell[CellGroupData[{
Cell[70590, 1806, 120, 3, 70, "Input",
 CellID->226929606],
Cell[70713, 1811, 162, 5, 36, "Output",
 CellID->43187262]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[70924, 1822, 161, 4, 70, "ExampleSubsubsection",
 CellID->332015634],
Cell[71088, 1828, 89, 1, 70, "ExampleText",
 CellID->93757185],
Cell[CellGroupData[{
Cell[71202, 1833, 151, 4, 70, "Input",
 CellID->25987035],
Cell[71356, 1839, 190, 5, 36, "Output",
 CellID->541322348]
}, Open  ]],
Cell[71561, 1847, 124, 3, 70, "ExampleDelimiter",
 CellID->17993225],
Cell[71688, 1852, 101, 2, 70, "ExampleText",
 CellID->12201578],
Cell[CellGroupData[{
Cell[71814, 1858, 155, 4, 70, "Input",
 CellID->224287922],
Cell[71972, 1864, 192, 5, 36, "Output",
 CellID->23598697]
}, Open  ]],
Cell[72179, 1872, 125, 3, 70, "ExampleDelimiter",
 CellID->332574725],
Cell[72307, 1877, 83, 1, 70, "ExampleText",
 CellID->165557687],
Cell[CellGroupData[{
Cell[72415, 1882, 153, 4, 70, "Input",
 CellID->288461389],
Cell[72571, 1888, 174, 5, 36, "Output",
 CellID->941911628]
}, Open  ]],
Cell[CellGroupData[{
Cell[72782, 1898, 151, 4, 70, "Input",
 CellID->51887334],
Cell[72936, 1904, 176, 5, 36, "Output",
 CellID->31938098]
}, Open  ]],
Cell[73127, 1912, 125, 3, 70, "ExampleDelimiter",
 CellID->198424648],
Cell[73255, 1917, 88, 1, 70, "ExampleText",
 CellID->575434581],
Cell[CellGroupData[{
Cell[73368, 1922, 156, 4, 70, "Input",
 CellID->428178773],
Cell[73527, 1928, 168, 5, 36, "Output",
 CellID->97857245]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[73744, 1939, 146, 3, 70, "ExampleSubsubsection",
 CellID->42165130],
Cell[73893, 1944, 96, 1, 70, "ExampleText",
 CellID->187325701],
Cell[CellGroupData[{
Cell[74014, 1949, 156, 4, 70, "Input",
 CellID->265754125],
Cell[74173, 1955, 280, 8, 36, "Output",
 CellID->1287019696]
}, Open  ]],
Cell[74468, 1966, 89, 1, 70, "ExampleText",
 CellID->537757506],
Cell[CellGroupData[{
Cell[74582, 1971, 275, 8, 70, "Input",
 CellID->11470067],
Cell[74860, 1981, 371, 10, 54, "Output",
 CellID->1484738088]
}, Open  ]],
Cell[75246, 1994, 125, 3, 70, "ExampleDelimiter",
 CellID->646995844],
Cell[75374, 1999, 85, 1, 70, "ExampleText",
 CellID->346300904],
Cell[CellGroupData[{
Cell[75484, 2004, 204, 6, 70, "Input",
 CellID->898062766],
Cell[75691, 2012, 315, 9, 36, "Output",
 CellID->233066876]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[76055, 2027, 149, 3, 70, "ExampleSubsubsection",
 CellID->206196781],
Cell[76207, 2032, 95, 1, 70, "ExampleText",
 CellID->53789131],
Cell[CellGroupData[{
Cell[76327, 2037, 121, 3, 70, "Input",
 CellID->37322416],
Cell[76451, 2042, 385, 10, 54, "Output",
 CellID->313618589]
}, Open  ]],
Cell[76851, 2055, 125, 3, 70, "ExampleDelimiter",
 CellID->135005035],
Cell[76979, 2060, 87, 1, 70, "ExampleText",
 CellID->16994087],
Cell[CellGroupData[{
Cell[77091, 2065, 157, 4, 70, "Input",
 CellID->54396352],
Cell[77251, 2071, 159, 5, 36, "Output",
 CellID->148380073]
}, Open  ]],
Cell[77425, 2079, 125, 3, 70, "ExampleDelimiter",
 CellID->118942676],
Cell[77553, 2084, 98, 1, 70, "ExampleText",
 CellID->284166854],
Cell[CellGroupData[{
Cell[77676, 2089, 158, 4, 70, "Input",
 CellID->351923496],
Cell[77837, 2095, 621, 13, 54, "Output",
 CellID->144812804]
}, Open  ]],
Cell[78473, 2111, 124, 3, 70, "ExampleDelimiter",
 CellID->77961267],
Cell[78600, 2116, 82, 1, 70, "ExampleText",
 CellID->336628928],
Cell[CellGroupData[{
Cell[78707, 2121, 155, 4, 70, "Input",
 CellID->91794653],
Cell[78865, 2127, 198, 5, 36, "Output",
 CellID->40942134]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[79112, 2138, 161, 4, 70, "ExampleSubsubsection",
 CellID->313174033],
Cell[79276, 2144, 139, 5, 70, "ExampleText",
 CellID->151207433],
Cell[CellGroupData[{
Cell[79440, 2153, 120, 3, 70, "Input",
 CellID->333743693],
Cell[79563, 2158, 166, 5, 36, "Output",
 CellID->261108333]
}, Open  ]],
Cell[CellGroupData[{
Cell[79766, 2168, 154, 4, 70, "Input",
 CellID->180640064],
Cell[79923, 2174, 194, 5, 36, "Output",
 CellID->6871051]
}, Open  ]],
Cell[80132, 2182, 125, 3, 70, "ExampleDelimiter",
 CellID->342023374],
Cell[80260, 2187, 115, 3, 70, "ExampleText",
 CellID->384057746],
Cell[CellGroupData[{
Cell[80400, 2194, 120, 3, 70, "Input",
 CellID->65799697],
Cell[80523, 2199, 161, 5, 36, "Output",
 CellID->2099440]
}, Open  ]],
Cell[CellGroupData[{
Cell[80721, 2209, 155, 4, 70, "Input",
 CellID->250026892],
Cell[80879, 2215, 182, 5, 36, "Output",
 CellID->75952996]
}, Open  ]],
Cell[81076, 2223, 124, 3, 70, "ExampleDelimiter",
 CellID->67761741],
Cell[81203, 2228, 60, 1, 70, "ExampleText",
 CellID->657420572],
Cell[CellGroupData[{
Cell[81288, 2233, 122, 3, 70, "Input",
 CellID->212197307],
Cell[81413, 2238, 163, 5, 36, "Output",
 CellID->529210167]
}, Open  ]],
Cell[CellGroupData[{
Cell[81613, 2248, 156, 4, 70, "Input",
 CellID->332990830],
Cell[81772, 2254, 194, 5, 36, "Output",
 CellID->246831888]
}, Open  ]],
Cell[81981, 2262, 125, 3, 70, "ExampleDelimiter",
 CellID->228816305],
Cell[82109, 2267, 83, 1, 70, "ExampleText",
 CellID->307577191],
Cell[CellGroupData[{
Cell[82217, 2272, 120, 3, 70, "Input",
 CellID->47970543],
Cell[82340, 2277, 164, 5, 36, "Output",
 CellID->54388660]
}, Open  ]],
Cell[CellGroupData[{
Cell[82541, 2287, 121, 3, 70, "Input",
 CellID->448851325],
Cell[82665, 2292, 163, 5, 36, "Output",
 CellID->251571347]
}, Open  ]],
Cell[CellGroupData[{
Cell[82865, 2302, 155, 4, 70, "Input",
 CellID->430589200],
Cell[83023, 2308, 186, 5, 36, "Output",
 CellID->408367039]
}, Open  ]],
Cell[83224, 2316, 125, 3, 70, "ExampleDelimiter",
 CellID->212108301],
Cell[83352, 2321, 143, 5, 70, "ExampleText",
 CellID->246308691],
Cell[CellGroupData[{
Cell[83520, 2330, 123, 3, 70, "Input",
 CellID->190823870],
Cell[83646, 2335, 163, 5, 36, "Output",
 CellID->10911257]
}, Open  ]],
Cell[83824, 2343, 92, 1, 70, "ExampleText",
 CellID->942555933],
Cell[CellGroupData[{
Cell[83941, 2348, 174, 5, 70, "Input",
 CellID->2985750],
Cell[84118, 2355, 164, 5, 36, "Output",
 CellID->151882719]
}, Open  ]],
Cell[84297, 2363, 125, 3, 70, "ExampleDelimiter",
 CellID->360520360],
Cell[84425, 2368, 77, 1, 70, "ExampleText",
 CellID->188236723],
Cell[CellGroupData[{
Cell[84527, 2373, 122, 3, 70, "Input",
 CellID->70810601],
Cell[84652, 2378, 163, 5, 36, "Output",
 CellID->465624813]
}, Open  ]],
Cell[84830, 2386, 61, 1, 70, "ExampleText",
 CellID->5065487],
Cell[CellGroupData[{
Cell[84916, 2391, 123, 3, 70, "Input",
 CellID->134129235],
Cell[85042, 2396, 163, 5, 36, "Output",
 CellID->29776333]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[85266, 2408, 227, 7, 70, "ExampleSubsection",
 CellID->72887098],
Cell[85496, 2417, 76, 1, 70, "ExampleText",
 CellID->6745604],
Cell[CellGroupData[{
Cell[85597, 2422, 118, 3, 70, "Input",
 CellID->324218402],
Cell[85718, 2427, 162, 5, 36, "Output",
 CellID->61347497]
}, Open  ]],
Cell[CellGroupData[{
Cell[85917, 2437, 153, 4, 70, "Input",
 CellID->196401469],
Cell[86073, 2443, 163, 5, 36, "Output",
 CellID->283557429]
}, Open  ]],
Cell[86251, 2451, 124, 3, 70, "ExampleDelimiter",
 CellID->84142534],
Cell[86378, 2456, 64, 1, 70, "ExampleText",
 CellID->17331403],
Cell[CellGroupData[{
Cell[86467, 2461, 159, 4, 70, "Input",
 CellID->329558717],
Cell[86629, 2467, 310, 10, 36, "Output",
 CellID->495829465]
}, Open  ]],
Cell[86954, 2480, 125, 3, 70, "ExampleDelimiter",
 CellID->225106620],
Cell[87082, 2485, 68, 1, 70, "ExampleText",
 CellID->38450493],
Cell[CellGroupData[{
Cell[87175, 2490, 152, 4, 70, "Input",
 CellID->49765797],
Cell[87330, 2496, 163, 5, 36, "Output",
 CellID->245598005]
}, Open  ]],
Cell[87508, 2504, 125, 3, 70, "ExampleDelimiter",
 CellID->276075381],
Cell[87636, 2509, 99, 1, 70, "ExampleText",
 CellID->124756331],
Cell[CellGroupData[{
Cell[87760, 2514, 286, 8, 70, "Input",
 CellID->185965031],
Cell[88049, 2524, 212, 7, 36, "Output",
 CellID->284595998]
}, Open  ]],
Cell[CellGroupData[{
Cell[88298, 2536, 153, 4, 70, "Input",
 CellID->326529512],
Cell[88454, 2542, 211, 7, 36, "Output",
 CellID->33143547]
}, Open  ]],
Cell[88680, 2552, 125, 3, 70, "ExampleDelimiter",
 CellID->392425290],
Cell[88808, 2557, 64, 1, 70, "ExampleText",
 CellID->12690591],
Cell[CellGroupData[{
Cell[88897, 2562, 160, 4, 70, "Input",
 CellID->85805936],
Cell[89060, 2568, 165, 5, 36, "Output",
 CellID->176543103]
}, Open  ]],
Cell[89240, 2576, 125, 3, 70, "ExampleDelimiter",
 CellID->137422354],
Cell[89368, 2581, 67, 1, 70, "ExampleText",
 CellID->434204360],
Cell[CellGroupData[{
Cell[89460, 2586, 162, 4, 70, "Input",
 CellID->35467110],
Cell[89625, 2592, 162, 5, 36, "Output",
 CellID->127317523]
}, Open  ]],
Cell[89802, 2600, 125, 3, 70, "ExampleDelimiter",
 CellID->144022931],
Cell[89930, 2605, 77, 1, 70, "ExampleText",
 CellID->48678768],
Cell[CellGroupData[{
Cell[90032, 2610, 176, 5, 70, "Input",
 CellID->500033543],
Cell[90211, 2617, 164, 5, 36, "Output",
 CellID->517476370]
}, Open  ]],
Cell[90390, 2625, 124, 3, 70, "ExampleDelimiter",
 CellID->13154962],
Cell[90517, 2630, 102, 2, 70, "ExampleText",
 CellID->385569970],
Cell[CellGroupData[{
Cell[90644, 2636, 285, 7, 70, "Input",
 CellID->106671],
Cell[90932, 2645, 5361, 92, 70, "Output",
 Evaluatable->False,
 CellID->120439119]
}, Open  ]],
Cell[96308, 2740, 125, 3, 70, "ExampleDelimiter",
 CellID->260027542],
Cell[96436, 2745, 123, 3, 70, "ExampleText",
 CellID->226668632],
Cell[CellGroupData[{
Cell[96584, 2752, 305, 8, 70, "Input",
 CellID->54403038],
Cell[96892, 2762, 175, 5, 36, "Output",
 CellID->72339279]
}, Open  ]],
Cell[97082, 2770, 124, 3, 70, "ExampleDelimiter",
 CellID->18459002]
}, Closed]],
Cell[CellGroupData[{
Cell[97243, 2778, 239, 7, 70, "ExampleSubsection",
 CellID->47965353],
Cell[97485, 2787, 69, 1, 70, "ExampleText",
 CellID->158686579],
Cell[CellGroupData[{
Cell[97579, 2792, 117, 3, 70, "Input",
 CellID->47890169],
Cell[97699, 2797, 163, 5, 36, "Output",
 CellID->108215363]
}, Open  ]],
Cell[97877, 2805, 125, 3, 70, "ExampleDelimiter",
 CellID->180309490],
Cell[98005, 2810, 75, 1, 70, "ExampleText",
 CellID->278238793],
Cell[CellGroupData[{
Cell[98105, 2815, 325, 11, 70, "Input",
 CellID->68140534],
Cell[98433, 2828, 721, 27, 54, "Output",
 CellID->538537496]
}, Open  ]],
Cell[99169, 2858, 70, 1, 70, "ExampleText",
 CellID->235460803],
Cell[CellGroupData[{
Cell[99264, 2863, 350, 11, 70, "Input",
 CellID->117134940],
Cell[99617, 2876, 245, 8, 36, "Output",
 CellID->217365398]
}, Open  ]],
Cell[99877, 2887, 124, 3, 70, "ExampleDelimiter",
 CellID->19845487],
Cell[100004, 2892, 72, 1, 70, "ExampleText",
 CellID->32229768],
Cell[CellGroupData[{
Cell[100101, 2897, 237, 7, 70, "Input",
 CellID->211214615],
Cell[100341, 2906, 719, 27, 54, "Output",
 CellID->4726197]
}, Open  ]],
Cell[101075, 2936, 125, 3, 70, "ExampleDelimiter",
 CellID->490111333],
Cell[101203, 2941, 203, 7, 70, "ExampleText",
 CellID->102397045],
Cell[CellGroupData[{
Cell[101431, 2952, 183, 5, 70, "Input",
 CellID->566637411],
Cell[101617, 2959, 2766, 49, 70, "Output",
 Evaluatable->False,
 CellID->360235877]
}, Open  ]],
Cell[104398, 3011, 97, 1, 70, "ExampleText",
 CellID->1034216],
Cell[CellGroupData[{
Cell[104520, 3016, 245, 7, 70, "Input",
 CellID->182543731],
Cell[104768, 3025, 5487, 94, 70, "Output",
 Evaluatable->False,
 CellID->201833203]
}, Open  ]],
Cell[110270, 3122, 279, 11, 70, "ExampleText",
 CellID->143893669],
Cell[CellGroupData[{
Cell[110574, 3137, 208, 6, 70, "Input",
 CellID->95991783],
Cell[110785, 3145, 5487, 94, 70, "Output",
 Evaluatable->False,
 CellID->274423287]
}, Open  ]],
Cell[116287, 3242, 125, 3, 70, "ExampleDelimiter",
 CellID->412064084],
Cell[116415, 3247, 115, 3, 70, "ExampleText",
 CellID->1036457487],
Cell[CellGroupData[{
Cell[116555, 3254, 293, 9, 70, "Input",
 CellID->65159324],
Cell[116851, 3265, 1968, 67, 88, "Output",
 CellID->147203428]
}, Open  ]],
Cell[118834, 3335, 61, 1, 70, "ExampleText",
 CellID->507504431],
Cell[CellGroupData[{
Cell[118920, 3340, 293, 9, 70, "Input",
 CellID->572331369],
Cell[119216, 3351, 1651, 57, 88, "Output",
 CellID->206942306]
}, Open  ]],
Cell[120882, 3411, 124, 3, 70, "ExampleDelimiter",
 CellID->69506496],
Cell[121009, 3416, 77, 1, 70, "ExampleText",
 CellID->402751654],
Cell[CellGroupData[{
Cell[121111, 3421, 229, 7, 70, "Input",
 CellID->25599404],
Cell[121343, 3430, 299, 11, 36, "Output",
 CellID->686459304]
}, Open  ]],
Cell[121657, 3444, 63, 1, 70, "ExampleText",
 CellID->401392888],
Cell[CellGroupData[{
Cell[121745, 3449, 256, 8, 70, "Input",
 CellID->193041198],
Cell[122004, 3459, 182, 6, 36, "Output",
 CellID->34008531]
}, Open  ]],
Cell[122201, 3468, 87, 1, 70, "ExampleText",
 CellID->30386929],
Cell[CellGroupData[{
Cell[122313, 3473, 279, 8, 70, "Input",
 CellID->572532350],
Cell[122595, 3483, 187, 6, 36, "Output",
 CellID->242598719]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[122831, 3495, 234, 7, 70, "ExampleSubsection",
 CellID->71471726],
Cell[123068, 3504, 65, 1, 70, "ExampleText",
 CellID->204052994],
Cell[CellGroupData[{
Cell[123158, 3509, 210, 6, 70, "Input",
 CellID->60522201],
Cell[123371, 3517, 3779, 66, 70, "Output",
 Evaluatable->False,
 CellID->274667918]
}, Open  ]],
Cell[127165, 3586, 63, 1, 70, "ExampleText",
 CellID->99209185],
Cell[CellGroupData[{
Cell[127253, 3591, 326, 9, 70, "Input",
 CellID->456871292],
Cell[127582, 3602, 4339, 75, 70, "Output",
 Evaluatable->False,
 CellID->12803459]
}, Open  ]],
Cell[131936, 3680, 52, 1, 70, "ExampleText",
 CellID->653599555],
Cell[CellGroupData[{
Cell[132013, 3685, 495, 15, 70, "Input",
 CellID->186058442],
Cell[132511, 3702, 5056, 87, 70, "Output",
 Evaluatable->False,
 CellID->125759440]
}, Open  ]],
Cell[137582, 3792, 125, 3, 70, "ExampleDelimiter",
 CellID->151322591],
Cell[137710, 3797, 74, 1, 70, "ExampleText",
 CellID->368987215],
Cell[CellGroupData[{
Cell[137809, 3802, 489, 14, 70, "Input",
 CellID->670556123],
Cell[138301, 3818, 7106, 121, 70, "Output",
 Evaluatable->False,
 CellID->141555012]
}, Open  ]],
Cell[145422, 3942, 125, 3, 70, "ExampleDelimiter",
 CellID->307643591],
Cell[145550, 3947, 93, 1, 70, "ExampleText",
 CellID->71892613],
Cell[CellGroupData[{
Cell[145668, 3952, 966, 28, 70, "Input",
 CellID->124995501],
Cell[146637, 3982, 6230, 106, 70, "Output",
 Evaluatable->False,
 CellID->85983140]
}, Open  ]],
Cell[CellGroupData[{
Cell[152904, 4093, 916, 27, 70, "Input",
 CellID->5101232],
Cell[153823, 4122, 2510, 45, 70, "Output",
 Evaluatable->False,
 CellID->135093639]
}, Open  ]],
Cell[CellGroupData[{
Cell[156370, 4172, 970, 28, 70, "Input",
 CellID->24200996],
Cell[157343, 4202, 2582, 46, 70, "Output",
 Evaluatable->False,
 CellID->59400019]
}, Open  ]],
Cell[159940, 4251, 125, 3, 70, "ExampleDelimiter",
 CellID->985935685]
}, Closed]],
Cell[CellGroupData[{
Cell[160102, 4259, 232, 7, 70, "ExampleSubsection",
 CellID->85828566],
Cell[160337, 4268, 70, 1, 70, "ExampleText",
 CellID->451483001],
Cell[CellGroupData[{
Cell[160432, 4273, 157, 4, 70, "Input",
 CellID->144800147],
Cell[160592, 4279, 166, 5, 36, "Output",
 CellID->17500742]
}, Open  ]],
Cell[160773, 4287, 124, 3, 70, "ExampleDelimiter",
 CellID->70698397],
Cell[160900, 4292, 97, 1, 70, "ExampleText",
 CellID->9502744],
Cell[CellGroupData[{
Cell[161022, 4297, 159, 4, 70, "Input",
 CellID->230900282],
Cell[161184, 4303, 167, 5, 36, "Output",
 CellID->63256558]
}, Open  ]],
Cell[161366, 4311, 125, 3, 70, "ExampleDelimiter",
 CellID->138311976],
Cell[161494, 4316, 63, 1, 70, "ExampleText",
 CellID->93081590],
Cell[CellGroupData[{
Cell[161582, 4321, 165, 4, 70, "Input",
 CellID->453802474],
Cell[161750, 4327, 164, 5, 36, "Output",
 CellID->348559992]
}, Open  ]],
Cell[161929, 4335, 125, 3, 70, "ExampleDelimiter",
 CellID->281422811],
Cell[162057, 4340, 96, 1, 70, "ExampleText",
 CellID->140895079],
Cell[CellGroupData[{
Cell[162178, 4345, 164, 4, 70, "Input",
 CellID->116609142],
Cell[162345, 4351, 163, 5, 36, "Output",
 CellID->499445470]
}, Open  ]],
Cell[162523, 4359, 125, 3, 70, "ExampleDelimiter",
 CellID->177329415],
Cell[162651, 4364, 93, 1, 70, "ExampleText",
 CellID->92044503],
Cell[CellGroupData[{
Cell[162769, 4369, 154, 4, 70, "Input",
 CellID->94552650],
Cell[162926, 4375, 163, 5, 36, "Output",
 CellID->145313565]
}, Open  ]],
Cell[163104, 4383, 72, 1, 70, "ExampleText",
 CellID->289858732],
Cell[CellGroupData[{
Cell[163201, 4388, 162, 4, 70, "Input",
 CellID->328471706],
Cell[163366, 4394, 160, 5, 36, "Output",
 CellID->455376]
}, Open  ]],
Cell[163541, 4402, 124, 3, 70, "ExampleDelimiter",
 CellID->76926374],
Cell[163668, 4407, 79, 1, 70, "ExampleText",
 CellID->30122779],
Cell[CellGroupData[{
Cell[163772, 4412, 156, 4, 70, "Input",
 CellID->574160085],
Cell[163931, 4418, 161, 5, 36, "Output",
 CellID->70885543]
}, Open  ]],
Cell[164107, 4426, 125, 3, 70, "ExampleDelimiter",
 CellID->143408027],
Cell[164235, 4431, 78, 1, 70, "ExampleText",
 CellID->70384877],
Cell[CellGroupData[{
Cell[164338, 4436, 210, 6, 70, "Input",
 CellID->64150158],
Cell[164551, 4444, 2201, 40, 70, "Output",
 Evaluatable->False,
 CellID->261546325]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[166801, 4490, 233, 7, 70, "ExampleSubsection",
 CellID->871755336],
Cell[167037, 4499, 124, 3, 70, "ExampleText",
 CellID->43443820],
Cell[CellGroupData[{
Cell[167186, 4506, 409, 12, 70, "Input",
 CellID->603077345],
Cell[167598, 4520, 175, 5, 36, "Output",
 CellID->709550607]
}, Open  ]],
Cell[167788, 4528, 138, 4, 70, "ExampleText",
 CellID->510284768],
Cell[CellGroupData[{
Cell[167951, 4536, 421, 12, 70, "Input",
 CellID->81549659],
Cell[168375, 4550, 175, 5, 36, "Output",
 CellID->41555066]
}, Open  ]],
Cell[168565, 4558, 87, 1, 70, "ExampleText",
 CellID->299693805],
Cell[CellGroupData[{
Cell[168677, 4563, 412, 12, 70, "Input",
 CellID->33503638],
Cell[169092, 4577, 176, 5, 36, "Output",
 CellID->932703993]
}, Open  ]],
Cell[169283, 4585, 125, 3, 70, "ExampleDelimiter",
 CellID->220175310],
Cell[169411, 4590, 134, 4, 70, "ExampleText",
 CellID->57534863],
Cell[CellGroupData[{
Cell[169570, 4598, 630, 18, 70, "Input",
 CellID->201574402],
Cell[170203, 4618, 175, 5, 36, "Output",
 CellID->111202725]
}, Open  ]],
Cell[170393, 4626, 125, 3, 70, "ExampleDelimiter",
 CellID->510092804],
Cell[170521, 4631, 113, 3, 70, "ExampleText",
 CellID->460617968],
Cell[CellGroupData[{
Cell[170659, 4638, 433, 11, 70, "Input",
 CellID->294789194],
Cell[171095, 4651, 174, 5, 36, "Output",
 CellID->531938029]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[171318, 4662, 237, 7, 70, "ExampleSubsection",
 CellID->454059883],
Cell[171558, 4671, 77, 1, 70, "ExampleText",
 CellID->53133309],
Cell[CellGroupData[{
Cell[171660, 4676, 120, 3, 70, "Input",
 CellID->10775634],
Cell[171783, 4681, 164, 5, 36, "Output",
 CellID->67411612]
}, Open  ]],
Cell[CellGroupData[{
Cell[171984, 4691, 121, 3, 70, "Input",
 CellID->279976511],
Cell[172108, 4696, 164, 5, 36, "Output",
 CellID->15625594]
}, Open  ]],
Cell[172287, 4704, 125, 3, 70, "ExampleDelimiter",
 CellID->831960422],
Cell[172415, 4709, 76, 1, 70, "ExampleText",
 CellID->142119902],
Cell[CellGroupData[{
Cell[172516, 4714, 218, 7, 70, "Input",
 CellID->54000930],
Cell[172737, 4723, 395, 10, 36, "Output",
 CellID->149416337]
}, Open  ]],
Cell[173147, 4736, 124, 3, 70, "ExampleDelimiter",
 CellID->63661671],
Cell[173274, 4741, 93, 1, 70, "ExampleText",
 CellID->13041951],
Cell[CellGroupData[{
Cell[173392, 4746, 219, 7, 70, "Input",
 CellID->560901108],
Cell[173614, 4755, 449, 10, 54, "Output",
 CellID->101961137]
}, Open  ]],
Cell[174078, 4768, 125, 3, 70, "ExampleDelimiter",
 CellID->446995658],
Cell[174206, 4773, 86, 1, 70, "ExampleText",
 CellID->33817476],
Cell[CellGroupData[{
Cell[174317, 4778, 221, 7, 70, "Input",
 CellID->6430592],
Cell[174541, 4787, 456, 11, 54, "Output",
 CellID->769824777]
}, Open  ]],
Cell[175012, 4801, 105, 2, 70, "ExampleText",
 CellID->402144433],
Cell[CellGroupData[{
Cell[175142, 4807, 199, 6, 70, "Input",
 CellID->241470852],
Cell[175344, 4815, 162, 5, 36, "Output",
 CellID->268388356]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[175555, 4826, 234, 7, 70, "ExampleSubsection",
 CellID->374873301],
Cell[175792, 4835, 114, 3, 70, "ExampleText",
 CellID->122635831],
Cell[CellGroupData[{
Cell[175931, 4842, 403, 12, 70, "Input",
 CellID->49952296],
Cell[176337, 4856, 327, 9, 36, "Output",
 CellID->271633376]
}, Open  ]],
Cell[CellGroupData[{
Cell[176701, 4870, 188, 6, 70, "Input",
 CellID->138084224],
Cell[176892, 4878, 455, 11, 54, "Output",
 CellID->71661753]
}, Open  ]],
Cell[177362, 4892, 123, 3, 70, "ExampleDelimiter",
 CellID->3656482],
Cell[177488, 4897, 102, 2, 70, "ExampleText",
 CellID->110245438],
Cell[CellGroupData[{
Cell[177615, 4903, 359, 11, 70, "Input",
 CellID->26927731],
Cell[177977, 4916, 223, 7, 36, "Output",
 CellID->344368164]
}, Open  ]],
Cell[CellGroupData[{
Cell[178237, 4928, 186, 6, 70, "Input",
 CellID->6883866],
Cell[178426, 4936, 265, 8, 36, "Output",
 CellID->74253831]
}, Open  ]],
Cell[178706, 4947, 124, 3, 70, "ExampleDelimiter",
 CellID->98966124],
Cell[178833, 4952, 95, 1, 70, "ExampleText",
 CellID->12442663],
Cell[CellGroupData[{
Cell[178953, 4957, 493, 15, 70, "Input",
 CellID->251992734],
Cell[179449, 4974, 436, 11, 36, "Output",
 CellID->87597851]
}, Open  ]],
Cell[CellGroupData[{
Cell[179922, 4990, 187, 6, 70, "Input",
 CellID->24965078],
Cell[180112, 4998, 693, 15, 105, "Output",
 CellID->451336175]
}, Open  ]],
Cell[180820, 5016, 125, 3, 70, "ExampleDelimiter",
 CellID->253526391],
Cell[180948, 5021, 135, 4, 70, "ExampleText",
 CellID->92272593],
Cell[CellGroupData[{
Cell[181108, 5029, 358, 11, 70, "Input",
 CellID->148009114],
Cell[181469, 5042, 406, 10, 54, "Output",
 CellID->194350598]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[181924, 5058, 239, 7, 70, "ExampleSubsection",
 CellID->10369721],
Cell[182166, 5067, 78, 1, 70, "ExampleText",
 CellID->12647730],
Cell[CellGroupData[{
Cell[182269, 5072, 126, 3, 70, "Input",
 CellID->590075703],
Cell[182398, 5077, 3575, 60, 190, "Output",
 CellID->73559336]
}, Open  ]],
Cell[185988, 5140, 124, 3, 70, "ExampleDelimiter",
 CellID->78454681],
Cell[186115, 5145, 97, 1, 70, "ExampleText",
 CellID->9684183],
Cell[CellGroupData[{
Cell[186237, 5150, 121, 3, 70, "Input",
 CellID->9944848],
Cell[186361, 5155, 164, 5, 36, "Output",
 CellID->132385911]
}, Open  ]],
Cell[186540, 5163, 81, 1, 70, "ExampleText",
 CellID->229798417],
Cell[CellGroupData[{
Cell[186646, 5168, 175, 5, 70, "Input",
 CellID->75280672],
Cell[186824, 5175, 163, 5, 36, "Output",
 CellID->29819732]
}, Open  ]],
Cell[187002, 5183, 125, 3, 70, "ExampleDelimiter",
 CellID->408339838],
Cell[187130, 5188, 115, 3, 70, "ExampleText",
 CellID->151252491],
Cell[CellGroupData[{
Cell[187270, 5195, 122, 3, 70, "Input",
 CellID->10801005],
Cell[187395, 5200, 163, 5, 36, "Output",
 CellID->302367457]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[187619, 5212, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[187849, 5221, 89, 1, 70, "ExampleText",
 CellID->522670720],
Cell[CellGroupData[{
Cell[187963, 5226, 481, 14, 70, "Input",
 CellID->99878051],
Cell[188447, 5242, 3839, 67, 70, "Output",
 Evaluatable->False,
 CellID->52970150]
}, Open  ]],
Cell[192301, 5312, 124, 3, 70, "ExampleDelimiter",
 CellID->22398025],
Cell[192428, 5317, 115, 3, 70, "ExampleText",
 CellID->56114065],
Cell[CellGroupData[{
Cell[192568, 5324, 381, 12, 70, "Input",
 CellID->143688033],
Cell[192952, 5338, 2139, 44, 144, "Output",
 CellID->121433221]
}, Open  ]],
Cell[195106, 5385, 125, 3, 70, "ExampleDelimiter",
 CellID->238838549],
Cell[195234, 5390, 124, 3, 70, "ExampleText",
 CellID->96863732],
Cell[CellGroupData[{
Cell[195383, 5397, 585, 15, 70, "Input",
 CellID->81022563],
Cell[195971, 5414, 6260, 107, 70, "Output",
 Evaluatable->False,
 CellID->750832308]
}, Open  ]],
Cell[202246, 5524, 125, 3, 70, "ExampleDelimiter",
 CellID->571789222],
Cell[202374, 5529, 143, 4, 70, "ExampleText",
 CellID->57888114],
Cell[CellGroupData[{
Cell[202542, 5537, 753, 22, 70, "Input",
 CellID->56078898],
Cell[203298, 5561, 156873, 2620, 70, "Output",
 CellID->254856070]
}, Open  ]],
Cell[360186, 8184, 125, 3, 70, "ExampleDelimiter",
 CellID->434753880],
Cell[360314, 8189, 145, 4, 70, "ExampleText",
 CellID->775102060],
Cell[CellGroupData[{
Cell[360484, 8197, 539, 15, 70, "Input",
 CellID->4537665],
Cell[361026, 8214, 2087, 42, 141, "Output",
 CellID->201941089]
}, Open  ]],
Cell[363128, 8259, 125, 3, 70, "ExampleDelimiter",
 CellID->354303428],
Cell[363256, 8264, 121, 3, 70, "ExampleText",
 CellID->404604247],
Cell[CellGroupData[{
Cell[363402, 8271, 1049, 31, 70, "Input",
 CellID->719098037],
Cell[364454, 8304, 4447, 105, 303, "Output",
 CellID->215204516]
}, Open  ]],
Cell[368916, 8412, 126, 3, 70, "ExampleDelimiter",
 CellID->1028363587],
Cell[369045, 8417, 137, 4, 70, "ExampleText",
 CellID->76722058],
Cell[CellGroupData[{
Cell[369207, 8425, 531, 15, 70, "Input",
 CellID->283057930],
Cell[369741, 8442, 1816, 38, 123, "Output",
 CellID->450381436]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[371606, 8486, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[371847, 8495, 122, 3, 70, "ExampleText",
 CellID->560639371],
Cell[CellGroupData[{
Cell[371994, 8502, 119, 3, 70, "Input",
 CellID->484418457],
Cell[372116, 8507, 162, 5, 36, "Output",
 CellID->24159429]
}, Open  ]],
Cell[CellGroupData[{
Cell[372315, 8517, 121, 3, 70, "Input",
 CellID->607678962],
Cell[372439, 8522, 163, 5, 36, "Output",
 CellID->257936795]
}, Open  ]],
Cell[CellGroupData[{
Cell[372639, 8532, 123, 3, 70, "Input",
 CellID->135458308],
Cell[372765, 8537, 164, 5, 36, "Output",
 CellID->112325819]
}, Open  ]],
Cell[372944, 8545, 115, 3, 70, "ExampleText",
 CellID->761357570],
Cell[CellGroupData[{
Cell[373084, 8552, 193, 5, 70, "Input",
 CellID->84837171],
Cell[373280, 8559, 176, 5, 36, "Output",
 CellID->373922157]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[373505, 8570, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[373739, 8579, 122, 3, 70, "ExampleText",
 CellID->163621777],
Cell[CellGroupData[{
Cell[373886, 8586, 151, 4, 70, "Input",
 CellID->569611999],
Cell[374040, 8592, 210, 6, 36, "Output",
 CellID->72074793]
}, Open  ]],
Cell[374265, 8601, 125, 3, 70, "ExampleDelimiter",
 CellID->174359581],
Cell[374393, 8606, 219, 7, 70, "ExampleText",
 CellID->342256626],
Cell[CellGroupData[{
Cell[374637, 8617, 118, 3, 70, "Input",
 CellID->19576045],
Cell[374758, 8622, 392, 7, 70, "Message",
 CellID->48956927],
Cell[375153, 8631, 208, 6, 36, "Output",
 CellID->510173495]
}, Open  ]],
Cell[375376, 8640, 116, 3, 70, "ExampleText",
 CellID->233017874],
Cell[CellGroupData[{
Cell[375517, 8647, 121, 3, 70, "Input",
 CellID->729483323],
Cell[375641, 8652, 162, 5, 36, "Output",
 CellID->91322303]
}, Open  ]],
Cell[375818, 8660, 125, 3, 70, "ExampleDelimiter",
 CellID->121223470],
Cell[375946, 8665, 124, 3, 70, "ExampleText",
 CellID->174677099],
Cell[CellGroupData[{
Cell[376095, 8672, 121, 3, 70, "Input",
 CellID->298282296],
Cell[376219, 8677, 163, 5, 36, "Output",
 CellID->351349438]
}, Open  ]],
Cell[376397, 8685, 125, 3, 70, "ExampleDelimiter",
 CellID->813057292],
Cell[376525, 8690, 93, 1, 70, "ExampleText",
 CellID->623115820],
Cell[CellGroupData[{
Cell[376643, 8695, 229, 7, 70, "Input",
 CellID->899477076],
Cell[376875, 8704, 210, 6, 36, "Output",
 CellID->70775367]
}, Open  ]],
Cell[377100, 8713, 96, 1, 70, "ExampleText",
 CellID->77428252],
Cell[CellGroupData[{
Cell[377221, 8718, 218, 7, 70, "Input",
 CellID->85641797],
Cell[377442, 8727, 2874, 107, 139, "Output",
 CellID->55181212]
}, Open  ]],
Cell[380331, 8837, 125, 3, 70, "ExampleDelimiter",
 CellID->140140145],
Cell[380459, 8842, 85, 1, 70, "ExampleText",
 CellID->281263559],
Cell[CellGroupData[{
Cell[380569, 8847, 157, 4, 70, "Input",
 CellID->62622579],
Cell[380729, 8853, 163, 5, 36, "Output",
 CellID->615049915]
}, Open  ]],
Cell[380907, 8861, 116, 3, 70, "ExampleText",
 CellID->983404076],
Cell[CellGroupData[{
Cell[381048, 8868, 352, 10, 70, "Input",
 CellID->128956369],
Cell[381403, 8880, 162, 5, 36, "Output",
 CellID->68184022]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[381614, 8891, 228, 7, 70, "ExampleSection",
 CellID->589267740],
Cell[381845, 8900, 127, 3, 70, "ExampleText",
 CellID->559244587],
Cell[CellGroupData[{
Cell[381997, 8907, 642, 18, 70, "Input",
 CellID->462795108],
Cell[382642, 8927, 3006, 53, 70, "Output",
 Evaluatable->False,
 CellID->532817396]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[385709, 8987, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[386028, 8998, 469, 17, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[386534, 9020, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[386855, 9031, 299, 9, 70, "Tutorials",
 CellID->341631938]
}, Open  ]],
Cell[CellGroupData[{
Cell[387191, 9045, 319, 9, 70, "MoreAboutSection",
 CellID->149354000],
Cell[387513, 9056, 150, 3, 70, "MoreAbout",
 CellID->147604963],
Cell[387666, 9061, 185, 3, 70, "MoreAbout",
 CellID->549668142]
}, Open  ]],
Cell[387866, 9067, 27, 0, 70, "History"],
Cell[387896, 9069, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

