(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    402170,       8628]
NotebookOptionsPosition[    386833,       8106]
NotebookOutlinePosition[    388910,       8155]
CellTagsIndexPosition[    388823,       8150]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Unconstrained Optimization" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/UnconstrainedOptimizationOverview"], 
          "Numerical Mathematics: Basic Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalMathematics-BasicOperations"], 
          "Numerical Optimization" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalOptimization"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Unconstrained Optimization\"\>", 
       2->"\<\"Numerical Mathematics: Basic Operations\"\>", 
       3->"\<\"Numerical Optimization\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"FindMaximum" :> 
          Documentation`HelpLookup["paclet:ref/FindMaximum"], "NMinimize" :> 
          Documentation`HelpLookup["paclet:ref/NMinimize"], "Minimize" :> 
          Documentation`HelpLookup["paclet:ref/Minimize"], "FindFit" :> 
          Documentation`HelpLookup["paclet:ref/FindFit"], "FindRoot" :> 
          Documentation`HelpLookup["paclet:ref/FindRoot"], "Min" :> 
          Documentation`HelpLookup["paclet:ref/Min"], "LinearProgramming" :> 
          Documentation`HelpLookup["paclet:ref/LinearProgramming"], "D" :> 
          Documentation`HelpLookup["paclet:ref/D"], "CholeskyDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/CholeskyDecomposition"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"FindMaximum\"\>", 
       2->"\<\"NMinimize\"\>", 3->"\<\"Minimize\"\>", 4->"\<\"FindFit\"\>", 
       5->"\<\"FindRoot\"\>", 6->"\<\"Min\"\>", 
       7->"\<\"LinearProgramming\"\>", 8->"\<\"D\"\>", 
       9->"\<\"CholeskyDecomposition\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Mathematics" :> 
          Documentation`HelpLookup["paclet:guide/DiscreteMathematics"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Discrete Mathematics\"\>", 
       2->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FindMinimum", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMinimum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMinimum"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["0", "TR"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]searches for a local minimum in ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     ", starting from the point ",
     Cell[BoxData[
      RowBox[{
       StyleBox["x", "TI"], "=", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["0", "TR"]]}]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMinimum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMinimum"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            StyleBox["x", "TI"], ",", 
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            StyleBox["y", "TI"], ",", 
            SubscriptBox[
             StyleBox["y", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]searches for a local minimum in a function of several \
variables. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMinimum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMinimum"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["cons", "TI"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            StyleBox["x", "TI"], ",", 
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            StyleBox["y", "TI"], ",", 
            SubscriptBox[
             StyleBox["y", "TI"], 
             StyleBox["0", "TR"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]searches for a local minimum subject to the \
constraints ",
     Cell[BoxData[
      StyleBox["cons", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FindMinimum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FindMinimum"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["cons", "TI"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          StyleBox["y", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]starts from a point within the region defined by the \
constraints."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->27950]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " returns a list of the form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    SubscriptBox[
     StyleBox["f", "TI"], 
     StyleBox["min", "TI"]], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], "->", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["min", "TI"]]}], "}"}]}], "}"}]], "InlineFormula"],
 ", where ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["f", "TI"], 
   StyleBox["min", "TI"]]], "InlineFormula"],
 " is the minimum value of ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " found, and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["min", "TI"]]], "InlineFormula"],
 " is the value of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " for which it is found. "
}], "Notes",
 CellID->1268],

Cell["\<\
If the starting point for a variable is given as a list, the values of the \
variable are taken to be lists with the same dimensions. \
\>", "Notes",
 CellID->25479],

Cell[TextData[{
 "The constraints ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 " can contain equations, inequalities or logical combinations of these. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->19298],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " first localizes the values of all variables, then evaluates ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " with the variables being symbolic, and then repeatedly evaluates the \
result numerically."
}], "Notes",
 CellID->7438],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " has attribute ",
 Cell[BoxData[
  ButtonBox["HoldAll",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HoldAll"]], "InlineFormula"],
 ", and effectively uses ",
 Cell[BoxData[
  ButtonBox["Block",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Block"]], "InlineFormula"],
 " to localize variables."
}], "Notes",
 CellID->23526],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FindMinimum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMinimum"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]]}], "}"}]}], "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " searches for a local minimum in ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " using ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["0", "TR"]]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["1", "TR"]]], "InlineFormula"],
 " as the first two values of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ", avoiding the use of derivatives. "
}], "Notes",
 CellID->23431],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FindMinimum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMinimum"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["min", "TI"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " searches for a local minimum, stopping the search if ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " ever gets outside the range ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["min", "TI"]]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["max", "TI"]]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->27865],

Cell[TextData[{
 "Except when ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 " are both linear, the results found by ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " may correspond only to local, but not global, minima. "
}], "Notes",
 CellID->31288],

Cell["By default, all variables are assumed to be real. ", "Notes",
 CellID->23646],

Cell[TextData[{
 "For linear ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["cons", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "\[Element]", 
   ButtonBox["Integers",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Integers"]}]], "InlineFormula"],
 " can be used to specify that a variable can take on only integer values. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->21374],

Cell["The following options can be given: ", "Notes",
 CellID->30796],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the accuracy sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["EvaluationMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/EvaluationMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "expression to evaluate whenever ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " is evaluated "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "Gradient", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "the list of gradient functions ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["x", "TI"]}], "]"}], ",", 
        RowBox[{"D", "[", 
         RowBox[{
          StyleBox["f", "TI"], ",", 
          StyleBox["y", "TI"]}], "]"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["MaxIterations",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxIterations"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum number of iterations to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the precision sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["StepMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/StepMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "expression to evaluate whenever a step is taken ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "the precision used in internal computations ", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "3ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True}}}},
 CellID->3307],

Cell[TextData[{
 "The default settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " are ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["WorkingPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WorkingPrecision"], "/", "2"}]], 
  "InlineFormula"],
 ". "
}], "Notes",
 CellID->24106],

Cell[TextData[{
 "The settings for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " specify the number of digits to seek in both the value of the position of \
the minimum, and the value of the function at the minimum. "
}], "Notes",
 CellID->18946],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " continues until either of the goals specified by ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " is achieved. "
}], "Notes",
 CellID->237],

Cell[TextData[{
 "Possible settings for ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " include ",
 Cell[BoxData["\"\<ConjugateGradient\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Gradient\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<LevenbergMarquardt\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Newton\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<QuasiNewton\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<InteriorPoint\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<LinearProgramming\>\""], "InlineFormula"],
 ", with the default being ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->547912957]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->9409493],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->263648285],

Cell[TextData[{
 "Find a local minimum starting the search at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "2"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->499988759],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->376441841],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "3.2883713955908966`"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "3.425618459492147`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->19645436]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->347120420],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->65591389]
}, Open  ]],

Cell[TextData[{
 "Extract the value of ",
 Cell[BoxData["x"], "InlineFormula"],
 " at the local minimum:"
}], "ExampleText",
 CellID->452807755],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "/.", 
  RowBox[{"Last", "[", 
   RowBox[{"FindMinimum", "[", 
    RowBox[{
     RowBox[{"x", " ", 
      RowBox[{"Cos", "[", "x", "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->28565156],

Cell[BoxData["3.425618459492147`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->53541514]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->986451359],

Cell[TextData[{
 "Find a local minimum, starting at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "7"}], TraditionalForm]], "InlineMath"],
 ", subject to constraints ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "\[LessEqual]", " ", "x", " ", "\[LessEqual]", " ", "15"}], 
   TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1560461396],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", " ", 
      RowBox[{"Cos", "[", "x", "]"}]}], ",", 
     RowBox[{"1", "\[LessEqual]", "x", "\[LessEqual]", "15"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "7"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->678564151],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "9.477294259479795`"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "9.529334406144022`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->18197748]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1519398755],

Cell["\<\
Find the minimum of a linear function, subject to linear and integer \
constraints:\
\>", "ExampleText",
 CellID->1896901598],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "3"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[Element]", "Integers"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1202084729],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.`"}], ",", 
     RowBox[{"y", "\[Rule]", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{136, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->104164648]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26449],

Cell["\<\
With different starting points you may get different local minima:\
\>", "ExampleText",
 CellID->62007641],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->66723803],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "3.2883713955908966`"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "3.425618459480375`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->599965091]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->386171924],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "9.477294259479795`"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "9.529334405361269`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->134194003]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->666042395],

Cell[TextData[{
 "Local minimum of a 2-variable function starting from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "2"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "=", "2"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->516388244],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", " ", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "2"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->46634681],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "2.170803763674703`"}], ",", 
     RowBox[{"y", "\[Rule]", "2.170803763674703`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{207, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->463680148]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->242324886],

Cell["Local minimum constrained within a disk:", "ExampleText",
 CellID->681774368],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"x", " ", "y"}], "]"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "3"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "2"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->852578689],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.9974950175794361`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.2247450501615211`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "1.2247450501615211`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{270, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->655892637]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10112781],

Cell["Starting point does not have to be provided:", "ExampleText",
 CellID->369156266],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"x", " ", "y"}], "]"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "3"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->59114888],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.9974950017768699`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.2247449589659993`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "1.2247449589523258`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{270, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->475825784]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->685666782],

Cell["\<\
For linear objective and constraints, integer constraints can be imposed:\
\>", "ExampleText",
 CellID->169051774],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"3", "x"}], "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "7"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1991020309],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.3333333333333335`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "2.3333333333333335`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6104705]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"3", "x"}], "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "7"}], "&&", 
      RowBox[{"x", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"x", "\[Element]", "Integers"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->833915335],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.5`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "2"}], ",", 
     RowBox[{"y", "\[Rule]", "0.5`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{150, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->29511136]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5114145],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Or",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Or"]], "InlineFormula"],
 " constraints can be specified:"
}], "ExampleText",
 CellID->284316099],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}], "||", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "+", "2"}], ")"}], "^", "2"}], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"y", "+", "2"}], ")"}], "^", "2"}]}], "\[LessEqual]", 
       "1"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1963030529],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "5.414213562385342`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "2.707106781192671`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "2.707106781192671`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{271, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->423130414]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->7079],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "AccuracyGoal & PrecisionGoal",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->25956],

Cell[TextData[{
 "This enforces a convergence criteria ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     SubscriptBox["x", "k"], "-", 
     SuperscriptBox["x", "*"]}], "||", " ", 
    RowBox[{"\[LessEqual]", " ", 
     RowBox[{"max", "(", 
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "9"}]], ",", 
       RowBox[{
        SuperscriptBox["10", 
         RowBox[{"-", "8"}]], "||", 
        SubscriptBox["x", "k"], "||"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[EmptyDownTriangle]f", "(", 
     SubscriptBox["x", "k"], ")"}], "\[LessEqual]", 
    SuperscriptBox["10", 
     RowBox[{"-", "9"}]]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->764461149],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", "/", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1"}], "}"}], ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "9"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "8"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1625295724],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.`"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "3.1415926535897953`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{145, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->545725341]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1055000278],

Cell[TextData[{
 "This enforces a convergence criteria ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     SubscriptBox["x", "k"], "-", 
     SuperscriptBox["x", "*"]}], "||", " ", 
    RowBox[{"\[LessEqual]", " ", 
     RowBox[{"max", "(", 
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "20"}]], ",", 
       RowBox[{
        SuperscriptBox["10", 
         RowBox[{"-", "18"}]], "||", 
        SubscriptBox["x", "k"], "||"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[EmptyDownTriangle]f", "(", 
     SubscriptBox["x", "k"], ")"}], "\[LessEqual]", 
    SuperscriptBox["10", 
     RowBox[{"-", "20"}]]}], TraditionalForm]], "InlineMath"],
 ", which is not achievable with the default machine precision computation:"
}], "ExampleText",
 CellID->1329553122],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", "/", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1"}], "}"}], ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "20"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "18"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1468387723],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"fmdig\"\>"}], ":", 
  " ", "\<\"\\!\\(MachinePrecision\\) working digits is insufficient to \
achieve the requested accuracy or precision. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/FindMinimum/fmdig\\\", ButtonNote -> \
\\\"FindMinimum::fmdig\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->476395254],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.`"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "3.141592653589793`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{145, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->445600503]
}, Open  ]],

Cell[TextData[{
 "Setting a high ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 " makes the process convergent:"
}], "ExampleText",
 CellID->1199314865],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", "/", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1"}], "}"}], ",", 
   RowBox[{"AccuracyGoal", "\[Rule]", "20"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "18"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "40"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->39134653],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
   "-", "0.9999999999999999999999999999999999999999999999999999999924`40."}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
     "-", "3.1415926535897932384626433830324952267597214891697646653845`40."}]\
}], "}"}]}], "}"}]], "Output",
 ImageSize->{350, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->74943998]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "EvaluationMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->31705],

Cell["Plot convergence to the local minimum:", "ExampleText",
 CellID->28034064],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Last", "[", 
   RowBox[{"Reap", "[", 
    RowBox[{"FindMinimum", "[", 
     RowBox[{
      RowBox[{"x", " ", 
       RowBox[{"Cos", "[", "x", "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "2"}], "}"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19607354],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], 
    PointBox[{{1., 2.}, {2., 3.}, {3., 4.720685606045544}, {4., 
     3.370059693787118}, {5., 3.4349100179215992`}, {6., 
     3.4254835803311354`}, {7., 3.4256181498088467`}, {8., 
     3.425618459492147}}]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 2.},
  ImageMargins->0.,
  ImageSize->Small,
  PlotRange->{{0, 8.}, {2., 4.720685606045544}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->36865338]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Gradient",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->12053],

Cell["\<\
This uses the given gradient, and computes the Hessian using finite \
differences of the gradient:\
\>", "ExampleText",
 CellID->858592029],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", " ", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Gradient", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"y", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}], ",", " ", 
      RowBox[{"x", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}]}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<Newton\>\""}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2076542040],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.60711446908485`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "0.9774016456312932`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{235, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->154322767]
}, Open  ]],

Cell["This uses the supplied gradient and Hessian:", "ExampleText",
 CellID->1859244329],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", " ", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"Gradient", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"y", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}], ",", " ", 
      RowBox[{"x", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}]}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<Newton\>\"", ",", 
      RowBox[{"Hessian", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"-", 
             RowBox[{"y", "^", "2"}]}], 
            RowBox[{"Sin", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}], ",", 
           RowBox[{
            RowBox[{"-", " ", "x"}], " ", "y", " ", 
            RowBox[{"Cos", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "x"}], " ", "y", " ", 
            RowBox[{"Cos", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}], ",", " ", 
           RowBox[{
            RowBox[{"-", 
             RowBox[{"x", "^", "2"}]}], 
            RowBox[{"Sin", "[", 
             RowBox[{"x", " ", "y"}], "]"}]}]}], "}"}]}], "}"}]}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1427703802],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "8.118188590213629`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "11.028986285836702`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->207428351]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->16079],

Cell["\<\
In this case the default derivative-based methods have difficulties:\
\>", "ExampleText",
 CellID->993924491],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1"}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1.01"}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"y", "+", "1"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->782571855],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"lstol\"\>"}], ":", 
  " ", "\<\"The line search decreased the step size to within tolerance \
specified by AccuracyGoal and PrecisionGoal but was unable to find a \
sufficient decrease in the function.  You may need more than \
\\!\\(MachinePrecision\\) digits of working precision to meet these \
tolerances. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/FindMinimum/lstol\\\", ButtonNote -> \
\\\"FindMinimum::lstol\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->440544294],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.01000000452670391`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.0068264489265601`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.000000004526704`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{207, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->40644518]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1"}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1.01"}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"y", "+", "1"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<ConjugateGradient\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->939473422],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"sdprec\"\>"}], ":", 
  " ", "\<\"Line search unable to find a sufficient decrease in the function \
value with \\!\\(MachinePrecision\\) digit precision.\"\>"}]], "Message", \
"MSG",
 CellID->290095149],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.7373153897224387`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.6363423052199573`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.0000000001623532`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{242, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->85583272]
}, Open  ]],

Cell["\<\
Direct search methods which do not require derivatives can be helpful in \
these cases:\
\>", "ExampleText",
 CellID->1614271486],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1"}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"x", "+", "1.01"}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"y", "+", "1"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<PrincipalAxis\>\""}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->736570868],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.010000000000000009`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.0067107133234492`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{207, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->73519019]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " also uses a range of direct search methods:"
}], "ExampleText",
 CellID->1954260595],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Abs", "[", 
       RowBox[{"x", "+", "1"}], "]"}], "+", 
      RowBox[{"Abs", "[", 
       RowBox[{"x", "+", "1.01"}], "]"}], "+", 
      RowBox[{"Abs", "[", 
       RowBox[{"y", "+", "1"}], "]"}]}], ",", 
     RowBox[{"0", ">", "x", ">", 
      RowBox[{"-", "2"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->46889232],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.010000000037185819`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "1.0018199202089693`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.0000000000371858`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{207, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->43239344]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "StepMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->2025],

Cell[TextData[{
 "Steps taken by ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " in finding the minimum of the classic Rosenbrock function:"
}], "ExampleText",
 CellID->1691254454],

Cell[BoxData[{
 RowBox[{
  RowBox[{"pts", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"FindMinimum", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"1", "-", "x"}], ")"}], "^", "2"}], "+", 
        RowBox[{"100", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", 
             RowBox[{"x", "^", "2"}]}], "-", "y"}], ")"}], "^", "2"}]}]}], 
       ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"x", ",", 
           RowBox[{"-", "1.2"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"y", ",", "1"}], "}"}]}], "}"}], ",", 
       RowBox[{"StepMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"pts", " ", "=", 
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.2"}], ",", "1"}], "}"}], "}"}], ",", "pts"}], "]"}]}],
   ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1514338379],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", "x"}], ")"}], "^", "2"}], "+", 
    RowBox[{"100", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{"x", "^", "2"}]}], "-", "y"}], ")"}], "^", "2"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1.3"}], ",", "1.5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1.5"}], ",", "1.4"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Arrow", "[", "pts", "]"}], ",", 
      RowBox[{"Point", "[", "pts", "]"}]}], "}"}]}], ",", 
   RowBox[{"Contours", "\[Rule]", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"10", "^", 
       RowBox[{"-", "i"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{"-", "2"}], ",", "10"}], "}"}]}], "]"}]}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Hue", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Log", "[", 
           RowBox[{"10", ",", "#"}], "]"}], "+", "10"}], ")"}], "/", "12"}], 
       "]"}], "&"}], ")"}]}], ",", 
   RowBox[{"ColorFunctionScaling", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1350533918],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJxknQm4zsUXx+37WpZU1siWVAqRBiFE+duiKHuIUAkpIXtCsmTfsmSpZBfG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   "], {{{}, {}, {}, {}, {}, {}, {}, {}, 
     {Hue[0.3950302241245204], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{3693, 4678, 1968}, {4702, 4704, 1985}, {4515, 5312, 
         1825}, {4865, 4867, 2152}, {4102, 4700, 1290}, {4501, 4000, 1144}, {
         5026, 3693, 1968}, {4871, 4868, 2152}, {4700, 5050, 1290}, {4138, 
         4901, 1315}, {4034, 4857, 1985}, {4670, 4140, 1315}, {4719, 4515, 
         1825}, {3598, 4506, 1626}, {4519, 4001, 1144}, {4868, 5309, 2152}, {
         4506, 4300, 1626}, {4495, 4299, 1626}, {4854, 4852, 2141}, {4140, 
         4792, 1315}, {4514, 4719, 1825}, {4324, 4519, 1144}, {4873, 4670, 
         1315}, {4797, 3716, 1967}, {4792, 4138, 1315}, {4104, 4617, 1290}, {
         5052, 4702, 1985}, {4300, 4508, 1626}, {4861, 4676, 1968}, {4001, 
         4539, 1144}, {4795, 4674, 1967}, {4672, 4797, 1967}, {4804, 4671, 
         1967}, {4298, 4102, 1290}, {4517, 4717, 1825}, {4674, 4804, 1967}, {
         4000, 4503, 1144}, {4539, 3998, 1144}, {4118, 4512, 1823}, {4678, 
         4861, 1968}, {4499, 4106, 1291}, {5311, 4873, 1315}, {4105, 4499, 
         1291}, {4704, 4034, 1985}, {3998, 4501, 1144}, {4852, 4853, 2141}, {
         4507, 4108, 1291}, {4867, 3794, 2152}, {5309, 4865, 2152}, {3794, 
         4871, 2152}, {4106, 4493, 1291}, {4510, 4118, 1823}, {4299, 3598, 
         1626}, {4717, 4514, 1825}, {3716, 4795, 1967}, {4503, 4324, 1144}, {
         4903, 4517, 1825}},
         VertexColors->None], 
        PolygonBox[{{4617, 4854, 2141, 1290}, {5050, 5052, 1985, 1290}, {4508,
          4507, 1291, 1626}, {4671, 5026, 1968, 1967}, {4857, 4104, 1290, 
         1985}, {4853, 4298, 1290, 2141}, {4493, 4495, 1626, 1291}, {5312, 
         5311, 1315, 1825}, {4676, 4672, 1967, 1968}, {4108, 4510, 1823, 
         1291}, {4901, 4903, 1825, 1315}, {4512, 4105, 1291, 1823}},
         VertexColors->None]}]}, 
     {Hue[0.311696890791187], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlmdsVlUcxg/jbd+31LfvvbdllCpDlgwLxoQhagIpfDAqBUEkURIiwxZB
gQ8alQCxGLYrEVxxkTgiYlBJscowUQSUEU0YDlTAxPHFwRLR3+NzEj+cnHPP
+c/nv26vGfMnzmsfQviR1YE1KQuhLA1hKR9La0JIWK2VITxcHsJyVr92IbzO
+0PFEF7m+0RFCJ0vCaE2H8LWSp8nc54NTZdSCFkuhLPs86tCGN4xhGXIq68O
YUchhPuQcQN0DyKnCrq+8E3BmFZseIC3kdAM4q5/3vb8DO0B5DTy3YHzLZl5
f+PcDh3tcrarPrVti8pC6IXOBJmj4LmGdQH7DychrIZ+G/SNyJuUmq6cPc9a
DO8w1kH4W+CdBt/aMstvhncvPH/wvYT9QGq5x9H7d+l/Ou3j4f2C+9Pcryiz
HwPztqe63Nj04HyOtz6cF3PeV+5vnSeD0xTW7wVjK8yEc4bOJHVMhP/JCmMu
fOvYvwWr0bwPAOtN0LwETtfz3RP+/tB0heabDrZ3QmpbFYeOMRaH8Cuf2Y7p
+Ns+Yiv5oyKG8mVAjEsTPl6Brrl8/4OMAO9szg2yjf3Ljo6Z7D0M/b28L8j8
9gi8s9BxO/5OQO/O1HIf425lYpz6sHcrOvcy9KQ19rs/MvqyVkGfx8Zz8E7n
/Bd7WnJO9YS3NvLu4e5P3m7Dr3rOlfD2yzlXXom2vQPeldBPRU4neGtYP+DS
BmiH89a14Pr4vmjfb+VuN28N3K9nfzozzQbZjo5LkduGjlBuHxvhnchq5XsX
tG+ljkdffK+FZyO27chcP8qPkxFDxXcQ7wu53wLNLOibOQ+NMVFsFPf7satr
4twbk3e8Fes2ZG5jjca2+XnLkO9vImMTqzN343kbV7BM+a+8V9xlk+yVzKnI
H5oar8H4PoS1HZ6785YrmcrB2piHR7GziN6Lyv+cd9VUZc5xELb3IG8e6zNh
hLw7WOeROZO7cfhyALr3kbGkaPu/Qma31HmlHDsV8fkUvM/E+L7BXltl33vA
+3ms9ybu72LtRledco231pz9Ph99192qku9fK/Od8ufdTo6j9O6BZl3RmC1D
96sVpl+Tuq+pjo6h/9fUPBOFU+q6UT8Ym7gnCCfVtbDqljgPhPepWD/SdWW1
+6boBiOjpsq5qpgJe/l4MtILB/WmjbH/jCi4f6i/9cPmx0uul5nY2FjtXiSc
jqaWM5f7RxP3peuqnHPKlf2cf4JmQt518FTMc9Xu8qJzewiyc5l72UrO3VNj
sg6aNawu5cZd9SD67ZoDrGbsXgN9r8xv51PHUnE8zr4rdS9Xbm2J+aOc6B7n
hs7aNTsy6Nejq463Gs7Pcb5MdJyfVC/gPAN5xZJzUL3tv/mR8+xSn1VMNK8W
pa4r9S35Lcxvhr4NOVfF+dC73HndBu0vqW04zV6Ic0Q+rS3Zr23w3VTybPmu
ZN/kl+aq8kw5tg/6t1PX0JHEuIj3RnDaqh4E5h9ybmM1gdvHqWOmeA2GvqVo
Gy5wP7LKM/W9zPzilU97M/ulua4+Kn+7Q3swdR7Wc96d2ib1TvVf2XZ14rxR
zjTlnb+K9WjkbcbO57nrkZhfvM9wfjbiP427Oan70RlsOVvtmhFmsls2L0g8
ezR3jnF/OHUPu7No3dKrPG+pdF0IF+WT6nqMbEw8N9ch4wi8YzVrS54Hyg/1
2c5x9h2M/agl9smxca4J+yU1xr935hpW/apn9MysT3P12tSzVf8oI2J9qffP
KbouVifu4+rhm1PjK2wbEtske+SHeop8+QiaSVX+35E+3atGvi4ZO+E2t8J9
VnXQXHTsFXfl7YlY+xXofDHzLB4qPZn9FP+Y6OPFWEvyXfFbEf83FkI7sOje
o1jtzMynOq6LtfwE7wMT8yaZ+7vknSpZn3R9kLnGVF9DEs94+aJ/H+W74qhc
Ud9U7FYl5hH9LvJiZ7V9le3DosxPMs9azdnLM/cX2XMIPftTz0D5rX8R+V7g
/YX4//AvLUBTBw==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlllsVGUUxz8Knd4p5c7ce6cICEJRi1RaqC+4IBGw5U1LEaJGIDFuaUEU
fHEJghETRSnGCOIDImhEjUjiAgVZihpRUIrRpLigqPCg0ZiAUgmiv3/+83Ay
59yzfmedutsXt99bEUJYCwwSZCFUAR1RCJdUhtAVh/ApdGs+hJnA1jSEawsh
LII/FOiBtx8YBj4OA3ExhLPI3JILoQb9WmR/g54Fv2+gba3BZkveNo+DPwP/
kaoQVsBvLIXwJvQx8L3wtyQhnMTmOGyNRWYtdAtQgL4QegayHwMfAdOBJ7DX
we8BePMq7Gu94svbZ4qtc/AvjWxjIv62Qf+Av33w+8C7iHkG/OnAQXQPASPB
J/O+M+j3F42fGxBCHvkIegH8ZfjcRGyL+HYIfBn+m6Dn1obw3uAQVtaA4+8Y
8nXon8JfA/ydQDPyK5GvB78P/c/L+rtLrolq0VHNN2w1lRzrduw9ja2j6HwA
fyeQwM+AX5Dthn8aW3+nju1n6juNt92PvRPEewKZ8chOgb8V/kK+NcOfh/2z
5fqktdaRrGymyJaUZ+QfB5bDfxQYETlH89GtI55Jg2xjOPnPUsvWk6/D4Mux
dw2yV0eWHZOYJ52nVFveNAL5EjAMuhM4TT/NJB/D0V9K/CfR/Rr5Fdj/j293
R36DYhuN/Os5x9iMrQT58zn3UBP8m9F5Bd5mIIIenDg3ytFFyJ7hDVt462OR
e30I3xZH7vlvkP82dW+oRw/CH47+qznH2I/uxch3V9rGVGTzmXOr+NqgH4z9
ltV6E7o3EuMV8I5Ar4PeiM1e8JF8O5q659Rr6skbitaRrHqmt+CaKL+asV3w
Xi6YL5ut8BbhbzT+MmJqh56f2H83/F3ShT+3wjPRAj0NuA64HngO2XzRtVBN
+vlWVe4n9fxKdO9MHa9s/A7+XcG1CcgMgv5TOY8843vB2+Bfpn0BrML2qMy5
Vs4qwfeU3Ovq+W1F10T8DcBg8PGxd4/q8VfqHlf9tHM6wT/LvKvUT/J9Crg1
cgw16A4s7xvR4v0YuxaSGYPu6MyzpB2yDvx9YHLeO+SP1DOo2ZuDfK9sx66P
elq9MpX3vZNzz8TQE3lDfaV7cA35fDZxPlWTg/C6oC+o8r68B1ufZN5V8qfZ
WlHrWdaMvYhsHPu9o9C5CdnZmXen3tQQO0bFphmpRX59YlnpbAQfWnQvKIYh
4C/xbQNQA7449Q7T7tEOUi+9lnj3qqdmw+/G31V57zD16vOJd7F6IAfvK2J4
MucbsgP6odj5uzLvWvclzq9qOgN8etmfdM6n7kH13r/AhKJnSLOjGd+Ofn3B
tmRzAPx3S67lAGwuybxztWu14xaiuwT6n5x3snp/QWJZzcAI8FmpZ0EzrFnd
i73Oas9sT8k9p15byLcm/FWX51kyEXhj0b0qH6qNaqpaKt/qje+Lvo3qkZ7U
N0/10gws1YzF3v3qmUl6c+pbqR21D7yd9zaU50Vvn1Py7VAOdEvuQD+pcD32
69YWjD+AvQOZd3pr+f6tRn5s5tnUzWjl/VnRu0b160b/cugGYEfqWZtQ8i3W
zKmWL5TvqfL/E/SH0on8pi/Qm5J5VlXjWei2lxy7YtRu2QPsLu+YGv1/KDm3
mpG3+fZWal/6T/Gr7k3B+WiTDejDBePaeZvR7yzvtzHEcwT+l6lvi2767IJz
KP+KUbk8XnRulNMD8O4qeFfoZqwmH6sS108z+XDsHlau1fPapZsy+5JPxfZG
6luiGKuwPQ/6NiBX9O1vTPxfRP8B/geQoycz
         "],
         VertexColors->None], 
        PolygonBox[{{4701, 5051, 2364, 5050, 4700}, {3795, 4870, 2154, 4871, 
         3794}, {3712, 4703, 4702, 5052, 2364}, {4669, 4035, 4034, 4704, 
         1966}, {4504, 4325, 4324, 4503, 1820}, {4119, 4862, 4861, 4678, 
         1299}, {4798, 4796, 598, 3716, 4797}, {3897, 4675, 4674, 4795, 
         2085}, {4540, 3999, 3998, 4539, 1842}, {4703, 3713, 1966, 4704, 
         4702}, {4035, 4856, 2143, 4857, 4034}, {5308, 4866, 4865, 5309, 
         2666}, {4301, 4117, 1822, 4508, 4300}, {4718, 4516, 4515, 4719, 
         1997}, {4864, 4793, 4792, 4140, 2151}, {4500, 4107, 4106, 4499, 
         1818}, {4509, 4109, 4108, 4507, 1822}, {4107, 4494, 1815, 4493, 
         4106}, {4805, 5027, 2336, 4671, 4804}, {4518, 4716, 1996, 4717, 
         4517}, {4655, 3795, 3794, 4867, 1955}, {4242, 4518, 4517, 4903, 
         2187}, {4855, 4103, 4102, 4298, 1625}, {4139, 4902, 2187, 4901, 
         4138}, {3999, 4502, 1819, 4501, 3998}, {4516, 4142, 2667, 5312, 
         4515}, {4116, 4301, 4300, 4506, 1821}, {4325, 4520, 1826, 4519, 
         4324}, {4511, 4492, 1299, 4118, 4510}, {4866, 4654, 1955, 4867, 
         4865}},
         VertexColors->None]}]}, 
     {Hue[0.22836355745785367`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmgncl2O6x/963/e/vMv///yXt10lqaQ3YlBaCBlryE6rpIi02SLMkSUq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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmXm8leMWx99O++yzz4n9Dnuf6lQUmnUalBIJlzJXGiQNVEqJTkpEhi4q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         "],
         VertexColors->None]}]}, 
     {Hue[0.14503022412452032`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNnXecVEX29nu6Z7qnu2c6j+yu6BoXFASVJGJEBXQVMK5ZSfrTVcxxDQQX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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmnmgTdX7xg/33nPvOdy9zzn7iJIGlLHIkDm+Zg2apIRcc0kayFQRSkkl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         "],
         VertexColors->None]}]}, 
     {Hue[0.06169689079118701], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFnQeUVFW2hqu6uru6K1dXtQ+z6BgQHUVQQUWigaCCCGYxoAgmUIwkFRUD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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmXu8zWX2x7/HPvvssw/nu7/7YpIZt24qZibShV9y7eJWTKEkJfeUCiFy
OygVlaYZkaYSKnMocq0YavxGU4lcIvdbIndyb5r35/f5/bFe+1l7rWc9l+/z
rPVZ66n20GN/6lMmCILTeUGQz2/LdBAczQbBgGQQbCgIggPw+6GB8SAYBv9k
LggmpILgAZRzRUEwCN1Z0NXovwTfg3Z36LvCINgIFZQPgruiIBiH/ufIH0U2
HiqHflP4FLYjaBS2WzGHx5A9Du2h73jkT2eCYBd8F/Q3JhgfW5djs0LZIHg/
FgRx2jOZzxjsL0N/Gu1m2FuHvWH8NxG+ATod0V+FfoS9o8g3Y2sU9Dr8PqgV
4xVCX4VB0Aj9Luh3Qf8i+FTGuuu0HvgL9C/Bfnv260hxEMTYk/HMbzDjdaVv
c+Y4n7VsxkYN+jbkv060TyN/Af421jMbW0ex+QX2RvNfKWPdB+XT9wj2prCW
7loP7Z2s4Xv67mXMEvp+Ab8A/iB0BP42+oxE90nmMBS+KTaHMP5JxtjLXOfA
/8j3a894w6UPPxL5aeSf0fco9lrRvzX2KzNeFehDbC1ifbcgb4J8p74XNn7A
3o3YmELfe7A3ibmXQb6Qud/BfIqw/Qt7MgTZInQW0P8DqAN8E+w/TN9K6FzN
+N8j34W9WfA7sN0WnTnIW2HvIOPVhjojq6v9QZaGfw9bU/ivtIz7lMJvok/7
pNf4kPZTZwr9HP/9B9kg7MVZ3xj4cfD3ojMr7jN4Je0v0a+f8JmbyFhv8999
yPLh2/Dtntf6GW8m33AM/Y/BH2Ctb2OjNfLR8Enk7yLfy9qXwr+LvL8uFLbW
QGeQV4LdxN7ejI029C1hTgfQf5v/GtNux3/3IUvAv8T4Kxm/AvZ7sMbnkZ1C
pyeyr5njdPa7AH45cz9X7G93PWtojO5MxjvNeOt1pmhXx0Zl5vYre76W/g3R
Gaq7BS3Q2WDMXoxXCf0/Y/sHqCVjz2X+vdFfCz8HWRL997V3jHklY9VnTs2Q
9cd+rXKcKf7bC9808lmRjd3w67ExnL4tCn3XrmW8SYW+czn6dkV/BPr70PkI
+VzoXcabjr1ByK5A57fMfwX/LdV54b9O6F+LfD26TZjTzdhqAPVDtyF8X9Za
kTl9jv58+J6s7wb0d6L/Bt9nIGu7AB1F9ibzOY/tKdDt2F6Ytu+7kzFugq+L
zRaMn0DeCN3G0AzGepz+L2NvT8ZrK4Xvk7WPk2/bKR9IuyTlvWvHHFow3mXY
K4+9icxxHrK2jHeMvhOgLln7HPmatdAG9m4G/zWKew+L4T9BXpO5vI69kP75
0DTaazS/yN9Y33ZYkX3fp+jXjtkHfo2tp1M+O83pk0bWUT6e9X7FeB2R3591
+8Yi7/XN2LinrPf8LfgayF9CPgH6Dn4j9Le4z+g6bPfK+m43oP+vtLcwxnOM
NTbpu70O/R4FvuOP034C+it9myR8VofBj4n7zLahfTs25mBvFPbeYK+PQO3g
k9ArKZ9hnd2voL/AfwxfLWafOpX2jdhshq2l3Im36JtMWfch5tAf+4fYr620
JzPmYmQvwO+Cn8j3qId+EXQqYR/0LXu1JO2zdyu0G/3p6Ddnbi2hmqHPtM7y
bXzfm+BHpewbe8A3gr+J/isKvOby6P4S+m4epM8z2Jqecuyag/w4/Ajm+CHt
cbrfoc+8zvok+R/5EsVMZJPpMx/bW4vta3/Pfy/T/wB9/sJe/xH7A2nPzdq3
NmY/mqe9Bs1dMXMgY63nv9sUG/nvXvgFyHfIdzPepbRvYc5j2M9t9G+ArYY6
LwX2aQc1Xs57OZH+42hvZz5NkdcQPoC/njl0yrNPVls+Tr5N/5WGjgmKBdux
MQf9C4x5nrV1wsZ7yH9iTs3i9snCIu10H/ONSfqgP5b/2tP+F/rrsNUmtK/W
moQd7oBfkmcMkaEdZuy7yvHf4Zz3QGufDK1G/izyg8gPQ1OwPQqdVQX2IW31
bfhvC3xPxvhV+0+f4/JdCfuuu+G/zbMPk+w99EfkW2cu+vP473b61oCPkBVC
s/g+33JGb2DsG0XIPkAnpu/B+N2Y29vQLfRdG1mmmF+C7PWUY08l+Xj436Nz
kPNyMfy/adeHFmH7fuh95MfR/wn9H1nDL8xnY2hs8iS0FX5SaOzXBn4I+pXh
tyk2Ye8SnR2oIv2FI7che0aYCt0TUID+uIyxlHxAL3RrsSd12IuW/PdS1phP
WK8Y+fOKF9h4DNlW+j/N2trBb2ZtK2OO3S9mrasYPhr776SMDasofjDf1+C/
ZL41+e+a0HdMd+uuPGO7aoyxPGaM9xztsRljHX3jn7H9JX2GYv9p4Qfav6HP
YfpfxQLbw5/D/knWeyjPvvxkyr5CPv0H+u/L+lvqjI7C9srQ2G4WVA9ZX8VH
5G+wh12Z/6s6Y9gezHg7mf9b8B1o350wlj4CPzFpTC2s21rfqMCY9yHGv1kx
irE2MH55+Tvky5F/inw8+o/nHKs7wM+mXRad9sjvYU0fwJ9H5xyy+/nvEX1P
bE7B1nDmdCvj7+f+VmN/Shn/uqx9tnz15EJjw0a640ljxM3y/cL/ca/hNHO5
jD14Df0foX3w/2D8fsjuhe8r7Ml4b5b1HmSR59K++7sYsy2yZejMQDYPnZ+R
nYJ6C69BX2Prt4w/G3s/6D6hOxb+ctY2mu+TQ34D+stY2+J8Y9F5aZ8NYVLF
sjf57+VCxzRh4WFZY2Nh4mKdF2zUwtYG+KnYfgv6It+YbRWyMaxvb8JnvhTb
DbBXj/lvgS5oL0LftRJhaPgmOgP59pm/w9aorLGN1rCb9ofYyIs55xHWn5hz
LBbmF7acBD2Rb4z5FO07meM09uYy5niGvlcwn8nMZT+0rdg+WL5XmEWxfG5o
LKOYLmxwOfIFSWOE1fRfA/UssI9sy3wm5Bw75GPHpIx5hXX3o19HsQQ6EDem
G5YzRhI2qkifE7T/FfpsP6WYQ9+ilHOxAdi4O+1vpm+lGFwX/Rugc3HnGCH6
G5EvRr4Re09hf5BiCP37YO+FlDG6sLl8vr79R/x3RZHPQAL7hcKM+cZM92K7
A/TvAv+XoO/8lLHXCmFUZGeUv3D25jP+CeV/6Gwu8hxOYOvz0L6pe8K5nHyQ
fI9yumbwJ+nfMTBm2RwaAwn7tNf8s86ZlSsfQV6VsVdHvntV4Vfo7GPveeR1
sV+5vDGEsENBWceGi5H/PekYcReyTcLzMceYbsi65+y7hIGV6x6Duiac8/6V
s/EqNB/9GOOVpV0uY98wnv8qYL8G/ZfQfwt8NdpVoZ4x54TKbd/MGdspx/0U
/pPIWLk19DDz7QWVxp3D1kW3Xs61gKvQL4v9S+DnJX3mziuWsh8ttB70s/Ax
dN6BnwqdQtaPNdYrY5+su5wNjQV0p+Ohc3Dl3j2Vc6YdsxSrroFmKHfOGTsO
R2d95D7S1Rk/q3iedm4tn34m5Rim2CVM/r+hz7DOrvZEvqofOnXz7bO+j5zz
KtftrfiJ7qbQueZo+UPG3pLzWVINRL75N8gfKLCPvgzZpTnXPrQnG0L7DPmK
hdDn+rY536U2mmPaOaNyxR7M8Th8X77fKtpboNqKzynfzWcLHOsugSYmHPOU
y7wTGgsrp2kRuWahWoVirrB3RfRfTBiD16F9nc5Mwjnu6ZTPtM7yUdYzLGOf
LF+8KeFcRN9Q3045SUNkKcW/At+5j1VvKHbtRDHiInQr5FyrGMf4x9AtJ/xI
37VQA/ofQr85fb+AHqT/yrSxu2pSddC9mjGPx7ynNdm7g+jXC+zTazO/Whnn
kjOgs9h/Bf6buHPMKHTOrVz7RGBsVJxyLiWMtA77xbpjgXPm5SnHJMWiSTH7
qv/JOpbLZ1VCfk3Od2sv8t3F/mb6VrqDc5l/CXxI+wEoD/15GeuqhqLc4GTk
b6ccQbnZCKg44RxN2FuYS1hLGPwqxqoObaLvJ+zXCOYyMmsspxrFcmx9lnZt
oBvz60+7H3RNnmOYaifKUZWbqoYi35LAfmnSPqYU2ROhcxHV+KpnjMmExV7H
5tTIe6y9XYfONNpTobXIQ/agPmM9qz3Ld87XGf5B3YEyxvSPYL839FrSmFK5
+lb631ronL0Tsgegb+BXFTrWDVZMjDnmvYLsz1nXeoRhnkJ+Z9a5UC/y/3zW
UwZ6NuYanXLVtWnnbspZy6mWGDp2rimyr79U+XKeff5A1UKhvyWNobcjy8s5
FxiOzX/CH0M+MOkaygHaC/mvCrLe8LMZb3HavugDxfC0czTlZn9IGFvWCo1N
hTEj4TH4mTGP+SD2aipfTXoP/k57UMbYYixr3I/uAvhqyF6kTzf5+rSxqnLG
xchfpU/nfOfE8rWqASj3l8+tEhrTC8t3Vr6ErarQ0phrKrq772T8LXWH17Ce
34X25WWw+WXGNRfVWuQjWqp+Bw1OOCb9IXQOotyjK/3nZxzzFOuqqD6EPI58
dczfqLGwSujceYfqAynbUF/53B3Y3g7tZm3t9A2UH+dcKxGm6oZ+VyhTxj5B
tZPpWc9VNRRh1cYp2xZmLZeyz5Wv1RyEfR/lv2vzjYEHw2dSrqX1ZY0Pp7xn
2ivpqFYozC6srprhYfZnT9pz2xV3rjMhNJZWzvPPtGuoqp0qp9uhu8acjuS7
xivfdzx0bUc+MMvcm0WOrcK8H4au6aqWqxr0MviAMT4qMka+SHuP/nvCYwnX
SoconsZcM02rXpFxbNcdXRoakwuLqwb/SmgMIuzxDfQd8zmIvVZx15hnRM45
lGvov/87q4xZkO8zOxP5L/QZja29cZ+lKRnX1nWmVGutHrqWoZpred3drM9y
Z/TvT7tGptrYAWycDl1jU21NNU/VfpTDKHdRDUi5sGrmqpUrJ74UefPQa1fO
0U7fJnJuvr3AtZpd7EGLItdsWqSNEYQNlJPfCd86dGxTjvVM1jmfcr08vsee
rM+EzoJy9icy3jPtlWKwsPuctNcuDD8gck1BtQTV3C9XbTW07xRG74Xuw2nn
JqrZrowcIxQblKO1i7wGzb0XNm/PGEMLO2vPu+suaj0xn/lFyvWzxsbyQR3o
357+O+nfm/4dI9dEVQvtA7+DuW2D+seNOaulfEd1N+VTlmDvsGJA0pj7a/rW
Do1dGyQcm37KudajGHUX/Fli3guBayJByhhW2FUY7qfQPlq+WT58gOotOftK
+aiqyDfQZ1GeY9SP2Lola6xdGXNnaJ/NuvYnTLqN9oWcsaN8wiOM/2jaua9q
7l1Y71dpxybVmEdKlvLZUgxWLt4vci1NOfkS2ksjn0Xl1LsYf1HkvVIO3Jb5
bYns+1WTqMj4F2f9tiPMuyxyzFWsVQ74a+iaumrpqqEn4P+Tdm1FOUJ9dGeE
rqUpRpbAn8i5tiPMcTzrHFy5t2LMCGytQH+7cjvhh4xzbOXW8rnKRZUTKxdW
Tiqs/1rGbxnC/GUyzhGVG+qN5pxq39DHyKsz3qliv6npLa1H4NqhYrBir2qI
ehvUm5PemvRGWJG5/sycRwfGDAvZ2/sinzXVUAuQ7wl9d/SmMVRYMzJWbpZw
LWBoxthRNYE7sq5hqXalOSXl3zKupSkna6r8NnRtXTU45Z7XZ12LUA6q2s/G
nN9WVAPSWdQbpN4edSZnIDsbGevJB+hu3BO5tqk7cr7YOaFyQa1pcsoYXNhb
Ofc2+c6M3z6FUc6mjcGFvVWDlS/eWey3HvnkPPg9xd7b+UV+C1weuvalN0G9
VemNS29berNSrefKnLGcaj5lc36j0NuEMFId5nNTzthYbxZDmFuFrPdSPv9Q
aEwjLCOM9Ry6+0Jja+XEL2ac8ynXU037lGqN6DyTdAw6FDmGKXYJQ9+q/Dfl
WqRyiGORMbmwuHz8N4y9GvpTwphateL2kX2VasYnFcszfmtRDBrA3J4s79qz
3owKsHU4NJYWhs0p/kU+e8oBv0u5BqPai3zCJeH/1wSKXJO5NrRPki9SzjKS
sUoyvht6Y0xh71zot2LdmSJ0k2nXluTzTmB/Zs5vqaqBbk/7jUNvG1rToawx
nLCbfOD3oXM85Xbas9lpv7norUVvYtflXFNWLVmYTW8P0yLX/vQG8VnoM6Gz
IIz5X1dUcm0=
         "],
         VertexColors->None]}]}, 
     {Hue[1.0406922648421473`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmomT1dWVx++zu997vby1XwOaGGggZqmafyCZTHSMiQNOzCYmYSyrsorj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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwlltlvVVUUxndp6e1te8+57b04ATI68eYf4IAgwTJJy6gxJiIoqCAIRmiZ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         "],
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3551, 5260, 3550, 4930, 4929, 5246, 5245, 4291, 3994, 5458, 
         3993, 4030, 3557, 5378, 3995, 5250, 4942, 5115, 5114, 5249, 3585, 
         3556, 5377, 3778, 3619, 5391, 3827, 4173, 4056, 5390, 3618, 4469, 
         3617, 5389, 4055, 4172, 4054, 3616, 4467, 4468, 4053, 5280, 5282, 
         5281, 3562, 4596, 3555, 4593, 4592, 4624, 4623, 3607, 4963, 4964, 
         5248, 5247, 4931, 4932, 4417, 4928, 4927, 4926, 5337, 5336, 3992, 
         4531, 4532, 4271, 4278, 3549, 4533, 4534}],
        FractionBox["1", "10000000000"]],
       Annotation[#, 
        Rational[1, 10000000000], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3560, 4933, 4595, 3779, 4746, 3797, 4602, 4603, 4171, 5476, 
         4170, 5279, 4052, 4969, 4970, 3613, 4440, 4834, 4737, 4738, 4736, 
         3781, 5135, 3834, 4980, 4981, 4213, 5413, 3662, 4560, 4561, 4310, 
         5384, 4309, 3589, 3567, 5123, 5124, 5122, 3811, 5125, 3663, 3852, 
         3798, 5116, 3799, 3569, 5380, 3786, 3645, 3853, 5443, 3854, 3844, 
         5139, 3843, 5140, 4338, 4369, 3574, 5381, 4010, 4432, 4431, 3664, 
         4087, 3803, 5119, 3804, 5120, 4748, 5111, 3788, 5441, 4641, 4642, 
         3654, 3855, 5444, 3856, 3848, 5145, 3847, 5146, 4341, 4370, 3579, 
         5383, 4021, 4020, 4437, 3666, 4088, 3806, 4742, 3790, 4741, 3789, 
         5147, 5148, 4995, 4996, 4216, 5414, 3668, 4566, 3667, 4313, 4940, 
         4941, 3580, 5488, 4609, 4599, 4644, 4643, 3658, 4887, 4218, 4886, 
         4219, 4888, 4217, 5501, 4315, 5490, 4314, 3594, 3581, 5489, 4610, 
         4600, 4646, 4645, 3660, 4647, 4220, 3670, 4567, 3669, 4316, 5385, 
         4946, 3807, 5121, 3792, 4743, 4745, 4744, 4767, 3582, 5150, 5151, 
         5149, 5153, 5152, 3791, 5112, 5113, 4766, 4765, 4847, 5294, 3595, 
         4296, 4274, 4482, 4481, 4028, 4445, 3659, 5412, 4084, 4024, 5258, 
         4025, 5259, 4023, 4187, 5478, 4186, 4439, 4184, 5500, 4185, 4022, 
         4183, 5477, 4182, 4438, 4180, 5499, 4181, 4016, 4435, 3577, 5382, 
         4179, 4015, 5339, 5340, 4938, 4939, 3576, 4433, 3575, 3593, 4608, 
         4607, 3805, 4008, 5459, 4007, 4427, 4006, 5338, 4177, 4936, 4937, 
         3572, 4424, 3570, 3591, 4606, 4605, 3800, 5507, 4747, 4935, 5252, 
         4003, 4732, 5090, 5091, 4528, 4529, 4267, 4913, 4266, 5373, 4337, 
         4415, 3565}],
        FractionBox["1", "10000000000"]],
       Annotation[#, 
        Rational[1, 10000000000], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1Hloz3Ecx/GPf8hi7KDlP5QhfzhyZfwhJrIchea+RW0zmW2UaHOfc5Wa
ko1ph1iOUfgDuf2BcmTYnNtstYUiWx7fP557vd+vz/v49P3u++u7PGt2ZqcQ
QrI/ObEhDO4VwvQeIfTDZHF/WsaviAvhHB3fM4RdvPHyNPlLeZr8UkII1cjl
zU8MYRPN0b8JGzE1PoTuiEW8s5FqhuhbqmcJFqORf5hfhDnyeZiLInU5vHd6
36MWV9TecYeF9tc6W0RrUN89hHHyBpqqtw2tuKs+1z2+i6+Kj5g5yJyB6Co+
TZ/rf2DmUtpAGyPEJ9XXimtwHRd4zea04KazFbw6O/OQjyb+D9Q42yy/Rhvl
DdHziZ4Njtp5DNednaA36HH6wT0K8VBe6L4PaIH8Hy21dwzvjLjczhI6Vp7C
X4VnvGreBF6V/Jb8Jp7jBV7zXqFFvJquQZLaUXrWRPe3v8Md38jb6Vtaxf/N
r6C/aCX9SEtoHS2lN2iBmX9oM71v7l/xPP07zcmz47O4s9ouWMDrKY8T/1F7
jtarP0s/0WlmTEWpeAfvLN1Js/VU6q2KZiIfw3lP1Y6gfTynreYl27dAns5f
LG8X9+INpdvNGkZ7yzvoPflKdauwXW2rPdtoG012/pg/kK63KwuHcBDpvP30
ADKQiQG8SeoL9Bab+9F9TtE6WuZ8Cv2pJtb5E/EWZ8Xy8/alRO9MXK+/ko6T
l/ET5Qnoihgs4b2P3gE+4B7viPps8zagLXoPyNL/1Tcwy96ZmBE9L16T2hh9
P6L/Tftn8PJpkrybuJnejr5LM5ahXXxR/b/oG6cJzkfzE6M56h/x18nXopw3
Mfp9iJ6F2nhnm6N3L7/srFrNN/FXfEGq2iacd7ZX7Sy1++geZOjfTTPpBH3/
AYcKt9E=
         "]],
        FractionBox["1", "1000000000"]],
       Annotation[#, 
        Rational[1, 1000000000], "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1HlM13Ucx/GP6ayJBCgYtZLQWil2eIJEyaELN93K1dZhq5BIMxPFaS0Q
M1334VGmOfEuhcqjdEtLcK7WxNQB3WXOAy0VDLXLrR6fP56/1/t4vT+fz/f7
+/x+mSVTxz3VKYRQ5+PE5SG80SOEk3QbzUwK4RN6LT2QHMJ+fJQawr/yhT1D
WIQS3nkpITxKh/PmohteRwK641X+qeZeo+V0A+8HZj5EgTWb1PJpM03Umyvu
EHem22gX2kbb8URaCJNQpXaV/a9Elj3y1RrMVovbrDvJXgPVFqg9KR4jnkIn
4+G4Ds3Qe1bcO3qsMxkJcR95jXglVmAIb576WfXu9A+aSHfQS+11Gfbw3Md7
Px5AvlqS/hrnWYvVSOFZR1/Rfxnp8izrfMHXIf+SnqOlaovF0/XHOncFnYYJ
6uX0Zr1K8U10Fv9MJMfvRb5RXIti+xSa7ca/StwfS8Qj1HbyXc+fIO9Dr0Nf
3KKe5NzJaPd+P5e3infRKvOzcV5+AR2JIfS0z2i+ndiBE+rFdDvdEN95XIen
DXeqf2uPU3Q9SpzjDrqL57O4F8YhjTcVL/DWWuNpez6DOr4R+vW0njZgt7gB
/zjLbnk7/2yz1XE/Z260RjGdqXdGPIu28DfjbvX5fGfNvKc3R/wcHoxnj9+x
+vu0g26ko+O91P9KvkQ+XvwQUtTHm5mD3+Pz4TfMsNdpupn/Hr0t9EX5VvoS
bTV3BU1HpXWqMDLeQ/ko2mS+GsM8ezZysMc6x5HDOxzZuA25mM5bgT7ea6H5
Fv4i2lfeHH/DWI530YrB8V5jEIZhKMrNT0OmmQKzefY6Jx8UfyPx90Qz6Hzc
xTOPPsI3xHpDkYG9qNDLj+9d/zEzZThv/b1qF2gFDXq9+fLEz4t/Uv8ZS81f
Iu9hphP9WH4NvVX+lvhtLMY78T5jlfnDzrmS/mj+B6xx5k/jfXAvuqpf1G+T
bxIfE2+mt9u7Tu17/u8wN/432KMp7of/eBbR4ziGkbwTzIyiRbwFvIW0kGbx
5lp3AE3lKY1eNKrVqGWrrY7v0ToDsRVb8JfeAvo3XUj3O8cB/GLuUHwmM1ny
GjqAnvI8XXhPx/uMozxH8KveoTiHi/F/AF+L96HSGSfK91k/HX+af5MexRG0
6H2D9bzrsDbeKTMHrX91/E/Xy8VE9cdRFn/P6jeq9cMN6MXf1TnOqDeI7/W8
9bTIGsvU6uP3rXbY+YP4IGrjXTczBY28CfppajnyXvFOWbe/+uD4fuM9QinK
9PrR7db+H3XC9II=
         "]],
        FractionBox["1", "1000000000"]],
       Annotation[#, 
        Rational[1, 1000000000], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwV1nmQz3Ucx/EvpdFMtBEb6z5y1LLt2rXIMROSe7RYdh27KlfWESnnTMpR
ucK6zWhSzIgKI0z5AysahGI6yS2qaYqZjqHH+4/nvD7v9+v1/ny/39/32K1f
PL5fSbkkSeqXT5IRDybJ4SpJUvhwkhSgkXp51SRZgVLk6T2Lf1OS5D/c4XeX
74leGPVQkpThMGqjCT+bnpPtY64vdthnJ3YhjVcLfeR68nrT3vRupSQ5j1zr
dL0c2oKOQo9qSTKSjqycJBfsWwGDHHswCrDAfrN5A63zsQWbkY5c2Wzza/kT
5CbitL2q8aojFW300vjPO7+96mz+bppDB6Kb+QH0rr0S2RRaxB+B5/Cy3hEc
RVO0lX2HXozztH7c/Bz7N9F7jT6mztc/yL9HfcN177F+Um+zc9iCEv1iv8F4
Op93y3oBvU3foLVk5tE0OpdWt3cN8+nqjeopcV6YivO8DfY5y2uBM8hxDt/Q
UziJ2WZH2/siLmC6mfVmvuadRjt7HtO/SkuRaX4F/we5Zerm8czQZnQ5/Uc2
h9dKZrreNFzWu4QfHeskPYXj+AonUCg7hg6hS+WXoJbslHhGHG8ybWnPbH6q
+pz6J9RwvHz1I7R8PF90sLo2vRLPEK2pzjWXwW9Na6gv8y4hLe6TumacN9bg
xXhm6Eqsxlj1Kpplvo35NPU1s1dRV79AXSfuJ39ovCPqYfTpwDV1jeeI/inT
TqaDPerKjItj4YTee3qbrFu67iz3rZp8ddxndo26DrrEXtgu8xGG81rY5y2z
O8x2te6it9O6xHoX/RjtZe+au4PBjlVR/pbnbj/vpvoG9sXvTHthFT+Ldx3F
+kOQYb+WWMQrUlfAEuxQF9JneAV0aqzjutXd6SZ+W+u51petKzqHK/R++jYt
pV/wVtDD8Z47/mT9I+qj4aG18/iSbuX3439I8+y5Lb4PcTxeD3THQrMzZa5a
X4tjxXsU10K7yf4e77XZ3+is+HagGX+4XlM6zPyjoeom9Fe5ElpP3ZGXGe+G
epx6A22FnPgmqLPpJJkGdIi6If3F/Nh4Z9VteU+Yb0drq2/ybqAxf6i6USiv
Hi1U14/z5s8wM9p6lN4Y2llvXTyP6lL5LfyVtJU6U3+b+/2U6+2Mg9YHMJ+/
UW4BLVMfwj7+3vidzCxFuvk3+d/JfY8leo3kFtO/5VL1KqFyfPfjO6e3HMuw
BxPMHqSHcCDuJ8rwrmzfeD/id/fclVkfQkZ87+g0z/Z0bDO/3d7b6c+843qz
zG6lM2mx7Eh7FNFX5abF99Y6RbYNrULL6Z2VXyR/hk52/JfQSd2RdqSd4m+d
YzSTbUxT7Pmt/ivqqdhvn8/xgvlPZXfH9zd+Q6TKNVffim+07G36QNxT2dHW
R2mJzBFaJDccuer28RzTDvEd9RuUl53jnP9Sv07n22MeZukPMNvYOkt+PM2k
6+U2YC1Oxv2nDXgN0V+dr86jg+I9t99EOgnrnfM6rEUGf5XeMVqkLkad+Lul
vu6Yx+k1ukZmNUp4M5xPnl5/VMAHqCr3Pr2X/0f06Wd6C80sxiLswoj4LsXf
d94nNHFec2XfiP8BzPXG/3P5JQI=
         "]],
        FractionBox["1", "100000000"]],
       Annotation[#, 
        Rational[1, 100000000], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1FmIz1EUwPFrnywzZouxj2WyzEKRFArhQRiDITWmZB/ZHyxZC2XfspUk
kshWwjCWGdn35cEeigdbopjsn5uH73zPPfec87u/3/z+v8wx0wqmVgkhdPGn
cUoIK+uH0Co1hMq0EL6jmfioXHe+zc25BSapnYzliSGMTw5hBedwYXoIuTxT
zQz0SgohRS4VyViCRViMY/pvqrmB2eJpaitd4zseW6+zfsRP8QSNrA9yjnNl
o51r1tFbLq5AXXGe3kR+ojaZO1kncUd+i/3O9oa3cSr343FmLjMrW/xBXIZz
6IW++IMGaifwBbNWq9uFenI7OZ1vIQ3b4xp1kKC+Fipx0Xl6cFv33Z3b8WTX
3G1ebeTq3atnqr3pyBL3Rh98s/dQbU11DzjP/gI+wGnmvuM78X45FRn2Wutr
xC04E6M9nyJ0UnfVnBHiKzxRzQ7ehnniUzyfS3moWUPUV/BLZ3iFAusL1kP5
Ir+O54pr9cNQTVxX7rj5VdR/sa7KxfEMKIgz4/9S3MaMDL7BJdwQc/Xm2/+t
56X4q/gzqmMA5sit4Vdczvl6z3OB6w3BOPnO5ozlF2YUyg1HlrqTzjdQfIK3
oqt4O69Xe43X8XUuMW98fE6cx83NSxOn4K71e07nbO6Ikfpy433qLTPzHGqI
c1w/ge+rnWnGFDX7eKN8lnwHLpUL4r+8xJxiXsxn5Iri+xN/D3yNf3Aij+Gz
9leZtTa+a3KbOYlX8gasj9cz57LaEt6j/pK4D7o5z694P+p/cjL3Vb8lvtf2
yniwXH8exKe5PTaZM0t9B3FPdfXM7MGH5Q7hCCri7zG+V+JR9p7rH6Xvbrwv
6zt8H/fQ2PpofMae19L43eCF/EmufbyWORniZ+o+xu8At5Rba2Ymr4nfCWdZ
zY+4aXwm4rbcJOX/9+wfN5Ghow==
         "]],
        FractionBox["1", "10000000"]],
       Annotation[#, 
        Rational[1, 10000000], "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{5312, 4515, 4719, 4514, 4717, 4517, 4903, 4901, 4138, 4792, 
         4140, 4670, 4873, 5311, 5312}],
        FractionBox["1", "1000000"]],
       Annotation[#, 
        Rational[1, 1000000], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4797, 4672, 4676, 4861, 4678, 3693, 5026, 4671, 4804, 4674, 
         4795, 3716, 4797}],
        FractionBox["1", "1000000"]],
       Annotation[#, 
        Rational[1, 1000000], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4704, 4034, 4857, 4104, 4617, 4854, 4852, 4853, 4298, 4102, 
         4700, 5050, 5052, 4702, 4704}],
        FractionBox["1", "1000000"]],
       Annotation[#, 
        Rational[1, 1000000], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4510, 4118, 4512, 4105, 4499, 4106, 4493, 4495, 4299, 3598, 
         4506, 4300, 4508, 4507, 4108, 4510}],
        FractionBox["1", "1000000"]],
       Annotation[#, 
        Rational[1, 1000000], "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{4867, 3794, 4871, 4868, 5309, 4865, 4867}],
        FractionBox["1", "1000000"]],
       Annotation[#, 
        Rational[1, 1000000], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4324, 4503, 4000, 4501, 3998, 4539, 4001, 4519, 4324}],
        FractionBox["1", "1000000"]],
       Annotation[#, 
        Rational[1, 1000000], 
        "Tooltip"]& ]}, {}, {}, {}, {}, {}, {}, {}, {}, {}, {}}}],
  AspectRatio->1,
  Epilog->{
    ArrowBox[{{-1.2, 1}, {-0.8377335312327795, 
      0.1383578663016639}, {-0.38546788559588496`, 
      0.05641791466247163}, {-0.3069314914683546, -0.08277877382185601}, \
{-0.11656763796105155`, 0.021647383890019883`}, {0.07862586389284601, 
      0.031835306905749886`}, {0.2718555676826467, -0.036128088682171475`}, {
      0.4584963880522191, -0.17473278087913358`}, {
      0.6357975829441427, -0.37220556411385985`}, {
      0.8027271046850972, -0.6161274459842689}, {
      0.9594205771014539, -0.8958511169360397}, {1., -0.9983533104372209}, {
      1., -1.}}], 
    PointBox[{{-1.2, 1}, {-0.8377335312327795, 
      0.1383578663016639}, {-0.38546788559588496`, 
      0.05641791466247163}, {-0.3069314914683546, -0.08277877382185601}, \
{-0.11656763796105155`, 0.021647383890019883`}, {0.07862586389284601, 
      0.031835306905749886`}, {0.2718555676826467, -0.036128088682171475`}, {
      0.4584963880522191, -0.17473278087913358`}, {
      0.6357975829441427, -0.37220556411385985`}, {
      0.8027271046850972, -0.6161274459842689}, {
      0.9594205771014539, -0.8958511169360397}, {1., -0.9983533104372209}, {
      1., -1.}}]},
  Frame->True,
  PlotRange->{{-1.3, 1.5}, {-1.5, 1.4}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 169},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->97978994]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "WorkingPrecision",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->26397],

Cell[TextData[{
 "Set the working precision to ",
 Cell[BoxData[
  FormBox["20", TraditionalForm]], "InlineMath"],
 "; by default ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " are set to ",
 Cell[BoxData[
  FormBox[
   FractionBox["20", "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->549268746],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{"3", " ", "y"}]}], "]"}], "+", 
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<Newton\>\""}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1164918439],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.99999999999999999999999999987636450257`20."}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.37638497240654986291347727951333404923`20."}],
      ",", 
     RowBox[{
     "y", "\[Rule]", "1.67867608195212020647259190660275716844`20."}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{550, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->854186613]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6282],

Cell["\<\
Optimal asset allocation: Here are the annual returns (R) of long bonds and \
the S&P 500 from 1973 to 1994:\
\>", "ExampleText",
 CellID->671779067],

Cell[BoxData[
 RowBox[{
  RowBox[{"R", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "0.942", ",", "1.02", ",", "1.056", ",", "1.175", ",", "1.002", ",", 
       "0.982", ",", "0.978", ",", "0.947", ",", "1.003", ",", "1.465", ",", 
       "0.985", ",", "1.159", ",", "1.366", ",", "1.309", ",", "0.925", ",", 
       "1.086", ",", "1.212", ",", "1.054", ",", "1.193", ",", "1.079", ",", 
       "1.217", ",", "0.889"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
      "0.852", ",", "0.735", ",", "1.371", ",", "1.236", ",", "0.926", ",", 
       "1.064", ",", "1.184", ",", "1.323", ",", "0.949", ",", "1.215", ",", 
       "1.224", ",", "1.061", ",", "1.316", ",", "1.186", ",", "1.052", ",", 
       "1.165", ",", "1.316", ",", "0.968", ",", "1.304", ",", "1.076", ",", 
       "1.1", ",", "1.012"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->630754998],

Cell["Compute the mean and covariance from the returns:", "ExampleText",
 CellID->519863318],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[Mu]", "=", 
  RowBox[{"Mean", "[", 
   RowBox[{"Transpose", "@", "R"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->114999578],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.0929090909090908`", ",", "1.1197727272727274`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->495299524]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[CapitalSigma]", "=", 
  RowBox[{"Covariance", "[", 
   RowBox[{"Transpose", "[", "R", "]"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->225045130],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0.023152277056277062`", ",", "0.011331740259740264`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0.011331740259740264`", ",", "0.028282945887445888`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{332, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->137793996]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"X", "=", 
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Subscript", "[", 
      RowBox[{"x", ",", "#"}], "]"}], "&"}], ",", 
    RowBox[{"{", 
     RowBox[{"\"\<long T-bond\>\"", ",", "\"\<SP500\>\""}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->77110343],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   SubscriptBox["x", "\<\"long T-bond\"\>"], ",", 
   SubscriptBox["x", "\<\"SP500\"\>"]}], "}"}]], "Output",
 ImageSize->{125, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->55176005]
}, Open  ]],

Cell["\<\
To minimize the volatility and achieve a 10% return through a mix of stocks \
and bonds, the optimal percentages for the two assets are:\
\>", "ExampleText",
 CellID->1315314347],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"X", ".", "\[CapitalSigma]", ".", "X"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"Total", "[", "X", "]"}], "\[Equal]", "1"}], "&&", 
      RowBox[{
       RowBox[{"\[Mu]", ".", "X"}], "\[GreaterEqual]", "1.10"}], "&&", 
      RowBox[{"Apply", "[", 
       RowBox[{"And", ",", 
        RowBox[{"Thread", "[", 
         RowBox[{"X", "\[GreaterEqual]", 
          RowBox[{"{", 
           RowBox[{"0", ",", "0"}], "}"}]}], "]"}]}], "]"}]}]}], "}"}], ",", 
   "X"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->721185121],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.018295949255468652`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "\<\"long T-bond\"\>"], "\[Rule]", 
      "0.5888555295805812`"}], ",", 
     RowBox[{
      SubscriptBox["x", "\<\"SP500\"\>"], "\[Rule]", 
      "0.41114447041941876`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{357, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->276323235]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21468],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " tries to find a local minimum, ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " attempts to find a global minimum:"
}], "ExampleText",
 CellID->1650805990],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FormBox[
      RowBox[{
       RowBox[{
        RowBox[{"-", " ", "100"}], "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "-", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "-", "1"}], ")"}], "2"], "+", "1"}], ")"}]}], "-", 
       RowBox[{"200", "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "+", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "+", "2"}], ")"}], "2"], "+", "1"}], ")"}]}]}],
      TraditionalForm], ",", " ", 
     FormBox[
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ">", "3"}],
      TraditionalForm]}], " ", "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "2"}], "}"}], ",", " ", "y"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2092335573],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "103.06261052078153`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.2303719117773388`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.2190918583675474`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->33929099]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FormBox[
      RowBox[{
       RowBox[{
        RowBox[{"-", " ", "100"}], "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "-", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "-", "1"}], ")"}], "2"], "+", "1"}], ")"}]}], "-", 
       RowBox[{"200", "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "+", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "+", "2"}], ")"}], "2"], "+", "1"}], ")"}]}]}],
      TraditionalForm], ",", " ", 
     FormBox[
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ">", "3"}],
      TraditionalForm]}], " ", "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->827788029],

Cell[BoxData[
 RowBox[{
  RowBox[{"NMinimize", "::", "\<\"incst\"\>"}], ":", 
  " ", "\<\"\\!\\(NMinimize\\) was unable to generate any initial points \
satisfying the inequality constraints \\!\\({\\(\\(3 - x\\^2 - y\\^2\\)\\) \
\[LessEqual] 0}\\). The initial region specified may not contain any feasible \
points. Changing the initial region or specifying explicit initial points may \
provide a better solution. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NMinimize/incst\\\", ButtonNote -> \
\\\"NMinimize::incst\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->57058932],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "207.15989698200875`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.9948613347456736`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.9922920020949584`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{271, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->33288165]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "100"}], "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"x", "-", "1"}], ")"}], ")"}], "^", "2"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"y", "-", "1"}], ")"}], ")"}], "^", "2"}], "+", "1"}], 
      ")"}]}], "-", 
    RowBox[{"200", "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"x", "+", "1"}], ")"}], ")"}], "^", "2"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"(", 
          RowBox[{"y", "+", "2"}], ")"}], ")"}], "^", "2"}], "+", "1"}], 
      ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "3"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "3"}], ",", "2"}], "}"}], ",", 
   RowBox[{"RegionFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"#1", "^", "2"}], "+", 
        RowBox[{"#2", "^", "2"}]}], ">", "3"}], "&"}], ")"}]}], ",", 
   RowBox[{"Contours", "\[Rule]", "10"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{"{", 
      RowBox[{"Red", ",", 
       RowBox[{"PointSize", "[", ".02", "]"}], ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<global minimum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", ".995"}], ",", 
           RowBox[{"-", "2.092"}]}], "}"}]}], "]"}], ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", ".995"}], ",", 
          RowBox[{"-", "1.992"}]}], "}"}], "]"}], ",", 
       RowBox[{"Text", "[", 
        RowBox[{"\"\<local minimum\>\"", ",", 
         RowBox[{"{", 
          RowBox[{"0.5304", ",", "1.2191"}], "}"}]}], "]"}], ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1.2304", ",", "1.2191"}], "}"}], "]"}]}], "}"}], ")"}]}], 
   ",", 
   RowBox[{"ContourLabels", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->200111634],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyEfQVclMn//+7CLuqZ2N16nl18zXuwu8/kzu5ODE6xFbvPTkwO9ezCeWwx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   "], {{
     {RGBColor[0.293416, 0.0574044, 0.529412], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmG1sleUZx5/n9LQ9pT0v9JwOOoiLia3Qol+qoLHsfYyYcbqtloLoEvtB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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllMtvTWEUxfd9VG+199zHOTdIOzEoIWLk1cQrInGDtEJpRUJioEERuSpB
B1RURNKRV2ugf4FBGZkwELSUicSzLZVIm5LbEpEYqN+2Bjv59ln7W/u1vrPw
4Imdx+NmVsSSmOXNZnNm92NmlyvN+iOzO9hkwmwqZVYfmi3CBjk/4ltj1qwJ
e825hTuLA7MJ4gvVZkuqiEvzjfihlO6MgxUzZl9JGhDTRvxP8p0lVye2Anwl
9oXYfvh2cf4Gx5UKcs81ewLXDP5V/C74u7g/Q8xyuB4S/wz8BdYNlqCmV3AP
Y234++C/S+5W8CR+D3zb8K/R8zzyLcBq8Lvxo6R6usH5OtYRU8z/Xsj/OKme
+vLi8Lt12Ebwj+Al4s+R7wL1TVPfMuqboufdzKo5q1m10sMo5xGsAFaEcwv3
i9gPuMrkfwD2Gb4y/oY57AGuYfD2lGb8FvwNZtxfn1Su84Fm4Tm9l+ZIs/Oe
fFYXwYO4ZrbHa8mqFq+hgvhSoFp/x5Wrh/z1ppxj3F9Dv5PwV3L/Hfgt4pdS
axmOoVA78NlfYsZNxB6OhK0l/yy+ZXV2DXwCm4Z/AL4q+Fbjr3K9gU2Qb77v
C9tK/C/m+TyUZlwrvuP9+AeweEJ3xuA6xr7/MKvTMWFPQ2nFYz6AH8lptifB
R/E3h9K273gd5/dpYa7BFnIPhurFa3At9TKj2pQ0VU5rJj4L7+Em2O28tNBA
/N+MNOdacw3vhet7Wtp2Dbbn1KP35jUnAr0Rfxs+Y38bhwJpyd9IB1gpklYa
4exjlnUFNFyjN1rrufN6u76zo8TuyEgb/mZcSy9zeguuqQKxnYF27Tt0bZ+K
xO0ad+7erGbpOUbAN4XajWt8PC2NujZdsw0Z7dR36Zps47w9o1n4zAaIvZfV
v8P/MdXkPhPqX+Ga/AciDKCg
         "],
         VertexColors->None]}]}, 
     {RGBColor[
      0.37980202610186786`, 0.20760636954882677`, 0.6508380100637459], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmX2Q1lUVx3+7++w+L/vs87LPwwwiuGokOIPtmopkLEuorCKJrphAKJok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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllnlsVFUUxm+npTNtnXlvOkPcrRhcSIgQRUQDKItVEFFbdgGXGASVVqWo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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4598361905720326, 0.3467641671356469, 0.7633357102003145], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHmM1dUVx38z8+a9N4957/3eMtUMMjBix0SrGAS1xSWVUdFxhBk0se5i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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnlsVVUQxk/7Xl8rpO/de9+FGIUiQiDRKATENdEoRQUsS8EEUMQVNSAo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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5398703550421974, 0.48592196472246707`, 0.8758334103368832], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1mGmMVeUZx88wc++dGbjLnHvBIgVbB4bRKomBDy0kGMCgYGkEBEErLSN8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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllnuQzmUUx5+1l9du7fv+3t/7ImmZMmjKTIk/KoZxGUa1jfu1xKI/Ivf7
rTEuE2tdKkUlNAkzRSoNIqHEpnFZY8qlGu0gzXQjlp3o8+37x5l5zvM95zzn
Oc+5PPeWje01pk4I4WMoD/olHcJDcQjbEiGMg+IohAx0OjeENeBJsGHIfIXS
SvADrPenjc1FpguybyZD2A52JD+EG6kQZsLPLAxhPnxb1nuRn8R6HjZaF4fQ
BqqoG8ISaDP2K7BRryCEVvDPIzuWvSvYO4P9pqxnZEJ4AHsb2CsDr07b16+x
txP8JPbao9sBGsZ558GH5Nvn8fCPYn8f/Cmofr0QWmLvLfT/uAM+tk3Z0t5N
ZL9hrxRb0/DpMfhy8IZg+9HvgG576EKufT6A7H6oLcGsgd7n/jehhcjOBH+B
9a/YGAk2HOqN7kX8Lc8JIQebj6C7Gh8/Zf0J9BL4HKg/smXoXwPri8weztsN
7cyGsAtaWxRCLTYmI1uETGfkWxCj68h2gR9a6BiPBn8R6sr6AjKl+PM2Mtfy
7LNiexb/Bhc4xsvBVkA5yLfG/iDufh173+PbBWTO6C5Qd2HwzZFdAJ7Ct43I
bMBeZdpvLR9Wct4q5BsjuyTXsWwCvynfMT0EfgEaD78MmTvB22FvB3w5/LPw
JZyxDtlL8hHd4ez9yXobMleVK7FzcRE+z2A9E7qFfC17s/FnNvEaWOSYPont
jewN4Q1G4t9z8KexORB7ldg7Ab8Z7Ah8Bfo9wWawd5X77IK/AraEvSx4Mfqd
WN9krx7nfYv+U/DtoYOsT0J10f0BfgDyTZGfgm9TodqEY1aD7h7wTuBp9taB
jSAeg/C1GroM/xs0h7uNQX4a66bgtxO+Ywd0SyOf9TM+VmCvR+RcJ9XDUO46
LGNfHyQGE1lPyThXuhX5LZ8hnhcL/KZVxa4p1ZJyWL7dYC9O2MfDkWtEtVEK
9UP2Ov7sqOMcUS/5EHttE+4pE4j9eKgFvnfnvK7IViATY79NkWO9FHv1Cxxz
vUVNyr1Db6LYFRLD5oWO4UF0K6Gn0W+ETF9ka5GZXuAcnpbxnXXX+3Vn1t8V
u3cdRWev8gF7Nfg7C/wsvp5LO7dUk+PAJqVdm4ugLaw/gLoV+k7tOKstVIls
mWoucg4rd7fgz6XIMsJGKH+TfiO9zZdQb2THZR2LJ6Db+HYCvDeyDRLOpeaR
bSmnRsN3jtxb1yMzH18WpJ0rzfJcCx0j565q4i/Ouwf75/G3GmrMuiTrXqE9
xXpTsWtFMe8O/4Tih/4u8qWcdeukaysDvhjdefj8Wp5zfo/ekxjWwr+L/404
/25oLdg6qAC8KunaKYFvyFs1iB3rNQnXUpOsfVNNjYJ/nDMPcF4VtA3ss5R7
1Vz0a2L3NPWypVA91V5kX9WDb3GXsqxz6Rh33BF7hmh2aCbcB/9y0rPjPeg4
tquhsfmeEZexNQp+RJ579OdZ92z16vXYrAbvm3TvH4Y/tzKuSdXiTu7wDuvh
yA/Ic09thuwpdPoVuMflss6LPBt0hnr11IxzUz17UsY1qVpUjldiqzJybfXA
/1ex/0rsu2lGdAM/Dt4LPFPXsV+WdO/VGzRGdj78+oR7pnI3Yu/1hHP4MOvt
KcdWNfQ3slsj97ad0JjYM1mzWDNYtZ0TuxZU40fx9VjGsVQNVIFP4w06Bv8Z
zqXcg9R71MNUm31SnhWq0Z9SfnO99eh8/x00AzX79Ie4nHIOKvf0Z1hR7BpQ
7mtG6O0WIZ+s6zdUL1FMFAv1lIVpx1CxU8+aHrvm/q81+JWaZRn3avWki2BH
Oa8n9iew93vKM0izRzWxL3ZPVi/WDFXuT8i4l6sG+oD9k/RsVs9X7HdnXRt6
g8Wxa1a1Kp9z0M0D34f+YnRacfbD0OrgGmuJ/l2xa2VrvmfhvxnnmmZiFiwT
u5esQqY/+MXIvUY1NCvrmlQtaualYvdA9T7lwHKwFWn/LZSzfZLOceW2/kjd
wd6IXLvHoWORe5R6k3JOsZKP8k0xm4P92bFnrf4MmvWyIV3NfNXCjym/tWoi
N+uaVi0rBhOxXZJ27JVDg9P+c+ivoZmkXq0/m/5q6tk70P8i5drUH2gytibF
/qtoph6KPSM0G/SHq4r8J9NfTD1Wsd6U8exXzD+Cv5J07uuPpb9rHDu2+sP+
B4RdkIM=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6050553268026867, 0.5938177390693457, 0.9104387599908079], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1mH2QV2UVx+8uu7+33d/93d/9rfgSYLyYoBb5DyZmTikp6UxqLqSoKBOg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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllmds1lUUxm/79p30/a++IqOtDDGMIiEmVVBcCcQIiYgMFWQ0YrCDIQit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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6638989569447512, 0.6883630792934027, 0.9117802135171892], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuw1VUVx3/3nvfhnsfvnIOQDwbzFiI+gBQyxVCyaUYmqyEQQSBMQS3U
pgybQemi5dQk42OIbjbOKOJjpnSUh2EhRnGZsBk1fBDKRRwbvTwCAi63NPp+
+C7GP37nt8/+rb3W2uvxXWvv0+fe/I0F7UmStOsno/eZ2STZk0uSnrYkOar3
HZq8SE+hkSRfqybJb0U3dFCSrM8nyQ80nl1Ikrc1/pPo+8pJ8mHZ/xdUkmRi
PUlGiOcj+t8uvqPEZ38pSdbo/ysaL9Da4fo/Ud+GDk6SUi1J8kXJF//devaL
/xrRrBT9+aJf2ZEk2/LW4aD+r9W7S3LvaiXJZOn2Qs68kDNZ89Mlf0YzSXZo
/jHRrNBzd9k6VMVronQbp3Vf0d6eE83ZaZIUpcezkvFQznOTRfMr0XeXrMNB
vWsdn/BgPChrHasxLonPpJptubRgnUdLn2WSv1vyztX/HtFu0nOJ9vBLze/T
/GjNt2ntlWHnq9FfT6/0mBV8oD8kPT7Ke//wgNeDOb83x/j3kv+U1h6QDv+S
rvu0fq+e+2Xfh7Wn10VXlY+G6LlbfIZp37NE/57Wvpi1LpvbbLvB+l/JWPer
Wta/U3wuEL9V7ZbXrbmZGn9d88/l/X+t7Hq/6CcUHTvsjX3t0fwx+eVOza/P
mt/vJGuwaE5Kvceq3jXtYZvkTpKcK0uOt9XS8THN92jdRRXvAf2H6n2xnu7E
760dHq8M+8Bzo3TZpLVPKgb6xHOaeE/V83Pps1b6/EE0N2U9P0Zrd2l8hXQ7
Q++a5F5a8/jxsDO8t2p+jfhOEd1YyZ0mvlMLtt9EPUcytinfx2jtx3n7bgnx
UHZsvSWa5SWPibGXNb+l7BzbIv73V52TxBjxBc0lss/3687RJQ3nD7mzSro8
2zLve0WzWE9HwbyXh6yS1p0knstyXr857zy/SvuZpmexdFupvXwx9bdU33aJ
rq53f8H7YbxZfE4Wn279r2i+UnQMLNK6z9UcNxP1vqfhb9va7W98fYrsfXLT
/1tl647NN4vP2qwxZ7328ZLWH9P/OZKzQ3w3thsfiFHic2PBc+T7MPE7telv
D2vdkpZ5nU+8tYwln5Gc93PGjRlZ++Q9jUdmna/w/JHknK31Hfq/UXa4UGsv
bnPe/TNn3/1U+lzSMq+LRT9BzyLiRu/RqTHunZJtij3nis+fW5/kBu8O0TfJ
obzxFn+Q6+RaNePcY9wfvpstmr9WLQ9Z6Lc8YmaWeM/T3lv6P1+yllaMveDi
5xvGRugP5s1nmuhnyn87Nb846+9g3avi8Zqen5Ucd9fXHXvEBXvBh2D5/pxx
9kZ9vwmcIZbaTId/J8tHuYLjdUjWcqqB+WA0PuN9IPj0FByDYBM515l1rlEn
sOVcvZ/oMDaDN9/T/h6sGMPRa3rodoN0mdq0fuA3dmJfO0qmwSdgzJDAmQHp
2VJMjJasN2WT4Vq7Qf9H6j1HNFsyltkdcrHx/LAzetYK1nWS1l6mZ1DZ2NAT
+PA6sSD6a6ljRfuYWvBA0VjMnqk1/dJjnMatgmOCeCAuGINL2II8+3Ri2qNB
XyC/5acUzCw4D6nFxBZj8pHvv0hNM0JxPUvjjPj9Xe+vVl1b3td4nfS8tWD/
4SuJP+63HeG7q6XvgtBhl/g80/L/efLDa/o/O+feYIeennbjDr0AcnvFe7zi
8Ney2/iM94BfyDvWkXvk6uLI/fHiN7Xl2OkW/5dS4/lA7B0e+G0gxsjCJ+w9
odYoDpaK76fEZ6fW7864pwA7yHH8zXfqS1l6dei5r+CYJXaz1Mugpw+5XfIX
pp7/gvS6RfRHiG+9z5CssW2m3RvYdJ10eEd0r7UbO5/JGDMngCV68oqTv2nt
tdJtStF9BXLJl2WR98TJBPH+jnhtb7ce7IHYII9/k3Mug8GVuuOjX/xebDku
T9N8ve74IodbEVfoWw2dsSU2xB7ELvh2KPzzUGDCgPb93dABP+0OG06LnCNe
vlU3JoKH5M+Hgxy3+L8jYrJzkOfBD+xODUd/6P9X9Dw2xVbg7UDB9RK8fTzy
if6B/Ho8+hzeT8SY/mhl1PobZNuifL+nYOy/rWb8xydvVeyX00UzXM+jmr9T
+i/Wk8u7/vOmN9gi2h/XXV8O0z9oj7drfqHe65v2yQsl1y0wLBXtsKp9Qy85
KjV+LIo+Gnvyjdxl7+fp+7mpc536cE7q2vCKxqtS9z74qD/inB6TPgDcI1YO
54z/20Xf1B57NT6r6J4bXAXXx4fct0Xzben2qnR+LrUM+GN36KkX+B/sH5yx
nkdCV/SnV2APp0nORS334I/IfvP0f4RkrtB4tfhOb3PfsKFkfOtr2r7YdF3R
WIiP36sZR8CQmVn3GfWofZXAHzAMLCN+VsgXr7bcX9Fbbc26vzq9aewmjqCd
HfFGf4lv8MvW1FiM7H2in9Rw70AfuqFqWnS6NnraJ6l7VfcG9J7gIv3n+IL7
XvqHs1rGBTCBHAbLsBs+aS+53tKbToq1ffSHTX9bonVdqTEb3mAmNNgfHsQ8
eiCLXLhU+j5ddWymDfclzOODrpzPAttDv66oHRdGvp8JPjT8n1je23Q8r9Z7
atVyOBtsib4Xvz0QteljfZ+jtTvBQ41P0Z57S+6bDkTvBPaXU+P4rqjb1IjL
qAmiH1N27aUGgxec+Q7EWY36xnewYmzZPLAjtL1B/xPpfE7TvQC9J7KJE2p8
Js57J/oJ9jUz+jqxTk5rui98MfSiVo+Nsyp4PzZqAXhEnC8uOhboQ0Ypf7ek
7umpNReUTTu83X0EfR1+R290Zo5v2ASMBCuxP3blXIFtTxW/rQ3nG3tkDfQL
ZKvLGqZjL8QmeQRedgaW4pMH9fwj8f5mhh3+qP3dJt8sFK//isdHDdORJ8Qe
MVhquM6x58vr7p/AkDw42fA5lZhBFjhA/TgW9poevR+63ihZRzg3co7XXsal
PqfQo/2l4D6Nuj2u5toNVs0tWtbrkvNGw//Rf0b46O66+y3sxDnpQPiPPBgZ
PTm2p15j/66oUcR8J+eY1H3OM1XnCTmyu+BzBTa/uukzCecR6jv5CXZyRhlW
9zx4cHlgAhhHHwJm/kfrPlt1XrWDzS2foY7XmqJr9JOpc5J8pMZcU/a+qdvU
TvKSGsDZmD4SnCHnoH9C359KnedgHrrA842KMR085yyID4/jQ+peDcyYI92v
qbu/eLNifAfbwRK+04P1VF23qFmbok5iK74RZ8QYcYP/kXs4zgnUlClN94vk
WBc8arbLLZJ/c+rajB1YS11m3/iWvR+NnoKc4nw/JM59xMuGiBnuh+bHOYV7
lGrcqxDzG6IfAHfoO8GThTXrgQ7g/11l1wBiltglRx7R90db/sZZcEXNuU8d
eyhq2ZqG7ya4l+AM+mjN59D7Wj430odSz/m+Ns5YHdGrUKNmRtx2xxkOe9Jv
1lLHaK8+flhw/8C+qWfsnV6kL+bzknN9w3RHSz7jHe+H6r6LAeeIY+iJYWrn
u7GWud0x/7z4rG45Z7AhfSSxzbm5mDqHiYehUV/Kmr+x4fugzujFsD+4NSPw
nHMc+lJ/OQNlas7Ngt7zG5ZNr/lBwX7h7M4dEzWE8d7QjTV7AxPo78Br5IEV
+BlMoF88XsPafGdCPSffuW8jl4hD7po4t0NDn8F/8gdfbos+aqJscKmectm5
1Bn0X2q4LyHOqTn0RifOm5sj169rGH/BXs4nYP34qKvwA2velR12pj6/0kON
qTnX8Elv+IW6AV9qB/GNzafHPQP2JWbAtCNR33kPjv3iQ+7s8CN3qOAu5yZ6
bPxBraHf/GHVPecdsQf0/yB0wBe8d8YYrO+L+1jiAptTQztDBrWPu7CjEXvn
Sofz9AyUPDcQ90Xgy7roj77ZNC6ACVfUfY+zPvp27oaoR/DuC/6sORLzJ3qU
rrgfQCd6p6FxXsRW3M/S95PXd8kv16U+69C7sG/q6TrZ4MsN1xvO0A9ETeRe
7Wgh6kLWsY8NN9R8l3Qs+kV4Y/8F0a9gq6Xid2/Dvl+mfT1dMU9wF/zl7Dkk
zvys4Y5tbc33bNiXPSMLO/bHXQ3nitnRvy6PHMee7PvfsfdV4nFD0zzJlX2R
L3w/GP4iXjPR26A79kH/YVp7a8N7pj8a2TI9+r4cOnPO7Ioce77pXhls7Q8f
o89c2bM3tQ+O3zmWjRXY81DE6v8BdL07XA==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnmQl3Mcx79293fstr/n9zy//a1ydLmFSkOLSpeMGUYyKR3arKmUsck4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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7227425870868156, 0.7829084195174597, 0.9131216670435703], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmXm0lVUZxj/uPcM9595zzj0D19QsMCRzQlsgKoITFxAEXSpomozOutRl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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwll3uQlXUZx3972HPOnrN73ve87x52QTQXZ5UkMnEQxAS5CNrIpUlFhcRl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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7796540715823604, 0.852674740776912, 0.9064046791860324], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmgmUlNWVx79eqrqrqruqq6ua1mhOQlBWQQeC0KjglsMqIZCZxAXPqOgc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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3uQlWUdx1/27Nk957DnvOc97+EmNcOygMIGNCByETRE5SKgYhYJNnGz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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8324996452299218, 0.8702973314681988, 0.8827299095485677], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmgmQldWVx7/uRzfv9Xv9dmYGBx1wb4gYFROEaAImjhIdBBGXQDQjIAqK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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3twVdUVxrdJbl5w7zn3npu2WKpGdICqUPCVgkUd6wjpOBJo0WoYeclD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         "],
         VertexColors->None], PolygonBox[{{5415, 3495, 896, 3362, 3361}},
         VertexColors->None]}]}, 
     {RGBColor[0.8853452188774831, 0.8879199221594856, 0.8590551399111029], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmQlwldUVx7+XmJDl5eUtYeo20waKpQIOBJeAFcG2jtJiWxWnohVBQKBu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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnmQz3UYxz+7vz3Z7+9ck0b/VBQNjaOmHIWSUPqj1AwV5VyFlMjYLhVh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         "],
         VertexColors->None], PolygonBox[{{4240, 3512, 3333, 3376, 1560}},
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNl0ts1UUUxudSLrS9vY/2FlxAARNhYQBRRKOSUKCkJMaYkCAYZQNCLxBe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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwlk0lszXEQx6eevq7v/5b/qwsREkQEoQvBoSWRSBwkDkrExdr0hXqSWhIi
2kaI6pKIhFB9rxWi1YqDpamThCMOQlsHrdgOxHZEfMY3+U8y85/vzHxn+c3d
2bT5wDQz24hMR7oSZt3I0YhZosDsamh2BTmGb1uJ2cKk2UTKrKHYrA67Gmwu
ZtZcaDaKjOEbR/bhr8VfCX40MHtSZPYgajZErhHweYpm8HfGqYn0UvwWNW7j
H0By+Bvxr06qptd6HFWtwVA+r/mJWp+RFuL3EzNBrnGkD/79yE/0H3HlToAv
DcXRueXg+A57KlRsD/8uodfBtx59C/ISuwN7FrnWY5dXmK1KisuLIvV2lhkk
o+rxMvj7cCwvUI50Wpycyxh2N/o5/mVKzbJwek98Mq1ceTC1gTh4bceMozeR
/xfcL/BvN3o7mJkRcXrj+8GuRL8e0WxLQvXmM867HmqXO4i/AfYEHNqY1x9y
VpGvEtkElz3YPWCvhdr1DGKWg6+h3xH6q4NjCf1fpEYj+LYys4/488RbVDNf
kxLGfR5zCuxh/L8LtaNOcnchd8mVgd9p/GFas/Gb+YK+lpwDpJxNjj6w/aG4
+w1WJXVTfksPkW/08j2u3TqnHLGToXZZj70AbHNC+iD4eXD7So05ZarhtzYc
6hb95lrhcwT830LN6BX9VCdVaxf8JrHfIln0Q8jxuGbqs2wFvyylnfmuvMcG
/M/4txXs9lLlbqPGhohqZNCnUsq1GD5Z8E/hdw/sUjDrAs3EZ+E35dznV2g3
3sMZ8j2Kafd+gwcTyuGxfjO+29cpcfcdD9PLkkCz95vpjekN+9v1N58i/4dA
t+g7dS7ZmLg5p+fk2ovN97+n4ZRu2m95BXY7+h3+rSzWm74Z6Ib8drznDvxD
+GuKhXH9fFyx/s9vpSWmWfrN+KwXBXp7PnO//ZNpzdLfwD+C1aVd
         "],
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3464, 4827, 4392, 4410, 5387, 4409, 5388, 3490, 5137, 3813, 
         5136, 3814, 5138, 3753, 4258, 3815, 4363, 4058, 4059, 5140, 5139, 
         3462, 4826, 4825, 4874, 4873, 3514, 5038, 5039, 5035, 5037, 5036, 
         3489, 4148, 3488, 5327, 3781, 4241, 4240, 3512, 5025, 5026, 5024, 
         5028, 5027, 5414, 5020, 5021, 5019, 5023, 5022, 3513, 5413, 5358, 
         5412, 3487, 3649, 5357, 3650, 3511, 5018, 3648, 5017, 3647, 5259, 
         3985, 4783, 3981, 4782, 3982, 4784, 3980, 3984, 3983, 3486, 4082, 
         4083, 4035, 4081, 4080, 4679, 4668, 3478, 5385, 4667, 4405, 3633, 
         3752, 5325, 3460, 4215, 3459, 5324, 3965, 3751, 3964, 5323, 3458, 
         4214, 3457, 4841, 4840, 4839, 3632, 5090, 5091, 5089, 3745, 4213, 
         4390, 4389}],
        RowBox[{"-", "34"}]],
       Annotation[#, -34, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4055, 4056, 4147, 4146, 4824, 4823, 4404, 4403, 4402, 3772, 
         4233, 4232, 3517, 5084, 3740, 5083, 3739, 4527, 3629, 5007, 5008, 
         4657, 4656, 5082, 4207, 4208, 3953, 5319, 3441, 4200, 3435, 4834, 
         4833, 4832, 3626, 5077, 3726, 5076, 3725, 4196, 3724, 5247, 3944, 
         4815, 4816, 3430, 4195, 4326}],
        RowBox[{"-", "34"}]],
       Annotation[#, -34, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3465, 3481, 4123, 4124, 3667, 4435, 4436, 3520, 4593, 4611, 
         4591, 4592, 4590, 4589, 4434, 3666, 4057, 3463, 3480, 4121}],
        RowBox[{"-", "51"}]],
       Annotation[#, -51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3618, 4122, 4142, 4141, 4779, 3977, 4778, 4781, 4780, 3976, 
         3979, 5377, 3978, 3491, 3657, 5359, 3658}],
        RowBox[{"-", "51"}]],
       Annotation[#, -51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3973, 3645, 3779, 5364, 3778, 4239, 3508, 5333, 3803, 3507, 
         3802, 3800, 5127, 3801, 5128, 3799, 5262, 3994, 5260, 3993, 5261, 
         4206, 4204, 4205, 3952, 5318, 3440, 4199, 3434, 3472, 4831, 4830, 
         3625, 5006, 3429, 4814, 3723, 5075, 3722, 4267, 4268, 4034, 4047, 
         3428, 4269, 4108}],
        RowBox[{"-", "51"}]],
       Annotation[#, -51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4044, 5040, 3661, 4876, 4877, 4498, 3601, 4992, 4993, 4829, 
         4828, 3466, 3424, 5243, 3937, 5242, 3938, 4220, 4219, 3754, 5363, 
         3755, 3945, 4700, 4701, 4447, 4904, 3532, 4222, 5096, 5097, 4670, 
         4672, 4671, 4771, 4716, 4717, 4452, 4913, 3536, 4226, 3764, 5105, 
         3765, 5106, 3954, 3898, 5345, 3572, 4231, 3771, 4333, 3957, 4368, 
         4369, 4073, 3477, 3446, 4774, 3451, 5321, 4773, 4386, 5088, 3742, 
         4455, 4917, 4456, 5278, 5277, 3857, 5168, 3858, 5169, 3743, 5094, 
         3747, 5092, 3746, 5093, 4211, 3962, 5253, 3963, 5254, 3748, 3750, 
         5362, 3749, 3780, 4846, 4847, 3787, 3495, 5415, 5417}],
        RowBox[{"-", "51"}]],
       Annotation[#, -51, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4861, 4420, 4421, 4427, 4426, 4416, 4417, 3654, 4850, 4849, 
         4848, 3646, 5016, 4548, 4549, 4517, 5386, 4518, 5401, 4516, 5005, 
         5408, 5004}],
        RowBox[{"-", "51"}]],
       Annotation[#, -51, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3602, 4499, 3603, 4500, 4688, 5143, 3818, 4884, 4885, 4262, 
         3817, 5268, 4001, 5267, 4000, 3467, 3425, 4129, 4130, 4260, 4259, 
         4064, 4065, 4377, 4133, 3627, 4276, 4069, 4651, 4652, 4649, 3731, 
         4650, 3732, 4653, 4914, 3537, 4227, 3767, 5107, 5109, 5108, 5248, 
         3955, 5051, 3683, 4915, 4916, 4515, 3612, 4512, 4514, 4513, 4583, 
         5291, 5193, 5194, 5192, 3885, 5313, 5314, 4804, 4037, 5312, 3541, 
         4278, 4279, 3959, 4054, 3453, 4776, 3455, 5322, 4775, 4391, 4408, 
         4407, 3485, 4681, 4682, 3789, 5330, 3498, 4245, 3499, 5014, 5421}],
        RowBox[{"-", "68"}]],
       Annotation[#, -68, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3608, 4107, 3674, 4569, 4568, 5001, 3609, 5003, 5002, 5351, 
         5350, 4898, 5049, 3676, 5396, 4942, 4943, 3556, 4284, 4946, 4945, 
         4944, 3873, 4440, 3526, 4221, 5095, 3756, 4669, 3757, 3949, 5317, 
         3433, 5154, 3834, 5153, 5156, 5155, 3763, 5103, 5104, 4837, 4838, 
         3475, 3439, 4139, 3445, 5320, 3956, 4332, 4331, 3493, 4785, 4786, 
         4077, 4076, 4342, 3506, 5411, 5416}],
        RowBox[{"-", "68"}]],
       Annotation[#, -68, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3975, 3974, 4340, 4341, 3988, 4787, 3987, 4788, 4140, 4560, 
         4561, 4424, 4423, 4422, 3656, 4086, 3504, 5033, 5034, 4864, 4863, 
         4862, 3655, 4085, 3500, 4149, 4017, 5293, 4018, 5294, 4016, 4352, 
         4966, 4965, 4019, 5295, 3615, 5292, 4015, 4962, 4963, 4964, 4348, 
         4349, 4007, 4918, 4919, 4920, 4168, 4136, 4137, 4120, 3617, 4336, 
         3616, 5041, 5042, 4880, 4881, 5170, 3860, 4457, 4796, 4797, 4009, 
         4092, 4091, 4371, 3969, 4338, 3970, 5116, 3775, 5114, 3774, 5115, 
         4216, 4923, 4922, 4921, 3861, 5171, 4883, 4882, 5044, 5043, 4394, 
         4393, 3619, 4519, 4529, 4528, 4562, 3515, 5334, 3812, 3811, 3903, 
         5372, 3902, 3516, 3900, 5371, 3901, 3899, 5212, 5213, 5209, 5211, 
         5210, 4150, 3502, 5134, 5135, 5133, 3807, 4218, 4217, 4255, 3808, 
         4372, 4373, 4257, 4256, 3510, 3996, 5381, 5380, 5379, 3809, 4084}],
        RowBox[{"-", "68"}]],
       Annotation[#, -68, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3604, 4501, 4996, 4995, 4994, 5349, 5348, 4431, 5144, 5145, 
         4888, 4889, 3522, 4263, 3521, 4887, 4886, 4002, 5269, 4060, 4046, 
         4045, 4358, 4031, 4102, 4101, 3518, 4261, 3519, 4879, 5142, 5141, 
         3816, 5367, 4478, 4956, 4479, 4799, 4798, 3892, 5205, 3893, 4743, 
         5157, 4707, 3840, 4705, 3839, 4706, 3880, 4738, 5399, 3841, 5158, 
         3842, 4708, 3766, 4228, 4906, 5159, 3844, 5368, 4469, 4470, 4471, 
         4292, 4293, 3883, 5190, 3882, 5191, 3538, 5056, 5057, 4634, 5394, 
         3630, 5009, 3631, 4661, 4662, 4384, 4385, 3447, 4167, 3452, 4053, 
         4039, 4119, 4118, 3509, 5130, 3804, 5129, 5132, 5131, 3744, 4665, 
         4666, 4387, 4388, 3454, 4687, 3456, 4684, 4685, 4683, 3790}],
        RowBox[{"-", "85"}]],
       Annotation[#, -85, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3607, 4106, 3672, 4567, 4566, 4998, 5000, 4999, 4997, 4437, 
         4438, 4722, 3872, 4465, 4940, 4941, 4194, 4644, 4645, 4570, 3675, 
         4439, 4895, 4511, 3611, 4508, 4510, 4509, 4571, 5189, 4736, 4737, 
         4695, 5152, 3833, 4696, 3762, 4699, 3836, 4697, 3835, 4698, 3854, 
         4715, 5395, 3636, 5013, 3637, 5402, 4535, 4556, 5030, 3653, 4857}],
        RowBox[{"-", "85"}]],
       Annotation[#, -85, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4419, 4686, 3792, 5332, 3505, 4234, 3967, 5376, 3968, 3641, 
         5356, 4542, 5355, 3639, 5015, 3640, 5405, 4541, 4875, 5354, 4430, 
         5011, 5012, 3634, 5010, 4789, 5263, 5264, 4870, 4871, 4872, 4253, 
         4429, 4428, 4811, 3806, 4254, 3995, 4036, 5366, 3805, 4345, 3997, 
         5382, 3998, 3810, 5111, 3773, 5110, 5113, 5112, 5255, 4337, 3545, 
         5342, 4008, 4794, 4795, 4539, 5392, 4540, 5403, 4536, 4538, 5409, 
         4537, 3461, 3479, 3966, 4335, 4334, 3501, 5331, 3986, 4854, 4853}],
        RowBox[{"-", "85"}]],
       Annotation[#, -85, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3718, 4192, 4189, 3524, 4564, 4565, 5046, 5045, 3670, 4770, 
         3943, 4769, 3942, 4193, 3719, 5182, 3869, 5179, 5181, 5180, 3525, 
         5184, 5185, 5183, 5187, 5186, 3721, 5073, 5074, 4812, 4813, 3427, 
         5216, 3431, 4050, 4362, 4361, 3907, 4299, 3730, 4198, 4488, 4487, 
         3906, 5215, 4747, 5047, 5048, 4897, 5335, 4896, 3560, 4201, 4950, 
         4949, 4948, 3879, 4446, 3531, 4225, 5101, 5102, 4675, 4677, 4676, 
         4772, 4533, 4534, 4406, 4544, 5326, 3482, 4545, 5418}],
        RowBox[{"-", "102"}]],
       Annotation[#, -102, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{3865, 3552, 3783, 5365, 3782, 4242, 3542, 5339, 3859, 3563, 
         3685, 5341, 3544, 3701, 3543, 5340, 3684, 3889}],
        RowBox[{"-", "102"}]],
       Annotation[#, -102, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{4103, 4104, 4187, 4186, 3606, 4105, 4878, 4432, 3662, 4563, 
         4375, 4374, 3605, 5410, 4502, 3620, 4744, 3904, 4481, 4482, 4483, 
         4264, 4968, 4967, 5214, 4480, 4494, 4493, 3596, 4359, 4360, 4033, 
         5310, 4032, 5311, 3575, 4295, 3574, 4970, 4969, 4020, 5296, 4066, 
         4067, 4817, 4704, 3837, 4702, 3838, 4224, 4223, 5098, 3759, 4673, 
         3760, 4674, 3950, 4709, 3843, 4448, 4905, 4229, 3768, 4678, 3770, 
         4400, 4401, 3476, 3442, 4165, 4822, 4821, 5249, 3958, 5375, 4660, 
         5086, 5085, 4023, 5298, 5300, 5299, 4354, 5167, 5166, 5165, 4038, 
         5315, 4005, 4277, 3855, 4366, 4367, 4097, 3584, 5219, 4750, 4751, 
         4663, 5087, 3741, 4664, 3660, 5252, 3961, 5250, 3960, 5251, 3999, 
         5265, 5266, 4791, 4790, 3777, 5120, 3776, 5119, 4844, 4845}],
        RowBox[{"-", "102"}]],
       Annotation[#, -102, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3503, 4246, 4860, 4859, 4858, 3791, 4415, 3494, 4212, 4414, 
         4413, 4412, 3786, 4680, 3638, 4112, 3613, 4210, 3614, 4113, 4453, 
         4740, 3884, 4474, 4951, 3562, 4209, 3561, 4472, 4473, 5162, 5163, 
         5161, 4909, 4230, 3769, 4714, 3850, 5160, 3849, 5400, 4739, 3881, 
         4712, 3847, 4711, 3848, 4713, 3758, 4710, 3846, 4703, 3845, 4109, 
         3610, 4296, 3905, 4111, 4110, 4131, 4619, 4620, 4618, 4617, 4433, 
         3665, 4640, 4641, 4376, 4397, 5316, 3426, 3468, 4395, 5384, 4396, 
         4151, 5407, 3663, 4353, 3664, 4615, 4616, 4614, 3702, 5053, 4125, 
         4126, 3891, 4477, 4954, 4955, 4185, 4188, 5207, 3894, 5206, 3895, 
         5208, 3819, 5270, 4003, 4890, 4891, 4171, 3690, 5147, 3821, 5146, 
         3820, 4265, 4266, 4025, 5301, 4024, 5302, 4308, 4190, 5406, 4191, 
         3868, 4720, 5178, 5177, 3523, 4625, 4521, 4522, 4520, 3622, 4602, 
         3693, 4601, 3692, 4157, 3673, 4721, 3870, 4462, 4463, 4464, 3871, 
         5284, 5285, 5281, 5283, 5282, 4283, 3822, 5271, 4004, 4892, 4894, 
         4893, 5397, 3703, 5054, 3704, 4621, 3624, 4398, 4399, 3471, 3432, 
         4160, 3438, 4820, 4380, 3951, 4203, 4202, 3591, 4576, 4577, 4574, 
         3680, 4575, 3682, 5050, 4581, 4582, 4530, 4532, 5328, 4531, 5329, 
         3492, 4555, 5419}],
        RowBox[{"-", "119"}]],
       Annotation[#, -119, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{5031, 3659, 5032, 4144, 4145, 3972, 5257, 3971, 5258, 4236, 
         3571, 4961, 4960, 4984, 4983, 3594, 4981, 4982, 4979, 4980, 3593, 
         4313, 3592, 4978, 4977, 5309, 5308, 3716, 4638, 4637, 4445, 4573, 
         4572, 4648, 4132, 3559, 4947, 4468, 3696, 4609, 4087, 4694, 3831, 
         4442, 4443, 4444, 4197, 4315, 3924, 4347, 3925, 4316, 4986, 4643, 
         3720, 4642, 5072, 5071, 4128, 3579, 4155, 4156, 4328, 4327, 3671, 
         4154, 3691, 4063, 4062, 4310, 3621, 4127, 4491, 4153, 3669, 4324, 
         3668, 4152, 3697, 4176, 3469, 4061, 3941, 4767, 3940, 4768, 4801, 
         4800, 4973, 4972, 4971, 4022, 5297, 4021, 4809, 4810, 4806, 4808, 
         4807, 4346, 4297, 4298, 4042, 4486, 3577, 4285, 3576, 4485, 4484, 
         4746, 4745, 4068, 4049, 4048, 4578, 3681, 4449, 4622, 5336, 3533, 
         4907, 4623, 4624, 4524, 4526, 4525, 4655, 4579, 4580, 4450, 4630, 
         5337, 3534, 4908, 5398, 3709, 5055, 4631, 4633, 4632, 3628, 4303, 
         4071, 4072, 4381, 3443, 4166, 3448, 4051, 3856, 4115, 4114, 3483, 
         4546, 4547, 3788, 4851, 4852, 3496, 4238, 3497, 3794, 5125, 3795, 
         5126, 3793, 4247, 4425, 4248, 3989, 5420}],
        RowBox[{"-", "136"}]],
       Annotation[#, -136, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3484, 4843, 4842, 5118, 5117, 3644, 4364, 4078, 4079, 4170, 
         4169, 4116, 4117, 3862, 4459, 4926, 4927, 4237, 4777, 4718, 4719, 
         4454, 4792, 5338, 3540, 4793, 3539, 4090, 4006, 4307, 4306, 3583, 
         4304, 4305, 3911, 4749, 3910, 5218, 3582, 4754, 4755, 4659, 3737, 
         4658, 3738, 4164, 4163, 3436, 3473, 4070, 4040, 4135, 4134, 3578, 
         4162, 5080, 3733, 4654, 3734, 5233, 5234, 5232, 3923, 4729, 3875, 
         4728, 4731, 4730, 3874, 4733, 4734, 4732, 4753, 4752, 5220, 4724, 
         4725, 4723, 4727, 4726, 3717, 4646, 3727, 4378, 4379, 5244, 5245, 
         4989, 4990, 4991, 4325, 4497, 4496, 4495, 3939, 4766, 4028, 4764, 
         3917, 4492, 4802, 4803, 4029, 4100, 4099, 4357, 3947, 4330, 3948, 
         4329, 3946, 5307, 4030, 5304, 5306, 5305, 3589, 5238, 5149, 5150, 
         5148, 3825, 5151, 3706, 4177, 4441, 4271, 3826, 4365, 3827, 4272, 
         5078, 4693, 3830, 4691, 3829, 4692, 4158, 4172, 4173, 4159, 3677, 
         4302, 3678, 4627, 4628, 4626, 3707, 4089, 4088, 4314, 3679, 4161, 
         4451, 4138, 3635, 4235, 4074, 4543, 3643, 4411, 4554, 5029, 3652, 
         4855, 4856, 5123, 5124, 4865, 4866}],
        RowBox[{"-", "153"}]],
       Annotation[#, -153, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{4475, 3888, 4741, 4075, 4370, 3642, 4143, 4418, 5176, 3867, 
         4937, 4938, 4939, 4244, 3784, 5121, 3785, 5122, 3566, 5202, 3890, 
         5199, 5201, 5200, 4243, 3555, 4552, 4553, 4550, 3651, 4551, 3450, 
         4052, 3797, 4867, 4868, 4869, 3798, 3992, 5378, 3991, 4344, 3796, 
         4252, 3990, 5052, 3689, 4933, 4934, 5058, 3710, 4975, 4976, 4600, 
         3688, 4597, 4599, 4598, 3449, 4594, 3444, 4383, 4382, 4490, 4489, 
         3585, 4587, 4588, 4584, 4586, 4585, 3535, 4178, 4912, 4911, 4910, 
         3851, 5222, 3914, 5221, 3913, 4027, 5383, 4026, 4355, 4356, 4291, 
         4290, 4351, 4288, 4289, 4041, 4098, 3588, 5303, 5273, 5274, 5272, 
         5276, 5275, 3705, 4690, 3824, 4689, 3823, 4605, 3695, 4466, 4467, 
         3557, 4610, 4504, 4505, 4503, 4507, 4506, 4523, 4765, 3930, 4805, 
         3931, 4320, 3929, 4319, 4010, 5286, 4011, 5287, 3711, 5059, 5061, 
         5060, 3694, 4603, 4604, 5389, 5391, 5390, 3527, 5246, 3528, 4900, 
         4899, 4988, 4987, 3598, 4286, 3876, 5188, 3877, 3933, 5374, 3932, 
         3558, 4093, 3934, 5347, 3599, 4323, 3600, 5241, 3936, 5239, 3935, 
         5240, 4321, 4322, 4273, 3828, 4287, 5289, 4012, 5288, 4013, 5290, 
         3878, 5370, 4735, 5369, 3728, 5079, 3729, 4647, 4974, 3581, 4300, 
         4301, 3909, 4748, 3908, 5217, 3580, 5229, 3922, 5228, 5231, 5230, 
         3761, 5099, 5100, 4835, 4836, 3474, 3437, 5196, 5197, 5195, 3887, 
         5198, 3886, 4294, 4476, 4475}],
        RowBox[{"-", "170"}]],
       Annotation[#, -170, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{5346, 3912, 4095, 4096, 3918, 5224, 3919, 5225, 3735, 5081, 
         3736, 4818, 4819, 4629, 3708, 4613, 3699, 4612, 3700, 3713, 5360, 
         3712, 3573, 3714, 5361, 3715, 3595, 4985, 5393, 3623, 5352, 5353, 
         5069, 3470, 5070, 4639, 4635, 4636, 4606, 4608, 4607, 3570, 4274, 
         4275, 3832, 4901, 4902, 3529, 4903, 3530, 5236, 5237, 5235, 3926, 
         4318, 4317, 3597, 4182, 4183, 4043, 4181, 4180, 4184, 4174, 4175, 
         3853, 5164, 3852, 4742, 5204, 5203, 3564, 4094, 4014, 4952, 4953, 
         3565, 4339, 5256, 5172, 5173, 4924, 4925, 3546, 4280, 3547, 4458, 
         3866, 4935, 4936, 3554, 4282, 3553, 4558, 4559, 4557, 4761, 4760, 
         5223, 3916, 4759, 4763, 4762, 3915, 4756, 4758, 4757, 4281, 4350, 
         3550, 4932, 4931, 5280, 5279, 3863, 4250, 5404, 4249, 4343, 4251, 
         3551, 4461, 4460, 3864, 3928, 5373, 3927, 5065, 5066, 5064, 5068, 
         5067, 3687, 4595, 3686, 4596, 4928, 3548, 4179, 3549, 4930, 4929, 
         5175, 5174, 3698, 4957, 4958, 4959, 5063, 5062, 3569, 5227, 3920, 
         5226, 3921, 4312, 4311, 3590, 3897, 5344, 3568, 4270, 3567, 5343, 
         3896, 3586, 4309, 3587, 5346}],
        RowBox[{"-", "187"}]],
       Annotation[#, -187, "Tooltip"]& ]}, {}, {}}, {InsetBox[
      RowBox[{"-", "187"}], 5422], InsetBox[
      RowBox[{"-", "170"}], 5423], InsetBox[
      RowBox[{"-", "153"}], 5424], InsetBox[
      RowBox[{"-", "136"}], 5425], InsetBox[
      RowBox[{"-", "119"}], 5426], InsetBox[
      RowBox[{"-", "102"}], 5427], InsetBox[
      RowBox[{"-", "102"}], 5428], InsetBox[
      RowBox[{"-", "102"}], 5429], InsetBox[
      RowBox[{"-", "85"}], 5430], InsetBox[
      RowBox[{"-", "85"}], 5431], InsetBox[
      RowBox[{"-", "85"}], 5432], InsetBox[
      RowBox[{"-", "68"}], 5433], InsetBox[
      RowBox[{"-", "68"}], 5434], InsetBox[
      RowBox[{"-", "68"}], 5435], InsetBox[
      RowBox[{"-", "51"}], 5436], InsetBox[
      RowBox[{"-", "51"}], 5437], InsetBox[
      RowBox[{"-", "51"}], 5438], InsetBox[
      RowBox[{"-", "51"}], 5439], InsetBox[
      RowBox[{"-", "51"}], 5440], InsetBox[
      RowBox[{"-", "34"}], 5441], InsetBox[
      RowBox[{"-", "34"}], 5442]}}],
  AspectRatio->1,
  Epilog->{
    RGBColor[1, 0, 0], 
    PointSize[0.02], 
    InsetBox[
     FormBox["\"global minimum\"", TraditionalForm], {-0.995, -2.092}], 
    PointBox[{-0.995, -1.992}], 
    InsetBox[
     FormBox["\"local minimum\"", TraditionalForm], {0.5304, 1.2191}], 
    PointBox[{1.2304, 1.2191}]},
  Frame->True,
  ImageSize->{266., Automatic},
  PlotRange->{{-3, 2}, {-3, 2}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{270, 267},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->491078450]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Minimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Minimize"]], "InlineFormula"],
 " finds a global minimum and can work in infinite precision:"
}], "ExampleText",
 CellID->1708706374],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FormBox[
      RowBox[{
       RowBox[{
        RowBox[{"-", " ", "100"}], "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "-", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "-", "1"}], ")"}], "2"], "+", "1"}], ")"}]}], "-", 
       RowBox[{"200", "/", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "+", "1"}], ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"y", "+", "2"}], ")"}], "2"], "+", "1"}], ")"}]}]}],
      TraditionalForm], ",", " ", 
     FormBox[
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ">", "3"}],
      TraditionalForm]}], " ", "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1528684992],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"9450000000000", "-", 
       RowBox[{"1185300000000", " ", "#1"}], "+", 
       RowBox[{"39312000000", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       RowBox[{"1872000000", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       RowBox[{"18099900", " ", 
        SuperscriptBox["#1", "4"]}], "+", 
       RowBox[{"48841", " ", 
        SuperscriptBox["#1", "5"]}]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"289", "-", 
          RowBox[{"1785", " ", "#1"}], "+", 
          RowBox[{"234", " ", 
           SuperscriptBox["#1", "2"]}], "+", 
          RowBox[{"1326", " ", 
           SuperscriptBox["#1", "3"]}], "-", 
          RowBox[{"507", " ", 
           SuperscriptBox["#1", "4"]}], "+", 
          RowBox[{"507", " ", 
           SuperscriptBox["#1", "5"]}]}], "&"}], ",", "1"}], "]"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"-", "595"}], "-", 
          RowBox[{"2077", " ", "#1"}], "+", 
          RowBox[{"1586", " ", 
           SuperscriptBox["#1", "2"]}], "+", 
          RowBox[{"910", " ", 
           SuperscriptBox["#1", "3"]}], "+", 
          RowBox[{"169", " ", 
           SuperscriptBox["#1", "4"]}], "+", 
          RowBox[{"169", " ", 
           SuperscriptBox["#1", "5"]}]}], "&"}], ",", "1"}], "]"}]}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{460, 75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->472749798]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1056055790],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "207.1598969820089`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.9948613347360095`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "1.9922920021040142`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{271, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->15996015]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->15625],

Cell["\<\
If the constraint region is empty, the algorithm will not converge:\
\>", "ExampleText",
 CellID->687871453],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], "\[GreaterEqual]", " ", "2"}], " ", "&&", 
      " ", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], "\[LessEqual]", " ", "1"}]}]}], "}"}], ",",
    
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1311741547],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"eit\"\>"}], ":", 
  " ", "\<\"The algorithm does not converge to the tolerance of \
\\!\\(4.806217383937354`*^-6\\) in \\!\\(500\\) iterations. The best \
estimated solution, with feasibility residual, KKT residual or complementary \
residual of \\!\\({0.5094107859477752`, 1.0546072559086298`*^-6, \
8.290002787469562`*^-12}\\), is returned.\"\>"}]], "Message", "MSG",
 CellID->539395986],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.7374756314177864`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.8687378157094133`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.8687378157083729`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->931508770]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1437064019],

Cell["\<\
If the minimum value is not finite, the algorithm will not converge:\
\>", "ExampleText",
 CellID->107225626],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"Tan", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->749723501],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"lstol\"\>"}], ":", 
  " ", "\<\"The line search decreased the step size to within tolerance \
specified by AccuracyGoal and PrecisionGoal but was unable to find a \
sufficient decrease in the function.  You may need more than \
\\!\\(MachinePrecision\\) digits of working precision to meet these \
tolerances. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/FindMinimum/lstol\\\", ButtonNote -> \
\\\"FindMinimum::lstol\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->395290730],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "3.058628940849363`*^14"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "1.5707963267948999`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{201, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->420563618]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       RowBox[{"x", "^", "2"}]}], "-", 
      RowBox[{"y", "^", "2"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], ">", "10"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1357681257],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"eit\"\>"}], ":", 
  " ", "\<\"The algorithm does not converge to the tolerance of \
\\!\\(4.806217383937354`*^-6\\) in \\!\\(500\\) iterations. The best \
estimated solution, with feasibility residual, KKT residual or complementary \
residual of \\!\\({0.43151582754954465`, 7.312348927125921`, \
0.1593467131318663`}\\), is returned.\"\>"}]], "Message", "MSG",
 CellID->51181816],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "26.208433894735613`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "3.61997471639897`"}], ",", 
     RowBox[{"y", "\[Rule]", "3.619974716398971`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->270541581]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1615498026],

Cell["\<\
The integer linear programming algorithm is only available for machine number \
problems:\
\>", "ExampleText",
 CellID->2001017660],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "+", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "+", 
        RowBox[{"2", "y"}]}], "\[GreaterEqual]", "3"}], "&&", 
      RowBox[{"x", ">", "1"}], "&&", 
      RowBox[{"y", ">", "1"}], "&&", 
      RowBox[{"x", "\[Element]", "Integers"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1252111367],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"lpip\"\>"}], ":", 
  " ", "\<\"Warning: Problem specified contains integer variables. Only \
machine number algorithm is available; the problem will be converted into \
machine precision.\"\>"}]], "Message", "MSG",
 CellID->226262499],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1"}], ",", 
     RowBox[{"y", "\[Rule]", "1.`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{136, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->448448890]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1986880533],

Cell["\<\
Sometimes providing a suitable starting point can help the algorithm to \
converge:\
\>", "ExampleText",
 CellID->585941183],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Cos", "[", "x", "]"}], "-", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "0.5"}], ")"}], " ", "y"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->831389318],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"eit\"\>"}], ":", 
  " ", "\<\"The algorithm does not converge to the tolerance of \
\\!\\(4.806217383937354`*^-6\\) in \\!\\(500\\) iterations. The best \
estimated solution, with feasibility residual, KKT residual or complementary \
residual of \\!\\({2.6522780976498024`, 0.9410342449205663`, \
0.05992969652258977`}\\), is returned.\"\>"}]], "Message", "MSG",
 CellID->397531238],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.7117655966513841`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.762852369357621`"}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "0.5164658455857531`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{277, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->583573647]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Cos", "[", "x", "]"}], "-", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "0.5"}], ")"}], " ", "y"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "<", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->765500604],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.6004629968908979`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.6572187833876794`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "0.7536998547122109`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{271, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->65417947]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->15669],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindMaximum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMaximum"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Minimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Minimize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindFit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindFit"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindRoot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindRoot"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Min",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Min"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LinearProgramming",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearProgramming"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["D",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/D"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CholeskyDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CholeskyDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->7403]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->5025],

Cell[TextData[ButtonBox["Unconstrained Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/UnconstrainedOptimizationOverview"]], "Tutorials",
 CellID->123070281],

Cell[TextData[ButtonBox["Numerical Mathematics: Basic Operations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalMathematics-BasicOperations"]], "Tutorials",
 CellID->123709578],

Cell[TextData[ButtonBox["Numerical Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NumericalOptimization"]], "Tutorials",
 CellID->9521]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->32561],

Cell[TextData[{
 ButtonBox["Demonstrations with FindMinimum",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=FindMinimum"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#13028"], None}]], "RelatedLinks",
 CellID->15098],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
FindMinimum"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->72541878],

Cell[TextData[ButtonBox["Discrete Mathematics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteMathematics"]], "MoreAbout",
 CellID->360546794],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->499090032]
}, Open  ]],

Cell["New in 1  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FindMinimum - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 43, 2.4685064}", "context" -> "System`", 
    "keywords" -> {
     "conjugate gradient", "constrained local optimization", 
      "constrained optimization", "convex programming", 
      "Gauss-Newton algorithm", "GNA", "gradient", "gradient descent", 
      "interior point algorithm", "Levenberg-Marquardt method", 
      "linear fractional programming", "linear programming", 
      "local minimization", "local optimization", "minimization", 
      "Newton's method", "numerical minimization", "quadratic programming", 
      "quasi-Newton", "steepest-descent"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "FindMinimum[f, {x, x_0}] searches for a local minimum in f, starting \
from the point x = x_0. FindMinimum[f, {{x, x_0}, {y, y_0}, ...}] searches \
for a local minimum in a function of several variables. FindMinimum[{f, \
cons}, {{x, x_0}, {y, y_0}, ...}] searches for a local minimum subject to the \
constraints cons. FindMinimum[{f, cons}, {x, y, ...}] starts from a point \
within the region defined by the constraints.", 
    "synonyms" -> {"find minimum"}, "title" -> "FindMinimum", "type" -> 
    "Symbol", "uri" -> "ref/FindMinimum"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[22885, 699, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->9409493]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 388681, 8143}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3204, 69, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3827, 96, 4195, 83, 70, "ObjectNameGrid"],
Cell[8025, 181, 3402, 109, 70, "Usage",
 CellID->27950]
}, Open  ]],
Cell[CellGroupData[{
Cell[11464, 295, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[11942, 312, 935, 36, 70, "Notes",
 CellID->1268],
Cell[12880, 350, 175, 4, 70, "Notes",
 CellID->25479],
Cell[13058, 356, 330, 9, 70, "Notes",
 CellID->19298],
Cell[13391, 367, 386, 11, 70, "Notes",
 CellID->7438],
Cell[13780, 380, 474, 17, 70, "Notes",
 CellID->23526],
Cell[14257, 399, 947, 36, 70, "Notes",
 CellID->23431],
Cell[15207, 437, 958, 36, 70, "Notes",
 CellID->27865],
Cell[16168, 475, 416, 14, 70, "Notes",
 CellID->31288],
Cell[16587, 491, 83, 1, 70, "Notes",
 CellID->23646],
Cell[16673, 494, 575, 19, 70, "Notes",
 CellID->21374],
Cell[17251, 515, 69, 1, 70, "Notes",
 CellID->30796],
Cell[17323, 518, 3205, 90, 70, "3ColumnTableMod",
 CellID->3307],
Cell[20531, 610, 520, 20, 70, "Notes",
 CellID->24106],
Cell[21054, 632, 472, 14, 70, "Notes",
 CellID->18946],
Cell[21529, 648, 505, 17, 70, "Notes",
 CellID->237],
Cell[22037, 667, 811, 27, 70, "Notes",
 CellID->547912957]
}, Closed]],
Cell[CellGroupData[{
Cell[22885, 699, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->9409493],
Cell[CellGroupData[{
Cell[23268, 713, 148, 5, 70, "ExampleSection",
 CellID->263648285],
Cell[23419, 720, 196, 7, 70, "ExampleText",
 CellID->499988759],
Cell[CellGroupData[{
Cell[23640, 731, 235, 8, 28, "Input",
 CellID->376441841],
Cell[23878, 741, 314, 11, 36, "Output",
 CellID->19645436]
}, Open  ]],
Cell[CellGroupData[{
Cell[24229, 757, 239, 8, 28, "Input",
 CellID->347120420],
Cell[24471, 767, 3233, 57, 123, "Output",
 Evaluatable->False,
 CellID->65591389]
}, Open  ]],
Cell[27719, 827, 144, 5, 70, "ExampleText",
 CellID->452807755],
Cell[CellGroupData[{
Cell[27888, 836, 300, 10, 70, "Input",
 CellID->28565156],
Cell[28191, 848, 174, 5, 36, "Output",
 CellID->53541514]
}, Open  ]],
Cell[28380, 856, 125, 3, 70, "ExampleDelimiter",
 CellID->986451359],
Cell[28508, 861, 357, 12, 70, "ExampleText",
 CellID->1560461396],
Cell[CellGroupData[{
Cell[28890, 877, 347, 11, 70, "Input",
 CellID->678564151],
Cell[29240, 890, 313, 11, 36, "Output",
 CellID->18197748]
}, Open  ]],
Cell[29568, 904, 126, 3, 70, "ExampleDelimiter",
 CellID->1519398755],
Cell[29697, 909, 135, 4, 70, "ExampleText",
 CellID->1896901598],
Cell[CellGroupData[{
Cell[29857, 917, 522, 16, 70, "Input",
 CellID->1202084729],
Cell[30382, 935, 321, 11, 36, "Output",
 CellID->104164648]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[30752, 952, 216, 7, 70, "ExampleSection",
 CellID->26449],
Cell[30971, 961, 116, 3, 70, "ExampleText",
 CellID->62007641],
Cell[CellGroupData[{
Cell[31112, 968, 234, 8, 70, "Input",
 CellID->66723803],
Cell[31349, 978, 315, 11, 36, "Output",
 CellID->599965091]
}, Open  ]],
Cell[CellGroupData[{
Cell[31701, 994, 236, 8, 70, "Input",
 CellID->386171924],
Cell[31940, 1004, 314, 11, 36, "Output",
 CellID->134194003]
}, Open  ]],
Cell[32269, 1018, 125, 3, 70, "ExampleDelimiter",
 CellID->666042395],
Cell[32397, 1023, 300, 11, 70, "ExampleText",
 CellID->516388244],
Cell[CellGroupData[{
Cell[32722, 1038, 340, 12, 70, "Input",
 CellID->46634681],
Cell[33065, 1052, 375, 13, 36, "Output",
 CellID->463680148]
}, Open  ]],
Cell[33455, 1068, 125, 3, 70, "ExampleDelimiter",
 CellID->242324886],
Cell[33583, 1073, 83, 1, 70, "ExampleText",
 CellID->681774368],
Cell[CellGroupData[{
Cell[33691, 1078, 503, 18, 70, "Input",
 CellID->852578689],
Cell[34197, 1098, 415, 14, 36, "Output",
 CellID->655892637]
}, Open  ]],
Cell[34627, 1115, 124, 3, 70, "ExampleDelimiter",
 CellID->10112781],
Cell[34754, 1120, 87, 1, 70, "ExampleText",
 CellID->369156266],
Cell[CellGroupData[{
Cell[34866, 1125, 396, 14, 70, "Input",
 CellID->59114888],
Cell[35265, 1141, 415, 14, 36, "Output",
 CellID->475825784]
}, Open  ]],
Cell[35695, 1158, 125, 3, 70, "ExampleDelimiter",
 CellID->685666782],
Cell[35823, 1163, 124, 3, 70, "ExampleText",
 CellID->169051774],
Cell[CellGroupData[{
Cell[35972, 1170, 492, 16, 70, "Input",
 CellID->1991020309],
Cell[36467, 1188, 353, 11, 36, "Output",
 CellID->6104705]
}, Open  ]],
Cell[CellGroupData[{
Cell[36857, 1204, 545, 17, 70, "Input",
 CellID->833915335],
Cell[37405, 1223, 322, 11, 36, "Output",
 CellID->29511136]
}, Open  ]],
Cell[37742, 1237, 123, 3, 70, "ExampleDelimiter",
 CellID->5114145],
Cell[37868, 1242, 195, 7, 70, "ExampleText",
 CellID->284316099],
Cell[CellGroupData[{
Cell[38088, 1253, 660, 23, 70, "Input",
 CellID->1963030529],
Cell[38751, 1278, 431, 14, 36, "Output",
 CellID->423130414]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[39231, 1298, 217, 7, 70, "ExampleSection",
 CellID->7079],
Cell[CellGroupData[{
Cell[39473, 1309, 245, 7, 70, "ExampleSubsection",
 CellID->25956],
Cell[39721, 1318, 782, 28, 70, "ExampleText",
 CellID->764461149],
Cell[CellGroupData[{
Cell[40528, 1350, 339, 10, 70, "Input",
 CellID->1625295724],
Cell[40870, 1362, 319, 11, 36, "Output",
 CellID->545725341]
}, Open  ]],
Cell[41204, 1376, 126, 3, 70, "ExampleDelimiter",
 CellID->1055000278],
Cell[41333, 1381, 858, 28, 70, "ExampleText",
 CellID->1329553122],
Cell[CellGroupData[{
Cell[42216, 1413, 341, 10, 70, "Input",
 CellID->1468387723],
Cell[42560, 1425, 445, 8, 70, "Message",
 CellID->476395254],
Cell[43008, 1435, 318, 11, 36, "Output",
 CellID->445600503]
}, Open  ]],
Cell[43341, 1449, 244, 8, 70, "ExampleText",
 CellID->1199314865],
Cell[CellGroupData[{
Cell[43610, 1461, 395, 11, 70, "Input",
 CellID->39134653],
Cell[44008, 1474, 440, 15, 54, "Output",
 CellID->74943998]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44497, 1495, 234, 7, 70, "ExampleSubsection",
 CellID->31705],
Cell[44734, 1504, 80, 1, 70, "ExampleText",
 CellID->28034064],
Cell[CellGroupData[{
Cell[44839, 1509, 453, 14, 70, "Input",
 CellID->19607354],
Cell[45295, 1525, 691, 21, 138, "Output",
 CellID->36865338]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[46035, 1552, 225, 7, 70, "ExampleSubsection",
 CellID->12053],
Cell[46263, 1561, 149, 4, 70, "ExampleText",
 CellID->858592029],
Cell[CellGroupData[{
Cell[46437, 1569, 579, 18, 70, "Input",
 CellID->2076542040],
Cell[47019, 1589, 397, 14, 36, "Output",
 CellID->154322767]
}, Open  ]],
Cell[47431, 1606, 88, 1, 70, "ExampleText",
 CellID->1859244329],
Cell[CellGroupData[{
Cell[47544, 1611, 1584, 50, 70, "Input",
 CellID->1427703802],
Cell[49131, 1663, 398, 14, 36, "Output",
 CellID->207428351]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[49578, 1683, 223, 7, 70, "ExampleSubsection",
 CellID->16079],
Cell[49804, 1692, 119, 3, 70, "ExampleText",
 CellID->993924491],
Cell[CellGroupData[{
Cell[49948, 1699, 390, 13, 70, "Input",
 CellID->782571855],
Cell[50341, 1714, 616, 11, 70, "Message",
 CellID->440544294],
Cell[50960, 1727, 414, 13, 36, "Output",
 CellID->40644518]
}, Open  ]],
Cell[CellGroupData[{
Cell[51411, 1745, 462, 15, 70, "Input",
 CellID->939473422],
Cell[51876, 1762, 262, 6, 70, "Message",
 CellID->290095149],
Cell[52141, 1770, 414, 13, 36, "Output",
 CellID->85583272]
}, Open  ]],
Cell[52570, 1786, 139, 4, 70, "ExampleText",
 CellID->1614271486],
Cell[CellGroupData[{
Cell[52734, 1794, 455, 14, 70, "Input",
 CellID->736570868],
Cell[53192, 1810, 400, 13, 36, "Output",
 CellID->73519019]
}, Open  ]],
Cell[53607, 1826, 224, 7, 70, "ExampleText",
 CellID->1954260595],
Cell[CellGroupData[{
Cell[53856, 1837, 508, 17, 70, "Input",
 CellID->46889232],
Cell[54367, 1856, 416, 13, 36, "Output",
 CellID->43239344]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[54832, 1875, 227, 7, 70, "ExampleSubsection",
 CellID->2025],
Cell[55062, 1884, 263, 8, 70, "ExampleText",
 CellID->1691254454],
Cell[55328, 1894, 1206, 41, 70, "Input",
 CellID->1514338379],
Cell[CellGroupData[{
Cell[56559, 1939, 1351, 44, 70, "Input",
 CellID->1350533918],
Cell[57913, 1985, 147081, 2465, 70, "Output",
 CellID->97978994]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[205043, 4456, 233, 7, 70, "ExampleSubsection",
 CellID->26397],
Cell[205279, 4465, 546, 20, 70, "ExampleText",
 CellID->549268746],
Cell[CellGroupData[{
Cell[205850, 4489, 542, 17, 70, "Input",
 CellID->1164918439],
Cell[206395, 4508, 482, 15, 36, "Output",
 CellID->854186613]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[206938, 4530, 222, 7, 70, "ExampleSection",
 CellID->6282],
Cell[207163, 4539, 159, 4, 70, "ExampleText",
 CellID->671779067],
Cell[207325, 4545, 916, 20, 70, "Input",
 CellID->630754998],
Cell[208244, 4567, 92, 1, 70, "ExampleText",
 CellID->519863318],
Cell[CellGroupData[{
Cell[208361, 4572, 159, 5, 70, "Input",
 CellID->114999578],
Cell[208523, 4579, 243, 8, 36, "Output",
 CellID->495299524]
}, Open  ]],
Cell[CellGroupData[{
Cell[208803, 4592, 180, 5, 70, "Input",
 CellID->225045130],
Cell[208986, 4599, 389, 13, 36, "Output",
 CellID->137793996]
}, Open  ]],
Cell[CellGroupData[{
Cell[209412, 4617, 316, 11, 70, "Input",
 CellID->77110343],
Cell[209731, 4630, 279, 9, 38, "Output",
 CellID->55176005]
}, Open  ]],
Cell[210025, 4642, 188, 4, 70, "ExampleText",
 CellID->1315314347],
Cell[CellGroupData[{
Cell[210238, 4650, 630, 19, 70, "Input",
 CellID->721185121],
Cell[210871, 4671, 470, 15, 38, "Output",
 CellID->276323235]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[211390, 4692, 233, 7, 70, "ExampleSection",
 CellID->21468],
Cell[211626, 4701, 377, 12, 70, "ExampleText",
 CellID->1650805990],
Cell[CellGroupData[{
Cell[212028, 4717, 1138, 39, 70, "Input",
 CellID->2092335573],
Cell[213169, 4758, 392, 13, 36, "Output",
 CellID->33929099]
}, Open  ]],
Cell[CellGroupData[{
Cell[213598, 4776, 1079, 36, 70, "Input",
 CellID->827788029],
Cell[214680, 4814, 658, 11, 70, "Message",
 CellID->57058932],
Cell[215341, 4827, 433, 14, 36, "Output",
 CellID->33288165]
}, Open  ]],
Cell[CellGroupData[{
Cell[215811, 4846, 2165, 69, 70, "Input",
 CellID->200111634],
Cell[217979, 4917, 150682, 2532, 70, "Output",
 CellID->491078450]
}, Open  ]],
Cell[368676, 7452, 237, 7, 70, "ExampleText",
 CellID->1708706374],
Cell[CellGroupData[{
Cell[368938, 7463, 1079, 36, 70, "Input",
 CellID->1528684992],
Cell[370020, 7501, 1738, 52, 96, "Output",
 CellID->472749798]
}, Open  ]],
Cell[CellGroupData[{
Cell[371795, 7558, 98, 3, 70, "Input",
 CellID->1056055790],
Cell[371896, 7563, 432, 14, 36, "Output",
 CellID->15996015]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[372377, 7583, 226, 7, 70, "ExampleSection",
 CellID->15625],
Cell[372606, 7592, 118, 3, 70, "ExampleText",
 CellID->687871453],
Cell[CellGroupData[{
Cell[372749, 7599, 569, 20, 70, "Input",
 CellID->1311741547],
Cell[373321, 7621, 449, 8, 70, "Message",
 CellID->539395986],
Cell[373773, 7631, 374, 12, 36, "Output",
 CellID->931508770]
}, Open  ]],
Cell[374162, 7646, 126, 3, 70, "ExampleDelimiter",
 CellID->1437064019],
Cell[374291, 7651, 119, 3, 70, "ExampleText",
 CellID->107225626],
Cell[CellGroupData[{
Cell[374435, 7658, 210, 7, 70, "Input",
 CellID->749723501],
Cell[374648, 7667, 616, 11, 70, "Message",
 CellID->395290730],
Cell[375267, 7680, 319, 11, 39, "Output",
 CellID->420563618]
}, Open  ]],
Cell[CellGroupData[{
Cell[375623, 7696, 442, 16, 70, "Input",
 CellID->1357681257],
Cell[376068, 7714, 440, 8, 70, "Message",
 CellID->51181816],
Cell[376511, 7724, 390, 13, 36, "Output",
 CellID->270541581]
}, Open  ]],
Cell[376916, 7740, 126, 3, 70, "ExampleDelimiter",
 CellID->1615498026],
Cell[377045, 7745, 141, 4, 70, "ExampleText",
 CellID->2001017660],
Cell[CellGroupData[{
Cell[377211, 7753, 550, 17, 70, "Input",
 CellID->1252111367],
Cell[377764, 7772, 294, 6, 70, "Message",
 CellID->226262499],
Cell[378061, 7780, 321, 11, 36, "Output",
 CellID->448448890]
}, Open  ]],
Cell[378397, 7794, 126, 3, 70, "ExampleDelimiter",
 CellID->1986880533],
Cell[378526, 7799, 134, 4, 70, "ExampleText",
 CellID->585941183],
Cell[CellGroupData[{
Cell[378685, 7807, 520, 18, 70, "Input",
 CellID->831389318],
Cell[379208, 7827, 442, 8, 70, "Message",
 CellID->397531238],
Cell[379653, 7837, 411, 13, 36, "Output",
 CellID->583573647]
}, Open  ]],
Cell[CellGroupData[{
Cell[380101, 7855, 672, 24, 70, "Input",
 CellID->765500604],
Cell[380776, 7881, 433, 14, 36, "Output",
 CellID->65417947]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[381270, 7902, 311, 9, 70, "SeeAlsoSection",
 CellID->15669],
Cell[381584, 7913, 2191, 80, 70, "SeeAlso",
 CellID->7403]
}, Open  ]],
Cell[CellGroupData[{
Cell[383812, 7998, 313, 9, 70, "TutorialsSection",
 CellID->5025],
Cell[384128, 8009, 177, 4, 70, "Tutorials",
 CellID->123070281],
Cell[384308, 8015, 193, 4, 70, "Tutorials",
 CellID->123709578],
Cell[384504, 8021, 153, 3, 70, "Tutorials",
 CellID->9521]
}, Open  ]],
Cell[CellGroupData[{
Cell[384694, 8029, 321, 9, 70, "RelatedLinksSection",
 CellID->32561],
Cell[385018, 8040, 367, 12, 70, "RelatedLinks"],
Cell[385388, 8054, 267, 6, 70, "RelatedLinks",
 CellID->15098],
Cell[385658, 8062, 389, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[386084, 8080, 318, 9, 70, "MoreAboutSection",
 CellID->72541878],
Cell[386405, 8091, 151, 3, 70, "MoreAbout",
 CellID->360546794],
Cell[386559, 8096, 179, 3, 70, "MoreAbout",
 CellID->499090032]
}, Open  ]],
Cell[386753, 8102, 50, 0, 70, "History"],
Cell[386806, 8104, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

