(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    176866,       4166]
NotebookOptionsPosition[    165418,       3771]
NotebookOutlinePosition[    167003,       3814]
CellTagsIndexPosition[    166916,       3809]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Fourier Transforms" :> 
          Documentation`HelpLookup["paclet:tutorial/FourierTransforms"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Discrete Fourier Transforms\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"FourierDST" :> 
          Documentation`HelpLookup["paclet:ref/FourierDST"], "Fourier" :> 
          Documentation`HelpLookup["paclet:ref/Fourier"], 
          "FourierCosTransform" :> 
          Documentation`HelpLookup["paclet:ref/FourierCosTransform"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"FourierDST\"\>", 
       2->"\<\"Fourier\"\>", 
       3->"\<\"FourierCosTransform\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Image Processing" :> 
          Documentation`HelpLookup["paclet:guide/ImageProcessing"], 
          "Integral Transforms" :> 
          Documentation`HelpLookup["paclet:guide/IntegralTransforms"], 
          "Signal Processing" :> 
          Documentation`HelpLookup["paclet:guide/SignalProcessing"], 
          "New in 6.0: Numerical Data Handling" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumericalDataHandling"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Image Processing\"\>", 
       2->"\<\"Integral Transforms\"\>", 3->"\<\"Signal Processing\"\>", 
       4->"\<\"New in 6.0: Numerical Data Handling\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["FourierDCT", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FourierDCT",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FourierDCT"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]finds the Fourier discrete cosine transform of a list \
of real numbers."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["FourierDCT",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FourierDCT"], "[", 
       RowBox[{
        StyleBox["list", "TI"], ",", 
        StyleBox["m", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds the Fourier discrete cosine transform of type ",
     Cell[BoxData[
      FormBox["m", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "Possible types ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " of discrete cosine transform for a list ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["u", "r"], TraditionalForm]], "InlineMath"],
 " of length ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " giving a result ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["v", "s"], TraditionalForm]], "InlineMath"],
 " are:"
}], "Notes",
 CellID->321171497],

Cell[BoxData[GridBox[{
   {"1.", 
    RowBox[{"DCT", "-", "I"}], Cell[BoxData[
     FormBox[
      RowBox[{
       SqrtBox[
        FractionBox["2", 
         RowBox[{"n", "-", "1"}]]], 
       RowBox[{"(", 
        RowBox[{
         FractionBox[
          SubscriptBox["u", "1"], "2"], "+", 
         RowBox[{
          UnderoverscriptBox["\[Sum]", 
           RowBox[{"r", "=", "2"}], 
           RowBox[{"n", "-", "1"}]], 
          RowBox[{
           SubscriptBox["u", "r"], " ", 
           RowBox[{"cos", "(", 
            RowBox[{
             FractionBox["\[Pi]", 
              RowBox[{"n", "-", "1"}]], " ", 
             RowBox[{"(", 
              RowBox[{"r", "-", "1"}], ")"}], " ", 
             RowBox[{"(", 
              RowBox[{"s", "-", "1"}], ")"}]}], ")"}]}]}], "+", " ", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"-", "1"}], ")"}], 
           RowBox[{"s", "-", "1"}]], 
          FractionBox[
           SubscriptBox["u", "n"], "2"]}]}], " ", ")"}], " "}], 
      TraditionalForm]], "InlineMath"]},
   {"2.", 
    RowBox[{"DCT", "-", "II"}], Cell[BoxData[
     FormBox[
      RowBox[{
       FractionBox["1", 
        SqrtBox["n"]], 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"r", "=", "1"}], "n"], 
        RowBox[{
         SubscriptBox["u", "r"], " ", "cos", 
         RowBox[{"(", 
          RowBox[{
           FractionBox["\[Pi]", "n"], " ", 
           RowBox[{"(", 
            RowBox[{"r", "-", 
             FractionBox["1", "2"]}], ")"}], 
           RowBox[{"(", 
            RowBox[{"s", "-", "1"}], ")"}]}], " ", ")"}], " "}]}]}], 
      TraditionalForm]], "InlineMath"]},
   {"3.", 
    RowBox[{"DCT", "-", "III"}], Cell[BoxData[
     FormBox[
      RowBox[{
       FractionBox["1", 
        SqrtBox["n"]], 
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["u", "1"], "+", 
         RowBox[{"2", " ", 
          RowBox[{
           UnderoverscriptBox["\[Sum]", 
            RowBox[{"r", "=", "2"}], "n"], 
           RowBox[{
            SubscriptBox["u", "r"], " ", 
            RowBox[{"cos", "(", 
             RowBox[{
              FractionBox["\[Pi]", "n"], " ", 
              RowBox[{"(", 
               RowBox[{"r", "-", "1"}], ")"}], 
              RowBox[{"(", 
               RowBox[{"s", "-", 
                FractionBox["1", "2"]}], ")"}]}], " ", ")"}]}]}]}]}], ")"}], 
       " "}], TraditionalForm]], "InlineMath"]},
   {"4.", 
    RowBox[{"DCT", "-", "IV"}], Cell[BoxData[
     FormBox[
      RowBox[{
       SqrtBox[
        FractionBox["2", "n"]], 
       RowBox[{"(", 
        RowBox[{
         UnderoverscriptBox["\[Sum]", 
          RowBox[{"r", "=", "1"}], "n"], 
         RowBox[{
          SubscriptBox["u", "r"], " ", 
          RowBox[{"cos", "(", 
           RowBox[{
            FractionBox["\[Pi]", "n"], " ", 
            RowBox[{"(", 
             RowBox[{"r", "-", 
              FractionBox["1", "2"]}], ")"}], " ", 
            RowBox[{"(", 
             RowBox[{"s", "-", 
              FractionBox["1", "2"]}], ")"}]}], ")"}]}]}], ")"}], " "}], 
      TraditionalForm]], "InlineMath"]}
  }]], "Notes",
 GridBoxOptions->{GridBoxAlignment->{"Columns" -> {{"Left"}}},
 GridBoxItemSize->{"Columns" -> {
     Scaled[0.02], 
     Scaled[0.12], 
     Scaled[0.64]}}},
 CellID->57195839],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["FourierDCT",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDCT"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FourierDCT",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDCT"], "[", 
   RowBox[{
    StyleBox["list", "TI"], ",", "2"}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->411046922],

Cell["\<\
The inverse discrete cosine transforms for types 1, 2, 3, 4 are types 1, 3, \
2, 4, respectively.\
\>", "Notes",
 CellID->872153026],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 " given in ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["FourierDCT",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDCT"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " can be nested to represent an array of data in any number of dimensions. "
}], "Notes",
 CellID->29223],

Cell["The array of data must be rectangular. ", "Notes",
 CellID->15547],

Cell[TextData[{
 "If the elements of ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 " are exact numbers, ",
 Cell[BoxData[
  ButtonBox["FourierDCT",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FourierDCT"]], "InlineFormula"],
 " begins by applying ",
 Cell[BoxData[
  ButtonBox["N",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/N"]], "InlineFormula"],
 " to them. "
}], "Notes",
 CellID->21017],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FourierDCT",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FourierDCT"]], "InlineFormula"],
 " can be used on ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->20893]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->166596712],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->476867505],

Cell["Find a discrete cosine transform:", "ExampleText",
 CellID->786559098],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", 
  RowBox[{"{", 
   RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "1"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->147480397],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.8944271909999159`", ",", 
   RowBox[{"-", "0.4253254041760199`"}], ",", 
   RowBox[{"-", "0.08541019662496845`"}], ",", 
   RowBox[{"-", "0.2628655560595667`"}], ",", "0.5854101966249684`"}], 
  "}"}]], "Output",
 ImageSize->{366, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->216225253]
}, Open  ]],

Cell["Find the inverse discrete cosine transform:", "ExampleText",
 CellID->169504947],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", 
  RowBox[{"%", ",", "3"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->876521411],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.1102230246251565`*^-16", ",", 
   RowBox[{"-", "1.1102230246251565`*^-16"}], ",", "0.9999999999999999`", ",",
    
   RowBox[{"-", "1.1102230246251565`*^-16"}], ",", "0.9999999999999998`"}], 
  "}"}]], "Output",
 ImageSize->{376, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->434127498]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25285511],

Cell["Find a discrete cosine transform of type 1 (DCT-I):", "ExampleText",
 CellID->360961894],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "2"}], "}"}], ",", "1"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->170348084],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.7677669529663687`", ",", 
   RowBox[{"-", "0.8535533905932737`"}], ",", "1.0606601717798212`", ",", 
   "0.14644660940672627`", ",", "0.35355339059327373`"}], "}"}]], "Output",
 ImageSize->{337, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->488649771]
}, Open  ]],

Cell["Find the inverse discrete cosine transform:", "ExampleText",
 CellID->252433387],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", 
  RowBox[{"%", ",", "1"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->366710795],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.9999999999999998`", ",", 
   RowBox[{"-", "1.1644331332494318`*^-16"}], ",", 
   RowBox[{"-", "1.3877787807814457`*^-17"}], ",", "0.9999999999999999`", ",",
    "1.9999999999999998`"}], "}"}]], "Output",
 ImageSize->{300, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->139979391]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["\<\
Use machine arithmetic to compute the discrete cosine transform:\
\>", "ExampleText",
 CellID->75805794],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "1", ",", "2", ",", "3", ",", "4", ",", "3", ",", "2", ",", "1",
      ",", "0"}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->283753173],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", "v", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->158871569],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"5.333333333333333`", ",", 
   RowBox[{"-", "1.0779341286758226`*^-17"}], ",", 
   RowBox[{"-", "2.7636197897938626`"}], ",", 
   RowBox[{"-", "3.2051726724102725`*^-17"}], ",", "0.09437286095264925`", 
   ",", 
   RowBox[{"-", "1.4990789100506607`*^-16"}], ",", 
   RowBox[{"-", "0.3333333333333333`"}], ",", 
   RowBox[{"-", "3.312237636943216`*^-16"}], ",", "0.14200734925348715`"}], 
  "}"}]], "Output",
 ImageSize->{441, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->136790635]
}, Open  ]],

Cell["Use 24-digit precision arithmetic:", "ExampleText",
 CellID->8571778],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", 
  RowBox[{"N", "[", 
   RowBox[{"v", ",", " ", "24"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->343340528],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "5.33333333333333333333333333333333333332`23.698970004336015", ",", 
   "0``22.97063211792824", ",", 
   RowBox[{
   "-", "2.76361978979386320220707413739488395336`23.202608476446255"}], ",", 
   "0``22.889215187534685", ",", 
   "0.09437286095264951896462088339994358711`21.855720050811374", ",", 
   "0``22.87924626186094", ",", 
   RowBox[{
   "-", "0.33333333333333333333333333333277631253`22.442388234873203"}], ",", 
   "0``22.686926201294767", ",", 
   "0.14200734925348727882830497944843777253`22.149752410649025"}], 
  "}"}]], "Output",
 ImageSize->{505, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->233119860]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->450525711],

Cell["A 2-dimensional discrete cosine transform:", "ExampleText",
 CellID->312951839],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->62446164],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->80647354],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.663592499554336`", ",", "0.285490095667454`", ",", 
     RowBox[{"-", "0.18215854158460085`"}], ",", "0.05197600944631247`"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.48088253516152596`", ",", "0.17495467764446823`", ",", 
     RowBox[{"-", "0.03086279893139374`"}], ",", "0.12274786033971653`"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.029090664927172098`"}], ",", "0.08104950367167936`", ",",
      
     RowBox[{"-", "0.28603332357886696`"}], ",", "0.08150951474064427`"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{581, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->543014232]
}, Open  ]],

Cell["A 5-dimensional discrete cosine transform:", "ExampleText",
 CellID->26630158],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", 
      RowBox[{"5", ",", "5", ",", "5", ",", "5", ",", "5"}], "}"}]}], "]"}]}],
   ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->3199137],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{"FourierDCT", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->49014697],

Cell[BoxData["28.218282567805694`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->53809066]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell["The list may have complex values:", "ExampleText",
 CellID->294060611],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", 
    RowBox[{"2", "I"}], ",", "3", ",", 
    RowBox[{"4", "I"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->174976628],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"2.`", "\[InvisibleSpace]", "+", 
    RowBox[{"3.`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.1120853822919913`"}], "-", 
    RowBox[{"1.4650756326574836`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.7071067811865476`"}], "+", 
    RowBox[{"0.7071067811865476`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"1.577161014949475`", "\[InvisibleSpace]", "-", 
    RowBox[{"1.6892463972414664`", " ", "\[ImaginaryI]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{505, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->100946]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->190477669],

Cell["\<\
You can use \"I\", \"II\", \"III\", or \"IV\" for the types 1, 2, 3, and 4 \
respectively.\
\>", "ExampleText",
 CellID->258193052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierDCT", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   "\"\<IV\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->184074371],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.44721359549995804`", ",", 
   RowBox[{"-", "0.44721359549995804`"}], ",", 
   RowBox[{"-", "0.44721359549995804`"}], ",", "0.44721359549995815`", ",", 
   "0.4472135954999578`"}], "}"}]], "Output",
 ImageSize->{359, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->570153698]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Compressing Image Data",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->115292603],

Cell["Import some image data:", "ExampleText",
 CellID->84161279],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"data", "=", 
   RowBox[{"256", "-", 
    RowBox[{"Import", "[", 
     RowBox[{"\"\<ExampleData/Ocelot.jpg\>\"", ",", "\"\<Data\>\""}], 
     "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->275556337],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJxMuwd8XNd9Jqpkk01e8naTfXlJ7NiO7Xg3StzfOlacYrnJKra6RFGkSLCC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   "], {{0, 0}, {200, 200}}, {0, 256}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->150033468]
}, Open  ]],

Cell["The 2-dimensional DCT:", "ExampleText",
 CellID->270166069],

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{"FourierDCT", "[", "data", "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->314419586],

Cell["The diagonal spectra shows exponential decay:", "ExampleText",
 CellID->297063968],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogPlot", "[", 
  RowBox[{
   RowBox[{"Abs", "@", 
    RowBox[{"Diagonal", "[", "t", "]"}]}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->629957784],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->589151797]
}, Open  ]],

Cell[TextData[{
 "Truncate modes in each axis, effectively compressing by factor of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1237309],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"truncate", "[", 
    RowBox[{"data_", ",", "f_"}], "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"i", ",", "j"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i", ",", "j"}], "}"}], "=", 
       RowBox[{"Floor", "[", 
        RowBox[{
         RowBox[{"Dimensions", "[", "data", "]"}], "/", 
         RowBox[{"Sqrt", "[", "f", "]"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"PadRight", "[", 
       RowBox[{
        RowBox[{"Take", "[", 
         RowBox[{"data", ",", "i", ",", "j"}], "]"}], ",", 
        RowBox[{"Dimensions", "[", "data", "]"}], ",", "0."}], "]"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->735572926],

Cell["Invert the DCT:", "ExampleText",
 CellID->42795126],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{"FourierDCT", "[", 
     RowBox[{
      RowBox[{"truncate", "[", 
       RowBox[{"t", ",", "4"}], "]"}], ",", "3"}], "]"}], "]"}], ",", 
   RowBox[{"ArrayPlot", "[", 
    RowBox[{"FourierDCT", "[", 
     RowBox[{
      RowBox[{"truncate", "[", 
       RowBox[{"t", ",", "9"}], "]"}], ",", "3"}], "]"}], "]"}], ",", 
   RowBox[{"ArrayPlot", "[", 
    RowBox[{"FourierDCT", "[", 
     RowBox[{
      RowBox[{"truncate", "[", 
       RowBox[{"t", ",", "16"}], "]"}], ",", "3"}], "]"}], "]"}]}], 
  "}"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->73880455],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{337, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[5]=",
 CellID->109531366]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Cosine Series Expansion",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->99964165],

Cell["Get an expansion for an even function as a sum of cosines:", \
"ExampleText",
 CellID->948737116],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   RowBox[{"Exp", "[", 
    RowBox[{
     RowBox[{"-", "100"}], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", 
        RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->292652122],

Cell["\<\
The function values on a uniformly spaced grid with n points on [-L,L):\
\>", "ExampleText",
 CellID->47689303],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "25"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xg", "=", 
   RowBox[{
    RowBox[{"N", "[", 
     RowBox[{"Range", "[", 
      RowBox[{"0", ",", 
       RowBox[{"n", "-", "1"}]}], "]"}], "]"}], "/", "n"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"fg", "=", 
   RowBox[{"Map", "[", 
    RowBox[{"f", ",", "xg"}], "]"}]}], ";"}], "\n", 
 RowBox[{"fp", "=", 
  RowBox[{"ListPlot", "[", 
   RowBox[{
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"xg", ",", "fg"}], "}"}], "]"}], ",", 
    RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->160587250],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztVz9vEzEUd86XVCpCNBKFClDo3ExlSgfSAaVdmqlCzFGElAy0qM3HgA+A
xEJV+gVawQdggakrM1/ksM/ns+/nZ8dJk6pDI+Xufu+/33t+5+sPJqP3HwaT
8XCwuX8y+DgaD083945PBInXGGNX4j/ZYOI5E0/5/yrLxN29tsU1EQJcIi4l
JZXlj09tXl3x1I8hl5dclkzhPc/1t+Q1NRzG6hDBE0eiAdYeOxKpQ1mPkHnm
iQj90bbSSC83iWQlbuXB7MVEVC/q5qvR9Di0Bdo/j4zMlwdX0rXmj8DuMH89
qUj0LmkU943ijvlYt+g6BltP5VXtvDXQ/RagPsqp38+vBePzp19FtQXeff1F
Yg4Y+bPKHy7YHuIkX9VDcb0QK6pZa/zz+5+g/Pzxt9AR+PjokpWzRGHJz20U
U8lQua5gSeKU3dRo6Pn27u2FiK735qsqvFNb7B6cDsifS75R4jKccoYW8ZZ5
WXCeErL/WoBXwXoT+C8Br4H8A+C/IL1uQ7/sAN4OaJWJK7QMNlpyL76CyHYA
0z5Cu3RWqozi7Mb2OpCbNuBOQMvkpg24U4kSc9MGTPsIrUI1oPpRZ4+U1gqe
GGJ0cBdK3xJSsjgBkgj78+i4bx3PSqyc6TcKTpFFd9ft9PD8Xnazare3AHcD
WqbbW4C7Tr7pMzBmX9k1+6IFuEtUn+rImI5JIzseT4YyZl/H67MT3VUzyVq5
wz1Br/Du9thtevHNRfkYW5noebI0LV9/aV5sf9ky02eiPnXTksuufC+rTp4m
4F5Ay0yeJuBeJcq9rDpPmoBpH/f969eyp1ioa5cYz0xv9bs8uxbnpZ9V9w5+
Xx6QNnKqtXc44JAWfBlZmNYKnm1r/wHpyA62\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->90237700]
}, Open  ]],

Cell["Compute the DCT-III and renormalize:", "ExampleText",
 CellID->674066408],

Cell[BoxData[
 RowBox[{
  RowBox[{"cc", "=", 
   RowBox[{
    RowBox[{"FourierDCT", "[", 
     RowBox[{"fg", ",", "3"}], "]"}], "/", 
    RowBox[{"Sqrt", "[", "n", "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->60039364],

Cell["\<\
The function has, in effect, been periodized with a particular symmetry:\
\>", "ExampleText",
 CellID->275165810],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"fp", ",", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Sum", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"cc", "[", 
         RowBox[{"[", "r", "]"}], "]"}], "*", 
        RowBox[{"Cos", "[", 
         RowBox[{"Pi", 
          RowBox[{"(", 
           RowBox[{"r", "-", 
            RowBox[{"1", "/", "2"}]}], ")"}], "x"}], "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"r", ",", 
         RowBox[{"Length", "[", "cc", "]"}]}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", " ", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->118986235],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->53040623]
}, Open  ]],

Cell["Plot the expansion error where the points are defined:", "ExampleText",
 CellID->286287023],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], "-", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"cc", "[", 
        RowBox[{"[", "r", "]"}], "]"}], "*", 
       RowBox[{"Cos", "[", 
        RowBox[{"Pi", 
         RowBox[{"(", 
          RowBox[{"r", "-", 
           RowBox[{"1", "/", "2"}]}], ")"}], "x"}], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"r", ",", 
        RowBox[{"Length", "[", "cc", "]"}]}], "}"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"1", "-", 
      RowBox[{"1", "/", "n"}]}]}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->32402675],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->550028307]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Chebyshev Basis Expansion",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->277068117],

Cell["\<\
Get an expansion for a function in the Chebyshev polynomials:\
\>", "ExampleText",
 CellID->255684039],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"5", "x"}], ")"}], "^", "2"}]}], ")"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->306651978],

Cell["The values of the function at the Chebyshev nodes:", "ExampleText",
 CellID->27178759],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "47"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"cnodes", "=", 
   RowBox[{"N", "[", 
    RowBox[{"Cos", "[", 
     RowBox[{"Pi", " ", 
      RowBox[{
       RowBox[{"Range", "[", 
        RowBox[{"n", ",", "0", ",", 
         RowBox[{"-", "1"}]}], "]"}], "/", "n"}]}], "]"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"fc", "=", 
   RowBox[{"Map", "[", 
    RowBox[{"f", ",", "cnodes"}], "]"}]}], ";"}], "\n", 
 RowBox[{"pf", "=", 
  RowBox[{"ListPlot", "[", 
   RowBox[{
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"cnodes", ",", "fc"}], "}"}], "]"}], ",", 
    RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->54712444],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWM1OFEEQbnZ2GVhFggEEDMuaiGd/glGSDQgrCLrBqIiJYlyJEQ6KQXwL
velBIUQUTdSDBwk+gInRg/Hq2RcZq7pmuntremZnAUUlTE93/XxVXf11T0Op
vDh76055cW6mnB9dKN+bnZu5nx+ZXwCRUyeEWIdnvkNA34OefK56nicc7Mkf
cRkbYZc2SumDxY2BwlP4FSlsnLnb7+H96OEnMc68xuT41dp30IOVSEt7sPT1
joE6NLhkojYOFp7RGMwVgh+nyOIMYZMlK2Y+NroCLYBR7NPDy+HYbS9Wv50Z
WSmNP5+8uDZ97c2Tx5/BDuyhnZp8Tf40VDHAT/VFAUYpSKcg0Vge0CEQhlY8
tTR/9wPEunnj3URptTi8fOH8y48bP0RGZtaM7d6vX36CEUipA9Y0VIhgpfoq
Xj+rz0lsWqLSgU5UGLFb5rIP2z3KCEaqbyaiAhxn8fuwaTJrFxUZY7bKmDls
d5kxraHEURbriPSzxjLzFl0yigstOqcYCq5mC3ZcawbSD20cw08gDjwHGFY+
xtaVMewZtsTkRuxtYBopTeuxoFpWzEFXLhWydhlejo177B4ZkjBkR8+kvWqe
KXo5fmn86jhV/ZCZ6cC/6lwQMx87B73C+xPmXH0NOlg1VK5+tTZTd3lGuFRx
kmv87cu828gUa0JxbRy3Z4iW5pnbvb2Zufg1ozkLwdlXuf9CubommsXXx+1J
mHFwHtT7755g9mnPIJXjGRy0nBB26xzDpk1yjLGq19ceZPLD0tr+PT8hpXr2
PK8c0/cZXnqn2L203vTSedm9tL5P1fWtmnOW+WS2OK7fZrx/LX7GWtVDbNxb
o55Q5V3I4A4/lTqZvt/wij6FO5ne9Ko8AXEvdDE5Wddy4926dIjNs93PrY3J
BwxrPT/+5WxletMr+nvbyvQDm5obZj0VY3POM86njNJKnQhxj3zOYltxnvI9
we8AUX54dKb9d1J78zaG+eEwyj74ntnnMSrnbN46hQjf3JwIjGbGhaLGNNY6
5Xvz2wKvUDPzM9Eqb5dhL60vJl733yn9M7FrYja7icfwWt+LoYPrGr49cLZm
GRPGQmiwrtDBdU2KppkQRoP1hg6utx0NwbKMGQaK1T7pTud29p0b1D64Ydmt
dppB/yF7w8yK5VqUJ30N1B4gYcibxpr3E3FoaWK0EOa5TQia67EIGcX72Hw0
56PQktYlWIPgDm+v+04z6e9g8RVsGzQXgkOUhpdUPaP+NyJlDZoJwbFJQ/Iv
MT/JiiZzHT3zT2o2jMFIwIVkfvouUt1+uiY7zsXrUroeakXdLwjVpdM=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 111},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->770147473]
}, Open  ]],

Cell["Find the Chebyshev coefficients:", "ExampleText",
 CellID->514950679],

Cell[BoxData[{
 RowBox[{
  RowBox[{"cc", "=", 
   RowBox[{
    RowBox[{"FourierDCT", "[", 
     RowBox[{"fc", ",", "1"}], "]"}], "*", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"2", "/", "n"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"cc", "[", 
    RowBox[{"[", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}], "]"}], "]"}], "/=", "2"}], 
  ";"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->39251404],

Cell["Show the error:", "ExampleText",
 CellID->12404932],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], "-", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"ChebyshevT", "[", 
        RowBox[{
         RowBox[{"i", "-", "1"}], ",", "x"}], "]"}], "*", 
       RowBox[{"cc", "[", 
        RowBox[{"[", "i", "]"}], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{"Length", "[", "cc", "]"}]}], "}"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->391006214],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFW0vIJUcV7vz9uH/UlQuThXFmoaCgi2wS8YERdYKOOgmYCRhcjIMwg8RI
HDe60Y1udKMbA2IWUcHoImKICBHEgBgQA2JAVJAIEhcGweBiQGi/Ot+pr05V
9/+4mXv1MnP/7uquU995n1Pd9yOXrl351IOXrl29fOns3Q9f+uyVq5c/d/bc
Qw9jqL+p67on8f/Tt3Y4nnFk/5+cZ/zl9wPpu0+j9ukuavQAg/fb2Z1+1jf3
5rH2b7qlm2yqf3D806f+MLTz7fzFF//13G//lg7vtSl3pO+h3NV1Nm+KI/w+
CPS6fD7qPh8d813vefcjWGjJhZ1/+UtP33/f9wuK2/E9aHpYeodovvfd5yQv
zr78yR9h/OWXr+9bFp944Aff/MavjpDFB973bQB5/vd/3xOKbCtd//lrT339
a8988Qs/I5LDeCekcP6Dj+KOZ3/91/3hyKMPfeYnj37nN/gWOpszwDphF1/9
yi9gv0SRvWGal5Yf/w7iEgxA3pLvFHFAylevPHGAtUn9PkMOxl944Z/3XniM
GKsZoAahtDNgNviL7wY/YBM8vnn/xwJfnLKUPe/nSmHWqFn4xjHXqGbCph/5
1rPwM4gyzvzTH/8Bm6pnZpSJl67XHd09YR4p4RDXcMfKijCgHz/xPJHFFZ/5
5V9gWFRt8vvlTAqGdhZnknsc0vgqrN3AQUqutoht4mOm2Q3wQ2FZYiT4uNqu
vcBGNxxNirADCpXXbo4zqAYcwDhfeunfrqt9Rcs4KmxdspeDOeQZ2pQO7tmf
nHw0xc4aSy0lAYJvpzi6R0SuOdgPD374+O8SumMwuT/sHxNyapdzf8JX+dHI
iMXbfv70nx3PSdG168Eg+aj8qxvoJlxqiS1FRnJATzMm7bzQO2cIPmSjGMKU
lJk2Rkt+yLCywv3aCsy38BOndXdYAYRAFKGnXYEhr0SIxCLDm9NHgEWYLfQL
pUB/YCTGKAKccTrKAsIKgYewRrLqKVChswf5DOSR0eCVUycVOkmkzhhLwyB1
ImjoZOkwibp0ELcQvSSdAXKHgLhSXIN4OT7VFhmlXTiIfPXKEb4KEjwyNjUT
VyFNMJx5vqFViJmj5yp/OU3usYPsi+6Hqj2H69f/g8KP0xd3G3XWAwUNOaFD
EM2W0eRQp5AdJIgDUMtXp9Z2MlJJUHdKThKgIRxhMjAcHLK4cA9/hdmqoHUd
2cEO0aoCcgo3KFPXnQl3iVI8qOaHAWQjUlbrqRiDYBhVj9FcToURtx6FkeEF
B6mWqHNMtOJs+YVGQaia1NeUleJDrhjncYA4kQ7fdQrMzPCp4lzJfFavHpkJ
cZD7iG1wR6vgLOJGMEMQT0iIO2WL/ghZnZmX0cAQurGb1yeB934hRAfSSz1H
GbcI5OfqUFn/dYoREBVyqN9frJ2dCxXMIcb5KdsbLiUpT8dgPnMMr2ThsLnS
L7R622Jk01KzO+h9/DaMEzNH1MiQZeJRmTOVuYW0VAAHK1IpNqBegjZQYRhP
xH3bibwf8E/Q9lquWM67ZWaEPDil7JhzGRIoO8lEet+N7LqJrQmHeC+jJ0oF
FAzAAG9JEWfYmfx2ZmOjsHsXbMgnxYg5F+eDWwk1JTssPuc1M6+XGENkkqBT
oXxUNXpP8v+VD++BpTPQenzqRoKb1XsvJUTWKwkVjmVFxa4O1mQ0yfPm3DJx
fVKXaXnWI3Jmg2xbSRdn9yghVWWAzk6qqYQ97lJC8hJBv1EJRQ+m11JCrHTn
vNVTSYg1XNrXbPd6z5xSUpu57v/OzCE7DMztjI8anbiDE3XJ1TWuzMvxNsKE
8+xf3iP3jaT6hTWy+qVk2GtpUBnXOkDWobKcN8zLPrfr3mqLQOeAlKRIjQuf
Rw3WC6oW8WHsI3elrjiW66F0kdkipNmFhZBvWd6cy0ciEXf4sF4c57yrN2vn
bsw6gWiSZ7/Fln6TjbKkyPLpBmgRcOmjjdV55OPK5JtT2QSSVx4XTkpP2K/I
wpD68wrSoqDXtR89jbJe6oOZkMgVvSnw0pWCSUyG8MasdZe+RRWPvhiiTXXS
IkSRJEA61AkNbjjOG6pKjBojRVX0i55zkX0aWVFcUVZFrktZERfXLJ2kqk5a
lu/Jt57EGCM7i5EnRUjoGkIBI2OWmfcytxrQ1+VRzEvRW7EE+gLrWImrcjFG
3NSPkAdhxQUsn++etL9ce0OvDLKoX1r58lx0FtHmZNtsI1mJM2013voBz5Vt
vTtUt1fra6QdyeC7OIo1MVTkCuZBA5Qx1GcLw63p8LXGyKtsNuiDdXxDwLbH
kEgcpgcOFx6jI4AyNARNp+cdeQ5oY9x3Oak39SRuF9RPrIJowxxX3GtsspFF
yIc8lwz8vPjNmt7yk5c5xNdWG632SuQmRXnBYt+P56om/VzRcNmrGeex5iG3
lJ+8L+76j5R+kTW5gmYwihvxj3vMMFAwRkawHBAANCSRdh2sorTvx5XjqH1V
VJlDlTK2cYHomQxfWk+bN9ZWJyPhaLFl7hqRl9JRhS58E2MiLniHECtB4on2
QCCS3iZWUI0npHujrZxC97gABsuMYk1H1WfcrSl4Yu7DBZq486Stnzk/L+R4
qh5yHAk+MVKE0njBhAuwOqjfRl4D+cBwqAyKCxDIOVWZuwhqOdR77C0YliuL
hw7TPMYOXAdWQMc/xhhFx5j3i8e7lnOtS6ol+/erflNsZN3vGs8/hTaVWxee
2K+u2FZa5dnn6vVJW4pyUsqBNtJGYEwF/1jT+yTD4/j6Rg9tz0cXZF8DvWCl
RJd5FRDhpWmnfWzlmqvMKivGqrSNnk1VtZBoex6qlKai1j5VlVXbKqaNtU1s
neLuN6MAqzfFRqrEaw6iAU3IZ8wa9Crp9SbBN2fLooN4dysuIrVJ9jfnpk5a
q+J506GdJLNNjClzE7FOkFeVJWM1K8v0/WNWobybfjdlem4bbzQAt2cLwmia
x85Gu84evdYr0qKrZf9FbMPW/BZ7JE2dayL5led6FiS/9Ffq/22BP8wEJ0nf
5E9WHvXddGjLHa7IX+gwYldjPtDGuVJJVp3fEc8b1J/5Tgf1xk3ZdEi+3qE7
SzdNybTPsbgPoSqqxr4pPJtu3HrW9V3iCuWh6/gw6xOB+PW++A4hzh7KO5rn
k/4siHjjLger3+3xNl3PEDtB7b8NlB/p3hmQst9gtG6shql7BWnsN4I1mVW0
/URrbW2PFrv38DQ05Syskeppon2njQKS2wexqtuhYhxrrL0D1o2ibBffC1j3
ipIhmOsUK7xgm/Is7jw7yvfbKKOo0cryrzBRsHHnLAIinoJ1HY/syX37LqPy
9vno57HMNKnCto8AlXci+I6LsBiBvkbZ5Qd7y30/IhprG2LwdHQmo+Uzr1fH
EX67CeVHnO1zO68mMhZ2AqISRRrjZoWWmZW+QSuiJb53S6T+eNIq8BWkqtwI
hJlRVGhUua7WXrlsjcxwl86iYPH9bSUKOvFBaPU8NuJUHVueKHp8s/ederYN
c/bfMWaLjDU/kSHP20oUwYuP/daRyjQMaVc/oeVdRZJCx/Dn6Bgh6aXbogNb
akCE7madqwLwZjZTmnwbMj/dxcqNp3geI0I+R6JrE2H27PbNpLh7KU/ecCmu
Vu8vU5q+wRwCQYPG32lhfQ68vl9GLB81GrFuDHulhfZcPyFTrPZsxqcSpRe4
K9CO9TDpkba3YQva4tFz1nmjdcGuyeurp15edtiH5V9DybNDpCS/XKdUYy1P
vDwGRUqyS4+PpCSbqSipUuJrPJHOoHoHTOYoQasoPRzzhaowXyRSie+UKta4
N8Soro7AWY40OESbOI6GIpibyfnKurd5h7UbvGm1D/OUlwH2obaaVf3p4/nj
vP60bx+WN2WZOOt3WsrVuB3iEu5G7XW6Le0YEe3I96+PRhSemY+qsD0r7RgR
X62SgpocQ9Oy6Yw5AAGxpPdBPrwLHNUbbp1nj5UngKOU6PGYaHCEC+l8J2iK
VJiz5PjN1QYMsZTfChDLUZlh+f5d2AfKHjNqbwefwi+shj+O8DXs7dgUU6gW
r74113FyLuIeX6+p5kK7vmNy3FwYif9OJs6FZfp+W7RUYk7R/jDPddnTcj9u
Oko1Sa4TL55iXIQh4zhOsIzpcTxlyZxDL1b62C6WzTmQ5ZOBmifIi7v1AR8d
a2ubLSh3OR/+r9ZkJGI62OOa1ag9vcHAftbL2qUemRwY5ve33uLthrTH1Fnd
77+jO22cGOZovwzbFh1mWX3PIJret4ijfIsyeSRH69/x6bd+vpt2sR517Gu/
AFy/p7vpv/sPIF4=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->409008655]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell["DCT-I and DCT-IV are their own inverses:", "ExampleText",
 CellID->393142439],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "7"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->279848536],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDCT", "[", 
    RowBox[{
     RowBox[{"FourierDCT", "[", 
      RowBox[{"data", ",", "1"}], "]"}], ",", "1"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->279425620],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->323290440]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDCT", "[", 
    RowBox[{
     RowBox[{"FourierDCT", "[", 
      RowBox[{"data", ",", "4"}], "]"}], ",", "4"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->27979728],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->374458290]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->320017841],

Cell["DCT-II and DCT-III are inverses of each other:", "ExampleText",
 CellID->19940023],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "10"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->355011233],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDCT", "[", 
    RowBox[{
     RowBox[{"FourierDCT", "[", 
      RowBox[{"data", ",", "2"}], "]"}], ",", "3"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->678711840],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0", ",", "0"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->516499672]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FourierDCT", "[", 
    RowBox[{
     RowBox[{"FourierDCT", "[", 
      RowBox[{"data", ",", "3"}], "]"}], ",", "2"}], "]"}], "-", "data"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->132848064],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0", ",", "0"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->478883801]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217596405],

Cell["The DCT is equivalent to matrix multiplication:", "ExampleText",
 CellID->73266107],

Cell[BoxData[
 RowBox[{
  RowBox[{"dctII", "[", "n_", "]"}], ":=", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["n"]], 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Cos", "[", 
      RowBox[{"Pi", 
       RowBox[{"(", 
        RowBox[{"r", "-", 
         RowBox[{"1", "/", "2"}]}], ")"}], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"s", "-", "1"}], ")"}], "/", "n"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"s", ",", "n"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"r", ",", "n"}], "}"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->45951218],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"dctII", "[", "7", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->253948789],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      FractionBox["1", 
       SqrtBox["7"]], 
      FractionBox["1", 
       SqrtBox["7"]], 
      FractionBox["1", 
       SqrtBox["7"]], 
      FractionBox["1", 
       SqrtBox["7"]], 
      FractionBox["1", 
       SqrtBox["7"]], 
      FractionBox["1", 
       SqrtBox["7"]], 
      FractionBox["1", 
       SqrtBox["7"]]},
     {
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], "0", 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}]},
     {
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox["1", 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]]},
     {
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], "0", 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}]},
     {
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox["1", 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]]},
     {
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox["\[Pi]", "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox[
         RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
       SqrtBox["7"]], "0", 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Cos", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox["\[Pi]", "7"], "]"}], 
        SqrtBox["7"]]}]},
     {
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox["1", 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Pi]", "7"], "]"}], 
       SqrtBox["7"]], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", 
         FractionBox[
          RowBox[{"3", " ", "\[Pi]"}], "14"], "]"}], 
        SqrtBox["7"]]}], 
      FractionBox[
       RowBox[{"Sin", "[", 
        FractionBox["\[Pi]", "14"], "]"}], 
       SqrtBox["7"]]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{416, 269},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->176959709]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", "7"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->71580635],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"dctII", "[", "7", "]"}], ".", "data"}], "-", 
   RowBox[{"FourierDCT", "[", "data", "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->105998950],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->86497424]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FourierDCT",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FourierDCT"]], "InlineFormula"],
 " always returns normalized results:"
}], "ExampleText",
 CellID->146202875],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"FourierDCT", "[", 
     RowBox[{
      RowBox[{"ConstantArray", "[", 
       RowBox[{"1", ",", "5"}], "]"}], ",", "type"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"type", ",", "1", ",", "4"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->949419420],

Cell[BoxData[
 TagBox[GridBox[{
    {"2.82842712474619`", "0.`", "0.`", "0.`", "0.`"},
    {"2.23606797749979`", "9.444121133484361`*^-17", 
     "8.033649276373055`*^-17", "5.836787854364518`*^-17", 
     "3.06858096987851`*^-17"},
    {"2.8235955159711317`", 
     RowBox[{"-", "0.8777061007329484`"}], "0.44721359549995776`", 
     RowBox[{"-", "0.22786670826713562`"}], "0.07083167502878462`"},
    {"2.021471201601977`", 
     RowBox[{"-", "0.6965515053690705`"}], "0.4472135954999583`", 
     RowBox[{"-", "0.3549107188691969`"}], "0.32016958489789743`"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.7]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{445, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TableForm=",
 CellID->136591668]
}, Open  ]],

Cell["\<\
To get unnormalized results, you can multiply by the normalization:\
\>", "ExampleText",
 CellID->104315647],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"nc", "[", 
    RowBox[{"n_", ",", "1"}], "]"}], "=", 
   RowBox[{"Sqrt", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"n", "-", "1"}], ")"}], "/", "2"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"nc", "[", 
    RowBox[{"n_", ",", 
     RowBox[{"2", "|", "3"}]}], "]"}], "=", 
   RowBox[{"Sqrt", "[", "n", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"nc", "[", 
    RowBox[{"n_", ",", "4"}], "]"}], "=", 
   RowBox[{"Sqrt", "[", 
    RowBox[{"n", "/", "2"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->77124536],

Cell[BoxData[
 RowBox[{
  RowBox[{"unnormalizedDCT", "[", 
   RowBox[{"data_", ",", "type_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"FourierDCT", "[", 
    RowBox[{"data", ",", "type"}], "]"}], "*", 
   RowBox[{"nc", "[", 
    RowBox[{
     RowBox[{"Length", "[", "data", "]"}], ",", "type"}], "]"}]}]}]], "Input",\

 CellLabel->"In[3]:=",
 CellID->388475004],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"unnormalizedDCT", "[", 
     RowBox[{
      RowBox[{"ConstantArray", "[", 
       RowBox[{"1", ",", "5"}], "]"}], ",", "type"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"type", ",", "1", ",", "4"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->567054902],

Cell[BoxData[
 TagBox[GridBox[{
    {"4.`", "0.`", "0.`", "0.`", "0.`"},
    {"5.000000000000001`", "2.1117696842213397`*^-16", 
     "1.7963785889362148`*^-16", "1.3051454412604205`*^-16", 
     "6.861555643110583`*^-17"},
    {"6.313751514675044`", 
     RowBox[{"-", "1.9626105055051508`"}], "0.9999999999999997`", 
     RowBox[{"-", "0.5095254494944286`"}], "0.1583844403245368`"},
    {"3.196226610749831`", 
     RowBox[{"-", "1.1013446322926335`"}], "0.7071067811865481`", 
     RowBox[{"-", "0.5611631188171807`"}], "0.5062325628940022`"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.7]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{445, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]//TableForm=",
 CellID->42631871]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["FourierDST",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierDST"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Fourier",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Fourier"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FourierCosTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierCosTransform"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Discrete Fourier Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/FourierTransforms"]], "Tutorials",
 CellID->1899]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Image Processing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ImageProcessing"]], "MoreAbout",
 CellID->33777527],

Cell[TextData[ButtonBox["Integral Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/IntegralTransforms"]], "MoreAbout",
 CellID->92927196],

Cell[TextData[ButtonBox["Signal Processing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SignalProcessing"]], "MoreAbout",
 CellID->45169662],

Cell[TextData[ButtonBox["New in 6.0: Numerical Data Handling",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60NumericalDataHandling"]], "MoreAbout",
 CellID->274144357]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FourierDCT - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 43, 58.2824352}", "context" -> "System`", 
    "keywords" -> {
     "discrete cosine transform", "fast cosine transform", 
      "fast Fourier transform", "real Fourier transform", 
      "sequence transform", "trigonometric transform"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "FourierDCT[list] finds the Fourier discrete cosine transform of a list \
of real numbers. FourierDCT[list, m] finds the Fourier discrete cosine \
transform of type m.", 
    "synonyms" -> {
     "DCT", "DCTI", "DCTII", "DCTIII", "DCTIV", "DCT-I", "DCT-II", "DCT-III", 
      "DCT-IV", "Fourier DCT"}, "title" -> "FourierDCT", "type" -> "Symbol", 
    "uri" -> "ref/FourierDCT"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[11830, 382, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->166596712]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 166772, 3802}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2469, 59, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3092, 86, 1575, 37, 70, "ObjectNameGrid"],
Cell[4670, 125, 979, 30, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5686, 160, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6164, 177, 463, 17, 70, "Notes",
 CellID->321171497],
Cell[6630, 196, 3346, 106, 70, "Notes",
 CellID->57195839],
Cell[9979, 304, 455, 17, 70, "Notes",
 CellID->411046922],
Cell[10437, 323, 142, 4, 70, "Notes",
 CellID->872153026],
Cell[10582, 329, 383, 13, 70, "Notes",
 CellID->29223],
Cell[10968, 344, 72, 1, 70, "Notes",
 CellID->15547],
Cell[11043, 347, 420, 16, 70, "Notes",
 CellID->21017],
Cell[11466, 365, 327, 12, 70, "Notes",
 CellID->20893]
}, Closed]],
Cell[CellGroupData[{
Cell[11830, 382, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->166596712],
Cell[CellGroupData[{
Cell[12215, 396, 148, 5, 70, "ExampleSection",
 CellID->476867505],
Cell[12366, 403, 76, 1, 70, "ExampleText",
 CellID->786559098],
Cell[CellGroupData[{
Cell[12467, 408, 186, 6, 28, "Input",
 CellID->147480397],
Cell[12656, 416, 385, 11, 36, "Output",
 CellID->216225253]
}, Open  ]],
Cell[13056, 430, 86, 1, 70, "ExampleText",
 CellID->169504947],
Cell[CellGroupData[{
Cell[13167, 435, 129, 4, 28, "Input",
 CellID->876521411],
Cell[13299, 441, 384, 11, 39, "Output",
 CellID->434127498]
}, Open  ]],
Cell[13698, 455, 124, 3, 70, "ExampleDelimiter",
 CellID->25285511],
Cell[13825, 460, 94, 1, 70, "ExampleText",
 CellID->360961894],
Cell[CellGroupData[{
Cell[13944, 465, 211, 7, 70, "Input",
 CellID->170348084],
Cell[14158, 474, 349, 9, 36, "Output",
 CellID->488649771]
}, Open  ]],
Cell[14522, 486, 86, 1, 70, "ExampleText",
 CellID->252433387],
Cell[CellGroupData[{
Cell[14633, 491, 129, 4, 70, "Input",
 CellID->366710795],
Cell[14765, 497, 376, 10, 39, "Output",
 CellID->139979391]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[15190, 513, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[15414, 522, 114, 3, 70, "ExampleText",
 CellID->75805794],
Cell[15531, 527, 250, 8, 70, "Input",
 CellID->283753173],
Cell[CellGroupData[{
Cell[15806, 539, 106, 3, 70, "Input",
 CellID->158871569],
Cell[15915, 544, 577, 15, 58, "Output",
 CellID->136790635]
}, Open  ]],
Cell[16507, 562, 75, 1, 70, "ExampleText",
 CellID->8571778],
Cell[CellGroupData[{
Cell[16607, 567, 164, 5, 70, "Input",
 CellID->343340528],
Cell[16774, 574, 718, 19, 75, "Output",
 CellID->233119860]
}, Open  ]],
Cell[17507, 596, 125, 3, 70, "ExampleDelimiter",
 CellID->450525711],
Cell[17635, 601, 85, 1, 70, "ExampleText",
 CellID->312951839],
Cell[17723, 604, 240, 8, 70, "Input",
 CellID->62446164],
Cell[CellGroupData[{
Cell[17988, 616, 105, 3, 70, "Input",
 CellID->80647354],
Cell[18096, 621, 751, 21, 54, "Output",
 CellID->543014232]
}, Open  ]],
Cell[18862, 645, 84, 1, 70, "ExampleText",
 CellID->26630158],
Cell[18949, 648, 272, 9, 70, "Input",
 CellID->3199137],
Cell[CellGroupData[{
Cell[19246, 661, 135, 4, 70, "Input",
 CellID->49014697],
Cell[19384, 667, 175, 5, 36, "Output",
 CellID->53809066]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19608, 678, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[19855, 687, 76, 1, 70, "ExampleText",
 CellID->294060611],
Cell[CellGroupData[{
Cell[19956, 692, 213, 7, 70, "Input",
 CellID->174976628],
Cell[20172, 701, 661, 18, 36, "Output",
 CellID->100946]
}, Open  ]],
Cell[20848, 722, 125, 3, 70, "ExampleDelimiter",
 CellID->190477669],
Cell[20976, 727, 141, 4, 70, "ExampleText",
 CellID->258193052],
Cell[CellGroupData[{
Cell[21142, 735, 221, 7, 70, "Input",
 CellID->184074371],
Cell[21366, 744, 370, 10, 36, "Output",
 CellID->570153698]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21785, 760, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[CellGroupData[{
Cell[22037, 771, 243, 7, 70, "ExampleSubsection",
 CellID->115292603],
Cell[22283, 780, 65, 1, 70, "ExampleText",
 CellID->84161279],
Cell[CellGroupData[{
Cell[22373, 785, 264, 8, 70, "Input",
 CellID->275556337],
Cell[22640, 795, 43419, 717, 201, "Output",
 CellID->150033468]
}, Open  ]],
Cell[66074, 1515, 65, 1, 70, "ExampleText",
 CellID->270166069],
Cell[66142, 1518, 151, 5, 70, "Input",
 CellID->314419586],
Cell[66296, 1525, 88, 1, 70, "ExampleText",
 CellID->297063968],
Cell[CellGroupData[{
Cell[66409, 1530, 281, 8, 70, "Input",
 CellID->629957784],
Cell[66693, 1540, 5373, 92, 70, "Output",
 Evaluatable->False,
 CellID->589151797]
}, Open  ]],
Cell[72081, 1635, 192, 6, 70, "ExampleText",
 CellID->1237309],
Cell[72276, 1643, 858, 25, 70, "Input",
 CellID->735572926],
Cell[73137, 1670, 57, 1, 70, "ExampleText",
 CellID->42795126],
Cell[CellGroupData[{
Cell[73219, 1675, 635, 20, 70, "Input",
 CellID->73880455],
Cell[73857, 1697, 50341, 829, 70, "Output",
 Evaluatable->False,
 CellID->109531366]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[124247, 2532, 243, 7, 70, "ExampleSubsection",
 CellID->99964165],
Cell[124493, 2541, 103, 2, 70, "ExampleText",
 CellID->948737116],
Cell[124599, 2545, 328, 13, 70, "Input",
 CellID->292652122],
Cell[124930, 2560, 121, 3, 70, "ExampleText",
 CellID->47689303],
Cell[CellGroupData[{
Cell[125076, 2567, 645, 22, 70, "Input",
 CellID->160587250],
Cell[125724, 2591, 1102, 22, 70, "Output",
 Evaluatable->False,
 CellID->90237700]
}, Open  ]],
Cell[126841, 2616, 79, 1, 70, "ExampleText",
 CellID->674066408],
Cell[126923, 2619, 233, 8, 70, "Input",
 CellID->60039364],
Cell[127159, 2629, 123, 3, 70, "ExampleText",
 CellID->275165810],
Cell[CellGroupData[{
Cell[127307, 2636, 738, 23, 70, "Input",
 CellID->118986235],
Cell[128048, 2661, 3445, 61, 70, "Output",
 Evaluatable->False,
 CellID->53040623]
}, Open  ]],
Cell[131508, 2725, 97, 1, 70, "ExampleText",
 CellID->286287023],
Cell[CellGroupData[{
Cell[131630, 2730, 739, 24, 70, "Input",
 CellID->32402675],
Cell[132372, 2756, 5636, 96, 70, "Output",
 Evaluatable->False,
 CellID->550028307]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[138057, 2858, 246, 7, 70, "ExampleSubsection",
 CellID->277068117],
Cell[138306, 2867, 112, 3, 70, "ExampleText",
 CellID->255684039],
Cell[138421, 2872, 295, 11, 70, "Input",
 CellID->306651978],
Cell[138719, 2885, 92, 1, 70, "ExampleText",
 CellID->27178759],
Cell[CellGroupData[{
Cell[138836, 2890, 731, 25, 70, "Input",
 CellID->54712444],
Cell[139570, 2917, 1550, 29, 70, "Output",
 Evaluatable->False,
 CellID->770147473]
}, Open  ]],
Cell[141135, 2949, 75, 1, 70, "ExampleText",
 CellID->514950679],
Cell[141213, 2952, 447, 17, 70, "Input",
 CellID->39251404],
Cell[141663, 2971, 57, 1, 70, "ExampleText",
 CellID->12404932],
Cell[CellGroupData[{
Cell[141745, 2976, 651, 21, 70, "Input",
 CellID->391006214],
Cell[142399, 2999, 4560, 79, 70, "Output",
 Evaluatable->False,
 CellID->409008655]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[147020, 3085, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[147261, 3094, 83, 1, 70, "ExampleText",
 CellID->393142439],
Cell[147347, 3097, 176, 6, 70, "Input",
 CellID->279848536],
Cell[CellGroupData[{
Cell[147548, 3107, 262, 9, 70, "Input",
 CellID->279425620],
Cell[147813, 3118, 257, 8, 36, "Output",
 CellID->323290440]
}, Open  ]],
Cell[CellGroupData[{
Cell[148107, 3131, 261, 9, 70, "Input",
 CellID->27979728],
Cell[148371, 3142, 257, 8, 36, "Output",
 CellID->374458290]
}, Open  ]],
Cell[148643, 3153, 125, 3, 70, "ExampleDelimiter",
 CellID->320017841],
Cell[148771, 3158, 88, 1, 70, "ExampleText",
 CellID->19940023],
Cell[148862, 3161, 177, 6, 70, "Input",
 CellID->355011233],
Cell[CellGroupData[{
Cell[149064, 3171, 262, 9, 70, "Input",
 CellID->678711840],
Cell[149329, 3182, 291, 9, 36, "Output",
 CellID->516499672]
}, Open  ]],
Cell[CellGroupData[{
Cell[149657, 3196, 262, 9, 70, "Input",
 CellID->132848064],
Cell[149922, 3207, 291, 9, 36, "Output",
 CellID->478883801]
}, Open  ]],
Cell[150228, 3219, 125, 3, 70, "ExampleDelimiter",
 CellID->217596405],
Cell[150356, 3224, 89, 1, 70, "ExampleText",
 CellID->73266107],
Cell[150448, 3227, 588, 21, 70, "Input",
 CellID->45951218],
Cell[CellGroupData[{
Cell[151061, 3252, 138, 4, 70, "Input",
 CellID->253948789],
Cell[151202, 3258, 6176, 222, 304, "Output",
 CellID->176959709]
}, Open  ]],
Cell[157393, 3483, 175, 6, 70, "Input",
 CellID->71580635],
Cell[CellGroupData[{
Cell[157593, 3493, 225, 7, 70, "Input",
 CellID->105998950],
Cell[157821, 3502, 256, 8, 36, "Output",
 CellID->86497424]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[158126, 3516, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[158360, 3525, 216, 7, 70, "ExampleText",
 CellID->146202875],
Cell[CellGroupData[{
Cell[158601, 3536, 365, 11, 70, "Input",
 CellID->949419420],
Cell[158969, 3549, 1159, 29, 96, "Output",
 CellID->136591668]
}, Open  ]],
Cell[160143, 3581, 118, 3, 70, "ExampleText",
 CellID->104315647],
Cell[160264, 3586, 606, 22, 70, "Input",
 CellID->77124536],
Cell[160873, 3610, 358, 12, 70, "Input",
 CellID->388475004],
Cell[CellGroupData[{
Cell[161256, 3626, 370, 11, 70, "Input",
 CellID->567054902],
Cell[161629, 3639, 1144, 29, 96, "Output",
 CellID->42631871]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[162834, 3675, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[163153, 3686, 724, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[163914, 3717, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[164235, 3728, 154, 3, 70, "Tutorials",
 CellID->1899]
}, Open  ]],
Cell[CellGroupData[{
Cell[164426, 3736, 299, 8, 70, "MoreAboutSection"],
Cell[164728, 3746, 142, 3, 70, "MoreAbout",
 CellID->33777527],
Cell[164873, 3751, 148, 3, 70, "MoreAbout",
 CellID->92927196],
Cell[165024, 3756, 144, 3, 70, "MoreAbout",
 CellID->45169662],
Cell[165171, 3761, 175, 3, 70, "MoreAbout",
 CellID->274144357]
}, Open  ]],
Cell[165361, 3767, 27, 0, 70, "History"],
Cell[165391, 3769, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

