(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    152563,       3314]
NotebookOptionsPosition[    145452,       3079]
NotebookOutlinePosition[    147169,       3122]
CellTagsIndexPosition[    147082,       3117]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"The Representation of Solution Sets" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/TheRepresentationOfSolutionSets"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"The Representation of Solution Sets\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"CylindricalDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/CylindricalDecomposition"], 
          "Reduce" :> Documentation`HelpLookup["paclet:ref/Reduce"], 
          "SemialgebraicComponentInstances" :> 
          Documentation`HelpLookup[
           "paclet:ref/SemialgebraicComponentInstances"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"CylindricalDecomposition\"\>", 
       2->"\<\"Reduce\"\>", 
       3->"\<\"SemialgebraicComponentInstances\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Polynomial Systems" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialSystems"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60AlgebraicComputing"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Polynomial Systems\"\>", 
       2->"\<\"New in 6.0: Symbolic Computation\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["GenericCylindricalDecomposition", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["GenericCylindricalDecomposition",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/GenericCylindricalDecomposition"], "[", 
       RowBox[{
        StyleBox["ineqs", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds the full-dimensional part of the decomposition of \
the region represented by the inequalities ",
     Cell[BoxData[
      StyleBox["ineqs", "TI"]], "InlineFormula"],
     " into cylindrical parts whose directions correspond to the successive ",
     
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     ", together with any hypersurfaces containing the rest of the region."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->30562]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GenericCylindricalDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GenericCylindricalDecomposition"]], 
  "InlineFormula"],
 " assumes that all variables are real. "
}], "Notes",
 CellID->23865],

Cell["Lists or logical combinations of inequalities can be given. ", "Notes",
 CellID->17554],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GenericCylindricalDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GenericCylindricalDecomposition"]], 
  "InlineFormula"],
 " returns inequalities whose bounds in general involve algebraic functions. \
"
}], "Notes",
 CellID->20700]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->34455918],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->555575941],

Cell["\<\
Find a cylindrical decomposition of the full-dimensional part of the unit \
disc:\
\>", "ExampleText",
 CellID->504489806],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GenericCylindricalDecomposition", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], "+", 
     RowBox[{"y", "^", "2"}]}], "<=", "1"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->407112382],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "<", "x", "<", "1"}], "&&", 
    RowBox[{
     RowBox[{"-", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]]}], "<", "y", "<", 
     SqrtBox[
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}]]}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     SuperscriptBox["x", "2"], "+", 
     SuperscriptBox["y", "2"]}], "\[Equal]", "0"}]}], "}"}]], "Output",
 ImageSize->{340, 27},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->119351041]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GenericCylindricalDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GenericCylindricalDecomposition"]], 
  "InlineFormula"],
 " gives a full\[Hyphen]dimensional set of solutions and a hypersurface:"
}], "ExampleText",
 CellID->29891744],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ineqs", " ", "=", " ", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], " ", "+", " ", 
       RowBox[{"y", "^", "2"}]}], " ", "<=", " ", "1"}], " ", "&&", " ", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", " ", "-", " ", 
          RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}], " ", "+", " ", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"y", " ", "-", " ", 
          RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}]}], " ", "<=", " ", 
      "1"}]}]}], ";"}], " "}]], "Input",
 CellLabel->"In[1]:=",
 CellID->181235941],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"full", ",", "hyp"}], "}"}], "=", 
  RowBox[{"GenericCylindricalDecomposition", "[", 
   RowBox[{"ineqs", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3731316],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], "<", "x", "<", 
       RowBox[{
        FractionBox["1", "4"], " ", 
        RowBox[{"(", 
         RowBox[{"1", "-", 
          SqrtBox["7"]}], ")"}]}]}], "&&", 
      RowBox[{
       RowBox[{
        FractionBox["1", "2"], "-", 
        RowBox[{
         FractionBox["1", "2"], " ", 
         SqrtBox[
          RowBox[{"3", "+", 
           RowBox[{"4", " ", "x"}], "-", 
           RowBox[{"4", " ", 
            SuperscriptBox["x", "2"]}]}]]}]}], "<", "y", "<", 
       RowBox[{
        FractionBox["1", "2"], "+", 
        RowBox[{
         FractionBox["1", "2"], " ", 
         SqrtBox[
          RowBox[{"3", "+", 
           RowBox[{"4", " ", "x"}], "-", 
           RowBox[{"4", " ", 
            SuperscriptBox["x", "2"]}]}]]}]}]}]}], ")"}], "||", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{
        FractionBox["1", "4"], " ", 
        RowBox[{"(", 
         RowBox[{"1", "-", 
          SqrtBox["7"]}], ")"}]}], "<", "x", "<", 
       RowBox[{
        FractionBox["1", "4"], " ", 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          SqrtBox["7"]}], ")"}]}]}], "&&", 
      RowBox[{
       RowBox[{
        FractionBox["1", "2"], "-", 
        RowBox[{
         FractionBox["1", "2"], " ", 
         SqrtBox[
          RowBox[{"3", "+", 
           RowBox[{"4", " ", "x"}], "-", 
           RowBox[{"4", " ", 
            SuperscriptBox["x", "2"]}]}]]}]}], "<", "y", "<", 
       SqrtBox[
        RowBox[{"1", "-", 
         SuperscriptBox["x", "2"]}]]}]}], ")"}], "||", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{
        FractionBox["1", "4"], " ", 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          SqrtBox["7"]}], ")"}]}], "<", "x", "<", "1"}], "&&", 
      RowBox[{
       RowBox[{"-", 
        SqrtBox[
         RowBox[{"1", "-", 
          SuperscriptBox["x", "2"]}]]}], "<", "y", "<", 
       SqrtBox[
        RowBox[{"1", "-", 
         SuperscriptBox["x", "2"]}]]}]}], ")"}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"-", "3"}], "-", 
      RowBox[{"4", " ", "x"}], "+", 
      RowBox[{"8", " ", 
       SuperscriptBox["x", "2"]}]}], "\[Equal]", "0"}], "||", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], "\[Equal]", "0"}], "||", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "-", 
      RowBox[{"2", " ", "x"}], "+", 
      RowBox[{"2", " ", 
       SuperscriptBox["x", "2"]}], "-", 
      RowBox[{"2", " ", "y"}], "+", 
      RowBox[{"2", " ", 
       SuperscriptBox["y", "2"]}]}], "\[Equal]", "0"}]}]}], "}"}]], "Output",
 ImageSize->{479, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->92786582]
}, Open  ]],

Cell["\<\
The hypersurface contains the difference between all solutions and the full\
\[Hyphen]dimensional set:\
\>", "ExampleText",
 CellID->387256760],

Cell[BoxData[{
 RowBox[{
  RowBox[{"allsols", "=", 
   RowBox[{"RegionPlot", "[", 
    RowBox[{"ineqs", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", "0.5", "]"}], ",", "Yellow"}], "}"}]}]}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"fulldim", "=", 
   RowBox[{"RegionPlot", "[", 
    RowBox[{"full", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", "0.5", "]"}], ",", "Red"}], "}"}]}]}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"hypersurface", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"ContourPlot", "[", 
      RowBox[{"#", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", 
         RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
       RowBox[{"ContourStyle", "->", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Opacity", "[", "0.5", "]"}], ",", "Blue"}], "}"}]}]}], 
      "]"}], "&"}], "/@", 
    RowBox[{"List", "@@", "hyp"}]}]}], ";"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->221223872],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"allsols", ",", "fulldim", ",", "hypersurface"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->787286033],

Cell[BoxData[
 GraphicsBox[{GraphicsComplexBox[CompressedData["
1:eJyVWH1wVNUV3zJgoSIFQmuNA4bhTXTUMhiYIp3qlQjkIWHGQRrbSBEpCyPk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    "], {{
      {RGBColor[1, 1, 0], Opacity[0.5], EdgeForm[None], 
       GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHmY11UVxu8FHEA2BQETRXaQfQlZhnCbGUQDCQaRMEQWDYVBAQljCQOX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          "]], PolygonBox[CompressedData["
1:eJwtlWWrVFEYRvdroYiJgV3Y2N2NrdjdomIHdrder93d3YHdGKAIfhH8IaJi
r8dnPiz2OnPO7LPnrak0flb/mTlSStMhF/yB+5HSe9a2MBYfB1vxkTAenwDb
8AnQHr8H7/A3rC+gSXivT9AVfwwf8GnQHL8Br/A8UA+fByvw2TAYHwKT8DbQ
NLOn9hoKZfFBMB8fB6PxMTAXH6Y98SfwER8EDfDb8BYfAh3x1/AZnwir8VWw
MPk3TdFesAufBafCMVEsHugZriuH/QoUxAMW4+vgJP6LtZbiCMfCZ9ZZ78FA
vBzcwTfDFrxi+NmLcBSvEH72LhzBD+sz/DacwI/rO/g52IlXg4f49UzuOoVz
oRwq9rmgb3IOFJv64VgrRgXwv6yLYG1yLn+zNk7O6S2ub4Zjq5wpll9YWf7H
9HT4omvyGdbgdeCRzg4r8XaKOX4QiuPr4QR+A0riG+GM3gUz8Urh3CgndXVP
NaSzJ+89ErKT39EXLxbO7XDogZcK+0rFGN8UjtX55NgfCsdSOeiJl4ZR+Krk
s/aGNclnzg7nSLm5BAtUe+FaXgo18dowVbmE7eEaUW1chap4DZiCZ0EJfAOc
wm9Cf7yVegDfBM3wg+FYvIR+eAuYg2+Ehvj+cOyeQSO8cbg3cuqd4T303Req
EfwsDE6Ood59JXxPZ5gW7nH19naYEZkex3fAT5ge9kMwB18Ce5N7ZIRqH/Yk
50S9fyD82zQDdK8I7E5+ZhE+OTxv9FkvvIzOoF6EQuq9cC2rZlVbO8KxVI3t
xffBsUwN/sjUmGpLPaK9Z8PO5HcsDfe0enkf5MbXhmtTMVmOrwj/ds0Q1f7W
cK7VA93xZeHvakaVD/eoelM92Dmc46xMD7UO15RqSTltGa5J1aJyeA3PG559
qvn84Rgrtqq5fFx/Y92fPHOrhHtUvamZqN65BM+Te+hiuCZUC0/hcrgmVAt6
pk+4h9Q7G2AdfiH8rGKoWZUV7g3NrFp4YfVock1/zdSgak89ons5VQPJzwzH
h4VnrT5biHcL94ZyoFopGn63aqZtOOfKtWZ49fAM1exUj3QJ95R6STP1e6ZH
1BuqEcVmQHh2Kkaa5R3C/xWa6f8AxWPOSQ==
          "]]}]}, {}, {}, {}}, 
     {GrayLevel[0], Opacity[0.4], 
      LineBox[{250, 31, 411, 412, 410, 285, 340, 339, 247, 265, 353, 352, 363,
        362, 382, 381, 351, 350, 349, 280, 433, 434, 432, 312, 336, 431, 264, 
       430, 391, 279, 409, 311, 429, 428, 390, 278, 408, 309, 449, 310, 88, 
       308, 307, 427, 426, 389, 277, 407, 306, 425, 424, 388, 406, 405, 423, 
       263, 244, 435, 305, 276, 335, 245, 436, 281, 330, 314, 450, 313, 337, 
       246, 437, 282, 331, 316, 451, 315, 338, 248, 438, 283, 332, 318, 452, 
       317, 341, 284, 414, 287, 413, 286, 266, 249, 415, 416, 392, 393, 394, 
       252, 417, 288, 395, 396, 397, 253, 439, 320, 454, 323, 440, 254, 418, 
       290, 398, 399, 400, 256, 419, 294, 401, 402, 271, 258, 344, 260, 444, 
       327, 300, 358, 373, 374, 328, 375, 367, 368, 359, 376, 403, 273, 377, 
       347, 348, 301, 386, 387, 261, 86, 421, 422, 420, 302, 334, 274, 378, 
       369, 370, 360, 379, 404, 275, 380, 329, 371, 303, 361, 372, 304, 445, 
       262, 345, 346, 326, 443, 259, 272, 298, 448, 299, 297, 325, 442, 257, 
       270, 295, 447, 296, 293, 324, 441, 255, 269, 291, 446, 292, 343, 342, 
       321, 453, 322, 357, 356, 289, 366, 384, 385, 268, 355, 354, 365, 364, 
       251, 267, 333, 319, 383, 250}]}}], GraphicsComplexBox[CompressedData["
1:eJyVWH1wVNUV3zJgoSIFQmuNA4bhTXTUMhiYIp3qlQjkIWHGQRrbSBEpCyPk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    "], {{
      {RGBColor[1, 0, 0], Opacity[0.5], EdgeForm[None], 
       GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHmY11UVxu8FHEA2BQETRXaQfQlZhnCbGUQDCQaRMEQWDYVBAQljCQOX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          "]], PolygonBox[CompressedData["
1:eJwtlWWrVFEYRvdroYiJgV3Y2N2NrdjdomIHdrder93d3YHdGKAIfhH8IaJi
r8dnPiz2OnPO7LPnrak0flb/mTlSStMhF/yB+5HSe9a2MBYfB1vxkTAenwDb
8AnQHr8H7/A3rC+gSXivT9AVfwwf8GnQHL8Br/A8UA+fByvw2TAYHwKT8DbQ
NLOn9hoKZfFBMB8fB6PxMTAXH6Y98SfwER8EDfDb8BYfAh3x1/AZnwir8VWw
MPk3TdFesAufBafCMVEsHugZriuH/QoUxAMW4+vgJP6LtZbiCMfCZ9ZZ78FA
vBzcwTfDFrxi+NmLcBSvEH72LhzBD+sz/DacwI/rO/g52IlXg4f49UzuOoVz
oRwq9rmgb3IOFJv64VgrRgXwv6yLYG1yLn+zNk7O6S2ub4Zjq5wpll9YWf7H
9HT4omvyGdbgdeCRzg4r8XaKOX4QiuPr4QR+A0riG+GM3gUz8Urh3CgndXVP
NaSzJ+89ErKT39EXLxbO7XDogZcK+0rFGN8UjtX55NgfCsdSOeiJl4ZR+Krk
s/aGNclnzg7nSLm5BAtUe+FaXgo18dowVbmE7eEaUW1chap4DZiCZ0EJfAOc
wm9Cf7yVegDfBM3wg+FYvIR+eAuYg2+Ehvj+cOyeQSO8cbg3cuqd4T303Req
EfwsDE6Ood59JXxPZ5gW7nH19naYEZkex3fAT5ge9kMwB18Ce5N7ZIRqH/Yk
50S9fyD82zQDdK8I7E5+ZhE+OTxv9FkvvIzOoF6EQuq9cC2rZlVbO8KxVI3t
xffBsUwN/sjUmGpLPaK9Z8PO5HcsDfe0enkf5MbXhmtTMVmOrwj/ds0Q1f7W
cK7VA93xZeHvakaVD/eoelM92Dmc46xMD7UO15RqSTltGa5J1aJyeA3PG559
qvn84Rgrtqq5fFx/Y92fPHOrhHtUvamZqN65BM+Te+hiuCZUC0/hcrgmVAt6
pk+4h9Q7G2AdfiH8rGKoWZUV7g3NrFp4YfVock1/zdSgak89ons5VQPJzwzH
h4VnrT5biHcL94ZyoFopGn63aqZtOOfKtWZ49fAM1exUj3QJ95R6STP1e6ZH
1BuqEcVmQHh2Kkaa5R3C/xWa6f8AxWPOSQ==
          "]]}]}, {}, {}, {}}, 
     {GrayLevel[0], Opacity[0.4], 
      LineBox[{250, 31, 411, 412, 410, 285, 340, 339, 247, 265, 353, 352, 363,
        362, 382, 381, 351, 350, 349, 280, 433, 434, 432, 312, 336, 431, 264, 
       430, 391, 279, 409, 311, 429, 428, 390, 278, 408, 309, 449, 310, 88, 
       308, 307, 427, 426, 389, 277, 407, 306, 425, 424, 388, 406, 405, 423, 
       263, 244, 435, 305, 276, 335, 245, 436, 281, 330, 314, 450, 313, 337, 
       246, 437, 282, 331, 316, 451, 315, 338, 248, 438, 283, 332, 318, 452, 
       317, 341, 284, 414, 287, 413, 286, 266, 249, 415, 416, 392, 393, 394, 
       252, 417, 288, 395, 396, 397, 253, 439, 320, 454, 323, 440, 254, 418, 
       290, 398, 399, 400, 256, 419, 294, 401, 402, 271, 258, 344, 260, 444, 
       327, 300, 358, 373, 374, 328, 375, 367, 368, 359, 376, 403, 273, 377, 
       347, 348, 301, 386, 387, 261, 86, 421, 422, 420, 302, 334, 274, 378, 
       369, 370, 360, 379, 404, 275, 380, 329, 371, 303, 361, 372, 304, 445, 
       262, 345, 346, 326, 443, 259, 272, 298, 448, 299, 297, 325, 442, 257, 
       270, 295, 447, 296, 293, 324, 441, 255, 269, 291, 446, 292, 343, 342, 
       321, 453, 322, 357, 356, 289, 366, 384, 385, 268, 355, 354, 365, 364, 
       251, 267, 333, 319, 383, 250}]}}], GraphicsComplexBox[CompressedData["
1:eJxdlntMzWEYx4+syaWJ2WIooVyGaTNmxmlrQnOZS/7BXNMQZWcj/MHaovxB
JWxlhQ3LZQinnHBymblTpHSOEpFUx2WMLOWc3/N8v7+ts9rbZ9/39/Q77/s8
3+cJW5O0KN7PYrHYvb++NXvkHO9PjdNifL45+52yvemzsMY56Ldf59mtX8lx
DVe2TJzpcWL/Fpd9s8veSn1fxfXEvaNNvuvc6O7d3kx2+7ZvNvnXheSa1XVf
yJF5YbGNRU2MH2v80UR9bcao2dOPfCYXbKsuvrOhkVziC+/6RC5fV3tt4DyT
AxcdDM8f9pHxI3xLdgN1a9ThET+7m7xrfMyhHVXvyTnGcyZfGDw367mjntwU
8M8Sd+Ad4/sZ5/mO+mSPf0fch1rz+QcJb1ftd5PrjPNxkbuu6bsrS4L/VDG+
w9j+mnrLsvqrd29VktdNjs55Fv+SfMR4roL8oP+CzJ3jysm7vbfR68xTxv9g
vM9j6rO8p7Fm0kPz+3hvIyX1HjnXeO4OucN7Gi/6lpFrNkXmPim1M/5JYy2i
Xum4PaDt/Dly6NAToccfZZElPy/d4P0PMXQr+LA8b0X8VIlP3SH/n+yR9yPv
l/cnR8j3I0+R78/4L+V8qCfJ+ZHL5HzJB+T8ycvlfsgNcn+MXyT3S32P3D+5
p6zkKskf8inJL/IEyT/Gb+/0feqof5T8JRdKfpMzJf/J26U+yNOkfhh/uNSX
+b5Sf+SnUp/kq1K/5Dypb/JKqX/GjxF/oD5e/IP8XfyFXC3+Q3aKP5HTxL8Y
P1H8jfpi8T+y+iNZ/dNqK1gRlTi21dqVe9w7XZFp+0q+WDipwZ1gslyfh3wz
MjWjY77J8cFBebHRHmu6vifeD3rXdXBa88KSkFbul/tooT4mcOrvN/4ml/+4
HxDxt5n7cV5g9WvyGD1vI+zIVivuA7r6Nfmo3icY9w1Wvyb7a74gPvIJuvo1
2ab5CEa+gtWvyfWa74iPeoCufk3O13oCo97AQVqP4BStV8RHPUNXvyYvUT8A
wy/A6tfkVeo3iA8/gh6jfgUOVj8Dw+/A/9QPwcfULxEffgr9hfot80H9GAy/
BheK7uR5lYrfIz76AfQb2i/Agd2kn4DRb8CjtB+Bl2q/Qnz0M+jJ2u/4fbQf
gtEvwerX5J/abxEf/Ri6+jV5IPq4Mvo9+LTOA+AZOi8gPuYJ6OrX5Ms6j4Ax
r4DVr8nROu8gPuYh6OrX5Eqdp8CYt8Dq1+QEndcQH/McdPVrcpvOg2DMi/Cn
V+JP5HHiX9wfLv5GPUT8j7xNVu7H/Ap9vfgr+Zb4L/cXiz9TvyT+TQ4Qf+d+
zNP/AQpn2Vo=
    "], {{}, {}, 
     {RGBColor[0, 0, 1], Opacity[0.5], 
      TagBox[
       TooltipBox[
        LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
         18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 
         35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 
         52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 
         69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85}],
        
        RowBox[{
          RowBox[{
            RowBox[{"-", "3"}], "-", 
            RowBox[{"4", " ", "x"}], "+", 
            RowBox[{"8", " ", 
              SuperscriptBox["x", "2"]}]}], "\[Equal]", "0"}]],
       Annotation[#, -3 - 4 $CellContext`x + 8 $CellContext`x^2 == 0, 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 
         101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114,
          115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 
         128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141,
          142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 
         155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168,
          169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 
         182}],
        RowBox[{
          RowBox[{
            RowBox[{"-", "3"}], "-", 
            RowBox[{"4", " ", "x"}], "+", 
            RowBox[{"8", " ", 
              SuperscriptBox["x", "2"]}]}], "\[Equal]", "0"}]],
       Annotation[#, -3 - 4 $CellContext`x + 8 $CellContext`x^2 == 0, 
        "Tooltip"]& ]}}], GraphicsComplexBox[CompressedData["
1:eJx1mAlsVFUUhoeKoomtuMQEjFTaWlFBFCjWCBwMiQluUBTFhCjRiIqlKIZF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    "], {{}, {}, 
     {RGBColor[0, 0, 1], Opacity[0.5], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1FdUEAQYBWBAlmxENiJDAWXvJQhUlllZtG1YWJqZ0FAb5myn2VAr01Lb
u7RhO9vZsKysTG1PVzu39nF8+O7/fs+5f3ZHV3tnYEBAwEGi+wbRg2BCCCWM
cHoSQSRRRBNDLHHE04sEepNIEsmkkEoa6WTQh0z6kkU2OeTSj/7kkU8BAxhI
IUUUU0IpZZRTQSVVVFNDLXXU00Ajg2iimcG00EpbdxcczCEM4VAOYyiHM4wj
OJKjGM7RHEM7x3Icx3MCJ3ISJzOCUziV0zidkZzBmXQwirM4m9GM4RzGci7j
OI/xdNLF+VzAhVzEBCYyiYu5hEu5jMlczhSmMo3pzGAmV3AlV3E113At13E9
s5jNDczhRm7iZm5hLvOYz63cxu0s4A4Wsog7uYvFLGEpd3MP93If9/MAD/IQ
D/MIj/IYj/MET7KM5TzF0zzDs6zgOZ7nBV7kJV7mFV5lJa/xOm/wJm/xNu/w
Lqt4j/f5gA9ZzUd8zBo+4VM+Yy2f8wVfso6vWM8GNvI13/At3/E9P/AjP/Ez
v/Arv7GJzWxhK9v4nT/4k7/4m3/4l//Yzg52sovd7GEv+9hP9/gDCaIHwYQQ
Shjh9CSCSKKIJoZY4oinFwn0JpEkkkkhlTTSyaAPmfQli2xyyKUf/ckjnwIG
MJBCiiimhFLKKKeCSqqopoZa6qingUYG0UQzg2mhlbbAA7/wf46diqI=
         "]],
        RowBox[{
          RowBox[{
            RowBox[{"-", "1"}], "+", 
            SuperscriptBox["x", "2"], "+", 
            SuperscriptBox["y", "2"]}], "\[Equal]", "0"}]],
       Annotation[#, -1 + $CellContext`x^2 + $CellContext`y^2 == 0, 
        "Tooltip"]& ]}}], GraphicsComplexBox[CompressedData["
1:eJxdlXtMFFcUxokmGq1FCUZct65B/QP7SLQuqQ3iiSZiFlECG0JM+QMxIKKU
gEmRUAurjZJWtGJZER9IW0LWrNECobzsAbUI+FoeywILjNINlGzkYaJQ1NqZ
2fstCTezmZ3J3HvP/c53ficwIS06cZ6Pj0+6/FPuBT/eM9elONhHHRO0PyV+
vGhVD1tNdw7If2lffrYpx97Doe4FRlflGI3NSzAeaunlI87qVGf1C/IvO9q7
JKqPT3XUHjkV9IIk5XWqk5PUZdzUUde0/F+rkztqAqZj891kaNBMJt13co/6
nZt0iVPaxCkn10pB9VKQm1Jf1lR+P9DPH24MD87qH6UbXSXROeEDHFaw3lCw
fpQMEdlr548OcPKWhD5/zT908qJxImPxIL8q/X2bvnuESkayro1kDXKVGsYw
xczPftDVOMhxSjghwxRaTD3H9BIv+DajZsm0iz55VS7vIPFag7KBi97++q79
v2SJddrz+S1hLrpp/iJsY7vEW/d9lnYmfoh2v9sgz5D4JyUcwxDdjanaXOaQ
OC1y3adT+iHy6PhM6PnM+/z2vTIk7/O0eG4S81ftry33MzspXKzv8OhIVrF/
zp6ywJFtTnoj4qtvHXB9neGgDSL+CvXzbgoR5xtdqD03vLSbjOL8FPikLTKv
k64Kfc6q8XdQrtBvccOVExeNNtol9N2ryzTfvvyUyoX+nWo8DylF5OcOOT7W
nm2hAJG/mdw4f1PcPdop8vuDuv5deiryb/modM31tiavPw5+t+NKYUkj+Qn/
1B/eVPyovppWWm0aq62XT6jzKyhCzXsvt1xShoVihR9XRJyXl/6NvhJ+tbdG
5lW+vEC3LHpX/0H4+XZD7NFg+XLwjmTTimTTN1zn0YlPd25f3bU9n4PVD+y8
ujBKUxhl4UNbFQG7+BfP/rz5ywvy1ckrM9MDMtP/YLNHN/5Ap4w/OVqjTGxn
P9/ny3yfN3KIvvt084yNiz3n5zXjk7rxySdsj3ssX3/x3x4deWFRaPzr3Fbu
Kz8uX23eO97jO8zDOlgX+2BfxIG4ECfixjlwLpwT54YO0AU6QTfoCF2hM3RH
HpAX5Al5dAtuIM/IO3wAX8An8A189PNxX8ui5lmfwXfwIXwJn8K38DF8DZ/D
96iDVltzzPvc2TpB3aCOUFeoM9Qd6hB1iTpF3aKOZ/I+r0q3z9Y56n56Dhfm
cmIuR8AZcAccwvrgFLgFjoFr4By4Bw6Ci+AkuAmOPhT6gLPgLjgMLoPT4DY4
Dq6D8+A++kCRyC/6BPoG+gj8gT6DvoM+pBXcOJCkjDEvN9C30MfQ19Dn0PfQ
B/8H6CjK+g==
    "], {{}, {}, 
     {RGBColor[0, 0, 1], Opacity[0.5], 
      TagBox[
       TooltipBox[
        LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
         18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 
         35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 
         52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 
         69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 
         86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 
         102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113}],
        RowBox[{
          RowBox[{
            RowBox[{"-", "1"}], "-", 
            RowBox[{"2", " ", "x"}], "+", 
            RowBox[{"2", " ", 
              SuperscriptBox["x", "2"]}], "-", 
            RowBox[{"2", " ", "y"}], "+", 
            RowBox[{"2", " ", 
              SuperscriptBox["y", "2"]}]}], "\[Equal]", "0"}]],
       Annotation[#, -1 - 2 $CellContext`x + 2 $CellContext`x^2 - 
         2 $CellContext`y + 2 $CellContext`y^2 == 0, "Tooltip"]& ]}}]},
  AspectRatio->1,
  Frame->True,
  Method->{"AxesInFront" -> True},
  PlotRange->{{-1, 1}, {-1, 1}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 172},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->13693608]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->201923221],

Cell["There is no remaining hypersurface here:", "ExampleText",
 CellID->67884103],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GenericCylindricalDecomposition", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], "+", 
     RowBox[{"y", "^", "2"}]}], "<", "1"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23732627],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "<", "x", "<", "1"}], "&&", 
    RowBox[{
     RowBox[{"-", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]]}], "<", "y", "<", 
     SqrtBox[
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}]]}]}], ",", "False"}], "}"}]], "Output",
 ImageSize->{286, 27},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->152019689]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->120852689],

Cell["Here the whole solution set is lower-dimensional:", "ExampleText",
 CellID->293777608],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GenericCylindricalDecomposition", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], "==", "1"}], "&&", 
    RowBox[{"x", ">", "y"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->131306625],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"False", ",", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     SuperscriptBox["x", "2"], "+", 
     SuperscriptBox["y", "2"]}], "\[Equal]", "0"}]}], "}"}]], "Output",
 ImageSize->{154, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->42399086]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Plot a region described by inequalities:", "ExampleText",
 CellID->636227657],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ineqs", " ", "=", " ", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], " ", "+", " ", 
       RowBox[{"y", "^", "2"}], " ", "+", " ", 
       RowBox[{"z", "^", "2"}], " ", "-", " ", 
       RowBox[{"3", " ", "x", " ", "y", " ", "z"}]}], " ", "<", " ", "1"}], 
     " ", "&&", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], " ", "+", " ", 
       RowBox[{"y", "^", "2"}]}], " ", ">=", " ", 
      RowBox[{"2", " ", 
       RowBox[{"z", "^", "3"}]}]}], " ", "&&", " ", "\n", "    ", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "4"}], " ", "+", " ", 
       RowBox[{"y", "^", "4"}], " ", "+", " ", 
       RowBox[{"z", "^", "4"}]}], " ", "<=", " ", "3"}]}]}], ";"}], 
  " "}]], "Input",
 CellLabel->"In[1]:=",
 CellID->142404062],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{"full", ",", "hyp"}], "}"}], "=", 
    RowBox[{"GenericCylindricalDecomposition", "[", 
     RowBox[{"ineqs", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]}], ";"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->522106679],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.312999999999996`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->3406339]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"plots", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Plot3D", "[", 
      RowBox[{
       RowBox[{"Evaluate", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"#", "[", 
           RowBox[{"[", 
            RowBox[{"3", ",", "1"}], "]"}], "]"}], ",", 
          RowBox[{"#", "[", 
           RowBox[{"[", 
            RowBox[{"3", ",", "5"}], "]"}], "]"}]}], "}"}], "]"}], ",", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"#", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "1"}], "]"}], "]"}], ",", 
          RowBox[{"#", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "5"}], "]"}], "]"}]}], "}"}], "]"}], ",", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"#", "[", 
           RowBox[{"[", 
            RowBox[{"2", ",", "1"}], "]"}], "]"}], ",", 
          RowBox[{"#", "[", 
           RowBox[{"[", 
            RowBox[{"2", ",", "5"}], "]"}], "]"}]}], "}"}], "]"}], ",", 
       RowBox[{"Mesh", "->", "None"}]}], "]"}], "&"}], "/@", 
    RowBox[{"List", "@@", 
     RowBox[{"(", 
      RowBox[{"full", "//.", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"a___", "&&", "b_Or", "&&", "c___"}], ":>", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            RowBox[{"a", "&&", "#", "&&", "c"}], "&"}], "/@", "b"}], ")"}]}], 
        "}"}]}], ")"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->42018775],

Cell["The curves on the surface correspond to cylinder boundaries:", \
"ExampleText",
 CellID->144136462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"plots", ",", 
   RowBox[{"PlotRange", "->", "All"}], ",", 
   RowBox[{"BoxRatios", "->", "Automatic"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->88134385],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 189},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->65785604]
}, Open  ]],

Cell["\<\
Computing full cylindrical decomposition takes longer and is not necessary to \
plot the picture:\
\>", "ExampleText",
 CellID->332568270],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"CylindricalDecomposition", "[", 
    RowBox[{"ineqs", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}], ";"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->99052342],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"20.87499999999999`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->926533285]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RegionPlot3D",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RegionPlot3D"]], "InlineFormula"],
 " uses a numerical method and gives a somewhat less precise picture:"
}], "ExampleText",
 CellID->505795927],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RegionPlot3D", "[", 
  RowBox[{"ineqs", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1.33"}], ",", "1.33"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1.33"}], ",", "1.33"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", 
     RowBox[{"-", "1.33"}], ",", "1.33"}], "}"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "None"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->7398354],

Cell[BoxData[
 Graphics3DBox[GraphicsComplex3DBox[CompressedData["
1:eJyVnE2oXVcVx29UOggitASLk6BIIWYiqCAI8QgOmpEkbSZCR4FWBzU2oiih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   "], {{
     {EdgeForm[None], GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJxF13f8V1Mcx/F77ldZpaU06dfW0tAkFO2hvY2iQWkKLe2tSUuhtHcZkbYG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         "]], Polygon3DBox[CompressedData["
1:eJxFl3mwzmUUx5/f++Lei1IitFBcruW613IvUV171NQU7QqRfatmok3RTGiS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         "]], Polygon3DBox[CompressedData["
1:eJwt13W0FVUUx/GZedLdHQ8EA5BYkvLoBukuaRCEB6goKaCAlDTSLFDpbiUk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         "]], Polygon3DBox[CompressedData["
1:eJwt1FtMz2EYwPH3/8tc2IhQYkZLpaRU1mEOc8GIYopcMDYXTkMpiWKmbCoU
sdFYMoecjaWTUV3IFRuGkaRZoVioqHXg++x9Lj7bf//nPT7P8/58NiYn7HSM
MfMxHNddxtzAFVxCMlKwjEFxiMVijII7cpGHw8jGEh3zibWa0YQPKMY51LLW
Y1SiAoeQg5XMSUC8Y/fywFhsYc5mbMB6jIMnthO7hdu4gzJsQg6xI8hFHsIw
09gxv/AbXejENTSx90c04h0u693bmNOKFmPvUoJS/GRON3rwB3dxEw14inrU
yjk0J8nYhVSkIRIRCGaPMIRjFnpZuwcuN2McDDJmANWowXLGxGMpYjEeXpiK
KZiEiS6bcznzS7zAczxDPo4ZW1O540WUYiu24SqxK3rHC1iFJBxn7wLka63j
tEbRmIO5mIch1ujHfX7/RS/6NOdSowZiT1Dnsj2Qi3yks0cW9iMTAfCDDznw
gz8C0MoanzGMWLDWVGobrn3bRqwdHfiOiyjBaiQh0bE9NgHeUn/mdGvOu/AA
9zCdvQIxDb74ythv8NK50pNjHNszzS67Z4CecwaC3OwZpBbeWiNP6S99C55a
u9Fwx2v+e4tCFGiNjiIGc4zdy0Pf2ki8117tINaOL8b2qtSuDGcYcxancFLv
vgahxEIQhEDpM7jJmyR2HsWOnbsW66SGxAYwiH48Qo30BRMTscJle9Jbey+U
OSGYLvnAD/7rhD9z9mIfMpAGX0QTi8FsRGCI//7hjbE5eaU9LDk5IW9Ze0a+
Fdn6FuQMUv9IRMmaGKH9kMEZMpGF/VigveqjZ9iNVOzAZJQTq0AlqpCubzZb
3/JBHMBCLHJsz8ib7TM2V9WoMPaNy9w9jj2DvJEo+R6iRfpGvjlaI8l5uc6V
3FYhRb5txtauCIWOfYMJ2sOSwzrUy7cUD/Uu8o1t1F6SHBbhNP4DDmrC2g==

         "]]}]}, {}, {}, {}, {}}, {Line3DBox[{12, 10, 56, 12}], 
     Line3DBox[{51, 49, 137, 51}], Line3DBox[{675, 673, 748, 675}], 
     Line3DBox[{743, 741, 807, 743}]}},
   VertexNormals->CompressedData["
1:eJyNXAl0jlcTDkqDoCpRaW1VW6i2pCvxTxFE7EuT2KLV2EJRa1BFikRRpLQE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    "]],
  Axes->True,
  BoxRatios->{1, 1, 1},
  PlotRange->{{-1.33, 1.33}, {-1.33, 1.33}, {-1.33, 1.33}},
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 207},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->133391227,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GenericCylindricalDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GenericCylindricalDecomposition"]], 
  "InlineFormula"],
 " finds the solution set up to lower\[Hyphen]dimensional parts:"
}], "ExampleText",
 CellID->186020544],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GenericCylindricalDecomposition", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}], "&&", 
    RowBox[{"x", ">", "y"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->669879111],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", 
         SqrtBox["2"]]}], "<", "x", "<", 
       FractionBox["1", 
        SqrtBox["2"]]}], "&&", 
      RowBox[{
       RowBox[{"-", 
        SqrtBox[
         RowBox[{"1", "-", 
          SuperscriptBox["x", "2"]}]]}], "<", "y", "<", "x"}]}], ")"}], "||", 
    
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       FractionBox["1", 
        SqrtBox["2"]], "<", "x", "<", "1"}], "&&", 
      RowBox[{
       RowBox[{"-", 
        SqrtBox[
         RowBox[{"1", "-", 
          SuperscriptBox["x", "2"]}]]}], "<", "y", "<", 
       SqrtBox[
        RowBox[{"1", "-", 
         SuperscriptBox["x", "2"]}]]}]}], ")"}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      RowBox[{"2", " ", 
       SuperscriptBox["x", "2"]}]}], "\[Equal]", "0"}], "||", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], "\[Equal]", "0"}]}]}], "}"}]], "Output",
 ImageSize->{519, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->237898438]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->129177953],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["CylindricalDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CylindricalDecomposition"]], "InlineFormula"],
 " finds the complete solution set:"
}], "ExampleText",
 CellID->239265583],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CylindricalDecomposition", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}], "&&", 
    RowBox[{"x", ">", "y"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->424647270],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", 
       SqrtBox["2"]]}], "<", "x", "\[LessEqual]", 
     FractionBox["1", 
      SqrtBox["2"]]}], "&&", 
    RowBox[{
     RowBox[{"-", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]]}], "\[LessEqual]", "y", "<", "x"}]}], 
   ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     FractionBox["1", 
      SqrtBox["2"]], "<", "x", "<", "1"}], "&&", 
    RowBox[{
     RowBox[{"-", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]]}], "\[LessEqual]", "y", "\[LessEqual]", 
     SqrtBox[
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}]]}]}], ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "\[Equal]", "1"}], "&&", 
    RowBox[{"y", "\[Equal]", "0"}]}], ")"}]}]], "Output",
 ImageSize->{375, 75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->407361169]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->452064319],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Reduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reduce"]], "InlineFormula"],
 " uses cylindrical decomposition to solve inequalities:"
}], "ExampleText",
 CellID->285231363],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "+", 
      RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}], "&&", 
    RowBox[{"x", ">", "y"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30158986],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", 
       SqrtBox["2"]]}], "<", "x", "\[LessEqual]", 
     FractionBox["1", 
      SqrtBox["2"]]}], "&&", 
    RowBox[{
     RowBox[{"-", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]]}], "\[LessEqual]", "y", "<", "x"}]}], 
   ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     FractionBox["1", 
      SqrtBox["2"]], "<", "x", "<", "1"}], "&&", 
    RowBox[{
     RowBox[{"-", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]]}], "\[LessEqual]", "y", "\[LessEqual]", 
     SqrtBox[
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}]]}]}], ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "\[Equal]", "1"}], "&&", 
    RowBox[{"y", "\[Equal]", "0"}]}], ")"}]}]], "Output",
 ImageSize->{375, 75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->333839224]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["\<\
When the solution set is lower-dimensional no solutions are found:\
\>", "ExampleText",
 CellID->735379744],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GenericCylindricalDecomposition", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "y"}], ")"}], "^", "2"}], "\[LessEqual]", "0"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->36749648],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"False", ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "2"], "-", 
     RowBox[{"2", " ", "x", " ", "y"}], "+", 
     SuperscriptBox["y", "2"]}], "\[Equal]", "0"}]}], "}"}]], "Output",
 ImageSize->{166, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->316350489]
}, Open  ]],

Cell[TextData[{
 "To find lower-dimensional solution sets use ",
 Cell[BoxData[
  ButtonBox["CylindricalDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CylindricalDecomposition"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->334760405],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CylindricalDecomposition", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", "y"}], ")"}], "^", "2"}], "\[LessEqual]", "0"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->284709399],

Cell[BoxData[
 RowBox[{"y", "\[Equal]", "x"}]], "Output",
 ImageSize->{35, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->237107949]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["CylindricalDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CylindricalDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Reduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Reduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SemialgebraicComponentInstances",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SemialgebraicComponentInstances"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["The Representation of Solution Sets",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/TheRepresentationOfSolutionSets"]], "Tutorials",\

 CellID->227434737]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->138762797],

Cell[TextData[ButtonBox["Polynomial Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialSystems"]], "MoreAbout",
 CellID->15227048],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->109983990]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"GenericCylindricalDecomposition - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 45, 0.5340288}", "context" -> "System`", 
    "keywords" -> {
     "cylindrical algebraic decomposition", "cylindrical parts", 
      "full\\[Hyphen]dimensional inequality solutions", 
      "generic inequality solutions", "regular sets"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "GenericCylindricalDecomposition[ineqs, {x_1, x_2, ...}] finds the \
full-dimensional part of the decomposition of the region represented by the \
inequalities ineqs into cylindrical parts whose directions correspond to the \
successive x_i, together with any hypersurfaces containing the rest of the \
region.", "synonyms" -> {"generic cylindrical decomposition"}, "title" -> 
    "GenericCylindricalDecomposition", "type" -> "Symbol", "uri" -> 
    "ref/GenericCylindricalDecomposition"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6980, 206, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->34455918]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 146940, 3110}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2313, 56, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2936, 83, 1596, 37, 70, "ObjectNameGrid"],
Cell[4535, 122, 1247, 35, 70, "Usage",
 CellID->30562]
}, Open  ]],
Cell[CellGroupData[{
Cell[5819, 162, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6297, 179, 254, 8, 70, "Notes",
 CellID->23865],
Cell[6554, 189, 93, 1, 70, "Notes",
 CellID->17554],
Cell[6650, 192, 293, 9, 70, "Notes",
 CellID->20700]
}, Closed]],
Cell[CellGroupData[{
Cell[6980, 206, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->34455918],
Cell[CellGroupData[{
Cell[7364, 220, 148, 5, 70, "ExampleSection",
 CellID->555575941],
Cell[7515, 227, 132, 4, 70, "ExampleText",
 CellID->504489806],
Cell[CellGroupData[{
Cell[7672, 235, 301, 10, 28, "Input",
 CellID->407112382],
Cell[7976, 247, 627, 23, 48, "Output",
 CellID->119351041]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8652, 276, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[8876, 285, 295, 8, 70, "ExampleText",
 CellID->29891744],
Cell[9174, 295, 646, 21, 70, "Input",
 CellID->181235941],
Cell[CellGroupData[{
Cell[9845, 320, 279, 9, 70, "Input",
 CellID->3731316],
Cell[10127, 331, 2902, 100, 138, "Output",
 CellID->92786582]
}, Open  ]],
Cell[13044, 434, 153, 4, 70, "ExampleText",
 CellID->387256760],
Cell[13200, 440, 1546, 50, 70, "Input",
 CellID->221223872],
Cell[CellGroupData[{
Cell[14771, 494, 156, 4, 70, "Input",
 CellID->787286033],
Cell[14930, 500, 27557, 476, 70, "Output",
 CellID->13693608]
}, Open  ]],
Cell[42502, 979, 125, 3, 70, "ExampleDelimiter",
 CellID->201923221],
Cell[42630, 984, 82, 1, 70, "ExampleText",
 CellID->67884103],
Cell[CellGroupData[{
Cell[42737, 989, 299, 10, 70, "Input",
 CellID->23732627],
Cell[43039, 1001, 490, 18, 48, "Output",
 CellID->152019689]
}, Open  ]],
Cell[43544, 1022, 125, 3, 70, "ExampleDelimiter",
 CellID->120852689],
Cell[43672, 1027, 92, 1, 70, "ExampleText",
 CellID->293777608],
Cell[CellGroupData[{
Cell[43789, 1032, 355, 12, 70, "Input",
 CellID->131306625],
Cell[44147, 1046, 348, 12, 39, "Output",
 CellID->42399086]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44544, 1064, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[44774, 1073, 83, 1, 70, "ExampleText",
 CellID->636227657],
Cell[44860, 1076, 831, 25, 70, "Input",
 CellID->142404062],
Cell[CellGroupData[{
Cell[45716, 1105, 354, 12, 70, "Input",
 CellID->522106679],
Cell[46073, 1119, 221, 7, 36, "Output",
 CellID->3406339]
}, Open  ]],
Cell[46309, 1129, 1538, 47, 70, "Input",
 CellID->42018775],
Cell[47850, 1178, 105, 2, 70, "ExampleText",
 CellID->144136462],
Cell[CellGroupData[{
Cell[47980, 1184, 212, 6, 70, "Input",
 CellID->88134385],
Cell[48195, 1192, 29276, 484, 70, "Output",
 Evaluatable->False,
 CellID->65785604]
}, Open  ]],
Cell[77486, 1679, 148, 4, 70, "ExampleText",
 CellID->332568270],
Cell[CellGroupData[{
Cell[77659, 1687, 262, 9, 70, "Input",
 CellID->99052342],
Cell[77924, 1698, 224, 7, 36, "Output",
 CellID->926533285]
}, Open  ]],
Cell[78163, 1708, 252, 7, 70, "ExampleText",
 CellID->505795927],
Cell[CellGroupData[{
Cell[78440, 1719, 457, 14, 70, "Input",
 CellID->7398354],
Cell[78900, 1735, 56505, 941, 228, 22935, 386, "CachedBoxData", "BoxData", \
"Output",
 CellID->133391227]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[135454, 2682, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[135695, 2691, 288, 8, 70, "ExampleText",
 CellID->186020544],
Cell[CellGroupData[{
Cell[136008, 2703, 365, 12, 70, "Input",
 CellID->669879111],
Cell[136376, 2717, 1233, 46, 81, "Output",
 CellID->237898438]
}, Open  ]],
Cell[137624, 2766, 125, 3, 70, "ExampleDelimiter",
 CellID->129177953],
Cell[137752, 2771, 242, 7, 70, "ExampleText",
 CellID->239265583],
Cell[CellGroupData[{
Cell[138019, 2782, 358, 12, 70, "Input",
 CellID->424647270],
Cell[138380, 2796, 993, 37, 96, "Output",
 CellID->407361169]
}, Open  ]],
Cell[139388, 2836, 125, 3, 70, "ExampleDelimiter",
 CellID->452064319],
Cell[139516, 2841, 227, 7, 70, "ExampleText",
 CellID->285231363],
Cell[CellGroupData[{
Cell[139768, 2852, 339, 12, 70, "Input",
 CellID->30158986],
Cell[140110, 2866, 993, 37, 96, "Output",
 CellID->333839224]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[141152, 2909, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[141386, 2918, 117, 3, 70, "ExampleText",
 CellID->735379744],
Cell[CellGroupData[{
Cell[141528, 2925, 315, 11, 70, "Input",
 CellID->36749648],
Cell[141846, 2938, 364, 12, 39, "Output",
 CellID->316350489]
}, Open  ]],
Cell[142225, 2953, 259, 8, 70, "ExampleText",
 CellID->334760405],
Cell[CellGroupData[{
Cell[142509, 2965, 309, 11, 70, "Input",
 CellID->284709399],
Cell[142821, 2978, 187, 6, 36, "Output",
 CellID->237107949]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[143069, 2991, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[143388, 3002, 774, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[144199, 3033, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[144520, 3044, 183, 4, 70, "Tutorials",
 CellID->227434737]
}, Open  ]],
Cell[CellGroupData[{
Cell[144740, 3053, 319, 9, 70, "MoreAboutSection",
 CellID->138762797],
Cell[145062, 3064, 146, 3, 70, "MoreAbout",
 CellID->15227048],
Cell[145211, 3069, 169, 3, 70, "MoreAbout",
 CellID->109983990]
}, Open  ]],
Cell[145395, 3075, 27, 0, 70, "History"],
Cell[145425, 3077, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

