(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     40782,       1405]
NotebookOptionsPosition[     33703,       1159]
NotebookOutlinePosition[     35006,       1196]
CellTagsIndexPosition[     34920,       1191]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"HankelH2" :> Documentation`HelpLookup["paclet:ref/HankelH2"], 
          "BesselJ" :> Documentation`HelpLookup["paclet:ref/BesselJ"], 
          "BesselY" :> Documentation`HelpLookup["paclet:ref/BesselY"], 
          "SphericalHankelH1" :> 
          Documentation`HelpLookup["paclet:ref/SphericalHankelH1"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"HankelH2\"\>", 2->"\<\"BesselJ\"\>",
        3->"\<\"BesselY\"\>", 
       4->"\<\"SphericalHankelH1\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"Special Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["HankelH1", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["HankelH1",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/HankelH1"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Hankel function of the first kind ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubsuperscriptBox["H", "n", 
         RowBox[{"(", "1", ")"}]], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->3737]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["H", "n", 
     RowBox[{"(", "1", ")"}]], "(", "z", ")"}], TraditionalForm]], 
  "InlineMath"],
 " is given by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["J", "n"], "(", "z", ")"}], "+", 
    RowBox[{"i", "\[InvisibleSpace]", 
     RowBox[{
      SubscriptBox["Y", "n"], "(", "z", ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Notes",
 CellID->5030],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["HankelH1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HankelH1"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->19782],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["HankelH1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH1"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->32008081],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH1"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH1"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->723534962],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->472411498],

Cell["Evaluate numerically:", "ExampleText",
 CellID->284288149],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH1", "[", 
  RowBox[{"3", ",", " ", "1.2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->373363032],

Cell[BoxData[
 RowBox[{"0.03287433692499494`", "\[InvisibleSpace]", "-", 
  RowBox[{"3.5898996296131855`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->136653921]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->263037437],

Cell[TextData[{
 "Plot the absolute value of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     TagBox["H",
      HankelH1], "n", 
     RowBox[{"(", "1", ")"}]], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 " for various orders:"
}], "ExampleText",
 CellID->125334460],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Abs", "[", 
      RowBox[{"HankelH1", "[", 
       RowBox[{"n", ",", "x"}], "]"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "4"}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24505028],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 142},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->943841251]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->870569274],

Cell["Series at the origin:", "ExampleText",
 CellID->656988425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"HankelH1", "[", 
    RowBox[{"2", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->63700055],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    FractionBox[
     RowBox[{"4", " ", "\[ImaginaryI]"}], 
     RowBox[{"\[Pi]", " ", 
      SuperscriptBox["x", "2"]}]]}], "+", 
   RowBox[{"-", 
    FractionBox["\[ImaginaryI]", "\[Pi]"]}], "+", 
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "3"}], " ", "\[ImaginaryI]"}], "+", 
       RowBox[{"4", " ", "\[ImaginaryI]", " ", "EulerGamma"}], "+", 
       RowBox[{"2", " ", "\[Pi]"}], "-", 
       RowBox[{"4", " ", "\[ImaginaryI]", " ", 
        RowBox[{"Log", "[", "2", "]"}]}], "+", 
       RowBox[{"4", " ", "\[ImaginaryI]", " ", 
        RowBox[{"Log", "[", "x", "]"}]}]}], ")"}], " ", 
     SuperscriptBox["x", "2"]}], 
    RowBox[{"16", " ", "\[Pi]"}]], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "4"],
    SeriesData[$CellContext`x, 0, {}, -2, 4, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
   Complex[0, -4]/Pi, 0, Complex[0, -1]/Pi, 0, Rational[1, 16] 
    Pi^(-1) (Complex[0, -3] + Complex[0, 4] EulerGamma + 2 Pi + 
     Complex[0, -4] Log[2] + Complex[0, 4] Log[$CellContext`x])}, -2, 4, 1],
  Editable->False]], "Output",
 ImageSize->{476, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->490939981]
}, Open  ]],

Cell["Series at infinity:", "ExampleText",
 CellID->298449549],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"HankelH1", "[", 
    RowBox[{"2", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "\[Infinity]", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->146869830],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"\[ImaginaryI]", " ", "x"}]], " ", 
  RowBox[{"(", 
   InterpretationBox[
    RowBox[{
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"-", "1"}], ")"}], 
       RowBox[{"3", "/", "4"}]], " ", 
      SqrtBox[
       FractionBox["2", "\[Pi]"]], " ", 
      SqrtBox[
       FractionBox["1", "x"]]}], "-", 
     FractionBox[
      RowBox[{"15", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], 
        RowBox[{"1", "/", "4"}]], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         FractionBox["1", "x"], ")"}], 
        RowBox[{"3", "/", "2"}]]}], 
      RowBox[{"4", " ", 
       SqrtBox[
        RowBox[{"2", " ", "\[Pi]"}]]}]], "+", 
     InterpretationBox[
      SuperscriptBox[
       RowBox[{"O", "[", 
        FractionBox["1", "x"], "]"}], 
       RowBox[{"5", "/", "2"}]],
      SeriesData[$CellContext`x, 
       DirectedInfinity[1], {}, 1, 5, 2],
      Editable->False]}],
    SeriesData[$CellContext`x, 
     DirectedInfinity[
     1], {(-1)^Rational[3, 4] (2/Pi)^Rational[1, 2], 0, 
      Rational[-15, 4] (-1)^Rational[1, 4] (2 Pi)^Rational[-1, 2]}, 1, 5, 2],
    Editable->False], ")"}]}]], "Output",
 ImageSize->{335, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->216021696]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28825],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->87678633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH1", "[", 
  RowBox[{
   RowBox[{"1.1", "-", 
    RowBox[{"2.3", "I"}]}], ",", " ", 
   RowBox[{"1.2", "+", 
    RowBox[{"1.9", " ", "I"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->570002732],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "0.001727401455632105`"}], "-", 
  RowBox[{"0.00016502144878671743`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{176, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5448230]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->625191067],

Cell["Evaluate numerically to high precision:", "ExampleText",
 CellID->99433911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"HankelH1", "[", 
    RowBox[{"1", ",", "3"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->226681605],

Cell[BoxData[
 RowBox[{"0.\
3390589585259364589255145972064788969730804181980081754563417272149`50.\
00920959146974", "+", 
  RowBox[{
  "0.3246744247917999784370128392879532396692751433723549576619370894173`49.\
99038244758081", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{386, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->42589447]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->166034826],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->490047609],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH1", "[", 
  RowBox[{"1", ",", "3.0000000000000000000000"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18350283],

Cell[BoxData[
 RowBox[{"0.33905895852593645892551523244207964811`21.187718641069583", "+", 
  RowBox[{
  "0.32467442479179997843716866806492308285`20.848925293277727", " ", 
   "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{350, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->250303425]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->171778757],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH1"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->151242334],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH1", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", " ", "1."}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->505082822],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.44005058574493355`", "\[InvisibleSpace]", "-", 
    RowBox[{"0.7812128213002887`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.11490348493190047`", "\[InvisibleSpace]", "-", 
    RowBox[{"1.6506826068162546`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.019563353982668404`", "\[InvisibleSpace]", "-", 
    RowBox[{"5.821517605964729`", " ", "\[ImaginaryI]"}]}]}], "}"}]], "Output",\

 ImageSize->{441, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180167701]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->492188634],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->122431680],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"HankelH1", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->214910545],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubsuperscriptBox[
    TagBox["H",
     HankelH1], "n", 
    RowBox[{"(", "1", ")"}]], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{39, 19},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->220144584]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1244],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH1"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText",
 CellID->461756866],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH1", "[", 
  RowBox[{"2", ",", 
   RowBox[{
    RowBox[{"Log", "[", 
     RowBox[{"1", "+", "x"}], "]"}], "+", 
    RowBox[{
     RowBox[{"O", "[", "x", "]"}], "^", "2"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24445531],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    FractionBox[
     RowBox[{"4", " ", "\[ImaginaryI]"}], 
     RowBox[{"\[Pi]", " ", 
      SuperscriptBox["x", "2"]}]]}], "+", 
   RowBox[{"-", 
    FractionBox["\[ImaginaryI]", "\[Pi]"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "2"],
    SeriesData[$CellContext`x, 0, {}, -2, 2, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {Complex[0, -4]/Pi, 0, Complex[0, -1]/Pi}, -2,
    2, 1],
  Editable->False]], "Output",
 ImageSize->{118, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1973343]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31351],

Cell["Find series expansions at infinity:", "ExampleText",
 CellID->19447],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"HankelH1", "[", 
    RowBox[{"2", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "Infinity", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1378437836],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"\[ImaginaryI]", " ", "x"}]], " ", 
  RowBox[{"(", 
   InterpretationBox[
    RowBox[{
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"-", "1"}], ")"}], 
       RowBox[{"3", "/", "4"}]], " ", 
      SqrtBox[
       FractionBox["2", "\[Pi]"]], " ", 
      SqrtBox[
       FractionBox["1", "x"]]}], "-", 
     FractionBox[
      RowBox[{"15", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], 
        RowBox[{"1", "/", "4"}]], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         FractionBox["1", "x"], ")"}], 
        RowBox[{"3", "/", "2"}]]}], 
      RowBox[{"4", " ", 
       SqrtBox[
        RowBox[{"2", " ", "\[Pi]"}]]}]], "+", 
     InterpretationBox[
      SuperscriptBox[
       RowBox[{"O", "[", 
        FractionBox["1", "x"], "]"}], 
       RowBox[{"5", "/", "2"}]],
      SeriesData[$CellContext`x, 
       DirectedInfinity[1], {}, 1, 5, 2],
      Editable->False]}],
    SeriesData[$CellContext`x, 
     DirectedInfinity[
     1], {(-1)^Rational[3, 4] (2/Pi)^Rational[1, 2], 0, 
      Rational[-15, 4] (-1)^Rational[1, 4] (2 Pi)^Rational[-1, 2]}, 1, 5, 2],
    Editable->False], ")"}]}]], "Output",
 ImageSize->{335, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->24582805]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8529],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to convert to Bessel functions:"
}], "ExampleText",
 CellID->649204644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"HankelH1", "[", 
   RowBox[{"n", ",", "z"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->751371406],

Cell[BoxData[
 RowBox[{
  RowBox[{"BesselJ", "[", 
   RowBox[{"n", ",", "z"}], "]"}], "+", 
  RowBox[{"\[ImaginaryI]", " ", 
   RowBox[{"BesselY", "[", 
    RowBox[{"n", ",", "z"}], "]"}]}]}]], "Output",
 ImageSize->{204, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->65434311]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->282889322],

Cell[TextData[{
 "Integrate expressions with ",
 Cell[BoxData[
  ButtonBox["HankelH1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH1"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->228055373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"HankelH1", "[", 
     RowBox[{"2", ",", "x"}], "]"}], "^", "2"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->253099075],

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   SqrtBox["\[Pi]"]], 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"MeijerG", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", 
            FractionBox["3", "2"]}], ",", 
           FractionBox["1", "2"], ",", 
           FractionBox["5", "2"]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}], ",", "x", ",", 
       FractionBox["1", "2"]}], "]"}]}], "+", 
    RowBox[{"\[ImaginaryI]", " ", 
     RowBox[{"MeijerG", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "1"}], "}"}], ",", 
         RowBox[{"{", "0", "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           FractionBox["1", "2"], ",", 
           FractionBox["5", "2"]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", 
            FractionBox["3", "2"]}], ",", "0", ",", "0"}], "}"}]}], "}"}], 
       ",", "x", ",", 
       FractionBox["1", "2"]}], "]"}]}]}], ")"}]}]], "Output",
 ImageSize->{418, 64},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->289789271]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->20132],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH1"]], "InlineFormula"],
 " does not automatically evaluate symbolically for half-integer arguments:"
}], "ExampleText",
 CellID->907156500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH1", "[", 
  RowBox[{
   RowBox[{"1", "/", "2"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->594280542],

Cell[BoxData[
 RowBox[{"HankelH1", "[", 
  RowBox[{
   FractionBox["1", "2"], ",", "x"}], "]"}]], "Output",
 ImageSize->{104, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->24028302]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to obtain an expanded form:"
}], "ExampleText",
 CellID->377029976],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->122401508],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{"\[ImaginaryI]", " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"\[ImaginaryI]", " ", "x"}]], " ", 
    SqrtBox[
     FractionBox["2", "\[Pi]"]]}], 
   SqrtBox["x"]]}]], "Output",
 ImageSize->{80, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->99059741]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17752],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["HankelH2",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HankelH2"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselY",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselY"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalHankelH1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalHankelH1"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->9204]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->32315],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->32080]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->362408715],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->3778483],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->496452883],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->187932628],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->331705414]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"HankelH1 - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 47, 22.0845274}", "context" -> "System`", 
    "keywords" -> {
     "Hankel function of the first kind", "Hankel functions", "Hankel H1"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "HankelH1[n, z] gives the Hankel function of the first kind \
SubsuperscriptBox[H, n, (1)](z). ", "synonyms" -> {"hankel h1"}, "title" -> 
    "HankelH1", "type" -> "Symbol", "uri" -> "ref/HankelH1"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7921, 254, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->723534962]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 34777, 1184}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2606, 61, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3229, 88, 1573, 37, 70, "ObjectNameGrid"],
Cell[4805, 127, 730, 24, 70, "Usage",
 CellID->3737]
}, Open  ]],
Cell[CellGroupData[{
Cell[5572, 156, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6050, 173, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6173, 179, 467, 19, 70, "Notes",
 CellID->5030],
Cell[6643, 200, 566, 21, 70, "Notes",
 CellID->19782],
Cell[7212, 223, 247, 8, 70, "Notes",
 CellID->32008081],
Cell[7462, 233, 218, 7, 70, "Notes",
 CellID->31226],
Cell[7683, 242, 201, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[7921, 254, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->723534962],
Cell[CellGroupData[{
Cell[8306, 268, 148, 5, 70, "ExampleSection",
 CellID->472411498],
Cell[8457, 275, 64, 1, 70, "ExampleText",
 CellID->284288149],
Cell[CellGroupData[{
Cell[8546, 280, 134, 4, 28, "Input",
 CellID->373363032],
Cell[8683, 286, 274, 7, 36, "Output",
 CellID->136653921]
}, Open  ]],
Cell[8972, 296, 125, 3, 70, "ExampleDelimiter",
 CellID->263037437],
Cell[9100, 301, 296, 12, 70, "ExampleText",
 CellID->125334460],
Cell[CellGroupData[{
Cell[9421, 317, 392, 13, 28, "Input",
 CellID->24505028],
Cell[9816, 332, 4584, 79, 163, "Output",
 Evaluatable->False,
 CellID->943841251]
}, Open  ]],
Cell[14415, 414, 125, 3, 70, "ExampleDelimiter",
 CellID->870569274],
Cell[14543, 419, 64, 1, 70, "ExampleText",
 CellID->656988425],
Cell[CellGroupData[{
Cell[14632, 424, 244, 8, 70, "Input",
 CellID->63700055],
Cell[14879, 434, 1342, 38, 56, "Output",
 CellID->490939981]
}, Open  ]],
Cell[16236, 475, 62, 1, 70, "ExampleText",
 CellID->298449549],
Cell[CellGroupData[{
Cell[16323, 480, 255, 8, 70, "Input",
 CellID->146869830],
Cell[16581, 490, 1385, 46, 67, "Output",
 CellID->216021696]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[18015, 542, 216, 7, 70, "ExampleSection",
 CellID->28825],
Cell[18234, 551, 84, 1, 70, "ExampleText",
 CellID->87678633],
Cell[CellGroupData[{
Cell[18343, 556, 233, 8, 70, "Input",
 CellID->570002732],
Cell[18579, 566, 274, 8, 36, "Output",
 CellID->5448230]
}, Open  ]],
Cell[18868, 577, 125, 3, 70, "ExampleDelimiter",
 CellID->625191067],
Cell[18996, 582, 81, 1, 70, "ExampleText",
 CellID->99433911],
Cell[CellGroupData[{
Cell[19102, 587, 182, 6, 70, "Input",
 CellID->226681605],
Cell[19287, 595, 396, 11, 54, "Output",
 CellID->42589447]
}, Open  ]],
Cell[19698, 609, 125, 3, 70, "ExampleDelimiter",
 CellID->166034826],
Cell[19826, 614, 113, 3, 70, "ExampleText",
 CellID->490047609],
Cell[CellGroupData[{
Cell[19964, 621, 149, 4, 70, "Input",
 CellID->18350283],
Cell[20116, 627, 339, 9, 36, "Output",
 CellID->250303425]
}, Open  ]],
Cell[20470, 639, 125, 3, 70, "ExampleDelimiter",
 CellID->171778757],
Cell[20598, 644, 210, 7, 70, "ExampleText",
 CellID->151242334],
Cell[CellGroupData[{
Cell[20833, 655, 195, 7, 70, "Input",
 CellID->505082822],
Cell[21031, 664, 573, 14, 36, "Output",
 CellID->180167701]
}, Open  ]],
Cell[21619, 681, 125, 3, 70, "ExampleDelimiter",
 CellID->492188634],
Cell[21747, 686, 203, 7, 70, "ExampleText",
 CellID->122431680],
Cell[CellGroupData[{
Cell[21975, 697, 166, 5, 70, "Input",
 CellID->214910545],
Cell[22144, 704, 317, 11, 54, "Output",
 CellID->220144584]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22510, 721, 238, 7, 70, "ExampleSection",
 CellID->1244],
Cell[22751, 730, 211, 7, 70, "ExampleText",
 CellID->461756866],
Cell[CellGroupData[{
Cell[22987, 741, 264, 9, 70, "Input",
 CellID->24445531],
Cell[23254, 752, 666, 22, 53, "Output",
 CellID->1973343]
}, Open  ]],
Cell[23935, 777, 121, 3, 70, "ExampleDelimiter",
 CellID->31351],
Cell[24059, 782, 74, 1, 70, "ExampleText",
 CellID->19447],
Cell[CellGroupData[{
Cell[24158, 787, 253, 8, 70, "Input",
 CellID->1378437836],
Cell[24414, 797, 1384, 46, 67, "Output",
 CellID->24582805]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25847, 849, 232, 7, 70, "ExampleSection",
 CellID->8529],
Cell[26082, 858, 230, 8, 70, "ExampleText",
 CellID->649204644],
Cell[CellGroupData[{
Cell[26337, 870, 169, 5, 70, "Input",
 CellID->751371406],
Cell[26509, 877, 333, 11, 36, "Output",
 CellID->65434311]
}, Open  ]],
Cell[26857, 891, 125, 3, 70, "ExampleDelimiter",
 CellID->282889322],
Cell[26985, 896, 347, 12, 70, "ExampleText",
 CellID->228055373],
Cell[CellGroupData[{
Cell[27357, 912, 215, 7, 70, "Input",
 CellID->253099075],
Cell[27575, 921, 1510, 51, 85, "Output",
 CellID->289789271]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29134, 978, 226, 7, 70, "ExampleSection",
 CellID->20132],
Cell[29363, 987, 250, 7, 70, "ExampleText",
 CellID->907156500],
Cell[CellGroupData[{
Cell[29638, 998, 151, 5, 70, "Input",
 CellID->594280542],
Cell[29792, 1005, 237, 8, 51, "Output",
 CellID->24028302]
}, Open  ]],
Cell[30044, 1016, 226, 8, 70, "ExampleText",
 CellID->377029976],
Cell[CellGroupData[{
Cell[30295, 1028, 110, 3, 70, "Input",
 CellID->122401508],
Cell[30408, 1033, 374, 13, 80, "Output",
 CellID->99059741]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[30843, 1053, 311, 9, 70, "SeeAlsoSection",
 CellID->17752],
Cell[31157, 1064, 954, 35, 70, "SeeAlso",
 CellID->9204]
}, Open  ]],
Cell[CellGroupData[{
Cell[32148, 1104, 314, 9, 70, "TutorialsSection",
 CellID->32315],
Cell[32465, 1115, 144, 3, 70, "Tutorials",
 CellID->32080]
}, Open  ]],
Cell[CellGroupData[{
Cell[32646, 1123, 319, 9, 70, "MoreAboutSection",
 CellID->362408715],
Cell[32968, 1134, 156, 3, 70, "MoreAbout",
 CellID->3778483],
Cell[33127, 1139, 145, 3, 70, "MoreAbout",
 CellID->496452883],
Cell[33275, 1144, 174, 3, 70, "MoreAbout",
 CellID->187932628],
Cell[33452, 1149, 179, 3, 70, "MoreAbout",
 CellID->331705414]
}, Open  ]],
Cell[33646, 1155, 27, 0, 70, "History"],
Cell[33676, 1157, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

