(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     41029,       1398]
NotebookOptionsPosition[     33937,       1152]
NotebookOutlinePosition[     35240,       1189]
CellTagsIndexPosition[     35154,       1184]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"HankelH1" :> Documentation`HelpLookup["paclet:ref/HankelH1"], 
          "BesselJ" :> Documentation`HelpLookup["paclet:ref/BesselJ"], 
          "BesselY" :> Documentation`HelpLookup["paclet:ref/BesselY"], 
          "SphericalHankelH2" :> 
          Documentation`HelpLookup["paclet:ref/SphericalHankelH2"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"HankelH1\"\>", 2->"\<\"BesselJ\"\>",
        3->"\<\"BesselY\"\>", 
       4->"\<\"SphericalHankelH2\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "Functions for Separable Coordinate Systems" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsForSeparableCoordinateSystems"], 
          "Functions Used in Optics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInOptics"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"Functions for Separable Coordinate Systems\"\>", 
       3->"\<\"Functions Used in Optics\"\>", 
       4->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["HankelH2", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["HankelH2",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/HankelH2"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Hankel function of the second kind ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubsuperscriptBox["H", "n", 
         RowBox[{"(", "2", ")"}]], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->32058]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["H", "n", 
     RowBox[{"(", "2", ")"}]], "(", "z", ")"}], TraditionalForm]], 
  "InlineMath"],
 " is given by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["J", "n"], "(", "z", ")"}], "-", 
    RowBox[{"i", "\[InvisibleSpace]", 
     RowBox[{
      SubscriptBox["Y", "n"], "(", "z", ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Notes",
 CellID->8383],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["HankelH1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HankelH1"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->6846],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->209498233],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->6942811],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->9771575],

Cell["Evaluate numerically:", "ExampleText",
 CellID->100529053],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH2", "[", 
  RowBox[{"3", ",", " ", "1.2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->158701808],

Cell[BoxData[
 RowBox[{"0.03287433692499494`", "\[InvisibleSpace]", "+", 
  RowBox[{"3.5898996296131855`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->295029083]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->107801519],

Cell[TextData[{
 "Plot the absolute value of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     TagBox["H",
      HankelH1], "n", 
     RowBox[{"(", "2", ")"}]], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 " for various orders:"
}], "ExampleText",
 CellID->125334460],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"Abs", "[", 
       RowBox[{"HankelH2", "[", 
        RowBox[{"n", ",", "x"}], "]"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", "4"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->125209509],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 142},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->132483283]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->870569274],

Cell["Series at the origin:", "ExampleText",
 CellID->656988425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"HankelH2", "[", 
    RowBox[{"2", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->63700055],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox[
    RowBox[{"4", " ", "\[ImaginaryI]"}], 
    RowBox[{"\[Pi]", " ", 
     SuperscriptBox["x", "2"]}]], "+", 
   FractionBox["\[ImaginaryI]", "\[Pi]"], "+", 
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"3", " ", "\[ImaginaryI]"}], "-", 
       RowBox[{"4", " ", "\[ImaginaryI]", " ", "EulerGamma"}], "+", 
       RowBox[{"2", " ", "\[Pi]"}], "+", 
       RowBox[{"4", " ", "\[ImaginaryI]", " ", 
        RowBox[{"Log", "[", "2", "]"}]}], "-", 
       RowBox[{"4", " ", "\[ImaginaryI]", " ", 
        RowBox[{"Log", "[", "x", "]"}]}]}], ")"}], " ", 
     SuperscriptBox["x", "2"]}], 
    RowBox[{"16", " ", "\[Pi]"}]], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "3"],
    SeriesData[$CellContext`x, 0, {}, -2, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
   Complex[0, 4]/Pi, 0, Complex[0, 1]/Pi, 0, Rational[1, 16] 
    Pi^(-1) (Complex[0, 3] + Complex[0, -4] EulerGamma + 2 Pi + 
     Complex[0, 4] Log[2] + Complex[0, -4] Log[$CellContext`x])}, -2, 3, 1],
  Editable->False]], "Output",
 ImageSize->{453, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->578276534]
}, Open  ]],

Cell["Series at infinity:", "ExampleText",
 CellID->298449549],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"HankelH2", "[", 
    RowBox[{"2", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "\[Infinity]", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->824663884],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    RowBox[{"-", "\[ImaginaryI]"}], " ", "x"}]], " ", 
  RowBox[{"(", 
   InterpretationBox[
    RowBox[{
     RowBox[{"-", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], 
        RowBox[{"1", "/", "4"}]], " ", 
       SqrtBox[
        FractionBox["2", "\[Pi]"]], " ", 
       SqrtBox[
        FractionBox["1", "x"]]}]}], "+", 
     FractionBox[
      RowBox[{"15", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], 
        RowBox[{"3", "/", "4"}]], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         FractionBox["1", "x"], ")"}], 
        RowBox[{"3", "/", "2"}]]}], 
      RowBox[{"4", " ", 
       SqrtBox[
        RowBox[{"2", " ", "\[Pi]"}]]}]], "+", 
     InterpretationBox[
      SuperscriptBox[
       RowBox[{"O", "[", 
        FractionBox["1", "x"], "]"}], 
       RowBox[{"5", "/", "2"}]],
      SeriesData[$CellContext`x, 
       DirectedInfinity[1], {}, 1, 5, 2],
      Editable->False]}],
    SeriesData[$CellContext`x, 
     DirectedInfinity[
     1], {-(-1)^Rational[1, 4] (2/Pi)^Rational[1, 2], 0, 
      Rational[15, 4] (-1)^Rational[3, 4] (2 Pi)^Rational[-1, 2]}, 1, 5, 2],
    Editable->False], ")"}]}]], "Output",
 ImageSize->{348, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->176111851]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24061],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->841601362],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH2", "[", 
  RowBox[{
   RowBox[{"2", "+", "I"}], ",", 
   RowBox[{"-", "1.5"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->502585196],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "5.699596667719684`"}], "+", 
  RowBox[{"10.659222221692245`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{127, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->188342896]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->64247589],

Cell["Evaluate numerically to high precision:", "ExampleText",
 CellID->221025511],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"HankelH2", "[", 
    RowBox[{"1", ",", "3"}], "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->342375044],

Cell[BoxData[
 RowBox[{"0.339058958525936458925514597206478894095162259995`30.\
009209591469737", "-", 
  RowBox[{
  "0.324674424791799978437012839287953340364112708289`29.990382447580807", 
   " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{476, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17350437]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->382604780],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->27968742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH2", "[", 
  RowBox[{"1", ",", "3.000000000000000000000000000"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->926751074],

Cell[BoxData[
 RowBox[{"0.33905895852593645892551459720634610792`26.187718641069583", "-", 
  RowBox[{
  "0.32467442479179997843701283928791906465`25.84892529327772", " ", 
   "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{420, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->34460330]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->144915864],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->69029971],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH2", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", " ", "1."}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->189347627],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.44005058574493355`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.7812128213002887`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.11490348493190047`", "\[InvisibleSpace]", "+", 
    RowBox[{"1.6506826068162546`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.019563353982668404`", "\[InvisibleSpace]", "+", 
    RowBox[{"5.821517605964729`", " ", "\[ImaginaryI]"}]}]}], "}"}]], "Output",\

 ImageSize->{441, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->323375764]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->646195254],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand half-integer order ",
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 " into elementary functions:"
}], "ExampleText",
 CellID->44455969],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"HankelH2", "[", 
   RowBox[{
    RowBox[{"3", "/", "2"}], ",", "x"}], "]"}], "//", 
  "FunctionExpand"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->147946761],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "\[ImaginaryI]"}], " ", "x"}]], " ", 
   SqrtBox[
    FractionBox["2", "\[Pi]"]], " ", 
   RowBox[{"(", 
    RowBox[{"\[ImaginaryI]", "-", "x"}], ")"}]}], 
  SuperscriptBox["x", 
   RowBox[{"3", "/", "2"}]]]], "Output",
 ImageSize->{111, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->306474586]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->446428633],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->14207313],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"HankelH2", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24369327],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubsuperscriptBox[
    TagBox["H",
     HankelH2], "n", 
    RowBox[{"(", "2", ")"}]], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{39, 19},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->1412440]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->436799651],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText",
 CellID->461756866],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH2", "[", 
  RowBox[{"2", ",", 
   RowBox[{
    RowBox[{"Log", "[", 
     RowBox[{"1", "+", "x"}], "]"}], "+", 
    RowBox[{
     RowBox[{"O", "[", "x", "]"}], "^", "2"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24445531],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox[
    RowBox[{"4", " ", "\[ImaginaryI]"}], 
    RowBox[{"\[Pi]", " ", 
     SuperscriptBox["x", "2"]}]], "+", 
   FractionBox["\[ImaginaryI]", "\[Pi]"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "2"],
    SeriesData[$CellContext`x, 0, {}, -2, 2, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {Complex[0, 4]/Pi, 0, Complex[0, 1]/Pi}, -2, 
   2, 1],
  Editable->False]], "Output",
 ImageSize->{103, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->773399893]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8034],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to convert to Bessel functions:"
}], "ExampleText",
 CellID->649204644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"HankelH2", "[", 
   RowBox[{"n", ",", "z"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->751371406],

Cell[BoxData[
 RowBox[{
  RowBox[{"BesselJ", "[", 
   RowBox[{"n", ",", "z"}], "]"}], "-", 
  RowBox[{"\[ImaginaryI]", " ", 
   RowBox[{"BesselY", "[", 
    RowBox[{"n", ",", "z"}], "]"}]}]}]], "Output",
 ImageSize->{204, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->753108766]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->282889322],

Cell[TextData[{
 "Integrate expressions with ",
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->228055373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"HankelH2", "[", 
     RowBox[{"2", ",", "x"}], "]"}], "^", "2"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->253099075],

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   SqrtBox["\[Pi]"]], 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"MeijerG", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", 
            FractionBox["3", "2"]}], ",", 
           FractionBox["1", "2"], ",", 
           FractionBox["5", "2"]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}], ",", "x", ",", 
       FractionBox["1", "2"]}], "]"}]}], "-", 
    RowBox[{"\[ImaginaryI]", " ", 
     RowBox[{"MeijerG", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "1"}], "}"}], ",", 
         RowBox[{"{", "0", "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           FractionBox["1", "2"], ",", 
           FractionBox["5", "2"]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", 
            FractionBox["3", "2"]}], ",", "0", ",", "0"}], "}"}]}], "}"}], 
       ",", "x", ",", 
       FractionBox["1", "2"]}], "]"}]}]}], ")"}]}]], "Output",
 ImageSize->{418, 64},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->313073554]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32133],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HankelH2",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HankelH2"]], "InlineFormula"],
 " does not automatically evaluate symbolically for half-integer arguments. "
}], "ExampleText",
 CellID->907156500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"HankelH2", "[", 
  RowBox[{
   RowBox[{"1", "/", "2"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->594280542],

Cell[BoxData[
 RowBox[{"HankelH2", "[", 
  RowBox[{
   FractionBox["1", "2"], ",", "x"}], "]"}]], "Output",
 ImageSize->{104, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->210749265]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to obtain expanded form:"
}], "ExampleText",
 CellID->377029976],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->122401508],

Cell[BoxData[
 FractionBox[
  RowBox[{"\[ImaginaryI]", " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "\[ImaginaryI]"}], " ", "x"}]], " ", 
   SqrtBox[
    FractionBox["2", "\[Pi]"]]}], 
  SqrtBox["x"]]], "Output",
 ImageSize->{77, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->344593048]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->12251],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["HankelH1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HankelH1"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselY",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselY"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalHankelH2",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalHankelH2"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->24351]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->20943],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->32737]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->580009607],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->6029162],

Cell[TextData[ButtonBox["Functions for Separable Coordinate Systems",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/FunctionsForSeparableCoordinateSystems"]], "MoreAbout",
 CellID->229730554],

Cell[TextData[ButtonBox["Functions Used in Optics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInOptics"]], "MoreAbout",
 CellID->234918320],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->215716352]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"HankelH2 - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 47, 22.6157910}", "context" -> "System`", 
    "keywords" -> {
     "Hankel function of the second kind", "Hankel functions", "Hankel H2"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "HankelH2[n, z] gives the Hankel function of the second kind \
SubsuperscriptBox[H, n, (2)](z). ", "synonyms" -> {"hankel h2"}, "title" -> 
    "HankelH2", "type" -> "Symbol", "uri" -> "ref/HankelH2"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7966, 254, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->6942811]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 35013, 1177}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2650, 61, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3273, 88, 1573, 37, 70, "ObjectNameGrid"],
Cell[4849, 127, 732, 24, 70, "Usage",
 CellID->32058]
}, Open  ]],
Cell[CellGroupData[{
Cell[5618, 156, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6096, 173, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6219, 179, 467, 19, 70, "Notes",
 CellID->8383],
Cell[6689, 200, 564, 21, 70, "Notes",
 CellID->6846],
Cell[7256, 223, 248, 8, 70, "Notes",
 CellID->209498233],
Cell[7507, 233, 218, 7, 70, "Notes",
 CellID->31226],
Cell[7728, 242, 201, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[7966, 254, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->6942811],
Cell[CellGroupData[{
Cell[8349, 268, 146, 5, 70, "ExampleSection",
 CellID->9771575],
Cell[8498, 275, 64, 1, 70, "ExampleText",
 CellID->100529053],
Cell[CellGroupData[{
Cell[8587, 280, 134, 4, 28, "Input",
 CellID->158701808],
Cell[8724, 286, 274, 7, 36, "Output",
 CellID->295029083]
}, Open  ]],
Cell[9013, 296, 125, 3, 70, "ExampleDelimiter",
 CellID->107801519],
Cell[9141, 301, 296, 12, 70, "ExampleText",
 CellID->125334460],
Cell[CellGroupData[{
Cell[9462, 317, 436, 14, 28, "Input",
 CellID->125209509],
Cell[9901, 333, 5544, 95, 163, "Output",
 Evaluatable->False,
 CellID->132483283]
}, Open  ]],
Cell[15460, 431, 125, 3, 70, "ExampleDelimiter",
 CellID->870569274],
Cell[15588, 436, 64, 1, 70, "ExampleText",
 CellID->656988425],
Cell[CellGroupData[{
Cell[15677, 441, 244, 8, 70, "Input",
 CellID->63700055],
Cell[15924, 451, 1273, 35, 56, "Output",
 CellID->578276534]
}, Open  ]],
Cell[17212, 489, 62, 1, 70, "ExampleText",
 CellID->298449549],
Cell[CellGroupData[{
Cell[17299, 494, 255, 8, 70, "Input",
 CellID->824663884],
Cell[17557, 504, 1435, 48, 67, "Output",
 CellID->176111851]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[19041, 558, 216, 7, 70, "ExampleSection",
 CellID->24061],
Cell[19260, 567, 85, 1, 70, "ExampleText",
 CellID->841601362],
Cell[CellGroupData[{
Cell[19370, 572, 172, 6, 70, "Input",
 CellID->502585196],
Cell[19545, 580, 269, 8, 36, "Output",
 CellID->188342896]
}, Open  ]],
Cell[19829, 591, 124, 3, 70, "ExampleDelimiter",
 CellID->64247589],
Cell[19956, 596, 82, 1, 70, "ExampleText",
 CellID->221025511],
Cell[CellGroupData[{
Cell[20063, 601, 182, 6, 70, "Input",
 CellID->342375044],
Cell[20248, 609, 360, 10, 36, "Output",
 CellID->17350437]
}, Open  ]],
Cell[20623, 622, 125, 3, 70, "ExampleDelimiter",
 CellID->382604780],
Cell[20751, 627, 112, 3, 70, "ExampleText",
 CellID->27968742],
Cell[CellGroupData[{
Cell[20888, 634, 155, 4, 70, "Input",
 CellID->926751074],
Cell[21046, 640, 337, 9, 36, "Output",
 CellID->34460330]
}, Open  ]],
Cell[21398, 652, 125, 3, 70, "ExampleDelimiter",
 CellID->144915864],
Cell[21526, 657, 209, 7, 70, "ExampleText",
 CellID->69029971],
Cell[CellGroupData[{
Cell[21760, 668, 195, 7, 70, "Input",
 CellID->189347627],
Cell[21958, 677, 573, 14, 36, "Output",
 CellID->323375764]
}, Open  ]],
Cell[22546, 694, 125, 3, 70, "ExampleDelimiter",
 CellID->646195254],
Cell[22674, 699, 378, 13, 70, "ExampleText",
 CellID->44455969],
Cell[CellGroupData[{
Cell[23077, 716, 193, 7, 70, "Input",
 CellID->147946761],
Cell[23273, 725, 458, 16, 77, "Output",
 CellID->306474586]
}, Open  ]],
Cell[23746, 744, 125, 3, 70, "ExampleDelimiter",
 CellID->446428633],
Cell[23874, 749, 202, 7, 70, "ExampleText",
 CellID->14207313],
Cell[CellGroupData[{
Cell[24101, 760, 165, 5, 70, "Input",
 CellID->24369327],
Cell[24269, 767, 315, 11, 54, "Output",
 CellID->1412440]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24633, 784, 243, 7, 70, "ExampleSection",
 CellID->436799651],
Cell[24879, 793, 211, 7, 70, "ExampleText",
 CellID->461756866],
Cell[CellGroupData[{
Cell[25115, 804, 264, 9, 70, "Input",
 CellID->24445531],
Cell[25382, 815, 623, 20, 53, "Output",
 CellID->773399893]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26054, 841, 232, 7, 70, "ExampleSection",
 CellID->8034],
Cell[26289, 850, 230, 8, 70, "ExampleText",
 CellID->649204644],
Cell[CellGroupData[{
Cell[26544, 862, 169, 5, 70, "Input",
 CellID->751371406],
Cell[26716, 869, 334, 11, 36, "Output",
 CellID->753108766]
}, Open  ]],
Cell[27065, 883, 125, 3, 70, "ExampleDelimiter",
 CellID->282889322],
Cell[27193, 888, 347, 12, 70, "ExampleText",
 CellID->228055373],
Cell[CellGroupData[{
Cell[27565, 904, 215, 7, 70, "Input",
 CellID->253099075],
Cell[27783, 913, 1510, 51, 85, "Output",
 CellID->313073554]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29342, 970, 226, 7, 70, "ExampleSection",
 CellID->32133],
Cell[29571, 979, 251, 7, 70, "ExampleText",
 CellID->907156500],
Cell[CellGroupData[{
Cell[29847, 990, 151, 5, 70, "Input",
 CellID->594280542],
Cell[30001, 997, 238, 8, 51, "Output",
 CellID->210749265]
}, Open  ]],
Cell[30254, 1008, 223, 8, 70, "ExampleText",
 CellID->377029976],
Cell[CellGroupData[{
Cell[30502, 1020, 110, 3, 70, "Input",
 CellID->122401508],
Cell[30615, 1025, 372, 13, 80, "Output",
 CellID->344593048]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[31048, 1045, 311, 9, 70, "SeeAlsoSection",
 CellID->12251],
Cell[31362, 1056, 955, 35, 70, "SeeAlso",
 CellID->24351]
}, Open  ]],
Cell[CellGroupData[{
Cell[32354, 1096, 314, 9, 70, "TutorialsSection",
 CellID->20943],
Cell[32671, 1107, 144, 3, 70, "Tutorials",
 CellID->32737]
}, Open  ]],
Cell[CellGroupData[{
Cell[32852, 1115, 319, 9, 70, "MoreAboutSection",
 CellID->580009607],
Cell[33174, 1126, 156, 3, 70, "MoreAbout",
 CellID->6029162],
Cell[33333, 1131, 195, 4, 70, "MoreAbout",
 CellID->229730554],
Cell[33531, 1137, 157, 3, 70, "MoreAbout",
 CellID->234918320],
Cell[33691, 1142, 174, 3, 70, "MoreAbout",
 CellID->215716352]
}, Open  ]],
Cell[33880, 1148, 27, 0, 70, "History"],
Cell[33910, 1150, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

