(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     58362,       2065]
NotebookOptionsPosition[     48212,       1706]
NotebookOutlinePosition[     49615,       1746]
CellTagsIndexPosition[     49529,       1741]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Advanced Matrix Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AdvancedMatrixOperations"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Advanced Matrix Operations\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RowReduce" :> Documentation`HelpLookup["paclet:ref/RowReduce"],
           "LatticeReduce" :> 
          Documentation`HelpLookup["paclet:ref/LatticeReduce"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"RowReduce\"\>", 
       2->"\<\"LatticeReduce\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete & Integer Data" :> 
          Documentation`HelpLookup["paclet:guide/DiscreteAndIntegerData"], 
          "Matrix Decompositions" :> 
          Documentation`HelpLookup["paclet:guide/MatrixDecompositions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Discrete & Integer Data\"\>", 
       2->"\<\"Matrix Decompositions\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       4->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>", 
       5->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["HermiteDecomposition", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["HermiteDecomposition",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/HermiteDecomposition"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Hermite normal form decomposition of an \
integer matrix ",
     Cell[BoxData[
      FormBox["m", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The result is given in the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"u", ",", "r"}], "}"}], TraditionalForm]], "InlineMath"],
 " where ",
 Cell[BoxData[
  FormBox["u", TraditionalForm]], "InlineMath"],
 " is a unimodular matrix, ",
 Cell[BoxData[
  FormBox["r", TraditionalForm]], "InlineMath"],
 " is an upper\[Hyphen]triangular matrix, and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"u", ".", "m"}], "==", "r"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->15568]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->253770186],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->420383711],

Cell[TextData[{
 "Decompose ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " into a unimodular matrix ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " and an upper triangular matrix ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->14505552],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "7", ",", "9"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->863419787],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"u", ",", "r"}], "}"}], "=", 
    RowBox[{"HermiteDecomposition", "[", "m", "]"}]}], ")"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13266326],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWD1vGkEQXe7DGIw/YkgURYnkOpbyJyzFVX4EsiLZRZTIcUFHSUlJSeGC
kpKSkpKSkpKSkvIyuzt7uzzWPs53kSwllnx3s+y8fTM7O7uz39oPt99/tB/u
btoX1/ftX7d3N78vvv68p6awIoR4pP/L94K+E/pS/49JQm/32ZDPcDgcJupP
XCm52+2y3FTyYDBg+bOSN5uNVhFHCkU9w8ViMZlMTGtMA0rtUIhUPjGjzedz
+XmuZFJarVbyM+L+DXhXDSvuV3dGtdyOlSzt1LJCj1NZ6Qitu9N6YuzqdDqW
Wb/f5y5hDix/64GCWS6XmjDbUDfjcmvEnorBljq3V6G9BvKZknu9HkFamdw9
nU4NfjE7quCZWmpHMJvNToHPO48drqz7UwRQ8Fi+hM78/yZfFTscUSfAqyXt
SePCtmvZRnrqb/49L9+I3ocKYzQaMReXoV2JyAQ92wBZrwa7Uosy3Y06Cima
8f0Z51mPZ2Z+uOkUsPOwPwIe0t8BkXdXoEWWfpMz/zbZzkFvoB8yiqH/cWJz
lx0tYvQItK8z0L8wSsj6Kj/orORY9lJUzDbBf/RS0QPojzFX/mhBGvdljWLi
W+7pcol94Pch9EOc2hM4uMebVelnhx6MAdWfD1qg9RFkH/dA7OaEcnYhad0B
MKjo30P9ks/0rLPFEbIoyL4TA9q2XwbFnI27Ssx2SJbykCn5axuyZijLjqy5
KZe/fx6y1oDUMntD+dwbjG52oOdXRBbX1+Xvl9lQzqoucnZ4fjUXy6TmjPkp
8WedrBPGfpnzde2E/w56uSc9jA2NYquap1D8Z1ytbav+fbTpLL5er42f3Oq+
xW99ZqcKdzwey09fxRFznMd8PjHxK0JSIlX56VYTVGbxXQFmbH814W/V9pFB
WGVyF8TOX1/aSsVlTw5jXxRnb2tiyV7W2ngXgHUO1uKxlyneomA052dq40jX
TDbOEBt3C9xN9NqhslhPlXPvE1BUFPGpHCmguWkahuxbZIh3Smps4dwqbN1I
NcCXUUGGLpbev4nmaDTyMc3aiUOjrxck39w1jIdLY2yxmhnYeVZZbNhzDtLs
z5PtWt1kosvEnqTcnVKdHPjmU1T+ANRqYCg=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{163, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->704542135]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"u", ".", "m"}], "\[Equal]", "r"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->128631385],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->271031334]
}, Open  ]],

Cell[TextData[{
 "The determinant of ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " is a unit:"
}], "ExampleText",
 CellID->39512272],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", "u", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->525284373],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->516504952]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->51018452],

Cell[TextData[{
 "A singular matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->64860078],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"Partition", "[", 
    RowBox[{
     RowBox[{"Range", "[", "9", "]"}], ",", "3"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->115482812],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"HermiteDecomposition", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->205355614],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWDuPGjEQXvZ1HJDjmSghD52UMv8gkZIuqfIj0CnSXRElulxBR0lJSUm5
xRWUlJRXUlJSUlJSOjP2GJvBud3AotMpQWLt8WP8zXhm7PHXzs3lt++dm6uL
zvmX687Py6uLX+eff1xDU1DwPO8W/u+ee1AXUJP/WyGgtL8V/Aaj0UjIn/dB
0r1ej+impIfDIdHvJb1er9UUryy5yG8wn88nk4mswieCBdtUlqiM9Wqz2Qyr
DUnDpOVyidUQPoGnUGHZorKqUdG4krWqwRZJGuVUdFs2bmg5R0tsRsVaom63
azANBgPqD5xcSs7WhrP1TLJZLBYKKqEv6XWpNWa4GqSzkLU/YfSJpPv9PrDC
ak3SoODpdKp1ehj+KtPI6Qa/f3d3d8LwvGI017caD3sO5mLwAnfCf0y80lrI
hri+XzO67MRtbLym9U79eeFOkoQQ2riNR3LcbxhdcuI2Hnss3GBuYA3ZcWfz
zqLY9vEq45oPeh/A215p+GNM8WEUWoeOTL63O47jioQ7ktW2VguJW8hmf0rh
/pG4BDRfRj4VqSzd7cuVx1H/P/dcuftsPLel/FdDGynnugpGFex+IdQpVSb6
lI3jfHh/mp+40XENRoxrkfW3XRrfiQRFB3aXr+cXc3hMLKj+QBVaz3ztGPrq
pHeun2ySZYuiZ4x37V5ZECleO1EGJUfaLrlX5feHh5PIvTtpniGlDE3/cSXS
5xHe08NUv0nDbkv8IHth0fVHI4mbV9Mh32HxFvf5JZWYBbjibdo95KkwGdVj
OS//He753gf5uaW4mLzoT1zcN2E127wUZJkN9/bVaoVVba94T4jIO4KNx0Fu
PB6PhXBnJxGdo7vzYBJMxaqdc0CCRu8LMptIzTHcZ7Q6eUEgnqfSEH46tP6C
q8lnbNygKtKClprn1jy3u28Nk1HLyOPI1PWrjOsO4Vo9cmLnrzDVAzAba6qL
bWvjGLm2nzFaeRAk0mrDrBcjH2wDzx4dRdOx4U77sDNNjYn0moapRats9mPr
9arC9MbPw2x6ixkX9XoFAJMk2UdvgZ6vXJDe9ypamza2Pf0qZryajObvXdn8
KtK4Kd4o3Hjvt7N2ffq9tWi7Xa5FL6Ne4TdVwmp/\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{171, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->222915191]
}, Open  ]],

Cell[TextData[{
 "The number of nonzero rows in ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " is equal to the rank of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->113985872],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "m", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->87565153],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->357460901]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->248175087],

Cell[TextData[{
 "Rectangular matrices ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->81416039],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"Partition", "[", 
    RowBox[{
     RowBox[{"Range", "[", "12", "]"}], ",", "3"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->410616385],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"HermiteDecomposition", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->375971598],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWTtvGkEQPu44wNjm7ShxnMiSf0WalEmV/AdkRbKLKJHjgo6SkpKSktIl
khuXlJQuKa+kpNzM7M5yy3gvdxyHwChI3N7u3jy+mdnZ17f2/c2Pn+372+v2
5de79u+b2+s/l19+3UGTl3Mc5xH+n9468C7gTf4fhYDSfJ7g0xsOh0L+HBce
Pnz6HkrPUf1YHlN7QX7f7Xbp+6asDwYDqnuMrkVlRXYuFgslSvJTXB3v+fl5
PB5ragfFCP2Dd+fc2lq2tio8y1bSF4ROp1N8bcg6iAuCIFqinXfD2lrRFiGO
ZQNZaJc806tJ9iyy9tIaqMCanU4nRNXv9w0vZIFqNpspCISqrOVSa4Hp9TEV
GmWDXq8HjPG1JuvgsKenp6XtNkJTZfY5WqJxJ5NJiemDse9SzCfVHiIKwjjU
HmQRmm1qLyOM4o6juNjAF+GIrGnfUH9WaEajEeltognzil3OJplAIQsz0LaQ
QdBCTEUj84XKiBdiNVNi6TqGX+n7agQ9z7CIxwXhZavU76zOuX7+T71n1BiP
PsW8T3HhU1ZCfx/FcE8WJygFoy7PqHlmt488Psbso6UUo2k2Iw8xeYTGl+0e
0y4OY4MszOfq0z1D6TN93FRo+fywb77MBuUHVj8+SJR83VU+SJRZ5iDXarH9
wY5UmMkwfs3VwamwZ39XGGvWNWaBdNrxmY1/z7XTM51dGveFn0BauFJ6KYXH
TU55xFMFPosxEqJmaqTDrTVyzKXSPYqzXed1/RDFPRsvvybu6CnsfidWV016
HGy6ekqnFd+jnbO6Pb+8zuyN9QJ018kDnMv2sVaYxNqWUL7kyddWh4FVUuXD
/l3g1PMgnp/lY8djkt3NHvnLqNeFbUfDZ7zd47DzalrQRe90i4RRn7vpNU7a
XH1GZfWfUl/vycDuqcOzyDTU4W1DEurBYDCfz/FVx4W+i2iKVb+rCJ7NZg8P
D0LYT/h8mu1MekUHRECKr+a53Xg8ppsDuc+I3X3YZ1L7KFGrUIDLT47pE74m
bCWSVdRWoyYTDRiSbKNtwe8f+M5xfTzhybcc9ZPJxCYjakVg08m3IuI3LNXM
kYRxWRerccs15555w+pq/gmCQDnXuCNyIbpwLtB5L63GGCsu+LapNSUfxGna
ItlL363cYp0wG/NZaxMbFxhvdYcFao9GozQ29jS9GvB0w3iiLW9qnOkoLjAJ
TVbnp4nJRrGv0VDuU2hwv2CeBpyJ1Qx4ZbSb/VIm3fg6ub/p933D\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{186, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->105414976]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"Partition", "[", 
    RowBox[{
     RowBox[{"Range", "[", "12", "]"}], ",", "4"}], "]"}]}], ";"}], "\n", 
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"HermiteDecomposition", "[", "m", "]"}]}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->360524753],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWDuPGjEQXvYFt3BwgUsUpTopZcoUaVNESZUfgU6R7ooo0eUKOkpKSkrK
1VWUlJSUlJSUW1JSbsb2DPbO+ZYFlpPyQGLt8doz841nx+P52r2/+fa9e397
3b36ctf9eXN7/evq8487GPIqjuM8wP/dawf6KfTk/yFNoTWfDfH0xuNxKn/O
e0n3+32kO5IejUZIf5T0ZrNRS5y65CKf3nK5nE6nsguPAAS2sY2wDUnaYrEQ
3bakYVGSJKLrw8NzlFa2tkna4fzIkK51DCQt8CpaSgm2tFxDyPWskJD1ej2t
23A4xPeelUt09Oi5ZL5arRQAxBSRNjgaMm3Joj4bP2d0VdKDwQBYie6FpMH8
s9mMLH4KVE1mvbMtKnc+n1eZlpeM5nuj5oOfgItpFMAdUT0/Culv6IV8b14y
um5Fo7+WC9ojfH9aNHEco94mGv3FczSvGB1Z0eiI8LxowI3Bn4qjKRYLamk2
orQY11NicgGSGQO0VBHXXJgl/Iuious8nse1DdL8qNrKSPWRq8+4fNgh5RNy
8XC9jMIqahqWPZQrj+nuf+6lcnfZfO5T5UsTPlIvVQr5uchZAuQu6DM2j/Ph
74lPE9sI2zrydbGt5mrLLRowKfYYwk9Ce5yoWRDZIoFdRpPJqFln8ZjoWmeR
DXicraj3nmrEs8p0q1m0CChu7G2B4hE6JI/YiZj7SQ2xCiQiwRYYFc5de82x
tk+MsegoR8yzyfzdLfo98nyoTLQygviat/7qeSbM/SpMj401eTlfmRjNXSq+
l3zPgr8cb358Ohy1jyeKiz5lO1GeyriaqT7nzIyN3lNE+RMyhH+He7kZMD+T
FRd9p3yKi/0OoFbrek2R1XCPWa/XZCczM7pMtR/KGsRkMhFd2x3OrOPQesqQ
VKyFxcBCdM27GFxysdpjq2E8/mLtNzSVPQAwXgHAKcV4590k9doGWQ1pEw0Y
Em10PBpdvxBobHURXjloMVrsW7i1P9d4V11sf421n71Is37IefPoXCznVDiS
JFEba9T3XPChYywuT3vYuQ7pjZbnevOsQcqm0yaTZ6iZoGimFsl9h2dU++ps
8lJZIygex7FNd7sfM9nGudogfupDx1pug/agNASaV+cg3rvP/wwajHUKTTvN
ntwU8d6ycbPCLMbfyNWqRu5UfgNawXol\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{169, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->195848082]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2751910],

Cell[TextData[{
 "A rational matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->143033864],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", 
       RowBox[{"2", "/", "3"}], ",", 
       RowBox[{"3", "/", "4"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "7", ",", "9"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24622245],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"u", ",", "r"}], "}"}], "=", 
    RowBox[{"HermiteDecomposition", "[", "m", "]"}]}], ")"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->547156626],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"3", "3", 
        RowBox[{"-", "2"}]},
       {"8", "4", 
        RowBox[{"-", "3"}]},
       {
        RowBox[{"-", "18"}], 
        RowBox[{"-", "11"}], "8"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        FractionBox["1", "2"], "0", 
        RowBox[{"-", 
         FractionBox["27", "4"]}]},
       {"0", 
        FractionBox["1", "3"], 
        RowBox[{"-", "9"}]},
       {"0", "0", 
        FractionBox["51", "2"]}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Output",
 ImageSize->{204, 69},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->60522437]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " is a unimodular integer matrix, ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " is an upper triangular rational matrix:"
}], "ExampleText",
 CellID->554897715],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", "u", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->175712652],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->546209282]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"u", ".", "m"}], "\[Equal]", "r"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->20992379],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->584362656]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->59384568],

Cell[TextData[{
 "A Gaussian integer matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->136098556],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "+", "I"}], ",", 
       RowBox[{"2", "+", 
        RowBox[{"3", "I"}]}], ",", 
       RowBox[{"3", "+", 
        RowBox[{"4", "I"}]}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "7", ",", "9"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->215212562],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"u", ",", "r"}], "}"}], "=", 
    RowBox[{"HermiteDecomposition", "[", "m", "]"}]}], ")"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->547268617],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWjtvHDcQ3ru9vZdlvU6JDceF4ldj52/ElfsUaQQjgF0ECRwX6lQKKQKV
KlVeqfJKlSpVqrzySpUqaT5mjtxPwyV1twaMwAZ0S3KHM8OPM8PhrN8dfP7w
x58Hnz++P9h/++ng7w8f3/+z/+tfn/RQ2SmKzqOiKF4/LnRb6Zb9M/8e6z/X
xNaG+S3Pzs4snSr+tf2joyPqT2z/9PSU+r/Z/u3trZtSPLBc7G95fX09m81M
03AttfA9eu7oZ6WfPUvHWqmi0j8D3R0T3VB4b+Z16cn0fdb66urKNHdtXwtf
LBam2SN+rMcDml9G5PRpnOf1a3I0GsR3HKzWY2L71ZKbpSmKH0HGNvQRi0HQ
L0jnu1wHDbNyuA55/w4PDz1yJycnRF+KUsfi6AbI6mZpXIlU6ble+/l87sCn
/RjzmmjUYFAJkkeR8TIyXkG/n7VDVYRbamdGtn98fKwXopb2og384uJiOb/V
nWFNreSep3Z+EkPC6+qtZrTch+7l5eUOzDY21qMosD6+63qA013HCh2uPM56
LYR7+zhLdtgW3jYOUXTaNPhT1L2rxSro94RYuRre/mxweGs7D+eviHcsng8I
iVJYg9dpOp0SciGe/uzDiL2ThWtf3dWxYmsOxicJHGveGZy4beIn26vVrGaf
HDmdLjogaS+P47YR55pET9YIz9LdBHbWA4ITegs0bB+zLXEU/dro1dXwheeW
1+qhctb8g/JZk7FiPs/Qyjm72Sa6ip5joBuBHESjUvVsaSPgx3qUQR/1GNZW
1Qu0DqX8l9Did+JSBtLvIrgqV4x23e/cW+XeBXq07faldZcxtS0pxu7D2wjb
Pd8G9lU8TwxvJwPgi3LRH9mv+JYyUfHbUKjPZuNqcUcqkLoZvOfzPjeu4Rn/
LEm11CDJC9HheIAWJcdwjNa7WZrJvFCzbcIbT/iOe1+6h/lFC+AIjnndFnDi
XGEk7p2f90TJ9+k+jaOGcv4Tjw152I5V833qfqia8Y7uGzQdove1YESzKeM1
9C/gfepmkbLU1VDE3Aa9uQ3bTEU+RNJkIsbndumJ9vMS+DXZG75vD7lYnDSz
payfMVr9dKgyrIyt/1UmXXtYhH3UckDS1sPgqaiLfEa8uQft5KtggPe3n9Za
OyJqzsswS8CY/SzxfphYc5iVGx80t4Ie7aBk9Vtrre7/lIt+S9zbvQ1hDuS4
+LpQjIt833Szfe0/Z7a+U9/c3JhmKlv+WTXX9Ou1/Pl8fn5+rpRc3+D6SUmx
Qsp9pGwz1Me910K0KNMMaxez2Yy+JHyrWbFDUW8GVi1pCp7Aci1jD6hSGObl
Hi6y+WpLiKw2FtrXWA7cXEXPzYRjOWgqEx4yjkHV3dSwkZ+MJ2fA6PtSDpef
Ect44lenh1k44pmI+GGWgpqncmA8xVyW42MK8suzy6baNOOZ6zOLxcK5SfD1
rqu9HT2dq1+xnPe5ile/mnJdW4vWPjBhZMjSvh4yUu67tPPal0yXQXmbYnvO
syX2secwjjrXv695aa6Kq+GYTqerI4IIYFbIme+Apbmjhb5cO2oHimk+EWXK
8f4XkLQtUu2AhAn0MQbnrbqhCh1QLRGmM9ut+SlQ5WSyYeYpvTcew/XrPVWv
W8cyVrsm23H/F6HofAH91QfZ\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{279, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->204719926]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " is a unimodular Gaussian integer matrix, ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " is an upper triangular Gaussian integer matrix:"
}], "ExampleText",
 CellID->213399759],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", "u", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->108949387],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->975809843]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"u", ".", "m"}], "\[Equal]", "r"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->41888003],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->72242049]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12388067],

Cell[TextData[{
 "A Gaussian rational matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->22971386],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"1", "/", "2"}], "+", 
        RowBox[{"I", "/", "3"}]}], ",", 
       RowBox[{
        RowBox[{"2", "/", "5"}], "+", 
        RowBox[{
         RowBox[{"3", "/", "7"}], "I"}]}], ",", 
       RowBox[{
        RowBox[{"3", "/", "4"}], "+", 
        RowBox[{
         RowBox[{"4", "/", "5"}], "I"}]}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "7", ",", "9"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->887160195],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"u", ",", "r"}], "}"}], "=", 
    RowBox[{"HermiteDecomposition", "[", "m", "]"}]}], ")"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->88409972],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        RowBox[{"315", "-", 
         RowBox[{"140", " ", "\[ImaginaryI]"}]}], 
        RowBox[{"20", "+", 
         RowBox[{"129", " ", "\[ImaginaryI]"}]}], 
        RowBox[{
         RowBox[{"-", "38"}], "-", 
         RowBox[{"85", " ", "\[ImaginaryI]"}]}]},
       {
        RowBox[{
         RowBox[{"-", "126"}], "+", 
         RowBox[{"300", " ", "\[ImaginaryI]"}]}], 
        RowBox[{
         RowBox[{"-", "97"}], "-", 
         RowBox[{"76", " ", "\[ImaginaryI]"}]}], 
        RowBox[{"81", "+", 
         RowBox[{"34", " ", "\[ImaginaryI]"}]}]},
       {
        RowBox[{"630", " ", "\[ImaginaryI]"}], 
        RowBox[{
         RowBox[{"-", "230"}], "-", 
         RowBox[{"63", " ", "\[ImaginaryI]"}]}], "170"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        FractionBox["1", "6"], "0", 
        RowBox[{
         FractionBox["265", "4"], "-", 
         RowBox[{"231", " ", "\[ImaginaryI]"}]}]},
       {"0", 
        FractionBox["1", "35"], 
        RowBox[{
         FractionBox["207", "2"], "+", 
         FractionBox[
          RowBox[{"1011", " ", "\[ImaginaryI]"}], "5"]}]},
       {"0", "0", 
        RowBox[{"336", "+", 
         FractionBox[
          RowBox[{"567", " ", "\[ImaginaryI]"}], "2"]}]}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Output",
 ImageSize->{392, 71},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->5214314]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " is a unimodular Gaussian integer matrix, ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " is an upper triangular Gaussian rational matrix:"
}], "ExampleText",
 CellID->714262366],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", "u", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->43588313],

Cell[BoxData["\[ImaginaryI]"], "Output",
 ImageSize->{12, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->515882169]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"u", ".", "m"}], "\[Equal]", "r"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->98559397],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->882925957]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->39929908],

Cell[TextData[{
 "Solve a linear Diophantine equation ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"A", ".", 
    StyleBox["x", "TI"]}], "==", 
   StyleBox["b", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->22068174],

Cell[BoxData[
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}]}], ";", 
  RowBox[{"b", "=", "4"}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->290085031],

Cell["Form a homogenized system:", "ExampleText",
 CellID->43244501],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"m", "=", 
    RowBox[{"Transpose", "[", 
     RowBox[{"Prepend", "[", 
      RowBox[{
       RowBox[{"IdentityMatrix", "[", "4", "]"}], ",", 
       RowBox[{"Prepend", "[", 
        RowBox[{"A", ",", 
         RowBox[{"-", "b"}]}], "]"}]}], "]"}], "]"}]}], ")"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->718131251],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"-", "4"}], "1", "0", "0", "0"},
     {"1", "0", "1", "0", "0"},
     {"2", "0", "0", "1", "0"},
     {"3", "0", "0", "0", "1"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{99, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->27455389]
}, Open  ]],

Cell[TextData[{
 "Rows of ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " are integer combinations of rows of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->52553652],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"r", "=", 
    RowBox[{
     RowBox[{"HermiteDecomposition", "[", "m", "]"}], "[", 
     RowBox[{"[", "2", "]"}], "]"}]}], ")"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->178433943],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", "0", "0", 
      RowBox[{"-", "1"}], "1"},
     {"0", "1", "0", 
      RowBox[{"-", "1"}], "2"},
     {"0", "0", "1", "1", 
      RowBox[{"-", "1"}]},
     {"0", "0", "0", "3", 
      RowBox[{"-", "2"}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{107, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->206795928]
}, Open  ]],

Cell[TextData[{
 "The second row of ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " gives a solution of ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"A", ".", 
    StyleBox["x", "TI"]}], "==", 
   StyleBox["b", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->40053137],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"Drop", "[", 
   RowBox[{
    RowBox[{"r", "[", 
     RowBox[{"[", "2", "]"}], "]"}], ",", "2"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->106133069],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", 
   RowBox[{"-", "1"}], ",", "2"}], "}"}]], "Output",
 ImageSize->{70, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->266423191]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"A", ".", "sol"}], "\[Equal]", "b"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->287208247],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->44233833]
}, Open  ]],

Cell[TextData[{
 "The last two rows of ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " give a basis of solutions to the homogenous equation ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"A", ".", 
    StyleBox["x", "TI"]}], "==", "0"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->400431885],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"h1", ",", "h2"}], "}"}], "=", 
  RowBox[{
   RowBox[{
    RowBox[{"Drop", "[", 
     RowBox[{"#", ",", "2"}], "]"}], "&"}], "/@", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"r", "[", 
      RowBox[{"[", "3", "]"}], "]"}], ",", 
     RowBox[{"r", "[", 
      RowBox[{"[", "4", "]"}], "]"}]}], "}"}]}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->67943350],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "3", ",", 
     RowBox[{"-", "2"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{160, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->918363595]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"A", ".", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"h1", ",", "h2"}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->848296020],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->143930641]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Reduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reduce"]], "InlineFormula"],
 " uses ",
 Cell[BoxData[
  ButtonBox["HermiteDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HermiteDecomposition"]], "InlineFormula"],
 " to solve linear Diophantine equations:"
}], "ExampleText",
 CellID->2961031],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"A", ".", 
     RowBox[{"{", 
      RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "\[Equal]", "b"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", "Integers"}], 
  "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->1546448],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"C", "[", "1", "]"}], "|", 
     RowBox[{"C", "[", "2", "]"}]}], ")"}], "\[Element]", "Integers"}], "&&", 
  
  RowBox[{"x", "\[Equal]", 
   RowBox[{"C", "[", "1", "]"}]}], "&&", 
  RowBox[{"y", "\[Equal]", 
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    RowBox[{"C", "[", "1", "]"}], "+", 
    RowBox[{"3", " ", 
     RowBox[{"C", "[", "2", "]"}]}]}]}], "&&", 
  RowBox[{"z", "\[Equal]", 
   RowBox[{"2", "-", 
    RowBox[{"C", "[", "1", "]"}], "-", 
    RowBox[{"2", " ", 
     RowBox[{"C", "[", "2", "]"}]}]}]}]}]], "Output",
 ImageSize->{520, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->82851398]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28253308],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HermiteDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HermiteDecomposition"]], "InlineFormula"],
 " gives a unimodular matrix and an upper triangular matrix:"
}], "ExampleText",
 CellID->2253121],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"9", ",", "7", ",", "3", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "1", ",", "2"}], "}"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"u", ",", "r"}], "}"}], "=", 
    RowBox[{"HermiteDecomposition", "[", "m", "]"}]}], ")"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->94525042],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"4", "1", 
        RowBox[{"-", "3"}]},
       {"7", "1", 
        RowBox[{"-", "4"}]},
       {"17", "3", 
        RowBox[{"-", "11"}]}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0", "12", "12"},
       {"0", "1", "20", "22"},
       {"0", "0", "49", "52"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Output",
 ImageSize->{194, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->139380108]
}, Open  ]],

Cell[TextData[{
 "The matrices satisfy the equation ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["u", "TI"], ".", 
    StyleBox["m", "TI"]}], "==", 
   StyleBox["r", "TI"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->191766142],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"u", ".", "m"}], "\[Equal]", "r"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->106430844],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->441650186]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Det",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Det"]], "InlineFormula"],
 " of a unimodular matrix is a unit:"
}], "ExampleText",
 CellID->38029128],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", "u", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->349350027],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->194128857]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Inverse",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Inverse"]], "InlineFormula"],
 " of a unimodular integer matrix is an integer matrix:"
}], "ExampleText",
 CellID->33424756],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Inverse", "[", "u", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->274094359],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"9", ",", "7", ",", 
     RowBox[{"-", "5"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"4", ",", "5", ",", 
     RowBox[{"-", "3"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{236, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->737512734]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->135531421],

Cell[TextData[{
 "Elements of ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " form an upper triangular basis of the lattice generated by the rows of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->181806474],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"3", "^", "9"}], ",", 
       RowBox[{"3", "^", "10"}], ",", 
       RowBox[{"3", "^", "11"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"10", "^", "5"}], ",", 
       RowBox[{"10", "^", "6"}], ",", 
       RowBox[{"10", "^", "7"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"5", "^", "7"}], ",", 
       RowBox[{"5", "^", "8"}], ",", 
       RowBox[{"5", "^", "9"}]}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->709244953],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"u", ",", "r"}], "}"}], "=", 
    RowBox[{"HermiteDecomposition", "[", "m", "]"}]}], ")"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->278279817],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        RowBox[{"-", "1259653"}], 
        RowBox[{"-", "99175"}], "444304"},
       {"415625", "32723", 
        RowBox[{"-", "146599"}]},
       {"6250000", "492075", 
        RowBox[{"-", "2204496"}]}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "3", 
        RowBox[{"-", "347112499991"}]},
       {"0", "6250", "114530550000"},
       {"0", "0", "1722262500000"}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Output",
 ImageSize->{403, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->41704114]
}, Open  ]],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " provides an explicit representation of rows of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " in terms of the basis:"
}], "ExampleText",
 CellID->22751761],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Inverse", "[", "u", "]"}], ".", "r"}], "\[Equal]", "m"}]], "Input",\

 CellLabel->"In[3]:=",
 CellID->460609724],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->727160260]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LatticeReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LatticeReduce"]], "InlineFormula"],
 " gives a basis consisting of shorter vectors:"
}], "ExampleText",
 CellID->221225689],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeReduce", "[", "m", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->120794037],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"19683", ",", "59049", ",", "177147"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"243244", ",", "23482", ",", "39196"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"104856", ",", 
     RowBox[{"-", "235432"}], ",", "43704"}], "}"}]}], "}"}]], "Output",
 ImageSize->{520, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->124775419]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RowReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RowReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LatticeReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeReduce"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Advanced Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AdvancedMatrixOperations"]], "Tutorials",
 CellID->4897]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->226600209],

Cell[TextData[ButtonBox["Discrete & Integer Data",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteAndIntegerData"]], "MoreAbout",
 CellID->113358544],

Cell[TextData[ButtonBox["Matrix Decompositions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatrixDecompositions"]], "MoreAbout",
 CellID->363166485],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->37309052],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->343445759],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->141801811]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"HermiteDecomposition - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 47, 32.1160342}", "context" -> "System`", 
    "keywords" -> {
     "extended GCD", "linear Diophantine equations", "Hermite normal form"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "HermiteDecomposition[m] gives the Hermite normal form decomposition of \
an integer matrix m.", 
    "synonyms" -> {
     "Hermite decomposition", "HermiteForm", "HermiteNormalForm"}, "title" -> 
    "HermiteDecomposition", "type" -> "Symbol", "uri" -> 
    "ref/HermiteDecomposition"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6630, 194, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->253770186]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 49386, 1734}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2714, 63, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3337, 90, 1585, 37, 70, "ObjectNameGrid"],
Cell[4925, 129, 620, 19, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5582, 153, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6060, 170, 533, 19, 70, "Notes",
 CellID->15568]
}, Closed]],
Cell[CellGroupData[{
Cell[6630, 194, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->253770186],
Cell[CellGroupData[{
Cell[7015, 208, 148, 5, 70, "ExampleSection",
 CellID->420383711],
Cell[7166, 215, 312, 12, 70, "ExampleText",
 CellID->14505552],
Cell[7481, 229, 358, 12, 70, "Input",
 CellID->863419787],
Cell[CellGroupData[{
Cell[7864, 245, 248, 8, 28, "Input",
 CellID->13266326],
Cell[8115, 255, 1346, 26, 64, "Output",
 Evaluatable->False,
 CellID->704542135]
}, Open  ]],
Cell[CellGroupData[{
Cell[9498, 286, 122, 4, 28, "Input",
 CellID->128631385],
Cell[9623, 292, 161, 5, 36, "Output",
 CellID->271031334]
}, Open  ]],
Cell[9799, 300, 149, 6, 70, "ExampleText",
 CellID->39512272],
Cell[CellGroupData[{
Cell[9973, 310, 99, 3, 70, "Input",
 CellID->525284373],
Cell[10075, 315, 175, 6, 36, "Output",
 CellID->516504952]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10299, 327, 219, 7, 70, "ExampleSection",
 CellID->51018452],
Cell[10521, 336, 138, 6, 70, "ExampleText",
 CellID->64860078],
Cell[10662, 344, 207, 7, 70, "Input",
 CellID->115482812],
Cell[CellGroupData[{
Cell[10894, 355, 149, 4, 70, "Input",
 CellID->205355614],
Cell[11046, 361, 1350, 26, 70, "Output",
 Evaluatable->False,
 CellID->222915191]
}, Open  ]],
Cell[12411, 390, 238, 9, 70, "ExampleText",
 CellID->113985872],
Cell[CellGroupData[{
Cell[12674, 403, 105, 3, 70, "Input",
 CellID->87565153],
Cell[12782, 408, 158, 5, 36, "Output",
 CellID->357460901]
}, Open  ]],
Cell[12955, 416, 125, 3, 70, "ExampleDelimiter",
 CellID->248175087],
Cell[13083, 421, 141, 6, 70, "ExampleText",
 CellID->81416039],
Cell[13227, 429, 208, 7, 70, "Input",
 CellID->410616385],
Cell[CellGroupData[{
Cell[13460, 440, 149, 4, 70, "Input",
 CellID->375971598],
Cell[13612, 446, 1484, 28, 70, "Output",
 Evaluatable->False,
 CellID->105414976]
}, Open  ]],
Cell[CellGroupData[{
Cell[15133, 479, 300, 9, 70, "Input",
 CellID->360524753],
Cell[15436, 490, 1403, 27, 70, "Output",
 Evaluatable->False,
 CellID->195848082]
}, Open  ]],
Cell[16854, 520, 123, 3, 70, "ExampleDelimiter",
 CellID->2751910],
Cell[16980, 525, 139, 6, 70, "ExampleText",
 CellID->143033864],
Cell[17122, 533, 441, 15, 70, "Input",
 CellID->24622245],
Cell[CellGroupData[{
Cell[17588, 552, 249, 8, 70, "Input",
 CellID->547156626],
Cell[17840, 562, 1706, 53, 90, "Output",
 CellID->60522437]
}, Open  ]],
Cell[19561, 618, 250, 8, 70, "ExampleText",
 CellID->554897715],
Cell[CellGroupData[{
Cell[19836, 630, 99, 3, 70, "Input",
 CellID->175712652],
Cell[19938, 635, 175, 6, 36, "Output",
 CellID->546209282]
}, Open  ]],
Cell[CellGroupData[{
Cell[20150, 646, 121, 4, 70, "Input",
 CellID->20992379],
Cell[20274, 652, 161, 5, 36, "Output",
 CellID->584362656]
}, Open  ]],
Cell[20450, 660, 124, 3, 70, "ExampleDelimiter",
 CellID->59384568],
Cell[20577, 665, 147, 6, 70, "ExampleText",
 CellID->136098556],
Cell[20727, 673, 490, 17, 70, "Input",
 CellID->215212562],
Cell[CellGroupData[{
Cell[21242, 694, 249, 8, 70, "Input",
 CellID->547268617],
Cell[21494, 704, 1883, 35, 70, "Output",
 Evaluatable->False,
 CellID->204719926]
}, Open  ]],
Cell[23392, 742, 267, 8, 70, "ExampleText",
 CellID->213399759],
Cell[CellGroupData[{
Cell[23684, 754, 99, 3, 70, "Input",
 CellID->108949387],
Cell[23786, 759, 175, 6, 36, "Output",
 CellID->975809843]
}, Open  ]],
Cell[CellGroupData[{
Cell[23998, 770, 121, 4, 70, "Input",
 CellID->41888003],
Cell[24122, 776, 160, 5, 36, "Output",
 CellID->72242049]
}, Open  ]],
Cell[24297, 784, 124, 3, 70, "ExampleDelimiter",
 CellID->12388067],
Cell[24424, 789, 147, 6, 70, "ExampleText",
 CellID->22971386],
Cell[24574, 797, 666, 23, 70, "Input",
 CellID->887160195],
Cell[CellGroupData[{
Cell[25265, 824, 248, 8, 70, "Input",
 CellID->88409972],
Cell[25516, 834, 2505, 74, 92, "Output",
 CellID->5214314]
}, Open  ]],
Cell[28036, 911, 268, 8, 70, "ExampleText",
 CellID->714262366],
Cell[CellGroupData[{
Cell[28329, 923, 98, 3, 70, "Input",
 CellID->43588313],
Cell[28430, 928, 170, 5, 36, "Output",
 CellID->515882169]
}, Open  ]],
Cell[CellGroupData[{
Cell[28637, 938, 121, 4, 70, "Input",
 CellID->98559397],
Cell[28761, 944, 161, 5, 36, "Output",
 CellID->882925957]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[28971, 955, 226, 7, 70, "ExampleSection",
 CellID->39929908],
Cell[29200, 964, 226, 9, 70, "ExampleText",
 CellID->22068174],
Cell[29429, 975, 202, 7, 70, "Input",
 CellID->290085031],
Cell[29634, 984, 68, 1, 70, "ExampleText",
 CellID->43244501],
Cell[CellGroupData[{
Cell[29727, 989, 393, 13, 70, "Input",
 CellID->718131251],
Cell[30123, 1004, 840, 25, 92, "Output",
 CellID->27455389]
}, Open  ]],
Cell[30978, 1032, 227, 9, 70, "ExampleText",
 CellID->52553652],
Cell[CellGroupData[{
Cell[31230, 1045, 255, 8, 70, "Input",
 CellID->178433943],
Cell[31488, 1055, 908, 28, 92, "Output",
 CellID->206795928]
}, Open  ]],
Cell[32411, 1086, 291, 12, 70, "ExampleText",
 CellID->40053137],
Cell[CellGroupData[{
Cell[32727, 1102, 205, 7, 70, "Input",
 CellID->106133069],
Cell[32935, 1111, 232, 8, 36, "Output",
 CellID->266423191]
}, Open  ]],
Cell[CellGroupData[{
Cell[33204, 1124, 124, 4, 70, "Input",
 CellID->287208247],
Cell[33331, 1130, 160, 5, 36, "Output",
 CellID->44233833]
}, Open  ]],
Cell[33506, 1138, 308, 11, 70, "ExampleText",
 CellID->400431885],
Cell[CellGroupData[{
Cell[33839, 1153, 407, 15, 70, "Input",
 CellID->67943350],
Cell[34249, 1170, 364, 13, 36, "Output",
 CellID->918363595]
}, Open  ]],
Cell[CellGroupData[{
Cell[34650, 1188, 179, 6, 70, "Input",
 CellID->848296020],
Cell[34832, 1196, 203, 7, 36, "Output",
 CellID->143930641]
}, Open  ]],
Cell[35050, 1206, 364, 12, 70, "ExampleText",
 CellID->2961031],
Cell[CellGroupData[{
Cell[35439, 1222, 321, 11, 70, "Input",
 CellID->1546448],
Cell[35763, 1235, 735, 25, 36, "Output",
 CellID->82851398]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36547, 1266, 236, 7, 70, "ExampleSection",
 CellID->28253308],
Cell[36786, 1275, 257, 7, 70, "ExampleText",
 CellID->2253121],
Cell[CellGroupData[{
Cell[37068, 1286, 582, 19, 70, "Input",
 CellID->94525042],
Cell[37653, 1307, 1545, 46, 64, "Output",
 CellID->139380108]
}, Open  ]],
Cell[39213, 1356, 246, 10, 70, "ExampleText",
 CellID->191766142],
Cell[CellGroupData[{
Cell[39484, 1370, 122, 4, 70, "Input",
 CellID->106430844],
Cell[39609, 1376, 161, 5, 36, "Output",
 CellID->441650186]
}, Open  ]],
Cell[39785, 1384, 200, 7, 70, "ExampleText",
 CellID->38029128],
Cell[CellGroupData[{
Cell[40010, 1395, 99, 3, 70, "Input",
 CellID->349350027],
Cell[40112, 1400, 158, 5, 36, "Output",
 CellID->194128857]
}, Open  ]],
Cell[40285, 1408, 227, 7, 70, "ExampleText",
 CellID->33424756],
Cell[CellGroupData[{
Cell[40537, 1419, 103, 3, 70, "Input",
 CellID->274094359],
Cell[40643, 1424, 454, 16, 36, "Output",
 CellID->737512734]
}, Open  ]],
Cell[41112, 1443, 125, 3, 70, "ExampleDelimiter",
 CellID->135531421],
Cell[41240, 1448, 267, 9, 70, "ExampleText",
 CellID->181806474],
Cell[41510, 1459, 615, 21, 70, "Input",
 CellID->709244953],
Cell[CellGroupData[{
Cell[42150, 1484, 249, 8, 70, "Input",
 CellID->278279817],
Cell[42402, 1494, 1652, 48, 64, "Output",
 CellID->41704114]
}, Open  ]],
Cell[44069, 1545, 267, 9, 70, "ExampleText",
 CellID->22751761],
Cell[CellGroupData[{
Cell[44361, 1558, 159, 6, 70, "Input",
 CellID->460609724],
Cell[44523, 1566, 161, 5, 36, "Output",
 CellID->727160260]
}, Open  ]],
Cell[44699, 1574, 232, 7, 70, "ExampleText",
 CellID->221225689],
Cell[CellGroupData[{
Cell[44956, 1585, 109, 3, 70, "Input",
 CellID->120794037],
Cell[45068, 1590, 452, 14, 36, "Output",
 CellID->124775419]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[45581, 1611, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[45900, 1622, 467, 17, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[46404, 1644, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[46725, 1655, 160, 3, 70, "Tutorials",
 CellID->4897]
}, Open  ]],
Cell[CellGroupData[{
Cell[46922, 1663, 319, 9, 70, "MoreAboutSection",
 CellID->226600209],
Cell[47244, 1674, 157, 3, 70, "MoreAbout",
 CellID->113358544],
Cell[47404, 1679, 153, 3, 70, "MoreAbout",
 CellID->363166485],
Cell[47560, 1684, 178, 3, 70, "MoreAbout",
 CellID->37309052],
Cell[47741, 1689, 198, 4, 70, "MoreAbout",
 CellID->343445759],
Cell[47942, 1695, 198, 4, 70, "MoreAbout",
 CellID->141801811]
}, Open  ]],
Cell[48155, 1702, 27, 0, 70, "History"],
Cell[48185, 1704, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

