(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     54924,       1944]
NotebookOptionsPosition[     45638,       1613]
NotebookOutlinePosition[     46983,       1650]
CellTagsIndexPosition[     46897,       1645]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DiscreteDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Discrete Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BinomialDistribution"], 
          "HypergeometricPFQ" :> 
          Documentation`HelpLookup["paclet:ref/HypergeometricPFQ"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"BinomialDistribution\"\>", 
       2->"\<\"HypergeometricPFQ\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/DiscreteStatisticalDistributions"], 
          "Functions Used in Statistics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInStatistics"], 
          "Statistical Distributions" :> 
          Documentation`HelpLookup["paclet:guide/StatisticalDistributions"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Discrete Statistical Distributions\"\>", 
       2->"\<\"Functions Used in Statistics\"\>", 
       3->"\<\"Statistical Distributions\"\>", 
       4->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["HypergeometricDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["HypergeometricDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/HypergeometricDistribution"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        SubscriptBox[
         StyleBox["n", "TI"], 
         StyleBox["succ", "TI"]], ",", 
        SubscriptBox[
         StyleBox["n", "TI"], 
         StyleBox["tot", "TI"]]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a hypergeometric distribution."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->10467]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "A hypergeometric distribution gives the distribution of the number of \
successes in ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " draws from a population of size ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["n", "TI"], 
   StyleBox["tot", "TI"]]], "InlineFormula"],
 " containing ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["n", "TI"], 
   StyleBox["succ", "TI"]]], "InlineFormula"],
 " successes."
}], "Notes",
 CellID->656338067],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HypergeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HypergeometricDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["n", "TI"], 
   StyleBox["succ", "TI"]]], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["n", "TI"], 
   StyleBox["tot", "TI"]]], "InlineFormula"],
 " to be any integers such that ",
 Cell[BoxData[
  RowBox[{
   StyleBox["0", "TR"], "<", 
   StyleBox["n", "TI"], "\[LessEqual]", 
   SubscriptBox[
    StyleBox["n", "TI"], 
    StyleBox["tot", "TI"]]}]], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  RowBox[{
   StyleBox["0", "TR"], "\[LessEqual]", 
   SubscriptBox[
    StyleBox["n", "TI"], 
    StyleBox["succ", "TI"]], "\[LessEqual]", 
   SubscriptBox[
    StyleBox["n", "TI"], 
    StyleBox["tot", "TI"]]}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->6850],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HypergeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HypergeometricDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomInteger"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->11373,
  ButtonNote->"11373"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->392413691],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->33486935],

Cell["Mean and variance of a hypergeometric distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->11373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"n", ",", 
    SubscriptBox["n", "succ"], ",", 
    SubscriptBox["n", "tot"]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30750],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", 
   SubscriptBox["n", "succ"]}], 
  SubscriptBox["n", "tot"]]], "Output",
 ImageSize->{46, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->238714793]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"n", ",", 
    SubscriptBox["n", "succ"], ",", 
    SubscriptBox["n", "tot"]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->9341],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", 
   SubscriptBox["n", "succ"], " ", 
   RowBox[{"(", 
    RowBox[{"1", "-", 
     FractionBox[
      SubscriptBox["n", "succ"], 
      SubscriptBox["n", "tot"]]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "n"}], "+", 
     SubscriptBox["n", "tot"]}], ")"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     SubscriptBox["n", "tot"]}], ")"}], " ", 
   SubscriptBox["n", "tot"]}]]], "Output",
 ImageSize->{179, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->542841659]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->28548],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->10406],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"n", ",", 
     SubscriptBox["n", "succ"], ",", 
     SubscriptBox["n", "tot"]}], "]"}], ",", "k"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31918],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Binomial", "[", 
    RowBox[{
     SubscriptBox["n", "succ"], ",", "k"}], "]"}], " ", 
   RowBox[{"Binomial", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       SubscriptBox["n", "succ"]}], "+", 
      SubscriptBox["n", "tot"]}], ",", 
     RowBox[{
      RowBox[{"-", "k"}], "+", "n"}]}], "]"}]}], 
  RowBox[{"Binomial", "[", 
   RowBox[{
    SubscriptBox["n", "tot"], ",", "n"}], "]"}]]], "Output",
 ImageSize->{313, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->260487121]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"PDF", "[", 
      RowBox[{
       RowBox[{"HypergeometricDistribution", "[", 
        RowBox[{"30", ",", "50", ",", "100"}], "]"}], ",", "k"}], "]"}], ",", 
     
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "30"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "->", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->212],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW0lPFEEUbrpnJhJPHoABIeE3uNwkJuISNUaMAhc1mYwmcAAMcvXmvt2J
uETAH6CJHt0uokaueOAAHjigiUgIKsG2tu6qel1T080MPT1MdTLd86pefe97
71XVVE13n8iN9F0cyI3053PtR4Zzl/r685fbDw8NoyKnzrKsOfS5n7XQdxd9
I58510XX4HkXOjtIoZVdbXS10bWZyVaKqLEji4SUoOXJQS3c1mFa9aRyLz7j
YpccVgs+pbhMUUWNJiA7QG7EpzSV/XMApVGQfapCfRbUw/aZTWLfVDJCpZgX
s5sFsiojttdOYrKNXHaz+rZwTLkGa7czKj/0tXhUMj43sbQ1HMe0yNFi48vK
iO3UrRuK4rcU8Zby3iPExhbGZjO7plw+onX12Jvg2BZnlAZgf0pZSllOjH/Z
3zF65/Y7Niq57JRYf6VEvFD6BbzqPj2JtA4eGGOtuOyUpZ6yuHf3PZMnJ6Yl
luvr/5aX/5COQ+Slpd+oyGUj2V5dXUt75Yw3xVlZ+bu2ts5liIMEAcdykDJq
ItkR8LAdnw+tCo8D+IrRL+z3xuMcSt/agc7TKNl1ytzDWRL+zkStV8tTH+ZR
0auXMwGZ4vV2P0Pyoc4xji/lFMnfF1e09lmu/D7BphZfn+XK8XIF6yEes+fn
Oioe428r/Sscj3TE+EbNl63sBW1AbvRYDQ2+4FmSs2KjCFG2X2cWJesLC79g
Nr7N/9RGDzXh9TIetpPysjA7+yM8DlKWssD92V7E/3pllPa5wfG78SggoSxR
KIwD55sOjVeohyItr4cm3yvOV+eV3IOT7xXnq/aqM+FedPqs8SrrYNWwrc4Y
bw3WlvNg9CNbIamsVY79o4efGC8de7re4r/tSWHPeenYw9/4pLDnvNTsjyWE
7VGfHZ7xjieOVbJjZtgZdoadmUkMO8MuDDtSyna0fL24+ezEnfStm29plAqw
k3fS8bHjdjk7PJPQjMJ1XlysuF11zLpiYtNlrFfUOm1ODwcw8e6q4SL4/2fM
LKV7VkQnWKK5R5h8tlhMPsvkxtR7hiDZ/RTGEMa4cmM/48p3ZivLrJZnZG99
cKrGfDbWjXUz0mrL+kki0/3bjetvGFq5rI8/9Z7w0FmH+8JyWee4Outw/1cu
6xxXbb2nTNZ6Y0LH47S3yjgb9K2IbnpifOi6fyccza6vzCykPRx5Ala7q4Oy
YRP8r6PyLJIVk3jGkLfLd9w4xk01zzzmd8Sgl4b+5PFn6W5YeHQ6H9Bd0bWr
ry2dlZ4z8lNOUa3QXQp+Kl3VM+HuJCo6b6/24VxE1LMGJRYU3APObwlPDIrJ
cDwoxfYqtJl3wJXXBm0GV5qad07gOjMum1isav8C74RuWvYCb3ZCn0vvm8Xu
9JEWwjM1KWDBEerx6oYwCowhigL/W4UovF6HglYRgwPP+SoEovB6joI9uwC0
80ob+ZrW4nHiT9XrMPjT6zotkhGy7lNrqd/ipaVW3X89rLJt\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 142},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->124472560]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->10127],

Cell["\<\
Generate a set of pseudorandom numbers that are hypergeometrically \
distributed:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->18571],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomInteger", "[", 
  RowBox[{
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"5", ",", "5", ",", "10"}], "]"}], ",", "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13114],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "4", ",", "3", ",", "2", ",", "2", ",", "1", ",", "2", ",", "2", ",", "2", 
   ",", "2", ",", "2"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->957126505]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7750],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"n", ",", 
    SubscriptBox["n", "succ"], ",", 
    SubscriptBox["n", "tot"]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32749],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox[
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     SubscriptBox["n", "tot"]}]], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], " ", "n"}], "+", 
     SubscriptBox["n", "tot"]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], " ", 
      SubscriptBox["n", "succ"]}], "+", 
     SubscriptBox["n", "tot"]}], ")"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "2"}], "+", 
     SubscriptBox["n", "tot"]}], ")"}], " ", 
   SqrtBox[
    RowBox[{"n", " ", 
     SubscriptBox["n", "succ"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "n"}], "+", 
       SubscriptBox["n", "tot"]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", 
        SubscriptBox["n", "succ"]}], "+", 
       SubscriptBox["n", "tot"]}], ")"}]}]]}]]], "Output",
 ImageSize->{284, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26914300]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"n", ",", 
    SubscriptBox["n", "succ"], ",", 
    SubscriptBox["n", "tot"]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->9977],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      SubscriptBox["n", "tot"]}], ")"}], " ", 
    SubsuperscriptBox["n", "tot", "2"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       SubscriptBox["n", "tot"], " ", 
       RowBox[{"(", 
        RowBox[{"1", "+", 
         SubscriptBox["n", "tot"]}], ")"}]}], "-", 
      RowBox[{"6", " ", "n", " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "n"}], "+", 
         SubscriptBox["n", "tot"]}], ")"}]}], "+", 
      FractionBox[
       RowBox[{"3", " ", 
        SubscriptBox["n", "succ"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", 
           SubscriptBox["n", "succ"]}], "+", 
          SubscriptBox["n", "tot"]}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"-", 
            SuperscriptBox["n", "2"]}], " ", 
           SubscriptBox["n", "tot"]}], "+", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{
             RowBox[{"-", "2"}], "+", "n"}], ")"}], " ", 
           SubsuperscriptBox["n", "tot", "2"]}], "+", 
          RowBox[{"6", " ", "n", " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"-", "n"}], "+", 
             SubscriptBox["n", "tot"]}], ")"}]}]}], ")"}]}], 
       SubsuperscriptBox["n", "tot", "2"]]}], ")"}]}], ")"}], "/", 
  RowBox[{"(", 
   RowBox[{"n", " ", 
    SubscriptBox["n", "succ"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "3"}], "+", 
      SubscriptBox["n", "tot"]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "2"}], "+", 
      SubscriptBox["n", "tot"]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "n"}], "+", 
      SubscriptBox["n", "tot"]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       SubscriptBox["n", "succ"]}], "+", 
      SubscriptBox["n", "tot"]}], ")"}]}], ")"}]}]], "Output",
 ImageSize->{442, 98},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->475138028]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14025],

Cell["Second moment of a hypergeometric distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->1606],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"ExpectedValue", "[", 
    RowBox[{
     RowBox[{"k", "^", "2"}], ",", 
     RowBox[{"HypergeometricDistribution", "[", 
      RowBox[{"n", ",", 
       SubscriptBox["n", "succ"], ",", 
       SubscriptBox["n", "tot"]}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{
    RowBox[{"n", "+", 
     SubscriptBox["n", "succ"]}], "\[LessEqual]", 
    SubscriptBox["n", "tot"]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20465],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Gamma", "[", 
     RowBox[{"1", "+", 
      SubscriptBox["n", "succ"]}], "]"}], " ", 
    RowBox[{"Gamma", "[", 
     RowBox[{"1", "-", 
      SubscriptBox["n", "succ"], "+", 
      SubscriptBox["n", "tot"]}], "]"}], " ", 
    RowBox[{"HypergeometricPFQ", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"2", ",", 
        RowBox[{"1", "-", "n"}], ",", 
        RowBox[{"1", "-", 
         SubscriptBox["n", "succ"]}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", 
        RowBox[{"2", "-", "n", "-", 
         SubscriptBox["n", "succ"], "+", 
         SubscriptBox["n", "tot"]}]}], "}"}], ",", "1"}], "]"}]}], ")"}], "/",
   
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Binomial", "[", 
     RowBox[{
      SubscriptBox["n", "tot"], ",", "n"}], "]"}], " ", 
    RowBox[{"Gamma", "[", "n", "]"}], " ", 
    RowBox[{"Gamma", "[", 
     SubscriptBox["n", "succ"], "]"}], " ", 
    RowBox[{"Gamma", "[", 
     RowBox[{"2", "-", "n", "-", 
      SubscriptBox["n", "succ"], "+", 
      SubscriptBox["n", "tot"]}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{455, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->49679336]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox["0.9", TraditionalForm]], "InlineMath"],
 " quantile of a hypergeometric distribution:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->48521834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"10", ",", "25", ",", "100"}], "]"}], ",", ".9"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->4591],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->216492365]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11626],

Cell["\<\
The probability of 25 successes in 50 draws from 100 elements that include 40 \
successes:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->32514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"50", ",", "40", ",", "100"}], "]"}], ",", "25"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->10642],

Cell[BoxData[
 FractionBox["273015841246512", "13190020067434111"]], "Output",
 ImageSize->{138, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->293242353]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3834],

Cell[BoxData["0.020698667617692446`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->157144469]
}, Open  ]],

Cell["The probability of fewer than 25 successes:", "ExampleText",
 CellID->411747730],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CDF", "[", 
  RowBox[{
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"50", ",", "40", ",", "100"}], "]"}], ",", "24"}], "]"}]], "Input",\

 CellLabel->"In[3]:=",
 CellID->633017742],

Cell[BoxData[
 FractionBox["38272603773068849", "39570060202302333"]], "Output",
 ImageSize->{138, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->211152684]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->109966815],

Cell[BoxData["0.9672111585729153`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->750741742]
}, Open  ]],

Cell["The probability of more than 25 successes:", "ExampleText",
 CellID->213514598],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"HypergeometricDistribution", "[", 
     RowBox[{"50", ",", "40", ",", "100"}], "]"}], ",", "25"}], 
   "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->207586845],

Cell[BoxData[
 FractionBox["43491718681268", "3597278200209303"]], "Output",
 ImageSize->{131, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->118270948]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->134218711],

Cell[BoxData["0.012090173809392193`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->8079710]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->86062964],

Cell["\<\
Plot the cumulative distribution function of a hypergeometric distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"HypergeometricDistribution", "[", 
      RowBox[{"20", ",", "35", ",", "75"}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26124],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwPATiyRIMQPZ/IAuM
H/7/D6SRSQcgyQyU0oPSTECaCUjrQPkMLGBlUKAN5LAgqWIGSzqDSJDi/2AA
1svAguBD1COr0ELjM6PxNUEEK4QPJ8GmgDzGCKJpZLMWVhNobas2Eh+7Tx2h
Ia5PnF0IFVB9ekTZgGyOHnE2sSLbxABNOQxsyPpwhaMT1G26/xGpCZb2mKBJ
jwma6pDlIakOORULoLnsHDGi8+edASIGdhDB3NlxAIXf23145oyTCP7kSUcn
TzoGYnKA+bdvvYGauhRsKg+QvAL1HabN7CTyOSjUP9jso9R8ZqyhCs4LzD9/
/gEK2dvOZOAAEcCYOgaVV0VTb0CAj90WotIShigonZ4nQacVmJ+ctAboB5A3
2MF+AfKh8u0D6DZqipJqtwuY//Dh+/PnngIRAzuIYL565QVUvhaPrqVLzufn
bQYiBnYQwdzedgCYWBC6oBEOBsglMpgPq/owXIhaJrKi6SNGDxuaGvTaAJse
9BwBKk1BXGLVEjIf7G5MEUxdSGHGBo1PVqwqh3OqJEXUD8xfveoSNC1ygNMi
MEVD5Yvw6Pr8+SeOdF9ME7dj5AiM2h17XFOevkGpjYXI/IOsBuYmXHmB9T++
NEqSWqSwwVo/DepUOJB2R/xHTcsc4LQM5EPlM/DoArazUPINot2FXRelPsJV
J4CYxKYjostSmunClRtgcsTmBmQ1hOsD9v+IdvvgTpEDaXfSf2i6hrb0IK3W
w4fuQ+Xj6OjioZbakUtofGmchu4ZNO2fXDAf2CJAaSG8ffsVKh+CVVcOmA8s
e0tLtiFK1Y0bruLVNZjzKrJoOZiP3lt4/vwTVN4LxcUVYNWNDXtQegmI/qMX
ye5BHS1Db3fjTivoKsGJDG9qbATzq6t24ugvOpBpbjeYDww26GgIZLTj9KnH
0BBlQ2uH6pBgD0gBuvmQ0ZX62l25OZsQcZCcuAaldHR2nI3iTzeXuSj8wIDF
KHygIcguISMcyI0XUksJzBFZBkYAeR2qWQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 147},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->307328238]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->332082353],

Cell["\<\
The density functions of hypergeometric random variables are concentrated \
about their means:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->12249],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"PDF", "[", 
     RowBox[{
      RowBox[{"HypergeometricDistribution", "[", 
       RowBox[{"n", ",", "25", ",", "50"}], "]"}], ",", "x"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "25"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "0", ",", "50"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14426],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmjtoHFcUhlfxuQEHHIjBxNiQCDcBY3BjSBVIwNgY7CaQJpDAIoylgFGQ
DQkEocaQKpAmYBBsYXAj1tUiwkYser9fncQWLiQtW+gFeqD3I+c73FGuhBst
irGlu7Cr1ezc/xvmP/ecM3fmQfpp7cPH6ad1Nenquw3pn2vrap5U36lv0E3n
qlKp1JS+f7mc0u/7+s3ebNLXOb7bK/VaP6r+h61/64fs7e3J7u6ubG5uuqWl
JZmampLOzk5pamoS3dWdgpEtfqRj5NbWlqyursrs7KwUi0XX0dEhmUxGGhsb
UTl/ihVeoMDo7e1tWV9ftzNYLpdlcnJSurq6JJvNuubmZlPSs3ohqhxD5a8g
Ph3+rK2tyeLiopRKJTcxMSG9vb3S2toqLS0tpohnGr+fnHG1PxK1nZ0d5jVx
7RYWFmRmZkZQGxgYcG1tbZLL5UwVR4h5delSVDxQfJbkBxQ3NjZkZWVF5ufn
ZXp62lwZGhqS9vZ2yefzpoxLRDg5RI/gSlR9o+qvQbRbhlBlt7y8LHNzc1Z/
UB4eHnY4VCgUjIB7bCeLkKd17GdR/R1Sr98PqqnP+4K6xoJDnbgYGRmRnp4e
0fxlFOIHdfIcNYLcpxrXIqFiwqMjLruAIJ7gRkdHpa+vT7q7u42E6xDIldQd
araOvx4JFRN+fJPT6p4LCDI2Nib9/f0OdUg4D4H8Su2iD2Cs6tyMlBOhfL//
X2/qPCFx3SUEzXpWyyARBWznCKh99BXMQNX4MhIqJnybzMAgCx5yenx83A0O
DprLkCAyK/mdikUvQp+nhK8j4R0m3Ns/0kUGedZ5gkAgjvjrCXYE1EX2Zxzj
Ved2pJwI5ZvEeU9xR+ofqs5TjEYkBBTxFOcp9yPlRChfJbmXWcXs0lnmAoLN
Qp2NzhMS511CYBb7GvhdJFRMuOUJLlkbpK6FvSEuU/98VjzkNH1LQlCNHyKh
YsKNsEJpf2d9nvZ71vfR/+Ey/SCkIBM6T0ictr5SddKR8h5RvtAP24O9uerS
qy/nVyrsqoyrM0hhB+rneBhbEH6KhIoJnydu6zy3VYhSqSR+DdBWKViThoTz
SX94JNcmTjdEQsWEq0ndI9eWy2Xn17Ztta5QKBgF19nue/SDfjBw+beofiz1
T1Gn18AtZpDWNFvzzufzRsBhvxJhNZCeJOj1fo+qb1H1IvmQ+VcsFu1OD3d8
crmcKRMTfh3PahbxwhF4xT+j4oHix7hDzSZvZbNZuwOJKi75NWmrN9R/v96D
2vMzrvYRajxHkclk7E43ijhChJObgv4HpZdnROVDVHi2hOcqUOJsE63MfiKS
uunP7KtTrPABCvqvqXBWiTzOoF9JY+Q/Z3pkZU9j+We9qv4FsOXhwg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5300214]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1662],

Cell["\<\
The probability of getting an irrational number or negative number is zero:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->21598],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"n", ",", 
     SubscriptBox["n", "succ"], ",", 
     SubscriptBox["n", "tot"]}], "]"}], ",", "\[Pi]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14978],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->121972345]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12825],

Cell[TextData[{
 "The characteristic function of the hypergeometric distribution is defined \
in terms of ",
 Cell[BoxData[
  ButtonBox["Hypergeometric2F1Regularized",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Hypergeometric2F1Regularized"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->20460],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CharacteristicFunction", "[", 
  RowBox[{
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"n", ",", 
     SubscriptBox["n", "succ"], ",", 
     SubscriptBox["n", "tot"]}], "]"}], ",", "t"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26269],

Cell[BoxData[
 RowBox[{"\[Piecewise]", GridBox[{
    {
     FractionBox[
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", 
           SubscriptBox["n", "succ"]}], "+", 
          SubscriptBox["n", "tot"]}], ")"}], "!"}], " ", 
       RowBox[{"Hypergeometric2F1Regularized", "[", 
        RowBox[{
         RowBox[{"-", "n"}], ",", 
         RowBox[{"-", 
          SubscriptBox["n", "succ"]}], ",", 
         RowBox[{"1", "-", "n", "-", 
          SubscriptBox["n", "succ"], "+", 
          SubscriptBox["n", "tot"]}], ",", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"\[ImaginaryI]", " ", "t"}]]}], "]"}]}], 
      RowBox[{
       RowBox[{"Binomial", "[", 
        RowBox[{
         SubscriptBox["n", "tot"], ",", "n"}], "]"}], " ", 
       RowBox[{"n", "!"}]}]], 
     RowBox[{
      RowBox[{"n", "+", 
       SubscriptBox["n", "succ"]}], "\[LessEqual]", 
      SubscriptBox["n", "tot"]}]},
    {
     RowBox[{
      FractionBox["1", 
       RowBox[{
        SubscriptBox["n", "tot"], "!"}]], 
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"\[ImaginaryI]", " ", "t", " ", 
         RowBox[{"(", 
          RowBox[{"n", "+", 
           SubscriptBox["n", "succ"], "-", 
           SubscriptBox["n", "tot"]}], ")"}]}]], " ", 
       RowBox[{"n", "!"}], " ", 
       RowBox[{
        SubscriptBox["n", "succ"], "!"}], " ", 
       RowBox[{"Hypergeometric2F1Regularized", "[", 
        RowBox[{
         RowBox[{"n", "-", 
          SubscriptBox["n", "tot"]}], ",", 
         RowBox[{
          SubscriptBox["n", "succ"], "-", 
          SubscriptBox["n", "tot"]}], ",", 
         RowBox[{"1", "+", "n", "+", 
          SubscriptBox["n", "succ"], "-", 
          SubscriptBox["n", "tot"]}], ",", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"\[ImaginaryI]", " ", "t"}]]}], "]"}]}]}], 
     TagBox["True",
      "PiecewiseDefault",
      AutoDelete->False,
      DeletionWarning->True]}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxItemSize->{
    "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.84]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}]}]], "Output",
 ImageSize->{598, 77},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->45481213]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->307904394],

Cell[TextData[{
 "The infinite population limit of ",
 Cell[BoxData[
  ButtonBox["HypergeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HypergeometricDistribution"]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->100094394],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"hpdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"HypergeometricDistribution", "[", 
     RowBox[{"n", ",", 
      RowBox[{"p", "  ", "ntot"}], ",", "ntot"}], "]"}], ",", "k"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->343715506],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Binomial", "[", 
    RowBox[{
     RowBox[{"ntot", " ", "p"}], ",", "k"}], "]"}], " ", 
   RowBox[{"Binomial", "[", 
    RowBox[{
     RowBox[{"ntot", "-", 
      RowBox[{"ntot", " ", "p"}]}], ",", 
     RowBox[{
      RowBox[{"-", "k"}], "+", "n"}]}], "]"}]}], 
  RowBox[{"Binomial", "[", 
   RowBox[{"ntot", ",", "n"}], "]"}]]], "Output",
 ImageSize->{332, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->486490537]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"hlim", "=", 
  RowBox[{"Assuming", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"n", ">", "0"}], "&&", 
     RowBox[{"k", ">", "0"}], "&&", 
     RowBox[{"0", "<", "p", "<", "1"}]}], ",", 
    RowBox[{"Limit", "[", 
     RowBox[{"hpdf", ",", 
      RowBox[{"ntot", "\[Rule]", "Infinity"}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->662360908],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}], 
    RowBox[{
     RowBox[{"-", "k"}], "+", "n"}]], " ", 
   SuperscriptBox["p", "k"], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", "n"}], "]"}]}], 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", "k"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "-", "k", "+", "n"}], "]"}]}]]], "Output",
 ImageSize->{182, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->430205385]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"bpdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"BinomialDistribution", "[", 
     RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->308109750],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], 
   RowBox[{
    RowBox[{"-", "k"}], "+", "n"}]], " ", 
  SuperscriptBox["p", "k"], " ", 
  RowBox[{"Binomial", "[", 
   RowBox[{"n", ",", "k"}], "]"}]}]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->20376465]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Assuming", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"n", ">", "0"}], "&&", 
    RowBox[{"k", ">", "0"}], "&&", 
    RowBox[{"0", "<", "p", "<", "1"}]}], ",", 
   RowBox[{"FullSimplify", "[", 
    RowBox[{"hlim", "\[Equal]", "bpdf"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->592089036],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->690149466]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(4)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12223],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HypergeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HypergeometricDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["n", "TI"], 
   StyleBox["tot", "TI"]]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["n", "TI"], 
   StyleBox["succ", "TI"]]], "InlineFormula"],
 ", or ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is non-positive:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->17347],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"5", ",", 
    RowBox[{"-", "25"}], ",", "50"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6402],

Cell[BoxData[
 RowBox[{
  RowBox[{"HypergeometricDistribution", "::", "\<\"nngint\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-25\\) is expected to be a non-negative \
integer.\"\>"}]], "Message", "MSG",
 CellID->43591602],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"5", ",", 
    RowBox[{"-", "25"}], ",", "50"}], "]"}], "]"}]], "Output",
 ImageSize->{308, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->475251471]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->15423],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HypergeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HypergeometricDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  RowBox[{
   StyleBox["n", "TI"], ">", 
   SubscriptBox[
    StyleBox["n", "TI"], 
    StyleBox["tot", "TI"]]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->14055],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"10", ",", "5", ",", "5"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28187],

Cell[BoxData[
 RowBox[{
  RowBox[{"HypergeometricDistribution", "::", "\<\"nsize\"\>"}], ":", 
  " ", "\<\"The number \\!\\(10\\) must be less than the total number \\!\\(5\
\\).\"\>"}]], "Message", "MSG",
 CellID->322267766],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"10", ",", "5", ",", "5"}], "]"}], "]"}]], "Output",
 ImageSize->{293, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->439008440]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->21330],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["HypergeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HypergeometricDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["n", "TI"], 
    StyleBox["succ", "TI"]], ">", 
   SubscriptBox[
    StyleBox["n", "TI"], 
    StyleBox["tot", "TI"]]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->6091],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"10", ",", "50", ",", "20"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18111],

Cell[BoxData[
 RowBox[{
  RowBox[{"HypergeometricDistribution", "::", "\<\"nsize\"\>"}], ":", 
  " ", "\<\"The number \\!\\(50\\) must be less than the total number \
\\!\\(20\\).\"\>"}]], "Message", "MSG",
 CellID->522338060],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"10", ",", "50", ",", "20"}], "]"}], "]"}]], "Output",
 ImageSize->{307, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->318776727]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3075],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"HypergeometricDistribution", "[", 
    RowBox[{"n", ",", 
     SubscriptBox["n", "succ"], ",", 
     SubscriptBox["n", "tot"]}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"n", "->", "10"}], ",", 
    RowBox[{
     SubscriptBox["n", "succ"], "->", "80"}], ",", 
    RowBox[{
     SubscriptBox["n", "tot"], "\[Rule]", 
     RowBox[{"-", "40"}]}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23205],

Cell[BoxData[
 RowBox[{"-", "20"}]], "Output",
 ImageSize->{26, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->136096669]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->15787],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["HypergeometricPFQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HypergeometricPFQ"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->5465]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->26495],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Discrete Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteStatisticalDistributions"]], "MoreAbout",
 CellID->364332738],

Cell[TextData[ButtonBox["Functions Used in Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInStatistics"]], "MoreAbout",
 CellID->318302660],

Cell[TextData[ButtonBox["Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/StatisticalDistributions"]], "MoreAbout",
 CellID->299109787],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->157151776]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"HypergeometricDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 47, 57.8979442}", "context" -> "System`", 
    "keywords" -> {"sampling without replacement", "urn problem"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "HypergeometricDistribution[n, n_succ, n_tot] represents a hypergeometric \
distribution.", "synonyms" -> {"hypergeometric distribution"}, "title" -> 
    "HypergeometricDistribution", "type" -> "Symbol", "uri" -> 
    "ref/HypergeometricDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8143, 259, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->392413691]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 46754, 1638}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2475, 59, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3098, 86, 1591, 37, 70, "ObjectNameGrid"],
Cell[4692, 125, 724, 22, 70, "Usage",
 CellID->10467]
}, Open  ]],
Cell[CellGroupData[{
Cell[5453, 152, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5931, 169, 470, 17, 70, "Notes",
 CellID->656338067],
Cell[6404, 188, 979, 38, 70, "Notes",
 CellID->6850],
Cell[7386, 228, 720, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[8143, 259, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->392413691],
Cell[CellGroupData[{
Cell[8528, 273, 147, 5, 70, "ExampleSection",
 CellID->33486935],
Cell[8678, 280, 275, 6, 70, "ExampleText",
 CellID->11373],
Cell[CellGroupData[{
Cell[8978, 290, 236, 7, 28, "Input",
 CellID->30750],
Cell[9217, 299, 251, 9, 51, "Output",
 CellID->238714793]
}, Open  ]],
Cell[CellGroupData[{
Cell[9505, 313, 239, 7, 28, "Input",
 CellID->9341],
Cell[9747, 322, 620, 23, 65, "Output",
 CellID->542841659]
}, Open  ]],
Cell[10382, 348, 121, 3, 70, "ExampleDelimiter",
 CellID->28548],
Cell[10506, 353, 205, 5, 70, "ExampleText",
 CellID->10406],
Cell[CellGroupData[{
Cell[10736, 362, 262, 8, 70, "Input",
 CellID->31918],
Cell[11001, 372, 591, 21, 53, "Output",
 CellID->260487121]
}, Open  ]],
Cell[CellGroupData[{
Cell[11629, 398, 444, 14, 47, "Input",
 CellID->212],
Cell[12076, 414, 1725, 32, 163, "Output",
 Evaluatable->False,
 CellID->124472560]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13850, 452, 216, 7, 70, "ExampleSection",
 CellID->10127],
Cell[14069, 461, 265, 8, 70, "ExampleText",
 CellID->18571],
Cell[CellGroupData[{
Cell[14359, 473, 219, 6, 70, "Input",
 CellID->13114],
Cell[14581, 481, 291, 9, 36, "Output",
 CellID->957126505]
}, Open  ]],
Cell[14887, 493, 120, 3, 70, "ExampleDelimiter",
 CellID->7750],
Cell[15010, 498, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[15134, 505, 240, 7, 70, "Input",
 CellID->32749],
Cell[15377, 514, 1021, 39, 61, "Output",
 CellID->26914300]
}, Open  ]],
Cell[CellGroupData[{
Cell[16435, 558, 239, 7, 70, "Input",
 CellID->9977],
Cell[16677, 567, 2104, 70, 119, "Output",
 CellID->475138028]
}, Open  ]],
Cell[18796, 640, 121, 3, 70, "ExampleDelimiter",
 CellID->14025],
Cell[18920, 645, 222, 5, 70, "ExampleText",
 CellID->1606],
Cell[CellGroupData[{
Cell[19167, 654, 491, 15, 70, "Input",
 CellID->20465],
Cell[19661, 671, 1251, 40, 71, "Output",
 CellID->49679336]
}, Open  ]],
Cell[20927, 714, 119, 3, 70, "ExampleDelimiter",
 CellID->113],
Cell[21049, 719, 312, 10, 70, "ExampleText",
 CellID->48521834],
Cell[CellGroupData[{
Cell[21386, 733, 218, 7, 70, "Input",
 CellID->4591],
Cell[21607, 742, 158, 5, 36, "Output",
 CellID->216492365]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21814, 753, 223, 7, 70, "ExampleSection",
 CellID->11626],
Cell[22040, 762, 274, 8, 70, "ExampleText",
 CellID->32514],
Cell[CellGroupData[{
Cell[22339, 774, 214, 7, 70, "Input",
 CellID->10642],
Cell[22556, 783, 209, 6, 51, "Output",
 CellID->293242353]
}, Open  ]],
Cell[CellGroupData[{
Cell[22802, 794, 92, 3, 70, "Input",
 CellID->3834],
Cell[22897, 799, 178, 5, 36, "Output",
 CellID->157144469]
}, Open  ]],
Cell[23090, 807, 86, 1, 70, "ExampleText",
 CellID->411747730],
Cell[CellGroupData[{
Cell[23201, 812, 218, 7, 70, "Input",
 CellID->633017742],
Cell[23422, 821, 211, 6, 51, "Output",
 CellID->211152684]
}, Open  ]],
Cell[CellGroupData[{
Cell[23670, 832, 97, 3, 70, "Input",
 CellID->109966815],
Cell[23770, 837, 176, 5, 36, "Output",
 CellID->750741742]
}, Open  ]],
Cell[23961, 845, 85, 1, 70, "ExampleText",
 CellID->213514598],
Cell[CellGroupData[{
Cell[24071, 850, 246, 8, 70, "Input",
 CellID->207586845],
Cell[24320, 860, 207, 6, 51, "Output",
 CellID->118270948]
}, Open  ]],
Cell[CellGroupData[{
Cell[24564, 871, 97, 3, 70, "Input",
 CellID->134218711],
Cell[24664, 876, 176, 5, 36, "Output",
 CellID->8079710]
}, Open  ]],
Cell[24855, 884, 124, 3, 70, "ExampleDelimiter",
 CellID->86062964],
Cell[24982, 889, 259, 7, 70, "ExampleText",
 CellID->13370],
Cell[CellGroupData[{
Cell[25266, 900, 331, 10, 70, "Input",
 CellID->26124],
Cell[25600, 912, 1286, 25, 70, "Output",
 Evaluatable->False,
 CellID->307328238]
}, Open  ]],
Cell[26901, 940, 125, 3, 70, "ExampleDelimiter",
 CellID->332082353],
Cell[27029, 945, 278, 8, 70, "ExampleText",
 CellID->12249],
Cell[CellGroupData[{
Cell[27332, 957, 446, 13, 70, "Input",
 CellID->14426],
Cell[27781, 972, 1609, 30, 70, "Output",
 Evaluatable->False,
 CellID->5300214]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29439, 1008, 232, 7, 70, "ExampleSection",
 CellID->1662],
Cell[29674, 1017, 259, 7, 70, "ExampleText",
 CellID->21598],
Cell[CellGroupData[{
Cell[29958, 1028, 266, 8, 70, "Input",
 CellID->14978],
Cell[30227, 1038, 158, 5, 36, "Output",
 CellID->121972345]
}, Open  ]],
Cell[30400, 1046, 121, 3, 70, "ExampleDelimiter",
 CellID->12825],
Cell[30524, 1051, 444, 13, 70, "ExampleText",
 CellID->20460],
Cell[CellGroupData[{
Cell[30993, 1068, 281, 8, 70, "Input",
 CellID->26269],
Cell[31277, 1078, 2653, 79, 98, "Output",
 CellID->45481213]
}, Open  ]],
Cell[33945, 1160, 125, 3, 70, "ExampleDelimiter",
 CellID->307904394],
Cell[34073, 1165, 404, 13, 70, "ExampleText",
 CellID->100094394],
Cell[CellGroupData[{
Cell[34502, 1182, 278, 9, 70, "Input",
 CellID->343715506],
Cell[34783, 1193, 522, 18, 53, "Output",
 CellID->486490537]
}, Open  ]],
Cell[CellGroupData[{
Cell[35342, 1216, 385, 12, 70, "Input",
 CellID->662360908],
Cell[35730, 1230, 566, 20, 56, "Output",
 CellID->430205385]
}, Open  ]],
Cell[CellGroupData[{
Cell[36333, 1255, 224, 7, 70, "Input",
 CellID->308109750],
Cell[36560, 1264, 390, 14, 36, "Output",
 CellID->20376465]
}, Open  ]],
Cell[CellGroupData[{
Cell[36987, 1283, 326, 10, 70, "Input",
 CellID->592089036],
Cell[37316, 1295, 161, 5, 36, "Output",
 CellID->690149466]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[37526, 1306, 327, 9, 70, "ExampleSection",
 CellID->12223],
Cell[37856, 1317, 668, 24, 70, "ExampleText",
 CellID->17347],
Cell[CellGroupData[{
Cell[38549, 1345, 204, 6, 70, "Input",
 CellID->6402],
Cell[38756, 1353, 220, 5, 70, "Message",
 CellID->43591602],
Cell[38979, 1360, 298, 9, 36, "Output",
 CellID->475251471]
}, Open  ]],
Cell[39292, 1372, 121, 3, 70, "ExampleDelimiter",
 CellID->15423],
Cell[39416, 1377, 521, 18, 70, "ExampleText",
 CellID->14055],
Cell[CellGroupData[{
Cell[39962, 1399, 184, 5, 70, "Input",
 CellID->28187],
Cell[40149, 1406, 225, 5, 70, "Message",
 CellID->322267766],
Cell[40377, 1413, 277, 8, 36, "Output",
 CellID->439008440]
}, Open  ]],
Cell[40669, 1424, 121, 3, 70, "ExampleDelimiter",
 CellID->21330],
Cell[40793, 1429, 568, 20, 70, "ExampleText",
 CellID->6091],
Cell[CellGroupData[{
Cell[41386, 1453, 186, 5, 70, "Input",
 CellID->18111],
Cell[41575, 1460, 226, 5, 70, "Message",
 CellID->522338060],
Cell[41804, 1467, 279, 8, 36, "Output",
 CellID->318776727]
}, Open  ]],
Cell[42098, 1478, 120, 3, 70, "ExampleDelimiter",
 CellID->3075],
Cell[42221, 1483, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[42527, 1495, 483, 16, 70, "Input",
 CellID->23205],
Cell[43013, 1513, 176, 6, 36, "Output",
 CellID->136096669]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[43250, 1526, 311, 9, 70, "SeeAlsoSection",
 CellID->15787],
Cell[43564, 1537, 492, 17, 70, "SeeAlso",
 CellID->5465]
}, Open  ]],
Cell[CellGroupData[{
Cell[44093, 1559, 314, 9, 70, "TutorialsSection",
 CellID->26495],
Cell[44410, 1570, 153, 3, 70, "Tutorials",
 CellID->4738]
}, Open  ]],
Cell[CellGroupData[{
Cell[44600, 1578, 299, 8, 70, "MoreAboutSection"],
Cell[44902, 1588, 178, 3, 70, "MoreAbout",
 CellID->364332738],
Cell[45083, 1593, 165, 3, 70, "MoreAbout",
 CellID->318302660],
Cell[45251, 1598, 161, 3, 70, "MoreAbout",
 CellID->299109787],
Cell[45415, 1603, 151, 3, 70, "MoreAbout",
 CellID->157151776]
}, Open  ]],
Cell[45581, 1609, 27, 0, 70, "History"],
Cell[45611, 1611, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

