(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    177511,       4662]
NotebookOptionsPosition[    163190,       4171]
NotebookOutlinePosition[    165207,       4218]
CellTagsIndexPosition[    165120,       4213]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Operations on Polynomials" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AlgebraicOperationsOnPolynomials"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Algebraic Operations on Polynomials\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Interpolation" :> 
          Documentation`HelpLookup["paclet:ref/Interpolation"], 
          "FunctionInterpolation" :> 
          Documentation`HelpLookup["paclet:ref/FunctionInterpolation"], "Fit" :> 
          Documentation`HelpLookup["paclet:ref/Fit"], "Roots" :> 
          Documentation`HelpLookup["paclet:ref/Roots"], 
          "InterpolatingFunction" :> 
          Documentation`HelpLookup["paclet:ref/InterpolatingFunction"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Interpolation\"\>", 
       2->"\<\"FunctionInterpolation\"\>", 3->"\<\"Fit\"\>", 
       4->"\<\"Roots\"\>", 
       5->"\<\"InterpolatingFunction\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Curve Fitting & Approximate Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/CurveFittingAndApproximateFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Curve Fitting & Approximate Functions\"\>", 
       2->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["InterpolatingPolynomial", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["InterpolatingPolynomial",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/InterpolatingPolynomial"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["f", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["f", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]constructs an interpolating polynomial in ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " which reproduces the function values ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["f", "i"], TraditionalForm]], "InlineMath"],
     " at successive integer values ",
     Cell[BoxData[
      FormBox[
       RowBox[{"1", ",", "2", ",", "\[Ellipsis]"}], TraditionalForm]], 
      "InlineMath"],
     " of ",
     Cell[BoxData[
      FormBox["x", TraditionalForm]], "InlineMath"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["InterpolatingPolynomial",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/InterpolatingPolynomial"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["1", "TR"]], ",", 
            SubscriptBox[
             StyleBox["f", "TI"], 
             StyleBox["1", "TR"]]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            SubscriptBox[
             StyleBox["x", "TI"], 
             StyleBox["2", "TR"]], ",", 
            SubscriptBox[
             StyleBox["f", "TI"], 
             StyleBox["2", "TR"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]constructs an interpolating polynomial for the function \
values ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["f", "i"], TraditionalForm]], "InlineMath"],
     " corresponding to ",
     Cell[BoxData[
      FormBox["x", TraditionalForm]], "InlineMath"],
     " values ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["InterpolatingPolynomial",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/InterpolatingPolynomial"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              SubscriptBox[
               StyleBox["x", "TI"], 
               StyleBox["1", "TR"]], ",", 
              SubscriptBox[
               StyleBox["y", "TI"], 
               StyleBox["1", "TR"]], ",", 
              StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
            SubscriptBox[
             StyleBox["f", "TI"], 
             StyleBox["1", "TR"]]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              SubscriptBox[
               StyleBox["x", "TI"], 
               StyleBox["2", "TR"]], ",", 
              SubscriptBox[
               StyleBox["y", "TI"], 
               StyleBox["2", "TR"]], ",", 
              StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
            SubscriptBox[
             StyleBox["f", "TI"], 
             StyleBox["2", "TR"]]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          StyleBox["y", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]constructs a multidimensional interpolating polynomial \
in the variables ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", ",", "y", ",", "\[Ellipsis]"}], TraditionalForm]], 
      "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["InterpolatingPolynomial",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/InterpolatingPolynomial"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              SubscriptBox[
               StyleBox["x", "TI"], 
               StyleBox["1", "TR"]], ",", 
              StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
            SubscriptBox[
             StyleBox["f", "TI"], 
             StyleBox["1", "TR"]], ",", 
            SubscriptBox[
             StyleBox["df", "TI"], 
             StyleBox["1", "TR"]], ",", 
            StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]constructs an interpolating polynomial that reproduces \
derivatives as well as function values."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->18504]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The function values ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["f", "i"], TraditionalForm]], "InlineMath"],
 " and sample points ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 ", etc. can be arbitrary real or complex numbers, and in 1D can be arbitrary \
symbolic expressions. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->28445701],

Cell[TextData[{
 "With a 1D list of data of length ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  ButtonBox["InterpolatingPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingPolynomial"]], "InlineFormula"],
 " gives a polynomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "-", "1"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->1405],

Cell[TextData[{
 "With any given specified set of data, there are infinitely many possible \
interpolating polynomials; ",
 Cell[BoxData[
  ButtonBox["InterpolatingPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingPolynomial"]], "InlineFormula"],
 " always tries to find the one with lowest total degree."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->13741],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["InterpolatingPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingPolynomial"]], "InlineFormula"],
 " gives the interpolating polynomial in a Horner form, suitable for \
numerical evaluation. "
}], "Notes",
 CellID->18513],

Cell["\<\
Different elements in the data can have different numbers of derivatives \
specified.\
\>", "Notes",
 CellID->44352622],

Cell[TextData[{
 "For multidimensional data, the ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " derivative can be given as a tensor with a structure corresponding to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["D",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/D"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        StyleBox["y", "TI"], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      StyleBox["n", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 ". ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->188072664,
  ButtonNote->"188072664"]
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->719953730],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["InterpolatingPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingPolynomial"]], "InlineFormula"],
 " allows any function value or derivative to be given as ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", in which case it will attempt to fill in the necessary information from \
derivatives or other function values. ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->177875428,
  ButtonNote->"177875428"]
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->563880862],

Cell[TextData[{
 "The option setting ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Modulus",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Modulus"], "->", 
   StyleBox["n", "TI"]}]], "InlineFormula"],
 " specifies that the interpolating polynomial should be found modulo ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ". ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->57022563,
  ButtonNote->"57022563"]
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->79223399]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->76115320],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->394451355],

Cell["Construct an interpolating polynomial for the squares:", "ExampleText",
 CellID->25620],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "4", ",", "9", ",", "16"}], "}"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18886],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "x"}], ")"}]}]}]], "Output",
 ImageSize->{116, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->497578883]
}, Open  ]],

Cell["Check the result:", "ExampleText",
 CellID->15767],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Expand", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15332],

Cell[BoxData[
 SuperscriptBox["x", "2"]], "Output",
 ImageSize->{17, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->99813704]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24719],

Cell["Construct an interpolating polynomial through three points:", \
"ExampleText",
 CellID->20172],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "6"}], "}"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20633],

Cell[BoxData[
 RowBox[{"4", "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"1", "+", "x"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "2"}], "+", 
     RowBox[{"3", " ", "x"}]}], ")"}]}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->699325116]
}, Open  ]],

Cell["Check the result at a single point:", "ExampleText",
 CellID->12703],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "/.", 
  RowBox[{"x", "->", "0"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->14456],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->749825420]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9349],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"4", ",", "7", ",", "2", ",", "8", ",", "9"}], "}"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20031],

Cell[BoxData[
 RowBox[{"4", "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"3", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "4"}], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           FractionBox["19", "6"], "-", 
           RowBox[{
            FractionBox["35", "24"], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", "4"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "3"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "2"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}]}]}]], "Output",
 ImageSize->{367, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->42877572]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{"%", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "6"}], "}"}]}], "]"}], ",", 
   RowBox[{"ListPlot", "[", 
    RowBox[{"{", 
     RowBox[{"4", ",", "7", ",", "2", ",", "8", ",", "9"}], "}"}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23760],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 135},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->545231217]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19328],

Cell["Make the polynomial have derivative 0 when it has value 8:", \
"ExampleText",
 CellID->10193],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"4", ",", "7", ",", "2", ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "0"}], "}"}], ",", "9"}], "}"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31863],

Cell[BoxData[
 RowBox[{"4", "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"3", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "4"}], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           FractionBox["19", "6"], "+", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", 
               FractionBox["107", "36"]}], "+", 
              RowBox[{
               FractionBox["109", "72"], " ", 
               RowBox[{"(", 
                RowBox[{
                 RowBox[{"-", "4"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", "4"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "3"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "2"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}]}]}]], "Output",
 ImageSize->{485, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->22608299]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{"%", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "6"}], "}"}]}], "]"}], ",", 
   RowBox[{"ListPlot", "[", 
    RowBox[{"{", 
     RowBox[{"4", ",", "7", ",", "2", ",", "8", ",", "9"}], "}"}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4322],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9WVtoHVUUPcmd3FQlNu/cxNimRRB//PHbB/bho7SpfRjSWiGGavKhlRjB
x49i86haBP0x0jZIWtuqYMVSFaqIAbFQrIgFUUEqSP2wCBY/AsJ17bPPmT2z
77kzSa5YksmcmX3Wfq29zznTzUPjI3sfHxofHR7qu2ds6MmR0eGn+jbuG8Oj
Qp0x5jR+95YM7su4s7/t5XLZFOjO/jOH6WL002Y7vnz5r6tXF+j25ViqHkIt
9u177373xutfpd9WYnx29ucXXzibJ3Xx+99HR07lSV258vcD/XOV9rTat3fe
PuOkDwQx2OqBne/ArTypfU98dOGb3/Kkpia/+PjMD2kpsWfmzXPHjl7Iw/jw
1MWDr87nScEYmJQnBcfgXqU9bSo+0/ZtI64EUe+kigqTvdiz++SlS3/KrBV0
jUSK5yfnFYI4T4+fOff1r4LDYvyv3g9zcQtxdBE0hK52u4TFteHMf/nL8899
WjsOgo2Qp7Pk4+OzJDldWPhnw7pDIq15wVIoGpQO3U5lSAnns6RQzChpn7VO
PIvcXxLtVnM63POSszqKkWlkNUQoEpQKx6iUmG8CeJ1qXFLj9oqoF6r7EknO
KjWFkKOA3NI0SnaN6QnEiqLSpWKg5RarkcegB0gSwtGx7HT6Swk7ChU567Cz
hh95/6cf/zDe2pQ1IQ543OTzLjXO8gKsc71CW6e1dSkvslCFeZMBH4UdkxkY
0nPDUh2JLNDtRIaUVCBLUVefs15red1L8nqLfV+Mx7wo+9zWirucMfzEj6mn
C63+ju614v9n9v1P+BqvrlydHSg4lF2oK15DlwjvsSii14OPEahkGrGlwqKP
6+yR827TU6VissZhm7h7bLpv1u0Q19KlATZgoYeZWJux5NjVyxSRX1gAswYH
jsMgWLk1xpSOJ+zvs/4AGCszsLBvaLJP4Bum33XHDEqzP8MqTMOyTrerE5HD
I7PS2gNAKHt79jzqGy+gdXMG2sT+z9los8qOMRshRSd3nRjoQEM9weFNGTg7
tx8DDjqBudGOsXVErKg5s/8oSqwN6Enm/kB0oHXDusMnjn9rev1sbD/ptttH
C70MABC8N2gF48ieldcDRMXhWFxaoWAVAgPCbMzAUfoTYxtzWlthDNIOB9en
/OF1QPpmyfsHvXS7xjIJ05gzMJBixFJgEsKEF6aeLqTVEXtt0FaehXRhYlxJ
tONAJrZumQPwTVYbUgfSwGDwFpHkvVgR0FCAWIDmwECKC6Q1gqMwDOYCdU1V
vWQhZ37HtqPQt/7uQ+Zmqw24XKkAgt/GsrsRNOQMAhsvEECypDHBOWJwX4af
zHO6bfec2PPQycce/QBe3EJP2uAw0w40xBUWAhmKXzs4DwuffeYT3EML7ACl
IYMrgkVcb/KVhTl0uyqVU+aSnK14LwttUEIt/lYaN2EMdQDAUyAjILac38I9
gKEIopRrPs3I+cnWi91RJ88rDSoG3WqWPifc5iMJdQiC43Hckpj/rpcm9zN5
PbzbM9OdYPXpeEuCA1T9y9XDMeadF92uVHXT78e7Bk+Mj51ekp5CnEOx+3rP
KcSLmLjNjuU82FOTH8gzCoBwmVkgDvTgaraXXTd03w2Wq+cGO5Y93nV2DI3I
E21ed3g7QFtaq3piliXPWw3ub6+SvjbBcKrhB1X0LCsq6rRX+bbC2+RQBuwY
+NBClRBG4fVDOnkxEUGqtkE75pNa2XeHKijQggZH9dDgPXCb+112LLtgQaF4
rPazYQZtS2w/jlBHsBwhgEu7VZ7DVth+RvsGzMcPapU48LDysCs49yVlYZYU
7EAXj1fb2A+70058negMYuxX2c2SkpUtLMUapVd2pOyxp4hE1+K3tApEGb2P
MWUN7oi5bBqMP0nyNbduJpQXjLX4PjKp/LP+u3pZvBWTigG1oEje2uNoJk5D
Mdzi0NQqW7VjTCnetQfZwFLS1dsypIQVWVLCApEie6aV9VkY8l0pLDWtrG7N
kJI+1Jqy54DKTRaGrHJZUvJ9OK3Jfi1OnFVaghhsj+wVW2KuFFR2K2dJZnjW
UnmqY7F4lEIcSYlRS4qXdlaqcvWpn+fLtymev/QVl+Ms35mblxULRpF9RnPV
PNDfV6w0FgAsAyKtc8uYsjesLkVfJML8kP8pIb1H2Ju6fwGhMUhN\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->37268126]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->405723598],

Cell["Interpolate values depending on 2 variables: ", "ExampleText",
 CellID->385391469],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0"}], "}"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "0"}], "}"}], ",", "7"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}], ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"2", ",", "1"}], "}"}], ",", "40"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"3", ",", "3"}], "}"}], ",", "151"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "2"}], "}"}], ",", "47"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->163495851],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{"x", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", 
     RowBox[{"4", " ", "x"}], "+", 
     RowBox[{"5", " ", "y"}]}], ")"}]}], "+", 
  RowBox[{"y", " ", 
   RowBox[{"(", 
    RowBox[{"3", "+", 
     RowBox[{"6", " ", "y"}]}], ")"}]}]}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->238656603]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{"%", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "3"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Red", ",", 
      RowBox[{"Map", "[", 
       RowBox[{"Point", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0", ",", "0"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", "0"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"2", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"3", ",", "3"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", "2"}], "}"}]}], "}"}]}], "]"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->387408667],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJx1XAlcjsv3f/c3+xLZd5dky5a9M6hIV7YQ2d1QlihkF9cNl2TJvu+5hBRl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   "], {{{}, 
     {RGBColor[0.3465645412680248, 0.14981535970418544`, 0.6041187042912054], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNlbtvVFcQxo/v3gV7i7CsX4RAjL22iQRe44eABjDUBqQ0FLwkioUACgUV
NUmMef0DiCBACAQVfwKPwrwaEwkkuthIgIKFXFjCCInvY36ruJh75s6ZM2ce
38zpPnLq19+zlNJDUUH0XvSvyLJyU0oXxOwXX8tT+rGY0qT+9+i/rnW31vNa
z4lK6F4UHZd8AvlB8ZeR257PTSD3mQnsWXcS+VbdtUZ3XdL/R/Hzcqwg3bfa
+w//ZkUfRCtFt6TzII97P6HThJ51yvDviOuw7I2KmUf+Hjsz8Nb5TZ+j+OO1
Dl9HfgD5Mfiruv+W7K4Sf038Hd8h/rr4e+J3if9b/G3x/eLP6ty0nDwhuivZ
PsV7Uvxa6fwl+mo/mkK2N0X81t0pfqP2J0Wn9T9HvM6NHT+Nfpv2/xAtsP+J
+M3PkZ9+7bcVw7Zzckg0Ihs3Jb9fiHgd35mmiPEH6Z7NI28HwIexYVkZ+Wqt
f4oWU/jrGB1HFzVbhw/vyP1gij3Lv+jc50Lk3jHsEI05FpG2kq7/jpXt7PWy
1y7qY+0QDYt6RFWodYmebSxDz+tyUSdrs+iVLlpRDHy0aq0Uo1av89gbBb91
cDtD31TIyXiKvmhFbr+rYh5KvyVFfUrUaBNxd4uGWO33CH47PuOiLw8cnONu
nx/j/BC5GOT/LXktL8F0hT6YRf4BnZX4OAPmr+ieGxKUUuBsiFw6ZuOiRO5r
3Dcg3eks/O3hHse9XvKpLPK6QfyzLOrTzZ0VcrQNe7PInYuBPPDtPG3RujkP
/JnK4HC1cjKYx+yx/DV7zqlz47q059Fv/WDDPjifGXpj5G0vmPo5j151nz6R
7tMsMOxzz4nRq/9H4V9k0acv0Bklnj3ENwu2G7XoJkb3mfttntwepu8K5K8C
Zg6h0woeepZg3bH4jHVaONPDHW3s+4zjcDz9YLENe1PIbetRFvj07PKcvsCs
/kfryyxyYn96mW8+u5455n3reYaM0wOeCbZXRb/OjNxN7FV8NjY2MicX+W/U
agNy+/Y4izlmf6eIxfYfgbFOfOqgd4zJYXybxn/7+ws+L/Dfia71jPfH2LTP
RXLtfJ1ghlmnMS866JGMXvQc9VvlPDgm+++YVuTxNqzC/0Xq10UP1ajbMP32
DKz1sr8DTDVmVjtnttM/rtcl3ly/G128Hc/B5kj6f754PryRX+fz6JmX1M1x
TPPWOB6/gTNg1nqXwYP3E7O5im37fbMQs6OET67zOLVbIF/2xfPCZ2qF4Buy
Gu+jazEAvw5/nYOf8nhTjQfX0LVbQH8nPvSCq3mw0Ee+/AZ28A76XVrD2zRE
/Vy7FvDUzGp/m7l/jBp8A33A38k=
         "],
         VertexColors->None], 
        PolygonBox[{{1365, 1053, 253, 717}, {1040, 1398, 804, 242}, {1368, 
         1062, 259, 724}, {1362, 1044, 246, 708}, {939, 1368, 724, 66}, {1046,
          1400, 808, 246}, {1364, 1050, 250, 714}, {1298, 982, 466, 643}, {
         1017, 1274, 615, 617}, {1295, 1294, 242, 564}, {1016, 1079, 377, 
         616}, {933, 1362, 708, 24}, {1400, 1045, 38, 808}, {1162, 1081, 654, 
         472}, {1061, 1013, 512, 364}, {1010, 1272, 363, 259}, {1398, 1039, 
         24, 804}, {1274, 1016, 616, 615}, {1052, 1402, 812, 250}, {930, 1240,
          564, 10}, {1055, 1403, 814, 253}, {1160, 1306, 650, 266}, {1071, 
         1304, 648, 650}, {936, 1365, 717, 52}, {1081, 1307, 652, 654}, {1013,
          1069, 371, 512}, {1402, 1051, 52, 812}, {1403, 1054, 66, 814}, {
         1161, 1160, 266, 371}, {1078, 987, 472, 377}, {1304, 1070, 649, 
         648}, {1307, 1080, 653, 652}, {1272, 1299, 643, 363}, {1308, 1189, 
         106, 653}, {935, 1364, 714, 38}, {945, 1275, 617, 93}, {982, 938, 
         364, 466}, {1305, 1187, 93, 649}},
         VertexColors->None]}]}, 
     {RGBColor[0.4584562070954453, 0.34436474855510213`, 0.7613959764832167], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNV0tsVkUU/u+9cytJlVcFSlueCdCfBEFefchKaI0aGqWYgCKlRTAB2rKA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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllElsj1EUxb/ve+8vEonSyVzapEVibGLeWRBDREjUrK1pUdQCC0okWgkW
IiQkqBhqFiqWLCxsTI0YG0LEUsQUqmj5nZzFSc5590333PteaW39gk1ZkiRV
IIKhuSQpAbvghwg0MlgIVqZJ8hC0hCQ5if6Vemw4/CdjKXPrpVn7Hv0BfpSx
dvh1sA39AgwjXprz3gPQF4mdYo8/8LVgLHwg8efwLeA78U7GNsF7gD3w3sR3
s/4+ehR6NOil+3DnwcQmo7drvhJDj8j5LocZ60fsGXs+gleDiehJoC+8TvnD
94I29DpQyNoK9EZ4I3s06X6gFv0YvIS/AhOILUbnM78ArEHPZ7/18IXERxLr
YqwaPT0694/opbpfdG7t6EXoYeh/rG1DF6Ffct+nxJvkGXpMtDcBnZfzHXS2
POkT7ZG8WQLK4VXMeZLZo+5gT+SF7lQHX8GcSnhPYq3wG+AY046k9mJIdG3k
yV/mzEbvg2/IfLerwbXXHb/A+0d72wlOw0uivexKXetBYHXqmqtXbgefpZ7Z
F11T1VI9cCV4D61VDQ7C54Kh8HfgM/F89DJ4BxgX3XPqta26E/wcWA5/AFrh
Z6O925G6t4pz9lY9Von+zZyYuWfL0A3ExwXX5FZwDVW7zeiC6JyVq84YHN0j
6g3lOAD+lXgt82tS372R9TOCc8jQGbo5dQ3uBtdEtVAN1Kvng9eqZ+8E95R6
6QDxS8E9o16Zgp7Kfm8Ya2duA2hGF4FV8N/gG7FP6HmZe1S9cTm4V9Ujyn1a
9Fp5cA9+ItrbCnTfaE/lpTw/Az+uPyD1G7gAPx/tzU7lA2+JfotloJP5f6K9
15u7FvwG1PszE/8l46Nrpz9Fe59lrDzzGfvRr4Pfqt64cr0ZfJZynkX8bXBv
qEfl9Zxor+S5avEDXZO5JvrL8qLfiv604ug/T3+dPJNX+gP09uXZ3+g3oN6v
U77BHstb/QnqldLoXNQzHfImujY6swveHb1Wf+x/ZheyaA==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.568619724553345, 0.5352758645688142, 0.9096081404827117], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmGmMl9UVxt//fe8rpsPQDEsXSZXFdaCtLcsHhkUrjbIoRBCIDS3tDAxa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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlW1ol2UUxp+X+7HQLfJlasVyc6Ju4uucH/I1Z4SlYb6kQir5sqmwzYVR
+6DkFAVbiEslcZKoTUybaUVGiWhpXxPEZEqKKPbJoprDt83fxfX/cPhf1znn
vp/7Pve5zr94ed3c2iSKonVYwEZmUbQEsD+Oon0E/kuj6B58DngV9jzx+/AV
4E3YYPgNcm6Svwv+N7iF+BJ4B/t1wf+BzydWh40Bb2TNBPwriA8AD8e3jvyt
xBPwv9gicD02BNybnG7ia7Gp8E3wKawvgL8Gb4RPg78EfxZei50Af8D6UmLz
4Lvhn8NfAX+MbxL59Xy/CNyE7zrxz4g/Af9A7BN4DbwTPB7fh/AEvoz8ycHf
/gu+G/wqvjZwJb4X4PPgv8Efs/8T1nfjayJWgs0Ev4EVgZuIXwO3B9/tEtaL
tXmqEes3JLm7wtdHvnNf8Nv4zhF7gK+3zgYfSqxQb5i5Brr7emwpuIfOCz7A
/lXEfyZ3L7kP8Z0HJ5lr3YovA/fIvPY0vIK1j8jJ4B+xfimxU7na3Ulcq2Pk
9Excszzi+ZnPfkHvS6wBXkbeCHg7v9+k3usP7Cj4ADmPwdXYnuCaqpZ6w4vE
61jfl9+vWD+Y2Dbsd2KrY/emeki9ox79n7wyfPWxe+ZXeI3OxO/+yGd9Gd6c
+Mx6q9bc2fRmqu3hXC1V4zby38S2s1c+vJvYAvge+HPw7+HvseZe4je6A/8l
da3vYsPIXUz8CvH82HeVhqQd3XlEcI+pt96HH4QXB9+tCzsCnoS9CH4mca9O
xzbH7tmvU69RrjSkbx3XGSN/U9o8jX0aW6M7yP0zde/pTmdS31F3k0a+Td3j
6u0qcr6DtwZ/W9/UW1TBJ8Z+E9V2Gbwido1Xwt8K7vWOxFofG3x3ab4A3j/z
LNEZ5wb3pHqxXN8PfnO9tXr2x9ye2qtRb6L+0UxIPKOGaX5kni3q6Wpi5fBD
8C8Ta30ivn6xNa9eHBisHfVkCfhssPbWxH7LyuBZojetgI/PrAXtWUdsimqc
WPOd8I5grWsmSot9gmefNDkNPjWzFrTmdXAD8Z9ia3yh+hnfLPUgvnfBJ3Kz
8YZ6RG8bPIs0MzTrZgTXQjMvBaf4voj9BuoVaVTaVM+UB2tOWtMMU+5orCXx
mkJwcebZtyX2rCrJPMs1swZlzlFMmjkc3CPqDc2E+fBb8NuxNaFZr5mrWauZ
PztYM9KKelCzXf8x+m/RjD8Z3AN6e72pavFOsFZVk53BNVJt9A1pvyh4VmgG
lIKvknc5tuaF21JrSb5m4hMy94J6dlxwjVVb9cSozJqVVlWTp3hE2aQ=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6508859897604071, 0.6674548619247941, 0.9114835579458729], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNVktslkUU/f5v5keeMSoPFxJAcCFUCK8WtYBVoAUjUFBLWYgQIj8LWyWU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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlGtoz3EUxn+X74/NtoRaXlDGvDBCtjGXXTDbTC5byMhlE6Mwl1zLPXsj
conZC8olc11u08xlyi1JjLyw4cVEJpnGCyXl8zh78fz/5znn/L6Xc57zTSqr
KF4VeJ5XChy4xE8uqPQ9L45AC/ab0POmEMuAX4CfBxvhO8nZi50JtmE3g/6R
5w0AWvQAOT72YFANP0Z8KPYwsBZ+kfhH1r4Hqol9AlWstQ7kYbeDVOzt5GeQ
U+6ZnYavu28+na0R3i+wM84jXouvBPsdvn3EJoI95L8HZ7D/EE/07cx7yJ+B
7zD5Bfgr4bPgQ4gVwg9it/J/gngVvkPwpeAs9lfQlfxY+Gb22gRixCOz7/DN
W2LNzmo3CjhiXcAG7FzVT7Ug/pTcOezzGTwCp4l9AYOIbSVnROednsNfgWFa
n/ho1SayvWrw7cbO133hE/imAns8vu/kXgEN+JbAfweWM8lZjVQb9Twdez7x
1sBq1ET+enhSaGcc7KzGqq16UA4fGVktanw7awq+WN/O/BN7OL7j5M8FU+GL
4R2B1XgBfAW8G7njfNtrC74i3/aUFlQj1UaaKHLWM/VqILxNtQd1rDcZ5BOL
h0/zzRcPf0FuA3x7YGdZSPykb2d6gv0Y3IV3gGTi0+D7sY8QT4lMo9KmNNxE
7FtouepBPTyNnDPYvUEO/D5oIb4D3MbOJn6ZWB8QOquJaqEZSHamEWljF6jD
vgFqPdOAtLUGXPNNY3/Zux1eHNiMncPO0owQj4F/UC+dzabOcCu0mdYs604F
zjQibfSF3+S/DP4jsJnOwc6KTCs687PQ7qi7XcV3FHulauaZpsaQOzYy7WXC
VxNrC212deaZzmaqsHOG9O0qZ9rXGr/w94js7tJMXWh30l00s3qU9Ib8fzt8
0059aG+LNBQHT4ist+r5HPId6BXYm6LeqweqvTTQCF9OfhDazKdjp0amXfUw
wdmMabakwSiyGujumlnVKs9Z7VSz1fBlzu6e2Dlb051pUzOmXklT0pJ6dh3e
xZnW9YZkO3tD9Xaqh9ewZzt7a3pKn6HtobWlCc3Kg9DeTs2M7loKTvl254eh
raFvpXHN8pXOt1MzrbuXOKularDI2Zujt0ZrvAxtpjRLr/nmH7Xsvvg=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7331522549674694, 0.7996338592807739, 0.913358975409034], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1lk9oXVUQxu9791y0omkWonWRQhETqEKjq7wUWlGxSTZNXkgwVURrQ2wF
rXkWkiqSICYBC4ml/stLF7YRGjdVbCJUmnRjK9YKdWMVjLZCJdRFF65bvx/z
ZTGcc+fMmTNnvm/m3C1736i+Xs6y7E9JLjmcsuxQkWXLpSz7T/N7JF2aL0pu
STolpyQ7JOdkv0/rj8t+Xt/DclDR+LX0Vc17JDv1/aZsNkh2aT4nuV8HnpTN
v1q/Iv2vmr8oXavkquabZVPRfDKLczbKblnzM7JdlDygtf0aj6Xwz3qDbQrp
nijiDGL9KYs4pzW/V7pmjUN57Dktfbt02yUjml/X2lFJi+RBx/5VFnER31ZJ
k+RIHvlZ0FlfcjfpZolD4zaNL0uXJG2a12TbL/0J+TlYjvnnWZzTqPlH3Evz
DsklzSdL4Y8ccL/XfMfv5OcvzXeUI05GfJDvL5xz4sbfBHeR7bUUa89rnFJO
6j5nwdjdNJbMl2TzQx5+iIHzJ5w3cnZM82HZfCA/3+i7WorxR+lXS7EH+zX5
OJhCzz2mfZfbGvvKkc8DWj8rWckil+S06pxjQwyPOd9wAy60G6Npz/H5h3ys
psCvzl7Jb7K5IN1FSYuxGjVe/Y6B/C9xx3Lw4x3ZduheK9I1aN5ZxNoe7RuQ
9JYCo4fKkT9yBgfh5qDsv3WesWXP71n47TWmYNLqHMKL58yNj8vBAXCEv03m
8MPG+EPJtVJgwN5Z8wwudcvH+4pzBjzLgSW4UnPUHrmdNCfAhbv1uh7g6Ix9
bva5K7Y9ag6hY43ap1YrPpf8gRM4rpl7rN1wbXPPcecNPODOlGPgjru9l15B
beBzUx4+wZ0anTLnsG1zDU64L3CnbvuhfsG9ah5yn2bHj79NrmFwr5lbyLDP
qrhnEUO/48UPfarfc/jUZ0zh6Tr3OX/IMYzlYb++b9y+jkvmJCPuTz3O86fC
7t0UePSaN5d8zoDPGrdP9DuNK3ng3G7nhX4DjuBZk79PUsQHhzpd1wOOBz/0
vEbHvNHxr/Ouy/ueKaLvd/kuY96LdJg/vANLfguoiQX3FWpkzHkY9b3p58dd
g+Rt2WvoeRPuKsL+nHv9smvlpPHE3xnXEbVSN76Xtfe+FH1iQOP+Impjg8a7
JYfdJyruFaPGg5h+SbGfvdjhnxje076OFPzb5vogjrp5zbnNsnmkiNrdrrG9
iPp7WuNTktOalwoKPfo9+Nfdl+gxb6eoxfPSHSiiZu6Uoz+C6bNa31tEP513
bRHDoHRn8+gFi849udqdYm3aHGkwTxr9xpHTV4mliL5VNyfBAt3FPPT09Zr5
02E8wJoe05OCZ7whe1Lk49Ei7skdaxrfSsGJOecZ/9xjKY+7nNf6k5ItfoMn
3BdHHA/5od+1uMb7nDdqEGxX8sCXf5Mr7ufk74UUOeQf5JUUOaPWh13vu1Lg
CqaX84h1yXw44jhZW3VM9IpZv9fLWv87j/ei7h7XalzWjA3rW4uwmTVv2Dvj
3g2H2rReIf/leFdWXEeHUtQbtcZbPeh3hLfxZ/eK7zXecM1SZzeNDbm67nxR
g6dch/8Qc4r4Ol2nnIWPl1L44X+G/F11rc2ZY03mD+cOucc2uNfc9tvNf9W8
7eHyZ+ZznzlMHk74jcEG3R3r1/8j4S11csvf/wNIcEbn
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlN1rj2Ecxu/7+d1PbHnZwRIHVkuocSAnZqsJ5YcTMyXGAePAqL1oSgun
XhKL8jJOFLWf0hL7yUublCZvhQMlmxnRcMBfwOfqcnA9z/f6Xt/7fu7vy/3U
tnW2dGQhhCJI4CaP1eAszgqwHvsMGI8hnIgO+gO2onWBr4UQHoOraN/ARbQp
+HNC78B/YL8CZezf4Dz6PjCA/ROMYv9Cf0/8YvbbBT+Zh9CMbxDfCuxGUEIr
Eb8TPQfVmb/5AvsdWAavA0PYbaAJbQ78Gft0s34u79vsN6J90Wej9RPTgz0L
1GQ+4zadj/i/8Hr4J/gi+AX4OfiY8sVXCz/Mfq/hPeg1vEfgH9HGQCN6AxhP
XqNY+aYT+xJ+l7164X3om1Rv+GfQgf70f+5VvJfD96Jfj855D/Z3/FPRvlPo
+5O/3YJ/Te6aq9aDxFeiVeIbgm8EG7CPKD465wfE3oLPyHyGieSclWtT5tq1
w1dF11CzoRyVm2ZEs7CUmP7MMyF7e3Jt5RtVrsm9UM+noVfkzr0I70LrTrY1
Mytz91y9HiDmDesfoc/HHsa3EC3CqzL3RLW8D5+Xuaabk2uqWvZF9+ZQcu7q
kXo7zPtacI81K6q5aq2Z0Ww0J6/VjOhbO8CV6G/ewy4nz9KN6LMdTJ51nVFn
qc99dp1Js7w7eZY001+wJ5PXqocT+B4WXOtJ+GW0S2BLdI8OJPdYvVUP1Kti
8l1Uz54k76G1C1Qv+Fv1JPOdash953TXdIeOYZ/G11rwN1qT77DurnKUdhTf
h+CYMigl/wuOR9+13uRe6M5pttbmnjXN2MzkO6m7qDPo37Euea3+IXXELsn9
71AP/gFUWJP+
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8100377758620689, 0.8628068965517242, 0.8927928068965517], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFVTtrlFEQvd/33Q8hgq2YTWNAOwstJSEPH+DuJoWYRpNGJeBGFMEX6EaL
pBFbRZtEQZs8QDRkjYuSWKmQxN6IwVfEJEXwD3gO5ywpZu/95s7MnXNm5u7e
s5dPXkpDCA8gGWQPfrqTEF5hX8H+DQ6XgtY65Cv2/am+ue/NQ7gbQ3iL71HY
17CeSrS+9v4Ozouwm0uln4V8ge9v6G9C5mAzCN8eSJYpB64zYfub+bTDrw1y
MFEOBegXoR/A/oXzLKW6py+RX7PxjGSyX4FNFWt0zNu4/0SuXImpNVOcDuy7
4Lfg+NTzzvuwHYbPPM7G8f0Echj73VifQjqxH850Rv152B7KZUd8xPksiK+6
8/zLHHNh4T0DxtIEXRP8b+G71/zQ9wZ0x5gzfMuQZejmg/C1mJPM9s+xr8Gm
lCgmMY2lwj5oHhiz2TzPOA75Yu2YC3Ez5oprz7q3mE/a5I5DvJnvZJwN5LkO
KbturN/DoPqPuBbki1wvODfuXwZzaT5b3Ye8+zvi/YB0peKYXI/ZtsP20Twz
l4+QX7A/mkqfG+8F6K6Aw4lE/A4YI/tnCDbFVJwVXQvOAvWsF22oJ+/UlVxH
5lixDTGXbFO0nnE2E+UynagPh81bzfNDboin6v4kT6PmmRi7zM9UIr5ocw1Y
ruaqc6MnyCfzZ18wB+Ls9Df7gv3EOfoG39Wo2nCGau6TA6lynXK8OfdZt3W8
d38uf/qewZpExWRe086hkCqnRj6TzqEC23u5vqtRc8WZWot6F3gfe/x6lP+R
XPNwy3XMzE+jTsxtDbpzUfz8xP50VK1L8OvJNW/kaSkTzos4/xyFaRnrUtzG
PW0cBc/9imes01jG/D7wLuIYisLCeJ8ckzEK7u1VxHkf1f851v6ouve4Do05
bTaeCb8pvI/xdjjmLNZa1Fndb1af35yqZ2Enzo/nuvsR9o8h/zwXVfcb+7bi
Pt9yfcuu7aaxL2Z6mzgXE6niEVM7Yrfl0nGGPmTKl7lP+q2Zz1QD8j/q94S9
zff+netLXsruh3omjsjPOtY/UXkx3oZtdrHPo2ahwzNP/skf3xdyOO75oZ79
uC/XOXuLHPB/h/20Zbwb5prx/wNTEeH2
         "],
         VertexColors->None], 
        PolygonBox[{{974, 1210, 538, 439}, {1251, 999, 178, 577}, {1142, 1003,
          497, 440}, {1123, 1329, 671, 178}, {1416, 1115, 305, 846}, {1228, 
         1231, 557, 428}, {1252, 1341, 682, 429}, {1330, 1029, 314, 672}, {
         1177, 1147, 219, 499}, {1210, 1130, 427, 538}, {1317, 1318, 573, 
         305}, {1394, 1132, 322, 794}, {1352, 1148, 696, 695}, {1214, 1213, 
         207, 441}, {1353, 1038, 340, 696}, {1395, 973, 205, 796}, {1386, 960,
          164, 773}, {1031, 1228, 428, 322}, {1258, 1151, 697, 587}, {1150, 
         1176, 499, 448}, {968, 1206, 535, 426}, {1259, 1258, 587, 588}, {
         1140, 1395, 796, 340}, {1172, 1129, 426, 492}, {1348, 1141, 439, 
         688}, {1350, 1143, 440, 539}, {1125, 1249, 576, 673}, {970, 1394, 
         794, 180}, {1034, 1268, 605, 635}, {1147, 1352, 695, 219}, {1268, 
         1348, 688, 605}, {1207, 1125, 673, 535}, {1131, 1172, 492, 427}, {
         1354, 1178, 341, 697}, {1256, 1255, 441, 497}, {1000, 969, 580, 
         493}, {1213, 1260, 588, 207}, {1249, 1250, 577, 576}, {1329, 1124, 
         672, 671}, {1357, 1356, 221, 699}, {1114, 1416, 846, 164}, {1152, 
         1355, 699, 701}, {1230, 1000, 493, 557}, {1116, 1386, 773, 314}, {
         1340, 1035, 539, 682}, {1248, 955, 150, 573}, {1178, 1358, 701, 
         341}, {1253, 1252, 429, 580}, {1346, 1347, 635, 205}},
         VertexColors->None]}]}, 
     {RGBColor[0.8839184655172414, 0.8874441379310345, 0.859694324137931], 
      EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{690, 899, 689}, {1000, 1230, 194}, {548, 455, 454}, {342,
          581, 223}, {578, 579, 194}, {555, 553, 194}, {453, 578, 194}, {548, 
         607, 455}, {887, 886, 553}, {343, 581, 208}, {581, 343, 223}, {449, 
         342, 222}, {1360, 1359, 702}, {1005, 1225, 501}, {558, 603, 195}, {
         1178, 1354, 500}, {581, 582, 208}, {1237, 1238, 562}, {898, 578, 
         209}, {455, 562, 451}, {887, 558, 195}, {1231, 1228, 555}, {454, 455,
          209}, {978, 1153, 210}, {1260, 1213, 543}, {702, 501, 451}, {578, 
         453, 209}, {1003, 1142, 208}, {603, 558, 430}, {690, 689, 208}, {451,
          501, 343}, {450, 500, 222}, {969, 1000, 194}, {453, 454, 209}, {500,
          449, 222}, {899, 690, 333}, {1143, 1350, 691}, {690, 691, 333}, {
         607, 562, 455}, {1035, 1340, 333}, {343, 452, 223}, {343, 689, 
         209}, {860, 454, 453}, {553, 453, 194}, {1356, 1357, 700}, {543, 692,
          342}, {886, 887, 195}, {451, 343, 209}, {582, 581, 342}, {449, 543, 
         342}, {898, 899, 333}, {578, 898, 333}, {1132, 1394, 430}, {1151, 
         1258, 449}, {899, 898, 689}, {1394, 970, 430}, {1358, 1178, 500}, {
         861, 860, 210}, {1228, 1031, 555}, {501, 452, 343}, {859, 860, 
         453}, {222, 342, 223}, {689, 898, 209}, {1341, 1252, 579}, {689, 343,
          208}, {886, 859, 453}, {860, 861, 454}, {1252, 1253, 579}, {859, 
         886, 195}, {1213, 1214, 543}, {1220, 1219, 548}, {582, 583, 208}, {
         579, 578, 333}, {1258, 1259, 449}, {692, 582, 342}, {455, 451, 
         209}, {1355, 1152, 450}, {1255, 1256, 583}, {860, 859, 210}, {558, 
         887, 553}, {553, 886, 453}},
         VertexColors->None], 
        PolygonBox[{{1152, 1358, 500, 450}, {1153, 1421, 861, 210}, {1253, 
         969, 194, 579}, {1006, 1237, 562, 607}, {1236, 1360, 702, 451}, {
         1226, 1232, 558, 553}, {1214, 1144, 692, 543}, {1421, 1154, 454, 
         861}, {1232, 1132, 430, 558}, {1259, 1260, 543, 449}, {1350, 1035, 
         333, 691}, {1031, 1226, 553, 555}, {1225, 1224, 452, 501}, {1357, 
         1355, 450, 700}, {1219, 1006, 607, 548}, {1154, 1220, 548, 454}, {
         1359, 1005, 501, 702}, {1238, 1236, 451, 562}, {1144, 1254, 582, 
         692}, {1142, 1349, 690, 208}, {1349, 1143, 691, 690}, {1354, 1151, 
         449, 500}, {1340, 1341, 579, 333}, {1230, 1231, 555, 194}, {1256, 
         1003, 208, 583}, {1254, 1255, 583, 582}},
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{239, 237, 235}, {1360, 1236, 561}, {1224, 1225, 552}, {
         237, 241, 231}, {1006, 1219, 230}, {1237, 1006, 230}, {241, 237, 
         225}, {456, 457, 235}, {1154, 1421, 456}, {1220, 1154, 456}, {1421, 
         1153, 456}, {1236, 1238, 561}, {561, 231, 224}, {231, 561, 230}, {
         235, 237, 230}, {456, 235, 230}, {237, 231, 230}, {237, 239, 225}, {
         1005, 1359, 224}},
         VertexColors->None], 
        PolygonBox[{{1359, 1360, 561, 224}, {1238, 1237, 230, 561}, {1153, 
         978, 457, 456}, {1219, 1220, 456, 230}, {1225, 1005, 224, 552}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{978, 1153, 1421, 1154, 1220, 1219, 1006, 1237, 1238, 1236, 
         1360, 1359, 1005, 1225, 1224}],
        "140"],
       Annotation[#, 140, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{970, 1394, 1132, 1232, 1226, 1031, 1228, 1231, 1230, 1000, 
         969, 1253, 1252, 1341, 1340, 1035, 1350, 1143, 1349, 1142, 1003, 
         1256, 1255, 1254, 1144, 1214, 1213, 1260, 1259, 1258, 1151, 1354, 
         1178, 1358, 1152, 1355, 1357, 1356}],
        "120"],
       Annotation[#, 120, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1149, 1420, 1150, 1176, 1177, 1147, 1352, 1148, 1353, 1038, 
         1140, 1395, 973, 1002, 1346, 1347, 1290, 1034, 1268, 1348, 1141, 
         1396, 974, 1210, 1130, 1418, 1131, 1172, 1129, 1393, 968, 1206, 1207,
          1125, 1249, 1250, 1251, 999, 1123, 1329, 1124, 1330, 1029, 1116, 
         1386, 960, 996, 1114, 1416, 1115, 1025, 1317, 1318, 1247, 1248, 955, 
         992, 1218}],
        "100"],
       Annotation[#, 100, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1088, 1216, 989, 1246, 1245, 1244, 1089, 1194, 1193, 950, 
         1378, 1094, 1166, 1096, 1411, 1095, 1197, 954, 1381, 1103, 1168, 
         1105, 1413, 1104, 1199, 959, 1385, 1113, 1324, 1265, 1024, 1281, 
         1323, 1322, 995, 958, 1384, 1112, 1028, 1121, 1417, 1120, 998, 965, 
         1390, 1122, 1205, 1204, 967, 1392, 1128, 1171, 1338, 1339, 1335, 
         1337, 1336, 1209, 972, 1288, 1289, 1287, 1033, 1137, 1136, 1001, 
         1175, 1139, 1419, 1138, 1212, 977, 1397, 1146, 1235, 1227, 1037, 
         1229, 1234, 1233, 1004, 1223, 1222}],
        "80"],
       Annotation[#, 80, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{976, 1257, 1145, 1351, 1036, 1291, 1293, 1292, 975, 1211, 
         1343, 1344, 1342, 1135, 1345, 1173, 1174, 1133, 1134, 1032, 1284, 
         1286, 1285, 971, 1208, 1332, 1333, 1331, 1127, 1334, 1170, 1126, 
         1391, 966, 1203, 1030, 1118, 1388, 963, 997, 1326, 1327, 1283, 1027, 
         1267, 1328, 1119, 1389, 964, 1202, 1110, 1415, 1111, 1169, 994, 1108,
          1414, 1109, 1023, 1101, 1379, 952, 991, 1314, 1315, 1279, 1021, 
         1263, 1316, 1102, 1380, 953, 1196, 1092, 1410, 1093, 1165, 1091, 
         1377, 949, 1192, 1086, 1409, 1087, 1163, 1085, 1375, 947, 1188, 1019,
          1077, 1373, 944, 986, 1075, 1407, 1076, 1015, 1067, 1371, 942, 984, 
         1065, 1405, 1066, 1012, 1297, 1060, 1241, 1242, 1243, 981, 1215, 
         1059}],
        "60"],
       Annotation[#, 60, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{962, 1282, 1026, 1266, 1325, 1117, 1387, 961, 1200, 1201, 
         1106, 1382, 956, 993, 1319, 1320, 1280, 1022, 1264, 1321, 1107, 1383,
          957, 1198, 1099, 1412, 1100, 1167, 990, 1097, 1098, 1020, 1276, 
         1278, 1277, 951, 1195, 1310, 1311, 1309, 1313, 1312, 1164, 1090, 
         1376, 948, 1190, 1191, 1084, 1374, 946, 988, 1082, 1408, 1083, 1018, 
         1074, 1372, 943, 985, 1072, 1406, 1073, 1014, 1063, 1369, 940, 983, 
         1301, 1302, 1273, 1011, 1262, 1303, 1064, 1370, 941, 1186, 1057, 
         1404, 1058, 1159, 1056, 1366, 937, 1184, 1296, 1270, 1271, 1269, 
         1009, 1049, 1363, 934, 980, 1047, 1401, 1048, 1008, 1043, 1361, 932, 
         979, 1041, 1399, 1042, 1007, 1180, 1221}],
        "40"],
       Annotation[#, 40, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{931, 1179, 930, 1240, 1239, 1295, 1294, 1155, 1040, 1398, 
         1039, 1181, 933, 1362, 1044, 1156, 1046, 1400, 1045, 1182, 935, 1364,
          1050, 1157, 1052, 1402, 1051, 1183, 936, 1365, 1053, 1158, 1055, 
         1403, 1054, 1185, 939, 1368, 1062, 1300, 1261, 1010, 1272, 1299, 
         1298, 982, 938, 1367, 1061, 1013, 1069, 1068, 1161, 1160, 1306, 1071,
          1304, 1070, 1305, 1187, 945, 1275, 1017, 1274, 1016, 1079, 1078, 
         987, 1162, 1081, 1307, 1080, 1308, 1189, 1217}],
        "20"],
       Annotation[#, 20, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Epilog->{
    RGBColor[1, 0, 0], {
     PointBox[{0, 0}], 
     PointBox[{1, 0}], 
     PointBox[{0, 1}], 
     PointBox[{2, 1}], 
     PointBox[{3, 3}], 
     PointBox[{1, 2}]}},
  Frame->True,
  PlotRange->{{0, 3}, {0, 3}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 179},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->455981983]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28313],

Cell["\<\
Make the polynomial have zero derivative at -1 and 1 without specifying the \
values there:\
\>", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->177875428],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "Automatic", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "Automatic", ",", "0"}], "}"}]}], "}"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->891653599],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{"x", " ", 
   RowBox[{"(", 
    RowBox[{"1", "-", 
     FractionBox[
      SuperscriptBox["x", "2"], "3"]}], ")"}]}]}]], "Output",
 ImageSize->{84, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->224688373]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{"%", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->200576512],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->76663258]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->754120929],

Cell["Specify some of the partial derivatives in 2 dimensions:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->188072664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0"}], "}"}], ",", "1", ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", "3"}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "0"}], "}"}], ",", "7"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}], ",", "10", ",", 
       RowBox[{"{", 
        RowBox[{"Automatic", ",", "15"}], "}"}]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->93515997],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{"x", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", 
     RowBox[{"4", " ", "x"}]}], ")"}]}], "+", 
  RowBox[{"y", " ", 
   RowBox[{"(", 
    RowBox[{"3", "+", 
     RowBox[{"6", " ", "y"}]}], ")"}]}]}]], "Output",
 ImageSize->{158, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->505326363]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->316195888],

Cell["Interpolate values and a gradient in 3 variables:", "ExampleText",
 CellID->25187625],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", "4", ",", 
       RowBox[{"{", 
        RowBox[{"5", ",", "6", ",", "7"}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"3", ",", "2", ",", "1"}], "}"}], ",", "0"}], "}"}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13698880],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "34"}], "+", 
  RowBox[{"5", " ", "x"}], "+", 
  RowBox[{"6", " ", "y"}], "+", 
  RowBox[{"7", " ", "z"}]}]], "Output",
 ImageSize->{115, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->714633937]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31773],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Modulus",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->24623],

Cell["\<\
Find a polynomial interpolating the given points in arithmetic mod 47:\
\>", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->57022563],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "=", 
  RowBox[{"InterpolatingPolynomial", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4", ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9", ",", "3"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16", ",", "4"}], "}"}]}], "}"}], ",", "x", ",", 
    RowBox[{"Modulus", "\[Rule]", "47"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->79976998],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"12", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"6", "+", 
        RowBox[{"31", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "9"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "4"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
   "x"}]}]], "Output",
 ImageSize->{225, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->79368049]
}, Open  ]],

Cell["The polynomial takes on the specified values mod 47:", "ExampleText",
 CellID->623917411],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", " ", "/.", " ", 
  RowBox[{"x", "\[Rule]", 
   RowBox[{"{", 
    RowBox[{"0", ",", "4", ",", "9", ",", "16"}], "}"}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->16398935],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "49", ",", "379", ",", "43009"}], "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->270490817]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mod", "[", 
  RowBox[{"%", ",", "47"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->904431693],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "2", ",", "3", ",", "4"}], "}"}]], "Output",
 ImageSize->{80, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->164769456]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->907],

Cell[TextData[{
 "Construct a polynomial with roots ",
 Cell[BoxData["a"], "InlineFormula"],
 ", ",
 Cell[BoxData["b"], "InlineFormula"],
 " and ",
 Cell[BoxData["c"], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->2666],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"b", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"c", ",", "0"}], "}"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17380],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{"x", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", "a"]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "a"}], "+", "x"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        FractionBox["1", 
         RowBox[{"a", " ", "b"}]], "-", 
        FractionBox[
         RowBox[{
          RowBox[{"-", "b"}], "+", "x"}], 
         RowBox[{"a", " ", "b", " ", "c"}]]}], ")"}]}]}], ")"}]}]}]], "Output",\

 ImageSize->{217, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->228419607]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Factor", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1666],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "a"}], "+", "x"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "b"}], "+", "x"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "c"}], "+", "x"}], ")"}]}], 
   RowBox[{"a", " ", "b", " ", "c"}]]}]], "Output",
 ImageSize->{167, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->345849230]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->156371133],

Cell[TextData[{
 "Newton\[Dash]Cotes integration formulas with ",
 Cell[BoxData["n"], "InlineFormula"],
 " points:"
}], "ExampleText",
 CellID->374619358],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"abc", "=", 
      RowBox[{
       RowBox[{"Range", "[", 
        RowBox[{"0", ",", 
         RowBox[{"n", "-", "1"}]}], "]"}], "/", 
       RowBox[{"(", 
        RowBox[{"n", "-", "1"}], ")"}]}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"Integrate", "[", 
      RowBox[{
       RowBox[{"InterpolatingPolynomial", "[", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"x", ",", 
             RowBox[{"f", "[", "x", "]"}]}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", "abc"}], "}"}]}], "]"}], ",", "x"}], "]"}], ",",
        
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{"n", ",", "2", ",", "8"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->60724522],

Cell[BoxData[
 TagBox[
  TagBox[GridBox[{
     {
      RowBox[{
       FractionBox[
        RowBox[{"f", "[", "0", "]"}], "2"], "+", 
       FractionBox[
        RowBox[{"f", "[", "1", "]"}], "2"]}]},
     {
      RowBox[{
       FractionBox[
        RowBox[{"f", "[", "0", "]"}], "6"], "+", 
       RowBox[{
        FractionBox["2", "3"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "2"], "]"}]}], "+", 
       FractionBox[
        RowBox[{"f", "[", "1", "]"}], "6"]}]},
     {
      RowBox[{
       FractionBox[
        RowBox[{"f", "[", "0", "]"}], "8"], "+", 
       RowBox[{
        FractionBox["3", "8"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "3"], "]"}]}], "+", 
       RowBox[{
        FractionBox["3", "8"], " ", 
        RowBox[{"f", "[", 
         FractionBox["2", "3"], "]"}]}], "+", 
       FractionBox[
        RowBox[{"f", "[", "1", "]"}], "8"]}]},
     {
      RowBox[{
       FractionBox[
        RowBox[{"7", " ", 
         RowBox[{"f", "[", "0", "]"}]}], "90"], "+", 
       RowBox[{
        FractionBox["16", "45"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "4"], "]"}]}], "+", 
       RowBox[{
        FractionBox["2", "15"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "2"], "]"}]}], "+", 
       RowBox[{
        FractionBox["16", "45"], " ", 
        RowBox[{"f", "[", 
         FractionBox["3", "4"], "]"}]}], "+", 
       FractionBox[
        RowBox[{"7", " ", 
         RowBox[{"f", "[", "1", "]"}]}], "90"]}]},
     {
      RowBox[{
       FractionBox[
        RowBox[{"19", " ", 
         RowBox[{"f", "[", "0", "]"}]}], "288"], "+", 
       RowBox[{
        FractionBox["25", "96"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "5"], "]"}]}], "+", 
       RowBox[{
        FractionBox["25", "144"], " ", 
        RowBox[{"f", "[", 
         FractionBox["2", "5"], "]"}]}], "+", 
       RowBox[{
        FractionBox["25", "144"], " ", 
        RowBox[{"f", "[", 
         FractionBox["3", "5"], "]"}]}], "+", 
       RowBox[{
        FractionBox["25", "96"], " ", 
        RowBox[{"f", "[", 
         FractionBox["4", "5"], "]"}]}], "+", 
       FractionBox[
        RowBox[{"19", " ", 
         RowBox[{"f", "[", "1", "]"}]}], "288"]}]},
     {
      RowBox[{
       FractionBox[
        RowBox[{"41", " ", 
         RowBox[{"f", "[", "0", "]"}]}], "840"], "+", 
       RowBox[{
        FractionBox["9", "35"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "6"], "]"}]}], "+", 
       RowBox[{
        FractionBox["9", "280"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "3"], "]"}]}], "+", 
       RowBox[{
        FractionBox["34", "105"], " ", 
        RowBox[{"f", "[", 
         FractionBox["1", "2"], "]"}]}], "+", 
       RowBox[{
        FractionBox["9", "280"], " ", 
        RowBox[{"f", "[", 
         FractionBox["2", "3"], "]"}]}], "+", 
       RowBox[{
        FractionBox["9", "35"], " ", 
        RowBox[{"f", "[", 
         FractionBox["5", "6"], "]"}]}], "+", 
       FractionBox[
        RowBox[{"41", " ", 
         RowBox[{"f", "[", "1", "]"}]}], "840"]}]},
     {
      RowBox[{
       FractionBox[
        RowBox[{"751", " ", 
         RowBox[{"f", "[", "0", "]"}]}], "17280"], "+", 
       FractionBox[
        RowBox[{"3577", " ", 
         RowBox[{"f", "[", 
          FractionBox["1", "7"], "]"}]}], "17280"], "+", 
       RowBox[{
        FractionBox["49", "640"], " ", 
        RowBox[{"f", "[", 
         FractionBox["2", "7"], "]"}]}], "+", 
       FractionBox[
        RowBox[{"2989", " ", 
         RowBox[{"f", "[", 
          FractionBox["3", "7"], "]"}]}], "17280"], "+", 
       FractionBox[
        RowBox[{"2989", " ", 
         RowBox[{"f", "[", 
          FractionBox["4", "7"], "]"}]}], "17280"], "+", 
       RowBox[{
        FractionBox["49", "640"], " ", 
        RowBox[{"f", "[", 
         FractionBox["5", "7"], "]"}]}], "+", 
       FractionBox[
        RowBox[{"3577", " ", 
         RowBox[{"f", "[", 
          FractionBox["6", "7"], "]"}]}], "17280"], "+", 
       FractionBox[
        RowBox[{"751", " ", 
         RowBox[{"f", "[", "1", "]"}]}], "17280"]}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.5599999999999999]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}],
   Column],
  Function[BoxForm`e$, 
   TableForm[BoxForm`e$]]]], "Output",
 ImageSize->{478, 181},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TableForm=",
 CellID->380240317]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23812338],

Cell[TextData[{
 "Centered finite difference formula of order ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 " for approximating the first derivative: "
}], "ExampleText",
 CellID->755679668],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"abc", "=", 
      RowBox[{"Range", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "k"}], "/", "2"}], ",", 
        RowBox[{"k", "/", "2"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"Together", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"InterpolatingPolynomial", "[", 
           RowBox[{
            RowBox[{"Table", "[", 
             RowBox[{
              RowBox[{"{", 
               RowBox[{"s", ",", 
                RowBox[{"f", "[", 
                 RowBox[{"x", "+", 
                  RowBox[{"s", " ", "h"}]}], "]"}]}], "}"}], ",", 
              RowBox[{"{", 
               RowBox[{"s", ",", "abc"}], "}"}]}], "]"}], ",", "s"}], "]"}], 
          ",", "s"}], "]"}], "/", "h"}], " ", "/.", " ", 
       RowBox[{"s", "\[Rule]", "0"}]}], "]"}]}], ",", "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{"k", ",", "2", ",", "10", ",", "2"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->394635355],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{586, 127},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TableForm=",
 CellID->6486793]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->325782462],

Cell["\<\
Interpolate to find the characteristic polynomial of a matrix:\
\>", "ExampleText",
 CellID->475502054],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"RandomInteger", "[", 
   RowBox[{"1", ",", 
    RowBox[{"{", 
     RowBox[{"6", ",", "6"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->456555764],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "0", ",", "1"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{377, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->42357163]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Det", "[", 
      RowBox[{"m", "-", 
       RowBox[{"\[Lambda]", " ", 
        RowBox[{"IdentityMatrix", "[", "6", "]"}]}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"\[Lambda]", ",", "11"}], "}"}]}], "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->499969362],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFWdtuwjAMDU3pGNfdtPG0seunbE/7iApNgodpE+P/u7bE0B0cnLQpQ6LF
qetzbMdOWt7T9eLjM10v5+nsbZV+L5bzn9nr1yof0h2lolgpNZmq/HeW/yq/
9Enyr1Kz4qi34+oB5AHILyA/Cvq3nvrPLevfgxyHkNUdjD6BfCpEEb1AfYyi
pI/XQ+tjlJtGUZdR7OXHbj54AldR7hu9hLEa5WJkzoiCdkb5QRv94tyD62if
cNGOL642eLEj7sCi54sbGf6Ery3+ID7FqWm8EZ/8l/Ldlv++eWwqF3aNF91q
J1YOMRgKOeqWgDtZbUYOWBzv3aE5m0I9yJ74M4tYu/yoNHfb4sjmNhDnsZCH
47JFdlw3/r8I+9dBmCo2S1b5oV6KOeV6K/UgjhWtaZFD1Ku9LFE7PluGsN5w
6yXZQA+TzI8Lty744HProw++20yxZcjGiiLrm5kmtUZYtmyHjpTNd5u9MLXT
F1j6rn2u3vOdql6PRk6cfojakdZgaa0IO2ulHVa91fcYlSXZbiuOUn7brTR8
+pXYh99lDvbuqFdvo+DMaLfv+hQ2NPqo13xnQ88jXF87JO/H9pj72pjBc31y
a86LsufvcZjKwkqZgIzXR5nfs2r1fntXO4O7sIsNPVETFuUyMEq9nT2HWo3Q
uBUWuJe+ABk77KQhC67G/+udhOTrNcj4lrCqb5/D55yVPa0bJy3Njl7VR4Au
XXf3gT5iZMJgoxbmD/fd0yCoOAvCvAsuuGhL7iImYjg3qW5obcUeRvZZO3HV
R22qLmYqwt1KwaWb/X3viZyIM0b0sDU6c9xib2uhs7n5f4SQEk9JdX4BZ7lh
Nw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{516, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->52628845]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Expand", "[", "%", "]"}], "===", 
  RowBox[{"CharacteristicPolynomial", "[", 
   RowBox[{"m", ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->404413096],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->373488390]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->955660304],

Cell["Create a tensor product interpolation:", "ExampleText",
 CellID->846400484],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"m", ",", "n"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"4", ",", "4"}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"xc", ",", "yc"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Range", "[", "m", "]"}], ",", 
     RowBox[{"Range", "[", "n", "]"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"zc", "=", 
   RowBox[{"RandomInteger", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "9"}], ",", "9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"m", ",", "n"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->157133961],

Cell[TextData[{
 "Create an interpolating polynomial for each fixed ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " value:"
}], "ExampleText",
 CellID->23398873],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"yinterp", "=", 
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"InterpolatingPolynomial", "[", 
      RowBox[{
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"yc", ",", "#"}], "}"}], "]"}], ",", "y"}], "]"}], "&"}], 
    ",", "zc"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1427335],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "3"}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         FractionBox["11", "2"], "-", 
         RowBox[{
          FractionBox["23", "6"], " ", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", "3"}], "+", "y"}], ")"}]}]}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "2"}], "+", "y"}], ")"}]}]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "y"}], ")"}]}], ",", 
   RowBox[{"3", "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "11"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          FractionBox["23", "2"], "-", 
          RowBox[{"7", " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"-", "3"}], "+", "y"}], ")"}]}]}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "2"}], "+", "y"}], ")"}]}]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "y"}], ")"}]}]}], ",", 
   RowBox[{
    RowBox[{"-", "9"}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"4", "-", 
       RowBox[{
        FractionBox["11", "6"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "3"}], "+", "y"}], ")"}]}]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "2"}], "+", "y"}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "y"}], ")"}]}]}], ",", 
   RowBox[{"1", "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"8", "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "6"}], "+", 
          RowBox[{"2", " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"-", "3"}], "+", "y"}], ")"}]}]}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "2"}], "+", "y"}], ")"}]}]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "y"}], ")"}]}]}]}], "}"}]], "Output",
 ImageSize->{561, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->312572547]
}, Open  ]],

Cell[TextData[{
 "Show the interpolation curves in the ",
 Cell[BoxData[
  StyleBox["y", "TI"]], "InlineFormula"],
 " direction:"
}], "ExampleText",
 CellID->135694689],

Cell[BoxData[
 RowBox[{
  RowBox[{"points", "=", 
   RowBox[{"Graphics3D", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", "Medium", "]"}], ",", "Red", ",", 
      RowBox[{"Point", "@", 
       RowBox[{"Flatten", "[", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{"xc", "[", 
              RowBox[{"[", "i", "]"}], "]"}], ",", 
             RowBox[{"yc", "[", 
              RowBox[{"[", "j", "]"}], "]"}], ",", 
             RowBox[{"zc", "[", 
              RowBox[{"[", 
               RowBox[{"i", ",", "j"}], "]"}], "]"}]}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"i", ",", "m"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"j", ",", "n"}], "}"}]}], "]"}], ",", "1"}], "]"}]}]}], 
     "}"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->83262874],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"ParametricPlot3D", "[", 
    RowBox[{
     RowBox[{"Evaluate", "@", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"xc", "[", 
           RowBox[{"[", "i", "]"}], "]"}], ",", "y", ",", 
          RowBox[{"yinterp", "[", 
           RowBox[{"[", "i", "]"}], "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "m"}], "}"}]}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "1", ",", "n"}], "}"}], ",", 
     RowBox[{"BoxRatios", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"1", ",", "1", ",", "0.4"}], "}"}]}]}], "]"}], ",", "points"}],
   "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->31712146],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->89661972]
}, Open  ]],

Cell[TextData[{
 "Interpolate between the curves in the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " direction:"
}], "ExampleText",
 CellID->37608948],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"xyinterp", "=", 
  RowBox[{"InterpolatingPolynomial", "[", 
   RowBox[{
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"xc", ",", "yinterp"}], "}"}], "]"}], ",", "x"}], 
   "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->28513564],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{571, 347},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->120300620]
}, Open  ]],

Cell["Show the interpolating surface and data points:", "ExampleText",
 CellID->56965926],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{"xyinterp", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "1", ",", "m"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "1", ",", "n"}], "}"}]}], "]"}], ",", "points"}], 
  "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->511407523],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->810928204]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9259],

Cell["\<\
The interpolating polynomial always goes through the data points:\
\>", "ExampleText",
 CellID->166409219],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3", ",", "5", ",", "8", ",", "5"}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->420782050],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{"data", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4436254],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"1", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        FractionBox["1", "6"], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            FractionBox["1", "24"]}], "+", 
           FractionBox[
            RowBox[{"5", "-", "x"}], "20"]}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "4"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "3"}], "+", "x"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "2"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], ")"}]}]}]], "Output",
 ImageSize->{384, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->232407042]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "/.", " ", 
  RowBox[{"x", "\[Rule]", 
   RowBox[{"Range", "[", "6", "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->163355998],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "2", ",", "3", ",", "5", ",", "8", ",", "5"}], 
  "}"}]], "Output",
 ImageSize->{116, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->712179932]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->407943899],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ListInterpolation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ListInterpolation"]], "InlineFormula"],
 " creates a tensor product interpolation:"
}], "ExampleText",
 CellID->473096014],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"m", ",", "n"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"5", ",", "6"}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"xc", ",", "yc"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Range", "[", "m", "]"}], ",", 
     RowBox[{"Range", "[", "n", "]"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"zc", "=", 
   RowBox[{"RandomInteger", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "9"}], ",", "9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"m", ",", "n"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->34202644],

Cell[TextData[{
 "Create a numerical ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object:"
}], "ExampleText",
 CellID->205322194],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"linterp", "=", 
  RowBox[{"ListInterpolation", "[", 
   RowBox[{"zc", ",", 
    RowBox[{"{", 
     RowBox[{"xc", ",", "yc"}], "}"}], ",", 
    RowBox[{"InterpolationOrder", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"m", "-", "1"}], ",", 
       RowBox[{"n", "-", "1"}]}], "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21587785],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "6"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{294, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->447784873]
}, Open  ]],

Cell["\<\
Create a symbolic polynomial by interpolating in each dimension separately:\
\>", "ExampleText",
 CellID->937810632],

Cell[BoxData[{
 RowBox[{
  RowBox[{"yinterp", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"InterpolatingPolynomial", "[", 
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"{", 
          RowBox[{"yc", ",", "#"}], "}"}], "]"}], ",", "y"}], "]"}], "&"}], 
     ",", "zc"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"interp", "=", 
   RowBox[{"InterpolatingPolynomial", "[", 
    RowBox[{
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"xc", ",", "yinterp"}], "}"}], "]"}], ",", "x"}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->55371509],

Cell["Verify that results agree with random data points:", "ExampleText",
 CellID->185745966],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{"Abs", "[", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"interp", "-", 
      RowBox[{"linterp", "[", 
       RowBox[{"x", ",", "y"}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"RandomReal", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "m"}], "}"}], ",", 
         RowBox[{"2", "m"}]}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"RandomReal", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "n"}], "}"}], ",", 
         RowBox[{"2", "n"}]}], "]"}]}], "}"}]}], "]"}], "]"}], "]"}]], "Input",\

 CellLabel->"In[4]:=",
 CellID->526053930],

Cell[BoxData["4.440892098500626`*^-14"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->214460321]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17419],

Cell["Runge's function:", "ExampleText",
 CellID->450115561],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{"25", " ", 
       RowBox[{"x", "^", "2"}]}]}], ")"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->102724351],

Cell["\<\
Sampling at evenly spaced intervals in the interval from -1 to 1:\
\>", "ExampleText",
 CellID->644646945],

Cell[BoxData[
 RowBox[{
  RowBox[{"points", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"f", "[", "x", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "1"}], ",", "1", ",", ".1"}], "}"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->195197814],

Cell["\<\
The polynomial that interpolates these points has large oscillations:\
\>", "ExampleText",
 CellID->163770],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"InterpolatingPolynomial", "[", 
     RowBox[{"points", ",", "x"}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Red", ",", 
      RowBox[{"Map", "[", 
       RowBox[{"Point", ",", "points"}], "]"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->7635386],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->772094833]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Interpolation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Interpolation"]], "InlineFormula"],
 " uses a lower-order piecewise polynomial that does not have this problem:"
}], "ExampleText",
 CellID->54978621],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Interpolation", "[", "points", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->134926756],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1.`"}], ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
   "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{266, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->21499659]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"%", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Red", ",", 
      RowBox[{"Map", "[", 
       RowBox[{"Point", ",", "points"}], "]"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->226607093],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->133956055]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->466796740],

Cell["\<\
When derivatives are specified without function values an interpolant may not \
be found:\
\>", "ExampleText",
 CellID->154411273],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "Automatic", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->554349834],

Cell[BoxData[
 RowBox[{
  RowBox[{"InterpolatingPolynomial", "::", "\<\"noipf\"\>"}], ":", 
  " ", "\<\"Unable to find an interpolating polynomial of total degree \
\\!\\(2\\).\"\>"}]], "Message", "MSG",
 CellID->510286077],

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "Automatic", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "}"}], ",", "x"}], "]"}]], "Output",
 ImageSize->{441, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->146878770]
}, Open  ]],

Cell["\<\
There is no quadratic polynomial that satisfies the interpolation conditions:\
\
\>", "ExampleText",
 CellID->702970728],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Expand", "[", 
  RowBox[{"InterpolatingPolynomial", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "a", ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", 
        RowBox[{"-", "1"}]}], "}"}]}], "}"}], ",", "x"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->76220879],

Cell[BoxData[
 RowBox[{"a", "-", 
  RowBox[{"a", " ", 
   SuperscriptBox["x", "2"]}], "-", 
  SuperscriptBox["x", "3"]}]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1976488]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->815823841],

Cell["Points with abscissas lying on a line in 2 dimensions:", "ExampleText",
 CellID->19854453],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0"}], "}"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}], ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"2", ",", "2"}], "}"}], ",", "4"}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->591039839],

Cell["\<\
In multiple dimensions an interpolant may not be found for some arrangements \
of points:\
\>", "ExampleText",
 CellID->533158672],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{"data", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->65706371],

Cell[BoxData[
 RowBox[{
  RowBox[{"InterpolatingPolynomial", "::", "\<\"poised\"\>"}], ":", 
  " ", "\<\"The interpolation points \\!\\({\\(\\({0, 0}\\)\\), \\(\\({1, \
1}\\)\\), \\(\\({2, 2}\\)\\)}\\) are not poised, so an interpolating \
polynomial of total degree \\!\\(1\\) could not be found. \\!\\(\\*ButtonBox[\
\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/InterpolatingPolynomial/poised\\\", \
ButtonNote -> \\\"InterpolatingPolynomial::poised\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellID->91438957],

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0"}], "}"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}], ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"2", ",", "2"}], "}"}], ",", "4"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Output",
 ImageSize->{480, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->270381017]
}, Open  ]],

Cell["\<\
This polynomial interpolates the data above, but has total degree 2:\
\>", "ExampleText",
 CellID->409938236],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingPolynomial", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "4"}], "}"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->79009712],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"1", "+", 
     RowBox[{
      FractionBox["1", "2"], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "1"}], "+", "x"}], ")"}]}]}], ")"}], " ", 
   "x"}]}]], "Output",
 ImageSize->{130, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->102546439]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->32480],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Interpolation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Interpolation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FunctionInterpolation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FunctionInterpolation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Fit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Fit"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Roots",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Roots"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["InterpolatingFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InterpolatingFunction"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->30505]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->7675],

Cell[TextData[ButtonBox["Algebraic Operations on Polynomials",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AlgebraicOperationsOnPolynomials"]], "Tutorials",\

 CellID->20145]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with InterpolatingPolynomial",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
InterpolatingPolynomial"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->45608450],

Cell[TextData[ButtonBox["Curve Fitting & Approximate Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/CurveFittingAndApproximateFunctions"]], "MoreAbout",\

 CellID->95810050],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->206106957]
}, Open  ]],

Cell["New in 2  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"InterpolatingPolynomial - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 49, 20.7438150}", "context" -> "System`", 
    "keywords" -> {
     "Birkhoff interpolation", "Hermite interpolation", 
      "Newton interpolation", "polynomial", "polynomial interpolation"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "InterpolatingPolynomial[{f_1, f_2, ...}, x] constructs an interpolating \
polynomial in x which reproduces the function values f_i at successive \
integer values 1, 2, ... of x. InterpolatingPolynomial[{{x_1, f_1}, {x_2, \
f_2}, ...}, x] constructs an interpolating polynomial for the function values \
f_i corresponding to x values x_i. InterpolatingPolynomial[{{{x_1, y_1, ...}, \
f_1}, {{x_2, y_2, ...}, f_2}, ...}, {x, y, ...}] constructs a \
multidimensional interpolating polynomial in the variables x, y, .... \
InterpolatingPolynomial[{{{x_1, ...}, f_1, df_1, ...}, ...}, {x, ...}] \
constructs an interpolating polynomial that reproduces derivatives as well as \
function values.", "synonyms" -> {"interpolating polynomial"}, "title" -> 
    "InterpolatingPolynomial", "type" -> "Symbol", "uri" -> 
    "ref/InterpolatingPolynomial"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[17561, 505, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->76115320]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 164977, 4206}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2566, 62, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3189, 89, 4207, 83, 70, "ObjectNameGrid"],
Cell[7399, 174, 5427, 164, 70, "Usage",
 CellID->18504]
}, Open  ]],
Cell[CellGroupData[{
Cell[12863, 343, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[13341, 360, 504, 15, 70, "Notes",
 CellID->28445701],
Cell[13848, 377, 430, 15, 70, "Notes",
 CellID->1405],
Cell[14281, 394, 489, 12, 70, "Notes",
 CellID->13741],
Cell[14773, 408, 288, 8, 70, "Notes",
 CellID->18513],
Cell[15064, 418, 129, 4, 70, "Notes",
 CellID->44352622],
Cell[15196, 424, 961, 32, 70, "Notes",
 CellID->719953730],
Cell[16160, 458, 741, 20, 70, "Notes",
 CellID->563880862],
Cell[16904, 480, 620, 20, 70, "Notes",
 CellID->79223399]
}, Closed]],
Cell[CellGroupData[{
Cell[17561, 505, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->76115320],
Cell[CellGroupData[{
Cell[17945, 519, 148, 5, 70, "ExampleSection",
 CellID->394451355],
Cell[18096, 526, 93, 1, 70, "ExampleText",
 CellID->25620],
Cell[CellGroupData[{
Cell[18214, 531, 211, 7, 28, "Input",
 CellID->18886],
Cell[18428, 540, 323, 12, 36, "Output",
 CellID->497578883]
}, Open  ]],
Cell[18766, 555, 56, 1, 70, "ExampleText",
 CellID->15767],
Cell[CellGroupData[{
Cell[18847, 560, 98, 3, 28, "Input",
 CellID->15332],
Cell[18948, 565, 180, 6, 36, "Output",
 CellID->99813704]
}, Open  ]],
Cell[19143, 574, 121, 3, 70, "ExampleDelimiter",
 CellID->24719],
Cell[19267, 579, 100, 2, 70, "ExampleText",
 CellID->20172],
Cell[CellGroupData[{
Cell[19392, 585, 379, 13, 70, "Input",
 CellID->20633],
Cell[19774, 600, 349, 13, 36, "Output",
 CellID->699325116]
}, Open  ]],
Cell[20138, 616, 74, 1, 70, "ExampleText",
 CellID->12703],
Cell[CellGroupData[{
Cell[20237, 621, 113, 4, 70, "Input",
 CellID->14456],
Cell[20353, 627, 158, 5, 36, "Output",
 CellID->749825420]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[20560, 638, 215, 7, 70, "ExampleSection",
 CellID->9349],
Cell[CellGroupData[{
Cell[20800, 649, 220, 7, 70, "Input",
 CellID->20031],
Cell[21023, 658, 877, 31, 51, "Output",
 CellID->42877572]
}, Open  ]],
Cell[CellGroupData[{
Cell[21937, 694, 359, 12, 70, "Input",
 CellID->23760],
Cell[22299, 708, 3299, 58, 70, "Output",
 Evaluatable->False,
 CellID->545231217]
}, Open  ]],
Cell[25613, 769, 121, 3, 70, "ExampleDelimiter",
 CellID->19328],
Cell[25737, 774, 99, 2, 70, "ExampleText",
 CellID->10193],
Cell[CellGroupData[{
Cell[25861, 780, 273, 9, 70, "Input",
 CellID->31863],
Cell[26137, 791, 1156, 39, 51, "Output",
 CellID->22608299]
}, Open  ]],
Cell[CellGroupData[{
Cell[27330, 835, 358, 12, 70, "Input",
 CellID->4322],
Cell[27691, 849, 2399, 43, 70, "Output",
 Evaluatable->False,
 CellID->37268126]
}, Open  ]],
Cell[30105, 895, 125, 3, 70, "ExampleDelimiter",
 CellID->405723598],
Cell[30233, 900, 88, 1, 70, "ExampleText",
 CellID->385391469],
Cell[CellGroupData[{
Cell[30346, 905, 945, 32, 70, "Input",
 CellID->163495851],
Cell[31294, 939, 423, 15, 36, "Output",
 CellID->238656603]
}, Open  ]],
Cell[CellGroupData[{
Cell[31754, 959, 902, 28, 70, "Input",
 CellID->387408667],
Cell[32659, 989, 41042, 699, 70, "Output",
 CellID->455981983]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[73750, 1694, 239, 7, 70, "ExampleSection",
 CellID->28313],
Cell[73992, 1703, 190, 5, 70, "ExampleText",
 CellID->177875428],
Cell[CellGroupData[{
Cell[74207, 1712, 432, 14, 70, "Input",
 CellID->891653599],
Cell[74642, 1728, 304, 11, 54, "Output",
 CellID->224688373]
}, Open  ]],
Cell[CellGroupData[{
Cell[74983, 1744, 203, 7, 70, "Input",
 CellID->200576512],
Cell[75189, 1753, 2371, 43, 70, "Output",
 Evaluatable->False,
 CellID->76663258]
}, Open  ]],
Cell[77575, 1799, 125, 3, 70, "ExampleDelimiter",
 CellID->754120929],
Cell[77703, 1804, 147, 2, 70, "ExampleText",
 CellID->188072664],
Cell[CellGroupData[{
Cell[77875, 1810, 723, 24, 70, "Input",
 CellID->93515997],
Cell[78601, 1836, 387, 14, 36, "Output",
 CellID->505326363]
}, Open  ]],
Cell[79003, 1853, 125, 3, 70, "ExampleDelimiter",
 CellID->316195888],
Cell[79131, 1858, 91, 1, 70, "ExampleText",
 CellID->25187625],
Cell[CellGroupData[{
Cell[79247, 1863, 570, 19, 70, "Input",
 CellID->13698880],
Cell[79820, 1884, 289, 10, 36, "Output",
 CellID->714633937]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[80158, 1900, 218, 7, 70, "ExampleSection",
 CellID->31773],
Cell[CellGroupData[{
Cell[80401, 1911, 224, 7, 70, "ExampleSubsection",
 CellID->24623],
Cell[80628, 1920, 168, 4, 70, "ExampleText",
 CellID->57022563],
Cell[CellGroupData[{
Cell[80821, 1928, 505, 16, 70, "Input",
 CellID->79976998],
Cell[81329, 1946, 535, 20, 36, "Output",
 CellID->79368049]
}, Open  ]],
Cell[81879, 1969, 95, 1, 70, "ExampleText",
 CellID->623917411],
Cell[CellGroupData[{
Cell[81999, 1974, 201, 6, 70, "Input",
 CellID->16398935],
Cell[82203, 1982, 231, 7, 36, "Output",
 CellID->270490817]
}, Open  ]],
Cell[CellGroupData[{
Cell[82471, 1994, 123, 4, 70, "Input",
 CellID->904431693],
Cell[82597, 2000, 223, 7, 36, "Output",
 CellID->164769456]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[82881, 2014, 221, 7, 70, "ExampleSection",
 CellID->907],
Cell[83105, 2023, 224, 9, 70, "ExampleText",
 CellID->2666],
Cell[CellGroupData[{
Cell[83354, 2036, 419, 14, 70, "Input",
 CellID->17380],
Cell[83776, 2052, 641, 24, 52, "Output",
 CellID->228419607]
}, Open  ]],
Cell[CellGroupData[{
Cell[84454, 2081, 97, 3, 70, "Input",
 CellID->1666],
Cell[84554, 2086, 484, 18, 52, "Output",
 CellID->345849230]
}, Open  ]],
Cell[85053, 2107, 125, 3, 70, "ExampleDelimiter",
 CellID->156371133],
Cell[85181, 2112, 154, 5, 70, "ExampleText",
 CellID->374619358],
Cell[CellGroupData[{
Cell[85360, 2121, 994, 30, 70, "Input",
 CellID->60724522],
Cell[86357, 2153, 4789, 155, 216, "Output",
 CellID->380240317]
}, Open  ]],
Cell[91161, 2311, 124, 3, 70, "ExampleDelimiter",
 CellID->23812338],
Cell[91288, 2316, 205, 6, 70, "ExampleText",
 CellID->755679668],
Cell[CellGroupData[{
Cell[91518, 2326, 1147, 33, 70, "Input",
 CellID->394635355],
Cell[92668, 2361, 3011, 53, 70, "Output",
 Evaluatable->False,
 CellID->6486793]
}, Open  ]],
Cell[95694, 2417, 125, 3, 70, "ExampleDelimiter",
 CellID->325782462],
Cell[95822, 2422, 113, 3, 70, "ExampleText",
 CellID->475502054],
Cell[CellGroupData[{
Cell[95960, 2429, 207, 7, 70, "Input",
 CellID->456555764],
Cell[96170, 2438, 800, 25, 54, "Output",
 CellID->42357163]
}, Open  ]],
Cell[CellGroupData[{
Cell[97007, 2468, 408, 13, 70, "Input",
 CellID->499969362],
Cell[97418, 2483, 1069, 22, 70, "Output",
 Evaluatable->False,
 CellID->52628845]
}, Open  ]],
Cell[CellGroupData[{
Cell[98524, 2510, 202, 6, 70, "Input",
 CellID->404413096],
Cell[98729, 2518, 161, 5, 36, "Output",
 CellID->373488390]
}, Open  ]],
Cell[98905, 2526, 125, 3, 70, "ExampleDelimiter",
 CellID->955660304],
Cell[99033, 2531, 81, 1, 70, "ExampleText",
 CellID->846400484],
Cell[99117, 2534, 715, 26, 70, "Input",
 CellID->157133961],
Cell[99835, 2562, 176, 6, 70, "ExampleText",
 CellID->23398873],
Cell[CellGroupData[{
Cell[100036, 2572, 357, 12, 70, "Input",
 CellID->1427335],
Cell[100396, 2586, 2195, 79, 81, "Output",
 CellID->312572547]
}, Open  ]],
Cell[102606, 2668, 168, 6, 70, "ExampleText",
 CellID->135694689],
Cell[102777, 2676, 911, 27, 70, "Input",
 CellID->83262874],
Cell[CellGroupData[{
Cell[103713, 2707, 747, 23, 70, "Input",
 CellID->31712146],
Cell[104463, 2732, 3583, 63, 70, "Output",
 Evaluatable->False,
 CellID->89661972]
}, Open  ]],
Cell[108061, 2798, 168, 6, 70, "ExampleText",
 CellID->37608948],
Cell[CellGroupData[{
Cell[108254, 2808, 274, 9, 70, "Input",
 CellID->28513564],
Cell[108531, 2819, 6272, 107, 70, "Output",
 Evaluatable->False,
 CellID->120300620]
}, Open  ]],
Cell[114818, 2929, 89, 1, 70, "ExampleText",
 CellID->56965926],
Cell[CellGroupData[{
Cell[114932, 2934, 333, 11, 70, "Input",
 CellID->511407523],
Cell[115268, 2947, 23401, 388, 70, "Output",
 Evaluatable->False,
 CellID->810928204]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[138718, 3341, 232, 7, 70, "ExampleSection",
 CellID->9259],
Cell[138953, 3350, 116, 3, 70, "ExampleText",
 CellID->166409219],
Cell[139072, 3355, 205, 7, 70, "Input",
 CellID->420782050],
Cell[CellGroupData[{
Cell[139302, 3366, 143, 4, 70, "Input",
 CellID->4436254],
Cell[139448, 3372, 900, 32, 51, "Output",
 CellID->232407042]
}, Open  ]],
Cell[CellGroupData[{
Cell[140385, 3409, 165, 5, 70, "Input",
 CellID->163355998],
Cell[140553, 3416, 247, 8, 36, "Output",
 CellID->712179932]
}, Open  ]],
Cell[140815, 3427, 125, 3, 70, "ExampleDelimiter",
 CellID->407943899],
Cell[140943, 3432, 235, 7, 70, "ExampleText",
 CellID->473096014],
Cell[141181, 3441, 714, 26, 70, "Input",
 CellID->34202644],
Cell[141898, 3469, 235, 8, 70, "ExampleText",
 CellID->205322194],
Cell[CellGroupData[{
Cell[142158, 3481, 386, 12, 70, "Input",
 CellID->21587785],
Cell[142547, 3495, 439, 16, 36, "Output",
 CellID->447784873]
}, Open  ]],
Cell[143001, 3514, 126, 3, 70, "ExampleText",
 CellID->937810632],
Cell[143130, 3519, 623, 21, 70, "Input",
 CellID->55371509],
Cell[143756, 3542, 93, 1, 70, "ExampleText",
 CellID->185745966],
Cell[CellGroupData[{
Cell[143874, 3547, 716, 24, 70, "Input",
 CellID->526053930],
Cell[144593, 3573, 180, 5, 36, "Output",
 CellID->214460321]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[144822, 3584, 226, 7, 70, "ExampleSection",
 CellID->17419],
Cell[145051, 3593, 60, 1, 70, "ExampleText",
 CellID->450115561],
Cell[145114, 3596, 272, 10, 70, "Input",
 CellID->102724351],
Cell[145389, 3608, 116, 3, 70, "ExampleText",
 CellID->644646945],
Cell[145508, 3613, 359, 13, 70, "Input",
 CellID->195197814],
Cell[145870, 3628, 117, 3, 70, "ExampleText",
 CellID->163770],
Cell[CellGroupData[{
Cell[146012, 3635, 539, 16, 70, "Input",
 CellID->7635386],
Cell[146554, 3653, 2888, 51, 70, "Output",
 Evaluatable->False,
 CellID->772094833]
}, Open  ]],
Cell[149457, 3707, 259, 7, 70, "ExampleText",
 CellID->54978621],
Cell[CellGroupData[{
Cell[149741, 3718, 114, 3, 70, "Input",
 CellID->134926756],
Cell[149858, 3723, 386, 15, 36, "Output",
 CellID->21499659]
}, Open  ]],
Cell[CellGroupData[{
Cell[150281, 3743, 401, 13, 70, "Input",
 CellID->226607093],
Cell[150685, 3758, 3181, 56, 70, "Output",
 Evaluatable->False,
 CellID->133956055]
}, Open  ]],
Cell[153881, 3817, 125, 3, 70, "ExampleDelimiter",
 CellID->466796740],
Cell[154009, 3822, 140, 4, 70, "ExampleText",
 CellID->154411273],
Cell[CellGroupData[{
Cell[154174, 3830, 424, 14, 70, "Input",
 CellID->554349834],
Cell[154601, 3846, 223, 5, 70, "Message",
 CellID->510286077],
Cell[154827, 3853, 513, 17, 36, "Output",
 CellID->146878770]
}, Open  ]],
Cell[155355, 3873, 130, 4, 70, "ExampleText",
 CellID->702970728],
Cell[CellGroupData[{
Cell[155510, 3881, 462, 16, 70, "Input",
 CellID->76220879],
Cell[155975, 3899, 260, 9, 36, "Output",
 CellID->1976488]
}, Open  ]],
Cell[156250, 3911, 125, 3, 70, "ExampleDelimiter",
 CellID->815823841],
Cell[156378, 3916, 96, 1, 70, "ExampleText",
 CellID->19854453],
Cell[156477, 3919, 505, 19, 70, "Input",
 CellID->591039839],
Cell[156985, 3940, 140, 4, 70, "ExampleText",
 CellID->533158672],
Cell[CellGroupData[{
Cell[157150, 3948, 193, 6, 70, "Input",
 CellID->65706371],
Cell[157346, 3956, 571, 10, 70, "Message",
 CellID->91438957],
Cell[157920, 3968, 669, 23, 36, "Output",
 CellID->270381017]
}, Open  ]],
Cell[158604, 3994, 119, 3, 70, "ExampleText",
 CellID->409938236],
Cell[CellGroupData[{
Cell[158748, 4001, 359, 12, 70, "Input",
 CellID->79009712],
Cell[159110, 4015, 387, 15, 51, "Output",
 CellID->102546439]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[159558, 4037, 311, 9, 70, "SeeAlsoSection",
 CellID->32480],
Cell[159872, 4048, 1232, 44, 70, "SeeAlso",
 CellID->30505]
}, Open  ]],
Cell[CellGroupData[{
Cell[161141, 4097, 313, 9, 70, "TutorialsSection",
 CellID->7675],
Cell[161457, 4108, 180, 4, 70, "Tutorials",
 CellID->20145]
}, Open  ]],
Cell[CellGroupData[{
Cell[161674, 4117, 305, 8, 70, "RelatedLinksSection"],
Cell[161982, 4127, 388, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[162407, 4144, 318, 9, 70, "MoreAboutSection",
 CellID->45608450],
Cell[162728, 4155, 185, 4, 70, "MoreAbout",
 CellID->95810050],
Cell[162916, 4161, 179, 3, 70, "MoreAbout",
 CellID->206106957]
}, Open  ]],
Cell[163110, 4167, 50, 0, 70, "History"],
Cell[163163, 4169, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

