(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    790671,      20965]
NotebookOptionsPosition[    765696,      20147]
NotebookOutlinePosition[    769762,      20222]
CellTagsIndexPosition[    769674,      20217]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Mathematica Internet Connectivity" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/InternetConnectivity"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Mathematica Internet Connectivity\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ElementData" :> 
          Documentation`HelpLookup["paclet:ref/ElementData"], "ChemicalData" :> 
          Documentation`HelpLookup["paclet:ref/ChemicalData"], "ParticleData" :> 
          Documentation`HelpLookup["paclet:ref/ParticleData"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"ElementData\"\>", 
       2->"\<\"ChemicalData\"\>", 
       3->"\<\"ParticleData\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Integrated Data Sources" :> 
          Documentation`HelpLookup["paclet:guide/DataCollections"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Integrated Data Sources\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["IsotopeData", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["IsotopeData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/IsotopeData"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["Z", "TI"], ",", 
          StyleBox["A", "TI"]}], "}"}], ",", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the value of the specified property for the \
isotope with atomic number ",
     Cell[BoxData[
      StyleBox["Z", "TI"]], "InlineFormula"],
     " and mass number ",
     Cell[BoxData[
      StyleBox["A", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["IsotopeData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/IsotopeData"], "[", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the value of the property for the named \
isotope."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "Isotope names can be specified by the corresponding element name or \
abbreviation, followed by a mass number ",
 Cell[BoxData[
  StyleBox["A", "TI"]], "InlineFormula"],
 " (e.g. ",
 Cell[BoxData["\"\<Sodium23\>\""], "InlineFormula"],
 " or ",
 Cell[BoxData["\"\<Na23\>\""], "InlineFormula"],
 "). The form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", 
    StyleBox["A", "TI"]}], "}"}]], "InlineFormula"],
 " can also be used."
}], "Notes",
 CellID->823808400],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " gives a list of all known isotopes."
}], "Notes",
 CellID->485978669],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", 
   RowBox[{"{", 
    RowBox[{
     StyleBox["Z", "TI"], ",", 
     StyleBox["A", "TI"]}], "}"}], "]"}]], "InlineFormula"],
 "gives the isotope name."
}], "Notes",
 CellID->14574448],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", "\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", "]"}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", 
   StyleBox["Z", "TI"], "]"}]], "InlineFormula"],
 " gives a list of all known isotopes of the element with the specified name \
or atomic number."
}], "Notes",
 CellID->691884323],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", 
   RowBox[{"{", 
    RowBox[{
     StyleBox["Z", "TI"], ",", 
     StyleBox["A", "TI"]}], "}"}], "]"}]], "InlineFormula"],
 "gives ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Missing",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Missing"], "[", "\"\<Unknown\>\"", "]"}]], 
  "InlineFormula"],
 " when no isotope with the specified ",
 Cell[BoxData[
  StyleBox["Z", "TI"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["A", "TI"]], "InlineFormula"],
 " is known."
}], "Notes",
 CellID->530731578],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", "\"\<Properties\>\"", "]"}]], 
  "InlineFormula"],
 " gives a list of all properties available for the isotopes."
}], "Notes",
 CellID->54231339],

Cell[TextData[{
 "Properties that do not apply or are not known in a particular case are \
indicated by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Missing",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Missing"], "[", 
   StyleBox["\[Ellipsis]", "TR"], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->48425469],

Cell[TextData[{
 "When not noted otherwise, all relevant quantities are given in SI (MKS) \
units. ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"prop\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
 " gives the numerical value; ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"prop\", \"TI\"]\)\>\"", ",", "\"\<Units\>\""}], "]"}]], 
  "InlineFormula"],
 " gives the units used."
}], "Notes",
 CellID->193117324],

Cell["\<\
Numerical values that are known only to a certain precision are typically \
tagged with their precision, and given in arbitrary-precision form.\
\>", "Notes",
 CellID->286491208],

Cell["Basic isotope properties include:", "Notes",
 CellID->156536174],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<AtomicMass\>\"", Cell[
    "mass of isotope in atomic mass units", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<AtomicNumber\>\"", Cell[TextData[{
     "atomic number ",
     Cell[BoxData[
      FormBox["Z", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<BindingEnergy\>\"", Cell[
    "binding energy", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<IsotopeAbundance\>\"", Cell[
    "relative isotope abundance", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<MassExcess\>\"", Cell[TextData[{
     "mass excess ",
     Cell[BoxData[
      FormBox["\[CapitalDelta]", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<MassNumber\>\"", Cell[TextData[{
     "mass number ",
     Cell[BoxData[
      FormBox["A", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<NeutronNumber\>\"", Cell[
    "number of neutrons", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<QuantumStatistics\>\"", Cell[
    "+1 for bosons, -1 for fermions", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True}}}},
 CellID->120063746],

Cell["Quantum number and structural properties include: ", "Notes",
 CellID->729383543],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<MagneticMoment\>\"", Cell[
    "magnetic moment in nuclear magnetons", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Parity\>\"", Cell[TextData[{
     "parity ",
     Cell[BoxData[
      FormBox["\[Pi]", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<QuadrupoleMoment\>\"", Cell[
    "electric quadrupole moment in barns", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Spin\>\"", Cell[TextData[{
     "spin ",
     Cell[BoxData[
      FormBox["J", TraditionalForm]], "InlineMath"]
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True}}}},
 CellID->912550125],

Cell["Decay-related properties include:", "Notes",
 CellID->1093015083],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<BranchingRatios\>\"", Cell[
    "branching ratio of each decay mode", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DaughterNuclides\>\"", Cell[
    "final daughter nuclide of each mode", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DecayEnergies\>\"", Cell[TextData[{
     "decay energy ",
     Cell[BoxData[
      FormBox["Q", TraditionalForm]], "InlineMath"],
     " of each mode in keV"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DecayModes\>\"", Cell[
    "list of observed decay modes", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DecayModeSymbols\>\"", Cell[
    "standard symbols for observed decay modes", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DecayProducts\>\"", Cell[
    "list of all final decay products and branching ratios", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HalfLife\>\"", Cell[TextData[{
     "decay half-life ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["T", 
        RowBox[{"1", "/", "2"}]], TraditionalForm]], "InlineMath"],
     " in seconds"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Lifetime\>\"", Cell[TextData[{
     "decay lifetime ",
     Cell[BoxData[
      StyleBox["\[Tau]", "TR"]], "InlineFormula"],
     " in seconds"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Stable\>\"", Cell[
    "whether the isotope is stable", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Width\>\"", Cell[TextData[{
     "decay width ",
     Cell[BoxData[
      StyleBox["\[Gamma]", "TR"]], "InlineFormula"],
     " in MeV"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True}}}},
 CellID->2049227648],

Cell[TextData[{
 "Examples of decay mode designations include: ",
 Cell[BoxData["\"\<AlphaDecay\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<BetaDecay\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<BetaPlusDecay\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<ProtonEmission\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<NeutronEmission\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<SpontaneousFission\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<ElectronCapture\>\""], "InlineFormula"],
 ".",
 " ",
 "\"Delayed\" decay mode designations indicate the presence of an \
intermediate excited state."
}], "Notes",
 CellID->215660396],

Cell["\<\
Ordinary decay-related properties include only final ground-state decay \
products. \
\>", "Notes",
 CellID->506275057],

Cell["Properties of excited states include:", "Notes",
 CellID->270216217],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<ExcitedStateEnergies\>\"", Cell[
    "list of energies in keV", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExcitedStateHalfLives\>\"", Cell[
    "list of decay half-lives in seconds", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExcitedStateLifetimes\>\"", Cell[
    "list of decay lifetimes in seconds", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExcitedStateParities\>\"", Cell[
    "list of parities", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExcitedStateSpins\>\"", Cell[
    "list of spins", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExcitedStateWidths\>\"", Cell[
    "list of decay widths in MeV", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True, True}}}},
 CellID->981007280],

Cell["Names-related properties include:", "Notes",
 CellID->204992243],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<FullSymbol\>\"", Cell[
    "full abbreviation", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Name\>\"", Cell[TextData[{
     "English name (e.g. ",
     Cell[BoxData["\"\<hydrogen 3\>\""], "InlineFormula"],
     ")"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StandardName\>\"", Cell[TextData[{
     "standard ",
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " name (e.g. ",
     Cell[BoxData["\"\<Hydrogen3\>\""], "InlineFormula"],
     ")"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Symbol\>\"", Cell[
    "standard abbreviation", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True}}}},
 CellID->711624400],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", "\"\<\!\(\*
StyleBox[\"class\", \"TI\"]\)\>\"", "]"}]], "InlineFormula"],
 " gives a list of isotopes in the specified class. ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", 
   RowBox[{
    StyleBox["name", "TI"], ",", "\"\<\!\(\*
StyleBox[\"class\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " depending on whether the element corresponding to ",
 Cell[BoxData[
  StyleBox["name", "TI"]], "InlineFormula"],
 " is in the specified class."
}], "Notes",
 CellID->482514198],

Cell["Classes of isotopes include:", "Notes",
 CellID->91622357],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<AlphaDecay\>\"", Cell[TextData[{
     "decays via ",
     Cell[BoxData[
      FormBox["\[Alpha]", TraditionalForm]], "InlineMath"],
     " decay"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<BetaDecay\>\"", Cell[TextData[{
     "decays via ",
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Beta]", "-"], TraditionalForm]], "InlineMath"],
     "decay"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<BetaPlusDecay\>\"", Cell[TextData[{
     "decays via ",
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Beta]", "+"], TraditionalForm]], "InlineMath"],
     "decay"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Boson\>\"", Cell[
    "obeys Bose-Einstein quantum statistics", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ElectronCapture\>\"", Cell[
    "decays via electron capture", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Fermion\>\"", Cell[
    "obeys Fermi-Dirac quantum statistics", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<NeutronEmission\>\"", Cell[
    "decays via neutron emission", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ProtonEmission\>\"", Cell[
    "decays via proton emission", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<SpontaneousFission\>\"", Cell[
    "decays via spontaneous fission", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Stable\>\"", Cell[
    "stable with respect to radioactive decay", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Unstable\>\"", Cell[
    "unstable with respect to radioactive decay", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, 
      True}}}},
 CellID->1173142],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["IsotopeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IsotopeData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"ann\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
 " gives various annotations associated with a property. Typical annotations \
include:"
}], "Notes",
 CellID->276644756],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Description\>\"", Cell[TextData[{
     Cell[BoxData[Cell["", "TableText"]]],
     "short textual description of the property"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Interval\>\"", Cell[TextData[{
     "uncertainty range given by ",
     Cell[BoxData[
      RowBox[{
       ButtonBox["Interval",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Interval"], "[", 
       StyleBox["\[Ellipsis]", "TR"], "]"}]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<LongDescription\>\"", Cell[
    "longer textual description of the property ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Note\>\"", Cell[
    "additional information about the property", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Units\>\"", Cell[
    "units in which the value is given", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsName\>\"", Cell[
    "English name for the units used", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsNotation\>\"", Cell[
    "notation for the units used", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsStandardName\>\"", Cell[
    TextData[{
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " standard name for the units used"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True}}}},
 CellID->524183967]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->2939395],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(4)", "ExampleCount"]
}], "ExampleSection",
 CellID->16842629],

Cell["Find the atomic mass of uranium-235:", "ExampleText",
 CellID->994896104],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Uranium235\>\"", ",", "\"\<AtomicMass\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->911144386],

Cell[BoxData["235.043929918`11.999999999999998"], "Output",
 ImageSize->{95, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->239511853]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->51359002],

Cell["Find the magnetic moment of tritium:", "ExampleText",
 CellID->235209921],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Tritium\>\"", ",", "\"\<MagneticMoment\>\""}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->1803231767],

Cell[BoxData["2.97896248`9.000000000000002"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1091358893]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->255582612],

Cell["Find the decay modes of lead-181:", "ExampleText",
 CellID->1348723483],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Lead181\>\"", ",", "\"\<DecayModes\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1305684696],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AlphaEmission\"\>", ",", "\<\"BetaPlusDecay\"\>"}], 
  "}"}]], "Output",
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->811085624]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->125102124],

Cell["Plot the excited state energies of oxygen-15:", "ExampleText",
 CellID->751030269],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"IsotopeData", "[", 
   RowBox[{"\"\<Oxygen15\>\"", ",", "\"\<ExcitedStateEnergies\>\""}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1549403507],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdz8lrk1EYhfFL3QgOKAoaKFIHRESL2jpWm+tQxzqlFVqLG7FREMdVVnIX
KhU3dqEYcS24ELoQEVH4LNWqTRwREQtZCIKrqn+BNec5GwPh4ZeTN+RbePxc
ob8hhNA9+f5XvX7l68k/jHwQi5OvHyV7aqxWKpXOpsd4VrxTLpdz157gufXv
F8fs+bo//RQ31psuP8NNanOGF2mf8xwvUc/YS//zsvr/mX7dXq5eHcYr1AG7
Wb8/Yq9Sa/ZqNTeC16gL7Bbdn7Vb1ZK9Vr1ir9P3Cy/wevWYvUH7gL1RHrU3
qT/tNu23XuLN8m17izpkt2v/bufVP3ZUF4+qCRdx2Kr7Ie9yqHnfpn3CO57y
in273IeTHO5630Ffs8tpNg4d6invcjrvfafa8YYd9+KwS73hHQ97361OeMfz
xtj3qCdxwg+871Vr3nFDhX2f/m/ESQ6PvHeqLVV23I/DfvWSd7zyLfsB9ShO
uOT9oDroHY97P6TOfKdG3IsTvogzfB+Hw3q+ae+5x3044Xs4k8OMD9wXtOdw
xIM44SrOcNtH7rvkmzjixk/c4ws4k8NvHLp5/s/c4/Yv3OMTX7nHPd+4P6IW
xuNfvu3Wjw==
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 5000.},
  PlotRange->{{0., 84.}, {5183., 29000.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1837375059]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(15)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  InterpretationBox[Cell[TextData[{
    "Names and Classes",
    "\[NonBreakingSpace]\[NonBreakingSpace]",
    Cell["(8)", "ExampleCount"]
   }], "ExampleSubsection"],
   $Line = 0; Null], "  "}]], "ExampleSubsection",
 CellID->14003343],

Cell["\<\
Obtain a list of all the isotopes sorted by atomic number and mass number:\
\>", "ExampleText",
 CellID->56694407],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"IsotopeData", "[", "]"}], "//", "Short"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->174295882],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"\<\"Neutron\"\>", ",", "\<\"Hydrogen1\"\>", ",", 
    RowBox[{"\[LeftSkeleton]", "3178", "\[RightSkeleton]"}], 
    ",", "\<\"Ununseptium292\"\>", ",", "\<\"Ununoctium293\"\>"}], "}"}],
  Short]], "Output",
 ImageSize->{420, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//Short=",
 CellID->2095256400]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->49007095],

Cell[BoxData["3182"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->537471294]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->498555886],

Cell["Get a list of known isotopes for an element:", "ExampleText",
 CellID->309303337],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", "\"\<Potassium\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1617796782],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Potassium32\"\>", ",", "\<\"Potassium33\"\>", 
   ",", "\<\"Potassium34\"\>", ",", "\<\"Potassium35\"\>", 
   ",", "\<\"Potassium36\"\>", ",", "\<\"Potassium37\"\>", 
   ",", "\<\"Potassium38\"\>", ",", "\<\"Potassium39\"\>", 
   ",", "\<\"Potassium40\"\>", ",", "\<\"Potassium41\"\>", 
   ",", "\<\"Potassium42\"\>", ",", "\<\"Potassium43\"\>", 
   ",", "\<\"Potassium44\"\>", ",", "\<\"Potassium45\"\>", 
   ",", "\<\"Potassium46\"\>", ",", "\<\"Potassium47\"\>", 
   ",", "\<\"Potassium48\"\>", ",", "\<\"Potassium49\"\>", 
   ",", "\<\"Potassium50\"\>", ",", "\<\"Potassium51\"\>", 
   ",", "\<\"Potassium52\"\>", ",", "\<\"Potassium53\"\>", 
   ",", "\<\"Potassium54\"\>", ",", "\<\"Potassium55\"\>"}], "}"}]], "Output",\

 ImageSize->{536, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->963818976]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->365236494],

Cell["An isotope can be specified in several specific ways:", "ExampleText",
 CellID->492932380],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Uranium238\>\"", ",", "\"\<MassNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->374014864],

Cell[BoxData["238"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->436380311]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<U238\>\"", ",", "\"\<StandardName\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7393140],

Cell[BoxData["\<\"Uranium238\"\>"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->587232472]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<Uranium\>\"", ",", "238"}], "}"}], ",", 
   "\"\<AtomicNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->188296305],

Cell[BoxData["92"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1928798939]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"92", ",", "238"}], "}"}], ",", "\"\<NeutronNumber\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->399684212],

Cell[BoxData["146"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1058116308]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31217455],

Cell["Find the English name of an isotope:", "ExampleText",
 CellID->277763820],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Hydrogen2\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->221172202],

Cell[BoxData["\<\"deuterium\"\>"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->448707749]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Carbon12\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->33052035],

Cell[BoxData["\<\"carbon 12\"\>"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1616052826]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->86736583],

Cell["Get the standard symbol for an isotope:", "ExampleText",
 CellID->361353193],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Gold197\>\"", ",", "\"\<Symbol\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1820430401],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1UE0LwjAMTT+GiGDnzdPYr9GTP6IMYTuIMvf/axqXNYSCJwtN8rL3kr7d
4jLeH3GZhthf5/gap+HdX54ztpwBgA5vOAPWCSu6XUqYZQwYHX4CT3A9FoHN
hStdQ2WbY5Nn8WE+uK0LlnvEbxjVdXAS6m3pH7jsSmYgsfTplRen8J6o2nH9
HX7ddKhq+B16QyjsnZoo5krF8adC7s64/Wblm/+KKwzzAUqnE1A=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{30, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->485885536]
}, Open  ]],

Cell["Including atomic number and neutron number:", "ExampleText",
 CellID->1008028582],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Gold197\>\"", ",", "\"\<FullSymbol\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1237181837],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1VEsKwjAQTfNBdGWtoIhgvUIPoSsPUYrQLkSpvX+cpJMmaYORgoFkMtN5
L/Ojt7Kr74+ya6oyv7blq26qd355tmBiCSGkgL3bE7hLuOldSAlyei7VyZWH
WQyUFKQASZh20kvZKagUJWLZgO2dHT0NfRfuWwS5VNBJkJNiRJsZXCs/NxO3
w7KOZODnuwjzCddiCAi3lbN18+o5IyMVAUcsm77vYLwsEEExLFf2mWXY7zBf
puXWYR3i/4OvW3M6eI0n0a+viPTxoJ2DFf+h+8cg2kwmH9lz6x3tzJjzFMN+
neYebaLm0s6Igp1l+A9Akg+/sghE\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{50, 23},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->388382199]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->255481931],

Cell["Find the list of isotope classes:", "ExampleText",
 CellID->208354654],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"IsotopeData", "[", "\"\<Classes\>\"", "]"}], "//", 
  "Short"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28830905],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"\<\"AlphaEmission\"\>", ",", "\<\"BetaDecay\"\>", ",", 
    RowBox[{"\[LeftSkeleton]", "43", "\[RightSkeleton]"}], 
    ",", "\<\"TwoProtonEmission\"\>", ",", "\<\"Unstable\"\>"}], "}"}],
  Short]], "Output",
 ImageSize->{434, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//Short=",
 CellID->588247246]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->237929120],

Cell["Find the list of isotopes belonging to a class:", "ExampleText",
 CellID->40277343],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"IsotopeData", "[", "\"\<Stable\>\"", "]"}], "//", 
  "Short"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->795584317],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"\<\"Hydrogen1\"\>", ",", "\<\"Hydrogen2\"\>", 
    ",", "\<\"Helium3\"\>", ",", 
    RowBox[{"\[LeftSkeleton]", "250", "\[RightSkeleton]"}], 
    ",", "\<\"Lead206\"\>", ",", "\<\"Lead207\"\>", ",", "\<\"Lead208\"\>"}], 
   "}"}],
  Short]], "Output",
 ImageSize->{457, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//Short=",
 CellID->1804236381]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"IsotopeData", "[", "\"\<BetaDecay\>\"", "]"}], "//", 
  "Short"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23186614],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"\<\"Neutron\"\>", ",", "\<\"Hydrogen3\"\>", ",", 
    RowBox[{"\[LeftSkeleton]", "1192", "\[RightSkeleton]"}], 
    ",", "\<\"Mendelevium258\"\>", ",", "\<\"Mendelevium260\"\>"}], "}"}],
  Short]], "Output",
 ImageSize->{427, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//Short=",
 CellID->1331759827]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->128501363],

Cell["Test whether an isotope belongs to a class:", "ExampleText",
 CellID->278486034],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<Oxygen\>\"", ",", "20"}], "}"}], ",", 
   "\"\<AlphaEmission\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16562659],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->950269553]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties and Annotations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1961895733],

Cell["Get a list of properties for a particular isotope:", "ExampleText",
 CellID->63407444],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Short", "[", 
  RowBox[{
   RowBox[{"IsotopeData", "[", 
    RowBox[{"\"\<Helium7\>\"", ",", "\"\<Properties\>\""}], "]"}], ",", "4"}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->180587304],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"\<\"AtomicMass\"\>", ",", "\<\"AtomicNumber\"\>", 
    ",", "\<\"BindingEnergy\"\>", ",", "\<\"BranchingRatios\"\>", 
    ",", "\<\"DaughterNuclides\"\>", ",", "\<\"DecayEnergies\"\>", 
    ",", "\<\"DecayModes\"\>", ",", "\<\"DecayModeSymbols\"\>", ",", 
    RowBox[{"\[LeftSkeleton]", "17", "\[RightSkeleton]"}], 
    ",", "\<\"Parity\"\>", ",", "\<\"QuadrupoleMoment\"\>", 
    ",", "\<\"QuantumStatistics\"\>", ",", "\<\"Spin\"\>", 
    ",", "\<\"Stable\"\>", ",", "\<\"Symbol\"\>", ",", "\<\"Width\"\>"}], 
   "}"}],
  Short[#, 4]& ]], "Output",
 ImageSize->{506, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//Short=",
 CellID->1812843355]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->575071077],

Cell["Get a short textual description of a property:", "ExampleText",
 CellID->298397443],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"80", ",", "192"}], "}"}], ",", "\"\<BranchingRatios\>\"", ",", 
   "\"\<Description\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->44910077],

Cell[BoxData["\<\"branching ratios\"\>"], "Output",
 ImageSize->{116, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2107767496]
}, Open  ]],

Cell["Get a longer textual description:", "ExampleText",
 CellID->826150189],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"80", ",", "192"}], "}"}], ",", "\"\<BranchingRatios\>\"", ",", 
   "\"\<LongDescription\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17124444],

Cell[BoxData["\<\"branching ratio of each decay mode\"\>"], "Output",
 ImageSize->{242, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->358064360]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->336008823],

Cell["Find the units in which a property value is given:", "ExampleText",
 CellID->725709153],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<N25\>\"", ",", "\"\<Width\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1229972329],

Cell[BoxData["1.75000000000001`2.9999999999999973*^-15"], "Output",
 ImageSize->{73, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->738519626]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<N25\>\"", ",", "\"\<Width\>\"", ",", "\"\<Units\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->560816081],

Cell[BoxData["\<\"MegaelectronVolts\"\>"], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1991358169]
}, Open  ]],

Cell["Get units in proper textual form:", "ExampleText",
 CellID->98478505],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<N25\>\"", ",", "\"\<Width\>\"", ",", "\"\<UnitsName\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1250211495],

Cell[BoxData["\<\"megaelectron volts\"\>"], "Output",
 ImageSize->{130, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->224107888]
}, Open  ]],

Cell["Get units as notation:", "ExampleText",
 CellID->433423136],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<N25\>\"", ",", "\"\<Width\>\"", ",", "\"\<UnitsNotation\>\""}],
   "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->826445282],

Cell[BoxData["\<\"MeV\"\>"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->2042440425]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->59265178],

Cell[TextData[{
 "Uncertainty is represented using ",
 Cell[BoxData[
  ButtonBox["Interval",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Interval"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->244694112],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Lithium7\>\"", ",", "\"\<AtomicMass\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->493396966],

Cell[BoxData["7.016004548`10.000000000000004"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1917609327]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{
  "\"\<Lithium7\>\"", ",", "\"\<AtomicMass\>\"", ",", "\"\<Interval\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->431132459],

Cell[BoxData[
 RowBox[{"Interval", "[", 
  RowBox[{"{", 
   RowBox[{
   "7.0160044637671693563461303711`10.999999999985588", ",", 
    "7.01600463223283064365386962891`11.000000000014413"}], "}"}], 
  "]"}]], "Output",
 ImageSize->{268, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1972868360]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12996297],

Cell["Find additional notes about properties:", "ExampleText",
 CellID->17219685],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Erbium175\>\"", ",", "\"\<HalfLife\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->502847698],

Cell[BoxData["72.`1.9999999999999998"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->851016840]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Erbium175\>\"", ",", "\"\<HalfLife\>\"", ",", "\"\<Note\>\""}],
   "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->417695890],

Cell[BoxData["\<\"Value and uncertainty are estimates based on systematic \
trends.\"\>"], "Output",
 ImageSize->{445, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->798940860]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Property Values",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->29302194],

Cell[TextData[{
 "A property value can be any valid ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expression:"
}], "ExampleText",
 CellID->553589695],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Pb211\>\"", ",", "\"\<BindingEnergy\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->529027222],

Cell[BoxData["7.817`3.9999999999999996"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1731272976]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Helium4\>\"", ",", "\"\<Lifetime\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->609521478],

Cell[BoxData["\[Infinity]"], "Output",
 ImageSize->{12, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1248974429]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Radon208\>\"", ",", "\"\<DecayModes\>\""}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1563433475],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AlphaEmission\"\>", ",", "\<\"BetaPlusDecay\"\>"}], 
  "}"}]], "Output",
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->2008062425]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->180621357],

Cell["\<\
Values are listed in the same order for all excited state and decay \
properties: \
\>", "ExampleText",
 CellID->1207501607],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Cesium132\>\"", ",", "\"\<DecayModes\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->759044699],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"BetaPlusDecay\"\>", ",", "\<\"BetaDecay\"\>"}], 
  "}"}]], "Output",
 ImageSize->{184, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1799664370]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Cesium132\>\"", ",", "\"\<DecayModeSymbols\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->873970007],

Cell[GraphicsData["CompressedBitmap", "\<\
eJytUtsKwjAMzdZu4mUq+uLj/qH/oE9+RBnC9iDK3P/PpBVb0igTHDSX0+Tk
kPVsh/ZytUPX2PrU23vbNY/6eOsRUhkAGDybA2A8YuSOGUf0mqym3H0YVORU
QEqWr1iuXJuzxRv1VI4rRz9jPXxGHjhKxoGWCheso4hymkRTlp9ZOZooncdK
k1s+3TPn0iaS3p2Icq3ZD1plRlmjyKjD3gD2CTJtjyrq8628okoqpmn8H7MW
mOkv0y0tcP3KCdfCHP+2twytv6AmspA9AZoP0OM=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{51, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->366698316]
}, Open  ]],

Cell["Get properties for the second decay mode:", "ExampleText",
 CellID->770047961],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"IsotopeData", "[", 
     RowBox[{"\"\<Cesium132\>\"", ",", "prop"}], "]"}], "[", 
    RowBox[{"[", "2", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"prop", ",", 
     RowBox[{"{", 
      RowBox[{
      "\"\<BranchingRatios\>\"", ",", "\"\<DaughterNuclides\>\"", ",", 
       "\"\<DecayEnergies\>\""}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->713275645],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.0187`2.9999999999999996", ",", "\<\"Barium132\"\>", ",", 
   "1278.92`5.999999999999999"}], "}"}]], "Output",
 ImageSize->{195, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1352624240]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->94476489],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["Precision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Precision"]], "InlineFormula"],
 " to find out the number of significant digits for a property:"
}], "ExampleText",
 CellID->5758993],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Gallium71\>\"", ",", "\"\<IsotopeAbundance\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->607479222],

Cell[BoxData["0.39892`5.000000000000002"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->293714538]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Precision", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->388541687],

Cell[BoxData["5.000000000000002`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1542819360]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->528642111],

Cell[TextData[{
 "A property that is unknown for an isotope has the value ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Missing",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Missing"], "[", "\"\<Unknown\>\"", "]"}]], 
  "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->9731214],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Cu76\>\"", ",", "\"\<Parity\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->78474866],

Cell[BoxData[
 RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}]], "Output",
 ImageSize->{116, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1447429642]
}, Open  ]],

Cell["A list of possible values due to uncertainty:", "ExampleText",
 CellID->1278380685],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Iron61\>\"", ",", "\"\<Spin\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->170284560],

Cell[BoxData[
 RowBox[{"Missing", "[", 
  RowBox[{"\<\"Uncertain\"\>", ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["3", "2"], ",", 
     FractionBox["5", "2"]}], "}"}]}], "]"}]], "Output",
 ImageSize->{189, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->255239878]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->598708091],

Cell["\<\
Use a string expression to find isotopes with the same mass number:\
\>", "ExampleText",
 CellID->210372128],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{
   RowBox[{"LetterCharacter", ".."}], "~~", "\"\<87\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->77964217],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Germanium87\"\>", ",", "\<\"Arsenic87\"\>", 
   ",", "\<\"Selenium87\"\>", ",", "\<\"Bromine87\"\>", 
   ",", "\<\"Krypton87\"\>", ",", "\<\"Rubidium87\"\>", 
   ",", "\<\"Strontium87\"\>", ",", "\<\"Yttrium87\"\>", 
   ",", "\<\"Zirconium87\"\>", ",", "\<\"Niobium87\"\>", 
   ",", "\<\"Molybdenum87\"\>", ",", "\<\"Technetium87\"\>", 
   ",", "\<\"Ruthenium87\"\>"}], "}"}]], "Output",
 ImageSize->{569, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1291786487]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->699285692],

Cell["Find the distribution of lifetimes:", "ExampleText",
 CellID->334957664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{"i", ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", 
          RowBox[{"-", "30"}], ",", "30"}], "}"}]}], "]"}], ",", 
      RowBox[{"BinCounts", "[", 
       RowBox[{
        RowBox[{"Cases", "[", 
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"Log", "[", 
             RowBox[{"10", ",", 
              RowBox[{"IsotopeData", "[", 
               RowBox[{"#", ",", "\"\<Lifetime\>\""}], "]"}]}], "]"}], "&"}], 
           "/@", 
           RowBox[{"IsotopeData", "[", "]"}]}], ",", "_Real"}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "30.5"}], ",", "30.5", ",", "1"}], "}"}]}], "]"}]}], 
     "}"}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}], ",", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", 
   RowBox[{"Axes", "\[Rule]", "None"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"FrameLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\!\(\*SubscriptBox[\(log\), \(10\)]\) \[Tau]\>\"", ",", "None"}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->977521728],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{300, 210},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{300, Automatic}},
 CellLabel->"Out[1]=",
 CellID->106759051]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->754209157],

Cell["Make an isotope chart of lifetimes:", "ExampleText",
 CellID->341858881],

Cell[BoxData[
 RowBox[{
  RowBox[{"g", "=", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "-", "z"}], ",", "z", ",", 
        RowBox[{"Log", "[", 
         RowBox[{"10", ",", 
          RowBox[{"IsotopeData", "[", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"z", ",", "a"}], "}"}], ",", "\"\<Lifetime\>\""}], 
           "]"}]}], "]"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"z", ",", "1", ",", "118"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"a", ",", 
        RowBox[{
         RowBox[{
          RowBox[{"IsotopeData", "[", 
           RowBox[{"#", ",", "\"\<MassNumber\>\""}], "]"}], "&"}], "/@", 
         RowBox[{"IsotopeData", "[", "z", "]"}]}]}], "}"}]}], "]"}], "//", 
    RowBox[{
     RowBox[{"Flatten", "[", 
      RowBox[{"#", ",", "1"}], "]"}], "&"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12496365],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListDensityPlot", "[", 
  RowBox[{"g", ",", 
   RowBox[{"FrameLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<N\>\"", ",", "\"\<Z\>\""}], "}"}]}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", "\"\<Rainbow\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->61226800],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 227},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{225, Automatic}},
 CellLabel->"Out[2]=",
 CellID->987068872]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1532279],

Cell["Make an isotope chart of binding energies:", "ExampleText",
 CellID->331248974],

Cell[BoxData[
 RowBox[{
  RowBox[{"g", "=", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "-", "z"}], ",", "z", ",", 
        RowBox[{"IsotopeData", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"z", ",", "a"}], "}"}], ",", "\"\<BindingEnergy\>\""}], 
         "]"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"z", ",", "1", ",", "118"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"a", ",", 
        RowBox[{
         RowBox[{
          RowBox[{"IsotopeData", "[", 
           RowBox[{"#", ",", "\"\<MassNumber\>\""}], "]"}], "&"}], "/@", 
         RowBox[{"IsotopeData", "[", "z", "]"}]}]}], "}"}]}], "]"}], "//", 
    RowBox[{
     RowBox[{"Flatten", "[", 
      RowBox[{"#", ",", "1"}], "]"}], "&"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->343971853],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListDensityPlot", "[", 
  RowBox[{"g", ",", 
   RowBox[{"FrameLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<N\>\"", ",", "\"\<Z\>\""}], "}"}]}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", "\"\<Rainbow\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->108955647],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 227},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{225, Automatic}},
 CellLabel->"Out[2]=",
 CellID->225439762]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->174543350],

Cell["Draw a decay network:", "ExampleText",
 CellID->197058215],

Cell[BoxData[
 RowBox[{
  RowBox[{"is", "=", 
   RowBox[{"Select", "[", 
    RowBox[{
     RowBox[{"IsotopeData", "[", "]"}], ",", 
     RowBox[{
      RowBox[{"100", "\[LessEqual]", 
       RowBox[{"IsotopeData", "[", 
        RowBox[{"#", ",", "\"\<MassNumber\>\""}], "]"}], "\[LessEqual]", 
       "102"}], "&"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->303285230],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LayeredGraphPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"IsotopeData", "[", 
            RowBox[{"#", ",", "\"\<Symbol\>\""}], "]"}], "&"}], "/@", 
          RowBox[{"(", 
           RowBox[{"i", "\[Rule]", "#"}], ")"}]}], ")"}], "&"}], "/@", 
       RowBox[{"IsotopeData", "[", 
        RowBox[{"i", ",", "\"\<DaughterNuclides\>\""}], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "is"}], "}"}]}], "]"}], "//", "Flatten"}], ",", 
   RowBox[{"DirectedEdges", "\[Rule]", "True"}], ",", 
   RowBox[{"VertexLabeling", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1336166279],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{346, 346},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{400, Automatic}},
 CellLabel->"Out[2]=",
 CellID->172123439]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22216897],

Cell["Plot magnetic moment against spin:", "ExampleText",
 CellID->780810098],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"IsotopeData", "[", 
       RowBox[{"#", ",", "\"\<Spin\>\""}], "]"}], ",", 
      RowBox[{"IsotopeData", "[", 
       RowBox[{"#", ",", "\"\<MagneticMoment\>\""}], "]"}]}], "}"}], "&"}], "/@",
    
   RowBox[{"IsotopeData", "[", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->75959812],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->121851749]
}, Open  ]],

Cell["Quadrupole moment vs. spin:", "ExampleText",
 CellID->827681361],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"IsotopeData", "[", 
       RowBox[{"#", ",", "\"\<Spin\>\""}], "]"}], ",", 
      RowBox[{"IsotopeData", "[", 
       RowBox[{"#", ",", "\"\<QuadrupoleMoment\>\""}], "]"}]}], "}"}], "&"}], 
   "/@", 
   RowBox[{"IsotopeData", "[", "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->403426493],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->555095442]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(8)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31316719],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["Table",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Table"]], "InlineFormula"],
 " to obtain the property values for a list of isotopes:"
}], "ExampleText",
 CellID->214644472],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"IsotopeData", "[", 
    RowBox[{"i", ",", "\"\<BindingEnergy\>\""}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", 
     RowBox[{"IsotopeData", "[", "\"\<Neon\>\"", "]"}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1117571144],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "6.082572`7.000000000000001", ",", "6.642825`7.000000000000001", ",", 
   "7.341282`7.000000000000001", ",", "7.567375`7.000000000000001", ",", 
   "8.03224`5.999999999999999", ",", "7.971713`7.000000000000001", ",", 
   "8.080465`7.000000000000001", ",", "7.955255`7.000000000000001", ",", 
   "7.993319`6.999999999999997", ",", "7.842701`7.000000000000001", ",", 
   "7.753891`7.000000000000001", ",", "7.519709`7.000000000000001", ",", 
   "7.390315`7.000000000000001", ",", "7.178857`6.999999999999997", ",", 
   "7.040467`7.000000000000001", ",", "6.824`3.9999999999999996", ",", 
   "6.662`3.9999999999999996", ",", "6.44`3.9999999999999996", ",", 
   "6.279`3.9999999999999996"}], "}"}]], "Output",
 ImageSize->{461, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1459170272]
}, Open  ]],

Cell[TextData[{
 "The same thing using ",
 Cell[BoxData[
  ButtonBox["Map",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Map"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->215160751],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"IsotopeData", "[", 
    RowBox[{"#", ",", "\"\<BindingEnergy\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"IsotopeData", "[", "\"\<Neon\>\"", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->540657586],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "6.082572`7.000000000000001", ",", "6.642825`7.000000000000001", ",", 
   "7.341282`7.000000000000001", ",", "7.567375`7.000000000000001", ",", 
   "8.03224`5.999999999999999", ",", "7.971713`7.000000000000001", ",", 
   "8.080465`7.000000000000001", ",", "7.955255`7.000000000000001", ",", 
   "7.993319`6.999999999999997", ",", "7.842701`7.000000000000001", ",", 
   "7.753891`7.000000000000001", ",", "7.519709`7.000000000000001", ",", 
   "7.390315`7.000000000000001", ",", "7.178857`6.999999999999997", ",", 
   "7.040467`7.000000000000001", ",", "6.824`3.9999999999999996", ",", 
   "6.662`3.9999999999999996", ",", "6.44`3.9999999999999996", ",", 
   "6.279`3.9999999999999996"}], "}"}]], "Output",
 ImageSize->{461, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->11512764]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->183120514],

Cell["Get values of multiple properties for a list of isotopes:", \
"ExampleText",
 CellID->316513968],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"vals", "=", 
  RowBox[{
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"IsotopeData", "[", 
       RowBox[{"#", ",", "prop"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"prop", ",", 
        RowBox[{"{", 
         RowBox[{
         "\"\<Symbol\>\"", ",", "\"\<HalfLife\>\"", ",", 
          "\"\<BindingEnergy\>\""}], "}"}]}], "}"}]}], "]"}], "&"}], "/@", 
   RowBox[{
   "IsotopeData", "[", "\"\<TwoNeutronEmission\>\"", "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->241955977],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{521, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->21086418]
}, Open  ]],

Cell[TextData[{
 "Display the data as a formatted table using ",
 Cell[BoxData[
  ButtonBox["Grid",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Grid"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Text",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Text"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->613113541],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Text", "[", 
  RowBox[{"Grid", "[", 
   RowBox[{
    RowBox[{"Prepend", "[", 
     RowBox[{"vals", ",", 
      RowBox[{"{", 
       RowBox[{
       "\"\<\>\"", ",", "\"\<half-life\>\"", ",", "\"\<binding energy\>\""}], 
       "}"}]}], "]"}], ",", 
    RowBox[{"Frame", "\[Rule]", "All"}], ",", 
    RowBox[{"Background", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"None", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"LightBlue", ",", "White"}], "}"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", "\[Rule]", "LightYellow"}], "}"}]}], "}"}]}], 
      "}"}]}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1367561668],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{186, 191},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->385921769]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->135815337],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["DeleteCases",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DeleteCases"]], "InlineFormula"],
 " to filter out non-missing data:"
}], "ExampleText",
 CellID->146537355],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Short", "[", 
  RowBox[{
   RowBox[{"DeleteCases", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"IsotopeData", "[", 
          RowBox[{"#", ",", "prop"}], "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"prop", ",", 
           RowBox[{"{", 
            RowBox[{"\"\<Symbol\>\"", ",", "\"\<MagneticMoment\>\""}], 
            "}"}]}], "}"}]}], "]"}], "&"}], "/@", 
      RowBox[{"IsotopeData", "[", "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"_", ",", "_Missing"}], "}"}]}], "]"}], ",", "4"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1805150387],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWkuPFDcQ7u2emX3M7vJYIogS0CoIidc9QCQ2gJScoiDlAFqQkmGFYA8I
BIgTfyfHcM3Pc8Y9rln315/tcvcMGkJLuz1ll+thl6ur/fVvk/cvn7+avD8+
muz/+nby5uXx0bv9X16/nTZVa0Wx9k9RFFcuFNPfZvqr/rPXp+lfUYyn/8tp
05/u/sjdb7n7nrvfDtBD714UW/Z/JRpMMQH6D6DvAH0K6J8y+wdAl1+QVSjl
d6AroH8GGuX9MP1nyRLaTyfkbgJ9dcnj9bNj4+wh0W7b77n7GPov1yrmtOM6
E5CCWq8uiF/n48it2IxdLpxfyzV07aVjtt0jYmPl+Cq3Q4UeenTp9cds3mY+
eDSTKbqQF+kNoDGGzjp5A+fzbFJml5U1kEnzxqwr/ULdexFd6dVZba087tZl
+YxchWcJ76383kg2KOnoYVCmt1eJn4MkB0aWeLFl4tE+ypZcgcR1w3PCGepr
a0dIW+Y64AoLN9tBX6MV7ehHmzaorF3gQrqio5TxSvfDyNv1c2sjO2vHzYXs
RMwopUIm0iOQic/qyPopW3eC/qNvmB3/z3oxUtdqPl2exrrRjhoUyhzreMO7
LZ3LdWNtJMXybx/ZmIn3oD+SgXutItqBK7Gs3bIqevXVxZDkEp53WVxIJcmt
t/GLtW9tSSvvsrjDWEllSF6jBCM6Qp8DOjtK4Yn5eWKlJNrHK665TxWMb3Kc
K1A9JlrQAt+78FygNxcj3uqlXAK69nsEo7xV8KsE/11lW6WNxkJCm7+nZ7p2
V06XLtL8+dPm3k3ampPTeStWldymbjG1nMhsPBGSFQRyfavSiTEjOmXldJUR
vmmcWohu/1xnUTp1cdtav7kouXacbWPTPP8Rm+Wsyj+Lsnf/vYM9S783J+99
Jciz9D6zP5VZT/a3Yedu3wVsYevTmqWT/DJXUlIt4tHZPtoaT5VBJz26CMDd
9ARoPEmv+dch+vw18NbkRsJztt+xJrBybnaUg+1oT/4esT8PNTOUUTVe13in
nHHVTAVkdZstQZvQkkPTRJ/ofo3Q10wTncodf31B45toGL7x/Ju51vdJP4vU
B0q+H2ur7M9P8EvyqNj/AubjL6Clv7aQn3p59DHQGGfPEv04/mCFtaKUR0Cn
0DKMmRQKSJ/3Gfpvq72y6/2YSGNn3qdNHK0LoXsh+XdUVurwM1vJM/xsYHit
MpcKiEzoFCoX+RHsJg/1YycezAuxntX5WIHlWK2b6w3TrNOGpl0H+vyIHwpf
rI6MWX1LFTdx/CuE/XZDvnhrPvqke3cK4TD87E2H+GGk9MLysj3EFWdSmcfx
Nebz4WeIrmfT3U6G1C2R+RPfZhmevjckI8K+zwmu73MjTsJwpLi1o6JtE42s
yFqjFfyMBJE7HMXjl7cKpoazl8pDKcsjGKTyJIh9/dHEeUbilQKP3OR8EY/u
1mzdMu154BIEq/NeikQme+LE+LGe4iuFXOxbGeZRzjOCWt5C8lL1KF+B3pHW
FQnvlUNwxIHRxiDDEr4BmnsfesJGZqHhg+bchVU2LA4ZqsYRkdTu9eutuL3y
TEyfEzOrBQvk8Sbf6IS+aGujejL7sW/g9NYeGE38yFml/z5wAXh0sZMX4Wzv
8coEK1NdDsM6d5y0Lm9PB+3P2uOLz2fsW7qkD1mVbb6XByYYiY2Mb51BlEUX
e12/twhz6fAXmtdaPuV8lbAsZKyP17p3gxjSs9iY7oPqLau1W/bFfkGGtk38
pAERJv9EQVM3tevpau6EXIJ+bRl+WoG2lAE+FQaUsGXXNJG4gTn50kve6BBJ
i93Rpm6rdaSZZY9+msmP8r8kK/EMHkdNTPMMHvsPoR+fW8+Mf2YvJ64f3f0D
SPt7NmrtPxv4C+c=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{429, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//Short=",
 CellID->580580474]
}, Open  ]],

Cell[TextData[{
 "Alternatively use ",
 Cell[BoxData[
  ButtonBox["Cases",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Cases"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  ButtonBox["Except",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Except"]], "InlineFormula"],
 " to filter out missing data: "
}], "ExampleText",
 CellID->75974750],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Short", "[", 
  RowBox[{
   RowBox[{"Cases", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"IsotopeData", "[", 
          RowBox[{"#", ",", "prop"}], "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"prop", ",", 
           RowBox[{"{", 
            RowBox[{"\"\<Symbol\>\"", ",", "\"\<MagneticMoment\>\""}], 
            "}"}]}], "}"}]}], "]"}], "&"}], "/@", 
      RowBox[{"IsotopeData", "[", "]"}]}], ",", 
     RowBox[{"Except", "[", 
      RowBox[{"{", 
       RowBox[{"_", ",", "_Missing"}], "}"}], "]"}]}], "]"}], ",", "4"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->424408147],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWkuPFDcQ7u2emX3M7vJYIogS0CoIidc9QCQ2gJScoiDlAFqQkmGFYA8I
BIgTfyfHcM3Pc8Y9rln315/tcvcMGkJLuz1ll+thl6ur/fVvk/cvn7+avD8+
muz/+nby5uXx0bv9X16/nTZVa0Wx9k9RFFcuFNPfZvqr/rPXp+lfUYyn/8tp
05/u/sjdb7n7nrvfDtBD714UW/Z/JRpMMQH6D6DvAH0K6J8y+wdAl1+QVSjl
d6AroH8GGuX9MP1nyRLaTyfkbgJ9dcnj9bNj4+wh0W7b77n7GPov1yrmtOM6
E5CCWq8uiF/n48it2IxdLpxfyzV07aVjtt0jYmPl+Cq3Q4UeenTp9cds3mY+
eDSTKbqQF+kNoDGGzjp5A+fzbFJml5U1kEnzxqwr/ULdexFd6dVZba087tZl
+YxchWcJ76383kg2KOnoYVCmt1eJn4MkB0aWeLFl4tE+ypZcgcR1w3PCGepr
a0dIW+Y64AoLN9tBX6MV7ehHmzaorF3gQrqio5TxSvfDyNv1c2sjO2vHzYXs
RMwopUIm0iOQic/qyPopW3eC/qNvmB3/z3oxUtdqPl2exrrRjhoUyhzreMO7
LZ3LdWNtJMXybx/ZmIn3oD+SgXutItqBK7Gs3bIqevXVxZDkEp53WVxIJcmt
t/GLtW9tSSvvsrjDWEllSF6jBCM6Qp8DOjtK4Yn5eWKlJNrHK665TxWMb3Kc
K1A9JlrQAt+78FygNxcj3uqlXAK69nsEo7xV8KsE/11lW6WNxkJCm7+nZ7p2
V06XLtL8+dPm3k3ampPTeStWldymbjG1nMhsPBGSFQRyfavSiTEjOmXldJUR
vmmcWohu/1xnUTp1cdtav7kouXacbWPTPP8Rm+Wsyj+Lsnf/vYM9S783J+99
Jciz9D6zP5VZT/a3Yedu3wVsYevTmqWT/DJXUlIt4tHZPtoaT5VBJz26CMDd
9ARoPEmv+dch+vw18NbkRsJztt+xJrBybnaUg+1oT/4esT8PNTOUUTVe13in
nHHVTAVkdZstQZvQkkPTRJ/ofo3Q10wTncodf31B45toGL7x/Ju51vdJP4vU
B0q+H2ur7M9P8EvyqNj/AubjL6Clv7aQn3p59DHQGGfPEv04/mCFtaKUR0Cn
0DKMmRQKSJ/3Gfpvq72y6/2YSGNn3qdNHK0LoXsh+XdUVurwM1vJM/xsYHit
MpcKiEzoFCoX+RHsJg/1YycezAuxntX5WIHlWK2b6w3TrNOGpl0H+vyIHwpf
rI6MWX1LFTdx/CuE/XZDvnhrPvqke3cK4TD87E2H+GGk9MLysj3EFWdSmcfx
Nebz4WeIrmfT3U6G1C2R+RPfZhmevjckI8K+zwmu73MjTsJwpLi1o6JtE42s
yFqjFfyMBJE7HMXjl7cKpoazl8pDKcsjGKTyJIh9/dHEeUbilQKP3OR8EY/u
1mzdMu154BIEq/NeikQme+LE+LGe4iuFXOxbGeZRzjOCWt5C8lL1KF+B3pHW
FQnvlUNwxIHRxiDDEr4BmnsfesJGZqHhg+bchVU2LA4ZqsYRkdTu9eutuL3y
TEyfEzOrBQvk8Sbf6IS+aGujejL7sW/g9NYeGE38yFml/z5wAXh0sZMX4Wzv
8coEK1NdDsM6d5y0Lm9PB+3P2uOLz2fsW7qkD1mVbb6XByYYiY2Mb51BlEUX
e12/twhz6fAXmtdaPuV8lbAsZKyP17p3gxjSs9iY7oPqLau1W/bFfkGGtk38
pAERJv9EQVM3tevpau6EXIJ+bRl+WoG2lAE+FQaUsGXXNJG4gTn50kve6BBJ
i93Rpm6rdaSZZY9+msmP8r8kK/EMHkdNTPMMHvsPoR+fW8+Mf2YvJ64f3f0D
SPt7NmrtPxv4C+c=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{429, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//Short=",
 CellID->271614221]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5874536],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ListPlot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ListPlot"]], "InlineFormula"],
 " to plot a list of property values:"
}], "ExampleText",
 CellID->535512214],

Cell[BoxData[
 RowBox[{
  RowBox[{"spins", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"i", ",", 
       RowBox[{"IsotopeData", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"IsotopeData", "[", "i", "]"}], "[", 
          RowBox[{"[", "1", "]"}], "]"}], ",", "\"\<Spin\>\""}], "]"}]}], 
      "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "118"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1403006388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", "spins", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1864310337],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlMlKA0EURQtx4cKFiIgLFyoiIiJOOEa7nIc4duIUdRNw7S+8T+tPySe4
dKnS7wTfLQiXw6mqV0NXpruf5cdASqn7+/vLun0VdfY8R3LkMefkOQF7mxQ/
JX5G/Kz4uRzXMS/1F4QXZfySjF8WvyJ+Vfya+HXxG8KbsOeW+G3xO57fPv+u
87BnQ8bvOQ957osvxOfoDebeDmJ9gznPQ+fBOk04HUn/Y6kHs78TqQcz/jSy
wZzHWWSDRz3PI5twuhAPs5/LyAZz/82c/jdrir+KPl3LfDDncSP9b2X+Pnve
iReuYN7ZvafvN8Pjvh6YVjr7+edS+pexf9Vnz1b0uRW9ia9g7r/tzPfRlv4w
3/eD1IO5X9jnq2DqPcp42Oc32O+vgvlfe4rzZZj6MO05rsdgvr8XWQ/MexCf
Os68h474V/Ew7+FN1iOc3uP6DO4VP/5Ld9o=
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 118.}, {0., 8.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 121},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1012019439]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Missing",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Missing"]], "InlineFormula"],
 " data is automatically excluded in plotting functions: "
}], "ExampleText",
 CellID->94821444],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Cases", "[", 
   RowBox[{"spins", ",", 
    RowBox[{"{", 
     RowBox[{"_", ",", "_Missing"}], "}"}]}], "]"}], "//", 
  "Length"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->957919056],

Cell[BoxData["20"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->940227033]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4200064],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ListLogPlot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ListLogPlot"]], "InlineFormula"],
 " when the data spans many orders of magnitude:"
}], "ExampleText",
 CellID->671567693],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Max", "[", 
     RowBox[{"Cases", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"IsotopeData", "[", 
          RowBox[{"#", ",", "\"\<HalfLife\>\""}], "]"}], "&"}], "/@", 
        RowBox[{"IsotopeData", "[", "i", "]"}]}], ",", "_Real"}], "]"}], 
     "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "118"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1765939808],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5626181]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->28051594],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ParticleData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ParticleData"]], "InlineFormula"],
 " to find information about the elementary particles:"
}], "ExampleText",
 CellID->193367025],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParticleData", "[", 
  RowBox[{"\"\<Proton\>\"", ",", "\"\<Properties\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1129350733],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Antiparticle\"\>", ",", "\<\"BaryonNumber\"\>", 
   ",", "\<\"Bottomness\"\>", ",", "\<\"Charge\"\>", 
   ",", "\<\"ChargeRadius\"\>", ",", "\<\"ChargeStates\"\>", 
   ",", "\<\"Charm\"\>", ",", "\<\"CParity\"\>", ",", "\<\"DecayModes\"\>", 
   ",", "\<\"DecayType\"\>", ",", "\<\"Excitations\"\>", 
   ",", "\<\"FullDecayModes\"\>", ",", "\<\"FullSymbol\"\>", 
   ",", "\<\"GenericFullSymbol\"\>", ",", "\<\"GenericSymbol\"\>", 
   ",", "\<\"GFactor\"\>", ",", "\<\"GParity\"\>", ",", "\<\"HalfLife\"\>", 
   ",", "\<\"Hypercharge\"\>", ",", "\<\"Isospin\"\>", 
   ",", "\<\"IsospinMultiplet\"\>", ",", "\<\"IsospinProjection\"\>", 
   ",", "\<\"LeptonNumber\"\>", ",", "\<\"Lifetime\"\>", ",", "\<\"Mass\"\>", 
   ",", "\<\"Memberships\"\>", ",", "\<\"Parity\"\>", 
   ",", "\<\"PDGNumber\"\>", ",", "\<\"QuarkContent\"\>", ",", "\<\"Spin\"\>",
    ",", "\<\"Strangeness\"\>", ",", "\<\"Symbol\"\>", ",", "\<\"Topness\"\>",
    ",", "\<\"UnobservedDecayModes\"\>", ",", "\<\"Width\"\>"}], 
  "}"}]], "Output",
 ImageSize->{522, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->586760769]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParticleData", "[", 
  RowBox[{"\"\<Proton\>\"", ",", "\"\<Mass\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->776831264],

Cell[BoxData["938.27203`7.999999999999999"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->443337559]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->274708332],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ElementData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ElementData"]], "InlineFormula"],
 " to find physical and chemical properties of an element:"
}], "ExampleText",
 CellID->1950066989],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ElementData", "[", 
   RowBox[{"\"\<Gold\>\"", ",", "\"\<Properties\>\""}], "]"}], "//", 
  "Short"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2027947256],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"\<\"Abbreviation\"\>", ",", "\<\"AbsoluteBoilingPoint\"\>", ",", 
    RowBox[{"\[LeftSkeleton]", "83", "\[RightSkeleton]"}], 
    ",", "\<\"YoungModulus\"\>"}], "}"}],
  Short]], "Output",
 ImageSize->{402, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//Short=",
 CellID->1553582670]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ElementData", "[", 
  RowBox[{"\"\<Au\>\"", ",", "\"\<Density\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1641565555],

Cell[BoxData["19300.`2.9999999999999996"], "Output",
 ImageSize->{63, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1957754464]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->273159180],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ChemicalData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChemicalData"]], "InlineFormula"],
 " to find information about a molecule:"
}], "ExampleText",
 CellID->1030066181],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ChemicalData", "[", 
   RowBox[{"\"\<DeuteriumOxide\>\"", ",", "\"\<Properties\>\""}], "]"}], "//",
   "Short"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1456625600],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"\<\"AdjacencyMatrix\"\>", ",", "\<\"AdjacencyMatrixRules\"\>", 
    ",", 
    RowBox[{"\[LeftSkeleton]", "64", "\[RightSkeleton]"}], 
    ",", "\<\"VertexTypes\"\>", ",", "\<\"Viscosity\"\>"}], "}"}],
  Short]], "Output",
 ImageSize->{490, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//Short=",
 CellID->1979915459]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ChemicalData", "[", 
  RowBox[{"\"\<HeavyWater\>\"", ",", "\"\<BoilingPoint\>\""}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->345028468],

Cell[BoxData["101.3`3.9999999999999996"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1180009148]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->183057169],

Cell["\<\
Using non-standard isotope and property names will not work: \
\>", "ExampleText",
 CellID->1329051991],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<helium 4\>\"", ",", "\"\<AtomicMass\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->46500988],

Cell[BoxData[
 RowBox[{
  RowBox[{"IsotopeData", "::", "\<\"notent\"\>"}], ":", 
  " ", "\<\"\\!\\(\\\"\\\\\\\"helium 4\\\\\\\"\\\"\\) is not a known entity, \
class, or tag for \\!\\(IsotopeData\\). Use \\!\\(IsotopeData\\)[] for a list \
of entities.\"\>"}]], "Message", "MSG",
 CellID->402270391],

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\<\"helium 4\"\>", ",", "\<\"AtomicMass\"\>"}], "]"}]], "Output",
 ImageSize->{233, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->679940781]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Helium4\>\"", ",", " ", "\"\<atomic mass\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->222925477],

Cell[BoxData[
 RowBox[{
  RowBox[{"IsotopeData", "::", "\<\"notprop\"\>"}], ":", 
  " ", "\<\"\\!\\(\\\"\\\\\\\"atomic mass\\\\\\\"\\\"\\) is not a known \
property for \\!\\(IsotopeData\\). Use \
\\!\\(IsotopeData\\)[\\\"Properties\\\"] for a list of properties.\"\>"}]], \
"Message", "MSG",
 CellID->1004220580],

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\<\"Helium4\"\>", ",", "\<\"atomic mass\"\>"}], "]"}]], "Output",
 ImageSize->{233, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->861306287]
}, Open  ]],

Cell["Use string functions to find the standardized names: ", "ExampleText",
 CellID->173046581],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"IsotopeData", "[", "]"}], ",", 
   RowBox[{
    RowBox[{"StringMatchQ", "[", 
     RowBox[{"#", ",", 
      RowBox[{"___", "~~", "\"\<helium\>\"", "~~", "___", "~~", "\"\<4\>\""}],
       ",", 
      RowBox[{"IgnoreCase", "\[Rule]", "True"}]}], "]"}], "&"}]}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->135247669],

Cell[BoxData[
 RowBox[{"{", "\<\"Helium4\"\>", "}"}]], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1934984829]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"IsotopeData", "[", 
    RowBox[{"\"\<Helium4\>\"", ",", "\"\<Properties\>\""}], "]"}], ",", 
   RowBox[{
    RowBox[{"StringMatchQ", "[", 
     RowBox[{"#", ",", 
      RowBox[{"___", "~~", "\"\<atomic\>\"", "~~", "___"}], ",", 
      RowBox[{"IgnoreCase", "\[Rule]", "True"}]}], "]"}], "&"}]}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->57009403],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AtomicMass\"\>", ",", "\<\"AtomicNumber\"\>"}], 
  "}"}]], "Output",
 ImageSize->{184, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1945154482]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IsotopeData", "[", 
  RowBox[{"\"\<Helium4\>\"", ",", "\"\<AtomicMass\>\""}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1260968962],

Cell[BoxData["4.00260325415`12.000000000000004"], "Output",
 ImageSize->{95, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1067820693]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->90344104],

Cell[TextData[{
 "Arithmetical operations cannot be carried out on ",
 Cell[BoxData[
  ButtonBox["Missing",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Missing"]], "InlineFormula"],
 " entries:"
}], "ExampleText",
 CellID->46301994],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"45", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"IsotopeData", "[", 
     RowBox[{"i", ",", "\"\<MagneticMoment\>\""}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", 
      RowBox[{"IsotopeData", "[", "\"\<Li\>\"", "]"}]}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1739347100],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"45", " ", 
    RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}]}], ",", 
   RowBox[{"45", " ", 
    RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}]}], ",", 
   RowBox[{"45", " ", 
    RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}]}], ",", 
   "36.9921284999999999997`7.000000000000001", ",", 
   "146.5392059999999999992`7.999999999999999", ",", 
   "74.4101999999999999996`7.000000000000001", ",", 
   "154.759500000000000001`5.000000000000002", ",", 
   RowBox[{"45", " ", 
    RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}]}], ",", 
   "165.0509999999999999996`5.000000000000002", ",", 
   RowBox[{"45", " ", 
    RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}]}]}], "}"}]], "Output",
 ImageSize->{535, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->341549272]
}, Open  ]],

Cell[TextData[{
 "Remove the ",
 Cell[BoxData[
  ButtonBox["Missing",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Missing"]], "InlineFormula"],
 " entries before performing operations: "
}], "ExampleText",
 CellID->85509328],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DeleteCases", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"IsotopeData", "[", 
      RowBox[{"i", ",", "\"\<MagneticMoment\>\""}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", 
       RowBox[{"IsotopeData", "[", "\"\<Li\>\"", "]"}]}], "}"}]}], "]"}], ",",
    "_Missing"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->548645387],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.8220473`7.000000000000001", ",", "3.2564268`7.999999999999999", ",", 
   "1.65356`7.000000000000001", ",", "3.4391`5.000000000000002", ",", 
   "3.6678`5.000000000000002"}], "}"}]], "Output",
 ImageSize->{329, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->194826073]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"45", " ", "%"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->60605648],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "36.9921284999999999997`7.000000000000001", ",", 
   "146.5392059999999999992`7.999999999999999", ",", 
   "74.4101999999999999996`7.000000000000001", ",", 
   "154.759500000000000001`5.000000000000002", ",", 
   "165.0509999999999999996`5.000000000000002"}], "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1676730883]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->771168705],

Cell["\<\
Draw a decay network for all known isotopes, labeling each isotope with a \
tooltip:\
\>", "ExampleText",
 CellID->288858026],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GraphPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"i", "\[Rule]", "#"}], ")"}], "&"}], "/@", 
       RowBox[{"Flatten", "[", 
        RowBox[{"{", 
         RowBox[{"IsotopeData", "[", 
          RowBox[{"i", ",", "\"\<DaughterNuclides\>\""}], "]"}], "}"}], 
        "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", 
        RowBox[{"IsotopeData", "[", "]"}]}], "}"}]}], "]"}], "//", 
    "Flatten"}], ",", 
   RowBox[{"VertexLabeling", "\[Rule]", "Tooltip"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->280087899],

Cell[BoxData[
 GraphicsBox[
  TagBox[GraphicsComplexBox[CompressedData["
1:eJxEe3k01G/YPiVCtiwpUsiSSpJ11nvGUqGFFkUYZsYyM4YQbZYoKpFKSEj7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    "], {
     {RGBColor[0.5, 0., 0.], 
      LineBox[{{1, 2}, {3, 4}, {5, 3}, {6, 3}, {7, 6}, {7, 3}, {8, 6}, {9, 
        10}, {11, 12}, {11, 10}, {13, 11}, {14, 15}, {14, 16}, {14, 9}, {15, 
        10}, {17, 14}, {18, 14}, {19, 20}, {21, 4}, {22, 10}, {23, 24}, {23, 
        25}, {24, 10}, {26, 23}, {27, 28}, {27, 25}, {27, 24}, {27, 11}, {27, 
        15}, {27, 23}, {28, 33}, {29, 27}, {30, 21}, {31, 10}, {32, 16}, {34, 
        35}, {34, 16}, {36, 37}, {36, 35}, {37, 47}, {37, 24}, {38, 36}, {39, 
        40}, {39, 37}, {39, 35}, {39, 34}, {39, 26}, {40, 48}, {40, 47}, {41, 
        39}, {42, 39}, {43, 21}, {44, 31}, {45, 10}, {46, 24}, {49, 50}, {49, 
        48}, {50, 64}, {51, 50}, {51, 48}, {52, 51}, {53, 54}, {53, 55}, {53, 
        50}, {53, 48}, {54, 65}, {55, 65}, {56, 53}, {57, 58}, {57, 59}, {58, 
        68}, {58, 66}, {59, 66}, {59, 67}, {60, 31}, {61, 46}, {61, 24}, {61, 
        22}, {62, 33}, {63, 35}, {66, 80}, {66, 48}, {67, 80}, {67, 47}, {68, 
        81}, {68, 50}, {68, 82}, {69, 68}, {69, 70}, {69, 66}, {70, 81}, {70, 
        83}, {71, 70}, {71, 72}, {72, 83}, {72, 84}, {73, 71}, {74, 75}, {74, 
        72}, {75, 84}, {76, 61}, {77, 62}, {78, 47}, {78, 24}, {79, 48}, {83, 
        96}, {84, 97}, {85, 86}, {85, 87}, {86, 99}, {86, 98}, {87, 98}, {88, 
        86}, {88, 87}, {89, 88}, {90, 91}, {90, 88}, {90, 89}, {91, 102}, {92,
         62}, {92, 78}, {93, 79}, {93, 47}, {94, 64}, {95, 65}, {97, 114}, {
        98, 115}, {99, 116}, {99, 115}, {100, 101}, {100, 99}, {101, 117}, {
        101, 116}, {102, 103}, {102, 101}, {103, 118}, {103, 117}, {104, 
        102}, {104, 91}, {104, 105}, {105, 121}, {105, 120}, {106, 104}, {106,
         91}, {107, 108}, {107, 104}, {107, 106}, {108, 122}, {109, 93}, {110,
         94}, {111, 95}, {112, 82}, {113, 81}, {117, 135}, {118, 136}, {119, 
        120}, {119, 118}, {120, 137}, {121, 138}, {121, 137}, {122, 121}, {
        122, 123}, {123, 139}, {123, 138}, {124, 125}, {124, 126}, {124, 
        123}, {125, 140}, {125, 139}, {126, 141}, {126, 140}, {126, 139}, {
        127, 124}, {128, 129}, {128, 130}, {129, 143}, {129, 142}, {130, 
        142}, {130, 141}, {131, 94}, {132, 80}, {132, 79}, {133, 113}, {134, 
        96}, {136, 155}, {137, 156}, {138, 157}, {139, 158}, {139, 157}, {140,
         159}, {140, 158}, {141, 160}, {141, 159}, {142, 161}, {142, 160}, {
        142, 159}, {142, 121}, {143, 162}, {143, 161}, {143, 160}, {143, 
        159}, {144, 145}, {144, 143}, {145, 163}, {145, 162}, {145, 161}, {
        146, 144}, {147, 148}, {147, 149}, {148, 163}, {148, 165}, {148, 
        164}, {149, 163}, {149, 164}, {149, 162}, {150, 132}, {150, 133}, {
        151, 133}, {152, 114}, {152, 80}, {153, 115}, {154, 116}, {158, 
        177}, {159, 178}, {160, 179}, {161, 180}, {161, 179}, {162, 181}, {
        162, 180}, {163, 182}, {163, 181}, {164, 183}, {164, 182}, {165, 
        184}, {165, 183}, {166, 167}, {166, 165}, {167, 184}, {167, 185}, {
        168, 166}, {169, 170}, {169, 171}, {170, 187}, {170, 186}, {171, 
        186}, {172, 132}, {173, 152}, {173, 134}, {174, 153}, {174, 114}, {
        174, 112}, {175, 154}, {176, 135}, {178, 200}, {179, 201}, {180, 
        202}, {181, 203}, {181, 202}, {182, 204}, {182, 203}, {183, 205}, {
        183, 204}, {184, 206}, {184, 205}, {185, 207}, {185, 206}, {186, 
        208}, {186, 207}, {187, 209}, {188, 189}, {189, 210}, {190, 188}, {
        191, 192}, {191, 193}, {192, 212}, {192, 211}, {193, 211}, {194, 
        173}, {195, 175}, {195, 153}, {195, 114}, {195, 133}, {196, 176}, {
        196, 154}, {197, 155}, {197, 114}, {197, 135}, {198, 156}, {199, 
        157}, {203, 223}, {204, 224}, {205, 225}, {206, 226}, {207, 227}, {
        207, 226}, {208, 228}, {208, 227}, {209, 229}, {209, 228}, {210, 
        230}, {210, 229}, {211, 231}, {212, 232}, {212, 231}, {213, 214}, {
        214, 233}, {215, 216}, {215, 214}, {216, 234}, {216, 233}, {217, 
        195}, {217, 174}, {218, 175}, {218, 153}, {219, 197}, {219, 176}, {
        220, 198}, {220, 155}, {221, 199}, {222, 177}, {224, 246}, {225, 
        247}, {226, 248}, {227, 249}, {228, 250}, {229, 251}, {230, 252}, {
        230, 251}, {231, 253}, {231, 252}, {232, 254}, {232, 253}, {233, 
        255}, {233, 254}, {234, 255}, {234, 256}, {235, 236}, {235, 234}, {
        236, 256}, {237, 238}, {237, 236}, {238, 257}, {239, 218}, {239, 
        219}, {240, 219}, {241, 221}, {241, 155}, {241, 198}, {242, 222}, {
        242, 199}, {243, 200}, {243, 177}, {243, 155}, {244, 201}, {245, 
        202}, {249, 268}, {251, 269}, {252, 270}, {253, 271}, {254, 272}, {
        255, 273}, {255, 272}, {256, 274}, {256, 273}, {257, 277}, {257, 
        276}, {258, 259}, {259, 277}, {259, 278}, {260, 261}, {261, 279}, {
        262, 219}, {263, 242}, {263, 198}, {263, 221}, {264, 243}, {264, 
        222}, {265, 244}, {265, 200}, {266, 245}, {267, 223}, {270, 294}, {
        271, 295}, {272, 296}, {273, 297}, {274, 298}, {275, 276}, {275, 
        274}, {276, 299}, {276, 298}, {277, 300}, {277, 299}, {278, 301}, {
        278, 300}, {279, 301}, {279, 302}, {280, 281}, {281, 303}, {281, 
        302}, {282, 283}, {283, 304}, {284, 282}, {285, 263}, {286, 264}, {
        287, 265}, {288, 267}, {288, 245}, {289, 246}, {289, 200}, {289, 
        223}, {290, 247}, {291, 248}, {292, 293}, {292, 250}, {293, 250}, {
        295, 318}, {297, 319}, {298, 320}, {299, 321}, {300, 322}, {301, 
        323}, {302, 324}, {303, 325}, {303, 324}, {304, 326}, {305, 306}, {
        306, 326}, {306, 327}, {307, 308}, {308, 328}, {308, 327}, {309, 
        310}, {310, 329}, {311, 264}, {312, 266}, {312, 288}, {312, 244}, {
        312, 200}, {312, 221}, {312, 242}, {313, 289}, {313, 267}, {314, 
        290}, {314, 246}, {315, 291}, {316, 268}, {317, 269}, {320, 343}, {
        321, 344}, {322, 345}, {323, 346}, {324, 347}, {325, 348}, {326, 
        349}, {326, 348}, {327, 349}, {328, 350}, {328, 351}, {329, 352}, {
        329, 350}, {330, 331}, {331, 352}, {331, 350}, {332, 333}, {332, 
        331}, {333, 353}, {333, 352}, {334, 312}, {335, 313}, {336, 314}, {
        337, 316}, {337, 291}, {338, 293}, {338, 268}, {338, 246}, {339, 
        317}, {340, 294}, {341, 342}, {341, 296}, {342, 296}, {346, 366}, {
        347, 367}, {348, 368}, {349, 369}, {349, 370}, {350, 372}, {351, 
        371}, {352, 373}, {352, 372}, {353, 374}, {353, 373}, {354, 355}, {
        354, 356}, {355, 376}, {355, 375}, {356, 375}, {356, 374}, {357, 
        358}, {357, 355}, {358, 377}, {359, 313}, {360, 315}, {360, 290}, {
        361, 316}, {361, 338}, {362, 293}, {363, 340}, {364, 318}, {365, 
        319}, {368, 392}, {370, 369}, {371, 393}, {372, 394}, {373, 395}, {
        374, 396}, {375, 397}, {375, 396}, {376, 398}, {376, 397}, {377, 
        399}, {377, 398}, {378, 379}, {379, 400}, {380, 381}, {380, 379}, {
        381, 401}, {381, 400}, {382, 360}, {383, 361}, {384, 362}, {385, 
        363}, {386, 342}, {386, 318}, {386, 293}, {387, 365}, {388, 343}, {
        389, 344}, {390, 345}, {391, 367}, {395, 416}, {396, 417}, {397, 
        418}, {398, 419}, {399, 420}, {400, 421}, {400, 420}, {401, 422}, {
        401, 421}, {402, 403}, {403, 423}, {404, 405}, {404, 403}, {405, 
        424}, {406, 407}, {407, 425}, {408, 361}, {409, 363}, {409, 339}, {
        410, 364}, {411, 342}, {412, 388}, {413, 389}, {414, 390}, {415, 
        366}, {417, 444}, {418, 445}, {419, 446}, {420, 447}, {421, 448}, {
        421, 447}, {422, 449}, {422, 448}, {423, 450}, {423, 449}, {424, 
        451}, {424, 450}, {425, 452}, {425, 451}, {426, 427}, {426, 425}, {
        427, 453}, {427, 452}, {428, 429}, {429, 454}, {430, 431}, {430, 
        429}, {431, 455}, {431, 454}, {432, 409}, {433, 410}, {434, 411}, {
        435, 413}, {436, 414}, {436, 342}, {437, 415}, {438, 367}, {439, 
        392}, {440, 369}, {441, 393}, {442, 394}, {442, 443}, {443, 394}, {
        447, 469}, {448, 470}, {449, 471}, {450, 472}, {451, 473}, {452, 
        474}, {453, 475}, {453, 474}, {454, 476}, {454, 475}, {455, 477}, {
        455, 476}, {456, 457}, {457, 478}, {458, 459}, {458, 457}, {459, 
        479}, {459, 478}, {460, 434}, {460, 410}, {460, 411}, {461, 435}, {
        461, 412}, {461, 387}, {461, 385}, {462, 436}, {462, 413}, {463, 
        437}, {463, 414}, {464, 438}, {465, 439}, {466, 440}, {467, 441}, {
        468, 416}, {470, 496}, {471, 497}, {472, 498}, {473, 499}, {474, 
        500}, {475, 501}, {475, 500}, {476, 502}, {476, 501}, {477, 503}, {
        477, 502}, {478, 504}, {478, 503}, {479, 505}, {479, 504}, {480, 
        481}, {481, 506}, {481, 505}, {482, 483}, {483, 507}, {483, 506}, {
        484, 461}, {485, 462}, {486, 464}, {486, 437}, {486, 414}, {486, 
        412}, {487, 465}, {487, 438}, {488, 466}, {488, 439}, {488, 414}, {
        489, 467}, {490, 443}, {491, 468}, {492, 444}, {493, 445}, {494, 3183,
         3184, 3185, 3186, 3187, 3188, 3189, 3190, 3191, 3192, 3193, 3194, 
        3195, 3196, 3197, 3198, 446}, {494, 3199, 3200, 3201, 3202, 3203, 
        3204, 3205, 3206, 3207, 3208, 3209, 3210, 3211, 3212, 3213, 3214, 
        446}, {494, 495}, {495, 446}, {499, 522}, {500, 523}, {501, 524}, {
        502, 525}, {503, 526}, {504, 527}, {505, 528}, {506, 529}, {506, 
        528}, {507, 530}, {507, 529}, {508, 509}, {508, 507}, {509, 531}, {
        509, 530}, {510, 511}, {510, 509}, {511, 532}, {511, 531}, {512, 
        461}, {512, 462}, {513, 486}, {513, 463}, {514, 464}, {515, 488}, {
        515, 465}, {516, 466}, {516, 489}, {517, 490}, {517, 467}, {518, 
        491}, {519, 492}, {520, 493}, {521, 469}, {523, 551}, {524, 552}, {
        525, 553}, {526, 554}, {527, 555}, {528, 556}, {529, 557}, {530, 
        558}, {531, 559}, {532, 560}, {532, 559}, {533, 534}, {534, 561}, {
        534, 560}, {535, 536}, {536, 562}, {536, 561}, {537, 538}, {538, 
        563}, {538, 562}, {539, 513}, {540, 514}, {541, 515}, {542, 489}, {
        542, 517}, {543, 518}, {544, 519}, {545, 520}, {546, 495}, {547, 
        521}, {548, 496}, {549, 497}, {550, 498}, {554, 582}, {556, 583}, {
        557, 584}, {558, 585}, {559, 586}, {560, 587}, {561, 588}, {562, 
        589}, {563, 590}, {563, 589}, {564, 565}, {565, 591}, {565, 590}, {
        566, 567}, {567, 592}, {567, 591}, {568, 569}, {569, 593}, {569, 
        592}, {570, 513}, {571, 541}, {571, 515}, {572, 542}, {573, 543}, {
        574, 544}, {574, 518}, {575, 545}, {575, 519}, {576, 546}, {577, 
        547}, {578, 548}, {579, 549}, {580, 498}, {581, 522}, {584, 613}, {
        585, 614}, {586, 615}, {587, 616}, {588, 617}, {589, 618}, {590, 
        619}, {591, 620}, {591, 619}, {592, 621}, {593, 622}, {593, 621}, {
        594, 595}, {594, 593}, {595, 623}, {595, 622}, {596, 597}, {597, 
        624}, {598, 599}, {599, 625}, {600, 573}, {601, 574}, {601, 575}, {
        602, 575}, {603, 577}, {603, 576}, {604, 578}, {604, 547}, {605, 
        579}, {606, 580}, {607, 581}, {608, 551}, {609, 552}, {610, 553}, {
        611, 555}, {611, 612}, {612, 555}, {616, 641}, {617, 642}, {618, 
        643}, {619, 644}, {620, 645}, {621, 646}, {622, 647}, {623, 648}, {
        624, 649}, {625, 650}, {626, 625}, {626, 627}, {627, 651}, {627, 
        650}, {628, 629}, {629, 652}, {629, 651}, {630, 574}, {631, 603}, {
        632, 604}, {632, 577}, {633, 578}, {633, 605}, {634, 606}, {634, 
        579}, {635, 607}, {635, 580}, {636, 608}, {637, 609}, {638, 610}, {
        639, 582}, {640, 583}, {644, 673}, {645, 674}, {646, 675}, {647, 
        676}, {648, 677}, {649, 678}, {650, 679}, {651, 680}, {651, 679}, {
        652, 681}, {652, 680}, {653, 654}, {653, 652}, {654, 682}, {654, 
        681}, {655, 656}, {656, 683}, {656, 682}, {657, 658}, {658, 684}, {
        658, 683}, {659, 631}, {660, 632}, {661, 633}, {662, 634}, {663, 
        636}, {663, 607}, {663, 578}, {664, 637}, {664, 608}, {665, 638}, {
        666, 639}, {667, 612}, {668, 640}, {669, 613}, {670, 614}, {671, 
        615}, {672, 642}, {672, 617}, {676, 704}, {677, 705}, {678, 706}, {
        679, 707}, {680, 708}, {681, 709}, {682, 710}, {683, 711}, {684, 
        712}, {685, 684}, {685, 686}, {686, 713}, {686, 712}, {687, 688}, {
        688, 714}, {688, 713}, {689, 690}, {690, 715}, {690, 714}, {691, 
        632}, {692, 633}, {693, 663}, {694, 636}, {695, 665}, {696, 666}, {
        697, 667}, {698, 668}, {698, 612}, {699, 669}, {700, 670}, {701, 
        671}, {702, 641}, {703, 643}, {706, 739}, {707, 740}, {708, 741}, {
        709, 742}, {710, 743}, {711, 744}, {712, 745}, {713, 746}, {713, 
        745}, {714, 746}, {714, 747}, {715, 748}, {715, 747}, {716, 717}, {
        716, 715}, {717, 749}, {717, 748}, {718, 719}, {719, 750}, {719, 
        749}, {720, 721}, {721, 751}, {722, 692}, {723, 693}, {724, 694}, {
        725, 695}, {726, 697}, {727, 698}, {728, 699}, {729, 700}, {730, 
        701}, {731, 702}, {732, 642}, {733, 703}, {734, 673}, {735, 674}, {
        736, 675}, {736, 737}, {737, 675}, {738, 705}, {738, 677}, {741, 
        768}, {742, 769}, {743, 770}, {744, 771}, {745, 772}, {746, 773}, {
        747, 774}, {748, 775}, {749, 776}, {749, 775}, {750, 777}, {750, 
        776}, {751, 778}, {751, 777}, {752, 753}, {753, 779}, {753, 778}, {
        754, 755}, {755, 780}, {755, 779}, {756, 757}, {757, 781}, {758, 
        727}, {758, 697}, {759, 728}, {760, 729}, {760, 699}, {761, 730}, {
        762, 731}, {762, 701}, {763, 732}, {764, 733}, {765, 734}, {766, 
        735}, {767, 704}, {772, 801}, {773, 802}, {774, 803}, {775, 804}, {
        776, 805}, {776, 804}, {777, 806}, {777, 805}, {778, 807}, {778, 
        806}, {779, 808}, {779, 807}, {780, 809}, {780, 808}, {781, 810}, {
        781, 809}, {782, 783}, {783, 810}, {783, 811}, {784, 785}, {785, 
        811}, {785, 812}, {786, 759}, {787, 760}, {788, 761}, {789, 763}, {
        790, 764}, {791, 765}, {792, 766}, {793, 737}, {794, 767}, {795, 
        705}, {796, 739}, {797, 740}, {797, 798}, {798, 740}, {799, 769}, {
        799, 742}, {800, 771}, {803, 830}, {804, 831}, {805, 832}, {806, 
        833}, {807, 834}, {808, 835}, {809, 836}, {810, 837}, {810, 836}, {
        811, 838}, {811, 837}, {812, 839}, {812, 838}, {813, 814}, {814, 
        840}, {814, 839}, {815, 816}, {816, 841}, {816, 840}, {817, 818}, {
        818, 842}, {818, 841}, {819, 788}, {819, 761}, {820, 789}, {821, 
        790}, {821, 763}, {822, 791}, {823, 792}, {823, 765}, {824, 793}, {
        825, 794}, {826, 795}, {827, 796}, {828, 768}, {829, 770}, {832, 
        865}, {833, 866}, {834, 867}, {835, 868}, {836, 869}, {837, 870}, {
        837, 869}, {838, 871}, {838, 870}, {839, 872}, {839, 871}, {840, 
        873}, {840, 872}, {841, 874}, {841, 873}, {842, 875}, {842, 874}, {
        843, 844}, {843, 842}, {844, 876}, {844, 875}, {844, 874}, {845, 
        846}, {845, 844}, {846, 877}, {846, 876}, {847, 848}, {848, 878}, {
        848, 877}, {848, 876}, {849, 820}, {850, 821}, {851, 822}, {852, 
        824}, {853, 825}, {854, 826}, {854, 765}, {855, 827}, {856, 798}, {
        857, 828}, {858, 769}, {859, 829}, {860, 771}, {861, 801}, {862, 
        802}, {862, 863}, {863, 802}, {864, 831}, {864, 804}, {867, 895}, {
        868, 896}, {869, 897}, {870, 898}, {871, 899}, {872, 900}, {873, 
        901}, {874, 902}, {875, 903}, {875, 902}, {876, 904}, {876, 903}, {
        877, 905}, {877, 904}, {878, 906}, {878, 905}, {879, 880}, {879, 
        878}, {880, 907}, {880, 906}, {881, 882}, {881, 880}, {882, 908}, {
        882, 907}, {883, 851}, {883, 822}, {884, 852}, {885, 853}, {885, 
        824}, {886, 854}, {887, 855}, {887, 826}, {888, 856}, {889, 857}, {
        890, 858}, {891, 859}, {892, 860}, {893, 861}, {894, 830}, {896, 
        928}, {897, 929}, {898, 930}, {899, 931}, {900, 932}, {901, 933}, {
        902, 934}, {903, 935}, {903, 934}, {904, 936}, {904, 935}, {905, 
        937}, {905, 936}, {906, 938}, {906, 937}, {907, 939}, {907, 938}, {
        908, 940}, {908, 939}, {909, 910}, {910, 941}, {910, 940}, {911, 
        912}, {912, 942}, {912, 941}, {913, 914}, {914, 943}, {915, 886}, {
        915, 885}, {916, 886}, {916, 854}, {916, 887}, {917, 888}, {917, 
        855}, {918, 889}, {918, 856}, {919, 890}, {920, 891}, {921, 892}, {
        922, 893}, {923, 863}, {924, 894}, {925, 831}, {926, 865}, {927, 
        866}, {931, 962}, {932, 963}, {933, 964}, {934, 965}, {935, 966}, {
        936, 967}, {937, 968}, {938, 969}, {939, 970}, {940, 971}, {941, 
        972}, {942, 973}, {943, 974}, {943, 973}, {944, 945}, {945, 975}, {
        946, 947}, {947, 976}, {948, 949}, {948, 947}, {949, 977}, {949, 
        976}, {950, 917}, {950, 887}, {951, 918}, {951, 888}, {952, 919}, {
        952, 889}, {953, 920}, {953, 890}, {954, 921}, {955, 922}, {955, 
        892}, {956, 923}, {957, 924}, {958, 925}, {959, 926}, {960, 927}, {
        961, 895}, {964, 996}, {966, 998}, {967, 999}, {968, 1000}, {969, 
        1001}, {970, 1002}, {971, 1003}, {972, 1004}, {973, 1005}, {973, 
        1004}, {974, 1006}, {974, 1005}, {975, 1007}, {975, 1006}, {976, 
        1008}, {976, 1007}, {977, 1009}, {977, 1008}, {978, 979}, {979, 
        1010}, {979, 1009}, {980, 981}, {981, 1011}, {981, 1010}, {982, 
        952}, {982, 953}, {982, 919}, {983, 954}, {983, 920}, {984, 955}, {
        985, 956}, {985, 922}, {986, 957}, {987, 958}, {988, 959}, {989, 
        960}, {990, 961}, {991, 928}, {992, 3215, 3216, 3217, 3218, 3219, 
        3220, 3221, 3222, 3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 
        929}, {992, 3231, 3232, 3233, 3234, 3235, 3236, 3237, 3238, 3239, 
        3240, 3241, 3242, 3243, 3244, 3245, 3246, 929}, {993, 930}, {993, 
        994}, {994, 930}, {995, 963}, {997, 965}, {999, 1028}, {1000, 1029}, {
        1001, 1030}, {1002, 1031}, {1003, 1032}, {1004, 1033}, {1005, 1034}, {
        1006, 1035}, {1007, 1036}, {1008, 1037}, {1009, 1038}, {1010, 1039}, {
        1011, 1040}, {1011, 1039}, {1012, 1013}, {1013, 1041}, {1014, 1015}, {
        1015, 1042}, {1016, 1017}, {1017, 1043}, {1018, 984}, {1018, 954}, {
        1019, 985}, {1019, 955}, {1020, 986}, {1020, 956}, {1021, 987}, {1022,
         988}, {1022, 958}, {1023, 989}, {1024, 990}, {1025, 991}, {1026, 
        992}, {1027, 962}, {1029, 1062}, {1031, 1064}, {1032, 1065}, {1033, 
        1066}, {1034, 1067}, {1035, 1068}, {1036, 1069}, {1037, 1070}, {1038, 
        1071}, {1039, 1072}, {1039, 1071}, {1040, 1073}, {1040, 1072}, {1041, 
        1074}, {1041, 1073}, {1042, 1075}, {1042, 1074}, {1043, 1076}, {1043, 
        1075}, {1044, 1045}, {1045, 1077}, {1045, 1076}, {1046, 1047}, {1046, 
        1045}, {1047, 1078}, {1047, 1077}, {1048, 1019}, {1048, 1020}, {1048, 
        985}, {1049, 1021}, {1049, 986}, {1050, 1022}, {1050, 987}, {1051, 
        1023}, {1052, 1024}, {1053, 1025}, {1054, 1026}, {1055, 994}, {1056, 
        1027}, {1057, 963}, {1058, 996}, {1059, 965}, {1060, 998}, {1061, 
        1028}, {1061, 999}, {1063, 1030}, {1063, 1001}, {1066, 1096}, {1067, 
        1097}, {1068, 1098}, {1069, 1099}, {1070, 1100}, {1071, 1101}, {1072, 
        1102}, {1073, 1103}, {1074, 1104}, {1075, 1105}, {1076, 1106}, {1077, 
        1107}, {1077, 1106}, {1078, 1108}, {1078, 1107}, {1079, 1080}, {1080, 
        1109}, {1081, 1082}, {1082, 1110}, {1083, 1084}, {1084, 1111}, {1085, 
        1050}, {1086, 1051}, {1086, 1022}, {1087, 1052}, {1087, 1023}, {1088, 
        1053}, {1089, 1054}, {1090, 1055}, {1091, 1056}, {1092, 1057}, {1093, 
        1058}, {1094, 965}, {1095, 1060}, {1098, 1132}, {1099, 1133}, {1100, 
        1134}, {1101, 1135}, {1102, 1136}, {1103, 1137}, {1104, 1138}, {1105, 
        1139}, {1106, 1140}, {1107, 1141}, {1107, 1140}, {1108, 1142}, {1109, 
        1143}, {1109, 1142}, {1110, 1144}, {1111, 1145}, {1112, 1113}, {1113, 
        1146}, {1114, 1115}, {1115, 1147}, {1116, 1087}, {1117, 1088}, {1117, 
        1052}, {1118, 1089}, {1118, 1053}, {1119, 1090}, {1119, 1054}, {1120, 
        1091}, {1120, 1055}, {1121, 1092}, {1121, 1056}, {1122, 1093}, {1123, 
        1094}, {1124, 1095}, {1125, 1028}, {1126, 1062}, {1127, 1030}, {1128, 
        1064}, {1129, 1065}, {1129, 1130}, {1130, 1065}, {1131, 1097}, {1131, 
        1067}, {1134, 1164}, {1136, 1165}, {1137, 1166}, {1138, 1167}, {1139, 
        1168}, {1140, 1169}, {1141, 1170}, {1142, 1171}, {1143, 1172}, {1144, 
        1173}, {1145, 1174}, {1146, 1175}, {1147, 1176}, {1148, 1149}, {1149, 
        1177}, {1150, 1151}, {1151, 1178}, {1152, 1118}, {1153, 1119}, {1153, 
        1089}, {1154, 1120}, {1154, 1090}, {1155, 1121}, {1156, 1122}, {1157, 
        1123}, {1158, 1124}, {1159, 1125}, {1160, 1126}, {1161, 1127}, {1162, 
        1128}, {1163, 1096}, {1167, 1201}, {1168, 1202}, {1169, 1203}, {1170, 
        1204}, {1171, 1205}, {1172, 1206}, {1173, 1207}, {1174, 1208}, {1175, 
        1209}, {1176, 1210}, {1176, 1209}, {1177, 1211}, {1177, 1210}, {1178, 
        1212}, {1178, 1211}, {1179, 1180}, {1180, 1213}, {1180, 1212}, {1181, 
        1182}, {1182, 1214}, {1182, 1213}, {1183, 1153}, {1183, 1154}, {1184, 
        1155}, {1184, 1120}, {1185, 1156}, {1185, 1121}, {1186, 1157}, {1186, 
        1122}, {1187, 1158}, {1188, 1159}, {1188, 1124}, {1189, 1160}, {1190, 
        1161}, {1191, 1162}, {1192, 1130}, {1193, 1163}, {1194, 1097}, {1195, 
        1132}, {1196, 1133}, {1196, 1197}, {1197, 1133}, {1198, 1199}, {1198, 
        1135}, {1199, 1135}, {1200, 1166}, {1200, 1137}, {1203, 1240}, {1204, 
        1241}, {1205, 1242}, {1206, 1243}, {1207, 1244}, {1208, 1245}, {1209, 
        1246}, {1210, 1247}, {1211, 1248}, {1212, 1249}, {1213, 1250}, {1214, 
        1251}, {1215, 1216}, {1215, 1214}, {1216, 1252}, {1217, 1218}, {1217, 
        1216}, {1218, 1253}, {1219, 1220}, {1219, 1218}, {1220, 1254}, {1221, 
        1222}, {1221, 1220}, {1222, 1255}, {1223, 1224}, {1223, 1222}, {1224, 
        1256}, {1224, 1255}, {1225, 1226}, {1225, 1224}, {1226, 1257}, {1226, 
        1256}, {1227, 1185}, {1227, 1155}, {1228, 1186}, {1229, 1187}, {1229, 
        1157}, {1230, 1188}, {1230, 1158}, {1231, 1189}, {1231, 1159}, {1231, 
        1122}, {1232, 1190}, {1233, 1191}, {1234, 1192}, {1235, 1193}, {1236, 
        1194}, {1237, 1195}, {1238, 1164}, {1239, 1165}, {1242, 1280}, {1244, 
        1282}, {1245, 1283}, {1246, 1284}, {1247, 1285}, {1248, 1286}, {1249, 
        1287}, {1250, 1288}, {1251, 1289}, {1252, 1290}, {1253, 1291}, {1254, 
        1292}, {1254, 1291}, {1255, 1293}, {1255, 1292}, {1256, 1294}, {1256, 
        1293}, {1257, 1295}, {1257, 1294}, {1258, 1259}, {1258, 1257}, {1259, 
        1296}, {1259, 1295}, {1260, 1259}, {1260, 1261}, {1261, 1297}, {1261, 
        1296}, {1262, 1228}, {1262, 1229}, {1263, 1230}, {1263, 1187}, {1264, 
        1231}, {1264, 1188}, {1265, 1232}, {1265, 1189}, {1266, 1233}, {1266, 
        1190}, {1267, 1234}, {1267, 1191}, {1268, 1235}, {1269, 1236}, {1270, 
        1237}, {1271, 1197}, {1272, 1238}, {1273, 1199}, {1274, 1239}, {1275, 
        1166}, {1276, 1201}, {1277, 1202}, {1277, 1278}, {1278, 1202}, {1279, 
        1241}, {1279, 1204}, {1281, 1243}, {1281, 1206}, {1286, 1318}, {1287, 
        1319}, {1288, 1320}, {1289, 1321}, {1290, 1322}, {1291, 1323}, {1292, 
        1324}, {1293, 1325}, {1294, 1326}, {1295, 1327}, {1296, 1328}, {1297, 
        1329}, {1298, 1297}, {1299, 1300}, {1299, 1301}, {1299, 1297}, {1300, 
        1330}, {1300, 1329}, {1301, 1331}, {1301, 1330}, {1302, 1303}, {1302, 
        1301}, {1303, 1332}, {1303, 1331}, {1304, 1264}, {1304, 1230}, {1305, 
        1265}, {1305, 1231}, {1306, 1266}, {1306, 1232}, {1307, 1267}, {1307, 
        1233}, {1308, 1268}, {1308, 1234}, {1309, 1269}, {1310, 1270}, {1310, 
        1236}, {1311, 1271}, {1312, 1272}, {1313, 1273}, {1314, 1274}, {1315, 
        1275}, {1316, 1276}, {1317, 1240}, {1322, 1357}, {1323, 1358}, {1324, 
        1359}, {1325, 1360}, {1326, 1361}, {1327, 1362}, {1328, 1363}, {1329, 
        1364}, {1330, 1365}, {1331, 1366}, {1332, 1367}, {1332, 1366}, {1333, 
        1334}, {1333, 1332}, {1334, 1368}, {1334, 1367}, {1334, 1366}, {1335, 
        1334}, {1335, 1336}, {1336, 1369}, {1336, 1368}, {1337, 1308}, {1338, 
        1308}, {1338, 1268}, {1338, 1265}, {1339, 1309}, {1339, 1269}, {1339, 
        1310}, {1340, 1311}, {1341, 1312}, {1342, 1313}, {1342, 1272}, {1343, 
        1314}, {1344, 1315}, {1345, 1316}, {1346, 1278}, {1347, 1317}, {1348, 
        1241}, {1349, 1280}, {1350, 1243}, {1351, 1282}, {1352, 1283}, {1353, 
        1284}, {1354, 1285}, {1355, 1319}, {1355, 1287}, {1356, 1321}, {1359, 
        1392}, {1360, 1393}, {1361, 1394}, {1362, 1395}, {1363, 1396}, {1364, 
        1397}, {1365, 1398}, {1366, 1399}, {1367, 1400}, {1368, 1401}, {1368, 
        1400}, {1369, 1402}, {1369, 1401}, {1370, 1371}, {1370, 1369}, {1371, 
        1403}, {1371, 1402}, {1372, 1373}, {1373, 1404}, {1373, 1403}, {1374, 
        1306}, {1374, 1339}, {1375, 1307}, {1376, 1308}, {1376, 1341}, {1377, 
        1309}, {1377, 1312}, {1377, 1269}, {1378, 1313}, {1378, 1310}, {1378, 
        1270}, {1379, 1344}, {1379, 1311}, {1380, 1345}, {1380, 1315}, {1381, 
        1346}, {1382, 1347}, {1383, 1348}, {1384, 1349}, {1385, 1350}, {1386, 
        3247, 3248, 3249, 3250, 3251, 3252, 3253, 3254, 3255, 3256, 3257, 
        3258, 3259, 3260, 3261, 3262, 1351}, {1386, 3263, 3264, 3265, 3266, 
        3267, 3268, 3269, 3270, 3271, 3272, 3273, 3274, 3275, 3276, 3277, 
        3278, 1351}, {1387, 1352}, {1388, 1353}, {1389, 1285}, {1390, 1318}, {
        1391, 1320}, {1394, 1432}, {1396, 1434}, {1397, 1435}, {1398, 1436}, {
        1399, 1437}, {1400, 1438}, {1401, 1439}, {1402, 1440}, {1402, 1439}, {
        1403, 1441}, {1403, 1440}, {1404, 1442}, {1404, 1441}, {1405, 1406}, {
        1405, 1404}, {1406, 1443}, {1406, 1442}, {1407, 1408}, {1407, 1406}, {
        1408, 1444}, {1408, 1443}, {1409, 1410}, {1410, 1445}, {1410, 1444}, {
        1411, 1338}, {1411, 1377}, {1411, 1376}, {1412, 1377}, {1413, 1340}, {
        1413, 1343}, {1413, 1311}, {1414, 1380}, {1414, 1341}, {1415, 1381}, {
        1415, 1345}, {1415, 1313}, {1415, 1342}, {1416, 1382}, {1416, 1343}, {
        1416, 1314}, {1417, 1383}, {1417, 1347}, {1418, 1384}, {1419, 1385}, {
        1420, 3279, 3280, 3281, 3282, 3283, 3284, 3285, 3286, 3287, 3288, 
        3289, 3290, 3291, 3292, 3293, 3294, 1386}, {1420, 3295, 3296, 3297, 
        3298, 3299, 3300, 3301, 3302, 3303, 3304, 3305, 3306, 3307, 3308, 
        3309, 3310, 1386}, {1421, 1387}, {1422, 1388}, {1423, 1389}, {1424, 
        1390}, {1425, 1319}, {1426, 1391}, {1427, 1321}, {1428, 1357}, {1429, 
        1358}, {1429, 1430}, {1430, 1358}, {1431, 1393}, {1431, 1360}, {1433, 
        1395}, {1436, 1466}, {1437, 1467}, {1438, 1468}, {1439, 1469}, {1440, 
        1470}, {1441, 1471}, {1442, 1472}, {1443, 1473}, {1444, 1474}, {1444, 
        1473}, {1445, 1475}, {1445, 1474}, {1446, 1445}, {1447, 1448}, {1448, 
        1476}, {1448, 1475}, {1449, 1375}, {1449, 1413}, {1450, 1414}, {1450, 
        1376}, {1451, 1415}, {1451, 1377}, {1452, 1416}, {1452, 1381}, {1452, 
        1378}, {1452, 1343}, {1453, 1417}, {1454, 1418}, {1454, 1383}, {1454, 
        1345}, {1455, 1419}, {1456, 1420}, {1456, 1385}, {1457, 1421}, {1458, 
        3311, 3312, 3313, 3314, 3315, 3316, 3317, 3318, 3319, 3320, 3321, 
        3322, 3323, 3324, 3325, 3326, 1422}, {1458, 3327, 3328, 3329, 3330, 
        3331, 3332, 3333, 3334, 3335, 3336, 3337, 3338, 3339, 3340, 3341, 
        3342, 1422}, {1459, 1423}, {1460, 1424}, {1461, 1425}, {1462, 1426}, {
        1463, 1321}, {1464, 1428}, {1465, 1392}, {1468, 1509}, {1470, 1511}, {
        1471, 1512}, {1472, 1513}, {1473, 1514}, {1474, 1515}, {1474, 1514}, {
        1475, 1516}, {1475, 1515}, {1476, 1517}, {1476, 1516}, {1477, 1478}, {
        1477, 1476}, {1478, 1518}, {1478, 1517}, {1479, 1480}, {1479, 1478}, {
        1480, 1519}, {1480, 1518}, {1481, 1482}, {1481, 1480}, {1482, 1520}, {
        1482, 1519}, {1483, 1484}, {1483, 1482}, {1484, 1521}, {1484, 1520}, {
        1485, 1450}, {1486, 1451}, {1486, 1412}, {1487, 1453}, {1487, 1416}, {
        1487, 1379}, {1487, 1413}, {1488, 1454}, {1488, 1417}, {1489, 1455}, {
        1489, 1418}, {1489, 1381}, {1490, 1456}, {1491, 1457}, {1491, 1420}, {
        1491, 1383}, {1492, 1458}, {1492, 1384}, {1493, 1459}, {1493, 1385}, {
        1493, 1422}, {1494, 1460}, {1495, 1461}, {1495, 1387}, {1496, 1462}, {
        1497, 1463}, {1498, 1464}, {1499, 1430}, {1500, 1465}, {1501, 1393}, {
        1502, 1432}, {1503, 1395}, {1503, 1504}, {1504, 1395}, {1505, 1434}, {
        1506, 1435}, {1506, 1507}, {1507, 1435}, {1508, 1467}, {1508, 1437}, {
        1510, 1469}, {1513, 1548}, {1514, 1549}, {1515, 1550}, {1516, 1551}, {
        1517, 1552}, {1518, 1553}, {1519, 1554}, {1520, 1555}, {1521, 1556}, {
        1522, 1523}, {1522, 1521}, {1523, 1557}, {1523, 1556}, {1524, 1525}, {
        1524, 1523}, {1525, 1558}, {1525, 1557}, {1526, 1527}, {1526, 1525}, {
        1527, 1559}, {1527, 1558}, {1528, 1529}, {1528, 1527}, {1529, 1560}, {
        1530, 1487}, {1530, 1452}, {1530, 1449}, {1530, 1307}, {1531, 1488}, {
        1531, 1453}, {1532, 1489}, {1532, 1454}, {1533, 1490}, {1533, 1455}, {
        1533, 1416}, {1534, 1491}, {1534, 1456}, {1535, 1492}, {1535, 1457}, {
        1536, 1493}, {1537, 1494}, {1537, 1459}, {1538, 1495}, {1539, 1496}, {
        1540, 1497}, {1541, 1498}, {1542, 1499}, {1543, 1500}, {1544, 1501}, {
        1545, 1502}, {1546, 1505}, {1547, 1466}, {1549, 1588}, {1550, 1589}, {
        1551, 1590}, {1552, 1591}, {1553, 1592}, {1554, 1593}, {1555, 1594}, {
        1556, 1595}, {1556, 1594}, {1557, 1596}, {1557, 1595}, {1558, 1597}, {
        1558, 1596}, {1559, 1598}, {1559, 1597}, {1560, 1599}, {1561, 1562}, {
        1562, 1600}, {1563, 1564}, {1564, 1601}, {1565, 1532}, {1565, 1533}, {
        1566, 1534}, {1567, 1535}, {1568, 1536}, {1568, 1492}, {1569, 1537}, {
        1569, 1493}, {1570, 1538}, {1570, 1494}, {1571, 1539}, {1572, 1540}, {
        1573, 1541}, {1574, 1542}, {1575, 1543}, {1576, 1544}, {1577, 1545}, {
        1578, 1504}, {1579, 1546}, {1580, 1507}, {1581, 1547}, {1582, 1467}, {
        1583, 1509}, {1584, 1469}, {1585, 1511}, {1586, 1512}, {1586, 1587}, {
        1587, 1512}, {1589, 1626}, {1590, 1512}, {1590, 1627}, {1591, 1628}, {
        1592, 1629}, {1593, 1630}, {1594, 1631}, {1595, 1632}, {1596, 1633}, {
        1597, 1634}, {1598, 1635}, {1599, 1636}, {1600, 1637}, {1601, 1638}, {
        1602, 1603}, {1603, 1639}, {1604, 1605}, {1605, 1640}, {1606, 1567}, {
        1607, 1568}, {1608, 1569}, {1608, 1536}, {1609, 1570}, {1609, 1537}, {
        1610, 1571}, {1610, 1538}, {1611, 1572}, {1612, 1573}, {1612, 1540}, {
        1613, 1574}, {1614, 1575}, {1615, 1576}, {1616, 1577}, {1617, 1578}, {
        1618, 1579}, {1619, 1580}, {1620, 1581}, {1621, 1582}, {1622, 1583}, {
        1623, 1469}, {1624, 1585}, {1625, 1548}, {1628, 1666}, {1629, 1667}, {
        1630, 1668}, {1631, 1669}, {1632, 1670}, {1633, 1671}, {1634, 1672}, {
        1635, 1673}, {1636, 1674}, {1637, 1675}, {1638, 1676}, {1639, 1677}, {
        1640, 1678}, {1641, 1642}, {1642, 1679}, {1643, 1644}, {1644, 1680}, {
        1645, 1607}, {1645, 1608}, {1645, 1568}, {1646, 1609}, {1647, 1610}, {
        1648, 1611}, {1648, 1571}, {1649, 1612}, {1649, 1572}, {1650, 1613}, {
        1650, 1573}, {1651, 1614}, {1652, 1615}, {1652, 1575}, {1653, 1616}, {
        1654, 1617}, {1655, 1618}, {1656, 1619}, {1657, 1620}, {1658, 1621}, {
        1659, 1622}, {1660, 1623}, {1661, 1624}, {1662, 1587}, {1663, 1625}, {
        1664, 1588}, {1665, 1627}, {1665, 1590}, {1667, 1588}, {1669, 1703}, {
        1669, 1590}, {1670, 1704}, {1671, 1705}, {1671, 1592}, {1672, 1706}, {
        1673, 1707}, {1674, 1708}, {1675, 1709}, {1676, 1710}, {1677, 1711}, {
        1678, 1712}, {1679, 1713}, {1680, 1714}, {1681, 1682}, {1682, 1715}, {
        1683, 1684}, {1684, 1716}, {1685, 1648}, {1686, 1649}, {1687, 1650}, {
        1688, 1651}, {1688, 1613}, {1689, 1652}, {1689, 1614}, {1690, 1653}, {
        1690, 1615}, {1691, 1654}, {1692, 1655}, {1692, 1617}, {1693, 1656}, {
        1694, 1657}, {1695, 1658}, {1696, 1659}, {1697, 1660}, {1698, 1661}, {
        1699, 1662}, {1700, 1663}, {1701, 1664}, {1702, 1626}, {1703, 1627}, {
        1704, 1742}, {1705, 1667}, {1706, 1744}, {1708, 1746}, {1709, 1747}, {
        1710, 1748}, {1711, 1749}, {1712, 1750}, {1713, 1751}, {1714, 1752}, {
        1715, 1753}, {1716, 1754}, {1717, 1718}, {1718, 1755}, {1719, 1720}, {
        1720, 1756}, {1721, 1687}, {1722, 1688}, {1722, 1687}, {1723, 1689}, {
        1723, 1688}, {1723, 1651}, {1724, 1690}, {1725, 1691}, {1725, 1653}, {
        1726, 1692}, {1726, 1654}, {1727, 1693}, {1727, 1655}, {1728, 1694}, {
        1729, 1695}, {1730, 1696}, {1731, 1697}, {1732, 1698}, {1733, 1699}, {
        1734, 1700}, {1735, 1701}, {1736, 1702}, {1737, 1627}, {1738, 3343, 
        3344, 3345, 3346, 3347, 3348, 3349, 3350, 3351, 3352, 3353, 3354, 
        3355, 3356, 3357, 3358, 1666}, {1738, 3359, 3360, 3361, 3362, 3363, 
        3364, 3365, 3366, 3367, 3368, 3369, 3370, 3371, 3372, 3373, 3374, 
        1666}, {1739, 3375, 3376, 3377, 3378, 3379, 3380, 3381, 3382, 3383, 
        3384, 3385, 3386, 3387, 3388, 3389, 3390, 1667}, {1739, 3391, 3392, 
        3393, 3394, 3395, 3396, 3397, 3398, 3399, 3400, 3401, 3402, 3403, 
        3404, 3405, 3406, 1667}, {1740, 1668}, {1740, 1626}, {1741, 1669}, {
        1741, 1703}, {1742, 1666}, {1743, 1705}, {1744, 1668}, {1745, 1707}, {
        1746, 1779}, {1748, 1780}, {1749, 1781}, {1750, 1782}, {1751, 1783}, {
        1752, 1784}, {1753, 1785}, {1754, 1786}, {1755, 1787}, {1756, 1788}, {
        1757, 1758}, {1758, 1789}, {1759, 1760}, {1760, 1790}, {1761, 1723}, {
        1761, 1722}, {1762, 1724}, {1763, 1725}, {1764, 1726}, {1764, 1691}, {
        1765, 1727}, {1765, 1692}, {1766, 1728}, {1766, 1693}, {1767, 1729}, {
        1768, 1730}, {1768, 1695}, {1769, 1731}, {1770, 1732}, {1771, 1733}, {
        1772, 1734}, {1773, 1735}, {1774, 1736}, {1775, 1737}, {1776, 1738}, {
        1777, 1667}, {1778, 1740}, {1782, 1819}, {1783, 1820}, {1784, 1821}, {
        1785, 1822}, {1786, 1823}, {1787, 1824}, {1788, 1825}, {1789, 1826}, {
        1790, 1827}, {1791, 1792}, {1792, 1828}, {1793, 1794}, {1794, 1829}, {
        1795, 1762}, {1795, 1763}, {1796, 1763}, {1796, 1764}, {1797, 1765}, {
        1797, 1764}, {1798, 1766}, {1799, 1767}, {1799, 1728}, {1800, 1768}, {
        1800, 1729}, {1801, 1769}, {1801, 1730}, {1802, 1770}, {1803, 1771}, {
        1804, 1772}, {1805, 1773}, {1806, 1774}, {1807, 1775}, {1808, 1776}, {
        1809, 1777}, {1810, 1778}, {1811, 1703}, {1812, 1742}, {1812, 1738}, {
        1813, 1705}, {1813, 1739}, {1814, 1744}, {1815, 1707}, {1816, 1747}, {
        1816, 1817}, {1817, 1705}, {1817, 1747}, {1818, 1781}, {1818, 1749}, {
        1823, 1853}, {1824, 1854}, {1825, 1855}, {1826, 1856}, {1827, 1857}, {
        1828, 1858}, {1829, 1859}, {1830, 1831}, {1831, 1860}, {1832, 1833}, {
        1833, 1861}, {1834, 1799}, {1835, 1800}, {1835, 1767}, {1836, 1801}, {
        1837, 1802}, {1837, 1769}, {1838, 1803}, {1839, 1804}, {1839, 1771}, {
        1840, 1805}, {1841, 1806}, {1841, 1773}, {1842, 1807}, {1843, 1808}, {
        1843, 1734}, {1843, 1775}, {1844, 1809}, {1845, 1810}, {1846, 1811}, {
        1847, 1812}, {1848, 1777}, {1848, 1705}, {1849, 1814}, {1849, 1778}, {
        1850, 1703}, {1850, 1707}, {1851, 1779}, {1851, 1742}, {1852, 1780}, {
        1855, 1893}, {1856, 1894}, {1857, 1895}, {1858, 1896}, {1859, 1897}, {
        1860, 1898}, {1861, 1899}, {1862, 1863}, {1863, 1900}, {1864, 1865}, {
        1865, 1901}, {1866, 1836}, {1867, 1836}, {1867, 1837}, {1868, 1838}, {
        1868, 1837}, {1869, 1839}, {1869, 1803}, {1870, 1840}, {1870, 1804}, {
        1871, 1841}, {1872, 1842}, {1872, 1806}, {1873, 1843}, {1874, 1844}, {
        1874, 1808}, {1875, 1845}, {1876, 1846}, {1877, 1847}, {1878, 1848}, {
        1879, 1849}, {1879, 1810}, {1880, 1850}, {1880, 1811}, {1881, 1851}, {
        1881, 1812}, {1882, 1817}, {1882, 1813}, {1883, 1852}, {1884, 1781}, {
        1884, 1885}, {1885, 1850}, {1885, 1781}, {1886, 1819}, {1887, 1820}, {
        1887, 1888}, {1888, 1817}, {1888, 1820}, {1889, 1821}, {1890, 1822}, {
        1890, 1891}, {1891, 1781}, {1891, 1822}, {1892, 1854}, {1897, 1925}, {
        1898, 1926}, {1899, 1927}, {1900, 1928}, {1901, 1929}, {1902, 1903}, {
        1903, 1930}, {1904, 1905}, {1905, 1931}, {1906, 1868}, {1907, 1869}, {
        1907, 1838}, {1908, 1870}, {1909, 1871}, {1909, 1840}, {1910, 1872}, {
        1910, 1841}, {1911, 1873}, {1911, 1842}, {1912, 1874}, {1912, 1843}, {
        1913, 1875}, {1913, 1844}, {1914, 1876}, {1915, 1877}, {1915, 1846}, {
        1916, 1878}, {1917, 1879}, {1918, 1880}, {1918, 1846}, {1919, 1881}, {
        1919, 1847}, {1920, 1882}, {1920, 1848}, {1921, 1883}, {1921, 1849}, {
        1922, 1886}, {1923, 1889}, {1924, 1853}, {1926, 1962}, {1927, 1963}, {
        1928, 1964}, {1929, 1965}, {1930, 1966}, {1931, 1967}, {1932, 1933}, {
        1933, 1968}, {1934, 1935}, {1935, 1969}, {1936, 1907}, {1936, 1908}, {
        1937, 1908}, {1937, 1909}, {1938, 1910}, {1938, 1909}, {1938, 1871}, {
        1939, 1911}, {1940, 1912}, {1940, 1873}, {1941, 1913}, {1942, 1914}, {
        1942, 1875}, {1943, 1915}, {1943, 1876}, {1944, 1916}, {1945, 1917}, {
        1946, 1918}, {1947, 1919}, {1947, 1877}, {1948, 1920}, {1948, 1878}, {
        1949, 1921}, {1949, 1879}, {1950, 1885}, {1950, 1880}, {1951, 1922}, {
        1952, 1888}, {1953, 1923}, {1954, 1891}, {1954, 1884}, {1955, 1924}, {
        1956, 1854}, {1957, 1893}, {1958, 1894}, {1959, 1895}, {1960, 1896}, {
        1960, 1961}, {1961, 1854}, {1961, 1896}, {1965, 1996}, {1966, 1997}, {
        1966, 1898}, {1967, 1998}, {1968, 1999}, {1969, 2000}, {1970, 1971}, {
        1971, 2001}, {1972, 1973}, {1973, 2002}, {1974, 1939}, {1975, 1940}, {
        1976, 1941}, {1976, 1912}, {1977, 1942}, {1977, 1913}, {1978, 1943}, {
        1978, 1914}, {1979, 1944}, {1979, 1915}, {1980, 1945}, {1980, 1916}, {
        1981, 1946}, {1982, 1947}, {1983, 1916}, {1983, 1948}, {1984, 1917}, {
        1984, 1949}, {1985, 1950}, {1985, 1918}, {1986, 1951}, {1986, 1919}, {
        1987, 1952}, {1987, 1920}, {1988, 1953}, {1989, 1954}, {1990, 1955}, {
        1991, 1956}, {1992, 1957}, {1993, 1891}, {1993, 1894}, {1994, 1959}, {
        1995, 1925}, {1998, 2033}, {1999, 2034}, {2000, 2035}, {2001, 2036}, {
        2002, 2037}, {2003, 2004}, {2004, 2038}, {2005, 2006}, {2006, 2039}, {
        2007, 1975}, {2008, 1976}, {2008, 1977}, {2009, 1978}, {2009, 1977}, {
        2010, 1979}, {2011, 1980}, {2011, 1944}, {2012, 1981}, {2013, 1982}, {
        2014, 1983}, {2015, 1945}, {2015, 1984}, {2016, 1946}, {2016, 1985}, {
        2017, 1986}, {2017, 1947}, {2018, 1987}, {2018, 1948}, {2019, 1988}, {
        2020, 1989}, {2021, 1990}, {2022, 1991}, {2023, 1992}, {2024, 1993}, {
        2025, 1994}, {2026, 3407, 3408, 3409, 3410, 3411, 3412, 3413, 3414, 
        3415, 3416, 3417, 3418, 3419, 3420, 3421, 3422, 1961}, {2026, 3423, 
        3424, 3425, 3426, 3427, 3428, 3429, 3430, 3431, 3432, 3433, 3434, 
        3435, 3436, 3437, 3438, 1961}, {2027, 1995}, {2028, 1962}, {2029, 
        1963}, {2030, 1964}, {2030, 2031}, {2031, 1961}, {2031, 1964}, {2032, 
        1997}, {2032, 1966}, {2037, 2065}, {2038, 2066}, {2038, 1968}, {2039, 
        2067}, {2040, 2041}, {2041, 2068}, {2042, 2043}, {2043, 2069}, {2044, 
        2010}, {2045, 2011}, {2045, 1979}, {2046, 2012}, {2047, 2013}, {2047, 
        1981}, {2048, 2014}, {2048, 1982}, {2049, 1980}, {2049, 1983}, {2050, 
        1981}, {2050, 2016}, {2051, 1982}, {2051, 2017}, {2052, 2018}, {2052, 
        1983}, {2053, 2019}, {2053, 1984}, {2054, 2020}, {2054, 1985}, {2055, 
        2021}, {2056, 2022}, {2057, 2023}, {2058, 2024}, {2059, 2025}, {2060, 
        2026}, {2061, 2027}, {2062, 2028}, {2063, 2029}, {2064, 1996}, {2067, 
        2100}, {2068, 2101}, {2069, 2102}, {2070, 2071}, {2071, 2103}, {2072, 
        2073}, {2073, 2104}, {2074, 2045}, {2074, 2046}, {2075, 2046}, {2075, 
        2047}, {2076, 2048}, {2076, 2013}, {2077, 2011}, {2077, 2049}, {2077, 
        2048}, {2078, 2050}, {2079, 2013}, {2079, 2051}, {2080, 2014}, {2080, 
        2052}, {2081, 2053}, {2081, 2015}, {2082, 2054}, {2082, 2016}, {2083, 
        2055}, {2083, 2017}, {2084, 2056}, {2084, 2018}, {2085, 2057}, {2086, 
        2058}, {2087, 2059}, {2088, 2060}, {2089, 2061}, {2090, 2062}, {2091, 
        2063}, {2092, 2031}, {2093, 2064}, {2094, 1997}, {2095, 2033}, {2096, 
        2034}, {2097, 2035}, {2098, 2036}, {2099, 2066}, {2104, 2134}, {2105, 
        2106}, {2106, 2135}, {2107, 2108}, {2108, 2136}, {2109, 2110}, {2110, 
        2137}, {2111, 2077}, {2112, 2078}, {2112, 2046}, {2113, 2079}, {2113, 
        2047}, {2114, 2048}, {2114, 2080}, {2115, 2049}, {2115, 2081}, {2116, 
        2082}, {2116, 2050}, {2117, 2083}, {2117, 2051}, {2118, 2084}, {2118, 
        2052}, {2119, 2085}, {2119, 2053}, {2120, 2086}, {2120, 2054}, {2121, 
        2087}, {2121, 2055}, {2122, 2088}, {2123, 2089}, {2124, 2090}, {2125, 
        2091}, {2126, 3439, 3440, 3441, 3442, 3443, 3444, 3445, 3446, 3447, 
        3448, 3449, 3450, 3451, 3452, 3453, 3454, 2092}, {2126, 3455, 3456, 
        3457, 3458, 3459, 3460, 3461, 3462, 3463, 3464, 3465, 3466, 3467, 
        3468, 3469, 3470, 2092}, {2127, 2093}, {2128, 2094}, {2129, 2095}, {
        2130, 2096}, {2131, 2097}, {2132, 1997}, {2132, 2036}, {2133, 2065}, {
        2135, 2173}, {2136, 2174}, {2137, 2175}, {2138, 2139}, {2139, 2176}, {
        2140, 2141}, {2141, 2177}, {2142, 2143}, {2143, 2178}, {2144, 2145}, {
        2145, 2179}, {2146, 2112}, {2147, 2113}, {2147, 2114}, {2148, 2077}, {
        2148, 2114}, {2148, 2115}, {2149, 2078}, {2149, 2116}, {2150, 2117}, {
        2150, 2079}, {2151, 2118}, {2151, 2080}, {2152, 2119}, {2152, 2081}, {
        2153, 2120}, {2153, 2082}, {2154, 2121}, {2154, 2083}, {2155, 2122}, {
        2156, 2123}, {2157, 2124}, {2158, 2125}, {2159, 2126}, {2160, 2127}, {
        2161, 2128}, {2162, 2129}, {2163, 2130}, {2164, 2131}, {2165, 2132}, {
        2166, 2133}, {2167, 2066}, {2168, 2100}, {2169, 2101}, {2170, 2102}, {
        2171, 2103}, {2171, 2172}, {2172, 2066}, {2172, 2103}, {2173, 2101}, {
        2174, 2102}, {2175, 2103}, {2176, 2207}, {2177, 2208}, {2177, 2106}, {
        2178, 2209}, {2179, 2210}, {2180, 2181}, {2181, 2211}, {2182, 2183}, {
        2183, 2212}, {2184, 2112}, {2185, 2113}, {2185, 2150}, {2186, 2114}, {
        2186, 2151}, {2187, 2115}, {2187, 2152}, {2188, 2153}, {2188, 2116}, {
        2189, 2154}, {2189, 2117}, {2190, 2155}, {2190, 2118}, {2191, 2156}, {
        2191, 2119}, {2192, 2157}, {2192, 2120}, {2193, 2158}, {2193, 2121}, {
        2194, 2159}, {2194, 2122}, {2195, 2160}, {2196, 2161}, {2196, 2124}, {
        2197, 2162}, {2198, 2163}, {2199, 2164}, {2200, 2165}, {2201, 2166}, {
        2202, 2167}, {2203, 2168}, {2204, 2169}, {2205, 2170}, {2206, 2134}, {
        2208, 2173}, {2209, 2243}, {2209, 2136}, {2210, 2244}, {2211, 2245}, {
        2212, 2246}, {2213, 2214}, {2214, 2247}, {2215, 2216}, {2216, 2248}, {
        2217, 2185}, {2217, 2147}, {2218, 2186}, {2218, 2148}, {2219, 2149}, {
        2219, 2188}, {2220, 2189}, {2220, 2150}, {2221, 2151}, {2222, 2191}, {
        2222, 2152}, {2223, 2192}, {2223, 2153}, {2224, 2154}, {2224, 2193}, {
        2225, 2194}, {2225, 2155}, {2226, 2195}, {2226, 2156}, {2227, 2196}, {
        2227, 2157}, {2228, 2197}, {2229, 2198}, {2230, 2199}, {2231, 2200}, {
        2232, 2201}, {2233, 2202}, {2234, 2203}, {2235, 2204}, {2236, 2205}, {
        2237, 2172}, {2238, 2206}, {2239, 2173}, {2240, 2174}, {2241, 2175}, {
        2242, 2208}, {2242, 2177}, {2247, 2277}, {2248, 2278}, {2248, 2179}, {
        2249, 2250}, {2250, 2279}, {2251, 2252}, {2252, 2280}, {2253, 2184}, {
        2254, 2185}, {2254, 2188}, {2254, 2220}, {2255, 2186}, {2255, 2221}, {
        2256, 2187}, {2256, 2222}, {2257, 2188}, {2257, 2223}, {2258, 2189}, {
        2258, 2224}, {2259, 2225}, {2259, 2190}, {2260, 2226}, {2260, 2191}, {
        2261, 2227}, {2261, 2192}, {2262, 2228}, {2262, 2193}, {2263, 2229}, {
        2263, 2194}, {2264, 2230}, {2264, 2195}, {2265, 2231}, {2265, 2196}, {
        2266, 2232}, {2267, 2233}, {2268, 2234}, {2269, 2235}, {2270, 2236}, {
        2271, 2237}, {2272, 2238}, {2273, 2239}, {2274, 2240}, {2275, 2172}, {
        2275, 2175}, {2276, 2207}, {2280, 2313}, {2281, 2282}, {2282, 2314}, {
        2283, 2284}, {2284, 2315}, {2285, 2254}, {2285, 2217}, {2285, 2255}, {
        2286, 2255}, {2286, 2218}, {2287, 2219}, {2287, 2256}, {2288, 2220}, {
        2288, 2257}, {2288, 2258}, {2289, 2221}, {2290, 2222}, {2290, 2260}, {
        2291, 2261}, {2291, 2223}, {2292, 2224}, {2292, 2262}, {2293, 2263}, {
        2293, 2225}, {2294, 2264}, {2294, 2226}, {2295, 2265}, {2295, 2227}, {
        2296, 2266}, {2296, 2228}, {2297, 2267}, {2297, 2229}, {2298, 2268}, {
        2298, 2230}, {2299, 2269}, {2300, 2270}, {2301, 2271}, {2302, 2272}, {
        2303, 2273}, {2304, 2274}, {2304, 2236}, {2305, 2275}, {2306, 2276}, {
        2307, 2208}, {2308, 2243}, {2309, 2244}, {2310, 2245}, {2311, 3471, 
        3472, 3473, 3474, 3475, 3476, 3477, 3478, 3479, 3480, 3481, 3482, 
        3483, 3484, 3485, 3486, 2246}, {2311, 3487, 3488, 3489, 3490, 3491, 
        3492, 3493, 3494, 3495, 3496, 3497, 3498, 3499, 3500, 3501, 3502, 
        2246}, {2312, 2278}, {2312, 2248}, {2316, 2317}, {2317, 2349}, {2318, 
        2319}, {2319, 2350}, {2319, 2252}, {2320, 2321}, {2321, 2351}, {2322, 
        2253}, {2323, 2254}, {2324, 2255}, {2324, 2289}, {2325, 2256}, {2325, 
        2290}, {2326, 2257}, {2326, 2291}, {2327, 2258}, {2327, 2292}, {2328, 
        2259}, {2328, 2293}, {2329, 2260}, {2329, 2294}, {2330, 2261}, {2330, 
        2295}, {2331, 2262}, {2331, 2296}, {2332, 2297}, {2332, 2263}, {2333, 
        2298}, {2333, 2264}, {2334, 2299}, {2334, 2265}, {2335, 2300}, {2335, 
        2266}, {2336, 2301}, {2336, 2267}, {2337, 2302}, {2337, 2268}, {2338, 
        2303}, {2338, 2269}, {2339, 2304}, {2339, 2270}, {2340, 2305}, {2340, 
        2271}, {2341, 2306}, {2341, 2272}, {2342, 2307}, {2342, 2273}, {2343, 
        2308}, {2344, 2309}, {2344, 2275}, {2345, 2310}, {2346, 2208}, {2346, 
        2246}, {2347, 2277}, {2348, 2279}, {2351, 2388}, {2352, 2353}, {2353, 
        2389}, {2354, 2355}, {2355, 2390}, {2356, 2357}, {2357, 2391}, {2358, 
        2286}, {2358, 2325}, {2359, 2325}, {2359, 2287}, {2360, 2326}, {2360, 
        2288}, {2361, 2289}, {2361, 2327}, {2362, 2290}, {2362, 2329}, {2363, 
        2291}, {2363, 2330}, {2364, 2292}, {2364, 2331}, {2365, 2293}, {2365, 
        2332}, {2366, 2294}, {2366, 2333}, {2367, 2334}, {2367, 2295}, {2368, 
        2335}, {2368, 2296}, {2369, 2336}, {2369, 2297}, {2370, 2337}, {2370, 
        2298}, {2371, 2338}, {2371, 2299}, {2372, 2339}, {2372, 2300}, {2373, 
        2340}, {2373, 2301}, {2374, 2341}, {2374, 2302}, {2375, 2342}, {2375, 
        2303}, {2376, 2343}, {2376, 2304}, {2377, 2344}, {2378, 2345}, {2378, 
        2306}, {2379, 2346}, {2379, 2307}, {2380, 2347}, {2381, 2278}, {2382, 
        2348}, {2382, 2310}, {2383, 2313}, {2384, 2314}, {2385, 2315}, {2385, 
        2386}, {2386, 2315}, {2387, 2350}, {2392, 2393}, {2393, 2424}, {2394, 
        2395}, {2395, 2425}, {2396, 2397}, {2397, 2426}, {2398, 2323}, {2398, 
        2360}, {2399, 2324}, {2400, 2325}, {2401, 2326}, {2401, 2363}, {2402, 
        2327}, {2402, 2364}, {2403, 2328}, {2403, 2365}, {2404, 2329}, {2404, 
        2366}, {2405, 2330}, {2405, 2367}, {2406, 2331}, {2406, 2334}, {2407, 
        2369}, {2407, 2332}, {2408, 2333}, {2408, 2336}, {2408, 2298}, {2409, 
        2371}, {2409, 2334}, {2409, 2337}, {2410, 2372}, {2410, 2335}, {2410, 
        2338}, {2411, 2373}, {2411, 2336}, {2412, 2374}, {2412, 2337}, {2413, 
        2375}, {2413, 2338}, {2414, 2376}, {2414, 2339}, {2415, 2377}, {2415, 
        2340}, {2416, 2378}, {2416, 2341}, {2417, 3503, 3504, 3505, 3506, 
        3507, 3508, 3509, 3510, 3511, 3512, 3513, 3514, 3515, 3516, 3517, 
        3518, 2379}, {2417, 3519, 3520, 3521, 3522, 3523, 3524, 3525, 3526, 
        3527, 3528, 3529, 3530, 3531, 3532, 3533, 3534, 2379}, {2417, 2342}, {
        2418, 2380}, {2418, 2343}, {2419, 2381}, {2419, 2344}, {2420, 2382}, {
        2421, 2383}, {2422, 2384}, {2423, 2349}, {2425, 2389}, {2427, 2428}, {
        2428, 2466}, {2429, 2430}, {2430, 2467}, {2431, 2432}, {2432, 2468}, {
        2433, 2434}, {2434, 2469}, {2435, 2436}, {2436, 2470}, {2436, 2469}, {
        2437, 2361}, {2438, 2362}, {2438, 2403}, {2439, 2363}, {2439, 2405}, {
        2440, 2364}, {2440, 2406}, {2441, 2365}, {2441, 2442}, {2443, 2366}, {
        2443, 2408}, {2444, 2409}, {2444, 2367}, {2445, 2410}, {2445, 2368}, {
        2446, 2411}, {2446, 2369}, {2447, 2412}, {2447, 2370}, {2448, 2413}, {
        2449, 2414}, {2449, 2372}, {2450, 2415}, {2451, 2416}, {2452, 2417}, {
        2453, 2418}, {2454, 2419}, {2455, 2420}, {2456, 2421}, {2456, 2379}, {
        2457, 2422}, {2458, 2386}, {2459, 2423}, {2460, 2350}, {2461, 2388}, {
        2462, 2389}, {2463, 2390}, {2464, 2391}, {2465, 2425}, {2465, 2395}, {
        2469, 2502}, {2470, 2503}, {2470, 2427}, {2471, 2472}, {2472, 2504}, {
        2473, 2474}, {2474, 2505}, {2475, 2401}, {2475, 2439}, {2476, 2402}, {
        2477, 2403}, {2478, 2404}, {2478, 2443}, {2479, 2405}, {2479, 2444}, {
        2480, 2406}, {2480, 2445}, {2481, 2407}, {2481, 2446}, {2482, 2408}, {
        2482, 2447}, {2483, 2448}, {2483, 2409}, {2484, 2449}, {2484, 2410}, {
        2485, 2450}, {2485, 2411}, {2486, 2451}, {2486, 2412}, {2487, 2452}, {
        2487, 2413}, {2488, 2453}, {2488, 2414}, {2489, 2454}, {2489, 2415}, {
        2490, 2455}, {2491, 2456}, {2491, 2417}, {2492, 2457}, {2493, 2458}, {
        2493, 2419}, {2494, 2459}, {2495, 2460}, {2496, 2461}, {2497, 2462}, {
        2498, 2463}, {2499, 2464}, {2499, 2350}, {2500, 2424}, {2501, 2426}, {
        2502, 2426}, {2503, 2537}, {2503, 2428}, {2504, 2430}, {2504, 2538}, {
        2505, 2539}, {2505, 2432}, {2505, 2468}, {2506, 2507}, {2507, 2540}, {
        2507, 2434}, {2508, 2509}, {2509, 2541}, {2509, 2436}, {2509, 2470}, {
        2510, 2511}, {2511, 2542}, {2512, 2441}, {2513, 2481}, {2513, 2443}, {
        2514, 2444}, {2514, 2483}, {2515, 2445}, {2515, 2484}, {2516, 2446}, {
        2516, 2485}, {2517, 2486}, {2517, 2447}, {2518, 2448}, {2518, 2487}, {
        2519, 2449}, {2519, 2488}, {2520, 2489}, {2520, 2450}, {2521, 2490}, {
        2521, 2451}, {2522, 2491}, {2522, 2452}, {2523, 2492}, {2523, 2453}, {
        2524, 2493}, {2524, 2454}, {2525, 2494}, {2525, 2455}, {2526, 2495}, {
        2527, 2496}, {2528, 2497}, {2529, 2498}, {2529, 2459}, {2530, 2499}, {
        2530, 2460}, {2531, 2500}, {2531, 2461}, {2532, 2425}, {2533, 2501}, {
        2534, 2466}, {2535, 2467}, {2536, 2468}, {2537, 2466}, {2538, 2467}, {
        2539, 2468}, {2540, 2469}, {2541, 2470}, {2542, 2472}, {2542, 2571}, {
        2543, 2544}, {2544, 2474}, {2544, 2572}, {2545, 2546}, {2546, 2506}, {
        2546, 2573}, {2547, 2548}, {2548, 2508}, {2548, 2574}, {2549, 2481}, {
        2549, 2516}, {2550, 2482}, {2550, 2517}, {2551, 2483}, {2551, 2518}, {
        2552, 2484}, {2552, 2519}, {2553, 2485}, {2553, 2520}, {2554, 2486}, {
        2554, 2521}, {2555, 2487}, {2555, 2522}, {2556, 2488}, {2556, 2523}, {
        2557, 2489}, {2557, 2524}, {2558, 2525}, {2558, 2490}, {2559, 2491}, {
        2559, 2526}, {2560, 2527}, {2560, 2492}, {2561, 2528}, {2561, 2493}, {
        2562, 2529}, {2562, 2494}, {2563, 2530}, {2563, 2495}, {2564, 2531}, {
        2564, 2496}, {2565, 2532}, {2565, 2497}, {2566, 2533}, {2566, 2498}, {
        2567, 2534}, {2567, 2499}, {2568, 2535}, {2568, 2500}, {2569, 2425}, {
        2569, 2536}, {2570, 2501}, {2570, 2502}, {2571, 2504}, {2572, 2539}, {
        2573, 2507}, {2573, 2603}, {2574, 2509}, {2574, 2604}, {2575, 2576}, {
        2575, 2510}, {2576, 2511}, {2576, 2605}, {2577, 2578}, {2578, 2606}, {
        2578, 2543}, {2579, 2517}, {2580, 2518}, {2580, 2555}, {2581, 2519}, {
        2581, 2556}, {2582, 2520}, {2582, 2557}, {2583, 2521}, {2583, 2558}, {
        2584, 2522}, {2584, 2559}, {2585, 2523}, {2585, 2560}, {2586, 2524}, {
        2586, 2561}, {2587, 2525}, {2587, 2562}, {2588, 2563}, {2588, 2526}, {
        2589, 2564}, {2589, 2527}, {2590, 2565}, {2590, 2528}, {2591, 2566}, {
        2591, 2529}, {2592, 2567}, {2592, 2530}, {2593, 2568}, {2593, 2531}, {
        2594, 2569}, {2594, 2532}, {2595, 2570}, {2595, 2533}, {2596, 2537}, {
        2596, 2534}, {2597, 2538}, {2597, 2535}, {2598, 2536}, {2598, 2539}, {
        2598, 2599}, {2599, 2569}, {2599, 2539}, {2600, 2502}, {2601, 2503}, {
        2602, 2505}, {2602, 2572}, {2602, 2544}, {2603, 2540}, {2604, 2541}, {
        2605, 2542}, {2606, 2544}, {2606, 2634}, {2607, 2608}, {2608, 2635}, {
        2608, 2546}, {2609, 2610}, {2610, 2548}, {2611, 2612}, {2611, 2613}, {
        2612, 2636}, {2612, 2575}, {2614, 2552}, {2615, 2553}, {2615, 2582}, {
        2616, 2554}, {2616, 2583}, {2617, 2555}, {2617, 2584}, {2618, 2556}, {
        2618, 2585}, {2619, 2557}, {2619, 2586}, {2620, 2558}, {2620, 2587}, {
        2621, 2559}, {2621, 2588}, {2622, 2560}, {2622, 2589}, {2623, 2561}, {
        2623, 2590}, {2624, 2591}, {2624, 2562}, {2625, 2563}, {2625, 2592}, {
        2626, 2593}, {2626, 2564}, {2627, 2565}, {2627, 2594}, {2628, 2595}, {
        2628, 2566}, {2629, 2567}, {2629, 2596}, {2630, 2597}, {2630, 2568}, {
        2631, 2570}, {2632, 2537}, {2632, 2541}, {2633, 2538}, {2634, 2572}, {
        2635, 2573}, {2635, 2669}, {2635, 2430}, {2636, 2672}, {2636, 2576}, {
        2637, 2638}, {2638, 2673}, {2639, 2640}, {2640, 2674}, {2641, 2642}, {
        2642, 2675}, {2643, 2644}, {2644, 2676}, {2645, 2646}, {2646, 2677}, {
        2647, 2581}, {2647, 2618}, {2648, 2582}, {2649, 2583}, {2649, 2620}, {
        2650, 2584}, {2650, 2621}, {2651, 2585}, {2651, 2622}, {2652, 2586}, {
        2652, 2623}, {2653, 2587}, {2653, 2624}, {2654, 2625}, {2654, 2588}, {
        2655, 2589}, {2655, 2626}, {2656, 2590}, {2656, 2627}, {2657, 2591}, {
        2657, 2628}, {2658, 2592}, {2658, 2629}, {2659, 2593}, {2659, 2630}, {
        2660, 2599}, {2660, 2594}, {2661, 2595}, {2661, 2631}, {2662, 2596}, {
        2663, 2597}, {2664, 2598}, {2664, 2572}, {2665, 2600}, {2666, 2601}, {
        2667, 2571}, {2668, 2602}, {2668, 2634}, {2669, 2603}, {2669, 2467}, {
        2670, 2671}, {2671, 2604}, {2671, 2468}, {2672, 2605}, {2672, 2469}, {
        2673, 2606}, {2673, 2470}, {2674, 2704}, {2675, 2610}, {2675, 2474}, {
        2675, 2705}, {2676, 2706}, {2677, 2707}, {2678, 2679}, {2679, 2708}, {
        2680, 2681}, {2681, 2709}, {2682, 2683}, {2683, 2710}, {2684, 2685}, {
        2685, 2711}, {2686, 2617}, {2687, 2618}, {2687, 2651}, {2688, 2619}, {
        2688, 2652}, {2689, 2620}, {2689, 2653}, {2690, 2621}, {2691, 2622}, {
        2691, 2655}, {2692, 2623}, {2692, 2656}, {2693, 2624}, {2693, 2657}, {
        2694, 2625}, {2694, 2658}, {2695, 2626}, {2695, 2659}, {2696, 2627}, {
        2696, 2660}, {2697, 2628}, {2697, 2661}, {2698, 2629}, {2698, 2662}, {
        2699, 2630}, {2700, 2599}, {2700, 2664}, {2701, 2631}, {2702, 2632}, {
        2702, 2604}, {2703, 2633}, {2704, 2635}, {2704, 2504}, {2705, 2671}, {
        2705, 2468}, {2706, 2737}, {2706, 2636}, {2707, 2738}, {2708, 2739}, {
        2709, 2740}, {2709, 2442}, {2710, 2741}, {2711, 2742}, {2712, 2713}, {
        2713, 2743}, {2714, 2715}, {2715, 2744}, {2716, 2650}, {2716, 2690}, {
        2717, 2651}, {2718, 2652}, {2718, 2692}, {2719, 2653}, {2719, 2693}, {
        2720, 2654}, {2720, 2694}, {2721, 2655}, {2721, 2695}, {2722, 2656}, {
        2722, 2696}, {2723, 2657}, {2723, 2697}, {2724, 2658}, {2724, 2698}, {
        2725, 2659}, {2725, 2699}, {2726, 2660}, {2726, 2700}, {2727, 2661}, {
        2727, 2701}, {2728, 2662}, {2729, 2663}, {2729, 2703}, {2730, 2664}, {
        2730, 2634}, {2731, 2665}, {2732, 2666}, {2732, 2671}, {2733, 2667}, {
        2733, 2672}, {2734, 2673}, {2734, 2668}, {2734, 2735}, {2735, 2634}, {
        2735, 2673}, {2736, 2705}, {2736, 2675}, {2736, 2670}, {2737, 2672}, {
        2738, 2673}, {2738, 2468}, {2739, 2674}, {2740, 2675}, {2740, 2427}, {
        2740, 2442}, {2741, 2765}, {2741, 2676}, {2742, 2677}, {2742, 2442}, {
        2742, 2766}, {2742, 2767}, {2743, 2768}, {2744, 2769}, {2745, 2746}, {
        2746, 2770}, {2747, 2748}, {2748, 2771}, {2749, 2689}, {2749, 2719}, {
        2750, 2690}, {2750, 2720}, {2751, 2691}, {2751, 2721}, {2752, 2692}, {
        2752, 2722}, {2753, 2693}, {2753, 2723}, {2754, 2694}, {2754, 2724}, {
        2755, 2695}, {2756, 2696}, {2756, 2726}, {2757, 2697}, {2758, 2698}, {
        2759, 2699}, {2759, 2729}, {2760, 2700}, {2760, 2730}, {2761, 2701}, {
        2762, 2702}, {2762, 2732}, {2763, 2703}, {2764, 2669}, {2764, 2704}, {
        2765, 2706}, {2765, 2430}, {2765, 2442}, {2765, 2468}, {2767, 2738}, {
        2767, 2468}, {2767, 2395}, {2767, 2442}, {2768, 2797}, {2769, 2798}, {
        2769, 2442}, {2770, 2799}, {2771, 2800}, {2771, 2442}, {2772, 2773}, {
        2773, 2801}, {2774, 2775}, {2775, 2802}, {2775, 2442}, {2776, 2718}, {
        2777, 2719}, {2778, 2720}, {2779, 2721}, {2780, 2722}, {2780, 2756}, {
        2781, 2723}, {2782, 2724}, {2783, 2725}, {2783, 2759}, {2784, 2726}, {
        2784, 2760}, {2785, 2727}, {2786, 2728}, {2787, 2729}, {2787, 2763}, {
        2788, 2730}, {2788, 2735}, {2789, 2731}, {2790, 2732}, {2790, 2705}, {
        2791, 2733}, {2791, 2737}, {2792, 2738}, {2792, 2734}, {2793, 2739}, {
        2793, 2704}, {2794, 2740}, {2794, 2795}, {2794, 2736}, {2795, 2705}, {
        2795, 2442}, {2795, 2468}, {2795, 2740}, {2796, 2767}, {2796, 2742}, {
        2797, 2739}, {2797, 2442}, {2797, 2469}, {2797, 2397}, {2798, 2740}, {
        2798, 2442}, {2798, 2427}, {2798, 2110}, {2799, 2741}, {2799, 2442}, {
        2799, 2429}, {2799, 2470}, {2799, 2510}, {2800, 2742}, {2800, 2442}, {
        2801, 2821}, {2802, 2744}, {2802, 2442}, {2802, 2822}, {2803, 2804}, {
        2804, 2823}, {2805, 2806}, {2806, 2824}, {2806, 2748}, {2807, 2753}, {
        2808, 2754}, {2809, 2755}, {2809, 2783}, {2810, 2756}, {2810, 2784}, {
        2811, 2757}, {2811, 2785}, {2812, 2758}, {2812, 2786}, {2813, 2759}, {
        2813, 2787}, {2814, 2760}, {2814, 2788}, {2815, 2761}, {2816, 2762}, {
        2817, 2763}, {2817, 2791}, {2818, 2735}, {2818, 2792}, {2819, 2793}, {
        2819, 2764}, {2820, 2765}, {2820, 2737}, {2821, 2768}, {2821, 2442}, {
        2821, 2430}, {2822, 2798}, {2822, 2442}, {2822, 2427}, {2822, 2477}, {
        2823, 2843}, {2823, 2770}, {2824, 2844}, {2825, 2826}, {2826, 2845}, {
        2827, 2828}, {2828, 2846}, {2829, 2785}, {2830, 2786}, {2830, 2816}, {
        2831, 2787}, {2831, 2817}, {2832, 2818}, {2832, 2788}, {2832, 2442}, {
        2833, 2819}, {2833, 2789}, {2834, 2795}, {2834, 2790}, {2835, 2820}, {
        2835, 2791}, {2836, 2767}, {2836, 2792}, {2837, 2797}, {2837, 2793}, {
        2838, 2798}, {2839, 2799}, {2839, 2765}, {2840, 2800}, {2840, 2841}, {
        2840, 2796}, {2841, 2767}, {2841, 2442}, {2841, 2468}, {2841, 2800}, {
        2842, 2822}, {2843, 2799}, {2843, 2442}, {2844, 2800}, {2844, 2442}, {
        2844, 2427}, {2845, 2860}, {2845, 2801}, {2845, 2442}, {2846, 2802}, {
        2846, 2442}, {2847, 2848}, {2848, 2861}, {2849, 2850}, {2850, 2806}, {
        2850, 2442}, {2850, 2862}, {2851, 2814}, {2851, 2832}, {2852, 2815}, {
        2853, 2816}, {2853, 2834}, {2854, 2835}, {2854, 2817}, {2855, 2836}, {
        2855, 2818}, {2856, 2837}, {2856, 2819}, {2857, 2838}, {2857, 2795}, {
        2858, 2839}, {2858, 2820}, {2859, 2821}, {2859, 2797}, {2860, 2821}, {
        2860, 2442}, {2860, 2430}, {2861, 2823}, {2861, 2442}, {2862, 2844}, {
        2862, 2442}, {2863, 2864}, {2864, 2882}, {2865, 2866}, {2866, 2883}, {
        2867, 2868}, {2868, 2884}, {2869, 2854}, {2869, 2831}, {2870, 2832}, {
        2870, 2442}, {2871, 2856}, {2871, 2833}, {2872, 2857}, {2872, 2834}, {
        2872, 2442}, {2873, 2858}, {2873, 2835}, {2874, 2841}, {2874, 2836}, {
        2875, 2859}, {2875, 2837}, {2876, 2822}, {2876, 2838}, {2877, 2843}, {
        2877, 2839}, {2878, 2844}, {2878, 2840}, {2879, 2880}, {2879, 2846}, {
        2880, 2822}, {2880, 2442}, {2880, 2468}, {2880, 2846}, {2881, 2862}, {
        2882, 2845}, {2882, 2442}, {2883, 2846}, {2883, 2442}, {2884, 2848}, {
        2885, 2886}, {2886, 2850}, {2886, 2442}, {2886, 2899}, {2887, 2888}, {
        2888, 2900}, {2889, 2852}, {2889, 2871}, {2890, 2853}, {2890, 2872}, {
        2890, 2442}, {2891, 2873}, {2891, 2854}, {2892, 2874}, {2892, 2855}, {
        2893, 2875}, {2893, 2856}, {2894, 2876}, {2894, 2857}, {2895, 2877}, {
        2895, 2858}, {2896, 2841}, {2896, 2878}, {2896, 2442}, {2897, 2860}, {
        2897, 2859}, {2898, 2843}, {2898, 2861}, {2898, 2442}, {2899, 2862}, {
        2899, 2442}, {2900, 2921}, {2900, 2864}, {2900, 2442}, {2901, 2442}, {
        2901, 2865}, {2901, 2902}, {2902, 2922}, {2903, 2904}, {2904, 2923}, {
        2905, 2906}, {2906, 2924}, {2906, 2885}, {2907, 2891}, {2907, 2869}, {
        2908, 2892}, {2908, 2870}, {2909, 2893}, {2909, 2871}, {2910, 2894}, {
        2910, 2442}, {2910, 2872}, {2911, 2895}, {2911, 2873}, {2912, 2874}, {
        2912, 2442}, {2913, 2875}, {2913, 2897}, {2914, 2880}, {2914, 2442}, {
        2914, 2876}, {2915, 2898}, {2915, 2877}, {2916, 2862}, {2916, 2878}, {
        2917, 2882}, {2917, 2860}, {2918, 2883}, {2918, 2879}, {2919, 2861}, {
        2919, 2442}, {2920, 2881}, {2921, 2882}, {2921, 2442}, {2922, 2883}, {
        2922, 2442}, {2923, 2884}, {2923, 2442}, {2924, 2886}, {2924, 2442}, {
        2925, 2926}, {2926, 2940}, {2926, 2888}, {2927, 2928}, {2928, 2442}, {
        2928, 2901}, {2928, 2941}, {2929, 2442}, {2929, 2909}, {2929, 2889}, {
        2930, 2442}, {2930, 2890}, {2930, 2910}, {2931, 2891}, {2931, 2911}, {
        2932, 2892}, {2932, 2442}, {2932, 2912}, {2933, 2913}, {2933, 2893}, {
        2934, 2894}, {2934, 2914}, {2934, 2442}, {2935, 2915}, {2935, 2895}, {
        2936, 2896}, {2936, 2916}, {2937, 2917}, {2937, 2897}, {2938, 2880}, {
        2938, 2918}, {2938, 2442}, {2939, 2919}, {2939, 2898}, {2940, 2900}, {
        2940, 2442}, {2941, 2922}, {2941, 2442}, {2942, 2943}, {2942, 2442}, {
        2942, 2903}, {2943, 2960}, {2943, 2904}, {2943, 2442}, {2944, 2442}, {
        2944, 2905}, {2944, 2945}, {2945, 2442}, {2945, 2924}, {2946, 2908}, {
        2946, 2932}, {2947, 2909}, {2947, 2933}, {2948, 2442}, {2948, 2910}, {
        2949, 2935}, {2949, 2911}, {2950, 2912}, {2950, 2442}, {2951, 2937}, {
        2951, 2913}, {2952, 2938}, {2952, 2914}, {2952, 2442}, {2953, 2939}, {
        2953, 2915}, {2953, 2442}, {2954, 2899}, {2954, 2916}, {2954, 2442}, {
        2955, 2921}, {2955, 2917}, {2956, 2922}, {2956, 2918}, {2957, 2923}, {
        2957, 2919}, {2958, 2920}, {2958, 2924}, {2959, 2902}, {2959, 2928}, {
        2959, 2941}, {2959, 2442}, {2960, 2923}, {2960, 2442}, {2961, 2945}, {
        2962, 2963}, {2962, 2925}, {2963, 2926}, {2963, 2442}, {2964, 2965}, {
        2964, 2927}, {2965, 2442}, {2965, 2928}, {2966, 2442}, {2966, 2930}, {
        2967, 2931}, {2967, 2949}, {2967, 2442}, {2968, 2442}, {2968, 2932}, {
        2969, 2933}, {2969, 2951}, {2969, 2442}, {2970, 3535, 3536, 3537, 
        3538, 3539, 3540, 3541, 3542, 3543, 3544, 3545, 3546, 3547, 3548, 
        3549, 3550, 2442}, {2970, 3551, 3552, 3553, 3554, 3555, 3556, 3557, 
        3558, 3559, 3560, 3561, 3562, 3563, 3564, 3565, 3566, 2442}, {2970, 
        2934}, {2971, 2953}, {2971, 2935}, {2972, 2936}, {2972, 2954}, {2972, 
        2442}, {2973, 2955}, {2973, 2937}, {2974, 2938}, {2974, 2956}, {2974, 
        2442}, {2975, 2957}, {2975, 2939}, {2976, 2899}, {2976, 2442}, {2976, 
        2924}, {2977, 2940}, {2977, 2921}, {2978, 2442}, {2979, 2442}, {2980, 
        2442}, {2980, 2947}, {2981, 2948}, {2981, 2970}, {2981, 2442}, {2982, 
        2442}, {2982, 2949}, {2983, 2950}, {2983, 2442}, {2984, 2951}, {2984, 
        2973}, {2985, 2952}, {2985, 2442}, {2986, 2975}, {2986, 2953}, {2987, 
        2954}, {2987, 2976}, {2988, 2977}, {2988, 2955}, {2989, 2941}, {2989, 
        2956}, {2990, 2960}, {2990, 2957}, {2990, 2442}, {2991, 2945}, {2991, 
        2958}, {2991, 2442}, {2992, 2963}, {2992, 2940}, {2992, 2442}, {2993, 
        2959}, {2993, 2994}, {2993, 2965}, {2994, 2442}, {2994, 2941}, {2994, 
        2965}, {2995, 2442}, {2995, 2943}, {2996, 2979}, {2996, 2961}, {2996, 
        2997}, {2996, 2442}, {2997, 2442}, {2998, 2962}, {2999, 2442}, {2999, 
        2964}, {3000, 2442}, {3001, 2984}, {3001, 2969}, {3002, 2442}, {3002, 
        2970}, {3002, 2985}, {3003, 2971}, {3003, 2986}, {3003, 2442}, {3004, 
        2972}, {3004, 2442}, {3004, 2987}, {3005, 2973}, {3005, 2988}, {3005, 
        2442}, {3006, 2974}, {3006, 2989}, {3006, 2442}, {3007, 2975}, {3007, 
        2990}, {3008, 2976}, {3008, 2442}, {3008, 2991}, {3009, 2977}, {3009, 
        2992}, {3010, 2960}, {3010, 2995}, {3010, 2442}, {3011, 2963}, {3012, 
        2442}, {3012, 2965}, {3013, 2978}, {3013, 2442}, {3014, 2979}, {3014, 
        2442}, {3015, 3003}, {3015, 2982}, {3016, 3004}, {3016, 2983}, {3016, 
        2442}, {3017, 2984}, {3017, 2442}, {3017, 3005}, {3018, 2985}, {3018, 
        3006}, {3018, 2442}, {3019, 2986}, {3019, 3007}, {3019, 2442}, {3020, 
        2987}, {3020, 3008}, {3020, 2442}, {3021, 2988}, {3021, 3009}, {3021, 
        2442}, {3022, 2989}, {3022, 2994}, {3023, 2990}, {3023, 2442}, {3023, 
        3010}, {3024, 2991}, {3024, 2997}, {3025, 2442}, {3025, 2992}, {3026, 
        2442}, {3026, 2993}, {3026, 3012}, {3027, 2995}, {3028, 2996}, {3028, 
        2442}, {3029, 2998}, {3029, 2442}, {3030, 2999}, {3030, 2442}, {3031, 
        3001}, {3031, 2442}, {3032, 2442}, {3032, 3002}, {3033, 2442}, {3033, 
        3003}, {3034, 2442}, {3034, 3004}, {3035, 3005}, {3035, 3021}, {3035, 
        2442}, {3036, 2442}, {3036, 3006}, {3037, 3007}, {3037, 2442}, {3037, 
        3023}, {3038, 2442}, {3038, 3008}, {3038, 3024}, {3039, 3009}, {3039, 
        2442}, {3040, 2442}, {3040, 2994}, {3041, 2442}, {3041, 3010}, {3042, 
        2997}, {3043, 3011}, {3044, 3012}, {3044, 2442}, {3045, 3013}, {3045, 
        2442}, {3046, 3014}, {3046, 2442}, {3047, 3015}, {3047, 2442}, {3048, 
        3034}, {3048, 2442}, {3048, 3016}, {3049, 3017}, {3049, 2442}, {3049, 
        3035}, {3050, 3018}, {3050, 3036}, {3050, 2442}, {3051, 3019}, {3052, 
        3020}, {3052, 2442}, {3052, 3038}, {3053, 3021}, {3053, 2442}, {3054, 
        3022}, {3054, 2442}, {3054, 3040}, {3055, 2442}, {3055, 3023}, {3055, 
        3041}, {3056, 3024}, {3057, 3025}, {3058, 3026}, {3058, 2442}, {3059, 
        3027}, {3059, 2442}, {3060, 3028}, {3060, 2442}, {3061, 3029}, {3061, 
        2442}, {3062, 3030}, {3062, 2442}, {3063, 2442}, {3063, 3032}, {3064, 
        3033}, {3064, 2442}, {3065, 2442}, {3065, 3034}, {3066, 3035}, {3066, 
        2442}, {3067, 2442}, {3067, 3036}, {3068, 3037}, {3068, 2442}, {3069, 
        3038}, {3070, 3039}, {3070, 2442}, {3071, 2442}, {3071, 3040}, {3072, 
        3041}, {3073, 3042}, {3073, 2442}, {3074, 3043}, {3074, 2442}, {3075, 
        3044}, {3075, 2442}, {3076, 3045}, {3076, 2442}, {3077, 3046}, {3077, 
        2442}, {3078, 2442}, {3079, 3048}, {3080, 3049}, {3080, 2442}, {3081, 
        3050}, {3081, 2442}, {3082, 3051}, {3083, 3052}, {3083, 3069}, {3084, 
        3053}, {3085, 3054}, {3085, 2442}, {3085, 3071}, {3086, 3055}, {3087, 
        3056}, {3087, 2442}, {3088, 3057}, {3089, 3058}, {3089, 2442}, {3090, 
        3059}, {3090, 2442}, {3091, 3060}, {3091, 2442}, {3092, 3061}, {3092, 
        2442}, {3093, 3062}, {3093, 2442}, {3094, 2442}, {3095, 3064}, {3096, 
        2442}, {3096, 3065}, {3097, 3066}, {3097, 2442}, {3098, 3067}, {3098, 
        2442}, {3099, 3068}, {3100, 3069}, {3101, 3070}, {3102, 3071}, {3103, 
        3072}, {3103, 2442}, {3104, 3073}, {3104, 2442}, {3105, 3074}, {3106, 
        3075}, {3106, 2442}, {3107, 3076}, {3107, 2442}, {3108, 3077}, {3108, 
        2442}, {3109, 2442}, {3110, 3080}, {3111, 3081}, {3111, 2442}, {3112, 
        3082}, {3113, 3083}, {3114, 3084}, {3115, 3085}, {3116, 3086}, {3117, 
        3087}, {3117, 2442}, {3118, 3088}, {3118, 2442}, {3119, 3089}, {3119, 
        2442}, {3120, 3090}, {3120, 2442}, {3121, 3091}, {3121, 2442}, {3122, 
        3092}, {3122, 2442}, {3123, 3093}, {3123, 2442}, {3124, 3094}, {3124, 
        2442}, {3125, 3095}, {3126, 3096}, {3127, 3097}, {3128, 3098}, {3128, 
        2442}, {3129, 3099}, {3130, 2442}, {3131, 3101}, {3132, 3102}, {3132, 
        2442}, {3133, 3103}, {3133, 2442}, {3134, 3104}, {3134, 2442}, {3135, 
        3105}, {3136, 3106}, {3136, 2442}, {3137, 3107}, {3137, 2442}, {3138, 
        2442}, {3139, 3109}, {3140, 3113}, {3141, 3114}, {3142, 3115}, {3143, 
        3116}, {3144, 3117}, {3144, 2442}, {3145, 3118}, {3145, 2442}, {3146, 
        3119}, {3146, 2442}, {3147, 3120}, {3147, 2442}, {3148, 3121}, {3148, 
        2442}, {3149, 3122}, {3149, 2442}, {3150, 3123}, {3150, 2442}, {3151, 
        3124}, {3151, 2442}, {3152, 3131}, {3153, 3132}, {3153, 2442}, {3154, 
        3133}, {3154, 2442}, {3155, 3134}, {3155, 2442}, {3156, 3135}, {3157, 
        3136}, {3157, 2442}, {3158, 2442}, {3159, 3138}, {3160, 3139}, {3161, 
        3147}, {3161, 2442}, {3162, 3148}, {3162, 2442}, {3163, 3149}, {3163, 
        2442}, {3164, 3150}, {3164, 2442}, {3165, 3151}, {3165, 2442}, {3166, 
        3156}, {3167, 3157}, {3167, 2442}, {3168, 3158}, {3169, 3159}, {3170, 
        3160}, {3171, 3161}, {3171, 2442}, {3172, 3162}, {3172, 2442}, {3173, 
        3163}, {3173, 2442}, {3174, 3164}, {3174, 2442}, {3175, 3165}, {3175, 
        2442}, {3176, 3166}, {3177, 3167}, {3177, 2442}, {3178, 3168}, {3178, 
        2442}, {3179, 3169}, {3180, 3171}, {3180, 2442}, {3181, 3172}, {3181, 
        2442}, {3182, 3176}}]}, 
     {RGBColor[0, 0, 0.7], 
      TagBox[
       TooltipBox[PointBox[1],
        "\"Neutron\""],
       Annotation[#, "Neutron", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2],
        "\"Hydrogen1\""],
       Annotation[#, "Hydrogen1", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3],
        "\"Hydrogen3\""],
       Annotation[#, "Hydrogen3", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[4],
        "\"Helium3\""],
       Annotation[#, "Helium3", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[5],
        "\"Hydrogen4\""],
       Annotation[#, "Hydrogen4", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[6],
        "\"Hydrogen5\""],
       Annotation[#, "Hydrogen5", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[7],
        "\"Hydrogen6\""],
       Annotation[#, "Hydrogen6", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[8],
        "\"Hydrogen7\""],
       Annotation[#, "Hydrogen7", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[9],
        "\"Helium5\""],
       Annotation[#, "Helium5", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[10],
        "\"Helium4\""],
       Annotation[#, "Helium4", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[11],
        "\"Helium6\""],
       Annotation[#, "Helium6", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[12],
        "\"Lithium6\""],
       Annotation[#, "Lithium6", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[13],
        "\"Helium7\""],
       Annotation[#, "Helium7", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[14],
        "\"Helium8\""],
       Annotation[#, "Helium8", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[15],
        "\"Lithium8\""],
       Annotation[#, "Lithium8", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[16],
        "\"Lithium7\""],
       Annotation[#, "Lithium7", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[17],
        "\"Helium9\""],
       Annotation[#, "Helium9", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[18],
        "\"Helium10\""],
       Annotation[#, "Helium10", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[19],
        "\"Lithium3\""],
       Annotation[#, "Lithium3", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[20],
        "\"Helium2\""],
       Annotation[#, "Helium2", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[21],
        "\"Lithium4\""],
       Annotation[#, "Lithium4", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[22],
        "\"Lithium5\""],
       Annotation[#, "Lithium5", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[23],
        "\"Lithium9\""],
       Annotation[#, "Lithium9", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[24],
        "\"Beryllium8\""],
       Annotation[#, "Beryllium8", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[25],
        "\"Beryllium9\""],
       Annotation[#, "Beryllium9", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[26],
        "\"Lithium10\""],
       Annotation[#, "Lithium10", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[27],
        "\"Lithium11\""],
       Annotation[#, "Lithium11", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[28],
        "\"Beryllium10\""],
       Annotation[#, "Beryllium10", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[29],
        "\"Lithium12\""],
       Annotation[#, "Lithium12", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[30],
        "\"Beryllium5\""],
       Annotation[#, "Beryllium5", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[31],
        "\"Beryllium6\""],
       Annotation[#, "Beryllium6", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[32],
        "\"Beryllium7\""],
       Annotation[#, "Beryllium7", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[33],
        "\"Boron10\""],
       Annotation[#, "Boron10", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[34],
        "\"Beryllium11\""],
       Annotation[#, "Beryllium11", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[35],
        "\"Boron11\""],
       Annotation[#, "Boron11", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[36],
        "\"Beryllium12\""],
       Annotation[#, "Beryllium12", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[37],
        "\"Boron12\""],
       Annotation[#, "Boron12", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[38],
        "\"Beryllium13\""],
       Annotation[#, "Beryllium13", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[39],
        "\"Beryllium14\""],
       Annotation[#, "Beryllium14", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[40],
        "\"Boron13\""],
       Annotation[#, "Boron13", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[41],
        "\"Beryllium15\""],
       Annotation[#, "Beryllium15", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[42],
        "\"Beryllium16\""],
       Annotation[#, "Beryllium16", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[43],
        "\"Boron6\""],
       Annotation[#, "Boron6", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[44],
        "\"Boron7\""],
       Annotation[#, "Boron7", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[45],
        "\"Boron8\""],
       Annotation[#, "Boron8", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[46],
        "\"Boron9\""],
       Annotation[#, "Boron9", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[47],
        "\"Carbon12\""],
       Annotation[#, "Carbon12", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[48],
        "\"Carbon13\""],
       Annotation[#, "Carbon13", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[49],
        "\"Boron14\""],
       Annotation[#, "Boron14", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[50],
        "\"Carbon14\""],
       Annotation[#, "Carbon14", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[51],
        "\"Boron15\""],
       Annotation[#, "Boron15", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[52],
        "\"Boron16\""],
       Annotation[#, "Boron16", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[53],
        "\"Boron17\""],
       Annotation[#, "Boron17", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[54],
        "\"Carbon16\""],
       Annotation[#, "Carbon16", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[55],
        "\"Carbon15\""],
       Annotation[#, "Carbon15", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[56],
        "\"Boron18\""],
       Annotation[#, "Boron18", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[57],
        "\"Boron19\""],
       Annotation[#, "Boron19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[58],
        "\"Carbon18\""],
       Annotation[#, "Carbon18", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[59],
        "\"Carbon17\""],
       Annotation[#, "Carbon17", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[60],
        "\"Carbon8\""],
       Annotation[#, "Carbon8", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[61],
        "\"Carbon9\""],
       Annotation[#, "Carbon9", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[62],
        "\"Carbon10\""],
       Annotation[#, "Carbon10", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[63],
        "\"Carbon11\""],
       Annotation[#, "Carbon11", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[64],
        "\"Nitrogen14\""],
       Annotation[#, "Nitrogen14", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[65],
        "\"Nitrogen15\""],
       Annotation[#, "Nitrogen15", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[66],
        "\"Nitrogen17\""],
       Annotation[#, "Nitrogen17", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[67],
        "\"Nitrogen16\""],
       Annotation[#, "Nitrogen16", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[68],
        "\"Nitrogen18\""],
       Annotation[#, "Nitrogen18", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[69],
        "\"Carbon19\""],
       Annotation[#, "Carbon19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[70],
        "\"Nitrogen19\""],
       Annotation[#, "Nitrogen19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[71],
        "\"Carbon20\""],
       Annotation[#, "Carbon20", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[72],
        "\"Nitrogen20\""],
       Annotation[#, "Nitrogen20", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[73],
        "\"Carbon21\""],
       Annotation[#, "Carbon21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[74],
        "\"Carbon22\""],
       Annotation[#, "Carbon22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[75],
        "\"Nitrogen21\""],
       Annotation[#, "Nitrogen21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[76],
        "\"Nitrogen10\""],
       Annotation[#, "Nitrogen10", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[77],
        "\"Nitrogen11\""],
       Annotation[#, "Nitrogen11", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[78],
        "\"Nitrogen12\""],
       Annotation[#, "Nitrogen12", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[79],
        "\"Nitrogen13\""],
       Annotation[#, "Nitrogen13", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[80],
        "\"Oxygen16\""],
       Annotation[#, "Oxygen16", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[81],
        "\"Oxygen18\""],
       Annotation[#, "Oxygen18", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[82],
        "\"Oxygen17\""],
       Annotation[#, "Oxygen17", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[83],
        "\"Oxygen19\""],
       Annotation[#, "Oxygen19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[84],
        "\"Oxygen20\""],
       Annotation[#, "Oxygen20", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[85],
        "\"Nitrogen22\""],
       Annotation[#, "Nitrogen22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[86],
        "\"Oxygen22\""],
       Annotation[#, "Oxygen22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[87],
        "\"Oxygen21\""],
       Annotation[#, "Oxygen21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[88],
        "\"Nitrogen23\""],
       Annotation[#, "Nitrogen23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[89],
        "\"Nitrogen24\""],
       Annotation[#, "Nitrogen24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[90],
        "\"Nitrogen25\""],
       Annotation[#, "Nitrogen25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[91],
        "\"Oxygen25\""],
       Annotation[#, "Oxygen25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[92],
        "\"Oxygen12\""],
       Annotation[#, "Oxygen12", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[93],
        "\"Oxygen13\""],
       Annotation[#, "Oxygen13", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[94],
        "\"Oxygen14\""],
       Annotation[#, "Oxygen14", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[95],
        "\"Oxygen15\""],
       Annotation[#, "Oxygen15", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[96],
        "\"Fluorine19\""],
       Annotation[#, "Fluorine19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[97],
        "\"Fluorine20\""],
       Annotation[#, "Fluorine20", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[98],
        "\"Fluorine21\""],
       Annotation[#, "Fluorine21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[99],
        "\"Fluorine22\""],
       Annotation[#, "Fluorine22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[100],
        "\"Oxygen23\""],
       Annotation[#, "Oxygen23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[101],
        "\"Fluorine23\""],
       Annotation[#, "Fluorine23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[102],
        "\"Oxygen24\""],
       Annotation[#, "Oxygen24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[103],
        "\"Fluorine24\""],
       Annotation[#, "Fluorine24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[104],
        "\"Oxygen26\""],
       Annotation[#, "Oxygen26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[105],
        "\"Fluorine26\""],
       Annotation[#, "Fluorine26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[106],
        "\"Oxygen27\""],
       Annotation[#, "Oxygen27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[107],
        "\"Oxygen28\""],
       Annotation[#, "Oxygen28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[108],
        "\"Fluorine28\""],
       Annotation[#, "Fluorine28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[109],
        "\"Fluorine14\""],
       Annotation[#, "Fluorine14", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[110],
        "\"Fluorine15\""],
       Annotation[#, "Fluorine15", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[111],
        "\"Fluorine16\""],
       Annotation[#, "Fluorine16", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[112],
        "\"Fluorine17\""],
       Annotation[#, "Fluorine17", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[113],
        "\"Fluorine18\""],
       Annotation[#, "Fluorine18", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[114],
        "\"Neon20\""],
       Annotation[#, "Neon20", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[115],
        "\"Neon21\""],
       Annotation[#, "Neon21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[116],
        "\"Neon22\""],
       Annotation[#, "Neon22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[117],
        "\"Neon23\""],
       Annotation[#, "Neon23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[118],
        "\"Neon24\""],
       Annotation[#, "Neon24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[119],
        "\"Fluorine25\""],
       Annotation[#, "Fluorine25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[120],
        "\"Neon25\""],
       Annotation[#, "Neon25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[121],
        "\"Neon26\""],
       Annotation[#, "Neon26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[122],
        "\"Fluorine27\""],
       Annotation[#, "Fluorine27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[123],
        "\"Neon27\""],
       Annotation[#, "Neon27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[124],
        "\"Fluorine29\""],
       Annotation[#, "Fluorine29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[125],
        "\"Neon28\""],
       Annotation[#, "Neon28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[126],
        "\"Neon29\""],
       Annotation[#, "Neon29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[127],
        "\"Fluorine30\""],
       Annotation[#, "Fluorine30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[128],
        "\"Fluorine31\""],
       Annotation[#, "Fluorine31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[129],
        "\"Neon31\""],
       Annotation[#, "Neon31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[130],
        "\"Neon30\""],
       Annotation[#, "Neon30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[131],
        "\"Neon16\""],
       Annotation[#, "Neon16", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[132],
        "\"Neon17\""],
       Annotation[#, "Neon17", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[133],
        "\"Neon18\""],
       Annotation[#, "Neon18", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[134],
        "\"Neon19\""],
       Annotation[#, "Neon19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[135],
        "\"Sodium23\""],
       Annotation[#, "Sodium23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[136],
        "\"Sodium24\""],
       Annotation[#, "Sodium24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[137],
        "\"Sodium25\""],
       Annotation[#, "Sodium25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[138],
        "\"Sodium26\""],
       Annotation[#, "Sodium26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[139],
        "\"Sodium27\""],
       Annotation[#, "Sodium27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[140],
        "\"Sodium28\""],
       Annotation[#, "Sodium28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[141],
        "\"Sodium29\""],
       Annotation[#, "Sodium29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[142],
        "\"Sodium30\""],
       Annotation[#, "Sodium30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[143],
        "\"Sodium31\""],
       Annotation[#, "Sodium31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[144],
        "\"Neon32\""],
       Annotation[#, "Neon32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[145],
        "\"Sodium32\""],
       Annotation[#, "Sodium32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[146],
        "\"Neon33\""],
       Annotation[#, "Neon33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[147],
        "\"Neon34\""],
       Annotation[#, "Neon34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[148],
        "\"Sodium34\""],
       Annotation[#, "Sodium34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[149],
        "\"Sodium33\""],
       Annotation[#, "Sodium33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[150],
        "\"Sodium18\""],
       Annotation[#, "Sodium18", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[151],
        "\"Sodium19\""],
       Annotation[#, "Sodium19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[152],
        "\"Sodium20\""],
       Annotation[#, "Sodium20", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[153],
        "\"Sodium21\""],
       Annotation[#, "Sodium21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[154],
        "\"Sodium22\""],
       Annotation[#, "Sodium22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[155],
        "\"Magnesium24\""],
       Annotation[#, "Magnesium24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[156],
        "\"Magnesium25\""],
       Annotation[#, "Magnesium25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[157],
        "\"Magnesium26\""],
       Annotation[#, "Magnesium26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[158],
        "\"Magnesium27\""],
       Annotation[#, "Magnesium27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[159],
        "\"Magnesium28\""],
       Annotation[#, "Magnesium28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[160],
        "\"Magnesium29\""],
       Annotation[#, "Magnesium29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[161],
        "\"Magnesium30\""],
       Annotation[#, "Magnesium30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[162],
        "\"Magnesium31\""],
       Annotation[#, "Magnesium31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[163],
        "\"Magnesium32\""],
       Annotation[#, "Magnesium32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[164],
        "\"Magnesium33\""],
       Annotation[#, "Magnesium33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[165],
        "\"Magnesium34\""],
       Annotation[#, "Magnesium34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[166],
        "\"Sodium35\""],
       Annotation[#, "Sodium35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[167],
        "\"Magnesium35\""],
       Annotation[#, "Magnesium35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[168],
        "\"Sodium36\""],
       Annotation[#, "Sodium36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[169],
        "\"Sodium37\""],
       Annotation[#, "Sodium37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[170],
        "\"Magnesium37\""],
       Annotation[#, "Magnesium37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[171],
        "\"Magnesium36\""],
       Annotation[#, "Magnesium36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[172],
        "\"Magnesium19\""],
       Annotation[#, "Magnesium19", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[173],
        "\"Magnesium20\""],
       Annotation[#, "Magnesium20", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[174],
        "\"Magnesium21\""],
       Annotation[#, "Magnesium21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[175],
        "\"Magnesium22\""],
       Annotation[#, "Magnesium22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[176],
        "\"Magnesium23\""],
       Annotation[#, "Magnesium23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[177],
        "\"Aluminum27\""],
       Annotation[#, "Aluminum27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[178],
        "\"Aluminum28\""],
       Annotation[#, "Aluminum28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[179],
        "\"Aluminum29\""],
       Annotation[#, "Aluminum29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[180],
        "\"Aluminum30\""],
       Annotation[#, "Aluminum30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[181],
        "\"Aluminum31\""],
       Annotation[#, "Aluminum31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[182],
        "\"Aluminum32\""],
       Annotation[#, "Aluminum32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[183],
        "\"Aluminum33\""],
       Annotation[#, "Aluminum33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[184],
        "\"Aluminum34\""],
       Annotation[#, "Aluminum34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[185],
        "\"Aluminum35\""],
       Annotation[#, "Aluminum35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[186],
        "\"Aluminum36\""],
       Annotation[#, "Aluminum36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[187],
        "\"Aluminum37\""],
       Annotation[#, "Aluminum37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[188],
        "\"Magnesium38\""],
       Annotation[#, "Magnesium38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[189],
        "\"Aluminum38\""],
       Annotation[#, "Aluminum38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[190],
        "\"Magnesium39\""],
       Annotation[#, "Magnesium39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[191],
        "\"Magnesium40\""],
       Annotation[#, "Magnesium40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[192],
        "\"Aluminum40\""],
       Annotation[#, "Aluminum40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[193],
        "\"Aluminum39\""],
       Annotation[#, "Aluminum39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[194],
        "\"Aluminum21\""],
       Annotation[#, "Aluminum21", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[195],
        "\"Aluminum22\""],
       Annotation[#, "Aluminum22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[196],
        "\"Aluminum23\""],
       Annotation[#, "Aluminum23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[197],
        "\"Aluminum24\""],
       Annotation[#, "Aluminum24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[198],
        "\"Aluminum25\""],
       Annotation[#, "Aluminum25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[199],
        "\"Aluminum26\""],
       Annotation[#, "Aluminum26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[200],
        "\"Silicon28\""],
       Annotation[#, "Silicon28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[201],
        "\"Silicon29\""],
       Annotation[#, "Silicon29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[202],
        "\"Silicon30\""],
       Annotation[#, "Silicon30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[203],
        "\"Silicon31\""],
       Annotation[#, "Silicon31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[204],
        "\"Silicon32\""],
       Annotation[#, "Silicon32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[205],
        "\"Silicon33\""],
       Annotation[#, "Silicon33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[206],
        "\"Silicon34\""],
       Annotation[#, "Silicon34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[207],
        "\"Silicon35\""],
       Annotation[#, "Silicon35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[208],
        "\"Silicon36\""],
       Annotation[#, "Silicon36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[209],
        "\"Silicon37\""],
       Annotation[#, "Silicon37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[210],
        "\"Silicon38\""],
       Annotation[#, "Silicon38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[211],
        "\"Silicon39\""],
       Annotation[#, "Silicon39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[212],
        "\"Silicon40\""],
       Annotation[#, "Silicon40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[213],
        "\"Aluminum41\""],
       Annotation[#, "Aluminum41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[214],
        "\"Silicon41\""],
       Annotation[#, "Silicon41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[215],
        "\"Aluminum42\""],
       Annotation[#, "Aluminum42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[216],
        "\"Silicon42\""],
       Annotation[#, "Silicon42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[217],
        "\"Silicon22\""],
       Annotation[#, "Silicon22", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[218],
        "\"Silicon23\""],
       Annotation[#, "Silicon23", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[219],
        "\"Silicon24\""],
       Annotation[#, "Silicon24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[220],
        "\"Silicon25\""],
       Annotation[#, "Silicon25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[221],
        "\"Silicon26\""],
       Annotation[#, "Silicon26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[222],
        "\"Silicon27\""],
       Annotation[#, "Silicon27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[223],
        "\"Phosphorus31\""],
       Annotation[#, "Phosphorus31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[224],
        "\"Phosphorus32\""],
       Annotation[#, "Phosphorus32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[225],
        "\"Phosphorus33\""],
       Annotation[#, "Phosphorus33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[226],
        "\"Phosphorus34\""],
       Annotation[#, "Phosphorus34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[227],
        "\"Phosphorus35\""],
       Annotation[#, "Phosphorus35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[228],
        "\"Phosphorus36\""],
       Annotation[#, "Phosphorus36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[229],
        "\"Phosphorus37\""],
       Annotation[#, "Phosphorus37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[230],
        "\"Phosphorus38\""],
       Annotation[#, "Phosphorus38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[231],
        "\"Phosphorus39\""],
       Annotation[#, "Phosphorus39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[232],
        "\"Phosphorus40\""],
       Annotation[#, "Phosphorus40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[233],
        "\"Phosphorus41\""],
       Annotation[#, "Phosphorus41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[234],
        "\"Phosphorus42\""],
       Annotation[#, "Phosphorus42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[235],
        "\"Silicon43\""],
       Annotation[#, "Silicon43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[236],
        "\"Phosphorus43\""],
       Annotation[#, "Phosphorus43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[237],
        "\"Silicon44\""],
       Annotation[#, "Silicon44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[238],
        "\"Phosphorus44\""],
       Annotation[#, "Phosphorus44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[239],
        "\"Phosphorus24\""],
       Annotation[#, "Phosphorus24", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[240],
        "\"Phosphorus25\""],
       Annotation[#, "Phosphorus25", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[241],
        "\"Phosphorus26\""],
       Annotation[#, "Phosphorus26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[242],
        "\"Phosphorus27\""],
       Annotation[#, "Phosphorus27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[243],
        "\"Phosphorus28\""],
       Annotation[#, "Phosphorus28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[244],
        "\"Phosphorus29\""],
       Annotation[#, "Phosphorus29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[245],
        "\"Phosphorus30\""],
       Annotation[#, "Phosphorus30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[246],
        "\"Sulfur32\""],
       Annotation[#, "Sulfur32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[247],
        "\"Sulfur33\""],
       Annotation[#, "Sulfur33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[248],
        "\"Sulfur34\""],
       Annotation[#, "Sulfur34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[249],
        "\"Sulfur35\""],
       Annotation[#, "Sulfur35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[250],
        "\"Sulfur36\""],
       Annotation[#, "Sulfur36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[251],
        "\"Sulfur37\""],
       Annotation[#, "Sulfur37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[252],
        "\"Sulfur38\""],
       Annotation[#, "Sulfur38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[253],
        "\"Sulfur39\""],
       Annotation[#, "Sulfur39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[254],
        "\"Sulfur40\""],
       Annotation[#, "Sulfur40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[255],
        "\"Sulfur41\""],
       Annotation[#, "Sulfur41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[256],
        "\"Sulfur42\""],
       Annotation[#, "Sulfur42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[257],
        "\"Sulfur44\""],
       Annotation[#, "Sulfur44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[258],
        "\"Phosphorus45\""],
       Annotation[#, "Phosphorus45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[259],
        "\"Sulfur45\""],
       Annotation[#, "Sulfur45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[260],
        "\"Phosphorus46\""],
       Annotation[#, "Phosphorus46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[261],
        "\"Sulfur46\""],
       Annotation[#, "Sulfur46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[262],
        "\"Sulfur26\""],
       Annotation[#, "Sulfur26", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[263],
        "\"Sulfur27\""],
       Annotation[#, "Sulfur27", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[264],
        "\"Sulfur28\""],
       Annotation[#, "Sulfur28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[265],
        "\"Sulfur29\""],
       Annotation[#, "Sulfur29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[266],
        "\"Sulfur30\""],
       Annotation[#, "Sulfur30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[267],
        "\"Sulfur31\""],
       Annotation[#, "Sulfur31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[268],
        "\"Chlorine35\""],
       Annotation[#, "Chlorine35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[269],
        "\"Chlorine37\""],
       Annotation[#, "Chlorine37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[270],
        "\"Chlorine38\""],
       Annotation[#, "Chlorine38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[271],
        "\"Chlorine39\""],
       Annotation[#, "Chlorine39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[272],
        "\"Chlorine40\""],
       Annotation[#, "Chlorine40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[273],
        "\"Chlorine41\""],
       Annotation[#, "Chlorine41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[274],
        "\"Chlorine42\""],
       Annotation[#, "Chlorine42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[275],
        "\"Sulfur43\""],
       Annotation[#, "Sulfur43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[276],
        "\"Chlorine43\""],
       Annotation[#, "Chlorine43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[277],
        "\"Chlorine44\""],
       Annotation[#, "Chlorine44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[278],
        "\"Chlorine45\""],
       Annotation[#, "Chlorine45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[279],
        "\"Chlorine46\""],
       Annotation[#, "Chlorine46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[280],
        "\"Sulfur47\""],
       Annotation[#, "Sulfur47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[281],
        "\"Chlorine47\""],
       Annotation[#, "Chlorine47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[282],
        "\"Sulfur48\""],
       Annotation[#, "Sulfur48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[283],
        "\"Chlorine48\""],
       Annotation[#, "Chlorine48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[284],
        "\"Sulfur49\""],
       Annotation[#, "Sulfur49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[285],
        "\"Chlorine28\""],
       Annotation[#, "Chlorine28", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[286],
        "\"Chlorine29\""],
       Annotation[#, "Chlorine29", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[287],
        "\"Chlorine30\""],
       Annotation[#, "Chlorine30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[288],
        "\"Chlorine31\""],
       Annotation[#, "Chlorine31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[289],
        "\"Chlorine32\""],
       Annotation[#, "Chlorine32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[290],
        "\"Chlorine33\""],
       Annotation[#, "Chlorine33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[291],
        "\"Chlorine34\""],
       Annotation[#, "Chlorine34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[292],
        "\"Chlorine36\""],
       Annotation[#, "Chlorine36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[293],
        "\"Argon36\""],
       Annotation[#, "Argon36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[294],
        "\"Argon38\""],
       Annotation[#, "Argon38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[295],
        "\"Argon39\""],
       Annotation[#, "Argon39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[296],
        "\"Argon40\""],
       Annotation[#, "Argon40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[297],
        "\"Argon41\""],
       Annotation[#, "Argon41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[298],
        "\"Argon42\""],
       Annotation[#, "Argon42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[299],
        "\"Argon43\""],
       Annotation[#, "Argon43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[300],
        "\"Argon44\""],
       Annotation[#, "Argon44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[301],
        "\"Argon45\""],
       Annotation[#, "Argon45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[302],
        "\"Argon46\""],
       Annotation[#, "Argon46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[303],
        "\"Argon47\""],
       Annotation[#, "Argon47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[304],
        "\"Argon48\""],
       Annotation[#, "Argon48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[305],
        "\"Chlorine49\""],
       Annotation[#, "Chlorine49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[306],
        "\"Argon49\""],
       Annotation[#, "Argon49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[307],
        "\"Chlorine50\""],
       Annotation[#, "Chlorine50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[308],
        "\"Argon50\""],
       Annotation[#, "Argon50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[309],
        "\"Chlorine51\""],
       Annotation[#, "Chlorine51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[310],
        "\"Argon51\""],
       Annotation[#, "Argon51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[311],
        "\"Argon30\""],
       Annotation[#, "Argon30", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[312],
        "\"Argon31\""],
       Annotation[#, "Argon31", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[313],
        "\"Argon32\""],
       Annotation[#, "Argon32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[314],
        "\"Argon33\""],
       Annotation[#, "Argon33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[315],
        "\"Argon34\""],
       Annotation[#, "Argon34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[316],
        "\"Argon35\""],
       Annotation[#, "Argon35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[317],
        "\"Argon37\""],
       Annotation[#, "Argon37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[318],
        "\"Potassium39\""],
       Annotation[#, "Potassium39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[319],
        "\"Potassium41\""],
       Annotation[#, "Potassium41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[320],
        "\"Potassium42\""],
       Annotation[#, "Potassium42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[321],
        "\"Potassium43\""],
       Annotation[#, "Potassium43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[322],
        "\"Potassium44\""],
       Annotation[#, "Potassium44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[323],
        "\"Potassium45\""],
       Annotation[#, "Potassium45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[324],
        "\"Potassium46\""],
       Annotation[#, "Potassium46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[325],
        "\"Potassium47\""],
       Annotation[#, "Potassium47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[326],
        "\"Potassium48\""],
       Annotation[#, "Potassium48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[327],
        "\"Potassium49\""],
       Annotation[#, "Potassium49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[328],
        "\"Potassium50\""],
       Annotation[#, "Potassium50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[329],
        "\"Potassium51\""],
       Annotation[#, "Potassium51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[330],
        "\"Argon52\""],
       Annotation[#, "Argon52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[331],
        "\"Potassium52\""],
       Annotation[#, "Potassium52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[332],
        "\"Argon53\""],
       Annotation[#, "Argon53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[333],
        "\"Potassium53\""],
       Annotation[#, "Potassium53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[334],
        "\"Potassium32\""],
       Annotation[#, "Potassium32", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[335],
        "\"Potassium33\""],
       Annotation[#, "Potassium33", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[336],
        "\"Potassium34\""],
       Annotation[#, "Potassium34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[337],
        "\"Potassium35\""],
       Annotation[#, "Potassium35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[338],
        "\"Potassium36\""],
       Annotation[#, "Potassium36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[339],
        "\"Potassium37\""],
       Annotation[#, "Potassium37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[340],
        "\"Potassium38\""],
       Annotation[#, "Potassium38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[341],
        "\"Potassium40\""],
       Annotation[#, "Potassium40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[342],
        "\"Calcium40\""],
       Annotation[#, "Calcium40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[343],
        "\"Calcium42\""],
       Annotation[#, "Calcium42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[344],
        "\"Calcium43\""],
       Annotation[#, "Calcium43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[345],
        "\"Calcium44\""],
       Annotation[#, "Calcium44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[346],
        "\"Calcium45\""],
       Annotation[#, "Calcium45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[347],
        "\"Calcium46\""],
       Annotation[#, "Calcium46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[348],
        "\"Calcium47\""],
       Annotation[#, "Calcium47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[349],
        "\"Calcium48\""],
       Annotation[#, "Calcium48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[350],
        "\"Calcium50\""],
       Annotation[#, "Calcium50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[351],
        "\"Calcium49\""],
       Annotation[#, "Calcium49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[352],
        "\"Calcium51\""],
       Annotation[#, "Calcium51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[353],
        "\"Calcium52\""],
       Annotation[#, "Calcium52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[354],
        "\"Potassium54\""],
       Annotation[#, "Potassium54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[355],
        "\"Calcium54\""],
       Annotation[#, "Calcium54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[356],
        "\"Calcium53\""],
       Annotation[#, "Calcium53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[357],
        "\"Potassium55\""],
       Annotation[#, "Potassium55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[358],
        "\"Calcium55\""],
       Annotation[#, "Calcium55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[359],
        "\"Calcium34\""],
       Annotation[#, "Calcium34", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[360],
        "\"Calcium35\""],
       Annotation[#, "Calcium35", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[361],
        "\"Calcium36\""],
       Annotation[#, "Calcium36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[362],
        "\"Calcium37\""],
       Annotation[#, "Calcium37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[363],
        "\"Calcium38\""],
       Annotation[#, "Calcium38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[364],
        "\"Calcium39\""],
       Annotation[#, "Calcium39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[365],
        "\"Calcium41\""],
       Annotation[#, "Calcium41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[366],
        "\"Scandium45\""],
       Annotation[#, "Scandium45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[367],
        "\"Titanium46\""],
       Annotation[#, "Titanium46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[368],
        "\"Scandium47\""],
       Annotation[#, "Scandium47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[369],
        "\"Titanium48\""],
       Annotation[#, "Titanium48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[370],
        "\"Scandium48\""],
       Annotation[#, "Scandium48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[371],
        "\"Scandium49\""],
       Annotation[#, "Scandium49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[372],
        "\"Scandium50\""],
       Annotation[#, "Scandium50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[373],
        "\"Scandium51\""],
       Annotation[#, "Scandium51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[374],
        "\"Scandium52\""],
       Annotation[#, "Scandium52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[375],
        "\"Scandium53\""],
       Annotation[#, "Scandium53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[376],
        "\"Scandium54\""],
       Annotation[#, "Scandium54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[377],
        "\"Scandium55\""],
       Annotation[#, "Scandium55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[378],
        "\"Calcium56\""],
       Annotation[#, "Calcium56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[379],
        "\"Scandium56\""],
       Annotation[#, "Scandium56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[380],
        "\"Calcium57\""],
       Annotation[#, "Calcium57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[381],
        "\"Scandium57\""],
       Annotation[#, "Scandium57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[382],
        "\"Scandium36\""],
       Annotation[#, "Scandium36", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[383],
        "\"Scandium37\""],
       Annotation[#, "Scandium37", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[384],
        "\"Scandium38\""],
       Annotation[#, "Scandium38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[385],
        "\"Scandium39\""],
       Annotation[#, "Scandium39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[386],
        "\"Scandium40\""],
       Annotation[#, "Scandium40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[387],
        "\"Scandium41\""],
       Annotation[#, "Scandium41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[388],
        "\"Scandium42\""],
       Annotation[#, "Scandium42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[389],
        "\"Scandium43\""],
       Annotation[#, "Scandium43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[390],
        "\"Scandium44\""],
       Annotation[#, "Scandium44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[391],
        "\"Scandium46\""],
       Annotation[#, "Scandium46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[392],
        "\"Titanium47\""],
       Annotation[#, "Titanium47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[393],
        "\"Titanium49\""],
       Annotation[#, "Titanium49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[394],
        "\"Titanium50\""],
       Annotation[#, "Titanium50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[395],
        "\"Titanium51\""],
       Annotation[#, "Titanium51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[396],
        "\"Titanium52\""],
       Annotation[#, "Titanium52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[397],
        "\"Titanium53\""],
       Annotation[#, "Titanium53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[398],
        "\"Titanium54\""],
       Annotation[#, "Titanium54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[399],
        "\"Titanium55\""],
       Annotation[#, "Titanium55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[400],
        "\"Titanium56\""],
       Annotation[#, "Titanium56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[401],
        "\"Titanium57\""],
       Annotation[#, "Titanium57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[402],
        "\"Scandium58\""],
       Annotation[#, "Scandium58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[403],
        "\"Titanium58\""],
       Annotation[#, "Titanium58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[404],
        "\"Scandium59\""],
       Annotation[#, "Scandium59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[405],
        "\"Titanium59\""],
       Annotation[#, "Titanium59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[406],
        "\"Scandium60\""],
       Annotation[#, "Scandium60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[407],
        "\"Titanium60\""],
       Annotation[#, "Titanium60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[408],
        "\"Titanium38\""],
       Annotation[#, "Titanium38", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[409],
        "\"Titanium39\""],
       Annotation[#, "Titanium39", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[410],
        "\"Titanium40\""],
       Annotation[#, "Titanium40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[411],
        "\"Titanium41\""],
       Annotation[#, "Titanium41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[412],
        "\"Titanium42\""],
       Annotation[#, "Titanium42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[413],
        "\"Titanium43\""],
       Annotation[#, "Titanium43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[414],
        "\"Titanium44\""],
       Annotation[#, "Titanium44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[415],
        "\"Titanium45\""],
       Annotation[#, "Titanium45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[416],
        "\"Vanadium51\""],
       Annotation[#, "Vanadium51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[417],
        "\"Vanadium52\""],
       Annotation[#, "Vanadium52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[418],
        "\"Vanadium53\""],
       Annotation[#, "Vanadium53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[419],
        "\"Vanadium54\""],
       Annotation[#, "Vanadium54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[420],
        "\"Vanadium55\""],
       Annotation[#, "Vanadium55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[421],
        "\"Vanadium56\""],
       Annotation[#, "Vanadium56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[422],
        "\"Vanadium57\""],
       Annotation[#, "Vanadium57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[423],
        "\"Vanadium58\""],
       Annotation[#, "Vanadium58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[424],
        "\"Vanadium59\""],
       Annotation[#, "Vanadium59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[425],
        "\"Vanadium60\""],
       Annotation[#, "Vanadium60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[426],
        "\"Titanium61\""],
       Annotation[#, "Titanium61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[427],
        "\"Vanadium61\""],
       Annotation[#, "Vanadium61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[428],
        "\"Titanium62\""],
       Annotation[#, "Titanium62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[429],
        "\"Vanadium62\""],
       Annotation[#, "Vanadium62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[430],
        "\"Titanium63\""],
       Annotation[#, "Titanium63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[431],
        "\"Vanadium63\""],
       Annotation[#, "Vanadium63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[432],
        "\"Vanadium40\""],
       Annotation[#, "Vanadium40", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[433],
        "\"Vanadium41\""],
       Annotation[#, "Vanadium41", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[434],
        "\"Vanadium42\""],
       Annotation[#, "Vanadium42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[435],
        "\"Vanadium43\""],
       Annotation[#, "Vanadium43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[436],
        "\"Vanadium44\""],
       Annotation[#, "Vanadium44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[437],
        "\"Vanadium45\""],
       Annotation[#, "Vanadium45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[438],
        "\"Vanadium46\""],
       Annotation[#, "Vanadium46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[439],
        "\"Vanadium47\""],
       Annotation[#, "Vanadium47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[440],
        "\"Vanadium48\""],
       Annotation[#, "Vanadium48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[441],
        "\"Vanadium49\""],
       Annotation[#, "Vanadium49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[442],
        "\"Vanadium50\""],
       Annotation[#, "Vanadium50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[443],
        "\"Chromium50\""],
       Annotation[#, "Chromium50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[444],
        "\"Chromium52\""],
       Annotation[#, "Chromium52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[445],
        "\"Chromium53\""],
       Annotation[#, "Chromium53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[446],
        "\"Chromium54\""],
       Annotation[#, "Chromium54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[447],
        "\"Chromium55\""],
       Annotation[#, "Chromium55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[448],
        "\"Chromium56\""],
       Annotation[#, "Chromium56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[449],
        "\"Chromium57\""],
       Annotation[#, "Chromium57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[450],
        "\"Chromium58\""],
       Annotation[#, "Chromium58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[451],
        "\"Chromium59\""],
       Annotation[#, "Chromium59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[452],
        "\"Chromium60\""],
       Annotation[#, "Chromium60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[453],
        "\"Chromium61\""],
       Annotation[#, "Chromium61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[454],
        "\"Chromium62\""],
       Annotation[#, "Chromium62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[455],
        "\"Chromium63\""],
       Annotation[#, "Chromium63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[456],
        "\"Vanadium64\""],
       Annotation[#, "Vanadium64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[457],
        "\"Chromium64\""],
       Annotation[#, "Chromium64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[458],
        "\"Vanadium65\""],
       Annotation[#, "Vanadium65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[459],
        "\"Chromium65\""],
       Annotation[#, "Chromium65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[460],
        "\"Chromium42\""],
       Annotation[#, "Chromium42", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[461],
        "\"Chromium43\""],
       Annotation[#, "Chromium43", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[462],
        "\"Chromium44\""],
       Annotation[#, "Chromium44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[463],
        "\"Chromium45\""],
       Annotation[#, "Chromium45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[464],
        "\"Chromium46\""],
       Annotation[#, "Chromium46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[465],
        "\"Chromium47\""],
       Annotation[#, "Chromium47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[466],
        "\"Chromium48\""],
       Annotation[#, "Chromium48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[467],
        "\"Chromium49\""],
       Annotation[#, "Chromium49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[468],
        "\"Chromium51\""],
       Annotation[#, "Chromium51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[469],
        "\"Manganese55\""],
       Annotation[#, "Manganese55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[470],
        "\"Manganese56\""],
       Annotation[#, "Manganese56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[471],
        "\"Manganese57\""],
       Annotation[#, "Manganese57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[472],
        "\"Manganese58\""],
       Annotation[#, "Manganese58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[473],
        "\"Manganese59\""],
       Annotation[#, "Manganese59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[474],
        "\"Manganese60\""],
       Annotation[#, "Manganese60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[475],
        "\"Manganese61\""],
       Annotation[#, "Manganese61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[476],
        "\"Manganese62\""],
       Annotation[#, "Manganese62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[477],
        "\"Manganese63\""],
       Annotation[#, "Manganese63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[478],
        "\"Manganese64\""],
       Annotation[#, "Manganese64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[479],
        "\"Manganese65\""],
       Annotation[#, "Manganese65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[480],
        "\"Chromium66\""],
       Annotation[#, "Chromium66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[481],
        "\"Manganese66\""],
       Annotation[#, "Manganese66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[482],
        "\"Chromium67\""],
       Annotation[#, "Chromium67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[483],
        "\"Manganese67\""],
       Annotation[#, "Manganese67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[484],
        "\"Manganese44\""],
       Annotation[#, "Manganese44", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[485],
        "\"Manganese45\""],
       Annotation[#, "Manganese45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[486],
        "\"Manganese46\""],
       Annotation[#, "Manganese46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[487],
        "\"Manganese47\""],
       Annotation[#, "Manganese47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[488],
        "\"Manganese48\""],
       Annotation[#, "Manganese48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[489],
        "\"Manganese49\""],
       Annotation[#, "Manganese49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[490],
        "\"Manganese50\""],
       Annotation[#, "Manganese50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[491],
        "\"Manganese51\""],
       Annotation[#, "Manganese51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[492],
        "\"Manganese52\""],
       Annotation[#, "Manganese52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[493],
        "\"Manganese53\""],
       Annotation[#, "Manganese53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[494],
        "\"Manganese54\""],
       Annotation[#, "Manganese54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[495],
        "\"Iron54\""],
       Annotation[#, "Iron54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[496],
        "\"Iron56\""],
       Annotation[#, "Iron56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[497],
        "\"Iron57\""],
       Annotation[#, "Iron57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[498],
        "\"Iron58\""],
       Annotation[#, "Iron58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[499],
        "\"Iron59\""],
       Annotation[#, "Iron59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[500],
        "\"Iron60\""],
       Annotation[#, "Iron60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[501],
        "\"Iron61\""],
       Annotation[#, "Iron61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[502],
        "\"Iron62\""],
       Annotation[#, "Iron62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[503],
        "\"Iron63\""],
       Annotation[#, "Iron63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[504],
        "\"Iron64\""],
       Annotation[#, "Iron64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[505],
        "\"Iron65\""],
       Annotation[#, "Iron65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[506],
        "\"Iron66\""],
       Annotation[#, "Iron66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[507],
        "\"Iron67\""],
       Annotation[#, "Iron67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[508],
        "\"Manganese68\""],
       Annotation[#, "Manganese68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[509],
        "\"Iron68\""],
       Annotation[#, "Iron68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[510],
        "\"Manganese69\""],
       Annotation[#, "Manganese69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[511],
        "\"Iron69\""],
       Annotation[#, "Iron69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[512],
        "\"Iron45\""],
       Annotation[#, "Iron45", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[513],
        "\"Iron46\""],
       Annotation[#, "Iron46", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[514],
        "\"Iron47\""],
       Annotation[#, "Iron47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[515],
        "\"Iron48\""],
       Annotation[#, "Iron48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[516],
        "\"Iron49\""],
       Annotation[#, "Iron49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[517],
        "\"Iron50\""],
       Annotation[#, "Iron50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[518],
        "\"Iron51\""],
       Annotation[#, "Iron51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[519],
        "\"Iron52\""],
       Annotation[#, "Iron52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[520],
        "\"Iron53\""],
       Annotation[#, "Iron53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[521],
        "\"Iron55\""],
       Annotation[#, "Iron55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[522],
        "\"Cobalt59\""],
       Annotation[#, "Cobalt59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[523],
        "\"Cobalt60\""],
       Annotation[#, "Cobalt60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[524],
        "\"Cobalt61\""],
       Annotation[#, "Cobalt61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[525],
        "\"Cobalt62\""],
       Annotation[#, "Cobalt62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[526],
        "\"Cobalt63\""],
       Annotation[#, "Cobalt63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[527],
        "\"Cobalt64\""],
       Annotation[#, "Cobalt64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[528],
        "\"Cobalt65\""],
       Annotation[#, "Cobalt65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[529],
        "\"Cobalt66\""],
       Annotation[#, "Cobalt66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[530],
        "\"Cobalt67\""],
       Annotation[#, "Cobalt67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[531],
        "\"Cobalt68\""],
       Annotation[#, "Cobalt68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[532],
        "\"Cobalt69\""],
       Annotation[#, "Cobalt69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[533],
        "\"Iron70\""],
       Annotation[#, "Iron70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[534],
        "\"Cobalt70\""],
       Annotation[#, "Cobalt70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[535],
        "\"Iron71\""],
       Annotation[#, "Iron71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[536],
        "\"Cobalt71\""],
       Annotation[#, "Cobalt71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[537],
        "\"Iron72\""],
       Annotation[#, "Iron72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[538],
        "\"Cobalt72\""],
       Annotation[#, "Cobalt72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[539],
        "\"Cobalt47\""],
       Annotation[#, "Cobalt47", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[540],
        "\"Cobalt48\""],
       Annotation[#, "Cobalt48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[541],
        "\"Cobalt49\""],
       Annotation[#, "Cobalt49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[542],
        "\"Cobalt50\""],
       Annotation[#, "Cobalt50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[543],
        "\"Cobalt51\""],
       Annotation[#, "Cobalt51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[544],
        "\"Cobalt52\""],
       Annotation[#, "Cobalt52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[545],
        "\"Cobalt53\""],
       Annotation[#, "Cobalt53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[546],
        "\"Cobalt54\""],
       Annotation[#, "Cobalt54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[547],
        "\"Cobalt55\""],
       Annotation[#, "Cobalt55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[548],
        "\"Cobalt56\""],
       Annotation[#, "Cobalt56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[549],
        "\"Cobalt57\""],
       Annotation[#, "Cobalt57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[550],
        "\"Cobalt58\""],
       Annotation[#, "Cobalt58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[551],
        "\"Nickel60\""],
       Annotation[#, "Nickel60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[552],
        "\"Nickel61\""],
       Annotation[#, "Nickel61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[553],
        "\"Nickel62\""],
       Annotation[#, "Nickel62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[554],
        "\"Nickel63\""],
       Annotation[#, "Nickel63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[555],
        "\"Nickel64\""],
       Annotation[#, "Nickel64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[556],
        "\"Nickel65\""],
       Annotation[#, "Nickel65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[557],
        "\"Nickel66\""],
       Annotation[#, "Nickel66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[558],
        "\"Nickel67\""],
       Annotation[#, "Nickel67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[559],
        "\"Nickel68\""],
       Annotation[#, "Nickel68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[560],
        "\"Nickel69\""],
       Annotation[#, "Nickel69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[561],
        "\"Nickel70\""],
       Annotation[#, "Nickel70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[562],
        "\"Nickel71\""],
       Annotation[#, "Nickel71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[563],
        "\"Nickel72\""],
       Annotation[#, "Nickel72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[564],
        "\"Cobalt73\""],
       Annotation[#, "Cobalt73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[565],
        "\"Nickel73\""],
       Annotation[#, "Nickel73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[566],
        "\"Cobalt74\""],
       Annotation[#, "Cobalt74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[567],
        "\"Nickel74\""],
       Annotation[#, "Nickel74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[568],
        "\"Cobalt75\""],
       Annotation[#, "Cobalt75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[569],
        "\"Nickel75\""],
       Annotation[#, "Nickel75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[570],
        "\"Nickel48\""],
       Annotation[#, "Nickel48", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[571],
        "\"Nickel49\""],
       Annotation[#, "Nickel49", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[572],
        "\"Nickel50\""],
       Annotation[#, "Nickel50", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[573],
        "\"Nickel51\""],
       Annotation[#, "Nickel51", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[574],
        "\"Nickel52\""],
       Annotation[#, "Nickel52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[575],
        "\"Nickel53\""],
       Annotation[#, "Nickel53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[576],
        "\"Nickel54\""],
       Annotation[#, "Nickel54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[577],
        "\"Nickel55\""],
       Annotation[#, "Nickel55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[578],
        "\"Nickel56\""],
       Annotation[#, "Nickel56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[579],
        "\"Nickel57\""],
       Annotation[#, "Nickel57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[580],
        "\"Nickel58\""],
       Annotation[#, "Nickel58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[581],
        "\"Nickel59\""],
       Annotation[#, "Nickel59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[582],
        "\"Copper63\""],
       Annotation[#, "Copper63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[583],
        "\"Copper65\""],
       Annotation[#, "Copper65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[584],
        "\"Copper66\""],
       Annotation[#, "Copper66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[585],
        "\"Copper67\""],
       Annotation[#, "Copper67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[586],
        "\"Copper68\""],
       Annotation[#, "Copper68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[587],
        "\"Copper69\""],
       Annotation[#, "Copper69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[588],
        "\"Copper70\""],
       Annotation[#, "Copper70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[589],
        "\"Copper71\""],
       Annotation[#, "Copper71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[590],
        "\"Copper72\""],
       Annotation[#, "Copper72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[591],
        "\"Copper73\""],
       Annotation[#, "Copper73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[592],
        "\"Copper74\""],
       Annotation[#, "Copper74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[593],
        "\"Copper75\""],
       Annotation[#, "Copper75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[594],
        "\"Nickel76\""],
       Annotation[#, "Nickel76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[595],
        "\"Copper76\""],
       Annotation[#, "Copper76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[596],
        "\"Nickel77\""],
       Annotation[#, "Nickel77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[597],
        "\"Copper77\""],
       Annotation[#, "Copper77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[598],
        "\"Nickel78\""],
       Annotation[#, "Nickel78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[599],
        "\"Copper78\""],
       Annotation[#, "Copper78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[600],
        "\"Copper52\""],
       Annotation[#, "Copper52", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[601],
        "\"Copper53\""],
       Annotation[#, "Copper53", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[602],
        "\"Copper54\""],
       Annotation[#, "Copper54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[603],
        "\"Copper55\""],
       Annotation[#, "Copper55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[604],
        "\"Copper56\""],
       Annotation[#, "Copper56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[605],
        "\"Copper57\""],
       Annotation[#, "Copper57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[606],
        "\"Copper58\""],
       Annotation[#, "Copper58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[607],
        "\"Copper59\""],
       Annotation[#, "Copper59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[608],
        "\"Copper60\""],
       Annotation[#, "Copper60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[609],
        "\"Copper61\""],
       Annotation[#, "Copper61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[610],
        "\"Copper62\""],
       Annotation[#, "Copper62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[611],
        "\"Copper64\""],
       Annotation[#, "Copper64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[612],
        "\"Zinc64\""],
       Annotation[#, "Zinc64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[613],
        "\"Zinc66\""],
       Annotation[#, "Zinc66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[614],
        "\"Zinc67\""],
       Annotation[#, "Zinc67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[615],
        "\"Zinc68\""],
       Annotation[#, "Zinc68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[616],
        "\"Zinc69\""],
       Annotation[#, "Zinc69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[617],
        "\"Zinc70\""],
       Annotation[#, "Zinc70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[618],
        "\"Zinc71\""],
       Annotation[#, "Zinc71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[619],
        "\"Zinc72\""],
       Annotation[#, "Zinc72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[620],
        "\"Zinc73\""],
       Annotation[#, "Zinc73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[621],
        "\"Zinc74\""],
       Annotation[#, "Zinc74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[622],
        "\"Zinc75\""],
       Annotation[#, "Zinc75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[623],
        "\"Zinc76\""],
       Annotation[#, "Zinc76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[624],
        "\"Zinc77\""],
       Annotation[#, "Zinc77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[625],
        "\"Zinc78\""],
       Annotation[#, "Zinc78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[626],
        "\"Copper79\""],
       Annotation[#, "Copper79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[627],
        "\"Zinc79\""],
       Annotation[#, "Zinc79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[628],
        "\"Copper80\""],
       Annotation[#, "Copper80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[629],
        "\"Zinc80\""],
       Annotation[#, "Zinc80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[630],
        "\"Zinc54\""],
       Annotation[#, "Zinc54", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[631],
        "\"Zinc55\""],
       Annotation[#, "Zinc55", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[632],
        "\"Zinc56\""],
       Annotation[#, "Zinc56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[633],
        "\"Zinc57\""],
       Annotation[#, "Zinc57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[634],
        "\"Zinc58\""],
       Annotation[#, "Zinc58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[635],
        "\"Zinc59\""],
       Annotation[#, "Zinc59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[636],
        "\"Zinc60\""],
       Annotation[#, "Zinc60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[637],
        "\"Zinc61\""],
       Annotation[#, "Zinc61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[638],
        "\"Zinc62\""],
       Annotation[#, "Zinc62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[639],
        "\"Zinc63\""],
       Annotation[#, "Zinc63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[640],
        "\"Zinc65\""],
       Annotation[#, "Zinc65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[641],
        "\"Gallium69\""],
       Annotation[#, "Gallium69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[642],
        "\"Germanium70\""],
       Annotation[#, "Germanium70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[643],
        "\"Gallium71\""],
       Annotation[#, "Gallium71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[644],
        "\"Gallium72\""],
       Annotation[#, "Gallium72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[645],
        "\"Gallium73\""],
       Annotation[#, "Gallium73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[646],
        "\"Gallium74\""],
       Annotation[#, "Gallium74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[647],
        "\"Gallium75\""],
       Annotation[#, "Gallium75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[648],
        "\"Gallium76\""],
       Annotation[#, "Gallium76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[649],
        "\"Gallium77\""],
       Annotation[#, "Gallium77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[650],
        "\"Gallium78\""],
       Annotation[#, "Gallium78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[651],
        "\"Gallium79\""],
       Annotation[#, "Gallium79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[652],
        "\"Gallium80\""],
       Annotation[#, "Gallium80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[653],
        "\"Zinc81\""],
       Annotation[#, "Zinc81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[654],
        "\"Gallium81\""],
       Annotation[#, "Gallium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[655],
        "\"Zinc82\""],
       Annotation[#, "Zinc82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[656],
        "\"Gallium82\""],
       Annotation[#, "Gallium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[657],
        "\"Zinc83\""],
       Annotation[#, "Zinc83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[658],
        "\"Gallium83\""],
       Annotation[#, "Gallium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[659],
        "\"Gallium56\""],
       Annotation[#, "Gallium56", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[660],
        "\"Gallium57\""],
       Annotation[#, "Gallium57", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[661],
        "\"Gallium58\""],
       Annotation[#, "Gallium58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[662],
        "\"Gallium59\""],
       Annotation[#, "Gallium59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[663],
        "\"Gallium60\""],
       Annotation[#, "Gallium60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[664],
        "\"Gallium61\""],
       Annotation[#, "Gallium61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[665],
        "\"Gallium62\""],
       Annotation[#, "Gallium62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[666],
        "\"Gallium63\""],
       Annotation[#, "Gallium63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[667],
        "\"Gallium64\""],
       Annotation[#, "Gallium64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[668],
        "\"Gallium65\""],
       Annotation[#, "Gallium65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[669],
        "\"Gallium66\""],
       Annotation[#, "Gallium66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[670],
        "\"Gallium67\""],
       Annotation[#, "Gallium67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[671],
        "\"Gallium68\""],
       Annotation[#, "Gallium68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[672],
        "\"Gallium70\""],
       Annotation[#, "Gallium70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[673],
        "\"Germanium72\""],
       Annotation[#, "Germanium72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[674],
        "\"Germanium73\""],
       Annotation[#, "Germanium73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[675],
        "\"Germanium74\""],
       Annotation[#, "Germanium74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[676],
        "\"Germanium75\""],
       Annotation[#, "Germanium75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[677],
        "\"Germanium76\""],
       Annotation[#, "Germanium76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[678],
        "\"Germanium77\""],
       Annotation[#, "Germanium77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[679],
        "\"Germanium78\""],
       Annotation[#, "Germanium78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[680],
        "\"Germanium79\""],
       Annotation[#, "Germanium79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[681],
        "\"Germanium80\""],
       Annotation[#, "Germanium80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[682],
        "\"Germanium81\""],
       Annotation[#, "Germanium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[683],
        "\"Germanium82\""],
       Annotation[#, "Germanium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[684],
        "\"Germanium83\""],
       Annotation[#, "Germanium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[685],
        "\"Gallium84\""],
       Annotation[#, "Gallium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[686],
        "\"Germanium84\""],
       Annotation[#, "Germanium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[687],
        "\"Gallium85\""],
       Annotation[#, "Gallium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[688],
        "\"Germanium85\""],
       Annotation[#, "Germanium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[689],
        "\"Gallium86\""],
       Annotation[#, "Gallium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[690],
        "\"Germanium86\""],
       Annotation[#, "Germanium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[691],
        "\"Germanium58\""],
       Annotation[#, "Germanium58", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[692],
        "\"Germanium59\""],
       Annotation[#, "Germanium59", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[693],
        "\"Germanium60\""],
       Annotation[#, "Germanium60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[694],
        "\"Germanium61\""],
       Annotation[#, "Germanium61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[695],
        "\"Germanium62\""],
       Annotation[#, "Germanium62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[696],
        "\"Germanium63\""],
       Annotation[#, "Germanium63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[697],
        "\"Germanium64\""],
       Annotation[#, "Germanium64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[698],
        "\"Germanium65\""],
       Annotation[#, "Germanium65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[699],
        "\"Germanium66\""],
       Annotation[#, "Germanium66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[700],
        "\"Germanium67\""],
       Annotation[#, "Germanium67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[701],
        "\"Germanium68\""],
       Annotation[#, "Germanium68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[702],
        "\"Germanium69\""],
       Annotation[#, "Germanium69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[703],
        "\"Germanium71\""],
       Annotation[#, "Germanium71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[704],
        "\"Arsenic75\""],
       Annotation[#, "Arsenic75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[705],
        "\"Selenium76\""],
       Annotation[#, "Selenium76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[706],
        "\"Arsenic77\""],
       Annotation[#, "Arsenic77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[707],
        "\"Arsenic78\""],
       Annotation[#, "Arsenic78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[708],
        "\"Arsenic79\""],
       Annotation[#, "Arsenic79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[709],
        "\"Arsenic80\""],
       Annotation[#, "Arsenic80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[710],
        "\"Arsenic81\""],
       Annotation[#, "Arsenic81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[711],
        "\"Arsenic82\""],
       Annotation[#, "Arsenic82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[712],
        "\"Arsenic83\""],
       Annotation[#, "Arsenic83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[713],
        "\"Arsenic84\""],
       Annotation[#, "Arsenic84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[714],
        "\"Arsenic85\""],
       Annotation[#, "Arsenic85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[715],
        "\"Arsenic86\""],
       Annotation[#, "Arsenic86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[716],
        "\"Germanium87\""],
       Annotation[#, "Germanium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[717],
        "\"Arsenic87\""],
       Annotation[#, "Arsenic87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[718],
        "\"Germanium88\""],
       Annotation[#, "Germanium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[719],
        "\"Arsenic88\""],
       Annotation[#, "Arsenic88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[720],
        "\"Germanium89\""],
       Annotation[#, "Germanium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[721],
        "\"Arsenic89\""],
       Annotation[#, "Arsenic89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[722],
        "\"Arsenic60\""],
       Annotation[#, "Arsenic60", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[723],
        "\"Arsenic61\""],
       Annotation[#, "Arsenic61", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[724],
        "\"Arsenic62\""],
       Annotation[#, "Arsenic62", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[725],
        "\"Arsenic63\""],
       Annotation[#, "Arsenic63", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[726],
        "\"Arsenic64\""],
       Annotation[#, "Arsenic64", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[727],
        "\"Arsenic65\""],
       Annotation[#, "Arsenic65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[728],
        "\"Arsenic66\""],
       Annotation[#, "Arsenic66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[729],
        "\"Arsenic67\""],
       Annotation[#, "Arsenic67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[730],
        "\"Arsenic68\""],
       Annotation[#, "Arsenic68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[731],
        "\"Arsenic69\""],
       Annotation[#, "Arsenic69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[732],
        "\"Arsenic70\""],
       Annotation[#, "Arsenic70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[733],
        "\"Arsenic71\""],
       Annotation[#, "Arsenic71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[734],
        "\"Arsenic72\""],
       Annotation[#, "Arsenic72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[735],
        "\"Arsenic73\""],
       Annotation[#, "Arsenic73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[736],
        "\"Arsenic74\""],
       Annotation[#, "Arsenic74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[737],
        "\"Selenium74\""],
       Annotation[#, "Selenium74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[738],
        "\"Arsenic76\""],
       Annotation[#, "Arsenic76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[739],
        "\"Selenium77\""],
       Annotation[#, "Selenium77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[740],
        "\"Selenium78\""],
       Annotation[#, "Selenium78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[741],
        "\"Selenium79\""],
       Annotation[#, "Selenium79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[742],
        "\"Selenium80\""],
       Annotation[#, "Selenium80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[743],
        "\"Selenium81\""],
       Annotation[#, "Selenium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[744],
        "\"Selenium82\""],
       Annotation[#, "Selenium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[745],
        "\"Selenium83\""],
       Annotation[#, "Selenium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[746],
        "\"Selenium84\""],
       Annotation[#, "Selenium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[747],
        "\"Selenium85\""],
       Annotation[#, "Selenium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[748],
        "\"Selenium86\""],
       Annotation[#, "Selenium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[749],
        "\"Selenium87\""],
       Annotation[#, "Selenium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[750],
        "\"Selenium88\""],
       Annotation[#, "Selenium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[751],
        "\"Selenium89\""],
       Annotation[#, "Selenium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[752],
        "\"Arsenic90\""],
       Annotation[#, "Arsenic90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[753],
        "\"Selenium90\""],
       Annotation[#, "Selenium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[754],
        "\"Arsenic91\""],
       Annotation[#, "Arsenic91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[755],
        "\"Selenium91\""],
       Annotation[#, "Selenium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[756],
        "\"Arsenic92\""],
       Annotation[#, "Arsenic92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[757],
        "\"Selenium92\""],
       Annotation[#, "Selenium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[758],
        "\"Selenium65\""],
       Annotation[#, "Selenium65", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[759],
        "\"Selenium66\""],
       Annotation[#, "Selenium66", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[760],
        "\"Selenium67\""],
       Annotation[#, "Selenium67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[761],
        "\"Selenium68\""],
       Annotation[#, "Selenium68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[762],
        "\"Selenium69\""],
       Annotation[#, "Selenium69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[763],
        "\"Selenium70\""],
       Annotation[#, "Selenium70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[764],
        "\"Selenium71\""],
       Annotation[#, "Selenium71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[765],
        "\"Selenium72\""],
       Annotation[#, "Selenium72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[766],
        "\"Selenium73\""],
       Annotation[#, "Selenium73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[767],
        "\"Selenium75\""],
       Annotation[#, "Selenium75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[768],
        "\"Bromine79\""],
       Annotation[#, "Bromine79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[769],
        "\"Krypton80\""],
       Annotation[#, "Krypton80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[770],
        "\"Bromine81\""],
       Annotation[#, "Bromine81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[771],
        "\"Krypton82\""],
       Annotation[#, "Krypton82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[772],
        "\"Bromine83\""],
       Annotation[#, "Bromine83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[773],
        "\"Bromine84\""],
       Annotation[#, "Bromine84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[774],
        "\"Bromine85\""],
       Annotation[#, "Bromine85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[775],
        "\"Bromine86\""],
       Annotation[#, "Bromine86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[776],
        "\"Bromine87\""],
       Annotation[#, "Bromine87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[777],
        "\"Bromine88\""],
       Annotation[#, "Bromine88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[778],
        "\"Bromine89\""],
       Annotation[#, "Bromine89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[779],
        "\"Bromine90\""],
       Annotation[#, "Bromine90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[780],
        "\"Bromine91\""],
       Annotation[#, "Bromine91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[781],
        "\"Bromine92\""],
       Annotation[#, "Bromine92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[782],
        "\"Selenium93\""],
       Annotation[#, "Selenium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[783],
        "\"Bromine93\""],
       Annotation[#, "Bromine93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[784],
        "\"Selenium94\""],
       Annotation[#, "Selenium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[785],
        "\"Bromine94\""],
       Annotation[#, "Bromine94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[786],
        "\"Bromine67\""],
       Annotation[#, "Bromine67", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[787],
        "\"Bromine68\""],
       Annotation[#, "Bromine68", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[788],
        "\"Bromine69\""],
       Annotation[#, "Bromine69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[789],
        "\"Bromine70\""],
       Annotation[#, "Bromine70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[790],
        "\"Bromine71\""],
       Annotation[#, "Bromine71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[791],
        "\"Bromine72\""],
       Annotation[#, "Bromine72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[792],
        "\"Bromine73\""],
       Annotation[#, "Bromine73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[793],
        "\"Bromine74\""],
       Annotation[#, "Bromine74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[794],
        "\"Bromine75\""],
       Annotation[#, "Bromine75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[795],
        "\"Bromine76\""],
       Annotation[#, "Bromine76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[796],
        "\"Bromine77\""],
       Annotation[#, "Bromine77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[797],
        "\"Bromine78\""],
       Annotation[#, "Bromine78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[798],
        "\"Krypton78\""],
       Annotation[#, "Krypton78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[799],
        "\"Bromine80\""],
       Annotation[#, "Bromine80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[800],
        "\"Bromine82\""],
       Annotation[#, "Bromine82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[801],
        "\"Krypton83\""],
       Annotation[#, "Krypton83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[802],
        "\"Krypton84\""],
       Annotation[#, "Krypton84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[803],
        "\"Krypton85\""],
       Annotation[#, "Krypton85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[804],
        "\"Krypton86\""],
       Annotation[#, "Krypton86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[805],
        "\"Krypton87\""],
       Annotation[#, "Krypton87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[806],
        "\"Krypton88\""],
       Annotation[#, "Krypton88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[807],
        "\"Krypton89\""],
       Annotation[#, "Krypton89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[808],
        "\"Krypton90\""],
       Annotation[#, "Krypton90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[809],
        "\"Krypton91\""],
       Annotation[#, "Krypton91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[810],
        "\"Krypton92\""],
       Annotation[#, "Krypton92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[811],
        "\"Krypton93\""],
       Annotation[#, "Krypton93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[812],
        "\"Krypton94\""],
       Annotation[#, "Krypton94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[813],
        "\"Bromine95\""],
       Annotation[#, "Bromine95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[814],
        "\"Krypton95\""],
       Annotation[#, "Krypton95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[815],
        "\"Bromine96\""],
       Annotation[#, "Bromine96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[816],
        "\"Krypton96\""],
       Annotation[#, "Krypton96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[817],
        "\"Bromine97\""],
       Annotation[#, "Bromine97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[818],
        "\"Krypton97\""],
       Annotation[#, "Krypton97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[819],
        "\"Krypton69\""],
       Annotation[#, "Krypton69", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[820],
        "\"Krypton70\""],
       Annotation[#, "Krypton70", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[821],
        "\"Krypton71\""],
       Annotation[#, "Krypton71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[822],
        "\"Krypton72\""],
       Annotation[#, "Krypton72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[823],
        "\"Krypton73\""],
       Annotation[#, "Krypton73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[824],
        "\"Krypton74\""],
       Annotation[#, "Krypton74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[825],
        "\"Krypton75\""],
       Annotation[#, "Krypton75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[826],
        "\"Krypton76\""],
       Annotation[#, "Krypton76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[827],
        "\"Krypton77\""],
       Annotation[#, "Krypton77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[828],
        "\"Krypton79\""],
       Annotation[#, "Krypton79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[829],
        "\"Krypton81\""],
       Annotation[#, "Krypton81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[830],
        "\"Rubidium85\""],
       Annotation[#, "Rubidium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[831],
        "\"Strontium86\""],
       Annotation[#, "Strontium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[832],
        "\"Rubidium87\""],
       Annotation[#, "Rubidium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[833],
        "\"Rubidium88\""],
       Annotation[#, "Rubidium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[834],
        "\"Rubidium89\""],
       Annotation[#, "Rubidium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[835],
        "\"Rubidium90\""],
       Annotation[#, "Rubidium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[836],
        "\"Rubidium91\""],
       Annotation[#, "Rubidium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[837],
        "\"Rubidium92\""],
       Annotation[#, "Rubidium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[838],
        "\"Rubidium93\""],
       Annotation[#, "Rubidium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[839],
        "\"Rubidium94\""],
       Annotation[#, "Rubidium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[840],
        "\"Rubidium95\""],
       Annotation[#, "Rubidium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[841],
        "\"Rubidium96\""],
       Annotation[#, "Rubidium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[842],
        "\"Rubidium97\""],
       Annotation[#, "Rubidium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[843],
        "\"Krypton98\""],
       Annotation[#, "Krypton98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[844],
        "\"Rubidium98\""],
       Annotation[#, "Rubidium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[845],
        "\"Krypton99\""],
       Annotation[#, "Krypton99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[846],
        "\"Rubidium99\""],
       Annotation[#, "Rubidium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[847],
        "\"Krypton100\""],
       Annotation[#, "Krypton100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[848],
        "\"Rubidium100\""],
       Annotation[#, "Rubidium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[849],
        "\"Rubidium71\""],
       Annotation[#, "Rubidium71", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[850],
        "\"Rubidium72\""],
       Annotation[#, "Rubidium72", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[851],
        "\"Rubidium73\""],
       Annotation[#, "Rubidium73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[852],
        "\"Rubidium74\""],
       Annotation[#, "Rubidium74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[853],
        "\"Rubidium75\""],
       Annotation[#, "Rubidium75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[854],
        "\"Rubidium76\""],
       Annotation[#, "Rubidium76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[855],
        "\"Rubidium77\""],
       Annotation[#, "Rubidium77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[856],
        "\"Rubidium78\""],
       Annotation[#, "Rubidium78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[857],
        "\"Rubidium79\""],
       Annotation[#, "Rubidium79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[858],
        "\"Rubidium80\""],
       Annotation[#, "Rubidium80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[859],
        "\"Rubidium81\""],
       Annotation[#, "Rubidium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[860],
        "\"Rubidium82\""],
       Annotation[#, "Rubidium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[861],
        "\"Rubidium83\""],
       Annotation[#, "Rubidium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[862],
        "\"Rubidium84\""],
       Annotation[#, "Rubidium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[863],
        "\"Strontium84\""],
       Annotation[#, "Strontium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[864],
        "\"Rubidium86\""],
       Annotation[#, "Rubidium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[865],
        "\"Strontium87\""],
       Annotation[#, "Strontium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[866],
        "\"Strontium88\""],
       Annotation[#, "Strontium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[867],
        "\"Strontium89\""],
       Annotation[#, "Strontium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[868],
        "\"Strontium90\""],
       Annotation[#, "Strontium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[869],
        "\"Strontium91\""],
       Annotation[#, "Strontium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[870],
        "\"Strontium92\""],
       Annotation[#, "Strontium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[871],
        "\"Strontium93\""],
       Annotation[#, "Strontium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[872],
        "\"Strontium94\""],
       Annotation[#, "Strontium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[873],
        "\"Strontium95\""],
       Annotation[#, "Strontium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[874],
        "\"Strontium96\""],
       Annotation[#, "Strontium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[875],
        "\"Strontium97\""],
       Annotation[#, "Strontium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[876],
        "\"Strontium98\""],
       Annotation[#, "Strontium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[877],
        "\"Strontium99\""],
       Annotation[#, "Strontium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[878],
        "\"Strontium100\""],
       Annotation[#, "Strontium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[879],
        "\"Rubidium101\""],
       Annotation[#, "Rubidium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[880],
        "\"Strontium101\""],
       Annotation[#, "Strontium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[881],
        "\"Rubidium102\""],
       Annotation[#, "Rubidium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[882],
        "\"Strontium102\""],
       Annotation[#, "Strontium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[883],
        "\"Strontium73\""],
       Annotation[#, "Strontium73", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[884],
        "\"Strontium74\""],
       Annotation[#, "Strontium74", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[885],
        "\"Strontium75\""],
       Annotation[#, "Strontium75", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[886],
        "\"Strontium76\""],
       Annotation[#, "Strontium76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[887],
        "\"Strontium77\""],
       Annotation[#, "Strontium77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[888],
        "\"Strontium78\""],
       Annotation[#, "Strontium78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[889],
        "\"Strontium79\""],
       Annotation[#, "Strontium79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[890],
        "\"Strontium80\""],
       Annotation[#, "Strontium80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[891],
        "\"Strontium81\""],
       Annotation[#, "Strontium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[892],
        "\"Strontium82\""],
       Annotation[#, "Strontium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[893],
        "\"Strontium83\""],
       Annotation[#, "Strontium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[894],
        "\"Strontium85\""],
       Annotation[#, "Strontium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[895],
        "\"Yttrium89\""],
       Annotation[#, "Yttrium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[896],
        "\"Yttrium90\""],
       Annotation[#, "Yttrium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[897],
        "\"Yttrium91\""],
       Annotation[#, "Yttrium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[898],
        "\"Yttrium92\""],
       Annotation[#, "Yttrium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[899],
        "\"Yttrium93\""],
       Annotation[#, "Yttrium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[900],
        "\"Yttrium94\""],
       Annotation[#, "Yttrium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[901],
        "\"Yttrium95\""],
       Annotation[#, "Yttrium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[902],
        "\"Yttrium96\""],
       Annotation[#, "Yttrium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[903],
        "\"Yttrium97\""],
       Annotation[#, "Yttrium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[904],
        "\"Yttrium98\""],
       Annotation[#, "Yttrium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[905],
        "\"Yttrium99\""],
       Annotation[#, "Yttrium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[906],
        "\"Yttrium100\""],
       Annotation[#, "Yttrium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[907],
        "\"Yttrium101\""],
       Annotation[#, "Yttrium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[908],
        "\"Yttrium102\""],
       Annotation[#, "Yttrium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[909],
        "\"Strontium103\""],
       Annotation[#, "Strontium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[910],
        "\"Yttrium103\""],
       Annotation[#, "Yttrium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[911],
        "\"Strontium104\""],
       Annotation[#, "Strontium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[912],
        "\"Yttrium104\""],
       Annotation[#, "Yttrium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[913],
        "\"Strontium105\""],
       Annotation[#, "Strontium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[914],
        "\"Yttrium105\""],
       Annotation[#, "Yttrium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[915],
        "\"Yttrium76\""],
       Annotation[#, "Yttrium76", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[916],
        "\"Yttrium77\""],
       Annotation[#, "Yttrium77", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[917],
        "\"Yttrium78\""],
       Annotation[#, "Yttrium78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[918],
        "\"Yttrium79\""],
       Annotation[#, "Yttrium79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[919],
        "\"Yttrium80\""],
       Annotation[#, "Yttrium80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[920],
        "\"Yttrium81\""],
       Annotation[#, "Yttrium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[921],
        "\"Yttrium82\""],
       Annotation[#, "Yttrium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[922],
        "\"Yttrium83\""],
       Annotation[#, "Yttrium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[923],
        "\"Yttrium84\""],
       Annotation[#, "Yttrium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[924],
        "\"Yttrium85\""],
       Annotation[#, "Yttrium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[925],
        "\"Yttrium86\""],
       Annotation[#, "Yttrium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[926],
        "\"Yttrium87\""],
       Annotation[#, "Yttrium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[927],
        "\"Yttrium88\""],
       Annotation[#, "Yttrium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[928],
        "\"Zirconium90\""],
       Annotation[#, "Zirconium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[929],
        "\"Zirconium91\""],
       Annotation[#, "Zirconium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[930],
        "\"Zirconium92\""],
       Annotation[#, "Zirconium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[931],
        "\"Zirconium93\""],
       Annotation[#, "Zirconium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[932],
        "\"Zirconium94\""],
       Annotation[#, "Zirconium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[933],
        "\"Zirconium95\""],
       Annotation[#, "Zirconium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[934],
        "\"Zirconium96\""],
       Annotation[#, "Zirconium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[935],
        "\"Zirconium97\""],
       Annotation[#, "Zirconium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[936],
        "\"Zirconium98\""],
       Annotation[#, "Zirconium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[937],
        "\"Zirconium99\""],
       Annotation[#, "Zirconium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[938],
        "\"Zirconium100\""],
       Annotation[#, "Zirconium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[939],
        "\"Zirconium101\""],
       Annotation[#, "Zirconium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[940],
        "\"Zirconium102\""],
       Annotation[#, "Zirconium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[941],
        "\"Zirconium103\""],
       Annotation[#, "Zirconium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[942],
        "\"Zirconium104\""],
       Annotation[#, "Zirconium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[943],
        "\"Zirconium105\""],
       Annotation[#, "Zirconium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[944],
        "\"Yttrium106\""],
       Annotation[#, "Yttrium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[945],
        "\"Zirconium106\""],
       Annotation[#, "Zirconium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[946],
        "\"Yttrium107\""],
       Annotation[#, "Yttrium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[947],
        "\"Zirconium107\""],
       Annotation[#, "Zirconium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[948],
        "\"Yttrium108\""],
       Annotation[#, "Yttrium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[949],
        "\"Zirconium108\""],
       Annotation[#, "Zirconium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[950],
        "\"Zirconium78\""],
       Annotation[#, "Zirconium78", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[951],
        "\"Zirconium79\""],
       Annotation[#, "Zirconium79", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[952],
        "\"Zirconium80\""],
       Annotation[#, "Zirconium80", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[953],
        "\"Zirconium81\""],
       Annotation[#, "Zirconium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[954],
        "\"Zirconium82\""],
       Annotation[#, "Zirconium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[955],
        "\"Zirconium83\""],
       Annotation[#, "Zirconium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[956],
        "\"Zirconium84\""],
       Annotation[#, "Zirconium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[957],
        "\"Zirconium85\""],
       Annotation[#, "Zirconium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[958],
        "\"Zirconium86\""],
       Annotation[#, "Zirconium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[959],
        "\"Zirconium87\""],
       Annotation[#, "Zirconium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[960],
        "\"Zirconium88\""],
       Annotation[#, "Zirconium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[961],
        "\"Zirconium89\""],
       Annotation[#, "Zirconium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[962],
        "\"Niobium93\""],
       Annotation[#, "Niobium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[963],
        "\"Molybdenum94\""],
       Annotation[#, "Molybdenum94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[964],
        "\"Niobium95\""],
       Annotation[#, "Niobium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[965],
        "\"Molybdenum96\""],
       Annotation[#, "Molybdenum96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[966],
        "\"Niobium97\""],
       Annotation[#, "Niobium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[967],
        "\"Niobium98\""],
       Annotation[#, "Niobium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[968],
        "\"Niobium99\""],
       Annotation[#, "Niobium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[969],
        "\"Niobium100\""],
       Annotation[#, "Niobium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[970],
        "\"Niobium101\""],
       Annotation[#, "Niobium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[971],
        "\"Niobium102\""],
       Annotation[#, "Niobium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[972],
        "\"Niobium103\""],
       Annotation[#, "Niobium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[973],
        "\"Niobium104\""],
       Annotation[#, "Niobium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[974],
        "\"Niobium105\""],
       Annotation[#, "Niobium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[975],
        "\"Niobium106\""],
       Annotation[#, "Niobium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[976],
        "\"Niobium107\""],
       Annotation[#, "Niobium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[977],
        "\"Niobium108\""],
       Annotation[#, "Niobium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[978],
        "\"Zirconium109\""],
       Annotation[#, "Zirconium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[979],
        "\"Niobium109\""],
       Annotation[#, "Niobium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[980],
        "\"Zirconium110\""],
       Annotation[#, "Zirconium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[981],
        "\"Niobium110\""],
       Annotation[#, "Niobium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[982],
        "\"Niobium81\""],
       Annotation[#, "Niobium81", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[983],
        "\"Niobium82\""],
       Annotation[#, "Niobium82", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[984],
        "\"Niobium83\""],
       Annotation[#, "Niobium83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[985],
        "\"Niobium84\""],
       Annotation[#, "Niobium84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[986],
        "\"Niobium85\""],
       Annotation[#, "Niobium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[987],
        "\"Niobium86\""],
       Annotation[#, "Niobium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[988],
        "\"Niobium87\""],
       Annotation[#, "Niobium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[989],
        "\"Niobium88\""],
       Annotation[#, "Niobium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[990],
        "\"Niobium89\""],
       Annotation[#, "Niobium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[991],
        "\"Niobium90\""],
       Annotation[#, "Niobium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[992],
        "\"Niobium91\""],
       Annotation[#, "Niobium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[993],
        "\"Niobium92\""],
       Annotation[#, "Niobium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[994],
        "\"Molybdenum92\""],
       Annotation[#, "Molybdenum92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[995],
        "\"Niobium94\""],
       Annotation[#, "Niobium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[996],
        "\"Molybdenum95\""],
       Annotation[#, "Molybdenum95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[997],
        "\"Niobium96\""],
       Annotation[#, "Niobium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[998],
        "\"Molybdenum97\""],
       Annotation[#, "Molybdenum97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[999],
        "\"Molybdenum98\""],
       Annotation[#, "Molybdenum98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1000],
        "\"Molybdenum99\""],
       Annotation[#, "Molybdenum99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1001],
        "\"Molybdenum100\""],
       Annotation[#, "Molybdenum100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1002],
        "\"Molybdenum101\""],
       Annotation[#, "Molybdenum101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1003],
        "\"Molybdenum102\""],
       Annotation[#, "Molybdenum102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1004],
        "\"Molybdenum103\""],
       Annotation[#, "Molybdenum103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1005],
        "\"Molybdenum104\""],
       Annotation[#, "Molybdenum104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1006],
        "\"Molybdenum105\""],
       Annotation[#, "Molybdenum105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1007],
        "\"Molybdenum106\""],
       Annotation[#, "Molybdenum106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1008],
        "\"Molybdenum107\""],
       Annotation[#, "Molybdenum107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1009],
        "\"Molybdenum108\""],
       Annotation[#, "Molybdenum108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1010],
        "\"Molybdenum109\""],
       Annotation[#, "Molybdenum109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1011],
        "\"Molybdenum110\""],
       Annotation[#, "Molybdenum110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1012],
        "\"Niobium111\""],
       Annotation[#, "Niobium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1013],
        "\"Molybdenum111\""],
       Annotation[#, "Molybdenum111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1014],
        "\"Niobium112\""],
       Annotation[#, "Niobium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1015],
        "\"Molybdenum112\""],
       Annotation[#, "Molybdenum112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1016],
        "\"Niobium113\""],
       Annotation[#, "Niobium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1017],
        "\"Molybdenum113\""],
       Annotation[#, "Molybdenum113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1018],
        "\"Molybdenum83\""],
       Annotation[#, "Molybdenum83", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1019],
        "\"Molybdenum84\""],
       Annotation[#, "Molybdenum84", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1020],
        "\"Molybdenum85\""],
       Annotation[#, "Molybdenum85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1021],
        "\"Molybdenum86\""],
       Annotation[#, "Molybdenum86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1022],
        "\"Molybdenum87\""],
       Annotation[#, "Molybdenum87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1023],
        "\"Molybdenum88\""],
       Annotation[#, "Molybdenum88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1024],
        "\"Molybdenum89\""],
       Annotation[#, "Molybdenum89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1025],
        "\"Molybdenum90\""],
       Annotation[#, "Molybdenum90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1026],
        "\"Molybdenum91\""],
       Annotation[#, "Molybdenum91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1027],
        "\"Molybdenum93\""],
       Annotation[#, "Molybdenum93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1028],
        "\"Ruthenium98\""],
       Annotation[#, "Ruthenium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1029],
        "\"Technetium99\""],
       Annotation[#, "Technetium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1030],
        "\"Ruthenium100\""],
       Annotation[#, "Ruthenium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1031],
        "\"Technetium101\""],
       Annotation[#, "Technetium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1032],
        "\"Technetium102\""],
       Annotation[#, "Technetium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1033],
        "\"Technetium103\""],
       Annotation[#, "Technetium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1034],
        "\"Technetium104\""],
       Annotation[#, "Technetium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1035],
        "\"Technetium105\""],
       Annotation[#, "Technetium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1036],
        "\"Technetium106\""],
       Annotation[#, "Technetium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1037],
        "\"Technetium107\""],
       Annotation[#, "Technetium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1038],
        "\"Technetium108\""],
       Annotation[#, "Technetium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1039],
        "\"Technetium109\""],
       Annotation[#, "Technetium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1040],
        "\"Technetium110\""],
       Annotation[#, "Technetium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1041],
        "\"Technetium111\""],
       Annotation[#, "Technetium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1042],
        "\"Technetium112\""],
       Annotation[#, "Technetium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1043],
        "\"Technetium113\""],
       Annotation[#, "Technetium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1044],
        "\"Molybdenum114\""],
       Annotation[#, "Molybdenum114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1045],
        "\"Technetium114\""],
       Annotation[#, "Technetium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1046],
        "\"Molybdenum115\""],
       Annotation[#, "Molybdenum115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1047],
        "\"Technetium115\""],
       Annotation[#, "Technetium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1048],
        "\"Technetium85\""],
       Annotation[#, "Technetium85", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1049],
        "\"Technetium86\""],
       Annotation[#, "Technetium86", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1050],
        "\"Technetium87\""],
       Annotation[#, "Technetium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1051],
        "\"Technetium88\""],
       Annotation[#, "Technetium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1052],
        "\"Technetium89\""],
       Annotation[#, "Technetium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1053],
        "\"Technetium90\""],
       Annotation[#, "Technetium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1054],
        "\"Technetium91\""],
       Annotation[#, "Technetium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1055],
        "\"Technetium92\""],
       Annotation[#, "Technetium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1056],
        "\"Technetium93\""],
       Annotation[#, "Technetium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1057],
        "\"Technetium94\""],
       Annotation[#, "Technetium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1058],
        "\"Technetium95\""],
       Annotation[#, "Technetium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1059],
        "\"Technetium96\""],
       Annotation[#, "Technetium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1060],
        "\"Technetium97\""],
       Annotation[#, "Technetium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1061],
        "\"Technetium98\""],
       Annotation[#, "Technetium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1062],
        "\"Ruthenium99\""],
       Annotation[#, "Ruthenium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1063],
        "\"Technetium100\""],
       Annotation[#, "Technetium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1064],
        "\"Ruthenium101\""],
       Annotation[#, "Ruthenium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1065],
        "\"Ruthenium102\""],
       Annotation[#, "Ruthenium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1066],
        "\"Ruthenium103\""],
       Annotation[#, "Ruthenium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1067],
        "\"Ruthenium104\""],
       Annotation[#, "Ruthenium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1068],
        "\"Ruthenium105\""],
       Annotation[#, "Ruthenium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1069],
        "\"Ruthenium106\""],
       Annotation[#, "Ruthenium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1070],
        "\"Ruthenium107\""],
       Annotation[#, "Ruthenium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1071],
        "\"Ruthenium108\""],
       Annotation[#, "Ruthenium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1072],
        "\"Ruthenium109\""],
       Annotation[#, "Ruthenium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1073],
        "\"Ruthenium110\""],
       Annotation[#, "Ruthenium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1074],
        "\"Ruthenium111\""],
       Annotation[#, "Ruthenium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1075],
        "\"Ruthenium112\""],
       Annotation[#, "Ruthenium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1076],
        "\"Ruthenium113\""],
       Annotation[#, "Ruthenium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1077],
        "\"Ruthenium114\""],
       Annotation[#, "Ruthenium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1078],
        "\"Ruthenium115\""],
       Annotation[#, "Ruthenium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1079],
        "\"Technetium116\""],
       Annotation[#, "Technetium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1080],
        "\"Ruthenium116\""],
       Annotation[#, "Ruthenium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1081],
        "\"Technetium117\""],
       Annotation[#, "Technetium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1082],
        "\"Ruthenium117\""],
       Annotation[#, "Ruthenium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1083],
        "\"Technetium118\""],
       Annotation[#, "Technetium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1084],
        "\"Ruthenium118\""],
       Annotation[#, "Ruthenium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1085],
        "\"Ruthenium87\""],
       Annotation[#, "Ruthenium87", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1086],
        "\"Ruthenium88\""],
       Annotation[#, "Ruthenium88", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1087],
        "\"Ruthenium89\""],
       Annotation[#, "Ruthenium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1088],
        "\"Ruthenium90\""],
       Annotation[#, "Ruthenium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1089],
        "\"Ruthenium91\""],
       Annotation[#, "Ruthenium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1090],
        "\"Ruthenium92\""],
       Annotation[#, "Ruthenium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1091],
        "\"Ruthenium93\""],
       Annotation[#, "Ruthenium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1092],
        "\"Ruthenium94\""],
       Annotation[#, "Ruthenium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1093],
        "\"Ruthenium95\""],
       Annotation[#, "Ruthenium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1094],
        "\"Ruthenium96\""],
       Annotation[#, "Ruthenium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1095],
        "\"Ruthenium97\""],
       Annotation[#, "Ruthenium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1096],
        "\"Rhodium103\""],
       Annotation[#, "Rhodium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1097],
        "\"Palladium104\""],
       Annotation[#, "Palladium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1098],
        "\"Rhodium105\""],
       Annotation[#, "Rhodium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1099],
        "\"Rhodium106\""],
       Annotation[#, "Rhodium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1100],
        "\"Rhodium107\""],
       Annotation[#, "Rhodium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1101],
        "\"Rhodium108\""],
       Annotation[#, "Rhodium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1102],
        "\"Rhodium109\""],
       Annotation[#, "Rhodium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1103],
        "\"Rhodium110\""],
       Annotation[#, "Rhodium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1104],
        "\"Rhodium111\""],
       Annotation[#, "Rhodium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1105],
        "\"Rhodium112\""],
       Annotation[#, "Rhodium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1106],
        "\"Rhodium113\""],
       Annotation[#, "Rhodium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1107],
        "\"Rhodium114\""],
       Annotation[#, "Rhodium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1108],
        "\"Rhodium115\""],
       Annotation[#, "Rhodium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1109],
        "\"Rhodium116\""],
       Annotation[#, "Rhodium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1110],
        "\"Rhodium117\""],
       Annotation[#, "Rhodium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1111],
        "\"Rhodium118\""],
       Annotation[#, "Rhodium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1112],
        "\"Ruthenium119\""],
       Annotation[#, "Ruthenium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1113],
        "\"Rhodium119\""],
       Annotation[#, "Rhodium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1114],
        "\"Ruthenium120\""],
       Annotation[#, "Ruthenium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1115],
        "\"Rhodium120\""],
       Annotation[#, "Rhodium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1116],
        "\"Rhodium89\""],
       Annotation[#, "Rhodium89", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1117],
        "\"Rhodium90\""],
       Annotation[#, "Rhodium90", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1118],
        "\"Rhodium91\""],
       Annotation[#, "Rhodium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1119],
        "\"Rhodium92\""],
       Annotation[#, "Rhodium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1120],
        "\"Rhodium93\""],
       Annotation[#, "Rhodium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1121],
        "\"Rhodium94\""],
       Annotation[#, "Rhodium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1122],
        "\"Rhodium95\""],
       Annotation[#, "Rhodium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1123],
        "\"Rhodium96\""],
       Annotation[#, "Rhodium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1124],
        "\"Rhodium97\""],
       Annotation[#, "Rhodium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1125],
        "\"Rhodium98\""],
       Annotation[#, "Rhodium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1126],
        "\"Rhodium99\""],
       Annotation[#, "Rhodium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1127],
        "\"Rhodium100\""],
       Annotation[#, "Rhodium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1128],
        "\"Rhodium101\""],
       Annotation[#, "Rhodium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1129],
        "\"Rhodium102\""],
       Annotation[#, "Rhodium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1130],
        "\"Palladium102\""],
       Annotation[#, "Palladium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1131],
        "\"Rhodium104\""],
       Annotation[#, "Rhodium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1132],
        "\"Palladium105\""],
       Annotation[#, "Palladium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1133],
        "\"Palladium106\""],
       Annotation[#, "Palladium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1134],
        "\"Palladium107\""],
       Annotation[#, "Palladium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1135],
        "\"Palladium108\""],
       Annotation[#, "Palladium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1136],
        "\"Palladium109\""],
       Annotation[#, "Palladium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1137],
        "\"Palladium110\""],
       Annotation[#, "Palladium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1138],
        "\"Palladium111\""],
       Annotation[#, "Palladium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1139],
        "\"Palladium112\""],
       Annotation[#, "Palladium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1140],
        "\"Palladium113\""],
       Annotation[#, "Palladium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1141],
        "\"Palladium114\""],
       Annotation[#, "Palladium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1142],
        "\"Palladium115\""],
       Annotation[#, "Palladium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1143],
        "\"Palladium116\""],
       Annotation[#, "Palladium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1144],
        "\"Palladium117\""],
       Annotation[#, "Palladium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1145],
        "\"Palladium118\""],
       Annotation[#, "Palladium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1146],
        "\"Palladium119\""],
       Annotation[#, "Palladium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1147],
        "\"Palladium120\""],
       Annotation[#, "Palladium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1148],
        "\"Rhodium121\""],
       Annotation[#, "Rhodium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1149],
        "\"Palladium121\""],
       Annotation[#, "Palladium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1150],
        "\"Rhodium122\""],
       Annotation[#, "Rhodium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1151],
        "\"Palladium122\""],
       Annotation[#, "Palladium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1152],
        "\"Palladium91\""],
       Annotation[#, "Palladium91", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1153],
        "\"Palladium92\""],
       Annotation[#, "Palladium92", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1154],
        "\"Palladium93\""],
       Annotation[#, "Palladium93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1155],
        "\"Palladium94\""],
       Annotation[#, "Palladium94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1156],
        "\"Palladium95\""],
       Annotation[#, "Palladium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1157],
        "\"Palladium96\""],
       Annotation[#, "Palladium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1158],
        "\"Palladium97\""],
       Annotation[#, "Palladium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1159],
        "\"Palladium98\""],
       Annotation[#, "Palladium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1160],
        "\"Palladium99\""],
       Annotation[#, "Palladium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1161],
        "\"Palladium100\""],
       Annotation[#, "Palladium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1162],
        "\"Palladium101\""],
       Annotation[#, "Palladium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1163],
        "\"Palladium103\""],
       Annotation[#, "Palladium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1164],
        "\"Silver107\""],
       Annotation[#, "Silver107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1165],
        "\"Silver109\""],
       Annotation[#, "Silver109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1166],
        "\"Cadmium110\""],
       Annotation[#, "Cadmium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1167],
        "\"Silver111\""],
       Annotation[#, "Silver111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1168],
        "\"Silver112\""],
       Annotation[#, "Silver112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1169],
        "\"Silver113\""],
       Annotation[#, "Silver113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1170],
        "\"Silver114\""],
       Annotation[#, "Silver114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1171],
        "\"Silver115\""],
       Annotation[#, "Silver115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1172],
        "\"Silver116\""],
       Annotation[#, "Silver116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1173],
        "\"Silver117\""],
       Annotation[#, "Silver117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1174],
        "\"Silver118\""],
       Annotation[#, "Silver118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1175],
        "\"Silver119\""],
       Annotation[#, "Silver119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1176],
        "\"Silver120\""],
       Annotation[#, "Silver120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1177],
        "\"Silver121\""],
       Annotation[#, "Silver121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1178],
        "\"Silver122\""],
       Annotation[#, "Silver122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1179],
        "\"Palladium123\""],
       Annotation[#, "Palladium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1180],
        "\"Silver123\""],
       Annotation[#, "Silver123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1181],
        "\"Palladium124\""],
       Annotation[#, "Palladium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1182],
        "\"Silver124\""],
       Annotation[#, "Silver124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1183],
        "\"Silver93\""],
       Annotation[#, "Silver93", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1184],
        "\"Silver94\""],
       Annotation[#, "Silver94", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1185],
        "\"Silver95\""],
       Annotation[#, "Silver95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1186],
        "\"Silver96\""],
       Annotation[#, "Silver96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1187],
        "\"Silver97\""],
       Annotation[#, "Silver97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1188],
        "\"Silver98\""],
       Annotation[#, "Silver98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1189],
        "\"Silver99\""],
       Annotation[#, "Silver99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1190],
        "\"Silver100\""],
       Annotation[#, "Silver100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1191],
        "\"Silver101\""],
       Annotation[#, "Silver101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1192],
        "\"Silver102\""],
       Annotation[#, "Silver102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1193],
        "\"Silver103\""],
       Annotation[#, "Silver103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1194],
        "\"Silver104\""],
       Annotation[#, "Silver104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1195],
        "\"Silver105\""],
       Annotation[#, "Silver105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1196],
        "\"Silver106\""],
       Annotation[#, "Silver106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1197],
        "\"Cadmium106\""],
       Annotation[#, "Cadmium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1198],
        "\"Silver108\""],
       Annotation[#, "Silver108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1199],
        "\"Cadmium108\""],
       Annotation[#, "Cadmium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1200],
        "\"Silver110\""],
       Annotation[#, "Silver110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1201],
        "\"Cadmium111\""],
       Annotation[#, "Cadmium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1202],
        "\"Cadmium112\""],
       Annotation[#, "Cadmium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1203],
        "\"Cadmium113\""],
       Annotation[#, "Cadmium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1204],
        "\"Cadmium114\""],
       Annotation[#, "Cadmium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1205],
        "\"Cadmium115\""],
       Annotation[#, "Cadmium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1206],
        "\"Cadmium116\""],
       Annotation[#, "Cadmium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1207],
        "\"Cadmium117\""],
       Annotation[#, "Cadmium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1208],
        "\"Cadmium118\""],
       Annotation[#, "Cadmium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1209],
        "\"Cadmium119\""],
       Annotation[#, "Cadmium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1210],
        "\"Cadmium120\""],
       Annotation[#, "Cadmium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1211],
        "\"Cadmium121\""],
       Annotation[#, "Cadmium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1212],
        "\"Cadmium122\""],
       Annotation[#, "Cadmium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1213],
        "\"Cadmium123\""],
       Annotation[#, "Cadmium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1214],
        "\"Cadmium124\""],
       Annotation[#, "Cadmium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1215],
        "\"Silver125\""],
       Annotation[#, "Silver125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1216],
        "\"Cadmium125\""],
       Annotation[#, "Cadmium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1217],
        "\"Silver126\""],
       Annotation[#, "Silver126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1218],
        "\"Cadmium126\""],
       Annotation[#, "Cadmium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1219],
        "\"Silver127\""],
       Annotation[#, "Silver127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1220],
        "\"Cadmium127\""],
       Annotation[#, "Cadmium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1221],
        "\"Silver128\""],
       Annotation[#, "Silver128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1222],
        "\"Cadmium128\""],
       Annotation[#, "Cadmium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1223],
        "\"Silver129\""],
       Annotation[#, "Silver129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1224],
        "\"Cadmium129\""],
       Annotation[#, "Cadmium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1225],
        "\"Silver130\""],
       Annotation[#, "Silver130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1226],
        "\"Cadmium130\""],
       Annotation[#, "Cadmium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1227],
        "\"Cadmium95\""],
       Annotation[#, "Cadmium95", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1228],
        "\"Cadmium96\""],
       Annotation[#, "Cadmium96", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1229],
        "\"Cadmium97\""],
       Annotation[#, "Cadmium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1230],
        "\"Cadmium98\""],
       Annotation[#, "Cadmium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1231],
        "\"Cadmium99\""],
       Annotation[#, "Cadmium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1232],
        "\"Cadmium100\""],
       Annotation[#, "Cadmium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1233],
        "\"Cadmium101\""],
       Annotation[#, "Cadmium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1234],
        "\"Cadmium102\""],
       Annotation[#, "Cadmium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1235],
        "\"Cadmium103\""],
       Annotation[#, "Cadmium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1236],
        "\"Cadmium104\""],
       Annotation[#, "Cadmium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1237],
        "\"Cadmium105\""],
       Annotation[#, "Cadmium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1238],
        "\"Cadmium107\""],
       Annotation[#, "Cadmium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1239],
        "\"Cadmium109\""],
       Annotation[#, "Cadmium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1240],
        "\"Indium113\""],
       Annotation[#, "Indium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1241],
        "\"Tin114\""],
       Annotation[#, "Tin114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1242],
        "\"Indium115\""],
       Annotation[#, "Indium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1243],
        "\"Tin116\""],
       Annotation[#, "Tin116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1244],
        "\"Indium117\""],
       Annotation[#, "Indium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1245],
        "\"Indium118\""],
       Annotation[#, "Indium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1246],
        "\"Indium119\""],
       Annotation[#, "Indium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1247],
        "\"Indium120\""],
       Annotation[#, "Indium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1248],
        "\"Indium121\""],
       Annotation[#, "Indium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1249],
        "\"Indium122\""],
       Annotation[#, "Indium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1250],
        "\"Indium123\""],
       Annotation[#, "Indium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1251],
        "\"Indium124\""],
       Annotation[#, "Indium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1252],
        "\"Indium125\""],
       Annotation[#, "Indium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1253],
        "\"Indium126\""],
       Annotation[#, "Indium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1254],
        "\"Indium127\""],
       Annotation[#, "Indium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1255],
        "\"Indium128\""],
       Annotation[#, "Indium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1256],
        "\"Indium129\""],
       Annotation[#, "Indium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1257],
        "\"Indium130\""],
       Annotation[#, "Indium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1258],
        "\"Cadmium131\""],
       Annotation[#, "Cadmium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1259],
        "\"Indium131\""],
       Annotation[#, "Indium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1260],
        "\"Cadmium132\""],
       Annotation[#, "Cadmium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1261],
        "\"Indium132\""],
       Annotation[#, "Indium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1262],
        "\"Indium97\""],
       Annotation[#, "Indium97", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1263],
        "\"Indium98\""],
       Annotation[#, "Indium98", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1264],
        "\"Indium99\""],
       Annotation[#, "Indium99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1265],
        "\"Indium100\""],
       Annotation[#, "Indium100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1266],
        "\"Indium101\""],
       Annotation[#, "Indium101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1267],
        "\"Indium102\""],
       Annotation[#, "Indium102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1268],
        "\"Indium103\""],
       Annotation[#, "Indium103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1269],
        "\"Indium104\""],
       Annotation[#, "Indium104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1270],
        "\"Indium105\""],
       Annotation[#, "Indium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1271],
        "\"Indium106\""],
       Annotation[#, "Indium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1272],
        "\"Indium107\""],
       Annotation[#, "Indium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1273],
        "\"Indium108\""],
       Annotation[#, "Indium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1274],
        "\"Indium109\""],
       Annotation[#, "Indium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1275],
        "\"Indium110\""],
       Annotation[#, "Indium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1276],
        "\"Indium111\""],
       Annotation[#, "Indium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1277],
        "\"Indium112\""],
       Annotation[#, "Indium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1278],
        "\"Tin112\""],
       Annotation[#, "Tin112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1279],
        "\"Indium114\""],
       Annotation[#, "Indium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1280],
        "\"Tin115\""],
       Annotation[#, "Tin115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1281],
        "\"Indium116\""],
       Annotation[#, "Indium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1282],
        "\"Tin117\""],
       Annotation[#, "Tin117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1283],
        "\"Tin118\""],
       Annotation[#, "Tin118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1284],
        "\"Tin119\""],
       Annotation[#, "Tin119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1285],
        "\"Tin120\""],
       Annotation[#, "Tin120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1286],
        "\"Tin121\""],
       Annotation[#, "Tin121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1287],
        "\"Tin122\""],
       Annotation[#, "Tin122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1288],
        "\"Tin123\""],
       Annotation[#, "Tin123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1289],
        "\"Tin124\""],
       Annotation[#, "Tin124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1290],
        "\"Tin125\""],
       Annotation[#, "Tin125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1291],
        "\"Tin126\""],
       Annotation[#, "Tin126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1292],
        "\"Tin127\""],
       Annotation[#, "Tin127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1293],
        "\"Tin128\""],
       Annotation[#, "Tin128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1294],
        "\"Tin129\""],
       Annotation[#, "Tin129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1295],
        "\"Tin130\""],
       Annotation[#, "Tin130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1296],
        "\"Tin131\""],
       Annotation[#, "Tin131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1297],
        "\"Tin132\""],
       Annotation[#, "Tin132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1298],
        "\"Indium133\""],
       Annotation[#, "Indium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1299],
        "\"Indium134\""],
       Annotation[#, "Indium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1300],
        "\"Tin133\""],
       Annotation[#, "Tin133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1301],
        "\"Tin134\""],
       Annotation[#, "Tin134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1302],
        "\"Indium135\""],
       Annotation[#, "Indium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1303],
        "\"Tin135\""],
       Annotation[#, "Tin135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1304],
        "\"Tin99\""],
       Annotation[#, "Tin99", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1305],
        "\"Tin100\""],
       Annotation[#, "Tin100", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1306],
        "\"Tin101\""],
       Annotation[#, "Tin101", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1307],
        "\"Tin102\""],
       Annotation[#, "Tin102", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1308],
        "\"Tin103\""],
       Annotation[#, "Tin103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1309],
        "\"Tin104\""],
       Annotation[#, "Tin104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1310],
        "\"Tin105\""],
       Annotation[#, "Tin105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1311],
        "\"Tin106\""],
       Annotation[#, "Tin106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1312],
        "\"Tin107\""],
       Annotation[#, "Tin107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1313],
        "\"Tin108\""],
       Annotation[#, "Tin108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1314],
        "\"Tin109\""],
       Annotation[#, "Tin109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1315],
        "\"Tin110\""],
       Annotation[#, "Tin110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1316],
        "\"Tin111\""],
       Annotation[#, "Tin111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1317],
        "\"Tin113\""],
       Annotation[#, "Tin113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1318],
        "\"Antimony121\""],
       Annotation[#, "Antimony121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1319],
        "\"Tellurium122\""],
       Annotation[#, "Tellurium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1320],
        "\"Antimony123\""],
       Annotation[#, "Antimony123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1321],
        "\"Tellurium124\""],
       Annotation[#, "Tellurium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1322],
        "\"Antimony125\""],
       Annotation[#, "Antimony125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1323],
        "\"Antimony126\""],
       Annotation[#, "Antimony126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1324],
        "\"Antimony127\""],
       Annotation[#, "Antimony127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1325],
        "\"Antimony128\""],
       Annotation[#, "Antimony128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1326],
        "\"Antimony129\""],
       Annotation[#, "Antimony129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1327],
        "\"Antimony130\""],
       Annotation[#, "Antimony130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1328],
        "\"Antimony131\""],
       Annotation[#, "Antimony131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1329],
        "\"Antimony132\""],
       Annotation[#, "Antimony132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1330],
        "\"Antimony133\""],
       Annotation[#, "Antimony133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1331],
        "\"Antimony134\""],
       Annotation[#, "Antimony134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1332],
        "\"Antimony135\""],
       Annotation[#, "Antimony135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1333],
        "\"Tin136\""],
       Annotation[#, "Tin136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1334],
        "\"Antimony136\""],
       Annotation[#, "Antimony136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1335],
        "\"Tin137\""],
       Annotation[#, "Tin137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1336],
        "\"Antimony137\""],
       Annotation[#, "Antimony137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1337],
        "\"Antimony103\""],
       Annotation[#, "Antimony103", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1338],
        "\"Antimony104\""],
       Annotation[#, "Antimony104", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1339],
        "\"Antimony105\""],
       Annotation[#, "Antimony105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1340],
        "\"Antimony106\""],
       Annotation[#, "Antimony106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1341],
        "\"Antimony107\""],
       Annotation[#, "Antimony107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1342],
        "\"Antimony108\""],
       Annotation[#, "Antimony108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1343],
        "\"Antimony109\""],
       Annotation[#, "Antimony109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1344],
        "\"Antimony110\""],
       Annotation[#, "Antimony110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1345],
        "\"Antimony111\""],
       Annotation[#, "Antimony111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1346],
        "\"Antimony112\""],
       Annotation[#, "Antimony112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1347],
        "\"Antimony113\""],
       Annotation[#, "Antimony113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1348],
        "\"Antimony114\""],
       Annotation[#, "Antimony114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1349],
        "\"Antimony115\""],
       Annotation[#, "Antimony115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1350],
        "\"Antimony116\""],
       Annotation[#, "Antimony116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1351],
        "\"Antimony117\""],
       Annotation[#, "Antimony117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1352],
        "\"Antimony118\""],
       Annotation[#, "Antimony118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1353],
        "\"Antimony119\""],
       Annotation[#, "Antimony119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1354],
        "\"Antimony120\""],
       Annotation[#, "Antimony120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1355],
        "\"Antimony122\""],
       Annotation[#, "Antimony122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1356],
        "\"Antimony124\""],
       Annotation[#, "Antimony124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1357],
        "\"Tellurium125\""],
       Annotation[#, "Tellurium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1358],
        "\"Tellurium126\""],
       Annotation[#, "Tellurium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1359],
        "\"Tellurium127\""],
       Annotation[#, "Tellurium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1360],
        "\"Tellurium128\""],
       Annotation[#, "Tellurium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1361],
        "\"Tellurium129\""],
       Annotation[#, "Tellurium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1362],
        "\"Tellurium130\""],
       Annotation[#, "Tellurium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1363],
        "\"Tellurium131\""],
       Annotation[#, "Tellurium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1364],
        "\"Tellurium132\""],
       Annotation[#, "Tellurium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1365],
        "\"Tellurium133\""],
       Annotation[#, "Tellurium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1366],
        "\"Tellurium134\""],
       Annotation[#, "Tellurium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1367],
        "\"Tellurium135\""],
       Annotation[#, "Tellurium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1368],
        "\"Tellurium136\""],
       Annotation[#, "Tellurium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1369],
        "\"Tellurium137\""],
       Annotation[#, "Tellurium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1370],
        "\"Antimony138\""],
       Annotation[#, "Antimony138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1371],
        "\"Tellurium138\""],
       Annotation[#, "Tellurium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1372],
        "\"Antimony139\""],
       Annotation[#, "Antimony139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1373],
        "\"Tellurium139\""],
       Annotation[#, "Tellurium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1374],
        "\"Tellurium105\""],
       Annotation[#, "Tellurium105", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1375],
        "\"Tellurium106\""],
       Annotation[#, "Tellurium106", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1376],
        "\"Tellurium107\""],
       Annotation[#, "Tellurium107", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1377],
        "\"Tellurium108\""],
       Annotation[#, "Tellurium108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1378],
        "\"Tellurium109\""],
       Annotation[#, "Tellurium109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1379],
        "\"Tellurium110\""],
       Annotation[#, "Tellurium110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1380],
        "\"Tellurium111\""],
       Annotation[#, "Tellurium111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1381],
        "\"Tellurium112\""],
       Annotation[#, "Tellurium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1382],
        "\"Tellurium113\""],
       Annotation[#, "Tellurium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1383],
        "\"Tellurium114\""],
       Annotation[#, "Tellurium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1384],
        "\"Tellurium115\""],
       Annotation[#, "Tellurium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1385],
        "\"Tellurium116\""],
       Annotation[#, "Tellurium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1386],
        "\"Tellurium117\""],
       Annotation[#, "Tellurium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1387],
        "\"Tellurium118\""],
       Annotation[#, "Tellurium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1388],
        "\"Tellurium119\""],
       Annotation[#, "Tellurium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1389],
        "\"Tellurium120\""],
       Annotation[#, "Tellurium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1390],
        "\"Tellurium121\""],
       Annotation[#, "Tellurium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1391],
        "\"Tellurium123\""],
       Annotation[#, "Tellurium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1392],
        "\"Iodine127\""],
       Annotation[#, "Iodine127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1393],
        "\"Xenon128\""],
       Annotation[#, "Xenon128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1394],
        "\"Iodine129\""],
       Annotation[#, "Iodine129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1395],
        "\"Xenon130\""],
       Annotation[#, "Xenon130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1396],
        "\"Iodine131\""],
       Annotation[#, "Iodine131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1397],
        "\"Iodine132\""],
       Annotation[#, "Iodine132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1398],
        "\"Iodine133\""],
       Annotation[#, "Iodine133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1399],
        "\"Iodine134\""],
       Annotation[#, "Iodine134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1400],
        "\"Iodine135\""],
       Annotation[#, "Iodine135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1401],
        "\"Iodine136\""],
       Annotation[#, "Iodine136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1402],
        "\"Iodine137\""],
       Annotation[#, "Iodine137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1403],
        "\"Iodine138\""],
       Annotation[#, "Iodine138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1404],
        "\"Iodine139\""],
       Annotation[#, "Iodine139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1405],
        "\"Tellurium140\""],
       Annotation[#, "Tellurium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1406],
        "\"Iodine140\""],
       Annotation[#, "Iodine140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1407],
        "\"Tellurium141\""],
       Annotation[#, "Tellurium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1408],
        "\"Iodine141\""],
       Annotation[#, "Iodine141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1409],
        "\"Tellurium142\""],
       Annotation[#, "Tellurium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1410],
        "\"Iodine142\""],
       Annotation[#, "Iodine142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1411],
        "\"Iodine108\""],
       Annotation[#, "Iodine108", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1412],
        "\"Iodine109\""],
       Annotation[#, "Iodine109", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1413],
        "\"Iodine110\""],
       Annotation[#, "Iodine110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1414],
        "\"Iodine111\""],
       Annotation[#, "Iodine111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1415],
        "\"Iodine112\""],
       Annotation[#, "Iodine112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1416],
        "\"Iodine113\""],
       Annotation[#, "Iodine113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1417],
        "\"Iodine114\""],
       Annotation[#, "Iodine114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1418],
        "\"Iodine115\""],
       Annotation[#, "Iodine115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1419],
        "\"Iodine116\""],
       Annotation[#, "Iodine116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1420],
        "\"Iodine117\""],
       Annotation[#, "Iodine117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1421],
        "\"Iodine118\""],
       Annotation[#, "Iodine118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1422],
        "\"Iodine119\""],
       Annotation[#, "Iodine119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1423],
        "\"Iodine120\""],
       Annotation[#, "Iodine120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1424],
        "\"Iodine121\""],
       Annotation[#, "Iodine121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1425],
        "\"Iodine122\""],
       Annotation[#, "Iodine122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1426],
        "\"Iodine123\""],
       Annotation[#, "Iodine123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1427],
        "\"Iodine124\""],
       Annotation[#, "Iodine124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1428],
        "\"Iodine125\""],
       Annotation[#, "Iodine125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1429],
        "\"Iodine126\""],
       Annotation[#, "Iodine126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1430],
        "\"Xenon126\""],
       Annotation[#, "Xenon126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1431],
        "\"Iodine128\""],
       Annotation[#, "Iodine128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1432],
        "\"Xenon129\""],
       Annotation[#, "Xenon129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1433],
        "\"Iodine130\""],
       Annotation[#, "Iodine130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1434],
        "\"Xenon131\""],
       Annotation[#, "Xenon131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1435],
        "\"Xenon132\""],
       Annotation[#, "Xenon132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1436],
        "\"Xenon133\""],
       Annotation[#, "Xenon133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1437],
        "\"Xenon134\""],
       Annotation[#, "Xenon134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1438],
        "\"Xenon135\""],
       Annotation[#, "Xenon135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1439],
        "\"Xenon136\""],
       Annotation[#, "Xenon136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1440],
        "\"Xenon137\""],
       Annotation[#, "Xenon137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1441],
        "\"Xenon138\""],
       Annotation[#, "Xenon138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1442],
        "\"Xenon139\""],
       Annotation[#, "Xenon139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1443],
        "\"Xenon140\""],
       Annotation[#, "Xenon140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1444],
        "\"Xenon141\""],
       Annotation[#, "Xenon141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1445],
        "\"Xenon142\""],
       Annotation[#, "Xenon142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1446],
        "\"Iodine143\""],
       Annotation[#, "Iodine143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1447],
        "\"Iodine144\""],
       Annotation[#, "Iodine144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1448],
        "\"Xenon143\""],
       Annotation[#, "Xenon143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1449],
        "\"Xenon110\""],
       Annotation[#, "Xenon110", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1450],
        "\"Xenon111\""],
       Annotation[#, "Xenon111", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1451],
        "\"Xenon112\""],
       Annotation[#, "Xenon112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1452],
        "\"Xenon113\""],
       Annotation[#, "Xenon113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1453],
        "\"Xenon114\""],
       Annotation[#, "Xenon114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1454],
        "\"Xenon115\""],
       Annotation[#, "Xenon115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1455],
        "\"Xenon116\""],
       Annotation[#, "Xenon116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1456],
        "\"Xenon117\""],
       Annotation[#, "Xenon117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1457],
        "\"Xenon118\""],
       Annotation[#, "Xenon118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1458],
        "\"Xenon119\""],
       Annotation[#, "Xenon119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1459],
        "\"Xenon120\""],
       Annotation[#, "Xenon120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1460],
        "\"Xenon121\""],
       Annotation[#, "Xenon121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1461],
        "\"Xenon122\""],
       Annotation[#, "Xenon122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1462],
        "\"Xenon123\""],
       Annotation[#, "Xenon123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1463],
        "\"Xenon124\""],
       Annotation[#, "Xenon124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1464],
        "\"Xenon125\""],
       Annotation[#, "Xenon125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1465],
        "\"Xenon127\""],
       Annotation[#, "Xenon127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1466],
        "\"Cesium133\""],
       Annotation[#, "Cesium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1467],
        "\"Barium134\""],
       Annotation[#, "Barium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1468],
        "\"Cesium135\""],
       Annotation[#, "Cesium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1469],
        "\"Barium136\""],
       Annotation[#, "Barium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1470],
        "\"Cesium137\""],
       Annotation[#, "Cesium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1471],
        "\"Cesium138\""],
       Annotation[#, "Cesium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1472],
        "\"Cesium139\""],
       Annotation[#, "Cesium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1473],
        "\"Cesium140\""],
       Annotation[#, "Cesium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1474],
        "\"Cesium141\""],
       Annotation[#, "Cesium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1475],
        "\"Cesium142\""],
       Annotation[#, "Cesium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1476],
        "\"Cesium143\""],
       Annotation[#, "Cesium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1477],
        "\"Xenon144\""],
       Annotation[#, "Xenon144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1478],
        "\"Cesium144\""],
       Annotation[#, "Cesium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1479],
        "\"Xenon145\""],
       Annotation[#, "Xenon145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1480],
        "\"Cesium145\""],
       Annotation[#, "Cesium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1481],
        "\"Xenon146\""],
       Annotation[#, "Xenon146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1482],
        "\"Cesium146\""],
       Annotation[#, "Cesium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1483],
        "\"Xenon147\""],
       Annotation[#, "Xenon147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1484],
        "\"Cesium147\""],
       Annotation[#, "Cesium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1485],
        "\"Cesium112\""],
       Annotation[#, "Cesium112", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1486],
        "\"Cesium113\""],
       Annotation[#, "Cesium113", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1487],
        "\"Cesium114\""],
       Annotation[#, "Cesium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1488],
        "\"Cesium115\""],
       Annotation[#, "Cesium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1489],
        "\"Cesium116\""],
       Annotation[#, "Cesium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1490],
        "\"Cesium117\""],
       Annotation[#, "Cesium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1491],
        "\"Cesium118\""],
       Annotation[#, "Cesium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1492],
        "\"Cesium119\""],
       Annotation[#, "Cesium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1493],
        "\"Cesium120\""],
       Annotation[#, "Cesium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1494],
        "\"Cesium121\""],
       Annotation[#, "Cesium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1495],
        "\"Cesium122\""],
       Annotation[#, "Cesium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1496],
        "\"Cesium123\""],
       Annotation[#, "Cesium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1497],
        "\"Cesium124\""],
       Annotation[#, "Cesium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1498],
        "\"Cesium125\""],
       Annotation[#, "Cesium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1499],
        "\"Cesium126\""],
       Annotation[#, "Cesium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1500],
        "\"Cesium127\""],
       Annotation[#, "Cesium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1501],
        "\"Cesium128\""],
       Annotation[#, "Cesium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1502],
        "\"Cesium129\""],
       Annotation[#, "Cesium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1503],
        "\"Cesium130\""],
       Annotation[#, "Cesium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1504],
        "\"Barium130\""],
       Annotation[#, "Barium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1505],
        "\"Cesium131\""],
       Annotation[#, "Cesium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1506],
        "\"Cesium132\""],
       Annotation[#, "Cesium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1507],
        "\"Barium132\""],
       Annotation[#, "Barium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1508],
        "\"Cesium134\""],
       Annotation[#, "Cesium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1509],
        "\"Barium135\""],
       Annotation[#, "Barium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1510],
        "\"Cesium136\""],
       Annotation[#, "Cesium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1511],
        "\"Barium137\""],
       Annotation[#, "Barium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1512],
        "\"Barium138\""],
       Annotation[#, "Barium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1513],
        "\"Barium139\""],
       Annotation[#, "Barium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1514],
        "\"Barium140\""],
       Annotation[#, "Barium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1515],
        "\"Barium141\""],
       Annotation[#, "Barium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1516],
        "\"Barium142\""],
       Annotation[#, "Barium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1517],
        "\"Barium143\""],
       Annotation[#, "Barium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1518],
        "\"Barium144\""],
       Annotation[#, "Barium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1519],
        "\"Barium145\""],
       Annotation[#, "Barium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1520],
        "\"Barium146\""],
       Annotation[#, "Barium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1521],
        "\"Barium147\""],
       Annotation[#, "Barium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1522],
        "\"Cesium148\""],
       Annotation[#, "Cesium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1523],
        "\"Barium148\""],
       Annotation[#, "Barium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1524],
        "\"Cesium149\""],
       Annotation[#, "Cesium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1525],
        "\"Barium149\""],
       Annotation[#, "Barium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1526],
        "\"Cesium150\""],
       Annotation[#, "Cesium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1527],
        "\"Barium150\""],
       Annotation[#, "Barium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1528],
        "\"Cesium151\""],
       Annotation[#, "Cesium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1529],
        "\"Barium151\""],
       Annotation[#, "Barium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1530],
        "\"Barium114\""],
       Annotation[#, "Barium114", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1531],
        "\"Barium115\""],
       Annotation[#, "Barium115", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1532],
        "\"Barium116\""],
       Annotation[#, "Barium116", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1533],
        "\"Barium117\""],
       Annotation[#, "Barium117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1534],
        "\"Barium118\""],
       Annotation[#, "Barium118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1535],
        "\"Barium119\""],
       Annotation[#, "Barium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1536],
        "\"Barium120\""],
       Annotation[#, "Barium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1537],
        "\"Barium121\""],
       Annotation[#, "Barium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1538],
        "\"Barium122\""],
       Annotation[#, "Barium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1539],
        "\"Barium123\""],
       Annotation[#, "Barium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1540],
        "\"Barium124\""],
       Annotation[#, "Barium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1541],
        "\"Barium125\""],
       Annotation[#, "Barium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1542],
        "\"Barium126\""],
       Annotation[#, "Barium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1543],
        "\"Barium127\""],
       Annotation[#, "Barium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1544],
        "\"Barium128\""],
       Annotation[#, "Barium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1545],
        "\"Barium129\""],
       Annotation[#, "Barium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1546],
        "\"Barium131\""],
       Annotation[#, "Barium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1547],
        "\"Barium133\""],
       Annotation[#, "Barium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1548],
        "\"Lanthanum139\""],
       Annotation[#, "Lanthanum139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1549],
        "\"Lanthanum140\""],
       Annotation[#, "Lanthanum140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1550],
        "\"Lanthanum141\""],
       Annotation[#, "Lanthanum141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1551],
        "\"Lanthanum142\""],
       Annotation[#, "Lanthanum142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1552],
        "\"Lanthanum143\""],
       Annotation[#, "Lanthanum143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1553],
        "\"Lanthanum144\""],
       Annotation[#, "Lanthanum144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1554],
        "\"Lanthanum145\""],
       Annotation[#, "Lanthanum145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1555],
        "\"Lanthanum146\""],
       Annotation[#, "Lanthanum146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1556],
        "\"Lanthanum147\""],
       Annotation[#, "Lanthanum147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1557],
        "\"Lanthanum148\""],
       Annotation[#, "Lanthanum148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1558],
        "\"Lanthanum149\""],
       Annotation[#, "Lanthanum149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1559],
        "\"Lanthanum150\""],
       Annotation[#, "Lanthanum150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1560],
        "\"Lanthanum151\""],
       Annotation[#, "Lanthanum151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1561],
        "\"Barium152\""],
       Annotation[#, "Barium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1562],
        "\"Lanthanum152\""],
       Annotation[#, "Lanthanum152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1563],
        "\"Barium153\""],
       Annotation[#, "Barium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1564],
        "\"Lanthanum153\""],
       Annotation[#, "Lanthanum153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1565],
        "\"Lanthanum117\""],
       Annotation[#, "Lanthanum117", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1566],
        "\"Lanthanum118\""],
       Annotation[#, "Lanthanum118", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1567],
        "\"Lanthanum119\""],
       Annotation[#, "Lanthanum119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1568],
        "\"Lanthanum120\""],
       Annotation[#, "Lanthanum120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1569],
        "\"Lanthanum121\""],
       Annotation[#, "Lanthanum121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1570],
        "\"Lanthanum122\""],
       Annotation[#, "Lanthanum122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1571],
        "\"Lanthanum123\""],
       Annotation[#, "Lanthanum123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1572],
        "\"Lanthanum124\""],
       Annotation[#, "Lanthanum124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1573],
        "\"Lanthanum125\""],
       Annotation[#, "Lanthanum125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1574],
        "\"Lanthanum126\""],
       Annotation[#, "Lanthanum126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1575],
        "\"Lanthanum127\""],
       Annotation[#, "Lanthanum127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1576],
        "\"Lanthanum128\""],
       Annotation[#, "Lanthanum128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1577],
        "\"Lanthanum129\""],
       Annotation[#, "Lanthanum129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1578],
        "\"Lanthanum130\""],
       Annotation[#, "Lanthanum130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1579],
        "\"Lanthanum131\""],
       Annotation[#, "Lanthanum131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1580],
        "\"Lanthanum132\""],
       Annotation[#, "Lanthanum132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1581],
        "\"Lanthanum133\""],
       Annotation[#, "Lanthanum133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1582],
        "\"Lanthanum134\""],
       Annotation[#, "Lanthanum134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1583],
        "\"Lanthanum135\""],
       Annotation[#, "Lanthanum135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1584],
        "\"Lanthanum136\""],
       Annotation[#, "Lanthanum136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1585],
        "\"Lanthanum137\""],
       Annotation[#, "Lanthanum137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1586],
        "\"Lanthanum138\""],
       Annotation[#, "Lanthanum138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1587],
        "\"Cerium138\""],
       Annotation[#, "Cerium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1588],
        "\"Cerium140\""],
       Annotation[#, "Cerium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1589],
        "\"Cerium141\""],
       Annotation[#, "Cerium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1590],
        "\"Cerium142\""],
       Annotation[#, "Cerium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1591],
        "\"Cerium143\""],
       Annotation[#, "Cerium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1592],
        "\"Cerium144\""],
       Annotation[#, "Cerium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1593],
        "\"Cerium145\""],
       Annotation[#, "Cerium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1594],
        "\"Cerium146\""],
       Annotation[#, "Cerium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1595],
        "\"Cerium147\""],
       Annotation[#, "Cerium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1596],
        "\"Cerium148\""],
       Annotation[#, "Cerium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1597],
        "\"Cerium149\""],
       Annotation[#, "Cerium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1598],
        "\"Cerium150\""],
       Annotation[#, "Cerium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1599],
        "\"Cerium151\""],
       Annotation[#, "Cerium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1600],
        "\"Cerium152\""],
       Annotation[#, "Cerium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1601],
        "\"Cerium153\""],
       Annotation[#, "Cerium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1602],
        "\"Lanthanum154\""],
       Annotation[#, "Lanthanum154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1603],
        "\"Cerium154\""],
       Annotation[#, "Cerium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1604],
        "\"Lanthanum155\""],
       Annotation[#, "Lanthanum155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1605],
        "\"Cerium155\""],
       Annotation[#, "Cerium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1606],
        "\"Cerium119\""],
       Annotation[#, "Cerium119", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1607],
        "\"Cerium120\""],
       Annotation[#, "Cerium120", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1608],
        "\"Cerium121\""],
       Annotation[#, "Cerium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1609],
        "\"Cerium122\""],
       Annotation[#, "Cerium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1610],
        "\"Cerium123\""],
       Annotation[#, "Cerium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1611],
        "\"Cerium124\""],
       Annotation[#, "Cerium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1612],
        "\"Cerium125\""],
       Annotation[#, "Cerium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1613],
        "\"Cerium126\""],
       Annotation[#, "Cerium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1614],
        "\"Cerium127\""],
       Annotation[#, "Cerium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1615],
        "\"Cerium128\""],
       Annotation[#, "Cerium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1616],
        "\"Cerium129\""],
       Annotation[#, "Cerium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1617],
        "\"Cerium130\""],
       Annotation[#, "Cerium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1618],
        "\"Cerium131\""],
       Annotation[#, "Cerium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1619],
        "\"Cerium132\""],
       Annotation[#, "Cerium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1620],
        "\"Cerium133\""],
       Annotation[#, "Cerium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1621],
        "\"Cerium134\""],
       Annotation[#, "Cerium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1622],
        "\"Cerium135\""],
       Annotation[#, "Cerium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1623],
        "\"Cerium136\""],
       Annotation[#, "Cerium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1624],
        "\"Cerium137\""],
       Annotation[#, "Cerium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1625],
        "\"Cerium139\""],
       Annotation[#, "Cerium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1626],
        "\"Praseodymium141\""],
       Annotation[#, "Praseodymium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1627],
        "\"Neodymium142\""],
       Annotation[#, "Neodymium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1628],
        "\"Praseodymium143\""],
       Annotation[#, "Praseodymium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1629],
        "\"Praseodymium144\""],
       Annotation[#, "Praseodymium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1630],
        "\"Praseodymium145\""],
       Annotation[#, "Praseodymium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1631],
        "\"Praseodymium146\""],
       Annotation[#, "Praseodymium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1632],
        "\"Praseodymium147\""],
       Annotation[#, "Praseodymium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1633],
        "\"Praseodymium148\""],
       Annotation[#, "Praseodymium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1634],
        "\"Praseodymium149\""],
       Annotation[#, "Praseodymium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1635],
        "\"Praseodymium150\""],
       Annotation[#, "Praseodymium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1636],
        "\"Praseodymium151\""],
       Annotation[#, "Praseodymium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1637],
        "\"Praseodymium152\""],
       Annotation[#, "Praseodymium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1638],
        "\"Praseodymium153\""],
       Annotation[#, "Praseodymium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1639],
        "\"Praseodymium154\""],
       Annotation[#, "Praseodymium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1640],
        "\"Praseodymium155\""],
       Annotation[#, "Praseodymium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1641],
        "\"Cerium156\""],
       Annotation[#, "Cerium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1642],
        "\"Praseodymium156\""],
       Annotation[#, "Praseodymium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1643],
        "\"Cerium157\""],
       Annotation[#, "Cerium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1644],
        "\"Praseodymium157\""],
       Annotation[#, "Praseodymium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1645],
        "\"Praseodymium121\""],
       Annotation[#, "Praseodymium121", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1646],
        "\"Praseodymium122\""],
       Annotation[#, "Praseodymium122", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1647],
        "\"Praseodymium123\""],
       Annotation[#, "Praseodymium123", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1648],
        "\"Praseodymium124\""],
       Annotation[#, "Praseodymium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1649],
        "\"Praseodymium125\""],
       Annotation[#, "Praseodymium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1650],
        "\"Praseodymium126\""],
       Annotation[#, "Praseodymium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1651],
        "\"Praseodymium127\""],
       Annotation[#, "Praseodymium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1652],
        "\"Praseodymium128\""],
       Annotation[#, "Praseodymium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1653],
        "\"Praseodymium129\""],
       Annotation[#, "Praseodymium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1654],
        "\"Praseodymium130\""],
       Annotation[#, "Praseodymium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1655],
        "\"Praseodymium131\""],
       Annotation[#, "Praseodymium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1656],
        "\"Praseodymium132\""],
       Annotation[#, "Praseodymium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1657],
        "\"Praseodymium133\""],
       Annotation[#, "Praseodymium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1658],
        "\"Praseodymium134\""],
       Annotation[#, "Praseodymium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1659],
        "\"Praseodymium135\""],
       Annotation[#, "Praseodymium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1660],
        "\"Praseodymium136\""],
       Annotation[#, "Praseodymium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1661],
        "\"Praseodymium137\""],
       Annotation[#, "Praseodymium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1662],
        "\"Praseodymium138\""],
       Annotation[#, "Praseodymium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1663],
        "\"Praseodymium139\""],
       Annotation[#, "Praseodymium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1664],
        "\"Praseodymium140\""],
       Annotation[#, "Praseodymium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1665],
        "\"Praseodymium142\""],
       Annotation[#, "Praseodymium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1666],
        "\"Neodymium143\""],
       Annotation[#, "Neodymium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1667],
        "\"Neodymium144\""],
       Annotation[#, "Neodymium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1668],
        "\"Neodymium145\""],
       Annotation[#, "Neodymium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1669],
        "\"Neodymium146\""],
       Annotation[#, "Neodymium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1670],
        "\"Neodymium147\""],
       Annotation[#, "Neodymium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1671],
        "\"Neodymium148\""],
       Annotation[#, "Neodymium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1672],
        "\"Neodymium149\""],
       Annotation[#, "Neodymium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1673],
        "\"Neodymium150\""],
       Annotation[#, "Neodymium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1674],
        "\"Neodymium151\""],
       Annotation[#, "Neodymium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1675],
        "\"Neodymium152\""],
       Annotation[#, "Neodymium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1676],
        "\"Neodymium153\""],
       Annotation[#, "Neodymium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1677],
        "\"Neodymium154\""],
       Annotation[#, "Neodymium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1678],
        "\"Neodymium155\""],
       Annotation[#, "Neodymium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1679],
        "\"Neodymium156\""],
       Annotation[#, "Neodymium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1680],
        "\"Neodymium157\""],
       Annotation[#, "Neodymium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1681],
        "\"Praseodymium158\""],
       Annotation[#, "Praseodymium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1682],
        "\"Neodymium158\""],
       Annotation[#, "Neodymium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1683],
        "\"Praseodymium159\""],
       Annotation[#, "Praseodymium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1684],
        "\"Neodymium159\""],
       Annotation[#, "Neodymium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1685],
        "\"Neodymium124\""],
       Annotation[#, "Neodymium124", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1686],
        "\"Neodymium125\""],
       Annotation[#, "Neodymium125", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1687],
        "\"Neodymium126\""],
       Annotation[#, "Neodymium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1688],
        "\"Neodymium127\""],
       Annotation[#, "Neodymium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1689],
        "\"Neodymium128\""],
       Annotation[#, "Neodymium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1690],
        "\"Neodymium129\""],
       Annotation[#, "Neodymium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1691],
        "\"Neodymium130\""],
       Annotation[#, "Neodymium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1692],
        "\"Neodymium131\""],
       Annotation[#, "Neodymium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1693],
        "\"Neodymium132\""],
       Annotation[#, "Neodymium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1694],
        "\"Neodymium133\""],
       Annotation[#, "Neodymium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1695],
        "\"Neodymium134\""],
       Annotation[#, "Neodymium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1696],
        "\"Neodymium135\""],
       Annotation[#, "Neodymium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1697],
        "\"Neodymium136\""],
       Annotation[#, "Neodymium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1698],
        "\"Neodymium137\""],
       Annotation[#, "Neodymium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1699],
        "\"Neodymium138\""],
       Annotation[#, "Neodymium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1700],
        "\"Neodymium139\""],
       Annotation[#, "Neodymium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1701],
        "\"Neodymium140\""],
       Annotation[#, "Neodymium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1702],
        "\"Neodymium141\""],
       Annotation[#, "Neodymium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1703],
        "\"Samarium146\""],
       Annotation[#, "Samarium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1704],
        "\"Promethium147\""],
       Annotation[#, "Promethium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1705],
        "\"Samarium148\""],
       Annotation[#, "Samarium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1706],
        "\"Promethium149\""],
       Annotation[#, "Promethium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1707],
        "\"Samarium150\""],
       Annotation[#, "Samarium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1708],
        "\"Promethium151\""],
       Annotation[#, "Promethium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1709],
        "\"Promethium152\""],
       Annotation[#, "Promethium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1710],
        "\"Promethium153\""],
       Annotation[#, "Promethium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1711],
        "\"Promethium154\""],
       Annotation[#, "Promethium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1712],
        "\"Promethium155\""],
       Annotation[#, "Promethium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1713],
        "\"Promethium156\""],
       Annotation[#, "Promethium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1714],
        "\"Promethium157\""],
       Annotation[#, "Promethium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1715],
        "\"Promethium158\""],
       Annotation[#, "Promethium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1716],
        "\"Promethium159\""],
       Annotation[#, "Promethium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1717],
        "\"Neodymium160\""],
       Annotation[#, "Neodymium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1718],
        "\"Promethium160\""],
       Annotation[#, "Promethium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1719],
        "\"Neodymium161\""],
       Annotation[#, "Neodymium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1720],
        "\"Promethium161\""],
       Annotation[#, "Promethium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1721],
        "\"Promethium126\""],
       Annotation[#, "Promethium126", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1722],
        "\"Promethium127\""],
       Annotation[#, "Promethium127", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1723],
        "\"Promethium128\""],
       Annotation[#, "Promethium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1724],
        "\"Promethium129\""],
       Annotation[#, "Promethium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1725],
        "\"Promethium130\""],
       Annotation[#, "Promethium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1726],
        "\"Promethium131\""],
       Annotation[#, "Promethium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1727],
        "\"Promethium132\""],
       Annotation[#, "Promethium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1728],
        "\"Promethium133\""],
       Annotation[#, "Promethium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1729],
        "\"Promethium134\""],
       Annotation[#, "Promethium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1730],
        "\"Promethium135\""],
       Annotation[#, "Promethium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1731],
        "\"Promethium136\""],
       Annotation[#, "Promethium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1732],
        "\"Promethium137\""],
       Annotation[#, "Promethium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1733],
        "\"Promethium138\""],
       Annotation[#, "Promethium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1734],
        "\"Promethium139\""],
       Annotation[#, "Promethium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1735],
        "\"Promethium140\""],
       Annotation[#, "Promethium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1736],
        "\"Promethium141\""],
       Annotation[#, "Promethium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1737],
        "\"Promethium142\""],
       Annotation[#, "Promethium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1738],
        "\"Promethium143\""],
       Annotation[#, "Promethium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1739],
        "\"Promethium144\""],
       Annotation[#, "Promethium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1740],
        "\"Promethium145\""],
       Annotation[#, "Promethium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1741],
        "\"Promethium146\""],
       Annotation[#, "Promethium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1742],
        "\"Samarium147\""],
       Annotation[#, "Samarium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1743],
        "\"Promethium148\""],
       Annotation[#, "Promethium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1744],
        "\"Samarium149\""],
       Annotation[#, "Samarium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1745],
        "\"Promethium150\""],
       Annotation[#, "Promethium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1746],
        "\"Samarium151\""],
       Annotation[#, "Samarium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1747],
        "\"Samarium152\""],
       Annotation[#, "Samarium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1748],
        "\"Samarium153\""],
       Annotation[#, "Samarium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1749],
        "\"Samarium154\""],
       Annotation[#, "Samarium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1750],
        "\"Samarium155\""],
       Annotation[#, "Samarium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1751],
        "\"Samarium156\""],
       Annotation[#, "Samarium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1752],
        "\"Samarium157\""],
       Annotation[#, "Samarium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1753],
        "\"Samarium158\""],
       Annotation[#, "Samarium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1754],
        "\"Samarium159\""],
       Annotation[#, "Samarium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1755],
        "\"Samarium160\""],
       Annotation[#, "Samarium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1756],
        "\"Samarium161\""],
       Annotation[#, "Samarium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1757],
        "\"Promethium162\""],
       Annotation[#, "Promethium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1758],
        "\"Samarium162\""],
       Annotation[#, "Samarium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1759],
        "\"Promethium163\""],
       Annotation[#, "Promethium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1760],
        "\"Samarium163\""],
       Annotation[#, "Samarium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1761],
        "\"Samarium128\""],
       Annotation[#, "Samarium128", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1762],
        "\"Samarium129\""],
       Annotation[#, "Samarium129", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1763],
        "\"Samarium130\""],
       Annotation[#, "Samarium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1764],
        "\"Samarium131\""],
       Annotation[#, "Samarium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1765],
        "\"Samarium132\""],
       Annotation[#, "Samarium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1766],
        "\"Samarium133\""],
       Annotation[#, "Samarium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1767],
        "\"Samarium134\""],
       Annotation[#, "Samarium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1768],
        "\"Samarium135\""],
       Annotation[#, "Samarium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1769],
        "\"Samarium136\""],
       Annotation[#, "Samarium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1770],
        "\"Samarium137\""],
       Annotation[#, "Samarium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1771],
        "\"Samarium138\""],
       Annotation[#, "Samarium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1772],
        "\"Samarium139\""],
       Annotation[#, "Samarium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1773],
        "\"Samarium140\""],
       Annotation[#, "Samarium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1774],
        "\"Samarium141\""],
       Annotation[#, "Samarium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1775],
        "\"Samarium142\""],
       Annotation[#, "Samarium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1776],
        "\"Samarium143\""],
       Annotation[#, "Samarium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1777],
        "\"Samarium144\""],
       Annotation[#, "Samarium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1778],
        "\"Samarium145\""],
       Annotation[#, "Samarium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1779],
        "\"Europium151\""],
       Annotation[#, "Europium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1780],
        "\"Europium153\""],
       Annotation[#, "Europium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1781],
        "\"Gadolinium154\""],
       Annotation[#, "Gadolinium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1782],
        "\"Europium155\""],
       Annotation[#, "Europium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1783],
        "\"Europium156\""],
       Annotation[#, "Europium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1784],
        "\"Europium157\""],
       Annotation[#, "Europium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1785],
        "\"Europium158\""],
       Annotation[#, "Europium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1786],
        "\"Europium159\""],
       Annotation[#, "Europium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1787],
        "\"Europium160\""],
       Annotation[#, "Europium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1788],
        "\"Europium161\""],
       Annotation[#, "Europium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1789],
        "\"Europium162\""],
       Annotation[#, "Europium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1790],
        "\"Europium163\""],
       Annotation[#, "Europium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1791],
        "\"Samarium164\""],
       Annotation[#, "Samarium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1792],
        "\"Europium164\""],
       Annotation[#, "Europium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1793],
        "\"Samarium165\""],
       Annotation[#, "Samarium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1794],
        "\"Europium165\""],
       Annotation[#, "Europium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1795],
        "\"Europium130\""],
       Annotation[#, "Europium130", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1796],
        "\"Europium131\""],
       Annotation[#, "Europium131", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1797],
        "\"Europium132\""],
       Annotation[#, "Europium132", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1798],
        "\"Europium133\""],
       Annotation[#, "Europium133", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1799],
        "\"Europium134\""],
       Annotation[#, "Europium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1800],
        "\"Europium135\""],
       Annotation[#, "Europium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1801],
        "\"Europium136\""],
       Annotation[#, "Europium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1802],
        "\"Europium137\""],
       Annotation[#, "Europium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1803],
        "\"Europium138\""],
       Annotation[#, "Europium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1804],
        "\"Europium139\""],
       Annotation[#, "Europium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1805],
        "\"Europium140\""],
       Annotation[#, "Europium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1806],
        "\"Europium141\""],
       Annotation[#, "Europium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1807],
        "\"Europium142\""],
       Annotation[#, "Europium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1808],
        "\"Europium143\""],
       Annotation[#, "Europium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1809],
        "\"Europium144\""],
       Annotation[#, "Europium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1810],
        "\"Europium145\""],
       Annotation[#, "Europium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1811],
        "\"Europium146\""],
       Annotation[#, "Europium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1812],
        "\"Europium147\""],
       Annotation[#, "Europium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1813],
        "\"Europium148\""],
       Annotation[#, "Europium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1814],
        "\"Europium149\""],
       Annotation[#, "Europium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1815],
        "\"Europium150\""],
       Annotation[#, "Europium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1816],
        "\"Europium152\""],
       Annotation[#, "Europium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1817],
        "\"Gadolinium152\""],
       Annotation[#, "Gadolinium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1818],
        "\"Europium154\""],
       Annotation[#, "Europium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1819],
        "\"Gadolinium155\""],
       Annotation[#, "Gadolinium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1820],
        "\"Gadolinium156\""],
       Annotation[#, "Gadolinium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1821],
        "\"Gadolinium157\""],
       Annotation[#, "Gadolinium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1822],
        "\"Gadolinium158\""],
       Annotation[#, "Gadolinium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1823],
        "\"Gadolinium159\""],
       Annotation[#, "Gadolinium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1824],
        "\"Gadolinium160\""],
       Annotation[#, "Gadolinium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1825],
        "\"Gadolinium161\""],
       Annotation[#, "Gadolinium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1826],
        "\"Gadolinium162\""],
       Annotation[#, "Gadolinium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1827],
        "\"Gadolinium163\""],
       Annotation[#, "Gadolinium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1828],
        "\"Gadolinium164\""],
       Annotation[#, "Gadolinium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1829],
        "\"Gadolinium165\""],
       Annotation[#, "Gadolinium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1830],
        "\"Europium166\""],
       Annotation[#, "Europium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1831],
        "\"Gadolinium166\""],
       Annotation[#, "Gadolinium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1832],
        "\"Europium167\""],
       Annotation[#, "Europium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1833],
        "\"Gadolinium167\""],
       Annotation[#, "Gadolinium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1834],
        "\"Gadolinium134\""],
       Annotation[#, "Gadolinium134", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1835],
        "\"Gadolinium135\""],
       Annotation[#, "Gadolinium135", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1836],
        "\"Gadolinium136\""],
       Annotation[#, "Gadolinium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1837],
        "\"Gadolinium137\""],
       Annotation[#, "Gadolinium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1838],
        "\"Gadolinium138\""],
       Annotation[#, "Gadolinium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1839],
        "\"Gadolinium139\""],
       Annotation[#, "Gadolinium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1840],
        "\"Gadolinium140\""],
       Annotation[#, "Gadolinium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1841],
        "\"Gadolinium141\""],
       Annotation[#, "Gadolinium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1842],
        "\"Gadolinium142\""],
       Annotation[#, "Gadolinium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1843],
        "\"Gadolinium143\""],
       Annotation[#, "Gadolinium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1844],
        "\"Gadolinium144\""],
       Annotation[#, "Gadolinium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1845],
        "\"Gadolinium145\""],
       Annotation[#, "Gadolinium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1846],
        "\"Gadolinium146\""],
       Annotation[#, "Gadolinium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1847],
        "\"Gadolinium147\""],
       Annotation[#, "Gadolinium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1848],
        "\"Gadolinium148\""],
       Annotation[#, "Gadolinium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1849],
        "\"Gadolinium149\""],
       Annotation[#, "Gadolinium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1850],
        "\"Gadolinium150\""],
       Annotation[#, "Gadolinium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1851],
        "\"Gadolinium151\""],
       Annotation[#, "Gadolinium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1852],
        "\"Gadolinium153\""],
       Annotation[#, "Gadolinium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1853],
        "\"Terbium159\""],
       Annotation[#, "Terbium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1854],
        "\"Dysprosium160\""],
       Annotation[#, "Dysprosium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1855],
        "\"Terbium161\""],
       Annotation[#, "Terbium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1856],
        "\"Terbium162\""],
       Annotation[#, "Terbium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1857],
        "\"Terbium163\""],
       Annotation[#, "Terbium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1858],
        "\"Terbium164\""],
       Annotation[#, "Terbium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1859],
        "\"Terbium165\""],
       Annotation[#, "Terbium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1860],
        "\"Terbium166\""],
       Annotation[#, "Terbium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1861],
        "\"Terbium167\""],
       Annotation[#, "Terbium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1862],
        "\"Gadolinium168\""],
       Annotation[#, "Gadolinium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1863],
        "\"Terbium168\""],
       Annotation[#, "Terbium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1864],
        "\"Gadolinium169\""],
       Annotation[#, "Gadolinium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1865],
        "\"Terbium169\""],
       Annotation[#, "Terbium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1866],
        "\"Terbium136\""],
       Annotation[#, "Terbium136", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1867],
        "\"Terbium137\""],
       Annotation[#, "Terbium137", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1868],
        "\"Terbium138\""],
       Annotation[#, "Terbium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1869],
        "\"Terbium139\""],
       Annotation[#, "Terbium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1870],
        "\"Terbium140\""],
       Annotation[#, "Terbium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1871],
        "\"Terbium141\""],
       Annotation[#, "Terbium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1872],
        "\"Terbium142\""],
       Annotation[#, "Terbium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1873],
        "\"Terbium143\""],
       Annotation[#, "Terbium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1874],
        "\"Terbium144\""],
       Annotation[#, "Terbium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1875],
        "\"Terbium145\""],
       Annotation[#, "Terbium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1876],
        "\"Terbium146\""],
       Annotation[#, "Terbium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1877],
        "\"Terbium147\""],
       Annotation[#, "Terbium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1878],
        "\"Terbium148\""],
       Annotation[#, "Terbium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1879],
        "\"Terbium149\""],
       Annotation[#, "Terbium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1880],
        "\"Terbium150\""],
       Annotation[#, "Terbium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1881],
        "\"Terbium151\""],
       Annotation[#, "Terbium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1882],
        "\"Terbium152\""],
       Annotation[#, "Terbium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1883],
        "\"Terbium153\""],
       Annotation[#, "Terbium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1884],
        "\"Terbium154\""],
       Annotation[#, "Terbium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1885],
        "\"Dysprosium154\""],
       Annotation[#, "Dysprosium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1886],
        "\"Terbium155\""],
       Annotation[#, "Terbium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1887],
        "\"Terbium156\""],
       Annotation[#, "Terbium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1888],
        "\"Dysprosium156\""],
       Annotation[#, "Dysprosium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1889],
        "\"Terbium157\""],
       Annotation[#, "Terbium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1890],
        "\"Terbium158\""],
       Annotation[#, "Terbium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1891],
        "\"Dysprosium158\""],
       Annotation[#, "Dysprosium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1892],
        "\"Terbium160\""],
       Annotation[#, "Terbium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1893],
        "\"Dysprosium161\""],
       Annotation[#, "Dysprosium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1894],
        "\"Dysprosium162\""],
       Annotation[#, "Dysprosium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1895],
        "\"Dysprosium163\""],
       Annotation[#, "Dysprosium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1896],
        "\"Dysprosium164\""],
       Annotation[#, "Dysprosium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1897],
        "\"Dysprosium165\""],
       Annotation[#, "Dysprosium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1898],
        "\"Dysprosium166\""],
       Annotation[#, "Dysprosium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1899],
        "\"Dysprosium167\""],
       Annotation[#, "Dysprosium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1900],
        "\"Dysprosium168\""],
       Annotation[#, "Dysprosium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1901],
        "\"Dysprosium169\""],
       Annotation[#, "Dysprosium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1902],
        "\"Terbium170\""],
       Annotation[#, "Terbium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1903],
        "\"Dysprosium170\""],
       Annotation[#, "Dysprosium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1904],
        "\"Terbium171\""],
       Annotation[#, "Terbium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1905],
        "\"Dysprosium171\""],
       Annotation[#, "Dysprosium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1906],
        "\"Dysprosium138\""],
       Annotation[#, "Dysprosium138", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1907],
        "\"Dysprosium139\""],
       Annotation[#, "Dysprosium139", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1908],
        "\"Dysprosium140\""],
       Annotation[#, "Dysprosium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1909],
        "\"Dysprosium141\""],
       Annotation[#, "Dysprosium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1910],
        "\"Dysprosium142\""],
       Annotation[#, "Dysprosium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1911],
        "\"Dysprosium143\""],
       Annotation[#, "Dysprosium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1912],
        "\"Dysprosium144\""],
       Annotation[#, "Dysprosium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1913],
        "\"Dysprosium145\""],
       Annotation[#, "Dysprosium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1914],
        "\"Dysprosium146\""],
       Annotation[#, "Dysprosium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1915],
        "\"Dysprosium147\""],
       Annotation[#, "Dysprosium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1916],
        "\"Dysprosium148\""],
       Annotation[#, "Dysprosium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1917],
        "\"Dysprosium149\""],
       Annotation[#, "Dysprosium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1918],
        "\"Dysprosium150\""],
       Annotation[#, "Dysprosium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1919],
        "\"Dysprosium151\""],
       Annotation[#, "Dysprosium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1920],
        "\"Dysprosium152\""],
       Annotation[#, "Dysprosium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1921],
        "\"Dysprosium153\""],
       Annotation[#, "Dysprosium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1922],
        "\"Dysprosium155\""],
       Annotation[#, "Dysprosium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1923],
        "\"Dysprosium157\""],
       Annotation[#, "Dysprosium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1924],
        "\"Dysprosium159\""],
       Annotation[#, "Dysprosium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1925],
        "\"Holmium165\""],
       Annotation[#, "Holmium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1926],
        "\"Holmium166\""],
       Annotation[#, "Holmium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1927],
        "\"Holmium167\""],
       Annotation[#, "Holmium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1928],
        "\"Holmium168\""],
       Annotation[#, "Holmium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1929],
        "\"Holmium169\""],
       Annotation[#, "Holmium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1930],
        "\"Holmium170\""],
       Annotation[#, "Holmium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1931],
        "\"Holmium171\""],
       Annotation[#, "Holmium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1932],
        "\"Dysprosium172\""],
       Annotation[#, "Dysprosium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1933],
        "\"Holmium172\""],
       Annotation[#, "Holmium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1934],
        "\"Dysprosium173\""],
       Annotation[#, "Dysprosium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1935],
        "\"Holmium173\""],
       Annotation[#, "Holmium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1936],
        "\"Holmium140\""],
       Annotation[#, "Holmium140", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1937],
        "\"Holmium141\""],
       Annotation[#, "Holmium141", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1938],
        "\"Holmium142\""],
       Annotation[#, "Holmium142", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1939],
        "\"Holmium143\""],
       Annotation[#, "Holmium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1940],
        "\"Holmium144\""],
       Annotation[#, "Holmium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1941],
        "\"Holmium145\""],
       Annotation[#, "Holmium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1942],
        "\"Holmium146\""],
       Annotation[#, "Holmium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1943],
        "\"Holmium147\""],
       Annotation[#, "Holmium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1944],
        "\"Holmium148\""],
       Annotation[#, "Holmium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1945],
        "\"Holmium149\""],
       Annotation[#, "Holmium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1946],
        "\"Holmium150\""],
       Annotation[#, "Holmium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1947],
        "\"Holmium151\""],
       Annotation[#, "Holmium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1948],
        "\"Holmium152\""],
       Annotation[#, "Holmium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1949],
        "\"Holmium153\""],
       Annotation[#, "Holmium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1950],
        "\"Holmium154\""],
       Annotation[#, "Holmium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1951],
        "\"Holmium155\""],
       Annotation[#, "Holmium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1952],
        "\"Holmium156\""],
       Annotation[#, "Holmium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1953],
        "\"Holmium157\""],
       Annotation[#, "Holmium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1954],
        "\"Holmium158\""],
       Annotation[#, "Holmium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1955],
        "\"Holmium159\""],
       Annotation[#, "Holmium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1956],
        "\"Holmium160\""],
       Annotation[#, "Holmium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1957],
        "\"Holmium161\""],
       Annotation[#, "Holmium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1958],
        "\"Holmium162\""],
       Annotation[#, "Holmium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1959],
        "\"Holmium163\""],
       Annotation[#, "Holmium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1960],
        "\"Holmium164\""],
       Annotation[#, "Holmium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1961],
        "\"Erbium164\""],
       Annotation[#, "Erbium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1962],
        "\"Erbium166\""],
       Annotation[#, "Erbium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1963],
        "\"Erbium167\""],
       Annotation[#, "Erbium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1964],
        "\"Erbium168\""],
       Annotation[#, "Erbium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1965],
        "\"Erbium169\""],
       Annotation[#, "Erbium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1966],
        "\"Erbium170\""],
       Annotation[#, "Erbium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1967],
        "\"Erbium171\""],
       Annotation[#, "Erbium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1968],
        "\"Erbium172\""],
       Annotation[#, "Erbium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1969],
        "\"Erbium173\""],
       Annotation[#, "Erbium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1970],
        "\"Holmium174\""],
       Annotation[#, "Holmium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1971],
        "\"Erbium174\""],
       Annotation[#, "Erbium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1972],
        "\"Holmium175\""],
       Annotation[#, "Holmium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1973],
        "\"Erbium175\""],
       Annotation[#, "Erbium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1974],
        "\"Erbium143\""],
       Annotation[#, "Erbium143", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1975],
        "\"Erbium144\""],
       Annotation[#, "Erbium144", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1976],
        "\"Erbium145\""],
       Annotation[#, "Erbium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1977],
        "\"Erbium146\""],
       Annotation[#, "Erbium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1978],
        "\"Erbium147\""],
       Annotation[#, "Erbium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1979],
        "\"Erbium148\""],
       Annotation[#, "Erbium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1980],
        "\"Erbium149\""],
       Annotation[#, "Erbium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1981],
        "\"Erbium150\""],
       Annotation[#, "Erbium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1982],
        "\"Erbium151\""],
       Annotation[#, "Erbium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1983],
        "\"Erbium152\""],
       Annotation[#, "Erbium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1984],
        "\"Erbium153\""],
       Annotation[#, "Erbium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1985],
        "\"Erbium154\""],
       Annotation[#, "Erbium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1986],
        "\"Erbium155\""],
       Annotation[#, "Erbium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1987],
        "\"Erbium156\""],
       Annotation[#, "Erbium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1988],
        "\"Erbium157\""],
       Annotation[#, "Erbium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1989],
        "\"Erbium158\""],
       Annotation[#, "Erbium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1990],
        "\"Erbium159\""],
       Annotation[#, "Erbium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1991],
        "\"Erbium160\""],
       Annotation[#, "Erbium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1992],
        "\"Erbium161\""],
       Annotation[#, "Erbium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1993],
        "\"Erbium162\""],
       Annotation[#, "Erbium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1994],
        "\"Erbium163\""],
       Annotation[#, "Erbium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1995],
        "\"Erbium165\""],
       Annotation[#, "Erbium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1996],
        "\"Thulium169\""],
       Annotation[#, "Thulium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1997],
        "\"Ytterbium170\""],
       Annotation[#, "Ytterbium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1998],
        "\"Thulium171\""],
       Annotation[#, "Thulium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[1999],
        "\"Thulium172\""],
       Annotation[#, "Thulium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2000],
        "\"Thulium173\""],
       Annotation[#, "Thulium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2001],
        "\"Thulium174\""],
       Annotation[#, "Thulium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2002],
        "\"Thulium175\""],
       Annotation[#, "Thulium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2003],
        "\"Erbium176\""],
       Annotation[#, "Erbium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2004],
        "\"Thulium176\""],
       Annotation[#, "Thulium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2005],
        "\"Erbium177\""],
       Annotation[#, "Erbium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2006],
        "\"Thulium177\""],
       Annotation[#, "Thulium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2007],
        "\"Thulium145\""],
       Annotation[#, "Thulium145", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2008],
        "\"Thulium146\""],
       Annotation[#, "Thulium146", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2009],
        "\"Thulium147\""],
       Annotation[#, "Thulium147", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2010],
        "\"Thulium148\""],
       Annotation[#, "Thulium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2011],
        "\"Thulium149\""],
       Annotation[#, "Thulium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2012],
        "\"Thulium150\""],
       Annotation[#, "Thulium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2013],
        "\"Thulium151\""],
       Annotation[#, "Thulium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2014],
        "\"Thulium152\""],
       Annotation[#, "Thulium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2015],
        "\"Thulium153\""],
       Annotation[#, "Thulium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2016],
        "\"Thulium154\""],
       Annotation[#, "Thulium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2017],
        "\"Thulium155\""],
       Annotation[#, "Thulium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2018],
        "\"Thulium156\""],
       Annotation[#, "Thulium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2019],
        "\"Thulium157\""],
       Annotation[#, "Thulium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2020],
        "\"Thulium158\""],
       Annotation[#, "Thulium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2021],
        "\"Thulium159\""],
       Annotation[#, "Thulium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2022],
        "\"Thulium160\""],
       Annotation[#, "Thulium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2023],
        "\"Thulium161\""],
       Annotation[#, "Thulium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2024],
        "\"Thulium162\""],
       Annotation[#, "Thulium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2025],
        "\"Thulium163\""],
       Annotation[#, "Thulium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2026],
        "\"Thulium164\""],
       Annotation[#, "Thulium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2027],
        "\"Thulium165\""],
       Annotation[#, "Thulium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2028],
        "\"Thulium166\""],
       Annotation[#, "Thulium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2029],
        "\"Thulium167\""],
       Annotation[#, "Thulium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2030],
        "\"Thulium168\""],
       Annotation[#, "Thulium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2031],
        "\"Ytterbium168\""],
       Annotation[#, "Ytterbium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2032],
        "\"Thulium170\""],
       Annotation[#, "Thulium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2033],
        "\"Ytterbium171\""],
       Annotation[#, "Ytterbium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2034],
        "\"Ytterbium172\""],
       Annotation[#, "Ytterbium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2035],
        "\"Ytterbium173\""],
       Annotation[#, "Ytterbium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2036],
        "\"Ytterbium174\""],
       Annotation[#, "Ytterbium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2037],
        "\"Ytterbium175\""],
       Annotation[#, "Ytterbium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2038],
        "\"Ytterbium176\""],
       Annotation[#, "Ytterbium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2039],
        "\"Ytterbium177\""],
       Annotation[#, "Ytterbium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2040],
        "\"Thulium178\""],
       Annotation[#, "Thulium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2041],
        "\"Ytterbium178\""],
       Annotation[#, "Ytterbium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2042],
        "\"Thulium179\""],
       Annotation[#, "Thulium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2043],
        "\"Ytterbium179\""],
       Annotation[#, "Ytterbium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2044],
        "\"Ytterbium148\""],
       Annotation[#, "Ytterbium148", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2045],
        "\"Ytterbium149\""],
       Annotation[#, "Ytterbium149", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2046],
        "\"Ytterbium150\""],
       Annotation[#, "Ytterbium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2047],
        "\"Ytterbium151\""],
       Annotation[#, "Ytterbium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2048],
        "\"Ytterbium152\""],
       Annotation[#, "Ytterbium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2049],
        "\"Ytterbium153\""],
       Annotation[#, "Ytterbium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2050],
        "\"Ytterbium154\""],
       Annotation[#, "Ytterbium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2051],
        "\"Ytterbium155\""],
       Annotation[#, "Ytterbium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2052],
        "\"Ytterbium156\""],
       Annotation[#, "Ytterbium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2053],
        "\"Ytterbium157\""],
       Annotation[#, "Ytterbium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2054],
        "\"Ytterbium158\""],
       Annotation[#, "Ytterbium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2055],
        "\"Ytterbium159\""],
       Annotation[#, "Ytterbium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2056],
        "\"Ytterbium160\""],
       Annotation[#, "Ytterbium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2057],
        "\"Ytterbium161\""],
       Annotation[#, "Ytterbium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2058],
        "\"Ytterbium162\""],
       Annotation[#, "Ytterbium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2059],
        "\"Ytterbium163\""],
       Annotation[#, "Ytterbium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2060],
        "\"Ytterbium164\""],
       Annotation[#, "Ytterbium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2061],
        "\"Ytterbium165\""],
       Annotation[#, "Ytterbium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2062],
        "\"Ytterbium166\""],
       Annotation[#, "Ytterbium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2063],
        "\"Ytterbium167\""],
       Annotation[#, "Ytterbium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2064],
        "\"Ytterbium169\""],
       Annotation[#, "Ytterbium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2065],
        "\"Lutetium175\""],
       Annotation[#, "Lutetium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2066],
        "\"Hafnium176\""],
       Annotation[#, "Hafnium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2067],
        "\"Lutetium177\""],
       Annotation[#, "Lutetium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2068],
        "\"Lutetium178\""],
       Annotation[#, "Lutetium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2069],
        "\"Lutetium179\""],
       Annotation[#, "Lutetium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2070],
        "\"Ytterbium180\""],
       Annotation[#, "Ytterbium180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2071],
        "\"Lutetium180\""],
       Annotation[#, "Lutetium180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2072],
        "\"Ytterbium181\""],
       Annotation[#, "Ytterbium181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2073],
        "\"Lutetium181\""],
       Annotation[#, "Lutetium181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2074],
        "\"Lutetium150\""],
       Annotation[#, "Lutetium150", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2075],
        "\"Lutetium151\""],
       Annotation[#, "Lutetium151", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2076],
        "\"Lutetium152\""],
       Annotation[#, "Lutetium152", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2077],
        "\"Lutetium153\""],
       Annotation[#, "Lutetium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2078],
        "\"Lutetium154\""],
       Annotation[#, "Lutetium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2079],
        "\"Lutetium155\""],
       Annotation[#, "Lutetium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2080],
        "\"Lutetium156\""],
       Annotation[#, "Lutetium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2081],
        "\"Lutetium157\""],
       Annotation[#, "Lutetium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2082],
        "\"Lutetium158\""],
       Annotation[#, "Lutetium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2083],
        "\"Lutetium159\""],
       Annotation[#, "Lutetium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2084],
        "\"Lutetium160\""],
       Annotation[#, "Lutetium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2085],
        "\"Lutetium161\""],
       Annotation[#, "Lutetium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2086],
        "\"Lutetium162\""],
       Annotation[#, "Lutetium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2087],
        "\"Lutetium163\""],
       Annotation[#, "Lutetium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2088],
        "\"Lutetium164\""],
       Annotation[#, "Lutetium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2089],
        "\"Lutetium165\""],
       Annotation[#, "Lutetium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2090],
        "\"Lutetium166\""],
       Annotation[#, "Lutetium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2091],
        "\"Lutetium167\""],
       Annotation[#, "Lutetium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2092],
        "\"Lutetium168\""],
       Annotation[#, "Lutetium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2093],
        "\"Lutetium169\""],
       Annotation[#, "Lutetium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2094],
        "\"Lutetium170\""],
       Annotation[#, "Lutetium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2095],
        "\"Lutetium171\""],
       Annotation[#, "Lutetium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2096],
        "\"Lutetium172\""],
       Annotation[#, "Lutetium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2097],
        "\"Lutetium173\""],
       Annotation[#, "Lutetium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2098],
        "\"Lutetium174\""],
       Annotation[#, "Lutetium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2099],
        "\"Lutetium176\""],
       Annotation[#, "Lutetium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2100],
        "\"Hafnium177\""],
       Annotation[#, "Hafnium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2101],
        "\"Hafnium178\""],
       Annotation[#, "Hafnium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2102],
        "\"Hafnium179\""],
       Annotation[#, "Hafnium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2103],
        "\"Hafnium180\""],
       Annotation[#, "Hafnium180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2104],
        "\"Hafnium181\""],
       Annotation[#, "Hafnium181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2105],
        "\"Lutetium182\""],
       Annotation[#, "Lutetium182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2106],
        "\"Hafnium182\""],
       Annotation[#, "Hafnium182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2107],
        "\"Lutetium183\""],
       Annotation[#, "Lutetium183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2108],
        "\"Hafnium183\""],
       Annotation[#, "Hafnium183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2109],
        "\"Lutetium184\""],
       Annotation[#, "Lutetium184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2110],
        "\"Hafnium184\""],
       Annotation[#, "Hafnium184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2111],
        "\"Hafnium153\""],
       Annotation[#, "Hafnium153", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2112],
        "\"Hafnium154\""],
       Annotation[#, "Hafnium154", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2113],
        "\"Hafnium155\""],
       Annotation[#, "Hafnium155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2114],
        "\"Hafnium156\""],
       Annotation[#, "Hafnium156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2115],
        "\"Hafnium157\""],
       Annotation[#, "Hafnium157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2116],
        "\"Hafnium158\""],
       Annotation[#, "Hafnium158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2117],
        "\"Hafnium159\""],
       Annotation[#, "Hafnium159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2118],
        "\"Hafnium160\""],
       Annotation[#, "Hafnium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2119],
        "\"Hafnium161\""],
       Annotation[#, "Hafnium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2120],
        "\"Hafnium162\""],
       Annotation[#, "Hafnium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2121],
        "\"Hafnium163\""],
       Annotation[#, "Hafnium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2122],
        "\"Hafnium164\""],
       Annotation[#, "Hafnium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2123],
        "\"Hafnium165\""],
       Annotation[#, "Hafnium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2124],
        "\"Hafnium166\""],
       Annotation[#, "Hafnium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2125],
        "\"Hafnium167\""],
       Annotation[#, "Hafnium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2126],
        "\"Hafnium168\""],
       Annotation[#, "Hafnium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2127],
        "\"Hafnium169\""],
       Annotation[#, "Hafnium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2128],
        "\"Hafnium170\""],
       Annotation[#, "Hafnium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2129],
        "\"Hafnium171\""],
       Annotation[#, "Hafnium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2130],
        "\"Hafnium172\""],
       Annotation[#, "Hafnium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2131],
        "\"Hafnium173\""],
       Annotation[#, "Hafnium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2132],
        "\"Hafnium174\""],
       Annotation[#, "Hafnium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2133],
        "\"Hafnium175\""],
       Annotation[#, "Hafnium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2134],
        "\"Tantalum181\""],
       Annotation[#, "Tantalum181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2135],
        "\"Tantalum182\""],
       Annotation[#, "Tantalum182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2136],
        "\"Tantalum183\""],
       Annotation[#, "Tantalum183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2137],
        "\"Tantalum184\""],
       Annotation[#, "Tantalum184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2138],
        "\"Hafnium185\""],
       Annotation[#, "Hafnium185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2139],
        "\"Tantalum185\""],
       Annotation[#, "Tantalum185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2140],
        "\"Hafnium186\""],
       Annotation[#, "Hafnium186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2141],
        "\"Tantalum186\""],
       Annotation[#, "Tantalum186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2142],
        "\"Hafnium187\""],
       Annotation[#, "Hafnium187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2143],
        "\"Tantalum187\""],
       Annotation[#, "Tantalum187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2144],
        "\"Hafnium188\""],
       Annotation[#, "Hafnium188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2145],
        "\"Tantalum188\""],
       Annotation[#, "Tantalum188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2146],
        "\"Tantalum155\""],
       Annotation[#, "Tantalum155", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2147],
        "\"Tantalum156\""],
       Annotation[#, "Tantalum156", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2148],
        "\"Tantalum157\""],
       Annotation[#, "Tantalum157", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2149],
        "\"Tantalum158\""],
       Annotation[#, "Tantalum158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2150],
        "\"Tantalum159\""],
       Annotation[#, "Tantalum159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2151],
        "\"Tantalum160\""],
       Annotation[#, "Tantalum160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2152],
        "\"Tantalum161\""],
       Annotation[#, "Tantalum161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2153],
        "\"Tantalum162\""],
       Annotation[#, "Tantalum162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2154],
        "\"Tantalum163\""],
       Annotation[#, "Tantalum163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2155],
        "\"Tantalum164\""],
       Annotation[#, "Tantalum164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2156],
        "\"Tantalum165\""],
       Annotation[#, "Tantalum165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2157],
        "\"Tantalum166\""],
       Annotation[#, "Tantalum166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2158],
        "\"Tantalum167\""],
       Annotation[#, "Tantalum167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2159],
        "\"Tantalum168\""],
       Annotation[#, "Tantalum168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2160],
        "\"Tantalum169\""],
       Annotation[#, "Tantalum169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2161],
        "\"Tantalum170\""],
       Annotation[#, "Tantalum170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2162],
        "\"Tantalum171\""],
       Annotation[#, "Tantalum171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2163],
        "\"Tantalum172\""],
       Annotation[#, "Tantalum172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2164],
        "\"Tantalum173\""],
       Annotation[#, "Tantalum173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2165],
        "\"Tantalum174\""],
       Annotation[#, "Tantalum174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2166],
        "\"Tantalum175\""],
       Annotation[#, "Tantalum175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2167],
        "\"Tantalum176\""],
       Annotation[#, "Tantalum176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2168],
        "\"Tantalum177\""],
       Annotation[#, "Tantalum177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2169],
        "\"Tantalum178\""],
       Annotation[#, "Tantalum178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2170],
        "\"Tantalum179\""],
       Annotation[#, "Tantalum179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2171],
        "\"Tantalum180\""],
       Annotation[#, "Tantalum180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2172],
        "\"Tungsten180\""],
       Annotation[#, "Tungsten180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2173],
        "\"Tungsten182\""],
       Annotation[#, "Tungsten182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2174],
        "\"Tungsten183\""],
       Annotation[#, "Tungsten183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2175],
        "\"Tungsten184\""],
       Annotation[#, "Tungsten184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2176],
        "\"Tungsten185\""],
       Annotation[#, "Tungsten185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2177],
        "\"Tungsten186\""],
       Annotation[#, "Tungsten186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2178],
        "\"Tungsten187\""],
       Annotation[#, "Tungsten187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2179],
        "\"Tungsten188\""],
       Annotation[#, "Tungsten188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2180],
        "\"Tantalum189\""],
       Annotation[#, "Tantalum189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2181],
        "\"Tungsten189\""],
       Annotation[#, "Tungsten189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2182],
        "\"Tantalum190\""],
       Annotation[#, "Tantalum190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2183],
        "\"Tungsten190\""],
       Annotation[#, "Tungsten190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2184],
        "\"Tungsten158\""],
       Annotation[#, "Tungsten158", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2185],
        "\"Tungsten159\""],
       Annotation[#, "Tungsten159", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2186],
        "\"Tungsten160\""],
       Annotation[#, "Tungsten160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2187],
        "\"Tungsten161\""],
       Annotation[#, "Tungsten161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2188],
        "\"Tungsten162\""],
       Annotation[#, "Tungsten162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2189],
        "\"Tungsten163\""],
       Annotation[#, "Tungsten163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2190],
        "\"Tungsten164\""],
       Annotation[#, "Tungsten164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2191],
        "\"Tungsten165\""],
       Annotation[#, "Tungsten165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2192],
        "\"Tungsten166\""],
       Annotation[#, "Tungsten166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2193],
        "\"Tungsten167\""],
       Annotation[#, "Tungsten167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2194],
        "\"Tungsten168\""],
       Annotation[#, "Tungsten168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2195],
        "\"Tungsten169\""],
       Annotation[#, "Tungsten169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2196],
        "\"Tungsten170\""],
       Annotation[#, "Tungsten170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2197],
        "\"Tungsten171\""],
       Annotation[#, "Tungsten171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2198],
        "\"Tungsten172\""],
       Annotation[#, "Tungsten172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2199],
        "\"Tungsten173\""],
       Annotation[#, "Tungsten173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2200],
        "\"Tungsten174\""],
       Annotation[#, "Tungsten174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2201],
        "\"Tungsten175\""],
       Annotation[#, "Tungsten175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2202],
        "\"Tungsten176\""],
       Annotation[#, "Tungsten176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2203],
        "\"Tungsten177\""],
       Annotation[#, "Tungsten177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2204],
        "\"Tungsten178\""],
       Annotation[#, "Tungsten178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2205],
        "\"Tungsten179\""],
       Annotation[#, "Tungsten179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2206],
        "\"Tungsten181\""],
       Annotation[#, "Tungsten181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2207],
        "\"Rhenium185\""],
       Annotation[#, "Rhenium185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2208],
        "\"Osmium186\""],
       Annotation[#, "Osmium186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2209],
        "\"Rhenium187\""],
       Annotation[#, "Rhenium187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2210],
        "\"Rhenium188\""],
       Annotation[#, "Rhenium188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2211],
        "\"Rhenium189\""],
       Annotation[#, "Rhenium189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2212],
        "\"Rhenium190\""],
       Annotation[#, "Rhenium190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2213],
        "\"Tungsten191\""],
       Annotation[#, "Tungsten191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2214],
        "\"Rhenium191\""],
       Annotation[#, "Rhenium191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2215],
        "\"Tungsten192\""],
       Annotation[#, "Tungsten192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2216],
        "\"Rhenium192\""],
       Annotation[#, "Rhenium192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2217],
        "\"Rhenium160\""],
       Annotation[#, "Rhenium160", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2218],
        "\"Rhenium161\""],
       Annotation[#, "Rhenium161", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2219],
        "\"Rhenium162\""],
       Annotation[#, "Rhenium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2220],
        "\"Rhenium163\""],
       Annotation[#, "Rhenium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2221],
        "\"Rhenium164\""],
       Annotation[#, "Rhenium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2222],
        "\"Rhenium165\""],
       Annotation[#, "Rhenium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2223],
        "\"Rhenium166\""],
       Annotation[#, "Rhenium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2224],
        "\"Rhenium167\""],
       Annotation[#, "Rhenium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2225],
        "\"Rhenium168\""],
       Annotation[#, "Rhenium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2226],
        "\"Rhenium169\""],
       Annotation[#, "Rhenium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2227],
        "\"Rhenium170\""],
       Annotation[#, "Rhenium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2228],
        "\"Rhenium171\""],
       Annotation[#, "Rhenium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2229],
        "\"Rhenium172\""],
       Annotation[#, "Rhenium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2230],
        "\"Rhenium173\""],
       Annotation[#, "Rhenium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2231],
        "\"Rhenium174\""],
       Annotation[#, "Rhenium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2232],
        "\"Rhenium175\""],
       Annotation[#, "Rhenium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2233],
        "\"Rhenium176\""],
       Annotation[#, "Rhenium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2234],
        "\"Rhenium177\""],
       Annotation[#, "Rhenium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2235],
        "\"Rhenium178\""],
       Annotation[#, "Rhenium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2236],
        "\"Rhenium179\""],
       Annotation[#, "Rhenium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2237],
        "\"Rhenium180\""],
       Annotation[#, "Rhenium180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2238],
        "\"Rhenium181\""],
       Annotation[#, "Rhenium181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2239],
        "\"Rhenium182\""],
       Annotation[#, "Rhenium182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2240],
        "\"Rhenium183\""],
       Annotation[#, "Rhenium183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2241],
        "\"Rhenium184\""],
       Annotation[#, "Rhenium184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2242],
        "\"Rhenium186\""],
       Annotation[#, "Rhenium186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2243],
        "\"Osmium187\""],
       Annotation[#, "Osmium187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2244],
        "\"Osmium188\""],
       Annotation[#, "Osmium188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2245],
        "\"Osmium189\""],
       Annotation[#, "Osmium189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2246],
        "\"Osmium190\""],
       Annotation[#, "Osmium190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2247],
        "\"Osmium191\""],
       Annotation[#, "Osmium191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2248],
        "\"Osmium192\""],
       Annotation[#, "Osmium192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2249],
        "\"Rhenium193\""],
       Annotation[#, "Rhenium193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2250],
        "\"Osmium193\""],
       Annotation[#, "Osmium193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2251],
        "\"Rhenium194\""],
       Annotation[#, "Rhenium194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2252],
        "\"Osmium194\""],
       Annotation[#, "Osmium194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2253],
        "\"Osmium162\""],
       Annotation[#, "Osmium162", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2254],
        "\"Osmium163\""],
       Annotation[#, "Osmium163", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2255],
        "\"Osmium164\""],
       Annotation[#, "Osmium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2256],
        "\"Osmium165\""],
       Annotation[#, "Osmium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2257],
        "\"Osmium166\""],
       Annotation[#, "Osmium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2258],
        "\"Osmium167\""],
       Annotation[#, "Osmium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2259],
        "\"Osmium168\""],
       Annotation[#, "Osmium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2260],
        "\"Osmium169\""],
       Annotation[#, "Osmium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2261],
        "\"Osmium170\""],
       Annotation[#, "Osmium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2262],
        "\"Osmium171\""],
       Annotation[#, "Osmium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2263],
        "\"Osmium172\""],
       Annotation[#, "Osmium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2264],
        "\"Osmium173\""],
       Annotation[#, "Osmium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2265],
        "\"Osmium174\""],
       Annotation[#, "Osmium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2266],
        "\"Osmium175\""],
       Annotation[#, "Osmium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2267],
        "\"Osmium176\""],
       Annotation[#, "Osmium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2268],
        "\"Osmium177\""],
       Annotation[#, "Osmium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2269],
        "\"Osmium178\""],
       Annotation[#, "Osmium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2270],
        "\"Osmium179\""],
       Annotation[#, "Osmium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2271],
        "\"Osmium180\""],
       Annotation[#, "Osmium180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2272],
        "\"Osmium181\""],
       Annotation[#, "Osmium181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2273],
        "\"Osmium182\""],
       Annotation[#, "Osmium182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2274],
        "\"Osmium183\""],
       Annotation[#, "Osmium183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2275],
        "\"Osmium184\""],
       Annotation[#, "Osmium184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2276],
        "\"Osmium185\""],
       Annotation[#, "Osmium185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2277],
        "\"Iridium191\""],
       Annotation[#, "Iridium191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2278],
        "\"Platinum192\""],
       Annotation[#, "Platinum192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2279],
        "\"Iridium193\""],
       Annotation[#, "Iridium193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2280],
        "\"Iridium194\""],
       Annotation[#, "Iridium194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2281],
        "\"Osmium195\""],
       Annotation[#, "Osmium195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2282],
        "\"Iridium195\""],
       Annotation[#, "Iridium195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2283],
        "\"Osmium196\""],
       Annotation[#, "Osmium196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2284],
        "\"Iridium196\""],
       Annotation[#, "Iridium196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2285],
        "\"Iridium164\""],
       Annotation[#, "Iridium164", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2286],
        "\"Iridium165\""],
       Annotation[#, "Iridium165", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2287],
        "\"Iridium166\""],
       Annotation[#, "Iridium166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2288],
        "\"Iridium167\""],
       Annotation[#, "Iridium167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2289],
        "\"Iridium168\""],
       Annotation[#, "Iridium168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2290],
        "\"Iridium169\""],
       Annotation[#, "Iridium169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2291],
        "\"Iridium170\""],
       Annotation[#, "Iridium170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2292],
        "\"Iridium171\""],
       Annotation[#, "Iridium171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2293],
        "\"Iridium172\""],
       Annotation[#, "Iridium172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2294],
        "\"Iridium173\""],
       Annotation[#, "Iridium173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2295],
        "\"Iridium174\""],
       Annotation[#, "Iridium174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2296],
        "\"Iridium175\""],
       Annotation[#, "Iridium175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2297],
        "\"Iridium176\""],
       Annotation[#, "Iridium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2298],
        "\"Iridium177\""],
       Annotation[#, "Iridium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2299],
        "\"Iridium178\""],
       Annotation[#, "Iridium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2300],
        "\"Iridium179\""],
       Annotation[#, "Iridium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2301],
        "\"Iridium180\""],
       Annotation[#, "Iridium180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2302],
        "\"Iridium181\""],
       Annotation[#, "Iridium181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2303],
        "\"Iridium182\""],
       Annotation[#, "Iridium182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2304],
        "\"Iridium183\""],
       Annotation[#, "Iridium183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2305],
        "\"Iridium184\""],
       Annotation[#, "Iridium184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2306],
        "\"Iridium185\""],
       Annotation[#, "Iridium185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2307],
        "\"Iridium186\""],
       Annotation[#, "Iridium186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2308],
        "\"Iridium187\""],
       Annotation[#, "Iridium187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2309],
        "\"Iridium188\""],
       Annotation[#, "Iridium188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2310],
        "\"Iridium189\""],
       Annotation[#, "Iridium189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2311],
        "\"Iridium190\""],
       Annotation[#, "Iridium190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2312],
        "\"Iridium192\""],
       Annotation[#, "Iridium192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2313],
        "\"Platinum194\""],
       Annotation[#, "Platinum194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2314],
        "\"Platinum195\""],
       Annotation[#, "Platinum195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2315],
        "\"Platinum196\""],
       Annotation[#, "Platinum196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2316],
        "\"Iridium197\""],
       Annotation[#, "Iridium197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2317],
        "\"Platinum197\""],
       Annotation[#, "Platinum197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2318],
        "\"Iridium198\""],
       Annotation[#, "Iridium198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2319],
        "\"Platinum198\""],
       Annotation[#, "Platinum198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2320],
        "\"Iridium199\""],
       Annotation[#, "Iridium199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2321],
        "\"Platinum199\""],
       Annotation[#, "Platinum199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2322],
        "\"Platinum166\""],
       Annotation[#, "Platinum166", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2323],
        "\"Platinum167\""],
       Annotation[#, "Platinum167", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2324],
        "\"Platinum168\""],
       Annotation[#, "Platinum168", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2325],
        "\"Platinum169\""],
       Annotation[#, "Platinum169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2326],
        "\"Platinum170\""],
       Annotation[#, "Platinum170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2327],
        "\"Platinum171\""],
       Annotation[#, "Platinum171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2328],
        "\"Platinum172\""],
       Annotation[#, "Platinum172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2329],
        "\"Platinum173\""],
       Annotation[#, "Platinum173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2330],
        "\"Platinum174\""],
       Annotation[#, "Platinum174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2331],
        "\"Platinum175\""],
       Annotation[#, "Platinum175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2332],
        "\"Platinum176\""],
       Annotation[#, "Platinum176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2333],
        "\"Platinum177\""],
       Annotation[#, "Platinum177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2334],
        "\"Platinum178\""],
       Annotation[#, "Platinum178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2335],
        "\"Platinum179\""],
       Annotation[#, "Platinum179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2336],
        "\"Platinum180\""],
       Annotation[#, "Platinum180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2337],
        "\"Platinum181\""],
       Annotation[#, "Platinum181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2338],
        "\"Platinum182\""],
       Annotation[#, "Platinum182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2339],
        "\"Platinum183\""],
       Annotation[#, "Platinum183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2340],
        "\"Platinum184\""],
       Annotation[#, "Platinum184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2341],
        "\"Platinum185\""],
       Annotation[#, "Platinum185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2342],
        "\"Platinum186\""],
       Annotation[#, "Platinum186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2343],
        "\"Platinum187\""],
       Annotation[#, "Platinum187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2344],
        "\"Platinum188\""],
       Annotation[#, "Platinum188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2345],
        "\"Platinum189\""],
       Annotation[#, "Platinum189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2346],
        "\"Platinum190\""],
       Annotation[#, "Platinum190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2347],
        "\"Platinum191\""],
       Annotation[#, "Platinum191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2348],
        "\"Platinum193\""],
       Annotation[#, "Platinum193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2349],
        "\"Gold197\""],
       Annotation[#, "Gold197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2350],
        "\"Mercury198\""],
       Annotation[#, "Mercury198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2351],
        "\"Gold199\""],
       Annotation[#, "Gold199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2352],
        "\"Platinum200\""],
       Annotation[#, "Platinum200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2353],
        "\"Gold200\""],
       Annotation[#, "Gold200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2354],
        "\"Platinum201\""],
       Annotation[#, "Platinum201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2355],
        "\"Gold201\""],
       Annotation[#, "Gold201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2356],
        "\"Platinum202\""],
       Annotation[#, "Platinum202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2357],
        "\"Gold202\""],
       Annotation[#, "Gold202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2358],
        "\"Gold169\""],
       Annotation[#, "Gold169", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2359],
        "\"Gold170\""],
       Annotation[#, "Gold170", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2360],
        "\"Gold171\""],
       Annotation[#, "Gold171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2361],
        "\"Gold172\""],
       Annotation[#, "Gold172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2362],
        "\"Gold173\""],
       Annotation[#, "Gold173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2363],
        "\"Gold174\""],
       Annotation[#, "Gold174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2364],
        "\"Gold175\""],
       Annotation[#, "Gold175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2365],
        "\"Gold176\""],
       Annotation[#, "Gold176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2366],
        "\"Gold177\""],
       Annotation[#, "Gold177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2367],
        "\"Gold178\""],
       Annotation[#, "Gold178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2368],
        "\"Gold179\""],
       Annotation[#, "Gold179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2369],
        "\"Gold180\""],
       Annotation[#, "Gold180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2370],
        "\"Gold181\""],
       Annotation[#, "Gold181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2371],
        "\"Gold182\""],
       Annotation[#, "Gold182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2372],
        "\"Gold183\""],
       Annotation[#, "Gold183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2373],
        "\"Gold184\""],
       Annotation[#, "Gold184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2374],
        "\"Gold185\""],
       Annotation[#, "Gold185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2375],
        "\"Gold186\""],
       Annotation[#, "Gold186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2376],
        "\"Gold187\""],
       Annotation[#, "Gold187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2377],
        "\"Gold188\""],
       Annotation[#, "Gold188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2378],
        "\"Gold189\""],
       Annotation[#, "Gold189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2379],
        "\"Gold190\""],
       Annotation[#, "Gold190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2380],
        "\"Gold191\""],
       Annotation[#, "Gold191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2381],
        "\"Gold192\""],
       Annotation[#, "Gold192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2382],
        "\"Gold193\""],
       Annotation[#, "Gold193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2383],
        "\"Gold194\""],
       Annotation[#, "Gold194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2384],
        "\"Gold195\""],
       Annotation[#, "Gold195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2385],
        "\"Gold196\""],
       Annotation[#, "Gold196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2386],
        "\"Mercury196\""],
       Annotation[#, "Mercury196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2387],
        "\"Gold198\""],
       Annotation[#, "Gold198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2388],
        "\"Mercury199\""],
       Annotation[#, "Mercury199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2389],
        "\"Mercury200\""],
       Annotation[#, "Mercury200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2390],
        "\"Mercury201\""],
       Annotation[#, "Mercury201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2391],
        "\"Mercury202\""],
       Annotation[#, "Mercury202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2392],
        "\"Gold203\""],
       Annotation[#, "Gold203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2393],
        "\"Mercury203\""],
       Annotation[#, "Mercury203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2394],
        "\"Gold204\""],
       Annotation[#, "Gold204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2395],
        "\"Mercury204\""],
       Annotation[#, "Mercury204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2396],
        "\"Gold205\""],
       Annotation[#, "Gold205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2397],
        "\"Mercury205\""],
       Annotation[#, "Mercury205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2398],
        "\"Mercury171\""],
       Annotation[#, "Mercury171", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2399],
        "\"Mercury172\""],
       Annotation[#, "Mercury172", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2400],
        "\"Mercury173\""],
       Annotation[#, "Mercury173", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2401],
        "\"Mercury174\""],
       Annotation[#, "Mercury174", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2402],
        "\"Mercury175\""],
       Annotation[#, "Mercury175", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2403],
        "\"Mercury176\""],
       Annotation[#, "Mercury176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2404],
        "\"Mercury177\""],
       Annotation[#, "Mercury177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2405],
        "\"Mercury178\""],
       Annotation[#, "Mercury178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2406],
        "\"Mercury179\""],
       Annotation[#, "Mercury179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2407],
        "\"Mercury180\""],
       Annotation[#, "Mercury180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2408],
        "\"Mercury181\""],
       Annotation[#, "Mercury181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2409],
        "\"Mercury182\""],
       Annotation[#, "Mercury182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2410],
        "\"Mercury183\""],
       Annotation[#, "Mercury183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2411],
        "\"Mercury184\""],
       Annotation[#, "Mercury184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2412],
        "\"Mercury185\""],
       Annotation[#, "Mercury185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2413],
        "\"Mercury186\""],
       Annotation[#, "Mercury186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2414],
        "\"Mercury187\""],
       Annotation[#, "Mercury187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2415],
        "\"Mercury188\""],
       Annotation[#, "Mercury188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2416],
        "\"Mercury189\""],
       Annotation[#, "Mercury189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2417],
        "\"Mercury190\""],
       Annotation[#, "Mercury190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2418],
        "\"Mercury191\""],
       Annotation[#, "Mercury191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2419],
        "\"Mercury192\""],
       Annotation[#, "Mercury192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2420],
        "\"Mercury193\""],
       Annotation[#, "Mercury193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2421],
        "\"Mercury194\""],
       Annotation[#, "Mercury194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2422],
        "\"Mercury195\""],
       Annotation[#, "Mercury195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2423],
        "\"Mercury197\""],
       Annotation[#, "Mercury197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2424],
        "\"Thallium203\""],
       Annotation[#, "Thallium203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2425],
        "\"Lead204\""],
       Annotation[#, "Lead204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2426],
        "\"Thallium205\""],
       Annotation[#, "Thallium205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2427],
        "\"Mercury206\""],
       Annotation[#, "Mercury206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2428],
        "\"Thallium206\""],
       Annotation[#, "Thallium206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2429],
        "\"Mercury207\""],
       Annotation[#, "Mercury207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2430],
        "\"Thallium207\""],
       Annotation[#, "Thallium207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2431],
        "\"Mercury208\""],
       Annotation[#, "Mercury208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2432],
        "\"Thallium208\""],
       Annotation[#, "Thallium208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2433],
        "\"Mercury209\""],
       Annotation[#, "Mercury209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2434],
        "\"Thallium209\""],
       Annotation[#, "Thallium209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2435],
        "\"Mercury210\""],
       Annotation[#, "Mercury210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2436],
        "\"Thallium210\""],
       Annotation[#, "Thallium210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2437],
        "\"Thallium176\""],
       Annotation[#, "Thallium176", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2438],
        "\"Thallium177\""],
       Annotation[#, "Thallium177", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2439],
        "\"Thallium178\""],
       Annotation[#, "Thallium178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2440],
        "\"Thallium179\""],
       Annotation[#, "Thallium179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2441],
        "\"Thallium180\""],
       Annotation[#, "Thallium180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2442],
        RowBox[{"Missing", "[", "\"Variable\"", "]"}]],
       Annotation[#, 
        Missing["Variable"], "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2443],
        "\"Thallium181\""],
       Annotation[#, "Thallium181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2444],
        "\"Thallium182\""],
       Annotation[#, "Thallium182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2445],
        "\"Thallium183\""],
       Annotation[#, "Thallium183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2446],
        "\"Thallium184\""],
       Annotation[#, "Thallium184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2447],
        "\"Thallium185\""],
       Annotation[#, "Thallium185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2448],
        "\"Thallium186\""],
       Annotation[#, "Thallium186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2449],
        "\"Thallium187\""],
       Annotation[#, "Thallium187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2450],
        "\"Thallium188\""],
       Annotation[#, "Thallium188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2451],
        "\"Thallium189\""],
       Annotation[#, "Thallium189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2452],
        "\"Thallium190\""],
       Annotation[#, "Thallium190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2453],
        "\"Thallium191\""],
       Annotation[#, "Thallium191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2454],
        "\"Thallium192\""],
       Annotation[#, "Thallium192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2455],
        "\"Thallium193\""],
       Annotation[#, "Thallium193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2456],
        "\"Thallium194\""],
       Annotation[#, "Thallium194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2457],
        "\"Thallium195\""],
       Annotation[#, "Thallium195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2458],
        "\"Thallium196\""],
       Annotation[#, "Thallium196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2459],
        "\"Thallium197\""],
       Annotation[#, "Thallium197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2460],
        "\"Thallium198\""],
       Annotation[#, "Thallium198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2461],
        "\"Thallium199\""],
       Annotation[#, "Thallium199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2462],
        "\"Thallium200\""],
       Annotation[#, "Thallium200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2463],
        "\"Thallium201\""],
       Annotation[#, "Thallium201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2464],
        "\"Thallium202\""],
       Annotation[#, "Thallium202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2465],
        "\"Thallium204\""],
       Annotation[#, "Thallium204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2466],
        "\"Lead206\""],
       Annotation[#, "Lead206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2467],
        "\"Lead207\""],
       Annotation[#, "Lead207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2468],
        "\"Lead208\""],
       Annotation[#, "Lead208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2469],
        "\"Lead209\""],
       Annotation[#, "Lead209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2470],
        "\"Lead210\""],
       Annotation[#, "Lead210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2471],
        "\"Thallium211\""],
       Annotation[#, "Thallium211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2472],
        "\"Lead211\""],
       Annotation[#, "Lead211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2473],
        "\"Thallium212\""],
       Annotation[#, "Thallium212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2474],
        "\"Lead212\""],
       Annotation[#, "Lead212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2475],
        "\"Lead178\""],
       Annotation[#, "Lead178", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2476],
        "\"Lead179\""],
       Annotation[#, "Lead179", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2477],
        "\"Lead180\""],
       Annotation[#, "Lead180", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2478],
        "\"Lead181\""],
       Annotation[#, "Lead181", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2479],
        "\"Lead182\""],
       Annotation[#, "Lead182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2480],
        "\"Lead183\""],
       Annotation[#, "Lead183", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2481],
        "\"Lead184\""],
       Annotation[#, "Lead184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2482],
        "\"Lead185\""],
       Annotation[#, "Lead185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2483],
        "\"Lead186\""],
       Annotation[#, "Lead186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2484],
        "\"Lead187\""],
       Annotation[#, "Lead187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2485],
        "\"Lead188\""],
       Annotation[#, "Lead188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2486],
        "\"Lead189\""],
       Annotation[#, "Lead189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2487],
        "\"Lead190\""],
       Annotation[#, "Lead190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2488],
        "\"Lead191\""],
       Annotation[#, "Lead191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2489],
        "\"Lead192\""],
       Annotation[#, "Lead192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2490],
        "\"Lead193\""],
       Annotation[#, "Lead193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2491],
        "\"Lead194\""],
       Annotation[#, "Lead194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2492],
        "\"Lead195\""],
       Annotation[#, "Lead195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2493],
        "\"Lead196\""],
       Annotation[#, "Lead196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2494],
        "\"Lead197\""],
       Annotation[#, "Lead197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2495],
        "\"Lead198\""],
       Annotation[#, "Lead198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2496],
        "\"Lead199\""],
       Annotation[#, "Lead199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2497],
        "\"Lead200\""],
       Annotation[#, "Lead200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2498],
        "\"Lead201\""],
       Annotation[#, "Lead201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2499],
        "\"Lead202\""],
       Annotation[#, "Lead202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2500],
        "\"Lead203\""],
       Annotation[#, "Lead203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2501],
        "\"Lead205\""],
       Annotation[#, "Lead205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2502],
        "\"Bismuth209\""],
       Annotation[#, "Bismuth209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2503],
        "\"Bismuth210\""],
       Annotation[#, "Bismuth210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2504],
        "\"Bismuth211\""],
       Annotation[#, "Bismuth211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2505],
        "\"Bismuth212\""],
       Annotation[#, "Bismuth212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2506],
        "\"Lead213\""],
       Annotation[#, "Lead213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2507],
        "\"Bismuth213\""],
       Annotation[#, "Bismuth213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2508],
        "\"Lead214\""],
       Annotation[#, "Lead214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2509],
        "\"Bismuth214\""],
       Annotation[#, "Bismuth214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2510],
        "\"Lead215\""],
       Annotation[#, "Lead215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2511],
        "\"Bismuth215\""],
       Annotation[#, "Bismuth215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2512],
        "\"Bismuth184\""],
       Annotation[#, "Bismuth184", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2513],
        "\"Bismuth185\""],
       Annotation[#, "Bismuth185", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2514],
        "\"Bismuth186\""],
       Annotation[#, "Bismuth186", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2515],
        "\"Bismuth187\""],
       Annotation[#, "Bismuth187", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2516],
        "\"Bismuth188\""],
       Annotation[#, "Bismuth188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2517],
        "\"Bismuth189\""],
       Annotation[#, "Bismuth189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2518],
        "\"Bismuth190\""],
       Annotation[#, "Bismuth190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2519],
        "\"Bismuth191\""],
       Annotation[#, "Bismuth191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2520],
        "\"Bismuth192\""],
       Annotation[#, "Bismuth192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2521],
        "\"Bismuth193\""],
       Annotation[#, "Bismuth193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2522],
        "\"Bismuth194\""],
       Annotation[#, "Bismuth194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2523],
        "\"Bismuth195\""],
       Annotation[#, "Bismuth195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2524],
        "\"Bismuth196\""],
       Annotation[#, "Bismuth196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2525],
        "\"Bismuth197\""],
       Annotation[#, "Bismuth197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2526],
        "\"Bismuth198\""],
       Annotation[#, "Bismuth198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2527],
        "\"Bismuth199\""],
       Annotation[#, "Bismuth199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2528],
        "\"Bismuth200\""],
       Annotation[#, "Bismuth200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2529],
        "\"Bismuth201\""],
       Annotation[#, "Bismuth201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2530],
        "\"Bismuth202\""],
       Annotation[#, "Bismuth202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2531],
        "\"Bismuth203\""],
       Annotation[#, "Bismuth203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2532],
        "\"Bismuth204\""],
       Annotation[#, "Bismuth204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2533],
        "\"Bismuth205\""],
       Annotation[#, "Bismuth205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2534],
        "\"Bismuth206\""],
       Annotation[#, "Bismuth206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2535],
        "\"Bismuth207\""],
       Annotation[#, "Bismuth207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2536],
        "\"Bismuth208\""],
       Annotation[#, "Bismuth208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2537],
        "\"Polonium210\""],
       Annotation[#, "Polonium210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2538],
        "\"Polonium211\""],
       Annotation[#, "Polonium211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2539],
        "\"Polonium212\""],
       Annotation[#, "Polonium212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2540],
        "\"Polonium213\""],
       Annotation[#, "Polonium213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2541],
        "\"Polonium214\""],
       Annotation[#, "Polonium214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2542],
        "\"Polonium215\""],
       Annotation[#, "Polonium215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2543],
        "\"Bismuth216\""],
       Annotation[#, "Bismuth216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2544],
        "\"Polonium216\""],
       Annotation[#, "Polonium216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2545],
        "\"Bismuth217\""],
       Annotation[#, "Bismuth217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2546],
        "\"Polonium217\""],
       Annotation[#, "Polonium217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2547],
        "\"Bismuth218\""],
       Annotation[#, "Bismuth218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2548],
        "\"Polonium218\""],
       Annotation[#, "Polonium218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2549],
        "\"Polonium188\""],
       Annotation[#, "Polonium188", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2550],
        "\"Polonium189\""],
       Annotation[#, "Polonium189", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2551],
        "\"Polonium190\""],
       Annotation[#, "Polonium190", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2552],
        "\"Polonium191\""],
       Annotation[#, "Polonium191", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2553],
        "\"Polonium192\""],
       Annotation[#, "Polonium192", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2554],
        "\"Polonium193\""],
       Annotation[#, "Polonium193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2555],
        "\"Polonium194\""],
       Annotation[#, "Polonium194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2556],
        "\"Polonium195\""],
       Annotation[#, "Polonium195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2557],
        "\"Polonium196\""],
       Annotation[#, "Polonium196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2558],
        "\"Polonium197\""],
       Annotation[#, "Polonium197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2559],
        "\"Polonium198\""],
       Annotation[#, "Polonium198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2560],
        "\"Polonium199\""],
       Annotation[#, "Polonium199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2561],
        "\"Polonium200\""],
       Annotation[#, "Polonium200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2562],
        "\"Polonium201\""],
       Annotation[#, "Polonium201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2563],
        "\"Polonium202\""],
       Annotation[#, "Polonium202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2564],
        "\"Polonium203\""],
       Annotation[#, "Polonium203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2565],
        "\"Polonium204\""],
       Annotation[#, "Polonium204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2566],
        "\"Polonium205\""],
       Annotation[#, "Polonium205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2567],
        "\"Polonium206\""],
       Annotation[#, "Polonium206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2568],
        "\"Polonium207\""],
       Annotation[#, "Polonium207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2569],
        "\"Polonium208\""],
       Annotation[#, "Polonium208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2570],
        "\"Polonium209\""],
       Annotation[#, "Polonium209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2571],
        "\"Astatine215\""],
       Annotation[#, "Astatine215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2572],
        "\"Radon216\""],
       Annotation[#, "Radon216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2573],
        "\"Astatine217\""],
       Annotation[#, "Astatine217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2574],
        "\"Astatine218\""],
       Annotation[#, "Astatine218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2575],
        "\"Polonium219\""],
       Annotation[#, "Polonium219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2576],
        "\"Astatine219\""],
       Annotation[#, "Astatine219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2577],
        "\"Polonium220\""],
       Annotation[#, "Polonium220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2578],
        "\"Astatine220\""],
       Annotation[#, "Astatine220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2579],
        "\"Astatine193\""],
       Annotation[#, "Astatine193", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2580],
        "\"Astatine194\""],
       Annotation[#, "Astatine194", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2581],
        "\"Astatine195\""],
       Annotation[#, "Astatine195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2582],
        "\"Astatine196\""],
       Annotation[#, "Astatine196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2583],
        "\"Astatine197\""],
       Annotation[#, "Astatine197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2584],
        "\"Astatine198\""],
       Annotation[#, "Astatine198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2585],
        "\"Astatine199\""],
       Annotation[#, "Astatine199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2586],
        "\"Astatine200\""],
       Annotation[#, "Astatine200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2587],
        "\"Astatine201\""],
       Annotation[#, "Astatine201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2588],
        "\"Astatine202\""],
       Annotation[#, "Astatine202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2589],
        "\"Astatine203\""],
       Annotation[#, "Astatine203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2590],
        "\"Astatine204\""],
       Annotation[#, "Astatine204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2591],
        "\"Astatine205\""],
       Annotation[#, "Astatine205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2592],
        "\"Astatine206\""],
       Annotation[#, "Astatine206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2593],
        "\"Astatine207\""],
       Annotation[#, "Astatine207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2594],
        "\"Astatine208\""],
       Annotation[#, "Astatine208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2595],
        "\"Astatine209\""],
       Annotation[#, "Astatine209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2596],
        "\"Astatine210\""],
       Annotation[#, "Astatine210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2597],
        "\"Astatine211\""],
       Annotation[#, "Astatine211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2598],
        "\"Astatine212\""],
       Annotation[#, "Astatine212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2599],
        "\"Radon212\""],
       Annotation[#, "Radon212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2600],
        "\"Astatine213\""],
       Annotation[#, "Astatine213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2601],
        "\"Astatine214\""],
       Annotation[#, "Astatine214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2602],
        "\"Astatine216\""],
       Annotation[#, "Astatine216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2603],
        "\"Radon217\""],
       Annotation[#, "Radon217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2604],
        "\"Radon218\""],
       Annotation[#, "Radon218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2605],
        "\"Radon219\""],
       Annotation[#, "Radon219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2606],
        "\"Radon220\""],
       Annotation[#, "Radon220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2607],
        "\"Astatine221\""],
       Annotation[#, "Astatine221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2608],
        "\"Radon221\""],
       Annotation[#, "Radon221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2609],
        "\"Astatine222\""],
       Annotation[#, "Astatine222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2610],
        "\"Radon222\""],
       Annotation[#, "Radon222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2611],
        "\"Astatine223\""],
       Annotation[#, "Astatine223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2612],
        "\"Radon223\""],
       Annotation[#, "Radon223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2613],
        "\"Bismuth219\""],
       Annotation[#, "Bismuth219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2614],
        "\"Radon195\""],
       Annotation[#, "Radon195", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2615],
        "\"Radon196\""],
       Annotation[#, "Radon196", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2616],
        "\"Radon197\""],
       Annotation[#, "Radon197", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2617],
        "\"Radon198\""],
       Annotation[#, "Radon198", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2618],
        "\"Radon199\""],
       Annotation[#, "Radon199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2619],
        "\"Radon200\""],
       Annotation[#, "Radon200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2620],
        "\"Radon201\""],
       Annotation[#, "Radon201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2621],
        "\"Radon202\""],
       Annotation[#, "Radon202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2622],
        "\"Radon203\""],
       Annotation[#, "Radon203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2623],
        "\"Radon204\""],
       Annotation[#, "Radon204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2624],
        "\"Radon205\""],
       Annotation[#, "Radon205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2625],
        "\"Radon206\""],
       Annotation[#, "Radon206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2626],
        "\"Radon207\""],
       Annotation[#, "Radon207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2627],
        "\"Radon208\""],
       Annotation[#, "Radon208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2628],
        "\"Radon209\""],
       Annotation[#, "Radon209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2629],
        "\"Radon210\""],
       Annotation[#, "Radon210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2630],
        "\"Radon211\""],
       Annotation[#, "Radon211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2631],
        "\"Radon213\""],
       Annotation[#, "Radon213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2632],
        "\"Radon214\""],
       Annotation[#, "Radon214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2633],
        "\"Radon215\""],
       Annotation[#, "Radon215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2634],
        "\"Radium220\""],
       Annotation[#, "Radium220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2635],
        "\"Francium221\""],
       Annotation[#, "Francium221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2636],
        "\"Francium223\""],
       Annotation[#, "Francium223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2637],
        "\"Radon224\""],
       Annotation[#, "Radon224", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2638],
        "\"Francium224\""],
       Annotation[#, "Francium224", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2639],
        "\"Radon225\""],
       Annotation[#, "Radon225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2640],
        "\"Francium225\""],
       Annotation[#, "Francium225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2641],
        "\"Radon226\""],
       Annotation[#, "Radon226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2642],
        "\"Francium226\""],
       Annotation[#, "Francium226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2643],
        "\"Radon227\""],
       Annotation[#, "Radon227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2644],
        "\"Francium227\""],
       Annotation[#, "Francium227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2645],
        "\"Radon228\""],
       Annotation[#, "Radon228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2646],
        "\"Francium228\""],
       Annotation[#, "Francium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2647],
        "\"Francium199\""],
       Annotation[#, "Francium199", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2648],
        "\"Francium200\""],
       Annotation[#, "Francium200", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2649],
        "\"Francium201\""],
       Annotation[#, "Francium201", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2650],
        "\"Francium202\""],
       Annotation[#, "Francium202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2651],
        "\"Francium203\""],
       Annotation[#, "Francium203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2652],
        "\"Francium204\""],
       Annotation[#, "Francium204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2653],
        "\"Francium205\""],
       Annotation[#, "Francium205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2654],
        "\"Francium206\""],
       Annotation[#, "Francium206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2655],
        "\"Francium207\""],
       Annotation[#, "Francium207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2656],
        "\"Francium208\""],
       Annotation[#, "Francium208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2657],
        "\"Francium209\""],
       Annotation[#, "Francium209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2658],
        "\"Francium210\""],
       Annotation[#, "Francium210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2659],
        "\"Francium211\""],
       Annotation[#, "Francium211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2660],
        "\"Francium212\""],
       Annotation[#, "Francium212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2661],
        "\"Francium213\""],
       Annotation[#, "Francium213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2662],
        "\"Francium214\""],
       Annotation[#, "Francium214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2663],
        "\"Francium215\""],
       Annotation[#, "Francium215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2664],
        "\"Francium216\""],
       Annotation[#, "Francium216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2665],
        "\"Francium217\""],
       Annotation[#, "Francium217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2666],
        "\"Francium218\""],
       Annotation[#, "Francium218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2667],
        "\"Francium219\""],
       Annotation[#, "Francium219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2668],
        "\"Francium220\""],
       Annotation[#, "Francium220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2669],
        "\"Radium221\""],
       Annotation[#, "Radium221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2670],
        "\"Francium222\""],
       Annotation[#, "Francium222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2671],
        "\"Radium222\""],
       Annotation[#, "Radium222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2672],
        "\"Radium223\""],
       Annotation[#, "Radium223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2673],
        "\"Radium224\""],
       Annotation[#, "Radium224", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2674],
        "\"Radium225\""],
       Annotation[#, "Radium225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2675],
        "\"Radium226\""],
       Annotation[#, "Radium226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2676],
        "\"Radium227\""],
       Annotation[#, "Radium227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2677],
        "\"Radium228\""],
       Annotation[#, "Radium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2678],
        "\"Francium229\""],
       Annotation[#, "Francium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2679],
        "\"Radium229\""],
       Annotation[#, "Radium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2680],
        "\"Francium230\""],
       Annotation[#, "Francium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2681],
        "\"Radium230\""],
       Annotation[#, "Radium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2682],
        "\"Francium231\""],
       Annotation[#, "Francium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2683],
        "\"Radium231\""],
       Annotation[#, "Radium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2684],
        "\"Francium232\""],
       Annotation[#, "Francium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2685],
        "\"Radium232\""],
       Annotation[#, "Radium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2686],
        "\"Radium202\""],
       Annotation[#, "Radium202", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2687],
        "\"Radium203\""],
       Annotation[#, "Radium203", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2688],
        "\"Radium204\""],
       Annotation[#, "Radium204", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2689],
        "\"Radium205\""],
       Annotation[#, "Radium205", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2690],
        "\"Radium206\""],
       Annotation[#, "Radium206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2691],
        "\"Radium207\""],
       Annotation[#, "Radium207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2692],
        "\"Radium208\""],
       Annotation[#, "Radium208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2693],
        "\"Radium209\""],
       Annotation[#, "Radium209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2694],
        "\"Radium210\""],
       Annotation[#, "Radium210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2695],
        "\"Radium211\""],
       Annotation[#, "Radium211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2696],
        "\"Radium212\""],
       Annotation[#, "Radium212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2697],
        "\"Radium213\""],
       Annotation[#, "Radium213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2698],
        "\"Radium214\""],
       Annotation[#, "Radium214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2699],
        "\"Radium215\""],
       Annotation[#, "Radium215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2700],
        "\"Radium216\""],
       Annotation[#, "Radium216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2701],
        "\"Radium217\""],
       Annotation[#, "Radium217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2702],
        "\"Radium218\""],
       Annotation[#, "Radium218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2703],
        "\"Radium219\""],
       Annotation[#, "Radium219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2704],
        "\"Actinium225\""],
       Annotation[#, "Actinium225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2705],
        "\"Thorium226\""],
       Annotation[#, "Thorium226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2706],
        "\"Actinium227\""],
       Annotation[#, "Actinium227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2707],
        "\"Actinium228\""],
       Annotation[#, "Actinium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2708],
        "\"Actinium229\""],
       Annotation[#, "Actinium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2709],
        "\"Actinium230\""],
       Annotation[#, "Actinium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2710],
        "\"Actinium231\""],
       Annotation[#, "Actinium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2711],
        "\"Actinium232\""],
       Annotation[#, "Actinium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2712],
        "\"Radium233\""],
       Annotation[#, "Radium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2713],
        "\"Actinium233\""],
       Annotation[#, "Actinium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2714],
        "\"Radium234\""],
       Annotation[#, "Radium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2715],
        "\"Actinium234\""],
       Annotation[#, "Actinium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2716],
        "\"Actinium206\""],
       Annotation[#, "Actinium206", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2717],
        "\"Actinium207\""],
       Annotation[#, "Actinium207", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2718],
        "\"Actinium208\""],
       Annotation[#, "Actinium208", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2719],
        "\"Actinium209\""],
       Annotation[#, "Actinium209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2720],
        "\"Actinium210\""],
       Annotation[#, "Actinium210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2721],
        "\"Actinium211\""],
       Annotation[#, "Actinium211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2722],
        "\"Actinium212\""],
       Annotation[#, "Actinium212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2723],
        "\"Actinium213\""],
       Annotation[#, "Actinium213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2724],
        "\"Actinium214\""],
       Annotation[#, "Actinium214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2725],
        "\"Actinium215\""],
       Annotation[#, "Actinium215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2726],
        "\"Actinium216\""],
       Annotation[#, "Actinium216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2727],
        "\"Actinium217\""],
       Annotation[#, "Actinium217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2728],
        "\"Actinium218\""],
       Annotation[#, "Actinium218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2729],
        "\"Actinium219\""],
       Annotation[#, "Actinium219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2730],
        "\"Actinium220\""],
       Annotation[#, "Actinium220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2731],
        "\"Actinium221\""],
       Annotation[#, "Actinium221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2732],
        "\"Actinium222\""],
       Annotation[#, "Actinium222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2733],
        "\"Actinium223\""],
       Annotation[#, "Actinium223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2734],
        "\"Actinium224\""],
       Annotation[#, "Actinium224", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2735],
        "\"Thorium224\""],
       Annotation[#, "Thorium224", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2736],
        "\"Actinium226\""],
       Annotation[#, "Actinium226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2737],
        "\"Thorium227\""],
       Annotation[#, "Thorium227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2738],
        "\"Thorium228\""],
       Annotation[#, "Thorium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2739],
        "\"Thorium229\""],
       Annotation[#, "Thorium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2740],
        "\"Thorium230\""],
       Annotation[#, "Thorium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2741],
        "\"Thorium231\""],
       Annotation[#, "Thorium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2742],
        "\"Thorium232\""],
       Annotation[#, "Thorium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2743],
        "\"Thorium233\""],
       Annotation[#, "Thorium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2744],
        "\"Thorium234\""],
       Annotation[#, "Thorium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2745],
        "\"Actinium235\""],
       Annotation[#, "Actinium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2746],
        "\"Thorium235\""],
       Annotation[#, "Thorium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2747],
        "\"Actinium236\""],
       Annotation[#, "Actinium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2748],
        "\"Thorium236\""],
       Annotation[#, "Thorium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2749],
        "\"Thorium209\""],
       Annotation[#, "Thorium209", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2750],
        "\"Thorium210\""],
       Annotation[#, "Thorium210", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2751],
        "\"Thorium211\""],
       Annotation[#, "Thorium211", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2752],
        "\"Thorium212\""],
       Annotation[#, "Thorium212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2753],
        "\"Thorium213\""],
       Annotation[#, "Thorium213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2754],
        "\"Thorium214\""],
       Annotation[#, "Thorium214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2755],
        "\"Thorium215\""],
       Annotation[#, "Thorium215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2756],
        "\"Thorium216\""],
       Annotation[#, "Thorium216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2757],
        "\"Thorium217\""],
       Annotation[#, "Thorium217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2758],
        "\"Thorium218\""],
       Annotation[#, "Thorium218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2759],
        "\"Thorium219\""],
       Annotation[#, "Thorium219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2760],
        "\"Thorium220\""],
       Annotation[#, "Thorium220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2761],
        "\"Thorium221\""],
       Annotation[#, "Thorium221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2762],
        "\"Thorium222\""],
       Annotation[#, "Thorium222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2763],
        "\"Thorium223\""],
       Annotation[#, "Thorium223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2764],
        "\"Thorium225\""],
       Annotation[#, "Thorium225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2765],
        "\"Protactinium231\""],
       Annotation[#, "Protactinium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2766],
        "\"Ytterbium182\""],
       Annotation[#, "Ytterbium182", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2767],
        "\"Uranium232\""],
       Annotation[#, "Uranium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2768],
        "\"Protactinium233\""],
       Annotation[#, "Protactinium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2769],
        "\"Protactinium234\""],
       Annotation[#, "Protactinium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2770],
        "\"Protactinium235\""],
       Annotation[#, "Protactinium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2771],
        "\"Protactinium236\""],
       Annotation[#, "Protactinium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2772],
        "\"Thorium237\""],
       Annotation[#, "Thorium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2773],
        "\"Protactinium237\""],
       Annotation[#, "Protactinium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2774],
        "\"Thorium238\""],
       Annotation[#, "Thorium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2775],
        "\"Protactinium238\""],
       Annotation[#, "Protactinium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2776],
        "\"Protactinium212\""],
       Annotation[#, "Protactinium212", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2777],
        "\"Protactinium213\""],
       Annotation[#, "Protactinium213", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2778],
        "\"Protactinium214\""],
       Annotation[#, "Protactinium214", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2779],
        "\"Protactinium215\""],
       Annotation[#, "Protactinium215", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2780],
        "\"Protactinium216\""],
       Annotation[#, "Protactinium216", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2781],
        "\"Protactinium217\""],
       Annotation[#, "Protactinium217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2782],
        "\"Protactinium218\""],
       Annotation[#, "Protactinium218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2783],
        "\"Protactinium219\""],
       Annotation[#, "Protactinium219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2784],
        "\"Protactinium220\""],
       Annotation[#, "Protactinium220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2785],
        "\"Protactinium221\""],
       Annotation[#, "Protactinium221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2786],
        "\"Protactinium222\""],
       Annotation[#, "Protactinium222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2787],
        "\"Protactinium223\""],
       Annotation[#, "Protactinium223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2788],
        "\"Protactinium224\""],
       Annotation[#, "Protactinium224", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2789],
        "\"Protactinium225\""],
       Annotation[#, "Protactinium225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2790],
        "\"Protactinium226\""],
       Annotation[#, "Protactinium226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2791],
        "\"Protactinium227\""],
       Annotation[#, "Protactinium227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2792],
        "\"Protactinium228\""],
       Annotation[#, "Protactinium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2793],
        "\"Protactinium229\""],
       Annotation[#, "Protactinium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2794],
        "\"Protactinium230\""],
       Annotation[#, "Protactinium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2795],
        "\"Uranium230\""],
       Annotation[#, "Uranium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2796],
        "\"Protactinium232\""],
       Annotation[#, "Protactinium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2797],
        "\"Uranium233\""],
       Annotation[#, "Uranium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2798],
        "\"Uranium234\""],
       Annotation[#, "Uranium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2799],
        "\"Uranium235\""],
       Annotation[#, "Uranium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2800],
        "\"Uranium236\""],
       Annotation[#, "Uranium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2801],
        "\"Uranium237\""],
       Annotation[#, "Uranium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2802],
        "\"Uranium238\""],
       Annotation[#, "Uranium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2803],
        "\"Protactinium239\""],
       Annotation[#, "Protactinium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2804],
        "\"Uranium239\""],
       Annotation[#, "Uranium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2805],
        "\"Protactinium240\""],
       Annotation[#, "Protactinium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2806],
        "\"Uranium240\""],
       Annotation[#, "Uranium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2807],
        "\"Uranium217\""],
       Annotation[#, "Uranium217", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2808],
        "\"Uranium218\""],
       Annotation[#, "Uranium218", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2809],
        "\"Uranium219\""],
       Annotation[#, "Uranium219", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2810],
        "\"Uranium220\""],
       Annotation[#, "Uranium220", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2811],
        "\"Uranium221\""],
       Annotation[#, "Uranium221", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2812],
        "\"Uranium222\""],
       Annotation[#, "Uranium222", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2813],
        "\"Uranium223\""],
       Annotation[#, "Uranium223", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2814],
        "\"Uranium224\""],
       Annotation[#, "Uranium224", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2815],
        "\"Uranium225\""],
       Annotation[#, "Uranium225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2816],
        "\"Uranium226\""],
       Annotation[#, "Uranium226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2817],
        "\"Uranium227\""],
       Annotation[#, "Uranium227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2818],
        "\"Uranium228\""],
       Annotation[#, "Uranium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2819],
        "\"Uranium229\""],
       Annotation[#, "Uranium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2820],
        "\"Uranium231\""],
       Annotation[#, "Uranium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2821],
        "\"Neptunium237\""],
       Annotation[#, "Neptunium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2822],
        "\"Plutonium238\""],
       Annotation[#, "Plutonium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2823],
        "\"Neptunium239\""],
       Annotation[#, "Neptunium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2824],
        "\"Neptunium240\""],
       Annotation[#, "Neptunium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2825],
        "\"Uranium241\""],
       Annotation[#, "Uranium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2826],
        "\"Neptunium241\""],
       Annotation[#, "Neptunium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2827],
        "\"Uranium242\""],
       Annotation[#, "Uranium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2828],
        "\"Neptunium242\""],
       Annotation[#, "Neptunium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2829],
        "\"Neptunium225\""],
       Annotation[#, "Neptunium225", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2830],
        "\"Neptunium226\""],
       Annotation[#, "Neptunium226", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2831],
        "\"Neptunium227\""],
       Annotation[#, "Neptunium227", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2832],
        "\"Neptunium228\""],
       Annotation[#, "Neptunium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2833],
        "\"Neptunium229\""],
       Annotation[#, "Neptunium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2834],
        "\"Neptunium230\""],
       Annotation[#, "Neptunium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2835],
        "\"Neptunium231\""],
       Annotation[#, "Neptunium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2836],
        "\"Neptunium232\""],
       Annotation[#, "Neptunium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2837],
        "\"Neptunium233\""],
       Annotation[#, "Neptunium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2838],
        "\"Neptunium234\""],
       Annotation[#, "Neptunium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2839],
        "\"Neptunium235\""],
       Annotation[#, "Neptunium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2840],
        "\"Neptunium236\""],
       Annotation[#, "Neptunium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2841],
        "\"Plutonium236\""],
       Annotation[#, "Plutonium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2842],
        "\"Neptunium238\""],
       Annotation[#, "Neptunium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2843],
        "\"Plutonium239\""],
       Annotation[#, "Plutonium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2844],
        "\"Plutonium240\""],
       Annotation[#, "Plutonium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2845],
        "\"Plutonium241\""],
       Annotation[#, "Plutonium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2846],
        "\"Plutonium242\""],
       Annotation[#, "Plutonium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2847],
        "\"Neptunium243\""],
       Annotation[#, "Neptunium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2848],
        "\"Plutonium243\""],
       Annotation[#, "Plutonium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2849],
        "\"Neptunium244\""],
       Annotation[#, "Neptunium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2850],
        "\"Plutonium244\""],
       Annotation[#, "Plutonium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2851],
        "\"Plutonium228\""],
       Annotation[#, "Plutonium228", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2852],
        "\"Plutonium229\""],
       Annotation[#, "Plutonium229", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2853],
        "\"Plutonium230\""],
       Annotation[#, "Plutonium230", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2854],
        "\"Plutonium231\""],
       Annotation[#, "Plutonium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2855],
        "\"Plutonium232\""],
       Annotation[#, "Plutonium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2856],
        "\"Plutonium233\""],
       Annotation[#, "Plutonium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2857],
        "\"Plutonium234\""],
       Annotation[#, "Plutonium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2858],
        "\"Plutonium235\""],
       Annotation[#, "Plutonium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2859],
        "\"Plutonium237\""],
       Annotation[#, "Plutonium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2860],
        "\"Americium241\""],
       Annotation[#, "Americium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2861],
        "\"Americium243\""],
       Annotation[#, "Americium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2862],
        "\"Curium244\""],
       Annotation[#, "Curium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2863],
        "\"Plutonium245\""],
       Annotation[#, "Plutonium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2864],
        "\"Americium245\""],
       Annotation[#, "Americium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2865],
        "\"Plutonium246\""],
       Annotation[#, "Plutonium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2866],
        "\"Americium246\""],
       Annotation[#, "Americium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2867],
        "\"Plutonium247\""],
       Annotation[#, "Plutonium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2868],
        "\"Americium247\""],
       Annotation[#, "Americium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2869],
        "\"Americium231\""],
       Annotation[#, "Americium231", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2870],
        "\"Americium232\""],
       Annotation[#, "Americium232", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2871],
        "\"Americium233\""],
       Annotation[#, "Americium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2872],
        "\"Americium234\""],
       Annotation[#, "Americium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2873],
        "\"Americium235\""],
       Annotation[#, "Americium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2874],
        "\"Americium236\""],
       Annotation[#, "Americium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2875],
        "\"Americium237\""],
       Annotation[#, "Americium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2876],
        "\"Americium238\""],
       Annotation[#, "Americium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2877],
        "\"Americium239\""],
       Annotation[#, "Americium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2878],
        "\"Americium240\""],
       Annotation[#, "Americium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2879],
        "\"Americium242\""],
       Annotation[#, "Americium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2880],
        "\"Curium242\""],
       Annotation[#, "Curium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2881],
        "\"Americium244\""],
       Annotation[#, "Americium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2882],
        "\"Curium245\""],
       Annotation[#, "Curium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2883],
        "\"Curium246\""],
       Annotation[#, "Curium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2884],
        "\"Curium247\""],
       Annotation[#, "Curium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2885],
        "\"Americium248\""],
       Annotation[#, "Americium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2886],
        "\"Curium248\""],
       Annotation[#, "Curium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2887],
        "\"Americium249\""],
       Annotation[#, "Americium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2888],
        "\"Curium249\""],
       Annotation[#, "Curium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2889],
        "\"Curium233\""],
       Annotation[#, "Curium233", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2890],
        "\"Curium234\""],
       Annotation[#, "Curium234", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2891],
        "\"Curium235\""],
       Annotation[#, "Curium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2892],
        "\"Curium236\""],
       Annotation[#, "Curium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2893],
        "\"Curium237\""],
       Annotation[#, "Curium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2894],
        "\"Curium238\""],
       Annotation[#, "Curium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2895],
        "\"Curium239\""],
       Annotation[#, "Curium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2896],
        "\"Curium240\""],
       Annotation[#, "Curium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2897],
        "\"Curium241\""],
       Annotation[#, "Curium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2898],
        "\"Curium243\""],
       Annotation[#, "Curium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2899],
        "\"Californium248\""],
       Annotation[#, "Californium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2900],
        "\"Berkelium249\""],
       Annotation[#, "Berkelium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2901],
        "\"Curium250\""],
       Annotation[#, "Curium250", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2902],
        "\"Berkelium250\""],
       Annotation[#, "Berkelium250", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2903],
        "\"Curium251\""],
       Annotation[#, "Curium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2904],
        "\"Berkelium251\""],
       Annotation[#, "Berkelium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2905],
        "\"Curium252\""],
       Annotation[#, "Curium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2906],
        "\"Berkelium252\""],
       Annotation[#, "Berkelium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2907],
        "\"Berkelium235\""],
       Annotation[#, "Berkelium235", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2908],
        "\"Berkelium236\""],
       Annotation[#, "Berkelium236", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2909],
        "\"Berkelium237\""],
       Annotation[#, "Berkelium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2910],
        "\"Berkelium238\""],
       Annotation[#, "Berkelium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2911],
        "\"Berkelium239\""],
       Annotation[#, "Berkelium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2912],
        "\"Berkelium240\""],
       Annotation[#, "Berkelium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2913],
        "\"Berkelium241\""],
       Annotation[#, "Berkelium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2914],
        "\"Berkelium242\""],
       Annotation[#, "Berkelium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2915],
        "\"Berkelium243\""],
       Annotation[#, "Berkelium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2916],
        "\"Berkelium244\""],
       Annotation[#, "Berkelium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2917],
        "\"Berkelium245\""],
       Annotation[#, "Berkelium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2918],
        "\"Berkelium246\""],
       Annotation[#, "Berkelium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2919],
        "\"Berkelium247\""],
       Annotation[#, "Berkelium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2920],
        "\"Berkelium248\""],
       Annotation[#, "Berkelium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2921],
        "\"Californium249\""],
       Annotation[#, "Californium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2922],
        "\"Californium250\""],
       Annotation[#, "Californium250", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2923],
        "\"Californium251\""],
       Annotation[#, "Californium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2924],
        "\"Californium252\""],
       Annotation[#, "Californium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2925],
        "\"Berkelium253\""],
       Annotation[#, "Berkelium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2926],
        "\"Californium253\""],
       Annotation[#, "Californium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2927],
        "\"Berkelium254\""],
       Annotation[#, "Berkelium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2928],
        "\"Californium254\""],
       Annotation[#, "Californium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2929],
        "\"Californium237\""],
       Annotation[#, "Californium237", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2930],
        "\"Californium238\""],
       Annotation[#, "Californium238", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2931],
        "\"Californium239\""],
       Annotation[#, "Californium239", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2932],
        "\"Californium240\""],
       Annotation[#, "Californium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2933],
        "\"Californium241\""],
       Annotation[#, "Californium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2934],
        "\"Californium242\""],
       Annotation[#, "Californium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2935],
        "\"Californium243\""],
       Annotation[#, "Californium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2936],
        "\"Californium244\""],
       Annotation[#, "Californium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2937],
        "\"Californium245\""],
       Annotation[#, "Californium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2938],
        "\"Californium246\""],
       Annotation[#, "Californium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2939],
        "\"Californium247\""],
       Annotation[#, "Californium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2940],
        "\"Einsteinium253\""],
       Annotation[#, "Einsteinium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2941],
        "\"Fermium254\""],
       Annotation[#, "Fermium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2942],
        "\"Californium255\""],
       Annotation[#, "Californium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2943],
        "\"Einsteinium255\""],
       Annotation[#, "Einsteinium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2944],
        "\"Californium256\""],
       Annotation[#, "Californium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2945],
        "\"Fermium256\""],
       Annotation[#, "Fermium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2946],
        "\"Einsteinium240\""],
       Annotation[#, "Einsteinium240", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2947],
        "\"Einsteinium241\""],
       Annotation[#, "Einsteinium241", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2948],
        "\"Einsteinium242\""],
       Annotation[#, "Einsteinium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2949],
        "\"Einsteinium243\""],
       Annotation[#, "Einsteinium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2950],
        "\"Einsteinium244\""],
       Annotation[#, "Einsteinium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2951],
        "\"Einsteinium245\""],
       Annotation[#, "Einsteinium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2952],
        "\"Einsteinium246\""],
       Annotation[#, "Einsteinium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2953],
        "\"Einsteinium247\""],
       Annotation[#, "Einsteinium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2954],
        "\"Einsteinium248\""],
       Annotation[#, "Einsteinium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2955],
        "\"Einsteinium249\""],
       Annotation[#, "Einsteinium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2956],
        "\"Einsteinium250\""],
       Annotation[#, "Einsteinium250", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2957],
        "\"Einsteinium251\""],
       Annotation[#, "Einsteinium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2958],
        "\"Einsteinium252\""],
       Annotation[#, "Einsteinium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2959],
        "\"Einsteinium254\""],
       Annotation[#, "Einsteinium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2960],
        "\"Fermium255\""],
       Annotation[#, "Fermium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2961],
        "\"Einsteinium256\""],
       Annotation[#, "Einsteinium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2962],
        "\"Einsteinium257\""],
       Annotation[#, "Einsteinium257", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2963],
        "\"Fermium257\""],
       Annotation[#, "Fermium257", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2964],
        "\"Einsteinium258\""],
       Annotation[#, "Einsteinium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2965],
        "\"Fermium258\""],
       Annotation[#, "Fermium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2966],
        "\"Fermium242\""],
       Annotation[#, "Fermium242", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2967],
        "\"Fermium243\""],
       Annotation[#, "Fermium243", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2968],
        "\"Fermium244\""],
       Annotation[#, "Fermium244", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2969],
        "\"Fermium245\""],
       Annotation[#, "Fermium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2970],
        "\"Fermium246\""],
       Annotation[#, "Fermium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2971],
        "\"Fermium247\""],
       Annotation[#, "Fermium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2972],
        "\"Fermium248\""],
       Annotation[#, "Fermium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2973],
        "\"Fermium249\""],
       Annotation[#, "Fermium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2974],
        "\"Fermium250\""],
       Annotation[#, "Fermium250", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2975],
        "\"Fermium251\""],
       Annotation[#, "Fermium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2976],
        "\"Fermium252\""],
       Annotation[#, "Fermium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2977],
        "\"Fermium253\""],
       Annotation[#, "Fermium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2978],
        "\"Fermium259\""],
       Annotation[#, "Fermium259", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2979],
        "\"Fermium260\""],
       Annotation[#, "Fermium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2980],
        "\"Mendelevium245\""],
       Annotation[#, "Mendelevium245", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2981],
        "\"Mendelevium246\""],
       Annotation[#, "Mendelevium246", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2982],
        "\"Mendelevium247\""],
       Annotation[#, "Mendelevium247", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2983],
        "\"Mendelevium248\""],
       Annotation[#, "Mendelevium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2984],
        "\"Mendelevium249\""],
       Annotation[#, "Mendelevium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2985],
        "\"Mendelevium250\""],
       Annotation[#, "Mendelevium250", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2986],
        "\"Mendelevium251\""],
       Annotation[#, "Mendelevium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2987],
        "\"Mendelevium252\""],
       Annotation[#, "Mendelevium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2988],
        "\"Mendelevium253\""],
       Annotation[#, "Mendelevium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2989],
        "\"Mendelevium254\""],
       Annotation[#, "Mendelevium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2990],
        "\"Mendelevium255\""],
       Annotation[#, "Mendelevium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2991],
        "\"Mendelevium256\""],
       Annotation[#, "Mendelevium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2992],
        "\"Mendelevium257\""],
       Annotation[#, "Mendelevium257", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2993],
        "\"Mendelevium258\""],
       Annotation[#, "Mendelevium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2994],
        "\"Nobelium258\""],
       Annotation[#, "Nobelium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2995],
        "\"Mendelevium259\""],
       Annotation[#, "Mendelevium259", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2996],
        "\"Mendelevium260\""],
       Annotation[#, "Mendelevium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2997],
        "\"Nobelium260\""],
       Annotation[#, "Nobelium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2998],
        "\"Mendelevium261\""],
       Annotation[#, "Mendelevium261", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2999],
        "\"Mendelevium262\""],
       Annotation[#, "Mendelevium262", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3000],
        "\"Nobelium248\""],
       Annotation[#, "Nobelium248", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3001],
        "\"Nobelium249\""],
       Annotation[#, "Nobelium249", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3002],
        "\"Nobelium250\""],
       Annotation[#, "Nobelium250", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3003],
        "\"Nobelium251\""],
       Annotation[#, "Nobelium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3004],
        "\"Nobelium252\""],
       Annotation[#, "Nobelium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3005],
        "\"Nobelium253\""],
       Annotation[#, "Nobelium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3006],
        "\"Nobelium254\""],
       Annotation[#, "Nobelium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3007],
        "\"Nobelium255\""],
       Annotation[#, "Nobelium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3008],
        "\"Nobelium256\""],
       Annotation[#, "Nobelium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3009],
        "\"Nobelium257\""],
       Annotation[#, "Nobelium257", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3010],
        "\"Nobelium259\""],
       Annotation[#, "Nobelium259", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3011],
        "\"Nobelium261\""],
       Annotation[#, "Nobelium261", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3012],
        "\"Nobelium262\""],
       Annotation[#, "Nobelium262", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3013],
        "\"Nobelium263\""],
       Annotation[#, "Nobelium263", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3014],
        "\"Nobelium264\""],
       Annotation[#, "Nobelium264", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3015],
        "\"Lawrencium251\""],
       Annotation[#, "Lawrencium251", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3016],
        "\"Lawrencium252\""],
       Annotation[#, "Lawrencium252", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3017],
        "\"Lawrencium253\""],
       Annotation[#, "Lawrencium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3018],
        "\"Lawrencium254\""],
       Annotation[#, "Lawrencium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3019],
        "\"Lawrencium255\""],
       Annotation[#, "Lawrencium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3020],
        "\"Lawrencium256\""],
       Annotation[#, "Lawrencium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3021],
        "\"Lawrencium257\""],
       Annotation[#, "Lawrencium257", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3022],
        "\"Lawrencium258\""],
       Annotation[#, "Lawrencium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3023],
        "\"Lawrencium259\""],
       Annotation[#, "Lawrencium259", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3024],
        "\"Lawrencium260\""],
       Annotation[#, "Lawrencium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3025],
        "\"Lawrencium261\""],
       Annotation[#, "Lawrencium261", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3026],
        "\"Lawrencium262\""],
       Annotation[#, "Lawrencium262", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3027],
        "\"Lawrencium263\""],
       Annotation[#, "Lawrencium263", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3028],
        "\"Lawrencium264\""],
       Annotation[#, "Lawrencium264", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3029],
        "\"Lawrencium265\""],
       Annotation[#, "Lawrencium265", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3030],
        "\"Lawrencium266\""],
       Annotation[#, "Lawrencium266", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3031],
        "\"Rutherfordium253\""],
       Annotation[#, "Rutherfordium253", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3032],
        "\"Rutherfordium254\""],
       Annotation[#, "Rutherfordium254", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3033],
        "\"Rutherfordium255\""],
       Annotation[#, "Rutherfordium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3034],
        "\"Rutherfordium256\""],
       Annotation[#, "Rutherfordium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3035],
        "\"Rutherfordium257\""],
       Annotation[#, "Rutherfordium257", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3036],
        "\"Rutherfordium258\""],
       Annotation[#, "Rutherfordium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3037],
        "\"Rutherfordium259\""],
       Annotation[#, "Rutherfordium259", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3038],
        "\"Rutherfordium260\""],
       Annotation[#, "Rutherfordium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3039],
        "\"Rutherfordium261\""],
       Annotation[#, "Rutherfordium261", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3040],
        "\"Rutherfordium262\""],
       Annotation[#, "Rutherfordium262", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3041],
        "\"Rutherfordium263\""],
       Annotation[#, "Rutherfordium263", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3042],
        "\"Rutherfordium264\""],
       Annotation[#, "Rutherfordium264", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3043],
        "\"Rutherfordium265\""],
       Annotation[#, "Rutherfordium265", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3044],
        "\"Rutherfordium266\""],
       Annotation[#, "Rutherfordium266", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3045],
        "\"Rutherfordium267\""],
       Annotation[#, "Rutherfordium267", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3046],
        "\"Rutherfordium268\""],
       Annotation[#, "Rutherfordium268", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3047],
        "\"Dubnium255\""],
       Annotation[#, "Dubnium255", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3048],
        "\"Dubnium256\""],
       Annotation[#, "Dubnium256", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3049],
        "\"Dubnium257\""],
       Annotation[#, "Dubnium257", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3050],
        "\"Dubnium258\""],
       Annotation[#, "Dubnium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3051],
        "\"Dubnium259\""],
       Annotation[#, "Dubnium259", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3052],
        "\"Dubnium260\""],
       Annotation[#, "Dubnium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3053],
        "\"Dubnium261\""],
       Annotation[#, "Dubnium261", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3054],
        "\"Dubnium262\""],
       Annotation[#, "Dubnium262", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3055],
        "\"Dubnium263\""],
       Annotation[#, "Dubnium263", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3056],
        "\"Dubnium264\""],
       Annotation[#, "Dubnium264", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3057],
        "\"Dubnium265\""],
       Annotation[#, "Dubnium265", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3058],
        "\"Dubnium266\""],
       Annotation[#, "Dubnium266", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3059],
        "\"Dubnium267\""],
       Annotation[#, "Dubnium267", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3060],
        "\"Dubnium268\""],
       Annotation[#, "Dubnium268", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3061],
        "\"Dubnium269\""],
       Annotation[#, "Dubnium269", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3062],
        "\"Dubnium270\""],
       Annotation[#, "Dubnium270", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3063],
        "\"Seaborgium258\""],
       Annotation[#, "Seaborgium258", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3064],
        "\"Seaborgium259\""],
       Annotation[#, "Seaborgium259", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3065],
        "\"Seaborgium260\""],
       Annotation[#, "Seaborgium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3066],
        "\"Seaborgium261\""],
       Annotation[#, "Seaborgium261", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3067],
        "\"Seaborgium262\""],
       Annotation[#, "Seaborgium262", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3068],
        "\"Seaborgium263\""],
       Annotation[#, "Seaborgium263", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3069],
        "\"Seaborgium264\""],
       Annotation[#, "Seaborgium264", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3070],
        "\"Seaborgium265\""],
       Annotation[#, "Seaborgium265", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3071],
        "\"Seaborgium266\""],
       Annotation[#, "Seaborgium266", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3072],
        "\"Seaborgium267\""],
       Annotation[#, "Seaborgium267", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3073],
        "\"Seaborgium268\""],
       Annotation[#, "Seaborgium268", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3074],
        "\"Seaborgium269\""],
       Annotation[#, "Seaborgium269", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3075],
        "\"Seaborgium270\""],
       Annotation[#, "Seaborgium270", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3076],
        "\"Seaborgium271\""],
       Annotation[#, "Seaborgium271", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3077],
        "\"Seaborgium272\""],
       Annotation[#, "Seaborgium272", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3078],
        "\"Seaborgium273\""],
       Annotation[#, "Seaborgium273", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3079],
        "\"Bohrium260\""],
       Annotation[#, "Bohrium260", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3080],
        "\"Bohrium261\""],
       Annotation[#, "Bohrium261", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3081],
        "\"Bohrium262\""],
       Annotation[#, "Bohrium262", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3082],
        "\"Bohrium263\""],
       Annotation[#, "Bohrium263", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3083],
        "\"Bohrium264\""],
       Annotation[#, "Bohrium264", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3084],
        "\"Bohrium265\""],
       Annotation[#, "Bohrium265", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3085],
        "\"Bohrium266\""],
       Annotation[#, "Bohrium266", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3086],
        "\"Bohrium267\""],
       Annotation[#, "Bohrium267", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3087],
        "\"Bohrium268\""],
       Annotation[#, "Bohrium268", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3088],
        "\"Bohrium269\""],
       Annotation[#, "Bohrium269", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3089],
        "\"Bohrium270\""],
       Annotation[#, "Bohrium270", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3090],
        "\"Bohrium271\""],
       Annotation[#, "Bohrium271", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3091],
        "\"Bohrium272\""],
       Annotation[#, "Bohrium272", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3092],
        "\"Bohrium273\""],
       Annotation[#, "Bohrium273", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3093],
        "\"Bohrium274\""],
       Annotation[#, "Bohrium274", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3094],
        "\"Bohrium275\""],
       Annotation[#, "Bohrium275", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3095],
        "\"Hassium263\""],
       Annotation[#, "Hassium263", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3096],
        "\"Hassium264\""],
       Annotation[#, "Hassium264", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3097],
        "\"Hassium265\""],
       Annotation[#, "Hassium265", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3098],
        "\"Hassium266\""],
       Annotation[#, "Hassium266", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3099],
        "\"Hassium267\""],
       Annotation[#, "Hassium267", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3100],
        "\"Hassium268\""],
       Annotation[#, "Hassium268", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3101],
        "\"Hassium269\""],
       Annotation[#, "Hassium269", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3102],
        "\"Hassium270\""],
       Annotation[#, "Hassium270", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3103],
        "\"Hassium271\""],
       Annotation[#, "Hassium271", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3104],
        "\"Hassium272\""],
       Annotation[#, "Hassium272", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3105],
        "\"Hassium273\""],
       Annotation[#, "Hassium273", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3106],
        "\"Hassium274\""],
       Annotation[#, "Hassium274", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3107],
        "\"Hassium275\""],
       Annotation[#, "Hassium275", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3108],
        "\"Hassium276\""],
       Annotation[#, "Hassium276", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3109],
        "\"Hassium277\""],
       Annotation[#, "Hassium277", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3110],
        "\"Meitnerium265\""],
       Annotation[#, "Meitnerium265", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3111],
        "\"Meitnerium266\""],
       Annotation[#, "Meitnerium266", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3112],
        "\"Meitnerium267\""],
       Annotation[#, "Meitnerium267", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3113],
        "\"Meitnerium268\""],
       Annotation[#, "Meitnerium268", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3114],
        "\"Meitnerium269\""],
       Annotation[#, "Meitnerium269", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3115],
        "\"Meitnerium270\""],
       Annotation[#, "Meitnerium270", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3116],
        "\"Meitnerium271\""],
       Annotation[#, "Meitnerium271", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3117],
        "\"Meitnerium272\""],
       Annotation[#, "Meitnerium272", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3118],
        "\"Meitnerium273\""],
       Annotation[#, "Meitnerium273", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3119],
        "\"Meitnerium274\""],
       Annotation[#, "Meitnerium274", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3120],
        "\"Meitnerium275\""],
       Annotation[#, "Meitnerium275", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3121],
        "\"Meitnerium276\""],
       Annotation[#, "Meitnerium276", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3122],
        "\"Meitnerium277\""],
       Annotation[#, "Meitnerium277", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3123],
        "\"Meitnerium278\""],
       Annotation[#, "Meitnerium278", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3124],
        "\"Meitnerium279\""],
       Annotation[#, "Meitnerium279", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3125],
        "\"Darmstadtium267\""],
       Annotation[#, "Darmstadtium267", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3126],
        "\"Darmstadtium268\""],
       Annotation[#, "Darmstadtium268", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3127],
        "\"Darmstadtium269\""],
       Annotation[#, "Darmstadtium269", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3128],
        "\"Darmstadtium270\""],
       Annotation[#, "Darmstadtium270", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3129],
        "\"Darmstadtium271\""],
       Annotation[#, "Darmstadtium271", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3130],
        "\"Darmstadtium272\""],
       Annotation[#, "Darmstadtium272", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3131],
        "\"Darmstadtium273\""],
       Annotation[#, "Darmstadtium273", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3132],
        "\"Darmstadtium274\""],
       Annotation[#, "Darmstadtium274", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3133],
        "\"Darmstadtium275\""],
       Annotation[#, "Darmstadtium275", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3134],
        "\"Darmstadtium276\""],
       Annotation[#, "Darmstadtium276", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3135],
        "\"Darmstadtium277\""],
       Annotation[#, "Darmstadtium277", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3136],
        "\"Darmstadtium278\""],
       Annotation[#, "Darmstadtium278", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3137],
        "\"Darmstadtium279\""],
       Annotation[#, "Darmstadtium279", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3138],
        "\"Darmstadtium280\""],
       Annotation[#, "Darmstadtium280", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3139],
        "\"Darmstadtium281\""],
       Annotation[#, "Darmstadtium281", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3140],
        "\"Roentgenium272\""],
       Annotation[#, "Roentgenium272", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3141],
        "\"Roentgenium273\""],
       Annotation[#, "Roentgenium273", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3142],
        "\"Roentgenium274\""],
       Annotation[#, "Roentgenium274", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3143],
        "\"Roentgenium275\""],
       Annotation[#, "Roentgenium275", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3144],
        "\"Roentgenium276\""],
       Annotation[#, "Roentgenium276", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3145],
        "\"Roentgenium277\""],
       Annotation[#, "Roentgenium277", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3146],
        "\"Roentgenium278\""],
       Annotation[#, "Roentgenium278", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3147],
        "\"Roentgenium279\""],
       Annotation[#, "Roentgenium279", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3148],
        "\"Roentgenium280\""],
       Annotation[#, "Roentgenium280", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3149],
        "\"Roentgenium281\""],
       Annotation[#, "Roentgenium281", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3150],
        "\"Roentgenium282\""],
       Annotation[#, "Roentgenium282", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3151],
        "\"Roentgenium283\""],
       Annotation[#, "Roentgenium283", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3152],
        "\"Ununbium277\""],
       Annotation[#, "Ununbium277", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3153],
        "\"Ununbium278\""],
       Annotation[#, "Ununbium278", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3154],
        "\"Ununbium279\""],
       Annotation[#, "Ununbium279", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3155],
        "\"Ununbium280\""],
       Annotation[#, "Ununbium280", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3156],
        "\"Ununbium281\""],
       Annotation[#, "Ununbium281", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3157],
        "\"Ununbium282\""],
       Annotation[#, "Ununbium282", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3158],
        "\"Ununbium283\""],
       Annotation[#, "Ununbium283", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3159],
        "\"Ununbium284\""],
       Annotation[#, "Ununbium284", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3160],
        "\"Ununbium285\""],
       Annotation[#, "Ununbium285", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3161],
        "\"Ununtrium283\""],
       Annotation[#, "Ununtrium283", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3162],
        "\"Ununtrium284\""],
       Annotation[#, "Ununtrium284", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3163],
        "\"Ununtrium285\""],
       Annotation[#, "Ununtrium285", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3164],
        "\"Ununtrium286\""],
       Annotation[#, "Ununtrium286", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3165],
        "\"Ununtrium287\""],
       Annotation[#, "Ununtrium287", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3166],
        "\"Ununquadium285\""],
       Annotation[#, "Ununquadium285", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3167],
        "\"Ununquadium286\""],
       Annotation[#, "Ununquadium286", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3168],
        "\"Ununquadium287\""],
       Annotation[#, "Ununquadium287", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3169],
        "\"Ununquadium288\""],
       Annotation[#, "Ununquadium288", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3170],
        "\"Ununquadium289\""],
       Annotation[#, "Ununquadium289", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3171],
        "\"Ununpentium287\""],
       Annotation[#, "Ununpentium287", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3172],
        "\"Ununpentium288\""],
       Annotation[#, "Ununpentium288", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3173],
        "\"Ununpentium289\""],
       Annotation[#, "Ununpentium289", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3174],
        "\"Ununpentium290\""],
       Annotation[#, "Ununpentium290", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3175],
        "\"Ununpentium291\""],
       Annotation[#, "Ununpentium291", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3176],
        "\"Ununhexium289\""],
       Annotation[#, "Ununhexium289", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3177],
        "\"Ununhexium290\""],
       Annotation[#, "Ununhexium290", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3178],
        "\"Ununhexium291\""],
       Annotation[#, "Ununhexium291", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3179],
        "\"Ununhexium292\""],
       Annotation[#, "Ununhexium292", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3180],
        "\"Ununseptium291\""],
       Annotation[#, "Ununseptium291", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3181],
        "\"Ununseptium292\""],
       Annotation[#, "Ununseptium292", "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3182],
        "\"Ununoctium293\""],
       Annotation[#, "Ununoctium293", "Tooltip"]& ]}}],
   Annotation[#, VertexCoordinateRules -> CompressedData["
1:eJxEe3k01G/YPiVCtiwpUsiSSpJ11nvGUqGFFkUYZsYyM4YQbZYoKpFKSEj7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     "]]& ],
  AspectRatio->Automatic,
  FrameTicks->None,
  ImageSize->{581., Automatic},
  PlotRange->All,
  PlotRangePadding->Scaled[0.1]]], "Output",
 ImageSize->{585, 151},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1535534647]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ElementData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ElementData"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChemicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChemicalData"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ParticleData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ParticleData"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[{
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/InternetConnectivity"],
  FontSlant->"Italic"],
 ButtonBox[" Internet Connectivity",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/InternetConnectivity"]
}], "Tutorials",
 CellID->341631938]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->1584193535],

Cell[TextData[ButtonBox["IsotopeData Source Information",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
IsotopeDataSourceInformation.html"], None}]], "RelatedLinks",
 CellID->1038487239]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Integrated Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DataCollections"]], "MoreAbout",
 CellID->3403213]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"IsotopeData - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 51, 17.5749308}", "context" -> "System`", 
    "keywords" -> {
     "abbreviation", "actinium", "alpha decay", "aluminum", "americium", 
      "antimony", "argon", "arsenic", "astatine", "atomic mass", 
      "atomic mass units", "atomic number", "barium", "berkelium", 
      "beryllium", "binding energy", "bismuth", "bohrium", "boron", "boson", 
      "bosons", "branching ratio", "branching ratios", "bromine", "cadmium", 
      "calcium", "californium", "carbon", "cerium", "cesium", "chlorine", 
      "chromium", "cobalt", "copper", "curium", "darmstadtium", 
      "daughter nuclide", "daughter nuclides", "decay energies", 
      "decay energy", "decay mode", "decay mode symbol", "decay mode symbols",
       "decay modes", "decay time", "decay width", "deuterium", "dubnium", 
      "dysprosium", "einsteinium", "electric quadrupole moment", "erbium", 
      "europium", "excited state energies", "excited state energy", 
      "excited state lifetime", "excited state lifetimes", 
      "excited state parities", "excited state parity", "excited state spin", 
      "excited state spins", "fermion", "fermions", "fermium", "fluorine", 
      "francium", "gadolinium", "gallium", "germanium", "gold", "hafnium", 
      "half-life", "half life", "hassium", "helium", "holmium", "hydrogen", 
      "indium", "iodine", "iridium", "iron", "isotope abundance", "krypton", 
      "lanthanum", "lawrencium", "lead", "lifetime", "lithium", "lutetium", 
      "magnesium", "magnetic moment", "manganese", "mass excess", 
      "mass number", "mass of isotope", "meitnerium", "mendelevium", 
      "mercury", "molybdenum", "neodymium", "neon", "neptunium", "neutron", 
      "neutron number", "neutrons", "nickel", "niobium", "nitrogen", 
      "nobelium", "nuclear data", "nuclear isomers", "nuclear magnetons", 
      "nuclei", "nuclide", "osmium", "oxygen", "palladium", "parity", 
      "phosphorus", "platinum", "plutonium", "polonium", "potassium", 
      "praseodymium", "promethium", "protactinium", "protons", 
      "quadrupole moment", "quantum statistics", "radioactive decay", 
      "radioactivity", "radium", "radon", "relative abundance", "rhenium", 
      "rhodium", "roentgenium", "rubidium", "ruthenium", "rutherfordium", 
      "samarium", "scandium", "seaborgium", "selenium", "silicon", "silver", 
      "sodium", "spin", "stable isotope", "strontium", "sulfur", "symbol", 
      "tantalum", "technetium", "tellurium", "terbium", "thallium", "thorium",
       "thulium", "tin", "titanium", "tritium", "tungsten", 
      "unstable isotope", "ununbium", "ununhexium", "ununoctium", 
      "ununpentium", "ununquadium", "ununseptium", "ununtrium", "uranium", 
      "vanadium", "xenon", "ytterbium", "yttrium", "zinc", "zirconium"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "IsotopeData[{Z, A}, \" property\"] gives the value of the specified \
property for the isotope with atomic number Z and mass number A. IsotopeData[\
\"name\", \" property\"] gives the value of the property for the named \
isotope.", "synonyms" -> {"isotope data"}, "title" -> "IsotopeData", "type" -> 
    "Symbol", "uri" -> "ref/IsotopeData"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[22710, 665, 358, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->2939395]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 769532, 20210}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2010, 50, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2633, 77, 1576, 37, 70, "ObjectNameGrid"],
Cell[4212, 116, 1274, 39, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5523, 160, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6001, 177, 530, 17, 70, "Notes",
 CellID->823808400],
Cell[6534, 196, 263, 9, 70, "Notes",
 CellID->485978669],
Cell[6800, 207, 323, 12, 70, "Notes",
 CellID->14574448],
Cell[7126, 221, 529, 17, 70, "Notes",
 CellID->691884323],
Cell[7658, 240, 649, 25, 70, "Notes",
 CellID->530731578],
Cell[8310, 267, 286, 9, 70, "Notes",
 CellID->54231339],
Cell[8599, 278, 321, 11, 70, "Notes",
 CellID->48425469],
Cell[8923, 291, 746, 23, 70, "Notes",
 CellID->193117324],
Cell[9672, 316, 188, 4, 70, "Notes",
 CellID->286491208],
Cell[9863, 322, 70, 1, 70, "Notes",
 CellID->156536174],
Cell[9936, 325, 1504, 34, 70, "2ColumnTableMod",
 CellID->120063746],
Cell[11443, 361, 87, 1, 70, "Notes",
 CellID->729383543],
Cell[11533, 364, 901, 21, 70, "2ColumnTableMod",
 CellID->912550125],
Cell[12437, 387, 71, 1, 70, "Notes",
 CellID->1093015083],
Cell[12511, 390, 2027, 48, 70, "2ColumnTableMod",
 CellID->2049227648],
Cell[14541, 440, 661, 20, 70, "Notes",
 CellID->215660396],
Cell[15205, 462, 129, 4, 70, "Notes",
 CellID->506275057],
Cell[15337, 468, 74, 1, 70, "Notes",
 CellID->270216217],
Cell[15414, 471, 1037, 19, 70, "2ColumnTableMod",
 CellID->981007280],
Cell[16454, 492, 70, 1, 70, "Notes",
 CellID->204992243],
Cell[16527, 495, 943, 24, 70, "2ColumnTableMod",
 CellID->711624400],
Cell[17473, 521, 941, 31, 70, "Notes",
 CellID->482514198],
Cell[18417, 554, 64, 1, 70, "Notes",
 CellID->91622357],
Cell[18484, 557, 2049, 48, 70, "2ColumnTableMod",
 CellID->1173142],
Cell[20536, 607, 456, 13, 70, "Notes",
 CellID->276644756],
Cell[20995, 622, 1678, 38, 70, "2ColumnTableMod",
 CellID->524183967]
}, Closed]],
Cell[CellGroupData[{
Cell[22710, 665, 358, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->2939395],
Cell[CellGroupData[{
Cell[23093, 679, 147, 5, 70, "ExampleSection",
 CellID->16842629],
Cell[23243, 686, 79, 1, 70, "ExampleText",
 CellID->994896104],
Cell[CellGroupData[{
Cell[23347, 691, 164, 4, 28, "Input",
 CellID->911144386],
Cell[23514, 697, 189, 5, 36, "Output",
 CellID->239511853]
}, Open  ]],
Cell[23718, 705, 124, 3, 70, "ExampleDelimiter",
 CellID->51359002],
Cell[23845, 710, 79, 1, 70, "ExampleText",
 CellID->235209921],
Cell[CellGroupData[{
Cell[23949, 715, 168, 5, 70, "Input",
 CellID->1803231767],
Cell[24120, 722, 186, 5, 36, "Output",
 CellID->1091358893]
}, Open  ]],
Cell[24321, 730, 125, 3, 70, "ExampleDelimiter",
 CellID->255582612],
Cell[24449, 735, 77, 1, 70, "ExampleText",
 CellID->1348723483],
Cell[CellGroupData[{
Cell[24551, 740, 162, 4, 70, "Input",
 CellID->1305684696],
Cell[24716, 746, 247, 8, 36, "Output",
 CellID->811085624]
}, Open  ]],
Cell[24978, 757, 125, 3, 70, "ExampleDelimiter",
 CellID->125102124],
Cell[25106, 762, 88, 1, 70, "ExampleText",
 CellID->751030269],
Cell[CellGroupData[{
Cell[25219, 767, 212, 6, 70, "Input",
 CellID->1549403507],
Cell[25434, 775, 1100, 27, 130, "Output",
 CellID->1837375059]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[26583, 808, 222, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[CellGroupData[{
Cell[26830, 819, 261, 8, 70, "ExampleSubsection",
 CellID->14003343],
Cell[27094, 829, 124, 3, 70, "ExampleText",
 CellID->56694407],
Cell[CellGroupData[{
Cell[27243, 836, 130, 4, 70, "Input",
 CellID->174295882],
Cell[27376, 842, 396, 11, 36, "Output",
 CellID->2095256400]
}, Open  ]],
Cell[CellGroupData[{
Cell[27809, 858, 101, 3, 70, "Input",
 CellID->49007095],
Cell[27913, 863, 161, 5, 36, "Output",
 CellID->537471294]
}, Open  ]],
Cell[28089, 871, 125, 3, 70, "ExampleDelimiter",
 CellID->498555886],
Cell[28217, 876, 87, 1, 70, "ExampleText",
 CellID->309303337],
Cell[CellGroupData[{
Cell[28329, 881, 124, 3, 70, "Input",
 CellID->1617796782],
Cell[28456, 886, 902, 19, 88, "Output",
 CellID->963818976]
}, Open  ]],
Cell[29373, 908, 125, 3, 70, "ExampleDelimiter",
 CellID->365236494],
Cell[29501, 913, 96, 1, 70, "ExampleText",
 CellID->492932380],
Cell[CellGroupData[{
Cell[29622, 918, 164, 4, 70, "Input",
 CellID->374014864],
Cell[29789, 924, 160, 5, 36, "Output",
 CellID->436380311]
}, Open  ]],
Cell[CellGroupData[{
Cell[29986, 934, 158, 4, 70, "Input",
 CellID->7393140],
Cell[30147, 940, 175, 5, 36, "Output",
 CellID->587232472]
}, Open  ]],
Cell[CellGroupData[{
Cell[30359, 950, 218, 7, 70, "Input",
 CellID->188296305],
Cell[30580, 959, 160, 5, 36, "Output",
 CellID->1928798939]
}, Open  ]],
Cell[CellGroupData[{
Cell[30777, 969, 205, 7, 70, "Input",
 CellID->399684212],
Cell[30985, 978, 161, 5, 36, "Output",
 CellID->1058116308]
}, Open  ]],
Cell[31161, 986, 124, 3, 70, "ExampleDelimiter",
 CellID->31217455],
Cell[31288, 991, 79, 1, 70, "ExampleText",
 CellID->277763820],
Cell[CellGroupData[{
Cell[31392, 996, 157, 4, 70, "Input",
 CellID->221172202],
Cell[31552, 1002, 174, 5, 36, "Output",
 CellID->448707749]
}, Open  ]],
Cell[CellGroupData[{
Cell[31763, 1012, 155, 4, 70, "Input",
 CellID->33052035],
Cell[31921, 1018, 175, 5, 36, "Output",
 CellID->1616052826]
}, Open  ]],
Cell[32111, 1026, 124, 3, 70, "ExampleDelimiter",
 CellID->86736583],
Cell[32238, 1031, 82, 1, 70, "ExampleText",
 CellID->361353193],
Cell[CellGroupData[{
Cell[32345, 1036, 158, 4, 70, "Input",
 CellID->1820430401],
Cell[32506, 1042, 446, 11, 70, "Output",
 Evaluatable->False,
 CellID->485885536]
}, Open  ]],
Cell[32967, 1056, 87, 1, 70, "ExampleText",
 CellID->1008028582],
Cell[CellGroupData[{
Cell[33079, 1061, 162, 4, 70, "Input",
 CellID->1237181837],
Cell[33244, 1067, 544, 13, 70, "Output",
 Evaluatable->False,
 CellID->388382199]
}, Open  ]],
Cell[33803, 1083, 125, 3, 70, "ExampleDelimiter",
 CellID->255481931],
Cell[33931, 1088, 76, 1, 70, "ExampleText",
 CellID->208354654],
Cell[CellGroupData[{
Cell[34032, 1093, 151, 5, 70, "Input",
 CellID->28830905],
Cell[34186, 1100, 397, 11, 36, "Output",
 CellID->588247246]
}, Open  ]],
Cell[34598, 1114, 125, 3, 70, "ExampleDelimiter",
 CellID->237929120],
Cell[34726, 1119, 89, 1, 70, "ExampleText",
 CellID->40277343],
Cell[CellGroupData[{
Cell[34840, 1124, 151, 5, 70, "Input",
 CellID->795584317],
Cell[34994, 1131, 441, 13, 36, "Output",
 CellID->1804236381]
}, Open  ]],
Cell[CellGroupData[{
Cell[35472, 1149, 153, 5, 70, "Input",
 CellID->23186614],
Cell[35628, 1156, 397, 11, 36, "Output",
 CellID->1331759827]
}, Open  ]],
Cell[36040, 1170, 125, 3, 70, "ExampleDelimiter",
 CellID->128501363],
Cell[36168, 1175, 86, 1, 70, "ExampleText",
 CellID->278486034],
Cell[CellGroupData[{
Cell[36279, 1180, 216, 7, 70, "Input",
 CellID->16562659],
Cell[36498, 1189, 162, 5, 36, "Output",
 CellID->950269553]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36709, 1200, 248, 7, 70, "ExampleSubsection",
 CellID->1961895733],
Cell[36960, 1209, 92, 1, 70, "ExampleText",
 CellID->63407444],
Cell[CellGroupData[{
Cell[37077, 1214, 222, 7, 70, "Input",
 CellID->180587304],
Cell[37302, 1223, 740, 17, 71, "Output",
 CellID->1812843355]
}, Open  ]],
Cell[38057, 1243, 125, 3, 70, "ExampleDelimiter",
 CellID->575071077],
Cell[38185, 1248, 89, 1, 70, "ExampleText",
 CellID->298397443],
Cell[CellGroupData[{
Cell[38299, 1253, 235, 7, 70, "Input",
 CellID->44910077],
Cell[38537, 1262, 183, 5, 36, "Output",
 CellID->2107767496]
}, Open  ]],
Cell[38735, 1270, 76, 1, 70, "ExampleText",
 CellID->826150189],
Cell[CellGroupData[{
Cell[38836, 1275, 239, 7, 70, "Input",
 CellID->17124444],
Cell[39078, 1284, 200, 5, 36, "Output",
 CellID->358064360]
}, Open  ]],
Cell[39293, 1292, 125, 3, 70, "ExampleDelimiter",
 CellID->336008823],
Cell[39421, 1297, 93, 1, 70, "ExampleText",
 CellID->725709153],
Cell[CellGroupData[{
Cell[39539, 1302, 153, 4, 70, "Input",
 CellID->1229972329],
Cell[39695, 1308, 197, 5, 36, "Output",
 CellID->738519626]
}, Open  ]],
Cell[CellGroupData[{
Cell[39929, 1318, 177, 5, 70, "Input",
 CellID->560816081],
Cell[40109, 1325, 184, 5, 36, "Output",
 CellID->1991358169]
}, Open  ]],
Cell[40308, 1333, 75, 1, 70, "ExampleText",
 CellID->98478505],
Cell[CellGroupData[{
Cell[40408, 1338, 182, 5, 70, "Input",
 CellID->1250211495],
Cell[40593, 1345, 184, 5, 36, "Output",
 CellID->224107888]
}, Open  ]],
Cell[40792, 1353, 65, 1, 70, "ExampleText",
 CellID->433423136],
Cell[CellGroupData[{
Cell[40882, 1358, 185, 5, 70, "Input",
 CellID->826445282],
Cell[41070, 1365, 169, 5, 36, "Output",
 CellID->2042440425]
}, Open  ]],
Cell[41254, 1373, 124, 3, 70, "ExampleDelimiter",
 CellID->59265178],
Cell[41381, 1378, 216, 8, 70, "ExampleText",
 CellID->244694112],
Cell[CellGroupData[{
Cell[41622, 1390, 162, 4, 70, "Input",
 CellID->493396966],
Cell[41787, 1396, 188, 5, 36, "Output",
 CellID->1917609327]
}, Open  ]],
Cell[CellGroupData[{
Cell[42012, 1406, 193, 6, 70, "Input",
 CellID->431132459],
Cell[42208, 1414, 350, 11, 36, "Output",
 CellID->1972868360]
}, Open  ]],
Cell[42573, 1428, 124, 3, 70, "ExampleDelimiter",
 CellID->12996297],
Cell[42700, 1433, 81, 1, 70, "ExampleText",
 CellID->17219685],
Cell[CellGroupData[{
Cell[42806, 1438, 161, 4, 70, "Input",
 CellID->502847698],
Cell[42970, 1444, 179, 5, 36, "Output",
 CellID->851016840]
}, Open  ]],
Cell[CellGroupData[{
Cell[43186, 1454, 185, 5, 70, "Input",
 CellID->417695890],
Cell[43374, 1461, 231, 6, 36, "Output",
 CellID->798940860]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[43654, 1473, 235, 7, 70, "ExampleSubsection",
 CellID->29302194],
Cell[43892, 1482, 158, 6, 70, "ExampleText",
 CellID->553589695],
Cell[CellGroupData[{
Cell[44075, 1492, 162, 4, 70, "Input",
 CellID->529027222],
Cell[44240, 1498, 182, 5, 36, "Output",
 CellID->1731272976]
}, Open  ]],
Cell[CellGroupData[{
Cell[44459, 1508, 159, 4, 70, "Input",
 CellID->609521478],
Cell[44621, 1514, 169, 5, 36, "Output",
 CellID->1248974429]
}, Open  ]],
Cell[CellGroupData[{
Cell[44827, 1524, 163, 4, 70, "Input",
 CellID->1563433475],
Cell[44993, 1530, 248, 8, 36, "Output",
 CellID->2008062425]
}, Open  ]],
Cell[45256, 1541, 125, 3, 70, "ExampleDelimiter",
 CellID->180621357],
Cell[45384, 1546, 134, 4, 70, "ExampleText",
 CellID->1207501607],
Cell[CellGroupData[{
Cell[45543, 1554, 163, 4, 70, "Input",
 CellID->759044699],
Cell[45709, 1560, 244, 8, 36, "Output",
 CellID->1799664370]
}, Open  ]],
Cell[CellGroupData[{
Cell[45990, 1573, 172, 5, 70, "Input",
 CellID->873970007],
Cell[46165, 1580, 495, 12, 70, "Output",
 Evaluatable->False,
 CellID->366698316]
}, Open  ]],
Cell[46675, 1595, 84, 1, 70, "ExampleText",
 CellID->770047961],
Cell[CellGroupData[{
Cell[46784, 1600, 458, 14, 70, "Input",
 CellID->713275645],
Cell[47245, 1616, 286, 9, 36, "Output",
 CellID->1352624240]
}, Open  ]],
Cell[47546, 1628, 124, 3, 70, "ExampleDelimiter",
 CellID->94476489],
Cell[47673, 1633, 247, 8, 70, "ExampleText",
 CellID->5758993],
Cell[CellGroupData[{
Cell[47945, 1645, 172, 5, 70, "Input",
 CellID->607479222],
Cell[48120, 1652, 182, 5, 36, "Output",
 CellID->293714538]
}, Open  ]],
Cell[CellGroupData[{
Cell[48339, 1662, 105, 3, 70, "Input",
 CellID->388541687],
Cell[48447, 1667, 176, 5, 36, "Output",
 CellID->1542819360]
}, Open  ]],
Cell[48638, 1675, 125, 3, 70, "ExampleDelimiter",
 CellID->528642111],
Cell[48766, 1680, 283, 10, 70, "ExampleText",
 CellID->9731214],
Cell[CellGroupData[{
Cell[49074, 1694, 153, 4, 70, "Input",
 CellID->78474866],
Cell[49230, 1700, 207, 6, 36, "Output",
 CellID->1447429642]
}, Open  ]],
Cell[49452, 1709, 89, 1, 70, "ExampleText",
 CellID->1278380685],
Cell[CellGroupData[{
Cell[49566, 1714, 154, 4, 70, "Input",
 CellID->170284560],
Cell[49723, 1720, 328, 11, 51, "Output",
 CellID->255239878]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[50112, 1738, 243, 7, 70, "ExampleSection",
 CellID->598708091],
Cell[50358, 1747, 118, 3, 70, "ExampleText",
 CellID->210372128],
Cell[CellGroupData[{
Cell[50501, 1754, 173, 5, 70, "Input",
 CellID->77964217],
Cell[50677, 1761, 563, 13, 54, "Output",
 CellID->1291786487]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[51289, 1780, 227, 7, 70, "ExampleSection",
 CellID->699285692],
Cell[51519, 1789, 78, 1, 70, "ExampleText",
 CellID->334957664],
Cell[CellGroupData[{
Cell[51622, 1794, 1260, 37, 70, "Input",
 CellID->977521728],
Cell[52885, 1833, 2440, 44, 70, "Output",
 Evaluatable->False,
 CellID->106759051]
}, Open  ]],
Cell[55340, 1880, 125, 3, 70, "ExampleDelimiter",
 CellID->754209157],
Cell[55468, 1885, 78, 1, 70, "ExampleText",
 CellID->341858881],
Cell[55549, 1888, 938, 29, 70, "Input",
 CellID->12496365],
Cell[CellGroupData[{
Cell[56512, 1921, 307, 9, 70, "Input",
 CellID->61226800],
Cell[56822, 1932, 20346, 338, 70, "Output",
 Evaluatable->False,
 CellID->987068872]
}, Open  ]],
Cell[77183, 2273, 123, 3, 70, "ExampleDelimiter",
 CellID->1532279],
Cell[77309, 2278, 85, 1, 70, "ExampleText",
 CellID->331248974],
Cell[77397, 2281, 867, 27, 70, "Input",
 CellID->343971853],
Cell[CellGroupData[{
Cell[78289, 2312, 308, 9, 70, "Input",
 CellID->108955647],
Cell[78600, 2323, 15832, 264, 70, "Output",
 Evaluatable->False,
 CellID->225439762]
}, Open  ]],
Cell[94447, 2590, 125, 3, 70, "ExampleDelimiter",
 CellID->174543350],
Cell[94575, 2595, 64, 1, 70, "ExampleText",
 CellID->197058215],
Cell[94642, 2598, 387, 12, 70, "Input",
 CellID->303285230],
Cell[CellGroupData[{
Cell[95054, 2614, 778, 22, 70, "Input",
 CellID->1336166279],
Cell[95835, 2638, 13408, 224, 70, "Output",
 Evaluatable->False,
 CellID->172123439]
}, Open  ]],
Cell[109258, 2865, 124, 3, 70, "ExampleDelimiter",
 CellID->22216897],
Cell[109385, 2870, 77, 1, 70, "ExampleText",
 CellID->780810098],
Cell[CellGroupData[{
Cell[109487, 2875, 407, 13, 70, "Input",
 CellID->75959812],
Cell[109897, 2890, 2872, 51, 70, "Output",
 Evaluatable->False,
 CellID->121851749]
}, Open  ]],
Cell[112784, 2944, 70, 1, 70, "ExampleText",
 CellID->827681361],
Cell[CellGroupData[{
Cell[112879, 2949, 410, 13, 70, "Input",
 CellID->403426493],
Cell[113292, 2964, 2445, 44, 70, "Output",
 Evaluatable->False,
 CellID->555095442]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[115786, 3014, 236, 7, 70, "ExampleSection",
 CellID->31316719],
Cell[116025, 3023, 234, 8, 70, "ExampleText",
 CellID->214644472],
Cell[CellGroupData[{
Cell[116284, 3035, 313, 10, 70, "Input",
 CellID->1117571144],
Cell[116600, 3047, 879, 17, 71, "Output",
 CellID->1459170272]
}, Open  ]],
Cell[117494, 3067, 194, 8, 70, "ExampleText",
 CellID->215160751],
Cell[CellGroupData[{
Cell[117713, 3079, 244, 7, 70, "Input",
 CellID->540657586],
Cell[117960, 3088, 877, 17, 71, "Output",
 CellID->11512764]
}, Open  ]],
Cell[118852, 3108, 125, 3, 70, "ExampleDelimiter",
 CellID->183120514],
Cell[118980, 3113, 102, 2, 70, "ExampleText",
 CellID->316513968],
Cell[CellGroupData[{
Cell[119107, 3119, 535, 17, 70, "Input",
 CellID->241955977],
Cell[119645, 3138, 2878, 51, 70, "Output",
 Evaluatable->False,
 CellID->21086418]
}, Open  ]],
Cell[122538, 3192, 340, 13, 70, "ExampleText",
 CellID->613113541],
Cell[CellGroupData[{
Cell[122903, 3209, 735, 23, 70, "Input",
 CellID->1367561668],
Cell[123641, 3234, 3681, 64, 70, "Output",
 Evaluatable->False,
 CellID->385921769]
}, Open  ]],
Cell[127337, 3301, 125, 3, 70, "ExampleDelimiter",
 CellID->135815337],
Cell[127465, 3306, 224, 8, 70, "ExampleText",
 CellID->146537355],
Cell[CellGroupData[{
Cell[127714, 3318, 659, 21, 70, "Input",
 CellID->1805150387],
Cell[128376, 3341, 2126, 39, 70, "Output",
 Evaluatable->False,
 CellID->580580474]
}, Open  ]],
Cell[130517, 3383, 348, 13, 70, "ExampleText",
 CellID->75974750],
Cell[CellGroupData[{
Cell[130890, 3400, 690, 22, 70, "Input",
 CellID->424408147],
Cell[131583, 3424, 2126, 39, 70, "Output",
 Evaluatable->False,
 CellID->271614221]
}, Open  ]],
Cell[133724, 3466, 123, 3, 70, "ExampleDelimiter",
 CellID->5874536],
Cell[133850, 3471, 221, 8, 70, "ExampleText",
 CellID->535512214],
Cell[134074, 3481, 483, 16, 70, "Input",
 CellID->1403006388],
Cell[CellGroupData[{
Cell[134582, 3501, 109, 3, 70, "Input",
 CellID->1864310337],
Cell[134694, 3506, 931, 24, 142, "Output",
 CellID->1012019439]
}, Open  ]],
Cell[135640, 3533, 229, 7, 70, "ExampleText",
 CellID->94821444],
Cell[CellGroupData[{
Cell[135894, 3544, 219, 8, 70, "Input",
 CellID->957919056],
Cell[136116, 3554, 159, 5, 36, "Output",
 CellID->940227033]
}, Open  ]],
Cell[136290, 3562, 123, 3, 70, "ExampleDelimiter",
 CellID->4200064],
Cell[136416, 3567, 238, 8, 70, "ExampleText",
 CellID->671567693],
Cell[CellGroupData[{
Cell[136679, 3579, 509, 16, 70, "Input",
 CellID->1765939808],
Cell[137191, 3597, 2533, 46, 70, "Output",
 Evaluatable->False,
 CellID->5626181]
}, Open  ]],
Cell[139739, 3646, 124, 3, 70, "ExampleDelimiter",
 CellID->28051594],
Cell[139866, 3651, 246, 8, 70, "ExampleText",
 CellID->193367025],
Cell[CellGroupData[{
Cell[140137, 3663, 162, 4, 70, "Input",
 CellID->1129350733],
Cell[140302, 3669, 1187, 22, 122, "Output",
 CellID->586760769]
}, Open  ]],
Cell[CellGroupData[{
Cell[141526, 3696, 155, 4, 70, "Input",
 CellID->776831264],
Cell[141684, 3702, 184, 5, 36, "Output",
 CellID->443337559]
}, Open  ]],
Cell[141883, 3710, 125, 3, 70, "ExampleDelimiter",
 CellID->274708332],
Cell[142011, 3715, 249, 8, 70, "ExampleText",
 CellID->1950066989],
Cell[CellGroupData[{
Cell[142285, 3727, 191, 6, 70, "Input",
 CellID->2027947256],
Cell[142479, 3735, 378, 11, 36, "Output",
 CellID->1553582670]
}, Open  ]],
Cell[CellGroupData[{
Cell[142894, 3751, 154, 4, 70, "Input",
 CellID->1641565555],
Cell[143051, 3757, 183, 5, 36, "Output",
 CellID->1957754464]
}, Open  ]],
Cell[143249, 3765, 125, 3, 70, "ExampleDelimiter",
 CellID->273159180],
Cell[143377, 3770, 233, 8, 70, "ExampleText",
 CellID->1030066181],
Cell[CellGroupData[{
Cell[143635, 3782, 202, 6, 70, "Input",
 CellID->1456625600],
Cell[143840, 3790, 411, 12, 36, "Output",
 CellID->1979915459]
}, Open  ]],
Cell[CellGroupData[{
Cell[144288, 3807, 169, 5, 70, "Input",
 CellID->345028468],
Cell[144460, 3814, 182, 5, 36, "Output",
 CellID->1180009148]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[144691, 3825, 230, 7, 70, "ExampleSection",
 CellID->183057169],
Cell[144924, 3834, 113, 3, 70, "ExampleText",
 CellID->1329051991],
Cell[CellGroupData[{
Cell[145062, 3841, 161, 4, 70, "Input",
 CellID->46500988],
Cell[145226, 3847, 299, 6, 70, "Message",
 CellID->402270391],
Cell[145528, 3855, 251, 7, 36, "Output",
 CellID->679940781]
}, Open  ]],
Cell[CellGroupData[{
Cell[145816, 3867, 170, 5, 70, "Input",
 CellID->222925477],
Cell[145989, 3874, 313, 7, 70, "Message",
 CellID->1004220580],
Cell[146305, 3883, 251, 7, 36, "Output",
 CellID->861306287]
}, Open  ]],
Cell[146571, 3893, 96, 1, 70, "ExampleText",
 CellID->173046581],
Cell[CellGroupData[{
Cell[146692, 3898, 385, 12, 70, "Input",
 CellID->135247669],
Cell[147080, 3912, 195, 6, 36, "Output",
 CellID->1934984829]
}, Open  ]],
Cell[CellGroupData[{
Cell[147312, 3923, 419, 12, 70, "Input",
 CellID->57009403],
Cell[147734, 3937, 244, 8, 36, "Output",
 CellID->1945154482]
}, Open  ]],
Cell[CellGroupData[{
Cell[148015, 3950, 162, 4, 70, "Input",
 CellID->1260968962],
Cell[148180, 3956, 190, 5, 36, "Output",
 CellID->1067820693]
}, Open  ]],
Cell[148385, 3964, 124, 3, 70, "ExampleDelimiter",
 CellID->90344104],
Cell[148512, 3969, 237, 8, 70, "ExampleText",
 CellID->46301994],
Cell[CellGroupData[{
Cell[148774, 3981, 338, 11, 70, "Input",
 CellID->1739347100],
Cell[149115, 3994, 876, 22, 54, "Output",
 CellID->341549272]
}, Open  ]],
Cell[150006, 4019, 229, 8, 70, "ExampleText",
 CellID->85509328],
Cell[CellGroupData[{
Cell[150260, 4031, 393, 12, 70, "Input",
 CellID->548645387],
Cell[150656, 4045, 367, 10, 36, "Output",
 CellID->194826073]
}, Open  ]],
Cell[CellGroupData[{
Cell[151060, 4060, 92, 3, 70, "Input",
 CellID->60605648],
Cell[151155, 4065, 451, 12, 36, "Output",
 CellID->1676730883]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[151655, 4083, 228, 7, 70, "ExampleSection",
 CellID->771168705],
Cell[151886, 4092, 135, 4, 70, "ExampleText",
 CellID->288858026],
Cell[CellGroupData[{
Cell[152046, 4100, 659, 21, 70, "Input",
 CellID->280087899],
Cell[152708, 4123, 610068, 15909, 70, "Output",
 CellID->1535534647]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[762837, 20039, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[763156, 20050, 722, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[763915, 20081, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[764236, 20092, 299, 9, 70, "Tutorials",
 CellID->341631938]
}, Open  ]],
Cell[CellGroupData[{
Cell[764572, 20106, 326, 9, 70, "RelatedLinksSection",
 CellID->1584193535],
Cell[764901, 20117, 236, 5, 70, "RelatedLinks",
 CellID->1038487239]
}, Open  ]],
Cell[CellGroupData[{
Cell[765174, 20127, 299, 8, 70, "MoreAboutSection"],
Cell[765476, 20137, 148, 3, 70, "MoreAbout",
 CellID->3403213]
}, Open  ]],
Cell[765639, 20143, 27, 0, 70, "History"],
Cell[765669, 20145, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

