(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     39625,       1394]
NotebookOptionsPosition[     32553,       1148]
NotebookOutlinePosition[     33826,       1184]
CellTagsIndexPosition[     33740,       1179]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"KelvinBei" :> Documentation`HelpLookup["paclet:ref/KelvinBei"],
           "KelvinKer" :> Documentation`HelpLookup["paclet:ref/KelvinKer"], 
          "BesselJ" :> Documentation`HelpLookup["paclet:ref/BesselJ"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"KelvinBei\"\>", 
       2->"\<\"KelvinKer\"\>", 3->"\<\"BesselJ\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"Special Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["KelvinBer", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinBer",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinBer"], "[", 
       StyleBox["z", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"ber", "(", "z", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinBer",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinBer"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["ber", "n"], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->10639]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "For positive real values of parameters, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["ber", "n"], "(", "z", ")"}], "=", 
    RowBox[{"Re", "(", 
     RowBox[{
      SuperscriptBox["e", 
       RowBox[{"n", "\[InvisibleSpace]", "\[Pi]", "\[InvisibleSpace]", "i"}]],
       
      RowBox[{
       SubscriptBox["J", "n"], "(", 
       RowBox[{"z", "\[InvisibleSpace]", 
        SuperscriptBox["e", 
         RowBox[{
          RowBox[{"-", "\[Pi]"}], "\[InvisibleSpace]", 
          RowBox[{"i", "/", "4"}]}]]}], ")"}]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ". For other values, ",
 Cell[BoxData[
  FormBox["ber", TraditionalForm]], "InlineMath"],
 " is defined by analytic continuation. "
}], "Notes",
 CellID->11421],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinBer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBer"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->3968],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinBer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBer"], "[", 
   StyleBox["z", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinBer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBer"], "[", 
   RowBox[{"0", ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->646779718],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["KelvinBer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBer"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->119645454],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinBer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBer"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinBer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBer"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->324449405],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->349135805],

Cell["Evaluate numerically:", "ExampleText",
 CellID->387615674],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBer", "[", "2.5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->182740843],

Cell[BoxData["0.3999684171295315`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->194980704]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->112194837],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{"ber", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->691312742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"KelvinBer", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->359892538],

Cell[GraphicsData["CompressedBitmap", "\<\
eJytmFtoXFUUhk9yZnLrRSOWJpkkTlNv1RQV0Qel+CAqIiIK0gfxIQYhEUxq
mnpJtCZV9EVf9MWCtUi14OWhYqkILYiF0oJYEAuiglSQ+mARLD4EhPFbe505
65w9Z87cGibn7Mta/1p7rX+vvWcenlqcefrZqcXZ6anyAwtTu2Zmp3eX759f
YCjsCoLgKP/PDAW0K7Tc/9FKhXfts5dniEA3727eh91onzwLouf+ovkgtJFD
OhzqS/66q91cvTCyd6BJO6HX3+n6v/z819nv/+AThO5x4cI/0fyONnEfk0fP
+fN/f3Hk3OrK8Sef+OSeHfunn/q8OD/3ZVDc+fjh++59f8/iMWYvXVoL7m7B
isRlhLcIjnnzg/LoX1v778TxX99+6+SjjxzCMA26uCKTol10viF15vTvKrX/
vTO4cVWEXvJyGMaZrfbk3ePZHkn0gwzfrnBWT37725tvfPPQgwcJCou/ePHf
Qs2qe90I/n380Vn8/+rYT1fWyJQyYyVk7ZK3G7ra86CU4bFAbHCekXiCgWfL
S19jEvO9NTb63QhOswY+yGxs27PujBgNezFc5zyDjQc/+A7OwB0lzECNDScZ
4vfszBFh1MbIwmgbcRrNiJNgOX72sHqcwBUcwi2c2xDxxcVR9hKT4sP6y5K1
4Rqd0cgf5TGWWDYm4TEJJI2DHoLv17qOclaK3uMy2h/rEBbcgDswCGpIGdkU
STqOSGFhw0MZZz+JFFaS+8xf/11ulXjPbiC5rBJ4ttE2J/fZpz+8+84pad6W
qX2nPHrVObYc2hQhtvu5H/8MtjtZXAZNmrem1nm79PqwS7axyEZU09QS4Jxs
SAM4ngndpPVJeQwwz+LxFG1WoIWQVOESVemOanTIjjRvcUiSww+rFdBwoypd
dFXa+gWvH89jAo4GRfegjw394EHQjScugzpEeBDDTZRhdVCtRIn1FL1+ocF8
W/J9SU6xyaBMxeriZbTm8U2jEGySR0iqCIaErVCNncaWD5xIVXtprm/QH/TY
2ufN+zXHfBMuzrkhAkE4hG3bM9k259mYzJEyrMmUpV0exs2ZGPMeRraUYgEE
nDRvSll6rhLtIE1xYjaJoVJQk20nzW05UuQFWpuUxDh5jhej924njSgK0rzR
SevZlnf78PsLrk95YC93jkNlpsBIJFrBCeP1WIxuaNMPxeFSRJ3uBGfR9VkM
yZXm9XE2ktW+mo09TpoQEkiT9jOsmFYlTUownnezEEkvrsF1mRhqieLPSdJI
ivOBUp2WEksveGu7NhND/aGGUH8bSXH4cASlpcwSfOBAFVborNTAgsfpJKZq
kUDSKM2tcexdDdME6rNhHhXL+L21BT6E8QpICYkx/YGEdDNevOjldqIDFGPZ
RBzNoCe5lvorkmi/5GVsIo6uz+xa2+xvMmlaPhsUG3pDcmluSbFhSd6cRHm6
tju2ZEotuT5O4Io0yykLy27WbkPlHAw7H/KkrB6X42iHGRXgZc+va9rKsPpv
zG8eJYxWL/EtpzLqpFNc9+8Uy17k1WrrNVNx2Od8L27Ne/HzFTduJ9p43Xgn
tTXudj6PZ+ZSpYzzJmWWbY9nY6iU7eGxHCmrmmMpS3s95uVhGPOypRSLUEd3
y9GsrFXSLH3V6XAocDTU1Wki03u9qI/GuRIYTVICroZxqk+ICFRav4YpNcwN
4zjyxYNvNKbfPOM1DrZbS23pW50rNRH7FS9mpcysKrIx1aQMw5iajbHirW4k
R8pq5UjK0mpFa0meBfMyzwKXDK4aJlWvgu5z0nYrcb8ZtFxBV71VdYJit6zm
UcIodsnoDLfILtU3dpt+9f4TqzZcyT6PC0OVKlOrviRz8FolXR+GMvOqmMbk
PCm7eZmUWbIbah6GnSjZUopFsKLfRzenLL3uZu2bxeYcDNJN0uth6C/B9TEO
5IwGXf8DTbD9XQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->35376088]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->127075355],

Cell["Series at the origin:", "ExampleText",
 CellID->4255939],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"KelvinBer", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->258655594],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "-", 
   FractionBox[
    SuperscriptBox["x", "4"], "64"], "+", 
   FractionBox[
    SuperscriptBox["x", "8"], "147456"], "-", 
   FractionBox[
    SuperscriptBox["x", "12"], "2123366400"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "16"],
    SeriesData[$CellContext`x, 0, {}, 0, 16, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, 0, 0, 0, 
    Rational[-1, 64], 0, 0, 0, 
    Rational[1, 147456], 0, 0, 0, 
    Rational[-1, 2123366400]}, 0, 16, 1],
  Editable->False]], "Output",
 ImageSize->{247, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->160772109]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18557],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->420661709],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBer", "[", 
  RowBox[{
   RowBox[{"1", "-", "I"}], ",", 
   RowBox[{
    RowBox[{"-", "3.5"}], "+", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->427432274],

Cell[BoxData[
 RowBox[{"350.14832989809577`", "\[InvisibleSpace]", "+", 
  RowBox[{"68.09534711161068`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->354546206]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->41279313],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->12211142],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"KelvinBer", "[", 
    RowBox[{"1", ",", "10"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->107853740],

Cell[BoxData[
 RowBox[{"-", 
  "59.4776104262633285630978655250799137037008684908530995378356548483238`50.\
"}]], "Output",
 ImageSize->{369, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->430332763]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->136290494],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->7791327],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBer", "[", 
  RowBox[{"1", ",", "10.000000000000000000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->45488051],

Cell[BoxData[
 RowBox[{"-", 
  "59.477610426263328563097865525079913700632382850328`36.041101113316415"}]],\
 "Output",
 ImageSize->{271, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->187724356]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5197177],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinBer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBer"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->516289872],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBer", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", "1.0"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->340855058],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.39586826101971134`"}], ",", "0.01041124172310775`", ",", 
   "0.013787984049301446`"}], "}"}]], "Output",
 ImageSize->{224, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->20278712]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->227430004],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->98800477],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinBer", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->465261729],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["ber", "n"], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->435907693]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5589],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinBer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinBer"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText",
 CellID->49703899],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinBer", "[", 
  RowBox[{"0", ",", 
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], "+", 
    RowBox[{
     RowBox[{"O", "[", "x", "]"}], "^", "10"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->472082656],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "-", 
   FractionBox[
    SuperscriptBox["x", "4"], "64"], "+", 
   FractionBox[
    SuperscriptBox["x", "6"], "96"], "-", 
   FractionBox[
    RowBox[{"2299", " ", 
     SuperscriptBox["x", "8"]}], "737280"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "10"],
    SeriesData[$CellContext`x, 0, {}, 0, 10, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, 0, 0, 0, 
    Rational[-1, 64], 0, 
    Rational[1, 96], 0, 
    Rational[-2299, 737280]}, 0, 10, 1],
  Editable->False]], "Output",
 ImageSize->{185, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->809337766]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18859],

Cell["Solve the Kelvin differential equation:", "ExampleText",
 CellID->127419602],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "4"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "4", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["x", "3"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "3", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"2", " ", 
         SuperscriptBox["n", "2"]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "x"}], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
         SuperscriptBox["x", "2"], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}]}], ")"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["n", "2"], "-", "4"}], ")"}], " ", 
         SuperscriptBox["n", "2"]}], "+", 
        SuperscriptBox["x", "4"]}], ")"}], " ", 
      RowBox[{"f", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", 
   RowBox[{"f", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->61770610],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "4", "]"}], " ", 
      RowBox[{"KelvinBei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "3", "]"}], " ", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"KelvinKer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{471, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->545024718]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->809257294],

Cell["\<\
Plot the resistance of a wire with circular cross section versus AC frequency \
(skin effect):\
\>", "ExampleText",
 CellID->146335090],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"\[Omega]", "/", "2"}], " ", 
    RowBox[{
     RowBox[{"(", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"KelvinBer", "[", "\[Omega]", "]"}], 
        RowBox[{
         RowBox[{"KelvinBei", "'"}], "[", "\[Omega]", "]"}]}], "-", 
       RowBox[{
        RowBox[{"KelvinBei", "[", "\[Omega]", "]"}], 
        RowBox[{
         RowBox[{"KelvinBer", "'"}], "[", "\[Omega]", "]"}]}]}], ")"}], "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"KelvinBer", "'"}], "[", "\[Omega]", "]"}], "^", "2"}], "+", 
       
       RowBox[{
        RowBox[{
         RowBox[{"KelvinBei", "'"}], "[", "\[Omega]", "]"}], "^", "2"}]}], 
      ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Omega]", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->673371864],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWGtsVFUQPt272zZVFHxWfFW6Lb4VjcVHsCICglZERTS10bWibZCCpTUK
QWmQZ4qv+qDxQYyvoP7ASDAmJTGSGEmMJEYSoyQGE4M/JCYSfzQxOX4z59ye
e87O3m3Xfzbbe8/cM/PNnJlvZm/bVujrWrai0NfdWWiY31tY1dXdubph3spe
PIqqlFJ78Ntfr7DWWPHvHq1xL7624BpBIY971t5Jnk67tND8ozKsPTN4mk/I
SrDKjwulKZCbA3majS0SrUPtRlwyEJOoFFt2XNZhxHmLJluTL5V1cnEGGj37
Fis1J2Isne1knSYHGu+kPJ2E64d0ZLd/8Lvf8FE5ukxAzrL86e5DeLTz7W9V
hi51mzd9ubLnM/PBxtw5b7bOGl665AOI/X17WcdijI7+o3JBjP83uUqsBLOv
ekw+evQvpOPAN79ydmqRJiQLicMHi+2D+5FjKNRLKCmy801smkFSHXIOJOAB
FdgoTXfX7qGXv943chhRPCZG28LyoR9+//ij7wfWj6gMqqki1BlGx4+Pqk7R
6nq61AD0870/wuq2BTvhD46PHftbPeRF1srZQFiIafGid6EGEwB3iLizOc3Q
RgDQxhWRqXZRdw7r4swAxDk7H/4EC3W/530+6/z80x8GD2ckXt4r4t3KujgB
MgZdVIvOf4+ou5B1sQ+tjvZdFvcu1rUNb38ygWVkFSi6NnqQNR6Bsv+rXxaz
aWgT8i4K5DsYB9VYu+YL8OvIkT8XVYRzJ+OAqsjl8I4DOFLbuHFI5OizMHv/
vYOAQBlvryiOuxkHxwBbUDikeWFFOEtYNvQgHi0whYlcfaptJXL2vpQt4BkW
yIDhRFH172MZh4QWMX6eqMVMpM4yqVBzPWYypyNY44ygj7pFxHiAZdADaSCG
3SxqdcTnBBbFM9vz9GAcB6LFydRNIobRAodsrlo9jALvohDwAJqqG3U600mO
mZ70YnBwEjCVunWWWNdsgBSi8EyiGYUS0PKGCbH0ERZQXfSujmdZkXV1IIf7
yygzQKjM+lFrfZ1ozQT13i3iidFl7a7VaUxOYsUWM8Wqd9vdFq/ay+3Ta0Sb
5XH+bPavFrUej2uEMtPyKs9DD++CBCATLWeIGCsSbMFUKqHVE2uBuZj06kqX
spLcTPb8Km07Efbgv7qioooaFDM56Hvj8opQntC2V9Fn1O+XiSgxG+gE9LiX
95FK5ImWl4pW5ablapYxizBuaHlJRSfoY9m81tDyYh0yNY46zv+TbIHvLZSY
lheJVe7XdgLZKSZrGSxUETOXqnChx7qneBezGM4ou9NFDKNl3qJo2SxqPc0y
UoWE0bLJ87Qm5iQ80exvEjGMlmNeXtRaG3CiUZfgd27Mxm7FGV7HCGhEtCMt
p1XEj3VBZS8QUcgqa8OLxnwjbkRP3yulrZL9GcbyTFDXBhEl2RfJ58/GdUcM
tDy/rDXdB3gfQw6jjpbn6ZDJNWUyZhDclDtXrK/Rcsx2WhTFBm3nCk5O1T9H
xNgQVPhsUes5lt1Mnup52hhUaqqIsTGoxFmi1ibtT6R6XYKzJWbyFu1315li
xcrNos3BiSpD2RKc+Ixxo5C4NcjE6WW5l3y+LcjDaSWt0+RtQR4MSsjk+H8V
dB8MznyqWOXBgHWniFrbA9ZN8Vj3fBwbPBH7p4gYRst1wWRR6wWW3eQ+2fP0
kva78SQR40VtuxGeaCbLWgbLzeRJOp3fzIgEv4d4z33XnijWVX4PTGoMBWc6
4T/h0J/l7bvoL4iJ4JDCK7ztJmxdSfv4mz/5/FVt3jdlK4ndlMPX+Tkcwi0t
a3Uaq5MIr7GMc+K0xO5ascYG33VujcemYe13ZrWIsUP7nSdrGSzXbTnP0xu8
u2/k8MD6kbF8FGEYLddlUYqW67bI8/QW7+Id2rIpI2Kk/f9RVf0L+iNz2g==
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->628559539]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17393],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " to simplify expressions involving Kelvin functions:"
}], "ExampleText",
 CellID->540949068],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"KelvinBer", "[", 
     RowBox[{"1", ",", "x"}], "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->275163834],

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinBer", "[", 
   RowBox[{"1", ",", "x"}], "]"}], "+", 
  FractionBox[
   RowBox[{"x", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"KelvinBei", "[", 
        RowBox[{"0", ",", "x"}], "]"}]}], "+", 
      RowBox[{"KelvinBei", "[", 
       RowBox[{"2", ",", "x"}], "]"}], "-", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "+", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"2", ",", "x"}], "]"}]}], ")"}]}], 
   RowBox[{"2", " ", 
    SqrtBox["2"]}]]}]], "Output",
 ImageSize->{494, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->410383954]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->71003703],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{"x", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"KelvinBei", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "+", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"0", ",", "x"}], "]"}]}], ")"}]}], 
   SqrtBox["2"]]}]], "Output",
 ImageSize->{257, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->26934208]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->545301568],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand Kelvin functions of half-integer orders:"
}], "ExampleText",
 CellID->127473317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"KelvinBer", "[", 
   RowBox[{
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->788709929],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox[
    FractionBox["2", "\[Pi]"]], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "2"]}], " ", 
      SqrtBox[
       RowBox[{"2", "-", 
        SqrtBox["2"]}]], " ", 
      RowBox[{"Cosh", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}], " ", 
      RowBox[{"Sin", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}]}], "+", 
     RowBox[{
      FractionBox["1", "2"], " ", 
      SqrtBox[
       RowBox[{"2", "+", 
        SqrtBox["2"]}]], " ", 
      RowBox[{"Cos", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}], " ", 
      RowBox[{"Sinh", "[", 
       FractionBox["x", 
        SqrtBox["2"]], "]"}]}]}], ")"}]}], 
  SqrtBox["x"]]], "Output",
 ImageSize->{491, 62},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->308250045]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->370746783],

Cell["Integrate expressions involving Kelvin functions:", "ExampleText",
 CellID->122712053],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"r", " ", 
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox[
       RowBox[{"KelvinBer", "[", "r", "]"}], "2"], "+", 
      SuperscriptBox[
       RowBox[{"KelvinBei", "[", "r", "]"}], "2"]}], ")"}]}], ",", "r"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->586709475],

Cell[BoxData[
 RowBox[{"-", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["2"]], 
   RowBox[{"r", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"KelvinBer", "[", 
        RowBox[{"0", ",", "r"}], "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{"KelvinBei", "[", 
           RowBox[{"1", ",", "r"}], "]"}]}], "+", 
         RowBox[{"KelvinBer", "[", 
          RowBox[{"1", ",", "r"}], "]"}]}], ")"}]}], "+", 
      RowBox[{
       RowBox[{"KelvinBei", "[", 
        RowBox[{"0", ",", "r"}], "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"KelvinBei", "[", 
          RowBox[{"1", ",", "r"}], "]"}], "+", 
         RowBox[{"KelvinBer", "[", 
          RowBox[{"1", ",", "r"}], "]"}]}], ")"}]}]}], ")"}]}]}]}]], "Output",\

 ImageSize->{414, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->653231642]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->3467],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinBei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBei"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinKer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKer"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->629]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->768],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->29746]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->710636042],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->655419802],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->285561522],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->28698382]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"KelvinBer - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 51, 44.6381236}", "context" -> "System`", 
    "keywords" -> {"Kelvin ber", "Kelvin functions"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "KelvinBer[z] gives the Kelvin function ber(z). KelvinBer[n, z] gives the \
Kelvin function ber_n (z). ", "synonyms" -> {"kelvin ber"}, "title" -> 
    "KelvinBer", "type" -> "Symbol", "uri" -> "ref/KelvinBer"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8697, 284, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->324449405]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 33597, 1172}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2270, 54, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2893, 81, 1574, 37, 70, "ObjectNameGrid"],
Cell[4470, 120, 1081, 36, 70, "Usage",
 CellID->10639]
}, Open  ]],
Cell[CellGroupData[{
Cell[5588, 161, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6066, 178, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6189, 184, 771, 25, 70, "Notes",
 CellID->11421],
Cell[6963, 211, 567, 21, 70, "Notes",
 CellID->3968],
Cell[7533, 234, 445, 17, 70, "Notes",
 CellID->646779718],
Cell[7981, 253, 250, 8, 70, "Notes",
 CellID->119645454],
Cell[8234, 263, 220, 7, 70, "Notes",
 CellID->31226],
Cell[8457, 272, 203, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8697, 284, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->324449405],
Cell[CellGroupData[{
Cell[9082, 298, 148, 5, 70, "ExampleSection",
 CellID->349135805],
Cell[9233, 305, 64, 1, 70, "ExampleText",
 CellID->387615674],
Cell[CellGroupData[{
Cell[9322, 310, 107, 3, 28, "Input",
 CellID->182740843],
Cell[9432, 315, 176, 5, 36, "Output",
 CellID->194980704]
}, Open  ]],
Cell[9623, 323, 125, 3, 70, "ExampleDelimiter",
 CellID->112194837],
Cell[9751, 328, 164, 7, 70, "ExampleText",
 CellID->691312742],
Cell[CellGroupData[{
Cell[9940, 339, 220, 7, 28, "Input",
 CellID->359892538],
Cell[10163, 348, 2180, 40, 127, "Output",
 Evaluatable->False,
 CellID->35376088]
}, Open  ]],
Cell[12358, 391, 125, 3, 70, "ExampleDelimiter",
 CellID->127075355],
Cell[12486, 396, 62, 1, 70, "ExampleText",
 CellID->4255939],
Cell[CellGroupData[{
Cell[12573, 401, 222, 7, 70, "Input",
 CellID->258655594],
Cell[12798, 410, 718, 23, 54, "Output",
 CellID->160772109]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13565, 439, 216, 7, 70, "ExampleSection",
 CellID->18557],
Cell[13784, 448, 85, 1, 70, "ExampleText",
 CellID->420661709],
Cell[CellGroupData[{
Cell[13894, 453, 198, 7, 70, "Input",
 CellID->427432274],
Cell[14095, 462, 272, 7, 36, "Output",
 CellID->354546206]
}, Open  ]],
Cell[14382, 472, 124, 3, 70, "ExampleDelimiter",
 CellID->41279313],
Cell[14509, 477, 69, 1, 70, "ExampleText",
 CellID->12211142],
Cell[CellGroupData[{
Cell[14603, 482, 184, 6, 70, "Input",
 CellID->107853740],
Cell[14790, 490, 254, 8, 36, "Output",
 CellID->430332763]
}, Open  ]],
Cell[15059, 501, 125, 3, 70, "ExampleDelimiter",
 CellID->136290494],
Cell[15187, 506, 111, 3, 70, "ExampleText",
 CellID->7791327],
Cell[CellGroupData[{
Cell[15323, 513, 168, 5, 70, "Input",
 CellID->45488051],
Cell[15494, 520, 250, 8, 36, "Output",
 CellID->187724356]
}, Open  ]],
Cell[15759, 531, 123, 3, 70, "ExampleDelimiter",
 CellID->5197177],
Cell[15885, 536, 212, 7, 70, "ExampleText",
 CellID->516289872],
Cell[CellGroupData[{
Cell[16122, 547, 189, 6, 70, "Input",
 CellID->340855058],
Cell[16314, 555, 294, 9, 36, "Output",
 CellID->20278712]
}, Open  ]],
Cell[16623, 567, 125, 3, 70, "ExampleDelimiter",
 CellID->227430004],
Cell[16751, 572, 202, 7, 70, "ExampleText",
 CellID->98800477],
Cell[CellGroupData[{
Cell[16978, 583, 167, 5, 70, "Input",
 CellID->465261729],
Cell[17148, 590, 256, 8, 50, "Output",
 CellID->435907693]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17453, 604, 238, 7, 70, "ExampleSection",
 CellID->5589],
Cell[17694, 613, 212, 7, 70, "ExampleText",
 CellID->49703899],
Cell[CellGroupData[{
Cell[17931, 624, 241, 8, 70, "Input",
 CellID->472082656],
Cell[18175, 634, 721, 24, 54, "Output",
 CellID->809337766]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18945, 664, 223, 7, 70, "ExampleSection",
 CellID->18859],
Cell[19171, 673, 82, 1, 70, "ExampleText",
 CellID->127419602],
Cell[CellGroupData[{
Cell[19278, 678, 1516, 50, 70, "Input",
 CellID->61770610],
Cell[20797, 730, 813, 26, 54, "Output",
 CellID->545024718]
}, Open  ]],
Cell[21625, 759, 125, 3, 70, "ExampleDelimiter",
 CellID->809257294],
Cell[21753, 764, 145, 4, 70, "ExampleText",
 CellID->146335090],
Cell[CellGroupData[{
Cell[21923, 772, 892, 29, 70, "Input",
 CellID->673371864],
Cell[22818, 803, 2165, 40, 70, "Output",
 Evaluatable->False,
 CellID->628559539]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25032, 849, 233, 7, 70, "ExampleSection",
 CellID->17393],
Cell[25268, 858, 246, 8, 70, "ExampleText",
 CellID->540949068],
Cell[CellGroupData[{
Cell[25539, 870, 208, 7, 70, "Input",
 CellID->275163834],
Cell[25750, 879, 691, 23, 71, "Output",
 CellID->410383954]
}, Open  ]],
Cell[CellGroupData[{
Cell[26478, 907, 107, 3, 70, "Input",
 CellID->71003703],
Cell[26588, 912, 420, 15, 55, "Output",
 CellID->26934208]
}, Open  ]],
Cell[27023, 930, 125, 3, 70, "ExampleDelimiter",
 CellID->545301568],
Cell[27151, 935, 249, 8, 70, "ExampleText",
 CellID->127473317],
Cell[CellGroupData[{
Cell[27425, 947, 195, 6, 70, "Input",
 CellID->788709929],
Cell[27623, 955, 906, 35, 83, "Output",
 CellID->308250045]
}, Open  ]],
Cell[28544, 993, 125, 3, 70, "ExampleDelimiter",
 CellID->370746783],
Cell[28672, 998, 92, 1, 70, "ExampleText",
 CellID->122712053],
Cell[CellGroupData[{
Cell[28789, 1003, 344, 12, 70, "Input",
 CellID->586709475],
Cell[29136, 1017, 938, 32, 72, "Output",
 CellID->653231642]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[30135, 1056, 310, 9, 70, "SeeAlsoSection",
 CellID->3467],
Cell[30448, 1067, 696, 26, 70, "SeeAlso",
 CellID->629]
}, Open  ]],
Cell[CellGroupData[{
Cell[31181, 1098, 312, 9, 70, "TutorialsSection",
 CellID->768],
Cell[31496, 1109, 144, 3, 70, "Tutorials",
 CellID->29746]
}, Open  ]],
Cell[CellGroupData[{
Cell[31677, 1117, 319, 9, 70, "MoreAboutSection",
 CellID->710636042],
Cell[31999, 1128, 158, 3, 70, "MoreAbout",
 CellID->655419802],
Cell[32160, 1133, 145, 3, 70, "MoreAbout",
 CellID->285561522],
Cell[32308, 1138, 173, 3, 70, "MoreAbout",
 CellID->28698382]
}, Open  ]],
Cell[32496, 1144, 27, 0, 70, "History"],
Cell[32526, 1146, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

