(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     39067,       1375]
NotebookOptionsPosition[     32221,       1136]
NotebookOutlinePosition[     33493,       1172]
CellTagsIndexPosition[     33407,       1167]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"KelvinKer" :> Documentation`HelpLookup["paclet:ref/KelvinKer"],
           "KelvinBei" :> Documentation`HelpLookup["paclet:ref/KelvinBei"], 
          "BesselK" :> Documentation`HelpLookup["paclet:ref/BesselK"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"KelvinKer\"\>", 
       2->"\<\"KelvinBei\"\>", 3->"\<\"BesselK\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["KelvinKei", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinKei",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinKei"], "[", 
       StyleBox["z", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"kei", "(", "z", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinKei",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinKei"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["kei", "n"], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->3194]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "For positive real values of parameters, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["kei", "n"], "(", "z", ")"}], "=", 
    RowBox[{"Im", "(", 
     RowBox[{
      SuperscriptBox["e", 
       RowBox[{
        RowBox[{"-", "n"}], "\[InvisibleSpace]", "\[Pi]", "\[InvisibleSpace]",
         
        RowBox[{"i", "/", "2"}]}]], 
      RowBox[{
       SubscriptBox["K", "n"], "(", 
       RowBox[{"z", "\[InvisibleSpace]", 
        SuperscriptBox["e", 
         RowBox[{"\[Pi]", "\[InvisibleSpace]", 
          RowBox[{"i", "/", "4"}]}]]}], ")"}]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ". For other values, ",
 Cell[BoxData[
  FormBox["kei", TraditionalForm]], "InlineMath"],
 " is defined by analytic continuation. "
}], "Notes",
 CellID->26603],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinKei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKei"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->18534],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinKei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKei"], "[", 
   StyleBox["z", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinKei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKei"], "[", 
   RowBox[{"0", ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->521437675],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["KelvinKei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKei"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->401723354],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinKei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKei"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinKei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKei"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->47438784],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->566234308],

Cell["Evaluate numerically:", "ExampleText",
 CellID->387615674],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKei", "[", "2.5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->316194421],

Cell[BoxData[
 RowBox[{"-", "0.11069609915567502`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->890757558]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKei", "[", 
  RowBox[{"2", ",", "2.5"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->37912609],

Cell[BoxData["0.1242399363024985`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1188882]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->661358281],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["kei", "0"], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->543509965],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"KelvinKei", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->377531937],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 135},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->52644731]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->127075355],

Cell["Series at the origin:", "ExampleText",
 CellID->4255939],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"KelvinKei", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->258655594],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    FractionBox["\[Pi]", "4"]}], "+", 
   RowBox[{
    FractionBox["1", "4"], " ", 
    RowBox[{"(", 
     RowBox[{"1", "-", "EulerGamma", "+", 
      RowBox[{"Log", "[", "2", "]"}], "-", 
      RowBox[{"Log", "[", "x", "]"}]}], ")"}], " ", 
    SuperscriptBox["x", "2"]}], "+", 
   FractionBox[
    RowBox[{"\[Pi]", " ", 
     SuperscriptBox["x", "4"]}], "256"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "6"],
    SeriesData[$CellContext`x, 0, {}, 0, 6, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
   Rational[-1, 4] Pi, 0, 
    Rational[1, 4] (1 - EulerGamma + Log[2] - Log[$CellContext`x]), 0, 
    Rational[1, 256] Pi}, 0, 6, 1],
  Editable->False]], "Output",
 ImageSize->{366, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->446341787]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8509],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->550671982],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKei", "[", 
  RowBox[{
   RowBox[{"1", "-", "I"}], ",", "2.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->203028832],

Cell[BoxData[
 RowBox[{"0.12237027116644507`", "\[InvisibleSpace]", "-", 
  RowBox[{"0.36837531475058055`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->231943478]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->72025472],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->56601170],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"KelvinKei", "[", 
    RowBox[{"1", ",", "3"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3719057],

Cell[BoxData["0.\
0802702225239221937252898512884067283132193726969675788135920022348`50."], \
"Output",
 ImageSize->{375, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->226809519]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->912312743],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->167520595],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKei", "[", 
  RowBox[{"1", ",", "3.00000000000000000000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->360044869],

Cell[BoxData["0.080270222523922193725289851288406728315049391453`35.\
758470628614894"], "Output",
 ImageSize->{270, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15811761]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->33101005],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->560988411],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinKei", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->94325837],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["kei", "n"], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{37, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->98130048]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21273],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinKei",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKei"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText",
 CellID->93991909],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKei", "[", 
  RowBox[{"1", ",", 
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], "+", 
    RowBox[{
     RowBox[{"O", "[", "x", "]"}], "^", "3"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13251630],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    FractionBox["1", 
     RowBox[{
      SqrtBox["2"], " ", "x"}]]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      FractionBox["1", 
       RowBox[{"4", " ", 
        SqrtBox["2"]}]], "-", 
      FractionBox["EulerGamma", 
       RowBox[{"2", " ", 
        SqrtBox["2"]}]], "+", 
      FractionBox["\[Pi]", 
       RowBox[{"8", " ", 
        SqrtBox["2"]}]], "+", 
      FractionBox[
       RowBox[{"Log", "[", "2", "]"}], 
       RowBox[{"2", " ", 
        SqrtBox["2"]}]], "-", 
      FractionBox[
       RowBox[{"Log", "[", "x", "]"}], 
       RowBox[{"2", " ", 
        SqrtBox["2"]}]]}], ")"}], " ", "x"}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "3"],
    SeriesData[$CellContext`x, 0, {}, -1, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 
   0, {-2^Rational[-1, 2], 0, Rational[1, 4] 2^Rational[-1, 2] + 
    Rational[-1, 2] 2^Rational[-1, 2] EulerGamma + 
    Rational[1, 8] 2^Rational[-1, 2] Pi + 
    Rational[1, 2] 2^Rational[-1, 2] Log[2] + 
    Rational[-1, 2] 2^Rational[-1, 2] Log[$CellContext`x]}, -1, 3, 1],
  Editable->False]], "Output",
 ImageSize->{426, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17957439]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27412],

Cell["Solve the Kelvin differential equation:", "ExampleText",
 CellID->127419602],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "4"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "4", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["x", "3"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "3", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"2", " ", 
         SuperscriptBox["n", "2"]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "x"}], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
         SuperscriptBox["x", "2"], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}]}], ")"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "4"}], " ", 
         SuperscriptBox["n", "2"]}], "+", 
        SuperscriptBox["n", "4"], "+", 
        SuperscriptBox["x", "4"]}], ")"}], " ", 
      RowBox[{"f", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", 
   RowBox[{"f", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->61770610],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "4", "]"}], " ", 
      RowBox[{"KelvinBei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "3", "]"}], " ", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"KelvinKer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{471, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->49457396]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->809257294],

Cell["\<\
Plot the radial density profile for AC current in a hollow cylinder: \
\>", "ExampleText",
 CellID->104196068],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{
     RowBox[{"KelvinKer", "[", 
      RowBox[{"0", ",", "r"}], "]"}], "+", 
     RowBox[{"I", " ", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"0", ",", " ", "r"}], "]"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "1", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->106593776],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVx384E3gcwPExv9owaTmSn6Gia+rocPH5iubHg9QRM1dTU0hx0mW5dv2Q
JGrJulNd3abyszITU+ppCumpJoT2EOaR2k1Wfhd23/vj/byfl92ulG3x2gQC
IQz3//8OTzEhENTQtMRq+dIwBah0ht8+JaqBMcN5uzpIARvrY4rP6KshaeV4
KPJXgNyOvmEJRQ1+mt7oNC8FmI0vj3WyVkPUfrt+pZMCePznJSEb1VDSMDK+
hqCAE3JHKMpQg4uP7PyYeBD2sHv3r/+shsjf1AObbAchzvYw787UZ/jpSISf
7cd+UNIqq7qUX8B8C7OXNdIHHncrH7jJx6G+Q5yw1qwXJF3ys719E9DBc9f2
9pdD89ZihmvbJAxZ+px2rO4BvUNFqpddUyDcWfHXNetuIBswkwQ90+CrHinS
3H0D1z/VDtu2zgCfFuysldYJ7KPISflyFn7hKdzJ6zpAspPN3in9ClN2IYxR
i3Z4Lfo5n1T1Df49ZS2csHgNlpWpmZrqOeij2EwnmbXBxOzKvf/cmId044Xc
ub2vQDNmaH62fAGGLsUHSq++AFnkQ3FFhQbCX30Jzpx+Dn/qZzk1hBLQWwqv
U+t0K+zweBzZIiMgo/jseH2TZzBa0HpyVZAWYtGrLfJbmoEvTSy880ILcURN
3bs4TWBTmnJZFqqN7JhGnKbNTyFVsuk6vVkbRfl433v14xOQX63MVgQSUc3L
rVrxDo1ArhhK7mwkInXd94KE9VKw+fVYYJyXDrptwq9K3/YYQsg1eksadNBI
LQx1yB8BktU8Weaui+InSv+QBTyE0L0huwMkuig2t8d3quIBbKwLmjH21EMl
LMNPfrT7MO5KzVsj0kPFu83LJ/ol0Mg9pJ3ygz5K7pzMOJdTB4yp4sP6t/XR
vGFbTjejFtodPRReDgaIJdycJ0P3gL16N+3iTQN0UGXTvt+2Bh5uZ2b6Wi9C
obNCY4K9GGjRyjKqYBEyeJZj7mpVDQkn6d3eliRU0BVQptwggqw9Bl+1LpJQ
tYozOcCqgnqrOI2SSkYOy97nriLdhXsL2UTnc2QUHLGQ1XLqNjwX+ynzFxsi
d+fKWPZ0BUht9zTnFRiiavRusOZUORyrixQMkoxQCTd81QefMoi2793BzDdC
bw7sYtuPlcB0Lc3SRd8YSe57R6xovgUZxI8v+FxjFBboZ10ougmmDp5xjgQK
Sr6Skv6t8AYcPLB9gsShIB6PPLaiqBhG80hJS+coSOdz8KGQS0Iwz/KlOmIT
j0ti12L7Z2Y8csPu7In2N8G+nPjeNAL72Lt11E6+EOj0xgcF2FSPzbVM7Gsa
jpHJPAWJO9XEfYVC2JL2sYq0QEFcD0bTmQIhHEm0ZS7DfmNqI96HfYsVpeuM
XVr2nSAUey6siRGEHU4WHF2MXeoi0M7BbiikehZdEAJhODpCV0NBolRufSlP
CC695xeo2HYfUspzsbd3NJc6YMeZFV5Jxq6Uus37Yffz07mu2NHXTG6dwJ4z
u0SXnBfCSX5AeAH2pJeV52XsO3ncrwLsUdNul9+xdTJVYVLsqD7FYoRNS1sx
24ZtIXPTtceOSYwpHsBOOFg/Q8TOZl0IVWOrWlNVw+eEIIp6Nq3BDm6PGWjB
/g/KZUsz
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  ImageMargins->0.,
  ImageSize->Small,
  Method->{},
  PlotRange->{{1, 4}, {0., 0.5720315503209511}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->617359772]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31888],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " to simplify expressions involving Kelvin functions:"
}], "ExampleText",
 CellID->540949068],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"KelvinKei", "[", 
     RowBox[{"1", ",", 
      RowBox[{
       SqrtBox["2"], "x"}]}], "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->275163834],

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinKei", "[", 
   RowBox[{"1", ",", 
    RowBox[{
     SqrtBox["2"], " ", "x"}]}], "]"}], "+", 
  RowBox[{
   FractionBox["1", "2"], " ", "x", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"0", ",", 
        RowBox[{
         SqrtBox["2"], " ", "x"}]}], "]"}]}], "+", 
     RowBox[{"KelvinKei", "[", 
      RowBox[{"2", ",", 
       RowBox[{
        SqrtBox["2"], " ", "x"}]}], "]"}], "+", 
     RowBox[{"KelvinKer", "[", 
      RowBox[{"0", ",", 
       RowBox[{
        SqrtBox["2"], " ", "x"}]}], "]"}], "-", 
     RowBox[{"KelvinKer", "[", 
      RowBox[{"2", ",", 
       RowBox[{
        SqrtBox["2"], " ", "x"}]}], "]"}]}], ")"}]}]}]], "Output",
 ImageSize->{452, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->605434335]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->71003703],

Cell[BoxData[
 RowBox[{"x", " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"KelvinKei", "[", 
      RowBox[{"0", ",", 
       RowBox[{
        SqrtBox["2"], " ", "x"}]}], "]"}]}], "+", 
    RowBox[{"KelvinKer", "[", 
     RowBox[{"0", ",", 
      RowBox[{
       SqrtBox["2"], " ", "x"}]}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{307, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->2989014]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->545301568],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand Kelvin functions of half-integer orders:"
}], "ExampleText",
 CellID->127473317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"KelvinKei", "[", 
   RowBox[{
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->788709929],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", 
      FractionBox["x", 
       SqrtBox["2"]]}]], " ", 
    SqrtBox[
     FractionBox["\[Pi]", "2"]], " ", 
    RowBox[{"Cos", "[", 
     RowBox[{
      FractionBox["\[Pi]", "8"], "-", 
      FractionBox["x", 
       SqrtBox["2"]]}], "]"}]}], 
   SqrtBox["x"]]}]], "Output",
 ImageSize->{169, 64},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->946607654]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->370746783],

Cell["Integrate expressions involving Kelvin functions:", "ExampleText",
 CellID->122712053],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"r", " ", 
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox[
       RowBox[{"KelvinKer", "[", 
        RowBox[{"0", ",", "r"}], "]"}], "2"], "+", 
      SuperscriptBox[
       RowBox[{"KelvinKei", "[", 
        RowBox[{"0", ",", "r"}], "]"}], "2"]}], ")"}]}], ",", "r"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->586709475],

Cell[BoxData[
 RowBox[{"-", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["2"]], 
   RowBox[{"r", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"KelvinKer", "[", 
        RowBox[{"0", ",", "r"}], "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{"KelvinKei", "[", 
           RowBox[{"1", ",", "r"}], "]"}]}], "+", 
         RowBox[{"KelvinKer", "[", 
          RowBox[{"1", ",", "r"}], "]"}]}], ")"}]}], "+", 
      RowBox[{
       RowBox[{"KelvinKei", "[", 
        RowBox[{"0", ",", "r"}], "]"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"KelvinKei", "[", 
          RowBox[{"1", ",", "r"}], "]"}], "+", 
         RowBox[{"KelvinKer", "[", 
          RowBox[{"1", ",", "r"}], "]"}]}], ")"}]}]}], ")"}]}]}]}]], "Output",\

 ImageSize->{414, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->3444046]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->7987],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinKer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKer"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinBei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBei"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselK",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselK"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->28571]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->32541],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->21940]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->545050297],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->17649280],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->251363774]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"KelvinKei - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 51, 45.0600094}", "context" -> "System`", 
    "keywords" -> {"Kelvin functions", "Kelvin kei"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "KelvinKei[z] gives the Kelvin function kei(z). KelvinKei[n, z] gives the \
Kelvin function kei_n (z). ", "synonyms" -> {"kelvin kei"}, "title" -> 
    "KelvinKei", "type" -> "Symbol", "uri" -> "ref/KelvinKei"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8581, 282, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->47438784]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 33265, 1160}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2126, 51, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2749, 78, 1574, 37, 70, "ObjectNameGrid"],
Cell[4326, 117, 1080, 36, 70, "Usage",
 CellID->3194]
}, Open  ]],
Cell[CellGroupData[{
Cell[5443, 158, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5921, 175, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6044, 181, 800, 26, 70, "Notes",
 CellID->26603],
Cell[6847, 209, 567, 21, 70, "Notes",
 CellID->18534],
Cell[7417, 232, 445, 17, 70, "Notes",
 CellID->521437675],
Cell[7865, 251, 250, 8, 70, "Notes",
 CellID->401723354],
Cell[8118, 261, 220, 7, 70, "Notes",
 CellID->31226],
Cell[8341, 270, 203, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8581, 282, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->47438784],
Cell[CellGroupData[{
Cell[8965, 296, 148, 5, 70, "ExampleSection",
 CellID->566234308],
Cell[9116, 303, 64, 1, 70, "ExampleText",
 CellID->387615674],
Cell[CellGroupData[{
Cell[9205, 308, 107, 3, 28, "Input",
 CellID->316194421],
Cell[9315, 313, 194, 6, 36, "Output",
 CellID->890757558]
}, Open  ]],
Cell[CellGroupData[{
Cell[9546, 324, 129, 4, 28, "Input",
 CellID->37912609],
Cell[9678, 330, 174, 5, 36, "Output",
 CellID->1188882]
}, Open  ]],
Cell[9867, 338, 125, 3, 70, "ExampleDelimiter",
 CellID->661358281],
Cell[9995, 343, 191, 9, 70, "ExampleText",
 CellID->543509965],
Cell[CellGroupData[{
Cell[10211, 356, 219, 7, 28, "Input",
 CellID->377531937],
Cell[10433, 365, 2481, 45, 156, "Output",
 Evaluatable->False,
 CellID->52644731]
}, Open  ]],
Cell[12929, 413, 125, 3, 70, "ExampleDelimiter",
 CellID->127075355],
Cell[13057, 418, 62, 1, 70, "ExampleText",
 CellID->4255939],
Cell[CellGroupData[{
Cell[13144, 423, 221, 7, 70, "Input",
 CellID->258655594],
Cell[13368, 432, 917, 29, 54, "Output",
 CellID->446341787]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14334, 467, 215, 7, 70, "ExampleSection",
 CellID->8509],
Cell[14552, 476, 85, 1, 70, "ExampleText",
 CellID->550671982],
Cell[CellGroupData[{
Cell[14662, 481, 154, 5, 70, "Input",
 CellID->203028832],
Cell[14819, 488, 275, 7, 36, "Output",
 CellID->231943478]
}, Open  ]],
Cell[15109, 498, 124, 3, 70, "ExampleDelimiter",
 CellID->72025472],
Cell[15236, 503, 69, 1, 70, "ExampleText",
 CellID->56601170],
Cell[CellGroupData[{
Cell[15330, 508, 181, 6, 70, "Input",
 CellID->3719057],
Cell[15514, 516, 235, 7, 36, "Output",
 CellID->226809519]
}, Open  ]],
Cell[15764, 526, 125, 3, 70, "ExampleDelimiter",
 CellID->912312743],
Cell[15892, 531, 113, 3, 70, "ExampleText",
 CellID->167520595],
Cell[CellGroupData[{
Cell[16030, 538, 170, 5, 70, "Input",
 CellID->360044869],
Cell[16203, 545, 228, 6, 36, "Output",
 CellID->15811761]
}, Open  ]],
Cell[16446, 554, 124, 3, 70, "ExampleDelimiter",
 CellID->33101005],
Cell[16573, 559, 203, 7, 70, "ExampleText",
 CellID->560988411],
Cell[CellGroupData[{
Cell[16801, 570, 166, 5, 70, "Input",
 CellID->94325837],
Cell[16970, 577, 255, 8, 50, "Output",
 CellID->98130048]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17274, 591, 239, 7, 70, "ExampleSection",
 CellID->21273],
Cell[17516, 600, 212, 7, 70, "ExampleText",
 CellID->93991909],
Cell[CellGroupData[{
Cell[17753, 611, 239, 8, 70, "Input",
 CellID->13251630],
Cell[17995, 621, 1317, 43, 58, "Output",
 CellID->17957439]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19361, 670, 223, 7, 70, "ExampleSection",
 CellID->27412],
Cell[19587, 679, 82, 1, 70, "ExampleText",
 CellID->127419602],
Cell[CellGroupData[{
Cell[19694, 684, 1487, 49, 70, "Input",
 CellID->61770610],
Cell[21184, 735, 812, 26, 54, "Output",
 CellID->49457396]
}, Open  ]],
Cell[22011, 764, 125, 3, 70, "ExampleDelimiter",
 CellID->809257294],
Cell[22139, 769, 120, 3, 70, "ExampleText",
 CellID->104196068],
Cell[CellGroupData[{
Cell[22284, 776, 403, 13, 70, "Input",
 CellID->106593776],
Cell[22690, 791, 2165, 46, 136, "Output",
 CellID->617359772]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24904, 843, 233, 7, 70, "ExampleSection",
 CellID->31888],
Cell[25140, 852, 246, 8, 70, "ExampleText",
 CellID->540949068],
Cell[CellGroupData[{
Cell[25411, 864, 247, 9, 70, "Input",
 CellID->275163834],
Cell[25661, 875, 878, 31, 73, "Output",
 CellID->605434335]
}, Open  ]],
Cell[CellGroupData[{
Cell[26576, 911, 107, 3, 70, "Input",
 CellID->71003703],
Cell[26686, 916, 466, 17, 43, "Output",
 CellID->2989014]
}, Open  ]],
Cell[27167, 936, 125, 3, 70, "ExampleDelimiter",
 CellID->545301568],
Cell[27295, 941, 249, 8, 70, "ExampleText",
 CellID->127473317],
Cell[CellGroupData[{
Cell[27569, 953, 195, 6, 70, "Input",
 CellID->788709929],
Cell[27767, 961, 518, 20, 85, "Output",
 CellID->946607654]
}, Open  ]],
Cell[28300, 984, 125, 3, 70, "ExampleDelimiter",
 CellID->370746783],
Cell[28428, 989, 92, 1, 70, "ExampleText",
 CellID->122712053],
Cell[CellGroupData[{
Cell[28545, 994, 402, 14, 70, "Input",
 CellID->586709475],
Cell[28950, 1010, 936, 32, 72, "Output",
 CellID->3444046]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29947, 1049, 310, 9, 70, "SeeAlsoSection",
 CellID->7987],
Cell[30260, 1060, 698, 26, 70, "SeeAlso",
 CellID->28571]
}, Open  ]],
Cell[CellGroupData[{
Cell[30995, 1091, 314, 9, 70, "TutorialsSection",
 CellID->32541],
Cell[31312, 1102, 144, 3, 70, "Tutorials",
 CellID->21940]
}, Open  ]],
Cell[CellGroupData[{
Cell[31493, 1110, 319, 9, 70, "MoreAboutSection",
 CellID->545050297],
Cell[31815, 1121, 157, 3, 70, "MoreAbout",
 CellID->17649280],
Cell[31975, 1126, 174, 3, 70, "MoreAbout",
 CellID->251363774]
}, Open  ]],
Cell[32164, 1132, 27, 0, 70, "History"],
Cell[32194, 1134, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

