(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     37123,       1307]
NotebookOptionsPosition[     30488,       1076]
NotebookOutlinePosition[     31761,       1112]
CellTagsIndexPosition[     31675,       1107]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"KelvinKei" :> Documentation`HelpLookup["paclet:ref/KelvinKei"],
           "KelvinBer" :> Documentation`HelpLookup["paclet:ref/KelvinBer"], 
          "BesselK" :> Documentation`HelpLookup["paclet:ref/BesselK"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"KelvinKei\"\>", 
       2->"\<\"KelvinBer\"\>", 3->"\<\"BesselK\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"New in 6.0: Mathematical Functions\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["KelvinKer", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinKer",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinKer"], "[", 
       StyleBox["z", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"ker", "(", "z", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KelvinKer",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KelvinKer"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Kelvin function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["ker", "n"], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->151988642]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "For positive real values of parameters, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["ker", "n"], "(", "z", ")"}], "=", 
    RowBox[{"Re", "(", 
     RowBox[{
      SuperscriptBox["e", 
       RowBox[{
        RowBox[{"-", "n"}], "\[InvisibleSpace]", "\[Pi]", "\[InvisibleSpace]",
         
        RowBox[{"i", "/", "2"}]}]], 
      RowBox[{
       SubscriptBox["K", "n"], "(", 
       RowBox[{"z", "\[InvisibleSpace]", 
        SuperscriptBox["e", 
         RowBox[{"\[Pi]", "\[InvisibleSpace]", 
          RowBox[{"i", "/", "4"}]}]]}], ")"}]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ". For other values, ",
 Cell[BoxData[
  FormBox["ker", TraditionalForm]], "InlineMath"],
 " is defined by analytic continuation. "
}], "Notes",
 CellID->15190456],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinKer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKer"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->25396],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinKer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKer"], "[", 
   StyleBox["z", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["KelvinKer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKer"], "[", 
   RowBox[{"0", ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->416003445],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["KelvinKer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKer"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->262713514],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinKer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKer"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinKer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKer"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->701517911],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->21767794],

Cell["Evaluate numerically:", "ExampleText",
 CellID->387615674],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKer", "[", "2.5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->115455235],

Cell[BoxData[
 RowBox[{"-", "0.06968797258904491`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26470218]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKer", "[", 
  RowBox[{"2", ",", "2.5"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->316194421],

Cell[BoxData["0.18880430952581456`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->103649711]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->661358281],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["ker", "0"], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->543509965],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"KelvinKer", "[", 
    RowBox[{"0", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->377531937],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFWF1oHFUUvtmZTTR1jdFqY7Omsf6hYn0q4kMRFBUREQXxQXyIQUgeNDVN
/Un9aattrGlqfvpgQSz+489DxVIRKogFsSAWxIKoIBWkPlgEiw8BYf3OPXf2
zJw5O7NpFJfJzj1zz/nOd8/PvbO5a2hi5OFHhiZGh4cGbx8f2jgyOrxp8Lax
cTyKOpxzB/G3sc9h3MDI/x1sNHDPf6/DdwSFOu5xuJO8hmZp0PAf1x+ex97q
ejVbT8nOsK4reUDJHSbqJSVWqwNn7Y3RzlsiB1phBWLaK61l8F9B1/P14G0g
3OO2UImji0V2rpNu1RMn/jj2za/7X/sa1+aJQ64y9ujHvTdt2OeqoyMHoAUR
175XjroKNFz82eGfoI/r1Km/VqaYpH1FJqN+xbtCD7tOnvwTkAvzX8Id3D74
wHtwx76io1/9wr5cT6ihiIT47beOMeFtWw8TXxfdc/ebML71lldd5GXgYfqD
9791FbKIQDawOMuoFGIjnEna4Lkd/+63md1HQOn++96BK8DhibvCzy0u/s3c
cGEC3uDTpdlQEFme2vk582X+KTZrzUjdSF+dyAy8AwuR+ejA8R9/+H09sfMI
nxz6PkE01498hXnWB++Avcr0eIOvhSNf/AxfWC/cgaO72T9NrxSR97nx66py
yviiaiHfMTmtNjOHCxl2Z5te1/uaxFrgcsvkpyB5r3+CQdOa8VCZXIi4kAwX
UXAjrBcyfIXs1fD9LjVezlOk5HiZ823pd0FGGMEdi0M2ue247/5r3//32gvm
7V1Q79g8z7VBGe4250XW8xpPyzaLdZ4wKh4ti+v06cUxU+86LyOnYAeO7rGm
Fu0ePAvWaArouHET49pkhSgQOHKbTK1rvIwdBnRoOGFqXZ1gBa3NGT569nET
46pEC9VKwydMrSu9jA5F5yFO7klTy++RtM8NP/Qh7SRPmVqXexkoiABtU5MZ
1oyBXRWLp+GkiXGZl3GEYIOm4RavFTYE/0nXo5eTMzOFslbxfZp1M+dlVeHY
GFgKEk9DC8PqfRIv9QK2Xez4NHzGsO1SthUlDyY1iTVQZp49YwwUAI4CGloY
ft35JymUNV4GRKik53I56Qw5rirLgaQi0Fd0bmzNVATjohxQFDSU2XynI4vA
oL7aVqAl1WVr1RMsHDoU0+0ZPhpju4nRrzCeL9ASLFtrdRIfHPi0thcKtCRO
okWsmxjgQ7W+w8S4WOVwRy6HflhtWoQJnVHGwWaBBk/hlHZFWu7zMl6AUNk0
3GlgUA3G4U6mfSqerWzSHa25r1LRnjIwqg27ktlW9qWl20rMpnKx132sI3aR
l9FByDENX8xUAOPjXQ6vidnZPAaOMH55bKF1ofK0q0BLPO7K8GFPaNfwkmRj
rPQyn7k0fKlAC27grEwLlEG8tdYFam3TGdaMIXv2dAGG7KjTuVzq3FkVcX4j
VOKdd+ynPWS3UU3l+zKjoKhQWjRsH4VA2Fo6ajkcJNetUCIjCr0qHzOGrbbR
/hlD3oVmcvnoaqR+V+cs0c5oarFMqoHXhfRgo8jO5t9aZSfbU6Al+3eRllTW
ngyfXsXHxuhRfF4u0JJKL9JCWBHcMi3pPNEi1j0qw7MmxrlelrNpNpfDdnqK
UaSeZ41qKq9nRpFdsn0UAqk1sn09Z1i3OiVqKneWbTn/msrtclAk93O5jLR6
46up+M1l6oGji9/5CE52No1xjpflZJgv0JITuUhLzo/5DB9mK+/4RRh4aeB/
9LTQWuFlqECRhgsFWnAHp2Vacr6KFrFmPrJzLaSSmHz0LwErV+wF5YaiExxt
GZVWywqV1TPF6VZ537sEHFLoVhWxFPs8D4n+8nCkixinnS5iS7RQiOhes0pe
z1TFG17i/+W7jn8AamU+Sw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->211647488]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->127075355],

Cell["Series at the origin:", "ExampleText",
 CellID->4255939],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"KelvinKer", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->258655594],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "EulerGamma"}], "+", 
     RowBox[{"Log", "[", "2", "]"}], "-", 
     RowBox[{"Log", "[", "x", "]"}]}], ")"}], "+", 
   FractionBox[
    RowBox[{"\[Pi]", " ", 
     SuperscriptBox["x", "2"]}], "16"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "4"],
    SeriesData[$CellContext`x, 0, {}, 0, 4, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 
   0, {-EulerGamma + Log[2] - Log[$CellContext`x], 0, Rational[1, 16] Pi}, 0, 
   4, 1],
  Editable->False]], "Output",
 ImageSize->{292, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->207134093]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28004],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->550671982],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKer", "[", 
  RowBox[{
   RowBox[{"1", "-", "I"}], ",", "3.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->203028832],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "0.07416015678917912`"}], "-", 
  RowBox[{"0.13334846719333027`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{155, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->232659421]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->72025472],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->56601170],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"KelvinKer", "[", 
    RowBox[{"1", ",", "3"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3719057],

Cell[BoxData[
 RowBox[{"-", 
  "0.0498983077875149132543962741249984331229135449219314129665396355993`50."}\
]], "Output",
 ImageSize->{383, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->321038662]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->912312743],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->167520595],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKer", "[", 
  RowBox[{"1", ",", "3.000000000000000000000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->360044869],

Cell[BoxData[
 RowBox[{"-", 
  "0.049898307787514913254396274124998433122812621521`36.552001979238845"}]], \
"Output",
 ImageSize->{285, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->353892718]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->452982249],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->54933122],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinKer", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->126945261],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["ker", "n"], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{38, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->67906260]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17161],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KelvinKer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KelvinKer"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText",
 CellID->93991909],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KelvinKer", "[", 
  RowBox[{"1", ",", 
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], "+", 
    RowBox[{
     RowBox[{"O", "[", "x", "]"}], "^", "3"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13251630],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"-", 
    FractionBox["1", 
     RowBox[{
      SqrtBox["2"], " ", "x"}]]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", 
        RowBox[{"4", " ", 
         SqrtBox["2"]}]]}], "+", 
      FractionBox["EulerGamma", 
       RowBox[{"2", " ", 
        SqrtBox["2"]}]], "+", 
      FractionBox["\[Pi]", 
       RowBox[{"8", " ", 
        SqrtBox["2"]}]], "-", 
      FractionBox[
       RowBox[{"Log", "[", "2", "]"}], 
       RowBox[{"2", " ", 
        SqrtBox["2"]}]], "+", 
      FractionBox[
       RowBox[{"Log", "[", "x", "]"}], 
       RowBox[{"2", " ", 
        SqrtBox["2"]}]]}], ")"}], " ", "x"}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "3"],
    SeriesData[$CellContext`x, 0, {}, -1, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 
   0, {-2^Rational[-1, 2], 0, Rational[-1, 4] 2^Rational[-1, 2] + 
    Rational[1, 2] 2^Rational[-1, 2] EulerGamma + 
    Rational[1, 8] 2^Rational[-1, 2] Pi + 
    Rational[-1, 2] 2^Rational[-1, 2] Log[2] + 
    Rational[1, 2] 2^Rational[-1, 2] Log[$CellContext`x]}, -1, 3, 1],
  Editable->False]], "Output",
 ImageSize->{433, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->182785647]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->7064],

Cell["Solve the Kelvin differential equation:", "ExampleText",
 CellID->127419602],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "4"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "4", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["x", "3"], " ", 
      RowBox[{
       SuperscriptBox["f", 
        TagBox[
         RowBox[{"(", "3", ")"}],
         Derivative],
        MultilineFunction->None], "[", "x", "]"}]}], "-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"2", " ", 
         SuperscriptBox["n", "2"]}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "x"}], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
         SuperscriptBox["x", "2"], " ", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]\[Prime]",
           MultilineFunction->None], "[", "x", "]"}]}]}], ")"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "4"}], " ", 
         SuperscriptBox["n", "2"]}], "+", 
        SuperscriptBox["n", "4"], "+", 
        SuperscriptBox["x", "4"]}], ")"}], " ", 
      RowBox[{"f", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", 
   RowBox[{"f", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->61770610],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "4", "]"}], " ", 
      RowBox[{"KelvinBei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "3", "]"}], " ", 
      RowBox[{"KelvinBer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"KelvinKer", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{471, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->75949986]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->809257294],

Cell["\<\
Plot the radial density profile for AC current in a hollow cylinder: \
\>", "ExampleText",
 CellID->104196068],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{
     RowBox[{"KelvinKer", "[", 
      RowBox[{"0", ",", "r"}], "]"}], "+", 
     RowBox[{"I", " ", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"0", ",", " ", "r"}], "]"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "1", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->106593776],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVx384E3gcwPExv9owaTmSn6Gia+rocPH5iubHg9QRM1dTU0hx0mW5dv2Q
JGrJulNd3abyszITU+ppCumpJoT2EOaR2k1Wfhd23/vj/byfl92ulG3x2gQC
IQz3//8OTzEhENTQtMRq+dIwBah0ht8+JaqBMcN5uzpIARvrY4rP6KshaeV4
KPJXgNyOvmEJRQ1+mt7oNC8FmI0vj3WyVkPUfrt+pZMCePznJSEb1VDSMDK+
hqCAE3JHKMpQg4uP7PyYeBD2sHv3r/+shsjf1AObbAchzvYw787UZ/jpSISf
7cd+UNIqq7qUX8B8C7OXNdIHHncrH7jJx6G+Q5yw1qwXJF3ys719E9DBc9f2
9pdD89ZihmvbJAxZ+px2rO4BvUNFqpddUyDcWfHXNetuIBswkwQ90+CrHinS
3H0D1z/VDtu2zgCfFuysldYJ7KPISflyFn7hKdzJ6zpAspPN3in9ClN2IYxR
i3Z4Lfo5n1T1Df49ZS2csHgNlpWpmZrqOeij2EwnmbXBxOzKvf/cmId044Xc
ub2vQDNmaH62fAGGLsUHSq++AFnkQ3FFhQbCX30Jzpx+Dn/qZzk1hBLQWwqv
U+t0K+zweBzZIiMgo/jseH2TZzBa0HpyVZAWYtGrLfJbmoEvTSy880ILcURN
3bs4TWBTmnJZFqqN7JhGnKbNTyFVsuk6vVkbRfl433v14xOQX63MVgQSUc3L
rVrxDo1ArhhK7mwkInXd94KE9VKw+fVYYJyXDrptwq9K3/YYQsg1eksadNBI
LQx1yB8BktU8Weaui+InSv+QBTyE0L0huwMkuig2t8d3quIBbKwLmjH21EMl
LMNPfrT7MO5KzVsj0kPFu83LJ/ol0Mg9pJ3ygz5K7pzMOJdTB4yp4sP6t/XR
vGFbTjejFtodPRReDgaIJdycJ0P3gL16N+3iTQN0UGXTvt+2Bh5uZ2b6Wi9C
obNCY4K9GGjRyjKqYBEyeJZj7mpVDQkn6d3eliRU0BVQptwggqw9Bl+1LpJQ
tYozOcCqgnqrOI2SSkYOy97nriLdhXsL2UTnc2QUHLGQ1XLqNjwX+ynzFxsi
d+fKWPZ0BUht9zTnFRiiavRusOZUORyrixQMkoxQCTd81QefMoi2793BzDdC
bw7sYtuPlcB0Lc3SRd8YSe57R6xovgUZxI8v+FxjFBboZ10ougmmDp5xjgQK
Sr6Skv6t8AYcPLB9gsShIB6PPLaiqBhG80hJS+coSOdz8KGQS0Iwz/KlOmIT
j0ti12L7Z2Y8csPu7In2N8G+nPjeNAL72Lt11E6+EOj0xgcF2FSPzbVM7Gsa
jpHJPAWJO9XEfYVC2JL2sYq0QEFcD0bTmQIhHEm0ZS7DfmNqI96HfYsVpeuM
XVr2nSAUey6siRGEHU4WHF2MXeoi0M7BbiikehZdEAJhODpCV0NBolRufSlP
CC695xeo2HYfUspzsbd3NJc6YMeZFV5Jxq6Uus37Yffz07mu2NHXTG6dwJ4z
u0SXnBfCSX5AeAH2pJeV52XsO3ncrwLsUdNul9+xdTJVYVLsqD7FYoRNS1sx
24ZtIXPTtceOSYwpHsBOOFg/Q8TOZl0IVWOrWlNVw+eEIIp6Nq3BDm6PGWjB
/g/KZUsz
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  Method->{},
  PlotRange->{{1, 4}, {0., 0.5720315503209511}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->496192743]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->3660],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " to simplify expressions involving Kelvin functions:"
}], "ExampleText",
 CellID->540949068],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"KelvinKer", "[", 
     RowBox[{"1", ",", 
      RowBox[{
       SqrtBox["2"], "x"}]}], "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->275163834],

Cell[BoxData[
 RowBox[{
  RowBox[{"KelvinKer", "[", 
   RowBox[{"1", ",", 
    RowBox[{
     SqrtBox["2"], " ", "x"}]}], "]"}], "+", 
  RowBox[{
   FractionBox["1", "2"], " ", "x", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"KelvinKei", "[", 
       RowBox[{"0", ",", 
        RowBox[{
         SqrtBox["2"], " ", "x"}]}], "]"}]}], "+", 
     RowBox[{"KelvinKei", "[", 
      RowBox[{"2", ",", 
       RowBox[{
        SqrtBox["2"], " ", "x"}]}], "]"}], "-", 
     RowBox[{"KelvinKer", "[", 
      RowBox[{"0", ",", 
       RowBox[{
        SqrtBox["2"], " ", "x"}]}], "]"}], "+", 
     RowBox[{"KelvinKer", "[", 
      RowBox[{"2", ",", 
       RowBox[{
        SqrtBox["2"], " ", "x"}]}], "]"}]}], ")"}]}]}]], "Output",
 ImageSize->{452, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->155880464]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->71003703],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "x"}], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"KelvinKei", "[", 
     RowBox[{"0", ",", 
      RowBox[{
       SqrtBox["2"], " ", "x"}]}], "]"}], "+", 
    RowBox[{"KelvinKer", "[", 
     RowBox[{"0", ",", 
      RowBox[{
       SqrtBox["2"], " ", "x"}]}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{307, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->398010284]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->126804148],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand Kelvin functions of half-integer orders:"
}], "ExampleText",
 CellID->127473317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"KelvinKer", "[", 
   RowBox[{
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->788709929],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox["x", 
      SqrtBox["2"]]}]], " ", 
   SqrtBox[
    FractionBox["\[Pi]", "2"]], " ", 
   RowBox[{"Sin", "[", 
    RowBox[{
     FractionBox["\[Pi]", "8"], "-", 
     FractionBox["x", 
      SqrtBox["2"]]}], "]"}]}], 
  SqrtBox["x"]]], "Output",
 ImageSize->{162, 64},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->266971600]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->18299],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinKei",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinKei"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["KelvinBer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/KelvinBer"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselK",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselK"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->21127]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->90],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->18616]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->44731689],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->22011240],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->300676695],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->85435555]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"KelvinKer - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 51, 47.8882068}", "context" -> "System`", 
    "keywords" -> {"Kelvin functions", "Kelvin ker"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "KelvinKer[z] gives the Kelvin function ker(z). KelvinKer[n, z] gives the \
Kelvin function ker_n (z). ", "synonyms" -> {"kelvin ker"}, "title" -> 
    "KelvinKer", "type" -> "Symbol", "uri" -> "ref/KelvinKer"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8791, 286, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->701517911]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 31532, 1100}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2329, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2952, 82, 1574, 37, 70, "ObjectNameGrid"],
Cell[4529, 121, 1084, 36, 70, "Usage",
 CellID->151988642]
}, Open  ]],
Cell[CellGroupData[{
Cell[5650, 162, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6128, 179, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6251, 185, 803, 26, 70, "Notes",
 CellID->15190456],
Cell[7057, 213, 567, 21, 70, "Notes",
 CellID->25396],
Cell[7627, 236, 445, 17, 70, "Notes",
 CellID->416003445],
Cell[8075, 255, 250, 8, 70, "Notes",
 CellID->262713514],
Cell[8328, 265, 220, 7, 70, "Notes",
 CellID->31226],
Cell[8551, 274, 203, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8791, 286, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->701517911],
Cell[CellGroupData[{
Cell[9176, 300, 147, 5, 70, "ExampleSection",
 CellID->21767794],
Cell[9326, 307, 64, 1, 70, "ExampleText",
 CellID->387615674],
Cell[CellGroupData[{
Cell[9415, 312, 107, 3, 28, "Input",
 CellID->115455235],
Cell[9525, 317, 193, 6, 36, "Output",
 CellID->26470218]
}, Open  ]],
Cell[CellGroupData[{
Cell[9755, 328, 130, 4, 28, "Input",
 CellID->316194421],
Cell[9888, 334, 177, 5, 36, "Output",
 CellID->103649711]
}, Open  ]],
Cell[10080, 342, 125, 3, 70, "ExampleDelimiter",
 CellID->661358281],
Cell[10208, 347, 191, 9, 70, "ExampleText",
 CellID->543509965],
Cell[CellGroupData[{
Cell[10424, 360, 244, 8, 28, "Input",
 CellID->377531937],
Cell[10671, 370, 2128, 39, 133, "Output",
 Evaluatable->False,
 CellID->211647488]
}, Open  ]],
Cell[12814, 412, 125, 3, 70, "ExampleDelimiter",
 CellID->127075355],
Cell[12942, 417, 62, 1, 70, "ExampleText",
 CellID->4255939],
Cell[CellGroupData[{
Cell[13029, 422, 221, 7, 70, "Input",
 CellID->258655594],
Cell[13253, 431, 734, 24, 54, "Output",
 CellID->207134093]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14036, 461, 216, 7, 70, "ExampleSection",
 CellID->28004],
Cell[14255, 470, 85, 1, 70, "ExampleText",
 CellID->550671982],
Cell[CellGroupData[{
Cell[14365, 475, 154, 5, 70, "Input",
 CellID->203028832],
Cell[14522, 482, 272, 8, 36, "Output",
 CellID->232659421]
}, Open  ]],
Cell[14809, 493, 124, 3, 70, "ExampleDelimiter",
 CellID->72025472],
Cell[14936, 498, 69, 1, 70, "ExampleText",
 CellID->56601170],
Cell[CellGroupData[{
Cell[15030, 503, 181, 6, 70, "Input",
 CellID->3719057],
Cell[15214, 511, 253, 8, 36, "Output",
 CellID->321038662]
}, Open  ]],
Cell[15482, 522, 125, 3, 70, "ExampleDelimiter",
 CellID->912312743],
Cell[15610, 527, 113, 3, 70, "ExampleText",
 CellID->167520595],
Cell[CellGroupData[{
Cell[15748, 534, 171, 5, 70, "Input",
 CellID->360044869],
Cell[15922, 541, 249, 8, 36, "Output",
 CellID->353892718]
}, Open  ]],
Cell[16186, 552, 125, 3, 70, "ExampleDelimiter",
 CellID->452982249],
Cell[16314, 557, 202, 7, 70, "ExampleText",
 CellID->54933122],
Cell[CellGroupData[{
Cell[16541, 568, 167, 5, 70, "Input",
 CellID->126945261],
Cell[16711, 575, 255, 8, 50, "Output",
 CellID->67906260]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17015, 589, 239, 7, 70, "ExampleSection",
 CellID->17161],
Cell[17257, 598, 212, 7, 70, "ExampleText",
 CellID->93991909],
Cell[CellGroupData[{
Cell[17494, 609, 239, 8, 70, "Input",
 CellID->13251630],
Cell[17736, 619, 1343, 44, 58, "Output",
 CellID->182785647]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19128, 669, 222, 7, 70, "ExampleSection",
 CellID->7064],
Cell[19353, 678, 82, 1, 70, "ExampleText",
 CellID->127419602],
Cell[CellGroupData[{
Cell[19460, 683, 1487, 49, 70, "Input",
 CellID->61770610],
Cell[20950, 734, 812, 26, 54, "Output",
 CellID->75949986]
}, Open  ]],
Cell[21777, 763, 125, 3, 70, "ExampleDelimiter",
 CellID->809257294],
Cell[21905, 768, 120, 3, 70, "ExampleText",
 CellID->104196068],
Cell[CellGroupData[{
Cell[22050, 775, 403, 13, 70, "Input",
 CellID->106593776],
Cell[22456, 790, 2125, 44, 136, "Output",
 CellID->496192743]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24630, 840, 232, 7, 70, "ExampleSection",
 CellID->3660],
Cell[24865, 849, 246, 8, 70, "ExampleText",
 CellID->540949068],
Cell[CellGroupData[{
Cell[25136, 861, 247, 9, 70, "Input",
 CellID->275163834],
Cell[25386, 872, 878, 31, 73, "Output",
 CellID->155880464]
}, Open  ]],
Cell[CellGroupData[{
Cell[26301, 908, 107, 3, 70, "Input",
 CellID->71003703],
Cell[26411, 913, 462, 17, 43, "Output",
 CellID->398010284]
}, Open  ]],
Cell[26888, 933, 125, 3, 70, "ExampleDelimiter",
 CellID->126804148],
Cell[27016, 938, 249, 8, 70, "ExampleText",
 CellID->127473317],
Cell[CellGroupData[{
Cell[27290, 950, 195, 6, 70, "Input",
 CellID->788709929],
Cell[27488, 958, 487, 19, 85, "Output",
 CellID->266971600]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28036, 984, 311, 9, 70, "SeeAlsoSection",
 CellID->18299],
Cell[28350, 995, 698, 26, 70, "SeeAlso",
 CellID->21127]
}, Open  ]],
Cell[CellGroupData[{
Cell[29085, 1026, 311, 9, 70, "TutorialsSection",
 CellID->90],
Cell[29399, 1037, 144, 3, 70, "Tutorials",
 CellID->18616]
}, Open  ]],
Cell[CellGroupData[{
Cell[29580, 1045, 318, 9, 70, "MoreAboutSection",
 CellID->44731689],
Cell[29901, 1056, 157, 3, 70, "MoreAbout",
 CellID->22011240],
Cell[30061, 1061, 174, 3, 70, "MoreAbout",
 CellID->300676695],
Cell[30238, 1066, 178, 3, 70, "MoreAbout",
 CellID->85435555]
}, Open  ]],
Cell[30431, 1072, 27, 0, 70, "History"],
Cell[30461, 1074, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

