(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     68438,       2378]
NotebookOptionsPosition[     57778,       2005]
NotebookOutlinePosition[     59126,       2044]
CellTagsIndexPosition[     59040,       2039]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Vectors and Matrices" :> 
          Documentation`HelpLookup["paclet:tutorial/VectorsAndMatrices"], 
          "Basic Matrix Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/BasicMatrixOperations"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Vectors and Matrices\"\>", 
       2->"\<\"Basic Matrix Operations\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Outer" :> Documentation`HelpLookup["paclet:ref/Outer"], 
          "Cross" :> Documentation`HelpLookup["paclet:ref/Cross"], "Dot" :> 
          Documentation`HelpLookup["paclet:ref/Dot"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Outer\"\>", 2->"\<\"Cross\"\>", 
       3->"\<\"Dot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Matrix Operations" :> 
          Documentation`HelpLookup["paclet:guide/MatrixOperations"], 
          "Operations on Vectors" :> 
          Documentation`HelpLookup["paclet:guide/OperationsOnVectors"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Matrix Operations\"\>", 
       2->"\<\"Operations on Vectors\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["KroneckerProduct", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KroneckerProduct",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KroneckerProduct"], "[", 
       RowBox[{
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]constructs the Kronecker product of the arrays ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["m", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 " works on vectors, matrices or in general full arrays of any depth."
}], "Notes",
 CellID->5543913],

Cell[TextData[{
 "For matrices, ",
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 " gives the matrix direct product."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 " can be used on ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects, returning a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " object when possible. ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->775152956,
  ButtonNote->"775152956"]
}], "Notes",
 CellID->20799]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->348985882],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->232321335],

Cell["Kronecker product of vectors:", "ExampleText",
 CellID->247564214],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerProduct", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->571197403],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"a", " ", "x"}], ",", 
     RowBox[{"a", " ", "y"}], ",", 
     RowBox[{"a", " ", "z"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"b", " ", "x"}], ",", 
     RowBox[{"b", " ", "y"}], ",", 
     RowBox[{"b", " ", "z"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"c", " ", "x"}], ",", 
     RowBox[{"c", " ", "y"}], ",", 
     RowBox[{"c", " ", "z"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{303, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->59959681]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->173717615],

Cell["Matrix direct product:", "ExampleText",
 CellID->471934619],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"a", ",", "b"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"c", ",", "d"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"r", ",", "s"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "u"}], "}"}]}], "}"}]}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->593051083],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWL1u2zAQpiXLluMALYIiLTpl6NAHaIe0ezv1IYygRTIUNdIM3jx69OjR
j+DRo0ePHj169OjRI3sk7yzqTCp0pABqK0AijzreD3n8eOS33sPt95+9h7ub
3tXX+17/9u7m99WXX/fQFDeEEH14r98IqEuo6bcvJZT2N1HfeDqdSv2IGD4x
sKZYdrFsQxlBqRmEec6gmkDZsngVnTh4jZ7hcGjpEaKpq+v1ej6fH1qVCEmP
QPnHrR1nq9FzaBUR2lm91BaN3Gq14h5NJhPbT6tnyuguo9tO/QUecFlu2z19
9/v9YDDIrI+p1TiA3tTPeq0nHo1GYKqej4P10XK5LPL4afrOWN/IJ4/xFFkP
IQ+Bn7deR852u9Xd4NNEGQmuLkWrOVGyIyy7Mr/aaN4Sz7+Y9TfWZGvQtiZb
ry+Yd+8ZzSOF8xsapstAjUfLNdr0GssLGhP833D2+sHo/5dLxUqMsUnzT7SK
nXNst+OD6Cah36Na9FxXv74qlZiGe3ICvnE6LWF3h3G1nsViex9/qqW0v5eN
jggjsCzqHuQEoW1YJNhoy3MXQl5aRT7EdfXjfKQnzKpPzMtLRr8MkuJG1fqh
V125aAcOyXNT/EfrroO0K39uB2kvQNva5cQnoO4/sX/QauaazmWGJvWx9nnG
nXLJomyw6+Fzn/fCx96V6yQyn9/q0W4i/Aph9Tj+w/uGWcJz3XeMfmyPfhWk
5aPMo/hlUK+6YuopXOPxOIAru8OwueDssNvtpMxQ3Jf72rcZiYXctOubWdxs
NrPZTFXtEwwcnfAW4G/ANrPvLBYLx/kTBgv9K+MJz22rzJPNvgMxkTv783sX
mlUfMtt7edm8mKNEZlEEUVFtPuxDW5IVjrLaAphrbR+Oav7eKrbs5Hkt3QnS
yjlGWNdtYZ6L+rccuskicxmiqh+YVzwXvmD8RspnmUfNt7rV3H2Kxh/3E51r
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{112, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->362214281]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " are matrices with exact entries:"
}], "ExampleText",
 CellID->238706176],

Cell[BoxData[{
 RowBox[{
  RowBox[{"a", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->330562928],

Cell["Use exact arithmetic to compute the Kronecker product:", "ExampleText",
 CellID->24116838],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{"a", ",", "b"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->12531995],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"0", "0", "1", "2"},
     {"0", "0", "3", "4"},
     {
      RowBox[{"-", "1"}], 
      RowBox[{"-", "2"}], "0", "0"},
     {
      RowBox[{"-", "3"}], 
      RowBox[{"-", "4"}], "0", "0"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{92, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->25815915]
}, Open  ]],

Cell["Use machine arithmetic:", "ExampleText",
 CellID->1051321697],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{
    RowBox[{"N", "[", "a", "]"}], ",", 
    RowBox[{"N", "[", "b", "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->266741744],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"0.`", "0.`", "1.`", "2.`"},
     {"0.`", "0.`", "3.`", "4.`"},
     {
      RowBox[{"-", "1.`"}], 
      RowBox[{"-", "2.`"}], "0.`", "0.`"},
     {
      RowBox[{"-", "3.`"}], 
      RowBox[{"-", "4.`"}], "0.`", "0.`"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{120, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->552440390]
}, Open  ]],

Cell["Use 20-digit precision arithmetic:", "ExampleText",
 CellID->134884688],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{
    RowBox[{"N", "[", 
     RowBox[{"a", ",", "20"}], "]"}], ",", 
    RowBox[{"N", "[", 
     RowBox[{"b", ",", "20"}], "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->244180237],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"0", "0", "1.`19.69897000433602", "2.`19.69897000433602"},
     {"0", "0", "3.`19.69897000433602", "4.`19.69897000433602"},
     {
      RowBox[{"-", "1.`19.69897000433602"}], 
      RowBox[{"-", "2.`19.69897000433602"}], "0", "0"},
     {
      RowBox[{"-", "3.`19.69897000433602"}], 
      RowBox[{"-", "4.`19.69897000433602"}], "0", "0"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{652, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]//MatrixForm=",
 CellID->393795905]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->248818974],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["t", "TI"]], "InlineFormula"],
 " are sparse matrices:"
}], "ExampleText",
 CellID->310187793],

Cell[BoxData[{
 RowBox[{
  RowBox[{"s", " ", "=", " ", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"i_", ",", " ", "i_"}], "}"}], "\[Rule]", "i"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "3"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"t", " ", "=", " ", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", " ", "j_"}], "}"}], "/;", 
        RowBox[{
         RowBox[{"Abs", "[", 
          RowBox[{"i", "-", "j"}], "]"}], "==", "1"}]}], "\[Rule]", 
       RowBox[{"i", "-", "j"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "4"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->21156748],

Cell["Compute the sparse Kronecker product:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->775152956],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerProduct", "[", 
  RowBox[{"s", ",", "t"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->253117605],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "18", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 18, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"12", ",", "12"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->35939343]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", "%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->328089509],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztnb1y2kAQx4UwCcR2Yhw7k0lIxk+TVHkIxpMZu/Ak47igo6SkVMkjMKlU
UlJSUlJSUtJFuQ8JHTsnOKGv0916BnSclkP66Xbvr9VZ+tF/efj51H95vO/f
fX/u/354vP9z9+3XM6lqNhyn0XYc5+9Hh5QDUmIv+tchL16EpRZ9b04mE2ZH
iuStSb52ES5vEpanrn/HfmQ4HAq/5zhnrLhcLn3f39W2oq1nhuTvjbT2OnOt
vu2+jY7OYrGAtDzPExnWaK/StPuB7eCu1rk6wGm73Q4Gg5hTM6rlqDg3qzlx
7xuNRgQKLbo7Tu58Pm9psB9FtUvjj0uW7w9wIcGHhKB9LszP1ut1gP1nxymO
0yKnOKabzClNC5wWcTU+vCbQol5X5IjbDYK90aIh3Yp/4LO9VkUdj1647Cht
hX0+9Ela27aIFlcxx2Iw/S6tfA2sXyEpYPXVYkI0+iTrnV641qZYhH0mz3az
RGs9Rlhd9Uf9FU8PfO4qbYXJPiT3FrntlZG04PllO/G7Mm0jt7aZFBynbCIU
5XLOpWt7B9eaSQT7TJ7t5h+t9RhhddUf9VU8SXY2KZ7sv2ZbxFFXOMhLXTfa
zemwJkI+2I/K+rVyFVG2EVhXfaK/IqJ55Va4VL8KaYcPZW+3PhGHXV8Q4qr8
egNyugGcsD/J270FnLrISVqrNl8HOenQbjU5otNGYF31ib6KiJ5xiUroc/j5
PJfjoa8PpcmSmhlx2Lm2kvahlWfAWn6mbiapC7DvcqsesLKpL12CfZfrni/A
qmMRodPm78jzPGYSqnv0PlUBwathqIDKUUBRDihSOlD5mJMTSuMtZp5zwTxP
8nUGda1jJimY6ZG3qKZ1zCQEczxyFaOmdcwklOd8HjMJmRm90yogqHwwB1RN
DggqoXxzcuX6UFEeYG7Eyap4bOKlpo6Qk5pGQk5Z5vXYxKnuUV11JD424qIy
KkYZpZ0PVL4yqnYGRn1iDXJCTsgJOSEn/TjJVBCd9X8blDnq6qpJqlNBp84B
KlIF6TXnQsf4okYouqbVQlLYl5AQEkJCJSsdUeFEd3e4DFDplGlFe6yoaNLO
9cnnePC7cR3r+9XOrajWb9QIpdM0NpPCvoSEkBASKo4QjcGiooEKB3M6qlbj
8VjBKn7WhWjled5ms4mOR5Y5PcfW816xWq2m0yktincF930/fNKD/AxBx9kT
RbXL754+m80k9+Qnhyqkl5w5zqpv6smL+MDesx3UnhViX7+Ko0LMySW+h35H
/8/KJd7FSIQeFPcfF/kAIk2BE39QCHKKOcVEOCdxNJWpnKyjK5s3xzaEP9/K
afwHxIeX8Q==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{264, 177},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->284324807]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 "Solve the general linear matrix equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["a", "1"], ".", "x", ".", 
      SubscriptBox["b", "1"]}], "+", "\[CenterEllipsis]", "+", 
     RowBox[{
      SubscriptBox["a", "m"], ".", "x", ".", 
      SubscriptBox["b", "m"]}]}], "\[LongEqual]", "c"}], TraditionalForm]], 
  "InlineMath"],
 " for matrix ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->154989736],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["a", "1"], ",", 
     SubscriptBox["a", "2"], ",", 
     SubscriptBox["b", "1"], ",", 
     SubscriptBox["b", "2"], ",", "c"}], "}"}], "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "2", ",", "2"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->92801958],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "=", 
  RowBox[{"Partition", "[", 
   RowBox[{
    RowBox[{"LinearSolve", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"KroneckerProduct", "[", 
        RowBox[{
         SubscriptBox["a", "1"], ",", 
         RowBox[{
          SubscriptBox["b", "1"], "\[Transpose]"}]}], "]"}], "+", 
       RowBox[{"KroneckerProduct", "[", 
        RowBox[{
         SubscriptBox["a", "2"], ",", 
         RowBox[{
          SubscriptBox["b", "2"], "\[Transpose]"}]}], "]"}]}], ",", 
      RowBox[{"Flatten", "[", "c", "]"}]}], "]"}], ",", "2"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3349867],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "8.06834976578478`"}], ",", "6.192355626942089`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.3079531665076568`"}], ",", "2.9315333952978593`"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{298, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->354121891]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    SubscriptBox["a", "1"], ".", "x", ".", 
    SubscriptBox["b", "1"]}], "+", 
   RowBox[{
    SubscriptBox["a", "2"], ".", "x", ".", 
    SubscriptBox["b", "2"]}], "-", "c"}], "//", "Chop"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->289590767],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->2511561]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->237818671],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " are nonsingular matrices:"
}], "ExampleText",
 CellID->32209709],

Cell[BoxData[{
 RowBox[{
  RowBox[{"a", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->184146917],

Cell["The Kronecker product is also nonsingular:", "ExampleText",
 CellID->241337795],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", 
  RowBox[{"ab", " ", "=", " ", 
   RowBox[{"KroneckerProduct", "[", 
    RowBox[{"a", ",", "b"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->89819377],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->147588273]
}, Open  ]],

Cell[TextData[{
 "The inverse of the product can be computed from the simpler inverses of ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->599476156],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerProduct", "[", 
  RowBox[{
   RowBox[{"Inverse", "[", "a", "]"}], ",", " ", 
   RowBox[{"Inverse", "[", "b", "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->300624452],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "2", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"-", 
      FractionBox["3", "2"]}], ",", 
     FractionBox["1", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "2"}], ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["3", "2"], ",", 
     RowBox[{"-", 
      FractionBox["1", "2"]}], ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{402, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->460284671]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Inverse", "[", "ab", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->679572052],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "2", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"-", 
      FractionBox["3", "2"]}], ",", 
     FractionBox["1", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "2"}], ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["3", "2"], ",", 
     RowBox[{"-", 
      FractionBox["1", "2"]}], ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{402, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->64131251]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->451110104],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " is a differentiation matrix approximating the second derivative in 1 \
dimension:"
}], "ExampleText",
 CellID->561987471],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", " ", "=", " ", "4"}], ";"}], "\n", 
 RowBox[{"s", " ", "=", " ", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", " ", "i_"}], "}"}], "\[Rule]", 
       RowBox[{"-", "2"}]}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", " ", "j_"}], "}"}], "/;", 
        RowBox[{
         RowBox[{"Abs", "[", 
          RowBox[{"i", "-", "j"}], "]"}], "\[Equal]", "1"}]}], "\[Rule]", 
       "1"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "n"}], "}"}]}], "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->45633183],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "10", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 10, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "4"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->481672895]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", ".", 
  RowBox[{"Array", "[", 
   RowBox[{
    RowBox[{
     SubscriptBox["u", "#"], "&"}], ",", " ", "n"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13084088],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", 
     SubscriptBox["u", "1"]}], "+", 
    SubscriptBox["u", "2"]}], ",", 
   RowBox[{
    SubscriptBox["u", "1"], "-", 
    RowBox[{"2", " ", 
     SubscriptBox["u", "2"]}], "+", 
    SubscriptBox["u", "3"]}], ",", 
   RowBox[{
    SubscriptBox["u", "2"], "-", 
    RowBox[{"2", " ", 
     SubscriptBox["u", "3"]}], "+", 
    SubscriptBox["u", "4"]}], ",", 
   RowBox[{
    SubscriptBox["u", "3"], "-", 
    RowBox[{"2", " ", 
     SubscriptBox["u", "4"]}]}]}], "}"}]], "Output",
 ImageSize->{300, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->140687098]
}, Open  ]],

Cell["The identity matrix as a sparse array:", "ExampleText",
 CellID->68705780],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sid", " ", "=", " ", 
  RowBox[{"N", "[", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"i_", ",", " ", "i_"}], "}"}], "\[Rule]", "1"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "n"}], "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->19348395],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "4", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 4, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "4"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->271907544]
}, Open  ]],

Cell["A 2-dimensional array of values:", "ExampleText",
 CellID->40948944],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"Array", "[", 
    RowBox[{
     RowBox[{
      SubscriptBox["u", "##"], "&"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "n"}], "}"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->17929735],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWb2O2kAQNjYGEjjpgMvlEqW4F0mbVHkIdIp0V0SJ7q5AKUIJHaVLHoHS
JSUlJSUlJSWlM/tnr8ezvvUZnxACCXtn52dndmb3W5Yfg+f7n78Gzw93g9vv
j4M/9w93T7fffj9Cl1dzHGcC3683DrQjaPHvJIrg/ZE9PUbzj/OlIN3lVnrw
ZJ0u4voO4tVjnuztG7kuEFcGu67s7/DRfc6dzWaS2yakuSuxbWg7HyipxANo
XFMSOXSvoLzwezQaSdrjw9Z5c71eh2HIXZJzIRj8Aw3maBPZa7+pZEN789lf
rVY4iiAI9Nj0uUBZZYZbSOZ9RTrvkAymKZ0mSe/3++FwmETtqV4RuJiFTOV5
ck5J343S2IP2AaVbKY/G4zGEEKmKFVLL5dInbdRRFL601bGSbljNwWG0cG0L
K7DMYLGlo+WVu91uzTn8XLpX7LzJOtdHT+8Jqh7Z+1NBWowC6RTbo2GUfypm
SddypBL6FKU4npXAwwurUdSumlk70OzqvAxe9oxc1ts32FV42bLy7oLQPkb8
bFpFo/YjuUXzD20podtHpKNw1i53mczb7p4V6OBoKZyl0COL1VZRZyrU6qzw
opYdkh5GS+FwmajxSasR5eExlsY5M59JymvhfNit56rw+MZq9LJ4bDfKsaLj
KeOxR1SowuOYR+IxzVV4TNkthsf49yy9AvqUVAqPryiJHHy9LCh/xmM7PLba
PSvQwfsthccUepzx+IzHRaM+4/FpSr0NHrN69Ym9l/Vd6rwMHneNXJbtnsFu
Go+n02mudx2k7SnX4rFY/WK0RScEJ/sLmjol6DSe+5fkxfpN7sX1aIIg2O12
3PHoeLC1DB4L3c1mM5/P49mQ0YZhKG+nTw2PhY3FYkHcYUKC5VzYIdPrUAN7
ZL5pLq/FPEzuqWGdpu6pyf8cSIzX77xpHFZSOCfmM0pxaVwjYk6SfSaJyoXq
rQpn2S2dC3Ui/uDh9y/1eA6jcuh6TWVE3vGI63XW/ItoITXRnk7tP+vLhUs=
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{136, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]//MatrixForm=",
 CellID->440057243]
}, Open  ]],

Cell["A matrix that differentiates in the first dimension only:", \
"ExampleText",
 CellID->105732674],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m1", " ", "=", " ", 
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{"s", ",", "sid"}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->32411629],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "40", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 40, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"16", ",", "16"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->201984605]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Partition", "[", 
    RowBox[{
     RowBox[{"m1", ".", 
      RowBox[{"Flatten", "[", "v", "]"}]}], ",", "n"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{
     RowBox[{"2", ";;", 
      RowBox[{"-", "2"}]}], ",", 
     RowBox[{"2", ";;", 
      RowBox[{"-", "2"}]}]}], "]"}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->75204836],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWclOwzAQdbNQKlrUC4tAlUoFtGURgjNc4cRHRBVSe0Cg0v8PniyV9TKu
EycQJZUSe+zx88x4HPm578Fm+fEZbFaLYPy2Dr6Xq8XP+PVrLZvcjhCdZyHE
6FTIeihr0UO/F/kIMaO3m7aF4hHkJ5DvCuoPo1km8k2NLknetjdpncrSkaXH
9s6T0mF79ciEONMiU+uNFpl6B5HlPvhzBjIgy8otaDgZDYxhVuOSmwVHKPLU
0I8yrrpJ3+XsyVhFWn6yFhTB/aSMh8c/RO6CfFgBBsq+DtPbYsoKLQCO7LXO
I8xnl11JTsth0JzUDYP/wld3l1As5xDRmr2C0dmFrUZaJJFTdQ5KeI5Yg9Ke
o44pEkU874EOt5vNmdFnfcS5uiCTHzQI0fiI5UVDr3nbuBhRe7+1nmHWxCuL
2Yta+jl2ZeCwEhSTzKOavhu4t9vnIcpRflmu9AUXvYzWiZUf57VbcGwVKbIg
/ZZSOYH+HvRzHqj9GIeq8e28fGAjZmIC+UaxJ9WMRX5iP412k5Lke037PBcq
vdCeZmtiflTN3vCLyVtBWewLHUObhru4n56huUl+88gpu+OR05zQcz88q/Ge
jUCGmZjMy/JBM9u74mbBEYqM+iZ2WJTt4dmFj40XNowbFcBsHO8rgGl3Rvhf
Blj29FSGXdbLAE2IZRigiV3m2/d2PEEdRf118iQTWj4m2GQPM9lRYqXTsybe
J9TJk0yo+ZhgEz3jv+52tzjtZoBHoJXvvqtpDNDOy79kgNesRbob57wMEHll
fBFO1fh/LtH5BaZS/mY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{316, 29},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]//MatrixForm=",
 CellID->211593949]
}, Open  ]],

Cell["A matrix that approximates the Laplacian:", "ExampleText",
 CellID->73727274],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"lap", " ", "=", " ", 
  RowBox[{
   RowBox[{"KroneckerProduct", "[", 
    RowBox[{"s", ",", " ", "sid"}], "]"}], "+", " ", 
   RowBox[{"KroneckerProduct", "[", 
    RowBox[{"sid", ",", " ", "s"}], "]"}]}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->338802460],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "64", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 64, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"16", ",", "16"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->718456294]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Partition", "[", 
    RowBox[{
     RowBox[{"lap", ".", 
      RowBox[{"Flatten", "[", "v", "]"}]}], ",", "n"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{
     RowBox[{"2", ";;", 
      RowBox[{"-", "2"}]}], ",", 
     RowBox[{"2", ";;", 
      RowBox[{"-", "2"}]}]}], "]"}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->285829],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWutu0zAU9tKkMK3cxUWgSd0Eu3CRYBKvAL94iGpC2n4g0Nj7B5/ELunn
42vcJWobqXVsn/P5XHxJv/TH4vbq56/F7fXlYv79ZvHn6vry7/zb7xvZNNkT
opgJIQ5fCXlfy7vmo6+H8iPEGX1Plu3iC9QvPP1Y/5gZ73Fj5bH8psYJ1cqu
F9R6KstCliXoVtjn1lzppdZzJVWwunabSPfMOa6919Dt+FN0rObtIq0HTcQq
0H0NdRhZ3nxgRmtCuEQXNik/jqN+6kV8y9neCxFnaZKNGeUnnIeGnyRVqRlH
mb6vyla9vRD5nqdeZsB8Wv+fj6kYU6hXGewKxiyXmPKGJgoi7e8i78Scqrjl
izjuX+3gxg7PSBUMWqHNj8gP+uLCRrtQ91ldMzt1ChLWESPGSl/MY7C7M0Ko
THdlDnpkCrF8MRkmUzmj6ZrngtHpk6l9kOFOHf/KmzWN3bjhPoF2C7VfVMy+
EYaG80LvNlROZXmQZJNGwdileRiHxuWN2mcjiLYNbcho6/MkT5S5fcTcjcKi
1/e0CUOdKk+esNJplvlOmLtE9e3WeKoMmQtEecRqDZkTbr/MnxOsN2s2cRW9
TNKaU7nye4+Xe5Hkf5rWmy3zF7XCZgGtZ/2MUmjPHP1HnX4aNVbf149ZG5t9
aVF2cXJkZU6OLwSPZUsMLyrl84Uqv6ryM7R/CpQ7Dxq1Vhe2b5b0sSdDQ7PA
eM7zXtBqrYSNr9WsqTGvsM+tGcn0Hllt0rr2ce29hi7EhqonVrs0E+OP6CHU
wRJmFZU6pMvRhE3Kj4P2+faMFcR3nO0ORJQ3bXwfaSOLGOuTo47Pb3wOc/CM
m8xd3gXmjj92Y0bzxz3sC3tGGopJ9mGH85N9f1P2YWhzcp++3/xDMcnry1TO
aLrmuWB01vk+oQ8T4PNjxyTHoYUxyuuN+uYwyj6bjNVTp68EfVbje5sdo5wD
NYxRXm8OtoVJjrW1XTVpbyI3h1ndNib5OUiFvY8eG1M7diY5LcpjY5JPWC9s
/xLKxSQjb9P+ualWV/s/W7H3D4e6XP8=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{524, 29},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]//MatrixForm=",
 CellID->386526411]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->200860902],

Cell[TextData[{
 "Define a ",
 Cell[BoxData[
  RowBox[{"2", 
   StyleBox["l", "TI"], "\[Times]", "2", 
   StyleBox["l", "TI"]}]], "InlineFormula"],
 " \"butterfly\" matrix:"
}], "ExampleText",
 CellID->344424701],

Cell[BoxData[
 RowBox[{
  SubscriptBox["b", "l_"], ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"h", " ", "=", " ", 
       RowBox[{"l", "/", "2"}]}], ",", " ", "id", ",", "w"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"id", "  ", "=", " ", 
      RowBox[{"IdentityMatrix", "[", "h", "]"}]}], ";", "\[IndentingNewLine]",
      " ", 
     RowBox[{"w", " ", "=", " ", 
      RowBox[{"DiagonalMatrix", "[", 
       RowBox[{"Exp", "[", 
        RowBox[{"Pi", " ", "I", " ", 
         RowBox[{
          RowBox[{"Range", "[", 
           RowBox[{"0", ",", 
            RowBox[{"h", "-", "1"}]}], "]"}], "/", "h"}]}], "]"}], "]"}]}], 
     ";", "\[IndentingNewLine]", 
     RowBox[{"ArrayFlatten", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"id", ",", " ", "w"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"id", ",", " ", 
          RowBox[{"-", "w"}]}], "}"}]}], "}"}], "]"}]}]}], 
   "\[IndentingNewLine]", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->182487478],

Cell[TextData[{
 "Define the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["n", "TI"], "\[Times]", 
   StyleBox["n", "TI"]}]], "InlineFormula"],
 " \"bit reversal\" permutation matrix for ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " a power of 2:"
}], "ExampleText",
 CellID->314804321],

Cell[BoxData[
 RowBox[{
  SubscriptBox["p", "n_"], " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"r", " ", "=", " ", 
       RowBox[{"Range", "[", 
        RowBox[{"0", ",", " ", 
         RowBox[{"n", " ", "-", " ", "1"}]}], "]"}]}], ",", " ", 
      RowBox[{"k", " ", "=", " ", 
       RowBox[{"Log", "[", 
        RowBox[{"2", ",", "n"}], "]"}]}], ",", " ", "c"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"c", " ", "=", " ", 
      RowBox[{"Map", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"FromDigits", "[", 
          RowBox[{
           RowBox[{"Reverse", "[", 
            RowBox[{"PadLeft", "[", 
             RowBox[{
              RowBox[{"IntegerDigits", "[", 
               RowBox[{"#", ",", "2"}], "]"}], ",", " ", "k", ",", " ", "0"}],
              "]"}], "]"}], ",", " ", "2"}], "]"}], "&"}], ",", " ", 
        RowBox[{"Range", "[", 
         RowBox[{"0", ",", 
          RowBox[{"n", "-", "1"}]}], "]"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"SparseArray", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"r", ",", " ", "c"}], "}"}], "+", "1"}], ")"}], "]"}], 
        "\[Rule]", "1"}], ",", 
       RowBox[{"{", 
        RowBox[{"n", ",", "n"}], "}"}]}], "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7304874],

Cell[TextData[{
 "A compact notation for the identity matrix of size ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->907388934],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["i", "n_"], ":=", " ", 
   RowBox[{"IdentityMatrix", "[", "n", "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->35636705],

Cell["A compact notation for the direct matrix product:", "ExampleText",
 CellID->277203298],

Cell[BoxData[
 RowBox[{
  RowBox[{"a_", "\[CircleTimes]", "b_"}], " ", ":=", " ", 
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{"a", ",", "b"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->835322463],

Cell["\<\
Form the discrete Fourier transform matrix for length 16 from the Cooley\
\[Dash]Tukey factorization:\
\>", "ExampleText",
 CellID->37806682],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["f", "16"], "=", " ", 
   RowBox[{
    SubscriptBox["b", "16"], ".", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["i", "2"], "\[CircleTimes]", " ", 
      SubscriptBox["b", "8"]}], ")"}], ".", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["i", "4"], "\[CircleTimes]", " ", 
      SubscriptBox["b", "4"]}], ")"}], ".", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["i", "8"], "\[CircleTimes]", " ", 
      SubscriptBox["b", "2"]}], ")"}], ".", 
    SubscriptBox["p", "16"]}]}], ";"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->205882479],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 " is a random vector of length 16:"
}], "ExampleText",
 CellID->775384139],

Cell[BoxData[
 RowBox[{
  RowBox[{"r", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", "16"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->308123232],

Cell[TextData[{
 "The discrete Fourier transform of ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->405748513],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   SubscriptBox["f", "16"], ".", "r"}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->67563169],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"8.481751176687622`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.3945407599638251`"}], "-", 
    RowBox[{"0.08669998823240149`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.23915482913447167`", "\[InvisibleSpace]", "-", 
    RowBox[{"1.3468975030946435`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.044492198924579174`"}], "+", 
    RowBox[{"1.2238072373797608`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.24099708652701657`"}], "-", 
    RowBox[{"0.9896196484152786`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.4257682226871377`"}], "+", 
    RowBox[{"0.3146719441620598`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.9793769544071956`"}], "+", 
    RowBox[{"1.0400949755148932`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1.5569431264390428`"}], "+", 
    RowBox[{"0.1573576704433523`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1.496477119636687`"}], "+", 
    RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1.5569431264390428`"}], "-", 
    RowBox[{"0.15735767044335203`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.9793769544071955`"}], "-", 
    RowBox[{"1.0400949755148932`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.4257682226871377`"}], "-", 
    RowBox[{"0.31467194416205957`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.24099708652701657`"}], "+", 
    RowBox[{"0.9896196484152786`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.044492198924579285`"}], "-", 
    RowBox[{"1.2238072373797608`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.23915482913447172`", "\[InvisibleSpace]", "+", 
    RowBox[{"1.3468975030946435`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.39454075996382554`"}], "+", 
    RowBox[{"0.08669998823240144`", " ", "\[ImaginaryI]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{533, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->116023790]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Fourier",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Fourier"]], "InlineFormula"],
 " is fast because it effectively composes the factorization for a particular \
vector:"
}], "ExampleText",
 CellID->835354000],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Fourier", "[", 
  RowBox[{"r", ",", " ", 
   RowBox[{"FourierParameters", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->350723662],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"8.481751176687622`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.3945407599638252`"}], "-", 
    RowBox[{"0.08669998823240155`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.23915482913447159`", "\[InvisibleSpace]", "-", 
    RowBox[{"1.3468975030946435`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.044492198924579285`"}], "+", 
    RowBox[{"1.223807237379761`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.24099708652701657`"}], "-", 
    RowBox[{"0.9896196484152788`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.4257682226871377`"}], "+", 
    RowBox[{"0.3146719441620596`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.9793769544071955`"}], "+", 
    RowBox[{"1.0400949755148932`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1.5569431264390428`"}], "+", 
    RowBox[{"0.15735767044335236`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1.4964771196366868`"}], "+", 
    RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1.5569431264390428`"}], "-", 
    RowBox[{"0.15735767044335236`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.9793769544071955`"}], "-", 
    RowBox[{"1.0400949755148932`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.4257682226871377`"}], "-", 
    RowBox[{"0.3146719441620596`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.24099708652701657`"}], "+", 
    RowBox[{"0.9896196484152788`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.044492198924579285`"}], "-", 
    RowBox[{"1.223807237379761`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.23915482913447159`", "\[InvisibleSpace]", "+", 
    RowBox[{"1.3468975030946435`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.3945407599638252`"}], "+", 
    RowBox[{"0.08669998823240155`", " ", "\[ImaginaryI]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{533, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->313825799]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "For vectors, ",
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 " is a special case of ",
 Cell[BoxData[
  ButtonBox["Outer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Outer"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->18064987],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}]}], ";", " ", 
  RowBox[{"u", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"4", ",", "5"}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->440737606],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerProduct", "[", 
  RowBox[{"v", ",", "u"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->413642198],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"4", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"8", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"12", ",", "15"}], "}"}]}], "}"}]], "Output",
 ImageSize->{179, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->152428958]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Outer", "[", 
  RowBox[{"Times", ",", " ", "v", ",", " ", "u"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->819519502],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"4", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"8", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"12", ",", "15"}], "}"}]}], "}"}]], "Output",
 ImageSize->{179, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->223574920]
}, Open  ]],

Cell[TextData[{
 "If the vectors are formed into column and row matrices, ",
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 " is equivalent to a matrix product:"
}], "ExampleText",
 CellID->419642896],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Map", "[", 
   RowBox[{"List", ",", " ", "v"}], "]"}], ".", 
  RowBox[{"{", "u", "}"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->207596298],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"4", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"8", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"12", ",", "15"}], "}"}]}], "}"}]], "Output",
 ImageSize->{179, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->186370737]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->80337461],

Cell[TextData[{
 "For matrices, ",
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 " is a flattening of a special case of ",
 Cell[BoxData[
  ButtonBox["Outer",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Outer"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->661709],

Cell[BoxData[{
 RowBox[{
  RowBox[{"x", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"y", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"d", ",", "e", ",", "f"}], "}"}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->734390424],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"KroneckerProduct", "[", 
   RowBox[{"x", ",", "y"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->163842038],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"a", "b", "c", 
      RowBox[{"2", " ", "a"}], 
      RowBox[{"2", " ", "b"}], 
      RowBox[{"2", " ", "c"}]},
     {"d", "e", "f", 
      RowBox[{"2", " ", "d"}], 
      RowBox[{"2", " ", "e"}], 
      RowBox[{"2", " ", "f"}]},
     {
      RowBox[{"3", " ", "a"}], 
      RowBox[{"3", " ", "b"}], 
      RowBox[{"3", " ", "c"}], 
      RowBox[{"4", " ", "a"}], 
      RowBox[{"4", " ", "b"}], 
      RowBox[{"4", " ", "c"}]},
     {
      RowBox[{"3", " ", "d"}], 
      RowBox[{"3", " ", "e"}], 
      RowBox[{"3", " ", "f"}], 
      RowBox[{"4", " ", "d"}], 
      RowBox[{"4", " ", "e"}], 
      RowBox[{"4", " ", "f"}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{166, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->315324598]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"ArrayFlatten", "[", 
   RowBox[{"Outer", "[", 
    RowBox[{"Times", ",", " ", "x", ",", " ", "y"}], "]"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->233429184],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"a", "b", "c", 
      RowBox[{"2", " ", "a"}], 
      RowBox[{"2", " ", "b"}], 
      RowBox[{"2", " ", "c"}]},
     {"d", "e", "f", 
      RowBox[{"2", " ", "d"}], 
      RowBox[{"2", " ", "e"}], 
      RowBox[{"2", " ", "f"}]},
     {
      RowBox[{"3", " ", "a"}], 
      RowBox[{"3", " ", "b"}], 
      RowBox[{"3", " ", "c"}], 
      RowBox[{"4", " ", "a"}], 
      RowBox[{"4", " ", "b"}], 
      RowBox[{"4", " ", "c"}]},
     {
      RowBox[{"3", " ", "d"}], 
      RowBox[{"3", " ", "e"}], 
      RowBox[{"3", " ", "f"}], 
      RowBox[{"4", " ", "d"}], 
      RowBox[{"4", " ", "e"}], 
      RowBox[{"4", " ", "f"}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{166, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->9912404]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->57712655],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      StyleBox["a", "TI"], "\[CircleTimes]", 
      StyleBox["b", "TI"]}], ")"}], ".", 
    RowBox[{"(", 
     RowBox[{
      StyleBox["c", "TI"], "\[CircleTimes]", 
      StyleBox["d", "TI"]}], ")"}]}], "=", 
   RowBox[{
    RowBox[{"(", 
     StyleBox["ac", "TI"], ")"}], "\[CircleTimes]", 
    RowBox[{"(", 
     StyleBox["bd", "TI"], ")"}]}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->634132592],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c", ",", "d"}], "}"}], " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "3", ",", "3"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22927867],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"a", ",", "b"}], "]"}], ".", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"c", ",", "d"}], "]"}]}], " ", "-", " ", "\[IndentingNewLine]", 
   RowBox[{"KroneckerProduct", "[", 
    RowBox[{
     RowBox[{"a", ".", "c"}], ",", " ", 
     RowBox[{"b", ".", "d"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1807426],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW9FOwjAULRuEqUHmG/GJb/Aj9MmPWIgJPBgN8v+6lgzNpdh2vbfr9JAA
u7ez95zbs830hOfmsH15bQ67TbN+2jfv293mY/34tm9T5USpYqWUelip9viz
PTLv7nXfvr/DY3SjP8vTWeqKxEvmuA48f0ri60C8ofVi8cfi5e5faH9L/XGG
WsdFG9ae33eWmFbReVcdOo+rXt86Lvy28d+6WFi7qNHocEby84HyFdM8lH0V
yZsLl3Q/uHkPtX6x621Xu8nOfj5/2mPzt/oKmmc5WonMTHunz6bPgZhuyaAe
qpfS3cpPH5zaOr8uae9K61yLpNnb6BnoPWkRwTIejRR3TpZp1ydmLamOJ+Y8
v+fL2HRMudL/z0NY5qtjTpb56tjvfgwdy6OBjqFjHu7Q8VAYoWPoOCXLseuY
7uTrrA6Xnt/1hTytRvc+aZ1L87jqhdbxxe/iY98B7/pv80x4IjgvvPjhvPxH
56WrM7V2D45LWlxwXGTycFzCRuG45NSt/PTBra3jsxtOizwaOC2ca2muLuyA
JECDHRDOtTR3Wug2ARroFrqFbmVYQrc59AW6DWU5Nt3CGennjJjr/dRgOSfE
5cjAGYmb7284I6UHKzglvk6Jq5twTtLignMik7c7Jy71w0npRuGk5NSt/PQh
46QUllE4K7Jo4KxwruXRWen3nBmbjrGDEsoyXx3bd1Cg41CW0DF0PFSfoONQ
ltBxjn2CjkNZjk3HPr9s0SGcGkmnRk2+AKdIlGc=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{536, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->413817931]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Outer",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Outer"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Cross",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Cross"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Dot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Dot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Vectors and Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/VectorsAndMatrices"]], "Tutorials",
 CellID->26612],

Cell[TextData[ButtonBox["Basic Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/BasicMatrixOperations"]], "Tutorials",
 CellID->5521]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->230211943],

Cell[TextData[ButtonBox["Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatrixOperations"]], "MoreAbout",
 CellID->506989409],

Cell[TextData[ButtonBox["Operations on Vectors",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/OperationsOnVectors"]], "MoreAbout",
 CellID->69504760],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->122667096]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"KroneckerProduct - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 52, 6.8574424}", "context" -> "System`", 
    "keywords" -> {
     "Kronecker product", "matrix direct product", "outer product", 
      "tensor product"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "KroneckerProduct[m_1, m_2, ...] constructs the Kronecker product of the \
arrays m_i.", "synonyms" -> {"Kronecker product"}, "title" -> 
    "KroneckerProduct", "type" -> "Symbol", "uri" -> 
    "ref/KroneckerProduct"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7120, 217, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->348985882]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 58897, 2032}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2400, 57, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3023, 84, 1581, 37, 70, "ObjectNameGrid"],
Cell[4607, 123, 845, 27, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5489, 155, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5967, 172, 252, 7, 70, "Notes",
 CellID->5543913],
Cell[6222, 181, 240, 8, 70, "Notes",
 CellID->1067943069],
Cell[6465, 191, 618, 21, 70, "Notes",
 CellID->20799]
}, Closed]],
Cell[CellGroupData[{
Cell[7120, 217, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->348985882],
Cell[CellGroupData[{
Cell[7505, 231, 148, 5, 70, "ExampleSection",
 CellID->232321335],
Cell[7656, 238, 72, 1, 70, "ExampleText",
 CellID->247564214],
Cell[CellGroupData[{
Cell[7753, 243, 253, 8, 28, "Input",
 CellID->571197403],
Cell[8009, 253, 624, 22, 36, "Output",
 CellID->59959681]
}, Open  ]],
Cell[8648, 278, 125, 3, 70, "ExampleDelimiter",
 CellID->173717615],
Cell[8776, 283, 65, 1, 70, "ExampleText",
 CellID->471934619],
Cell[CellGroupData[{
Cell[8866, 288, 494, 18, 28, "Input",
 CellID->593051083],
Cell[9363, 308, 1200, 24, 92, "Output",
 Evaluatable->False,
 CellID->362214281]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10612, 338, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[10836, 347, 215, 8, 70, "ExampleText",
 CellID->238706176],
Cell[11054, 357, 516, 19, 70, "Input",
 CellID->330562928],
Cell[11573, 378, 96, 1, 70, "ExampleText",
 CellID->24116838],
Cell[CellGroupData[{
Cell[11694, 383, 172, 5, 70, "Input",
 CellID->12531995],
Cell[11869, 390, 886, 28, 92, "Output",
 CellID->25815915]
}, Open  ]],
Cell[12770, 421, 67, 1, 70, "ExampleText",
 CellID->1051321697],
Cell[CellGroupData[{
Cell[12862, 426, 233, 7, 70, "Input",
 CellID->266741744],
Cell[13098, 435, 920, 28, 92, "Output",
 CellID->552440390]
}, Open  ]],
Cell[14033, 466, 77, 1, 70, "ExampleText",
 CellID->134884688],
Cell[CellGroupData[{
Cell[14135, 471, 287, 9, 70, "Input",
 CellID->244180237],
Cell[14425, 482, 1040, 28, 92, "Output",
 CellID->393795905]
}, Open  ]],
Cell[15480, 513, 125, 3, 70, "ExampleDelimiter",
 CellID->248818974],
Cell[15608, 518, 203, 8, 70, "ExampleText",
 CellID->310187793],
Cell[15814, 528, 790, 26, 70, "Input",
 CellID->21156748],
Cell[16607, 556, 128, 2, 70, "ExampleText",
 CellID->775152956],
Cell[CellGroupData[{
Cell[16760, 562, 135, 4, 70, "Input",
 CellID->253117605],
Cell[16898, 568, 482, 17, 36, "Output",
 CellID->35939343]
}, Open  ]],
Cell[CellGroupData[{
Cell[17417, 590, 106, 3, 70, "Input",
 CellID->328089509],
Cell[17526, 595, 1579, 30, 70, "Output",
 Evaluatable->False,
 CellID->284324807]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19154, 631, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[19384, 640, 515, 18, 70, "ExampleText",
 CellID->154989736],
Cell[19902, 660, 414, 14, 70, "Input",
 CellID->92801958],
Cell[CellGroupData[{
Cell[20341, 678, 629, 20, 70, "Input",
 CellID->3349867],
Cell[20973, 700, 422, 15, 36, "Output",
 CellID->354121891]
}, Open  ]],
Cell[CellGroupData[{
Cell[21432, 720, 293, 10, 70, "Input",
 CellID->289590767],
Cell[21728, 732, 300, 11, 36, "Output",
 CellID->2511561]
}, Open  ]],
Cell[22043, 746, 125, 3, 70, "ExampleDelimiter",
 CellID->237818671],
Cell[22171, 751, 207, 8, 70, "ExampleText",
 CellID->32209709],
Cell[22381, 761, 516, 19, 70, "Input",
 CellID->184146917],
Cell[22900, 782, 85, 1, 70, "ExampleText",
 CellID->241337795],
Cell[CellGroupData[{
Cell[23010, 787, 201, 6, 70, "Input",
 CellID->89819377],
Cell[23214, 795, 158, 5, 36, "Output",
 CellID->147588273]
}, Open  ]],
Cell[23387, 803, 260, 9, 70, "ExampleText",
 CellID->599476156],
Cell[CellGroupData[{
Cell[23672, 816, 210, 6, 70, "Input",
 CellID->300624452],
Cell[23885, 824, 684, 24, 51, "Output",
 CellID->460284671]
}, Open  ]],
Cell[CellGroupData[{
Cell[24606, 853, 104, 3, 70, "Input",
 CellID->679572052],
Cell[24713, 858, 683, 24, 51, "Output",
 CellID->64131251]
}, Open  ]],
Cell[25411, 885, 125, 3, 70, "ExampleDelimiter",
 CellID->451110104],
Cell[25539, 890, 197, 6, 70, "ExampleText",
 CellID->561987471],
Cell[CellGroupData[{
Cell[25761, 900, 693, 23, 70, "Input",
 CellID->45633183],
Cell[26457, 925, 481, 17, 36, "Output",
 CellID->481672895]
}, Open  ]],
Cell[CellGroupData[{
Cell[26975, 947, 197, 7, 70, "Input",
 CellID->13084088],
Cell[27175, 956, 705, 26, 36, "Output",
 CellID->140687098]
}, Open  ]],
Cell[27895, 985, 80, 1, 70, "ExampleText",
 CellID->68705780],
Cell[CellGroupData[{
Cell[28000, 990, 342, 11, 70, "Input",
 CellID->19348395],
Cell[28345, 1003, 479, 17, 36, "Output",
 CellID->271907544]
}, Open  ]],
Cell[28839, 1023, 74, 1, 70, "ExampleText",
 CellID->40948944],
Cell[CellGroupData[{
Cell[28938, 1028, 297, 10, 70, "Input",
 CellID->17929735],
Cell[29238, 1040, 1261, 25, 70, "Output",
 Evaluatable->False,
 CellID->440057243]
}, Open  ]],
Cell[30514, 1068, 102, 2, 70, "ExampleText",
 CellID->105732674],
Cell[CellGroupData[{
Cell[30641, 1074, 171, 5, 70, "Input",
 CellID->32411629],
Cell[30815, 1081, 483, 17, 36, "Output",
 CellID->201984605]
}, Open  ]],
Cell[CellGroupData[{
Cell[31335, 1103, 403, 14, 70, "Input",
 CellID->75204836],
Cell[31741, 1119, 1037, 21, 70, "Output",
 Evaluatable->False,
 CellID->211593949]
}, Open  ]],
Cell[32793, 1143, 83, 1, 70, "ExampleText",
 CellID->73727274],
Cell[CellGroupData[{
Cell[32901, 1148, 284, 8, 70, "Input",
 CellID->338802460],
Cell[33188, 1158, 483, 17, 36, "Output",
 CellID->718456294]
}, Open  ]],
Cell[CellGroupData[{
Cell[33708, 1180, 402, 14, 70, "Input",
 CellID->285829],
Cell[34113, 1196, 1293, 25, 70, "Output",
 Evaluatable->False,
 CellID->386526411]
}, Open  ]],
Cell[35421, 1224, 125, 3, 70, "ExampleDelimiter",
 CellID->200860902],
Cell[35549, 1229, 212, 8, 70, "ExampleText",
 CellID->344424701],
Cell[35764, 1239, 1090, 33, 70, "Input",
 CellID->182487478],
Cell[36857, 1274, 299, 11, 70, "ExampleText",
 CellID->314804321],
Cell[37159, 1287, 1500, 45, 70, "Input",
 CellID->7304874],
Cell[38662, 1334, 172, 6, 70, "ExampleText",
 CellID->907388934],
Cell[38837, 1342, 181, 6, 70, "Input",
 CellID->35636705],
Cell[39021, 1350, 92, 1, 70, "ExampleText",
 CellID->277203298],
Cell[39116, 1353, 208, 6, 70, "Input",
 CellID->835322463],
Cell[39327, 1361, 151, 4, 70, "ExampleText",
 CellID->37806682],
Cell[39481, 1367, 606, 20, 70, "Input",
 CellID->205882479],
Cell[40090, 1389, 148, 5, 70, "ExampleText",
 CellID->775384139],
Cell[40241, 1396, 189, 6, 70, "Input",
 CellID->308123232],
Cell[40433, 1404, 155, 6, 70, "ExampleText",
 CellID->405748513],
Cell[CellGroupData[{
Cell[40613, 1414, 143, 5, 70, "Input",
 CellID->67563169],
Cell[40759, 1421, 2201, 53, 105, "Output",
 CellID->116023790]
}, Open  ]],
Cell[42975, 1477, 260, 8, 70, "ExampleText",
 CellID->835354000],
Cell[CellGroupData[{
Cell[43260, 1489, 228, 7, 70, "Input",
 CellID->350723662],
Cell[43491, 1498, 2199, 53, 105, "Output",
 CellID->313825799]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[45739, 1557, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[45980, 1566, 351, 13, 70, "ExampleText",
 CellID->18064987],
Cell[46334, 1581, 276, 9, 70, "Input",
 CellID->440737606],
Cell[CellGroupData[{
Cell[46635, 1594, 135, 4, 70, "Input",
 CellID->413642198],
Cell[46773, 1600, 364, 13, 36, "Output",
 CellID->152428958]
}, Open  ]],
Cell[CellGroupData[{
Cell[47174, 1618, 148, 4, 70, "Input",
 CellID->819519502],
Cell[47325, 1624, 364, 13, 36, "Output",
 CellID->223574920]
}, Open  ]],
Cell[47704, 1640, 289, 8, 70, "ExampleText",
 CellID->419642896],
Cell[CellGroupData[{
Cell[48018, 1652, 177, 6, 70, "Input",
 CellID->207596298],
Cell[48198, 1660, 364, 13, 36, "Output",
 CellID->186370737]
}, Open  ]],
Cell[48577, 1676, 124, 3, 70, "ExampleDelimiter",
 CellID->80337461],
Cell[48704, 1681, 366, 13, 70, "ExampleText",
 CellID->661709],
Cell[49073, 1696, 513, 18, 70, "Input",
 CellID->734390424],
Cell[CellGroupData[{
Cell[49611, 1718, 173, 5, 70, "Input",
 CellID->163842038],
Cell[49787, 1725, 1326, 42, 92, "Output",
 CellID->315324598]
}, Open  ]],
Cell[CellGroupData[{
Cell[51150, 1772, 230, 7, 70, "Input",
 CellID->233429184],
Cell[51383, 1781, 1324, 42, 92, "Output",
 CellID->9912404]
}, Open  ]],
Cell[52722, 1826, 124, 3, 70, "ExampleDelimiter",
 CellID->57712655],
Cell[52849, 1831, 497, 19, 70, "ExampleText",
 CellID->634132592],
Cell[53349, 1852, 314, 10, 70, "Input",
 CellID->22927867],
Cell[CellGroupData[{
Cell[53688, 1866, 469, 13, 70, "Input",
 CellID->1807426],
Cell[54160, 1881, 984, 20, 70, "Output",
 Evaluatable->False,
 CellID->413817931]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[55205, 1908, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[55524, 1919, 678, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[56239, 1950, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[56560, 1961, 149, 3, 70, "Tutorials",
 CellID->26612],
Cell[56712, 1966, 154, 3, 70, "Tutorials",
 CellID->5521]
}, Open  ]],
Cell[CellGroupData[{
Cell[56903, 1974, 319, 9, 70, "MoreAboutSection",
 CellID->230211943],
Cell[57225, 1985, 145, 3, 70, "MoreAbout",
 CellID->506989409],
Cell[57373, 1990, 151, 3, 70, "MoreAbout",
 CellID->69504760],
Cell[57527, 1995, 179, 3, 70, "MoreAbout",
 CellID->122667096]
}, Open  ]],
Cell[57721, 2001, 27, 0, 70, "History"],
Cell[57751, 2003, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

