(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     26976,        880]
NotebookOptionsPosition[     21976,        707]
NotebookOutlinePosition[     23251,        743]
CellTagsIndexPosition[     23166,        738]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"JacobiSymbol" :> 
          Documentation`HelpLookup["paclet:ref/JacobiSymbol"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"JacobiSymbol\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Number Theoretic Functions\"\>", 
       2->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["KroneckerSymbol", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["KroneckerSymbol",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/KroneckerSymbol"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Kronecker symbol ",
     Cell[BoxData[
      FormBox[
       RowBox[{"(", 
        FractionBox["n", "m"], ")"}], TraditionalForm]], "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Integer mathematical function, suitable for both symbolic and numerical \
manipulation.\
\>", "Notes",
 CellID->126685801],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KroneckerSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerSymbol"]], "InlineFormula"],
 " in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " is directly equivalent to ",
 Cell[BoxData[
  ButtonBox["JacobiSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/JacobiSymbol"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->209168635],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KroneckerSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerSymbol"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->555347927]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->31891578],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->79646112],

Cell["Compute Kronecker symbols:", "ExampleText",
 CellID->1641776611],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{"11", ",", "3"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1035753004],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1749087422]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"KroneckerSymbol", "[", 
    RowBox[{"n", ",", "m"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"m", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->322120774],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", 
     RowBox[{"-", "1"}], ",", "0", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}], ",", "0", ",", "1", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "1"}], ",", "1", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{578, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->8657771]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Evaluate for large arguments:", "ExampleText",
 CellID->77574531],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"10", "^", "11"}], "+", "1"}], ",", 
   RowBox[{"Prime", "[", "2000", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->272568788],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1629752848]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->70276092],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KroneckerSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerSymbol"]], "InlineFormula"],
 " works for negative arguments:"
}], "ExampleText",
 CellID->1715413169],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{
   RowBox[{"-", "3"}], ",", "7"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->981170408],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->102198851]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{"3", ",", 
   RowBox[{"-", "7"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->882386017],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1816654804]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2137823185],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KroneckerSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerSymbol"]], "InlineFormula"],
 " is defined for even and odd integers:"
}], "ExampleText",
 CellID->845372338],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{"5", ",", "4"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->421076214],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1742281429]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{"20", ",", "8"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1780819135],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1584450358]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->83566656],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["KroneckerSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerSymbol"]], "InlineFormula"],
 " threads over lists:"
}], "ExampleText",
 CellID->50482254],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", "3", ",", "5", ",", "7", ",", "11"}], "}"}], ",", " ", 
   "6"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->199202413],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1"}], "}"}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1059944106]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"KroneckerSymbol", "[", 
  RowBox[{"5", ",", " ", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", "7", ",", "8",
      ",", "9", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1711745644],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"-", "1"}], ",", 
   RowBox[{"-", "1"}], ",", "1", ",", "0", ",", "1", ",", 
   RowBox[{"-", "1"}], ",", 
   RowBox[{"-", "1"}], ",", "1", ",", "0"}], "}"}]], "Output",
 ImageSize->{220, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1103737677]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "Reduce equations containing ",
 Cell[BoxData[
  ButtonBox["KroneckerSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerSymbol"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1041026490],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"KroneckerSymbol", "[", 
      RowBox[{"p", ",", "3"}], "]"}], " ", "==", " ", 
     RowBox[{
      RowBox[{"2", 
       RowBox[{"KroneckerSymbol", "[", 
        RowBox[{
         RowBox[{"p", "+", "1"}], ",", "3"}], "]"}]}], "-", "1"}]}], "&&", 
    " ", 
    RowBox[{
     RowBox[{"-", "5"}], " ", "<", "p", " ", "<", " ", "5"}]}], " ", ",", 
   RowBox[{"{", "p", "}"}], ",", " ", "Integers"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->982945321],

Cell[BoxData[
 RowBox[{
  RowBox[{"p", "\[Equal]", 
   RowBox[{"-", "4"}]}], "||", 
  RowBox[{"p", "\[Equal]", 
   RowBox[{"-", "1"}]}], "||", 
  RowBox[{"p", "\[Equal]", "2"}]}]], "Output",
 ImageSize->{152, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->555394553]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->589267740],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"KroneckerSymbol", "[", 
      RowBox[{"n", ",", "m"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", 
       RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"m", ",", 
       RowBox[{"-", "50"}], ",", "50"}], "}"}]}], "]"}], ",", 
   RowBox[{"ColorRules", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "\[Rule]", "Red"}], ",", 
      RowBox[{"0", "\[Rule]", "Green"}], ",", 
      RowBox[{"1", "\[Rule]", "Blue"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->428546349],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{156, 156},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->24146277]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[Cell[BoxData[
 StyleBox[
  ButtonBox["JacobiSymbol",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/JacobiSymbol"],
  FontFamily->"Verdana"]], "InlineFormula"]], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->324055222],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->629026935],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->797999298]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"KroneckerSymbol - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 52, 7.4355822}", "context" -> "System`", 
    "keywords" -> {"Jacobi symbol", "Kronecker symbol", "Legendre symbol"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "KroneckerSymbol[n, m] gives the Kronecker symbol (n/m). ", 
    "synonyms" -> {"kronecker symbol"}, "title" -> "KroneckerSymbol", "type" -> 
    "Symbol", "uri" -> "ref/KroneckerSymbol"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[5614, 180, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->31891578]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 23024, 731}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1416, 35, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2039, 62, 1580, 37, 70, "ObjectNameGrid"],
Cell[3622, 101, 680, 22, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[4339, 128, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[4817, 145, 132, 4, 70, "Notes",
 CellID->126685801],
Cell[4952, 151, 403, 15, 70, "Notes",
 CellID->209168635],
Cell[5358, 168, 219, 7, 70, "Notes",
 CellID->555347927]
}, Closed]],
Cell[CellGroupData[{
Cell[5614, 180, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->31891578],
Cell[CellGroupData[{
Cell[5998, 194, 147, 5, 70, "ExampleSection",
 CellID->79646112],
Cell[6148, 201, 70, 1, 70, "ExampleText",
 CellID->1641776611],
Cell[CellGroupData[{
Cell[6243, 206, 136, 4, 28, "Input",
 CellID->1035753004],
Cell[6382, 212, 176, 6, 36, "Output",
 CellID->1749087422]
}, Open  ]],
Cell[CellGroupData[{
Cell[6595, 223, 300, 10, 28, "Input",
 CellID->322120774],
Cell[6898, 235, 753, 23, 36, "Output",
 CellID->8657771]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[7700, 264, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[7924, 273, 71, 1, 70, "ExampleText",
 CellID->77574531],
Cell[CellGroupData[{
Cell[8020, 278, 221, 7, 70, "Input",
 CellID->272568788],
Cell[8244, 287, 176, 6, 36, "Output",
 CellID->1629752848]
}, Open  ]],
Cell[8435, 296, 124, 3, 70, "ExampleDelimiter",
 CellID->70276092],
Cell[8562, 301, 222, 7, 70, "ExampleText",
 CellID->1715413169],
Cell[CellGroupData[{
Cell[8809, 312, 153, 5, 70, "Input",
 CellID->981170408],
Cell[8965, 319, 158, 5, 36, "Output",
 CellID->102198851]
}, Open  ]],
Cell[CellGroupData[{
Cell[9160, 329, 153, 5, 70, "Input",
 CellID->882386017],
Cell[9316, 336, 176, 6, 36, "Output",
 CellID->1816654804]
}, Open  ]],
Cell[9507, 345, 126, 3, 70, "ExampleDelimiter",
 CellID->2137823185],
Cell[9636, 350, 229, 7, 70, "ExampleText",
 CellID->845372338],
Cell[CellGroupData[{
Cell[9890, 361, 134, 4, 70, "Input",
 CellID->421076214],
Cell[10027, 367, 159, 5, 36, "Output",
 CellID->1742281429]
}, Open  ]],
Cell[CellGroupData[{
Cell[10223, 377, 136, 4, 70, "Input",
 CellID->1780819135],
Cell[10362, 383, 159, 5, 36, "Output",
 CellID->1584450358]
}, Open  ]],
Cell[10536, 391, 124, 3, 70, "ExampleDelimiter",
 CellID->83566656],
Cell[10663, 396, 210, 7, 70, "ExampleText",
 CellID->50482254],
Cell[CellGroupData[{
Cell[10898, 407, 223, 7, 70, "Input",
 CellID->199202413],
Cell[11124, 416, 234, 7, 36, "Output",
 CellID->1059944106]
}, Open  ]],
Cell[CellGroupData[{
Cell[11395, 428, 281, 8, 70, "Input",
 CellID->1711745644],
Cell[11679, 438, 361, 11, 36, "Output",
 CellID->1103737677]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12089, 455, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[12330, 464, 226, 8, 70, "ExampleText",
 CellID->1041026490],
Cell[CellGroupData[{
Cell[12581, 476, 544, 17, 70, "Input",
 CellID->982945321],
Cell[13128, 495, 321, 11, 36, "Output",
 CellID->555394553]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13498, 512, 228, 7, 70, "ExampleSection",
 CellID->589267740],
Cell[CellGroupData[{
Cell[13751, 523, 675, 21, 70, "Input",
 CellID->428546349],
Cell[14429, 546, 6165, 105, 70, "Output",
 Evaluatable->False,
 CellID->24146277]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[20655, 658, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[20974, 669, 208, 6, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[21219, 680, 319, 9, 70, "MoreAboutSection",
 CellID->324055222],
Cell[21541, 691, 162, 3, 70, "MoreAbout",
 CellID->629026935],
Cell[21706, 696, 198, 4, 70, "MoreAbout",
 CellID->797999298]
}, Open  ]],
Cell[21919, 703, 27, 0, 70, "History"],
Cell[21949, 705, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

