(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     48765,       1640]
NotebookOptionsPosition[     40116,       1337]
NotebookOutlinePosition[     41514,       1376]
CellTagsIndexPosition[     41428,       1371]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:tutorial/DescriptiveStatistics"], 
          "Discrete Distributions" :> 
          Documentation`HelpLookup["paclet:tutorial/DiscreteDistributions"], 
          "Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"Discrete Distributions\"\>", 
       3->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Skewness" :> Documentation`HelpLookup["paclet:ref/Skewness"], 
          "CentralMoment" :> 
          Documentation`HelpLookup["paclet:ref/CentralMoment"], "Variance" :> 
          Documentation`HelpLookup["paclet:ref/Variance"], "ExpectedValue" :> 
          Documentation`HelpLookup["paclet:ref/ExpectedValue"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Skewness\"\>", 
       2->"\<\"CentralMoment\"\>", 3->"\<\"Variance\"\>", 
       4->"\<\"ExpectedValue\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "Statistical Distributions" :> 
          Documentation`HelpLookup["paclet:guide/StatisticalDistributions"], 
          "New in 6.0: Data Handling & Data Sources" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60DataHandlingAndDataSources"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Numerical Data Handling" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumericalDataHandling"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"Statistical Distributions\"\>", 
       3->"\<\"New in 6.0: Data Handling & Data Sources\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       5->"\<\"New in 6.0: Numerical Data Handling\"\>", 
       6->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Kurtosis", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Kurtosis",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Kurtosis"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the coefficient of kurtosis for the elements in ",
     
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Kurtosis",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Kurtosis"], "[", 
       StyleBox["dist", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the coefficient of kurtosis for the symbolic \
distribution ",
     Cell[BoxData[
      StyleBox["dist", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->6122]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " measures the concentration of data around the peak and in the tails versus \
the concentration in the flanks."
}], "Notes",
 CellID->17901],

Cell["\<\
A kurtosis larger than 3 indicates a distribution that is more peaked and has \
heavier tails than a normal distribution with the same variance. A kurtosis \
smaller than 3 indicates a distribution that is flatter.\
\>", "Notes",
 CellID->22749],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " handles both numerical and symbolic data. "
}], "Notes",
 CellID->28487],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Kurtosis",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Kurtosis"], "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["1", "TR"]], ",", 
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["1", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["2", "TR"]], ",", 
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["2", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Kurtosis", "[", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["1", "TR"]], ",", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["2", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}], ",", 
    RowBox[{"Kurtosis", "[", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["1", "TR"]], ",", 
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["2", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->26555],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->22482],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Kurtosis",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Kurtosis"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["CentralMoment",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/CentralMoment"], "[", 
    RowBox[{
     StyleBox["list", "TI"], ",", "4"}], "]"}], "/", 
   SuperscriptBox[
    RowBox[{
     ButtonBox["CentralMoment",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/CentralMoment"], "[", 
     RowBox[{
      StyleBox["list", "TI"], ",", "2"}], "]"}], "2"]}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->15967]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->119478173],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->109256230],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " for a list of values:"
}], "ExampleText",
 CellID->196404163],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18121],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWstuEzEUnWQmJW3SNpRHy0vqBiTEiiIhlqxAPPoJQKMKqV1AUVs2SEg8
NvxBd/RTsuwn9FO6DLZnnKYn13Nsz6StRKRM5o59H+facz0+k/X+/tbHT/39
7c3+6svd/pet7c291Rc7u+pS2kiSxt8kSdZWEnU+VGfmqz+H6puf5mcP1bGp
Gn8XvzvF73fT+kgfU6s7TH6B/Bnkb0ZrFq5mIC+D/Abk26T/a9If/b0HeQ7k
eRN1m1i5RaJeIf0xauw/Exh11yvq8dypY7JOosD+b0n/DPpv1BI1eumA/BTk
FtFHGaMKtYftLD7mP8/KFZKVByA/I1HdBPl+Rf1rRJ+h7HihTNWhpUS8H64L
1nW/+eL3qvrVzYgC7XdBNvFkI7mwhlGi9w5473l6nwcZUS5FokQ7M45+cWPS
1IfWeI4SjxzNilqLAVcxDq+xpHH7zCzfCBei4g6dBfVnEj22vLTqnD3a4+l9
PhgMkvyTmoOSxyMDa1qv3KqM50xrSTYk3FJVkGPIKiJLKyBj44ztto7J/S7f
qPj0i193MmGMpeqQld99AfmfTgXrlMa4EBhjzJoVnyGxMtU+2nWvaE079zFv
U1nTJG9ox39Vk6x1o2Kvb12Lz+d5rmy6ZmhRP5VpF7jznSvae0Y5v57XU62R
eWQUn8GboiXtG/dUHafvXqRvW3+ro1x0XI8bBdxpo5W7pL+0ltlRncg9qw5j
8j2Q2S7S5ZdVPIYP/Vo/4i628t5cszH67kVWBr2lIGNt0DZwBiLyO4UvNqLM
F9vPuzBhLtCPq17MRGUauRjsj/MG9ZGxYvqYlQ8e6Mrk/wM1tht7lBPVM13P
MOTnkF9DDsbqlWVJ0ksLvQ2ih+gwOzm6H3D1J8jIG8+Cl3xm6KYD0b4+PXSe
5XX5dK+iTR6ob6PdbjeOj48bHjYeD2Ue/KtpfULwIA++a7TWArX2jBbOR3Z/
vAIZaxTWRtYfR/hdydgNR3d1j0QdyiuvBPZnPDRG3fKKmtUSxvmz/qzWxEWN
PG8ou408c6g+rq6oj1mSUeI+kaEMZbOrsumIEvUZSoPHHPVFtipaVhyjmiov
PcGKY5Qu791hDZx8yRrtixLtuFhxeWxkHi6OWZBZjqOjo+fmk6TmoOSiXeTt
L5ST9mNM6sElI5jOO4L0wkcCI/JjLOQ5K9oKnq1nGeyTk5OB+eQPWVp2WK/C
YPuyu74Mts9bh7qRZRx3ADLXW4f0Eo6KL8Z6Zq3l6i8XDx7HKdsYS2peQIwx
K2l8hti4+Y2+1tJ7LfYMVPfb3PPjvV2cPXuac6+O5/e+AWfMNPJ5kSsgcvjT
59EnOfwl4ntxQiP2/UFGPPmjdF2v81ma7aFCGXY71riDxll7g/gNfaPg8svq
I8OHfq0fkZMc7TUZF8n+x+XLuqM8kXV1iv/tRF++bxOYL/Y/KBcmEaM8jyuz
SfapBisuY5VceoxdsnriqlQLGivbGheKqkzfB924vshkU7bVxa/bZxbk1ZcD
+7t4eNsfxyZnW//AVWSWRb7dCId5Nhr/AAo08Zg=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{414, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->689051212]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->448710962],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " for a symbolic distribution:"
}], "ExampleText",
 CellID->25999718],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->127140819],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox["12", "\[Nu]"]}]], "Output",
 ImageSize->{42, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->524958168]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31371],

Cell["Compute results at machine precision:", "ExampleText",
 CellID->379986608],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", "20"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->216003675],

Cell[BoxData["1.9119601909387678`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->28299823]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158391064],

Cell["Obtain results at any precision: ", "ExampleText",
 CellID->160553898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "4", ",", "5", ",", "15", ",", "10", ",", "8"}], "}"}],
     ",", "30"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190699768],

Cell[BoxData["2.14692848818691788375257056275302898361`28.672374144810547"], \
"Output",
 ImageSize->{214, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->379527010]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->183534688],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " of a matrix will compute column-wise:"
}], "ExampleText",
 CellID->111413134],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", "4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "10"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"6", ",", "8"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->48341596],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["5477", "3481"], ",", 
   FractionBox["34", "25"]}], "}"}]], "Output",
 ImageSize->{80, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->279328463]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->650909219],

Cell["Compute results for a large vector or matrix:", "ExampleText",
 CellID->285474028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"10", "^", "7"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->315224713],

Cell[BoxData["1.7999430992883987`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->428024305]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"10", "^", "6"}], ",", "10"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->594479966],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.7981987714250456`", ",", "1.7983084222745067`", ",", 
   "1.8007498451995216`", ",", "1.7978701058232618`", ",", 
   "1.799891169650707`", ",", "1.7988473877173161`", ",", 
   "1.8006191450290896`", ",", "1.8012940165798867`", ",", 
   "1.7993903455761762`", ",", "1.801412499217092`"}], "}"}]], "Output",
 ImageSize->{368, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4831479]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->373178873],

Cell["Obtain results for continuous and discrete distributions:", \
"ExampleText",
 CellID->17058508],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"BetaDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->108321647],

Cell[BoxData[
 FractionBox[
  RowBox[{"3", " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"\[Alpha]", " ", "\[Beta]", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "6"}], "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"\[Alpha]", "+", "\[Beta]"}], ")"}], "2"]}]}], ")"}]}], 
  RowBox[{"\[Alpha]", " ", "\[Beta]", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"3", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}]]], "Output",
 ImageSize->{253, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->116078678]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->130372047],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox["1", "\[Mu]"]}]], "Output",
 ImageSize->{35, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->626115038]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32490],

Cell[TextData[{
 "Compute results for a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->625271644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sp", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], ":>", "i"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "j_"}], "}"}], "/;", 
        RowBox[{"j", "\[Equal]", 
         RowBox[{"i", "+", "1"}]}]}], "\[RuleDelayed]", 
       RowBox[{"i", "-", "1"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "5"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19878761],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "8", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 8, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "5"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->248710404]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", "sp", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->235263230],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["9703", "99"], ",", 
   FractionBox["9703", "99"], ",", 
   FractionBox["404246", "5043"], ",", 
   FractionBox["15946757", "241081"], ",", 
   FractionBox["3430486", "57963"]}], "}"}]], "Output",
 ImageSize->{297, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->577701629]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1281],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " computed from ",
 Cell[BoxData[
  ButtonBox["CentralMoment",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CentralMoment"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->209192808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->388198497],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.1538109734266375`", ",", "7.65024674041995`", ",", "2.196652518160167`", 
   ",", "1.7319153735056991`", ",", "1.6417664012782196`", ",", 
   "6.137817506088372`", ",", "9.86114919239061`", ",", "0.655690628558292`", 
   ",", "3.133716424871713`", ",", "0.7980430876076627`", ",", 
   "2.7418381244616463`", ",", "0.7467422100420906`", ",", 
   "5.1889074812987275`", ",", "5.43084515188252`", ",", 
   "3.4204218734087166`", ",", "6.665090447351927`", ",", 
   "1.4576962900454915`", ",", "4.767736225204315`", ",", 
   "1.325397804738433`", ",", "8.806330072666151`"}], "}"}]], "Output",
 ImageSize->{450, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->368476844]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", "data", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->934643318],

Cell[BoxData["2.3199157487225572`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->340669345]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CentralMoment", "[", 
   RowBox[{"data", ",", "4"}], "]"}], "/", 
  RowBox[{
   RowBox[{"CentralMoment", "[", 
    RowBox[{"data", ",", "2"}], "]"}], "^", "2"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->199964197],

Cell[BoxData["2.3199157487225572`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->121973022]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->203852233],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " for a distribution can be computed from ",
 Cell[BoxData[
  ButtonBox["ExpectedValue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ExpectedValue"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Variance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Variance"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->212776658],

Cell[BoxData[
 RowBox[{
  RowBox[{"dist", "=", 
   RowBox[{"GeometricDistribution", "[", "p", "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->107789233],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ev", "=", 
  RowBox[{
   RowBox[{"ExpectedValue", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"k", "-", 
        RowBox[{"Mean", "[", "dist", "]"}]}], ")"}], "^", "4"}], ",", "dist", 
     ",", "k"}], "]"}], "/", 
   RowBox[{
    RowBox[{"Variance", "[", "dist", "]"}], "^", "2"}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->273951061],

Cell[BoxData[
 FractionBox[
  RowBox[{"9", "-", 
   RowBox[{"18", " ", "p"}], "+", 
   RowBox[{"10", " ", 
    SuperscriptBox["p", "2"]}], "-", 
   SuperscriptBox["p", "3"]}], 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "2"]]], "Output",
 ImageSize->{119, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->191573611]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ks", "=", 
  RowBox[{"Kurtosis", "[", "dist", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->370222015],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox[
   RowBox[{"6", "-", 
    RowBox[{"6", " ", "p"}], "+", 
    SuperscriptBox["p", "2"]}], 
   RowBox[{"1", "-", "p"}]]}]], "Output",
 ImageSize->{90, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->271128780]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"ks", "==", "ev"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->3806373],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->23448660]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->246852425],

Cell[TextData[{
 "Normal distributions have ",
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " value 3:"
}], "ExampleText",
 CellID->421003384],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"NormalDistribution", "[", 
   RowBox[{"\[Mu]", ",", "\[Sigma]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->114256448],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->491306028]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->75092913],

Cell[TextData[{
 "Approximately normal distributions have ",
 Cell[BoxData[
  ButtonBox["Kurtosis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Kurtosis"]], "InlineFormula"],
 " values near 3:"
}], "ExampleText",
 CellID->659817507],

Cell[BoxData[
 RowBox[{
  RowBox[{"dist", "=", 
   RowBox[{"BinomialDistribution", "[", 
    RowBox[{
     RowBox[{"10", "^", "3"}], ",", ".5"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->524716893],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", "dist", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->929317708],

Cell[BoxData["2.998`"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->218502496]
}, Open  ]],

Cell[TextData[{
 "Plot the ",
 Cell[BoxData[
  ButtonBox["PDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PDF"]], "InlineFormula"],
 " for the distribution:"
}], "ExampleText",
 CellID->222429037],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"k", ",", 
       RowBox[{"PDF", "[", 
        RowBox[{"dist", ",", "k"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "450", ",", "550"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->207864196],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXWtsFUUUXrq9pS2PQltayrMComBFsYigYrE8RdDwjhGjFRD4UTBATCQI
Bl9g9Jci8hAFSw0S+oO3RMFqVHySGBPgR/2BP/gBJmJCMBJzPTNn9+zs7Ozc
vXe3tPdBwt2dvffunPN958z5dnbudlbD2uVLGxvWrljcUD1tdcPzy1csXlM9
ddVqOGR2MQzjCPxf39eA/Tjs8f9H4nHYel/vZa/57BP2vzxomNa2HLam0La3
fazjebDNt7bi5/OVZ+2l6tH6tmHiEUNqs91yzfuqzzPrjEKxB/FVefaYaCud
02VlCd/UWh4bXemc4smjsT2ANRXSediuEZNsEmx2fdbiI/FZA1krtAMxGZMI
8cV6jGhngRds+eylSX6+zPp8LIHV9nn9cUvdXh+uNdaUpfStUgVTOl/040KF
8G3xeKXP8Qqh7TcuiKNSP8nabZqjBfDawlw2CjkyC+c3w/7ECTvqJmyHVyMP
tkblpIk7Ybt0Cfsk7Lzx+pcXzl++du3fPy7+denS31ev/gPHxSa8C6+wD8ep
aeSxAxVO2zDgC/A1+jQ28TvYhJ22tj/Fc0Ez78rla0YhvQvH4QA1+bsm7Fy/
foNzKbTZp7GNJ4MvYbsae2FtfjZqoj/YhLfEpsZ5bIrOc9+rRd8lpLy+i8iQ
715kEEbRd2yS713p7Z3bf2B0Gsay51qQ2QJiGt+YMmkXjzKMhuLvz1xctfLw
1ne/xdeGpw8senL/YzP34Ff4GXpgbCxb2gKvr2w65RcMyL7pGw1en71N/2gw
nWhQt2X+Fe2k48Gv6RcPIhwB4yEgGH5N7ksRxQO0MAQgk+F1+tTdGAVFVhTU
7dj6zjcrGxnVzz5z4KlF+48fO4+jDo5rhRQP7G05EgpxfNiyuVXla3TMK6M9
Mubldv+kqPdyHTQT+kdOPtqfD+3dH/wIBC1ZfBB5QrYnP7wT8hlIxvQGqos4
01xlFBDTMCJYHHONREnePlmdmdwOj5xbymTIYRq/Wf7aXPbkXFbymk5cAoGz
Ht2DOW9iEc9xGJDDAZFzCARyJmKQiSsbDxF1Rm9Hr3UFyoA+TEKstMBbjrEg
jA2OkDEcQa1kq9vBWOrjsNQdVTXmGRZF5EocKnMcKTgaFAlH7287Q+xAMmEa
VRA/dKVmIkO+Y2COHXe7MjwekDgW1kSM0Vd5PWraukMa7HLMKJjpFwYP72DW
T8lIvqAecFQDanKZouKjKgwe8+fuEwcuo7+SDTZTY/QHNkArrHvxhFRdUqbB
B/c05SHFEQsLyNo1xwBbxsBAJQN8TqRqwbxmxlb9rhz27naf1ACZN6fJGlkG
uyo2vxqtgnh/YuEnNOeYQzsM2jjwz5ndxGJ8sDLGOep9F85vFktwboxn7fKk
EEesH+FFFoK7WoO2VGFzaCePtlRC1Wh3Yy8VvIIexgq66eUvsh3n0oA4CyWS
ITzENVbz2aZyi4T6XdmOaa+AmFLpG4Y3D3Gqnf8z5TuO0h1UU7rzynZ7cBao
Ym7Z3JpjIRELYkm0WHDdz+XXQAHuRidq8wwpw7oKhGe3dikJwgyOJkwX3tru
vGBNgBDI8ZKQF8Iqcl5MezVD731NZzXKMwuY6KFn4qMPfwJ83n7ra+O2MBwo
agjiT+NUDn8lAgvm2WNT0vi7PlmgRB+vB/wUajajTyLUmimz0Bd0E0NU1Er5
wtazoqvET7FmAcbd/DB25mNGxEWNz1fOlWBsNjedzSFGCJw4fsGKRwcxz5rD
nljTJFGuhC2jUCtWooayGwrYSA1iSrWcrYjhtDOAoUMMs1PSsdmKGIEx0jWO
8fvC3SWFmfkYFUoYkYSsUcYTX93QXdKB2YcSCT0dShhoLQd/y1aUvmr9HTNN
jRJfhVaMQz7o2czHp4DaKFih6I9yjUF8BUSxeCcxaxAheTlKEyukMrMJGZKR
amR4zBRR3fLCkllqO4ZtEorGXRpUqE5lCSokBgVU2LhSxdHA7Dl65FyWoEFC
z0IjyH0U+/eXqjkwXHn78d5fvAoxs/Aznewa7TeP5Tpieo4kmtdFLCleMxhL
535F+2KJ8++S2swwLE+farPuMUSJJWsOYI1YRo+KrG153V6RiChKij69UZQ1
Owz8te2KnqT+MwU90ve1AjzquXl7K9dfxEe6BsgUfEjlIz62chvEo0hU92nt
r0e9jyFvRab5Ott8Ub1ngteOEtB5jSu5UaVngtfoD6vZaq8Hc/Vy48Z/bfzn
i1eEn+ynl79WvWXujHXlMF91aF03vPbq6TT20MnbsRouMW+hgqWxp06u6jzF
WnTg01/T2FPQ1NZVstrTW1gUp/M4JOhed17ytaox/L10IrUa0rdBkftGanSc
L2tGjNYxpJdvpBV1vgXTgZ3ON9J5at+G8nY7Z9qAyL2C/ftduTU0oG4N6cnA
aD1x6tt4JTtDAurSzuWVMys83nPtpZsHxYta+598NTbEVqs0p5wOWDhzwhYW
Qa7i5V8ja67hhyVUsyHx6BctHg9GhgJrDkuodDuLRnAy/YF2jAOdEu4sSDhK
NwkkPE+bk0cH9F+njzuL/47+TcJ/z5MC5RW7w+N+qjmk31UR5r8xQXBJPReX
L235ylofzdzxnr239TvrSTjomVy/h8dxhlWlijve+rmz7RW2OuvVurfjrT96
+JyVR2rrb4+rlG1IuysjyQajjixmMc4tVWjXkLb2DWkrrcx7SIkvz0yFNu1g
q8kgndVeFdnBVpNMVFs9wqNzQtpbkbK9JGQmaiyle0odaSkJjYmubBtpK4WV
jYdQKXSYhfz2MpuqVmN5B9X0kBaWh4nOel/b7Lq8ccPnN9s6Krw66+zZqJ03
2zoqrPWuyKsR6unePT/fbKuoYNZ7VJjqqlx+KrysuO/k7ZCRWZpaZE4OpJzz
BOVMrwnvitcI1fjNLa26CqHzrCxZz5zfUk4K5F3y9/trglfsSD1z5oOCeSb+
Rs6PMdPOp2C1PFJ/nDmd6PwR23zdbaJqr/Ood3CPnHmJKX6+JPgFb3CPEqiC
iDxy5hemCImivtr0W/mB9uIKN1+NEJG9zhoztNeuFmzNVErxXJJUPBvTqFcR
gbvj2voeun/nylnXvz2f5angoft3rn11/fvW6tD9O1ev01ys3xP3rT66Pnsm
7JMSfbrS39H+tSFcvzT4a/q1VopvWH9SHPFS7Rc9fWGt9eNsdb+1qhFJ12P3
IEPODBeXY1SjSKp90DAxQ+kP76sA4wYQpG8q+ynWp+RMTQ/SLxyS64GSTtcD
xsLGl06iMUn0ID5DZ6aLifvc0e13ziJV+NrVZJanmuhUs/5e1lgh6tev+8wv
6os0YW3bE0QDC5Vafv6AXKn5XWLrPoIUSmhTVzFYptuPlTMeD2RNsnc1xtkZ
JD2L12UNAuN6cmkwaxJhYetMfufO9fxZMMT6uyVLFh+scz/DdnZqfSuihN8f
MKlf7DFGfzXB8nWOX38pPfmDz6WyHMQ/vEEPg58nEKjWUn5PuMC/7+P9m2VG
l/8BK+oesQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 124},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->744410417]
}, Open  ]],

Cell[TextData[{
 "Plot the ",
 Cell[BoxData[
  ButtonBox["PDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PDF"]], "InlineFormula"],
 " for the normal approximation:"
}], "ExampleText",
 CellID->428885874],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"NormalDistribution", "[", 
      RowBox[{"500", ",", 
       RowBox[{"5", "*", 
        RowBox[{"Sqrt", "[", "10", "]"}]}]}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "450", ",", "550"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->302802526],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 124},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->138815158]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2662],

Cell["\<\
The kurtosis coefficient is sometimes shifted by 3 to make the normal value \
0:\
\>", "ExampleText",
 CellID->35895314],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Kurtosis", "[", 
   RowBox[{"NormalDistribution", "[", 
    RowBox[{"\[Mu]", ",", "\[Sigma]"}], "]"}], "]"}], "-", "3"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->288434613],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->200040624]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->16710],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Skewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Skewness"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CentralMoment",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CentralMoment"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Variance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Variance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ExpectedValue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ExpectedValue"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->28908]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->30102],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DescriptiveStatistics"]], "Tutorials",
 CellID->4063544],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->96818115],

Cell[TextData[ButtonBox["Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/StatisticalDistributions"]], "MoreAbout",
 CellID->99019789],

Cell[TextData[ButtonBox["New in 6.0: Data Handling & Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60DataHandlingAndDataSources"]], "MoreAbout",
 CellID->67497505],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->158320269],

Cell[TextData[ButtonBox["New in 6.0: Numerical Data Handling",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60NumericalDataHandling"]], "MoreAbout",
 CellID->27247999],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->215775880]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Kurtosis - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 52, 7.8730934}", "context" -> "System`", 
    "keywords" -> {
     "Fisher kurtosis", "Fisher's kurtosis", "heavy tails", "kurtosis", 
      "kurtosis excess", "long tails"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "Kurtosis[list] gives the coefficient of kurtosis for the elements in \
list. Kurtosis[dist] gives the coefficient of kurtosis for the symbolic \
distribution dist.", "synonyms" -> {}, "title" -> "Kurtosis", "type" -> 
    "Symbol", "uri" -> "ref/Kurtosis"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10367, 319, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->119478173]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 41284, 1364}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3357, 73, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3980, 100, 1573, 37, 70, "ObjectNameGrid"],
Cell[5556, 139, 987, 32, 70, "Usage",
 CellID->6122]
}, Open  ]],
Cell[CellGroupData[{
Cell[6580, 176, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7058, 193, 277, 8, 70, "Notes",
 CellID->17901],
Cell[7338, 203, 255, 5, 70, "Notes",
 CellID->22749],
Cell[7596, 210, 210, 7, 70, "Notes",
 CellID->28487],
Cell[7809, 219, 1507, 54, 70, "Notes",
 CellID->26555],
Cell[9319, 275, 319, 12, 70, "Notes",
 CellID->22482],
Cell[9641, 289, 689, 25, 70, "Notes",
 CellID->15967]
}, Closed]],
Cell[CellGroupData[{
Cell[10367, 319, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->119478173],
Cell[CellGroupData[{
Cell[10752, 333, 148, 5, 70, "ExampleSection",
 CellID->109256230],
Cell[10903, 340, 199, 7, 70, "ExampleText",
 CellID->196404163],
Cell[CellGroupData[{
Cell[11127, 351, 157, 5, 28, "Input",
 CellID->18121],
Cell[11287, 358, 1777, 33, 71, "Output",
 Evaluatable->False,
 CellID->689051212]
}, Open  ]],
Cell[13079, 394, 125, 3, 70, "ExampleDelimiter",
 CellID->448710962],
Cell[13207, 399, 205, 7, 70, "ExampleText",
 CellID->25999718],
Cell[CellGroupData[{
Cell[13437, 410, 156, 4, 70, "Input",
 CellID->127140819],
Cell[13596, 416, 206, 7, 51, "Output",
 CellID->524958168]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13851, 429, 216, 7, 70, "ExampleSection",
 CellID->31371],
Cell[14070, 438, 80, 1, 70, "ExampleText",
 CellID->379986608],
Cell[CellGroupData[{
Cell[14175, 443, 166, 5, 70, "Input",
 CellID->216003675],
Cell[14344, 450, 175, 5, 36, "Output",
 CellID->28299823]
}, Open  ]],
Cell[14534, 458, 125, 3, 70, "ExampleDelimiter",
 CellID->158391064],
Cell[14662, 463, 76, 1, 70, "ExampleText",
 CellID->160553898],
Cell[CellGroupData[{
Cell[14763, 468, 255, 8, 70, "Input",
 CellID->190699768],
Cell[15021, 478, 219, 6, 36, "Output",
 CellID->379527010]
}, Open  ]],
Cell[15255, 487, 125, 3, 70, "ExampleDelimiter",
 CellID->183534688],
Cell[15383, 492, 215, 7, 70, "ExampleText",
 CellID->111413134],
Cell[CellGroupData[{
Cell[15623, 503, 375, 13, 70, "Input",
 CellID->48341596],
Cell[16001, 518, 255, 9, 51, "Output",
 CellID->279328463]
}, Open  ]],
Cell[16271, 530, 125, 3, 70, "ExampleDelimiter",
 CellID->650909219],
Cell[16399, 535, 88, 1, 70, "ExampleText",
 CellID->285474028],
Cell[CellGroupData[{
Cell[16512, 540, 191, 6, 70, "Input",
 CellID->315224713],
Cell[16706, 548, 176, 5, 36, "Output",
 CellID->428024305]
}, Open  ]],
Cell[CellGroupData[{
Cell[16919, 558, 245, 8, 70, "Input",
 CellID->594479966],
Cell[17167, 568, 479, 12, 54, "Output",
 CellID->4831479]
}, Open  ]],
Cell[17661, 583, 125, 3, 70, "ExampleDelimiter",
 CellID->373178873],
Cell[17789, 588, 101, 2, 70, "ExampleText",
 CellID->17058508],
Cell[CellGroupData[{
Cell[17915, 594, 184, 5, 70, "Input",
 CellID->108321647],
Cell[18102, 601, 808, 24, 58, "Output",
 CellID->116078678]
}, Open  ]],
Cell[CellGroupData[{
Cell[18947, 630, 154, 4, 70, "Input",
 CellID->130372047],
Cell[19104, 636, 205, 7, 53, "Output",
 CellID->626115038]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19358, 649, 239, 7, 70, "ExampleSection",
 CellID->32490],
Cell[19600, 658, 211, 8, 70, "ExampleText",
 CellID->625271644],
Cell[CellGroupData[{
Cell[19836, 670, 569, 19, 70, "Input",
 CellID->19878761],
Cell[20408, 691, 481, 17, 36, "Output",
 CellID->248710404]
}, Open  ]],
Cell[CellGroupData[{
Cell[20926, 713, 105, 3, 70, "Input",
 CellID->235263230],
Cell[21034, 718, 382, 12, 51, "Output",
 CellID->577701629]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21465, 736, 232, 7, 70, "ExampleSection",
 CellID->1281],
Cell[21700, 745, 327, 12, 70, "ExampleText",
 CellID->209192808],
Cell[CellGroupData[{
Cell[22052, 761, 158, 5, 70, "Input",
 CellID->388198497],
Cell[22213, 768, 765, 16, 71, "Output",
 CellID->368476844]
}, Open  ]],
Cell[CellGroupData[{
Cell[23015, 789, 107, 3, 70, "Input",
 CellID->934643318],
Cell[23125, 794, 176, 5, 36, "Output",
 CellID->340669345]
}, Open  ]],
Cell[CellGroupData[{
Cell[23338, 804, 251, 8, 70, "Input",
 CellID->199964197],
Cell[23592, 814, 176, 5, 36, "Output",
 CellID->121973022]
}, Open  ]],
Cell[23783, 822, 125, 3, 70, "ExampleDelimiter",
 CellID->203852233],
Cell[23911, 827, 482, 17, 70, "ExampleText",
 CellID->212776658],
Cell[24396, 846, 162, 5, 70, "Input",
 CellID->107789233],
Cell[CellGroupData[{
Cell[24583, 855, 386, 13, 70, "Input",
 CellID->273951061],
Cell[24972, 870, 396, 14, 57, "Output",
 CellID->191573611]
}, Open  ]],
Cell[CellGroupData[{
Cell[25405, 889, 131, 4, 70, "Input",
 CellID->370222015],
Cell[25539, 895, 310, 11, 56, "Output",
 CellID->271128780]
}, Open  ]],
Cell[CellGroupData[{
Cell[25886, 911, 128, 4, 70, "Input",
 CellID->3806373],
Cell[26017, 917, 160, 5, 36, "Output",
 CellID->23448660]
}, Open  ]],
Cell[26192, 925, 125, 3, 70, "ExampleDelimiter",
 CellID->246852425],
Cell[26320, 930, 217, 8, 70, "ExampleText",
 CellID->421003384],
Cell[CellGroupData[{
Cell[26562, 942, 184, 5, 70, "Input",
 CellID->114256448],
Cell[26749, 949, 158, 5, 36, "Output",
 CellID->491306028]
}, Open  ]],
Cell[26922, 957, 124, 3, 70, "ExampleDelimiter",
 CellID->75092913],
Cell[27049, 962, 237, 8, 70, "ExampleText",
 CellID->659817507],
Cell[27289, 972, 214, 7, 70, "Input",
 CellID->524716893],
Cell[CellGroupData[{
Cell[27528, 983, 107, 3, 70, "Input",
 CellID->929317708],
Cell[27638, 988, 163, 5, 36, "Output",
 CellID->218502496]
}, Open  ]],
Cell[27816, 996, 203, 8, 70, "ExampleText",
 CellID->222429037],
Cell[CellGroupData[{
Cell[28044, 1008, 411, 13, 70, "Input",
 CellID->207864196],
Cell[28458, 1023, 3763, 66, 70, "Output",
 Evaluatable->False,
 CellID->744410417]
}, Open  ]],
Cell[32236, 1092, 211, 8, 70, "ExampleText",
 CellID->428885874],
Cell[CellGroupData[{
Cell[32472, 1104, 385, 12, 70, "Input",
 CellID->302802526],
Cell[32860, 1118, 2864, 51, 70, "Output",
 Evaluatable->False,
 CellID->138815158]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[35773, 1175, 225, 7, 70, "ExampleSection",
 CellID->2662],
Cell[36001, 1184, 130, 4, 70, "ExampleText",
 CellID->35895314],
Cell[CellGroupData[{
Cell[36156, 1192, 209, 6, 70, "Input",
 CellID->288434613],
Cell[36368, 1200, 158, 5, 36, "Output",
 CellID->200040624]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[36587, 1212, 311, 9, 70, "SeeAlsoSection",
 CellID->16710],
Cell[36901, 1223, 961, 35, 70, "SeeAlso",
 CellID->28908]
}, Open  ]],
Cell[CellGroupData[{
Cell[37899, 1263, 314, 9, 70, "TutorialsSection",
 CellID->30102],
Cell[38216, 1274, 156, 3, 70, "Tutorials",
 CellID->4063544],
Cell[38375, 1279, 153, 3, 70, "Tutorials",
 CellID->4738],
Cell[38531, 1284, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[38725, 1292, 299, 8, 70, "MoreAboutSection"],
Cell[39027, 1302, 154, 3, 70, "MoreAbout",
 CellID->96818115],
Cell[39184, 1307, 160, 3, 70, "MoreAbout",
 CellID->99019789],
Cell[39347, 1312, 184, 3, 70, "MoreAbout",
 CellID->67497505],
Cell[39534, 1317, 179, 3, 70, "MoreAbout",
 CellID->158320269],
Cell[39716, 1322, 174, 3, 70, "MoreAbout",
 CellID->27247999],
Cell[39893, 1327, 151, 3, 70, "MoreAbout",
 CellID->215775880]
}, Open  ]],
Cell[40059, 1333, 27, 0, 70, "History"],
Cell[40089, 1335, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

