(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    154046,       4157]
NotebookOptionsPosition[    135726,       3545]
NotebookOutlinePosition[    137926,       3595]
CellTagsIndexPosition[    137839,       3590]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"LatticeReduce" :> 
          Documentation`HelpLookup["paclet:ref/LatticeReduce"], "GraphData" :> 
          Documentation`HelpLookup["paclet:ref/GraphData"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"LatticeReduce\"\>", 
       2->"\<\"GraphData\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Computational Geometry" :> 
          Documentation`HelpLookup["paclet:guide/ComputationalGeometry"], 
          "Cryptographic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/CryptographicNumberTheory"], 
          "Integrated Data Sources" :> 
          Documentation`HelpLookup["paclet:guide/DataCollections"], 
          "Discrete Mathematics" :> 
          Documentation`HelpLookup["paclet:guide/DiscreteMathematics"], 
          "Mathematical Data" :> 
          Documentation`HelpLookup["paclet:guide/MathematicalData"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Computational Geometry\"\>", 
       2->"\<\"Cryptographic Number Theory\"\>", 
       3->"\<\"Integrated Data Sources\"\>", 
       4->"\<\"Discrete Mathematics\"\>", 5->"\<\"Mathematical Data\"\>", 
       6->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["LatticeData", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["LatticeData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/LatticeData"], "[", 
       RowBox[{
        StyleBox["lattice", "TI"], ",", " ", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the specified property for a lattice."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["LatticeData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/LatticeData"], "[", 
       StyleBox["n", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of named lattices of dimension ",
     Cell[BoxData[
      FormBox["n", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "Lattices can be specified by standard names such as ",
 Cell[BoxData["\"\<FaceCenteredCubic\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<CoxeterTodd\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->1399648665],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LatticeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeData"], "[", "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " gives a list of classical named lattices."
}], "Notes",
 CellID->1291358921],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LatticeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeData"], "[", 
   StyleBox["patt", "TI"], "]"}]], "InlineFormula"],
 " gives a list of all named lattices that match the string pattern ",
 Cell[BoxData[
  StyleBox["patt", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->1682356610],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{"LatticeData", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\"\<\!\(\*
StyleBox[\"type\", \"TI\"]\)\>\"", ",", 
       StyleBox["id", "TI"]}], "}"}], ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "]"}], " "}]], "InlineFormula"],
 " gives data for the lattice of the specified type with identifier ",
 Cell[BoxData[
  StyleBox["id", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->255092973],

Cell["Type-related lattices include:", "Notes",
 CellID->1227577802],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\"\<BarnesWall\>\"", ",", 
       StyleBox["n", "TI"]}], "}"}], " "}], Cell[TextData[{
     "Barnes-Wall lattice ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["BW", "TI"], 
       StyleBox["n", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<Bravais\>\"", ",", 
      RowBox[{"{", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"system\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"centering\", \"TI\"]\)\>\""}], "}"}]}], "}"}], Cell[
    "Bravais lattice for the specified crystal system", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<CoxeterBarnes\>\"", ",", " ", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["r", "TI"], ",", " ", 
        StyleBox["n", "TI"]}], "}"}]}], "}"}], Cell[TextData[{
     "Coxeter-Barnes lattice ",
     Cell[BoxData[
      SubsuperscriptBox["A", 
       StyleBox["n", "TI"], 
       StyleBox["r", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<DualRootLattice\>\"", ",", 
      StyleBox["group", "TI"]}], "}"}], Cell[
    "dual root lattice for the specified Lie group", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<IntegerLattice\>\"", ",", 
      StyleBox["n", "TI"]}], "}"}], Cell[TextData[{
     "integer lattice ",
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[DoubleStruckCapitalZ]", "n"], TraditionalForm]], 
      "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<KappaLattice\>\"", ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["m", "TI"], ",", 
        StyleBox["n", "TI"]}], "}"}]}], "}"}], Cell[TextData[{
     Cell[BoxData[
      FormBox["\[Kappa]", TraditionalForm]], "InlineMath"],
     "-lattice"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<LaminatedLattice\>\"", ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["m", "TI"], ",", 
        StyleBox["n", "TI"]}], "}"}]}], "}"}], Cell[TextData[{
     "laminated lattice ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["\[CapitalLambda]", 
        RowBox[{"m", ",", "n"}]], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<MordellWeil\>\"", ",", 
      StyleBox["n", "TI"]}], "}"}], Cell[TextData[{
     "Mordell-Weil lattice ",
     Cell[BoxData[
      FormBox[
       SubscriptBox[
        StyleBox["MW", "TI"], "n"], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<Niemeier\>\"", ",", 
      StyleBox["n", "TI"]}], "}"}], Cell[
    "Niemeier lattice with", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<PerfectLattice\>\"", ",", 
      StyleBox["n", "TI"]}], "}"}], Cell["perfect lattice", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<Quebbemann\>\"", ",", 
      StyleBox["n", "TI"]}], "}"}], Cell[TextData[{
     "Quebbemann lattice ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["Q", "n"], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{"\"\<RootLattice\>\"", ",", 
      StyleBox["group", "TI"]}], "}"}], Cell[
    "root lattice for the specified Lie group", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, None, {None}},
    "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, True, 
      True}}}},
 CellID->1287804506],

Cell[TextData[{
 "Crystal systems are specified by standard names such as ",
 Cell[BoxData["\"\<Trigonal\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Monoclinic\>\""], "InlineFormula"],
 ", etc. Centering is ",
 Cell[BoxData["\"\<FaceCentered\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<BodyCentered\>\""], "InlineFormula"],
 " or ",
 Cell[BoxData["\"\<BaseCentered\>\""], "InlineFormula"],
 ". "
}], "Notes",
 CellID->288915648],

Cell[TextData[{
 "Groups can be specified either for example as ",
 Cell[BoxData["\"\<A5\>\""], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"\"\<A\>\"", ",", "5"}], "}"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->169971442],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LatticeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeData"], "[", 
   RowBox[{
    StyleBox["group", "TI"], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"LatticeData", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\"\<RootLattice\>\"", ",", 
       StyleBox["group", "TI"]}], "}"}], ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "]"}], TraditionalForm]], 
  "InlineFormula"],
 "."
}], "Notes",
 CellID->39805619],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LatticeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeData"], "[", "\"\<Properties\>\"", "]"}]], 
  "InlineFormula"],
 " gives a list of possible properties for lattices."
}], "Notes",
 CellID->1067943069],

Cell["Lattice points properties include:", "Notes",
 CellID->1024721485],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Basis\>\"", Cell[
    "basis vectors", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Determinant\>\"", Cell[
    "determinant of Gram matrix", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Dimension\>\"", Cell[
    "dimension of lattice", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Dual\>\"", Cell[
    "dual lattice", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<GeneratorMatrix\>\"", Cell[
    "matrix of generators", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Genus\>\"", Cell[
    "genus of lattice", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<GlueVectors\>\"", Cell[
    "glue vectors (when applicable)", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<GramMatrix\>\"", Cell[
    "Gram matrix", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Image\>\"", Cell[
    "configuration of points (when applicable)", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<KissingNumber\>\"", Cell[
    "kissing number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<MinimalNorm\>\"", Cell[
    "minimal norm", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<MinimalVectors\>\"", Cell[
    "minimal vectors", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ModularNumber\>\"", Cell[
    "modular number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<RadialFunction\>\"", Cell[TextData[{
     "lattice points as a function of radius (",
     Cell[BoxData[
      FormBox["\[CurlyTheta]", TraditionalForm]], "InlineMath"],
     " series coefficients)"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ThetaSeriesFunction\>\"", Cell[
    "pure function for the theta series ", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, None, None, 
     None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, True, 
      True, True, True, True}}}},
 CellID->601387462],

Cell["Lattice-packing-related properties include:", "Notes",
 CellID->1514415704],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<CenterDensity\>\"", Cell[
    "center density", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<CoveringRadius\>\"", Cell[
    "covering radius", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<CoxeterNumber\>\"", Cell[
    "Coxeter number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Density\>\"", Cell[
    "average sphere packing density", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HermiteInvariant\>\"", Cell[
    "Hermite invariant", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PackingRadius\>\"", Cell[
    "packing radius", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Thickness\>\"", Cell[
    "thickness", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Volume\>\"", Cell[
    "volume of the fundamental region", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True}}}},
 CellID->808552884],

Cell["Other properties include:", "Notes",
 CellID->1433705930],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<QuadraticForm\>\"", Cell[
    "quadratic form", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<AutomorphismGroupOrder\>\"", Cell[
    "order of the automorphism group", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True}}}},
 CellID->1529046170],

Cell["Naming-related properties include:", "Notes",
 CellID->1597847868],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<AlternateNames\>\"", Cell[
    "alternate English names", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Name\>\"", Cell[
    "English name", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Notation\>\"", Cell[
    "standard notation for display", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StandardName\>\"", Cell[TextData[{
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " name"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True}}}},
 CellID->746111741],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LatticeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeData"], "[", 
   RowBox[{
    StyleBox["lattice", "TI"], ",", " ", "\"\<Classes\>\""}], "]"}]], 
  "InlineFormula"],
 " gives a list of the classes in which ",
 Cell[BoxData[
  StyleBox["lattice", "TI"]], "InlineFormula"],
 " occurs."
}], "Notes",
 CellID->743016154],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LatticeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeData"], "[", "\"\<\!\(\*
StyleBox[\"class\", \"TI\"]\)\>\"", "]"}]], "InlineFormula"],
 " gives a list of named lattices in the specified class. "
}], "Notes",
 CellID->1461031052],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["LatticeData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeData"], "[", 
   RowBox[{
    StyleBox["lattice", "TI"], ",", " ", "\"\<\!\(\*
StyleBox[\"class\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " depending on whether ",
 Cell[BoxData[
  StyleBox["lattice", "TI"]], "InlineFormula"],
 " is in the specified class."
}], "Notes",
 CellID->1220871166],

Cell["Basic classes of lattices include:", "Notes",
 CellID->1822633102],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Even\>\"", Cell[
    "even", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Extremal\>\"", Cell[
    "extremal", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Integral\>\"", Cell[
    "integral", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Odd\>\"", Cell[
    "odd lattice", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Unimodular\>\"", Cell[
    "unimodular", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True}}}},
 CellID->2122923524],

Cell["Negative classes of lattices include:", "Notes",
 CellID->1185684217],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Nonextremal\>\"", Cell[
    "not extremal", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Nonintegral\>\"", Cell[
    "not integral", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Nonunimodular\>\"", Cell[
    "not unimodular", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True}}}},
 CellID->1576899322],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  ButtonBox["LatticeData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LatticeData"]], "InlineFormula"],
 " may require internet connectivity. "
}], "Notes",
 CellID->94188222]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->334793323],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->175436389],

Cell["The face-centered cubic lattice:", "ExampleText",
 CellID->1908265251],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<FaceCenteredCubic\>\"", ",", "\"\<Image\>\""}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->1693289259],

Cell[BoxData[
 Graphics3DBox[{
   GraphicsComplex3DBox[{{-1, -1, -1}, {-1, -1, 1}, {-1, 1, -1}, {-1, 1, 1}, {
    1, -1, -1}, {1, -1, 1}, {1, 1, -1}, {1, 1, 1}, {0, 0, 1}, {1, 0, 0}, {0, 
    1, 0}, {-1, 0, 0}, {0, 0, -1}, {0, -1, 0}}, {
     {Opacity[0.1], EdgeForm[GrayLevel[0.8]], 
      Polygon3DBox[{{8, 4, 2, 6}, {8, 6, 5, 7}, {8, 7, 3, 4}, {4, 3, 1, 2}, {
       1, 3, 7, 5}, {2, 1, 5, 6}}]}, 
     {GrayLevel[0.8], 
      Line3DBox[{{9, 8}, {9, 4}, {9, 2}, {9, 6}, {10, 8}, {10, 6}, {10, 5}, {
       10, 7}, {11, 8}, {11, 7}, {11, 3}, {11, 4}, {12, 4}, {12, 3}, {12, 
       1}, {12, 2}, {13, 1}, {13, 3}, {13, 7}, {13, 5}, {14, 2}, {14, 1}, {14,
        5}, {14, 6}}]}}], 
   {GrayLevel[0], Specularity[
     GrayLevel[1], 5], SphereBox[{-1, -1, -1}, 0.06], 
    SphereBox[{-1, -1, 1}, 0.06], SphereBox[{-1, 1, -1}, 0.06], 
    SphereBox[{-1, 1, 1}, 0.06], SphereBox[{1, -1, -1}, 0.06], 
    SphereBox[{1, -1, 1}, 0.06], SphereBox[{1, 1, -1}, 0.06], 
    SphereBox[{1, 1, 1}, 0.06], SphereBox[{0, 0, 1}, 0.06], 
    SphereBox[{1, 0, 0}, 0.06], SphereBox[{0, 1, 0}, 0.06], 
    SphereBox[{-1, 0, 0}, 0.06], SphereBox[{0, 0, -1}, 0.06], 
    SphereBox[{0, -1, 0}, 0.06]}},
  Boxed->False,
  ViewPoint->NCache[{4, 
     Rational[5, 3], 1}, {4, 1.6666666666666667`, 1}]]], "Output",
 ImageSize->{184, 172},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->931804828,
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJzFnPl7FMeZx0vT19z3KWmk0Uga3WJBB0gyCMJlFCDYmBtsi8tgcxgMwZhg
wMFZY7MQEmPzGBvjC5P4AOcBbEhiwAtkH54nf8H+Sp6HZ3eT4OfJr352q6qr
Z1rfaakHIbI8Yrqrp7veT7311ltVb1XPvMEdG9ZtHtyxcc1gZvb2wec3bFzz
QmbW1u30klRGCLlA/3+eJPT8f+kZ/3/hhx9+oF+U8HmAfjroIw6ZkHH8yn72
qbJLhMTKlAma+iO31CaTdv7tPvbpJOyfw++IdsUaN7W2HZ1Us7/V0+4ibXhP
OB6aPDBx3ZG1W6/uXXJxU8uKNkW/Zy+Te/v2bf0+R12sesPsJe+/dvyLS5de
//z9gWVzfPp9P6WfGr3PcefOP8S9OU92e/vghac+urn/xrsvfPSTlslu/d7n
6adK73Wxe7Ubf/wPQgJl0Z8EZx1u2vvuxGM7cmtaRSk2ilz97E4XvZMeGQzx
k+h4qb1Pbqn+69/+zq7oOa8W90fZ/VLhfokeWSkc9G4i0ZMf+D/SzJ9aKXjK
4SlZPCXBU038qUVCVhW7ixVC0h9jmRhPyvBkI39yvniyjt+qy9OfVTkBPRGl
wucb+POzxfPNRc9r8LwCz+f489NEeccJcsl4Ss+HKJCLCrnU81x6BUWnSWvs
1AG5YZk0yK2O59Yhcuu1yY2XUNZz0zPE/Gp5fq0iv6k2+TmL8nNCflmeX07o
bKbQmQz5uIrycUE+NTwfL/ukbZi4qNETItFPQToApFgrbl41eQlCmyglw6V4
DN8Ql9z9gcDyuGuOv4yK04SsBTayPJay3CCrmstyG17EKcU7Kyb+dPLMMwu7
35mVGCgvEzpbJHSmliTFA1KqhkjRPL5HpvS9eGjXsaundl1+o3/tDC+9dTmU
RwFJXktJXpCULkjSaHHU0GNtP/7wubdunrz5yfFzK368KEKcVMTjNrJ8Jntn
AvI+R6RRi5WFGtOIqyfY+28d+y7M//Stma8PVI9LMcfCbJ65SbQQlOy3kYya
rShIppotI4EutXOxd/b0cFMNvenenf+molhpWKmnijrUQGbARibquJzL9HH7
zAuhn/o5kxARpZ0EpUXrCdpI9oHkVEGylJdMHPRopNkpI0iKMo+zIQgBgWRD
kLQkkJhYJU8grlYLhnqhd9d9yvaD7ERBtpyXxXw/ky6bpTuEVFYHadCAEyjC
NhQBoIgPQyFjLfB0QW6T4CHS768MaWdqIp6rjuea4rlsPEcv0HvcwBgBRtlG
UybGolqx4mT6Gi/4tAJT27Sa7mdau59tabFgitowod5iIzApwCQJpm5DZ6pf
q8zFpw32rDy9fDm98PyEJFpTzIYnCDzREXhU4JGhLnt4N+7Xwn7n+Kam7WvX
fnDmnXcvv7f+paca+G3o1eKWbH/+07+LdAjYIkWtbIhFDaEaSsm0xscfqt/Z
Wdn66qxtV3d98c3Pv9w1sKKB9ZAyt0D6DD2y0wSQKaA1JAuXRMbqk40dVUE0
8Y4YJfu1lL98RfX8Yx173piweX4wGSbojZnHQI+MVGGgCo1AdevKdR2MaKAv
BWq1g+stn6Z//mCjP1WHHsMlrNJpQxkByqAlpQS6s6JkOpwAfsxj4U2dRu0O
QxQ17E4QBUoicgKRUat631LwZkhUbqSpQPpHFPYh6QMXBhID/fgtaWQbGg1q
sVn0Nj6gqbCsLYqkAxIH/XDcuHETbctXEpXLgopRNEGteYGqEqhUqDGmLicl
4+Me5qfoINVDSLaM1JeJcZ4tmxvYnKCxehvGtA0jY9MoI/dqaibDRl7tgern
6mt/0RpZECEuS0qlBEqmwVpRnwEbKu2vxZ5LZhWrJkKR2vK+2d1r335263f7
Bo4siJRG5AEilyDKgL78QFZlQ8Z6RlWQVWbqn56y7JN//ejS+W9ffml3zppM
tSFzQ41W2hBW2xCyGnZRQvX7v/wXo4wms8vTT3w65/Tlled2Nj8e1ym9QKkB
pdeCkumvXNRoEKgyQJWfxZr8R4ARsZNEqJp+4x1Y7OrLinZgx+MDHo/gSYC2
0NJqkIv7MZqmLAwnYjDRo36VtVOPJZPThskLNRmxYWM1rRpeX+jJyXhMfGFe
izqfGbE0Qr8FIdNa2IYsJvoj9e5//t1MM7Z0AaDzCTq/sLEQUPkFlR+oXEAV
ZB+ymcph3CH6Bes24LLh80P9ukGLyKsK3rgNb75d6Glhg6NjDlowF+ZvFqxi
LMnGRFU2nH5LTqkkTrcNJ9e1luejU5jyeC4Tz9ETYowI8m2FHhlqzobXx9N5
Xl2jSv77Eds68oYseHnUVaFTrZk1XRtbxjNvRGmNWlcMOnpkpw1A6wZaD9BK
D0AbttKuko3XNyVmbJ2++st1iz9Y1MrV7QXKRhtKtxWl6fvSrNZjw8tbmpqN
d/d2/2zzjnNffHbyozcX9LZFQ0DbZEPrAlp5TGgjFn6L0baxONXObzd88emy
QwvSiWwQaJv/X2ijVrQKC3R1rauY/0RlbRMzYztWD7BoNqyleQWvDau5LLzx
cyq9+bNTtN4WG2oVqJUxoY5ZjJXM1DJQu4G6zcYuFKBWR0Xts6HmvbOc/z7v
wQrcTuBut9G2ZMMtj4o7DtyqYc/3hkSBgF0F9nE27A5g18aEPQHsiiW7Buwy
sP+LFfuQnoKNJLE3VEoi9tsQY3whHz0aoudC2gHs463Y5Xyagltb//1yJy01
qwAt17SST9M/1sd2AKPXRMQ0OzaEKd7e8t+z9gZ0TpMuGVfncFy6mRbVuFoS
V8CKS4yy8jZqonJBjY6krbGjKgc7lC11pVOxmkZd5UdX+qWiOiyNKmhlZSLN
asjavvK6EmyoMc4yZL6COtNGoMMxNbTdIW2AMaLm0DNiH4vehrMVzQLuh3h4
fSasvEtRhF8DYr2f+varW2JegL7dCb7cMRzrfbGHLPvTQlouat/IjX4exwF2
YywcvThHxR2FtHn95NaV60X24QLq1n8KdRgoI5DmMVt6O9ujRGSViH1F9IoP
eHGMaDcmRH7XqHgxzeftGmHL3iRGlE6nNssrdal8a4YfiHEsbjceLI3YA8So
UbQTtyB2eB2RrljDlrb2X/dlD4739QUJCQAxzstkIMY4Es4iSyNGy0VvwnWs
20EgGuybM3HNkfXPf3tg2YUX2ga7tSAw48wXR6/o4dCLeIDZWRIz9nMeXctl
6XDF4PQnTv3i7a9+d+WXZ8/MW7owREJAjBEFHLNiH/dQiUlCK3+mYcmXS967
9eL10xs/Xpib6iVhIM4Z/WG+72DjKd+YcKL/xbG312hxxDXJ1bW76rljzQe2
VCxqZL1XBDjr+bDbPCrlMbrCFXrivw8q6kfNFEgpRghECpNojT+WLho1Y83X
jRlf3Ko2rdYBRN+KvQGS1ZpsktVt0OCien8QLoxwmdcn2JgA56bIlTW+F2Nl
PeaaCPnopRAhGUJSzPOOig1jsN6inh/HzEhXAy2ZcTgoHFEawsnJofiGdHRz
2tlJfZmvJEIcR2Hk3WNY3T3zuArjEkhZbaTFGJrrUKOYyVB1d+20Nb2Pfbhi
7qWnWnd0OUfJias+HlMt37tTPNu002saZlBBQRxMxft6J+3YuOHkZ6eOXT+z
ct+a1D+J2E7HFbxl59P0RoM5kEjOaJ12ZGDfdy///us3/rBt3jOVJTIngdFu
rVQCZo8Nc+qu1Sw6xIcmiXAzTVSuPt39qzc7XpmrVUqjY8Z1cWTGWQ1GBpE5
Aem8x/3etIbE0qRigqMxp9u6LXUKKDFaguvkOF/02VDrK0l0pmNHrbvaUVLj
jiCkVoHab0HN1kZiQC8Bre6DzaMBxlyaD+bMRfEdXLXWgDQApDg+QX3LY0jM
IxisZbOtZ1qYnqANO4EWY/JIi6NuZcz1m42n01qkxxub6Q83aeGiVofEuD6D
xDgXw/Vlv8l+rUaC3pLYue/QKHtTvHlKqmtXR99bU9q2tyTKaQlwr48LShC2
KYEH0ppNCXBmU9pY1ihBbXvN3AWP7Dixdf+11wZPrmufUBEu2kPlhhJEbEqA
83cnlABXTXE2eX8lyDXn1s9YdPaVd6588vWRA4emZzPhol1yHihB1KYEGDdx
WZagkH6AEqjUirKZpVTWrNcuLn3/YM+aLqT3An3Mhh4j5A+HPmV4m2z93GjP
0mh3N3L7gDtuwz3sKha2ze+toxLu0rVevJoD7H5gT1ixq7qvDWrhtBau4B7X
9/CYR+g/JRv6pFWvr3L0WLsvs6Iiu6kq2esLPxx6HNFar/AU7g8AfQrog4I+
mvR1zGtd/ObCVRfXTN3TV4H0uOKGMSGMAZVGLxt2f88IeIeBtxx4A4I3nUqu
nL/g5IlfnblydsfLz7Y9HF6cO5rH4qx3jgBtBdD6BW0qlFo3YelXq9+/tvP8
gR+tan44tBhTyY/89FuKaCuB1pdvh6728LgtNav21CyeYdEOMVKFsbYHob2d
X8lC2jTQesHD4UoIejykxjRGtnVqd6nUwvdFgboKqN1Aiav22L+gboe1lBGp
MSqo2VBXA7UTKLFPx1HJw6HW9w7kvxfsaCcZYLfuwYcfE44NO8bnsU/EXicE
paiBUmB/jiNbu72fOE7BVbPSSoFvD+E8PwilYHE8/k400DKPwt/xvE9q1L1m
SY2rN0iNERWkrjPpnlkYjq1wZjw21LiCg9S4Mx2p661GVSPETHAdNWBjIaVR
45tS1ruIkVrvo+4a86Gi6JTlHtz7ZsUVJ2S13lEMdjEk2hctujI2pLhqg7Zg
R1o7XN1b2AbTL1pwcEyoUb+4WoEj6yxQ406qsMlWrCw4BGncJWJNjTt9kBrX
gJC6ZjhPPIS6QIUr1uExoR521nLPev6YGa4HvwuxIbECMzbUuMqH1GhBSF0N
1BifD5p0zd9jGRNqtACkRr+C1FVAjdFuPqcZsj6IcagopHEPW2ncODfEfhJj
I8iN8W4+uylaH8Yo4OjY0Vegzu3Y08COsWSc05hX45lHjI+KGtc9kRrHU0hd
CdQYofUCNdpJ7D6oCxRIjXaCcx6MAFYANcY0PcNRC9+Cusbduta6xj4QqYdd
K79rjiwMHwnE+aNZ18zOH4C6aJdfObAa7z+iZjFyhrPF4ndmGClaRWmkTkvS
CgvS4vjS8PEmnCmit8BdMRH4HnfPW7NrluyVNuxJYMdojnOon6YnuBsN+5cH
oU0DLa7eIC1Gc7QiWtzrh+Ol0miLd/Az2iobWlzBxbiIWkTLxqTsTfHRURbv
1WeU1TaUcaDEWIJ5zY7lh2+B4Wgf3wEagXVIFLQGOHFlFNdqs8ApA6cXOIdd
uRuRszham7XhjAJnHXBKwImrWcPuNbPnNEVp64ASV8RxP1kOKNnsiUUxAkCH
Ef3S6CSgq7ehw3l8I2/cOp2+I9VFhRB946QVYdGassuSy1FUu3ZkuP7Od76y
PXsyIQkid6jKdLejQ+GXkKz4LZgCmctGYw3AhfsycK8o59J3lspBR3hiOLMh
V3dofPn+ZvfkgJjrwj7MEskcQNY4WrIyEqsI9cybsOTw0lVXtg78bl3Dxm4t
BGQuOzItT3bP9EtZpBmocPdNAKjY2Nl/W/8VR7lMbq2oWT/v8TePvP7eHz47
cPHUo88uC4etyIZ4bxOVClQ02WJDhHuq2Wg9RJ/z3PnLPyReaPZDlkpHMPez
ntVfb/7wxtGbHxw8t7R/UZh9iXTOkegK8WQdjbTasBksKcEi0yPVVFlaqlib
XHC669D5Rz9+beK+fjlCxB1oYdowPCwGKzErle4ZP3FG2oCGWRyT3iDydjLp
CpHSZdVznP1L3NO6WHNT6FUrj2WOmBRsh/spVW+3MUIyZbT1EuHdmSwi/c/f
7l3705+ZKC/7kPQEO60VHAlxRA+uWkpUubWykhKpSvXNiYQ3pn1Pp+QWanEh
IdVLhXJJ9MhPjbSQ7AaSDP1gSk2LsmOfpxRp3TmERfIp6b7qvj3TZv92ee/H
C9ODrQphHFXA4QAOF3BUCU3UCg4cI8gjcdDyB4LBmTOn7H5j99Hrp168cnzG
lkUhp0I8LMf7JKkQJI3iONIby7y5D9FH2B1/csKCT7a89d3JWx//8uKTA4NJ
ju2kUpl3UKigECNQ8kS6mRTVjJ9pwh1KEMkfStAjN1lRW+1CS/jWjzQiGyHe
R6P9J7oPfDPvzNv9r8/1U8OR6XcsxwSliILVOi2IiEpRKFB9qHxKMjU1magM
JRxCU53iiON7hyUV/5RJqFftXOKe2l+W8jArVyxqjP36pEyT+ZbLGNyh8sp4
69zm3r1TJh2e1si+6xVa4TarwdtbQ2SXi1JnWPZyXhI90aDMLO0UXp32N6ma
5Jz1j2767c5n6LVXHw/7VGJIZkdJjL5ZiR15G60S0iqLpKkgLcDVfpvvJCfy
uHTLnhVbv3z387PnL2/eubuW+Uw3ryP6xG3+j3deTaLc5rc52HfsPSF9NJUV
BMkiAgUIfEZa5O/g+iaOBm/jvo6tf3zy3KX1F7aPW16epNcjLEchmzvJRMhH
/zyENBJHj8p+BEq3Pf7Lrkx+zFSzzOZlkG70VFwvMvH2unq3xZ/eFJrX3irk
aUZfpuR8VYN1DUc7K19q0MK0qAqX1Cwkme3HbyPVKfKca9Q16wtcalt308qD
q7Zd27/s/NbcpCpDwjghwUdzczCrhNyMNvoIPcZYWUTr8yi+ue3T39555PJn
V4+fOD29vtEhcuwQOboMv+A2GjyjVaBGWgxK0zVmFRVGf0Tc06NT35ly+OrK
i8envzJRr4GJrJcEvXBPqFRlsr5QojaU6IgmMszdCG2wnloBOexNgvF6u6Xd
Z2yac+pT3jk9uo31GZqnOdbVJlqWtXYdnNzGSiCZ6o1pLEiPj/FnpuR7NEd6
XO0T+1dsufrKMiYjTccZ9Lie39XPS1amdOZ6jmw5fOXstRNHT09P1zm2Fr5V
SFnG2/5qx95vFvzmWMf28fz3rwmZykeeREqUZfulxlqyjV8t/uVvUvZ/XHjY
4Q==\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->129938682],

Cell["A basis for the face-centered cubic lattice:", "ExampleText",
 CellID->1653899797],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<FaceCenteredCubic\>\"", ",", "\"\<Basis\>\""}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->65561411],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", 
     RowBox[{"-", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1392990272]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(22)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Names and Classes",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(15)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->2127050445],

Cell["\<\
Obtain a list of classical named lattices, including non-lattice packings:\
\>", "ExampleText",
 CellID->1759462993],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->444704161],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"BaseCenteredMonoclinic\"\>", 
   ",", "\<\"BaseCenteredOrthorhombic\"\>", ",", "\<\"BodyCenteredCubic\"\>", 
   ",", "\<\"BodyCenteredOrthorhombic\"\>", ",", "\<\"CenteredTetragonal\"\>",
    ",", "\<\"CoxeterTodd\"\>", ",", "\<\"FaceCenteredCubic\"\>", 
   ",", "\<\"FaceCenteredOrthorhombic\"\>", 
   ",", "\<\"HexagonalClosePacking\"\>", ",", "\<\"HexagonalLattice\"\>", 
   ",", "\<\"KorkineZolotarev\"\>", ",", "\<\"Leech\"\>", 
   ",", "\<\"SimpleCubic\"\>", ",", "\<\"SimpleHexagonal\"\>", 
   ",", "\<\"SimpleMonoclinic\"\>", ",", "\<\"SimpleOrthorhombic\"\>", 
   ",", "\<\"SimpleTetragonal\"\>", ",", "\<\"SimpleTriclinic\"\>", 
   ",", "\<\"SimpleTrigonal\"\>", ",", "\<\"SquareLattice\"\>", 
   ",", "\<\"TetrahedralPacking\"\>"}], "}"}]], "Output",
 ImageSize->{504, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->120781910]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1537603642],

Cell[TextData[{
 "A lattice can be specified by its standard ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " name:"
}], "ExampleText",
 CellID->1833686743],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"LatticeData", "[", "\"\<SimpleCubic\>\"", "]"}], "//", 
  "Normal"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->235271504],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->609514955]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1314012897],

Cell[TextData[{
 "A Barnes-Wall lattice ",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    StyleBox["BW", "TI"], "n"], TraditionalForm]], "InlineMath"],
 " is specified by ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"\"\<BarnesWall\>\"", ",", 
    StyleBox["n", "TI"]}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1905478627],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<BarnesWall\>\"", ",", "32"}], "}"}], ",", " ", 
   "\"\<Determinant\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1630743499],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->620852495]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1096715949],

Cell["\<\
Bravais lattices can be specified by a crystal system with a lattice \
centering:\
\>", "ExampleText",
 CellID->652812642],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<Bravais\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<Cubic\>\"", ",", "\"\<BodyCentered\>\""}], "}"}]}], "}"}], 
   ",", "\"\<Basis\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1238301],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "2", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->44842692]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->328121571],

Cell[TextData[{
 "The integer lattice ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalZ]", "2"], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->985829595],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"LatticeData", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"\"\<IntegerLattice\>\"", ",", " ", "2"}], "}"}], ",", 
    "\"\<Basis\>\""}], "]"}], "//", "Normal"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1001666560],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2051847994]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1862655391],

Cell[TextData[{
 "Laminated lattices ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalLambda]", "31"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1780034218],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"LatticeData", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"\"\<LaminatedLattice\>\"", ",", "31"}], "}"}], ",", 
    "\"\<Dimension\>\""}], "]"}], " "}]], "Input",
 CellLabel->"In[1]:=",
 CellID->327928736],

Cell[BoxData["31"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1116831632]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->112150006],

Cell[TextData[{
 "Mordell-Weil lattice ",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    StyleBox["MW", "TI"], "24"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1561992423],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<MordellWeil\>\"", ",", " ", "24"}], "}"}], ",", 
   "\"\<Determinant\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1566552433],

Cell[BoxData["16777216"], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1582164244]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->101180043],

Cell[TextData[{
 "The root lattices ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["A", "2"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1873417924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<RootLattice\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<A\>\"", ",", "2"}], "}"}]}], "}"}], ",", 
   "\"\<KissingNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->129816354],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1291931754]
}, Open  ]],

Cell[TextData[{
 "The root lattice ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["D", "4"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1906796243],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<RootLattice\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<D\>\"", ",", "4"}], "}"}]}], "}"}], ",", 
   "\"\<CoxeterNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->625991920],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->289834661]
}, Open  ]],

Cell[TextData[{
 "The root lattice ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["E", "6"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1186042056],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<RootLattice\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<E\>\"", ",", "6"}], "}"}]}], "}"}], ",", 
   "\"\<PackingRadius\>\""}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->865623738],

Cell[BoxData[
 FractionBox["1", 
  SqrtBox["2"]]], "Output",
 ImageSize->{32, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->758236031]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2072225708],

Cell["Short string forms can be used for root lattices:", "ExampleText",
 CellID->7687310],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<A2\>\"", ",", "\"\<KissingNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->625471725],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->505011719]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<D(4)\>\"", ",", "\"\<CoxeterNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1601242085],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->876588289]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->228297362],

Cell["Find the English name of a lattice:", "ExampleText",
 CellID->78167828],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<IntegerLattice\>\"", ",", "5"}], "}"}], ",", 
   "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1654176879],

Cell[BoxData["\<\"5-dimensional integer lattice\"\>"], "Output",
 ImageSize->{207, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1081554803]
}, Open  ]],

Cell["A list of alternate names can also be found:", "ExampleText",
 CellID->1340823276],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<CoxeterTodd\>\"", ",", "\"\<AlternateNames\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1670891761],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"K12\"\>", ",", "\<\"Coxeter-Todd lattice\"\>", 
   ",", "\<\"Coxeter-Todd\"\>"}], "}"}]], "Output",
 ImageSize->{286, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1389985813]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1006267807],

Cell["Find the list of classes:", "ExampleText",
 CellID->1315236016],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", "\"\<Classes\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1122542341],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Even\"\>", ",", "\<\"Odd\"\>", ",", "\<\"Extremal\"\>", 
   ",", "\<\"Nonextremal\"\>", ",", "\<\"Integral\"\>", 
   ",", "\<\"Nonintegral\"\>", ",", "\<\"Unimodular\"\>", 
   ",", "\<\"Nonunimodular\"\>"}], "}"}]], "Output",
 ImageSize->{573, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1164447025]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1357621717],

Cell["Find the list of named lattices belonging to a class:", "ExampleText",
 CellID->1695025154],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", "\"\<Integral\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1917185823],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"BaseCenteredMonoclinic\"\>", 
   ",", "\<\"BaseCenteredOrthorhombic\"\>", ",", "\<\"BodyCenteredCubic\"\>", 
   ",", "\<\"BodyCenteredOrthorhombic\"\>", ",", "\<\"CenteredTetragonal\"\>",
    ",", "\<\"CoxeterTodd\"\>", ",", "\<\"FaceCenteredCubic\"\>", 
   ",", "\<\"FaceCenteredOrthorhombic\"\>", ",", "\<\"HexagonalLattice\"\>", 
   ",", "\<\"KorkineZolotarev\"\>", ",", "\<\"Leech\"\>", 
   ",", "\<\"SimpleCubic\"\>", ",", "\<\"SimpleHexagonal\"\>", 
   ",", "\<\"SimpleMonoclinic\"\>", ",", "\<\"SimpleOrthorhombic\"\>", 
   ",", "\<\"SimpleTetragonal\"\>", ",", "\<\"SimpleTriclinic\"\>", 
   ",", "\<\"SimpleTrigonal\"\>", ",", "\<\"SquareLattice\"\>"}], 
  "}"}]], "Output",
 ImageSize->{542, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->842674064]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", "\"\<Even\>\"", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1064432362],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"BaseCenteredMonoclinic\"\>", 
   ",", "\<\"BaseCenteredOrthorhombic\"\>", ",", "\<\"BodyCenteredCubic\"\>", 
   ",", "\<\"BodyCenteredOrthorhombic\"\>", ",", "\<\"CenteredTetragonal\"\>",
    ",", "\<\"FaceCenteredCubic\"\>", ",", "\<\"FaceCenteredOrthorhombic\"\>",
    ",", "\<\"KorkineZolotarev\"\>", ",", "\<\"Leech\"\>", 
   ",", "\<\"SimpleHexagonal\"\>", ",", "\<\"SimpleMonoclinic\"\>", 
   ",", "\<\"SimpleOrthorhombic\"\>", ",", "\<\"SimpleTetragonal\"\>", 
   ",", "\<\"SimpleTriclinic\"\>", ",", "\<\"SimpleTrigonal\"\>"}], 
  "}"}]], "Output",
 ImageSize->{454, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->168341683]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1324687773],

Cell["Test whether a lattice belongs to a class:", "ExampleText",
 CellID->1128472672],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<Leech\>\"", ",", "\"\<Odd\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1812111647],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1082703082]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1964625524],

Cell["Get a list of classes in which a lattice belongs:", "ExampleText",
 CellID->1900109243],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<BodyCenteredCubic\>\"", ",", "\"\<Classes\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2054216211],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Even\"\>", ",", "\<\"Extremal\"\>", 
   ",", "\<\"Nonintegral\"\>"}], "}"}]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1361758143]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1638048307],

Cell["A list of lattices which are even and integral:", "ExampleText",
 CellID->2085024856],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"{", 
   RowBox[{"\"\<Even\>\"", ",", " ", "\"\<Integral\>\""}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->374603046],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"BaseCenteredMonoclinic\"\>", 
   ",", "\<\"BaseCenteredOrthorhombic\"\>", ",", "\<\"BodyCenteredCubic\"\>", 
   ",", "\<\"BodyCenteredOrthorhombic\"\>", ",", "\<\"CenteredTetragonal\"\>",
    ",", "\<\"FaceCenteredCubic\"\>", ",", "\<\"FaceCenteredOrthorhombic\"\>",
    ",", "\<\"KorkineZolotarev\"\>", ",", "\<\"Leech\"\>", 
   ",", "\<\"SimpleHexagonal\"\>", ",", "\<\"SimpleMonoclinic\"\>", 
   ",", "\<\"SimpleOrthorhombic\"\>", ",", "\<\"SimpleTetragonal\"\>", 
   ",", "\<\"SimpleTriclinic\"\>", ",", "\<\"SimpleTrigonal\"\>"}], 
  "}"}]], "Output",
 ImageSize->{454, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1335883322]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->176557506],

Cell["Get a list of possible properties:", "ExampleText",
 CellID->575677773],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", "\"\<Properties\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1002501076],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AlternateNames\"\>", ",", "\<\"AutomorphismGroupOrder\"\>", 
   ",", "\<\"Basis\"\>", ",", "\<\"CenterDensity\"\>", 
   ",", "\<\"CoveringRadius\"\>", ",", "\<\"CoxeterNumber\"\>", 
   ",", "\<\"Density\"\>", ",", "\<\"Determinant\"\>", 
   ",", "\<\"Dimension\"\>", ",", "\<\"Dual\"\>", ",", "\<\"Even\"\>", 
   ",", "\<\"Extremal\"\>", ",", "\<\"GeneratorMatrix\"\>", 
   ",", "\<\"Genus\"\>", ",", "\<\"GlueVectors\"\>", 
   ",", "\<\"GramMatrix\"\>", ",", "\<\"HermiteInvariant\"\>", 
   ",", "\<\"Image\"\>", ",", "\<\"Integral\"\>", 
   ",", "\<\"KissingNumber\"\>", ",", "\<\"MinimalNorm\"\>", 
   ",", "\<\"MinimalVectors\"\>", ",", "\<\"ModularNumber\"\>", 
   ",", "\<\"Name\"\>", ",", "\<\"Nonextremal\"\>", 
   ",", "\<\"Nonintegral\"\>", ",", "\<\"Nonunimodular\"\>", 
   ",", "\<\"Notation\"\>", ",", "\<\"Odd\"\>", ",", "\<\"PackingRadius\"\>", 
   ",", "\<\"QuadraticForm\"\>", ",", "\<\"RadialFunction\"\>", 
   ",", "\<\"StandardName\"\>", ",", "\<\"ThetaSeriesFunction\"\>", 
   ",", "\<\"Thickness\"\>", ",", "\<\"Unimodular\"\>", 
   ",", "\<\"Volume\"\>"}], "}"}]], "Output",
 ImageSize->{576, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->249463469]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1142133792],

Cell[BoxData["37"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->844444539]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->318223146],

Cell["Get a list of available properties for a particular lattice:", \
"ExampleText",
 CellID->1639060664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<Leech\>\"", ",", " ", "\"\<Properties\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1919839779],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"AlternateNames\"\>", ",", "\<\"AutomorphismGroupOrder\"\>", 
   ",", "\<\"Basis\"\>", ",", "\<\"CenterDensity\"\>", 
   ",", "\<\"CoveringRadius\"\>", ",", "\<\"CoxeterNumber\"\>", 
   ",", "\<\"Density\"\>", ",", "\<\"Determinant\"\>", 
   ",", "\<\"Dimension\"\>", ",", "\<\"Dual\"\>", ",", "\<\"Even\"\>", 
   ",", "\<\"Extremal\"\>", ",", "\<\"GeneratorMatrix\"\>", 
   ",", "\<\"GramMatrix\"\>", ",", "\<\"HermiteInvariant\"\>", 
   ",", "\<\"Integral\"\>", ",", "\<\"KissingNumber\"\>", 
   ",", "\<\"MinimalNorm\"\>", ",", "\<\"ModularNumber\"\>", 
   ",", "\<\"Name\"\>", ",", "\<\"Nonextremal\"\>", 
   ",", "\<\"Nonintegral\"\>", ",", "\<\"Nonunimodular\"\>", 
   ",", "\<\"Notation\"\>", ",", "\<\"Odd\"\>", ",", "\<\"PackingRadius\"\>", 
   ",", "\<\"QuadraticForm\"\>", ",", "\<\"RadialFunction\"\>", 
   ",", "\<\"StandardName\"\>", ",", "\<\"ThetaSeriesFunction\"\>", 
   ",", "\<\"Thickness\"\>", ",", "\<\"Unimodular\"\>", 
   ",", "\<\"Volume\"\>"}], "}"}]], "Output",
 ImageSize->{522, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2007427597]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->255389705],

Cell["Certain 3-dimensional lattices have images:", "ExampleText",
 CellID->1662445229],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<BodyCenteredCubic\>\"", ",", " ", "\"\<Image\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1954716333],

Cell[BoxData[
 Graphics3DBox[{
   GraphicsComplex3DBox[{{-1, -1, -1}, {-1, -1, 1}, {-1, 1, -1}, {-1, 1, 1}, {
    1, -1, -1}, {1, -1, 1}, {1, 1, -1}, {1, 1, 1}}, 
    {Opacity[0.1], EdgeForm[GrayLevel[0.8]], 
     Polygon3DBox[{{8, 4, 2, 6}, {8, 6, 5, 7}, {8, 7, 3, 4}, {4, 3, 1, 2}, {1,
       3, 7, 5}, {2, 1, 5, 6}}]}], 
   {GrayLevel[0], Specularity[
     GrayLevel[1], 5], SphereBox[{-1, -1, -1}, 0.06], 
    SphereBox[{-1, -1, 1}, 0.06], SphereBox[{-1, 1, -1}, 0.06], 
    SphereBox[{-1, 1, 1}, 0.06], SphereBox[{1, -1, -1}, 0.06], 
    SphereBox[{1, -1, 1}, 0.06], SphereBox[{1, 1, -1}, 0.06], 
    SphereBox[{1, 1, 1}, 0.06], SphereBox[{0, 0, 0}, 0.06]}, 
   {GrayLevel[0.8], Line3DBox[{{0, 0, 0}, {-1, -1, -1}}], 
    Line3DBox[{{0, 0, 0}, {-1, -1, 1}}], Line3DBox[{{0, 0, 0}, {-1, 1, -1}}], 
    Line3DBox[{{0, 0, 0}, {-1, 1, 1}}], Line3DBox[{{0, 0, 0}, {1, -1, -1}}], 
    Line3DBox[{{0, 0, 0}, {1, -1, 1}}], Line3DBox[{{0, 0, 0}, {1, 1, -1}}], 
    Line3DBox[{{0, 0, 0}, {1, 1, 1}}]}},
  Boxed->False,
  ViewPoint->NCache[{4, 
     Rational[5, 3], 1}, {4, 1.6666666666666667`, 1}]]], "Output",
 ImageSize->{184, 172},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1867528409,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->938498346],

Cell["Get notation forms:", "ExampleText",
 CellID->74314007],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<IntegerLattice\>\"", ",", " ", "5"}], "}"}], ",", 
   "\"\<Notation\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1307516742],

Cell[GraphicsData["CompressedBitmap", "\<\
eJxlT9EKgzAMTBvBKpS9+ujX6JMfUYqgD2Oj+v81SbeoWyCX3PWatFPYl/kZ
9jWGfkzhvaxx64dXIgkNAHjKRwfUZ+okfc5UrwhYCgdS1kItY8U3JFSHqpSC
jXqtMvwwd5vO+K9+dwrFc5uRY3dR2SlvgpaQP8ee02d1qPuZZVVFnQHmAKpn
f8Q=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{13, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12510632]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Property Values",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1727086835],

Cell[TextData[{
 "A property value can be any valid ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expression:"
}], "ExampleText",
 CellID->435593338],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<FaceCenteredCubic\>\"", ",", "\"\<Basis\>\""}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->245479143],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", 
     RowBox[{"-", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->340675425]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<CoxeterTodd\>\"", ",", "\"\<Notation\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1085553006],

Cell[BoxData[
 SubscriptBox["\<\"K\"\>", "12"]], "Output",
 ImageSize->{22, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->540465073]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<Leech\>\"", ",", "\"\<Integral\>\""}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->884327185],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1888138748]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1953721738],

Cell[TextData[{
 Cell[BoxData["\"\<ThetaSeriesFunction\>\""], "InlineFormula"],
 " are given as pure functions:"
}], "ExampleText",
 CellID->1176287936],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<CoxeterTodd\>\"", ",", "\"\<ThetaSeriesFunction\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->136886070],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{435, 138},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->425654873]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->852451040],

Cell[TextData[{
 "A property that is not applicable to a lattice has the value ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Missing",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Missing"], "[", "\"\<NotApplicable\>\"", "]"}]], 
  "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1281220143],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<IntegerLattice\>\"", ",", "5"}], "}"}], ",", 
   "\"\<Image\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->648414932],

Cell[BoxData[
 RowBox[{"Missing", "[", "\<\"NotApplicable\"\>", "]"}]], "Output",
 ImageSize->{158, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1329994934]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->160146669],

Cell[TextData[{
 "A property that is not available for a lattice has the value ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Missing",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Missing"], "[", "\"\<NotAvailable\>\"", "]"}]], 
  "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->187464565],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<KorkineZolotarev\>\"", ",", "\"\<ThetaSeriesFunction\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10131782],

Cell[BoxData[
 RowBox[{"Missing", "[", "\<\"NotAvailable\"\>", "]"}]], "Output",
 ImageSize->{151, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->67574115]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1304052878],

Cell["Specify a list of properties for a lattice: ", "ExampleText",
 CellID->451374594],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<CoxeterTodd\>\"", ",", 
   RowBox[{"{", 
    RowBox[{"\"\<Dimension\>\"", ",", "\"\<Determinant\>\""}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1340560154],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"12", ",", "729"}], "}"}]], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1846785189]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LatticeData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LatticeData"]], "InlineFormula"],
 " works on nonlattice structures such as tetrahedral packings:"
}], "ExampleText",
 CellID->1044754157],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{"\"\<TetrahedralPacking\>\"", ",", "\"\<Image\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1493169506],

Cell[BoxData[
 Graphics3DBox[{
   GraphicsComplex3DBox[
    NCache[{{-1, -1, -1}, {-1, -1, 1}, {-1, 1, -1}, {-1, 1, 1}, {1, -1, -1}, {
      1, -1, 1}, {1, 1, -1}, {1, 1, 1}, {0, 0, 1}, {1, 0, 0}, {0, 1, 0}, {-1, 
      0, 0}, {0, 0, -1}, {0, -1, 0}, {
      Rational[1, 2], Rational[1, 2], Rational[1, 2]}, {
      Rational[-1, 2], Rational[-1, 2], Rational[1, 2]}, {
      Rational[1, 2], Rational[-1, 2], Rational[-1, 2]}, {
      Rational[-1, 2], Rational[1, 2], 
       Rational[-1, 2]}}, {{-1, -1, -1}, {-1, -1, 1}, {-1, 1, -1}, {-1, 1, 
      1}, {1, -1, -1}, {1, -1, 1}, {1, 1, -1}, {1, 1, 1}, {0, 0, 1}, {1, 0, 
      0}, {0, 1, 0}, {-1, 0, 0}, {0, 0, -1}, {0, -1, 0}, {0.5, 0.5, 
      0.5}, {-0.5, -0.5, 0.5}, {0.5, -0.5, -0.5}, {-0.5, 0.5, -0.5}}], {
     {Opacity[0.1], EdgeForm[GrayLevel[0.8]], 
      Polygon3DBox[{{8, 4, 2, 6}, {8, 6, 5, 7}, {8, 7, 3, 4}, {4, 3, 1, 2}, {
       1, 3, 7, 5}, {2, 1, 5, 6}}]}, 
     {GrayLevel[0.8], 
      Line3DBox[{{15, 8}, {15, 9}, {15, 11}, {15, 10}, {16, 9}, {16, 2}, {16, 
       12}, {16, 14}, {17, 10}, {17, 14}, {17, 13}, {17, 5}, {18, 11}, {18, 
       12}, {18, 13}, {18, 3}}]}}], 
   {GrayLevel[0], Specularity[
     GrayLevel[1], 5], SphereBox[{-1, -1, -1}, 0.06], 
    SphereBox[{-1, -1, 1}, 0.06], SphereBox[{-1, 1, -1}, 0.06], 
    SphereBox[{-1, 1, 1}, 0.06], SphereBox[{1, -1, -1}, 0.06], 
    SphereBox[{1, -1, 1}, 0.06], SphereBox[{1, 1, -1}, 0.06], 
    SphereBox[{1, 1, 1}, 0.06], SphereBox[{0, 0, 1}, 0.06], 
    SphereBox[{1, 0, 0}, 0.06], SphereBox[{0, 1, 0}, 0.06], 
    SphereBox[{-1, 0, 0}, 0.06], SphereBox[{0, 0, -1}, 0.06], 
    SphereBox[{0, -1, 0}, 0.06], 
    SphereBox[
     NCache[{Rational[1, 2], Rational[1, 2], Rational[1, 2]}, {0.5, 0.5, 
      0.5}], 0.06], 
    SphereBox[
     NCache[{Rational[-1, 2], Rational[-1, 2], Rational[1, 2]}, {-0.5, -0.5, 
      0.5}], 0.06], 
    SphereBox[
     NCache[{Rational[1, 2], Rational[-1, 2], Rational[-1, 2]}, {
      0.5, -0.5, -0.5}], 0.06], 
    SphereBox[
     NCache[{Rational[-1, 2], Rational[1, 2], Rational[-1, 2]}, {-0.5, 
      0.5, -0.5}], 0.06]}},
  Boxed->False,
  ViewPoint->NCache[{4, 
     Rational[5, 3], 1}, {4, 1.6666666666666667`, 1}]]], "Output",
 ImageSize->{184, 172},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1433327455,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
  "\"\<HexagonalClosePacking\>\"", ",", " ", "\"\<KissingNumber\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->396037781],

Cell[BoxData["12"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->38529636]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Number of lattice points in the unit sphere:", "ExampleText",
 CellID->1059341242],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"LatticeData", "[", 
   RowBox[{"\"\<FaceCenteredCubic\>\"", ",", "\"\<RadialFunction\>\""}], 
   "]"}], "[", "1", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1708820664],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->114137825]
}, Open  ]],

Cell["\<\
The number of lattice points on the sphere as a function of radius:\
\>", "ExampleText",
 CellID->1748383784],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"LatticeData", "[", 
     RowBox[{"\"\<FaceCenteredCubic\>\"", ",", "\"\<RadialFunction\>\""}], 
     "]"}], " ", "/@", " ", 
    RowBox[{"Range", "[", "30", "]"}]}], ",", 
   RowBox[{"Filling", "\[Rule]", " ", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1067515787],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJztk8EKgkAURYdWLl20cNHCJEQkIrXMNEtatXbRBwSt+/U+ocL7As8nRAPD
4cy8d2dEZn579PeJc+76nh8O43kaGHRaEBPRE0PR78b1U7HCuvnMct04x/Ij
8Sgu0B/Drc9yUuwvxYu4wvlr3DNDfS6exQL5G/gW/SX2dzi/wv5etO+vxV5s
xE48IK9FveXYfez/fgf6PbgPn8IDeAiP4At4DE/gKXwJX8MzeA4v4Bv4Fl7C
K/geXsMbeAs/wv/vb1z/O+/vBXU5QCs=
   "], {{{}, {}, {}, {}, 
     {Hue[0.67, 0.6, 0.6], Opacity[0.2], LineBox[{31, 1}], LineBox[{32, 2}], 
      LineBox[{33, 3}], LineBox[{34, 4}], LineBox[{35, 5}], LineBox[{36, 6}], 
      LineBox[{37, 8}], LineBox[{38, 9}], LineBox[{39, 10}], 
      LineBox[{40, 11}], LineBox[{41, 12}], LineBox[{42, 13}], 
      LineBox[{43, 14}], LineBox[{44, 16}], LineBox[{45, 17}], 
      LineBox[{46, 18}], LineBox[{47, 19}], LineBox[{48, 20}], 
      LineBox[{49, 21}], LineBox[{50, 22}], LineBox[{51, 24}], 
      LineBox[{52, 25}], LineBox[{53, 26}], LineBox[{54, 27}], 
      LineBox[{55, 29}], LineBox[{56, 30}]}}, {{}, 
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 
       72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86}]}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  PlotRange->{{0., 30.}, {0., 72.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->611528004]
}, Open  ]],

Cell["\<\
The number of lattice points in the ball as a function of radius: \
\>", "ExampleText",
 CellID->26445307],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Accumulate", "[", 
    RowBox[{
     RowBox[{"LatticeData", "[", 
      RowBox[{"\"\<FaceCenteredCubic\>\"", ",", "\"\<RadialFunction\>\""}], 
      "]"}], " ", "/@", " ", 
     RowBox[{"Range", "[", "30", "]"}]}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->45751088],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJzt088KAUEAx/HNyXEPe3BwQJIk+bv+Mw/gICnl4KAoJ0qkSCK5eiSP5BHQ
fCm/R5Cp7dtndpranSY6nvcmAcdxhs/nVTvubduQYYLmaZDWqEsN9WjHfO/T
p2FxhA5ozPYwwnFbM8YJPMFJPMUpPMNpPMcZcZYuaI4uKd9tVrhge1vjIu83
uIS32Gf9DpfFFdbvcdXWPWD+c/eI6/iEG+Km7fWMW+x3we/z/QzzzaDYFXvi
kDgsjohj4rg4IU6KU+K0OCPOinPivLggLopLYl9cFlfEVXFNXBc3xE1xS/y/
v7a/d38fChte4A==
   "], {{{}, {}, {}, {}, 
     {Hue[0.67, 0.6, 0.6], Opacity[0.2], LineBox[{31, 1}], LineBox[{32, 2}], 
      LineBox[{33, 3}], LineBox[{34, 4}], LineBox[{35, 5}], LineBox[{36, 6}], 
      LineBox[{37, 7}], LineBox[{38, 8}], LineBox[{39, 9}], LineBox[{40, 10}],
       LineBox[{41, 11}], LineBox[{42, 12}], LineBox[{43, 13}], 
      LineBox[{44, 14}], LineBox[{45, 15}], LineBox[{46, 16}], 
      LineBox[{47, 17}], LineBox[{48, 18}], LineBox[{49, 19}], 
      LineBox[{50, 20}], LineBox[{51, 21}], LineBox[{52, 22}], 
      LineBox[{53, 23}], LineBox[{54, 24}], LineBox[{55, 25}], 
      LineBox[{56, 26}], LineBox[{57, 27}], LineBox[{58, 28}], 
      LineBox[{59, 29}], LineBox[{60, 30}]}}, {{}, 
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 
       76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90}]}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 30.}, {0., 738.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->550503552]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->780948601],

Cell["Given a 2D lattice basis: ", "ExampleText",
 CellID->564854877],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", 
  RowBox[{"Normal", "@", 
   RowBox[{"LatticeData", "[", 
    RowBox[{"\"\<A2\>\"", ",", "\"\<Basis\>\""}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->128961002],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"-", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12094353]
}, Open  ]],

Cell["Generate and display some of the elements in the lattice: ", \
"ExampleText",
 CellID->599206258],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"l1", "=", 
  RowBox[{"Flatten", "[", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"i", " ", 
        RowBox[{"b", "[", 
         RowBox[{"[", "1", "]"}], "]"}]}], "+", 
       RowBox[{"j", " ", 
        RowBox[{"b", "[", 
         RowBox[{"[", "2", "]"}], "]"}]}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "0", ",", "3"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"j", ",", "0", ",", "3"}], "}"}]}], "]"}], ",", "1"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->68126967],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWdtu2kAQXXAIBBogSGnoG/2Z9qkfgaJK4SFKRPnb9l+SXRtX6HTGc7GN
GydImB175nh2dnZ29/Bje3j4+bg97O63m+/77fPD7v7X5tvTPt7KBiEMfocQ
vq5DbL/EVv5Nnz/xWzSL1ixds/LpS1iCvAD5AuS27VFG+6mgf2e0v/nP+t92
/IbpQkZhGEWMRna8j2/16q8Yfc6fL0r91Zn8r9tfHI0BORpX8TKKImpfMvdR
vuoIbxIvSRwJeBMl3lzpH+LNGD/GSv+wv1PGzhu/c+Fpx4PPwjD6K+c6hW8p
s8fsU3wTNV+rsbF/zaBOTB5TOc9jY13Qok47i3Gfx+/f3EaPUt1OUBncx9X3
EuS5EgdXYZyBWpwF2WdEuzagUfVFizcWooXv43qJUf3kjDrqod/aqJP+iFFP
UYwb7zzD6AqKuwJpb4dZ07T9nOwXouBbEEXaYaM9Rh2fD4Ve4KiiLEWhaXs6
O3R1CDMEM1+aUegb2mOspb6h/bWqb+glonwWvKDsU4VfHn9xplF5ze3GNXbS
GYmzo6uxb+Qxq7qeFbqsRi8R5Vbwou3a8DaiWGf/rZuh6CeNZV1PaBTfqkJj
WdcWev/cZIzearzpHMO1PL0hO3qWESjIX5zqU5Wa018y+FjBS/07Bh/1y5Ov
x/8u+kuzcamJPK29ZWUNpGwk69GJbGVau7a3Mr1t29f1X7Ivcgv3RNRbqphN
9KLU1zKzbetLzG9T+lZ/OP1MNSrpeUbUGy2DiYwoh+dlWM+Ft2DuS3hTxg8t
w6rF844Hh+eNX128YncgZWVeURpiFLGaV2N7+dVqVP157tzYeKK2MqtavHpj
xqF6R4vDs8WyWPfw5EdVWO1uX8OP0lgjwroeGlXR/GjI6GIF4Jhgqk5b+0jh
+OPPoXlHoApPGoM8qs5q6mPgaCzrabpNFNwP4fyUOEVpty3ZN8NsWmNhtc/X
9d5kzhq0pH/beF+os/f7yBgphv3KGGuV+Kg1vL0UhX5lzm2DHvmw+plFulgU
uaRhO5PIsa94bij1tWzwqb4VX6NfclPc/5RV+hb8tRF/zeDn56tc0PDHYfAK
YasOHg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{475, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->994329606]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"l2", "=", 
  RowBox[{
   RowBox[{"Tuples", "[", 
    RowBox[{
     RowBox[{"Range", "[", 
      RowBox[{"0", ",", "3"}], "]"}], ",", "2"}], "]"}], ".", 
   "b"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->80914975],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWdtu2kAQXXAIBBogSGnoG/2Z9qkfgaJK4SFKRPnb9l+SXRtX6HTGc7GN
GydImB175nh2dnZ29/Bje3j4+bg97O63m+/77fPD7v7X5tvTPt7KBiEMfocQ
vq5DbL/EVv5Nnz/xWzSL1ixds/LpS1iCvAD5AuS27VFG+6mgf2e0v/nP+t92
/IbpQkZhGEWMRna8j2/16q8Yfc6fL0r91Zn8r9tfHI0BORpX8TKKImpfMvdR
vuoIbxIvSRwJeBMl3lzpH+LNGD/GSv+wv1PGzhu/c+Fpx4PPwjD6K+c6hW8p
s8fsU3wTNV+rsbF/zaBOTB5TOc9jY13Qok47i3Gfx+/f3EaPUt1OUBncx9X3
EuS5EgdXYZyBWpwF2WdEuzagUfVFizcWooXv43qJUf3kjDrqod/aqJP+iFFP
UYwb7zzD6AqKuwJpb4dZ07T9nOwXouBbEEXaYaM9Rh2fD4Ve4KiiLEWhaXs6
O3R1CDMEM1+aUegb2mOspb6h/bWqb+glonwWvKDsU4VfHn9xplF5ze3GNXbS
GYmzo6uxb+Qxq7qeFbqsRi8R5Vbwou3a8DaiWGf/rZuh6CeNZV1PaBTfqkJj
WdcWev/cZIzearzpHMO1PL0hO3qWESjIX5zqU5Wa018y+FjBS/07Bh/1y5Ov
x/8u+kuzcamJPK29ZWUNpGwk69GJbGVau7a3Mr1t29f1X7Ivcgv3RNRbqphN
9KLU1zKzbetLzG9T+lZ/OP1MNSrpeUbUGy2DiYwoh+dlWM+Ft2DuS3hTxg8t
w6rF844Hh+eNX128YncgZWVeURpiFLGaV2N7+dVqVP157tzYeKK2MqtavHpj
xqF6R4vDs8WyWPfw5EdVWO1uX8OP0lgjwroeGlXR/GjI6GIF4Jhgqk5b+0jh
+OPPoXlHoApPGoM8qs5q6mPgaCzrabpNFNwP4fyUOEVpty3ZN8NsWmNhtc/X
9d5kzhq0pH/beF+os/f7yBgphv3KGGuV+Kg1vL0UhX5lzm2DHvmw+plFulgU
uaRhO5PIsa94bij1tWzwqb4VX6NfclPc/5RV+hb8tRF/zeDn56tc0PDHYfAK
YasOHg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{475, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->216715561]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Graphics", "[", 
    RowBox[{
     RowBox[{"Point", "[", "l1", "]"}], ",", 
     RowBox[{"Frame", "\[Rule]", "True"}], ",", 
     RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}], ",", 
   RowBox[{"Graphics", "[", 
    RowBox[{
     RowBox[{"Point", "[", "l2", "]"}], ",", 
     RowBox[{"Frame", "\[Rule]", "True"}], ",", 
     RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->22467214],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   GraphicsBox[
    PointBox[{{0, 0}, {0, -1}, {0, -2}, {0, -3}, {-1, 1}, {-1, 
     0}, {-1, -1}, {-1, -2}, {-2, 2}, {-2, 1}, {-2, 0}, {-2, -1}, {-3, 
     3}, {-3, 2}, {-3, 1}, {-3, 0}}],
    AspectRatio->1,
    Frame->True], ",", 
   GraphicsBox[
    PointBox[{{0, 0}, {0, -1}, {0, -2}, {0, -3}, {-1, 1}, {-1, 
     0}, {-1, -1}, {-1, -2}, {-2, 2}, {-2, 1}, {-2, 0}, {-2, -1}, {-3, 
     3}, {-3, 2}, {-3, 1}, {-3, 0}}],
    AspectRatio->1,
    Frame->True]}], "}"}]], "Output",
 ImageSize->{390, 182},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->741398235]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->82786105],

Cell["Given a 3D lattice basis: ", "ExampleText",
 CellID->34351434],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", 
  RowBox[{"Normal", "@", 
   RowBox[{"LatticeData", "[", 
    RowBox[{"\"\<FaceCenteredCubic\>\"", ",", "\"\<Basis\>\""}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->76540591],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", 
     RowBox[{"-", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->350796424]
}, Open  ]],

Cell["Generate and display some of the elements in the lattice: ", \
"ExampleText",
 CellID->86968521],

Cell[BoxData[
 RowBox[{
  RowBox[{"l1", "=", 
   RowBox[{"Flatten", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"i", " ", 
         RowBox[{"b", "[", 
          RowBox[{"[", "1", "]"}], "]"}]}], "+", 
        RowBox[{"j", " ", 
         RowBox[{"b", "[", 
          RowBox[{"[", "2", "]"}], "]"}]}], "+", 
        RowBox[{"k", " ", 
         RowBox[{"b", "[", 
          RowBox[{"[", "3", "]"}], "]"}]}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "0", ",", "5"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "0", ",", "5"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k", ",", "0", ",", "5"}], "}"}]}], "]"}], ",", "2"}], 
    "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->37651330],

Cell[BoxData[
 RowBox[{
  RowBox[{"l2", "=", 
   RowBox[{
    RowBox[{"Tuples", "[", 
     RowBox[{
      RowBox[{"Range", "[", 
       RowBox[{"0", ",", "5"}], "]"}], ",", "3"}], "]"}], ".", "b"}]}], 
  ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->651228490],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Graphics3D", "[", 
    RowBox[{
     RowBox[{"Map", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Sphere", "[", 
         RowBox[{"#", ",", ".2"}], "]"}], "&"}], ",", "l1"}], "]"}], ",", 
     RowBox[{"Boxed", "\[Rule]", "True"}], ",", 
     RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}], ",", 
   RowBox[{"Graphics3D", "[", 
    RowBox[{
     RowBox[{"Map", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Sphere", "[", 
         RowBox[{"#", ",", ".2"}], "]"}], "&"}], ",", "l2"}], "]"}], ",", 
     RowBox[{"Boxed", "\[Rule]", "True"}], ",", 
     RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->76730316],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   Graphics3DBox[{SphereBox[{0, 0, 0}, 0.2], SphereBox[{0, 1, -1}, 0.2], 
     SphereBox[{0, 2, -2}, 0.2], SphereBox[{0, 3, -3}, 0.2], 
     SphereBox[{0, 4, -4}, 0.2], SphereBox[{0, 5, -5}, 0.2], 
     SphereBox[{1, -1, 0}, 0.2], SphereBox[{1, 0, -1}, 0.2], 
     SphereBox[{1, 1, -2}, 0.2], SphereBox[{1, 2, -3}, 0.2], 
     SphereBox[{1, 3, -4}, 0.2], SphereBox[{1, 4, -5}, 0.2], 
     SphereBox[{2, -2, 0}, 0.2], SphereBox[{2, -1, -1}, 0.2], 
     SphereBox[{2, 0, -2}, 0.2], SphereBox[{2, 1, -3}, 0.2], 
     SphereBox[{2, 2, -4}, 0.2], SphereBox[{2, 3, -5}, 0.2], 
     SphereBox[{3, -3, 0}, 0.2], SphereBox[{3, -2, -1}, 0.2], 
     SphereBox[{3, -1, -2}, 0.2], SphereBox[{3, 0, -3}, 0.2], 
     SphereBox[{3, 1, -4}, 0.2], SphereBox[{3, 2, -5}, 0.2], 
     SphereBox[{4, -4, 0}, 0.2], SphereBox[{4, -3, -1}, 0.2], 
     SphereBox[{4, -2, -2}, 0.2], SphereBox[{4, -1, -3}, 0.2], 
     SphereBox[{4, 0, -4}, 0.2], SphereBox[{4, 1, -5}, 0.2], 
     SphereBox[{5, -5, 0}, 0.2], SphereBox[{5, -4, -1}, 0.2], 
     SphereBox[{5, -3, -2}, 0.2], SphereBox[{5, -2, -3}, 0.2], 
     SphereBox[{5, -1, -4}, 0.2], SphereBox[{5, 0, -5}, 0.2], 
     SphereBox[{-1, -1, 0}, 0.2], SphereBox[{-1, 0, -1}, 0.2], 
     SphereBox[{-1, 1, -2}, 0.2], SphereBox[{-1, 2, -3}, 0.2], 
     SphereBox[{-1, 3, -4}, 0.2], SphereBox[{-1, 4, -5}, 0.2], 
     SphereBox[{0, -2, 0}, 0.2], SphereBox[{0, -1, -1}, 0.2], 
     SphereBox[{0, 0, -2}, 0.2], SphereBox[{0, 1, -3}, 0.2], 
     SphereBox[{0, 2, -4}, 0.2], SphereBox[{0, 3, -5}, 0.2], 
     SphereBox[{1, -3, 0}, 0.2], SphereBox[{1, -2, -1}, 0.2], 
     SphereBox[{1, -1, -2}, 0.2], SphereBox[{1, 0, -3}, 0.2], 
     SphereBox[{1, 1, -4}, 0.2], SphereBox[{1, 2, -5}, 0.2], 
     SphereBox[{2, -4, 0}, 0.2], SphereBox[{2, -3, -1}, 0.2], 
     SphereBox[{2, -2, -2}, 0.2], SphereBox[{2, -1, -3}, 0.2], 
     SphereBox[{2, 0, -4}, 0.2], SphereBox[{2, 1, -5}, 0.2], 
     SphereBox[{3, -5, 0}, 0.2], SphereBox[{3, -4, -1}, 0.2], 
     SphereBox[{3, -3, -2}, 0.2], SphereBox[{3, -2, -3}, 0.2], 
     SphereBox[{3, -1, -4}, 0.2], SphereBox[{3, 0, -5}, 0.2], 
     SphereBox[{4, -6, 0}, 0.2], SphereBox[{4, -5, -1}, 0.2], 
     SphereBox[{4, -4, -2}, 0.2], SphereBox[{4, -3, -3}, 0.2], 
     SphereBox[{4, -2, -4}, 0.2], SphereBox[{4, -1, -5}, 0.2], 
     SphereBox[{-2, -2, 0}, 0.2], SphereBox[{-2, -1, -1}, 0.2], 
     SphereBox[{-2, 0, -2}, 0.2], SphereBox[{-2, 1, -3}, 0.2], 
     SphereBox[{-2, 2, -4}, 0.2], SphereBox[{-2, 3, -5}, 0.2], 
     SphereBox[{-1, -3, 0}, 0.2], SphereBox[{-1, -2, -1}, 0.2], 
     SphereBox[{-1, -1, -2}, 0.2], SphereBox[{-1, 0, -3}, 0.2], 
     SphereBox[{-1, 1, -4}, 0.2], SphereBox[{-1, 2, -5}, 0.2], 
     SphereBox[{0, -4, 0}, 0.2], SphereBox[{0, -3, -1}, 0.2], 
     SphereBox[{0, -2, -2}, 0.2], SphereBox[{0, -1, -3}, 0.2], 
     SphereBox[{0, 0, -4}, 0.2], SphereBox[{0, 1, -5}, 0.2], 
     SphereBox[{1, -5, 0}, 0.2], SphereBox[{1, -4, -1}, 0.2], 
     SphereBox[{1, -3, -2}, 0.2], SphereBox[{1, -2, -3}, 0.2], 
     SphereBox[{1, -1, -4}, 0.2], SphereBox[{1, 0, -5}, 0.2], 
     SphereBox[{2, -6, 0}, 0.2], SphereBox[{2, -5, -1}, 0.2], 
     SphereBox[{2, -4, -2}, 0.2], SphereBox[{2, -3, -3}, 0.2], 
     SphereBox[{2, -2, -4}, 0.2], SphereBox[{2, -1, -5}, 0.2], 
     SphereBox[{3, -7, 0}, 0.2], SphereBox[{3, -6, -1}, 0.2], 
     SphereBox[{3, -5, -2}, 0.2], SphereBox[{3, -4, -3}, 0.2], 
     SphereBox[{3, -3, -4}, 0.2], SphereBox[{3, -2, -5}, 0.2], 
     SphereBox[{-3, -3, 0}, 0.2], SphereBox[{-3, -2, -1}, 0.2], 
     SphereBox[{-3, -1, -2}, 0.2], SphereBox[{-3, 0, -3}, 0.2], 
     SphereBox[{-3, 1, -4}, 0.2], SphereBox[{-3, 2, -5}, 0.2], 
     SphereBox[{-2, -4, 0}, 0.2], SphereBox[{-2, -3, -1}, 0.2], 
     SphereBox[{-2, -2, -2}, 0.2], SphereBox[{-2, -1, -3}, 0.2], 
     SphereBox[{-2, 0, -4}, 0.2], SphereBox[{-2, 1, -5}, 0.2], 
     SphereBox[{-1, -5, 0}, 0.2], SphereBox[{-1, -4, -1}, 0.2], 
     SphereBox[{-1, -3, -2}, 0.2], SphereBox[{-1, -2, -3}, 0.2], 
     SphereBox[{-1, -1, -4}, 0.2], SphereBox[{-1, 0, -5}, 0.2], 
     SphereBox[{0, -6, 0}, 0.2], SphereBox[{0, -5, -1}, 0.2], 
     SphereBox[{0, -4, -2}, 0.2], SphereBox[{0, -3, -3}, 0.2], 
     SphereBox[{0, -2, -4}, 0.2], SphereBox[{0, -1, -5}, 0.2], 
     SphereBox[{1, -7, 0}, 0.2], SphereBox[{1, -6, -1}, 0.2], 
     SphereBox[{1, -5, -2}, 0.2], SphereBox[{1, -4, -3}, 0.2], 
     SphereBox[{1, -3, -4}, 0.2], SphereBox[{1, -2, -5}, 0.2], 
     SphereBox[{2, -8, 0}, 0.2], SphereBox[{2, -7, -1}, 0.2], 
     SphereBox[{2, -6, -2}, 0.2], SphereBox[{2, -5, -3}, 0.2], 
     SphereBox[{2, -4, -4}, 0.2], SphereBox[{2, -3, -5}, 0.2], 
     SphereBox[{-4, -4, 0}, 0.2], SphereBox[{-4, -3, -1}, 0.2], 
     SphereBox[{-4, -2, -2}, 0.2], SphereBox[{-4, -1, -3}, 0.2], 
     SphereBox[{-4, 0, -4}, 0.2], SphereBox[{-4, 1, -5}, 0.2], 
     SphereBox[{-3, -5, 0}, 0.2], SphereBox[{-3, -4, -1}, 0.2], 
     SphereBox[{-3, -3, -2}, 0.2], SphereBox[{-3, -2, -3}, 0.2], 
     SphereBox[{-3, -1, -4}, 0.2], SphereBox[{-3, 0, -5}, 0.2], 
     SphereBox[{-2, -6, 0}, 0.2], SphereBox[{-2, -5, -1}, 0.2], 
     SphereBox[{-2, -4, -2}, 0.2], SphereBox[{-2, -3, -3}, 0.2], 
     SphereBox[{-2, -2, -4}, 0.2], SphereBox[{-2, -1, -5}, 0.2], 
     SphereBox[{-1, -7, 0}, 0.2], SphereBox[{-1, -6, -1}, 0.2], 
     SphereBox[{-1, -5, -2}, 0.2], SphereBox[{-1, -4, -3}, 0.2], 
     SphereBox[{-1, -3, -4}, 0.2], SphereBox[{-1, -2, -5}, 0.2], 
     SphereBox[{0, -8, 0}, 0.2], SphereBox[{0, -7, -1}, 0.2], 
     SphereBox[{0, -6, -2}, 0.2], SphereBox[{0, -5, -3}, 0.2], 
     SphereBox[{0, -4, -4}, 0.2], SphereBox[{0, -3, -5}, 0.2], 
     SphereBox[{1, -9, 0}, 0.2], SphereBox[{1, -8, -1}, 0.2], 
     SphereBox[{1, -7, -2}, 0.2], SphereBox[{1, -6, -3}, 0.2], 
     SphereBox[{1, -5, -4}, 0.2], SphereBox[{1, -4, -5}, 0.2], 
     SphereBox[{-5, -5, 0}, 0.2], SphereBox[{-5, -4, -1}, 0.2], 
     SphereBox[{-5, -3, -2}, 0.2], SphereBox[{-5, -2, -3}, 0.2], 
     SphereBox[{-5, -1, -4}, 0.2], SphereBox[{-5, 0, -5}, 0.2], 
     SphereBox[{-4, -6, 0}, 0.2], SphereBox[{-4, -5, -1}, 0.2], 
     SphereBox[{-4, -4, -2}, 0.2], SphereBox[{-4, -3, -3}, 0.2], 
     SphereBox[{-4, -2, -4}, 0.2], SphereBox[{-4, -1, -5}, 0.2], 
     SphereBox[{-3, -7, 0}, 0.2], SphereBox[{-3, -6, -1}, 0.2], 
     SphereBox[{-3, -5, -2}, 0.2], SphereBox[{-3, -4, -3}, 0.2], 
     SphereBox[{-3, -3, -4}, 0.2], SphereBox[{-3, -2, -5}, 0.2], 
     SphereBox[{-2, -8, 0}, 0.2], SphereBox[{-2, -7, -1}, 0.2], 
     SphereBox[{-2, -6, -2}, 0.2], SphereBox[{-2, -5, -3}, 0.2], 
     SphereBox[{-2, -4, -4}, 0.2], SphereBox[{-2, -3, -5}, 0.2], 
     SphereBox[{-1, -9, 0}, 0.2], SphereBox[{-1, -8, -1}, 0.2], 
     SphereBox[{-1, -7, -2}, 0.2], SphereBox[{-1, -6, -3}, 0.2], 
     SphereBox[{-1, -5, -4}, 0.2], SphereBox[{-1, -4, -5}, 0.2], 
     SphereBox[{0, -10, 0}, 0.2], SphereBox[{0, -9, -1}, 0.2], 
     SphereBox[{0, -8, -2}, 0.2], SphereBox[{0, -7, -3}, 0.2], 
     SphereBox[{0, -6, -4}, 0.2], SphereBox[{0, -5, -5}, 0.2]},
    Axes->True,
    Boxed->True,
    ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]], ",", 
   Graphics3DBox[{SphereBox[{0, 0, 0}, 0.2], SphereBox[{0, 1, -1}, 0.2], 
     SphereBox[{0, 2, -2}, 0.2], SphereBox[{0, 3, -3}, 0.2], 
     SphereBox[{0, 4, -4}, 0.2], SphereBox[{0, 5, -5}, 0.2], 
     SphereBox[{1, -1, 0}, 0.2], SphereBox[{1, 0, -1}, 0.2], 
     SphereBox[{1, 1, -2}, 0.2], SphereBox[{1, 2, -3}, 0.2], 
     SphereBox[{1, 3, -4}, 0.2], SphereBox[{1, 4, -5}, 0.2], 
     SphereBox[{2, -2, 0}, 0.2], SphereBox[{2, -1, -1}, 0.2], 
     SphereBox[{2, 0, -2}, 0.2], SphereBox[{2, 1, -3}, 0.2], 
     SphereBox[{2, 2, -4}, 0.2], SphereBox[{2, 3, -5}, 0.2], 
     SphereBox[{3, -3, 0}, 0.2], SphereBox[{3, -2, -1}, 0.2], 
     SphereBox[{3, -1, -2}, 0.2], SphereBox[{3, 0, -3}, 0.2], 
     SphereBox[{3, 1, -4}, 0.2], SphereBox[{3, 2, -5}, 0.2], 
     SphereBox[{4, -4, 0}, 0.2], SphereBox[{4, -3, -1}, 0.2], 
     SphereBox[{4, -2, -2}, 0.2], SphereBox[{4, -1, -3}, 0.2], 
     SphereBox[{4, 0, -4}, 0.2], SphereBox[{4, 1, -5}, 0.2], 
     SphereBox[{5, -5, 0}, 0.2], SphereBox[{5, -4, -1}, 0.2], 
     SphereBox[{5, -3, -2}, 0.2], SphereBox[{5, -2, -3}, 0.2], 
     SphereBox[{5, -1, -4}, 0.2], SphereBox[{5, 0, -5}, 0.2], 
     SphereBox[{-1, -1, 0}, 0.2], SphereBox[{-1, 0, -1}, 0.2], 
     SphereBox[{-1, 1, -2}, 0.2], SphereBox[{-1, 2, -3}, 0.2], 
     SphereBox[{-1, 3, -4}, 0.2], SphereBox[{-1, 4, -5}, 0.2], 
     SphereBox[{0, -2, 0}, 0.2], SphereBox[{0, -1, -1}, 0.2], 
     SphereBox[{0, 0, -2}, 0.2], SphereBox[{0, 1, -3}, 0.2], 
     SphereBox[{0, 2, -4}, 0.2], SphereBox[{0, 3, -5}, 0.2], 
     SphereBox[{1, -3, 0}, 0.2], SphereBox[{1, -2, -1}, 0.2], 
     SphereBox[{1, -1, -2}, 0.2], SphereBox[{1, 0, -3}, 0.2], 
     SphereBox[{1, 1, -4}, 0.2], SphereBox[{1, 2, -5}, 0.2], 
     SphereBox[{2, -4, 0}, 0.2], SphereBox[{2, -3, -1}, 0.2], 
     SphereBox[{2, -2, -2}, 0.2], SphereBox[{2, -1, -3}, 0.2], 
     SphereBox[{2, 0, -4}, 0.2], SphereBox[{2, 1, -5}, 0.2], 
     SphereBox[{3, -5, 0}, 0.2], SphereBox[{3, -4, -1}, 0.2], 
     SphereBox[{3, -3, -2}, 0.2], SphereBox[{3, -2, -3}, 0.2], 
     SphereBox[{3, -1, -4}, 0.2], SphereBox[{3, 0, -5}, 0.2], 
     SphereBox[{4, -6, 0}, 0.2], SphereBox[{4, -5, -1}, 0.2], 
     SphereBox[{4, -4, -2}, 0.2], SphereBox[{4, -3, -3}, 0.2], 
     SphereBox[{4, -2, -4}, 0.2], SphereBox[{4, -1, -5}, 0.2], 
     SphereBox[{-2, -2, 0}, 0.2], SphereBox[{-2, -1, -1}, 0.2], 
     SphereBox[{-2, 0, -2}, 0.2], SphereBox[{-2, 1, -3}, 0.2], 
     SphereBox[{-2, 2, -4}, 0.2], SphereBox[{-2, 3, -5}, 0.2], 
     SphereBox[{-1, -3, 0}, 0.2], SphereBox[{-1, -2, -1}, 0.2], 
     SphereBox[{-1, -1, -2}, 0.2], SphereBox[{-1, 0, -3}, 0.2], 
     SphereBox[{-1, 1, -4}, 0.2], SphereBox[{-1, 2, -5}, 0.2], 
     SphereBox[{0, -4, 0}, 0.2], SphereBox[{0, -3, -1}, 0.2], 
     SphereBox[{0, -2, -2}, 0.2], SphereBox[{0, -1, -3}, 0.2], 
     SphereBox[{0, 0, -4}, 0.2], SphereBox[{0, 1, -5}, 0.2], 
     SphereBox[{1, -5, 0}, 0.2], SphereBox[{1, -4, -1}, 0.2], 
     SphereBox[{1, -3, -2}, 0.2], SphereBox[{1, -2, -3}, 0.2], 
     SphereBox[{1, -1, -4}, 0.2], SphereBox[{1, 0, -5}, 0.2], 
     SphereBox[{2, -6, 0}, 0.2], SphereBox[{2, -5, -1}, 0.2], 
     SphereBox[{2, -4, -2}, 0.2], SphereBox[{2, -3, -3}, 0.2], 
     SphereBox[{2, -2, -4}, 0.2], SphereBox[{2, -1, -5}, 0.2], 
     SphereBox[{3, -7, 0}, 0.2], SphereBox[{3, -6, -1}, 0.2], 
     SphereBox[{3, -5, -2}, 0.2], SphereBox[{3, -4, -3}, 0.2], 
     SphereBox[{3, -3, -4}, 0.2], SphereBox[{3, -2, -5}, 0.2], 
     SphereBox[{-3, -3, 0}, 0.2], SphereBox[{-3, -2, -1}, 0.2], 
     SphereBox[{-3, -1, -2}, 0.2], SphereBox[{-3, 0, -3}, 0.2], 
     SphereBox[{-3, 1, -4}, 0.2], SphereBox[{-3, 2, -5}, 0.2], 
     SphereBox[{-2, -4, 0}, 0.2], SphereBox[{-2, -3, -1}, 0.2], 
     SphereBox[{-2, -2, -2}, 0.2], SphereBox[{-2, -1, -3}, 0.2], 
     SphereBox[{-2, 0, -4}, 0.2], SphereBox[{-2, 1, -5}, 0.2], 
     SphereBox[{-1, -5, 0}, 0.2], SphereBox[{-1, -4, -1}, 0.2], 
     SphereBox[{-1, -3, -2}, 0.2], SphereBox[{-1, -2, -3}, 0.2], 
     SphereBox[{-1, -1, -4}, 0.2], SphereBox[{-1, 0, -5}, 0.2], 
     SphereBox[{0, -6, 0}, 0.2], SphereBox[{0, -5, -1}, 0.2], 
     SphereBox[{0, -4, -2}, 0.2], SphereBox[{0, -3, -3}, 0.2], 
     SphereBox[{0, -2, -4}, 0.2], SphereBox[{0, -1, -5}, 0.2], 
     SphereBox[{1, -7, 0}, 0.2], SphereBox[{1, -6, -1}, 0.2], 
     SphereBox[{1, -5, -2}, 0.2], SphereBox[{1, -4, -3}, 0.2], 
     SphereBox[{1, -3, -4}, 0.2], SphereBox[{1, -2, -5}, 0.2], 
     SphereBox[{2, -8, 0}, 0.2], SphereBox[{2, -7, -1}, 0.2], 
     SphereBox[{2, -6, -2}, 0.2], SphereBox[{2, -5, -3}, 0.2], 
     SphereBox[{2, -4, -4}, 0.2], SphereBox[{2, -3, -5}, 0.2], 
     SphereBox[{-4, -4, 0}, 0.2], SphereBox[{-4, -3, -1}, 0.2], 
     SphereBox[{-4, -2, -2}, 0.2], SphereBox[{-4, -1, -3}, 0.2], 
     SphereBox[{-4, 0, -4}, 0.2], SphereBox[{-4, 1, -5}, 0.2], 
     SphereBox[{-3, -5, 0}, 0.2], SphereBox[{-3, -4, -1}, 0.2], 
     SphereBox[{-3, -3, -2}, 0.2], SphereBox[{-3, -2, -3}, 0.2], 
     SphereBox[{-3, -1, -4}, 0.2], SphereBox[{-3, 0, -5}, 0.2], 
     SphereBox[{-2, -6, 0}, 0.2], SphereBox[{-2, -5, -1}, 0.2], 
     SphereBox[{-2, -4, -2}, 0.2], SphereBox[{-2, -3, -3}, 0.2], 
     SphereBox[{-2, -2, -4}, 0.2], SphereBox[{-2, -1, -5}, 0.2], 
     SphereBox[{-1, -7, 0}, 0.2], SphereBox[{-1, -6, -1}, 0.2], 
     SphereBox[{-1, -5, -2}, 0.2], SphereBox[{-1, -4, -3}, 0.2], 
     SphereBox[{-1, -3, -4}, 0.2], SphereBox[{-1, -2, -5}, 0.2], 
     SphereBox[{0, -8, 0}, 0.2], SphereBox[{0, -7, -1}, 0.2], 
     SphereBox[{0, -6, -2}, 0.2], SphereBox[{0, -5, -3}, 0.2], 
     SphereBox[{0, -4, -4}, 0.2], SphereBox[{0, -3, -5}, 0.2], 
     SphereBox[{1, -9, 0}, 0.2], SphereBox[{1, -8, -1}, 0.2], 
     SphereBox[{1, -7, -2}, 0.2], SphereBox[{1, -6, -3}, 0.2], 
     SphereBox[{1, -5, -4}, 0.2], SphereBox[{1, -4, -5}, 0.2], 
     SphereBox[{-5, -5, 0}, 0.2], SphereBox[{-5, -4, -1}, 0.2], 
     SphereBox[{-5, -3, -2}, 0.2], SphereBox[{-5, -2, -3}, 0.2], 
     SphereBox[{-5, -1, -4}, 0.2], SphereBox[{-5, 0, -5}, 0.2], 
     SphereBox[{-4, -6, 0}, 0.2], SphereBox[{-4, -5, -1}, 0.2], 
     SphereBox[{-4, -4, -2}, 0.2], SphereBox[{-4, -3, -3}, 0.2], 
     SphereBox[{-4, -2, -4}, 0.2], SphereBox[{-4, -1, -5}, 0.2], 
     SphereBox[{-3, -7, 0}, 0.2], SphereBox[{-3, -6, -1}, 0.2], 
     SphereBox[{-3, -5, -2}, 0.2], SphereBox[{-3, -4, -3}, 0.2], 
     SphereBox[{-3, -3, -4}, 0.2], SphereBox[{-3, -2, -5}, 0.2], 
     SphereBox[{-2, -8, 0}, 0.2], SphereBox[{-2, -7, -1}, 0.2], 
     SphereBox[{-2, -6, -2}, 0.2], SphereBox[{-2, -5, -3}, 0.2], 
     SphereBox[{-2, -4, -4}, 0.2], SphereBox[{-2, -3, -5}, 0.2], 
     SphereBox[{-1, -9, 0}, 0.2], SphereBox[{-1, -8, -1}, 0.2], 
     SphereBox[{-1, -7, -2}, 0.2], SphereBox[{-1, -6, -3}, 0.2], 
     SphereBox[{-1, -5, -4}, 0.2], SphereBox[{-1, -4, -5}, 0.2], 
     SphereBox[{0, -10, 0}, 0.2], SphereBox[{0, -9, -1}, 0.2], 
     SphereBox[{0, -8, -2}, 0.2], SphereBox[{0, -7, -3}, 0.2], 
     SphereBox[{0, -6, -4}, 0.2], SphereBox[{0, -5, -5}, 0.2]},
    Axes->True,
    Boxed->True,
    ImageCache->GraphicsData["CompressedBitmap", "\<\
eJydPAdYVUe6bMCexE22JbubTfa95O3b996xYDSaRJOYrImJF2tsq1HjoamI
BQsCIlhAQEUjKgqoCApSBOmI9C4dAanSe6+Xe8//vTdnZs7hOvem7Lvfp9wz
8///zP/PX2fmHsUOKzNjix1W5jt3vPPlgR37zMx3HnxnmeUB1KT/Kz09vSj0
z+8NPfT9f9E3/C8KAFCH1v88+l8fARjhJ1Pxf/ER8Efvm3+51Rj9/5JMz5iB
UeBWEwrzOh37Jfr8Pu61Ef83mMAh/ZpU5uJmK/S/CKb/C+H3Mq3TfyHebjpL
PX1jY2PS+YtxzSlneI7/D3wTDQnpTZZ7ESGCNU2D/oQEd1KsqXgUMir5fI77
d1DoGeLTJJkqIam3FP/9TuaaNMuznkQGZ+G3UJq/ljilACxfn2DohRT6bTrK
6wyWPoO1BGN9oCW1P1M6v/t/4r9Jx/8Dg2/A4C/+EfzfU/w//sT4oox+bPzf
0vm/xeBPZsb/6EfwX9OxSn/+RbSkFfg1/ftvDNYUButDjPUlHe9dBnoqA70I
Q/+DQv8HAz2NgV6IoZdS6L/roC3OkMzgEwr1PwzUdIYmlhVeNxF3NgM9g4Fe
gKE/BA1r0+hlx9KNLUl00s9gv8z0z8fY8+nYCxjoVxhoYuHz6Fia685ivkwp
Evqv0Kd36N+PGOhXmXHmYaxfM61/pNhLGOyZDJwhxhZXTlRX1preoFQ+Zaiw
o83VOYffUezPGezXGLg5GPtVKqnf07/LfgZrBqX+bzK2Zq8UsZbrmLlInZXD
25jKTHjR1lYw2K8zWPpUbqwmEWp/AY2I/RNUWD0j2H+i2KsZ7N8w0FJkIXJ4
EyR9m8BgKfyWocD6hL9iSpLHXM9g/04HtjgiwZL85CYdPIuS4jDU66BtFRsZ
jN8z4/wXBpW1gEbWfzJYf2Cw/i5jsVJhZ/jjmBqeHIFvZvDeYPD+cwLvhYgt
Rnd2zDfovDTG0qD0EgPNrsQfGXjsvWXdncRAf/uLsTVbWSrrmOc/MfDv6aQy
mcFayzz/mYF/V4uKuHhTGCxRr19Cj28z2P+O4dk4I2Ve0xgqYqaLEz59kPO1
tyjsez9KibVATYrLKUVWdm8xeMRiJR8i2YL4eUmHL57OUFtGuWdHZ0chdqlp
43o6PNYMhspSysPLP7NShLrkgwzo31cZrCWUGptzstTe0aKm2ctSxXHU4AeO
B+kjdv5GxxgT/oldSzaDmMlg4yiuf8TwXpXNY1FJRImzOSgr8b/gkaRY8hoD
TTIcd46/gyeuR3WvdZj2v/EzMpqgrtnKjoL97FTzuXeKDyUb4w9qMDoOERVQ
NQJECqc4++odZ3v28FqSZcd8S+eYbC2APZ/+RWk9JM4axzuSXcRnrEsGCs7r
pmX7UFQ/FGey0mbHnQQaVYPW+OxKYw821Xq2r7H0QQ2KvRBUALWjlGcFZxr7
hcvA/iN05Z1vQF0DKEcHqWYgbcpbxE9ohrYeYcvRqg1ezF808f5KNAhLRdIg
Pf3nMQDPB0HSagOzT4uKr/RBVr0q8jjOJA3WcffSDhUKQemNzrw+Q1PTd+v9
zJw0JcbO7S90RXhZYoTrjY6QUAYdSsnrIAgv7kj9lgNUKuvNhaR0UA7R/imo
34fjk2aJPK5e3Z8UBkKbduxgV/elXzhr1t5wZoSlGHpyRJmkhGfF1Adv5Hg/
alWY9mouKsG2QYiqhVQ/MvMtwUCpEIw9c0JLbHPhfjTl7EwSPO6SdU9vijwq
YvH8fJ8W60C4cIDYqFY9zvqS13Vyw9r3Hyg3RBqK2aUeAVDfAsqxceyzJym4
wyHfxoxdinTl+IJPeSo1o697w0JhpBXUvdVkDVZJvKPPHpOxkjiAdjXx65Op
KXBh121VA2kA9RX/Cgdi9JaqA7HzTYYDnPOJ62Fwfn1Zk1czpBaMBljiWU1z
lKakmFsc4QlG3O34PXnqO9kQ9gMdzTQYAhsmVmQqstBrn4d2nooGH1cRbw/4
FkLBILHfKTaz3CpMvcnYh72hsAXK2mroWm7hTFOXneg150ktZqDJ3YQX0ayX
9XREGXaHYCbotpGTp6EyD2CAzluEuLHYv9s+0IbjMcQkxzBYs6w91h/UNQCd
lXTk7znrvE0equOHyMgIL+nWKfVAFkBzwyu05fI58+72CDUai1SV1ivKa717
ILcUryebOb9BvjL5A5ulsM9/Yvh8GSSrqrZLgtv3e52prukrjgreJRDXLhBO
Jyu4+HtnBJ5zz9wRoj5/s92a8Kt/NgbyejAtzOlk241N9UFKKKvbydllbrg+
Zn+SwB27DfktUNffR7zH1KPcweJ1jj27RE05cAEKa2BIqZYym8nunGnlSrHL
xAaeFIJ6DMhqI3nf5viMebwWZ9jLae1dsLrN7r5MlfifjE3F5/yh8QZkQ3Uj
UBowscLz3Fot3Hr30xV28AOrDa2l/gAlg1AaSrKsGZ6zjzVtP0LIbrGDzKcw
PA42655X3h6B/CZljD3Nh71m7apdi72VpAmXruyo7brXAbk5rO+cRCECI/dX
CqHPIP4yycx2zgrPtyuGsMSfksILliCC/IWhbgCaGauZG6SUQY/k/ZEHio1w
6HXh9j1dbdW8VdKKdVYQlw39w7R+nGzERcbatQpxTUhMd+eY1ayiWsHbQMYT
UI7QKuDlVVxEyvHnQkw5hIkbyMc5PvtDHov84ZmLUFsBwhhgyBny/NBnK8dH
zMFgS9uDAmAEOcjWbFkfeW/sZrZ9r8yJB+gG1qaJ5U9+URY6bORtRi4TOavI
1GyTaiPK/SSFYaWC8w/dVT7s9wwSo8lMZphyfPICyeEtbrkXAv2dsNswNu94
qRCaDSEnqOz4CNDUyhleSJHnY+5W9EdGwji2IgXnEbIta8wL+cqrdFTXQopn
zvGxs3iiN9Mk727psSyuwy0d7t8QrSUY/J9LK2g3L7Da5jH4eilWlLvnTuRV
REemgO58+23QyKgNzA1jixzLIDwLgo7RWLWoPsAL+ioA+odekqXE8eXLqE0i
zbH2WpfVd60QogIOcrTVwJQfy4lC7g6ExlRaGXIhhTZ5EBxH+JxiIkeNRXXB
12G0HKCrXraSE/c2pCg9U8H/OKYnR7eUq6eFrlzkajvlTNNx/3h9IkDbAOsN
ME9ae3ZvAYlkbO73ssQdSW5pDN7JK5+IURbGnz8imnQwABJbIKGxhULw78c9
da6E2AxnGhOQriZeOw/dpbCXc8nd6qs6c4FKjlfdyIaiQeoLpp+SJLBzx2g+
GqRD/L6bc8rZclN12oXgHH8A2V2y/Yq5Q/HdazBWhaZUREY7MftchYkX4WOX
FyQ3wUS+tuhyp5VH0x5+oyvE1sCTzh4tjzkhI1E7pKyL9UzTqazN/Zf6dtvd
gfNWRBY/ZEJW38Tsph3l+BCJqU071bkpcPEDn6bD94SL9kRaUzddRYto5bi5
serOKOS3Qrb3Xo4P53g84sso+2o+GihcssNmUnc1F1K6BQV3yH5TYyWCL2xV
PTptIHF3eaFvu20AuO9HDXsCIKEDJLvl7vywq7M9bAzKqnE2TmfPX/vIr/tE
oMrVlHgJxazUB45D6uRhKEtkteR1nboj5XFsPU5jx5zsyLNKVZoSqopIy2rO
PdkkQX0jvN+Vl2YRfstmdADpa0014WS3PWQ9EfNsBecUtS1J5RWr9CAbFSKX
D7xslP1JqI6oQfNHhUvBEsk3m9lDZh6MjUzEjdhbDurhDICGpxji1X0cn7qI
robtJSitAuW4+F2ypQTf0+oxZEsNecTGNnB8zPvGld/wis/aPYOhuRN0VqBa
UvktlcorjFSIjqBUKWsxn/+JlNUq5ld5hkBjOwyPDhMprVnWmh4I5zmTp9+Y
EBwHbyhrhAGlivA21SpRJKj4oDrGF4TnsIbjoxYdattGZDrlmBcUN0H7yMh3
nH/ynkzVzSQIPENGm85Hgvxx4w5Xbz725BtRJDa3obgNpDVx9lqb3nU5HyKC
8JpM3sxZpH59lnBi4w9FXaBPIc0clqbWuJVBdKLsDT3murTvdcP+Q7Qh3V7n
NSojLMGpmlFPwV0M2pY15JkLD+9Kldiu2C8vDh911JNGvRu9t1R9rxjifMke
z6FZruWmN9VO1vQs0i4S0nrg4OKssvP1kFDQd9sU8zHpu7U9qXcBKgVoKRo5
zcveNeD+vsbx8BbIiaK6M7vwuht05qO4OkAihNU3z6p8BnDCj0fYwpH4fcvO
GirSAXoFSXYxgXYD6pRhKHnI8oyjqnzyMZNKAMdpjZprQUXQVRgqRt6/i/qZ
HUtKC68NwZNWyLhKR1na+tAPxmrRuM3kHNXZpKcNxZfqOkuOVs6mlpCXCtBH
7eF3iu3jPslQNQDqAgHqUEEE9hwfTdiY8+y6L3Q2gnqwDXNlErDIrcPCqZ4X
ey1uQVIT9SVTUDB4slBs3XoEMvNhXAmyR3n5W87u0SqfUYeL6MnoJPiXA6mR
UQahmFXoHgiN3VoWNFOWiWYr9r+IMK66JsuysQuEnDYo6x+ZJOlayCw+3VCu
D7epDnIn8zdfHbW3JzKwfgDJHRq2hzAC55jVraEYk3c6Q3Y1rOeckjYFKl28
pax+ytksuF05JOkj7zfXtp3fTyXgEASlvWLHeu5Swvfx41fDlRepBKYczxon
GBvnZiaebBfiWiA3gmrZes4vxTJXfTely5030ODL/MuqJ1dHIbcbcryJDa2d
HZFmWwmRhVrnHq/qlNZ0qklsdk9qL6clcY1ncyAkRI4Gx922NTT690NejRB3
inpPK86z+mAoXDkr5SvzH5c4VUNsbo8XT1tW1OfdBCgagqfhJF6s4RJTz3RA
cgMkXiA8fru8NfWOINpXTbwBHS169WZleDgMobK9X8pn1q3qSwoGoQnantzA
47+CZHYBJYTvixXn2r1CTCoMiTtF4IIVdNX6kUdRIHRrnwTplsgUKpF3GImQ
zMvZRt2UiVKmMaoVR52guAiEEUlLHHaPtibCZil3m7SFV+Ukiqvqy/FFcrwx
MhMepkH3EMWaifpvyincioEHMTDSL37fx5nlfCk6Z7H920HvZKgeknJglB/m
fEJxzt8Hywi6LYD8atDii71WZ6nXdkuGjF5xBl6zj7V+v/87zifZ5LHKMwqu
75Eyt0kOc7xrDoaAhxN6OhgH8qkUsbDJMJHv6kt/Ee67GpqI6AcvOtlhbp32
Fa1jVjlCeCXk90t3DhwyIW6xbbeJnEPYwTrOIXWjn8r5MpbC1BPJcMDwQbF1
FtyNVd0w2ck5FG33Up+2Jvi284NrjyeBv6/sec96bCzp8K6D5BQpajvd2lo4
4lsGj3xJy4YFucmu/UJaF+QF6FNdiYtz6BSSWnau7cxA6lYuwPMUiYU7bvtG
W1FB1NCidUqFtUTHviFzR4dKBu/sTkJftszLyHFpEx7XQYYfzRctPi2o9OyE
rDKKa7aivtgXoLQP8nxIpnLYbKwUhdoGtVAdZb2lozEc+eZGKsUTB8brk1HC
OixLId7NFeqRyx9US1JIv3YVempQldGEJf3mWRTi8VajYr0yOAb6sAdaiR2g
2Phpm38E9PWD0R64mwYNQ1hbZ5sQoZhdgsw6qB8aCTHc1fAtr3XqN+Pn5CLv
Av0NqBd10qwJVm1WxseBshuUXU9JfuKicT6xcY8gul1U8s6nWqUwEyKzoWdU
ypIVhyE4H+pGReg11pBUIc7+niHSdEuq/ytPwMNKyOtV7eRs0tZcGbU9WbjB
dfiQseIs3KkkWYuYAXt/7N1j46M8hTXhHNyolHZAtnKeacax6mvBssR5lyXR
bS7pEHyD+LKtc6MzbSuE8Dy5EtoxL6XItRGSC7TkpWlDPyo3Da/zNzrm3gdf
+fTbXR8/Y0Yg+BC4XilT1NPfNcvn6cFY8L4t+aGFkTWOmXCf7rEgCj5RlmXq
kBKIcSdrs35WcubpZuFRNSR6EH9ru7r6mVc/5DQIyRegrFTid1ZK8BmVKksN
NVlkvvzazlwxE1JDeYiUVXm72COHDNDTjWUybdPW0eRIEHAAJdysMxbSUOYw
on0CzN7j+dWPyMaAkc1/ED4+KExxHxEyBqA8jYWgVebO4epYVD+P0ueTVqo6
VAa0KSXZGNYEBcBYKyjbnhB+Vm4Yi40RtbK1Jo5oEfLoMYa82THIyobxIcir
hv4xFepBZUT+R0Q3X3cLgppuqrlbOT52AV/yBbawXRCYCY0jYvsWlA0vdxra
L969W+UGYc9JNjxd3J1b4NpucbZsq4hyJhXCmgVHwzsNxx4idtjzC/YO049J
jK093yP8KgbvR0NXNwwPdRCI3cjClu6qXi3XFbvBnTMrV5imLqG11ipkReVQ
MQKC014iRae5lxosvcGZ2tkkoyvI9HYHfuHVZ+upPEm9/LTvgmRjRt3G1xTJ
Ax5Z8ODqwcfSmcNm7m76gVzhXtK4t6lkYQfO/SOxwSUPwiLI/I6seFbtgxzk
M7ontXp2avSJLuFRJxRESbmzl8shVXsqyLnMi6di2tKSohp7Sv4/lN7B65uL
+m9VQlK0NEJArCWqHIog/g6RwPm9g32omHpeIVfFnjcODw0+RpZSRvXIMC/e
bVTIHIKKFFoncyFXzkFfuZh9Y25f3rBuIC1U3N0ZffYAC4m/ReuDuHP2QgOq
DzqUmP4f5D2O7daQXQAqJY7qnGXjZmNUbp4lWCsG/EOhpx36u8o8Z5nVrBEb
t7hDYgO1RJ2nSvJultYtR3LH66I7tFUADI1KeeAX7ZF3QVkrcpH/sZTVbHMD
I44PlfetjnhCaROgPCWIkyr2f3TejoC2HlQPV8o1X8iHp3ot6PniZCN3CK4X
Rz7M7S5YcbhtO8GceSGF7o6gj8tHoe2nH8Gd86KdXYSYGigZENtNOY98szDB
4zqts1ZxUYknmo04y7uf+yDfPu5kRleFC398tFqIeCo8sGY9BrY0rXMerEFa
Z8bvUQ7MYj53GdhvQ/2TV4UkacLzsTCxjqh1ewoxcURv3JdGt7ukwn0fIqep
aznnuI2ho273hIvmCu7yzaND46njUJVOZrthXnb2+V7IaBIS3SRve8Fza1mP
73NIS5JivvOF40J3Lir8Woi2H9zaU4G0qnaI7hWRqKr4ois4ClbNy49yHlGn
jsCzTJKlGin6I8JgvBM2cXzKx2ZanlaMYtKNZPmshNxKQkDEq5h9UlTi0QMZ
zyHdE+v6RO1lbgnl2ShDAbrT5Ui2HXsInWlbvqhL9xKgSAU1Ymlp7Q3lOG6c
4+QTlW/HIh4jd0VjsriD7b/QVcHFnneHJmRLw+Ok/QhnXLR8Nxl07QWIaSDx
fYaYnSKHjDfEVx6BqGJoVoowbgvutNmFwJWjerSSMBq8FQatndDRXXOGc641
uTRqK/KwPUSEPvF1Yat3I2Q81pLPm4xNYf7pvbEJKHx/xWCdiZCcCspBKC4O
Jf7un9z+7NVuY0ct5fzBjuPzPucTFkj8b3CHuHqIrmuQatGNY3cToWUA5F0N
/srC2x029+DSEbIT9j1nmbL8zMA+y4Rzo/CkV+GgvlEBsp8657OjYjigDtLD
5Zzm5KLIOsd0uH9f3tfwsFP1ZwG0VEk6FuBl2TUQOQxl+WQW+xygogyEsTGd
8hA1A/tWrTvMf6L0zB8svdx35Fz/UZ6cfdqurWu8OwTFlcR+kCkE7SwY8y2C
OH96H2fDxuG8OPHAQuY7+o6DejQDoL6c7Luh5NJe8bTGsx3Syzr9d7UOA579
6xevQnsdCMPi/u8mzjJntevwYVPksE8fVon5dIeYQK7i+Mj36emK8TVIbyXn
ItOcJb976gZUNcDgyID3xz799rfq9vN0XhO84Tucxi/ejiLawN4lwJ7GYPPC
/Ez3ASG9E4qjMdyUsxr572ZryCmFsXF5N3ZO6V0vGKxCuX0HXLWn2uDBHWrY
Ku7cUm05lQrkYIH2292GkjYo72qmVDZy3rn7UvbZQmEO8ukSFkIJ+OhC70Fn
Mt8ZduSgqVxcjXXOEFkl5tDo+/FPk+vPl0JsNM3bbGZfKd8dABdd5DVJue8s
qPNQHZ/A7u2K0UW8ts/qC/Gv7L2FNwm9taPhcTDQA/29tcS34XsFAcJl+4mZ
h75v0bBBijvbbkJEM/SgMr2tQdL4B7G2zUJ8A2TS2O36UUizQzzc8STx32Dj
uv68KPEGgGMG3Z9cbZiV6DooZHRDQSDxLm40Ll8O3pox6pkJYZ54PpOPWEM1
8meDKjEczz/dZXFC8lLXz5r3NUUA1LSS01d6Eh4MZWSnn72jJp0ps1pF5MPe
hHiH8nbN21Y1mIoKxhqyvgcWJZU5P4OobAi1Lf1c3nOwhxgcWSUvYPgMZQoD
3dDcmCVJ6dYlW1Un8sGtXURKJkgHPjMv/UbKC89lQHiTqASHJV4UX/XcjYHu
fmlXgp99s9QqAXx9FQtrg6/ByFMUksTU4zvONO2LY92m0u7ZxSLAsoyPPD2E
ZHNJOu8xd4OsSugZB4s5oWV22RASxt55mkJ1iL2BR2TE3rUgp/a2HP9Ao+rU
338N8puhbnCEWsSuPepClMl1QW9lOMQ9njjl8Frg3nnAfdCa7GNN3YjozJb2
abiM7y9B8nMoHxiX4E1/2FDS51sLadE0A+L4Y7PdK8xuCa4nSEw02z5SEY/8
zRh6sguFrE6gJ0+K0gavVsjKw5T5a1LGdOUh2M+5UbH3Plwm+2STd5uPl6G5
dqusONvKbeLZOnvnUNqfYXM5kuGJfuclqaYw2MRdyjSOgqv3Bl15Ug1dzIL4
DijJFl2DxFf8w1NDQtoglCfJnF77x8OOM/HgR3aV9Q8HQybeV6A+5/Q5aKoE
YWSYSngnd0XBuZwz72l7qILKeroOq5HHnSdWSfTZpwg05zrT0h5KisQDbvQ5
NT+wxSEWbp/FEvJz2zOALatNPDXg+EqFdNLsWYS6D7mZdPZGK6EyV0/rJMVY
+84jkQy7B4r9kcH2VR0F99FAaNoPid9CSW7i5yf695hRXne+eDqt7xEMTd1Q
3Vom69Hjm6cFMZa2tsl7ydGfnuzfu4/emuKxclrsGKlNAItzkFcFA5LP17f5
JLXevQIe0bPeyZul+HCMO/p045mhA+S8T//sMXUnWjTJy79scQFyKqFvHEkj
+JFVufCgCMJP7MBbpajf3B0Km+AVaktsbCLyeItpJfnvTpzPT3CrWN7mlwm1
0tmZ/r5lpbU+PZBXLnNv8fAr7yH7yz02ZKYTeeFBb8hpgor+EWIt/McFJR7d
kFXTcm834RVF5E3cidRvfcZPicm+31MazfcfxXeJhgWsC0fC1keOXwlRcB5X
dtb33O+CglxiRcFz+Ex84LDSGSLqyS2JyYrlvREhoG5HADahayOUP4S8Qu2F
3Xth8zciFfYuiHTf6XjEutARt0C4ZkW8+pq9QlYBKJWSv10zvyTdYxzyBqEs
Ss7a/OccaduxW877LOJf1KSVJkJSBj6XJHm629eZlkZ1pTfHoLBTyKK3HFC2
uKvUyLJ4JbWA1+wzKRWF4dOoWyg3Er+fvAQ11aBWqtF3q4XxNWeLIUr8oSRS
Zn8pU5jmwplXrDQnyHwoJjG/PNILab8gnxSw9bJuW2JvhpBbmysXViR6CkKx
APV5RFsOcQ7l2y8qj1vQU9p1GtmP4vO2u4+gXTrjn7aGe5zl3AYpNULcaXnX
+f4ip+69dnQnx9EJVu+HhGxUqcq+y8PXpGo0qB6yIohO2eODX7HMnmvXvPNI
VBvRqVct9kNFlriNip4uLY3qv/AY/I7jMfxDd1eo7z+D5DsIK0yKfHZRkEpu
b02iVsRGc92SYc8VyH3HazFQ2wfJVU+oFbmbtvdGjED5U7rGB7hTxVuvqBxt
6C6y4qveiEcwOihlhojeesPMDNceSG2GHF/qma5x+xo27xK/YvnM/OE6tNej
LLgT83UlzLxEHVy4lktION4kxNRB1j3Ujnxy9PsmtdI516+dn9DluHQWBlCt
1flURF5Y7+cLAw2g7q2x/zi+0TUfIu8rsGeQog8bt9nbz0Qaf2VaSa2w1w1K
amFQqabcu+5oaQ0agpJqWmmgke7MP9W525bm34pVw+HxMNQ/URc8FFl6VAtJ
l2SPG/7BofYd5Lqa3m8UXMm9uzCGvIDR4IMwGGmH4c4S1HNkSUatezU8TiXa
FTaXL12Gs76TalQGlCyjUpm2ygc8ysUZ2xyD50hCgyP2H8a2ncuFcC+dEYbd
OyDcixXhS5KsRH8QvNi9z8qpyYInVmbpBIVlMD5O4Q2MZiUlOfcIaW3wxF/O
fYMW2LTzlgRi4iwYL9Lz+w9huI9iTzX7IKPkXAMkljR6U+O5we1r/Kcp1rjf
rLklI8KeQ1CKagN8xd5iYUoNkkhiEka5HWZaoL5TALHXyZlV+ZdS3L1cKtY7
x2yhSTyM6NTK5nVnI+8yreQm0pp52ekXBsXsqCiU2Ifig5p70dArr/B3hmlP
nJsg4RkkeZAVtuQsi9Yc790t1WQbzITMVBAkCzFYyUUm2jcJCTVC7CmiRSgY
piziy6Ts9rXDKZT9rXuh8AkI+FBBsbg55BY+Lu+p3jz7Ya5tKTzMQPM9x/FV
ksfVP5xKMX+Wa7Jz8h7TOoOs/+Lm+3HQMwDxuaHync0HHzn27LGi/m31d6rE
RzDeC7UVkYTrS9vr+oI6IT+TWgaK7mXLJ6LzlH9K97dFVj5pDY4UTxapPii4
fReMcjqulcPjKKIPwfOt2nfw2EKm7wuBmHYRbe2WsXRUdPeInjDiYK0QXTn2
4DB7G1Y3n+zvY4lvc5l1pGqzTfcunsbOze/n5F/ohLQ6SP2BrN2mANCkY2Bz
AaprUPBSGcv3vrm7cXuL1QH5EHuFxkqOj198rAdVGFh2U5ycoBFp4pCYPG6c
k55/rhXSyskMpiMy7hxf9BlPpnYoAYvH8NmdYBjqhbeYWbOejPD2N12taF63
g/fXq8KaIPcRwVNsHI9LgrFBqdrbwPFJnx7pNub1NHfJFJ+3RwaAqhEl9uWE
n72fFVRdbYesEio1B47P/GBiZSd7PKUyMtqiepQKShLx8N7DFCPOL3Fvnvpe
usrPcjtnmv2VdScaj/W17K9YCF//qatV/8Zt8Y6HMCL97mM7x6cu2dO0ma6h
vt3yonrPJkh/Qutpgr83FMJb6K7fJksh+4l4HVnWviuRO1LHvRLh/kmifaHz
j3TuxLeW5GxsiuKbnkdBIDSLU3FY86zVD+l6ttZvunXeltL6vTWZ6YXV+R3X
KyExgT7vOgYlBaKxT7xVAP09xZmVKczpDfsljdG3QKgGaCtADf80zCgQlalC
nuV0R3y9lAxjEweheLrmJ6HkGYwpVf/OzIOt18hsxd9ryrFA//QZaBC1V03l
vYbjI+bx5V/R3x6J5x1Bu56pg6sgldy2nuIYBVk4H0E0jLapUQRTDku/yljP
/ZC4I0Z1NaT1jJj5PFhwtMvYREuDdc+L/ZU3kbW7dLOW5iOOnyY2uBZAZJQ8
vyc3b0NfI4z3N8hyFVOL+WYN63ky4/XftGX4CVCtHn8auOX9rILzbZD+TEv/
2DqfzIpjWsneyPFjUIdqoD7VxO9wUL6WNF/aCdOukfEXFPRr3Kvgcdo0qnMt
XmlQgaOf0Xb14zR8G5vVJjbbI/OapXNe3qUvSuvskphm5wwIuae1c6ov747t
tsIxeEjCweGar0Exhx2D3WUjM2HfDEC00MVOaEES6hkmd5rlqgdVdTGzNKo6
ywRpxmT1VnIP06wrICJ/+K7Fz42PKZBfqbO73/pGDhBajZ+xZjK3947eWhEz
cP4RBJwjWrZy1WB8BL67w1Jjd8jImHOZ1r/rGCPb74Iwgkyrs47c9XbSsN11
fqBFQ/dIhlR6rFbQOu4OpLZryGYjdzXNOA5uBJN3Fmjds2Kpv8+0snGUeO5N
q7pyAgGeq7XekMD+fphQZcdmsxDieW5kive0xK8s1Uk6qbJQbEbHvlGA1XhC
5QOmlfWZv4zKQqaVjRNfM8+sVyBUFjGtbNz85meokPNl8Z0booaw2Ow7El7S
if0x08ruOuD3DU3S1GxjeuhA3g2iCfvWv4DLvv+C3RPD72YwkJ8l/0HefsOe
bmg8r9J4nhiNfU8Gu3sr/WKdQH/G9LL74Wt0SnIppcFCv0j7C6aXPbMSZSay
jW0Av//lp3aKllNobP1ab+Zga0OxX3zEPg+/iean6uallDa5EfwV08vWmUso
NPmlqK63e2g+f0ihyS/kvqYSYqEWUCisG1r6zOZi80DjPiXRvZ/IOWZT2tj+
td4EwuYC/02hcSajt5LpZSP03yj0f2PoVUwvG8XepfPGcVTrnSBsVHiH0ibv
h1lDJcdC/ZlCkffqsO+GYP3rGxSavB+IfR8F6wF/S6HJG5nYd1+wPkq6Q78U
Q29gelmfJO0TY70nbx7R8h8zKAzWMfLmjhf8hHRuQ96UtpmOL+VhWL74XVsi
YSwZPb1f/R8lnWvi\
\>"]]}], "}"}]], "Output",
 ImageSize->{390, 166},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->168992869]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell[TextData[{
 "The generator matrix of a lattice may be given as a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1544459259],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<RootLattice\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<D\>\"", ",", "4"}], "}"}]}], "}"}], ",", 
   "\"\<GeneratorMatrix\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1091924724],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "8", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 8, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "4"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->153100893]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["Normal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Normal"]], "InlineFormula"],
 " to convert to a matrix expression:"
}], "ExampleText",
 CellID->29629415],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Normal", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1912855448],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "1"}], ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}], ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", 
     RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", 
     RowBox[{"-", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{393, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->153534084]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->912570862],

Cell[TextData[{
 "Extra coordinates may be used to describe an ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 "-dimensional lattice:"
}], "ExampleText",
 CellID->250270760],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Dimensions", " ", "@", " ", 
  RowBox[{"LatticeData", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"\"\<RootLattice\>\"", ",", 
      RowBox[{"{", 
       RowBox[{"\"\<E\>\"", ",", " ", "6"}], "}"}]}], "}"}], ",", 
    "\"\<GeneratorMatrix\>\""}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1527682494],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6", ",", "8"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1060224634]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LatticeData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<RootLattice\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<E\>\"", ",", " ", "6"}], "}"}]}], "}"}], ",", 
   "\"\<Dimension\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->445193126],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->101845684]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["LatticeReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GraphData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GraphData"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->1584193535],

Cell[TextData[ButtonBox["LatticeData Source Information",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
LatticeDataSourceInformation.html"], None}]], "RelatedLinks",
 CellID->1038487239]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Computational Geometry",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ComputationalGeometry"]], "MoreAbout",
 CellID->88100761],

Cell[TextData[ButtonBox["Cryptographic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/CryptographicNumberTheory"]], "MoreAbout",
 CellID->53464611],

Cell[TextData[ButtonBox["Integrated Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DataCollections"]], "MoreAbout",
 CellID->118303756],

Cell[TextData[ButtonBox["Discrete Mathematics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteMathematics"]], "MoreAbout",
 CellID->71198167],

Cell[TextData[ButtonBox["Mathematical Data",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MathematicalData"]], "MoreAbout",
 CellID->189559105],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->109466970]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"LatticeData - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 52, 15.0764028}", "context" -> "System`", 
    "keywords" -> {
     "automorphism group order", "Barnes-Wall lattice", "lattice basis", 
      "Bravais crystal lattice", "center density", "covering radius", 
      "Coxeter-Barnes", "coxeter number", "sphere packing density", 
      "lattice dimension", "dual lattice", "dual root lattice", 
      "extremal lattice", "generator matrix", "genus", "glue vectors", 
      "gram matrix", "hermite invariant", "integer lattice", "kappa lattice", 
      "kissing number", "laminated lattice", "lattice", "Lie algebras", 
      "Lie groups", "minimal norm", "minimal vectors", "modular number", 
      "Mordell-Weil", "Niemeier", "nonextremal", "non-extremal", 
      "nonintegral", "non-integral", "nonunimodular", "non-unimodular", 
      "packing radius", "perfect lattice", "quadratic form", 
      "Quebbemann lattice", "radial function", "root lattice", 
      "root lattices", "theta series function", "unimodular"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "LatticeData[lattice, \" property\"] gives the specified property for a \
lattice. LatticeData[n] gives a list of named lattices of dimension n.", 
    "synonyms" -> {"lattice data"}, "title" -> "LatticeData", "type" -> 
    "Symbol", "uri" -> "ref/LatticeData"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[20895, 622, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->334793323]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 137695, 3583}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2100, 47, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2723, 74, 1576, 37, 70, "ObjectNameGrid"],
Cell[4302, 113, 972, 29, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5311, 147, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5789, 164, 239, 7, 70, "Notes",
 CellID->1399648665],
Cell[6031, 173, 270, 9, 70, "Notes",
 CellID->1291358921],
Cell[6304, 184, 364, 12, 70, "Notes",
 CellID->1682356610],
Cell[6671, 198, 454, 15, 70, "Notes",
 CellID->255092973],
Cell[7128, 215, 68, 1, 70, "Notes",
 CellID->1227577802],
Cell[7199, 218, 4005, 116, 70, "2ColumnTableMod",
 CellID->1287804506],
Cell[11207, 336, 442, 13, 70, "Notes",
 CellID->288915648],
Cell[11652, 351, 253, 9, 70, "Notes",
 CellID->169971442],
Cell[11908, 362, 576, 21, 70, "Notes",
 CellID->39805619],
Cell[12487, 385, 279, 9, 70, "Notes",
 CellID->1067943069],
Cell[12769, 396, 72, 1, 70, "Notes",
 CellID->1024721485],
Cell[12844, 399, 2210, 46, 70, "2ColumnTableMod",
 CellID->601387462],
Cell[15057, 447, 81, 1, 70, "Notes",
 CellID->1514415704],
Cell[15141, 450, 1193, 25, 70, "2ColumnTableMod",
 CellID->808552884],
Cell[16337, 477, 63, 1, 70, "Notes",
 CellID->1433705930],
Cell[16403, 480, 497, 11, 70, "2ColumnTableMod",
 CellID->1529046170],
Cell[16903, 493, 72, 1, 70, "Notes",
 CellID->1597847868],
Cell[16978, 496, 776, 18, 70, "2ColumnTableMod",
 CellID->746111741],
Cell[17757, 516, 395, 14, 70, "Notes",
 CellID->743016154],
Cell[18155, 532, 308, 9, 70, "Notes",
 CellID->1461031052],
Cell[18466, 543, 670, 24, 70, "Notes",
 CellID->1220871166],
Cell[19139, 569, 72, 1, 70, "Notes",
 CellID->1822633102],
Cell[19214, 572, 758, 17, 70, "2ColumnTableMod",
 CellID->2122923524],
Cell[19975, 591, 75, 1, 70, "Notes",
 CellID->1185684217],
Cell[20053, 594, 579, 13, 70, "2ColumnTableMod",
 CellID->1576899322],
Cell[20635, 609, 223, 8, 70, "Notes",
 CellID->94188222]
}, Closed]],
Cell[CellGroupData[{
Cell[20895, 622, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->334793323],
Cell[CellGroupData[{
Cell[21280, 636, 148, 5, 70, "ExampleSection",
 CellID->175436389],
Cell[21431, 643, 76, 1, 70, "ExampleText",
 CellID->1908265251],
Cell[CellGroupData[{
Cell[21532, 648, 169, 5, 28, "Input",
 CellID->1693289259],
Cell[21704, 655, 8559, 148, 193, 1288, 24, "CachedBoxData", "BoxData", \
"Output",
 CellID->931804828]
}, Open  ]],
Cell[30278, 806, 125, 3, 70, "ExampleDelimiter",
 CellID->129938682],
Cell[30406, 811, 88, 1, 70, "ExampleText",
 CellID->1653899797],
Cell[CellGroupData[{
Cell[30519, 816, 167, 5, 70, "Input",
 CellID->65561411],
Cell[30689, 823, 476, 17, 36, "Output",
 CellID->1392990272]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[31214, 846, 222, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[CellGroupData[{
Cell[31461, 857, 240, 7, 70, "ExampleSubsection",
 CellID->2127050445],
Cell[31704, 866, 126, 3, 70, "ExampleText",
 CellID->1759462993],
Cell[CellGroupData[{
Cell[31855, 873, 102, 3, 70, "Input",
 CellID->444704161],
Cell[31960, 878, 937, 18, 122, "Output",
 CellID->120781910]
}, Open  ]],
Cell[32912, 899, 126, 3, 70, "ExampleDelimiter",
 CellID->1537603642],
Cell[33041, 904, 162, 6, 70, "ExampleText",
 CellID->1833686743],
Cell[CellGroupData[{
Cell[33228, 914, 157, 5, 70, "Input",
 CellID->235271504],
Cell[33388, 921, 391, 13, 36, "Output",
 CellID->609514955]
}, Open  ]],
Cell[33794, 937, 126, 3, 70, "ExampleDelimiter",
 CellID->1314012897],
Cell[33923, 942, 341, 13, 70, "ExampleText",
 CellID->1905478627],
Cell[CellGroupData[{
Cell[34289, 959, 225, 7, 70, "Input",
 CellID->1630743499],
Cell[34517, 968, 158, 5, 36, "Output",
 CellID->620852495]
}, Open  ]],
Cell[34690, 976, 126, 3, 70, "ExampleDelimiter",
 CellID->1096715949],
Cell[34819, 981, 132, 4, 70, "ExampleText",
 CellID->652812642],
Cell[CellGroupData[{
Cell[34976, 989, 291, 9, 70, "Input",
 CellID->1238301],
Cell[35270, 1000, 390, 13, 36, "Output",
 CellID->44842692]
}, Open  ]],
Cell[35675, 1016, 125, 3, 70, "ExampleDelimiter",
 CellID->328121571],
Cell[35803, 1021, 198, 8, 70, "ExampleText",
 CellID->985829595],
Cell[CellGroupData[{
Cell[36026, 1033, 255, 8, 70, "Input",
 CellID->1001666560],
Cell[36284, 1043, 303, 11, 36, "Output",
 CellID->2051847994]
}, Open  ]],
Cell[36602, 1057, 126, 3, 70, "ExampleDelimiter",
 CellID->1862655391],
Cell[36731, 1062, 187, 7, 70, "ExampleText",
 CellID->1780034218],
Cell[CellGroupData[{
Cell[36943, 1073, 245, 8, 70, "Input",
 CellID->327928736],
Cell[37191, 1083, 160, 5, 36, "Output",
 CellID->1116831632]
}, Open  ]],
Cell[37366, 1091, 125, 3, 70, "ExampleDelimiter",
 CellID->112150006],
Cell[37494, 1096, 196, 8, 70, "ExampleText",
 CellID->1561992423],
Cell[CellGroupData[{
Cell[37715, 1108, 226, 7, 70, "Input",
 CellID->1566552433],
Cell[37944, 1117, 166, 5, 36, "Output",
 CellID->1582164244]
}, Open  ]],
Cell[38125, 1125, 125, 3, 70, "ExampleDelimiter",
 CellID->101180043],
Cell[38253, 1130, 170, 7, 70, "ExampleText",
 CellID->1873417924],
Cell[CellGroupData[{
Cell[38448, 1141, 282, 9, 70, "Input",
 CellID->129816354],
Cell[38733, 1152, 159, 5, 36, "Output",
 CellID->1291931754]
}, Open  ]],
Cell[38907, 1160, 169, 7, 70, "ExampleText",
 CellID->1906796243],
Cell[CellGroupData[{
Cell[39101, 1171, 282, 9, 70, "Input",
 CellID->625991920],
Cell[39386, 1182, 158, 5, 36, "Output",
 CellID->289834661]
}, Open  ]],
Cell[39559, 1190, 169, 7, 70, "ExampleText",
 CellID->1186042056],
Cell[CellGroupData[{
Cell[39753, 1201, 282, 9, 70, "Input",
 CellID->865623738],
Cell[40038, 1212, 190, 7, 55, "Output",
 CellID->758236031]
}, Open  ]],
Cell[40243, 1222, 126, 3, 70, "ExampleDelimiter",
 CellID->2072225708],
Cell[40372, 1227, 90, 1, 70, "ExampleText",
 CellID->7687310],
Cell[CellGroupData[{
Cell[40487, 1232, 159, 4, 70, "Input",
 CellID->625471725],
Cell[40649, 1238, 158, 5, 36, "Output",
 CellID->505011719]
}, Open  ]],
Cell[CellGroupData[{
Cell[40844, 1248, 162, 4, 70, "Input",
 CellID->1601242085],
Cell[41009, 1254, 158, 5, 36, "Output",
 CellID->876588289]
}, Open  ]],
Cell[41182, 1262, 125, 3, 70, "ExampleDelimiter",
 CellID->228297362],
Cell[41310, 1267, 77, 1, 70, "ExampleText",
 CellID->78167828],
Cell[CellGroupData[{
Cell[41412, 1272, 216, 7, 70, "Input",
 CellID->1654176879],
Cell[41631, 1281, 196, 5, 36, "Output",
 CellID->1081554803]
}, Open  ]],
Cell[41842, 1289, 88, 1, 70, "ExampleText",
 CellID->1340823276],
Cell[CellGroupData[{
Cell[41955, 1294, 173, 5, 70, "Input",
 CellID->1670891761],
Cell[42131, 1301, 275, 8, 36, "Output",
 CellID->1389985813]
}, Open  ]],
Cell[42421, 1312, 126, 3, 70, "ExampleDelimiter",
 CellID->1006267807],
Cell[42550, 1317, 69, 1, 70, "ExampleText",
 CellID->1315236016],
Cell[CellGroupData[{
Cell[42644, 1322, 122, 3, 70, "Input",
 CellID->1122542341],
Cell[42769, 1327, 401, 10, 36, "Output",
 CellID->1164447025]
}, Open  ]],
Cell[43185, 1340, 126, 3, 70, "ExampleDelimiter",
 CellID->1357621717],
Cell[43314, 1345, 97, 1, 70, "ExampleText",
 CellID->1695025154],
Cell[CellGroupData[{
Cell[43436, 1350, 123, 3, 70, "Input",
 CellID->1917185823],
Cell[43562, 1355, 858, 17, 105, "Output",
 CellID->842674064]
}, Open  ]],
Cell[CellGroupData[{
Cell[44457, 1377, 119, 3, 70, "Input",
 CellID->1064432362],
Cell[44579, 1382, 731, 15, 105, "Output",
 CellID->168341683]
}, Open  ]],
Cell[45325, 1400, 126, 3, 70, "ExampleDelimiter",
 CellID->1324687773],
Cell[45454, 1405, 86, 1, 70, "ExampleText",
 CellID->1128472672],
Cell[CellGroupData[{
Cell[45565, 1410, 153, 4, 70, "Input",
 CellID->1812111647],
Cell[45721, 1416, 163, 5, 36, "Output",
 CellID->1082703082]
}, Open  ]],
Cell[45899, 1424, 126, 3, 70, "ExampleDelimiter",
 CellID->1964625524],
Cell[46028, 1429, 93, 1, 70, "ExampleText",
 CellID->1900109243],
Cell[CellGroupData[{
Cell[46146, 1434, 172, 5, 70, "Input",
 CellID->2054216211],
Cell[46321, 1441, 263, 8, 36, "Output",
 CellID->1361758143]
}, Open  ]],
Cell[46599, 1452, 126, 3, 70, "ExampleDelimiter",
 CellID->1638048307],
Cell[46728, 1457, 91, 1, 70, "ExampleText",
 CellID->2085024856],
Cell[CellGroupData[{
Cell[46844, 1462, 188, 6, 70, "Input",
 CellID->374603046],
Cell[47035, 1470, 732, 15, 105, "Output",
 CellID->1335883322]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[47816, 1491, 231, 7, 70, "ExampleSubsection",
 CellID->176557506],
Cell[48050, 1500, 77, 1, 70, "ExampleText",
 CellID->575677773],
Cell[CellGroupData[{
Cell[48152, 1505, 125, 3, 70, "Input",
 CellID->1002501076],
Cell[48280, 1510, 1273, 24, 122, "Output",
 CellID->249463469]
}, Open  ]],
Cell[CellGroupData[{
Cell[49590, 1539, 103, 3, 70, "Input",
 CellID->1142133792],
Cell[49696, 1544, 159, 5, 36, "Output",
 CellID->844444539]
}, Open  ]],
Cell[49870, 1552, 125, 3, 70, "ExampleDelimiter",
 CellID->318223146],
Cell[49998, 1557, 106, 2, 70, "ExampleText",
 CellID->1639060664],
Cell[CellGroupData[{
Cell[50129, 1563, 165, 4, 70, "Input",
 CellID->1919839779],
Cell[50297, 1569, 1163, 22, 122, "Output",
 CellID->2007427597]
}, Open  ]],
Cell[51475, 1594, 125, 3, 70, "ExampleDelimiter",
 CellID->255389705],
Cell[51603, 1599, 87, 1, 70, "ExampleText",
 CellID->1662445229],
Cell[CellGroupData[{
Cell[51715, 1604, 175, 5, 70, "Input",
 CellID->1954716333],
Cell[51893, 1611, 6083, 106, 193, 1105, 20, "CachedBoxData", "BoxData", \
"Output",
 CellID->1867528409]
}, Open  ]],
Cell[57991, 1720, 125, 3, 70, "ExampleDelimiter",
 CellID->938498346],
Cell[58119, 1725, 61, 1, 70, "ExampleText",
 CellID->74314007],
Cell[CellGroupData[{
Cell[58205, 1730, 225, 7, 70, "Input",
 CellID->1307516742],
Cell[58433, 1739, 397, 11, 70, "Output",
 Evaluatable->False,
 CellID->12510632]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[58879, 1756, 237, 7, 70, "ExampleSubsection",
 CellID->1727086835],
Cell[59119, 1765, 158, 6, 70, "ExampleText",
 CellID->435593338],
Cell[CellGroupData[{
Cell[59302, 1775, 168, 5, 70, "Input",
 CellID->245479143],
Cell[59473, 1782, 475, 17, 36, "Output",
 CellID->340675425]
}, Open  ]],
Cell[CellGroupData[{
Cell[59985, 1804, 164, 4, 70, "Input",
 CellID->1085553006],
Cell[60152, 1810, 188, 6, 36, "Output",
 CellID->540465073]
}, Open  ]],
Cell[CellGroupData[{
Cell[60377, 1821, 157, 4, 70, "Input",
 CellID->884327185],
Cell[60537, 1827, 162, 5, 36, "Output",
 CellID->1888138748]
}, Open  ]],
Cell[60714, 1835, 126, 3, 70, "ExampleDelimiter",
 CellID->1953721738],
Cell[60843, 1840, 152, 4, 70, "ExampleText",
 CellID->1176287936],
Cell[CellGroupData[{
Cell[61020, 1848, 177, 5, 70, "Input",
 CellID->136886070],
Cell[61200, 1855, 3063, 54, 70, "Output",
 Evaluatable->False,
 CellID->425654873]
}, Open  ]],
Cell[64278, 1912, 125, 3, 70, "ExampleDelimiter",
 CellID->852451040],
Cell[64406, 1917, 297, 10, 70, "ExampleText",
 CellID->1281220143],
Cell[CellGroupData[{
Cell[64728, 1931, 216, 7, 70, "Input",
 CellID->648414932],
Cell[64947, 1940, 213, 6, 36, "Output",
 CellID->1329994934]
}, Open  ]],
Cell[65175, 1949, 125, 3, 70, "ExampleDelimiter",
 CellID->160146669],
Cell[65303, 1954, 295, 10, 70, "ExampleText",
 CellID->187464565],
Cell[CellGroupData[{
Cell[65623, 1968, 181, 5, 70, "Input",
 CellID->10131782],
Cell[65807, 1975, 210, 6, 36, "Output",
 CellID->67574115]
}, Open  ]],
Cell[66032, 1984, 126, 3, 70, "ExampleDelimiter",
 CellID->1304052878],
Cell[66161, 1989, 87, 1, 70, "ExampleText",
 CellID->451374594],
Cell[CellGroupData[{
Cell[66273, 1994, 235, 7, 70, "Input",
 CellID->1340560154],
Cell[66511, 2003, 207, 7, 36, "Output",
 CellID->1846785189]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[66779, 2017, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[67026, 2026, 245, 7, 70, "ExampleText",
 CellID->1044754157],
Cell[CellGroupData[{
Cell[67296, 2037, 171, 5, 70, "Input",
 CellID->1493169506],
Cell[67470, 2044, 8588, 153, 193, 2199, 44, "CachedBoxData", "BoxData", \
"Output",
 CellID->1433327455]
}, Open  ]],
Cell[CellGroupData[{
Cell[76095, 2202, 189, 6, 70, "Input",
 CellID->396037781],
Cell[76287, 2210, 158, 5, 36, "Output",
 CellID->38529636]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[76494, 2221, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[76724, 2230, 88, 1, 70, "ExampleText",
 CellID->1059341242],
Cell[CellGroupData[{
Cell[76837, 2235, 209, 6, 70, "Input",
 CellID->1708820664],
Cell[77049, 2243, 158, 5, 36, "Output",
 CellID->114137825]
}, Open  ]],
Cell[77222, 2251, 119, 3, 70, "ExampleText",
 CellID->1748383784],
Cell[CellGroupData[{
Cell[77366, 2258, 359, 10, 70, "Input",
 CellID->1067515787],
Cell[77728, 2270, 1483, 31, 140, "Output",
 CellID->611528004]
}, Open  ]],
Cell[79226, 2304, 116, 3, 70, "ExampleText",
 CellID->26445307],
Cell[CellGroupData[{
Cell[79367, 2311, 395, 11, 70, "Input",
 CellID->45751088],
Cell[79765, 2324, 1633, 34, 136, "Output",
 CellID->550503552]
}, Open  ]],
Cell[81413, 2361, 125, 3, 70, "ExampleDelimiter",
 CellID->780948601],
Cell[81541, 2366, 69, 1, 70, "ExampleText",
 CellID->564854877],
Cell[CellGroupData[{
Cell[81635, 2371, 205, 6, 70, "Input",
 CellID->128961002],
Cell[81843, 2379, 343, 13, 36, "Output",
 CellID->12094353]
}, Open  ]],
Cell[82201, 2395, 103, 2, 70, "ExampleText",
 CellID->599206258],
Cell[CellGroupData[{
Cell[82329, 2401, 565, 19, 70, "Input",
 CellID->68126967],
Cell[82897, 2422, 1318, 26, 70, "Output",
 Evaluatable->False,
 CellID->994329606]
}, Open  ]],
Cell[CellGroupData[{
Cell[84252, 2453, 240, 9, 70, "Input",
 CellID->80914975],
Cell[84495, 2464, 1318, 26, 70, "Output",
 Evaluatable->False,
 CellID->216715561]
}, Open  ]],
Cell[CellGroupData[{
Cell[85850, 2495, 490, 14, 70, "Input",
 CellID->22467214],
Cell[86343, 2511, 653, 19, 203, "Output",
 CellID->741398235]
}, Open  ]],
Cell[87011, 2533, 124, 3, 70, "ExampleDelimiter",
 CellID->82786105],
Cell[87138, 2538, 68, 1, 70, "ExampleText",
 CellID->34351434],
Cell[CellGroupData[{
Cell[87231, 2543, 224, 7, 70, "Input",
 CellID->76540591],
Cell[87458, 2552, 475, 17, 36, "Output",
 CellID->350796424]
}, Open  ]],
Cell[87948, 2572, 102, 2, 70, "ExampleText",
 CellID->86968521],
Cell[88053, 2576, 781, 25, 70, "Input",
 CellID->37651330],
Cell[88837, 2603, 263, 10, 70, "Input",
 CellID->651228490],
Cell[CellGroupData[{
Cell[89125, 2617, 722, 22, 70, "Input",
 CellID->76730316],
Cell[89850, 2641, 39389, 645, 187, "Output",
 CellID->168992869]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[129288, 3292, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[129522, 3301, 242, 8, 70, "ExampleText",
 CellID->1544459259],
Cell[CellGroupData[{
Cell[129789, 3313, 285, 9, 70, "Input",
 CellID->1091924724],
Cell[130077, 3324, 479, 17, 36, "Output",
 CellID->153100893]
}, Open  ]],
Cell[130571, 3344, 216, 8, 70, "ExampleText",
 CellID->29629415],
Cell[CellGroupData[{
Cell[130812, 3356, 103, 3, 70, "Input",
 CellID->1912855448],
Cell[130918, 3361, 605, 20, 36, "Output",
 CellID->153534084]
}, Open  ]],
Cell[131538, 3384, 125, 3, 70, "ExampleDelimiter",
 CellID->912570862],
Cell[131666, 3389, 193, 6, 70, "ExampleText",
 CellID->250270760],
Cell[CellGroupData[{
Cell[131884, 3399, 338, 10, 70, "Input",
 CellID->1527682494],
Cell[132225, 3411, 204, 7, 36, "Output",
 CellID->1060224634]
}, Open  ]],
Cell[CellGroupData[{
Cell[132466, 3423, 283, 9, 70, "Input",
 CellID->445193126],
Cell[132752, 3434, 158, 5, 36, "Output",
 CellID->101845684]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[132971, 3446, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[133290, 3457, 467, 17, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[133794, 3479, 326, 9, 70, "RelatedLinksSection",
 CellID->1584193535],
Cell[134123, 3490, 236, 5, 70, "RelatedLinks",
 CellID->1038487239]
}, Open  ]],
Cell[CellGroupData[{
Cell[134396, 3500, 299, 8, 70, "MoreAboutSection"],
Cell[134698, 3510, 154, 3, 70, "MoreAbout",
 CellID->88100761],
Cell[134855, 3515, 163, 3, 70, "MoreAbout",
 CellID->53464611],
Cell[135021, 3520, 150, 3, 70, "MoreAbout",
 CellID->118303756],
Cell[135174, 3525, 150, 3, 70, "MoreAbout",
 CellID->71198167],
Cell[135327, 3530, 145, 3, 70, "MoreAbout",
 CellID->189559105],
Cell[135475, 3535, 179, 3, 70, "MoreAbout",
 CellID->109466970]
}, Open  ]],
Cell[135669, 3541, 27, 0, 70, "History"],
Cell[135699, 3543, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

