(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     37473,       1388]
NotebookOptionsPosition[     29975,       1126]
NotebookOutlinePosition[     31407,       1166]
CellTagsIndexPosition[     31321,       1161]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Partitioning Data into Clusters" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/PartitioningDataIntoClusters"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Partitioning Data into Clusters\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"EuclideanDistance" :> 
          Documentation`HelpLookup["paclet:ref/EuclideanDistance"], 
          "SquaredEuclideanDistance" :> 
          Documentation`HelpLookup["paclet:ref/SquaredEuclideanDistance"], 
          "HammingDistance" :> 
          Documentation`HelpLookup["paclet:ref/HammingDistance"], 
          "ChebyshevDistance" :> 
          Documentation`HelpLookup["paclet:ref/ChebyshevDistance"], 
          "BrayCurtisDistance" :> 
          Documentation`HelpLookup["paclet:ref/BrayCurtisDistance"], 
          "CanberraDistance" :> 
          Documentation`HelpLookup["paclet:ref/CanberraDistance"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"EuclideanDistance\"\>", 
       2->"\<\"SquaredEuclideanDistance\"\>", 3->"\<\"HammingDistance\"\>", 
       4->"\<\"ChebyshevDistance\"\>", 5->"\<\"BrayCurtisDistance\"\>", 
       6->"\<\"CanberraDistance\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Exploratory Data Analysis" :> 
          Documentation`HelpLookup["paclet:guide/DataClassification"], 
          "Distance and Dissimilarity Measures" :> 
          Documentation`HelpLookup[
           "paclet:guide/DistanceAndDissimilarityMeasures"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Exploratory Data Analysis\"\>", 
       2->"\<\"Distance and Dissimilarity Measures\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       4->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["ManhattanDistance", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ManhattanDistance",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ManhattanDistance"], "[", 
       RowBox[{
        StyleBox["u", "TI"], ",", 
        StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Manhattan or \"city block\" distance between \
vectors ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->26299]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["ManhattanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ManhattanDistance"], "[", 
   RowBox[{
    StyleBox["u", "TI"], ",", 
    StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Total",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Total"], "[", 
   RowBox[{
    ButtonBox["Abs",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Abs"], "[", 
    RowBox[{
     StyleBox["u", "TI"], "-", 
     StyleBox["v", "TI"]}], "]"}], "]"}]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->709349618,
  ButtonNote->"709349618"]
}], "Notes",
 CellID->24221]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->573195194],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->181065938],

Cell["Manhattan distance between two vectors:", "ExampleText",
 CellID->235641551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ManhattanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->196650888],

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"a", "-", "x"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"b", "-", "y"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"c", "-", "z"}], "]"}]}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->281578458]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->125461358],

Cell["Manhattan distance between numeric vectors:", "ExampleText",
 CellID->319587507],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ManhattanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "4", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190454384],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->253436332]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->15589],

Cell["Compute distance between any vectors of equal length:", "ExampleText",
 CellID->86273121],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ManhattanDistance", "[", 
  RowBox[{
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", "100"}], "]"}], ",", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", "100"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->488208944],

Cell[BoxData["180.69120635017342`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->239046793]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->204991544],

Cell["Compute distance between vectors of any precision:", "ExampleText",
 CellID->14044010],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ManhattanDistance", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "5", ",", "2", ",", "3", ",", "10"}], "}"}], ",", 
     "50"}], "]"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"4", ",", "15", ",", "20", ",", "5", ",", "5"}], "}"}], ",", 
     "50"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->323525768],

Cell[BoxData["38.`49.39794000867204"], "Output",
 ImageSize->{354, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->732307862]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26031],

Cell["Cluster data using Manhattan distance:", "ExampleText",
 CellID->245478961],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindClusters", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"DistanceFunction", "\[Rule]", "ManhattanDistance"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->108189764],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"5", ",", "10"}], "}"}], "}"}]}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->21681597]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->124651963],

Cell["Demonstrate the triangle inequality:", "ExampleText",
 CellID->162085005],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d1", "=", 
  RowBox[{"ManhattanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->395493032],

Cell[BoxData[
 RowBox[{"Abs", "[", 
  RowBox[{"b", "-", "c"}], "]"}]], "Output",
 ImageSize->{66, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->258315378]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d2", "=", 
  RowBox[{"ManhattanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "c"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"d", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13424126],

Cell[BoxData[
 RowBox[{"Abs", "[", 
  RowBox[{"a", "-", "d"}], "]"}]], "Output",
 ImageSize->{66, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->48648408]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d3", "=", 
  RowBox[{"ManhattanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"d", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->51893016],

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"b", "-", "c"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"a", "-", "d"}], "]"}]}]], "Output",
 ImageSize->{140, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->712339073]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"d3", "<=", 
   RowBox[{"d1", "+", "d2"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->175054461],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->366517162]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26268],

Cell["Manhattan distance is a sum of absolute differences:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->709349618],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ManhattanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->418619957],

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"a", "-", "x"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"b", "-", "y"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"c", "-", "z"}], "]"}]}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->404939381]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Total", "[", 
  RowBox[{"Abs", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], "-", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->290219153],

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"a", "-", "x"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"b", "-", "y"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"c", "-", "z"}], "]"}]}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->17103728]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->533993557],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ManhattanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ManhattanDistance"]], "InlineFormula"],
 " is equivalent to a ",
 Cell[BoxData[
  ButtonBox["Norm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Norm"]], "InlineFormula"],
 " of a difference:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->214729278],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ManhattanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->218507111],

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"a", "-", "x"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"b", "-", "y"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"c", "-", "z"}], "]"}]}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->359739626]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], "-", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21346996],

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"a", "-", "x"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"b", "-", "y"}], "]"}], "+", 
  RowBox[{"Abs", "[", 
   RowBox[{"c", "-", "z"}], "]"}]}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->312441923]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->263875481],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ManhattanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ManhattanDistance"]], "InlineFormula"],
 " is greater than or equal to ",
 Cell[BoxData[
  ButtonBox["ChebyshevDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChebyshevDistance"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->581795014],

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", "=", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->625730726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"ManhattanDistance", "[", 
    RowBox[{"u", ",", "v"}], "]"}], "\[GreaterEqual]", 
   RowBox[{"ChebyshevDistance", "[", 
    RowBox[{"u", ",", "v"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->196694015],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->502315653]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->604768347],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BrayCurtisDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BrayCurtisDistance"]], "InlineFormula"],
 " is a ratio of Manhattan distances:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->116722353],

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", "=", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->658943477],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ManhattanDistance", "[", 
   RowBox[{"u", ",", "v"}], "]"}], "/", 
  RowBox[{"ManhattanDistance", "[", 
   RowBox[{"u", ",", 
    RowBox[{"-", "v"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->281060260],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "x"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "y"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"c", "-", "z"}], "]"}]}], 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"a", "+", "x"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"b", "+", "y"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"c", "+", "z"}], "]"}]}]]], "Output",
 ImageSize->{219, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->804848397]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"BrayCurtisDistance", "[", 
  RowBox[{"u", ",", "v"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->378344330],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "x"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "y"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"c", "-", "z"}], "]"}]}], 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"a", "+", "x"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"b", "+", "y"}], "]"}], "+", 
   RowBox[{"Abs", "[", 
    RowBox[{"c", "+", "z"}], "]"}]}]]], "Output",
 ImageSize->{219, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->12746286]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->70099947],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MeanDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MeanDeviation"]], "InlineFormula"],
 " as a scaled ",
 Cell[BoxData[
  ButtonBox["ManhattanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ManhattanDistance"]], "InlineFormula"],
 " from the ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->394865487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->463545505],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"a", ",", "b", ",", "c"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->4692422]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mean", "=", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Mean", "[", "data", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"Length", "[", "data", "]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->132468290],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{264, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->129212415]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MeanDeviation", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->917053855],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "3"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Abs", "[", 
     RowBox[{"a", "+", 
      RowBox[{
       FractionBox["1", "3"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"b", "+", 
      RowBox[{
       FractionBox["1", "3"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{
      RowBox[{
       FractionBox["1", "3"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}], "+", "c"}], 
     "]"}]}], ")"}]}]], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{468, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->26988691]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ManhattanDistance", "[", 
   RowBox[{"data", ",", "mean"}], "]"}], "/", 
  RowBox[{"Length", "[", "data", "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->42727618],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "3"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Abs", "[", 
     RowBox[{"a", "+", 
      RowBox[{
       FractionBox["1", "3"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{"b", "+", 
      RowBox[{
       FractionBox["1", "3"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], "]"}], "+", 
    RowBox[{"Abs", "[", 
     RowBox[{
      RowBox[{
       FractionBox["1", "3"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}], "+", "c"}], 
     "]"}]}], ")"}]}]], "Output",
 ImageSize->{468, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->787225579]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->16893],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["EuclideanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/EuclideanDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SquaredEuclideanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SquaredEuclideanDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["HammingDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HammingDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChebyshevDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChebyshevDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BrayCurtisDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BrayCurtisDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CanberraDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CanberraDistance"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->32525]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->1615],

Cell[TextData[ButtonBox["Partitioning Data into Clusters",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/PartitioningDataIntoClusters"]], "Tutorials",
 CellID->22256]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with ManhattanDistance",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
ManhattanDistance"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->636469],

Cell[TextData[ButtonBox["Exploratory Data Analysis",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DataClassification"]], "MoreAbout",
 CellID->67049200],

Cell[TextData[ButtonBox["Distance and Dissimilarity Measures",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DistanceAndDissimilarityMeasures"]], "MoreAbout",
 CellID->18755088],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->148710273],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->564933387]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ManhattanDistance - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 59, 25.5561728}", "context" -> "System`", 
    "keywords" -> {
     "city block distance", "grid distance", "Manhattan distance", 
      "Manhattan metric", "taxi cab distance", "taxicab distance", 
      "taxicab metric"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "ManhattanDistance[u, v] gives the Manhattan or \"city block\" distance \
between vectors u and v.", "synonyms" -> {"manhattan distance"}, "title" -> 
    "ManhattanDistance", "type" -> "Symbol", "uri" -> 
    "ref/ManhattanDistance"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7324, 215, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->573195194]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 31178, 1154}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3092, 70, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3715, 97, 1582, 37, 70, "ObjectNameGrid"],
Cell[5300, 136, 733, 24, 70, "Usage",
 CellID->26299]
}, Open  ]],
Cell[CellGroupData[{
Cell[6070, 165, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6548, 182, 739, 28, 70, "Notes",
 CellID->24221]
}, Closed]],
Cell[CellGroupData[{
Cell[7324, 215, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->573195194],
Cell[CellGroupData[{
Cell[7709, 229, 148, 5, 70, "ExampleSection",
 CellID->181065938],
Cell[7860, 236, 82, 1, 70, "ExampleText",
 CellID->235641551],
Cell[CellGroupData[{
Cell[7967, 241, 254, 8, 28, "Input",
 CellID->196650888],
Cell[8224, 251, 353, 12, 36, "Output",
 CellID->281578458]
}, Open  ]],
Cell[8592, 266, 125, 3, 70, "ExampleDelimiter",
 CellID->125461358],
Cell[8720, 271, 86, 1, 70, "ExampleText",
 CellID->319587507],
Cell[CellGroupData[{
Cell[8831, 276, 254, 8, 70, "Input",
 CellID->190454384],
Cell[9088, 286, 158, 5, 36, "Output",
 CellID->253436332]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9295, 297, 216, 7, 70, "ExampleSection",
 CellID->15589],
Cell[9514, 306, 95, 1, 70, "ExampleText",
 CellID->86273121],
Cell[CellGroupData[{
Cell[9634, 311, 266, 8, 70, "Input",
 CellID->488208944],
Cell[9903, 321, 176, 5, 36, "Output",
 CellID->239046793]
}, Open  ]],
Cell[10094, 329, 125, 3, 70, "ExampleDelimiter",
 CellID->204991544],
Cell[10222, 334, 92, 1, 70, "ExampleText",
 CellID->14044010],
Cell[CellGroupData[{
Cell[10339, 339, 427, 14, 70, "Input",
 CellID->323525768],
Cell[10769, 355, 179, 5, 36, "Output",
 CellID->732307862]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[10997, 366, 223, 7, 70, "ExampleSection",
 CellID->26031],
Cell[11223, 375, 81, 1, 70, "ExampleText",
 CellID->245478961],
Cell[CellGroupData[{
Cell[11329, 380, 477, 16, 70, "Input",
 CellID->108189764],
Cell[11809, 398, 497, 18, 36, "Output",
 CellID->21681597]
}, Open  ]],
Cell[12321, 419, 125, 3, 70, "ExampleDelimiter",
 CellID->124651963],
Cell[12449, 424, 79, 1, 70, "ExampleText",
 CellID->162085005],
Cell[CellGroupData[{
Cell[12553, 429, 263, 9, 70, "Input",
 CellID->395493032],
Cell[12819, 440, 210, 7, 36, "Output",
 CellID->258315378]
}, Open  ]],
Cell[CellGroupData[{
Cell[13066, 452, 262, 9, 70, "Input",
 CellID->13424126],
Cell[13331, 463, 209, 7, 36, "Output",
 CellID->48648408]
}, Open  ]],
Cell[CellGroupData[{
Cell[13577, 475, 262, 9, 70, "Input",
 CellID->51893016],
Cell[13842, 486, 289, 10, 36, "Output",
 CellID->712339073]
}, Open  ]],
Cell[CellGroupData[{
Cell[14168, 501, 155, 5, 70, "Input",
 CellID->175054461],
Cell[14326, 508, 161, 5, 36, "Output",
 CellID->366517162]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14536, 519, 233, 7, 70, "ExampleSection",
 CellID->26268],
Cell[14772, 528, 143, 2, 70, "ExampleText",
 CellID->709349618],
Cell[CellGroupData[{
Cell[14940, 534, 254, 8, 70, "Input",
 CellID->418619957],
Cell[15197, 544, 353, 12, 36, "Output",
 CellID->404939381]
}, Open  ]],
Cell[CellGroupData[{
Cell[15587, 561, 277, 9, 70, "Input",
 CellID->290219153],
Cell[15867, 572, 352, 12, 36, "Output",
 CellID->17103728]
}, Open  ]],
Cell[16234, 587, 125, 3, 70, "ExampleDelimiter",
 CellID->533993557],
Cell[16362, 592, 396, 13, 70, "ExampleText",
 CellID->214729278],
Cell[CellGroupData[{
Cell[16783, 609, 254, 8, 70, "Input",
 CellID->218507111],
Cell[17040, 619, 353, 12, 36, "Output",
 CellID->359739626]
}, Open  ]],
Cell[CellGroupData[{
Cell[17430, 636, 268, 9, 70, "Input",
 CellID->21346996],
Cell[17701, 647, 353, 12, 36, "Output",
 CellID->312441923]
}, Open  ]],
Cell[18069, 662, 125, 3, 70, "ExampleDelimiter",
 CellID->263875481],
Cell[18197, 667, 415, 13, 70, "ExampleText",
 CellID->581795014],
Cell[18615, 682, 281, 10, 70, "Input",
 CellID->625730726],
Cell[CellGroupData[{
Cell[18921, 696, 281, 8, 70, "Input",
 CellID->196694015],
Cell[19205, 706, 161, 5, 36, "Output",
 CellID->502315653]
}, Open  ]],
Cell[19381, 714, 125, 3, 70, "ExampleDelimiter",
 CellID->604768347],
Cell[19509, 719, 280, 8, 70, "ExampleText",
 CellID->116722353],
Cell[19792, 729, 281, 10, 70, "Input",
 CellID->658943477],
Cell[CellGroupData[{
Cell[20098, 743, 248, 8, 70, "Input",
 CellID->281060260],
Cell[20349, 753, 581, 20, 53, "Output",
 CellID->804848397]
}, Open  ]],
Cell[CellGroupData[{
Cell[20967, 778, 137, 4, 70, "Input",
 CellID->378344330],
Cell[21107, 784, 580, 20, 53, "Output",
 CellID->12746286]
}, Open  ]],
Cell[21702, 807, 124, 3, 70, "ExampleDelimiter",
 CellID->70099947],
Cell[21829, 812, 469, 17, 70, "ExampleText",
 CellID->394865487],
Cell[CellGroupData[{
Cell[22323, 833, 152, 5, 70, "Input",
 CellID->463545505],
Cell[22478, 840, 211, 7, 36, "Output",
 CellID->4692422]
}, Open  ]],
Cell[CellGroupData[{
Cell[22726, 852, 251, 8, 70, "Input",
 CellID->132468290],
Cell[22980, 862, 538, 19, 51, "Output",
 CellID->129212415]
}, Open  ]],
Cell[CellGroupData[{
Cell[23555, 886, 112, 3, 70, "Input",
 CellID->917053855],
Cell[23670, 891, 923, 33, 51, "Output",
 CellID->26988691]
}, Open  ]],
Cell[CellGroupData[{
Cell[24630, 929, 201, 6, 70, "Input",
 CellID->42727618],
Cell[24834, 937, 874, 31, 51, "Output",
 CellID->787225579]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[25769, 975, 311, 9, 70, "SeeAlsoSection",
 CellID->16893],
Cell[26083, 986, 1549, 53, 70, "SeeAlso",
 CellID->32525]
}, Open  ]],
Cell[CellGroupData[{
Cell[27669, 1044, 313, 9, 70, "TutorialsSection",
 CellID->1615],
Cell[27985, 1055, 170, 3, 70, "Tutorials",
 CellID->22256]
}, Open  ]],
Cell[CellGroupData[{
Cell[28192, 1063, 305, 8, 70, "RelatedLinksSection"],
Cell[28500, 1073, 376, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[28913, 1090, 316, 9, 70, "MoreAboutSection",
 CellID->636469],
Cell[29232, 1101, 154, 3, 70, "MoreAbout",
 CellID->67049200],
Cell[29389, 1106, 178, 3, 70, "MoreAbout",
 CellID->18755088],
Cell[29570, 1111, 179, 3, 70, "MoreAbout",
 CellID->148710273],
Cell[29752, 1116, 151, 3, 70, "MoreAbout",
 CellID->564933387]
}, Open  ]],
Cell[29918, 1122, 27, 0, 70, "History"],
Cell[29948, 1124, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

