(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     69756,       2448]
NotebookOptionsPosition[     58080,       2038]
NotebookOutlinePosition[     59530,       2077]
CellTagsIndexPosition[     59444,       2072]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Basic Matrix Operations" :> 
          Documentation`HelpLookup["paclet:tutorial/BasicMatrixOperations"], 
          "Vectors and Matrices" :> 
          Documentation`HelpLookup["paclet:tutorial/VectorsAndMatrices"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Basic Matrix Operations\"\>",
        2->"\<\"Vectors and Matrices\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Dot" :> Documentation`HelpLookup["paclet:ref/Dot"], 
          "MatrixExp" :> Documentation`HelpLookup["paclet:ref/MatrixExp"], 
          "LinearSolve" :> 
          Documentation`HelpLookup["paclet:ref/LinearSolve"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Dot\"\>", 2->"\<\"MatrixExp\"\>", 
       3->"\<\"LinearSolve\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Matrix Operations" :> 
          Documentation`HelpLookup["paclet:guide/MatrixOperations"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Matrix Operations\"\>", 
       2->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["MatrixPower", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["MatrixPower",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/MatrixPower"], "[", 
       RowBox[{
        StyleBox["m", "TI"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the ",
     Cell[BoxData[
      StyleBox[
       RowBox[{
        StyleBox["n", "TI"], "\[Null]"}]]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " matrix power of the matrix ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["MatrixPower",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/MatrixPower"], "[", 
       RowBox[{
        StyleBox["m", "TI"], ",", 
        StyleBox["n", "TI"], ",", 
        StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the  ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " matrix power of the matrix ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     " applied to the vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->14262]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixPower",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixPower"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " effectively evaluates the product of a matrix with itself ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " times. "
}], "Notes",
 CellID->21409],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is negative, ",
 Cell[BoxData[
  ButtonBox["MatrixPower",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixPower"]], "InlineFormula"],
 " finds powers of the inverse of the matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ". ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->350054922,
  ButtonNote->"350054922"]
}], "Notes",
 CellID->28607],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixPower",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixPower"]], "InlineFormula"],
 " works only on square matrices. "
}], "Notes",
 CellID->18038],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MatrixPower",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixPower"]], "InlineFormula"],
 " can be used on ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->47279091,
  ButtonNote->"47279091"]
}], "Notes",
 CellID->4360]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->694643546],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->71898446],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}]}], "}"}], ",", "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->312626573],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"4181", ",", "6765"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"6765", ",", "10946"}], "}"}]}], "}"}]], "Output",
 ImageSize->{201, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->261945118]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->615723909],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"-", "2"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->419809931],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"5", ",", 
     RowBox[{"-", "3"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "3"}], ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->321410070]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->705433],

Cell["Compute for a symbolic (assumed to be integer) power:", "ExampleText",
 CellID->328803256],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "2", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "0", ",", "0"}], "}"}]}], "}"}], ",", " ", 
   "n"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->318208870],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      FractionBox[
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "\[ImaginaryI]"}], ")"}], "n"], "2"], "+", 
      FractionBox[
       SuperscriptBox["\[ImaginaryI]", "n"], "2"]}], ",", "0", ",", 
     RowBox[{
      RowBox[{
       FractionBox["1", "2"], " ", "\[ImaginaryI]", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "\[ImaginaryI]"}], ")"}], "n"]}], "-", 
      FractionBox[
       RowBox[{"\[ImaginaryI]", " ", 
        SuperscriptBox["\[ImaginaryI]", "n"]}], "2"]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     SuperscriptBox["2", "n"], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], " ", "\[ImaginaryI]", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "\[ImaginaryI]"}], ")"}], "n"]}], "+", 
      FractionBox[
       RowBox[{"\[ImaginaryI]", " ", 
        SuperscriptBox["\[ImaginaryI]", "n"]}], "2"]}], ",", "0", ",", 
     RowBox[{
      FractionBox[
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "\[ImaginaryI]"}], ")"}], "n"], "2"], "+", 
      FractionBox[
       SuperscriptBox["\[ImaginaryI]", "n"], "2"]}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{540, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->143473489]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->126398342],

Cell["\<\
Find the second inverse matrix power applied to a particular vector:\
\>", "ExampleText",
 CellID->350054922],

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", 
      RowBox[{"10", ",", " ", "10"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", 
   RowBox[{"ConstantArray", "[", 
    RowBox[{"1.", ",", " ", "10"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->413669739],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{"m", ",", " ", 
   RowBox[{"-", "2"}], ",", " ", "b"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->800763783],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "39.558953003213475`"}], ",", 
   RowBox[{"-", "9.55795117931637`"}], ",", "35.013837830761965`", ",", 
   RowBox[{"-", "91.94232072063757`"}], ",", "25.920413603825196`", ",", 
   "52.00713672074219`", ",", 
   RowBox[{"-", "56.01491251049597`"}], ",", "25.73259377609933`", ",", 
   "31.6383158887273`", ",", "32.93365743878073`"}], "}"}]], "Output",
 ImageSize->{376, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->374667947]
}, Open  ]],

Cell[TextData[{
 "This is a more efficient way of computing ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["m", 
     RowBox[{"-", "1"}]], ".", 
    SuperscriptBox["m", 
     RowBox[{"-", "1"}]], ".", "b"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->334205820],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"minv", " ", "=", " ", 
   RowBox[{"Inverse", "[", "m", "]"}]}], ";"}], "\n", 
 RowBox[{"minv", ".", "minv", ".", "b"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->250417994],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "39.558953003210604`"}], ",", 
   RowBox[{"-", "9.557951179315943`"}], ",", "35.013837830759414`", ",", 
   RowBox[{"-", "91.94232072062945`"}], ",", "25.92041360382344`", ",", 
   "52.00713672073806`", ",", 
   RowBox[{"-", "56.01491251049137`"}], ",", "25.732593776097247`", ",", 
   "31.638315888725337`", ",", "32.93365743877696`"}], "}"}]], "Output",
 ImageSize->{376, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->30620134]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->180344676],

Cell["Use exact arithmetic to compute the matrix power:", "ExampleText",
 CellID->70436859],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"1", "/", "2"}], ",", 
       RowBox[{"1", "/", "3"}], ",", 
       RowBox[{"1", "/", "4"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", 
       RowBox[{"1", "/", "3"}], ",", 
       RowBox[{"1", "/", "4"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1", ",", 
       RowBox[{"1", "/", "4"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26382475],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{"m", ",", "10"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->117248049],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "5", ",", 
     FractionBox["65", "6"], ",", 
     FractionBox["135", "8"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", 
     FractionBox["10", "3"], ",", 
     FractionBox["25", "4"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", 
     FractionBox["5", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{410, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->109708579]
}, Open  ]],

Cell["Use machine arithmetic:", "ExampleText",
 CellID->301159253],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"N", "[", "m", "]"}], ",", " ", "4"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->11405883],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1.`", ",", "2.`", ",", "2.333333333333333`", ",", "2.416666666666666`"}],
     "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.`", ",", "1.`", ",", "1.3333333333333333`", ",", 
     "1.4999999999999998`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "1.`", ",", "1.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0.`", ",", "0.`", ",", "1.`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{578, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->306430282]
}, Open  ]],

Cell["Use 20-digit precision arithmetic:", "ExampleText",
 CellID->157396404],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{"m", ",", "20"}], "]"}], ",", "4"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->54166464],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1.`19.397940008672037", ",", "2.`19.397940008672037", ",", 
     "2.33333333333333333333333333333`19.397940008672034", ",", 
     "2.41666666666666666666666666666`19.397940008672034"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "1.`19.397940008672037", ",", 
     "1.33333333333333333333333333333333333333`19.397940008672034", ",", 
     "1.5`19.397940008672037"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "0", ",", "1.`19.397940008672037", ",", 
     "1.`19.397940008672037"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "1.`19.397940008672037"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{514, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->445148139]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12775758],

Cell["Compute the matrix power for a complex matrix:", "ExampleText",
 CellID->292734790],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", " ", "I"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"I", ",", " ", "1"}], "}"}]}], "}"}], ",", " ", "4"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->489153450],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "4"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"-", "4"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->157303236]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->207950016],

Cell["Matrix power for a sparse matrix:", "ExampleText",
 CellID->47279091],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", " ", "=", " ", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], "\[Rule]", 
       RowBox[{"-", "2"}]}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", " ", "j_"}], "}"}], "/;", 
        RowBox[{
         RowBox[{"Abs", "[", 
          RowBox[{"i", "-", "j"}], "]"}], "\[Equal]", "1"}]}], "\[Rule]", 
       "1"}]}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"7", ",", "7"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->483701800],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "19", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 19, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "7"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->857055145]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{"m", ",", "3"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->119653575],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "37", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 37, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "7"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->85976132]
}, Open  ]],

Cell["\<\
The middle row is a stencil for a second-order approximation to the sixth \
derivative:\
\>", "ExampleText",
 CellID->343639214],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", "%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->71220229],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmr9T2zAUxx3bobRAC/0JCaWUu/4r7dQ/Isf1CkOvPcqQLWPGjB7zJzBm
ZGSkQ++ykTEjY0ZXkiUsPz9j2ZJs5a7cJfKTnyV99Z71sYm+Di7Pvv0YXJ6f
Do6/XAx+nZ2f/j7+/POCVAUdz/PuyOf7vkeOY3LEPndxTEr5u0u/g+l0GrM/
7w+zR6MRtwPmFbLD+Xw+m83oIbUD0mAflD1Sdkl5wu2X3O5y+wDYD73f3t7C
3qIoksdAL4vFn8fbUq3ts2Year2PqNcr4LUB7F7J+WTcq9VqOBymagJRmwhK
vIyqEbOKzxNU5ZOvDWLuSbHAzifqxuMxGbqoT/rwb25uui3FxEe9DpW8njAv
ksIkkbOKWLYtl0s78aHKQmK+BvWs5zC18zryHomG9E6UNaR3rR0Nsv0G2KFk
YxGB5xMdJLWSpacxHSfAhjGpqmMzzq5VHVSHWO/geklLn5Sf4nS99Pns+NJ5
YW+CdvHe/v73Kpj3Ij4Fkp9sb8VZXr1Q6rVK1ppcBXtoLe67DVqELHtuXWk1
asHx7fKrQ1CPtwbVCsa5Es+yJw21GOJeu41oLOIbNY/QVgLg/Qy5Osu+ZpQ8
Rjk45g+5MZapgvywr6mMeHDEfbRVtQyspw6jI0ZF6hdKfuItZDvOUlKtV1eZ
1R4l69JR2Ie81MlpV3hYpUUdirhExaLW8KtNqm6SkrCV/UbU2eGjuCtDG3ed
VQ6i1JHsPYNqmiQgbAu+P1YnYh0SivNiRVbLZlf51LwXnU8TJBTXvVPqFa5M
68A+OJp6SteHemb0usc7c7rskw62rhGDVhkHe6ynwyW61YsMpZUJyonrDpR6
dZU9zXvtxGYo1+P2vlKvLLfWjG74f1zUslw8U6wf43RUq0W5TdLpqmued9rx
cIp6Omp2QFsusU9HF/31xQQJha22IrvKp/be/+C+laokFPFSm3+4fqmtZ21R
Tye/oTK4J8UdyplWWXfvik3WmdSY/9XuPdq6HQaaVAJ/yzrKKWuGhDY16TxH
m+Ghjjrx3ijvlqlDx2qrtKvMquI1mUwUvNJdobJXFEX39/f0kM6vTMmnwBYU
LKKiKJNoLxaLq6sreijvBZvNZnxvaLIjpXgFhHef/RUSr33LWry+vkZ2GZKJ
4xrlJwyMhFsF9YKQ5pn2mBqSL5ldoHCHLnwOd+uZ5DFlac6nynyScTDb8lxD
18OMB1Wiyy68lpLDJ5nE4sBzKI1G2a6MsvPw7VuHU8Xjz+3w5jqSbbgmdEDe
2omDPOJEB81VmUtbwC7jkCgZUfk+fa/zDyWjt9g=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{222, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->30088172]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26632],

Cell["Use a symbolic matrix with a symbolic power:", "ExampleText",
 CellID->433473940],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "a"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"b", ",", "0"}], "}"}]}], "}"}], ",", "n"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->381025762],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       FractionBox["1", "2"], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", 
           SqrtBox["a"]}], " ", 
          SqrtBox["b"]}], ")"}], "n"]}], "+", 
      RowBox[{
       FractionBox["1", "2"], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          SqrtBox["a"], " ", 
          SqrtBox["b"]}], ")"}], "n"]}]}], ",", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{
         SqrtBox["a"], " ", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", 
             SqrtBox["a"]}], " ", 
            SqrtBox["b"]}], ")"}], "n"]}], 
        RowBox[{"2", " ", 
         SqrtBox["b"]}]]}], "+", 
      FractionBox[
       RowBox[{
        SqrtBox["a"], " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SqrtBox["a"], " ", 
           SqrtBox["b"]}], ")"}], "n"]}], 
       RowBox[{"2", " ", 
        SqrtBox["b"]}]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", 
             SqrtBox["a"]}], " ", 
            SqrtBox["b"]}], ")"}], "n"], " ", 
         SqrtBox["b"]}], 
        RowBox[{"2", " ", 
         SqrtBox["a"]}]]}], "+", 
      FractionBox[
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SqrtBox["a"], " ", 
           SqrtBox["b"]}], ")"}], "n"], " ", 
        SqrtBox["b"]}], 
       RowBox[{"2", " ", 
        SqrtBox["a"]}]]}], ",", 
     RowBox[{
      RowBox[{
       FractionBox["1", "2"], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", 
           SqrtBox["a"]}], " ", 
          SqrtBox["b"]}], ")"}], "n"]}], "+", 
      RowBox[{
       FractionBox["1", "2"], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          SqrtBox["a"], " ", 
          SqrtBox["b"]}], ")"}], "n"]}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{463, 88},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->932986364]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->102445823],

Cell["\<\
Construct a rotation matrix as a limit of repeated infinitesimal \
transformations:\
\>", "ExampleText",
 CellID->20666],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"\[Theta]", "/", "n"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "\[Theta]"}], "/", "n"}], ",", "1"}], "}"}]}], "}"}], 
   ",", "n"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8522],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW81SE0EQ3vysoKIYAwloFYSIVrQELxw968mHSFFWwcHSQh5Dn4nH4hi3
Z3eS9Usn3ds72eWPKkhmp6f/pqdntufj2/jy7PuP8eX56Xjw9WL86+z89Pfg
y8+L5FGrEUWNP1EUfd6Jku+T5Jv7pZ+/yW/6Nf22k/yNk84PyWcr+Yza7nH2
c7S092PWu8f2Hi/tPZpKj2Kv2yQj6NFna/o0eslSdYEqJK8+S/UUqDos1ZZK
Lxsvp1e0C0/Rxq5er9Jc+tBGu3h/bwu62Lg4C9zYZvLwGfQitc4+9LWNC9r3
xGQf6mLjsp75J859RnO8nkMbLepmo5ucDrn2q0B80PbHDD3HB1c26mPlg/5q
Q7vJenVTsNJ7I2plk8dkMLTAwgO9ubbAC3keuNJRDwsP9L7Oiy47iGvuGNo9
06htdhTqrZMljbJ5Y2OS7q+4zpF7X0nXy+hQziJ+Ep3NqkOB66eCUjHykF6S
V58VtIZw/6hGe/LZWtJ8JHDfgHZXOQ7lYZaS5EiRrLMaqfJ5jVbtegmtNOPJ
C00mN5a1XtqhGk4/1I6oYp+0W1dXV1nmbrtMnrQzuliwEscP3fjRaHR9fT3r
/1/beBoz1cndKiCvKrsscvDETPOYvAa6WU5nuu44f1OCP0evi2+kyp98bqPV
9kweK7QdKukkq27mblqP1tI5GaW8LUj/MBd6rSlLUnzjqZ+yacul5clkVqOi
d+C17CHla5yJXiFueBqKBe7bGXe0tJyuOm5FdbXNBlZo0I+Oy9xbVCdHRW3c
OfHNHXXBs7ROKlqcb1u0UL0Jr9x2ndTQtlcZLaRbDDZUFzWhpNuiZVW2Vytd
Fy3aqm+L1VVaV3jaRp2bE75q+MKkTdH1ZtUOq5G44ur1lU6blfqqPeOD9XI8
1XEzK/ncc/U765ye7Vk7mqvzaiWiBzgP6TToAAU3Q6u1WSsxnM2bqtn2UZOv
dVM8x4YILD7zq5UuzXq1tlcrnZ99rFMecHzZ/ZCY401SOW7oH6wDldOtHDeX
0V3FhWboBMY+yqgGQv9+1v9eGL+of89pQY8Q+zD7RjFuRTJ4FIQFQXEylW5F
COCM7QbkxaMg6kdn7LvnUuTzvKT7fRsXPJ/XichI453CLH/Lj2/0OuwA8sZM
aOOCdtaJzPBoCPd8bofA84FU91iEHsBYfR2ID/qgbpQGxkNH5VWsNmuQBLg2
0KMWHujNutAZGA06L/Jv4cjrCNp8jpdG8aiI1SAwbN6gvEdNjEi0ZEdJ51ER
oVAWNqswmlBLvDvAfpQ6FLQsirJgbyUrsYrWlHSnYLGG+G4EsUKqTCE9RiSt
Ew2qYqikk+pqTr5YncOzWQirEQ1BWUK6kTwsSL+okpBGrM/O9BffItN2uJt8
9LC/owqNGEA5RZAPnEdvC8IDT/TpO33VcZ3PTVXEc/ruXXXOWrWV9eyaVPUI
iZ6oxwrUouzuWI3WiAMIjZ64T3PhsX06DACt/WV3/v1C3PA00ha4e/TDbUBq
2GYDpehuVfO3ZNQuepsq/R+FDgGQ192iBdbL7hPyo8po8XfwZefLFjWhpNui
5W4gP/hoKVrd5e/OpVxivcnX3BOXzzGhUA8PCJFlqIe0Li7999uie/j8fxF6
vXT37laJaDkXRTa0w31AePRVs83duZPVYe7c65UuzfpdRnrws4/1kwHHdyEG
AHfLctzQPzcJieHOCdN7phGM9WiEA6F/f1IOLfHOaUGPUmRD1PgHTHSmgw==
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{396, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->571695309]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Limit", "[", 
   RowBox[{"%", ",", " ", 
    RowBox[{"n", " ", "->", " ", "Infinity"}]}], "]"}], " ", "//", " ", 
  "FullSimplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->8160],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cos", "[", "\[Theta]", "]"}], ",", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"Sin", "[", "\[Theta]", "]"}]}], ",", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->17553]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->170524937],

Cell[TextData[{
 "Solve the constant coefficient system of difference equations ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"x", "(", 
     RowBox[{"n", "+", "1"}], ")"}], "\[LongEqual]", 
    RowBox[{"a", ".", 
     RowBox[{"x", "(", "n", ")"}]}]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->680839285],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "1"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->539487851],

Cell[TextData[{
 "This computes the fundamental solution ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Phi]", "(", "n", ")"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->182770929],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"\[Phi]", "[", "n_", "]"}], "=", 
  RowBox[{"MatrixPower", "[", 
   RowBox[{"A", ",", "n"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->63574845],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWd1PFDEQ32PvTgHx+DCBmKgnUUl89smvN+XJ+KoYxQsxgQejQRKjiPCm
6B/FH6bJ2em2x/Lb2ZvZ7i4c4CXX7nTbmV9nptNt51lva/3d+97Wxlqvu7zZ
+7i+sfap+/TDpmmKG1HU+BNF0cOFyDz3zZP90++v+SePydOMKVvm5QtXb5s6
NvWuq38OekUtz6Nv25Ix2dZvVMeDVsspTf+w/WahdQXoL6ZoO27tAY5JU44Z
qi2Mfu36zbm6ySLdYVu33ZhWqo6iCZBwQUDwBugO0JZr85COou+Zlh0Y0wR6
jEV2UaGbYcjSuo2cJ6TpMFS8BXgPWj0RhGQRIldc/VZAQR5A/rHn6jCpV4Ce
F6SOM6jT9NWC/DgtpOlpoGMBD87nhiB/qeL5SPxwPjorzQkoZgStkBTyEWk2
yIdbRcQH+8WuXcKN1kA+qL1Q3Hl8sJ9O+7hXUC8KGmjLCdeOXDlrDJuN548x
HnedPHls7IPIEfudZUg/jOSoBy8/FOcloFupftScDPA/XPlaLWqtVC2ado4U
3uPQQ3Dd4CicE2LXeCCthwVhnGRZSW5e/LB8M/su+qOPKuiHqB1EURY15wk5
aDOtnL01mi5q0TowSns7jmtYfmidZNTBwYFbKU27XAyt5Eo0F1X8epbG4yw6
JfH4OIBepMWDa2CKtQlynyyp1Tx0VcyGG99mZ4WjLwuzQK5V+0Y59MPOAV6a
OfradZGsjeOO69eF/lXHc96X0QpTghbqjt/ITxcVJV2WjdhhqNhzvBijpS/a
s+mHku5QC3X7Ie6Uft8dg3ak6/bEUFyoT9237qj5YlH+x+N7RVHh9yXpmmyY
HFSSH8aKqjUTgqEaH8IdBU9neO7Hryvs75FbaUfuRBfrlJTxomm2VfIt1AZi
KIUR/SGDDnf7bp0aK4xGun8rcGvaTMsiRHj3ON5P3f6jjHbaq5LyVnmuR3wV
NURzjxVrME9iOJKOk4zWWKx4xnlypvt83kZn7eWMnJeZlmuCTp9nRnzOtNwP
QufPRI9MTezmgcYoz0Vc6v/E9b/n6MeOfhCESsqQrQqocPwu0GHZjtFAFZpF
lKTzmUPcDVDKV4Hr/shxmXXze+VqP+89V/+2vWgAZpsPnzDvTG/2B2+0mcLD
UVI+mYpftqf/uvZW0noBEXctB+nuuWhumIolFeeiuV095zL5Wb0UKceq53Ta
86ZU3FTN9LTkHolYVM0oL5+HKxHXUdX5PW1mqqp8HhVdqyH8MpRszmW3hmmq
rnxU2bN6WP6JiGSt+JNsn1lx//Mew3Wvy3tQcbs/ij56MvdIqGUq7th3RaPc
2dQPF+N0O/j5uHfU6+N836Hp9XS27n708w65BdFzL3qLoedc9Mx/HJzLnwGl
kzQVujN3SE88gdMbPHVHjX8TozcI\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{507, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->307927071]
}, Open  ]],

Cell["Show that it satisfies equations and initial conditions:", "ExampleText",
 CellID->148574018],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"Thread", "[", 
   RowBox[{
    RowBox[{"\[Phi]", "[", 
     RowBox[{"n", "+", "1"}], "]"}], " ", "\[Equal]", 
    RowBox[{"a", ".", 
     RowBox[{"\[Phi]", "[", "n", "]"}]}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->30307428],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "True", ",", "True"}], "}"}]], "Output",
 ImageSize->{125, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->634187053]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"Thread", "[", 
   RowBox[{
    RowBox[{"\[Phi]", "[", "0", "]"}], "\[Equal]", 
    RowBox[{"IdentityMatrix", "[", "3", "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->301295695],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->19183858]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->328813987],

Cell["\<\
Find the matrix exponential for a matrix without a full set of eigenvectors:\
\>", "ExampleText",
 CellID->590398838],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", " ", "1", ",", " ", "1", ",", " ", "1"}], "}"}], ",", 
     " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", " ", "1", ",", " ", "1", ",", " ", "1"}], "}"}], ",", 
     " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", " ", "0", ",", " ", "1", ",", " ", "1"}], "}"}], ",", 
     " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", " ", "0", ",", " ", "0", ",", " ", "1"}], "}"}]}], 
    "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->312728311],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mp", " ", "=", " ", 
  RowBox[{"MatrixPower", "[", 
   RowBox[{"m", ",", " ", "n"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->123414079],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "n", ",", 
     RowBox[{
      FractionBox["1", "2"], " ", "n", " ", 
      RowBox[{"(", 
       RowBox[{"1", "+", "n"}], ")"}]}], ",", 
     RowBox[{
      FractionBox["1", "6"], " ", "n", " ", 
      RowBox[{"(", 
       RowBox[{"1", "+", "n"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{"2", "+", "n"}], ")"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "n", ",", 
     RowBox[{
      FractionBox["1", "2"], " ", "n", " ", 
      RowBox[{"(", 
       RowBox[{"1", "+", "n"}], ")"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "n"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{571, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->102679399]
}, Open  ]],

Cell["Compute the exponential as the power series for each term:", \
"ExampleText",
 CellID->124517736],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{"#", "/", 
       RowBox[{"n", "!"}]}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"n", ",", " ", "0", ",", " ", "Infinity"}], "}"}]}], "]"}], 
    " ", "&"}], ",", " ", "mp", ",", " ", 
   RowBox[{"{", "2", "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->284378455],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\[ExponentialE]", ",", "\[ExponentialE]", ",", 
     FractionBox[
      RowBox[{"3", " ", "\[ExponentialE]"}], "2"], ",", 
     FractionBox[
      RowBox[{"13", " ", "\[ExponentialE]"}], "6"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "\[ExponentialE]", ",", "\[ExponentialE]", ",", 
     FractionBox[
      RowBox[{"3", " ", "\[ExponentialE]"}], "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "\[ExponentialE]", ",", "\[ExponentialE]"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "\[ExponentialE]"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{404, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->128928016]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "==", " ", 
  RowBox[{"MatrixExp", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->112623608],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->227752745]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->38021800],

Cell["\<\
Get a sparse identity matrix with size, precision and data type consistent \
with an input matrix:\
\>", "ExampleText",
 CellID->207865196],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", " ", "=", " ", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"i_", ",", " ", "i_"}], "}"}], "\[Rule]", "i"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"5", ",", "5"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->44693186],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "5", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 5, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "5"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->690148407]
}, Open  ]],

Cell["Sparse identity matrix with exact values:", "ExampleText",
 CellID->365568973],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{"m", ",", "0"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->49627659],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "5", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 5, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "5"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 CellAutoOverwrite->False,
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->141250841]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", "%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->29284910],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmD1vwjAQhp2EtLTA3pFf04qBmTlClWCoWlEGNkbGjBnzEzJmzMjImDFj
RkZG9/yFTWpSEJGhwpHCna3DfuI7v8MNg/nk/SOYT8dB/20WfE2m4+/+6+cM
pjwHITSCd/CCwMfg0XeEMVj11ye/XhzHmD5oQMfL5ZKPPRrVom6e52maEpeM
PViwe8R2wPpgH8Tqm82muloUReoeJByLB3z0fPFsly6+n+U0u91usVhIGk/M
MiBGZ4DmkY5XqxVsTVx3T+Ou12vfKAWkFZJ7SEEzVJal+TORdabSyJo0SwPp
YdfjyjRtfHhrHC0NqZu6m0msK/ZU/qdfbWij/jhXoXQdPn46abXLa6ZaHfqo
tqWxNHdCU6d8RPF6+Jwbeqs6dH3lqyqemG9e+U6rDlMVbGkszW3S6JRPVbzz
buit6tD1lO+Y4jWhfJdUR/MVbGkszX+iUZVPp3j3p3xhGJ4QJft9alQURdvt
FvMs1ymesCxLRVEkSUJctUOSpinv+rG+W5PVoZ9l/ZosyzS9LPgwzmiWBvJx
0N+r9j59ozSyDiSNC1kydSZkHxfyQE+Bf7s8i5ZBil/9Z07DGp7maeS+jIbc
vDpFE7ZHo1kvHzk/uo3DBw==\
\>"], "Output",
 Evaluatable->False,
 CellAutoOverwrite->False,
 ImageSize->{87, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->809454498]
}, Open  ]],

Cell["Sparse identity matrix with machine-number values:", "ExampleText",
 CellID->38111446],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPower", "[", 
  RowBox[{
   RowBox[{"N", "[", "m", "]"}], ",", "0"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->269505908],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "5", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 5, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "5"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 CellAutoOverwrite->False,
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->8078015]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", "%", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->274230965],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmb1uwjAQx01Cqn5QaFWGjjxNqw59iAhVAqmoiDIgFkZGxow8AmNGRkZG
RkZGRkb3/IUd46aBNC5SHSnYZ5/tX+zzfzhew37rrRP2282w8dwLu61287Px
9NGDJr+EEBrC+/KIoI6hRt8hxlCqvwH59afTKaYPeqf2aDTitk+9yrS6Wq3i
OCZVYvuItXtQ3h5pV6AMoPS5fSkolsulvmoURSoLGYbFg5Q5KoX1Vuni+16N
erfbDQYDSe2LVgbOvuJg7vvcrQ8aFaMcj8ewNKl6expvsVgEVikgTCBYkhT0
JDebjf09kXGr0sgYt0sDx8Ou2x/T1HDydpWMNCRufD76u9ITayrjzLN1nZe2
r6cqpt5/nWlVc2wVqZHpo/JQ54l9s1fN0TiawmjSlJTchTq37zLNdq66dn5K
+pNy2lfS07TStpJmi31b99PROBrRalJSVUFFu1PS47x0JT1WOYtQ0iK0sjgl
zRP7v38/HY2jSadRldSkoP9PSSeTSQYvmS9VvaIo2m63mJ9yHuXU7Ss6/3q9
ns1mpKpmjuI45lnTC5yMouK1Mr2XUc/nc0MuEDaKf4tOnT/2za1Vcb6J/Kie
Yw6s0si4kjQenKatPSFx5sE50F3g3y73omyR4iDPz2lYwtg+jVyX0dzgdIUU
ZZ16s/9WUOkL+shing==\
\>"], "Output",
 Evaluatable->False,
 CellAutoOverwrite->False,
 ImageSize->{122, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]//MatrixForm=",
 CellID->718797821]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->104332963],

Cell["\<\
Inverse power iteration for the smallest eigenvalue of a sparse positive \
definite matrix:\
\>", "ExampleText",
 CellID->356320375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"n", " ", "=", " ", "100"}], ";", 
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", " ", "i_"}], "}"}], "\[Rule]", "2."}], ",", 
       RowBox[{
        RowBox[{
         RowBox[{"{", 
          RowBox[{"i_", ",", " ", "j_"}], "}"}], "/;", 
         RowBox[{
          RowBox[{"Abs", "[", 
           RowBox[{"i", "-", "j"}], "]"}], "\[Equal]", "1"}]}], "\[Rule]", 
        RowBox[{"-", "1."}]}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "n"}], "}"}], ",", " ", "0."}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->301255886],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "298", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 298, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "100"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{209, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->370401567]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"Normalize", "[", 
    RowBox[{"MatrixPower", "[", 
     RowBox[{"m", ",", " ", 
      RowBox[{"-", "20"}], ",", " ", 
      RowBox[{"RandomReal", "[", 
       RowBox[{"1", ",", "n"}], "]"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"val", " ", "=", " ", 
  RowBox[{"Norm", "[", 
   RowBox[{"m", ".", "v"}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->660841586],

Cell[BoxData["0.0009674354160238703`"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1038468673]
}, Open  ]],

Cell[TextData[{
 "Check the error in ",
 Cell[BoxData[
  RowBox[{
   RowBox[{Cell["", "InlineFormula"], 
    RowBox[{
     StyleBox["m", "TI"], ".", 
     StyleBox["v", "TI"]}]}], " ", "-", " ", 
   RowBox[{
    StyleBox["val", "TI"], " ", 
    StyleBox["v", "TI"]}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->201133892],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{
   RowBox[{"m", ".", "v"}], " ", "-", " ", 
   RowBox[{"val", " ", "v"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->66227421],

Cell[BoxData["2.2668761580187807`*^-16"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->352185467]
}, Open  ]],

Cell["Shifted inverse power iteration for the largest eigenvalue:", \
"ExampleText",
 CellID->389195041],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{"Normalize", "[", 
    RowBox[{"MatrixPower", "[", 
     RowBox[{
      RowBox[{"m", " ", "-", " ", 
       RowBox[{"4", " ", 
        RowBox[{"MatrixPower", "[", 
         RowBox[{"m", ",", "0"}], "]"}]}]}], ",", " ", 
      RowBox[{"-", "20"}], ",", " ", 
      RowBox[{"RandomReal", "[", 
       RowBox[{"1", ",", "n"}], "]"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"val", " ", "=", " ", 
  RowBox[{"Norm", "[", 
   RowBox[{"m", ".", "v"}], "]"}]}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->476834489],

Cell[BoxData["3.999032564583976`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->82926824]
}, Open  ]],

Cell[TextData[{
 "Check the error in ",
 Cell[BoxData[
  RowBox[{
   RowBox[{Cell["", "InlineFormula"], 
    RowBox[{
     StyleBox["m", "TI"], ".", 
     StyleBox["v", "TI"]}]}], " ", "-", " ", 
   RowBox[{
    StyleBox["val", "TI"], " ", 
    StyleBox["v", "TI"]}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->19708664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{
   RowBox[{"m", ".", "v"}], " ", "-", " ", 
   RowBox[{"val", " ", "v"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->597431369],

Cell[BoxData["2.0124108275421474`*^-15"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->377139958]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->317847321],

Cell["An easy way to evaluate a matrix polynomial:", "ExampleText",
 CellID->90477123],

Cell[BoxData[
 RowBox[{
  RowBox[{"mpe", "[", 
   RowBox[{"p_", ",", " ", "x_", ",", " ", "m_"}], "]"}], ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"cl", " ", "=", " ", 
      RowBox[{"CoefficientList", "[", 
       RowBox[{"p", ",", "x"}], "]"}]}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"MatrixPower", "[", 
        RowBox[{"m", ",", 
         RowBox[{"i", "-", "1"}]}], "]"}], " ", 
       RowBox[{"cl", "[", 
        RowBox[{"[", "i", "]"}], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", " ", 
        RowBox[{"Length", "[", "cl", "]"}]}], "}"}]}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->94635654],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mpe", "[", 
  RowBox[{
   RowBox[{"1", " ", "+", " ", 
    RowBox[{"2", " ", "x"}], " ", "+", " ", 
    RowBox[{"3", " ", 
     RowBox[{"x", "^", "3"}]}]}], ",", " ", "x", ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->111753507],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"114", ",", "166"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"249", ",", "363"}], "}"}]}], "}"}]], "Output",
 ImageSize->{164, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->77182896]
}, Open  ]],

Cell["Evaluate a characteristic polynomial:", "ExampleText",
 CellID->60551487],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"Reverse", "[", 
    RowBox[{"IdentityMatrix", "[", "4", "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->191748700],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", " ", "=", " ", 
  RowBox[{"CharacteristicPolynomial", "[", 
   RowBox[{"m", ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->193476587],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"2", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  SuperscriptBox["x", "4"]}]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->275255748]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mpe", "[", 
  RowBox[{"p", ",", "x", ",", "m"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->21369588],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{354, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->190176411]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->63258234],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is nonsingular, ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"MatrixPower", "[", 
    RowBox[{
     StyleBox["m", "TI"], ",", " ", 
     StyleBox["n", "TI"]}], "]"}], ".", 
   RowBox[{"MatrixPower", "[", 
    RowBox[{
     StyleBox["m", "TI"], ",", 
     RowBox[{"-", 
      StyleBox["n", "TI"]}]}], "]"}]}]], "InlineFormula"],
 " is the identity:"
}], "ExampleText",
 CellID->300670215],

Cell[BoxData[{
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", 
      RowBox[{"5", ",", "5"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"n", " ", "=", " ", 
   RowBox[{"RandomInteger", "[", 
    RowBox[{"{", 
     RowBox[{"1", ",", "9"}], "}"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->676152031],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"MatrixPower", "[", 
    RowBox[{"m", ",", "n"}], "]"}], ".", 
   RowBox[{"MatrixPower", "[", 
    RowBox[{"m", ",", 
     RowBox[{"-", "n"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->423883213],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.0000000000015916`", ",", "0", ",", "0", ",", "0", ",", "0"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0.999999999998181`", ",", "0", ",", "0", ",", "0"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1.0000000000013642`", ",", "0", ",", "0"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "1.`", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "1.0000000000009095`"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{567, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->448276079]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->11514],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Dot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Dot"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MatrixExp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixExp"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LinearSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearSolve"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->14523]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->5471],

Cell[TextData[ButtonBox["Basic Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/BasicMatrixOperations"]], "Tutorials",
 CellID->11171],

Cell[TextData[ButtonBox["Vectors and Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/VectorsAndMatrices"]], "Tutorials",
 CellID->39801072]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->513445982],

Cell[TextData[ButtonBox["Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatrixOperations"]], "MoreAbout",
 CellID->477429409],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->85563818]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
MatrixPower"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell["New in 2  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"MatrixPower - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 0, 4.7915522}", "context" -> "System`", 
    "keywords" -> {
     "fundamental solution of difference equations", "inverse matrix power", 
      "matrix function", "powers of matrices"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "MatrixPower[m, n] gives the n\\[Null]\\[Null]^th matrix power of the \
matrix m. MatrixPower[m, n, v] gives the n\\[Null]^th matrix power of the \
matrix m applied to the vector v.", "synonyms" -> {"matrix power"}, "title" -> 
    "MatrixPower", "type" -> "Symbol", "uri" -> "ref/MatrixPower"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10770, 306, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->694643546]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 59300, 2065}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2300, 54, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2923, 81, 4195, 83, 70, "ObjectNameGrid"],
Cell[7121, 166, 1564, 52, 70, "Usage",
 CellID->14262]
}, Open  ]],
Cell[CellGroupData[{
Cell[8722, 223, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[9200, 240, 399, 14, 70, "Notes",
 CellID->21409],
Cell[9602, 256, 480, 18, 70, "Notes",
 CellID->28607],
Cell[10085, 276, 205, 7, 70, "Notes",
 CellID->18038],
Cell[10293, 285, 440, 16, 70, "Notes",
 CellID->4360]
}, Closed]],
Cell[CellGroupData[{
Cell[10770, 306, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->694643546],
Cell[CellGroupData[{
Cell[11155, 320, 147, 5, 70, "ExampleSection",
 CellID->71898446],
Cell[CellGroupData[{
Cell[11327, 329, 286, 10, 28, "Input",
 CellID->312626573],
Cell[11616, 341, 315, 11, 36, "Output",
 CellID->261945118]
}, Open  ]],
Cell[11946, 355, 125, 3, 70, "ExampleDelimiter",
 CellID->615723909],
Cell[CellGroupData[{
Cell[12096, 362, 304, 11, 28, "Input",
 CellID->419809931],
Cell[12403, 375, 344, 13, 36, "Output",
 CellID->321410070]
}, Open  ]],
Cell[12762, 391, 122, 3, 70, "ExampleDelimiter",
 CellID->705433],
Cell[12887, 396, 96, 1, 70, "ExampleText",
 CellID->328803256],
Cell[CellGroupData[{
Cell[13008, 401, 410, 14, 70, "Input",
 CellID->318208870],
Cell[13421, 417, 1475, 48, 52, "Output",
 CellID->143473489]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14945, 471, 220, 7, 70, "ExampleSection",
 CellID->126398342],
Cell[15168, 480, 119, 3, 70, "ExampleText",
 CellID->350054922],
Cell[15290, 485, 384, 12, 70, "Input",
 CellID->413669739],
Cell[CellGroupData[{
Cell[15699, 501, 169, 5, 70, "Input",
 CellID->800763783],
Cell[15871, 508, 539, 13, 54, "Output",
 CellID->374667947]
}, Open  ]],
Cell[16425, 524, 298, 11, 70, "ExampleText",
 CellID->334205820],
Cell[CellGroupData[{
Cell[16748, 539, 209, 6, 70, "Input",
 CellID->250417994],
Cell[16960, 547, 541, 13, 54, "Output",
 CellID->30620134]
}, Open  ]],
Cell[17516, 563, 125, 3, 70, "ExampleDelimiter",
 CellID->180344676],
Cell[17644, 568, 91, 1, 70, "ExampleText",
 CellID->70436859],
Cell[17738, 571, 651, 21, 70, "Input",
 CellID->26382475],
Cell[CellGroupData[{
Cell[18414, 596, 131, 4, 70, "Input",
 CellID->117248049],
Cell[18548, 602, 625, 20, 51, "Output",
 CellID->109708579]
}, Open  ]],
Cell[19188, 625, 66, 1, 70, "ExampleText",
 CellID->301159253],
Cell[CellGroupData[{
Cell[19279, 630, 163, 5, 70, "Input",
 CellID->11405883],
Cell[19445, 637, 618, 20, 36, "Output",
 CellID->306430282]
}, Open  ]],
Cell[20078, 660, 77, 1, 70, "ExampleText",
 CellID->157396404],
Cell[CellGroupData[{
Cell[20180, 665, 184, 6, 70, "Input",
 CellID->54166464],
Cell[20367, 673, 845, 24, 105, "Output",
 CellID->445148139]
}, Open  ]],
Cell[21227, 700, 124, 3, 70, "ExampleDelimiter",
 CellID->12775758],
Cell[21354, 705, 89, 1, 70, "ExampleText",
 CellID->292734790],
Cell[CellGroupData[{
Cell[21468, 710, 308, 11, 70, "Input",
 CellID->489153450],
Cell[21779, 723, 344, 13, 36, "Output",
 CellID->157303236]
}, Open  ]],
Cell[22138, 739, 125, 3, 70, "ExampleDelimiter",
 CellID->207950016],
Cell[22266, 744, 75, 1, 70, "ExampleText",
 CellID->47279091],
Cell[CellGroupData[{
Cell[22366, 749, 631, 21, 70, "Input",
 CellID->483701800],
Cell[23000, 772, 481, 17, 36, "Output",
 CellID->857055145]
}, Open  ]],
Cell[CellGroupData[{
Cell[23518, 794, 130, 4, 70, "Input",
 CellID->119653575],
Cell[23651, 800, 480, 17, 36, "Output",
 CellID->85976132]
}, Open  ]],
Cell[24146, 820, 138, 4, 70, "ExampleText",
 CellID->343639214],
Cell[CellGroupData[{
Cell[24309, 828, 105, 3, 70, "Input",
 CellID->71220229],
Cell[24417, 833, 1545, 29, 70, "Output",
 Evaluatable->False,
 CellID->30088172]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26011, 868, 239, 7, 70, "ExampleSection",
 CellID->26632],
Cell[26253, 877, 87, 1, 70, "ExampleText",
 CellID->433473940],
Cell[CellGroupData[{
Cell[26365, 882, 285, 10, 70, "Input",
 CellID->381025762],
Cell[26653, 894, 2306, 89, 109, "Output",
 CellID->932986364]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29008, 989, 227, 7, 70, "ExampleSection",
 CellID->102445823],
Cell[29238, 998, 130, 4, 70, "ExampleText",
 CellID->20666],
Cell[CellGroupData[{
Cell[29393, 1006, 378, 14, 70, "Input",
 CellID->8522],
Cell[29774, 1022, 1981, 37, 70, "Output",
 Evaluatable->False,
 CellID->571695309]
}, Open  ]],
Cell[CellGroupData[{
Cell[31792, 1064, 216, 7, 70, "Input",
 CellID->8160],
Cell[32011, 1073, 480, 16, 36, "Output",
 CellID->17553]
}, Open  ]],
Cell[32506, 1092, 125, 3, 70, "ExampleDelimiter",
 CellID->170524937],
Cell[32634, 1097, 337, 11, 70, "ExampleText",
 CellID->680839285],
Cell[32974, 1110, 358, 12, 70, "Input",
 CellID->539487851],
Cell[33335, 1124, 201, 7, 70, "ExampleText",
 CellID->182770929],
Cell[CellGroupData[{
Cell[33561, 1135, 187, 6, 70, "Input",
 CellID->63574845],
Cell[33751, 1143, 1643, 31, 70, "Output",
 Evaluatable->False,
 CellID->307927071]
}, Open  ]],
Cell[35409, 1177, 99, 1, 70, "ExampleText",
 CellID->148574018],
Cell[CellGroupData[{
Cell[35533, 1182, 294, 9, 70, "Input",
 CellID->30307428],
Cell[35830, 1193, 223, 7, 36, "Output",
 CellID->634187053]
}, Open  ]],
Cell[CellGroupData[{
Cell[36090, 1205, 246, 7, 70, "Input",
 CellID->301295695],
Cell[36339, 1214, 160, 5, 36, "Output",
 CellID->19183858]
}, Open  ]],
Cell[36514, 1222, 125, 3, 70, "ExampleDelimiter",
 CellID->328813987],
Cell[36642, 1227, 127, 3, 70, "ExampleText",
 CellID->590398838],
Cell[36772, 1232, 579, 18, 70, "Input",
 CellID->312728311],
Cell[CellGroupData[{
Cell[37376, 1254, 170, 5, 70, "Input",
 CellID->123414079],
Cell[37549, 1261, 913, 29, 51, "Output",
 CellID->102679399]
}, Open  ]],
Cell[38477, 1293, 103, 2, 70, "ExampleText",
 CellID->124517736],
Cell[CellGroupData[{
Cell[38605, 1299, 391, 13, 70, "Input",
 CellID->284378455],
Cell[38999, 1314, 802, 23, 51, "Output",
 CellID->128928016]
}, Open  ]],
Cell[CellGroupData[{
Cell[39838, 1342, 139, 4, 70, "Input",
 CellID->112623608],
Cell[39980, 1348, 161, 5, 36, "Output",
 CellID->227752745]
}, Open  ]],
Cell[40156, 1356, 124, 3, 70, "ExampleDelimiter",
 CellID->38021800],
Cell[40283, 1361, 149, 4, 70, "ExampleText",
 CellID->207865196],
Cell[CellGroupData[{
Cell[40457, 1369, 310, 10, 70, "Input",
 CellID->44693186],
Cell[40770, 1381, 479, 17, 36, "Output",
 CellID->690148407]
}, Open  ]],
Cell[41264, 1401, 84, 1, 70, "ExampleText",
 CellID->365568973],
Cell[CellGroupData[{
Cell[41373, 1406, 129, 4, 70, "Input",
 CellID->49627659],
Cell[41505, 1412, 506, 18, 36, "Output",
 CellID->141250841]
}, Open  ]],
Cell[CellGroupData[{
Cell[42048, 1435, 105, 3, 70, "Input",
 CellID->29284910],
Cell[42156, 1440, 884, 19, 70, "Output",
 Evaluatable->False,
 CellID->809454498]
}, Open  ]],
Cell[43055, 1462, 92, 1, 70, "ExampleText",
 CellID->38111446],
Cell[CellGroupData[{
Cell[43172, 1467, 159, 5, 70, "Input",
 CellID->269505908],
Cell[43334, 1474, 504, 18, 36, "Output",
 CellID->8078015]
}, Open  ]],
Cell[CellGroupData[{
Cell[43875, 1497, 106, 3, 70, "Input",
 CellID->274230965],
Cell[43984, 1502, 942, 20, 70, "Output",
 Evaluatable->False,
 CellID->718797821]
}, Open  ]],
Cell[44941, 1525, 125, 3, 70, "ExampleDelimiter",
 CellID->104332963],
Cell[45069, 1530, 142, 4, 70, "ExampleText",
 CellID->356320375],
Cell[CellGroupData[{
Cell[45236, 1538, 711, 22, 70, "Input",
 CellID->301255886],
Cell[45950, 1562, 487, 17, 36, "Output",
 CellID->370401567]
}, Open  ]],
Cell[CellGroupData[{
Cell[46474, 1584, 437, 13, 70, "Input",
 CellID->660841586],
Cell[46914, 1599, 180, 5, 36, "Output",
 CellID->1038468673]
}, Open  ]],
Cell[47109, 1607, 332, 13, 70, "ExampleText",
 CellID->201133892],
Cell[CellGroupData[{
Cell[47466, 1624, 182, 6, 70, "Input",
 CellID->66227421],
Cell[47651, 1632, 181, 5, 36, "Output",
 CellID->352185467]
}, Open  ]],
Cell[47847, 1640, 104, 2, 70, "ExampleText",
 CellID->389195041],
Cell[CellGroupData[{
Cell[47976, 1646, 576, 17, 70, "Input",
 CellID->476834489],
Cell[48555, 1665, 174, 5, 36, "Output",
 CellID->82926824]
}, Open  ]],
Cell[48744, 1673, 331, 13, 70, "ExampleText",
 CellID->19708664],
Cell[CellGroupData[{
Cell[49100, 1690, 183, 6, 70, "Input",
 CellID->597431369],
Cell[49286, 1698, 181, 5, 36, "Output",
 CellID->377139958]
}, Open  ]],
Cell[49482, 1706, 125, 3, 70, "ExampleDelimiter",
 CellID->317847321],
Cell[49610, 1711, 86, 1, 70, "ExampleText",
 CellID->90477123],
Cell[49699, 1714, 744, 23, 70, "Input",
 CellID->94635654],
Cell[CellGroupData[{
Cell[50468, 1741, 427, 14, 70, "Input",
 CellID->111753507],
Cell[50898, 1757, 309, 11, 36, "Output",
 CellID->77182896]
}, Open  ]],
Cell[51222, 1771, 79, 1, 70, "ExampleText",
 CellID->60551487],
Cell[51304, 1774, 198, 6, 70, "Input",
 CellID->191748700],
Cell[CellGroupData[{
Cell[51527, 1784, 177, 5, 70, "Input",
 CellID->193476587],
Cell[51707, 1791, 262, 9, 36, "Output",
 CellID->275255748]
}, Open  ]],
Cell[CellGroupData[{
Cell[52006, 1805, 131, 4, 70, "Input",
 CellID->21369588],
Cell[52140, 1811, 500, 15, 36, "Output",
 CellID->190176411]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[52689, 1832, 236, 7, 70, "ExampleSection",
 CellID->63258234],
Cell[52928, 1841, 466, 18, 70, "ExampleText",
 CellID->300670215],
Cell[53397, 1861, 396, 13, 70, "Input",
 CellID->676152031],
Cell[CellGroupData[{
Cell[53818, 1878, 272, 9, 70, "Input",
 CellID->423883213],
Cell[54093, 1889, 722, 21, 36, "Output",
 CellID->448276079]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[54876, 1917, 311, 9, 70, "SeeAlsoSection",
 CellID->11514],
Cell[55190, 1928, 694, 26, 70, "SeeAlso",
 CellID->14523]
}, Open  ]],
Cell[CellGroupData[{
Cell[55921, 1959, 313, 9, 70, "TutorialsSection",
 CellID->5471],
Cell[56237, 1970, 155, 3, 70, "Tutorials",
 CellID->11171],
Cell[56395, 1975, 152, 3, 70, "Tutorials",
 CellID->39801072]
}, Open  ]],
Cell[CellGroupData[{
Cell[56584, 1983, 319, 9, 70, "MoreAboutSection",
 CellID->513445982],
Cell[56906, 1994, 145, 3, 70, "MoreAbout",
 CellID->477429409],
Cell[57054, 1999, 197, 4, 70, "MoreAbout",
 CellID->85563818]
}, Open  ]],
Cell[CellGroupData[{
Cell[57288, 2008, 305, 8, 70, "RelatedLinksSection"],
Cell[57596, 2018, 389, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[58000, 2034, 50, 0, 70, "History"],
Cell[58053, 2036, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

