(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    945784,      21448]
NotebookOptionsPosition[    901031,      19868]
NotebookOutlinePosition[    903600,      19924]
CellTagsIndexPosition[    903512,      19919]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numerical Differential Equations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalDifferentialEquations"], 
          "Numerical Solution of Differential Equations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalSolutionOfDifferentialEquations"], 
          "Advanced Numerical Differential Equation Solving in Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/NDSolveOverview"], 
          "Numerical Mathematics in Mathematica" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalMathematicsInMathematica"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Numerical Differential Equations\"\>", 
       2->"\<\"Numerical Solution of Differential Equations\"\>", 
       3->"\<\"Advanced Numerical Differential Equation Solving in \
Mathematica\"\>", 
       4->"\<\"Numerical Mathematics in Mathematica\"\>"}, \
"\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"DSolve" :> Documentation`HelpLookup["paclet:ref/DSolve"], 
          "NIntegrate" :> Documentation`HelpLookup["paclet:ref/NIntegrate"], 
          "NSolve" :> Documentation`HelpLookup["paclet:ref/NSolve"]}, #, 2]& ,
         Evaluator -> Automatic], {1->"\<\"DSolve\"\>", 
       2->"\<\"NIntegrate\"\>", 3->"\<\"NSolve\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Calculus" :> Documentation`HelpLookup["paclet:guide/Calculus"],
           "Differential Equations" :> 
          Documentation`HelpLookup["paclet:guide/DifferentialEquations"], 
          "Equation Solving" :> 
          Documentation`HelpLookup["paclet:guide/EquationSolving"], 
          "Numerical Evaluation & Precision" :> 
          Documentation`HelpLookup[
           "paclet:guide/NumericalEvaluationAndPrecision"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Calculus\"\>", 
       2->"\<\"Differential Equations\"\>", 3->"\<\"Equation Solving\"\>", 
       4->"\<\"Numerical Evaluation & Precision\"\>", 
       5->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["NDSolve", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NDSolve",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NDSolve"], "[", 
       RowBox[{
        StyleBox["eqns", "TI"], ",", 
        StyleBox["y", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds a numerical solution to the ordinary differential \
equations ",
     Cell[BoxData[
      StyleBox["eqns", "TI"]], "InlineFormula"],
     " for the function ",
     Cell[BoxData[
      StyleBox["y", "TI"]], "InlineFormula"],
     " with the independent variable ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " in the range ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["min", "TI"]]], "InlineFormula"],
     " to ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["max", "TI"]]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NDSolve",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NDSolve"], "[", 
       RowBox[{
        StyleBox["eqns", "TI"], ",", 
        StyleBox["y", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["max", "TI"]]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["t", "TI"], ",", 
          SubscriptBox[
           StyleBox["t", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["t", "TI"], 
           StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds a numerical solution to the partial differential \
equations ",
     Cell[BoxData[
      StyleBox["eqns", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NDSolve",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NDSolve"], "[", 
       RowBox[{
        StyleBox["eqns", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["y", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["y", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds numerical solutions for the functions\
\[NonBreakingSpace]",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["y", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->16552]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " gives results in terms of ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->15874],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["NDSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NDSolve"], "[", 
   RowBox[{
    StyleBox["eqns", "TI"], ",", 
    RowBox[{
     StyleBox["y", "TI"], "[", 
     StyleBox["x", "TI"], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["min", "TI"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
 " gives solutions for ",
 Cell[BoxData[
  RowBox[{
   StyleBox["y", "TI"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " rather than for the function ",
 Cell[BoxData[
  StyleBox["y", "TI"]], "InlineFormula"],
 " itself. "
}], "Notes",
 CellID->16977],

Cell[TextData[{
 "Differential equations must be stated in terms of derivatives such as ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["y", "TI"], "'"}], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 ", obtained with ",
 Cell[BoxData[
  ButtonBox["D",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/D"]], "InlineFormula"],
 ", not total derivatives obtained with ",
 Cell[BoxData[
  ButtonBox["Dt",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Dt"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->13697],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " solves a wide range of ordinary differential equations as well as many \
partial differential equations. "
}], "Notes",
 CellID->15370],

Cell[TextData[{
 "In ordinary differential equations the functions ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["y", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " must depend only on the single variable ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ". In partial differential equations they may depend on more than one \
variable. "
}], "Notes",
 CellID->9941],

Cell[TextData[{
 "The differential equations must contain enough initial or boundary \
conditions to determine the solutions for the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["y", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " completely. "
}], "Notes",
 CellID->20267],

Cell[TextData[{
 "Initial and boundary conditions are typically stated in the form ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["y", "TI"], "[", 
    SubscriptBox[
     StyleBox["x", "TI"], 
     StyleBox["0", "TR"]], "]"}], "==", 
   SubscriptBox[
    StyleBox["c", "TI"], 
    StyleBox["0", "TR"]]}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{
     StyleBox["y", "TI"], "'"}], "[", 
    SubscriptBox[
     StyleBox["x", "TI"], 
     StyleBox["0", "TR"]], "]"}], "==", 
   SubscriptBox[
    StyleBox["dc", "TI"], 
    StyleBox["0", "TR"]]}]], "InlineFormula"],
 ", etc., but may consist of more complicated equations. "
}], "Notes",
 CellID->12341],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["c", "TI"], 
   StyleBox["0", "TR"]]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["dc", "TI"], 
   StyleBox["0", "TR"]]], "InlineFormula"],
 ", etc. can be lists, specifying that ",
 Cell[BoxData[
  RowBox[{
   StyleBox["y", "TI"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " is a function with vector or general list values. "
}], "Notes",
 CellID->617],

Cell[TextData[{
 "Periodic boundary conditions can be specified using ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["y", "TI"], "[", 
    SubscriptBox[
     StyleBox["x", "TI"], 
     StyleBox["0", "TR"]], "]"}], "==", 
   RowBox[{
    StyleBox["y", "TI"], "[", 
    SubscriptBox[
     StyleBox["x", "TI"], 
     StyleBox["1", "TR"]], "]"}]}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->24968],

Cell[TextData[{
 "The point ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["0", "TR"]]], "InlineFormula"],
 " that appears in the initial or boundary conditions need not lie in the \
range ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["min", "TI"]]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["max", "TI"]]], "InlineFormula"],
 " over which the solution is sought. "
}], "Notes",
 CellID->25065],

Cell[TextData[{
 "The differential equations in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " can involve complex numbers. "
}], "Notes",
 CellID->27321],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " can solve many differential\[Hyphen]algebraic equations, in which some of \
the ",
 Cell[BoxData[
  StyleBox["eqns", "TI"]], "InlineFormula"],
 " are purely algebraic, or some of the variables are implicitly algebraic. "
}], "Notes",
 CellID->18655],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["y", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " can be functions of the dependent variables, and need not include all such \
variables. "
}], "Notes",
 CellID->16727],

Cell["The following options can be given: ", "Notes",
 CellID->26914],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "digits of absolute accuracy sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["DependentVariables",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/DependentVariables"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the list of all dependent variables ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["EvaluationMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/EvaluationMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "expression to evaluate whenever the function is evaluated ", 
     "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    ButtonBox["InterpolationOrder",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/InterpolationOrder"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the continuity degree of the final output ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["MaxStepFraction",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxStepFraction"], 
    RowBox[{"1", "/", "10"}], Cell[
    "maximum fraction of range to cover in each step ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["MaxSteps",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxSteps"], "10000", Cell[
    "maximum number of steps to take ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["MaxStepSize",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxStepSize"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum size of each step ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["NormFunction",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/NormFunction"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the norm to use for error estimation ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "digits of precision sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["StartingStepSize",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/StartingStepSize"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "initial step size used ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["StepMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/StepMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "expression to evaluate when a step is taken ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "precision to use in internal computations ", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "3ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], None, None, None, None, None, None, 
     None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, True, 
      True, True}}}},
 CellID->8283],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " adapts its step size so that the estimated error in the solution is just \
within the tolerances specified by ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->2137],

Cell[TextData[{
 "The option ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["NormFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NormFunction"], "->", 
   StyleBox["f", "TI"]}]], "InlineFormula"],
 " specifies that the estimated errors for each of the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["y", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " should be combined using ",
 Cell[BoxData[
  RowBox[{
   StyleBox["f", "TI"], "[", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox[
      StyleBox["e", "TI"], 
      StyleBox["1", "TR"]], ",", 
     SubscriptBox[
      StyleBox["e", "TI"], 
      StyleBox["2", "TR"]], ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->6932],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " effectively specifies the absolute local error allowed at each step in \
finding a solution, while ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " specifies the relative local error. "
}], "Notes",
 CellID->26549],

Cell[TextData[{
 "If solutions must be followed accurately when their values are close to \
zero, ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " should be set larger, or to ",
 Cell[BoxData[
  ButtonBox["Infinity",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Infinity"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->1162],

Cell[TextData[{
 "The default setting of ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["WorkingPrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WorkingPrecision"], "/", "2"}]], 
  "InlineFormula"],
 ". "
}], "Notes",
 CellID->12882],

Cell[TextData[{
 "The setting for ",
 Cell[BoxData[
  ButtonBox["MaxStepFraction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxStepFraction"]], "InlineFormula"],
 " specifies the maximum step to be taken by ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " as a fraction of the range of values for each independent variable. "
}], "Notes",
 CellID->14035],

Cell[TextData[{
 "With ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["DependentVariables",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DependentVariables"], "->", 
   ButtonBox["Automatic",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Automatic"]}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " attempts to determine the dependent variables by analyzing the equations \
given. "
}], "Notes",
 CellID->28592],

Cell[TextData[{
 "Possible explicit settings for the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option include: "
}], "Notes",
 CellID->25135],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Adams\>\"", Cell["\<\
predictor\[Hyphen]corrector Adams method with orders 1 through 12 \
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<BDF\>\"", Cell["\<\
implicit backward differentiation formulas with orders 1 through 5 \
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExplicitRungeKutta\>\"", Cell["\<\
adaptive embedded pairs of 2(1) through 9(8) Runge\[Hyphen]Kutta methods \
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ImplicitRungeKutta\>\"", Cell["\<\
families of arbitrary\[Hyphen]order implicit Runge\[Hyphen]Kutta methods \
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    StyleBox["\"\<SymplecticPartitionedRungeKutta\>\"",
     GridElementStyleOptions -> {ColumnAlignments -> Left}], 
    "\[SpanFromLeft]"},
   {Cell["      ", "TableRowIcon"], "", Cell["\<\
interleaved Runge\[Hyphen]Kutta methods for separable Hamiltonian systems \
\>", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, False, True}}}},
 CellID->8930],

Cell[TextData[{
 "With ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   RowBox[{"{", 
    RowBox[{"\"\<\!\(\*
StyleBox[\"controller\", \"TI\"]\)\>\"", ",", 
     RowBox[{
      ButtonBox["Method",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Method"], "->", "\"\<\!\(\*
StyleBox[\"submethod\", \"TI\"]\)\>\""}]}], "}"}]}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   RowBox[{"{", 
    RowBox[{"\"\<\!\(\*
StyleBox[\"controller\", \"TI\"]\)\>\"", ",", 
     RowBox[{
      ButtonBox["Method",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Method"], "->", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["m", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]}]}], "}"}]}]], 
  "InlineFormula"],
 " possible controller methods include: "
}], "Notes",
 CellID->2534],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Composition\>\"", Cell[
    "compose a list of submethods ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DoubleStep\>\"", Cell["\<\
adapt step size by the double\[Hyphen]step method \
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<EventLocator\>\"", Cell[
    "respond to specified events ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Extrapolation\>\"", Cell[
    "adapt order and step size using polynomial extrapolation ", 
     "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FixedStep\>\"", Cell[
    "use a constant step size ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<OrthogonalProjection\>\"", Cell[
    "project solutions to fulfill orthogonal constraints ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Projection\>\"", Cell[
    "project solutions to fulfill general constraints ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Splitting\>\"", Cell[
    "split equations and use different submethods ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StiffnessSwitching\>\"", Cell["\<\
switch from explicit to implicit methods if stiffness is detected \
\>", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True, True}}}},
 CellID->17520],

Cell["Methods used mainly as submethods include: ", "Notes",
 CellID->17728],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<ExplicitEuler\>\"", Cell[
    "forward Euler method ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExplicitMidpoint\>\"", Cell[
    "midpoint rule method ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ExplicitModifiedMidpoint\>\"", Cell[
    "midpoint rule method with Gragg smoothing ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<LinearlyImplicitEuler\>\"", Cell[
    "linearly implicit Euler method ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<LinearlyImplicitMidpoint\>\"", Cell[
    "linearly implicit midpoint rule method ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    StyleBox["\"\<LinearlyImplicitModifiedMidpoint\>\"",
     GridElementStyleOptions -> {ColumnAlignments -> Left}], 
    "\[SpanFromLeft]"},
   {Cell["      ", "TableRowIcon"], "", Cell["\<\
linearly implicit Bader\[Hyphen]smoothed midpoint rule method \
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<LocallyExact\>\"", Cell[
    "numerical approximation to locally exact symbolic solution ", 
     "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, False, True, True}}}},
 CellID->29688],

Cell[TextData[{
 "The setting ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["InterpolationOrder",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InterpolationOrder"], "->", 
   ButtonBox["All",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/All"]}]], "InlineFormula"],
 " specifies that ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " should generate solutions that use interpolation of the same order as the \
underlying method used.",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->124638342]
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->323647842]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->397145522],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(4)", "ExampleCount"]
}], "ExampleSection",
 CellID->68938457],

Cell["Solve a first-order ordinary differential equation:", "ExampleText",
 CellID->28264],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
       RowBox[{
        RowBox[{"y", "[", "x", "]"}], 
        RowBox[{"Cos", "[", 
         RowBox[{"x", "+", 
          RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "30"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5423],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "30.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->471562520]
}, Open  ]],

Cell["Use the solution in a plot:", "ExampleText",
 CellID->4486],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "/.", "s"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->28908],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->13675044]
}, Open  ]],

Cell["Use the function and its derivative in a plot:", "ExampleText",
 CellID->8631],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], ",", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}]}], "}"}], "/.", "s"}], "]"}], ",",
    
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1565],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->29432243]
}, Open  ]],

Cell["Find specific values:", "ExampleText",
 CellID->4362],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "10.5", "]"}], ",", 
    RowBox[{
     RowBox[{"y", "'"}], "[", "12.5", "]"}]}], "}"}], "/.", "s"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->563],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"0.046930201372603834`", ",", "0.11379779267554291`"}], "}"}], 
  "}"}]], "Output",
 ImageSize->{163, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->136858501]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14419],

Cell["Second-order nonlinear ordinary differential equation:", "ExampleText",
 CellID->31851],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "''"}], "[", "x", "]"}], "+", " ", 
        RowBox[{
         RowBox[{"Sin", "[", 
          RowBox[{"y", "[", "x", "]"}], "]"}], 
         RowBox[{"y", "[", "x", "]"}]}]}], "==", "0"}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "==", "1"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", "y", 
    ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "30"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8741],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "30.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->147730850]
}, Open  ]],

Cell["Plot the function and its first two derivatives:", "ExampleText",
 CellID->13076],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], ",", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}], ",", 
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}]}], "}"}], "/.", "s"}], "]"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
   RowBox[{"PlotStyle", "->", "Automatic"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21168],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->339716780]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30098],

Cell["System of ordinary differential equations:", "ExampleText",
 CellID->19587],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "t", "]"}], "==", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{"y", "[", "t", "]"}]}], "-", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], "^", "2"}]}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], "==", 
       RowBox[{
        RowBox[{"2", 
         RowBox[{"x", "[", "t", "]"}]}], "-", 
        RowBox[{
         RowBox[{"y", "[", "t", "]"}], "^", "3"}]}]}], ",", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], "==", 
       RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "20"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20531],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->572529817]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", 
       RowBox[{"y", "[", "t", "]"}]}], "}"}], "/.", "s"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->8339],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 211},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->32042046]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26830],

Cell["This solves the heat equation in 1 dimension:", "ExampleText",
 CellID->9613550],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}], "==", 
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "==", "0"}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0"}], "]"}], "==", 
      RowBox[{"Sin", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "5"}], "]"}], "==", "0"}]}], "}"}], ",", "u", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20781],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "5.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{380, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->793882145]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"t", ",", "x"}], "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2900],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNfXd4Y9d1JwKQ06SR5BJbshOnbvpIcYrj1C+bZJNttB07cRw7sS2LsmU5
tmxZsmX1MuplZjRN03vvnTMsw947QRCVANELCRAECBbg9+2ec+59IAdDyXK+
/WPnu/PwcF+793fqPec8ouLeRx64/7v3PvLt++795X94+N7vP/Dt+374y3//
0MPUZfk5k8l0kf5vuNNE+/+H9uT/RQB04Kbto7S10Alm+jTT56el90e8LaOv
kH/6uMmy0PPH/5+ft0zPZ01xlvzNoj9X3HT978nH2v9H/c/ytrzYrw7dNMbf
kf6nNAVMlvvvv18dLFJDjf4J/Y17+WGflN7H9HVli8ag7vDJGyi7Up5KR/Qo
fl8+f6jveatcrY6qE+6Rj+/rZ61+l+Pf09d/QGZaPK6fcbd8fkeP4EOLRmDM
bjEOvytnf1vf8SPv6+xv6nvfeeP4dK+iwt/ob7+k7/zxkjuXldz5t+Wq/1bS
e5e+y8f1XX72O3xE3+ETS9yhvOTc31ryDh9eNBNF62YbWux0mxUlZ/6mXP+3
+knGdb9avK7Fhtb3vm5x7wf0fX4dBj+3DqHVhja+heashbN/Q+7xd/qa2/Sz
pXcZn9U2hDYb2u2KS0uv/i9y9ef11YLDcrlqEO1WtNvQoS8U6eVz1Jg/q5/z
O8Ur2gfRYUWHDZ0OdLn4olUlz5L5mCr0fYRbVvLRjn50DKJzGF0j6HKg262f
WHr9r8n1/1M/+e7i9Z0D6BpC1zC67eh2oue9r/8Hff3vF6/v6kfXILqt6Lah
x45eJ3rlFuYl7vGrRU4xNNvio6I5buHv3QPoHkKPFb029DrQ50Kfh/lvYGyp
kam7GpLzR/ztVj7a04+eQfQOoXcYfSPoc6DfhYFRvtFggE64peQ+vyL3+Ws9
OubyZQbSq/ms3gH0DaLPin4b+u0YcGHQQ/18P2sQ1pBJcwjfknV5mSHrpr/U
veWGNC2vOoIDu/H6RnzvBdY+FV+b5+7fLT6LHtQ/hIFhDIxg0IkhN6xe/azh
EIbDpiXGv1guTcVn/7l+5nL+top23BfRexbNp3D+MPbsxstv88lfeRg0hiIX
38536R/g7eAwBkcw5IDVjeFR7rEFeBi2EEYiGIn9tJHw7kdlJJ/WKBiyYLo1
WIURGs151J/CqUPYthNrN/JovvqD/Ge/Nlfxd2OGxMh4Bgd5O2SD1Y5hF2we
jPgw4oc9yOOxh2GPwh43GRbiPcYj+tr0KT0O4afbaCdbPUPb+DWMXcXIZfRc
wLUTOHwAb2/nQT30Qv7e789+/qs5vkYk/wMyniEMWWG1CTJOjHhgZ0rBEYQj
xANzROCMwZmAc/z9YKXG9iE9tk9oPlplPC9XMzVdm8vUztF+pBreq7BVoe08
Lh3HwYPYuJNH+s0XC//68FzF13g2i0Y6PMzNNsL7djccQkxnAK4QnGEeqSsK
VxyuBFwTRRR/qkegxvtBoMSr+JimNutU0wd55LWTM3XZmbpZ2k/UIlAN1zVY
r6LtIq6cxuEjfNETb+MbL+IfH8lXfJ0HLzrnQ3zXkWGMjHCz2+FwcY/Lx1t3
CO4w3BEevjsGdwIeRhmeyeIMyhbPgDG9611GfKc+fochhXN18Zm61Oz1zFx9
LlfPEhqpg7cGzmoMXEXzJVw5q9S3aCCxFw4ankOaE063WRHcE4QnDE8EniiP
0xPHqAxyNIXRtKHRDGu71Mh+Xh//gKHb8teDc9djs/XJufr0fMP0XAODOl6P
YB18dXDWKvlacf4MX71+Nx7fgPvX4nM/LlTcl6OeXy7S0DkCl5No6nKZbofb
DfcoPCx18IbgDcMbgTfGY/Ym4GWmgC8N3xTffHUJH3z8XcZucPKHhVlop1Af
mK+PzjVMzDdM5hszhSam9GQjYg0I1fMVI3Vor8K1Czh5mr+u24fHN+H+l/Ev
P57/QmXGkAplN90OjLrh8cAzilEvRn0mpVngC8MXgS8KX5zHPyaDH0vDn4E/
C//0EjP42LvM4A6N/s/LSXykwVto8M83RuYbE/mmZKFpqtA8PdfMIpls4ivC
jfA2YKAGTVdw+Tz37DmON2keW1H5Gr76GM+47c8rafsLBirwOOB1wzcKnxc+
H3xj8PkxFsQY63r4Y/An4J/Q/nYgjUAGgSyC0wjOvOtcDOxv15+ih1fJDNxo
HEXjWL4plG+KF5onCi2ThZZsoXVG8Lsj1YJ4M8LNcNSjqxoNV3DtMs9j13G8
tg+PvYNvvV74yhN56mn5dGXdH1QK9T8qDOLC2Cj8PvjH4PfDH5DxhhGI8k5w
HMEkQpNMk1AaoQxCWYRyCM0gNIvQ/FKcpeT1VzQV7irOodGFplE0jRWag4Xm
aKElUWhNFVqnCm2ioU23ZdqQbEWiFf5mWK+jqwbN1TyHUxew/Rhe3osfb8P9
6/Avzy7c907B2g3/KAI+BP0IBhAMIhhGKIpQDKEEwsJJkUlEMiZlkMPTiOQQ
mUFkFpF5RPKG1ls0B2WLb/Dq5VlNdt42e9DsK7QECq2RQmu80JYstLFeKLSz
WOfakWlHqh2JdniaMNyA3nq01fFMjp3D1uNiH0xlnX9W2fonlfWfZK76iKFz
EfQi6ENoDKEAwkGEw4gIHaIJRMcRTSKaQjSNWJY7YzmajhnRGXJfYnOIzSOW
l5OxBFXUjH5BU2WBj5uG0SyTavGgxYfWANrCaIuhXXDryKKD+TXfiZlOZDqR
6kS4Dd5W2JvR36RWlSu3ncYbR/DkbnxL3AGaWtun9NSUFK4QGvgRCSAaRDSM
aBSxuFineJKGGk8hnkZ8CgmZ2PgsErNMqvgs4nOIzyNRULYsfsOcFnuWpqKl
+Jimmqxkfk3oNUwrGd5p9aDNh7YA2sPoiKND5tiZRVcOXawO5rox3YWpLiS7
EOvEWAffdqCFt5eqcewSdp7F+hN4cg8e2MSd7Z+ubPmjyvp7Ki+vqVRrULG/
8aBsw0jEaMyJcXKhxpMYn8R4GhPUpjCRxcQ0JmZ4WUkuxfg8xvMYL1DjeVgw
fsM8S/0jZcPvxKIV2y/JPIfQMkyLL7SKyW33oSOAjgg6GWp0pdGVQfc0uplN
53sw04NMDyZ7kOjmyXg74WjnneZGVNXgxBXsOY91J/DEHigtXd72x5VNf8BU
vbpGaPuBIm3HI5hg1xKpFFJpJNNCrVSGppLKIjUtKww119S8nJdHqiC9SGr9
nQS3CZgLEzfMttQDU7NfgYWo0uKjxlrA8IQWohJihX5T2MCqVphoF5w6fOgM
oCvC+91JdKfRPYWeafTMoJcFarYXuV5M9fJdIt3wdcHZgWGBqrUZ167j1DXs
u4S3iDH2c2fbH1W2/GFlwz2M0TWFFGt0tf5KxpEax2SqkGaMMMX2Helpmkc6
JwjdgvSc9OWRLiAN2QcmdeSFdhhZwqyQQj6JQhJL6C+RupvWv5abbGUZjHjB
AlS/WNRGvJoeVgtqdLjQKR5olzB3dwI9SfRMomcKvVn0zqCP6Wqe76eH006y
F5EejHXDRWh18aN629FMaNXjbC32XcaG03hefMiWT1Y2/X5l/RrNV2doZ7WB
ViE9gakUptLIZPOZLKYFmeycCmTdiuw8snlkC4Us8hnMTQEZwSsDTPF0ClMw
M4RlhTTyk9wKk8XIQWn0YyH6VbrSXRq3G31tHtACchxOGEbHCDoc6HShS5Dr
FuR6E+idQG8KvWn0ZdCfQ/8cBgrKD5ruR7IP0V74e+DrgbcHrh5+Xnc7GptR
1YAzdcp+WBrvZszqCC1Fv4sCYFErsnebz2YK09OYnsnnZudzs4KZBbl55PLI
FWi2OTpvfhpz05jJIpdBPgtkFT+U0z4DmAG1/BTxXD6tMCkIS74Lgu+f4xTG
C1Gqu4parGPIiK2MqPAKukfRI15ubxR9cfSNoy+Jvkn0ZzAwjYFZDLKYFoYw
PYhUP2J9iPQh1IdgH3x9/FBrNzrbtRaznKvHwWvYesnQajVrKqvXVJKgVq2p
vKTxXGmMey43k5+Zm5+ZF/Rum53F/CzfsTCLuVnMzADylXamc0hniU8xO41C
TscNCNp5ATIvHDqfUWsyBtSigDT0O5//s0ptWQmGdxb1cadVQkw2dEmUqNuN
nlEZ0YfQF0V/HP0J9E9gYBIDUxicxiAbeQyx9ckNYXIQyUGMDyI+iNggQgMY
G9BoLevoRGM7qltxrgEHa/DOZb12ubKGjSDhR4x4bk3l6TWVJ4Ujlxmzyc7k
i1bgY6mZ/NQcP216jsQZGQFxytjJyU4qR+4AJqeRzXHn/AxmZWdmmgFlZKel
PwsLQfte9vJDPwOui88oxfcjRS3OITirROFG0O1ADwfi0DuqMPog+iPoj2Eg
gYEJDKYwmMaQODbWOVh51rNWZK3IWJG2ImVF0oq4VeH4AWc/BnrQ2YXmTtS2
43wTjojvuO4Cb88LuGeVmjSZCOJjmmN5NsKjy72ZgidTCOQK4Vl+lm+KfV2T
OUwbS1TwFR4kJ8t028QseZGITmNcoCXUszPIzPA+8XMup40OQW4mFWHOMc4z
GYNvS31MWYsV40VGfsTg1uUlaAq33yOK0YruYfSogKQDvRw/5P5+PwZEZw5E
MRjH4DiGkhgSu2nNYXgOwyz3eRvmbZi1IWdD1oaMjYeXGEZkmHccAxjqQ3cP
WrpQ24GLLTjewP1vX+Lt8TWVR9dUHlpTuV/ZaTN/3rVvTeX9xj+m3RT60+if
KvSJxLYm0BRBO+mhFGxZOIQLnVneeqbhnYY/h0AOwZyS9TihuTqeY4ST0pWS
Ld16MoupaUxlWW9MZZDJKBkr1aziZemcxUJvKZrC478tSnIIvVYOonI01o4+
J/rEzen3YsCvQqkYjGAoDus4rEkMC6C2HGxzGMljhJkmb8e8HXN2zNiFL+xI
jSAxwvtjw3ANwTqAnj609qC2ExdbcbxRiZLJsnVN5ZY1lRsJRh6c8PSvngsa
XEeruQDvnxzDMQ+OunFmDFfCqI0LsEm0pdA1KU73FMM+Iqi6snBTy8CXRWAa
oWlewlF/JEuLOubQcfGmzcmsQiiZWdJCLeBoRDUX4ye8y54LR4z7htFn4yB0
vwP9Tgy4BTMfBv0YCmJIlvjDCQxPYDgp4GUxksPIHNlUe4GcPQcPIO/AnOzk
HMg4kLIjYUdMYWiDy4rhIY5RK9zK9l/Hlqt4VaIPrCNlZfWhR67PPFiVu+/C
9FfPsqP4H1Uzj1xnH2htF17vx/ohbJX77XHigBsn/LgQxqUwqqJaWy9vmUDn
JHonmYEZzwwcmstMt44KTF5xlIIZhAVN6olmEJ3ixdgSGEqsTXivNCMgvG4y
3RQfXVmCs9xBVp79Q7IdNuLzDgwInw55MTQGqx/WEIbFKbeNY0RwHknDnoU9
B8ecGY4CEdEp3Orkx8+5kHNiyomUA+ZxQn5VTOD32uAQVdAziLZ+1PfgSid/
FW25cu1ZPH4S3zvEPf+yFZ9dV/jsK4XPPJuv+EleQlamOx+8ikcb8HQbXurB
mwPYZMVOue/hUZwY450rEdTE0JhAyzjaxuVJSTatw6Iu7FNwpmHIwrKxKfin
EJhCcAohwfhWzZHLl5JwZT9vsFKl5xmZ3FtKkBba3SHqcxADArZOUDhUjkL0
qA/DAQwHYQvDFsVIXPlEsE/CnoaD1NsMnPP0DCej7SrAJaztxpwbORemXEg5
YU5R7/Jx54KU24cxaOVEk0L5jrPtONKEXXXYSDx+EU+fwSPHJHOwE/+0GZ95
CxUvo+I5VDxRqPjRXMUPMiuF+59qxcvdWD+ILcMa82M+nBEFcjWC2hga4mhO
oH2c1yLUSaZWaVaTmcZvKnek4EzBlYIntaRGKOVVhbhia/VPowEsJsCSFLh5
xbpAidVFSgwOMS6DwwK9A0NOWIUMwz7Y/BgJYkTSNPYY7AnFL8w8TmKhLFwz
cM0TCcxwq9WCJe/BvAc5D40g46YnTrmRdiPlNiTc70DRjnUMoqkfNbKIPdeB
Y63Y14ittRCkyyqeL1Q8NV/xE2pzcu2qimczFa9MV6zjWMGX9uPeE/j2RTxa
h2fb8Eq39vl22rHXgYNOHHHj5CjOj+FqEPVR7ft2jqNrnAfQk+BFD3mcQzqs
UUoHY9214mfg+Jv9vQW8BX/JoQwOSsJnGFZJo9hcGBa8bV6MjMEehD3ECR9H
FI44J3w0vV1puDJwTcM9Azdzv6eg4teeAkZF14xiZlQhn6MOS3a0yHnig/mc
cNphtXFStJ2QH0Bdv9b0K7ddx4ZqvCzx2UdO4MFDuG8P/m0b/nmjWlISNZ4q
VDyTr3h2ruKFGdVz/3l8vxqPN+K5NoVu2YY+bBnELhvDLz2rq0KoifBNG6KM
OvkoLVG0x7inL8HOE5mrohQsL6JXZG4Z3LtQwMiEGr6cRKjpIYK0rNqsQ5Kw
GoHNgRHR43ZZ9Np9cATgDMEZgVMnrLjfnZLtFJt2cps8sxidxyhrc9rCS02c
e9pa5rzKs/1EehRJATrkxqiTkzfWETbU5Ie3DHH/tT5c6MaJduxvxvZ6bKzF
G1e5//Ez+P4xfPsAvrEbX5bs4GfXo+J10TovomKtzm9WvFioeCVf8QZ7lZ/f
iX89gK8fx7fO43tX8UgdnlhAf+XmQey0Yb+DvRhyYS6KTqoKwsikm8qboxxF
7YyDuH8p3WNo7fdC/KbVSqklXsTzIiMfF962wqbSbw7YXXCMwqkynD44/XAF
4YrAHYU7DrdQwpOEZxKjUwL9NEZzGJ2FV6J03gI91yebG/ibs/5zY8j5kPUh
LZSKjSLogc8NlwsjTq2BPtImhremHxd7cKIDB1uwsxGbr3PnS1fw9AX86DQe
OoZvHcLX9nLnF7bic2+jguzBG0wf1sAVrygdteqRWv7+HNmEDrzVg00D2GHF
Pjt7kSe9WiPR4EgF1YRwPYxGEYCOWNGPuSmzqLA3vJnFXg1/frAEYaGR5JPs
I9w4a+iC0wO3D24JT7j9cAc5wemJYjSO0cRC4tCbhlcQpvWBbwY+CWT5ChaM
FcRPLvhVloQ+5/yYGUN2DOkxpAT3uBdhPcPb6JFWJ/v19KXJitpBXOrFqU4c
bsOeZrwja5tXr+H5S2Aime6oWCdwvokK4ngJWn9+F764H185jK+fwrcu4LtV
IiH1eK4ZL7er6NKKwy5N77NeXPDhMml2P2oC4qcY/kopRyrr+fOLek1LoFhW
5FOnAy6nNBfcHnh8GPULWrLkGw1jNApvXFKZgqIvBd8kaDlLy4AxTghibA5+
yXj4C/IICwKqiq1A8vjhuQByfmT9mPIj7efBjIu7FvJizMtPtLthdXGdCXW2
2FA3hCv9ONuD450KhVteqsKzl/D4eTxyGt87wad9bR++tBNf2AZm1M2KZCbL
V0/gvjNFD9i0qgjmW93YTGw6jD0jGs9jLpUzKasSPOtD3L/qJulfXdQKS1Xe
GH52MU+7CN+Fcz1uamy7PESEUS+8gq4vBJ+sXzjXGsNYwsi1TsK/kG5FYAYB
8QTMKmoYLCAIaryr1kmFIKjlg5gLIhdENoBMAFMBTArUiTFExjgp5/WyiNjc
GHRxmIShHlEDNN1zugdHOrG3jXs3XMfr1Vhbhacv4bFzeFhyyd88jK/uw5d3
M+j/vEP09lZUbEHFZt5+5h3u+cdd+Kc9+NJB/Psx3Hua1TV1PnwNj9XhqQa8
0IJXRV0QHbYNaSIcdOCoEyfdODuKiz5U+ZfU0EIVpYdvjhK9exRds77w/bKS
qkV43Wb4vPTfR8tRSRgL+0bgj8IvCRT/OAITCIh1pCVCUOWMcwjOIKji3maE
CuUI6QoGYaBCGPkQZkMw5+j7rdNBZGTJnRK7FPNzRnGMtJQXdg8XBg241Xh/
vdWOehuqrbg8gHN9OCGJoF2t2NKIdXV4tRovVOEpiZR8/zQePI5K0hoH8O/7
8WVR1/+0C/+4A5/Zhopt+JzUsvzTXnzxAP71EL5yBF8/ifvPyrVV+FGNVGeR
9zPERnIv+S32BSJc8C1JgZXvmwKGzr5Bd5cXcz/weUjmSRb8fjKc/gD8QQRC
AnEUwRiCcc50h2SNGZqUbYaDDOEcwjPiewjW4l0hUqAdhGHh7ytVP31VR/Nh
zIUxG8ZMWNvAlVMhTAo1EkFOgAbG4PXBRaTwajm4tdOJBUIM4lw/TkjUf287
trdgYwPerMNH5czvXsLDV/FYLZ6qxwvNeEVUjGLqbeQNkj206Urfs6MGJ5ZG
Q382VJf0u2v+orKo2283ahx8nH0PBREKIRRGSFbuoRjCcYTHEU4ikkIkrWPw
4NjNQlbQgug8onl1KKqfGQXvmpf6zvYjH8V8hBvtz0QwHUYmjHQYkyIRsSDC
hLWfhYy+ku9DLv+gYNLlQqsD9SOoHma0zxLavdo3/5W3G/HGdbwkdQePX8Qj
5/C9U/jOCTxwHPcflQXrQXzlAL60D1/ci38hYyAJvy8fwr8fwdeO4xvE8WII
vkd0IqYnl70OzzTixRa81lH0oPhJy3YNMa0O2XHMiVPuos5fhPaKn5n7i/Hr
U58q5meV18BgRMKIRBCJIhJDlDPChehEIZpEbBKxNOIZxDlGrbh9IfMezyNe
kGfEjew2uIlbxYUHqsKpIN/no5iNIhdFNoKpCCbDSIbFgQmpcazyBeDxwzkG
m0h8nwdd7gVqCOd+4mgPDnRhN/F+KzY3Y30DXpcI+LNVUCT5wRl8l7TRSe6s
PIp7D+PfD+HfDvKWKUSUOIqvHWPf/b6T+JbQ46HL+OFVrYFogm/3YEs/tg9i
txX7bThsx3EVvLgFiyraf3bs+Ujrpysv/kHlobsrJed7m3BuEFHRNvEY4nHE
E4iLR5NIFxKpQiKNRAbj0zo/NyEezTjHEsfnuRQgr6sCyrgqoAyynFBbC31w
TYS5EGdzJUHUfBxzMczEMB1DJoqpKCYjSEY0/tEwiydpP2+AvVWHQYj+UXR5
NK/UjeDqMC6KsTzRh8Pd2NeJne3YSqqoCW9JzdfaajxTBU51mlY/dAk/rMKP
iddr8XQ9nm/CS62a1zd0Y1Mv3unHjkHFmKU8vhDjKs0Y3JSHYd5r/qPKa/dU
XrxbuFvuJfXHiRjG4xiX1cxEEhOiwZPZQjJTmJgyaVyT4nKn8pw9LyhOTual
6dqSJMphVB5MQEowZFMYRyHBTcdU5xOYjWMmjuk4MjGkY5jUKsq0Ihbhmp0A
+VhBeAK83hoZg3VMry+7PWh3o9mJegeqR3BFFken+nG0Fwe6sacD29uU5/nx
56/h6So8cRmPifH9AemhM/gO+esn8M0TuF980ftP4pvE39R5Gg+cwYPnhNFJ
8VzBo9fwkxo8dR3Pie5hAFSUefsAdpPiGV5K38j6RvuTJdiz4DT9YeX1eypr
7668fHfl4TWVtxj9SI4jNYFUspBK5VOTah2G9Ewx8lSYnCmk55CeF8qmC0p+
JguQQL+ZPsqR0m+fJImfJ2DhCpAPMzOPIy8B1dkEcglkE5iKIy18noxhIopx
qXUitRYMYyxESwe4AhxvUNw+6EOfF12jaHOjyYXrDtTYUSX5oTODStvctqsD
74gDuq4Rr13HizUg7J+5iievcOejF/HweTx0Fv9xGuYHTysJ//ZpcDtDPeTe
WL6rPM4rCg7RL6+TXe4Ecf9m4f6dg9hjhWkJzFVo8UNLUYIjNHV3V9bcXXnt
7spLd1eeWFO5a02lsgqS401PIp0uTKXzU5mZ9NTcVDafmVHWtJCZo/18ZpZW
Qpl5hXcGyBSo0UHtqxamwIUlnNbnTbmiR0HKJJREpIgSSVXRm5/AvLRZCSJN
jyMzjinRQpNxLmqZIBkUGxCNku2HP8wlqXo9ZA/AJp7owBh6fegcRatBkGq7
Gs0dZ4dwcgBH+3CoB3u7sFPM5dvNeKsBr5JFrsUL1Xj2mtjly/jRJdwq1HuU
FI/w+pOke4jdG7C2idn91Ta80YH1XdhImr4PxPbFN5FuQt9YSS2VzSjDwvsB
kg2w7C7WWRSrjArZbD47PZ/NzWRzujJldpqVSiGXz0/P5afztMMX5ZCfxlwW
hWneMed1rUVZIQuhDzczkUXVDKehmqoTUHQpSCA7n8I8Scwc0Wb1rES+pyeQ
FcKkxzGZQDKBiTjrxLiQJGzkg1Z5w/CEuILcHlwgSY8PHaNcG6e0VLUsiC8O
45wVpwZxrB+He7G/G7uljGdLKzY0K0tqefwKfnwZj1yku//wopKNhy/iB9LM
36eeW34g6ktR6XGh0jNCJQLpZfKJ2oys4bZ+w1u9ZZEtWMreLjdQh/FPhM6Q
GFXGcXlN5QWh03GdHS9alJnczPzMrKJSeTrHNmEyV8jkCrmZgtbveUmbz0pF
h8rPzuRohvM59dzcNFd4MBWmuTdL45uTNOWc1HeY56cg9R0cGpgX0s1PYm4S
s9RSmEkhl8R0UmH9wakJTI5zQRgp0fEEEgn2D6Ki4Mh1DkQxFuGibk+YI6h2
WUUMqSiM6RYiWJMb9S7UCMEu2XB+GKeHcHwAR/pwoEfbpNs2tWB9M94k9VaP
lyX09lw1nr6KJ6rw2GUm4I8kFP3oJZgfof8/JJJZHhXt99hVHZFeWaTXmyRV
ndhoaDZTUaruL75ZWPZelFs6BserqPNS13BqDSs6+np0TeXBNZV7OY9s0l7p
nVOzSM8WEjMsTNEc7RSSs5icRWqG+vkRU7I/nkMsK7UjOeSkliGbQ1rqSGYk
926emVYTm8liVhpT0EwENK2alZwcH0sjl8b0JLKTyExyGVompZBfPjmB5AQm
JjA+Tp4cYgmuz1Xc8+FQHIEYxqJMNncEzjCnGGxCucEAVzT0jKHTh7ZRriJu
EMf/mgNXRnBBr9zuOCg5ml1d2NaBLW14u0g+ie2trQWT7xqeFAo+XgU2Bqbb
n6jGkzV4ug7PkoCRF1aUsVa82Y51nXibaNaDd6QeSmhRpvyiYt3fe9LtJolb
ZdDtmJRQHBZyqSoKItoenf83rQrOwJOFY4q3XkGfw585joNSj9k9pe4SEjqF
ZzjB78tw2CM2rbRwko6UTeS4HChNAqkqq8wZ0p8ZzvdPSc1VVhcAmSyZKbKK
mCTBM6fIjq1KicacTIFaSnTleBKJCcQnuKJaeSvhBFeK014gjrEYxx2Zctpn
XjksC8qBAPr86B7jatFWL5oU8dT6/aIN54ZxSkTvsAjFnm7s6MQ77dgsBFzX
rKhU9hTR7RqeuIrHr8L8ExKvWx4Xm/YUka4Wz9Zp0q1tZKK8ItQz6ir6DA3J
q3DGttyQO00Uy7vQrzR3s8qgJ7kTO9dU7lhTud2gFk+nPVW0fuamCeVh10Rw
PYpWKSZoT+okd3eKqxiH0oZtGclwLof2VP1LQMInXGk/zQVGKkeVmKaFDsaz
3CaySMr5k1mkMkiK4E2kuUjaZE5MKnC5aJoGZJ5IqTHHab2aRFRlssxhGsgH
iHTBBAJCQF8cozHObtC+PYKRML9+ZxUC9gfQ40fXGJdF6/zodSV+Tlyx49II
zttwxoqTQzg6wP37ekBCKPrvA2+I+JH2XFuHF2rxvKQinq3BMySK1Xiqmqn6
ZPUCJZ+p07yxSulF+vdGG94S5akEcatIOftrpZUo70XH98oIFaPIRc9bRlD+
1gCeqc/9uHrmmdb8+iHlfptu2TKM3Q4clojI+TDOB7lw5GoUdXE0JNA8btT7
cZnlpFQzTMGe4bSo8lk8U/AKxUhUzWGiY3lEiBnNIJbhaILJHJtS44mm+Z0F
ZuIYUXVFLMX1/jFhs4hIZHgCoQmYg7y+DYzzmzOKlEo6XVHOzxI1bWFYJZww
EESf9vFvafOh2csvx1z3aGJWOQxiDjMxFR1u39ODnaJSt7Zjk3L4m0FkfbUB
L5HbL5SlTqLsc7XKlTCtfv46XqjH2ga82KgL4V5X2rQDG5QR7FnQhTdR0JA7
I69kxkJsQcVx3yW6tk1CeuLkm5Y93owXO7FugCt9djqwR2d7bz/rx2UBoz6B
pnG0TaBDhKIvhcFJLjq0TnK5DPWwilU0I4L5jPITS3BKjSrEC4DQJMLSSKZS
ugI4lOR3fzgYPcFRaZN5zNCXo3F+Kc4V44SuPcoFC8NCGqUHegPo8qNjDK0G
aai31o1rLiaNUGP1SYkukK480Ie9PdhN6lLZO6mq39CCt4Q4yvu4Yy3RoQ7P
1zFpnhcyMVnq8aJQ5qVGvCxO/2stUPQpN5luqjtSNPlgCeZCGyOPzUTf0YO3
WvBcDS8xaL399cPKPJnKX+7GG73YOIjtNuxzaa28/FoEdTE0JVglEv7dSfQm
Ndq3DE/CRjSYhD0FRwquSU4C0gHvpLG1eAlrX1L5iv4k/CINYxPwTcBLbRyj
omnNnoQanTvO76Y6YhiJwhaBNaK5e3lfED0Kdj/axtDsE9iVdapy4pId50dE
HmQZemQAB/uwrxd7RMVt1/muFW81Meivk65r5NNeIY1HINfDQgQQUhDgL1Gn
AK6j20X9tk4pt65FcrG0ZvsgFt7Y5oO36e+mFetbWRBpVUEL7kqpD+Lswla9
2vjWJfy4Ac+0aq4o3+/CEQ9OeTn7Xh0xojNNMbSK/9yRQGeC31Po07rMbGcb
YheMHUlwmzCuck9wGtybxKgIkWcCbn2VxUXYl7vklo44RmJcJqXnPhjGQBj9
Yl56gugKaPhbfBr7lTWK7YkCDlwYwVkbTkns5eig5qDbd8iqamuHeAqtWC+s
/2Yzd77exIQgBaUI8VK9nvmtrzbjtWatkd5qw7p2VkeE/KZFlqUUe/MC/2ur
orZ8SvHNJdI861qZ4uRcKh24quJNfG4TPi/Jna+cwAOX8MMa/KQezzTjpQ5l
HZef9uFCAJcDXP5zTdC4btCjrD2OrgR6x7kU2jw4od7UHRKUh8dhm8DIBNPB
OQGXdDqF4x2MviMBs53/x/V7ZkNRLlwdiKA/jD5Rfj0hdAf57R1NkSYfGry4
LipHYX9FYW/XFFl9bBCHB1jtaPYXXfuOaJ5NBgXWtehYzupXRRBeI0I0MeSv
U2vBGy0aewJ+fQfe7lSQLo24Be/9npKRY/2oPo+ksBWvNOKZGjwqi+f7jnJs
+4t7dfzhvnN6faIl/80ebBzA9mHsc+CoBydGcdrwb1ZUBVEdQl0YDRE0x6BI
QRKheL4vgX4iyziXBlnHwdQwJIV2ymx0ni2u1N9wDNYYGP6owB9h+HvDBpe3
B9DqR/MYGn2ol2qQWg+q3bjqMjyt83bmfdo7YYWiwMF+7Cfd34vdoisMHbRy
g8SOiQKkhkkM3iDEmxh3Pmgm5E3lb9KBVm7E+obTairfeCP731QfbV7wqt7H
OwAl76VwvmlnL7Z0sngQT5ARelL89ofO4YFT+MYxfEVyD5/diYrt+NwufEHS
o186gq+fxgMX8dBV/KgWTylm6sL6XiNefNjFFUSKnhfGcNmPa0HU6mzJ8tYY
1xDRXmcMXXF0x9ETN95iGUhgSFUWmenTMmR4S0NyAW2ZVFEhVYRJpbilM8iv
BLYF0OJHU5FeylOtcuGyExcdOEemokisIZC1ONSvqLOarXQnWFuJuGxUqXsS
mlYUSfaWIT3a9327Axs7sakLm7uxhehDi1Ah0bL/JIVK1z1CIX7grj5sIXDb
mGNIWX5YZvxSO97oxsZ+bLdytdw+OxfMaevNcAdQF0I9iYjok2ZDb61oj6Ij
iq4YemL8zhF18ZtHMY1yb4z3qfUL3P1Rfq2mV68YV3SF0RniRl/agxpuEg/1
1BoPrrmxgLZd0Fb4HiLV1I99stYjwdjZje1deEcQ3yKgUz+DLvjeSlPdQK1d
WodWSgx0F7Z0a6AtSyD2fpFepr2jjxln7h0org1Nd5EcviIc/TQZhCo8cgkP
nceDkm776hF8SaoAvnyYCwGo5+snUXkW376Ah67gEVkoPdOEtW14jUTKyL4u
57oMF056oKqOrgh5ND2YSiE0htEURksEbREmT6emh6WTSCXNZO6OwtwT1bqU
KRFGR4hbOzWJw7Qa+YHGMdT7cN2LWjEY1zy46sYVFy45ccGho6Fkr7UYDEJR
Z79BnV3Ka9Way8Ii0Y6N/J7Z20Qiy9uKWrS1bOzgTaemkBHz7MV2FY35z9Kn
1OLcaXixRCcShu09LA/0bOITheJt5ECT4vrxFY7Q/odYkm8cx9eP4d7j+IYk
le47jcoz+OZZ5XOVMXnIwPRh66BGer8dh504bmj2FRfHuDqM0QughnRXENdD
LEzq7FZJz7exVLWTdHTQTnlnxLBgpNj4SAedTcQxrW4LodV4VaQ5gCY/GoRC
dUQh7wKFtL5imyJlVYtJpCwLmXclT9vFu90m9f5bi2LUwYTaJBkG2t/caVTm
KvKw3PSAqLOtjwm0Q0U5S+3C+6VR6TrE+KslpmWKQlsNCpE4yYrvg988hftV
kxqob53FA+fw4AV87zLXflDP49e5pmBtq1Q2dWNDLzb3a34vFtWcGcU5gezS
GK7o/GN5DSm7IBpEjESOm+mzrCWsIW8lOrSFlSfIfwSAGhHEyOKsbBHSEF0a
FV3GUOdjutSMGhGgKjcuG9LDak2oc3IYx4U6gvEqZfSV50ULchKg7eIBE41I
z5m3KmkyEyzmreSbLVc02SYEYWoogizUHy7lby3OJZfW4t9lUGy3kj1lFzeI
rL4hGo0Wls/WSTRHebDfPYfvSHvwnMbxwfP4Dmmyy7rGQ+f91osjs3UAO4aw
ZxgHRnDYgeNupoZeZZz3crnZZR+qxlDtR20A14NqMOYG2lnWGERTCM0htIjp
aFF+gLkpCHNjQPFVgx/1flwf0x5xtSguwv2KG5dcuODEeQfOKuiVlBxdJBb0
vQi/0l9msjDyxvM7XcXYA4cfLNt6ijH9Ikpluwawm9oglkD/5wT9u5ZC+zal
kfb0a6X0TrfInMlEXM+QN/GK47k6PFWjalZXPCq+1ZPE5g14vpnLCV5tx5td
WqdsHTD0h9JAR5Q6MplOe7iWjEC+5GOeV5Qlnq+VVido1xP/Bw0fqlG+NLCN
qSebQMAqVGt9qPGi2ouro6gS9+yywHvRqVG95bSw9glh7aOGbSgiTEsLXcm/
i2x4D3Z0c7Ns71Y03EE91Hp1JE2hvIvQ6YeZ8DVZCOE9g4o7l0T5Y4vtMuv7
fQa65KQqgFX0yPI4KXvi0Sqa4SNXVCz34StcIfADMsN04Boeq+EHPXUdzzZy
RZnWhJv78I4w8m4pKSO1cliSTCdchud6wYCZvlwlQ+1njq7xaxrVSW7xOiNL
G0udRnZ57RhqfIzsNUJ2FFc8zLiGPielIcx7+ka9ofBVHsI+g30VB5vEDpss
BLOZcC4z9CohubtfW9s9AzATmua9Q8Uo3M2I/sJSlrSsiOxubU9uJbVErEuW
fV0bL0peFv9HaYsnqvETgvMapCZ41QtNXHnxciteFzujKmG2SiUMYarQOOLg
6q+TLpx246wH50dx0asea75KwJZdEzVBX2sJV42tuYaOLKsW3Kt9uKaYdFRz
7kWlA5yM5BlB8pR48oSk1nRKGdAem0lxZphbzXt5SbJH/PLdhj9mUSpyD23N
BIJ5z0BRJ+zV8YtSq6jQ/MUl0FQaRGG5s6/I/eWE6KZOdmBp2Utra8W35Y9X
4zFpP6nBE7XGm/rPNYBgXdvMsL7WbuQmt0vh416Jqxwc4WKuo6IRThmctZx4
lYC95MVlL66Q9vWB4VX5Itq76uNGmOuwzVUvqryG1bvswSU3LkqmgHF14qwB
rdYFxw3fQ0F7qATdPoVU+V7asewVFcFutHkfv/q5T8RcYVrqzSksP7EUZzJI
+wdBt6Bb0T218PcpD+LWlxrYiD1PSvU6mzKW71rOvT8hGfin6rhUQq3Q3ljE
nFv6sG2A+XOXBAmNStIVxw0WpS/EpedGmVEvjGrtdVnqxwhVamVXvPqay6O4
5GHULlBzMUPyxTr6u+LUCE7acEIodsxQoIdJRgk7sUbsYvOp+wlCQszMG8u+
fp6wPHQJOS5dTSr0fklryrsMr8uiMTPeu72VNOY28cI2GXJNva+JaL/UyBHm
tQ0K1VUvNHJxyYtkj8hmEbOKy7C+C2/3MHpKPxEX7id1qbjQwVWFJ4xYcfli
7LjDzMJuuThatGa8y9r8IiHnYeTOu3DOhbP6bRrzaTuK2Nlw/Eb4lL1RbKc4
z3xAv3LGPQPcjEqWn47br7ybBN+iDPkeUbAmWXPvUAh287KTFzpqJC836Xm9
0AAzsaKp7HmSXklhv9gE80tkaiwvt+DV1gUvrGxjjy4r0PU6JC/DbH7IldIr
RELzpEg2seMZUZrnPIpVys7TTtl5JadunHNrjXjGgdPysgwDN6KBOzaMo4ZG
XHFYxLYouWzIxZZbDvQv/C3HEsR+jr4yUgqtX30368EyKmJqZoaz7BHIdhtv
Aq7epLygdo5kvdnKvtBrhIjEul5q4tTG2kbe6mjKa+QtFXmuGwqsrSK023U1
2woF1iE7jqjXZZ3Cgk5dE65lmP1R2pgJQNPqM66F1dgZJ05Tc+CUHSftDBd1
MmI2jVXZUZW3EWktY7y4W8Elu7eXIFF+E1a/thRWFoUVNdGLKm+hbYXh0WyR
ZQIxGNmK9SKoBBnzmvkV5qVX5K90MTuZOWle/rqBFuu3Lo34iu1C4l1DXI+3
z5BVgkueuOyY2GOF2EnBxXxKva9jPuVUsnnKgZMOBucEtREc11bAzOy07IgV
R4a4KYwIH/NBpdFK/z512U24/PoiTSX6jXXdChVw2Ws4ITsNaVMZkPWGvnpD
2U4LsY+p7HURKdoSBMpCKm7ZJKVwJF3b5KY7h7gUVMvAwREN1WFhHVJdx0R7
ydyYf04QP5UdF6Y6bsdxjVjZUVFCNO/DVpgPD8FySHlqH/2p8/2NkjPkivIi
HyjJUSjoyNPefiOjtV38FMUTm5Xm7uAAmElC8iRMb2jOeJM4YMVbkrBdL7K2
sZthUHKjOGKX+LhFdjggQBwa0RJzRNS4aGcCxHKMNEnZUfuCxJiPkGiUHxnG
4WGGgC8lTM0HFQqlfw3cchMKv7kUCoulYb/2fMtLeECvYAiDrUo0eL5l69rV
1TRfaiYzbyw08w1dC1p25dbFTEDWXgDYqzwoNesDIhbME+ZDzOCHZcZH7No3
PWxTsnBoGNysOEhtCAc0N/1yyZzMN836t5aYtbL7Osq0TFlpZWXUskY4kSMC
5UoRMDGF7iYzU96yQei7wYjlrNooiZ3NxPO9nBJgPSl2RU1a5QzYuoiBYcBp
az7A1CzKgmIEoqaN5JjAKTtAq3ohMm35Kepvli9+x0ve75A5/raWZyPes2y/
LoHXPoxa8+wSD3uH6LhtEndicqqhd4mPZtlI0122ycipqqm806cke5mmn+gz
NRuFo00xBA14v2pWhUsp8otpo/+e9U/hRjXuovOpSKQWCjuFUDu0L2RRM9nM
gYXN3Wo89LlZZrKlx9DITBUhDDmeOwe1dqYJifzSlCx72VDvsy5e869ZEnc1
h997D94y5sIkWGyRjfualivhYnqIilEkoalojbuM4Cfst8l01bDNO+g+y9TI
1b/dIlBmrvguU6uSvSJfPAr53YSF3xIoIr/mpyBftmj0xmjNhhO72xAT8072
m3f0GmMRmdmmk03b1LjNvLHQ2I24Jp+zS4vubmUTzLt57HusC3P643cZ993v
k2P402z4QfzIog+peGWniuzJ8LXvv0MNUfhgF91FjNXCmPiUP1309AUOuGeR
5BUpX1Yc0z41Epk3r2p3a7OiAjAm8041JssuGu1ufSULmDQ+8a/0/dXTPlmC
wEfeB+WInweNtYxl74BeWd/IiyoStEdO4mCQxDD2qCjq39wgtX+wxAjMGnNZ
ecow1EqTHkqt3OCeZXR/WsqrxoyqZvj3N9z/D5egsdQTam1dpmlcvk+jtU95
verp/OsBZUa1k/y1f4M2pSiVGXyix2xZdLWqBv/UT8FaWS8FNG/5LjzXv9N3
UW/0/sm74FV+I5/oERjXyrup6rdr3msEeh5/q69SfxHtT/WsF0uHQWe+P99Q
/VXTPy+5f/HtbpMq2fgbfV/1l27/ouTsUi/zr/TZvyhn/2XJ0dI38/9Mn61+
V8Hg8tKzPqXPUr+l8dclR0v9/z/Us1O/WvJfS47eVvL9Hn3vhd9Wuemv6Cz6
/rv6bPVrM3+rx1t61m/qs+R3JvSvniwcLV3b/Zo+W/1G0d+XHC3N5fySPlvo
IL+1sfhoaTznFzQWgpnpv5ccLa1XuFPfW34vxfQ/9OxKz/qwPkuebfpfJUdL
o3N36LPlPNP/LjlaGmu+VZ/9WTm7ouRoafzFqGr8gpz9GT3e0rPK9T2/KGd9
btHR+3UxOh/9shz9R9qykf+qfDP93P8Fy1/MBw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->79463305]
}, Open  ]],

Cell["Alternative form of equation:", "ExampleText",
 CellID->29700],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SubscriptBox["\[PartialD]", "t"], 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "x"}], "]"}]}], "==", 
      RowBox[{
       SubscriptBox["\[PartialD]", 
        RowBox[{"x", ",", "x"}]], 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "x"}], "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0"}], "]"}], "\[Equal]", 
      RowBox[{"Sin", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "5"}], "]"}], "\[Equal]", "0"}]}], "}"}], ",", "u", 
   ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->24234],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "5.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{380, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->77337803]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(10)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->10680],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Ordinary Differential Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(8)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->99904585],

Cell["\<\
Specify any order equation. Reduction to normal form is done automatically:\
\>", "ExampleText",
 CellID->5110363],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
       RowBox[{
        RowBox[{"1", "/", "10"}], 
        RowBox[{
         RowBox[{"x", "'"}], "[", "t", "]"}]}], "+", 
       RowBox[{"Sin", "[", 
        RowBox[{"x", "[", "t", "]"}], "]"}]}], "\[Equal]", 
      RowBox[{
       RowBox[{"1", "/", "2"}], " ", 
       RowBox[{"Cos", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"x", "'"}], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], ",", 
   "x", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "100"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->39034423],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "100.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{323, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5929889]
}, Open  ]],

Cell["Directly differentiate the solution to make a phase-plot:", \
"ExampleText",
 CellID->13663552],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", 
       RowBox[{
        RowBox[{"x", "'"}], "[", "t", "]"}]}], "}"}], "/.", "%"}], "]"}], ",",
    
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "100"}], "}"}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", "Hue"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->235254161],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 177},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->555469401]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->656801886],

Cell["Directly specify a system of equations:", "ExampleText",
 CellID->257759957],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "\[Prime]",
        MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"-", "3"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], "-", 
         RowBox[{"y", "[", "t", "]"}]}], ")"}]}]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["y", "\[Prime]",
        MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{
        RowBox[{"-", 
         RowBox[{"x", "[", "t", "]"}]}], " ", 
        RowBox[{"z", "[", "t", "]"}]}], "+", 
       RowBox[{"26.5", " ", 
        RowBox[{"x", "[", "t", "]"}]}], "-", 
       RowBox[{"y", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["z", "\[Prime]",
        MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "[", "t", "]"}], " ", 
        RowBox[{"y", "[", "t", "]"}]}], "-", 
       RowBox[{"z", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", 
      RowBox[{"z", "[", "0", "]"}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->332826321],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "5.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "5.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"z", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "5.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{309, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1850539]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", 
       RowBox[{"y", "[", "t", "]"}], ",", 
       RowBox[{"z", "[", "t", "]"}]}], "}"}], "/.", 
     RowBox[{"First", "[", "%", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->431684294],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->299067117]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->73400561],

Cell["Solve for a vector-valued function:", "ExampleText",
 CellID->37159131],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox["y", "\[Prime]\[Prime]",
          MultilineFunction->None], "[", "x", "]"}], "+", 
        RowBox[{
         TagBox[
          RowBox[{"(", "\:f3a2", GridBox[{
             {"1", "1", "1", "1"},
             {"1", "2", "1", "2"},
             {"1", "1", "3", "1"},
             {"1", "2", "1", "4"}
            },
            
            GridBoxAlignment->{
             "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, 
              "Rows" -> {{Baseline}}, "RowsIndexed" -> {}},
            
            GridBoxSpacings->{
             "Columns" -> {0.28, {0.7}, 0.28}, "ColumnsIndexed" -> {}, 
              "Rows" -> {0.2, {0.4}, 0.2}, "RowsIndexed" -> {}}], "\:f3a2", 
           ")"}],
          Function[BoxForm`e$, 
           MatrixForm[BoxForm`e$]]], ".", 
         RowBox[{"y", "[", "x", "]"}]}]}], "==", "0"}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", 
       RowBox[{
        SuperscriptBox["y", "\[Prime]",
         MultilineFunction->None], "[", "0", "]"}], "\[Equal]", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "1", ",", "1"}], "}"}]}]}], "}"}], ",", 
    "y", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "8"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->853395542],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "8.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{309, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->49679485]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"y", "[", "5", "]"}], "/.", "s"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->426580444],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
   "0.3631035939468866`", ",", "0.6584610985987999`", ",", 
    "1.215740226661983`", ",", "1.4692304185298557`"}], "}"}], 
  "}"}]], "Output",
 ImageSize->{276, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->623800703]
}, Open  ]],

Cell["Plot the four components of the solution:", "ExampleText",
 CellID->202781194],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], " ", "/.", " ", 
    RowBox[{"First", "[", "s", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "8"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->53724152],

Cell[GraphicsData["CompressedBitmap", "\<\
eJylXIuT1cWVvs59zgMRTSK6GCGJbqK7tX9BhGEGGB4DwxsRhRFQiCIKCBIF
ZV2EVYkGg8rMMMzA8BxeAgoBBFQQFVwIi2Y3ShJJdqVSodzS2q2yaqvufn2+
7tN9e+4M6FJ4+T36cR7fOX3O6f5ZWz9vxvRZ9fNmTq3vPWhO/cMzZk6d23vg
7Dl4lLwqkUjsxX/TeiZwnceV/Lc3n8e/15vfpLmXP4mGok+/J/ev7fpoQ9vp
r7/+38QSbVWCRvI2/fjPf73i+XdeWvnuzBm7vvrq68RT0iaLXzNQSTBiwtzj
v0w0y3fMT6Z961kMwkdvHvp0yVOHOFIuah33Tkb318n9pUv/PX7sRkMz389+
aM/pf/kPc7mYlCQtQZameJRis5hbjg5CQa6f/f33Ljw67w1zebvcYypMbS7r
vhUP18o9hhg6eK3hIS33U+/d9snv/moufyySx83IEevBGX5BQSdzpay24jk5
x743/m3ZM0dlOLmHole/+r65/Fvzk/388y8xLaSJO2gXXIK3xIhOZwrv03Zm
ygxdQaSf6bPPvrj7ri3m8lbHLWbCUysNzMmZE8MVUWY09286mq2H3INGSMPI
LBvJ7BbzU6ooA8QgOSAXbQFwXNh5YhvguESkiMTMv7b5lIxXjvnABah85+0/
UDgYDCaBUUGHof4nji6wZi776CyGC2oBKoAi3OiCHxBnLn8kAiPVGA1zGRsT
q0pRV3wniEhBwJgVjTBgny64AawwjLnMRfc/lHFIOlEu2kiDH9jTxYtf4tdw
JTrLfHTuIuj87W//ggs0hRB7F531GrmHIqAOUYPcQ0yQmrn8gcxBrj3qUwGX
GBqCvdXpdPGig+AyUQKfk0gumL/PjvP9Atn2sJoqk16qec6Wosz5VLCUAl9g
nFTe4qxj0l1b0Mb6MbBZP3nbvfXtiZu74NP7GuEgC9GC/L4/XQ3ZAUecwPJM
nKAFOMHotNIUbYXYFQQkx4/d8Ohc8TBJR9nYMRvAdsBzSEl3Z1OhLEs5DwSJ
O1g2EA3xDKhqMi0ELyninLaZDZ5QC4KO5JnT/1ld1WRQQBucNHHLQ7N2m8ub
1FJj328unRcy/17j5Al6vO/wiEi6+ymT243HIXUgBORQZ6QOHIFD8ALZkrr7
79s5b85e72cgwv79Gg21pC72WjJTyt939CzdI/yQNhAK8jztoKR2aGtjwweJ
3oWaJRw4b4o3EC5E3Mf1q+zbeOHCf1nbH1m3/qWVx81lL5JzmZWqOP2mQfdI
piUOE9bPckX45+VH75ywyVzeHFih96O5wAqBFvwVunPNTSdHDF+HSzQCGvEc
v7jmLOtaPhxe2+q5+LZSpyfy1NNVm0tSD48E6RntCvUp+iyux+JlUjQ3Ik0o
70YJ8zm8ACjHL1TChQIs9LtjtdHHt6X8arknJrVHRlci/IJEvCZuhg1pgQry
znul0AnvyEdZ8ISSoI5wC0y/d+Iz8KQrZgZX0Bwnuefu9qeXHPomHJQoYvxa
JzLP6WoEyYMLkAZhQdlE4sLH9tfVrlP7SivfZDhBHuht+UtNvfjCsdohLWAA
HdKuJ9fS1pYPa4e1Wvm79T6TLx7HUNo+IisJ5gPB0K7inu7DIgmIxuT4Jd0p
Iov+pFyeKMH4FZqTdcPXQQ3GmsVW00SVWHCD0cPf5Iv5YqEwp/rPSySFcY4f
+yNQyjVCrRJ3QO/Ds01QZymjBVMLpIwy5nr1fYe2qv5NWCK4tlr+x4/buGzp
EU+V6jhHp09q0HDpPx2B6J0msqpxioPLEfWRYqxAT1AhT4hM8icUJ/HK2hCp
4WvAfPiw1uIy6pbPF0QI1BEmXvHc29At5Z1RXwZid+w4h1sZLa30WkorArkR
GTdZ/a23XomoxIvFTxwYULUGnN/YOV059ToUCYYZN8aIX6jKKtWY8Mjh8/T6
CTOeiT7E3Cg0SxdXEq4fItMkJAV5vfjC8QT1C9Vggur+jZs3nfF0Ge0JlrKM
hlR7D83aA2hT1tbPwFkD2mwnI9iYgrR2C6TmYwSRZXL+o/uwjmFUu9JgAHBh
pErLA/UjatdNrd+W6EJi2RDtmPneKbKKk0JaFL0Mabghb1dMUsIGHMk8BVhx
addRmbUcT6sqm0AoNEelMJoHqZjOiD3tJDl0cAvRfkNReiuc/XHuvKwAgwc1
QwFESZYrOaJc5it0x4mehcgjzKifFLmnjQi9WbyrGbQWw3J8aN5IlBKmrqSj
3C96/AAamtiOFDsPGK78YS5t/u1WiER6kQnjNxn/JpjKEAuUBGckB5rxWX/N
CBSzg3boCK0Zh7M/+SWixoxqGz60lUGQQTrXQQayebfOWGzDwVhuOH64+mQj
jcQRToWToNYHcFE7tMVogbzRD5MBXpM3egCPcPEqpWAD+kQ8PqB6zcMP7QHl
ACK0atDFbIf27vvJaFkMBdjfN30HesBTYzqfm3vPRR0OrmkeNkQ8TU8yGUVx
7BYI4cqkkNF1Dn9G1a2fNnW7aD4DaIIJOgX64+8566E/slmHkUCJSXZLT538
M+R0+M1PVWHoZP2nyCmtPFnLE04yaPKzmbswMRpSvqQ8jt2RolVVNhhE9fwG
eje35FTJwhB1I9YBpuQUYF24YB+hQEwKp9erJWpJA2pCY6AEQ2kFgxETrZAJ
O68dQtBJPESG3AMWsMXfnPkc17lI05qJVVeteWXViU75dNFVHLmQT12JwQlc
BMjt5jB996QtLCuRN+GzTO2VUFC2qSShqQI0gk8OC55BL90kLYRdrnd4al5z
EpYMRhkeMTf3dROuxOhfM7AZiUKXXBar/lQE0uQr2NyEcRtNHGIGCKsPUgks
VSSBB+SajHRo/r7GZP0tI1TWIcAvAaHq/G7ergGwgV07zz0ye6/YXlJDH4tB
n1VQp1jg4LMMHnpYLmTey9ag6I0UgPgD7YLS7g65aGoi5USC8ZnQV6r84+IX
K46BLV/pSTk5a0FTOM6SffWJ1kJlPOspzQqaSMBUOK2tufi8TjPsQQPXbGg7
w4y8QKN0WfF6We5wQz8ExdYMasbk5BEXw4a2sjWzMOpU9Q/jmzhhE1mxOkw6
JJSG3o0VpxPv/tFWCIxlkj9iEy/RZdzoDSbwxh9WeYhhj1t0mnz31ol3buqS
P49T9Tpt608PGtAMIqRfprp/E56gmTpUqWr6eiIQC3+M36jynNX1Oi8e1r4T
6jJojvSJ4Q2tUqSVo7QY5aPngw/s5hDkMcrPTNUP7njgABPFXlOA16K6S6vH
wVI264HXWBPM/GzGLhgm2xKdwmFOrfG5Z98aIcEzteTsm3LWiNRyRtcJ1agt
ymg2L8Hfp548ZAgTipLRCm2qOkNq1oJOYLf7FeKyrIOfqapshNFf47Swse20
QpPBNyvTObU/uDfkKGoihRxGuYlB+ojh6wg2mp/UcrPaHay+sOKd6qqmhIvP
gho4+YSIJt21eXRdG5pffRl8Rr7FlLUqTeHDeShZ6/PhWtYjwBLBV9m38eLF
L4P6WUbny7vVgRHQ5Hu2IgxVZbNO7WMweKcF8/fdM0nq90pd4EfJH2CKdGLa
1G0QU/cukSnaS6v9Y5SRdeupH2N7hw5+woa0b3Km1OCXNsTIpygt6Z3bzw2u
Wet1LxxlWIljmgv7bXj1/bh6Q04w18IF+2fcvxNdv9sFB1r7nXjnZqSWlgNI
c8Gj+9iQ1iscZDU+HTVi/fRpOyxl4omSsa/A/YD+TWE4S2RrAIBX0Ell34YL
F75gll+q1OTFvzz4wGuQUB33Jb7TEV1pWgKG7t+v4eQHfyL12eXLjiLa5ige
XZydPSCdR2ab9MvGj0XRnkYgVT+lXc2FI/icHlp8ctEBGK34MFe3sfkfuixb
esTkdZ7uUPrMLHVPCcT379fovI7BArJAs1DQlv0qwXxabQS/T//jYURExvES
CZG3TTL+NdmZoVHknFVdcYgp97TXT9lma9x+L4f84P2K5955edUJq93rOucn
rXMve+bI+HEbybupmo8e2WYyu5JohqsDKyWYQO1Tiw+atuQm8rTG4lpbToEU
SAyDEDdZtUM6YijO1CYEsz5aEbKOrn7lPfSzlTXyYqhskUwn5or1Et4TBKAE
iP34o7903C0QG8eo0CjoR1tTk5DICZ00lh5S06w7YnEVMN5xzfEeHDCgh5HX
DV8HERVtHdNSptrIi/vVoGzWg7tBXbz7yRmN73SPQB08JhYZ52UvO2OY83JG
So2eOp4xzqeu6hxZtop38MAnsDf2g3SBF4057b3PcDHC3EdeNzqI18OKAIXG
f5UY40iOHtVmbUwwVRZVWHityVyFQ6fdX+Qc903fOXjgGpg/vXWZemUdo6nx
JKyQWL9W+S1x8a7PYbkOWTl7i+F9vJcDgp5ecshE0NSTz4bET2Yx5Usrj5OU
xoYPYCK63paq18u7FRXvsMRPmbyVksqFWsW7gdVrHpu/H03FespYXuJbENm+
5SxJ5Vg9iupUMtO0xgbDhra0rrUVsKTPJXKRjWh9b/mytyBKn8/aEwGMODD7
4icOsgskCDlSF2GWo/4UgsQihSnL3Nx2D5+rKLODX75wDEsFudWdb9rH5k1n
GF3T1ogKiMPujpd0wT35QldkbqYyRO79GpRzCEOIb2yJuyCsFtGqdJ/AVnC1
+gcfM26Mde8Dq5sOHzpvowdW4EkGtcoxRo/asOpX76q1ddidAOyRTcOn4qnI
oTyUJnnXddHW5RjVRLt5hWi3dccd288hJSP/fvUqc1bLyoCPtKF61hSSEa0y
pql6wWVpsmfzYqE6rdkSUz2/R4cVwyPO7/JS5qasMfcN9iX3oV3zWm2Epmqp
lxhmt1lpiucA3JVR7/LAzNcQllib9r6lophOzE7j2bOfm1MFev7ErtCMLZD+
DBvSQhA8s/QIFg6zzmmFn4K0Y+EGbefOed36ZD0lktDTFIyw7N67rOvZ0PJB
zamTf3YVuzBi4b4394swOSypME+gNajM4V327P7YysCf7qmI6CIaaC2MiWM0
yLgZwBY2mhfHVVe7zuX3BXkXo3DSDd09/+xbpnKTdlqwtW9WNpDfV1U2MW6T
uLoi9AhEu8pFS1mUAWlj5tkVHoglRMX97mjwlQ+vX4mMCs6fcWSj27gtI5jD
b346oHqNWitWdRNhCf1ppbaglspsBK3w1MehwDO0ZFpxtwJrIXJy2qyMlgv9
gtZ18DCO/LiDihe0duszupCK3R1D+AZ1pSySuTcIPrkZmJeiJOimfoghfw5C
NTpq5HpYR4f4mPLQykBYsUup5wTzJWJ0kFvBGZghg9du3XKWnSiL0EsQb5SU
rluY2tsYZQ3qmEDYvDOX73wvxeX7PPFAPU6t3/74z/cnIw8ikX1OowY1tjAu
8/sgcp+DHCZN3Mx0jNzTziglPUFIluyOfpi/oalDvolqEDXzPJ5Bj1SGc6H/
gC+6ePErFiI4FtqhAU9r4gJ+IMYQ0UJ0dFWnLxot+31/UIbw2krYo4UZLyVE
Z6mQ9idobGZIjLBCiL8YV6t5CPiMNovvbsQRDHPyxYsOjhndltddWEpLV9K8
y0X4m3JzQxMALtiKOfl/ycmeKnj7rd9XVzXFuLq2EFd0pFpQIo64h4G/aMOF
kl6ds+EJhjbH4sTq07pu2JVG1ydbBS51ckRcg3jElhylbyr0iFzn/PrB9Tja
87lCmSTdXoj14I+J14ulfF2AF43ibJ4pvcthXqy35mWrgNZG4dkTnZSB4sH6
F87kz1xyxcTMY0ZvAABsFUb65sJ1jRCh7ZY7adPiOuXf+ZR4p4D8E3FDB7e0
b/mNRp8WC4wJSIyuwpY20VpOqx15t1LS15Jr1TEfWq59LKZxBEJbBLhmZwE6
lt0dXxHVUIqiYzTra2OpLri+3MlgoRCDX7r0PzamjE94iWc0frS15UPjE/me
XlM5yKlslCzS2jOQMMckFT6XEx1keAaELpBSpAxC/8ARiU3KwPuYwnXF7QwV
nLpOUW0rXzw2dszGOKMkl7iZdNcWH+95O5ARStXT5d1C6m2anJJH6pec+qyW
p3qIrZdXnYCe88672N4dbV1XaBvFe3spXsFlBEOwjR2zAcyWRFRIj+zCx/ZX
VTayk0c7qyXqX5RdYlxODtiYgaghwrz/EZoyPNdAbbG19CztaMV5rULmIkvu
iju0On/+EuAK0JZEerze6REJsI/wIpyUakacd4uft2qeqiD1YSThzxpwR4k5
HEjgGmp739ABJ3ptz9MwdlAzwSxma7AweyCflHrDq++PrFuvJ5+t35B5shPG
bfzF82+rFiFL41nJoyJF1YtrqiY4FeV3jKRXGSMgKsGctBApqfhK3Ema0tAu
4/MdjETUg/B1J1FweLp84oTNzy4/WhJhSjhNNjZ8UDu01dcYqUFTYWBdvmOE
XnQcK1PSxvUL6IN4+v50NePWfFBHsZST49A2lTcbTZNj9ot2HotznIWqECFU
3tEAAOVdIG3PfkEH9ZPbte5l4kzQDPJ4UrZAAr5K5/fvhV7jsX+979/NqKwg
hZhk2xs7WKXSTjHZFUJ72jXM26buJFmv8OTiA/WTt8WSvzFowRhCIy6ehbNx
dphz0j56OXS8vOo9gouxpMbpNlKSdgU18ThWSQX9vCUX9zK0b2KShSZv3zwL
CkbOn/8rxZ91OrPfHsX+VnRqz4rSq5AH5db6BeGhLLQrxbSqhZUb9iSH6Y66
MN6wufFUzaDmROwdZY40ewL3zMxsnKAndDqch+vl9LlwwX6DdtKvZ8KtDZD+
qKqqtWZajGYUeL592786GxTv39m3CnEUw5MXlDf3CtXk5ey2+DUjCDSBlWnm
wNMpxmNQSr4m2ctpjBuLttKkGs27jLKXk1O4J1kengLFNcQGEO7Z/fH0aTus
fMljaTGUFcRrcXxGTkH0K786USNnFn0VWc4bl+sqmpeFhKeHwD+4YO3ASs/7
hl7O8saMahs7eoPlVVGnKhOcZ8Jcln5Cs3pEPf36rl4u20Bc066QQ9NAFA6z
XPTEgZpBa2lg1MfN0lEzU4tFNOX+gt0p8DrguWvQM6TGHNqyJ5TDnUDKiO18
9BT7qBI3j63T5r6JzsyOVr++De8e/4zfs00Yt+npJYcsFvl9iY+Xyx3arH8g
ZaC4ZmCzqRGSg/AMDi2R7XwkQ3Twy0AqVwtzpPtKqS/Fxb2T2/ve0TC4Zm11
ZdMzSw9zUuOV5PuooO5J6r1voL7mzdl7/307BRqOm3DXPPgW66Zi+k3zMysI
K/xa4Qrpl1MXmZW/PI706cnFB0Eatyp/6JBhc+XyiBJSfuHCF/a7JEbnodx9
fs628ZlaSBokG3kTlTzwrbrRbMSdDk4X0o+OJSYYSxIY5iWl7WtBpFlPCvGb
JnMubt7c1z0d0W5CgK2bLQZ5EiCqtXdY9+TrxWBvWvcq7A5jH2dDC+a/YWJW
zsbV1EghrlVTaj5mZbXbx/iFeye3RTKm9n0GJrPbTBwsYRgIid8LJeKcsnc0
U1lESfHI7CcR/fH3kUKB39fA2CZz0dXTeiRKycfVZZHFdDW31xzn9n6byPBr
ViZCd59IV5zVe53C/OK2SNPxmVrajvc1mch2fhCNHu8wFY+rYg1nIx44q/cd
6ktsBhxbdLyT2vmsxgZyEQ/8otN7Nld9/1FERdbJyubIjLiKRSxxBfv2SKqZ
SOq3RO/jOuQtEf7Y31dmSEnsJYueYQjub49sI96hvDWaNa6ExFRzPm97V05V
UmXkdZiOtCLUZDViz4vb5IrHb+Pj6kNgQ8XjriuTT/wlgccDv/oHeQAy4GI9
0I9dL+uJqUu/s1c8QnLrVxzj/l0k02SEfHoLmBEIMKPfFr0vultZdPZi1Px9
JMe4JjpBJICpZ8tnGyDTnEcpVbuIK3DhHDK2/woTf2zuL+1gmhjQRwCxLbM3
0MD9OpgCxX2/tjZz/oMww4I1PyacUXS04v+3DT5NXPV/JE4zxw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->143354055]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->241093779],

Cell["\<\
Different equivalent ways of specifying a harmonic oscillator as a \
second-order equation:\
\>", "ExampleText",
 CellID->265604313],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
       RowBox[{"x", "[", "t", "]"}]}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", "1"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", "x", 
   ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->723423364],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14214380]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", 
       RowBox[{
        RowBox[{"x", "'"}], "[", "t", "]"}]}], "}"}], "/.", 
     RowBox[{"First", "[", "%", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->443931913],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->109474841]
}, Open  ]],

Cell["As a system of first-order equations:", "ExampleText",
 CellID->781148539],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"y", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"-", 
       RowBox[{"x", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", "1"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->323930071],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->943728904]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", " ", 
       RowBox[{"y", "[", "t", "]"}]}], "}"}], " ", "/.", " ", 
     RowBox[{"First", "[", "%", "]"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->396563124],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->282295871]
}, Open  ]],

Cell["\<\
Using a vector variable with the dimension deduced from the initial \
condition:\
\>", "ExampleText",
 CellID->515271795],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"z", "'"}], "[", "t", "]"}], "==", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}], ".", 
       RowBox[{"z", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"z", "[", "0", "]"}], "\[Equal]", 
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}]}]}], "}"}], ",", "z", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->493937889],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"z", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->764640490]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"z", "[", "t", "]"}], "/.", 
    RowBox[{"First", "[", "%", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->207087099],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->677241]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->125568132],

Cell["\<\
Use matrix-valued variables to compute the fundamental matrix solution:\
\>", "ExampleText",
 CellID->36857027],

Cell[BoxData[
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "5"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->79547994],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fs", "=", 
  RowBox[{"x", "/.", 
   RowBox[{"First", "[", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"x", "'"}], "[", "t", "]"}], "\[Equal]", 
         RowBox[{"A", ".", 
          RowBox[{"x", "[", "t", "]"}]}]}], ",", 
        RowBox[{
         RowBox[{"x", "[", "0", "]"}], "\[Equal]", 
         RowBox[{"IdentityMatrix", "[", "5", "]"}]}]}], "}"}], ",", "x", ",", 
      
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}], "]"}]}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->4535099],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{258, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->63540726]
}, Open  ]],

Cell["Compare to the exact solution:", "ExampleText",
 CellID->270259617],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Norm", "[", 
    RowBox[{"Flatten", "[", 
     RowBox[{
      RowBox[{"fs", "[", "t", "]"}], "-", 
      RowBox[{"MatrixExp", "[", 
       RowBox[{"A", " ", "t"}], "]"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->215987619],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->254608145]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->122523365],

Cell["Define a van der Pol equation:", "ExampleText",
 CellID->5349239],

Cell[BoxData[
 RowBox[{
  RowBox[{"vdp", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"y", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"-", 
        RowBox[{"x", "[", "t", "]"}]}], "+", 
       RowBox[{"1000", 
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{
           RowBox[{"x", "[", "t", "]"}], "^", "2"}]}], ")"}], 
        RowBox[{"y", "[", "t", "]"}]}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", "2"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->201941871],

Cell["\<\
The solution's \"stiff\" behavior that the default solver automatically \
handles:\
\>", "ExampleText",
 CellID->377329246],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"vdp", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "2000"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->12698415],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "2000.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "2000.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{330, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->24444224]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "/.", "sol"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "2000"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->86660993],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmV2IVVUUx/fcc+6de2csS50wrBwqKsuw70+nsg9ThpqkZCIzuA7CjJCG
Tj3US73ki77oS0I4lB/gx8OIYgQG4oA4IA2IA6GBjBDjg4Og+HBBuP322uec
e2bfc+7d3jCnoYE596x91v+/PvY6+5yz9nvF/t61nxf7+3qK7e9uLH7R29ez
qX3pho0MeU1KqSP8r5urOC9zJv9HymV+48ccR59LWS15Wkf+1M9ytTk2qlK1
lBdc5i8TijEtPxFVsEY1SouvaiH3zde/vvPWjxvWHx7Yefq3Y3+OjV0ppqLi
sqBbwQ2fujjy+1+gv/v22OpV+zqXDxgyxkulG+rTKHrlV9DVEQqfBxSyyF7h
2rWSIf+y/yhu9qw5uHXL0C9H/8BN9XGUOZ2LXB2+vBzOn7t8aHB08/fH456i
8ZFwzazDMWOyT1x4vWMHxx0/DA+duMC46op4tE+LLfzd4gPCnt0j5B0HCGj7
tpMmVZ1OPrTpQzNxHNh/xiSlr3cQB7isljkx3CsMo2cv4QWjMOALbDqnbzsx
3CcMuEzUuE8uu1fuZWK0D0tqZqBdkOPjV5kFRskAc0HZ6UrpcLL9kDCQa6qA
ioOBY8DwihPDI1L36IMyDPjAFLzkhF4QorFP/gmc5E9MXH/eCb1QH7LET+5X
dO3CPIbVszVztkjuHZzFHPWC3aedbImWT6UQHm5CsMgJJ974+Ih31BaF8qQT
TjLgow6IqiBFTzjhXhSZAgSHVfV4zWzILHlExZDWXuBk42WR0SeB+j59rByu
HZ7DWiR15Zly0aePCjpvoezV12YxMjOBG1EdOq3p0WpdxWZqv8LWmE8UCFU1
NVlc1/dkdFoVpWs3ZmPqo/7tLOhnhBfUcyb2O8tiC99GMpZe+NsWXDflE/7Z
WnJiXBILfg1GL9Hf+TFZBeg0WZ/OtOIQ9ya94dRiSGKcbcmeKz6IW/DZ6LqD
1VDWL7FN+ndStbT/5zMypyEPqrMxveqk7SbsueXk/qBiBJmLx1Yd4S3MSYXB
1Z9bmpHwXoyM365aqfLDc8zLLIf5bCw/D8RmqN7TIW0ND58O+tdP4Ys/F2bX
4ao8F/5/Y9Bfuj+poC9RxTHHkmdY8h2WnLVs2vr1+HKWj/abpI3PO0VuW1lo
WZnXEOvUeQObmqjploXpFs/UzIK+97xgnb95tFMv8zayaNFo96w5eP7c5cZ9
MJ2K8fGr3Sv36tPF/4Bl65ahQ4Oj6Sz5YN7t7onptaAKQJ921Kwaoz2w83Sg
/ZpT1bwg8sTEddPBVW84oZ7TB79UugGEBA2furjECfeMPmTBHdh/Bhxo3W18
0wn7lGDxdM/uEbDbt53k3PRZU3Mifbrs6NlLm6VzZxqMaqmTPem55Ux/lGrq
6x3kBHeXOaGlm5aNN0YpR9XphJXRLMWPYtSWVe87YR/Uh2bTkzVwJhaE7tV9
UDNX8pXaPDZ2JeqHYzhAfuhke15oe+jEBWqehK9etc905VW3E8NcYWC+qAwS
bjrq3Du6P/mJxSC6Pv5yHS3CbEnklHeaoM9v9gpghh8r6jOLU3TzhpPoV3Tt
IgJCMZsSFX6dr7u0VCDaaF8l2oYI9jCKFrsggp2TODtTRVnemeh9q1ghfpRg
NkmJ9oW0lbWWldYUvyJD5ttA3kj9hCeB2UEz+cJLlWEepT+KrHO2zrIn+gVy
howNLNke3lMOV7n4t41tNxPymEjNnpDJJyO6AtZblg2CiMi32UXCMilFgwzr
+00qMnmVLiBbOBOj3nvZpDVa4GHFMGkgCcwX9WO2uUjvfOFuSYjCk+8lo6XH
vpL7GR4sVaJ5OBUfl2O7mpO+X5N3KNOeJOZq9V6qavobKEGMDQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->659258409]
}, Open  ]],

Cell["Other methods may not be able to solve this system:", "ExampleText",
 CellID->601641067],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "\[Prime]",
        MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"y", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["y", "\[Prime]",
        MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"-", 
        RowBox[{"x", "[", "t", "]"}]}], "+", 
       RowBox[{"1000", " ", 
        RowBox[{"(", 
         RowBox[{"1", "-", 
          SuperscriptBox[
           RowBox[{"x", "[", "t", "]"}], "2"]}], ")"}], " ", 
        RowBox[{"y", "[", "t", "]"}]}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", "2"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "2000"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->156182478],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ndstf\"\>"}], ":", 
  " ", "\<\"At \\!\\(t\\) == \\!\\(0.029952638056252905`\\), system appears \
to be stiff. Methods Automatic, BDF or StiffnessSwitching may be more \
appropriate.\"\>"}]], "Message", "MSG",
 CellID->42902036],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.029952638056252905`"}], "}"}], "}"}], 
        ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.029952638056252905`"}], "}"}], "}"}], 
        ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{358, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->6800184]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->16425158],

Cell["Equations can have multiple distinct solutions:", "ExampleText",
 CellID->596309688],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox[
        RowBox[{
         SuperscriptBox["y", "\[Prime]",
          MultilineFunction->None], "[", "x", "]"}], "2"], "-", 
       SuperscriptBox[
        RowBox[{"y", "[", "x", "]"}], "2"]}], "\[Equal]", "0"}], ",", 
     RowBox[{
      SuperscriptBox[
       RowBox[{"y", "[", "0", "]"}], "2"], "\[Equal]", "4"}]}], "}"}], ",", 
   RowBox[{"y", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->131565440],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWNtu1DAQ9eaybSkSFFWlIFWUD0BCSPQBiSek9okvgJdVhdQ+IFDp/4c4
G28nk4k9djar3fZE2vV1LrZPJuPzfXF/8+v34v72enF+dbf4e3N7/e/88s9d
3ZXPjJn9MMZ8OzV1vaprzc8+P+vfsrqsHdr/wo3axzbKuszMctSWF3WZ1+XH
tvza9s+tdL6SNgVrv2Ttk8B83j4hdqg/ZUAub9b2LNLaIWt/Zu1PrP0lUv9R
6/2pwnvNPO5vyH6m35UOIqwvWVvy2VybPZnSuCNyj5tkV+bKotVatmXRanfj
JStzMl/WnhGtOUFxC4fmsf3LRbiHzqY2qM2+VB65905bUclvzpxY1bxZ4jk8
nFpd2R91rulYkt7PrOcdn8GxzNffaCip96tT2GO97diecAJ9DbJecbWydbk3
KOvRyH2mugYlN+VhEk5ED2hcPxDWrX0LnLxDsUYujEcd/rnfdJYRtPrWdSBg
Ng0lktUJkO+xOBrzStks4McYv8ScIpRz9LTwjOd5QIsdtwjZr3RxeQhRITzP
Gv94/G7W0HkvtPZGxG+mvRzoH3NuxYSYXbW3ELNp0ZljNpRnvqjiom9qjiPn
HBFxzBvxuV4pFg9/f0ux1620oLIdL8Rv8Vrir2zPjdBMXEaXx9/efN13q6PV
s+bNoPaotzYxY54ga2C5ejV4v1Bn68fi7M69p+reahwyiVVByt3HpIyC3tMy
rpXp4H30xDkyZK9sB78x8lsbt5J0OqNxdSaNT3zv4j7x9jFrc/n3gfE34s68
ZrPebs9+R8jLO8yl7I5QBsruqHX9A2nT+Zb5mtfNd6z/VaW/oeTtbtj5obzx
jOiNZ8pslTOD8TVwicOo23UukWYy4BANOETFuYJDVFoHh5iQJ+wmh+juXuAO
hyyCO3RatoU7bL4C4ArXgNFVewsxuttcYRNNwQ0qsQtu0K9lE9wgQSy4QBIf
wQXGxThwf8PjIfnljj5Wrq85yaYBbk+WA7dH9YDbA7enPVdwe0rr4PYS8gBw
ew9y4PbivQW3B24P3J4/Wq3/NgZuz/WC2xvU6lkzuL2q8mfd4PbA7fXw4Ill
4PbA7YVrYYbn8XB9sbzM0+L+knYHXCC4wPD3QLUL4AY1suAGd5kbjLn1GEE7
uEJwhX4t03KFqjWAOwzaAHdI29NyhyHMgksEl5iC2s1xiSKCwS2S+AlucVwM
BNeo3xt5hzX37qfHPZrZfzjMw6w=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{347, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1061009135]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->496817457],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->68497495]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->871106438],

Cell[TextData[{
 "The solution ",
 Cell[BoxData[
  RowBox[{
   StyleBox["y", "TI"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " is continuous, as it integrates the piecewise function once:"
}], "ExampleText",
 CellID->23617400],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}], "+", 
       RowBox[{"Cos", "[", 
        RowBox[{"y", "[", "x", "]"}], "]"}]}], "==", 
      RowBox[{"Floor", "[", "x", "]"}]}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6317317],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "3.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{309, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->864697541]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->107509006],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->38047015]
}, Open  ]],

Cell[TextData[{
 "The solution ",
 Cell[BoxData[
  RowBox[{
   StyleBox["y", "TI"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " is differentiable, whereas ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["y", "TI"], "'"}], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " is continuous only:"
}], "ExampleText",
 CellID->24835091],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "+", 
       RowBox[{"y", "[", "x", "]"}]}], "==", 
      RowBox[{"Floor", "[", "x", "]"}]}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "==", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->206682691],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "5.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{309, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->25026241]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], ",", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}]}], "}"}], "/.", "%"}], "]"}], ",",
    
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->216482258],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->173216352]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Boundary Value Problems",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->5187909],

Cell["A nonlinear multipoint boundary value problem:", "ExampleText",
 CellID->51965684],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"y", "[", "t", "]"}], " ", 
        RowBox[{"x", "[", "t", "]"}]}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{"2", "-", 
        RowBox[{"x", "[", "t", "]"}]}]}], ",", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], "\[Equal]", 
       RowBox[{"x", "[", "4", "]"}], "\[Equal]", "1"}], ",", 
      RowBox[{
       RowBox[{"y", "[", "1", "]"}], "\[Equal]", "1"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], ",", " ", "t"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->77218768],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindRoot", "::", "\<\"cvmit\"\>"}], ":", 
  " ", "\<\"Failed to converge to the requested accuracy or precision within \
\\!\\(100\\) iterations. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/FindRoot/cvmit\\\", ButtonNote -> \
\\\"FindRoot::cvmit\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->220104526],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"berr\"\>"}], ":", 
  " ", "\<\"There are significant errors \
\\!\\({\\(\\(-0.23438463259491904`\\)\\), \
\\(\\(-0.0031120658361388776`\\)\\), \\(\\(-0.05555409522286525`\\)\\)}\\) in \
the boundary value residuals.  Returning the best solution found.\"\>"}]], \
"Message", "MSG",
 CellID->205823825],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "4.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "4.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{309, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->235329110]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", " ", 
       RowBox[{"y", "[", "t", "]"}]}], "}"}], " ", "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}], ",", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Point", "[", 
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}], "]"}], ",", 
      RowBox[{"Point", "[", 
       RowBox[{"{", 
        RowBox[{"4", ",", "1"}], "}"}], "]"}], ",", "Red", ",", 
      RowBox[{"Point", "[", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->139000656],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->16517113]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Differential-Algebraic Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->16331],

Cell["A differential equation with an algebraic constraint:", "ExampleText",
 CellID->218712133],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "t", "]"}], "==", 
       RowBox[{
        RowBox[{
         RowBox[{"y", "[", "t", "]"}], "^", "2"}], "+", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], 
         RowBox[{"y", "[", "t", "]"}]}]}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"2", 
         RowBox[{
          RowBox[{"x", "[", "t", "]"}], "^", "2"}]}], "+", 
        RowBox[{
         RowBox[{"y", "[", "t", "]"}], "^", "2"}]}], "==", "1"}], ",", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], "==", "0"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18805],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5868487]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", 
       RowBox[{"y", "[", "t", "]"}]}], "}"}], "/.", "s"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13139],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmL1rFEEUwPd29+5iEpMYvZj4EaMW2von2CgiIhZiIRZnEJJCI/HSaKON
Ntpoo5WFH+BHEREiQgQxIKYxIAaCChJBYmEQDBYBYXwzs7tvZm7m7V684o7k
4O5mdn/zvubNe7d3pFwZOnO2XBkeLA8cHC2fHxoevDBwYGQULgU5z/P2w3uu
14Mxg5F4b2aMeQEfiZe3h394tV714eJeK1MQ8xcTcwsLv/mwn6Cm3ny9dPGl
myqK+eDpp58//eTD7UiFkvLEK1DW8bmfXJQvMQ/lXIrORvjwHUbf6LdvWBnG
VkYGCFl5VZYXryqoVsuBr3ijXFG9yaAdIlkn7YEhP0dqz8fzaIf8WEJelZli
lyKPz3nicq20Zq4lEDIhhVSXkl2Ltel3guhGdi1Krp04/lDNnNhihTByKyEw
Hoot1Tboec3zvxDZG3/HViaRTmxvEwYvL/8FKxcX/3hbUKrikaSePP5w6+Zb
PuwjKJRlp9rFHASBOD7sJSiQArJAYkR1wOe+JLs7lfzEyOl5W9N10PU/ckKr
Jy227OD5B8cPd7bVoIpWWR1iDscGCgcf9lgpEZdgrDIx/e4bH5a0He9iTKui
JasMSWEV3URQULevXX2dRmFe2KkNTM+xjQS1tLR87Og9nhd2qpvpOdZNUPPz
v06dfOSmhIYg2S9pgYPCqNop4XkA6kBpEplkZ0pGBLoIGeD74UN3eQTsVMmI
QCdBYQTsVI8RgQ6CwgjYKTmHRIg6/XqCenB/5s7taaSKTKlgRucuWCVg/ZAS
BKV1rqKxzuxkvYbv7XKZ+Sugqkdlkzvz/vvIuecoN5t9fNpnZFKbY31c/c34
yPWYa9R6a+/W5OB5bNN2Kv42V4kuE0DRgNLBh63WLJDU7Mcfw0PjbmqrmGOl
WkdQz8Znb1yfSqPwbNmpbWIOgkAcH7YQFNZqpHhE+g2r7fVe/HZVzgFFQa7D
HrgpqRG7QoGgXk1+uXJ5Mo3CONmpHcYu5wkKT4KdGhBzcBDc5EN7r5UUBAtC
lkZh1UOK78wucRdSDhKPxSe3SsZOpp8fisIqZKekRqwxPkHhzlAU6IpywU7t
ZlEGR/McQckwuKl6PwuuTF4jXG1en7TOWvXUYfYOuaa6V5hcaF2X3jv5ujC6
wZe79Knya7HT1RNXxqu9Lt37Rtjt5s7V1eXTmqeN72ktzyWuGpP2vJC1drqe
F+pVC7PV+JX6pz4vBE6rGmHPmyU31zxdTT41r6euGhp41f8eULVMXWtfl15D
61njqNqbrSY6fveS/9LoMS0wqsY3wu57uX/YjTev\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{65, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->52613757]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Partial Differential Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->484808727],

Cell["Nonlinear advection-diffusion equation in 1 dimension:", "ExampleText",
 CellID->249566227],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x"}], "]"}], ",", " ", "t"}], "]"}], " ", 
       "\[Equal]", " ", 
       RowBox[{
        RowBox[{"0.5", " ", 
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], 
        "+", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x"}], "]"}], 
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], ",", "x"}], "]"}]}]}]}], ",", 
      "  ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", 
         RowBox[{"-", "Pi"}]}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", "Pi"}], "]"}], " ", "\[Equal]", " ", "0"}], 
      " ", ",", " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"Sin", "[", "x", "]"}]}]}], "}"}], ",", " ", "u", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "Pi"}], ",", " ", "Pi"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->197355455],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "2.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "3.141592653589793`"}], ",", "3.141592653589793`"}], 
          "}"}]}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{451, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->86502572]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"t", ",", "x"}], "]"}], "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Pi"}], ",", " ", "Pi"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->22369196],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->54459521]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->378646315],

Cell["Define a system of PDEs of mixed parabolic-hyperbolic type:", \
"ExampleText",
 CellID->13424634],

Cell[BoxData[
 RowBox[{
  RowBox[{"pde", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SubscriptBox["\[PartialD]", "t"], 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "x"}], "]"}]}], "==", 
      RowBox[{
       RowBox[{
        SubscriptBox["\[PartialD]", "x"], 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"v", "[", 
             RowBox[{"t", ",", "x"}], "]"}], "-", "1"}], ")"}], " ", 
          RowBox[{
           SubscriptBox["\[PartialD]", "x"], 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}]}]}], ")"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"16", " ", "x", " ", "t"}], "-", 
          RowBox[{"2", " ", "t"}], "-", 
          RowBox[{"16", " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"v", "[", 
              RowBox[{"t", ",", "x"}], "]"}], "-", "1"}], ")"}]}]}], ")"}], 
        " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}], "-", "1"}], ")"}]}], "+", 
       RowBox[{"10", " ", "x", " ", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "4"}], " ", "x"}]]}]}]}], ",", 
     RowBox[{
      RowBox[{
       SubscriptBox["\[PartialD]", "t"], 
       RowBox[{"v", "[", 
        RowBox[{"t", ",", "x"}], "]"}]}], "==", 
      RowBox[{
       RowBox[{
        SubscriptBox["\[PartialD]", 
         RowBox[{"{", 
          RowBox[{"x", ",", "2"}], "}"}]], 
        RowBox[{"v", "[", 
         RowBox[{"t", ",", "x"}], "]"}]}], "+", 
       RowBox[{
        SubscriptBox["\[PartialD]", "x"], 
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}]}], "+", 
       RowBox[{"4", " ", 
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}]}], "-", "4", "+", 
       SuperscriptBox["x", "2"], "-", 
       RowBox[{"2", " ", "t"}], "-", 
       RowBox[{"10", " ", "t", " ", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "4"}], " ", "x"}]]}]}]}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->44479087],

Cell[BoxData[
 RowBox[{
  RowBox[{"bc", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", "1"}], ",", 
     RowBox[{
      RowBox[{"v", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0"}], "]"}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{"v", "[", 
       RowBox[{"t", ",", "0"}], "]"}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"3", " ", 
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "1"}], "]"}]}], "+", 
       RowBox[{
        SuperscriptBox["u", 
         TagBox[
          RowBox[{"(", 
           RowBox[{"0", ",", "1"}], ")"}],
          Derivative],
         MultilineFunction->None], "[", 
        RowBox[{"t", ",", "1"}], "]"}]}], "==", "3"}], ",", 
     RowBox[{
      RowBox[{"5", " ", 
       RowBox[{
        SuperscriptBox["v", 
         TagBox[
          RowBox[{"(", 
           RowBox[{"0", ",", "1"}], ")"}],
          Derivative],
         MultilineFunction->None], "[", 
        RowBox[{"t", ",", "1"}], "]"}]}], "==", 
      RowBox[{
       SuperscriptBox["\[ExponentialE]", "4"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "1"}], "]"}], "-", "1"}], ")"}]}]}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->36520046],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"pde", ",", "bc"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"u", ",", "v"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->62472787],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"u", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0.`", ",", "2.`"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"v", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0.`", ",", "2.`"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{373, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->311532987]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "x"}], "]"}], ",", 
       RowBox[{"v", "[", 
        RowBox[{"t", ",", "x"}], "]"}]}], "}"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Directive", "[", 
       RowBox[{
        RowBox[{"Opacity", "[", "0.7", "]"}], ",", "Red"}], "]"}], ",", 
      RowBox[{"Directive", "[", 
       RowBox[{
        RowBox[{"Opacity", "[", "0.7", "]"}], ",", "Blue"}], "]"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->22821469],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 140},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->98558905]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->861634095],

Cell["\<\
Nonlinear sine-Gordon equation in two spatial dimensions with periodic \
boundary conditions:\
\>", "ExampleText",
 CellID->102297288],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"L", " ", "=", " ", "4"}], ";"}], "\n", 
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", " ", "t", ",", " ", 
         "t"}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", " ", "x", ",", 
          "x"}], "]"}], " ", "+", " ", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", " ", "x", ",", " ", "y"}], "]"}], ",", " ", "y", 
          ",", " ", "y"}], "]"}], " ", "+", " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x", ",", "y"}], "]"}], "]"}]}]}], ",", " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", 
         RowBox[{"-", "L"}], ",", " ", "y"}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", "L", ",", " ", "y"}], "]"}]}], ",", " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", "x", ",", " ", 
         RowBox[{"-", "L"}]}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", "x", ",", " ", "L"}], "]"}]}], ",", " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", "x", ",", " ", "y"}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"Exp", "[", 
        RowBox[{"-", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"x", "^", "2"}], " ", "+", " ", 
           RowBox[{"y", "^", "2"}]}], ")"}]}], "]"}]}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"Derivative", "[", 
          RowBox[{"1", ",", "0", ",", "0"}], "]"}], "[", "u", "]"}], "[", 
        RowBox[{"0", ",", "x", ",", "y"}], "]"}], " ", "\[Equal]", " ", 
       "0"}]}], "}"}], ",", " ", "u", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", 
      RowBox[{"L", "/", "2"}]}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "L"}], ",", "L"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{"-", "L"}], ",", "L"}], "}"}]}], "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->211175700],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "2.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "4.`"}], ",", "4.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "4.`"}], ",", "4.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{453, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->455443847]
}, Open  ]],

Cell["Plot the solution at the final time:", "ExampleText",
 CellID->168065660],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"u", "[", 
     RowBox[{
      RowBox[{"L", "/", "2"}], ",", " ", "x", ",", " ", "y"}], "]"}], " ", "/.",
     " ", 
    RowBox[{"First", "[", "sol", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "L"}], ",", "L"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "L"}], ",", "L"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->104977509],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->343625809]
}, Open  ]],

Cell["\<\
Plot the time evolution of a radial cross section of the solution:\
\>", "ExampleText",
 CellID->414994633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"u", "[", 
     RowBox[{"t", ",", " ", "x", ",", " ", "0"}], "]"}], " ", "/.", " ", 
    RowBox[{"First", "[", "sol", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", " ", 
     RowBox[{"L", "/", "2"}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "L"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->176636832],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->24957344]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24370],

Cell["The names of functions need not be symbols:", "ExampleText",
 CellID->6676],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqns", "=", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         RowBox[{"y", "[", "i", "]"}], "\[Prime]",
         MultilineFunction->None], "[", "x", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{
         RowBox[{"y", "[", 
          RowBox[{"i", "-", "1"}], "]"}], "[", "x", "]"}], "-", 
        RowBox[{
         RowBox[{"y", "[", "i", "]"}], "[", "x", "]"}]}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "2", ",", "4"}], "}"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         RowBox[{"y", "[", "1", "]"}], "\[Prime]",
         MultilineFunction->None], "[", "x", "]"}], "\[Equal]", 
       RowBox[{"-", 
        RowBox[{
         RowBox[{"y", "[", "1", "]"}], "[", "x", "]"}]}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         RowBox[{"y", "[", "5", "]"}], "\[Prime]",
         MultilineFunction->None], "[", "x", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"y", "[", "4", "]"}], "[", "x", "]"}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "[", "1", "]"}], "[", "0", "]"}], "\[Equal]", "1"}]}], 
     "}"}], ",", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "[", "i", "]"}], "[", "0", "]"}], "\[Equal]", "0"}], ",",
       
      RowBox[{"{", 
       RowBox[{"i", ",", "2", ",", "5"}], "}"}]}], "]"}]}], "}"}]}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->10143],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdW81u00AQ3tpJ+kebQn+IWop6QJw48QAoJzhx4BGiCqk9IFDpM/BKgIQQ
Z94B+ijBXsfUmezu/Oys6zpS0o69uzPz7eyP51u/nV1fvP8wu748n529uZp9
urg8/3z2+uNVcSlfMyZ7ZYx5OTHF//PiP/utP9PieytW0nb5O2iWKoVh8Tcz
1d3y7wjIp2Wt/H8ts35P6g2AfGMx2EJKQXmbef9kYQ2mp+/tPkHK/xX1Rlb8
DAsRK5cXP/kiZprXDzzXN4TtpdY7XNTLe+IvPQoyR+3MQtKcvwwhFmi1aqSr
G/Vn3Vu3tG/znts5clnQKQtTIjly2jkSaazieDNQyji8g6MaGy00v/adpXYR
W9LEgsyWsu2RwWe9vuCB9fsfWw7Okhad290cIUrbn4F2URvvfg7CbezKfH5X
aOrPlDCW6ViF5k/NMblD9s61brVvCX82aMcmyi4wxhJGvC09+a7GM4ymNleA
B/fUIrjDSRlzNIva3FPJ5jr4pJxydXHbfdQ5C9qMI6i7exEDI6SMoGxROze3
T00jz/Wjhfw4sl5dLhfoGzDqYajcWFRqHKfJpPHcnWncBPKhpxzMSD4klsMi
bnX1olh5KtTuy7duzWUoUNvn5nPtuLUYh0rBsbSO3D9ktofd59rHvV+Psdi8
NoYL1htcP6h2w3owc5m6/6R2YvdhfrzWU21usKjx5YVtLy7NFGWbWbNtBlbB
/TtZi9QXnhZMxjLerlXexTTwEMZGTZDPEPsuze5vsLRi0ZOmXzHfYhmifWJ9
uwu0sS3JrMPZNlvGxNNiKS4//Wx49Efs6INWSP3la/JlWlw5dsoeyjmOFXpg
T6hfhkg8VwLHD67XnaFU5ZSi/BYhLczLhhiepq3V3nSvcdd4eszFc8DYbD5H
Na8fIO1z+8Knp2x3CK5z/OHoaco7gXZc+rlj3meHaw2Q6KP6HctFwTEtjZfY
/sP8SDNS4T7EuaupxqICy0ONIfdqAXaPjva1xqyWv1R9sNzYUS+OrYxDfhyt
nzUKovqYOp7daxTYxSfvedzfLq3Q9n4VDQqcGn8XSc8Lk3nFoD4tT+N38HBG
kEVhfA+cqOjl4xHHmfL37bRIS9HzdO48/W7APf535xrsJi8X9Iikk7T+k3Xq
+BmXsRmv1OZZ5JJ5yB+r6OShoMNW8zJytAjT7W3+OQGKrDnWJ0z/4YiDMpbJ
l61rsFWu1Vh72GqMlY9F6RmzfcwfLosP7YPlOe8vSPAPnXgg4604Ilz8vktn
k5F92iiXm9XzBHW5Y0/7GDMMmaLmuYe6PMXO5556UP+kUW4QwCGfu3HA/Jsg
dkAcjzy4WO7RvABXfwP5C5C/21p1tEwVJXzF6MIpCt85g7bt+GYxi10R+3aO
Qcqr98XvanRq7JPiWWoMMx2WGkOyW1z4fURQ57wGlL/OqZEqZyVT8+H62Rwf
E56Kea/xjWN/+3buIDXa1dqNcTDw2YD7np8W38uN8li+l3tyOhX/qYWflj9U
/LS4vp+2fQpre5cMHM59asUvzn1qaYrn+vqDabvRrPPGYEr+Tne/QePxRG9I
KnE5vmfwFPjq+0nD1+cnX18gypXfU0zBUpHiW5ml0mXGdN647AsDqKuTNoPL
3oOMZWdi826Ql9Fhe2RvZHaTqdJptYohrJbvzUgXUj4mI8QQHHvaD2X6pXaE
2qfa8cuiBpmCH0Auf97ZknUPTVHJrP0DZ+X1nw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{574, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->370638629]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{"eqns", ",", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"y", "[", "i", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "5"}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17973],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmttu1DAQhr05bLssomyFSkFClAdAQkhwgcQtXJUrXmBVIbUXCFT6/iGO
MxtnMvFMkj0lnZWK49N4bP/x4SPX64fbX7/XD3c366vv9+u/t3c3/66+/bnP
k+KZMbNrY8zPS5M/Z/lT8Wd/P/I/9+iezvJ/kzwzzcMoD5dl+DkP4zz8UIZf
y/S5rRWDtcwkKP4cxd8w5XH8omwnZcrFhfcrptQSxT+h+EcU/9LR21Xp7VuB
t7bcZUf7kaiXcTlHeMyeoLjNTw0UhB8UAis2tIqIytJRGU+8/BSFsVeeth55
VsFXOyLlRBY/m+46Bz+/tN+G32azVtxxlMFaktEan6NW28px7RTzkVRxY057
64F6R4rcdBMvh4R7J3AvaCvFmJ6g1DLvhBjPpgWZd4HW6VS2bsAi9tm31Vpz
Xx5yWpCPHGjVhovOtTgfwGo04M1wuvb3DkPom1Qqsrog9NhPAbj1Hak60OJg
PQvr4nHEfgzxi9y3uX29YeVC5MFTxrbNt+o8zSq1hvqJ82fZRqW1VRyX43Q6
YAVGI5W2pA+ZnWSHymxfKQ+vzH7rq0yZeMaxbXsKH75+non6KT0tUqto+66Y
kqmwjyR+3dq7Q+6QW1k56fYgxz/t0ooJ+NsoL9txalaZfexQSlw1etzvjPos
k5zMuXl8UfcmI+4MWf1sDorz2iVqwV2G2uP9O06ErSIbOM2fSTzjtFc2Ad+2
8I0Ht8LtU/vUS9f7vVMHvgeLdBBYq94x+a/Ivr1EpV5PYlx9VYFu7fhZF957
cb+2PdPMTZNjnGey/QhU69ObWNhLOe2xj5hfdX8aC/EqNDx6wgXpS1FvQDd4
PVCSJZut4yVZi838U1rf4Y1JmVXYnjKrQD2b7tZhjk1Zn+Ytb6wyqe7eKpNy
+YXuBMxp5+wfjYeSp+3p75jJk9OfnCyFVaqMqbKvjKnqMT49OtXJGJJvnVKl
MqSxMSQ3+6FvJVrnObDWTJ8RVePmn57HwoDOC+9tkjKeenxajEe/VnqcjGfn
NxQlPWF7SnoC9eSkR79C6qjbI7xxT4H46FdGbX63r3iHV95UWA/Y0K+I4JSp
hKeKb5fwcCudEp5pEB7ZCqqcp855QKWgNOU8ynno+H45D6xV3P8BKedpnwX/
vVbOo5xHZE85T6CenPNwtwPlPP28Vc7j8qWch4sr5zlG5U2F83CrovKeyr7y
nqrHynuqNOU9ynu2M3qPnfeQs1JbCcZDg7rygHHToqG9VZo0bZok6Q/1Lu3w
dqScKWxPOVOgXsWZupzTTQ+FK4fq561yKJfvTpciD5VTKafaO6filLmdfirH
quwrx6p6THMsUpO1e+jwE7tSrnFSLm6eOavKv+hxlazg/moLt+9p8TEz+w/9
/HPZ\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{333, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->298837429]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "[", "i", "]"}], "[", "x", "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "5"}], "}"}]}], "]"}], "/.", "\[InvisibleSpace]", 
     "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->11957],

Cell[GraphicsData["CompressedBitmap", "\<\
eJylmg1wFdUVx9e3770kgG112o71o9rWTse22tZaa+0wVuMH+NFUKQIGVIJK
EAkKgSpMkaoE0Ig0ChIICRo+GkzbZ4EoGkdKhCHKJJYhlQmMSGoK2mQyJuIY
Zeb1d+657+3bfZuAyCybe3fvPfec//mfc+7d5KaJpcV3T5tYOmXSxPOumzHx
/uIpk2aed+30GTxyT3EcZzP/HzzDoZ2kZf5vTib5mX2/kHuUAedIz5WR5p9z
NreIfS4/pS+vI7bvmtkXZcxyQqScJbeo13dCnpyd9UTWEDNOkZ/m0U8Ccs+0
WmfPPTn5Pw3I/1aYXVlSI6Fvji/9zIy5bqiGJ6LzhQPq9O3jeuFE5F80gLRM
RmT+dH3MyOTY1wISnhvk6anc11l97JjT6D6+8qm5S6ZVvLB6QWLdzA3LRy2f
e9OSh3i+bMMbZSu2lTy2afhtlUUP1a/4S3P7e13isZyA9FigHz3O+9B+TPv7
27taWzpfatjnRGqqdzvu0xU7eTG9ZJMT4Ra9YnilmUlbnsXkJv0ny5tkuJkT
q1n15uoFW7h4UVVa60SqSmqcWNXklatuXaSXWBGvHP6A466b8DjtDUVLnKi5
rZvyCP11Mx+uX7QoumFxqeOueWIcT+QekVu8qvy6yvLhevGCuxmSs7p83KKl
4xY/czsP/7hs/P3LC2dWTnIis1be57hTVxaPW1kkq0bkp3tDZeHwylu5nIjc
MrBwPd8IIgcaW4593t+4ef6+vQ3qt2GHez+s3r2xoPae8qZV3Ud7eIZbqut3
46LC0rqNDXv6Pzt2agDhbxq3HD7c27yrA4gqVzRHDHyMANFJRfX0FpZts5i/
sHEPPuByItyMN3PSsvh3rP/zzpb9HbvecSK7q1+ObCpZ5rg1I+eAZ1X+LMf0
RfLOigSvufZs3Oa4zHCiXe3vq4j+T/sinR0tkd07qp0o5m2qK+GqWpofEVSj
ddWF+mRrYk5UxsQZjEjuO/a+9GprYkvb1ggoONGybctLNv2Jq6h+tsEyll9V
qE+4eOtEZZybaHulpbONy4nIzQVFi00kgFWGB+I8bV6xWV9se7kMJyi3kVZY
98Da1r/r6Nihzp6K2p0F99aufbEV+J1z5WkeeL/WeACwQReYx4xeP6u0AYTX
rW01yH7FjDra3Qs2wNQ4/3mgUxRpNJRWKr6CnNvf94nVLy633N6PDgNGa/Pa
5u0rwKn++SKAq3l6ZISOonrgnUZGcEEgE2HRvUfad3W0AseK5vUR0DFYgRnQ
0ZvfuNRRTBv2va5YtXcdPMtDQziwdU615cDn/ax05L97Ndqjff0fIxVJh3re
d/XJ0X5SBpTc9XbHmUbn7u6jBDY8u7mgFjBYESTAwUgfAg4Ht+8BbmyHTbUF
82hgPyEAAkMN91kVg/bs3tj0ajnLY/L6VaNpAAJQ8Krrg/ZcGTmUyMAAPESU
YBzhgqHFibmlDWXYKJ5THvQf+8zi+r20pelM6x79uBtkuducBSCYuHHPFmv1
4Q97Sxc3kCUx9hsGI3xO+BBR2Iil2IvVip94GmO2lW1YP/pRzAPL1rWN2IZv
nWFmBHhiHEwjAAgGLNv5ekXb2wlh/xBjF/DiHqDGqJE1E7GLBl3lN1Zr7fIq
AFCKfLUHOchM59wo1lfsXIPncZ8yu+Gf+0gke/cfcU43wyDw3Dlb8RaJmNzs
nGasJI7xTH1ROWZA230Nzb2Hu019cXKxAf6pdxLrijGAsOnpPmT0H0LcNR7Y
ofrDPqKIGMUdKI8OP/frDgFM2hCfIFPrDtPxIzjYOoMPqFHo7Zh4ivX3H8MD
wA+/mrYfVI3jPYc+QGOAR2kaGGC0zSGOwJc0A9xoyyqC9VCV3PshsJq4GL1+
qhOlpZoC22V+Tcl1kuWUJdiOkHSsuuiKxuhtNSZDQBlStq4jGhMIaAzGabbE
8VpbYkdd4UIuNAZg1RiKQ3cIAvUJg4P7txMVyg4X/7OMZD2jsQv56WOHM9yv
Lx6Dh1Zf+I04UrKt7DgCU/GRfY+m5aubjHMNa+AEhCCv9fX1a6VCOcRBhqby
F0BamRBHU7yPmsQQ4QkHjL055CFYh+thLwvBZ9HwSl/8OS6OspGpqMEiAiOt
hYsr5mx9Al3t+0RjG6h63CYSSbeQAEhVS7hEzMEBrKeGaczFYQBeVy0JProG
yxwkK5aEGe4j6IVz13g42j2eBph6WmuEp+P2g2/6uApLrY4aj6QHUgVF2Uok
AcJ5LhqqHb5FIoxCQdRMa4en4aYmAcVQkBiRpZ2GkGqHLPyMCCcvjJc5ae0g
JLiRvnCw81UjjWqPQ4APacPU5x8dxru1zxZgdDo/xdECXciyukEhWG7M0kmD
RblGtsA4YXBeqo5gDjOJMa0jlNSla96YUpyAdqoLDKM4chHFqgvrE8JAhAOF
x3kZukASMMKP1L6CLF0AGqNybYYZmqruWsjs3uDQoZ6bJ6798+odmmGicIcy
RVgCillfai3swRCiUddGfYDFFHBgbYhwS4DfkAbqGOBT3mn4a2k6++SyPlrA
8ra9R7AeOs9/urFx5wGzosvSKED9sDmE2eQxEGAzRdxZBFgWFZQiqINSv/cQ
iKX3PEiSqFXOgh+2CEtUDyYtfHn1iCenvdN5ULjx2TFKXsu/O9VfWEHk4wtS
wBDNZp/2aTyRnESKkRqHCDgVTXCHajLGaGK3W+afm8pO9pE0o35NbZ5V/pJQ
SNnib7NyHvWJndaOA/8itZBgcD9lmWpGhbAYH9n7Hiwm/J2Artylyudk6Kql
FS4g5zbVyHeqiwW0DWhKpacy2rwMLeCnlxdwJppyV4bFSS7ka+4HOv5XPC+B
2iodmiWKl0ptzcvSV+w2cR2H2Jn6khomhOgbz9A3nYE8fZWOtvJBJbyXTFcw
8gAUJA6slUACwKi88G9108teTO2WunvRVjip2oKo5n+rbTSVuYhKpSV5QhV2
7gzROHjOFKzcFFFYCW8afSVyVU+2KpT8tN65SKaMXTd/3vhFy1h0345WqM60
3IwM5mUN44tcbANQ3cxq8MJXHk46QQ3NqSGZ+gcaOFFjFjVtBVO82KZQ8aWp
Of3uRzcUlT129fLC+xIPsz7LUikVtVPhOnNV130ftENRtrGACNUVRxIFu1Nn
coiWulfNeuLTk/gHGq3bFByNK9G/pROm2kzQ98k/ZlaOnVzDcUc31yhABXHN
GXZs9fhbVo5Jn2ljeqpAS+yQXDo1K97FR5FAJKVRdKnTEF+amuOpFFSvd9t3
sveQXZLWG0oAEU02RCdCneykPtC6qUn5tRaBZmYymcz+MhJJxRmOMkAbrGDu
M08UPV9ZAkU012hllkg0M2ytJNPiFa2BknVnp9fwzhCKbFp2XBElI+BMptEw
XB1KXqMS6TkP7Tve3UUI3jVj/iMLZ2OJ8PPBUAt0FQ0HI8j0KeHsLCRijb6n
I0I3udQYeAiL/vPuWw0Ly19aukhpqcdv3hL7XV2dYAmiDw2yIptTdnveimxh
2RIa+Ayl1QaEN72yZM2CUXXP3klpoosmFDndD6WdNqht1FvIQHZJ7XK7j8IB
LLTxpxyQbKDvvV3WH0Kl6trgTTWRpnpYT2vp3OGSwG3u1ejV6itNv5djAR01
w1OLwN+TxlKUA9l16rcZrzLNDtUxltKBWYZwSX9+syeJvk9gpVREnc+5jbOQ
4DkrVKrqQrUmYLxVPF1Vqu4t5ORlzoAuOzBO8tIsHUQqSNkIiQV0NQyUnQJ8
YW1A+roZgarsI6QZHptp7+IH2Z+4AW0jKR6iLXf9yuV29xwtLK0D3wyp4ifj
5WHKFc5TaMED8i9ZmIeEvx5feH6GkbP9rYNzlmyV5oxQ7Yy8PDyKMKauHfXI
wqvmVeXPQjzpRJhgvp24UBFCSvOBZDADBmqwfRHMh8q/6optqxdsEQbpV2O8
rEGaknv8ap8lkzMnwSTNc5P+eLw/RKawI2p/iiiNe5yMq6X5HdPf2LCnotZU
tekDyMjcNQVtVZkc3DgUSfO7po8jcIc0S0JkxpLhdUSzAbsssoU0zReejMj7
4rJIcKQ5aZ6f9MfbtCzfBvcGQQ+oRFRDQWl+P4DefT726mitHdL8QQCXqQNz
1H2yvOnFRJs3iwgBAe4DzLLngP5j0ENOgBeYPqGKrdK8d5BZHht+KJpX1+8e
bDSnTE510vxR0p8Rin3WpzW6YWSN5P0fB+woDl1DPQCRoJM0Lwz4bPIgs7xq
oL+HSzS22W8g9wwyy4sn/a0aU5gozbtDZ2mU6gFPmvrbLHRDQ/kG582KWEbJ
F/qLA9bfNYhs6AJppHlxwPpJWYwNMjQsDrQPK/CENC8xfQoj5VGaRckBYn7A
LOBJ9bjzC9MnEAgHaU78ElI9T16a9MdM2JkjXKqbRsKL10sDvhpImhuCot39
HOqZUFgnzcuS/sx0R4is4/2WTfvemeJXpo80ZEo1vD3L3zmWV+GSvPp9uem3
v9eFpSJpQijfdBa0gBzS/HVqFgda4dv49CxvTa+Wm++T7q63O3CPjC4MXcP8
HpQIuCIw+rYBR2faEZw1bpBZnh2/CcwaGzor/aXIevSqpI0LO2uMz/r01z/I
Kc38jDUE4VsHWcPL5lenZlm/jB5kFjADtjSvSfnF7h4yvgj5ToQa4yNSLMKL
pGZn1ACjtTKNTPG4dHEDic/xf/OKWO9db+gCKqRHLhq/G0Cq+s58QZQMjFSx
syCLyQNmLp809elNZiy2sEsUi36bPOHc4u3U1NGqSZQcjAfYImPJjScjTYng
3Gw0Y7eNNPHMDSGyMvcn+i1Et7DyVueTi/AsUpzrT0YXLRjqZ/ERZuF66DLi
ZKRpQVRmRlUzsjpOvDbLh76vAVYmc5FgIk52RnhMWOV9g9cvqV6FH5vyB7Ri
MHk+32OWlannVJNfoqBMxOmu48qskco+k+2iuvvRkVdkjYQLMOKOjJEYyq4/
8FsXDSmtkTEg5WhQcG8tCEvOvzxglW4J5KEZH8cqnDp6+nqiANFg+Eu/dE39
uquJkxVUuh5bceQl/tGaRpwp0s/htZbwkXfVcEe87Ct+5p+BQuxUdF+aqzN0
Cf3jEFbkYeC3j9fkV8l+zZwmcrEYC1A//44qJQIiMP0Cn+Xm5JrHWByOLhiM
UiDKVpIl5E9QzvevYfcM5gvCMGDBAYxlGdbQP13R2VCHV7j8nAyqpv4F/wrA
t/ex3zXmyW0IGugKcAzB0IxF0JI2cMiO1+RkXcoxJ93wvzEI37vY/fCCpM27
yJHfbapcjNIlnQiLOlFaykpYHGWEkM/sDDPX+SKr42BpPpWKDuRxEUdGOguZ
7yAnIV0GPPdFtbFPvty8sBxjSnTIX+w5p/wfwaAs5Q==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 120},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->62607028]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(9)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->771143190],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "AccuracyGoal and PrecisionGoal",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->10843],

Cell["\<\
Use defaults to solve a celestial mechanics equation with sensitive \
dependence on initial conditions:\
\>", "ExampleText",
 CellID->30672],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"z", "''"}], "[", "t", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"z", "[", "t", "]"}]}], "/", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox[
         RowBox[{"z", "[", "t", "]"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{"1", "+", " ", 
             RowBox[{
              RowBox[{"Sin", "[", 
               RowBox[{"2", " ", "\[Pi]", " ", "t"}], "]"}], "/", "2"}]}], 
            ")"}], "/", "2"}], ")"}], "2"]}], ")"}], 
      RowBox[{"3", "/", "2"}]]}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19814],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"z", "[", "t", "]"}], "/.", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"eqn", ",", 
         RowBox[{
          RowBox[{"z", "[", "0", "]"}], "==", "1"}], ",", 
         RowBox[{
          RowBox[{
           RowBox[{"z", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", 
       "z", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "40"}], "}"}]}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "40"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21298],

Cell[GraphicsData["CompressedBitmap", "\<\
eJydWluIVVUYPp7rXNVxZnRu6ox2ITSt3sUiKsLsRtJD9DBJoA9pmL3US71U
3vNSOiZJNyHrwUiMwCASIiESIiEqCIOwhyRIehCC0/evb6317/3vdc4cO4xz
9tr7X//l+69rj+unt296+pnp7Zs3Tk/et2362U2bNz43ee/WbbhVmVMqlU7j
35aREq6buHL/Tjeb+M7+buB3GY/q/vs9d1dWparscB9c1OSroneOufXly3/7
dYVk/ks+wo3bMmwKfN5y659/+nPjUx87oW3kyzbSHzzw9Ucnv8/TV7wVWe5H
3frKlX8e33Di2rV/vToNQ2V1OmpkcFeXR6jmv0kFruANCZRdmmd4zbg1GIGd
XCqV8JgxPBpJHkfc+txXv774wudy2Y4KMALMPJVI4tOLP/yxedMpuexK8jjs
1hADYa2p3gyoPvKQC5fugg9E4mGDIqm6Da+KWb8REAHvq1evlXqScVXtiIui
3tNStixJ/eQTH1669JdSZzMju+uQW5//5rfnt5+Ry96OYuqQQbY3GVOkQlYh
HuSyL4n/QYOsUgmPgyam+pM8Drj1Z2d+fPWVL+WyHZXGVH9OEp9q7s5N8tjv
1nt2n/vk1MXWVK8Hrdfdf1wydV7EJ0tV9ZL3G+1bUxdlIAGQBrqrULPM2npy
n7F6fjNEf6AqlTQb9hlfzW8bW3uN7wYK1OXIU+vBQEfW7zUROJD0w55mM1cx
F+Q8Th6aVwva8FCc01S73Roph8STy8GcJPKYOXL+g/cv5J8WeaBUQOHWVLvc
+sJ3v2/d8qlcDkWPZevVbhOnpJqtXu0yGgwlY2q2erXTaDhc8LsQ7zQaDnfk
951Gw+GWdpWjDNVkYQGr7K4dRqOFSfx3GA2USiTuMBIXJXm8ZiS1o1JJi3KS
Nsiqhoj74uwvx9/+FiIlx1a5PdoZR5KcH5Vf3aj6iFb8vPzSWXCAxpCFa/As
rRQpuDma3P+wcw0yCtRoHNjAugJVUEdgVGmFW2sXGonIVzMdwvGpoDNCW+z1
3gdfVEIIL93i1tpBRmMkZauTrWkPBi7gisrG7pTJvpub+bwfTUa5jeqqjxpy
h9WATy5dv6oACBgglze5NcCER5T7bJm33ujUH2wAnuKPGwNScJFUsrGOuD7g
1sAASMQ+lKn2N7i11t6x6KXivCqYUEvtoK7+Z/rBchPXY9Ffqf6wzkh3NVyq
NcJGpqXlRtpYMhbvkF91uBbIrF0zgx/4AiEzaLJngUF4ysTWeC67bg8RDt0Q
OeBRDhEO7SSqBoz2g8FjvtuRv3b18aT2t4XoAWGIspAR4vf5BoMhg/ikW0OE
35+WstqtgRFMcXC5NfiDscSFm2JsxRZsYKs8pxQYxiOSwWq14dZlpPWbDFlo
1kuMJyaSNrCymWk+g3efyRBXLTOT7ZLgPwjOSxEbVpnM6DHeYw1BWCO45XJR
kO4n+yXGLxMx8iuJCnWryZNeE6uUrjPHiKFfbKKC0mbrnStDfvlpqM/4qTs8
9yehURMVE8FmP9U6LTqumytN/vUZad0G8VGD8ITRZvF12KxIMha1OneZqBoz
63GjdWdSV5iInmuQ5n7ACMNUqkoZNxnSWqogPuURLvt11cTqXBPbjbDOeC3s
ZebrHNvleYeJipGqfd3VI+0VnqoSoz9V96fye7z0Uj3LhR5g9dO8cJwaxCn2
fx9+ZSNl0GDck3meoicqIMYWuRxOaimvoubIdzNVozqxjNbQMq3b7Lm69jtz
vOgbjc3OLNJZUBCpxoEp+x6rHK0Rfy1LW1GLdxgevgNqxtI7sUt04B2N+PDu
oGg164JG82DK6qK2nfpGraIh3irN3ZqJi3JKfsEyrVPdabkFP2nvGGqJxf+0
jI3DT0CaEawTnGCsJa0tUySc3nXUER+PYWPaPp2YO7FPni91zzkDs1KFKsQe
o1V9wEigZVpvy5m97NxawWpBSil8AiXzy5w1W9Q66xNqgI1+YhoI2PnnbleV
ZxLqbXuC7JATGONPs56ThvYG97xGL+rE1m98tTSHbDXYzPMfXFgJ9AAV4eJt
xxqPJSvc81qcHXwJpmc1VxhhOnMRBZ1TqLtmPXWPcw3Ln/cuuPrzmq1jakvh
7FOjbD3xMMaIFisT85H1hALLHmnHwte2RiY31DLFkzpCwj13H4s6ZSynZbEW
svl6bYgyCKFEwyDS2janoSiPb/zgIs6yHjdnWRdtjaeoZug4ZRNfNc9zyER5
OYV2Zk6rmrWjb8RcZBnzRcVWNlonk9+JltXMIdsArExJrFBLIQvYet0Aqa81
cbJgCHp8WTHEiz47a7ESNFnZeDfa7O8m33r7fKcN9LrkB+Xbub6jtdZljqE6
MWfjv8qTDO4idljPI8bXb0Wbdbp+EWmdGWN9yD3XEyE9obNA9JSvWtQ1RpXX
tRy4+qxiLOoZjc+1fljdR0x8VYxWXEvk0F8h021PH2uJiXYbtY2V2sxu9m9D
mUzjfvPGthGrJj56prJvDSKOvsJ2G17MRs3WyTbeVI27jJS6wZlStJ/SAq1S
rhPW4nOqw7nRVoaMFHLVE13VcG2nu6LZZRCpmTU7l/Z+6q79k7qDHubiFl+9
8F1TPVI1wwmk1yBHbtp9Vec40WROT9RFZw17Musx3iN3825ZbIMy0m/d+7p6
7G4xbHpN3tg8yCCbO4sQe/t+tGq8757Xs3M9+9ydjhibAA92g1Nf8LCP5rKJ
1qkEWionjYa+0XHc61HP6KO7QhxBjzj71KPP8KGMZSkUIvs+g72NXvdcJiBo
B6G4Kx5xb9JrfPeFZ1BO2vRylVQ4X/SbqLc1rj/jc4nMx4JvoRoMZwzEc3l4
B+DAIhNvP7sgGwaxcX+FSr8z9l2OETQvZAgsXbtmxqM6y24ENZwMNN9pSSfL
d1tzyayvhyqcHE+6VfH/hJTm/AdSDOPs\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 110},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->580111716]
}, Open  ]],

Cell["Higher accuracy and precision goals give a different result:", \
"ExampleText",
 CellID->11261],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"z", "[", "t", "]"}], "/.", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"eqn", ",", 
         RowBox[{
          RowBox[{"z", "[", "0", "]"}], "==", "1"}], ",", 
         RowBox[{
          RowBox[{
           RowBox[{"z", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", 
       "z", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "40"}], "}"}], ",", 
       RowBox[{"AccuracyGoal", "->", "10"}], ",", 
       RowBox[{"PrecisionGoal", "->", "10"}]}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "40"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->12809],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->16378611]
}, Open  ]],

Cell["Increasing the goals extends the correct solution further:", \
"ExampleText",
 CellID->17327],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"z", "[", "t", "]"}], "/.", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"eqn", ",", 
         RowBox[{
          RowBox[{"z", "[", "0", "]"}], "==", "1"}], ",", 
         RowBox[{
          RowBox[{
           RowBox[{"z", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", 
       "z", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "40"}], "}"}], ",", 
       RowBox[{"AccuracyGoal", "->", "20"}], ",", 
       RowBox[{"PrecisionGoal", "->", "20"}], ",", 
       RowBox[{"WorkingPrecision", "->", "35"}], ",", 
       RowBox[{"MaxSteps", "->", "Infinity"}]}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "40"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->28441],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->416140495]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "DependentVariables",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->80839691],

Cell["Set up a very large system of equations:", "ExampleText",
 CellID->285735470],

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", " ", "=", " ", "1000"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "i"], "[", "t", "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"i", ",", " ", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"eqns", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"j", " ", "=", " ", 
       RowBox[{
        RowBox[{"Mod", "[", 
         RowBox[{"i", ",", " ", "n"}], "]"}], " ", "+", " ", "1"}]}], ";", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           SubscriptBox["x", "i"], "'"}], "[", "t", "]"}], " ", "\[Equal]", 
         " ", 
         RowBox[{"1", "/", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              SubscriptBox["x", "i"], "[", "t", "]"}], " ", "+", " ", 
             RowBox[{
              SubscriptBox["x", "j"], "[", "t", "]"}]}], ")"}], "2"]}]}], ",",
         " ", 
        RowBox[{
         RowBox[{
          SubscriptBox["x", "i"], "[", "0", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"1", "/", "i"}]}]}], "}"}]}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"i", ",", " ", "n"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->136919303],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Short", "[", 
  RowBox[{"eqns", ",", " ", "3"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->468903557],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["x", "1"], "\[Prime]",
         MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
       FractionBox["1", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            SubscriptBox["x", "1"], "[", "t", "]"}], "+", 
           RowBox[{
            SubscriptBox["x", "2"], "[", "t", "]"}]}], ")"}], "2"]]}], ",", 
      RowBox[{
       RowBox[{
        SubscriptBox["x", "1"], "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], 
    ",", 
    RowBox[{"\[LeftSkeleton]", "998", "\[RightSkeleton]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["x", "1000"], "\[Prime]",
         MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
       FractionBox["1", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            SubscriptBox["x", "1"], "[", "t", "]"}], "+", 
           RowBox[{
            SubscriptBox["x", "1000"], "[", "t", "]"}]}], ")"}], "2"]]}], ",",
       
      RowBox[{
       RowBox[{
        SubscriptBox["x", "1000"], "[", "0", "]"}], "\[Equal]", 
       FractionBox["1", "1000"]}]}], "}"}]}], "}"}],
  Short[#, 3]& ]], "Output",
 ImageSize->{389, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//Short=",
 CellID->758789964]
}, Open  ]],

Cell[TextData[{
 "Solve for all the dependent variables, but save only the solution for ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["x", "1"], "."}]], "InlineFormula"]
}], "ExampleText",
 CellID->11419481],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"eqns", ",", " ", 
    SubscriptBox["x", "1"], ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "100"}], "}"}], ",", " ", 
    RowBox[{"DependentVariables", "\[Rule]", "vars"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->3202441],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    SubscriptBox["x", "1"], "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "100.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{328, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->501685373]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["x", "1"], "[", "t", "]"}], " ", "/.", " ", "sol"}], ",", 
   " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "100"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->211402592],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->28691989]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "EvaluationMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->5491],

Cell["Total number of evaluations:", "ExampleText",
 CellID->24597],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Module", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"c", "=", "0"}], "}"}], ",", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], 
          RowBox[{"Cos", "[", 
           RowBox[{"x", "+", 
            RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
      RowBox[{"EvaluationMonitor", ":>", 
       RowBox[{"c", "++"}]}]}], "]"}], ";", "c"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7099],

Cell[BoxData["722"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->75891397]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->21823],

Cell["\<\
The distance between successive evaluations; negative distance means a \
rejected step:\
\>", "ExampleText",
 CellID->126660969],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], 
           RowBox[{"Cos", "[", 
            RowBox[{"x", "+", 
             RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13310],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 99},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->7878603]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "MaxStepFraction",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->65990228],

Cell["\<\
Features with small relative size in the integration interval can be missed:\
\>", "ExampleText",
 CellID->338881393],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Evaluate", "[", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], "/.", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"y", "''"}], "[", "x", "]"}], "+", 
             RowBox[{"y", "[", "x", "]"}]}], " ", "\[Equal]", 
            RowBox[{"Exp", "[", 
             RowBox[{"-", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"1000", " ", 
                 RowBox[{"(", 
                  RowBox[{
                   RowBox[{"x", "/", "L"}], " ", "-", " ", 
                   RowBox[{"1", "/", "3"}]}], ")"}]}], ")"}], "^", "2"}]}], 
             "]"}]}], ",", " ", 
           RowBox[{
            RowBox[{"y", "[", "0", "]"}], "\[Equal]", 
            RowBox[{
             RowBox[{"y", "'"}], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
         ",", "y", ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "L"}], "}"}]}], "]"}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"L", "/", "4"}], ",", 
       RowBox[{"L", "/", "2"}]}], "}"}], ",", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"L", ",", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", " ", "10", ",", " ", "100", ",", " ", "1000"}], 
      "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->52497343],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 218},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5667811]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["MaxStepFraction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxStepFraction"]], "InlineFormula"],
 " to ensure features are not missed, independent of interval size:"
}], "ExampleText",
 CellID->175305345],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Evaluate", "[", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], "/.", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"y", "''"}], "[", "x", "]"}], "+", 
             RowBox[{"y", "[", "x", "]"}]}], "\[Equal]", 
            RowBox[{"Exp", "[", 
             RowBox[{"-", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"1000", " ", 
                 RowBox[{"(", 
                  RowBox[{
                   RowBox[{"x", "/", "L"}], " ", "-", " ", 
                   RowBox[{"1", "/", "3"}]}], ")"}]}], ")"}], "^", "2"}]}], 
             "]"}]}], ",", " ", 
           RowBox[{
            RowBox[{"y", "[", "0", "]"}], "\[Equal]", 
            RowBox[{
             RowBox[{"y", "'"}], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
         ",", "y", ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "L"}], "}"}], ",", 
         RowBox[{"MaxStepFraction", "\[Rule]", "0.001"}]}], "]"}]}], "]"}], 
     ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"L", "/", "4"}], ",", 
       RowBox[{"L", "/", "2"}]}], "}"}], ",", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"L", ",", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", " ", "10", ",", " ", "100", ",", " ", "1000"}], 
      "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->47417740],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->294481900]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "MaxSteps",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->47654750],

Cell["Integration stops short of the requested interval:", "ExampleText",
 CellID->18297381],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
       RowBox[{"x", " ", 
        RowBox[{"y", "[", "x", "]"}]}]}], "  ", "\[Equal]", " ", "0"}], ",", 
     " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
    "}"}], ",", " ", "y", ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "200"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->41796065],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"mxst\"\>"}], ":", 
  " ", "\<\"Maximum number of \\!\\(10000\\) steps reached at the point \
\\!\\(x\\) == \\!\\(132.39427800746785`\\). \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NDSolve/mxst\\\", ButtonNote -> \
\\\"NDSolve::mxst\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->429698689],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "132.39427800746785`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{344, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->19227784]
}, Open  ]],

Cell["More steps are needed to resolve the solution:", "ExampleText",
 CellID->28461920],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
        RowBox[{"x", " ", 
         RowBox[{"y", "[", "x", "]"}]}]}], "  ", "\[Equal]", " ", "0"}], ",", 
      " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
     "}"}], ",", " ", "y", ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "200"}], "}"}], ",", " ", 
    RowBox[{"MaxSteps", "\[Rule]", 
     RowBox[{"10", "^", "6"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->358373314],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "200.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{323, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->306251049]
}, Open  ]],

Cell["Plot the solution in the phase plane:", "ExampleText",
 CellID->358078506],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], ",", " ", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}]}], "}"}], " ", "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", "200"}], "}"}], ",", " ", 
   RowBox[{"ColorFunction", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{"Hue", "[", "#3", "]"}], "]"}]}], ",", " ", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->69667603],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNXQe4VcXV3Z57X3+P8qQ8eg8oihoBkaYBxRZ7LKCoBMUWEQFR7MZuFI0F
a4yxoEaNiKj8UTRgQWOJhSiKRk3U+GuMUaJR48v8e609c87c8h5gku/7+Q6H
e+49Z9Zea/bM7JkzM+w6ZfYRhx0zZfaRU6f03mHmlBlHHDl1Vu8Jx87Ur3Ib
iMhi/Xtvk+hnp5/4d3Fzc7PITjjn9LqZf+SI/9C32/FbJ5VOeuohCa7zxO7g
pMZJ7eHpfYneJtVOBvDeDXmT3tFHb3KyyTTeNw7nBiftmGR/3qkfNnQyk/f3
Ybp6f18nU+WwMs+043GgmcRnBvOZqQYjakXnqdlzbcNzCjQxfZTPzeRDBtfZ
yXZ6FntyW5wbnXSNHj7OyT40VCkN5vM9+Yw+fzST2IGpjPkhU9iGOjGFQZaI
2tVV6kIKyrMdE6m0p4Idm0DshilMY6zXfhCe1D8JTnaNRIQp5mE1EgIGnsUd
tbypM+7IHZKmVaFXzGGHe3L+lF1X+i/VmBweOphPjtIzMhw5m+rREDJXs2UH
Rble6vnN/dE9A0I+bkeKemfOUhzjtUn4ZQO08cxqeOf1+Ph3flfLJBeFJL1G
RemNxrmNk3ommeKbLNs6ufFf4n4t7jZxk1lkknuE6fyTKU/kzbV80CfrZPeD
spTrmOygKGWqL3KfuCmWZjJITJOJ4ux4QJyTvwX3rg2ZnEOWHZTqKhV0mySA
VzI7JKfJ9ueheT7KSQUk1D9bOT064Nt99FT9D2AYQF8+DsOddJucpV8T0u8c
bNdv/nGYuN4h7UqkPZuH/tHzni4BUE2TuH0A8F5g0C6I000MYSTO7ViIiqS/
7TArypr+hU7mObncyVVOXqb6cr2TOjkRSAqkMIcC5s5QFMpgKHinCAM5AABN
/UwAJLLKaZ30vpNPeKiA8g4wfuGAtSNgiFEbMACgxVkOzDCqiZGReERYrs+0
1HNI3b0p7nm1X09tIZb+614W90f5yMmbDminwt8eyaByTGukx9na12PjIyhw
kW1IRGEA8htxD4m7HzgXKSv3U0JdJe76nLinpRpov8QD3T1OZ4PS9NodkOHU
EWQ4cTTPKyHYKU7eUIRVBDEETfME1ufijuExMwFkrbjHceObAAraNQTh2pfg
WN5AtwDyBEEuEne4iuf2A5Ph4gYTaXP9Zk+tMdxMceeJexRgRAp8rMFpr3XI
JCKNwFkxhhGshmCV8NI7DEkTORVIbrTCDABWBwJpWRmsfDYH1vfEHSjuFI+o
Ym9pcEapshCrlq2K5dKGwLoxBQJKggKkJHJ96OaS4NReETs6NDhtmaq4LoB2
ozKeq2LIkUg3wqwhZk2mpPstH5sMsET9Mz9Qf9vGI+m/WxPcIGs8Xi8aeArA
RqdgWkdUeaStPLuT+Ugnr6RHUom6Kk5uU/16giaY17rxYB4ot9P02731Y8PW
0E429aC9yFAz+vmIWxXa84kZYo3z9DYMiJeJ20Lh2kA9TU1yE1gVTwPsLMe2
Ln+MXtTvTaIGKFoa4UYXRVDqJVURUl0gBpckzF5W0jSRvZz8CFAn6anqPCfz
w6F/9HwJftOCmAC3dmIAHQjcAQTNsQnKEId7v7yQoCm3vdpEcGc4UajczZp/
iwCgWItY3Umip+QWcDzfyUk8JmSI+8VwTu7ZP0PUyx/H+WdwkpCc4t3MGkKS
/9Ekluktqzzu0w6HJHpKfu2xaw37EAKrO4VibqgXRqjVgSRQVdBtAr0EYHkF
e4rHC9pgvWU5SOS2LziPrIfC3ufkZ8ScRr7ETAF7iSEOo9fchZqnJi2LijiL
zQnZPQVWmvZbVunnP4EYzcb0S4dDcji10Wb4bZql+MuIP5/t3bYeudaQf6ma
Z9h1jPhqGKNq5ZDMYluyKDD8MxAtamkW9zGCZkX/i6G7z3HotZ7y+rtZo2Yk
aofUrAoazC40QCtoiSyoCbFif2ssDPtVw07+AQMAIIDXPxXuXTNJ/7UD9sAG
qlJrKrwdJAj5XEUfO9Vwh9K7ZIoL0FB9EUGNbV6x3GocpP6S/n3Kc35K3Ivg
/KJ+97r++ydRDf8iim3Iqzxsink0snXfDLd6e0lxobgxdf/StD7Wv6v170sA
fZqcH4Uxj5qXaVum3z9CG2AObfgTIta/Q/hVBTJ32FGcoW7JfJbHFfXqGFgh
QfNdMn0J6T9tYIjqHjCt7/Ss78ShP96pfx8BYzXhdVFwpf5lgWcr4y/lYY06
MuxqLUPF2EQFYg5QzM9f4CDqLzyqfZPXUw7w9SoA6JsCrwM7AtbYsIPWbYb7
XXqW1otnQuz+1rp53KepakJcd5dH0cgDGle6n3js+Tj0ej6EvjEowDyoMSeA
BT6bVfIOJ+qVBgA/yPC1ivzs4kL8BHmpoCSFaIdHjqj6iDtD9NDrM7wmp/uj
gne4C4NdiZqkdtyJ+AbWFNuxILajWo61KL1/CI0T450Da025gqDAny166Hez
vQp6PQvXs/TB02F1sEAP2lAVQzt5VptjcRvK3kTfgq2wuB9qgMbaq2dmwFVs
TH+inwqxK3CqxPXhYgfsOJy3VMGYHBS5EIc+Oz+yYCaNMAtujC2o0k7RGOTD
wcEClHeqnFO0ap96GBUAZBVOeTeN17MgQgL+NQD2cL0M7kcCaREY7UXEzU1x
N11+444DaI+Ms+El7giwODyMQNS4SV5t9qn07okcf8gpvNSlBoA2oHtZ/1Dj
GFRx2jUz3M2Y1P5oUu+S69wtRdDIUOIivUkCyERPNUWjIRA+DyMacN808ccs
g+5s0BogynKF/hHCpD2JPoQ5vYSdVhiwf4kBR6ickwhf63bxfHfBofrvTL7E
hUFR5akK/1EmKdZ14qoKsM4F0P7i/hmwbMyEWJOYXoUmXlvEr5o4hqeJT5RC
rKXAWqlwe053sgfRNrVW8gtjtr+85BTQ4Z8I8PtQcxeqV6NhorEbhUNZj9Lf
RvK3nCIDpluIefTzHEBq3eh+2pZldHeibmL2vYEQgMAXI4zwwBUWg+WY4mgi
lMvHKkXN+Tg4g7tcKc9QuEvV8Fv138WyW4S4guGchv4ecQ8H1BRRv29XhNRA
K0aiTSiD8o64bUeatywW2ZVIg01RcvsuwHaVvyJKOlUKwJhypRvk1Rwk/Gin
gQG91scwGfJzhcjaFn/gAvLGFsfejYZ+HrM9BXfyNbPTg+fJaiMBfmUR4zxz
eaBpW0/sBmJv96xhPwkHEtd0PgMC7Th/n+gbUS53Jrq/KzgsIK4PdH7IDHiU
PZzAPq/obctqPVCAVxXaelkprq1kqJMYcymq+7nILkQe1MyRI20L35NXGMvG
4MOg+/0BHP3oDYtw29nzEaaTK4EJoHPVzYc6ucBhMAKwV8rORB3YzHHJM9FD
Vo3v5UADM3wM4ouHUA05WRwBd5KyyG0LkecH5EsNeYpFXB+Je03cBX6EV75j
Tx4v7ib0+/7KuPZoJKAN0jgYMN8MOCY1QDoUobf36Bn0aA+9ipVXMpTQXzlC
a2B9msiORB9A44F+Nej/y8kfnFzsZIQ34Ag8cRniitiAsvjtI/x5wJ/q5BnT
vWEoe3OvmQkv0ISZqKRkAq3ob88fj3xGUPeyfEE3O90bQhm0LEzGEA+dIihR
D09DLFbOosaQH7Bohp72tRx4XNzIIexzv2gWfYRSr1liRm1Pm/rivCFc4EiB
b9zMAZT3oM+9bMPbxJaNjy2DaypsOxWjra8V2uLQslJ43eDa6HcNgjrBLG3S
RJcJE1ps2XegWXoX5dMAGvLdEYzdT9wIGU97++DcQasJ2DuH+fmgV1JNvs7J
Lk42gMlHYEDrMicbmcmDUpPZpooUxBihHinRtlOwuMFbPIIjbu59cQdohu/q
5NI4w5eilKvbpUZrD/57tLuXpbYxajiMCM2h2nfT+je8H8yk9eK6IgkN4y/D
aPYHsH5gan1DGasRKTfiomNsrcYj3lqtSt3J4noNobW/dfKpOcNSSnylIDA6
TNxOaDVHoD4fEFIfQ9t7AENNV/vNeHOVIL02xir9Eto/FsDdvP3zMLTzd9jf
tVX7S1RvCiEsnHoNh7PgKp7HsCHsw3pX+SYIfwW95TgKv5O4OsifKBXj0B3n
Lm4wI8gxkjmQ+fxvfS4okZucHOSktydyPohcgAJQRETWiUm3mImTlzcTtuiB
yaDAREuohk6ZC11BLzqsgAzOA5BHdTZ2LN1w7qqcQGiMZJnzY2bO3aT1RQGt
LjBjC09rFAb5mHNdyEmrmpp670/FrWyLbEZwFBqdQM9Go5iHzMW08Vkh7r4o
awIhZMIIP3oxQFOr039rbfxButDbBosyG8MbY1oXB7d7Q9w/0IosYaEfGzN7
HsXmIDC7ImZWtjYvYvTr7UoYHe7keDL6Xys0L5DRz8noVNpnpHb0pCyWjcqQ
xdK11veRTmTXqIXK++LYjCM+zAk0b85ovk+a52Y0dxS3DDT7uLklNEtazcaM
pyd5hpFc4Enqt7sGki+lOWeVwxUc4j6Ob0v2o4Et8qzwPBmh04pEe57SE7cP
9jfFbH35uzj4Kpsda5k1Ep1LtqzH9gLb51Akby9k27Ec0y5k2mRMfyUcjvBM
J+Q4+nY8ffS3RT4acjQlq0wDx63SvHT9GdvnXI1Fr8QET97ZszzRo6Q4W7Vc
/hXl8ncsl3N9vZnzXJ8G14fAdU3rXLvHXK/iMDW0C1zVe6cHrv/bKtcsY7fy
9ZgS9Ug1zNcai+AYAYIvW89exXx34zmlfDEvLaR4GZQt1D6X/YrvwOxNSPkJ
UO5hlGeTbz1eP5bl3KNlzvuK66+cf8Bx6uVpiY2d+QJaVI52CW/WhMpbv6sO
3zH6kDZeAz7RO9UglsGX5iADNAgubgXaZNiP76szGc5xmqcs0zunMpRVoVuk
wlyvwp1GFyrkIxXQDVgTMv+OFjO/1NeL9GhDv6+W3tTAWj8Vwf9apEV5f4iL
wDIEUVrcn+frtEItfoahzLNRX12fadF2XUrB7l6LH5sWP9GCkGrxkGnxz6gg
FNZwsDXIsTY1rBxUs3qzYaHEdbBWM68UUhUSy61SIa6mEMtYF7yPlmxFJATr
gk3QV2p2KsSmYkLMToUoXzZMifbFSnzu2BXxSkxn7+CdtEvwWqREueKxNiUa
7boaI0J6WFTiR3pUEn8Xey41pozwT+oi+HygdRg0wlRprmFhWcY64335OEhz
ou8ntysnzdTWfaRzkTKdC5TZN1bmNXOTj9gtsYrjBl9kbC5MReonBkKwXJEq
1v5VpT23pMiqJDzjx4oypehDfZWiH/4cGz6U+BCEuyYUJtavsVbbg2xH0YxT
rVZjmGHrTKsNWtGqR9BKVZHTU62+8FppJptWN6RaraFWT9KRboAjwbLjvEL6
RJkaRspo1tFrpt9VxiOEsYV8Jp9d+1T9fZmKuI/vjvqm+WRKpjI2p/WzKXlz
gZLvlVfyNlNyJKlylktrSnaLlXTyDZQ82JS8tUjJx9IiuSrEmKmSp2ZKGtsK
a7WareGS0rarM5WsXGf1qsr5YP+0HlsP/diR+pfz+l1B/Thims/0m+J+6PWr
p36dWtKvZ3h71qJ+Ix3Ga28IVRr6VR+K+x0LbqSfOqPnEJwx9cVQhu26X8Dv
4hUs7ge1pKDdp17iv6ei9bGCVt/FCqp8zdbbUgmuKaPgoxykOpFvosV9h528
2xDW/CD1QFOwS0sK9ipScA9T8M+ZgtpmHMBpUotLFbyjNQWbfS0Wtw3FKna1
68rC37+lmv3XqqY7IRh5TRhbeTnr2C+kmnt6NQeKmypupTzu1Vy8Dmp2L1Jz
d1PzNVNzTqrmBbGaa6jmwy2qaXNbUT+KD0UrilTs5n1xXdWrKKdeLlMvCV6H
H4r0Q2Hmbx29gCvRBscCHu0FzFPA35uAPaDeO5xZsQFfwrQQoXSOFXyAVQLe
FC9NFdy4XaTga+UU1B6+9zvYfWhWK+bdDll0UVmkYXdqWCFhbCNrf/NFarHO
rEjVAsg2Evp3RZXfCSGmTf2Ncn1eXq4TU7mOhFxvx3KtvfTezzFN+JtXS7vx
G3UqVevDSK0boJYfpkhDXCvAO6Tt8fCCGD7SrKddV4Tr2v+CXm+gsf2cIy4L
ORIe9KqlXs9YbXeU12vw+ul1uHDo1Ot1tOpl7ydO/nZ65Yr0Kq7tetl1PlzX
lNWrXGmUPqqZF6+sZPOiqLhEskMzyc6CZNrFmYJRQyd/iSUrXyJNs7wFx18w
BTyDOu0lataUamYjrp/+RzXrXaQZvSnc1akwCtRgv1XNJpdodnPQ7E10u02z
JUGzLfxLt6DZdLczNPtoPTVTRb5frNlmptkprWh2xbprVlwuvWa5Ao9ab812
/3c0m86boNnxXrNOofVcD83OK6PZBdTspSLNQjhsrgbNpq6fZn3sOimIcGVh
iJ1L2taqcG0C2u/9YvXw22Qfw2Xq3eLVw6+fePVuYiB8UKbeRPz8G4T+24v7
GwTM1Gu2Fqx4BkFV0XVxXMrfe/kQwivNF+py3y7CUezlnHDqNALcbAh7Lpc6
uZtKf0al8aN14W4vo7RIYSsLmOHFEYjrW96u9biu+Ja+XCY3rslyA81yK7mh
Nz2K3DgMWbFi7b7cI1b4C46WYNYuplU8R4WbyiqMSRBPtujL5RX2cUzfgN/3
P6RXM8t+Ge99sFivJUGvrTK9/gC9tjG9nqRefdap7EOvDwv0ujbVa8eglx+t
/rB4UAF6nbIeZT/VzEbq80UaVa2jRs1p/XhWaIeXBI04gqkaPci3UIUavWUa
TeOIY1hEsI4aOVkNjYaaRu9wdgM02o0aPdySRpevu0a5Io0Gl9XIx3ZBo+aC
2K4VgX5VINB7QaBdvEA25eN3iOjGYrCczcC2qUDros9T2vfAa6tMnwbT5yrq
81Ia2xXpc0poc/8f6PNMsT47F+jzQok+Neuuz+odTZ+VNjHH62NjTbfGPf1V
bF8XhTjO9JkR9Nln/fTxPc/OrZax3lbG1qLPEq9Pc6TPZE7GKtBnD/fdddOn
e6zPUlFORfrUp/o8XKrP7VHhmkGLp2KhWnF9PSwokOqySVldyvjNtt5vepSp
nOeFxqxQlBVszCZzwGIDP01nHlzfi/IFV0vUtFbrdKckHQpUGW2qPGWqbFY4
QlnQY/o3Vaks5y0Vbsg6q4IPqSpvFqsy1qsy3quyO1YqPQphTBU08eVV6Rmr
8ksuNvrWqhQUpAmh3AxrqRyZMlVFynilhhSUo16qTmvl6BaWo0iZJ8OAdqky
Y/4NZba3yY5QRn9rUZlFWRPVXFrFTAijEcXK9AnXNq+gupwy5X2me+wzZWXR
GCeWZYyf8zMek284DTXIclwqS0vj/PmwaErOM1Vm0BMxLfCAfHgVb6r80V5C
xhXv5cVFKJShCVkZyhfpMeTf0OPokur2GR/zfdyKHoe5sV6PGWFRVnk9ehTp
MdT0+GMZPV634HhNVHhuaLHwrNVHTBP/Rq2pQJO+RZrEJUf1aC6taUMM/PtI
D06/0cr1XtNDo7r10qMq00Nd6k9K/BzT43uRHl9/ez2SIj24CoHjdonKUWU+
FXRQEfzb6d3FVx+pX8QivO9FWEgR9uOSB4qwCC+PZ+LN2MWFIrTW1lj3/2hV
YUv99xqb7KkqDC2jwmeM3RaVqVJbrDWKFeD6D76FjT2CZalPGY/YIyohJ0Y1
Rmhi0mG3iwuUOA1KzE+V6L0WJXpGSvQ6j/FMpEQPU8IGdX1l8U6JEiXOUL6W
sFlnDeX49y7L3x0UPOEsfrg2RKt/wHBQSv5EP1ujgDwX2FwdlvC1VldmXoDx
a8/9ZOOeDmh/WMT9Z1lFGbtAeCc6LGLTx6/u83N1miwC6RX7fpbdB0XZXUj3
8+gdXBFdLf0zQVf7G9VcWdNCXveI+P7oPJsvvjjl29g3vH7zowSfFUYR5xW0
CnFGl9R9XNvn14NkOZ0zfyvhHWez93HrkZD0I9w7ISI9qyXSrZR064jcdXTG
+X3OAAXnY5mGr/zXlsFG2PPtHZAsf9szf7uYR/cwnq1lrhbk5T6CVl++luOa
2/lZJbOxhvBVzGNep5ztFrM8OGP5WMrygrIsS7LVUxxqnpwoRa4p96svlJxH
9ZHwZiUcrfmyjPwVCWq1/SUIakZelBHcCZ71KiaW7wuCjwWCFeDY0vzYfHip
ogyfvSGj+Yhg9LLxAK5Df5g04b0xzZKcDBzjfDSuHXw+5sx/CjlmmTgvZOIz
PliJCOZJ8OqI4LushKuN4Drxu9z4vZDyqyrgZ/1B89TzyC/UwUaOuaTUbH42
5voi+/jyMaWU+WVc4xQ65R7+jc6mhXw+iflw9VE696OIV5dCXh8PLMfrp6x1
PjTffKIk034gIceUEfek8bMiumBudkQny55LQmP5Z8/lpuyF3qaYRPUKembD
PJfGlEubIg7Fo/ERJ1scGHH6i+NEhn7d+frAxlvhiBZfRxkFQtsz+OrtdzkK
M+i7itLBkdKJs+YZcPmYXE7zXDoxsWXfnkvnMlwOTWexgssR5PKhOd2rgYtl
TiBiO0ZxfYY0eRYxhSg7PuYYymncBia2fzgqCqXQjxRasj9dYdKpjN1zU7sr
R/Gd18Op3b9jO5ZmAI0uWA+zGbf/MIPPCn2zlRD8QS5dOdR7MK29zLXni9y1
WVuidodgNZZtPWvr8zQ33f+o1bXc0WcBF3Eh5nw1lO+TsVBaTS5Yd2TGqrOf
TUuegbTv0dKjuAKV6zRuk+dgKVeTjVg/SzcsshSLRYKl7SjuglTch4K4NNP2
TJN+bJ68kea/v/cDaNf6Vx4j8I7vOayMooXTUgsbfE63tMos7wNe6XCyWfYH
x3kzG41iXfJY2tu/jRUJzbJVZn7l2570ymthkur2PHWbnE7GaY5NarVVahfZ
MueAQlt2pESPcZ8tSHSZuGNlB1oxILZiCUK8d9m1m+yzbrfIhHuDCdWwongd
ZFiJmKcrqgG3jDIrMDVtn3YciF5gwfvzlOPYsBLRr8I8mzX+J+hrh7equ8k3
To60RZALC9CrXHffg8ks8H3cHuJ64lBL9NQ2eI83yq82cl+p1w5z8nOH2ScI
qRfRIluZ6VejXgI91Jzb/Rj0bmjHz0dfhjFwZg56kTVqEcDtDWkPHBr097Ae
ZptgRUda8QKs0IoH21rACq/LEyhhtjLVr8RdLn/P4DmX4SmD/7rVvGhnTwOz
jWG+pTWRnw//c3FNwyxe+Rq9dr8C1688fgzd2nM83Fy0Gr1RZQU4lXODsmj1
RGsqQIPIZ4jb4HiL9p7wq4z96uovkaUkNhed+7OxcUEMow64RcmKMvTbKzX6
xorqPMGwZdFPhN1DvjXfh5mqzbdfUW2rxt/0c8j74sXm2XhXHxZyAwqjIXWK
Fs9IBkitnjhSUhvQ6iI0LbRD5nBdcrw+XevAE5nJqBDvxnZFMVLOjfP9ry28
p3yXEZKtm95OMiSs5d3PT9vXeqv+LJZdQ+K4X+0jrKXwYu1uTITkMl0Po53N
hEqNK7u/gDrqeFE0ACZ68nsN+GlCBr4M674wTWjkZmwCbNU/R5NqLvIEn9GE
6EkpbLUmsJetwFTsaoUxluNx8JqzkN3eKBwer87wUOFgEubWGlv6nQ04VlNb
byQB9WaAagSadjAO8WvJ9irhWeVxrP8whbsmFuIdjPccGzm/ZwOxqq+1MeBD
3EkQ1LDwKiFxx+ijhlaraMZqbxya6t7eiimq5vQipPk2qcNdemxA4vhLFWBu
xyxLzrOvMBh9fq4SPcbP5DsEBPX6kJI5ATnm7hR/33TMPtH7TpIMGIs/NERg
jL6z34Ej7DXyDWEvSmGRb4k7B0nMpe/DAD2BVRVNANtgzRTPdjoRE3duIWMA
oyFy72ghKcSdp+ECV3DZIm7i5tylxM4rdDWBgcUt8PSe6T7/TtL7zvV7b1wv
hTl5Jup25Rjv6NJff7m5GMvdwwjlOsW7VB8/Rz1urlfwJM6QrPZ8ctjrrxZA
C1BDBjis+9YoGsK5Hobmd69Zg1i0AOoxBCg5hUsUjyQBcw4y6BzP8RwxyXk6
F5v9ATbBKeexF+D6Xjz0eIEVmCO63PkddDjKUVsDA0an5dC9xZCCZlBBswQT
U/JEgAWGaNdeWSIuQP16L/fqfJzH8yXorjFGr5TpMbqW2OpELcgpfs4tBMSt
nI5ynRD/eo9/vfCjnoCZ4M76DFSPt8V9atBW328M33JPFezc5I50eGNsfowW
UZtM9zcBvrF6DDIoyq896kIcer3AXy8QM3GhPoN7APycuFegxNtwik8ZOGVW
YJldagXHXCo4GzMzQZJ3sfjmK6T2AZJCajlNlakD9HH7yHUyiprob4neRMQa
kP4aY+HPetxuGW60T1dNT4860PL8AYcYVpFVgYD+NzEDmLrxfQX0iGjXlLrQ
gno89CmONRyTf5w9mNgUucLv3RV2SOvCToPnryHLE8EYSTSI02+/Iu8PcJgX
vOXx38KhZfEDeF1ANNDbANqxADfal62hCPU0xkpLib2Sm19qRxjgjOA1acP7
Cgf1yWm+Sv1HRFtNte8n5qmea3vWZoCNUKsWZ3mNAj+eHR1JFFpyS8l6Ja4V
PlH8/BpnuP9k/C0JTrmPCLga13pKFFaqbyTwuAwbwJeEvfCG+zZ653R9SE/2
XI/ipggKfo+mo/CSN+p25P5k+/9VrvQoNTdybHkOS2tdIZZEewzqTzulSNre
aoW6OXdsnoElZkhCknvA5AHq9iyucco/wTl1TzATIEfyK/2i7ho+NacAuK4g
XyPstl2gsGH3xnt6t2V/lv19CX8C07qMRtwIyAe8wkuZeZLglLuGt5yN6zng
vRP3YGWLMak8dti10iqwTgSep9iVxB5OQWYgvROQ/tkpfkIjNJnbCKlHoj8m
uLXWUDt74H5a4wfBUZFGu2U2RLjVhD6dr4M6bUzhM/ScoienWq7mtOqThgP5
23De195HZoOIVBeDycTyLA3tBa7n04BhUKM14clwgI3mRsa743qqV/lAuoEk
6qjJ5sjbjT27QQarFUxAbuP1NeSwn6tlbJdSZHF1CYK9XBN3QRsNlNEedZxV
/jmc6jsDrjOf2EcQqq0golUUexhohJkuqRriMdFBWBpgxzKdvnkg53vSe/vb
Vo8VwCLQEALtw13dVsiVjnC90nn1hXvHmrg2VF6hzTJ3slgOuATz/PKGZ4d+
092WeWIX4e64YwgwtxZEt8cTcDm6ZENdEDXPKi/GrOZUujrf8HPoAvMZXsRi
1+WaOrrWJ6F7q9eTEgC3J+KWAeco8TvWLS1EaxNe1GBhukfc2iMODYiA+6Fw
q5NXgLhCCZ6dAK4dUY4iD0t/CSFWQ5HjsBQhtKh+IWrhbsa2O7OBoFvYM4Cg
VUFLpVg5jOgZziU87vMIHzsD6QoQq8Q3LcIZ6X2jE8lUh2akGWIqzpWEwmvn
VxTlWY/yLPdVXI31Bi9y/GiOqdUcqNiC40nlUMYRxQi1nSbOA/2U24lCu9ex
v/UHdMEPUMLfwt7ZN2QwVKxN6KR5MtHe2Yl+DwhzAgN6KAVK5HyXw0xy7pl9
M3enPYkpH4rO/lQk/mCIq79TnH7YxfwqYpsDWLD79jSOnqq947D79858gXQ6
nszJ3k5qZQJcncl/FpLvGLLD7y7OsfQKVvdmfEc4MQtGbi9LHiUSu+u2wZdb
uzZc/Xcw1w6EhJtC2r1K0q4JJbIxjDGYPF8QAh3tXCNqTsHYWYJXV5VM+M8h
6lVB6lTfRrHd+NhrN2XG+PrbEj8g6g2P0/S+4R4HS9jzIBb6EzVcndk9Mrox
FcTvSj/GK2KpHmRW+9XWfOImzlrU9v2vfu9VWYh0LdHG4AEdy6ZbGyntR1zg
3LAZKWtatyoYvrkjpJVH/xzdtbQPauqO9fWBJdY7pNfb0svhmRp+tnvyPDfx
TpgHu3uXTxXP0gPasbXoF8xrZHLdQ8b0Y3L9AluM8x3CNLbxCjZyyLkHw+QK
YR42sDVBckzLEvo+E7LWbFMbBhH7/x+2NWsa/fZ7bTlNsoulESXQkQ2xJYDX
oy5+HphZRjKRSUxEiWzMROo416xJWIpzgY+I/T8W4f/QOCBKwKzIR1m3TYqP
J+3/0Bjv0dPHtjCD+djG4Unfisuh2TP1NG4gH7NWx55pih7z/zPI9r6M5QkA
Me1OqsBem/2vI/+t/x1FNvg/Ykvb6Q==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 177},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->76577998]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "MaxStepSize",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->154690044],

Cell["\<\
The default step control may miss a suddenly varying feature:\
\>", "ExampleText",
 CellID->576831980],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], " ", "/.", " ", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
             RowBox[{
              RowBox[{"(", 
               RowBox[{"1", " ", "+", "  ", 
                RowBox[{"Sech", "[", 
                 RowBox[{"1000", " ", 
                  RowBox[{"(", 
                   RowBox[{"x", " ", "-", " ", "Pi"}], ")"}]}], "]"}]}], 
               ")"}], " ", 
              RowBox[{"y", "[", "x", "]"}]}]}], " ", "\[Equal]", " ", "0"}], 
           ",", " ", 
           RowBox[{
            RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", 
           " ", 
           RowBox[{
            RowBox[{
             RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", 
            "0"}]}], "}"}], ",", " ", "y", ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]}], ")"}], " ", 
     "-", " ", 
     RowBox[{"Cos", "[", "x", "]"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->85406985],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 87},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->399446377]
}, Open  ]],

Cell[TextData[{
 "A smaller ",
 Cell[BoxData[
  ButtonBox["MaxStepSize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxStepSize"]], "InlineFormula"],
 " setting ensures that ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " catches the feature:"
}], "ExampleText",
 CellID->245961639],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], " ", "/.", " ", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
             RowBox[{
              RowBox[{"(", 
               RowBox[{"1", " ", "+", "  ", 
                RowBox[{"Sech", "[", 
                 RowBox[{"1000", " ", 
                  RowBox[{"(", 
                   RowBox[{"x", " ", "-", " ", "Pi"}], ")"}]}], "]"}]}], 
               ")"}], " ", 
              RowBox[{"y", "[", "x", "]"}]}]}], " ", "\[Equal]", " ", "0"}], 
           ",", " ", 
           RowBox[{
            RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", 
           " ", 
           RowBox[{
            RowBox[{
             RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", 
            "0"}]}], "}"}], ",", " ", "y", ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", " ", 
         RowBox[{"MaxStepSize", "\[Rule]", "0.01"}]}], "]"}]}], ")"}], " ", 
     "-", " ", 
     RowBox[{"Cos", "[", "x", "]"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->90211088],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 96},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->285638674]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->309908885],

Cell[TextData[{
 "Attempt to compute the number of positive integers less than ",
 Cell[BoxData[
  SuperscriptBox["\[ExponentialE]", "5"]], "InlineFormula"],
 " misses several events:"
}], "ExampleText",
 CellID->534866714],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", " ", "=", " ", "0"}], "}"}], ",", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"y", "[", "t", "]"}]}], ",", " ", 
        RowBox[{
         RowBox[{"y", "[", 
          RowBox[{"-", "1"}], "]"}], " ", "\[Equal]", " ", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "1"}]]}]}], "}"}], ",", " ", "y", ",", " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "5"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<EventLocator\>\"", ",", " ", 
         RowBox[{"\"\<Event\>\"", "\[Rule]", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"\[Pi]", " ", 
            RowBox[{"y", "[", "t", "]"}]}], "]"}]}], ",", " ", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
          RowBox[{"n", "++"}]}]}], "}"}]}]}], "]"}], ";", "n"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->81999207],

Cell[BoxData["18"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->135026800]
}, Open  ]],

Cell[TextData[{
 "Setting a small enough ",
 Cell[BoxData[
  ButtonBox["MaxStepSize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxStepSize"]], "InlineFormula"],
 " ensures that none of the events are missed:"
}], "ExampleText",
 CellID->132064070],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", " ", "=", " ", "0"}], "}"}], ",", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"y", "[", "t", "]"}]}], ",", " ", 
        RowBox[{
         RowBox[{"y", "[", 
          RowBox[{"-", "1"}], "]"}], " ", "\[Equal]", " ", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "1"}]]}]}], "}"}], ",", " ", "y", ",", " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "5"}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"MaxStepSize", "\[Rule]", "0.001"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<EventLocator\>\"", ",", " ", 
         RowBox[{"\"\<Event\>\"", "\[Rule]", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"\[Pi]", " ", 
            RowBox[{"y", "[", "t", "]"}]}], "]"}]}], ",", " ", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
          RowBox[{"n", "++"}]}]}], "}"}]}]}], "]"}], ";", "n"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->138136159],

Cell[BoxData["148"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->309303379]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->24247],

Cell[TextData[{
 "Differences between values of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " at successive steps with the default solution method:"
}], "ExampleText",
 CellID->32113],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], 
           RowBox[{"Cos", "[", 
            RowBox[{"x", "+", 
             RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
       RowBox[{"StepMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27422],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxd12tslEUUxvFTtFgJmooXvNIXFMRK6kLlJpceEIFSoEvLpQiWaaUUCoWF
AtEPxGMNConoBhGbKHWiqEEUFzECBel4C9+wCTFEE82LQEUNyUaigsHLkM4m
M0+/NL/+z9mTTbrt7sD6NVUNvYgomUd09XvPV7bs/OW+JY1LhpS5H3CPRzkX
gAvBt3D9mPRl/mul8+3gu1kFjnhp4EGcGj/9wnuzXnO+DzwEPBRcDB7GanPN
hkNP73YuASf4qRcuvnyhqN15OHgEpwKXgh9mCTyS29450bFttnYexTrwaPAY
8FjwI+BxnBo77fnNR/c4jwdPYAk8EXoZmMN5YeiTOFs69MCNW/a7Pokv+abJ
0MH0KHQwTYEOpsdY73h2/ZWLB10H01ToYJoGfRpngj4dOpjKoZcztU7ud+zm
r12fEVpmcHp3d1HzAuN6BYtvqYA+M7TMDOdpFnQwzYbHnw29EvYr4fGTrH1z
Eh4PbJJM+Xtb+/z+ldufw/G1nnlO2GUOR6d+3ffLd7l9MFUx++aqsAt0U8VS
O6CjOv+o269m9s3VYRewAdNcpnP76/Y07XT7YAEbMM1j6vb358E+2OT8rtuf
z+Kb54ddwAbmaUHYGSwLmD/bevD8IPd6MmCqCc1gqWEJ9qHTQtgHC9iA6XHY
BwvYgGkR7INlEctxOdcyzr1+je339u6smHDA7S9memjjqXuij9z+YpYSzwI2
YHoC9sFiHexDp1qmwd+/8tPMQ26/lsW3QDfQaQnsg2VJOG+gk2LppP+yFzt7
HIFZMflW0AWsYd5Aj8FUx7z95DVXln7h7tcx+WboCixgDTbgGEz1cL+e6fDO
b/uOPO7ug5Wd9y3QNXRju6QK2uYNd/fB9GToCMxgZW2uepK7n+vOGmysD73Y
59/3N7j7V/c909KwR2C27tj7Z962t9196yOeBbqGbqDH1ud+2zq0NuPuN4SO
GljOemboCrpA12AD8zF0Whb2aFnYGbqCLmAN88b656kNAz/92N2385VrRtwx
MPf6b2RKeo4aw85gBRawbmS5/oPyhjPungHHdv71pl03bP7Q3V/O9IbnCMzL
w3kFXaBr6GY5y8Tm9K76T9x9233TirBHK8LO0BVYYF7bPmLj0Sv/HHb3bS/1
HEOnptARmMGqKXw8ga5t7+5/f/fk3PuVJmbfMXRaGToCM1itZPEt0LX1ydOX
6MSX7j44tvu+aVXYIzCvCucVdAFrmDfQYzA1h/NRM1OvvnnFM3L/f6zzPCvr
C3HbWwUd7j5YW/d/s+h0vvt9MOAYTKtDR2AGK7CANdiAYzCtgftgBiuwgDXY
gGMwpZhuSoz+o8a9Hyi07uc5gp6AztCT0BX0FHSxvu6l0k3FR3qcBmtwBmzA
XeDYusBzFjqtZSpqP73v7DH3/MEROAFm6wGek9AVOAUW62G3Xj640L1fSIP1
WpZXz5R+/rd7f5IBGzvvuwt6DM6CaV24X7gu7BE4AWa7/0DxaqnLPX/rYs/K
eshtU07Mcn8fU9aDPQs4DfMaega6gd4FjmE+C51awl7YEvYInIB5hp6ErqCn
wALzaeganIF5A70Legw9C6b1THM3/bj9Gfd5uNC62nMEToAZ9pPQFTgFFuvn
ek/c8aD7+58Ga3AGbMBd61laPcfQs9Z3tVf+8E3u88cGpjt7/D9qJxD2
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 336.}, {0., 0.18242549574128475`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1863007]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25974],

Cell["\<\
With an explicit Runge\[Dash]Kutta method the step size is changed more \
often:\
\>", "ExampleText",
 CellID->23488],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], 
           RowBox[{"Cos", "[", 
            RowBox[{"x", "+", 
             RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
       RowBox[{"StepMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}], ",", 
       RowBox[{"Method", "->", "\"\<ExplicitRungeKutta\>\""}]}], "]"}], "]"}],
     "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3022],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxVkn1MjWEYxh82Dktko7JpjqSlzJKE0umq03epc45Ki7ajljB9kdB0eimS
6GPYaobD0vI1Kh8bR3vFyMcoilRymI/JpmP1h53+yDH3PXq2d+9+7/U+1309
9/3MT8vRZUwUQsTYnj/vv8sSJMYvtBbmzKnSvqLvU5CbrD545NxTYgc036i2
tj3sIJ4F53xlWHFvJ7EzOhb55xuameei8f4SXfval8RKOPiEe7e8YXaFQ0Li
5IhUrucGsaDW3quT2R0de1/OuFnRRewBTVTClUtpr4k9YZleuzOr5g3xYuiP
F4epL/cQLwEO1OzZZeon9ga6m6amFbwjXgppoZNv6f73xD7IlS88S3z0lngZ
RHKZNd99gNgX+hUfA+1X9hIvh0WbpMuzcj0/WGKfKM7cYl4BZakp8UMO+63E
tReN3xuUnHcVzEMnq5wqOY8/5N06xeNN3cQBMJ+yHj7VxH6rYczNcA0Z4PqB
MHr2dVkfsa6CZdpZVXKFmTgIxp7U83e2cH5A/L8kQOP8+vM05UfSgyGy1CEJ
Y+QnBaPaTltpv579QmCcHXPr+XXWQyBOy9p6K+dRQ9nqPuZpYl0NpIgHEwpZ
D4WcW3e16C7roRCbUlw2Z/F8wqD0vTDQaaL5SmEwligiAjK5f+GQChxUPzvI
TwoHplRu9Ctg/wjoy+1Myw0fSI+AaL04a3sd548EUu3XPB6hfkiRkD2PNah+
cX+iIJcZ6lsyWbexPC8vNJv9o4E6g8uPij7SoyFP8hiuT+X5xcD4KSrhi5nq
STGQtg66qLifIhbCUp19wsj5bWw8umu4i8+3BpLH0Killv1tHOt171sV9ycO
+h2KJPnpV9LjYE4vOaQu5vPFQ3m8Pd/Om+vHj5+30EC/T7EhpYz6Aw2kZ+WS
m5ruo6SBuNTQuC6a6skaYEPdxtBKzqeFGJHbvK5QPmiBnsjRotukSzZ2PJfT
v4332zjAr3xSH993HcTlz4Mz06lfsLEhTy218H4b9/VmOvrR//I//g3HeFCf

     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 75.}, {0., 0.7436219864135687}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->887793283]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17519],

Cell["Difference order of 8:", "ExampleText",
 CellID->24083],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], 
           RowBox[{"Cos", "[", 
            RowBox[{"x", "+", 
             RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
       RowBox[{"StepMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}], ",", 
       RowBox[{"Method", "->", 
        RowBox[{"{", 
         RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
          RowBox[{"\"\<DifferenceOrder\>\"", "->", "8"}]}], "}"}]}]}], "]"}], 
     "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15966],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{1., 0.27928190880409304`}, {2., 0.20114493895359448`}, {3., 
     0.25664988327144744`}, {4., 0.2660061391999726}, {5., 
     0.35912930151571043`}, {6., 0.3712671281058948}, {7., 
     0.32884821394023667`}, {8., 0.35195849044691885`}, {9., 
     0.35451000455480575`}, {10., 0.4111107114114887}, {11., 
     0.3975605655433938}, {12., 0.4707450387807306}, {13., 
     0.43003937288713967`}, {14., 0.48956618086631387`}, {15., 
     0.527746085376247}, {15.686085666712222`, 0.16566706053611924`}}], 
    LineBox[{{16.37906745846528, 0.16566706053611924`}, {17., 
     0.4370384659417903}, {18., 0.34768686274846683`}, {19., 
     0.33691682762482333`}, {20., 0.32034174116096814`}, {21., 
     0.3839742896380862}, {22., 0.30613812073403857`}, {23., 
     0.344154061928692}, {24., 0.3377081881273476}, {25., 
     0.35381388505023637`}, {26., 0.3613133547539942}, {27., 
     0.42377364649160754`}, {28., 0.3901775861646648}, {29., 
     0.46147087038362855`}, {30., 0.45540234563363846`}, {31., 
     0.48057436406477194`}, {32., 0.543737358509782}, {32.69531786267149, 
     0.16566706053611924`}}], 
    LineBox[{{33.33646149942604, 0.16566706053611924`}, {34., 
     0.4923804382335675}, {35., 0.38580343045581955`}, {36., 
     0.37339791038325565`}, {37., 0.3541976294148981}, {38., 
     0.4133952603308959}, {39., 0.3395982573297296}, {40., 
     0.3670838983387501}, {41., 0.35555428656681265`}, {42., 
     0.37071229306711295`}, {43., 0.40065682120686574`}, {44., 
     0.410791635750412}, {45., 0.43686745468297516`}, {46., 
     0.5015362308760594}, {47., 0.45885237722536587`}, {48., 
     0.585226201061019}, {49., 0.533531033794997}, {50., 
     0.44139535438834443`}, {51., 0.40483982543699426`}, {52., 
     0.3891781861335062}, {53., 0.45939945659599246`}, {54., 
     0.36074366725798157`}, {55., 0.3927011352451828}, {56., 
     0.3723452039657076}, {57., 0.3996933184809812}, {58., 
     0.4773040047118222}, {59., 0.418337327740506}, {60., 
     0.554644198269088}, {61., 0.4528544934086156}, {62., 
     0.6353147409660274}, {63., 0.5323226572951221}, {64., 
     0.46794773432950976`}, {65., 0.4163130084369975}, {66., 
     0.4068502351592649}, {67., 0.4763441968346207}, {68., 
     0.3732552915852807}, {69., 0.4030535680167695}, {70., 
     0.38242181918290896`}, {71., 0.41983782841134243`}, {72., 
     0.44143482316767546`}, {73., 0.45584738863156105`}, {74., 
     0.3817618853689595}, {75., 0.3817618853689595}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0.2},
  PlotRange->{{0., 75.}, {0.16566706053611924`, 0.6353147409660274}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 110},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->11084001]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2794],

Cell["With a difference order of 3, the steps are much smaller:", \
"ExampleText",
 CellID->7646],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], 
           RowBox[{"Cos", "[", 
            RowBox[{"x", "+", 
             RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
       RowBox[{"StepMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}], ",", 
       RowBox[{"Method", "->", 
        RowBox[{"{", 
         RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
          RowBox[{"\"\<DifferenceOrder\>\"", "->", "3"}]}], "}"}]}]}], "]"}], 
     "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8324],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 111},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->329765648]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31089],

Cell["Extrapolation tends to take very large steps:", "ExampleText",
 CellID->27098],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], 
           RowBox[{"Cos", "[", 
            RowBox[{"x", "+", 
             RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
       RowBox[{"StepMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}], ",", 
       RowBox[{"Method", "->", "\"\<Extrapolation\>\""}]}], "]"}], "]"}], "[",
     
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21797],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{1., 0.5249058264063586}, {2., 1.0823463299488527`}, {3., 
     0.8066112529652791}, {4., 1.0206483031042035`}, {5., 
     1.811580402799418}, {6., 1.431511206688528}, {7., 1.2813244438707843`}, {
     8., 1.8111561260882638`}, {9., 0.}, {10., 1.8632767826254177`}, {11., 
     1.8632767826254195`}, {12., 1.8710899866989479`}, {13., 
     1.8690805992883597`}, {14., 1.8066610249452317`}, {15., 
     1.648587298361278}, {16., 1.3794939584657904`}, {17., 
     1.5681572104288968`}, {18., 1.51608687642252}, {19., 0.}, {20., 
     1.3449695686685814`}, {21., 1.3356041862039305`}, {22., 
     1.9602889424332588`}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 22.}, {0., 1.9602889424332588`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 123},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->98812527]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "NormFunction",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->448770909],

Cell["\<\
Plot the actual solution error when using different error estimation norms:\
\>", "ExampleText",
 CellID->340610123],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"L", " ", "=", " ", "10"}], ";", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"sol", "[", "p", "]"}], " ", "=", " ", 
      RowBox[{"NDSolve", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           RowBox[{"D", "[", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"t", ",", "x"}], "]"}], ",", "t", ",", "t"}], "]"}], 
           " ", "\[Equal]", " ", 
           RowBox[{"D", "[", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], 
          ",", " ", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", 
           RowBox[{"Exp", "[", 
            RowBox[{"-", 
             RowBox[{"x", "^", "2"}]}], "]"}]}], ",", " ", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{"Derivative", "[", 
              RowBox[{"1", ",", "0"}], "]"}], "[", "u", "]"}], "[", 
            RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", 
           RowBox[{"-", 
            RowBox[{"Exp", "[", 
             RowBox[{"-", 
              RowBox[{"x", "^", "2"}]}], "]"}]}]}], ",", " ", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"t", ",", 
             RowBox[{"-", "L"}]}], "]"}], " ", "\[Equal]", " ", 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "L"}], "]"}]}]}], "}"}], ",", " ", "u", ",", 
        " ", 
        RowBox[{"{", 
         RowBox[{"t", ",", "0", ",", 
          RowBox[{"4", " ", "L"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "L"}], ",", "L"}], "}"}], ",", " ", 
        RowBox[{"NormFunction", "\[Rule]", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"Norm", "[", 
            RowBox[{"#", ",", " ", "p"}], "]"}], "&"}], ")"}]}]}], "]"}]}], 
     ";", "\n", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{"Evaluate", "[", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"40", ",", "x"}], "]"}], " ", "/.", " ", 
           RowBox[{"sol", "[", "p", "]"}]}], ")"}], " ", "-", "  ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"Exp", "[", 
            RowBox[{"-", 
             RowBox[{"x", "^", "2"}]}], "]"}], " ", "-", " ", 
           RowBox[{"2", 
            SqrtBox["\[Pi]"]}]}], ")"}]}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "L"}], ",", "L"}], "}"}], ",", " ", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"p", ",", " ", 
      RowBox[{"{", 
       RowBox[{"1", ",", "2", ",", "3", ",", "\[Infinity]"}], "}"}]}], 
     "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->93900919],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 227},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->220154982]
}, Open  ]],

Cell["A plot of the best solution:", "ExampleText",
 CellID->63035132],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"u", "[", 
     RowBox[{"t", ",", "x"}], "]"}], " ", "/.", " ", 
    RowBox[{"sol", "[", "1", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "40"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}], ",", " ", 
   RowBox[{"Mesh", "\[Rule]", "False"}], ",", " ", 
   RowBox[{"PlotPoints", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->452718695],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 144},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->861775330]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "SolveDelayed",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->421266211],

Cell["\<\
The solution cannot be completed because the square root function is not \
sufficiently smooth:\
\>", "ExampleText",
 CellID->431102374],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox["x", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "^", "2"}], "+", 
       RowBox[{
        RowBox[{"x", "[", "t", "]"}], "^", "2"}]}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", 
      RowBox[{"1", "/", "2"}]}]}], "}"}], ",", "x", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", 
     RowBox[{"10", " ", "Pi"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->149718405],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"mxst\"\>"}], ":", 
  " ", "\<\"Maximum number of \\!\\(10000\\) steps reached at the point \
\\!\\(t\\) == \\!\\(2.1309737431864053`\\). \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NDSolve/mxst\\\", ButtonNote -> \
\\\"NDSolve::mxst\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->23795732],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"mxst\"\>"}], ":", 
  " ", "\<\"Maximum number of \\!\\(10000\\) steps reached at the point \
\\!\\(t\\) == \\!\\(1.085288769586131`\\). \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NDSolve/mxst\\\", ButtonNote -> \
\\\"NDSolve::mxst\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->160434433],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "2.1309737431864053`"}], "}"}], "}"}], 
        ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.085288769586131`"}], "}"}], "}"}], 
        ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}], "}"}]}], "}"}]], "Output",
 ImageSize->{344, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63102960]
}, Open  ]],

Cell["\<\
When the solving is delayed the equation is treated as a DAE instead:\
\>", "ExampleText",
 CellID->907901123],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox["x", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "^", "2"}], "+", 
       RowBox[{
        RowBox[{"x", "[", "t", "]"}], "^", "2"}]}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", 
      RowBox[{"1", "/", "2"}]}]}], "}"}], ",", "x", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", 
     RowBox[{"10", " ", "Pi"}]}], "}"}], ",", " ", 
   RowBox[{"SolveDelayed", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->68376451],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "31.41592653589793`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{344, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->760797536]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], " ", "/.", " ", "%"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", " ", "0", ",", " ", 
     RowBox[{"10", " ", "Pi"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->86376055],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->23126441]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "StartingStepSize",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->274331389],

Cell["\<\
For a very large interval, a short-lived feature near the start may be \
missed:\
\>", "ExampleText",
 CellID->585109129],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{
         RowBox[{"10", "^", 
          RowBox[{"-", "4"}]}], " ", 
         RowBox[{"x", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{"t", " ", "<", " ", "1"}], ",", " ", 
         RowBox[{"t", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"10", " ", "Pi", " ", "t"}], "]"}]}], ",", " ", "0"}], 
        "]"}]}], ",", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", 
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
     "}"}], ",", " ", "x", ",", "  ", 
    RowBox[{"{", 
     RowBox[{"t", ",", " ", "0", ",", " ", "10000"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->54109572],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10000.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{339, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->184329392]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{"Abs", "[", 
   RowBox[{
    RowBox[{"x", "[", 
     RowBox[{"RandomReal", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", " ", "10000"}], "}"}], ",", " ", "10000"}], "]"}], 
     "]"}], " ", "/.", " ", "sol"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->748158840],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->2312514]
}, Open  ]],

Cell["\<\
Setting a sufficiently small step size to start with ensures that the input \
is not missed:\
\>", "ExampleText",
 CellID->131514743],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{
         RowBox[{"10", "^", 
          RowBox[{"-", "4"}]}], " ", 
         RowBox[{"x", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{"t", " ", "<", " ", "1"}], ",", " ", 
         RowBox[{"t", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"10", " ", "Pi", " ", "t"}], "]"}]}], ",", " ", "0"}], 
        "]"}]}], ",", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", 
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
     "}"}], ",", " ", "x", ",", "  ", 
    RowBox[{"{", 
     RowBox[{"t", ",", " ", "0", ",", " ", "10000"}], "}"}], ",", " ", 
    RowBox[{"StartingStepSize", "\[Rule]", "0.1"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->273984165],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10000.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{339, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->85900149]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "[", "t", "]"}], " ", "/.", " ", "sol"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "2"}], "}"}]}], "]"}], ",", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "[", "t", "]"}], " ", "/.", " ", "sol"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"t", ",", " ", "0", ",", " ", "10000"}], "}"}]}], "]"}]}], 
  "}"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->108070529],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->322912500]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "StepMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->26605],

Cell["\<\
Plot the solution at each point where a step is taken in the solution \
process:\
\>", "ExampleText",
 CellID->26806],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{"Reap", "[", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], 
          RowBox[{"Cos", "[", 
           RowBox[{"x", "+", 
            RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
      RowBox[{"StepMonitor", ":>", 
       RowBox[{"Sow", "[", 
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"y", "[", "x", "]"}]}], "}"}], "]"}]}]}], "]"}], "]"}], "[",
    
   RowBox[{"[", 
    RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17438],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->272758055]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27518],

Cell["Total number of steps involved in finding the solution:", "ExampleText",
 CellID->28262],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Module", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"c", "=", "0"}], "}"}], ",", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], 
          RowBox[{"Cos", "[", 
           RowBox[{"x", "+", 
            RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
      RowBox[{"StepMonitor", ":>", 
       RowBox[{"c", "++"}]}]}], "]"}], ";", "c"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27944],

Cell[BoxData["337"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->170342971]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17015],

Cell[TextData[{
 "Differences between values of ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " at successive steps:"
}], "ExampleText",
 CellID->22893],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}], "==", 
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], 
           RowBox[{"Cos", "[", 
            RowBox[{"x", "+", 
             RowBox[{"y", "[", "x", "]"}]}], "]"}]}]}], ",", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
       RowBox[{"StepMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19323],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxd12tslEUUxvFTtFgJmooXvNIXFMRK6kLlJpceEIFSoEvLpQiWaaUUCoWF
AtEPxGMNConoBhGbKHWiqEEUFzECBel4C9+wCTFEE82LQEUNyUaigsHLkM4m
M0+/NL/+z9mTTbrt7sD6NVUNvYgomUd09XvPV7bs/OW+JY1LhpS5H3CPRzkX
gAvBt3D9mPRl/mul8+3gu1kFjnhp4EGcGj/9wnuzXnO+DzwEPBRcDB7GanPN
hkNP73YuASf4qRcuvnyhqN15OHgEpwKXgh9mCTyS29450bFttnYexTrwaPAY
8FjwI+BxnBo77fnNR/c4jwdPYAk8EXoZmMN5YeiTOFs69MCNW/a7Pokv+abJ
0MH0KHQwTYEOpsdY73h2/ZWLB10H01ToYJoGfRpngj4dOpjKoZcztU7ud+zm
r12fEVpmcHp3d1HzAuN6BYtvqYA+M7TMDOdpFnQwzYbHnw29EvYr4fGTrH1z
Eh4PbJJM+Xtb+/z+ldufw/G1nnlO2GUOR6d+3ffLd7l9MFUx++aqsAt0U8VS
O6CjOv+o269m9s3VYRewAdNcpnP76/Y07XT7YAEbMM1j6vb358E+2OT8rtuf
z+Kb54ddwAbmaUHYGSwLmD/bevD8IPd6MmCqCc1gqWEJ9qHTQtgHC9iA6XHY
BwvYgGkR7INlEctxOdcyzr1+je339u6smHDA7S9memjjqXuij9z+YpYSzwI2
YHoC9sFiHexDp1qmwd+/8tPMQ26/lsW3QDfQaQnsg2VJOG+gk2LppP+yFzt7
HIFZMflW0AWsYd5Aj8FUx7z95DVXln7h7tcx+WboCixgDTbgGEz1cL+e6fDO
b/uOPO7ug5Wd9y3QNXRju6QK2uYNd/fB9GToCMxgZW2uepK7n+vOGmysD73Y
59/3N7j7V/c909KwR2C27tj7Z962t9196yOeBbqGbqDH1ud+2zq0NuPuN4SO
GljOemboCrpA12AD8zF0Whb2aFnYGbqCLmAN88b656kNAz/92N2385VrRtwx
MPf6b2RKeo4aw85gBRawbmS5/oPyhjPungHHdv71pl03bP7Q3V/O9IbnCMzL
w3kFXaBr6GY5y8Tm9K76T9x9233TirBHK8LO0BVYYF7bPmLj0Sv/HHb3bS/1
HEOnptARmMGqKXw8ga5t7+5/f/fk3PuVJmbfMXRaGToCM1itZPEt0LX1ydOX
6MSX7j44tvu+aVXYIzCvCucVdAFrmDfQYzA1h/NRM1OvvnnFM3L/f6zzPCvr
C3HbWwUd7j5YW/d/s+h0vvt9MOAYTKtDR2AGK7CANdiAYzCtgftgBiuwgDXY
gGMwpZhuSoz+o8a9Hyi07uc5gp6AztCT0BX0FHSxvu6l0k3FR3qcBmtwBmzA
XeDYusBzFjqtZSpqP73v7DH3/MEROAFm6wGek9AVOAUW62G3Xj640L1fSIP1
WpZXz5R+/rd7f5IBGzvvuwt6DM6CaV24X7gu7BE4AWa7/0DxaqnLPX/rYs/K
eshtU07Mcn8fU9aDPQs4DfMaega6gd4FjmE+C51awl7YEvYInIB5hp6ErqCn
wALzaeganIF5A70Legw9C6b1THM3/bj9Gfd5uNC62nMEToAZ9pPQFTgFFuvn
ek/c8aD7+58Ga3AGbMBd61laPcfQs9Z3tVf+8E3u88cGpjt7/D9qJxD2
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 336.}, {0., 0.18242549574128475`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->369956043]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "WorkingPrecision",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->386402526],

Cell["\<\
Error in the solution to a harmonic oscillator over 100 periods:\
\>", "ExampleText",
 CellID->12825289],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"T", " ", "=", " ", 
   RowBox[{"200", " ", "Pi"}]}], ";"}], "\n", 
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"y", "[", "T", "]"}], " ", "/.", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
         RowBox[{"y", "[", "x", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", 
       " ", 
       RowBox[{
        RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
       RowBox[{
        RowBox[{
         RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], 
      "}"}], ",", "y", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "T"}], "}"}], ",", " ", 
     RowBox[{"MaxSteps", "\[Rule]", "Infinity"}]}], "]"}]}], "]"}]}], "Input",\

 CellLabel->"In[1]:=",
 CellID->458275453],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.030000000000001872`", ",", 
   RowBox[{"{", 
    RowBox[{"-", "3.597502940932694`*^-6"}], "}"}]}], "}"}]], "Output",
 ImageSize->{157, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->328871731]
}, Open  ]],

Cell["\<\
When the working precision is increased, the local tolerances are \
correspondingly increased:\
\>", "ExampleText",
 CellID->61400925],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"y", "[", "T", "]"}], " ", "/.", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
         RowBox[{"y", "[", "x", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", 
       " ", 
       RowBox[{
        RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
       RowBox[{
        RowBox[{
         RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], 
      "}"}], ",", "y", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "T"}], "}"}], ",", " ", 
     RowBox[{"WorkingPrecision", "\[Rule]", "32"}], ",", " ", 
     RowBox[{"MaxSteps", "\[Rule]", 
      RowBox[{"10", "^", "6"}]}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->166953977],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.806999999999998`", ",", 
   RowBox[{"{", 
    RowBox[{
    "-", "9.4968389190802633536459684998384587`16.179399203409254*^-14"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{246, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->454478345]
}, Open  ]],

Cell[TextData[{
 "With a large working precision, sometimes the ",
 Cell[BoxData["\"\<Extrapolation\>\""], "InlineFormula"],
 " method is quite effective:"
}], "ExampleText",
 CellID->534795734],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"y", "[", "T", "]"}], " ", "/.", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
         RowBox[{"y", "[", "x", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", 
       " ", 
       RowBox[{
        RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
       RowBox[{
        RowBox[{
         RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], 
      "}"}], ",", "y", ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "T"}], "}"}], ",", " ", 
     RowBox[{"WorkingPrecision", "\[Rule]", "32"}], ",", " ", 
     RowBox[{"Method", "\[Rule]", "\"\<Extrapolation\>\""}]}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->49257934],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.5719999999999947`", ",", 
   RowBox[{"{", 
    RowBox[{
    "-", "2.4116395064013052074053723123359`13.584132521311709*^-16"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{232, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->190981274]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->4218],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Duffing Equation",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->25369],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
        RowBox[{"0.15", 
         RowBox[{
          RowBox[{"x", "'"}], "[", "t", "]"}]}], "-", 
        RowBox[{"x", "[", "t", "]"}], "+", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], "^", "3"}]}], "==", 
       RowBox[{"0.3", 
        RowBox[{"Cos", "[", "t", "]"}]}]}], ",", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], "==", 
       RowBox[{"-", "1"}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "x", 
    ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30472],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "50.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->189090723]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "/.", "s"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->27161],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->415242567]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11135042],

Cell["The solution depends strongly on initial conditions:", "ExampleText",
 CellID->126108385],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], "+", 
        RowBox[{"0.15", 
         RowBox[{
          RowBox[{"x", "'"}], "[", "t", "]"}]}], "-", 
        RowBox[{"x", "[", "t", "]"}], "+", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], "^", "3"}]}], "==", 
       RowBox[{"0.3", 
        RowBox[{"Cos", "[", "t", "]"}]}]}], ",", "\n", " ", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], "==", 
       RowBox[{"-", "1"}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], "==", "1.001"}]}], "}"}], ",", 
    "x", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27825],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "50.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->387525007]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "/.", "s"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "50"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1494],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->381498695]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Lotka\[Dash]Volterra Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->24195],

Cell[TextData[{
 "The Lotka\[Dash]Volterra predator-prey equations [",
 ButtonBox["more info",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://mathworld.wolfram.com/Lotka-VolterraEquations.html"], None}],
 "]:"
}], "ExampleText",
 CellID->27796781],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], "-", "1"}], ")"}]}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], 
       RowBox[{"(", 
        RowBox[{"2", "-", 
         RowBox[{"y", "[", "t", "]"}]}], ")"}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", "1"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "2.7"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->101383601],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->16513986]
}, Open  ]],

Cell["Phase plane plot:", "ExampleText",
 CellID->149801889],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", 
       RowBox[{"y", "[", "t", "]"}]}], "}"}], "/.", 
     RowBox[{"First", "[", "%", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25831712],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{169, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->55907042]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Lorenz Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->3830],

Cell[TextData[{
 "The Lorenz equations [",
 ButtonBox["more info",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://mathworld.wolfram.com/LorenzEquations.html"], None}],
 "]:"
}], "ExampleText",
 CellID->11628256],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "\[Prime]",
         MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"-", "3"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"x", "[", "t", "]"}], "-", 
          RowBox[{"y", "[", "t", "]"}]}], ")"}]}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["y", "\[Prime]",
         MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{
         RowBox[{"-", 
          RowBox[{"x", "[", "t", "]"}]}], " ", 
         RowBox[{"z", "[", "t", "]"}]}], "+", 
        RowBox[{"26.5", " ", 
         RowBox[{"x", "[", "t", "]"}]}], "-", 
        RowBox[{"y", "[", "t", "]"}]}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["z", "\[Prime]",
         MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], " ", 
         RowBox[{"y", "[", "t", "]"}]}], "-", 
        RowBox[{"z", "[", "t", "]"}]}]}], ",", 
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], "\[Equal]", 
       RowBox[{"z", "[", "0", "]"}], "\[Equal]", "0"}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "200"}], "}"}], ",", 
    RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16549],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "200.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "200.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"z", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "200.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{323, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->405630080]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], ",", 
       RowBox[{"y", "[", "t", "]"}], ",", 
       RowBox[{"z", "[", "t", "]"}]}], "}"}], "/.", "s"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "200"}], "}"}], ",", 
   RowBox[{"PlotPoints", "->", "1000"}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Hue", "[", "#4", "]"}], "&"}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1701],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{147, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->188297404]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Heat Equation",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->18561],

Cell[TextData[{
 "Simple model for soil temperature at depth ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " with periodic heating at the surface:"
}], "ExampleText",
 CellID->1002821242],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}], "==", 
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "==", "0"}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0"}], "]"}], "==", 
      RowBox[{"Sin", "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "5"}], "]"}], "==", "0"}]}], "}"}], ",", "u", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->562818297],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "5.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{380, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->38553498]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"t", ",", "x"}], "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->811388219],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->408728140]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Wave Equation",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->13303],

Cell["Simple wave evolution with periodic boundary conditions:", "ExampleText",
 CellID->3422311],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["\[PartialD]", 
         RowBox[{"t", ",", "t"}]], 
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}]}], "\[Equal]", 
       RowBox[{
        SubscriptBox["\[PartialD]", 
         RowBox[{"x", ",", "x"}]], 
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}]}]}], ",", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"-", 
         SuperscriptBox["x", "2"]}]]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["u", 
         TagBox[
          RowBox[{"(", 
           RowBox[{"1", ",", "0"}], ")"}],
          Derivative],
         MultilineFunction->None], "[", 
        RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", "0"}], ",", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", 
         RowBox[{"-", "10"}]}], "]"}], "\[Equal]", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "10"}], "]"}]}]}], "}"}], ",", "u", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "40"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14992],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "40.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "10.`"}], ",", "10.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{402, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->239769635]
}, Open  ]],

Cell["Plot the solution:", "ExampleText",
 CellID->513315368],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DensityPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"First", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x"}], "]"}], "/.", "\[InvisibleSpace]", "s"}], 
     "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "40"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "30"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->28969],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9vQd7VFe2LVpH2JgMJjjn0A7tDu7gTqdPt7ttN+AIGDBgwBiTEQIFMigH
hHIAlANJOeccS1mqUqmkklQ5nfPOvfcn+HtvzLX23rWrJIzs79zXXzc0UtXe
K8w1Zhpzrk8P+B895H3A/9jBAy/9y/fA6aPHDvq99JGPL3606N8UCkUN/lfx
lAL////F/2P/q/n+++/xC9mfG/CnF371Mv5ehL+9hH+/gL8f8fg3/X6R8O+X
6Nv0ge/F/3gJv1Uo1rv/Tni6+0/ouYpFrp88R388yv8t/TnnUy/I/j3f75+f
9z20GP9Gf//k0c3/3B8zsmeF9VQ8Kk1Q/smfNKqHvfM5YVe85jzp+fm+Ofdd
/1dG9fC1fPC4f/qoaP0Vi+Wf+J9eUz42+XkSTwT9/aLHv3/MeZp7auf++dj3
7ud3Mf7Olka15KHrs2jOJ17z+MSrD/n3K9+7TrD0EOH9wkTYf6R5PXw8sn+n
f+95jhc2J3ob+z8bV4Tjv4qn6I+lm1aGb1oZsWlVxKcbrn/5Usq+3965eHAo
/455VG1gK6dY0tFvzbg1cWF/w553sr98OfWzJ2LwaXzHC199F89YhL8VS/CD
zWsitzyb8NUb6We21ifHaJp67f9t07BDvnRCO3uvxHrl6NjBvxRtf/nGZ0/E
bl4VwR4TrniGxvGIa1V+5E7J/v2MOCM8fPPqyE/XRW97MeXAH+7jvXcKzKoJ
cUY9Y9acnMkrh9v2//bOV29mfP5U3OY1UcKMfoNnePEZrYr4+PGoz5+M/fqX
ub7b21NiZ9qG7f/LLsxoSjdTWGEN9tEc+lvp9ldufLr+Ot64ebU0o0U/cZfk
/34Sz3mEjQUP/nhN1BfPJOz+efaR98uvXewvrjZpZgx8Pv1a6+27k8HeHQf/
XLT751lbnk38ZG00doPm8Vu+JmxB8NMdr9365vf3Lh5oy0zXdmns/2XVchhe
Nq2fLW+yxIUMndxciwlvfS4JL8R3SD5WhD8974we+979JC/+gfHj5de2vpD8
9a/yTm9tToqerOlwTMzy8S8bmrIUFU1E+bad2lxG734h+dMNMfg8fz3tiGLZ
ZrYbn6yLxm5hz478oyL0zPDdIluf1oZJsDks1+lnG5SOm0nTZ/d07PvNbYgz
Ps8eEsGnwWR+7jzWzzNuJtWQJXrrZ0/GYouPvF9x+fBwZpq1pcemN87wdVv+
3/aJ8VljZZn6ZkRn4IGqw/+O16ZijDhOfAYKOiOK5UyarkFMIFBYXEie387O
+AhDaYV9dNr8XzZhH5aazNMdA9bcHFvIafXxj2p2vHYTX/h47TV8X0H7+SNn
sIyG/0TslucScQ78d3VFX50puGdQjpp0Br3DOsmO3QqLWTehN3Z0Td7P6os7
33B2W+HeX6ZD2DBvDBhPwMt/RbNYyc/Ep+uj8asvnon/6o2Mk5vqAk+oM1L0
jS3Gcb0Rw//fdg2X5GVO6+S0cXZQYyopMcSHz57bp8TxB0R8tiFGkM6Fzwbv
XhFOuwikWn8d2wGcgRBcOTKYdsPc0O8YmTbjXWazTrFo1jCjWIkdMpqnpwz6
PpW5sno6LW4w6Fjzob/eg1Dg9SQXa0i4FMK8gBlMRCF4215IBgqe2doaEzJb
VO3o1dqwVgbTtNE0rfBiz54xzJjMuhnj7NiMuXnEkZ1jDjo5fPSDyp0/Swc6
8vVauLQ9JRsBvr7z9TQIx+ktDTFBqoIyR9uATTVjondNmyDrCi8l/ljVP2mF
1OGnQ2pTY6ft9h191Pm+k5vKAQ/YaggaST0bxM8ZngmYtgI/xvCwtbvezjz0
H6WBx3sy0vT1vY4hnQUPw2oNTlkX0RtWjOrM04ZZvKRjyFZa74yPGPfb0YRD
hZWXDhWNXNSyXg/euVUkNWuisLp0dt9I/+6vJVe/a0pLUte02vtGzVhdrGPP
pL2m2+lVWudUrG4ecg5MWSGSEzOGrkFLeaMz6boOZxpf3POLHGyRMDt+pt8W
UeY5voVYg4N/Kfbd3hIXOlFY6cRpwpMmDYahaWur2lnW4MQ7qtqdXVrHyIwF
Lx9Um+o77NnZ1hCfQWDrrrcyv3g6Hs/BkPGSJxe0h08wjBJPx3V+4vb+Ou/M
trbrF/rv5Y11DNu8NHqjYu3gtLVO6cy/67yR5Ey76SwodXSO2UdnzDg7YzOm
lmFnbo4j1HccX8RZ4QPBY2n33hbP9FoJlXCmT29tDTwxmn7LXtftGNZZ8A71
rKlzwllc48xId6YmOHPznTVdzn4df/8y6L6iMntcuNF/ZyfkDFiLwUqn/IkF
zXaDx2wFLEg/sbH26nHVzfiZ6jrDouFpi2JR+4D1foE9Psqh8IoNcygW3Uhw
lFXalBqrwot+v6RPYy0uMcaHzVw4MHD4b2Vbn0/CaPBUt/kuwVHFmYVaw5Kc
26uMCZy5f8/YNWxhz3gEolJVb0u/4cAbvOLoLXfu2Ft6bfRL9ollDV32zExL
kLfm8PvlO3+WBvH/XIZsG37MnAWciMZ4gOI4D1j/6xhPqbVL4/AanzUpNmA8
lU2OW6mOq8fGfLd3nN3TE3Fel5PjaOyyj+pM+AjOc/OQPSfXHOwzdvxfNbve
zGCTvibYQzRvxRMC2q2MwO9gdQFajn9UjZ3OSDPV9dix03iQGgLTZ8u/7YgO
1J/f3+e3o+Py4ZHUREdZjaNv0sZGs6xXay+pdWK/fb9sg+bfRvvN4Q8WT/jC
5v44O2EY1Ro+fzKLmBY+9NeSK0d60m4a63qdgCQcMghfi9J2p8B57fK4z+dN
Xof/Xq5Yfe7rrsRr0yXVjs4By/i0YUJv6BqzlTU4EiInAna17v/d3e2v3BRB
JVxBiKV4hb/p8yfpNVhmPzrOmsJKe/uwHeun1Ru6hyzlDY7UeP2lg73AXrzI
+5OGiHOq3NuOxh67ataM0QBXGvudWVnWYO8+OtpvZ2HvBXRcSei4fsHzV6wX
d0TYF1qHtdcwugN/LDizrSk2eLygnMAGq47R9UMIWp2p8ZOXvu08+OdCL0xS
seboPypCfPqys4x1nY4+tWWKYU9Dhy03zxrmN3jsw0oXfDPseWsjW/NVZFnj
h0ADgNPhv5cFH2/LvDVZ22pXjprpIRPmhm57/h1L5Lmhkx/Xsnc9jlGd29uW
FD0BwIMwqGeMQMHuESuEgS/7t38qhBLgdqsgeT9iNZ51Ww2G75Ar4NGpz5qu
Hh9Ov+WASMCuw0uhQ9rVztv3LBFn1Sc/roeBCh1Mduh792BWx0dMl9Taeibs
Cq8Jjk+WkipHYrQZ5+bAe/dgwOO50ll9wwN5cAqfp1MI0Y+93Ftwf6Jz2Mqe
8yTmXN5sT47Rn/9GCb2BYw/R3/VWBg5c6JnRnFxLy6gThhc+C8uhaQhC4gzx
Ufl80QyMgXDz1zGLa93CV2WFfFXIA9lwHQLHUet64Oy9uyYaH9SOYjH0QEWt
KSVWD3cLdgyOuBemongUewI5Dj6lyUwz1LebFYvo08sau205ebbQM5MYPkTO
A8FefwAewzYLOjmenjxV2zjLnrOyRWnJzTaEBUwBtvb+Oh8QhpduezEZtsD5
b/oTo/Ql5bAygM706SW949bCIlNcmN6FzpLltu6hun+thN8YEBYfW37sg6or
R0fSbxprOwmevVRAqMVt/eb8TFVkwIjP581Q75ibFyRdsRifB+Sd398fH2ku
KrUByNjnlyknbWWNzsRoi/9XnRBikpC5a7EynCyOddFAT7K2f3fXb0dnbKix
sMICUePvBU6XVUNCzJid98f1WPhF9N5HccYgpqGnh7NvqBo79Ar+Vqjoxl5b
VpYpyJtAG/LENRX3LRYmI/Qp0t4QjQgOoOQObWmOCdIUlNvbR+0AdKAWTnVV
k/VG9ODFfU2Y4PZXb2LZvSBLirWYLH5yYlNdyOnR7GxnY59jZJK+giGS6ZDn
DDk9iEXe/XbWlmcSBA9qVcRr9N51TDa5G3iNvx1QEni8LzPD3tDnHGDwDfBq
GbDn33Pys/rdX4phjir4u7HEkNJvfn/37K6mxFBlaaWhe9iC78Cs6FRDydjj
w7Wwuva9m0+Ol+i9LWxl1vIRspUhR5chHXDz6pH29JtT1c32niGzdlY/OG5q
7LLdLbJeuzjk/XE1JBgjAjIqvPAlxTqMEEKDzfb/qis+YgZQ16uxAX5wyJXj
Vph9SdGac1+3AKVwOHCAXMj3kpvOFTCNHeCYHa/DY77r+2VDXKiqqMrZNWqD
qYY5D+qslc2OG4naiwdaAWVk+z8Zu4jGQbuEjT32YUWYX19unrGu3d6vIjNX
OWIGaGdnG0JO9R7+Wyl31SU3e+3CceZNSb6xMziWODant7RcP995/562Y9AK
xxoqCOe4qh0DNFz8thcbiQXd9lIKfxv/LoQAtsfRf1YGnRzMyrI0DjmHdWat
wTA6ZQI4wm4M81cDzoENOL3cPqGT/yKN4Mm5+I9NA9J5f9wQdHIEaNo46Bye
NGMcMGvb1M78e9aYwBG/HdUkm88mwIXE4VF4kYP82FZC5swD793329meEKUr
a7D3TsDTwXcVy7qHrcWVjoQoo9+OdgD9NooNXMfg+Wl/fEFrtoZ74C5MiMFY
4Vkf+7D6ytGxW/ETNXVTI2Qt4pArHmsbtN6+Y4q8oDtDGJmHwWGh+EiXMg8R
LnwcFv3bPxUE7O6JC5stKjOTy8S+vaRvylZSbkmMMl46NHzswyq4S+QSromS
Vm6ujiCPE0cRMBQbYigosvRobOxZa4FQ5bXm5Fj9lSP9xz4sgVixwEKMdLD5
QCBHUBkhpzU52camXusi+u4ymBXZOfYg73HyS19Pw4TJzse6sUjXmoWvm5te
wc7BaYDahsV5v8jcNW7zGps1K9YN6KDNbKlxJhigwA3RpI0ShEN0ijAKchNI
LY1lpow3tMBbp+8vbe6z5uaaQs9oTjIgxjS53yOeTa5H8GNMFycNjkbgcVVG
urG+04rZ8jG0dM/mpmvC/ccgszCkaMaCXri2mT2JuxHYz4N/JoWcGGUqqbDC
TlCw75NeKamDpWzC9HCg8Sa5jv0RK7ZegllMGaPFAbp8qDftph2OH2AWJwzW
WNuIA1Zy1EUttB50K3CLdpWfML7m4vnGtkPw/Xe0xl/tKi6dAeh64RGKpVBk
ZY32xGs6v52w7O8CCV02tBzPAKnYNpg+eAqcgNgQbVGlrWvcDlTEc3pHLeU1
hpQI5fm9rQDGL19KlZ8vNpr1fAv55h3/qCbMT5172wknfXTSRHAxbW4eJvsp
8MQALBkyT+C28gALX7cfthNWsGD0XOyFdY3Tg0Gf2VIXEzRyv8zeOWpTM+t6
kLlYNxMnLx1sJzuWuez8fWzE5MYKrsqaqK3PJe55J/vI30tCfXpycvRkNo8Q
GveNmeva7Ln5xjA/5dF/lgFDseWEj0xY2Fl9WYglr4ncwh5y6D9Kgk52Z6TP
QCP1DpPtPTBmqm+33b5rijzXd3JTxde/zCWsFCA9goaxwjUjSB+OIryws1/3
JF03ltU5+iZgazG/vBeecI0zPlzlv7Nh/+/uQB7kpvnqBUkfW8mnPVdyTSQW
B+4AeefHBtNu2shZUzH3aNrcrnbcvm+LCxnx31m7550c2EqfcIzgZ8/LTY4o
vI/TyeSoBdq+pM7O5MiAZ/UMW0prHInX9AG7Ka5AcrT++ici3jzPnyNprfXX
t7FYsO+XzbGXuwoKKeapmWXPmbBVtDhSYmfO7+/G1gLbsCtSDHcRP19sVvDC
WSArCzAeemYoN9/aMkIaDA+BUcT8Pmewz8jJzXU47zhELoletfDVdEdrFnvE
eIDW8LevB+nv3bdiBbiFCLSuaLSmxhtgU53YWILzCHiW8GeRmFegmDKADPOn
0/SvmiBvVXa2CbYlnFXB1myCtZ/rCD0zAQMe+OGB3c96arPYz5+O3/n6LTwr
8MRIeqK6thHaTHjWkpZBW95tc3jABLeusQ54nHhOSIvzbJCE0p8zWx2m19mv
exOvGUprLEBIeNN4GCzwohIbABKaCqptx6s3eShOsfA1XenyCVhAh2z8f9Vc
PaZKSzXUtFighJnmW9bSNZOXORkRoDm9hXSwzN8R19ItMvRcIkTl7J7e+DBD
UbEZxgN/Sv+UDd5v4jVTwO5uYNs2psmlwPyzrrwCj8vH0XP+CKXeHRuqLyg2
dY/z5yzBc0orLEnXDJePjMIlgUVI2m1NFEUdvebzvSC1pz5tDD0znn1zsrFt
mmnkxxo6LVnpRmhk0nFvCTruR64dx2OMGeY9JGMvZVNaY4K0BeW2znEHITrs
PpWlomY2Jbznwn5KCLF0W4y0do+KT9kg5rfiMB9sZ8jpMXgUTcPOkSnBamsZ
4x7FGOYsREUlbFgh5O3WS5oKG/TVGxlH/1EReHwoM9PZMOAc0lnoQdOm1jFn
3h1n+FnN6S9ajrxfDmzg2v0R8US7efSwUmEKnt3TkRjcU1I23TNiZZYg6buS
Wlt8+BSLZN2FJDN9x3Bq4SuIeYuwiO1mseKiq0e70m/NVLc6oAwpUDzOYup3
DVHnlCc3lQmYTkEZQmPFYyKWCUm1ddHw0AB5Abs64iOMcDq6x6yAdMBiH/NU
k65pzu5p/uZ3dwnTN1znT8HXn3iA3gOu8Qf6bW/kPkfPuI3HvPrGreWNzpTY
yfP72ynXx0xpriRoLR/h8+Pa8/EorrOOfVAWdqY7N2emvs0GbcUyIpbadntW
liHER3nk/TLuhmBYXPHxM7owdHyCW1XQ1FiFrc8nA2B8Pq+/fqHrfuFMe59V
raO4O3ywqnZnavz0xQNdOKZcz0r5iSUSojOLiSM6zIgg72HySQYExxRS1Kh0
QBzD/MagyHBMsUZcWXsJccy5a7kmEhsHUw42S7D3cGaGrb7HMawlBQE10Tri
yLtjiwkcOL2lCstEmk80qynW+agoK6T2KO+aBb0dsLstMWqytMHRpbJhOzQz
hs4BS3GVIyFy2n9XO4CKC6WQPKCEy4LWcjlH9BWu2DJOJcs4ll8+NHArdqSm
fnpoEoiuJkTvUDvu3LdEnJ/w+aIZsonzjY0WEH2JEP2IkHTDtheTKfP3VWd8
xGxxjZWiKOwp8FhKKu2J0Zbz+2C41EEXkWyujpRWcx7NhyWHroXpHBdmLCy2
UySFnrWsfxJiaU2OMQQe6z7+YSGRBGSW8xI54gBE8GsYERTDOKPKyTU3D9qh
W/CcUZ25gfIwUNpqQtlXbwIi5cFqtk4/bEsu55gupPKu4wE4IJAAzD/6yszd
u4ZOIWuwZGjaWlFlSrmuB6ZjUvS2JykIT+foMXdM+pwwnTLvwPRgn4nM1On6
FgPPLAzPWBo7zNnp+pAzk3A7IEefM5Tk1soGcV+ldcTTyIx+M+Pk5vogb016
qr62xcyzFMvxrJZuQ27GTJj/JLx5YDtTey4TQrFE1HvrWbg1Fufo0N9KL3wz
kBAxW1Jq6tda8SA8BicOMBQfYfDbQXlwnG5aSdFvXL5wmXShO94GNQG7BwIJ
T6a+j5JNEyzZ1D7uvHPPEXlh0vfLdkgsPwICIi3xzIuRrwtoowDN1b6igonu
UR6vXQp5LGu0JV6b9d/VRbr6xRSXN/OExMnAKpAH+HoaTGRMLi5suqjK1g05
ZLFjpdpSXqlLDus7t7cHBi+lep9wWQ3L3DQfi4Ht/FkawCTUdzwn29Ey6oTO
w3PInxlxZuc4g7xH4TzzUJEkzcw9W9AKPibmJV3oBtHf++v801/UX786er/M
2T1uB64Bjvo1lspW543EmYvfdh36awlOtRuuLxVxHXvxsYCRmVjqkBNtWWla
Ci+NMV9kwtLQ48jNt4T59h/9Z4UrvMRHvcFd+5EB/stcuJ7B3sqsTJaSGKeU
xBCLseXnTUf4dh3/SEhJcESDsnXhvhAgY6iAHYW+TogyQd31jlmh+ijkxvxp
V+L41ZuSU4ivz7+Gy7/n3C1PJon7WmLwzCeGmrh6rC/tpqmmwzE8JeB6+ygc
a3tM0NDpLdVM1yTLcJ2/V8R1HgUlTbujOS5svLjG0TFM8UINSzvDPeNpZ571
FuSIswbW8aesELJhmBiMGuC/79bauKC+onJTB9N+PNNV0QKVPXV+Xzs0CE9B
8eGI9ogYl8XjuUly9J+VIT5jcGiaBx1DU7Qp8A0auoQcNWRyP4wJUf0xC3Lp
vKu5ymMVPflF0qrSuYJ/wUNiOJvXA2fuFTkoo8l8U6jgqnZ7aoIx2Lvn+IdF
LKYVK9mCK+XxQIpJxTJ2RsaxD6oCTw5nZZqbBh04U3gShL2p156bSwlunDsc
YxnCs3VlceD1MjUYC6MJEw481peRMFjfZhyeNPMUd5vKkX/XEn523PuTBpwG
vNLl16yUcrVrInm8D7vDtGBXUrQBMAOwYTNb0qu1FVc64sKNAbt7SAu+mcG9
7gevKUkmkbgeWejaPiaF2+GJBp5Q30yYraozKRYNkf5oVZrzcwyR56d9t7fv
/+3tLRzwN1z/mIms5C1tkFg83/17MenfkJnCImOv2oKHAPL7Jqxl9Y7EKANc
F8rpP5fIXEbZuq7xWFfSi4nk5O3pjQmcLrin7x4182dhr8sqjMnXZi8dHCI0
h8/E+EdsHIskf5FhoJv3RDoSgAKjJNRvKidd39hp9mJzxEPrW00ZqfpgH+2p
zxo9dOTCJdeTv7h4o4ypB63AcQhK6Hrg1P1Si5iPXwZwraiaTI0aunigH+BK
9CzguagNV3lYjqsiaL/gFFH2qQamSHamranPQQwNlrlvGXHmkFOkPvZRNecO
StslxGk4usgWBnKOE3312EhGmrW+xw7ffJwZPGSB5tkjzk367ezEvsJvxO4Q
LniutWjVkNbC23g6jSV4lUlhQ6Ulml61lc11Kcy74mprXNgMlCPQZusLyS4t
yrTIPDzLFd/Pz95bIqETQ3quLw79R+nVo71pNwy13U6lhizzUa2ppcd2+7Yu
KqDT++M6bC/pCwogMvbbGnd04A4TA8t8/6/aE6LMpXVOgC2QUmtgDlODMyFS
6/9VC/aSqwyXBmPW9CY3LUYPg1J9N//M1qaYIDWcJVgD6hnmd4xaCHnj9Re+
6YYk4zOficyhOSu8QjwfBOZCXCnz+L9qwn3acrO00IdDGtLTOBz1vc7MTFPQ
yT6JKsRRcH45XuGxpo/Ot8bredILmwuLBUrk9OeVMVf67xebW3qsIxPk0neP
WCtbYB9MXTrYgX2XK5G1Lh0iKCOcbKh+ogn4jGdnG+E2wVQjHTJrbui05eTC
PBiAPHKtJtMhyyUdu1nUsRgMnhN0UpmZbqxptY/M0HMgv62D9tv3LdcuDZz6
tIoTNvFePh7PlV0p8+84DMIcP/d1KzETGp0dg1ZslmrK2K60Flc5EyI02Hk2
MnHnV0cumXdlVwpS67mi3DsXPABmcUKKMImrh7vTYvpqmo2DGrOLlHDfEnlh
nBvuMG+5BtnEaGVrJR0i8QJfTIHeBb5IlAVmdZInVVrvTIw2A31PbKolhpqk
Q+QrsUSubTE7qEggDM5+XJi+qNLZrRGtWC0xF1Li9MHeHUfeLyBtS2AV7coS
SpnKVRGfMQ25++0sgdeQZ2mWeA2TJtgMWZmOYJ8xqHWubaXo12MLsmEXz/Wn
nkkAsJCNcEV77/Z056CecwWGRD7Dpe+GjzI+A1bz0/XRi8Qxu3mlPMFLuUKR
6CDwHDos2WlGOFNntrVBDL6QKQpMeZmHTsWuYe+YH2WobbYIXInmHuI4hJ+F
P9K8/7e5HjxWFyuBYlZPxwPWDv1Hyfn9AvlBqeX8jGWAVDj28KGwq/vevQ2d
KvehFi9o9R51+fUMulmQEYYbMYkPDaffNHMahBCpbR+15+dbI89PBezpgTqA
B8D9KM8TtUbck49lyUhYNzBh4oNGi+6re1RW9sSlsMBLa62J1wx4ImNKJAs5
BQFLhWjiGooUbX/5BvOTu2JDZzlLgnvlfRpraYk6KXz0/P4+jMojB7leroP5
pgj0Cd+J7Exrk5LHxpcTe2IAPpUj8MQYsEdUnC6SN1urh6+ol4cG4AJF+fZf
5p76tCH6CpGwoPk5kaJvzFzdAdg0Xfy2F9boN7+/S4UH82sAllffwFdVYJm/
kXH0g6qQ4y1ZN1UNnXauAWCONw3AArByogUhsagBNomaWtCZUIA4BYADaIqg
k/2ZGeZ6kWYBndnUY8vL0oT7tMAQxpLx3RYY4xvkOnOl4GYJ3IuvuxKjLVCS
PSMUqoRu6GXwEx8+6beD6cyXuaci6syFr6sc+8kPeYdotBQavTlT02YfnrFw
Ql37sP1esTX68iBh/6/yiFPwuJDu8lzTtTLsl3JfAbuaEiLVJbXONqUVwK+a
JOwvrRW4GRQmfV0Ik2Ioy2VPoLgoywf6bquNCxkpqrRDP45OGMenjR39VixJ
SuzE+f2UxBRqOthGP+mhH9dEURXCL3Io4n1mkuJPfUS/1Rn0+BPbnJ1tCD2t
PPKPclkqj+vHR378ShLKv0kx8TPEEtXB5e8cE/KufZPW6g7HzWS8rfPI+4Vw
veTH4SmZr85ZLVDp8IOCvImpQUF6xtQY0ZJDlZPrCPNTc1LS5wS6YuXBUndd
yL0pSGPgkY6MxKH6Dlg19JSxGRahv20N9x+Dj4KNJ3bTk7GSLnzSpQsFExQb
ceA98qcSo3RljUTXYDnpJd1qGyMYGvx3dWKFmS6MFnXhI5JkPDSOJ63iEgrU
PBmLBxGx8JjqVpKppskq5YnahoirEXVxCmbDN+/lemTKnpRjPcxrzp4i0ka4
nkgbk0KeCP5TabUjMdp0bl8f5Isy/zKsf8xNBwq+E599zNXxgruTPSo+miXE
KKsxJ13Xn/+mH4YyrFxoS3yFxiCtoVwHujE3lFb2lOXk67fbstJMIacnznzZ
tu/dfJYqcqlBxaIFS6Is+kh54GcSsARAEb8dndFXZziBQ2A/DGgtFRXa1Gvq
iweHDv+tjOO8ZHg85RnJjJDY/DB5grzHM2+ZGzpsnM2xrGXQkZPnCDkzAaiH
He3hjApaUcak5pE4GENXj41JpA4+riUtfbbcbEuY/ySWG0eblmJerbhhzi5J
RI+EUHVJkRbeCnsiacXiSnNCpJ5nBbHbLq24sDO+SBYDA/gBmZnBXX71aH/a
DVNtj8DyUOmMLUrb7VxNlF+79yeNHOV5VpyvxNMeOySWrBGhdWdHfISekkkq
YkLhbC1jEVLYmXrO93CFpFgKzJWvEIOtVM337m2iFwZPEXFNZHwo1Vbyi+LM
5/b2wC7EyySmwRyt+ISoFTmCSBSQUO/2nDRVU48dmkxLYTfCpOwcS9BJgQIC
q8SlFRcmr4tkkblPJO7HVoH70dxlG9aYxnWGzn5LRYPjZqL20kFgfQG0NKvf
E7NXz7rhfYSL//F+GcQ0O9va2C3gPRXvKOfjfxAljQX5mC7k/pSL+lHX6eB+
EISppc+ef9cccbb/+L8qSBe+KPDrFI94rOOT82nCc3uakqLVpbUOKECowbFJ
Y1sv52er/b9qhC6Xk0B+xJnfwIMPQtT+tVuAoivftaTFD9e02iTiRyvxj2xR
F1UAIKg6VroUK6zh83Nl6eXUfSy2Eh+hLa2zw1cRWB+jVpJJxvo4+s8Klyyt
jFAs5lpwNWlBvAF6h9hHodOFlc7OYRuvyeFsjxuJusDjbQf/fA+umlymaQGf
maMJ95Dyrg71JcZH06DA+IBskLOT7Qw+NQLU2flGOkcKHpnhduP8q7dKWj18
Rh62AXaTsbG9Pfri6N08NWWOmJ3O6jusqQmGiwepXghjJgjhKP+cG7Zd49jG
K2OCTzGeh9LF81gC8zg7wwL4B7bB+BZgflXEYpfmi6Mz8GVb4AlV+g1LbZt9
RGBNL2sdsuXfMUdd1PjvrN/7qyyPQOpz7shC2dnnknCiOCAmRRtKay2M+cas
c/g6RWWOuHBTwK5uYjwy/SeFtehRD107ebCVyJWviuTKFEN1k5m43sxLbFVa
83NNkRdmCHv/WACdjLUW0Px5GYILMcrnEvmA40O1RYW63nHhKYxjaQKC41fM
J0kmX3Nd9KNc16+N5vwaTtCMCZ4tKDJ1q7hft7x/0lpWNp0cqb1wYODQ38o4
u9MtVvqcp64L554i9zbD/CZzMkyNnQJHnnk2vY6sLFvQyXHoTR4ScoXHiUM1
19J5ym3dRJnDugNEfD5vvh6kgxvTOe7QMEJHv9Za2eZMTbBd/Lbv5OZ6AIMQ
yOIW9wuu2K7gozNCR3XIya6slKGGLuuw1sR4GOaWMWdOLuRt+PhHRMMlHguz
8haJWTF+wKCy4doEnRzKSDe7aBxTJqBlXvpo+OluDAIyTWu+waXMFC94ajN6
nEDm+LonKdpU2kB8bIHX20tsDid0HEBh/+/uQllJoQk6+fPL25OydYvg1rxI
FS/inkptJ9Uy8kKk1j7b7fu2qAvD3h/XcjTlngpDZ8bve5trCBeFY2ddQvhI
SbW9jSGyesbYMUCehUDheO+eoG2k8sOVAkmcA7rfjsa4MFVRlaNVaRudMmmm
SVmV1ztSrqvO722Ezt5Jrk2MxPh+aaOM0SORDGE3h/pqc/JsADRgBc/+MBkz
hPj0MsJGtos3jt2bf61kMiamH4GcOC4+XzTFBGshYO0DxFOhggMtMTRupeiC
TrR/99cCTIaXsbpQmK3WW5JfCyeTEP0fFcHe/fBHGojXypgzGlNjl52zNI5x
0tAzCTwWTJbv45JnnU4+8ZHGzFR1XYd9QCWk7poGiJIRETAMuMRJ2MEzhlwn
vSLXSWzL8GvmTnZzSkbPhJ3znzoHuRIlSgZOJB4kxMDJvpl/rVbI1wrDxLGA
2X1mW9vlw8Npqdbadgqhcq4DZ19EX4ZANOx7l6pVPpP5ba9I8RAyPJ5Pwhh5
3smdffE4ELekgtgXZ7/uhVjxLAs3bAm+qEQ8kZeIx14ZKrgz1jVqE9gfEtfi
/L6+7/5aAqHGl/F+z2jji+7WFDeJ3XgXAyRdxLvg0pVuC/ZR+27vIEWEsfCK
8wes13JpvfhiUXk5c4iIZHGHSBYCp2FIZ62o1KdE6ygq+M9KHhV0Q8eXXTE9
+Ghwid4RaRY3DPWtZgWrI10NTZmbZws5PQkfmEMsszMipEgN/FYqdUrV1zYL
31nS0mPJzTSE+euAMDyW6NI1L0t6Wqa0XHSKcF1xsV45bhKqVHvVlsIiY1zY
DPTNwb8UQZXyFXooSmGmPBfKK9VvpZrqWK21hgWg2wZs+ZkjkX5KTHg3C+lz
PSSszKuS98rZN8Rl+j0rbYk0Flc5AKAsh7la8AuumSEt0Kk8O8vPGi+OO721
NTZUX1Rl76YqO5b3hBRVNDmSY63n9vZyA5BTOR8RZJg47OLaCP4Z0fZDTymz
UwcbuyxjOpOGMSWaBh3Z2eYg7xFoDJj92xlp5wHr8oS0LtyPEMrcA8dg3rf2
ExVCO2voGbJUNjluxI5d3Fd/4L17O4lrGSMl3l8Xz9erHDElKkSwjyory17f
ISAm1UMNOmAghvn2H/ugkhNfsYbATaHy/URvZoaxvtcxyPLr+HxTly3/rj3i
7AiUI4PqNCH2Q2tCeyHUJ0Vyi1hgPXxVkxg+VFJtA9AD7tUzpo4hW0mdIyFi
3H9n8/7f3dnFivUedI42SCuCEwS5PuReGc9RsbnPfqfAHnVh5OTm2r3v5u9g
NrKAiqwe8jV5Nmz7Kze51RoXqimuoe4FkDQtWeqW8iZncsz0uX2dQEU8CGuB
fQeGUKl8mLa42gnklIiLcBFvpWgvH2r95j2BuMhTRbQarJ5srej/kjbA44i0
cLo/O8fcgEXV0qIOsML6nFxb6GkqcYSIQx7nsYaWSqsAi4SRy9ulonlBXqnG
mzEQLrAc7s6fpTESabQn6rGRvS6NjBvyEDXo1CAPLgIV2ztycxxh/hN+OzoO
v1+OWUI7AvQDT45l3LI3KKmnhEA2GCOyQfRl1dndNbt/LsCPVFfCdkFCfQga
ZAeeqf9XnQmRs2UNEttgcc+YtbjCnhBlInufidgD5GKJtCIwjXy+aA48OX4r
Yaa6zshYA0sE1sAFYg0QslGQibwuQouf8RVY4RZdwVY/iDhQUmpMjJy9fHgE
x3vPOxQWhWEcEzRbcN/YPWrx5AR8h40s5/a2m838Mw/bjwEWrHtGCG0OOTOZ
laZv7IDfQ+NfjEc29hCbJfjUxNEPqqBqH7oOsGfhMsQE6++X2cTMPvPEGh2p
CdaL3w5glfAyXmtFMspOxqtyhCfDmCf1TwxkJg83tBtGdWJSf9iel28OJ7o5
EZSpZP/cVEa6qb5XzNbjM0pzXtpIhO8AEU5YgNlNj73pQmtW90sZ+j/cx5OS
oim4SDJAY36UwvXEiiJO+vF/1TwIGdZKM4eFEB4wlpFmqle619gXOq9dGvf+
pP7gn6kAhksjzZxGwipXeM8NV7p9e31CSF9JpalzwKKZYVX4KltFsyMlVoez
AZEM9dWkJpCeaB+hFDpMqZ5hmK/WlGtDF76uh13AQ04uduvbotyvFioAgcZv
ZUKEwvypQqixl3JIPGtAeiLHFnlBdW5f38PnHH1lIjPDUFbv7BiyjjOjjgCg
1ZmWOnnlSDv8OhxpmB086kNzZpXyL24Sq3UgocRt+3tZiI8yO5u4bf1iuX1j
l+1+sTk1ZirMV50UOXnnrqmunbCWfqthdfR3rZHnBk9sqgamsepSwQxWvMOS
b5tkzDdebBpAe6yDCoZbysvscc7LG53w9mFnP2Cu66S55uQYgbXYUO6mSYnp
mCDVuT11e35Btr4b+/AdcdWfxcB4PZxHXprIFHAb1ea6RuPd3Km01Jnb2bqq
Gj10gJaoeYY+lmJOjtWf398LgwarCf8L68byHPJ5CgRY+FhvCpXzuWKGmfuE
8LGwL3m3TQ+dZ3WHs33QNqglkxMCIeWLrxwZOv5R+fZXUoV4s4QprA/EC3I/
nNWMJHgkjuGFj83SQ5UjxqYuS1mDvaHT0jNEjTbwc/yW54PD/Kc4qRaaFmCw
mduBbK6r3aLGzyd99+8l5/f3ScXwsI8hxsOTZqij2k7nfP68S3KVo2asCoSJ
6OF6Y/uA+ba8tt1Vk0i7+QsRM57mmEFRE1nFu3LSxv2zGePs+JRBOWyGhOK5
eDqEjJINVUL5Oq8ooWmtiWL2Cp+XPOPOi32lwvamDj2WDKKgM+jV08a+MfOD
TuXj0tzMhimTedps1hnN0wMqU3Wz9cZ1V436p2zvhL4JvxJl9GmODG516zlO
OHpjE9QrymaZ+k+b1qSfsrCn4tkanb51CIdQqEA/yCrQBf4zzYzWTLFKmNna
aFdidHdTUpiyrMrQO2I2mGbwPBqrUbeAmCnzaP6PXTM1NdOhtNwvtUVfGTr1
SY14/sUMNevjtWLu+T/3dVPydXV5g6NvwGicnfrfdg0jWS2zmKYGR4xVVCM+
cfFAi1QjzimwNJdfSl6IxMj9Ve7xjyrC/fvybhtbuqwT2lnqpvXoQ+fA+21B
MJr77RnptqvHh+DJwNL9klV8814nvI8aMxSjuK8FmTn9ReX1q4O37zoGhwxW
/STvaeawTkLT3C9xxIWO+u+o5vEGklqpLvBdyeLhGETl238oCNjdmRpvgLcM
zF48Z8wPYuywsS/VTc9UVVniI63Qd1hUXqcmrPxvN4pcQzHChaU69s+iwBOD
t5LsbQ0Gh4l3gFsGYerqMmakOUJOj5zYWCa2YhMcdFlvP7EGb8872We2tl67
rC8uto2rZ+eOmjG43Lr2zcv8fGx8fBaPCA+YhpBTxwYqJohSSB34PKsJAAZA
7LgIa22tRT87zUdvM042NZmT42wBu7oZb99F/8fXfyOixWNU9fZEDISTave9
NXfu2IeGeW/DnzT2JaoJw+17pstHRqmM+CX4SdF8m39PI18qHznOIhy9qIvT
pXVWw6xO6FlnN03WtVtjQ2ZwYBld3jVoxe/ENX+MM0R4kBP4CgXfD9RZ6Ki9
xNEvUfeN30lRAfUYLSdJUhjvMZKVNFpILYYT4jNWmacyacfZSJc49Jq6QnVU
ANWIYxiSFS98V4yWULnHO9mw0LJiVENtanrvIoWcL7kQiWCraoRl7SpWE5h2
v+Pryiy0T9dRLz04PtjLyAtTZdVm93VtMccE606wAh15GYJsXTmlkq8rrKvM
dP2PWdd5x11Yag/1nRLbF1Cpo1ySpRw2q3VkINISG2aurrPqZ3UuSW61JMVY
4KHAEnPrV7NSJsm8jgXHFC7p1eOq/HvOgWHjPGP/YeyYGtdVlppjQy3+Ozux
CAJrjCPVb+dm759LPPz3oitHBlLjHc0NRpt5io/5v62arubZjGRb0MmRYx+W
UUzuaRcL4N05WQEcQQrYsoMcdclYdN+qHpuZZ+zzYvVS6F34QGk3gHdU50ym
tcTUdSGGUFlJES6fz8uvXRrMu0394qAnOVI7TROjo/r79+0xQaN+26tYwDXe
ZS2ucENqxpL+6s2Mb/9YcP6b/pQ4U3WLXTMfUj9Au+hmO/osdwutkecHT2wk
bsdWkR8j6rJV81QzirH1s7sbkq6p4Pv0DpkNhmneefL/2MchKkCv6iZ7avz4
hW+auV/M+wzxLXx3Pi157IPKML/+3HxTc49tYnL2/zxYS7ok3mae9LJYdIrl
sAz6R02VzdQ97Pz+bubZ3JR5NhEKZmWs9JwNe/kBXk94ajgry9agdIxqjAov
aju5ym6ZwtNh0eDxMD/GJw1QxnCDhIrFPxUICQheo/ArwY5x89IA+gE7G5iX
ZuwesswaZqwWMo0w8of6TIMqEww5iBUMRpiHrf02yEp4gBbuKm/JIHdX35kz
v/kr/LQUU1d4wbBQrBqdNPWPW4YnyKPTMQMXDk+pVEO4mUxQHkX0EuVh9SZZ
zxWKIr5O9VYh3oNZySMN7XqK+EEUjHrNjBGu2YMsUJcPUdtFqePhKTLIYZYP
6sjPSo41ntun/Ib1CvOMS/yc+4U/WIFX12wanrGMslg4JtjSZ4c/1txrU6oE
V2VEVuQn2fESL1XxS5cPIaut+EsxsDg+bLaomMIteIRq1jyqM3eNWuu6HfP5
EK5Z5t81VbU52/oph8RL7FpV1DEy3F8DRbCbUQ7d4k+SR/g4P+PQJqyoqSM2
VFdUae2esLOwHlGf69psd4sc0BH3bk/X1umVKotCqr8rrWf1dyyKvO0lWf3d
L+Z4hJ8JUbXq4FNj2dl2HlWjsU4bOwatNe2OOwXmh+5mbOhkTp6posnRNSQk
onjXvOTr2nN727ChIhMwSjZP5uXLK9hg6BJBPcNY2yVUsEFEW3psBeXOGwn6
yAualGjtvQJjQ6cNIiZVyeXkmkPP9EPnsiq5BFeNwVsiw9Tl6UvEQ78dzfHh
49RWapJadRGXRmUBjOTfMSdETj30hF4+3B8fps7NNwI2BqessrCzI+Lc6ImN
NcxFSHNVNb/NIzlSldlascpsS01cYH9hmam9jyiU8OK6x6xV7c6MNFPUJe3V
Y6qIsxO3UgyldWI92zTVKZfWUj3bub2dUhdXVz3bmzy+PQfvdjDoZqHdoewc
a32nEF8f1llahigmmhg1Hnh84EEn1xW/8/m8NvhkV2LUdCGTRx7V7Zu0VjQ4
UuJsMAe9P22Uert6RnXf8DhhxGO8iUEFHlWmJwzWtbpVhd2+Z4kJmlEsCvaZ
UKyJuqTLyjI39otBX51QgBbmR0dJKECTBR3e4D7eHFzk1VsBu3sSokwl1Q7i
O9IMHqVedA225Bh96OneM9vqHr4OB/5w78y2VowtPVlf02TiVWDN3ebcTEP4
2WlO95MyVZsY3e/NefFFqN3SzVO7FT2r8Arzn14UFjCtWBMXOls0p1QsKdpI
btZ8pWI/m2PncF9xzzvEwAw5rc28oae8nCuSC2hM04f6Tfrt7Dj4l8IHrIAr
xg8Jh5z77eyMuTx4/zbPbLLaJcpJNdtSYmloB/9cxDvICAjE1uDxTQ+qq0q3
Nijnqavy/6pLsQjuneJxYGHwKXVmuq3RvZQr98GlXB5Rfda/kHzcXd1x4bMl
tdZeIaK7WKm2lhRrEkKGzjKW/c430h+a7eEe8+G/lwcebsxIUde2SzlwS1M/
jhX8yCEMSkj0S1zu10Rsel08oFFz6596Juw48u71Tz1H/lGB3QZawmeCqU2V
V5G6Eo/Kq0ZnYhTWrHWeyqs33DM+X758A3Y2K1MdyMpkGR9X/t+WkzkR6t1K
bQh/ngUTcR5d55kJpAZ72xpiQ8YKKx2tfTbKLjLIKid0mDq/v32/QAuPEbnm
LCP3iCwTyJGRVytlpBtqW4nlN2+1Es8EMt8/B3Z+uL8yN98gr5LCFFgR9QBn
tLlXSbHzsXaz2EMPC7Xvt3f8d9TGhw4XV9raGaedeCODZJokRU+c3dNCZb9v
ZW55JmEh+WLi37xFGc0rRwfSb5kbGcENtpZKJ/T6BbYDikT+jczWeWWjvJp3
/XU85vi/qL6IKjoqnV0aO2PerO7T2ipa3OqLuDHBGw1/+6c7V4+0JEXrWChZ
YOtQjZPUSeeHapxY7ngdxyoeqaGIwUfVwSe6M5MH6zstI5MCFwkmTW6eJdR3
jNc/YNrUZe+hfAOe8OcFL/67uq4Hzd4vNHarRDYW9WBqcCTHmuDgUBTphTmZ
KhnfgPHEvv4lZag8aohWN/e61xCxUibOzyUs/AW8sfpQX01WmqG+Q6w7auqy
Zqc/sHbpVZcfKqD41ueSyEqkpNxkYcF0j8oksNu01pJyU2LULFXc/LWEzwGP
Wgh3hdWuEufg6AdVV4+Npt+01HW4OIMsjekM89fCgaEyXBnBe5PAiXDF2z/m
ZUGv3OAr7VkWND5PWZCMwEJbBJiHuHjUJPV51iSluFD+1Xk5K29mMPakmhR5
r41H4ymjpLRnZ1pDfSeIGP7efU6e3PxgzsocPtS6aMKAX+Sc2dZCFXhVzs4h
oVM8aaIWZ3LMzNmvOzB7TsOVWKmvcC6UDAvxkKPUsWZAXuMz8uAaH1pdV5VR
+tEPKnmVEcyroXGToAdkVUaMHCKrMnp5o6sGUKDoUuL/Bmtz2xofMQVk79aQ
0Q9k7x5hDLZo/dk9nVADRNEV6VALZ9lh6BgENjrwBLSoqb7LAc0/yXkWSiKi
RpwdOLGxSmjIyjnQXqJMveUiJzC09NteFx86p36nfv76Han+eG4BUavSOjZp
VE+5FxD93q2AiPGg3KpamdNw9J/loaeV2dkGLLlU/QOJyrtthD44/lG5nA2+
II6dzDShwtl3809vaaVWi1W2Lt6YC87CuLW8Ge6q+fy+HiAz1KxElpa6g4o8
TuZiUbUO5Opoe0bScJ1UrTNN3VmpWidgDD7sHqrWSZfxONfLSoayAQNSyRCE
SigZ6nGVDAl6RCwZemGje/US41pXBezqTIgy8GIELTMXgAOl9faUWF3gsdaD
f7lHRWcy02mhbGGp+OP1NOoCd1yVkWas67COUEdPqqeBE56bZQ4PmOI9YaS4
Hq3WC25xPd7pJ1GoYAxUF9zVdo9Z+FMoUVZLVTkXvhmAswgBxSfxYoEtTJVB
CbAeWAe5nvgIfVG5Gf4d++4ypcZK3civmc7tU0qVQVI+4XlJ4/JqGBhXvl+2
B/toMtPM9e22UWEey5qU1pxsY3jAuO+X9Xt/nSWpJB6Lmcc+cuekzzvHq/I5
sk53tRZiBe7vP/Qf1ClWiKAx3fO8Gytdtugf0aJnyipuZsSKGzNV3Liv+uJN
nt0sqEv5LZqpYhEbRfMA/GpHyGkN1f28Ja/7oXnO5aSzhafEICx+LHxxBZVk
8VEsJ2pKsTYhdFzg3Lmv/A9w0udWQ/BKWX6iXqcTFXikPSNxqK7dNjhu5g1M
caLy71jD/UeA1LBaCW2lCOrcaggq0aED3hwXqiuq9qysSY4xndvbc/wj98qa
xZtEbsh6saPrV1TiU1zj7B7j1XPLhH5x1/UBu6hXzJcvu1X4SONwHXAiC1eH
nB7OzrY0DQkHfJTXBKaNhXq3YxAeB/yHKiE8a3HkdZeMVPZlXVzISGGFvaWb
cFutM3b0WeBNpsZNXPimlXfV/kzsqv2MeDbWccIu70oA2Qz27s7KnK3vss9T
RfORexWNvApHXsWTlWlt6FpAFY+XKHXCTESi9t2AXY2JUawGtZdUiFCDWge/
WH12TyPUKNEOxRD0j6xeclX+vpNNfQeOKjPSZJW/kyagWv4dZ+R5DU4QZ0jD
thR252kX/jILRaqbiQudJkkbsQp1N/K6mb941s08uslTYjEc3hslLmyiuMat
fofdrmOAceBZv0Or94x4gniI+nPmrTF7pZWqorvswxqTwFIcchLd0o9qC7AJ
VM8qxjB+cr0iZIYn9+G+xwTPFJRZerR2oUdqn8ZaXg0vxnLhm34W+02W48NT
G93qR92qaOraSXQWUkXjXqkoVfMQEybo5FhWllvX1iUQw5xMS8gZz2oeWsen
52grsWa/Oy5orKhgnDq4sKcw+9iSGG3guAerjrdTId2+sLMr1c6KWM3Kj2C4
pacY61pEbwVOcf5dR3iADggCHAHiyZkpsspZ5iXBjWedUYVqmh71vNU0pVRN
4+FvyTtIyKt6APlxwdqiAldVD3RYSYUpIUqo6hFvm4gS7wmbUz3LOw1Q8Wma
oaHTwua1pKnTmpNhDPWjbnTYdanV0MLXT9ZdlnXFwkBg1+JxcIF4fxlewcL8
aGdKnDVgV9e3f5T6dMgruJ9wofWrNzFcmGOBJ4YYHlC4QXTr7XkZoxFnunn2
3MOtX+I6gTKP+sMqeNRZKUON3VaZR/2fOTmW4FPD2FFOUHS/iYxzukXPCooX
Bwuualw4lZkCUDhrjLPJk66bz37dA08f7jFHBLHX7Q/XbT8ixsw2iPxs2iXG
tx7NzLDWtQlRGYg6E0AYsQPHPqwUUFvsI7BJrJgUfEEBt39315d1RJ0nSBSj
urCvwVVgIwaJFELnvBXhbgEb6qZemxA+XFJjpy6lYsCGurhEjgfsavbov+3R
j8HVSYBXhlKnnVPKrEzPGBIrqB88/lGVK4YksMkXJoeP8rfK+lwI7QO3tMQE
q4uqqf+m1OmoshXuk/78vq7Df3frdMQNQLH3H79DUUQFfDToAfHHiLNkCe39
NdXgfCrV4Cyfo+OgrqkK6FQ/LJAGpYNXAfFazJw8R5jvKCRVVgXEVmC9p477
FeVweEgS1hB2lJEUjb0kiXTBxuVDMC7u7nAPSVLQ6UesI+vCImv1TISLf1Rc
PTpM3iz15LII9TVtI468XFt4gNZfaMqV4ELx9S4mj2BPCnfLyMp0+FPo9iFe
pbP/B6t0pN6ovF4omWqaWLdSZhlK9ULLXPVCe1z1QqLLFbFORKwNxItjCu/A
Hwp8t3cE+6izMmwNvQ5e6UO20gBdPBZ5fsx/JzHZ5R2BNi24W4h7VyAhI/Ht
nwrgP8UE6QruzXaPGHnlzKDOWlZpSro2e/HbQYgbu81JvN9tnTsvitgkrsQs
L+Ux8Uqepi5zTsYDKnmWzpevwIh2v53l/YlnHRE8McilLeS0VvBQ3O9RcvUG
wgZTEdbzSdg8bKGYx7Xyqial2lRcPJsQpuN9HvAxzyjj/OtIDJxF87BYHpNZ
WTi55Ke8SvmtoGO9GQkDlN/SCvkt1qfDEuankhLBrnvj1ok6UnrQVn6T5bY2
qfxHzLvZyhtY9c8+pWf1z7KN81Yd7epKiDQVV1GVHe8JKUaEhaojONDyBVgr
R8xVdIcdcJdH8kJ8xiiSNyhUV4zqTE3d5uyUgdBTSugbamHiwQOev9fSak+r
x+PfYmc1eeI0329bdVwQS5wqhcRp16i1olmWcZbB/hqXB8NvzXxa0mW95DP3
CklcNUvi5t1htUP/4kncdAHr6PvL59GJb2fC2g45rcrO9qxZysmzConpdzyv
8lSslelEoXfp7+DLtCZETpTUOtqHbOoZQSeW1liTIobOflXDG6+69S6dv/PX
gtZT1j32RWLhHP5bWdCJXkq3dMm6x3ZiMRxMg9R4do9dLUqX644UiDHMQd8v
m+GaUBDWvQbpZvIEwT8vBOOls5w7u0KmHTfLm6bv7KSbKWodvZN2qfKJBUCn
zn0t3SB63U07rtkodk1gXBy+0diCUO+27HRtPbu+ieoDtAJzIMx34NgHVQI/
eK3Iwpi/T9186/rgTpYCf+M14m/QuQ3RyfLl1N+4rMaOcwtI+vZPhW79jVdH
rHJpgVWy+0G3tJDtleZoUDql+0Hb1AQj1y6NBeyar55p9Ua5no+SVVFVB3mP
ZGXS6RWrqOjul5xsSqjDwSKFKV5ZyvzH1R4a84nYraz/M/yehKD+4iLW/1kj
ddGzJUTNwsmAKoG9x2+rkrqaz9O58od6AMprSslR2FwfeHI8LVlf0yj2XmVh
B1v4WWLsUpcaqdk206mumyrggrjdPho0w2ujWAZ8Odz60nJjYuTMxW8Hjrxf
Nk9t1GqZDbrQiqwIrjDLYBJwhSndEyFbTaEvOWtLQCVWN2cbWo1sVEt5Xj7r
lh6qzufzJk5Z4mWDrMfL/Biw0tOanwcD5LqByks5zyhEX1DsIDkdF/rBA1RT
4mwBe3p4Jt+zX7DUhxtzwK9Z95/qwBMjGekmVyZ/xtTSa869NRJ2mgocvppb
c/W46GMtsMYrN98cckbFr1BkBG1Bda7y9C5jsI2cpJAQZS6ps4m5/iXU07rK
gR9i5ygL+2YGx5JHHrimq76XcZdla7lUvgoYv2CSA1Cp7wLd5jvCOlpzPkP+
PadACaCbChKljtaEXivnaAUWh26ODdEUVdo7RhgM6qmwtrzemhw5eG53Pbty
1b2Wa530FPE+hwdWjY3BtnUkRmkFqoDsQmjmITCGwnqppTWJ6NtZR94nLyEn
19444ORcAeI+9ZKXEOo7gg3jeTYXvi+sP+UyDy2wTmBqATtjgifhKPZo2NUZ
BkOfxlLR5ORsAhxkFuRNcNMCy0UfabN0STZT90HH2jJuTvJro2RVYpaIs+Tc
iXcQilViUpf1+SvROuy8Em1AxS5+zYctN8B6tUsXvzKPc5lLT/Lk1xMxGAw1
L/uqBdq+tM7Zwy4Zgf9Hof9GZ2qc5sJ+0XEVLxlZ+Dp6dvt/ku4t4ySEjDQZ
CUG4yo/d5bKlBdPibo3gHy1zs+ijuQUKNem7vT32Uk/BvYnOIX6XyxKBgxCr
P7ePssCQfLgGws2iGzbKumIQY/kNqm/DjlJ9W51NilT0jllKWW4GyoRT7OU3
xS710IbYTup98XlzMNmfzuYRsYZNpCJcuzTs+2XVLkaV/Qnd/j16NAtd4v5w
P2BXtxAzG+cxsyUDOmtphZluEzpMVfNULCL5REulFeSVAa4r8yiAl5Y4WdMw
w+NcLUpLXo4xPICq2ui+AKpqixWq2tbN8YmIq/5sIq+cC/bRZGVQwEzkLdBt
U+QTHePZAvd+qTLMZ3ctYZnhyEE7xcGnKDYrJ2y8XyqP3iVG6y8d7DvyfrFH
vuZH3DnBuzK4OMNESSYyw3B6gqquSTeiE+KtS1uH7Xl3zDA84MLg/LjdGSis
o6z+jfgI1KGlg26BLZdugaWoIy+jw5Sorw00A4s6PiL4+/KaXh6zpSyhcCOt
TYhFTwq+kP9XXXNvpHXd2oG9xO/E3Jk6K1MIIKuYgmrqmM2+qQo9PcLpPe7K
LuLhd5+sEJGHW7ECx+kXOWe2NsVe6iosnqHLqWeEG3kqWtjlBl+3Y0LCzQw8
AuDR9Vvo6fpmBrWN5HfF9gupDNW0qbnfkXeHKvUgVgf/Ukx3EjCyGvN7XOeY
y7NIdqia/+7afNndtc8mCKwUfo5ZrQyOFruFJYvd79qeEDFZUmvvUtnFe0KH
LKWVhsRQ5dldTfvo/rkUpt6E1VvYPTwr3bGbX8YA1Ag60Z2ZPlvdYu+lpquE
3fXtLtKDeD+UQBVgMdD57nX6TNYmKJ7aBDlh/PK7ZFlTF4bh3/A6wTRWyyli
+BOyWLK88tD9mlvTPNfcvunWrugxmaXA6T173sk++o+SUJ/u7KyZWtZzQrgY
sd2em2cM8+s79oGsEeqPuR9qpZyjwm6gxGB49FKgRWgc0g2UZbUOTougSMLL
qYCNT8VSUpZX87zTkNWax2NncPThkQjXjWkFegTwPO+29frVYb/t1Zze/Yn8
TkPp3jJuNX7OvT2JKCHlUSdNzYwgGuanPvVZI+vnmeK6BfdRSWM+7rqnE/rX
/6s2uqezZLpr2CrnSiRemz67h/Lpu97K4PdfiTdV/Khb81jI0Z0vYRH5Esua
e8jzhk6gFgU8/y7DIZZfm+eGcnb7LCE6kLGgkN0+y54m8SauHOk79qHU+Vmg
ecy5PS+WXYkr41Bo+XMo2lRSZqUbcb8bhoqB6PJCZbGnLhBf5DAJ+o7FAINP
qTJTJ+qbp1zMhKZe4lGEnpmghjkwhJmOEdgznndT/JhV5esgdHUlqpvbOkBD
VDDazlmRsSZfB1e2iC8Bv1Io8KQ6I13OZVnGbqedCPNVUbf5OZXgz8r01afy
/BbWwVuVmaLBOjCOxVK+BiFnuNbLErQer0R5ZN49+WMB2xNjMfFaXHtSXGZN
iDReOjhEt1+9JqS7xcjKw1dvldx/oTI+xscmutLJkcwMdh4n6DxCsfHqFV4G
xS8l+uTBd4wKSEEuzPa2uLBJNwLVGN1VmxzWe24Pu6uWIYVUdP68u84XTuRv
7/jvbI2/0lVUTCfSdXNuA92cCz2CLXe7OfeRuejA7sgOPjVEhetydBimm0BC
fVXenzYST1O8YXTha8giA79wi2EJRAt4L3ri7I4KuE681qY5RIs1Lu9lo0gG
5N3cd7O7aoO9e7IyZ2ta3O6qzb9rigjoO/5RuftdtULHsHc2ebRN/XtJyKnu
rMwZPEV+bW5evjHcH0+p8CgD3Oh+462rSYbIuFCyG9mh8/o01KYtMWr87O4m
UecJ3JGFr+Aaabfcr7W/erSfsy4GuakyQ6FV+C2R5+ncQHEwU+XaZrH3odRM
QLiWdltz3OWuopIZikZ6XEu7j0Cc9zJyZStfkmlWdkluFqNbtHC6hfySXFJ1
MYZze7tZ00Pxklxe0yu3liBOB/9cBFMw9MxYTi51TODWEhyxFrqdzBp9ud/n
swoeuPpEvFiLWmyveKiluGYO2kCpSDSLQkazYJbt+n7ogGobWccHReuYddiR
ZyWlsAvAPfD4SHoimelwrrh9zRkW4QFafssG1eg9I97T9KKIIXQnLjOOkx/E
rlgHL4XVUmoxDCFJQLHPaMlP43li3nj0wjfUQqO4zMbuZRdt9NIaS0Kk/vKh
rqP/pLtgP/tpt3t7orSLXpE0WdfIvbO15J3lknfG2xKRd/YkeWceOTOYO8w/
1Hv4h2WV5uRow5WjdG+H5B/SqXjBwxtjXAqizxGD2eWNrYXOKm+CHch943se
Hc03icXLLEnGWqmkSwyKZY1tM1k3J0N8BA3j4Q8ubKUel0cCWA85F4vick9h
gZZhMez253gkIDlWH7CrE6oK+88jAZvFi9V4qSG76m0oI91Z3+8UmBM8FnHX
GXFu4vTWVnbzYLLrXllJttbxUUjEiRAfVXauzP9n+gmiFXJaRcSJt1z+v8KL
KFnrtrKbmunSE0otCHURPATRM2opLdMlBPVg8EIIYoMrBMFW4eFrxbI662Rx
JyHNFHS8PTNtilvl8rgTp0xIcSeFF5Xnr5U6Zvt+SYU0dKGMdGssC32lxk1d
+KYDXhp8Gin0RSv1En+/e8E2cSR2tSdEGaVQE2ZMUsXvs93dvF+8z5a/f91m
Bllf/4r4aiE+yqwsQ22Ho0+MdUHv5ObMhJ3uOvqPcs9Y19oFrRKLaax1j3LC
mWiMZVSIDtaxapJ1rKqotyZHDZ7fXY8to7ruDde9sCWKtSyTeY/lMIYyM6yy
u2AtFGGlu2AHz2ypnucu2Jf5u8Vb6YWmg8R7IOYlu6NWIMNTn8p8oNQoQAyH
Gyu0iN79OPXPfve2/476+OC+onKjGEs1dqlsZQ2OpOjJs7vboDE4mgux1HUL
lh7XveIUTaf2c8RtOObJbVjcojTnpo2Enxmkk82yx15UdLOYgyZDEbd7XSUW
Q+DxHuFeV367AhPv1+dGzdltuwJ7oVpiLywm7K0i9sI5sU58Eb13DTWV3Fgb
7D2YmTxS36bHgZSTFELPqLnKgZp03e23MGlZ64mTzyZw3yQ2mHe245n8xXJ+
AruZOsULQKV4VKSWiTe0CuyBTkNOung96689r2d93Z3PwLMzuxjCsltjZ8Rb
Yx8bnrE0dZmzMwxUUM4u8YXEKdibVx/6a8l5V+G4cKEr3PriEmNCuJQVSuJq
eE6//tUP1f3rRHl5UsgSPiGSDU6OZbhnCdm13fYwf81JliADvAJh2OVPnURH
rbJK/USXKVWWsvLJpLD+c3t7PDOTqyLe8GB6cElhrIL4wP6iQikL+CRjFbAq
dBZJw/nhAeDjH7kqzUfcM4+hvuNntrUC14gMIXStnWdVHi4zjN3z+CaxqzNd
18pJAzta4RFKqWktlanbGM9be35/O065wgtQpFjDAKEvM8NYJyXDWTlkXq4u
wrfz+L+qeEZfSoaznnWvbHLd6spoAf+oCDlJWec6j6xzHss6f0Rd6fA2VlrV
mhChKalzwqlT08W1hp4RIdF99usOmLh73slx3R7w01fkFfndcnyQh/9WFnh8
DDaDxIEA9DWyQvbI82Pen9TjAxBTeFfx4VpoIqlMndgWWLjYSboM9b17FOoT
NRHr1PCyO7cBrhsvuBe5Dey+7dI6WNC68/u7+FuAIGF+w/Li9CGdpXnAkZtv
jrrQ571ZlluZ/46Lha3FBhHtJHYN3WTM8hKxIUKXBSbD6xlqOlIT7Ze+G8Lp
hlEd5jeRleGo77BLdertahwtaHI1R2KcQBfCvSXzhwXuzCs3sB5C9l3GnaFy
9h57bo4j6qIuYHcP/FAYvknXLSXVbjyd0npbUvTs1WMdh/9234Ons2Hhc1++
Sc7UesbFvuVMLXaPGaBKsail13rnjj0mFNgQfRXQmRzjKC61dY9Z+e85J8wo
Ya5QOjj3htGfiRi+0sXG+nfqqhEXNltYZOxRWzg4EhO8ynYzySG8LzrQkp9r
b+y0AbV5XSb17gj1BWo37f1Vjgfna2ErwCKwL2yWM/+eT/Jo0A1PfRye+rS1
pssJjy0lznEjyXGn0Nk8LJScS6Xp8Ib8JHbhvLfZSLXnL2xaJTD6drx+C0ZF
sA9VsQMQRlmOHALfOWYrKnfcuuHAC7OynJVtzj6dbZwaTJt4ZXoiq0ynO7U5
a1BkODyxoJkzHu7zInM0WmSOlnHmaB0xR6mrBmbYrXVUtTsLK52FFU6qd9AJ
3QuFOvS42bns1DlnkWkKqV+L522hceEmThDliWhAT8uIAz/BOyuanZ0TVJRI
zSkY7OZmToSxKnRP/umTC5r3UzxaulLIvrIrx+9y27qw0tHWb1OxZjsjOvMi
nDXFSpwzmNozxllsTOsQKzi/2O+92Z3aPM+MWWeDVfL7ySRWcdDJ8cxMQ12H
o19tnjbOjs9SnsCLva1HYx/WEdZpZo2dQ1bm6/FS87s8dSS87amH2gPiPBlP
fcP1Ha/dZOXnFVePDmSkm5uGnIAavNxoml6k088qVk4bZk3mafyEmu+22lPj
9cEnCVg86O+sA6Ri5SZ3xjnHS2KcQ21pbRAOPHfWOKPgzzaYprGAFDFUO1lq
fOzEplroGxYydl1HtsAdlGoYlgtF4+zOXDjEsSGzsKb6J2j97NZJ6h3lRW0W
if5iMetglnf0m/PzjBHnp3y3N+/7TY5HcQSf24pN81cjaLLTDc1dBM9Yp0fo
ucuc1kksGISzvMrE6hCph7WrrnsTvx1sgWdR3rVJqPdmlJfqwBNj2RmW1h6r
wTRDr2WMk+X/j20CMlJdb72RQJ1CD7FApTw1y/taLpdHJni3zrN7ehOCx8oK
1XCR/9OqFXrSYYFgB+XnWcL8td4iK0d2b/H8M1nyvauD3nyMLY/7xaTybGp2
s701OZbu9dDO8h6HiqXDU5biKue1y0KEwu3uxN9IOWRRT/NwKBYo7GTrvXwV
ztB/2SaE2UCaG5WOWykGYBNrfpj6qdtdWPPPZin96dZJz5Mt6flvXv/jdspZ
6ziYe9GX+0rrjOMzrA8bq/7PvzMVfLLz4J8LOU2JD+e3kp/HI8Vc4r79U8Hl
71pysjVdGru0R0twmCpbzfHh8PNqsIJSlPeR/8EZPSPlZYVAL1Un4HxdPT52
t9Cs1gp79ViPypp2Q8+7zWwRmEs80il1AWQ+GmOc7PvN7fP7mtJvatpG7P/L
rmGzWTw1PQvzKvTMOJQHLytjcZlFPNq8Uahu+alz8uRiPiXOi9x9zoBmWYRL
B4fy75pH1eK8OgasN5NgdbVwRpaADDSod6V58dIfCBXw3H9nzY1EVZPS/t82
Pq+lE9rZ+6XWq8dU8F3Itd5ATYnlOcp5GI8/fp/Sf+peu33iNY9PvPqQf79C
fzwq/VuYiJeEBV7CoktY4PY2z5vIPdHCk0v3P/37/7/f9z/y+wesNumQLGG1
FYoa2Z+Kf/v/AOBQ6bM=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 176},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->256349595]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Wolfram's Nonlinear Wave Equation",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->537312911],

Cell[TextData[{
 "Wolfram's nonlinear wave equation [",
 ButtonBox["more info",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/page-923a-text"], None}],
 "]:"
}], "ExampleText",
 CellID->6561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["\[PartialD]", 
         RowBox[{"t", ",", "t"}]], 
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}]}], "\[Equal]", 
       RowBox[{
        RowBox[{
         SubscriptBox["\[PartialD]", 
          RowBox[{"x", ",", "x"}]], 
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x"}], "]"}]}], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"1", "-", 
           SuperscriptBox[
            RowBox[{"u", "[", 
             RowBox[{"t", ",", "x"}], "]"}], "2"]}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{"1", "+", 
           RowBox[{"2", " ", 
            RowBox[{"u", "[", 
             RowBox[{"t", ",", "x"}], "]"}]}]}], ")"}]}]}]}], ",", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"-", 
         SuperscriptBox["x", "2"]}]]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["u", 
         TagBox[
          RowBox[{"(", 
           RowBox[{"1", ",", "0"}], ")"}],
          Derivative],
         MultilineFunction->None], "[", 
        RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", "0"}], ",", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", 
         RowBox[{"-", "10"}]}], "]"}], "\[Equal]", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "10"}], "]"}]}]}], "}"}], ",", "u", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2651],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "10.`"}], ",", "10.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{402, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->503219777]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"t", ",", "x"}], "]"}], "/.", "\[InvisibleSpace]", "s"}], "]"}],
    ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32677],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->105104449]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DensityPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{
       RowBox[{"-", "t"}], ",", "x"}], "]"}], "/.", "\[InvisibleSpace]", 
     "s"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", 
     RowBox[{"-", "10"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->22804650],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 176},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->174509184]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->246788166],

Cell[TextData[{
 "This solves a ",
 Cell[BoxData[
  RowBox[{"2", "+", "1"}]], "InlineFormula"],
 " dimensional version of the equation:"
}], "ExampleText",
 CellID->43447455],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "t", ",", "t"}], 
       "]"}], "==", 
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "x", ",", "x"}], 
        "]"}], "+", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "y", ",", "y"}], 
         "]"}], "/", "2"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x", ",", "y"}], "]"}], "^", "2"}]}], ")"}], 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          RowBox[{"2", 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x", ",", "y"}], "]"}]}]}], ")"}]}]}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "x", ",", "y"}], "]"}], "\[Equal]", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"-", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"]}], ")"}]}]]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", 
        RowBox[{"-", "5"}], ",", "y"}], "]"}], "\[Equal]", 
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "5", ",", "y"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x", ",", 
        RowBox[{"-", "5"}]}], "]"}], "\[Equal]", 
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x", ",", "5"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["u", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"1", ",", "0", ",", "0"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"0", ",", "x", ",", "y"}], "]"}], "\[Equal]", "0"}]}], "}"}], 
   ",", "u", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6849],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "4.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "5.`"}], ",", "5.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "5.`"}], ",", "5.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{453, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->111557589]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"Evaluate", "[", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "x", ",", "y"}], "]"}], "/.", "\[InvisibleSpace]", 
       "%"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
     RowBox[{"PlotRange", "->", "All"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "1", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13912],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 309},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->548876126]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   StyleBox["n",
    FontSlant->"Italic"],
   "-Body Problems",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->5714],

Cell[TextData[{
 "Reduced 3-body problem ",
 ButtonBox["[more info]",
  BaseStyle->"NKSLink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/page-973a-text"], None}],
 ":"
}], "ExampleText",
 CellID->5371],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"z", "''"}], "[", "t", "]"}], "\[Equal]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"z", "[", "t", "]"}], 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox[
          RowBox[{"z", "[", "t", "]"}], "2"], "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            FractionBox["1", "2"], " ", 
            RowBox[{"(", 
             RowBox[{"1", "+", 
              RowBox[{"0.1", " ", 
               RowBox[{"Sin", "[", 
                RowBox[{"2", " ", "Pi", " ", "t"}], "]"}]}]}], ")"}]}], ")"}],
           "2"]}], ")"}], 
       RowBox[{"3", "/", "2"}]]]}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"eqn", ",", 
     RowBox[{
      RowBox[{"z", "[", "0", "]"}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"z", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", "z", 
   ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "30"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1620],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"z", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "30.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->363241442]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"z", "[", "t", "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "30"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->13156],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->419229927]
}, Open  ]],

Cell["\<\
A formulation suitable for a number of different initial conditions:\
\>", "ExampleText",
 CellID->30149],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"z", "[", 
       RowBox[{"t", ",", "s"}], "]"}], ",", "t", ",", "t"}], "]"}], 
    "\[Equal]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"z", "[", 
       RowBox[{"t", ",", "s"}], "]"}], 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox[
          RowBox[{"z", "[", 
           RowBox[{"t", ",", "s"}], "]"}], "2"], "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            FractionBox["1", "2"], " ", 
            RowBox[{"(", 
             RowBox[{"1", "+", 
              RowBox[{"0.1", " ", 
               RowBox[{"Sin", "[", 
                RowBox[{"2", " ", "Pi", " ", "t"}], "]"}]}]}], ")"}]}], ")"}],
           "2"]}], ")"}], 
       RowBox[{"3", "/", "2"}]]]}]}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->228004849],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"eqn", ",", 
      RowBox[{
       RowBox[{"z", "[", 
        RowBox[{"0", ",", "s"}], "]"}], "\[Equal]", "s"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"Derivative", "[", 
          RowBox[{"1", ",", "0"}], "]"}], "[", "z", "]"}], "[", 
        RowBox[{"0", ",", "s"}], "]"}], "==", "0"}]}], "}"}], ",", "z", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"s", ",", "1", ",", "2"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->4469050],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"z", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"1.`", ",", "2.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{380, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->327664038]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"z", "[", 
      RowBox[{"tp", ",", " ", "z0"}], "]"}], "/.", "sol"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"tp", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"z0", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->538369189],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 144},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->279324866]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Nonlinear Schr\[ODoubleDot]dinger Equation",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->467881331],

Cell["A soliton profile perturbed by a periodic potential:", "ExampleText",
 CellID->237473526],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"L", "=", "50"}], ";"}], "\n", 
 RowBox[{"sol", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"I", " ", 
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}]}], "+", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}], "+", 
        RowBox[{"2", " ", 
         RowBox[{
          RowBox[{"Abs", "[", 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], "]"}], "^", "2"}], " ", 
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x"}], "]"}]}]}], "\[Equal]", 
       RowBox[{"0.1", " ", 
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{"Cos", "[", 
           RowBox[{"Pi", " ", 
            RowBox[{"x", "/", "L"}]}], "]"}]}], ")"}]}]}], ",", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"Sech", "[", "x", "]"}], " ", 
        RowBox[{"Exp", "[", 
         RowBox[{"I", " ", "x"}], "]"}]}]}], ",", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", 
         RowBox[{"-", "L"}]}], "]"}], "\[Equal]", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "L"}], "]"}]}]}], "}"}], ",", "u", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "L"}], ",", "L"}], "}"}]}], "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->244609007],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "10.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "50.`"}], ",", "50.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{402, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->542382761]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"Abs", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x"}], "]"}], "/.", 
      RowBox[{"First", "[", "sol", "]"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "L"}], ",", "L"}], "}"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "60"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "3"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "None"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->270329133],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 143},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->267816204]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26392],

Cell["Symbolic versus numerical differential equation solving:", "ExampleText",
 CellID->13531],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "+", 
       RowBox[{"y", "[", "x", "]"}]}], "==", 
      RowBox[{"Cos", "[", "x", "]"}]}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "==", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", "y", 
   ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10204],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       FractionBox["1", "4"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "2"}], " ", 
          RowBox[{"Cos", "[", "x", "]"}]}], "+", 
         RowBox[{"2", " ", 
          SuperscriptBox[
           RowBox[{"Cos", "[", "x", "]"}], "3"]}], "+", 
         RowBox[{"2", " ", "x", " ", 
          RowBox[{"Sin", "[", "x", "]"}]}], "+", 
         RowBox[{
          RowBox[{"Sin", "[", "x", "]"}], " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"2", " ", "x"}], "]"}]}]}], ")"}]}]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{500, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->275648464]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "+", 
       RowBox[{"y", "[", "x", "]"}]}], "==", 
      RowBox[{"Cos", "[", "x", "]"}]}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "==", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", "y", 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->27294],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->59230442]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->19149],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->534324774]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19634],

Cell[TextData[{
 "The defining equation for ",
 Cell[BoxData[
  ButtonBox["JacobiSN",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/JacobiSN"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->25455],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}], "^", "2"}], "==", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "-", 
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], "^", "2"}]}], ")"}], 
       RowBox[{"(", 
        RowBox[{"1", "-", " ", 
         RowBox[{
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], "^", "2"}], "/", "2"}]}], ")"}]}]}], 
     ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "==", "0"}]}], "}"}], ",", "y", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1.5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7544],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.5`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.5`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], "}"}]}], "}"}]], "Output",
 ImageSize->{316, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->946039128]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1.5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->30708],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 111},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->149718147]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"JacobiSN", "[", 
    RowBox[{"x", ",", 
     RowBox[{"1", "/", "2"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1.5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->78234261],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJw1x20wFHgAx/GVPJzOpWaZjRgu9ohopkxJ/Ji6jK6Ey9OuPEZ2T8uYLnUb
ytOpliubpq7xsGekhh6upkzcukOsxoW93R7ExorV7rKtFnuth71/L+7Fd77z
cUnJikxbRaFQwkif/+93nOrwrKTA/1969Bv3AXogJBdTrD//uWaCXU4PQ1mo
1Mgj1z7cw8ukJ2Btk/kYm7zQTrcxns7BU8c1mVHkfaL9k9/T89GcJHXxI8+q
O/nsEL0CdkdWCSfcKtDB47UdpNcgz76pWuBaAzPzHAsl+x68abbS2NR7CC1I
LnkU3QLhrryGyP0taDmdJmsQt2NLWcOk16N2DDF+8MnJ7UL3SOvCAKML60LK
0zuCRGCUqs7b2ohgI017sIPZh7zE9S92f9mHTc6Zt0tSBsAV/XPmibofO2XH
+4IrxajmRnn6QowOw9kD0hsSRLGE5xbdJcj4s7e3vfAF2Ba8WoVaiuupjfa2
Sa8wNHa0sqL2JbbEyhdW0obg+YeF4Wr2a7zVHeoPCh3GE51Xn9z+DT6Obthq
FyXDlWFc2DY2jEtNPx2LiRnFM+1EvpYvA32t5cFTN8YQ4lf4uNllFI+zzzZV
1MpBbV26dDN1DFI3vkVE5TiEjZYBDzfJ4VokcT9y/R1y24ILSjRy+IWPbq+9
MYENTBft1vvjkGWXBJbzJqH0DryjSHyHB35x1t2XFWDTntUoaROwixYN3SqZ
QmrWt/2cvyaAccrlzhPvYdTRTNsSJ3HdLsOafkqJnZ1LJ5fNFSisYf3sy1bh
2OCg0+tqBar7r/VUZKvRmne3W+A/hTwOX+uYM434SWrC08EpqJPdaXEpMwg/
n9HVGv0eF5dVq9+ma5C9wPU4Pv4erdMhH75ifIBlwMagukQlXrrtHN/lqkXL
23Bp8QslMjxSffgNWtiokxnrw1XoiGZyg51mYapp5mcKVdgWq7pNFcyCoRQL
r3qrkVm071WAw0dEUk507uWrUZZuaTDhf4SEqzzLnFVD6JhsVFF1uOBenx4R
OY0nK6Wmmyt0WD3K3k27NY3+h3tU5evmUOdk+WHROI1u5/QeXuUcrCkan10R
MyhuiRLIrebRO5XrfuXaDOK/Hklgls8DAoOjj3wGi499HDwtFjDA8Zddctbg
jKny76r8BTjJ38TXMzSwc/VLdqPo0St6HuxRpUEuJ3rO6rQejuLs4VqRBrM8
K7btkh62hr062ScNNhYHU92IEzi8u++IQ7in2rcTp3l2sZTE1SzF+sPEsoKV
0Tni0H2dbZXEO/jNfWsMGgiMp61tlvUYYClq/YgjcpT3rVb0MPlC7F9FfIbl
zLQnbq43mf+VuDEpxmwz8fLPLnfriJfDuuNCidu2+Dg3E9/2FKwqIz6mOWfa
RWwyGXvYzKjHyH4zkZbYa+SXFSrxuaLNBQvEMZKeW67Ee82xY4n4Tsf25T3E
ASrfRotFDeJqbG4WEnd5MUudiIurQsIriWkthgBX4nu8fIOAWHTx/LwHsRl3
OqyDOJD641Ff4q05mz4NEjv0ih38iZksRv0YMbvHURJEXJp0+aCW2Htt3IV9
xL/H9OqNxPm/FQUfIP4POyiJ+w==
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0, 1.5}, {0., 0.9681760100912269}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->111514838]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5202541],

Cell["\<\
Numerically compute values of an integral at different points in an interval:\
\
\>", "ExampleText",
 CellID->455313870],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"iv", " ", "=", " ", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"x", ",", " ", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"Sin", "[", 
         RowBox[{"Exp", "[", "s", "]"}], "]"}], ",", " ", 
        RowBox[{"{", 
         RowBox[{"s", ",", " ", "0", ",", " ", "x"}], "}"}]}], "]"}]}], "}"}],
     ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "3", ",", ".25"}], "}"}]}], "]"}]}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->140835562],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0.`", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.25`", ",", "0.2259882299920528`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.5`", ",", "0.4730479976194906`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.75`", ",", "0.709540439767007`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1.`", ",", "0.8749571987803856`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1.25`", ",", "0.887999017257458`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1.5`", ",", "0.7120350249749674`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1.75`", ",", "0.4950837858565847`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2.`", ",", "0.5509351737392814`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2.25`", ",", "0.728469448976715`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2.5`", ",", "0.5519902414453262`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2.75`", ",", "0.6877455856287152`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"3.`", ",", "0.6061244734187703`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{518, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->150891179]
}, Open  ]],

Cell[TextData[{
 "For functions of the independent variable, ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " effectively gives an indefinite integral:"
}], "ExampleText",
 CellID->788121558],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}], " ", "\[Equal]", " ", 
       RowBox[{"Sin", "[", 
        RowBox[{"Exp", "[", "x", "]"}], "]"}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}], 
    ",", "  ", "y", ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", " ", "0", ",", " ", "3"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->200296671],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "3.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{309, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->232299251]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], " ", "/.", " ", "sol"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}], ",", " ", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{"Point", ",", " ", "iv"}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->137449308],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->622249486]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->176340561],

Cell["\<\
Finding an event is related to finding a root of a function of the solution:\
\>", "ExampleText",
 CellID->233343035],

Cell[BoxData[
 RowBox[{
  RowBox[{"ev", "[", 
   RowBox[{"t_", "?", "NumberQ"}], "]"}], " ", ":=", " ", 
  RowBox[{"First", "[", 
   RowBox[{
    RowBox[{"y", "[", "t", "]"}], " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
          RowBox[{"y", "[", "x", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", 
        " ", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
        
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
       "}"}], ",", "y", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", " ", "t"}], "}"}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->39932360],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"ev", "[", "t", "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->363650466],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->423019911]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{"ev", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->561096089],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"t", "\[Rule]", "1.570796372248926`"}], "}"}]], "Output",
 ImageSize->{83, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->47187379]
}, Open  ]],

Cell["Event location finds the root accurately and efficiently:", \
"ExampleText",
 CellID->508676218],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
       RowBox[{"y", "[", "x", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", " ", 
     
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
    "}"}], ",", "y", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "\[Infinity]"}], "}"}], ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<EventLocator\>\"", ",", " ", 
      RowBox[{"\"\<Event\>\"", "\[Rule]", 
       RowBox[{"y", "[", "x", "]"}]}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->194334875],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "1.5707963800400648`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{337, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->201919792]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->155559896],

Cell[TextData[{
 "This gives ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", 
    RowBox[{"(", "1", ")"}]}], TraditionalForm]], "InlineMath"],
 " as a function of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"y", "'"}], 
    RowBox[{"(", "0", ")"}]}], TraditionalForm]], "InlineMath"],
 " for a differential equation:"
}], "ExampleText",
 CellID->456027276],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["y", "10"], "[", 
   RowBox[{"yp0_", "?", "NumberQ"}], "]"}], " ", ":=", " ", 
  RowBox[{"First", "[", 
   RowBox[{
    RowBox[{"y", "[", "10", "]"}], " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"y", "[", "x", "]"}], "]"}]}], " ", "\[Equal]", " ", "0"}],
         ",", " ", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
        
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", 
         "yp0"}]}], "}"}], ",", "y", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "10"}], "}"}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->170598371],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    SubscriptBox["y", "10"], "[", 
    RowBox[{
     SubscriptBox["y", "0"], "'"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["y", "0"], "'"}], ",", " ", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->495578518],

Cell[BoxData[
 RowBox[{
  RowBox[{"Unset", "::", "\<\"norep\"\>"}], ":", 
  " ", "\<\"Assignment on \\!\\(Derivative\\) for \\!\\(Derivative[1]\\) not \
found. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/Unset/norep\\\", ButtonNote -> \
\\\"Unset::norep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->886134120],

Cell[BoxData[
 RowBox[{
  RowBox[{"Unset", "::", "\<\"norep\"\>"}], ":", 
  " ", "\<\"Assignment on \\!\\(Derivative\\) for \\!\\(Derivative[1]\\) not \
found. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/Unset/norep\\\", ButtonNote -> \
\\\"Unset::norep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->381763645],

Cell[BoxData[
 RowBox[{
  RowBox[{"Unset", "::", "\<\"norep\"\>"}], ":", 
  " ", "\<\"Assignment on \\!\\(Derivative\\) for \\!\\(Derivative[1]\\) not \
found. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/Unset/norep\\\", ButtonNote -> \
\\\"Unset::norep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->754038167],

Cell[BoxData[
 RowBox[{
  RowBox[{"General", "::", "\<\"stop\"\>"}], ":", 
  " ", "\<\"Further output of \\!\\(Unset :: \\\"norep\\\"\\) will be \
suppressed during this calculation. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\
\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/stop\\\", ButtonNote -> \
\\\"General::stop\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->660494860],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->249686577]
}, Open  ]],

Cell[TextData[{
 "Find a root of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["y", "10"], "(", 
     SuperscriptBox[
      SubscriptBox["y", "0"], "\[Prime]",
      MultilineFunction->None], ")"}], "\[LongEqual]", "0"}], 
   TraditionalForm]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->685856719],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["y", "10"], "[", 
     RowBox[{
      SubscriptBox["y", "0"], "'"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", 
   " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["y", "0"], "'"}], ",", 
     RowBox[{"-", "1.5"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->279554871],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   SuperscriptBox[
    SubscriptBox["y", "0"], "\[Prime]",
    MultilineFunction->None], "\[Rule]", 
   RowBox[{"-", "1.5608325027523304`"}]}], "}"}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->77403466]
}, Open  ]],

Cell["Solve the equivalent boundary value problem:", "ExampleText",
 CellID->344144216],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], ",", " ", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}]}], "}"}], " ", "/.", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{"y", "[", "x", "]"}], "]"}]}], " ", "\[Equal]", " ", 
          "0"}], ",", " ", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", 
         RowBox[{
          RowBox[{"y", "[", "10", "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}],
        ",", " ", "y", ",", "x"}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->240243937],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->98555213]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32581],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Numerical Error",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->31598],

Cell["\<\
The error tends to grow as one goes further from the initial conditions:\
\>", "ExampleText",
 CellID->21845],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "+", 
       RowBox[{"y", "[", "x", "]"}]}], "==", "0"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], "==", "0"}]}], "}"}], ",", "y", 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "100"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13675],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "100.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{323, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->3607412]
}, Open  ]],

Cell["Find the difference between numerical and exact solutions:", \
"ExampleText",
 CellID->31584],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "-", 
      RowBox[{"Cos", "[", "x", "]"}]}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "100"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->6100],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9W02oZEcVvtPdr19m5s1M4rgQsxkUDIoOqCgKCSoag0QZRXTEoDAOkRnR
vBjHjUbQjW50oxsDYhb+gD+LiCFBiCAGxIA4IAZEBYkgcWEQDC4GhOtX9Z06
56u6t/t2v/eSIenXdbtunXO+81t/77t07cq9n7l07erlS+fueuDS/VeuXv7c
uXfvP4BH82Nd1z2K/y+8rMP3Ht/y/2/q+36GL1/C365LrW6enud/3Wdz+5ln
/m3tm0Z73Z/bf/nzv952x0PPPvuf7njutYvP1GkmvbvUxv+L9H3HR7Bf5vJG
6nWL/f1U+nvjxv928kiPP/Yn0Hnqt383Ojc13Cybdks/j7KI32sO09fTTftq
ov+Nrz9J+j/4/nXQBxdGP7Pd8d+sNNfw0/6+2IifgkM3v/yJn37snh/5m/OH
vv3Une/8zsPf/d2Lhsel3IZCwMpzz/2XI8w/cOF74CP4+tpXf3XxQz9czdfC
pBq3hWlUlvb+0qyGXD39x3+adsgVGne/5+Hnn79hXH3ly098/tpj4G0NV9re
m+RjiN9O0+Ojuf3kr/8GGwoav3zir2Au4UNZwSM4BWRwIfbYv+/n3/rmb774
hV+IRyWJy9+T9ncufxeuUmEoP0g97ikYQFnBCSwa7YQJOQEDYAPKhUs7J2AU
n8bJaydl/mBu4yUYSFDCCJA5tIHfYcvgxyhDe1evPIJe13//D/YAFz/58R9q
ynPTepKIdDAixoUcxgfYhy0ibhkdAA8iaRTSwfBoKB28AQWBmtF53ZiWpX13
oYu3Qj7GwIQk+4MJyAMi4RPQaKJCPhDFaI0w2EWxV8JvfJw3OYsG31tww9Dw
QBsXTMB2MFRYPeSFhdnv+OVnjzydRiVdkAPRdXT3eoneo5Fa9N8P47Zi9a7c
xvBgOLACgyAKxKwNVgBMYEN2lEfoC99bKYAF/nMp8BefJsWphhONhZvHviTT
ncVKQCtkgHKBe8pFbs8Me5UMUDls2GWAOSQOU3uBF4gl3IQ+iZ/I+9K8uc0X
u017s/yh7TuK7Vp7Lj4UNgTGaTYmG6UonBdfVfQZu0qPWQptc7yCF9PPM0FE
5TwaHWn79twGzFXkRzSAzyTdUD78CM4i/tLS6FbUCb0TCCB8sA8HhKKgrrnL
SDt0GXdcRgJiVVMrZ+biCPLe7cXygCjigEkL3mFqkCYiAgWppCWDlJZS8JN9
6E3aB+9jFPbJT3YYSadkPT0p1XQWZdWJagdh3u3echmsrIoJoaHiWSoXfRCJ
ByaxXnYYOMx8LoghpoJkIzuHMtn3TD9n+m2z85uLF1X1HUjACiG3eBUJtl5F
eyX/7EFJyT/Vzz7z2k7ZrVM5KVsjp/WjnNM1wPne4rq1OT7rHPqZ53iIV6oA
SkhSISF54CclpMGFhHzPeW19sZIREACIzWTUaoMSMYC7ry0ANOAm4pQIAuLn
qDeIKanORR7GUJUH1R7eDHk49og8HNLkIdMcjfLQTlt5pqqa2/qxzADvYrCs
qhatngAHiVsPSkEWySGaeEju2YcWoH0YXdiHUrj0lbT0Su3nEdiqHkrb1k63
lQhiFuK1IaRJVbZGEA6kEUR5p+zKO9+CQBBLI4j24Xfl262uko+WMCqf9aN8
J8062edwNdqrchtgV/U5mING8NSwYTZnJTCXyNPnfL4Q7bAuUGzYh7wSJ3xl
OK8ij/ZzfEiywodaYT+vMip82rpArX276u9cfk7jdL9YUAp69aLgB9nhLIYX
2RYsdlDvItzhq9XvKhGQAB58wu+CxdyR7maOrEdJ70XfrHtx9OjFyNKiVerN
dgbcrh9sX2+esZGL7e0WeWyuxBUAyscYUOx3lr/Qk3rPBWSb+YyoutRWhczK
ANJmT2oEX1kicMSlY+uwcwTJQi6nVHuuVQ5GhIvw3cKmCp1KMz9Cy0zNm/vw
/Xi/7e0rDlw8MsTpXRTZ30pjmRRdl7CjDTGia+XnOKSSury95Nv45M8+qvt2
0UjuzaSBr3XvE/iSklbXWQWfBXGCYdluGfgyK5jR3dJY2c5rVNIy6LH0Nz8r
tdquoOiQb1iLr8unt0yO4Gt4XFaMuTDhoCUt+a4t2mRs1P+1jnGQUjIeaoVd
h1qxCpa9PerWWtnDFygMn6bDWitiGydcL9QfuXbX3kwvbQTZXhNJs2fleeUZ
wCf5v9YrXPEiju4VaUKsvsHobQZKLRA81YKHEYIdWkipYaAFUu+9Vmu0wNLJ
5On2HHzVAoNhrYUzrgVPAOhsMb/rmDPXaeGU0SxZsJ3JjK5kS/slq7CvZmxE
XWpjW2kocduRt6oth5Yqb7JKbDNbppi9xxa2uo7J18f06F7h7rmk1tIuvjAU
ma9QLKdovpL73uz5ZBX47llhzTX25yewPVMsCkac7FhHyv1gglHTMj/RAYkx
bS5nQMXYMlu2jk5jrFSjpzhO7/VDpu3YOtotyiRpKHvYqFHec0FMJ7npk352
fiExTmjknJqquy7/w5dSueSZHQtPs36vZKUa0Kos7N7KGot/w/yVnyw9IrvM
me7Cjb/43Q79JrRAXD0QePgY1FDD6G86aGqhvqQFjsAZURmhrmem1wIYwUko
5rjdjlcAVg83s1zTIfv6qofUEKP+t8M1BofWdMUYU3TFnu7t2nPH7UmwdUpm
4OzpuHLwtch67d+XkNN4CJmRufq8i9nrid5iI0c276SZaC3scyEaiHHUZJNW
zqqv18iK6dLn4q4A2rXUyI6qe3eFqlfJNpsbQ5W0VH/Ech2uLtsqXK33+O41
pQBj9KbOV8tg8FIRWK1FHTZzCY1ssfJiEbGxE801G8gjSIfHV3pRrIkwJRKP
d714UGHYGvH4UbSp+gm03ZYGaHvESSspEAIOX/zfVgvUfhkI+joijPquV1mV
7fhOSl+Wg8atYSgNeWmkWYm1asaxtjX2Bj/zgiHazKQNf5U0yh/NbArt7iwr
HdUncMQLdD7HB1/wkFirho8TB7WoVjK3o0oyjW/TkrmfNHbkFEVPoXPPAQwR
I3pRC3a9mC/WY03zpWM54oYVEX8jPh8s+Y74n2ZVBnogYNvDuYrAWxjBQm2O
Px79WIkbup7DWWxW8ZfvCV7H8YSIkHF2sOLeK1dy3vsChVUhbqceDl2bpzjD
4FML2EKbqDnPgZqXbo4aX1OvSzzTc3ufeuZOQd3rStcCx/YJpVGvo0zegbf9
RuZ8tGFQyXKb9bcq43kVgiFSdhxGEvV2R2BlbGDoamzQ1jqrrKdWfGoYhwmB
xz5bY18fh91DzB587rk+coksYRn+q3nIeAbl3LvdNQB04CNJ16z52tSa8d+j
va9ghraPokpbnxOm86/mhGm9S/4N7MQXR5C1TFzbMfdPARSUGfsuNFmpnu2J
roR7ZLbVlllt9VXt6vWE7Zo00Web+sUq12EemPSbHV/eUN3OY+GoYDaCnb0x
bpVcQwK8wN9Pytgek+5pE0/dO/O6oqrp3TsrXEbXEKL+NbtqMKRRra22h7Pb
dRiqxQaGMStWDPOPZnXsF5XisrLCl+ZfYRngN860kF9dCXYNmlw6t9VVMZ9B
VvJ75hifmUzN9SqktP71VTWJI5viJLMo9ZgBUtUOXDX/bdfGZv36/bezxVph
Wn5uy3aBZF8oMpXtX2gUVZ2MrreH/NW82uXXPZHwAovX62bWYzHbael6ScSc
Kj65v1Roh16IdrtromtiKfkvB1qYNW+Ulcxbc08anMbT4W4meaA1NLO+ClmP
eYrsyP7cyO7RcBVoFM/xGO4UdBUo8LSicTM82zWb7U9evTy/QYXr3JnhZriD
Hjuj7ttWoDYxt6oZtkGV9trkxgorp7EBrh6rNK6/QHaq7VsLnepkENpgLU6Z
0JJlldf26xUDr8LM/9iPkVn7ufdW/ZiVYy243Ulelym9iq2qDadTIRrnm4ho
2afLrcE+3bSdju0OzS0OzMAlLR/4gck4I0dP1/hA/asdH90pD8eyqjo83laV
hMeHquoYxzLW5wLLw++tafsVuc0zyUk+PSuiJ+iGZ56YcPSsDTGPHR2Je4aL
rhtpdeEZuzrpNaxCJrDTKmSA3UF2xNLvr8zPebwralGS0rOKlJvPD3/Kbc3u
8IEwanfrxzGa2rl6dW5Dq+A4TsowZ+hZ2/3BKU0mgdETjFXlyTeJYiO3zfq1
atJ+niF0P3x8N0JWJFfLnXT/mvyccTd5pd8MALE4fczR1BY2OY9Lz1l3Hldj
TfTzWFPVMD5LruSMWoRyTu/1UGIAaXbh2QpaijsJHDfOk1MP1PEqefnO4eVV
u5fabiBruwfzhqJ5HniM0xuw2pghkafVJ+NLlNO63ndX7MSLRjk9F+azc69Q
KYWcMRtIUd3lONQs5fVF3ur2CXjiqv7Kmxu1/HHrYVhvt3VhfXpwlcRek0uc
Xne+avOZAjXe3vNBAwjEvSVmaXqj3wzB97AJ6k1rM8quJ0Y3kZ3mwThVLHt3
VObRrL1V1fmWIrvdU/ObBBZt/YwosI97O4/L3aJV1f+q88GMQJr5os9qSQ+m
3dR8a7FfW2tya7WbrOO3x1gF6rnPuD2l571VPsYExWBz+Xha72BVr1fN8vzt
vUUxVpZxd4leu/KOHFQD0QOHYX28Ssqo8GboOy7lQe+IJAnfkXuCRHU3BHqt
bliykrwoNyzrGdT4zQNKc11uHtD82S9X0we886Gy3ZXbMCi7vapVEasCvysB
2ENLzF461x7yu18uTji/0zmc/MAt5I7vDF7MfhiLPl75vGZqck5P0Qqn5WLe
xe2K9xed2Bk7rxMu8h6x0mIcdO+zikW9jyUcKU3de7hQKNnekd94g8RxA44g
pizmN54RiGN9nnSpLdJddweeeb/16kVfZ+ATfX2HIWEUd+F5IS/uH3K3K622
+R0+MOXYyTnT7H3jd8V3G576baKqtj9cbJrR33gAOGDgot/Dg5728zUD8cFY
Zz25ms+mbtk+1yU8P5Kf+/Zo5xa1r3dE0GCoiMgCXNMK+ouB48eLLivr5P3Z
6363mIEs4TrOU5mPjNvddtkkPSZXoF/lTrAIruLmNSc1qTpYzdVR37e/tw/f
iLss1/OtrbRLRU4Of5v+k73FSJ6GNDpxr4F0pmPtp3MbTm132OM9jY739ebB
tjK2WWR7MPfSm1DDXx8lW8f+D1oyQEs=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 79},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->263792847]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->21567],

Cell["Error for a nonlinear equation:", "ExampleText",
 CellID->32297],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}], "==", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], "^", "2"}], "-", 
         RowBox[{"3", "/", "2"}]}], ")"}]}]}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "==", "0"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26009],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->349627463]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "-", 
      RowBox[{"JacobiSN", "[", 
       RowBox[{"x", ",", 
        RowBox[{"1", "/", "2"}]}], "]"}]}], "/.", "%"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->26231],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1WUtoHlUUnvzv5k/TtGl/Y+sjPmqr+NoqqGgfUWsbW6suxEUMQrLQSpq0
tbqwCLrRjW4siF34AB+LiqVSqCAWpAWxIBZEBakgdWERLC4Cwvidc+6de+fc
O/mTSRr+/DNn5p73Od+5N9k+Nj3x7HNj05PjY8MjU2MvTEyO7x3eumcKj6o9
SZIcw+/uoQT3Ke7491ia4irfd9J3lZ7yT3KInzbwTQ+rRNWzt/zOPK35T2lt
BdcKrivN9WWWfOS97wwjcyXLlL66R5Psel62ketz9Cv6INOXLs1ue+gIMwtb
1QgkCfitKa6Gomsl9Z45/fvM9HGnt2m8by2JXxWEjzlz8rK4BZnp7oX2u26y
J968dPDEqW9+c97obC0vJT+MGlKFhDk9tt5WmOhVvWvNJlNny6NfZPrChX+e
fPyjTGtyc4Srkvn65fGfXn/t67lXhzrefuvbTz/5wXFtCCKiPRa+8Wc+++Xn
vxzfRuNx3dgkq2Zn/0No8C1xDLrzANPnfvxzcuJoFu2CVTATxuZXOU2HXjn5
1clfu8lAOBHU4lX7VdybWTZrtjZKoscBlSOR3PT0kyxNL7zLxAMogSqnR6OH
rr+moheOHvtVfEXv0qPjPpWhVlRPLZqp8miyT2VPtDYCeStKaQj9c7glmiwK
l8UTkepQSqQW4YmsPvv9H3ue/4JulxWu9umZVPC9N50fjsykeRwRLRpHZlS+
Rbru22mVn7lWYbxhyOVXkSZ5iwDtHH2fEKsdlbGXaUANAIdu51rlstjOsmix
oSyOiGSHq33MraepX2Xl+mxK+Sl6Lj+OSN2g9pBvfFAbIyprl8vjm1gMNGMW
od7On/8btYnvLbzu4sV/YQ5FfHlU/1LvX9ZbrU8/9TGsMXxud7Y5lW7rj1rT
X0qvT4t+N1Ul0wAEBIhuNzGNVoGJdCtRWSxO3ci02w/IezQRWoluH1C0eF+E
YyINOYM0FJLx2c2o+9M8AhVL8+lB40ufyYDU/OF3znz4wVnK+ZDRfoW5cmSq
UAFFfoZtZ0tLdcfL1R5NetpWLryjuhQ7ULComMToz0tcEzzhXvJwKpxoZCUd
u3royo80QneUnb0qqk0Tr8487Gmr/DfnYY/GXm1Py8uPo/He5CuMU8d4pqOz
FNZIlvKTqsLwjxx6ma5GIyaxRcuh8TJf5mERaRDNcAV+U730BrxZpOCynYBi
mwSAXq9R/oQ5FBvduWK+Nob7N8xiAA6MAeZRsuQpLINkvMBny6Z3k8p99xwm
K+matBAYefXmG6dQgEAvVCJcytDeIIftTptrsdrN1VYaIliBnW2YJ7ttGIx7
UQhkgHL0IiKRVICafoaZr4k84jk+8AdMAFtCqYE0j1J3M+1OP6tyWa3ZaMsf
JsxfOCAHND4wh2Rep3IiMh16r4p6JzMDXph94C1Mux3SXbaiTEwHnZQmpEj/
4weOIA5QldzKHMglAuTyDMyCXbAWokjuHbwKEAAgUHIbkCsTCKMQQmHcbfwc
fPD186PnQEmFVAgH6xIZGAibIT7h1R4aM6omt+P71XyWa6IJzHBYT89eReu5
1u19tzmY7QURJxgKX7UFUY6FWkQaso2NPkNFJQY1ci/LQQYRfSk2mmQ0FwcK
+aluefbW4BjcAxSiRLJpDXOoVjpRfbwDqKHdsEr4rrdVanaqcb7NqcFdVAWC
ajrC7Wc6Oeu2shbYglIFA9y7htfLdE0NfoRaRuwqFC98S65O830X53owNX2F
MiWudSoSQznbHrZvzeq1yv+hqI6MC0kiMLqSaTel41zbmEaQ4Q9ZwvZ785ml
5M81OaTXNVRJw1ON//6RWC69CSz6ontvb2cU38Xo/THp38HP3S5X9lgOHdZG
9UX3zgs6a2xn2p0pVis/43rL+enT5C/15qDK47qovvhfUbp7W3SmkWi7ul5p
aWmx5KqoFYs/y4wy7SbogMq66F3s2WVUeSexR1ejlhDyhBGk8KyyU9nYZ7kN
klxbyO3TjyobsjMCIkxdP8xSup81tDUyG2AHAJPkMnoGJ4FdSrtkEpgLzIEn
CSN1gDG77CqzI5ZsImKGviHK9RjTCI7B76rVjX0fRXt9xkWW7U5Nv5loZqdp
0NgzyNk/0MFcDaAf7Me2EDEU8NwQXf2EzZhB1I3RVfL/qKTnf5Ur4m4=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->95730909]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->73370311],

Cell["\<\
For high-order methods the default interpolation may have large errors \
between steps:\
\>", "ExampleText",
 CellID->330352906],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "==", 
       RowBox[{
        RowBox[{"y", "[", "x", "]"}], 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], "^", "2"}], "-", 
          RowBox[{"3", "/", "2"}]}], ")"}]}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "==", "0"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", 
    ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", " ", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "8"}]}], "}"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->104587127],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->184574004]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}], " ", "-", " ", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], "^", "2"}], "-", 
         RowBox[{"3", "/", "2"}]}], ")"}]}]}], ")"}], " ", "/.", " ", "sol"}],
    ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->722335240],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 94},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->232144355]
}, Open  ]],

Cell["\<\
Interpolation with the order corresponding to the method reduces the error \
between steps:\
\>", "ExampleText",
 CellID->329570386],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "==", 
       RowBox[{
        RowBox[{"y", "[", "x", "]"}], 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"y", "[", "x", "]"}], "^", "2"}], "-", 
          RowBox[{"3", "/", "2"}]}], ")"}]}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "==", "0"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}], "==", "1"}]}], "}"}], ",", "y", 
    ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", " ", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "8"}]}], "}"}]}], ",", 
    " ", 
    RowBox[{"InterpolationOrder", "\[Rule]", "All"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->160838699],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->28573644]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}], " ", "-", " ", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"y", "[", "x", "]"}], "^", "2"}], "-", 
         RowBox[{"3", "/", "2"}]}], ")"}]}]}], ")"}], " ", "/.", " ", 
    "dsol"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->26199686],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 89},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->468230346]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Differential Algebraic Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->522396580],

Cell["Here is a system of differential-algebraic equations:", "ExampleText",
 CellID->120943363],

Cell[BoxData[
 RowBox[{
  RowBox[{"DAE", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["x", "1"], "\[Prime]",
        MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       SubscriptBox["x", "3"], "[", "t", "]"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["x", "2"], "[", "t", "]"}], " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", 
         RowBox[{
          SubscriptBox["x", "2"], "[", "t", "]"}]}], ")"}]}], "\[Equal]", 
      "0"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["x", "1"], "[", "t", "]"}], " ", 
        RowBox[{
         SubscriptBox["x", "2"], "[", "t", "]"}]}], "+", 
       RowBox[{
        RowBox[{
         SubscriptBox["x", "3"], "[", "t", "]"}], " ", 
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{
           SubscriptBox["x", "2"], "[", "t", "]"}]}], ")"}]}]}], "\[Equal]", 
      "t"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->782754975],

Cell[TextData[{
 "Find the solution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["x", "1"], "\[Prime]",
      MultilineFunction->None], "(", "0", ")"}], "\[LongEqual]", "1"}], 
   TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->51816239],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"DAE", ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["x", "1"], "'"}], "[", "0", "]"}], "\[Equal]", " ", 
       "1"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["x", "1"], ",", 
      SubscriptBox["x", "2"], ",", 
      SubscriptBox["x", "3"]}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->189859804],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     SubscriptBox["x", "1"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{
     SubscriptBox["x", "2"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{
     SubscriptBox["x", "3"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{314, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->414809325]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " cannot find the solution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["x", "1"], "\[Prime]",
      MultilineFunction->None], "(", "0", ")"}], "\[LongEqual]", "0"}], 
   TraditionalForm]], "InlineMath"],
 " because the solution does not exist:"
}], "ExampleText",
 CellID->435729729],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"DAE", ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["x", "1"], "'"}], "[", "0", "]"}], "\[Equal]", " ", 
       "0"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["x", "1"], ",", 
      SubscriptBox["x", "2"], ",", 
      SubscriptBox["x", "3"]}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->239796030],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"nderr\"\>"}], ":", 
  " ", "\<\"Error test failure at \\!\\(t\\) == \\!\\(0.`\\); unable to \
continue. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NDSolve/nderr\\\", ButtonNote -> \
\\\"NDSolve::nderr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->917131994],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     SubscriptBox["x", "1"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{
     SubscriptBox["x", "2"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{
     SubscriptBox["x", "3"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{314, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->375451713]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " is limited to index 1, but the solution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["x", "1"], "\[Prime]",
      MultilineFunction->None], "(", "0", ")"}], "\[LongEqual]", "0"}], 
   TraditionalForm]], "InlineMath"],
 " has index 2:"
}], "ExampleText",
 CellID->9706804],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"DAE", ",", 
      RowBox[{
       RowBox[{
        SubscriptBox["x", "1"], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], 
      ",", " ", 
      RowBox[{
       RowBox[{
        SubscriptBox["x", "2"], "[", "0", "]"}], "\[Equal]", " ", "1"}], ",", 
      " ", 
      RowBox[{
       RowBox[{
        SubscriptBox["x", "3"], "[", "0", "]"}], " ", "\[Equal]", " ", 
       "0"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["x", "1"], ",", 
      SubscriptBox["x", "2"], ",", 
      SubscriptBox["x", "3"]}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->137638465],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"nderr\"\>"}], ":", 
  " ", "\<\"Error test failure at \\!\\(t\\) == \\!\\(0.`\\); unable to \
continue. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NDSolve/nderr\\\", ButtonNote -> \
\\\"NDSolve::nderr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->150540322],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     SubscriptBox["x", "1"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{
     SubscriptBox["x", "2"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{
     SubscriptBox["x", "3"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{314, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->45753838]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Partial Differential Equations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->190271123],

Cell["Define a nonlinear PDE:", "ExampleText",
 CellID->100790804],

Cell[BoxData[{
 RowBox[{
  RowBox[{"L", "=", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"wpde", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SubscriptBox["\[PartialD]", 
        RowBox[{"t", ",", "t"}]], 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "x"}], "]"}]}], "\[Equal]", 
      RowBox[{
       RowBox[{
        SubscriptBox["\[PartialD]", 
         RowBox[{"x", ",", "x"}]], 
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", 
          SuperscriptBox[
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], "2"]}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          RowBox[{"2", " ", 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}]}]}], ")"}]}]}]}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"-", 
        SuperscriptBox["x", "2"]}]]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["u", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"1", ",", "0"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"0", ",", "x"}], "]"}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", 
        RowBox[{"-", "L"}]}], "]"}], "\[Equal]", 
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "L"}], "]"}]}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->170657928],

Cell["\<\
The spatial discretization is based on the initial value, which varies less \
than the final value:\
\>", "ExampleText",
 CellID->438720576],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s1", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"wpde", ",", "u", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "L"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "L"}], ",", "L"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->981350694],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"eerr\"\>"}], ":", 
  " ", "\<\"Warning: Scaled local spatial error estimate of \
\\!\\(13.608662762824927`\\) at \\!\\(t\\) = \\!\\(20.`\\) in the direction \
of independent variable \\!\\(x\\) is much greater than prescribed error \
tolerance. Grid spacing with \\!\\(327\\) points may be too large to achieve \
the desired accuracy or precision.  A singularity may have formed or you may \
want to specify a smaller grid spacing using the MaxStepSize or MinPoints \
method options. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NDSolve/eerr\\\", ButtonNote -> \
\\\"NDSolve::eerr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->149935379],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "20.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "20.`"}], ",", "20.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{402, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->8789718]
}, Open  ]],

Cell["\<\
By increasing the minimal number of spatial grid points, you can accurately \
compute the final value:\
\>", "ExampleText",
 CellID->27919707],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s2", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"wpde", ",", "u", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "20"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "L"}], ",", "L"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
       RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<TensorProductGrid\>\"", ",", " ", 
          RowBox[{"\"\<MinPoints\>\"", "\[Rule]", "750"}]}], "}"}]}]}], 
      "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->774659245],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "20.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "20.`"}], ",", "20.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{402, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->387096426]
}, Open  ]],

Cell["\<\
The plot demonstrates the onset of a spatially more complex solution:\
\>", "ExampleText",
 CellID->210550763],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DensityPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{
       RowBox[{"-", "t"}], ",", "x"}], "]"}], "/.", 
     RowBox[{"First", "[", "s2", "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "L"}], ",", "L"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", 
     RowBox[{"-", "L"}]}], "}"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "50"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1063819358],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 176},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->218268380]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->563384740],

Cell["\<\
Define a heat equation with an initial value that is a step function:\
\>", "ExampleText",
 CellID->627218989],

Cell[BoxData[
 RowBox[{
  RowBox[{"hpde", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}], " ", "\[Equal]", 
      "  ", 
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], ",", 
     " ", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", 
      RowBox[{"UnitStep", "[", 
       RowBox[{
        RowBox[{"1", "/", "2"}], " ", "-", " ", "x"}], "]"}]}], ",", " ", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0"}], "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
     
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "1"}], "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}]}],
   ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->107661612],

Cell["\<\
Discontinuities in the initial value may result in too many spatial grid \
points:\
\>", "ExampleText",
 CellID->65725047],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"t", ",", "x"}], "]"}], " ", "/.", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{"hpde", ",", " ", "u", ",", " ", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", ".01"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", ".01"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->265407733],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"mxsst\"\>"}], ":", 
  " ", "\<\"Using maximum number of grid points \\!\\(10000\\) allowed by the \
MaxPoints or MinStepSize options for independent variable \\!\\(x\\). \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NDSolve/mxsst\\\", \
ButtonNote -> \\\"NDSolve::mxsst\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->355775486],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->22079881]
}, Open  ]],

Cell["\<\
Setting the number of spatial grid points smaller results in essentially as \
good a solution:\
\>", "ExampleText",
 CellID->328051666],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"t", ",", "x"}], "]"}], " ", "/.", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{"hpde", ",", " ", "u", ",", " ", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", ".01"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
          RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<TensorProductGrid\>\"", ",", " ", 
             RowBox[{"\"\<MaxPoints\>\"", "\[Rule]", "100"}]}], "}"}]}]}], 
         "}"}]}]}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", ".01"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->43414940],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"mxsst\"\>"}], ":", 
  " ", "\<\"Using maximum number of grid points \\!\\(100\\) allowed by the \
MaxPoints or MinStepSize options for independent variable \\!\\(x\\). \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NDSolve/mxsst\\\", \
ButtonNote -> \\\"NDSolve::mxsst\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->134873555],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->222764338]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->106183979],

Cell["Define a Laplace equation with initial values:", "ExampleText",
 CellID->128702275],

Cell[BoxData[
 RowBox[{
  RowBox[{"lpde", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"x", ",", "y"}], "]"}], ",", "x", ",", "x"}], "]"}], " ", 
       "+", " ", 
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"x", ",", "y"}], "]"}], ",", "y", ",", "y"}], "]"}]}], " ", 
      "\[Equal]", " ", "0"}], ",", " ", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"x", ",", "0"}], "]"}], " ", "\[Equal]", " ", 
      RowBox[{"Sin", "[", 
       RowBox[{"Pi", " ", "x"}], "]"}]}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"Derivative", "[", 
         RowBox[{"0", ",", "1"}], "]"}], "[", "u", "]"}], "[", 
       RowBox[{"x", ",", "0"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
     
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"0", ",", "y"}], "]"}], " ", "\[Equal]", " ", 
      RowBox[{"u", "[", 
       RowBox[{"1", ",", "y"}], "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}]}],
   ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->117120600],

Cell["\<\
The solver only works for equations well posed as initial value (Cauchy) \
problems:\
\>", "ExampleText",
 CellID->309144532],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"lpde", ",", "u", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->120870374],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"eerr\"\>"}], ":", 
  " ", "\<\"Warning: Scaled local spatial error estimate of \
\\!\\(681.1649630786735`\\) at \\!\\(y\\) = \\!\\(1.`\\) in the direction of \
independent variable \\!\\(x\\) is much greater than prescribed error \
tolerance. Grid spacing with \\!\\(25\\) points may be too large to achieve \
the desired accuracy or precision.  A singularity may have formed or you may \
want to specify a smaller grid spacing using the MaxStepSize or MinPoints \
method options. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NDSolve/eerr\\\", ButtonNote -> \
\\\"NDSolve::eerr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->60679217],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{373, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->298218024]
}, Open  ]],

Cell["The ill-posedness shows up as numerical instability:", "ExampleText",
 CellID->16354085],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"x", ",", "y"}], "]"}], " ", "/.", " ", "sol"}], "]"}], ",", 
   " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", "0", ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->702053895],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->16252919]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Boundary Value Problems",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->565542508],

Cell["This finds a trivial solution of a boundary value problem:", \
"ExampleText",
 CellID->237261224],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "x", "]"}], ",", " ", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}]}], "}"}], " ", "/.", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{"y", "[", "x", "]"}], "]"}]}], " ", "\[Equal]", " ", 
          "0"}], ",", " ", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", 
         RowBox[{
          RowBox[{"y", "[", "10", "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}],
        ",", " ", "y", ",", "x"}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4180016],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlV81OwzAMTv+2cUei7Ya0p4ETD1FNSBsSAo09DUgc4MR9L8CNtwpxWy+p
47ZZGVUQ1ZrEjusvsb/l56bYrW/vi91mVSyvt8XjerN6Wl49bJUqCoQQe/Xe
pUK1pWqV715KVZvlVJWR6gpBgoYsHxGq5ntpMSM9ERSxloVIiMVzpY6qCp4Q
xQF+3PDB4Bj7oTj2dxBBmN/EqG2ri7r35SDxNj/X/k2UPn8YZZNFyCw+4hMr
f1Muw728rdJuuHH044aveWvbJ7KLUby9adfNV6x5K38y76ZNVQHBWBD9vEZY
1HUu9Xp34JPlLTNkwXjNrdi7aAATFuYA6sb85wx+jPoT4dCo0OhZ8234xDja
IzoeLTNyEjnOsh8lbR3z5YDs9eNlLd6RaYieS820JuNO+U86V+VrPW7D5vPj
S/1EAoUhx0T2rd+b8dQx5lcJymsqc2tRl8yj/M/9fhyUmWw/VbTvnt27Oj1J
j/GdPkXY9niKcJ0PtYs7I4enCN7Kn0z7ieLCz7bbGZ/PM6K1bzF8pn73O83P
sceHPMXb2Vupte/BIvgGvNjkEA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->234289763]
}, Open  ]],

Cell["\<\
You can find other solutions by giving starting conditions for the solution \
search:\
\>", "ExampleText",
 CellID->6166462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Evaluate", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"y", "[", "x", "]"}], ",", " ", 
         RowBox[{
          RowBox[{"y", "'"}], "[", "x", "]"}]}], "}"}], " ", "/.", " ", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", " ", 
             RowBox[{"Sin", "[", 
              RowBox[{"y", "[", "x", "]"}], "]"}]}], " ", "\[Equal]", " ", 
            "0"}], ",", " ", 
           RowBox[{
            RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", 
           RowBox[{
            RowBox[{"y", "[", "10", "]"}], " ", "\[Equal]", " ", "0"}]}], 
          "}"}], ",", " ", "y", ",", "x", ",", " ", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<Shooting\>\"", ",", " ", 
            RowBox[{"\"\<StartingInitialConditions\>\"", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{
                RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], 
               ",", " ", 
               RowBox[{
                RowBox[{
                 RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", 
                "i"}]}], "}"}]}]}], "}"}]}]}], "]"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->49683823],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->17914711]
}, Open  ]]
}, Closed]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->6122],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["DSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DSolve"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NIntegrate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NIntegrate"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NSolve"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->12782]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->19724],

Cell[TextData[ButtonBox["Numerical Differential Equations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NumericalDifferentialEquations"]], "Tutorials",
 CellID->81285119],

Cell[TextData[ButtonBox["Numerical Solution of Differential Equations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalSolutionOfDifferentialEquations"]], "Tutorials",
 CellID->25233],

Cell[TextData[{
 ButtonBox["Advanced Numerical Differential Equation Solving in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
  FontSlant->"Italic"]
}], "Tutorials",
 CellID->64016169],

Cell[TextData[{
 ButtonBox["Numerical Mathematics in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
  FontSlant->"Italic"]
}], "Tutorials",
 CellID->26634]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->5603],

Cell[TextData[{
 ButtonBox["Demonstrations with NDSolve",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=NDSolve"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#3826"], None}]], "RelatedLinks",
 CellID->17125],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
NDSolve"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->3638660],

Cell[TextData[ButtonBox["Calculus",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Calculus"]], "MoreAbout",
 CellID->254891512],

Cell[TextData[ButtonBox["Differential Equations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DifferentialEquations"]], "MoreAbout",
 CellID->388269326],

Cell[TextData[ButtonBox["Equation Solving",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/EquationSolving"]], "MoreAbout",
 CellID->30841973],

Cell[TextData[ButtonBox["Numerical Evaluation & Precision",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumericalEvaluationAndPrecision"]], "MoreAbout",
 CellID->919461540],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->294875945]
}, Open  ]],

Cell["New in 2  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 1, 32.0281604}", "context" -> "System`", 
    "keywords" -> {
     "Adams methods", "adaptive procedure", "BDF methods", 
      "boundary conditions", "BVP", "chasing method", "DAE", 
      "differential equations", "differential-algebraic equations", 
      "double-step", "event locator", "explicit Euler", "explicit midpoint", 
      "explicit modified midpoint", "explicit Runge-Kutta", "extrapolation", 
      "fixed step", "gear method", "Gel'fand-Lokutsiyevskii method", 
      "Gelfand-Lokutsiyevskii method", "implicit Runge-Kutta", 
      "initial conditions", "linearly implicit Euler", 
      "linearly implicit midpoint", "linearly implicit modified midpoint", 
      "locally-exact", "LSODA", "method of lines", 
      "numerical solution of differential equations", "ODE", 
      "ordinary differential equations", "orthogonal projection", 
      "partial differential equations", "PDE", "phase plots", 
      "Runge-Kutta methods", "singularities", 
      "solving differential equations", "splitting", "stiffness switching", 
      "symplectic partitioned Runge-Kutta"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "NDSolve[eqns, y, {x, x_min, x_max}] finds a numerical solution to the \
ordinary differential equations eqns for the function y with the independent \
variable x in the range x_min to x_max. NDSolve[eqns, y, {x, x_min, x_max}, \
{t, t_min, t_max}] finds a numerical solution to the partial differential \
equations eqns. NDSolve[eqns, {y_1, y_2, ...}, {x, x_min, x_max}] finds \
numerical solutions for the functions y_i. ", "synonyms" -> {}, "title" -> 
    "NDSolve", "type" -> "Symbol", "uri" -> "ref/NDSolve"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[32861, 993, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->397145522]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 903368, 19912}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3301, 73, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3924, 100, 4191, 83, 70, "ObjectNameGrid"],
Cell[8118, 185, 3457, 113, 70, "Usage",
 CellID->16552]
}, Open  ]],
Cell[CellGroupData[{
Cell[11612, 303, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[12090, 320, 352, 12, 70, "Notes",
 CellID->15874],
Cell[12445, 334, 797, 30, 70, "Notes",
 CellID->16977],
Cell[13245, 366, 527, 19, 70, "Notes",
 CellID->13697],
Cell[13775, 387, 271, 8, 70, "Notes",
 CellID->15370],
Cell[14049, 397, 385, 12, 70, "Notes",
 CellID->9941],
Cell[14437, 411, 281, 9, 70, "Notes",
 CellID->20267],
Cell[14721, 422, 692, 26, 70, "Notes",
 CellID->12341],
Cell[15416, 450, 462, 18, 70, "Notes",
 CellID->617],
Cell[15881, 470, 405, 16, 70, "Notes",
 CellID->24968],
Cell[16289, 488, 497, 19, 70, "Notes",
 CellID->25065],
Cell[16789, 509, 230, 8, 70, "Notes",
 CellID->27321],
Cell[17022, 519, 385, 11, 70, "Notes",
 CellID->18655],
Cell[17410, 532, 246, 9, 70, "Notes",
 CellID->16727],
Cell[17659, 543, 69, 1, 70, "Notes",
 CellID->26914],
Cell[17731, 546, 4426, 118, 70, "3ColumnTableMod",
 CellID->8283],
Cell[22160, 666, 548, 18, 70, "Notes",
 CellID->2137],
Cell[22711, 686, 747, 28, 70, "Notes",
 CellID->6932],
Cell[23461, 716, 446, 13, 70, "Notes",
 CellID->26549],
Cell[23910, 731, 414, 14, 70, "Notes",
 CellID->1162],
Cell[24327, 747, 662, 25, 70, "Notes",
 CellID->12882],
Cell[24992, 774, 436, 13, 70, "Notes",
 CellID->14035],
Cell[25431, 789, 517, 18, 70, "Notes",
 CellID->28592],
Cell[25951, 809, 220, 8, 70, "Notes",
 CellID->25135],
Cell[26174, 819, 1403, 29, 70, "2ColumnTableMod",
 CellID->8930],
Cell[27580, 850, 1124, 40, 70, "Notes",
 CellID->2534],
Cell[28707, 892, 1696, 33, 70, "2ColumnTableMod",
 CellID->17520],
Cell[30406, 927, 76, 1, 70, "Notes",
 CellID->17728],
Cell[30485, 930, 1572, 32, 70, "2ColumnTableMod",
 CellID->29688],
Cell[32060, 964, 764, 24, 70, "Notes",
 CellID->323647842]
}, Closed]],
Cell[CellGroupData[{
Cell[32861, 993, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->397145522],
Cell[CellGroupData[{
Cell[33246, 1007, 147, 5, 70, "ExampleSection",
 CellID->68938457],
Cell[33396, 1014, 90, 1, 70, "ExampleText",
 CellID->28264],
Cell[CellGroupData[{
Cell[33511, 1019, 573, 19, 28, "Input",
 CellID->5423],
Cell[34087, 1040, 467, 17, 36, "Output",
 CellID->471562520]
}, Open  ]],
Cell[34569, 1060, 65, 1, 70, "ExampleText",
 CellID->4486],
Cell[CellGroupData[{
Cell[34659, 1065, 322, 10, 28, "Input",
 CellID->28908],
Cell[34984, 1077, 2806, 50, 136, "Output",
 Evaluatable->False,
 CellID->13675044]
}, Open  ]],
Cell[37805, 1130, 84, 1, 70, "ExampleText",
 CellID->8631],
Cell[CellGroupData[{
Cell[37914, 1135, 397, 14, 28, "Input",
 CellID->1565],
Cell[38314, 1151, 4067, 71, 167, "Output",
 Evaluatable->False,
 CellID->29432243]
}, Open  ]],
Cell[42396, 1225, 59, 1, 70, "ExampleText",
 CellID->4362],
Cell[CellGroupData[{
Cell[42480, 1230, 221, 8, 70, "Input",
 CellID->563],
Cell[42704, 1240, 270, 9, 36, "Output",
 CellID->136858501]
}, Open  ]],
Cell[42989, 1252, 121, 3, 70, "ExampleDelimiter",
 CellID->14419],
Cell[43113, 1257, 93, 1, 70, "ExampleText",
 CellID->31851],
Cell[CellGroupData[{
Cell[43231, 1262, 682, 23, 70, "Input",
 CellID->8741],
Cell[43916, 1287, 467, 17, 36, "Output",
 CellID->147730850]
}, Open  ]],
Cell[44398, 1307, 87, 1, 70, "ExampleText",
 CellID->13076],
Cell[CellGroupData[{
Cell[44510, 1312, 507, 17, 28, "Input",
 CellID->21168],
Cell[45020, 1331, 8069, 136, 125, "Output",
 Evaluatable->False,
 CellID->339716780]
}, Open  ]],
Cell[53104, 1470, 121, 3, 70, "ExampleDelimiter",
 CellID->30098],
Cell[53228, 1475, 81, 1, 70, "ExampleText",
 CellID->19587],
Cell[CellGroupData[{
Cell[53334, 1480, 897, 30, 70, "Input",
 CellID->20531],
Cell[54234, 1512, 774, 28, 54, "Output",
 CellID->572529817]
}, Open  ]],
Cell[CellGroupData[{
Cell[55045, 1545, 369, 12, 28, "Input",
 CellID->8339],
Cell[55417, 1559, 6547, 111, 232, "Output",
 Evaluatable->False,
 CellID->32042046]
}, Open  ]],
Cell[61979, 1673, 121, 3, 70, "ExampleDelimiter",
 CellID->26830],
Cell[62103, 1678, 86, 1, 70, "ExampleText",
 CellID->9613550],
Cell[CellGroupData[{
Cell[62214, 1683, 910, 29, 70, "Input",
 CellID->20781],
Cell[63127, 1714, 563, 20, 36, "Output",
 CellID->793882145]
}, Open  ]],
Cell[CellGroupData[{
Cell[63727, 1739, 424, 13, 28, "Input",
 CellID->2900],
Cell[64154, 1754, 18735, 311, 166, "Output",
 Evaluatable->False,
 CellID->79463305]
}, Open  ]],
Cell[82904, 2068, 68, 1, 70, "ExampleText",
 CellID->29700],
Cell[CellGroupData[{
Cell[82997, 2073, 945, 31, 70, "Input",
 CellID->24234],
Cell[83945, 2106, 562, 20, 36, "Output",
 CellID->77337803]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[84556, 2132, 217, 7, 70, "ExampleSection",
 CellID->10680],
Cell[CellGroupData[{
Cell[84798, 2143, 251, 7, 70, "ExampleSubsection",
 CellID->99904585],
Cell[85052, 2152, 124, 3, 70, "ExampleText",
 CellID->5110363],
Cell[CellGroupData[{
Cell[85201, 2159, 785, 26, 70, "Input",
 CellID->39034423],
Cell[85989, 2187, 466, 17, 36, "Output",
 CellID->5929889]
}, Open  ]],
Cell[86470, 2207, 101, 2, 70, "ExampleText",
 CellID->13663552],
Cell[CellGroupData[{
Cell[86596, 2213, 457, 15, 70, "Input",
 CellID->235254161],
Cell[87056, 2230, 10778, 181, 70, "Output",
 Evaluatable->False,
 CellID->555469401]
}, Open  ]],
Cell[97849, 2414, 125, 3, 70, "ExampleDelimiter",
 CellID->656801886],
Cell[97977, 2419, 82, 1, 70, "ExampleText",
 CellID->257759957],
Cell[CellGroupData[{
Cell[98084, 2424, 1503, 46, 70, "Input",
 CellID->332826321],
Cell[99590, 2472, 1052, 38, 71, "Output",
 CellID->1850539]
}, Open  ]],
Cell[CellGroupData[{
Cell[100679, 2515, 441, 14, 70, "Input",
 CellID->431684294],
Cell[101123, 2531, 5454, 93, 70, "Output",
 Evaluatable->False,
 CellID->299067117]
}, Open  ]],
Cell[106592, 2627, 124, 3, 70, "ExampleDelimiter",
 CellID->73400561],
Cell[106719, 2632, 77, 1, 70, "ExampleText",
 CellID->37159131],
Cell[CellGroupData[{
Cell[106821, 2637, 1433, 42, 70, "Input",
 CellID->853395542],
Cell[108257, 2681, 465, 17, 36, "Output",
 CellID->49679485]
}, Open  ]],
Cell[CellGroupData[{
Cell[108759, 2703, 121, 4, 70, "Input",
 CellID->426580444],
Cell[108883, 2709, 331, 11, 36, "Output",
 CellID->623800703]
}, Open  ]],
Cell[109229, 2723, 84, 1, 70, "ExampleText",
 CellID->202781194],
Cell[CellGroupData[{
Cell[109338, 2728, 280, 9, 70, "Input",
 CellID->53724152],
Cell[109621, 2739, 8256, 139, 70, "Output",
 Evaluatable->False,
 CellID->143354055]
}, Open  ]],
Cell[117892, 2881, 125, 3, 70, "ExampleDelimiter",
 CellID->241093779],
Cell[118020, 2886, 142, 4, 70, "ExampleText",
 CellID->265604313],
Cell[CellGroupData[{
Cell[118187, 2894, 547, 19, 70, "Input",
 CellID->723423364],
Cell[118737, 2915, 466, 17, 36, "Output",
 CellID->14214380]
}, Open  ]],
Cell[CellGroupData[{
Cell[119240, 2937, 423, 14, 70, "Input",
 CellID->443931913],
Cell[119666, 2953, 4434, 77, 70, "Output",
 Evaluatable->False,
 CellID->109474841]
}, Open  ]],
Cell[124115, 3033, 80, 1, 70, "ExampleText",
 CellID->781148539],
Cell[CellGroupData[{
Cell[124220, 3038, 707, 23, 70, "Input",
 CellID->323930071],
Cell[124930, 3063, 774, 28, 54, "Output",
 CellID->943728904]
}, Open  ]],
Cell[CellGroupData[{
Cell[125741, 3096, 419, 13, 70, "Input",
 CellID->396563124],
Cell[126163, 3111, 4434, 77, 70, "Output",
 Evaluatable->False,
 CellID->282295871]
}, Open  ]],
Cell[130612, 3191, 131, 4, 70, "ExampleText",
 CellID->515271795],
Cell[CellGroupData[{
Cell[130768, 3199, 713, 24, 70, "Input",
 CellID->493937889],
Cell[131484, 3225, 467, 17, 36, "Output",
 CellID->764640490]
}, Open  ]],
Cell[CellGroupData[{
Cell[131988, 3247, 277, 9, 70, "Input",
 CellID->207087099],
Cell[132268, 3258, 4069, 71, 70, "Output",
 Evaluatable->False,
 CellID->677241]
}, Open  ]],
Cell[136352, 3332, 125, 3, 70, "ExampleDelimiter",
 CellID->125568132],
Cell[136480, 3337, 121, 3, 70, "ExampleText",
 CellID->36857027],
Cell[136604, 3342, 278, 10, 70, "Input",
 CellID->79547994],
Cell[CellGroupData[{
Cell[136907, 3356, 631, 21, 70, "Input",
 CellID->4535099],
Cell[137541, 3379, 359, 13, 36, "Output",
 CellID->63540726]
}, Open  ]],
Cell[137915, 3395, 73, 1, 70, "ExampleText",
 CellID->270259617],
Cell[CellGroupData[{
Cell[138013, 3400, 377, 12, 70, "Input",
 CellID->215987619],
Cell[138393, 3414, 4396, 76, 70, "Output",
 Evaluatable->False,
 CellID->254608145]
}, Open  ]],
Cell[142804, 3493, 125, 3, 70, "ExampleDelimiter",
 CellID->122523365],
Cell[142932, 3498, 71, 1, 70, "ExampleText",
 CellID->5349239],
Cell[143006, 3501, 798, 27, 70, "Input",
 CellID->201941871],
Cell[143807, 3530, 133, 4, 70, "ExampleText",
 CellID->377329246],
Cell[CellGroupData[{
Cell[143965, 3538, 268, 9, 70, "Input",
 CellID->12698415],
Cell[144236, 3549, 777, 28, 54, "Output",
 CellID->24444224]
}, Open  ]],
Cell[CellGroupData[{
Cell[145050, 3582, 329, 10, 70, "Input",
 CellID->86660993],
Cell[145382, 3594, 2095, 38, 70, "Output",
 Evaluatable->False,
 CellID->659258409]
}, Open  ]],
Cell[147492, 3635, 94, 1, 70, "ExampleText",
 CellID->601641067],
Cell[CellGroupData[{
Cell[147611, 3640, 1096, 34, 70, "Input",
 CellID->156182478],
Cell[148710, 3676, 284, 6, 70, "Message",
 CellID->42902036],
Cell[148997, 3684, 808, 28, 54, "Output",
 CellID->6800184]
}, Open  ]],
Cell[149820, 3715, 124, 3, 70, "ExampleDelimiter",
 CellID->16425158],
Cell[149947, 3720, 90, 1, 70, "ExampleText",
 CellID->596309688],
Cell[CellGroupData[{
Cell[150062, 3725, 616, 20, 70, "Input",
 CellID->131565440],
Cell[150681, 3747, 1461, 28, 70, "Output",
 Evaluatable->False,
 CellID->1061009135]
}, Open  ]],
Cell[CellGroupData[{
Cell[152179, 3780, 275, 9, 70, "Input",
 CellID->496817457],
Cell[152457, 3791, 3534, 62, 70, "Output",
 Evaluatable->False,
 CellID->68497495]
}, Open  ]],
Cell[156006, 3856, 125, 3, 70, "ExampleDelimiter",
 CellID->871106438],
Cell[156134, 3861, 242, 8, 70, "ExampleText",
 CellID->23617400],
Cell[CellGroupData[{
Cell[156401, 3873, 517, 17, 70, "Input",
 CellID->6317317],
Cell[156921, 3892, 466, 17, 36, "Output",
 CellID->864697541]
}, Open  ]],
Cell[CellGroupData[{
Cell[157424, 3914, 275, 9, 70, "Input",
 CellID->107509006],
Cell[157702, 3925, 2464, 44, 70, "Output",
 Evaluatable->False,
 CellID->38047015]
}, Open  ]],
Cell[160181, 3972, 360, 14, 70, "ExampleText",
 CellID->24835091],
Cell[CellGroupData[{
Cell[160566, 3990, 554, 19, 70, "Input",
 CellID->206682691],
Cell[161123, 4011, 465, 17, 36, "Output",
 CellID->25026241]
}, Open  ]],
Cell[CellGroupData[{
Cell[161625, 4033, 391, 14, 70, "Input",
 CellID->216482258],
Cell[162019, 4049, 3173, 56, 70, "Output",
 Evaluatable->False,
 CellID->173216352]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[165241, 4111, 242, 7, 70, "ExampleSubsection",
 CellID->5187909],
Cell[165486, 4120, 88, 1, 70, "ExampleText",
 CellID->51965684],
Cell[CellGroupData[{
Cell[165599, 4125, 823, 25, 70, "Input",
 CellID->77218768],
Cell[166425, 4152, 421, 8, 70, "Message",
 CellID->220104526],
Cell[166849, 4162, 356, 8, 70, "Message",
 CellID->205823825],
Cell[167208, 4172, 772, 28, 54, "Output",
 CellID->235329110]
}, Open  ]],
Cell[CellGroupData[{
Cell[168017, 4205, 812, 25, 70, "Input",
 CellID->139000656],
Cell[168832, 4232, 2839, 51, 70, "Output",
 Evaluatable->False,
 CellID->16517113]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[171720, 4289, 249, 7, 70, "ExampleSubsection",
 CellID->16331],
Cell[171972, 4298, 96, 1, 70, "ExampleText",
 CellID->218712133],
Cell[CellGroupData[{
Cell[172093, 4303, 872, 29, 70, "Input",
 CellID->18805],
Cell[172968, 4334, 772, 28, 54, "Output",
 CellID->5868487]
}, Open  ]],
Cell[CellGroupData[{
Cell[173777, 4367, 370, 12, 70, "Input",
 CellID->13139],
Cell[174150, 4381, 1532, 29, 70, "Output",
 Evaluatable->False,
 CellID->52613757]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[175731, 4416, 251, 7, 70, "ExampleSubsection",
 CellID->484808727],
Cell[175985, 4425, 97, 1, 70, "ExampleText",
 CellID->249566227],
Cell[CellGroupData[{
Cell[176107, 4430, 1427, 44, 70, "Input",
 CellID->197355455],
Cell[177537, 4476, 622, 21, 36, "Output",
 CellID->86502572]
}, Open  ]],
Cell[CellGroupData[{
Cell[178196, 4502, 497, 15, 70, "Input",
 CellID->22369196],
Cell[178696, 4519, 17938, 298, 70, "Output",
 Evaluatable->False,
 CellID->54459521]
}, Open  ]],
Cell[196649, 4820, 125, 3, 70, "ExampleDelimiter",
 CellID->378646315],
Cell[196777, 4825, 103, 2, 70, "ExampleText",
 CellID->13424634],
Cell[196883, 4829, 2177, 68, 70, "Input",
 CellID->44479087],
Cell[199063, 4899, 1407, 48, 70, "Input",
 CellID->36520046],
Cell[CellGroupData[{
Cell[200495, 4951, 364, 12, 70, "Input",
 CellID->62472787],
Cell[200862, 4965, 970, 34, 54, "Output",
 CellID->311532987]
}, Open  ]],
Cell[CellGroupData[{
Cell[201869, 5004, 821, 26, 70, "Input",
 CellID->22821469],
Cell[202693, 5032, 13579, 227, 70, "Output",
 Evaluatable->False,
 CellID->98558905]
}, Open  ]],
Cell[216287, 5262, 125, 3, 70, "ExampleDelimiter",
 CellID->861634095],
Cell[216415, 5267, 144, 4, 70, "ExampleText",
 CellID->102297288],
Cell[CellGroupData[{
Cell[216584, 5275, 2377, 66, 70, "Input",
 CellID->211175700],
Cell[218964, 5343, 692, 24, 36, "Output",
 CellID->455443847]
}, Open  ]],
Cell[219671, 5370, 79, 1, 70, "ExampleText",
 CellID->168065660],
Cell[CellGroupData[{
Cell[219775, 5375, 479, 16, 70, "Input",
 CellID->104977509],
Cell[220257, 5393, 26162, 433, 70, "Output",
 Evaluatable->False,
 CellID->343625809]
}, Open  ]],
Cell[246434, 5829, 117, 3, 70, "ExampleText",
 CellID->414994633],
Cell[CellGroupData[{
Cell[246576, 5836, 436, 13, 70, "Input",
 CellID->176636832],
Cell[247015, 5851, 13737, 229, 70, "Output",
 Evaluatable->False,
 CellID->24957344]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[260813, 6087, 239, 7, 70, "ExampleSection",
 CellID->24370],
Cell[261055, 6096, 81, 1, 70, "ExampleText",
 CellID->6676],
Cell[CellGroupData[{
Cell[261161, 6101, 1551, 50, 70, "Input",
 CellID->10143],
Cell[262715, 6153, 1960, 36, 70, "Output",
 Evaluatable->False,
 CellID->370638629]
}, Open  ]],
Cell[CellGroupData[{
Cell[264712, 6194, 327, 11, 70, "Input",
 CellID->17973],
Cell[265042, 6207, 1619, 31, 70, "Output",
 Evaluatable->False,
 CellID->298837429]
}, Open  ]],
Cell[CellGroupData[{
Cell[266698, 6243, 454, 15, 70, "Input",
 CellID->11957],
Cell[267155, 6260, 5624, 96, 70, "Output",
 Evaluatable->False,
 CellID->62607028]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[272828, 6362, 222, 7, 70, "ExampleSection",
 CellID->771143190],
Cell[CellGroupData[{
Cell[273075, 6373, 247, 7, 70, "ExampleSubsection",
 CellID->10843],
Cell[273325, 6382, 150, 4, 70, "ExampleText",
 CellID->30672],
Cell[273478, 6388, 743, 25, 70, "Input",
 CellID->19814],
Cell[CellGroupData[{
Cell[274246, 6417, 652, 21, 70, "Input",
 CellID->21298],
Cell[274901, 6440, 3173, 56, 70, "Output",
 Evaluatable->False,
 CellID->580111716]
}, Open  ]],
Cell[278089, 6499, 101, 2, 70, "ExampleText",
 CellID->11261],
Cell[CellGroupData[{
Cell[278215, 6505, 755, 23, 70, "Input",
 CellID->12809],
Cell[278973, 6530, 3892, 68, 70, "Output",
 Evaluatable->False,
 CellID->16378611]
}, Open  ]],
Cell[282880, 6601, 99, 2, 70, "ExampleText",
 CellID->17327],
Cell[CellGroupData[{
Cell[283004, 6607, 863, 25, 70, "Input",
 CellID->28441],
Cell[283870, 6634, 3881, 68, 70, "Output",
 Evaluatable->False,
 CellID->416140495]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[287800, 6708, 238, 7, 70, "ExampleSubsection",
 CellID->80839691],
Cell[288041, 6717, 83, 1, 70, "ExampleText",
 CellID->285735470],
Cell[288127, 6720, 1365, 43, 70, "Input",
 CellID->136919303],
Cell[CellGroupData[{
Cell[289517, 6767, 132, 4, 70, "Input",
 CellID->468903557],
Cell[289652, 6773, 1492, 49, 87, "Output",
 CellID->758789964]
}, Open  ]],
Cell[291159, 6825, 206, 6, 70, "ExampleText",
 CellID->11419481],
Cell[CellGroupData[{
Cell[291390, 6835, 336, 9, 70, "Input",
 CellID->3202441],
Cell[291729, 6846, 492, 18, 36, "Output",
 CellID->501685373]
}, Open  ]],
Cell[CellGroupData[{
Cell[292258, 6869, 340, 11, 70, "Input",
 CellID->211402592],
Cell[292601, 6882, 2326, 42, 70, "Output",
 Evaluatable->False,
 CellID->28691989]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[294976, 6930, 233, 7, 70, "ExampleSubsection",
 CellID->5491],
Cell[295212, 6939, 67, 1, 70, "ExampleText",
 CellID->24597],
Cell[CellGroupData[{
Cell[295304, 6944, 789, 25, 70, "Input",
 CellID->7099],
Cell[296096, 6971, 159, 5, 36, "Output",
 CellID->75891397]
}, Open  ]],
Cell[296270, 6979, 121, 3, 70, "ExampleDelimiter",
 CellID->21823],
Cell[296394, 6984, 138, 4, 70, "ExampleText",
 CellID->126660969],
Cell[CellGroupData[{
Cell[296557, 6992, 880, 26, 70, "Input",
 CellID->13310],
Cell[297440, 7020, 3020, 54, 70, "Output",
 Evaluatable->False,
 CellID->7878603]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[300509, 7080, 235, 7, 70, "ExampleSubsection",
 CellID->65990228],
Cell[300747, 7089, 127, 3, 70, "ExampleText",
 CellID->338881393],
Cell[CellGroupData[{
Cell[300899, 7096, 1533, 45, 70, "Input",
 CellID->52497343],
Cell[302435, 7143, 2285, 42, 70, "Output",
 Evaluatable->False,
 CellID->5667811]
}, Open  ]],
Cell[304735, 7188, 265, 8, 70, "ExampleText",
 CellID->175305345],
Cell[CellGroupData[{
Cell[305025, 7200, 1598, 47, 70, "Input",
 CellID->47417740],
Cell[306626, 7249, 13933, 232, 70, "Output",
 Evaluatable->False,
 CellID->294481900]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[320608, 7487, 228, 7, 70, "ExampleSubsection",
 CellID->47654750],
Cell[320839, 7496, 92, 1, 70, "ExampleText",
 CellID->18297381],
Cell[CellGroupData[{
Cell[320956, 7501, 660, 21, 70, "Input",
 CellID->41796065],
Cell[321619, 7524, 429, 8, 70, "Message",
 CellID->429698689],
Cell[322051, 7534, 482, 17, 36, "Output",
 CellID->19227784]
}, Open  ]],
Cell[322548, 7554, 88, 1, 70, "ExampleText",
 CellID->28461920],
Cell[CellGroupData[{
Cell[322661, 7559, 794, 24, 70, "Input",
 CellID->358373314],
Cell[323458, 7585, 468, 17, 36, "Output",
 CellID->306251049]
}, Open  ]],
Cell[323941, 7605, 80, 1, 70, "ExampleText",
 CellID->358078506],
Cell[CellGroupData[{
Cell[324046, 7610, 643, 18, 70, "Input",
 CellID->69667603],
Cell[324692, 7630, 12505, 209, 70, "Output",
 Evaluatable->False,
 CellID->76577998]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[337246, 7845, 232, 7, 70, "ExampleSubsection",
 CellID->154690044],
Cell[337481, 7854, 112, 3, 70, "ExampleText",
 CellID->576831980],
Cell[CellGroupData[{
Cell[337618, 7861, 1380, 40, 70, "Input",
 CellID->85406985],
Cell[339001, 7903, 2794, 50, 70, "Output",
 Evaluatable->False,
 CellID->399446377]
}, Open  ]],
Cell[341810, 7956, 363, 13, 70, "ExampleText",
 CellID->245961639],
Cell[CellGroupData[{
Cell[342198, 7973, 1444, 41, 70, "Input",
 CellID->90211088],
Cell[343645, 8016, 2473, 45, 70, "Output",
 Evaluatable->False,
 CellID->285638674]
}, Open  ]],
Cell[346133, 8064, 125, 3, 70, "ExampleDelimiter",
 CellID->309908885],
Cell[346261, 8069, 223, 6, 70, "ExampleText",
 CellID->534866714],
Cell[CellGroupData[{
Cell[346509, 8079, 1123, 32, 70, "Input",
 CellID->81999207],
Cell[347635, 8113, 159, 5, 36, "Output",
 CellID->135026800]
}, Open  ]],
Cell[347809, 8121, 255, 8, 70, "ExampleText",
 CellID->132064070],
Cell[CellGroupData[{
Cell[348089, 8133, 1204, 33, 70, "Input",
 CellID->138136159],
Cell[349296, 8168, 160, 5, 36, "Output",
 CellID->309303379]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[349505, 8179, 223, 7, 70, "ExampleSubsection",
 CellID->24247],
Cell[349731, 8188, 200, 6, 70, "ExampleText",
 CellID->32113],
Cell[CellGroupData[{
Cell[349956, 8198, 874, 26, 70, "Input",
 CellID->27422],
Cell[350833, 8226, 2185, 44, 135, "Output",
 CellID->1863007]
}, Open  ]],
Cell[353033, 8273, 121, 3, 70, "ExampleDelimiter",
 CellID->25974],
Cell[353157, 8278, 127, 4, 70, "ExampleText",
 CellID->23488],
Cell[CellGroupData[{
Cell[353309, 8286, 947, 28, 70, "Input",
 CellID->3022],
Cell[354259, 8316, 1518, 34, 138, "Output",
 CellID->887793283]
}, Open  ]],
Cell[355792, 8353, 121, 3, 70, "ExampleDelimiter",
 CellID->17519],
Cell[355916, 8358, 61, 1, 70, "ExampleText",
 CellID->24083],
Cell[CellGroupData[{
Cell[356002, 8363, 1062, 31, 70, "Input",
 CellID->15966],
Cell[357067, 8396, 2928, 55, 131, "Output",
 CellID->11084001]
}, Open  ]],
Cell[360010, 8454, 120, 3, 70, "ExampleDelimiter",
 CellID->2794],
Cell[360133, 8459, 97, 2, 70, "ExampleText",
 CellID->7646],
Cell[CellGroupData[{
Cell[360255, 8465, 1061, 31, 70, "Input",
 CellID->8324],
Cell[361319, 8498, 6272, 107, 70, "Output",
 Evaluatable->False,
 CellID->329765648]
}, Open  ]],
Cell[367606, 8608, 121, 3, 70, "ExampleDelimiter",
 CellID->31089],
Cell[367730, 8613, 84, 1, 70, "ExampleText",
 CellID->27098],
Cell[CellGroupData[{
Cell[367839, 8618, 943, 28, 70, "Input",
 CellID->21797],
Cell[368785, 8648, 1075, 25, 144, "Output",
 CellID->98812527]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[369909, 8679, 233, 7, 70, "ExampleSubsection",
 CellID->448770909],
Cell[370145, 8688, 126, 3, 70, "ExampleText",
 CellID->340610123],
Cell[CellGroupData[{
Cell[370296, 8695, 2908, 84, 70, "Input",
 CellID->93900919],
Cell[373207, 8781, 10513, 176, 70, "Output",
 Evaluatable->False,
 CellID->220154982]
}, Open  ]],
Cell[383735, 8960, 70, 1, 70, "ExampleText",
 CellID->63035132],
Cell[CellGroupData[{
Cell[383830, 8965, 512, 15, 70, "Input",
 CellID->452718695],
Cell[384345, 8982, 20387, 338, 70, "Output",
 Evaluatable->False,
 CellID->861775330]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[404781, 9326, 233, 7, 70, "ExampleSubsection",
 CellID->421266211],
Cell[405017, 9335, 146, 4, 70, "ExampleText",
 CellID->431102374],
Cell[CellGroupData[{
Cell[405188, 9343, 615, 20, 70, "Input",
 CellID->149718405],
Cell[405806, 9365, 428, 8, 70, "Message",
 CellID->23795732],
Cell[406237, 9375, 428, 8, 70, "Message",
 CellID->160434433],
Cell[406668, 9385, 828, 29, 54, "Output",
 CellID->63102960]
}, Open  ]],
Cell[407511, 9417, 120, 3, 70, "ExampleText",
 CellID->907901123],
Cell[CellGroupData[{
Cell[407656, 9424, 673, 21, 70, "Input",
 CellID->68376451],
Cell[408332, 9447, 482, 17, 36, "Output",
 CellID->760797536]
}, Open  ]],
Cell[CellGroupData[{
Cell[408851, 9469, 327, 10, 70, "Input",
 CellID->86376055],
Cell[409181, 9481, 4909, 85, 70, "Output",
 Evaluatable->False,
 CellID->23126441]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[414139, 9572, 237, 7, 70, "ExampleSubsection",
 CellID->274331389],
Cell[414379, 9581, 131, 4, 70, "ExampleText",
 CellID->585109129],
Cell[CellGroupData[{
Cell[414535, 9589, 986, 30, 70, "Input",
 CellID->54109572],
Cell[415524, 9621, 470, 17, 36, "Output",
 CellID->184329392]
}, Open  ]],
Cell[CellGroupData[{
Cell[416031, 9643, 344, 11, 70, "Input",
 CellID->748158840],
Cell[416378, 9656, 158, 5, 36, "Output",
 CellID->2312514]
}, Open  ]],
Cell[416551, 9664, 143, 4, 70, "ExampleText",
 CellID->131514743],
Cell[CellGroupData[{
Cell[416719, 9672, 1046, 30, 70, "Input",
 CellID->273984165],
Cell[417768, 9704, 469, 17, 36, "Output",
 CellID->85900149]
}, Open  ]],
Cell[CellGroupData[{
Cell[418274, 9726, 524, 17, 70, "Input",
 CellID->108070529],
Cell[418801, 9745, 10233, 172, 70, "Output",
 Evaluatable->False,
 CellID->322912500]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[429083, 9923, 228, 7, 70, "ExampleSubsection",
 CellID->26605],
Cell[429314, 9932, 127, 4, 70, "ExampleText",
 CellID->26806],
Cell[CellGroupData[{
Cell[429466, 9940, 913, 29, 70, "Input",
 CellID->17438],
Cell[430382, 9971, 2791, 50, 70, "Output",
 Evaluatable->False,
 CellID->272758055]
}, Open  ]],
Cell[433188, 10024, 121, 3, 70, "ExampleDelimiter",
 CellID->27518],
Cell[433312, 10029, 94, 1, 70, "ExampleText",
 CellID->28262],
Cell[CellGroupData[{
Cell[433431, 10034, 784, 25, 70, "Input",
 CellID->27944],
Cell[434218, 10061, 160, 5, 36, "Output",
 CellID->170342971]
}, Open  ]],
Cell[434393, 10069, 121, 3, 70, "ExampleDelimiter",
 CellID->17015],
Cell[434517, 10074, 174, 6, 70, "ExampleText",
 CellID->22893],
Cell[CellGroupData[{
Cell[434716, 10084, 874, 26, 70, "Input",
 CellID->19323],
Cell[435593, 10112, 2187, 44, 135, "Output",
 CellID->369956043]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[437829, 10162, 237, 7, 70, "ExampleSubsection",
 CellID->386402526],
Cell[438069, 10171, 114, 3, 70, "ExampleText",
 CellID->12825289],
Cell[CellGroupData[{
Cell[438208, 10178, 903, 28, 70, "Input",
 CellID->458275453],
Cell[439114, 10208, 289, 9, 39, "Output",
 CellID->328871731]
}, Open  ]],
Cell[439418, 10220, 144, 4, 70, "ExampleText",
 CellID->61400925],
Cell[CellGroupData[{
Cell[439587, 10228, 895, 26, 70, "Input",
 CellID->166953977],
Cell[440485, 10256, 334, 11, 39, "Output",
 CellID->454478345]
}, Open  ]],
Cell[440834, 10270, 194, 5, 70, "ExampleText",
 CellID->534795734],
Cell[CellGroupData[{
Cell[441053, 10279, 887, 26, 70, "Input",
 CellID->49257934],
Cell[441943, 10307, 332, 11, 39, "Output",
 CellID->190981274]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[442336, 10325, 222, 7, 70, "ExampleSection",
 CellID->4218],
Cell[CellGroupData[{
Cell[442583, 10336, 233, 7, 70, "ExampleSubsection",
 CellID->25369],
Cell[CellGroupData[{
Cell[442841, 10347, 836, 28, 70, "Input",
 CellID->30472],
Cell[443680, 10377, 467, 17, 36, "Output",
 CellID->189090723]
}, Open  ]],
Cell[CellGroupData[{
Cell[444184, 10399, 272, 9, 70, "Input",
 CellID->27161],
Cell[444459, 10410, 4613, 80, 70, "Output",
 Evaluatable->False,
 CellID->415242567]
}, Open  ]],
Cell[449087, 10493, 124, 3, 70, "ExampleDelimiter",
 CellID->11135042],
Cell[449214, 10498, 95, 1, 70, "ExampleText",
 CellID->126108385],
Cell[CellGroupData[{
Cell[449334, 10503, 851, 28, 70, "Input",
 CellID->27825],
Cell[450188, 10533, 467, 17, 36, "Output",
 CellID->387525007]
}, Open  ]],
Cell[CellGroupData[{
Cell[450692, 10555, 271, 9, 70, "Input",
 CellID->1494],
Cell[450966, 10566, 4141, 72, 70, "Output",
 Evaluatable->False,
 CellID->381498695]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[455156, 10644, 247, 7, 70, "ExampleSubsection",
 CellID->24195],
Cell[455406, 10653, 258, 8, 70, "ExampleText",
 CellID->27796781],
Cell[CellGroupData[{
Cell[455689, 10665, 916, 30, 70, "Input",
 CellID->101383601],
Cell[456608, 10697, 773, 28, 54, "Output",
 CellID->16513986]
}, Open  ]],
Cell[457396, 10728, 60, 1, 70, "ExampleText",
 CellID->149801889],
Cell[CellGroupData[{
Cell[457481, 10733, 408, 13, 70, "Input",
 CellID->25831712],
Cell[457892, 10748, 4461, 77, 70, "Output",
 Evaluatable->False,
 CellID->55907042]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[462402, 10831, 232, 7, 70, "ExampleSubsection",
 CellID->3830],
Cell[462637, 10840, 222, 8, 70, "ExampleText",
 CellID->11628256],
Cell[CellGroupData[{
Cell[462884, 10852, 1625, 48, 70, "Input",
 CellID->16549],
Cell[464512, 10902, 1060, 38, 71, "Output",
 CellID->405630080]
}, Open  ]],
Cell[CellGroupData[{
Cell[465609, 10945, 595, 18, 70, "Input",
 CellID->1701],
Cell[466207, 10965, 12713, 212, 70, "Output",
 Evaluatable->False,
 CellID->188297404]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[478969, 11183, 230, 7, 70, "ExampleSubsection",
 CellID->18561],
Cell[479202, 11192, 202, 6, 70, "ExampleText",
 CellID->1002821242],
Cell[CellGroupData[{
Cell[479429, 11202, 914, 29, 70, "Input",
 CellID->562818297],
Cell[480346, 11233, 562, 20, 36, "Output",
 CellID->38553498]
}, Open  ]],
Cell[CellGroupData[{
Cell[480945, 11258, 429, 13, 70, "Input",
 CellID->811388219],
Cell[481377, 11273, 18736, 311, 70, "Output",
 Evaluatable->False,
 CellID->408728140]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[500162, 11590, 230, 7, 70, "ExampleSubsection",
 CellID->13303],
Cell[500395, 11599, 97, 1, 70, "ExampleText",
 CellID->3422311],
Cell[CellGroupData[{
Cell[500517, 11604, 1370, 44, 70, "Input",
 CellID->14992],
Cell[501890, 11650, 593, 21, 36, "Output",
 CellID->239769635]
}, Open  ]],
Cell[502498, 11674, 61, 1, 70, "ExampleText",
 CellID->513315368],
Cell[CellGroupData[{
Cell[502584, 11679, 512, 16, 70, "Input",
 CellID->28969],
Cell[503099, 11697, 29025, 480, 70, "Output",
 Evaluatable->False,
 CellID->256349595]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[532173, 12183, 254, 7, 70, "ExampleSubsection",
 CellID->537312911],
Cell[532430, 12192, 236, 8, 70, "ExampleText",
 CellID->6561],
Cell[CellGroupData[{
Cell[532691, 12204, 1762, 56, 70, "Input",
 CellID->2651],
Cell[534456, 12262, 593, 21, 36, "Output",
 CellID->503219777]
}, Open  ]],
Cell[CellGroupData[{
Cell[535086, 12288, 418, 14, 70, "Input",
 CellID->32677],
Cell[535507, 12304, 23547, 390, 70, "Output",
 Evaluatable->False,
 CellID->105104449]
}, Open  ]],
Cell[CellGroupData[{
Cell[559091, 12699, 472, 16, 70, "Input",
 CellID->22804650],
Cell[559566, 12717, 35466, 585, 70, "Output",
 Evaluatable->False,
 CellID->174509184]
}, Open  ]],
Cell[595047, 13305, 125, 3, 70, "ExampleDelimiter",
 CellID->246788166],
Cell[595175, 13310, 174, 6, 70, "ExampleText",
 CellID->43447455],
Cell[CellGroupData[{
Cell[595374, 13320, 2333, 74, 70, "Input",
 CellID->6849],
Cell[597710, 13396, 692, 24, 36, "Output",
 CellID->111557589]
}, Open  ]],
Cell[CellGroupData[{
Cell[598439, 13425, 634, 20, 70, "Input",
 CellID->13912],
Cell[599076, 13447, 70668, 1163, 70, "Output",
 Evaluatable->False,
 CellID->548876126]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[669793, 14616, 273, 9, 70, "ExampleSubsection",
 CellID->5714],
Cell[670069, 14627, 223, 8, 70, "ExampleText",
 CellID->5371],
Cell[670295, 14637, 778, 26, 70, "Input",
 CellID->16808],
Cell[CellGroupData[{
Cell[671098, 14667, 397, 14, 70, "Input",
 CellID->1620],
Cell[671498, 14683, 467, 17, 36, "Output",
 CellID->363241442]
}, Open  ]],
Cell[CellGroupData[{
Cell[672002, 14705, 272, 9, 70, "Input",
 CellID->13156],
Cell[672277, 14716, 5040, 87, 70, "Output",
 Evaluatable->False,
 CellID->419229927]
}, Open  ]],
Cell[677332, 14806, 115, 3, 70, "ExampleText",
 CellID->30149],
Cell[677450, 14811, 921, 31, 70, "Input",
 CellID->228004849],
Cell[CellGroupData[{
Cell[678396, 14846, 666, 20, 70, "Input",
 CellID->4469050],
Cell[679065, 14868, 563, 20, 36, "Output",
 CellID->327664038]
}, Open  ]],
Cell[CellGroupData[{
Cell[679665, 14893, 385, 12, 70, "Input",
 CellID->538369189],
Cell[680053, 14907, 24051, 398, 70, "Output",
 Evaluatable->False,
 CellID->279324866]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[704153, 15311, 263, 7, 70, "ExampleSubsection",
 CellID->467881331],
Cell[704419, 15320, 95, 1, 70, "ExampleText",
 CellID->237473526],
Cell[CellGroupData[{
Cell[704539, 15325, 1657, 51, 70, "Input",
 CellID->244609007],
Cell[706199, 15378, 593, 21, 36, "Output",
 CellID->542382761]
}, Open  ]],
Cell[CellGroupData[{
Cell[706829, 15404, 615, 18, 70, "Input",
 CellID->270329133],
Cell[707447, 15424, 23380, 387, 70, "Output",
 Evaluatable->False,
 CellID->267816204]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[730888, 15818, 233, 7, 70, "ExampleSection",
 CellID->26392],
Cell[731124, 15827, 95, 1, 70, "ExampleText",
 CellID->13531],
Cell[CellGroupData[{
Cell[731244, 15832, 488, 17, 70, "Input",
 CellID->10204],
Cell[731735, 15851, 868, 28, 51, "Output",
 CellID->275648464]
}, Open  ]],
Cell[CellGroupData[{
Cell[732640, 15884, 549, 19, 70, "Input",
 CellID->27294],
Cell[733192, 15905, 466, 17, 36, "Output",
 CellID->59230442]
}, Open  ]],
Cell[CellGroupData[{
Cell[733695, 15927, 272, 9, 70, "Input",
 CellID->19149],
Cell[733970, 15938, 3141, 56, 70, "Output",
 Evaluatable->False,
 CellID->534324774]
}, Open  ]],
Cell[737126, 15997, 121, 3, 70, "ExampleDelimiter",
 CellID->19634],
Cell[737250, 16002, 205, 8, 70, "ExampleText",
 CellID->25455],
Cell[CellGroupData[{
Cell[737480, 16014, 722, 25, 70, "Input",
 CellID->7544],
Cell[738205, 16041, 798, 29, 54, "Output",
 CellID->946039128]
}, Open  ]],
Cell[CellGroupData[{
Cell[739040, 16075, 273, 9, 70, "Input",
 CellID->30708],
Cell[739316, 16086, 2941, 52, 70, "Output",
 Evaluatable->False,
 CellID->149718147]
}, Open  ]],
Cell[CellGroupData[{
Cell[742294, 16143, 270, 9, 70, "Input",
 CellID->78234261],
Cell[742567, 16154, 2094, 43, 138, "Output",
 CellID->111514838]
}, Open  ]],
Cell[744676, 16200, 123, 3, 70, "ExampleDelimiter",
 CellID->5202541],
Cell[744802, 16205, 130, 4, 70, "ExampleText",
 CellID->455313870],
Cell[CellGroupData[{
Cell[744957, 16213, 526, 17, 70, "Input",
 CellID->140835562],
Cell[745486, 16232, 1205, 33, 71, "Output",
 CellID->150891179]
}, Open  ]],
Cell[746706, 16268, 265, 8, 70, "ExampleText",
 CellID->788121558],
Cell[CellGroupData[{
Cell[746996, 16280, 559, 17, 70, "Input",
 CellID->200296671],
Cell[747558, 16299, 466, 17, 36, "Output",
 CellID->232299251]
}, Open  ]],
Cell[CellGroupData[{
Cell[748061, 16321, 410, 12, 70, "Input",
 CellID->137449308],
Cell[748474, 16335, 2787, 50, 70, "Output",
 Evaluatable->False,
 CellID->622249486]
}, Open  ]],
Cell[751276, 16388, 125, 3, 70, "ExampleDelimiter",
 CellID->176340561],
Cell[751404, 16393, 127, 3, 70, "ExampleText",
 CellID->233343035],
Cell[751534, 16398, 866, 27, 70, "Input",
 CellID->39932360],
Cell[CellGroupData[{
Cell[752425, 16429, 217, 7, 70, "Input",
 CellID->363650466],
Cell[752645, 16438, 2282, 41, 70, "Output",
 Evaluatable->False,
 CellID->423019911]
}, Open  ]],
Cell[CellGroupData[{
Cell[754964, 16484, 206, 7, 70, "Input",
 CellID->561096089],
Cell[755173, 16493, 225, 7, 36, "Output",
 CellID->47187379]
}, Open  ]],
Cell[755413, 16503, 102, 2, 70, "ExampleText",
 CellID->508676218],
Cell[CellGroupData[{
Cell[755540, 16509, 827, 25, 70, "Input",
 CellID->194334875],
Cell[756370, 16536, 483, 17, 36, "Output",
 CellID->201919792]
}, Open  ]],
Cell[756868, 16556, 125, 3, 70, "ExampleDelimiter",
 CellID->155559896],
Cell[756996, 16561, 362, 14, 70, "ExampleText",
 CellID->456027276],
Cell[757361, 16577, 932, 29, 70, "Input",
 CellID->170598371],
Cell[CellGroupData[{
Cell[758318, 16610, 359, 13, 70, "Input",
 CellID->495578518],
Cell[758680, 16625, 394, 8, 70, "Message",
 CellID->886134120],
Cell[759077, 16635, 394, 8, 70, "Message",
 CellID->381763645],
Cell[759474, 16645, 394, 8, 70, "Message",
 CellID->754038167],
Cell[759871, 16655, 419, 8, 70, "Message",
 CellID->660494860],
Cell[760293, 16665, 2368, 43, 70, "Output",
 Evaluatable->False,
 CellID->249686577]
}, Open  ]],
Cell[762676, 16711, 332, 13, 70, "ExampleText",
 CellID->685856719],
Cell[CellGroupData[{
Cell[763033, 16728, 399, 15, 70, "Input",
 CellID->279554871],
Cell[763435, 16745, 331, 11, 36, "Output",
 CellID->77403466]
}, Open  ]],
Cell[763781, 16759, 87, 1, 70, "ExampleText",
 CellID->344144216],
Cell[CellGroupData[{
Cell[763893, 16764, 971, 29, 70, "Input",
 CellID->240243937],
Cell[764867, 16795, 3904, 68, 70, "Output",
 Evaluatable->False,
 CellID->98555213]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[768820, 16869, 226, 7, 70, "ExampleSection",
 CellID->32581],
Cell[CellGroupData[{
Cell[769071, 16880, 232, 7, 70, "ExampleSubsection",
 CellID->31598],
Cell[769306, 16889, 119, 3, 70, "ExampleText",
 CellID->21845],
Cell[CellGroupData[{
Cell[769450, 16896, 542, 19, 70, "Input",
 CellID->13675],
Cell[769995, 16917, 466, 17, 36, "Output",
 CellID->3607412]
}, Open  ]],
Cell[770476, 16937, 99, 2, 70, "ExampleText",
 CellID->31584],
Cell[CellGroupData[{
Cell[770600, 16943, 333, 11, 70, "Input",
 CellID->6100],
Cell[770936, 16956, 5429, 93, 70, "Output",
 Evaluatable->False,
 CellID->263792847]
}, Open  ]],
Cell[776380, 17052, 121, 3, 70, "ExampleDelimiter",
 CellID->21567],
Cell[776504, 17057, 70, 1, 70, "ExampleText",
 CellID->32297],
Cell[CellGroupData[{
Cell[776599, 17062, 687, 24, 70, "Input",
 CellID->26009],
Cell[777289, 17088, 467, 17, 36, "Output",
 CellID->349627463]
}, Open  ]],
Cell[CellGroupData[{
Cell[777793, 17110, 395, 13, 70, "Input",
 CellID->26231],
Cell[778191, 17125, 2402, 43, 70, "Output",
 Evaluatable->False,
 CellID->95730909]
}, Open  ]],
Cell[780608, 17171, 124, 3, 70, "ExampleDelimiter",
 CellID->73370311],
Cell[780735, 17176, 138, 4, 70, "ExampleText",
 CellID->330352906],
Cell[CellGroupData[{
Cell[780898, 17184, 936, 30, 70, "Input",
 CellID->104587127],
Cell[781837, 17216, 467, 17, 36, "Output",
 CellID->184574004]
}, Open  ]],
Cell[CellGroupData[{
Cell[782341, 17238, 609, 20, 70, "Input",
 CellID->722335240],
Cell[782953, 17260, 4945, 85, 70, "Output",
 Evaluatable->False,
 CellID->232144355]
}, Open  ]],
Cell[787913, 17348, 142, 4, 70, "ExampleText",
 CellID->329570386],
Cell[CellGroupData[{
Cell[788080, 17356, 1003, 31, 70, "Input",
 CellID->160838699],
Cell[789086, 17389, 466, 17, 36, "Output",
 CellID->28573644]
}, Open  ]],
Cell[CellGroupData[{
Cell[789589, 17411, 610, 20, 70, "Input",
 CellID->26199686],
Cell[790202, 17433, 2965, 53, 70, "Output",
 Evaluatable->False,
 CellID->468230346]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[793216, 17492, 253, 7, 70, "ExampleSubsection",
 CellID->522396580],
Cell[793472, 17501, 96, 1, 70, "ExampleText",
 CellID->120943363],
Cell[793571, 17504, 1090, 37, 70, "Input",
 CellID->782754975],
Cell[794664, 17543, 310, 12, 70, "ExampleText",
 CellID->51816239],
Cell[CellGroupData[{
Cell[794999, 17559, 563, 19, 70, "Input",
 CellID->189859804],
Cell[795565, 17580, 1129, 41, 71, "Output",
 CellID->414809325]
}, Open  ]],
Cell[796709, 17624, 472, 16, 70, "ExampleText",
 CellID->435729729],
Cell[CellGroupData[{
Cell[797206, 17644, 563, 19, 70, "Input",
 CellID->239796030],
Cell[797772, 17665, 398, 8, 70, "Message",
 CellID->917131994],
Cell[798173, 17675, 1129, 41, 71, "Output",
 CellID->375451713]
}, Open  ]],
Cell[799317, 17719, 461, 16, 70, "ExampleText",
 CellID->9706804],
Cell[CellGroupData[{
Cell[799803, 17739, 792, 26, 70, "Input",
 CellID->137638465],
Cell[800598, 17767, 398, 8, 70, "Message",
 CellID->150540322],
Cell[800999, 17777, 1128, 41, 71, "Output",
 CellID->45753838]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[802176, 17824, 251, 7, 70, "ExampleSubsection",
 CellID->190271123],
Cell[802430, 17833, 66, 1, 70, "ExampleText",
 CellID->100790804],
Cell[802499, 17836, 1569, 52, 70, "Input",
 CellID->170657928],
Cell[804071, 17890, 150, 4, 70, "ExampleText",
 CellID->438720576],
Cell[CellGroupData[{
Cell[804246, 17898, 318, 10, 70, "Input",
 CellID->981350694],
Cell[804567, 17910, 776, 13, 70, "Message",
 CellID->149935379],
Cell[805346, 17925, 591, 21, 36, "Output",
 CellID->8789718]
}, Open  ]],
Cell[805952, 17949, 152, 4, 70, "ExampleText",
 CellID->27919707],
Cell[CellGroupData[{
Cell[806129, 17957, 655, 18, 70, "Input",
 CellID->774659245],
Cell[806787, 17977, 593, 21, 36, "Output",
 CellID->387096426]
}, Open  ]],
Cell[807395, 18001, 120, 3, 70, "ExampleText",
 CellID->210550763],
Cell[CellGroupData[{
Cell[807540, 18008, 530, 17, 70, "Input",
 CellID->1063819358],
Cell[808073, 18027, 35218, 581, 70, "Output",
 Evaluatable->False,
 CellID->218268380]
}, Open  ]],
Cell[843306, 18611, 125, 3, 70, "ExampleDelimiter",
 CellID->563384740],
Cell[843434, 18616, 120, 3, 70, "ExampleText",
 CellID->627218989],
Cell[843557, 18621, 936, 31, 70, "Input",
 CellID->107661612],
Cell[844496, 18654, 132, 4, 70, "ExampleText",
 CellID->65725047],
Cell[CellGroupData[{
Cell[844653, 18662, 623, 18, 70, "Input",
 CellID->265407733],
Cell[845279, 18682, 464, 8, 70, "Message",
 CellID->355775486],
Cell[845746, 18692, 13900, 232, 70, "Output",
 Evaluatable->False,
 CellID->22079881]
}, Open  ]],
Cell[859661, 18927, 145, 4, 70, "ExampleText",
 CellID->328051666],
Cell[CellGroupData[{
Cell[859831, 18935, 987, 26, 70, "Input",
 CellID->43414940],
Cell[860821, 18963, 462, 8, 70, "Message",
 CellID->134873555],
Cell[861286, 18973, 14116, 235, 70, "Output",
 Evaluatable->False,
 CellID->222764338]
}, Open  ]],
Cell[875417, 19211, 125, 3, 70, "ExampleDelimiter",
 CellID->106183979],
Cell[875545, 19216, 89, 1, 70, "ExampleText",
 CellID->128702275],
Cell[875637, 19219, 1142, 36, 70, "Input",
 CellID->117120600],
Cell[876782, 19257, 135, 4, 70, "ExampleText",
 CellID->309144532],
Cell[CellGroupData[{
Cell[876942, 19265, 312, 9, 70, "Input",
 CellID->120870374],
Cell[877257, 19276, 772, 13, 70, "Message",
 CellID->60679217],
Cell[878032, 19291, 562, 20, 36, "Output",
 CellID->298218024]
}, Open  ]],
Cell[878609, 19314, 94, 1, 70, "ExampleText",
 CellID->16354085],
Cell[CellGroupData[{
Cell[878728, 19319, 464, 14, 70, "Input",
 CellID->702053895],
Cell[879195, 19335, 5563, 95, 70, "Output",
 Evaluatable->False,
 CellID->16252919]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[884807, 19436, 244, 7, 70, "ExampleSubsection",
 CellID->565542508],
Cell[885054, 19445, 103, 2, 70, "ExampleText",
 CellID->237261224],
Cell[CellGroupData[{
Cell[885182, 19451, 969, 29, 70, "Input",
 CellID->4180016],
Cell[886154, 19482, 790, 17, 70, "Output",
 Evaluatable->False,
 CellID->234289763]
}, Open  ]],
Cell[886959, 19502, 134, 4, 70, "ExampleText",
 CellID->6166462],
Cell[CellGroupData[{
Cell[887118, 19510, 1673, 46, 70, "Input",
 CellID->49683823],
Cell[888794, 19558, 7161, 121, 70, "Output",
 Evaluatable->False,
 CellID->17914711]
}, Open  ]]
}, Closed]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[896028, 19687, 310, 9, 70, "SeeAlsoSection",
 CellID->6122],
Cell[896341, 19698, 692, 26, 70, "SeeAlso",
 CellID->12782]
}, Open  ]],
Cell[CellGroupData[{
Cell[897070, 19729, 314, 9, 70, "TutorialsSection",
 CellID->19724],
Cell[897387, 19740, 176, 3, 70, "Tutorials",
 CellID->81285119],
Cell[897566, 19745, 198, 4, 70, "Tutorials",
 CellID->25233],
Cell[897767, 19751, 318, 9, 70, "Tutorials",
 CellID->64016169],
Cell[898088, 19762, 324, 9, 70, "Tutorials",
 CellID->26634]
}, Open  ]],
Cell[CellGroupData[{
Cell[898449, 19776, 320, 9, 70, "RelatedLinksSection",
 CellID->5603],
Cell[898772, 19787, 359, 12, 70, "RelatedLinks"],
Cell[899134, 19801, 266, 6, 70, "RelatedLinks",
 CellID->17125],
Cell[899403, 19809, 385, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[899825, 19827, 317, 9, 70, "MoreAboutSection",
 CellID->3638660],
Cell[900145, 19838, 128, 3, 70, "MoreAbout",
 CellID->254891512],
Cell[900276, 19843, 155, 3, 70, "MoreAbout",
 CellID->388269326],
Cell[900434, 19848, 142, 3, 70, "MoreAbout",
 CellID->30841973],
Cell[900579, 19853, 175, 3, 70, "MoreAbout",
 CellID->919461540],
Cell[900757, 19858, 179, 3, 70, "MoreAbout",
 CellID->294875945]
}, Open  ]],
Cell[900951, 19864, 50, 0, 70, "History"],
Cell[901004, 19866, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

