(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    208943,       6212]
NotebookOptionsPosition[    181029,       5271]
NotebookOutlinePosition[    185207,       5375]
CellTagsIndexPosition[    185036,       5367]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Advanced Numerical Integration in Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/NIntegrateOverview"], 
          "Numerical Mathematics: Basic Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalMathematics-BasicOperations"], 
          "Numerical Integration" :> 
          Documentation`HelpLookup["paclet:tutorial/NumericalIntegration"], 
          "Numerical Sums, Products and Integrals" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalSumsProductsAndIntegrals"], 
          "Numerical Mathematics in Mathematica" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalMathematicsInMathematica"], 
          "The Uncertainties of Numerical Mathematics" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/TheUncertaintiesOfNumericalMathematics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Advanced Numerical Integration in Mathematica\"\>", 
       2->"\<\"Numerical Mathematics: Basic Operations\"\>", 
       3->"\<\"Numerical Integration\"\>", 
       4->"\<\"Numerical Sums, Products and Integrals\"\>", 
       5->"\<\"Numerical Mathematics in Mathematica\"\>", 
       6->"\<\"The Uncertainties of Numerical Mathematics\"\>"}, \
"\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"NDSolve" :> Documentation`HelpLookup["paclet:ref/NDSolve"], 
          "NSum" :> Documentation`HelpLookup["paclet:ref/NSum"], "Integrate" :> 
          Documentation`HelpLookup["paclet:ref/Integrate"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"NDSolve\"\>", 2->"\<\"NSum\"\>", 
       3->"\<\"Integrate\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Calculus" :> Documentation`HelpLookup["paclet:guide/Calculus"],
           "Functions of Complex Variables" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsOfComplexVariables"], 
          "Numerical Evaluation & Precision" :> 
          Documentation`HelpLookup[
           "paclet:guide/NumericalEvaluationAndPrecision"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Calculus\"\>", 
       2->"\<\"Functions of Complex Variables\"\>", 
       3->"\<\"Numerical Evaluation & Precision\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["NIntegrate", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NIntegrate",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NIntegrate"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a numerical approximation to the integral ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubsuperscriptBox["\[Integral]", 
         SubscriptBox["x", 
          StyleBox["min",
           FontSlant->"Italic"]], 
         SubscriptBox["x", 
          StyleBox["max",
           FontSlant->"Italic"]]], " ", 
        RowBox[{"f", " ", "d", "\[InvisibleSpace]", "x"}]}], 
       TraditionalForm]], "InlineMath"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NIntegrate",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NIntegrate"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["max", "TI"]]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["y", "TI"], ",", 
          SubscriptBox[
           StyleBox["y", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["y", "TI"], 
           StyleBox["max", "TI"]]}], "}"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives a numerical approximation to the multiple \
integral ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubsuperscriptBox["\[Integral]", 
         SubscriptBox["x", 
          StyleBox["min",
           FontSlant->"Italic"]], 
         SubscriptBox["x", 
          StyleBox["max",
           FontSlant->"Italic"]]], 
        RowBox[{"d", "x", 
         RowBox[{
          SubsuperscriptBox["\[Integral]", 
           SubscriptBox["y", 
            StyleBox["min",
             FontSlant->"Italic"]], 
           SubscriptBox["y", 
            StyleBox["max",
             FontSlant->"Italic"]]], 
          RowBox[{
          "d", "\[InvisibleSpace]", "y", " ", "\[Ellipsis]", " ", "f"}]}]}]}],
        TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->30088]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Multiple integrals use a variant of the standard iterator notation. The first \
variable given corresponds to the outermost integral, and is done last.\
\>", "Notes",
 CellID->3224718],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " by default tests for singularities at the boundaries of the integration \
region, and at the boundaries of regions specified by settings for the ",
 Cell[BoxData[
  ButtonBox["Exclusions",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exclusions"]], "InlineFormula"],
 " option."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->18461],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["NIntegrate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NIntegrate"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["k", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
 " tests for singularities in a one-dimensional integral at each of the \
intermediate points ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 ". If there are no singularities, the result is equivalent to an integral \
from ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["0", "TR"]]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["k", "TI"]]], "InlineFormula"],
 ". You can use complex numbers ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["x", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " to specify an integration contour in the complex plane. "
}], "Notes",
 CellID->19295],

Cell["The following options can be given: ", "Notes",
 CellID->32652],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Infinity",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Infinity"], Cell[
    "digits of absolute accuracy sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["EvaluationMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/EvaluationMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "expression to evaluate whenever ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " is evaluated "
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    ButtonBox["Exclusions",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Exclusions"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[
    "parts of the integration region to exclude", "TableText"]},
   {Cell["      ", "TableRowIcon"], "MaxPoints", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum total number of sample points ", "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    ButtonBox["MaxRecursion",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxRecursion"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum number of recursive subdivisions ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method to use ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "MinRecursion", "0", Cell[
    "minimum number of recursive subdivisions ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "digits of precision sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "the precision used in internal computations ", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "3ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, None, 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], None, 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True, True}}}},
 CellID->23643],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " usually uses adaptive algorithms, which recursively subdivide the \
integration region as needed. ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " specifies the minimum number of recursive subdivisions to try. ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " gives the maximum number. "
}], "Notes",
 CellID->90],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " usually continues doing subdivisions until the error estimate it gets \
implies that the final result achieves either the ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " or the ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " specified. "
}], "Notes",
 CellID->12747],

Cell[TextData[{
 "For low-dimensional integrals, the default setting for ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " is related to ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 ". For high-dimensional integrals, it is typically taken to be a fixed \
value, usually ",
 Cell[BoxData["2"], "InlineFormula"],
 ". "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->536560],

Cell[TextData[{
 "You should realize that with sufficiently pathological functions, the \
algorithms used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can give wrong answers. In most cases, you can test the answer by looking \
at its sensitivity to changes in the setting of options for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->32525],

Cell[TextData[{
 "\[FilledSmallSquare] Possible explicit settings for the ",
 ButtonBox["Method",
  BaseStyle->"Link",
  ButtonData->"paclet:ref/Method"],
 " option include:"
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellTags->{"S0.1651", "NDSolve"},
 CellID->306876545],

Cell[BoxData[GridBox[{
   {"\"\<GlobalAdaptive\>\"", Cell[
    "global adaptive integration strategy", "TableText"]},
   {"\"\<LocalAdaptive\>\"", Cell[
    "local adaptive integration strategy", "TableText"]},
   {"\"\<DoubleExponential\>\"", Cell[
    "double exponential quadrature", "TableText"]},
   {"\"\<MonteCarlo\>\"", Cell["Monte Carlo integration", "TableText"]},
   {"\"\<AdaptiveMonteCarlo\>\"", Cell[
    "adaptive Monte Carlo integration", "TableText"]},
   {"\"\<QuasiMonteCarlo\>\"", Cell[
    "quasi Monte Carlo integration", "TableText"]},
   {"\"\<AdaptiveQuasiMonteCarlo\>\"", Cell[
    "adaptive quasi Monte Carlo integration", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
    "RowsIndexed" -> {}}]], "2ColumnTable",
 CellTags->{"NDSolve", "S0.1651"},
 CellID->629005005],

Cell[TextData[{
 "\[FilledSmallSquare] With ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   RowBox[{"{", 
    RowBox[{"\"\<\!\(\*
StyleBox[\"strategy\", \"TI\"]\)\>\"", ",", 
     RowBox[{
      ButtonBox["Method",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Method"], "->", "\"\<\!\(\*
StyleBox[\"rule\", \"TI\"]\)\>\""}]}], "}"}]}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   RowBox[{"{", 
    RowBox[{"\"\<\!\(\*
StyleBox[\"strategy\", \"TI\"]\)\>\"", ",", 
     RowBox[{
      ButtonBox["Method",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Method"], "->", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["rule", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["rule", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]}]}], "}"}]}]], 
  "InlineFormula"],
 " possible strategy methods include: "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellTags->{"S0.1651", "NDSolve"},
 CellID->12228510],

Cell[BoxData[GridBox[{
   {"\"\<GlobalAdaptive\>\"", Cell[
    "subdivide based on global error estimates", "TableText"]},
   {"\"\<LocalAdaptive\>\"", Cell[
    "subdivide based only on local error estimates", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
    "RowsIndexed" -> {}}]], "2ColumnTable",
 CellTags->{"NDSolve", "S0.1651"},
 CellID->604156095],

Cell["\<\
\[FilledSmallSquare] Methods used as rules include: \
\>", "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellTags->{"S0.1651", "NDSolve"},
 CellID->100636363],

Cell[BoxData[GridBox[{
   {"\"\<CartesianRule\>\"", Cell[
    "multidimensional Cartesian product of rules", "TableText"]},
   {"\"\<ClenshawCurtisRule\>\"", Cell[
    "Clenshaw\[Dash]Curtis rule", "TableText"]},
   {"\"\<GaussKronrodRule\>\"", Cell[
    "Gauss points with Kronrod extension", "TableText"]},
   {"\"\<LobattoKronrodRule\>\"", Cell["\<\
Gauss\[Dash]Lobatto points with Kronrod extension\
\>", "TableText"]},
   {"\"\<MultidimensionalRule\>\"", Cell[
    "multidimensional symmetric rule", "TableText"]},
   {"\"\<MultipanelRule\>\"", Cell["combination of 1D rules", "TableText"]},
   {"\"\<NewtonCotesRule\>\"", Cell["Newton\[Dash]Cotes rule", "TableText"]},
   {"\"\<TrapezoidalRule\>\"", Cell[
    "uniform points in one dimension", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
    "RowsIndexed" -> {}}]], "2ColumnTable",
 CellTags->{"NDSolve", "S0.1651"},
 CellID->578720793],

Cell[TextData[{
 "Additional method suboptions can be given in the form ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   RowBox[{"{", 
    RowBox[{
     StyleBox["\[Ellipsis]", "TR"], ",", 
     StyleBox["opts", "TI"]}], "}"}]}]], "InlineFormula"],
 "."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->94879207],

Cell[TextData[{
 "The method suboption ",
 Cell[BoxData["\"\<SymbolicProcessing\>\""], "InlineFormula"],
 " specifies the maximum number of seconds for which to attempt performing \
symbolic analysis of the integrand. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->272103621],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "[", 
   RowBox[{
    ButtonBox["Integrate",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Integrate"], "[", 
    StyleBox["\[Ellipsis]", "TR"], "]"}], "]"}]], "InlineFormula"],
 " calls ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " for integrals that cannot be done symbolically. "
}], "Notes",
 CellID->15953],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " first localizes the values of all variables, then evaluates ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " with the variables being symbolic, and then repeatedly evaluates the \
result numerically."
}], "Notes",
 CellID->21792],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " has attribute ",
 Cell[BoxData[
  ButtonBox["HoldAll",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HoldAll"]], "InlineFormula"],
 ", and effectively uses ",
 Cell[BoxData[
  ButtonBox["Block",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Block"]], "InlineFormula"],
 " to localize variables."
}], "Notes",
 CellID->12110]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->83853679],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->150599599],

Cell["Compute a numerical integral:", "ExampleText",
 CellID->1875922869],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"Sin", "[", "x", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1692469869],

Cell[BoxData["1.2470560582440045`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->392329368]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->48604888],

Cell["\<\
Compute a multi dimensional integral (with singularity at the origin):\
\>", "ExampleText",
 CellID->800751445],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    SqrtBox[
     RowBox[{
      FractionBox["x1", "3"], "+", 
      FractionBox["x2", "2"], "+", 
      FractionBox["x3", "2"], "+", 
      FractionBox["x4", "10"]}]]], ",", 
   RowBox[{"{", 
    RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x2", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x3", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x4", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->888011411],

Cell[BoxData["1.2391040468429684`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25014712]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2155493],

Cell["Compute a high dimensional integral:", "ExampleText",
 CellID->1631390226],

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[Xi]", "=", 
   RowBox[{"Array", "[", 
    RowBox[{"x", ",", 
     RowBox[{"{", "1000", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{"1", "/", 
    RowBox[{"\[Xi]", ".", " ", 
     RowBox[{"RandomReal", "[", 
      RowBox[{"1", ",", "1000"}], "]"}]}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"lims", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"i", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "\[Xi]"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1137508336],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{"f", ",", "##"}], "]"}], "&"}], "@@", "lims"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->800901481],

Cell[BoxData["0.004111378705011539`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->803899748]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(12)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27236],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "One-dimensional quadrature-based integration",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->99904585],

Cell["Integration over an infinite region:", "ExampleText",
 CellID->1319942373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1396450524],

Cell[BoxData["1.000000000053296`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->146158737]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->301226],

Cell["Line integral in the complex plane:", "ExampleText",
 CellID->1381566442],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   SqrtBox["x"], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"1", "+", "\[ImaginaryI]"}], " ", ",", 
     RowBox[{"1", "-", "\[ImaginaryI]"}], " ", ",", " ", 
     RowBox[{
      RowBox[{"-", "1"}], "-", "\[ImaginaryI]"}], ",", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "\[ImaginaryI]"}], ",", 
     RowBox[{"1", "+", "\[ImaginaryI]"}]}], " ", "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2056248844],

Cell[BoxData[
 RowBox[{"6.938893903907228`*^-18", "+", 
  RowBox[{"1.333333333333334`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{167, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->437454782]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Multidimensional cubature-based integration",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->660056409],

Cell["Integration over an infinite strip:", "ExampleText",
 CellID->987442109],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{
     RowBox[{"-", "x"}], "+", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1449220138],

Cell[BoxData["1.7182818217912894`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->404652130]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->124462083],

Cell["Integration over a region with functional boundaries:", "ExampleText",
 CellID->1247122900],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"x", "-", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", 
     SqrtBox["x"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->845371828],

Cell[BoxData["0.1833333332091156`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->777025848]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->328358415],

Cell["Integration with complex numbers:", "ExampleText",
 CellID->1575881246],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"x", "/", "y"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "1", ",", "\[ImaginaryI]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2123707557],

Cell[BoxData[
 RowBox[{"0.`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.7853981643835539`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->870817440]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Singular integrals",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1651179549],

Cell["\<\
One-dimensional integral with singularities at both ends of the integration \
region:\
\>", "ExampleText",
 CellID->979465677],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox[
     RowBox[{"Log", "[", "x", "]"}], 
     SqrtBox["x"]], "+", 
    FractionBox["1", 
     SqrtBox[
      RowBox[{"1", "-", "x"}]]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1492018970],

Cell[BoxData[
 RowBox[{"-", "1.9999999999999827`"}]], "Output",
 ImageSize->{26, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->13052430]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19547005],

Cell["\<\
Three-dimensional integral with singularities at two of the corners:\
\>", "ExampleText",
 CellID->1728697943],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{"x1", "+", "x2", "+", "x3"}]]], "+", 
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "-", "x1"}], ")"}], "+", 
       RowBox[{"(", 
        RowBox[{"1", "-", "x3"}], ")"}], "+", 
       RowBox[{"(", 
        RowBox[{"1", "-", "x2"}], ")"}]}]]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x2", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x3", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5813051],

Cell[BoxData["1.725753879844789`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->205278710]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10805887],

Cell["\<\
One-dimensional integral with a singularity inside the integration region:\
\>", "ExampleText",
 CellID->41213695],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", "x"}], ")"}], "^", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30135056],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->630146756],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({0.9961248332810647`}\\). NIntegrate obtained \\!\\(-3.999950393363153`\
\\) and \\!\\(0.00027851572443338536`\\) for the integral and error \
estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\
\\\", ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\
\", ButtonNote -> \\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->219068490],

Cell[BoxData[
 RowBox[{"-", "3.999950393363153`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->298620374]
}, Open  ]],

Cell["\<\
Specify the location of the singularity to handle it with appropriate \
transformations:\
\>", "ExampleText",
 CellID->542194155],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", "x"}], ")"}], "^", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->102933999],

Cell[BoxData[
 RowBox[{"-", "3.9999999999999907`"}]], "Output",
 ImageSize->{26, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->348156940]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Oscillatory integrals",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->1055007759],

Cell["\<\
Compute a highly oscillatory integral with a singularity at the origin:\
\>", "ExampleText",
 CellID->1881162760],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox[
    RowBox[{"Sin", "[", 
     RowBox[{"2000", " ", "x"}], "]"}], 
    SqrtBox["x"]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->346592293],

Cell[BoxData["0.028283061499272774`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->710761742]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->135153400],

Cell["Fourier integral finite range:", "ExampleText",
 CellID->173809221],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"10", 
      SuperscriptBox["x", "3"]}], "]"}], 
    SqrtBox["x"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->377732425],

Cell[BoxData[
 RowBox[{"-", "0.027647549723782673`"}]], "Output",
 ImageSize->{75, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->42390940]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->166759994],

Cell["Fourier integral finite range:", "ExampleText",
 CellID->432778268],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"20000", 
      SuperscriptBox["x", "3"]}], "]"}], 
    SqrtBox["x"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "200"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1472879406],

Cell[BoxData["0.000013555276647957477`"], "Output",
 ImageSize->{88, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->714385086]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1921328018],

Cell["Fourier integral infinite range:", "ExampleText",
 CellID->875859325],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"20", 
       SuperscriptBox["x", "2"]}], "+", "200"}], "]"}], "/", 
    SqrtBox["x"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1343633055],

Cell[BoxData["0.02490431202637944`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->22916172]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1710898025],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJ"]], "InlineFormula"],
 " integral infinite range:"
}], "ExampleText",
 CellID->68293982],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"BesselJ", "[", 
     RowBox[{"4", ",", 
      RowBox[{
       RowBox[{"20", 
        SuperscriptBox["x", "3"]}], "+", "4"}]}], "]"}], "/", 
    SqrtBox["x"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1753504606],

Cell[BoxData["0.002747242221735678`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->289777912]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->73975753],

Cell["Piecewise oscillatory integral:", "ExampleText",
 CellID->902514234],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"\[Piecewise]", GridBox[{
      {
       FractionBox[
        RowBox[{"Sin", "[", 
         RowBox[{"20", " ", "x"}], "]"}], "x"], 
       RowBox[{"x", "<", "0"}]},
      {
       FractionBox[
        RowBox[{"Cos", "[", 
         RowBox[{
          RowBox[{"2", " ", 
           SuperscriptBox["x", "5"]}], "+", "4"}], "]"}], 
        SqrtBox["x"]], 
       RowBox[{"0", "<", "x", "<", "1"}]},
      {
       RowBox[{"BesselJ", "[", 
        RowBox[{"30", ",", 
         RowBox[{
          RowBox[{"20", 
           SuperscriptBox["x", "3"]}], "+", "4"}]}], "]"}], 
       RowBox[{"x", ">", "1"}]}
     },
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxItemSize->{
      "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.27999999999999997`], {
          Offset[0.84]}, 
         Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->759892428],

Cell[BoxData["0.5999380723346605`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->106419010]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Monte Carlo integration",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->839801663],

Cell["\<\
Compute a 10-dimensional integral using adaptive Monte Carlo integration:\
\>", "ExampleText",
 CellID->796164971],

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[Xi]", "=", 
   RowBox[{"Array", "[", 
    RowBox[{"x", ",", 
     RowBox[{"{", "10", "}"}]}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"lims", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"i", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "\[Xi]"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->80021104],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     FractionBox[
      RowBox[{"Exp", "[", 
       RowBox[{"-", 
        RowBox[{"\[Xi]", ".", "\[Xi]"}]}], "]"}], 
      SqrtBox[
       RowBox[{"\[Xi]", ".", "\[Xi]"}]]], ",", "##", ",", 
     RowBox[{"Method", "\[Rule]", "\"\<AdaptiveMonteCarlo\>\""}]}], "]"}], 
   "&"}], "@@", "lims"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->683155504],

Cell[BoxData["0.03508831654857796`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->237544071]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->292072618],

Cell["Find the volume of the unit ball in 5 dimensions:", "ExampleText",
 CellID->12891935],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Boole", "[", 
    RowBox[{
     RowBox[{
      SubsuperscriptBox["x", "1", "2"], "+", 
      SubsuperscriptBox["x", "2", "2"], "+", 
      SubsuperscriptBox["x", "3", "2"], "+", 
      SubsuperscriptBox["x", "4", "2"], "+", 
      SubsuperscriptBox["x", "5", "2"]}], "\[LessEqual]", "1"}], "]"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "2"], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "3"], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "4"], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "5"], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<AdaptiveQuasiMonteCarlo\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->188498931],

Cell[BoxData["0.1620217399296002`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->275956426]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->539005362],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "AccuracyGoal",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->474483189],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " option can be used to change the default absolute tolerance:"
}], "ExampleText",
 CellID->150473182],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"10", "^", 
      RowBox[{"(", 
       RowBox[{"-", "5"}], ")"}]}], "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        RowBox[{"10", "^", "2"}], 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", 
           RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}]}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->447216101],

Cell[BoxData[
 FractionBox[
  RowBox[{"ArcTan", "[", "5", "]"}], "500000"]], "Output",
 ImageSize->{72, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->830559185]
}, Open  ]],

Cell["\<\
The integration process stops once the accuracy goal criterion has been \
exceeded: \
\>", "ExampleText",
 CellID->1979756711],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"10", "^", 
      RowBox[{"(", 
       RowBox[{"-", "5"}], ")"}]}], "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        RowBox[{"10", "^", "2"}], 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", 
           RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}]}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"AccuracyGoal", "\[Rule]", "8"}]}], "]"}], "-", 
  "exact"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->963214242],

Cell[BoxData[
 RowBox[{"-", "4.9579365563708815`*^-12"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->2885668]
}, Open  ]],

Cell["\<\
The result with the default settings is different since the default uses only \
a precision criterion:\
\>", "ExampleText",
 CellID->1068336026],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"10", "^", 
      RowBox[{"(", 
       RowBox[{"-", "5"}], ")"}]}], "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        RowBox[{"10", "^", "2"}], 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", 
           RowBox[{"1", "/", "2"}]}], ")"}], "^", "2"}]}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], "-", 
  "exact"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->663400401],

Cell[BoxData["4.466404730871926`*^-18"], "Output",
 ImageSize->{87, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->115583411]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "EvaluationMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->328322763],

Cell["\<\
Get the number of evaluation points used in a numerical integration:\
\>", "ExampleText",
 CellID->353531660],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"k", "=", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        SqrtBox["x"]], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"k", "++"}]}]}], "]"}], ",", "k"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1406978729],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.000000000000003`", ",", "132"}], "}"}]], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->433772487]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->43543895],

Cell["Show the evaluation points used in a numerical integration:", \
"ExampleText",
 CellID->17998260],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"1", "/", 
        RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "1"}], ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}], ",", 
   RowBox[{"PlotRange", "->", "All"}]}], "]"}]], "Input",
 CellTags->"S3.9.10",
 CellLabel->"In[1]:=",
 CellID->369070442],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlmto1WUAxo8psgRrxj6ERE7TLBnhNe/u9a5T5zbd/Xbut51bhJlheIgU
taK0QhOGC0VNzFtec85jGt6mLi8omrqalmiasEqldCHv7yl0X8bju+c8z/v7
P3+O3d3xAt9TDocjrZ3D8ei3/bmT/fkEV05Tw+/7+AczsPbVxI55t9FpZvma
Xt27dPkNnW7eu9j4cEntdXSGcbTO3ls55yr6eWN/N6NfMI6u6fPHdT6HzjT3
ljR22LXoOLqHMcdO/fXW7H3oniaV5X+wvMM69MsmVbYxcGtKEv2KdLbVffT3
6Cx9Hvo15aH7qg+6n/qi++s+6AG6L3qgeKAHiRf6dQNP9GD6JtBD6LscPZS+
m9DD6LsXPZy+h9Aj6NuEHknfM+hR9D2Pzqbvj2hD38tWJw19r3A+2uSs6tsr
eIX+ydFmxYmMIRezdZ8xZurppZ8sTOO+yTFmf9TfdjhXPMaa5P09lzeOuMH5
WPqK3ziT7Diue/6Ba5yPM9lLv3AM3tLC+XizLSsan333J87HG3+/m4d6pvR8
JpgdFScvLMtEJyeoL/uYqL5WJyeqL+eT1JfzSerL+WT15Xyy+nKeo76c56gv
51PUl/Mp6sv5VO2B86naA+fTtAfOp2kPnOdqD5zn0reJ8+naA+fTtQfO87QH
q02e9sDf52kPVqfy6OvEn0/fhfjz6bsSfz59v8afT9+d+Avom8JfQN+D+Avo
exh/AX0b8c+g7wn8M+h7Ev8M9qD+M9gD2jGTPXBfM5M9wCM5kz1cwD+TvuJX
yB7O4i9kD6fxF7KHH/AXsgc9nyL2gDZF4suei8TX6lSR+LL/YvG12hSLL/5i
8cVfLL74S8QXf4n44i8RX/wl4ou/VHzxl4ov/lLxxV8qvvjLxBd/mfjiLxNf
/GXii79cfPGXiy/+cvHFXw7fPPwV8H0bfwV8P8ZfAd9a/BXwXYu/kr6b8VfC
dxv+Svjuwl8J33r8VfBtwF8FX76vklXwhXeqCr7iXw1fno+phq+eXzV8j+Kv
pi/P2+GE7/dWZzrhe4DPc8J3v9VOJ3zZU9IJX3SdU3zZu1N8rW52ii/7d4mv
1Zku8eV9cImv1U6X+PJ+ucTX6jqX+JLvEl/yXeJLvlt8yXeLL/lu8SXfLb7k
u8WXfLf4ku8WX/Ld4ku+R3zJ94gv+R7xJd8DX0O+B75B8j3wnUe+B74fke+B
7zLyvfStI98L39Xke+HL/7+cXvhuIN8L343ke+HL91XKC194N3vhK/4++PJ8
Mn3w5fkZH3y/Jd9HX5530gff7eT74PsN+T74biHfB1/25PDDF53pF1/27xdf
9u8XX/bvF1/27xdf3je/+Frd7Bdf3r+A+PL+BcSX/ID4kh8QX/ID4kt+QHzJ
D4gv+QHxJT8ovuQHxZf8oPiSHxRf8oPiS35QfMkPii/5+nx4OEJmbt67vzYd
WU9+yHSaM6b2+BDuZ0ImnFnx0qk/0c6Q+eDNDPf1Nt0/ZL5c3P5Bm3RdyGS3
PfpBp0LkKT9k0hs6uevPv09+2Gy9vLl46i8l5IfNz1n7Ju4uzyA/bM7mzvrw
Zu3aUTY/bNZ1vfP3+pa7I21+2Ezber5xUdb5ETY/bDZ2O9g+ONox3OaHTbsb
9zqeudQ62OaHTY/y5zI6P72G/BrTv2743GMpvr8za0xkQmunm0d1/xrzzMvt
u+1+qPvXmKrW+X3+u1+yxqz+7p//71tX8/h9U0/o5hpTP/niuWu3PiU/YiIP
e7y4+v4b5EfMH0cqFryzeBj5EbOg98rjz8aucv+IWXN72GejN5VZnYyYdttX
zLp0Y47lURcxs1oGbOhzP83ySEXMyf6rTO9LXw21+RFTeHBydXrL/kE2P6o9
8P5FtQfe/6j2wPsf1R54/6PaA+9/VHvg/Y9qD7z/ytP7H9MeyI9pD+THtAfy
Y9pDg82PaQ97bX5Me6i3+THtYY/Nj2kPO21+XHsgP649kB/XHsiPaw/cP649
cP+49sD944/ft/kJ7UhoD1anJ7QH+iS0B6v7JrQHe3+T0B6szktoD5aHM6E9
WB6JhPaw2/ZPaA/b/wUB1g7Z
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{184, 105},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"S3.9.10",
 CellLabel->"Out[1]=",
 CellID->192777593]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Exclusions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->96867093],

Cell["\<\
Integration by excluding the curves on which the integrand's denominator is \
zero:\
\>", "ExampleText",
 CellID->1830794989],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    SqrtBox[
     RowBox[{"Sin", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", "y"}], "]"}]]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "2"}], ",", "4"}], "}"}], ",", 
   RowBox[{"Exclusions", "\[Rule]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Sin", "[", 
       RowBox[{
        SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}], 
     ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1049131592],

Cell[BoxData[
 RowBox[{"30.493301555347387`", "\[InvisibleSpace]", "-", 
  RowBox[{"29.907341357128896`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->84628818]
}, Open  ]],

Cell["The curves on which the integrand is singular:", "ExampleText",
 CellID->124341040],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", "y"}], "]"}], "==", "0"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "2"}], ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1067034414],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJxEnAk4Vc//x+13v1el0irKNy1EtoqaD6VNUqhEq8jWJhFtsiYKFVHK2kaS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   "], {{}, {}, 
    {Hue[0.67, 0.6, 0.6], 
     TagBox[
      TooltipBox[
       LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
         19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
        36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 
        53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 
        70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82}],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ], 
     TagBox[
      TooltipBox[LineBox[CompressedData["
1:eJwl1OO/EAYYBeB7s23fbGstLdt2N/Nmt7Rs27Ztt2VzS1tYXtzSnn59eM77
D5zzBgWHVOsYGBAQkElUd2tQk1rUpg51qUd9GtCQRjSmCU0JphnNaUFLWtGa
NrSlHe3pQEdC6ERnutCVbnSnBz3pRW/60Jd+9OdnBjCQQQxmCL8wlGEMZwQj
GcVoxjCWcYxnAhOZxGSmMJVpTGcGM5nFbOYwl3nMZwELWcRilrCUZSxnBStZ
xWrWsJZ1rGcDG9nEZrawlW1sZwc72cVu9rCXfeznAAc5xGGOcJRjHOcEv/Ib
JznFac5wlnOc5wIXucRlrnCVa1znBje5xe/8wW3ucJd73OdP/uIBD3nEY/7m
CU95xnNe8JJX/MNr3vCWd7znA//yHx/5xGe+8JUAvQ0kFKEJQ1jCEZ4IRCQS
kYlCVKIRnRjEJBaxiUNc4hGfBCQkEYlJQlKSkZwUBJGSVKQmDWlJR3oykDHw
+8Yyk4WsZCM7OchJLnKTh7zkIz8/UIAfKUghClOEovxEMYpTgpKUojRlKEs5
ylOBilSiMlWoSrVv+6cGNalFbepQl3rUpwENaURjmtCUYJrRnBa0pBWtaUNb
2tGeDnx7OiF0ojNd6Eo3utODnvSiN33oSz/68z+EH5NW
        "]],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ], 
     TagBox[
      TooltipBox[LineBox[CompressedData["
1:eJwNxFVgFQQAAMDHSJHuLumW7i7pUrpVOkS6c2MjN2J0l4IKKN2dKt0NAtLd
dR93Gdt2r98tSiAQyKIhDGUYwxnBSEYxmmBCGEMoYYxlHOOZwEQmEU4Ek5nC
VKYRyXRmMJNZzGYOc5nHfBawkEUsZglLWcZyfuFXVrCS3/idP1jFatbwJ3+x
lnWsZwMb2cRmtrCVbWxnBzvZxW72sJd97OcABznEYY7wN//wL0c5xnFOcJJT
nOYMZznHeS5wkUtc5gpXucZ1bnCT/7jFbe7wP3e5x30e8JBHPOYJT3nGc17w
kle85g1vecd7PvCRTwSCAoEoBBGVaEQnBjGJxRfE5kviEJd4xCcBCUlEYpKQ
lGQkJwUpSUVq0pCWdKQnAxnJxFdkJgtZyUZ2cpCTXOQmD3nJR36+pgAFKURh
ilCUYhSnBCUpRWnKUJZylKcCFalEZapQlWp8Q3VqUJNa1KYOdalHfRrQkG/5
jkY0pglNaUZzWtCSVrSmDW1px/f8wI+0pwMd6URnutCVbnSnBz/Rk5/pRW/6
0Jd+9GcAAxnEYIYwlGEMZwQjGcVogglhDKGEMZZxjGcCE5lEOBFMZgpTmUYk
05nBTGYxmznMZR7zWcBCPgNXjpfy
        "]],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ], 
     TagBox[
      TooltipBox[
       LineBox[{675, 676, 677, 678, 679, 680, 681, 682, 683, 684, 685, 686, 
        687, 688, 689, 690, 691, 692, 693, 694, 695, 696, 697, 698, 699, 700, 
        701, 702, 703, 704, 705, 706, 707, 708, 709, 710, 711, 712, 713, 714, 
        715, 716, 717, 718, 719, 720, 721, 722, 723, 724, 725, 726, 727, 728, 
        729, 730, 731, 732, 733, 734, 735, 736, 737, 738, 739, 740, 741, 742, 
        743, 744, 745, 746, 747, 748, 749, 750, 751, 752, 753, 754, 755, 756, 
        757, 758, 759, 760, 761, 762, 763, 764, 765, 766, 767, 768, 769, 770, 
        771, 772, 773, 774, 775, 776, 777, 778, 779, 780, 781, 782, 783, 784, 
        785, 786, 787, 788, 789, 790, 791, 792, 793, 794, 795, 796, 797, 798, 
        799, 800, 801, 802, 803, 804, 805, 806, 807, 808, 809, 810, 811, 812, 
        813, 814, 815, 816, 817, 818, 819, 820, 821, 822, 823, 824, 825, 826, 
        827, 828, 829, 830, 831, 832, 833, 834, 835, 836, 837, 838, 839, 840, 
        841, 842, 843, 844, 845, 846, 847, 848, 849, 850, 851, 852, 853, 854, 
        855, 856, 857, 858, 859, 860, 861, 862, 863, 864, 865, 866, 867, 868, 
        869, 870, 871, 872, 873, 874, 875, 876, 877, 878, 879, 880, 881, 882, 
        883, 884, 885, 886, 887, 888}],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ], 
     TagBox[
      TooltipBox[
       LineBox[{889, 890, 891, 892, 893, 894, 895, 896, 897, 898, 899, 900, 
        901, 902, 903, 904, 905, 906, 907, 908, 909, 910, 911, 912, 913, 914, 
        915, 916, 917, 918, 919, 920, 921, 922, 923, 924, 925, 926, 927, 928, 
        929, 930, 931, 932, 933, 934, 935, 936, 937, 938, 939, 940, 941, 942, 
        943, 944, 945, 946, 947, 948, 949, 950, 951, 952}],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ], 
     TagBox[
      TooltipBox[LineBox[CompressedData["
1:eJwN0+OD0AcAgOFf3V22XZdt67J52XVZl7faloawhS1sYQszwpa2bNu2bWvP
h+f9D97wqH6R0TGCIFghK0OCYBWrWcNa1rGeDWxkE5vZwla2sZ0d7GQXu9nD
XvaxnwMc5BCHOcJRjnGcE5zkFKc5w1nOcZ4LXOQSl7nCVa5xnRvc5Ba3ucNd
7nGfBzzkEY95wlOe8ZwXvOQVr3nDW97xniA0CGIQkxBCCSMWsYlDXOIRnwQk
JBGJSUJSkpGcFKQkFalJQ1rSkZ4MZCQTmclCOFnJRnZykJNc5CYPeclHfgpQ
kEIUpghFKUZxSlCSUpSmDGUpR3kqUJEIKlGZKlSlGtWpQU1qUZs61KUe9WlA
QxrRmEia0JRmNKcFLWlFa9rQlna0pwMd6UQUnelCV7rRnR70pBe96UNfoulH
fwYwkEEMZggf8CFDGcZHfMwnDGcEIxnFaD7lMz7nC8YwlnGM50u+YgITmcRk
vuYbpjCVaUznW75jBjOZxWy+5wfmMJd5zOdHfuJnfuFXfuN3/uBP/mIBC1nE
Yv7mH5awlGUsZwX/8h8rWcVq1rCWdaxnAxvZxGa2sJVtbGcHO9nFbvawl33s
5wAHOcRhjnCUYxznBCc5xWnOcJZznOcCF7nEZa5wlWtc5wY3ucVt7nCXe9zn
AQ95xGOe8JRnPOcFL3nFa97wlne8JwjzPzEJIZQwYhGbOMQlHvFJQEISkZgk
JCUZyUlBSlKRmjSkJR3pyUBGMpGZLISTlWxkJwc5yUVu8pCXfOSnAAUpRGGK
UJRiFKcEJSlFacpQlnKUpwIViaASlalCVapRnRrUpBa1qUNd6lGfBjSkEY2J
pAlNaUZzWtCSVrSmDW1pR3s60JFORNGZLnSlG93pQU960Zs+9CWa/wH+Ft3I

        "]],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ], 
     TagBox[
      TooltipBox[LineBox[CompressedData["
1:eJwNxGWADgYAANDvtJlmpma6h+kaZjrGMUxMTzvdcVyf6e7p2KanayE30x3T
m+7O9+O9rG1CgrsHBQKBzAqJHwj0oCe96E0f+tKP/gxgIIMYzBCGMozhhDKC
kYQRTgSRRBFNDLGM4gdGM4axjGM8E5jIJCYzhalMYzozmMksZjOHufzIPOaz
gIUsYjFLWMoylvMTP/MLK1jJKlazhrWs41fWs4GNbGIzW9jKNrazg538xu/8
wZ/sYjd72Ms+9vMXf3OAfzjIIQ5zhKMc4zgnOMkpTnOGs5zjPBf4l4tc4jJX
uMo1rvMf/3ODm9ziNne4yz3u84CHPOIxT3jKM57zgpe84jVveMs7AgkCgSDi
EJd4xCcBCUlEYj4gCR+SlGQkJwUpSUVq0pCWj0jHx6QnAxnJRGY+IQufkpVs
ZCcHOclFbvKQl3zkpwAF+YxCFKYIn1OUYhSnBCUpRWnKUJZylOcLKlCRSnxJ
Zb6iClWpRnVqUJNa1KYOdfmaetQnmAY05Bsa0ZgmfEtTmtGcFnxHS1rRmja0
pR3t+Z4OdKQTnelCV7rRnRB60JNe9KYPfelHfwYwkEEMZghDGcZwQhnBSMII
J4JIoogmhlhG8R6n8ZWY
        "]],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ], 
     TagBox[
      TooltipBox[
       LineBox[{1676, 1677, 1678, 1679, 1680, 1681, 1682, 1683, 1684, 1685, 
        1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693, 1694, 1695, 1696, 
        1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704, 1705, 1706, 1707, 
        1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715, 1716, 1717, 1718, 
        1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726, 1727, 1728, 1729, 
        1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737, 1738, 1739, 1740, 
        1741, 1742, 1743, 1744, 1745, 1746, 1747, 1748, 1749, 1750, 1751, 
        1752, 1753, 1754, 1755, 1756, 1757, 1758, 1759, 1760, 1761, 1762, 
        1763, 1764, 1765, 1766, 1767, 1768, 1769, 1770, 1771, 1772, 1773, 
        1774, 1775, 1776, 1777, 1778, 1779, 1780, 1781, 1782, 1783, 1784, 
        1785, 1786, 1787, 1788, 1789, 1790, 1791, 1792, 1793, 1794, 1795, 
        1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803, 1804, 1805, 1806, 
        1807, 1808, 1809, 1810, 1811, 1812, 1813, 1814, 1815, 1816, 1817, 
        1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825, 1826, 1827, 1828, 
        1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836, 1837, 1838, 1839, 
        1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847, 1848, 1849, 1850, 
        1851, 1852, 1853, 1854, 1855, 1856, 1857}],
       RowBox[{
         RowBox[{"Sin", "[", 
           RowBox[{
             SuperscriptBox["x", "2"], "+", "y"}], "]"}], "\[Equal]", "0"}]],
      Annotation[#, Sin[$CellContext`x^2 + $CellContext`y] == 0, 
       "Tooltip"]& ]}}],
  AspectRatio->1,
  Frame->True,
  PlotRange->{{-2, 4}, {-2, 4}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 183},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->88069420]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "MaxPoints",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->125944362],

Cell["\<\
Stop integration after a specified number of points has been exceeded:\
\>", "ExampleText",
 CellID->1831979707],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"MaxPoints", " ", "->", " ", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->969050975],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"maxp\"\>"}], ":", 
  " ", "\<\"The integral failed to converge after \\!\\(33\\) integrand \
evaluations. NIntegrate obtained \\!\\(1.9558072180392028`\\) and \
\\!\\(0.06781302015519788`\\) for the integral and error estimates.\"\>"}]], \
"Message", "MSG",
 CellID->202512233],

Cell[BoxData["1.9558072180392028`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->405578365]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"Abs", "[", 
      RowBox[{"x", " ", "+", " ", "y", " ", "+", " ", "z"}], "]"}], "]"}]}], 
   ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", 
     RowBox[{"-", "1"}], ",", " ", "1"}], "}"}], ",", " ", "\n", "  ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", 
     RowBox[{"-", "1"}], ",", " ", "10"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"z", ",", " ", 
     RowBox[{"-", "1"}], ",", " ", "30"}], "}"}], ",", " ", "\n", "  ", 
   RowBox[{"Method", " ", "->", " ", "\"\<AdaptiveMonteCarlo\>\""}], ",", " ",
    
   RowBox[{"MaxPoints", " ", "->", " ", "1000"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->751519308],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"maxp\"\>"}], ":", 
  " ", "\<\"The integral failed to converge after \\!\\(1100\\) integrand \
evaluations. NIntegrate obtained \\!\\(197.45748669654552`\\) and \
\\!\\(2.6568252158578005`\\) for the integral and error estimates.\"\>"}]], \
"Message", "MSG",
 CellID->444265883],

Cell[BoxData["197.45748669654552`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->155002268]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "MaxRecursion",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->68379317],

Cell["\<\
Without enough adaptive recursion, the following gives a poor result:\
\>", "ExampleText",
 CellID->313888073],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"Sin", "[", "x", "]"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1988258193],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->218108592],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({9.43625271977563`15.954589770191005}\\). NIntegrate obtained \\!\\(\\(\
\\(10.303166074630907` \[InvisibleSpace]\\)\\) - \\(\\(6.742554183083223`\\\\ \
\[ImaginaryI]\\)\\)\\) and \\!\\(0.3168981771605228`\\) for the integral and \
error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\
\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->551780816],

Cell[BoxData[
 RowBox[{"10.303166074630907`", "\[InvisibleSpace]", "-", 
  RowBox[{"6.742554183083223`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->244994430]
}, Open  ]],

Cell[TextData[{
 "Specifying a larger value for ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " gives a much better result:"
}], "ExampleText",
 CellID->416954400],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"Sin", "[", "x", "]"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "100"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->682978387],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->244035193],

Cell[BoxData[
 RowBox[{"10.488229310959058`", "\[InvisibleSpace]", "-", 
  RowBox[{"6.7694653297438805`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->477526979]
}, Open  ]],

Cell["Specifying the singularity locations is even more efficient:", \
"ExampleText",
 CellID->4837156],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"Sin", "[", "x", "]"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Pi]", ",", 
     RowBox[{"2", " ", "\[Pi]"}], ",", 
     RowBox[{"3", " ", "\[Pi]"}], ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->730714105],

Cell[BoxData[
 RowBox[{"10.488230183119377`", "\[InvisibleSpace]", "-", 
  RowBox[{"6.769465582609065`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->149202581]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->20747351],

Cell["Timing for the default multidimensional rule:", "ExampleText",
 CellID->948959124],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "\[Pi]"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->609781982],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.6009999999999999`", ",", "0.8741681576720639`"}], 
  "}"}]], "Output",
 ImageSize->{121, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->222419039]
}, Open  ]],

Cell["Using a higher order multidimensional rule is faster:", "ExampleText",
 CellID->1718627431],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "\[Pi]"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<MultiDimensionalRule\>\"", ",", 
       RowBox[{"\"\<Generators\>\"", "\[Rule]", "9"}]}], "}"}]}]}], "]"}], "//",
   "Timing"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1385782705],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.06999999999999837`", ",", "0.8741681932947578`"}], 
  "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->306457727]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "MinRecursion",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->250171837],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " may miss sharp peaks of integrands: "
}], "ExampleText",
 CellID->813625102],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{
     RowBox[{"-", "100"}], 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], ")"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "50"}], ",", "60"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "50"}], ",", "60"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->358756462],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(18\\) recursive bisections in \\!\\(x\\) near \\!\\({x, y}\\) = \
\\!\\({23.33340779722629`, 42.85359608864269`}\\). NIntegrate obtained \
\\!\\(0.`\\) and \\!\\(0.`\\) for the integral and error estimates. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", \
ButtonNote -> \\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->70840692],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->644736936]
}, Open  ]],

Cell[TextData[{
 "Increasing ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " forces a finer subdivision of the integration region:"
}], "ExampleText",
 CellID->947812824],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{
     RowBox[{"-", "100"}], 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], ")"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "50"}], ",", "60"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "50"}], ",", "60"}], "}"}], ",", 
   RowBox[{"MinRecursion", "\[Rule]", "4"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->378241225],

Cell[BoxData["0.031415926458653085`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->170615515]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "PrecisionGoal",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->585718500],

Cell[TextData[{
 "The number of samples used to evaluate ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{"x", "+", "y"}]]], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " for different relative tolerances:"
}], "ExampleText",
 CellID->2038745],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"sampleCount", "[", "pg_", "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"k", "=", "0"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"1", "/", 
         RowBox[{"Sqrt", "[", 
          RowBox[{"x", "+", "y"}], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"PrecisionGoal", "\[Rule]", "pg"}], ",", 
        RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
         RowBox[{"k", "++"}]}]}], "]"}], ";", "\[IndentingNewLine]", "k"}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->79454938],

Cell[TextData[{
 "The number of samples needed typically increases exponentially with the ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->184300085],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"sampleCount", "[", "pg", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"pg", ",", "2", ",", "12"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->772163697],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "119", ",", "289", ",", "459", ",", "901", ",", "2125", ",", "5287", ",", 
   "12716", ",", "32436", ",", "82348", ",", "201518", ",", "440963"}], 
  "}"}]], "Output",
 ImageSize->{470, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->293817646]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogPlot", "[", 
  RowBox[{"%", ",", 
   RowBox[{"Joined", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->62454419],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFWGuIVVUU3jPnzpSlqZVdR8mmopdWjvawMsvIHM3UzCaryew6Dc0tdWwa
lcbKVLIiK7KCHpBBD+jxw0iMwCASIiESIiEqCIOwH0mQ9GMgOH3r23ufu91n
3ds4OTTouXft8+219vrWY+9955d6uzpXlnrLHaXm1p7S6q5yx0PNs7t7MJTU
GWN24v/asQbfU3zj/51pis/8swXPAgDjREoEyT9TxKPejcunyPK63smjOHtK
MMsoWk6TR6EiGzM2N1LMjdRHWkZweKqywoI6f/A2LtHWH/iX19zk+Mi/GZiF
YjA/cfoGt/aWqmtrivCD52eK0zhe0VgfWPReJWmYLWHeFSMNr9UYPRXPN4X+
3PtCNbncteOD97+t/v4YyckQ6/+/5To1KsMJ7e//+8VtX67v+/Tw4f5hEULN
uBrvY3mkavckynu++Hlp+3v7v/vNPK2iODc5dOgv5MA7b+8zT6mo0ZRffWVv
96qPgTVbVNTJlPd98yss7v3qF/OEijqFfICGNb27oBHEbFZxYyhjTbB54MAf
ZpOKsj0H/mH9n+3+yWxUUUVvc+PjuxEDkLJBxZHZBrwGaNHCt7C8gwf/NI+o
WFZ1I9jAEpe0vQv7n+z6Ht48rKInUDPW+eSWz+fN3Y6nxGSdim2mZqiCQqgF
mzABQ70q+mxqBkNYLdaMlSMCpkfFnkse4B84Bf2YtFrFnU8cYgjbz27dA9ur
VNxE4j7asR+47W98jSWvUHGTPP8AgSv49YCKu4g4mAM/Hcs/hCNlFTeZuB9/
+D3z4z7ibEn6P7/7JtFng/ucmvOyk9PZzav2d/MvsnyNZbuPWZ6k0947RHYu
9Tyj14AXeLT8GFoS8XKOW86kxktD5Mk07wncQGyQWcuGyNKVPjZIOKnJu+28
pJJJcf6EeRRquooykhFZKb10qZq70+kBHEK9oroRo3YVNyN1nQ1Mo2DMnSrq
Gs8TKgb/8OV2FTeTMvS43WCJirqOst0NUKrmNhV1PWW8h58waG5VUbMo291A
LC5WUTcEFoWzW1RUK2UQhmwQizerqDk+N10kF6qouZTR17F6qcUFKupGH6Vs
175JxXFUdlnYRD8y81TUfG8TXiLTrPYcaoG3iYTHRgWbc1QcvU+w2zk/W1XU
osCmMDtbRS32zLqdfVaGkgxvC2Io1c7IM+/D282Jkc42n5OIFuiDQ8wr05jW
ql35enwkM0sTRAk+yBlg5iD1sCYSxAeMSS1dexR6RLzDRwYlZiMzo5aGhmzE
DlpcoLfdR9DVxtX/Sdtd6ZFVOz2LUtitRFdB6Vrsd0E1sx+602Oli8Y2l/nM
cDl4BXEjna0YfU9aqTnJ/2k10SXPNuoAhOPLZcSPinDx+bsjdT3OZQt3wyyX
O30O4C3q1e7JuXqwKPCAokdV2DNCDsXThjubStW0qKiuiKPJKqrso+f60cUq
6n7PCQrKZuCFKm6FzyzHwSQVtZIy3rtT88QjmOr2HMAQzJkLVB08ObraFA7O
U1EPUob39jhnT6A5VE/EwTkqiuffJGvI9uSbQ62hjAhjXdKRz1JRayMGzkwH
0tnW+SiAGXsqOSMdeCcRjX0cRw3AU7lbTDiK+aHcF/Aq6z99kHrWc4aNtfVo
/AA08VzlKleGHvVeYTWSdeMGuJpYjn8heMxHyfXtpixKtc5hsZYNPmrZGYnZ
4G611efxLun6iWQS75pVu9amIKMkh8dk6BAVr3WzX1uW1rwfs9dpvvFO7c4b
0lN4585lN+/nwd4/WkXxri97v9Ol/47A3w0Ktm/a+/oIFfcMZVhz2oarqK3U
ZllCvsHfE1Tccz6j3MljmIp6ntrQN3DSsJE9TsW94CPpztKNKmqb1wZV9vZU
UHEvUa7cgRIV9TJlLAnOpj7Xsx77ujrH/oJX7ddn+zR1/wDmjaBt\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->709142616]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "WorkingPrecision",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->364871850],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can compute integrals using higher working precision:"
}], "ExampleText",
 CellID->321720514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Exp", "[", 
     RowBox[{"-", 
      RowBox[{"t", "^", "2"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "12"}], "}"}], ",", 
    RowBox[{"WorkingPrecision", "\[Rule]", "100"}]}], "]"}], 
  RowBox[{"2", "/", 
   RowBox[{"Sqrt", "[", "Pi", "]"}], " "}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->448363667],

Cell[BoxData["0.\
999999999999999999999999999999999999999999999999999999999999999864373883079409\
5787219693843409582427333218723397521`100."], "Output",
 ImageSize->{578, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->103115510]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " used is 10 less than the ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 ": "
}], "ExampleText",
 CellID->395073585],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Exp", "[", 
     RowBox[{"-", 
      RowBox[{"t", "^", "2"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "12"}], "}"}], ",", 
    RowBox[{"WorkingPrecision", "\[Rule]", "100"}], ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "90"}]}], "]"}], 
  RowBox[{"2", "/", 
   RowBox[{"Sqrt", "[", "Pi", "]"}], " "}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->105397796],

Cell[BoxData["0.\
999999999999999999999999999999999999999999999999999999999999999864373883079409\
5787219693843409582427333217783048361`100."], "Output",
 ImageSize->{578, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->141116667]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->669772564],

Cell[TextData[{
 "This finds the 3",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "rd"], TraditionalForm]]],
 " quartile of the ",
 Cell[BoxData[
  ButtonBox["NormalDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NormalDistribution"]], "InlineFormula"],
 " using ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->136564390],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"a_", "?", "NumberQ"}], "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"PDF", "[", 
      RowBox[{
       RowBox[{"InverseGaussianDistribution", "[", 
        RowBox[{"2", ",", "1"}], "]"}], ",", "x"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "a"}], "}"}], ",", 
     RowBox[{"MaxRecursion", "\[Rule]", "100"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->957268230],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", "a", "]"}], "\[Equal]", 
    RowBox[{"3", "/", "4"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2492432],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"a", "\[Rule]", "2.284173981552925`"}], "}"}]], "Output",
 ImageSize->{90, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->270059300]
}, Open  ]],

Cell[TextData[{
 "This finds the 3",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "rd"], TraditionalForm]]],
 " quartile of the ",
 Cell[BoxData[
  ButtonBox["NormalDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NormalDistribution"]], "InlineFormula"],
 " using ",
 Cell[BoxData[
  ButtonBox["Quantile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quantile"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->236725317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"InverseGaussianDistribution", "[", 
    RowBox[{"2", ",", "1"}], "]"}], ",", ".75"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->945494830],

Cell[BoxData["2.2841739814566586`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->474826046]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14335361],

Cell["\<\
Use Green's theorem to find the area of a parametrically defined \
2-dimensional region:\
\>", "ExampleText",
 CellID->544988733],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"x", "[", "t_", "]"}], ":=", 
    RowBox[{
     RowBox[{"Cos", "[", "t", "]"}], "+", 
     RowBox[{
      RowBox[{"1", "/", "7"}], 
      RowBox[{"Cos", "[", 
       RowBox[{
        RowBox[{"7", "t"}], "+", 
        RowBox[{"Pi", "/", "3"}]}], "]"}]}]}]}], ";"}], " "}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"y", "[", "t_", "]"}], ":=", 
    RowBox[{
     RowBox[{"Sin", "[", "t", "]"}], "+", 
     RowBox[{
      RowBox[{"1", "/", "7"}], 
      RowBox[{"Sin", "[", 
       RowBox[{"7", "t"}], "]"}]}]}]}], ";"}], " "}], "\n", 
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], ",", " ", 
     RowBox[{"y", "[", "t", "]"}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", " ", "0", ",", " ", 
     RowBox[{"2", "\[Pi]"}]}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1392412280],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 168},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->377873674]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "[", "t", "]"}], 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"y", "[", "t", "]"}], ",", "t"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", 
     RowBox[{"2", "\[Pi]"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->180897631],

Cell[BoxData["3.365992128846209`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->258479703]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->258581796],

Cell[TextData[{
 "Find the Fourier coefficients of a periodic function on ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->128269499],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], " ", ":=", 
   RowBox[{"(", 
    RowBox[{"1", "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{"100", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"x", "-", 
           RowBox[{"1", "/", "3"}]}], ")"}], "^", "2"}]}]}], ")"}]}], ")"}]}],
   ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->271271658],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->198431993]
}, Open  ]],

Cell[BoxData[{
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{"2", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Cos", "[", 
          RowBox[{"2", "\[Pi]", " ", "j", " ", "x"}], "]"}], 
         RowBox[{"f", "[", "x", "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", " ", "0", ",", " ", "1"}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"j", ",", " ", "0", ",", " ", "20"}], "}"}]}], "]"}]}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"b", "=", 
   RowBox[{"2", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Sin", "[", 
          RowBox[{"2", "\[Pi]", " ", "j", " ", "x"}], "]"}], 
         RowBox[{"f", "[", "x", "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", " ", "0", ",", " ", "1"}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"j", ",", " ", "1", ",", " ", "20"}], "}"}]}], "]"}]}]}], 
  ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->16513875],

Cell["\<\
Show the difference between the function and the Fourier series:\
\>", "ExampleText",
 CellID->438948190],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], " ", "-", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"a", "[", 
         RowBox[{"[", "1", "]"}], "]"}], "/", "2"}], " ", "+", " ", 
       RowBox[{"Sum", "[", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"a", "[", 
            RowBox[{"[", 
             RowBox[{"j", "+", " ", "1"}], "]"}], "]"}], " ", 
           RowBox[{"Cos", "[", 
            RowBox[{"2", "\[Pi]", " ", "j", " ", "x"}], "]"}]}], " ", "+", 
          " ", 
          RowBox[{
           RowBox[{"b", "[", 
            RowBox[{"[", "j", "]"}], "]"}], 
           RowBox[{"Sin", "[", 
            RowBox[{"2", "\[Pi]", " ", "j", " ", "x"}], "]"}]}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"j", ",", "1", ",", "20"}], "}"}]}], "]"}]}], ")"}]}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->734908557],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 99},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->138879792]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->251350421],

Cell[TextData[{
 "When a closed-form solution is available, ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 " can be used instead of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1300109756],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{"x", "+", "y"}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2104185480],

Cell[BoxData[
 RowBox[{
  FractionBox["8", "3"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    SqrtBox["2"]}], ")"}]}]], "Output",
 ImageSize->{84, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->168744022]
}, Open  ]],

Cell[TextData[{
 "The result of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " is close to the exact value:"
}], "ExampleText",
 CellID->1161879132],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "-", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{"x", "+", "y"}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->693688742],

Cell[BoxData[
 RowBox[{"-", "4.57992288538378`*^-9"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->309170504]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->39928257],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " can be used instead of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->16875335],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"f", "[", "x", "]"}], ",", "x"}], "]"}], "\[Equal]", 
        SqrtBox["x"]}], ",", 
       RowBox[{
        RowBox[{"f", "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], ",", "f", 
     ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"1", ",", "1"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->800886492],

Cell[BoxData[
 RowBox[{"f", "\[Rule]", 
  TagBox[
   RowBox[{"InterpolatingFunction", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
   False,
   Editable->False]}]], "Output",
 ImageSize->{281, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23531646]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"2", "/", "3"}], " ", "-", " ", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"f", "/.", "sol"}], ")"}], "[", "1", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->791354346],

Cell[BoxData[
 RowBox[{"-", "3.640009038630154`*^-8"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->35156568]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " will typically give a more precise result since it uses global error \
control:"
}], "ExampleText",
 CellID->427358720],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"2", "/", "3"}], " ", "-", " ", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    SqrtBox["x"], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1299148167],

Cell[BoxData[
 RowBox[{"-", "3.4416913763379853`*^-15"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->53526948]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->516429567],

Cell["\<\
Approximate the tail of a sum using the Euler\[Dash]Maclaurin formula:\
\>", "ExampleText",
 CellID->743109189],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], " ", ":=", " ", 
   RowBox[{"1", "/", 
    RowBox[{"x", "^", "2"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1445681194],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "15"}], "}"}]}], "]"}], "+", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "15", ",", "\[Infinity]"}], "}"}]}], "]"}], "-", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"f", "[", "15", "]"}], "+", 
     RowBox[{"f", "[", "\[Infinity]", "]"}]}], ")"}], "/", "2"}], "+", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"BernoulliB", "[", 
        RowBox[{"2", "i"}], "]"}], 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          SuperscriptBox["f", 
           TagBox[
            RowBox[{"(", 
             RowBox[{
              RowBox[{"2", "i"}], "-", "1"}], ")"}],
            Derivative],
           MultilineFunction->None], "[", "\[Infinity]", "]"}], "-", 
         RowBox[{
          SuperscriptBox["f", 
           TagBox[
            RowBox[{"(", 
             RowBox[{
              RowBox[{"2", "i"}], "-", "1"}], ")"}],
            Derivative],
           MultilineFunction->None], "[", "15", "]"}]}], ")"}]}], ")"}], "/", 
     
     RowBox[{
      RowBox[{"(", 
       RowBox[{"2", "i"}], ")"}], "!"}]}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"i", ",", "1", ",", "6"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->672681917],

Cell[BoxData["1.6449340668482269`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->228174824]
}, Open  ]],

Cell["\<\
The Euler\[Dash]Maclaurin formula approximation compares well with the exact \
result:\
\>", "ExampleText",
 CellID->59565138],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"x", "^", "2"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}], "-", 
  "%"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1740679851],

Cell[BoxData[
 RowBox[{"-", "6.661338147750939`*^-16"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->275496570]
}, Open  ]],

Cell[TextData[{
 "The Euler\[Dash]Maclaurin method of ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " uses ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->51850576],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    SuperscriptBox["x", "2"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<EulerMaclaurin\>\"", ",", 
      RowBox[{"Method", "\[Rule]", "NIntegrate"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->2001488439],

Cell[BoxData["1.6449340667600105`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->235034182]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->92174765],

Cell["\<\
Find a cumulative distribution function from a probability distribution \
function:\
\>", "ExampleText",
 CellID->651107854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"NormalDistribution", "[", 
      RowBox[{"2", ",", "5"}], "]"}], ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "4."}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1510519880],

Cell[BoxData["0.6554217416098364`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->843275400]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " gives the same result: "
}], "ExampleText",
 CellID->568523091],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CDF", "[", 
  RowBox[{
   RowBox[{"NormalDistribution", "[", 
    RowBox[{"2", ",", "5"}], "]"}], ",", "4."}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1311869428],

Cell[BoxData["0.6554217416103242`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->5458053]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->96388514],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Residue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Residue"]], "InlineFormula"],
 " of a function at a pole is equivalent to a contour integral:"
}], "ExampleText",
 CellID->94267959],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Residue", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "/", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", "1"}], ")"}], "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "\[ImaginaryI]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->225328397],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox["\[ImaginaryI]", "4"]}]], "Output",
 ImageSize->{24, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->38510876]
}, Open  ]],

Cell[TextData[{
 "Numerically integrate on a contour that includes the real axis from ",
 Cell[BoxData[
  RowBox[{"-", 
   StyleBox["\[Infinity]", "TR"]}]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  StyleBox["\[Infinity]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->781895585],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   RowBox[{"2", " ", "\[Pi]", " ", "I"}]], " ", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], "/", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", "1"}], ")"}], "^", "2"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->67972935],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "0.2500000000000003`"}], " ", "\[ImaginaryI]"}]], "Output",
 ImageSize->{51, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->18582942]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->209142943],

Cell["Several special functions are defined as integrals: ", "ExampleText",
 CellID->487508788],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "=", "10."}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"1", "/", "t"}], ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "1", ",", "x"}], "}"}]}], "]"}], ",", 
     RowBox[{
      FractionBox["2", 
       SqrtBox["\[Pi]"]], 
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"Exp", "[", 
         RowBox[{"-", 
          RowBox[{"t", "^", "2"}]}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"t", ",", "0", ",", "x"}], "}"}]}], "]"}]}], ",", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"t", "^", 
         RowBox[{"(", 
          RowBox[{"x", "-", "1"}], ")"}]}], "/", 
        RowBox[{"E", "^", "t"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "Infinity"}], "}"}]}], "]"}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->175075343],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.3025850929940557`", ",", "0.9999999999999952`", ",", 
   "362879.9999999157`"}], "}"}]], "Output",
 ImageSize->{155, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->449554829]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "=", "10."}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Log", "[", "x", "]"}], ",", 
     RowBox[{"Erf", "[", "x", "]"}], ",", 
     RowBox[{"Gamma", "[", "x", "]"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->201420606],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.302585092994046`", ",", "1.`", ",", "362880.`"}], 
  "}"}]], "Output",
 ImageSize->{155, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->306056226]
}, Open  ]],

Cell["\<\
For high precision, using the special function is usually preferable:\
\>", "ExampleText",
 CellID->47202774],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Erf", "[", "12", "]"}], ",", "100"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->26534081],

Cell[BoxData["0.\
999999999999999999999999999999999999999999999999999999999999999864373883079409\
5787219693843409582427333217766711895`100."], "Output",
 ImageSize->{585, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->370621071]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  FractionBox["2", 
   SqrtBox["\[Pi]"]], 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Exp", "[", 
     RowBox[{"-", 
      RowBox[{"t", "^", "2"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "12"}], "}"}], ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "100"}], ",", 
    RowBox[{"WorkingPrecision", "\[Rule]", "110"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->58913301],

Cell[BoxData["0.\
999999999999999999999999999999999999999999999999999999999999999864373883079409\
57872196938434095824273332177667109001743034964`110."], "Output",
 ImageSize->{585, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->859292989]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->171909922],

Cell["\<\
For integrands with complicated variation, many levels of adaptive recursion \
may be required:\
\>", "ExampleText",
 CellID->24555619],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox[
    RowBox[{"Cos", "[", 
     RowBox[{"20000", "x"}], "]"}], 
    SqrtBox["x"]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1939490000],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \\!\\({0.`}\
\\). NIntegrate obtained \\!\\(0.00579916354919178`\\) and \
\\!\\(0.014875547652988541`\\) for the integral and error estimates. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", \
ButtonNote -> \\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->444464910],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->196502837]
}, Open  ]],

Cell["\<\
Specifying higher recursive subdivision levels gives a convergent result:\
\>", "ExampleText",
 CellID->549328100],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox[
    RowBox[{"Cos", "[", 
     RowBox[{"20000", "x"}], "]"}], 
    SqrtBox["x"]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->153548073],

Cell[BoxData["0.008891367476073194`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->556991540]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->127029480],

Cell["\<\
Since the default is to use relative tolerance, integrals that are zero may \
be computed slowly:\
\>", "ExampleText",
 CellID->273091344],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Sin", "[", "x", "]"}], 
     RowBox[{"Cos", "[", "y", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", 
      RowBox[{"2", "\[Pi]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", 
      RowBox[{"2", "\[Pi]"}]}], "}"}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1855351150],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->284986777],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"eincr\"\>"}], ":", 
  " ", "\<\"The global error of the strategy GlobalAdaptive has increased \
more than \\!\\(2000\\) times. The global error is expected to decrease \
monotonically after a number of integrand evaluations. Suspect one of the \
following: the working precision is insufficient for the specified precision \
goal; the integrand is highly oscillatory or it is not a (piecewise) smooth \
function; or the true value of the integral is 0. Increasing the value of the \
GlobalAdaptive option MaxErrorIncreases might lead to a convergent numerical \
integration. NIntegrate obtained \\!\\(-1.468009741545373`*^-16\\) and \
\\!\\(1.4974799404526401`*^-12\\) for the integral and error estimates. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NIntegrate/eincr\\\", \
ButtonNote -> \\\"NIntegrate::eincr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->86330940],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"32.397000000000006`", ",", "0``11.824638986507319"}], 
  "}"}]], "Output",
 ImageSize->{126, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->539713215]
}, Open  ]],

Cell[TextData[{
 "Specifying an absolute tolerance with ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " will reduce the amount of work:"
}], "ExampleText",
 CellID->197471965],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Sin", "[", "x", "]"}], 
     RowBox[{"Cos", "[", "y", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", 
      RowBox[{"2", "\[Pi]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", 
      RowBox[{"2", "\[Pi]"}]}], "}"}], ",", " ", 
    RowBox[{"AccuracyGoal", "\[Rule]", "8"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->89566389],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.010000000000002479`", ",", "0``14.058950510203026"}], 
  "}"}]], "Output",
 ImageSize->{112, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->180079319]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->111789566],

Cell["\<\
Higher-dimensional cubature-based integration to default precision may be \
slow:\
\>", "ExampleText",
 CellID->339226259],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     SqrtBox[
      RowBox[{
      "x1", "+", "x2", "+", "x3", "+", "x4", "+", "x5", "+", "x6", "+", "x7", 
       "+", "x8", "+", "x9"}]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x2", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x3", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x4", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x5", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x6", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x7", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x8", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x9", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1691852158],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"109.507`", ",", "0.4785483747985543`"}], "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->170810035]
}, Open  ]],

Cell[TextData[{
 "Decreasing the ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " speeds up the computation:"
}], "ExampleText",
 CellID->141219965],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     SqrtBox[
      RowBox[{
      "x1", "+", "x2", "+", "x3", "+", "x4", "+", "x5", "+", "x6", "+", "x7", 
       "+", "x8", "+", "x9"}]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x2", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x3", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x4", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x5", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x6", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x7", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x8", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x9", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "2"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->121416176],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.009999999999990433`", ",", "0.4785535693171402`"}], 
  "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->867229128]
}, Open  ]],

Cell[TextData[{
 "Using a quasi Monte Carlo method for the same low ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " may be even faster:"
}], "ExampleText",
 CellID->586272819],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     SqrtBox[
      RowBox[{
      "x1", "+", "x2", "+", "x3", "+", "x4", "+", "x5", "+", "x6", "+", "x7", 
       "+", "x8", "+", "x9"}]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x2", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x3", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x4", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x5", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x6", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x7", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x8", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x9", ",", "0", ",", "1"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", "\"\<QuasiMonteCarlo\>\""}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->86508208],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.010000000000000425`", ",", "0.47758668191979603`"}], 
  "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->72714948]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " automatically uses Monte Carlo integration for dimensions higher than ",
 Cell[BoxData[
  FormBox["15", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1689058454],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"(", 
      RowBox[{"\[Sqrt]", 
       RowBox[{"(", 
        RowBox[{
        "x1", "+", "x2", "+", "x3", "+", "x4", "+", "x5", "+", "x6", "+", 
         "x7", "+", "x8", "+", "x9", "+", "x10", "+", "x11", "+", "x12", "+", 
         "x13", "+", "x14", "+", "x15", "+", "x16"}], ")"}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x2", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x3", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x4", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x5", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x6", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x7", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x8", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x9", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x10", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x11", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x12", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x13", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x14", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x15", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x16", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1485004164],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.010000000000001508`", ",", "0.357578944131031`"}], 
  "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->349937733]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->102225208],

Cell["\<\
Sampling points for a 3-dimensional integration, using a low-order method:\
\>", "ExampleText",
 CellID->17442091],

Cell[BoxData[
 RowBox[{
  RowBox[{"points", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Boole", "[", 
         RowBox[{
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", "x1", ")"}], "2"], "+", 
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"x2", "-", "1"}], ")"}], "2"]}], "<", "1"}], "]"}], "+", 
        
        RowBox[{"Boole", "[", 
         RowBox[{
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", "x1", ")"}], "2"], "+", 
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"x3", "-", "1"}], ")"}], "2"]}], "<", "1"}], "]"}]}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"x2", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"x3", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<TrapezoidalRule\>\"", ",", 
             RowBox[{"\"\<Points\>\"", "\[Rule]", "3"}]}], "}"}]}], ",", 
          RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], 
         "}"}]}], ",", 
       RowBox[{"PrecisionGoal", "\[Rule]", "4"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x1", ",", "x2", ",", "x3"}], "}"}], "]"}]}]}], "]"}], 
     "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->277430143],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{"Point", "[", "points", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->36883129],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXfmPZcdZLb996b17umd6Vns8toO34ARCCAHjxDYB2zhxEhPjxJmYjmxD
ZDAGBQGKFIEgAgkJwf/qX0rwajlVp05993ULIfglI83r9+rW8p2qr7617r2v
P/z0g4ufPPz0w/cf3nv1k4d//sGH7//lvVc+/mRTNH7EOfeZc498dN1tvv/X
5lf8/5n3fnPhl5//T58PN5+ji4uL/4y/fhg+J5ufm78X8d9/xPL3wuc0ladL
oSC0+3freqriRqF0XErdv8W6P6Ax0r/RhhG4Xug31f1+V3esdWWMf43t3pXS
8UDLMFJq8cfSYmK0mEidX5gtp1do+c+x5TtSOtt8TIyWgcrU4nvSYm60mEmd
f4wt/0hKlwMtw1hXb7GQOj+PLd+W0tXmY2q0DGOlFt+VFmtusfmDXrjOz2LL
70jpzuZjJmOt8ljDLebSYi11/i62/LaU7vFYmcqdPFZq8Za02DfG2pU6fxNb
fktKD7ilS//2pM5PB1sujJaBytTim9LicPOxFFz7UuevYss3pfRIWoZ/B3ms
1OIPpcXx5mMlLQ6lzifbW05qy9Q2jJbavCFtTjYfa3MWE0xvSJSPY0+vS+k1
o6cgx+L6B6IyHan1a0br3Y720DYQMpXafxb7+AMpPUMfuQwUrHIv2M1269PN
x47LhG7+pCuh3Tr/nUuLD2PF3zeo2EM/hV1Ce8gL7eeD2M83pPT65mNfeG6c
0aCXgCa1/j1pfQNoZnki4r/U0yjTs8g9LaXtxWCP4IrQV+Tqaek9gx3RDKZ/
Sxol0Jt6f1V6P8/cM8rcojw3zuUB9TTPAl//Uez1FSm9WTiiUlR6zlBG0vM6
05l6fNnocaQ9yu+oE+Y8B4UTUrVJncW0QNPM6zzSe+b4YQGue+jT1En4nOX1
5N0HHR8WaYaduLmcev56pEQtibjKU6YvzpEgnLarnPvFuo2EtvA1SGqV69Em
cV8zVjpQcgqUDTeNcl9lnEnZ65svsDQmAyNGy8S9JKUTGXEknD7LqyQ22ubH
WMaDFtk+znmhXZHNUFoWsc5nvxfCtaDHVBNFO8X9rpTO8ujXqDXk8gIzWaiZ
CkJe/XkeM1xPY70oY81lrKnvd02dabG3UC8vbKyTd0z6FyTPgbSMNoz7HSmF
hLtuzviYS7sZ51nGTAQJBX2aRvxtGRG65tjohXUq23I8qsrinczF6IuvfWuQ
goD5yFduVgoiaOKq8HXF9cNuznhVf6dRvyqjrvOoZ4Iz8o6hg+aECTuo6c9V
vavaP9o97rd0pnySowfSc7/m1rxDJ0L677iqabEX0rhfkXH3cq0Dmr9KtYMh
4XQXjZUjfN0rTOFM2g1ZI68NUodZ4dGanZZXJaBYSD0LTairVIF3LfsmWlru
N6V0P1O2b4zIK4LdmS6PXatDa+lE+mn81Uzl5BKqwyonar8s1Ib5m+W9CBp4
f3d+NCHA3xm1CSOG0USTutZe4l6jPeZ+Q0oPia6yS439ZOsbni/sf1AHGRy/
Egq154bpCnoiyoNJsba8JYeXQkmZn4wNtgV29MLbNmC0ldyXpPQ497KmmWm0
fMYI24FXdkzlI8zLlGmHzYvaakUmmn5daDrJs7Myaaq2KTgGc688XiMrlaZ5
nrF+h4A2tTSjneJ+TUqvZRqXNOuNL+It+QreBz+qxherMdOKvma+tVSHKVtk
ihpJCBqcSI+G0zDOvFtLTG5js+VKljUZrQ33RSk9zdTBBqq2T9W1RYqJptT1
ZckB72iMHUE8AduT6Yh2gfuClJ5dQh0IgLXI+pl5iGXcVNCNaM5gESdqXhBq
gl22dFUyFmo6v6Sxxuj3lOjq+YvlWdoTvN5jb9vPX9lKK0ZkjYiB1d6Cv84z
13iKvo2XMfcGyhApShT9qlAUfLUVjcyWgxc5OkTRxFm6rFJYdqiUh7XVqFTU
Ae7zUnpu9Dp1VVNFqhvfY5LpAldwTfAmeFB5o1i1uS+2I8d5NhOdzwudwbPn
PaZ22CSPqlTN6O/MtZYt+A5UTFyvmaN0c89J6S1j1ma0lvas9dYESwneYcX+
M6U3ajW7O9dZutYyD/iGMcwNftIdoHbKNHM/ry/bT4hnsAcLjJjx4kkXbGvq
hUeM+9w9i/n0+Bd6Bu+q7zTfQjlbxp5kuPKLpbsRh0J9nfs58RGimYn+Zzr6
m0iDRqdkzLGJfkaow9epq3sNcz6T35etNbhmmtdpYq6T5UXE/eqe1lkXpGrV
ztB9h86af5YmjJC5UOWYSiWKSLlW5lb/qtL/nIkKY90gVJYMHZmrvg0Zc6L6
4MoluA7JxjE+eALTjClcS1h+RbBMhrA0mpo9DrRiSQRpqvYIVod9LY1ysMbm
SFedMayOehRxXdznpHTqa6yDNX4XITVidkpvq7tdZ9lbuDzZpWi7xPrI2iDT
l5A8JUggaxiJ5rmYW9Q2ZesBK7XAbJi2NVpxpFYlkkam4H3Ai2METw3iKtGo
xg8a0mqWZ6W6RfVn7w2iE9ZY+L0mDg/j8irB30lonhQ08Fo4Pm3K2sYesGjQ
naNeN/cO74r9Yl2pouvywFh/RCQYxYPY7Akpxc5DpBi5gsaOdXWP7Qka5qOF
a20C5rNiGzZSh+mHJz/P5fBcEt0PhO610M38U9aDZYbMHGQr06/8hTwHz36H
RzJbbPdclEgrIvZM0WODuAKtJ7Qeeu5guYV+tm44TwNrpshgoRvyC7g1vv9o
rP64lO7mWTj2Ik2mLOExxtT1UgwtlJ/YxmRrrGRLhH7LMlD5zx5dwnNf8CBa
e+TbvdpIwzxDrJ11R7DWZnsG1tC83Qm0TiW/lqMygeYSy6mR2M3nvdj8MUGw
n0c5FL5QSxm8uhhAwV4V+3qgvsqJimJMfcxaejOHYddq3ulO7ORRKT3MYx9S
v310YsCmcX0sfUqjK0bLn2l7K7GCQf9OpVlAvDOA+PYg4pmrOZQi1SjGYUVq
sXrQimztQKZy1Ij9y8a2ztjqzGoOm/MVi4yv13vqhTp3T0qPfI1PNWsr+XPm
VaCcSxu1kxs54FsbcWzU4dgx9h6iw3bUsI1aOHdXSo8NdNEOKuAmvt3vQ7EV
5VHF1kTttniZwLj0/amtku1o9pXFtQntHSlFpBq5GY3sM4dhLbfhbbw5Yy3V
1kDfK1q/YUy6gueDmMDfI26F+RRMtrzAjPNem8lf9RLZ8lWLapQpSvqDd536
teyvaczSJeljRKXXRKui5fjqMFqVpaol1WZGebUcuFeftRWwB/pgqShi9KBx
oeuxw1tG6cK1WffG63KsO1uaNe4Ji3UIs658gLgSVOXkHqFRn3dqoIseVtqX
sspLV/MtjY9A86To+nW1MCoWK6LFdpyN1VdPTKjH3r8a1vOMtclAFNZlr0vl
Cfv2HPmzfUq2uDmqr6vK0mgXo2zoWfpq/zH9p3Gwcym9mXsr+8b0Lf93UPH+
0/WF58yoIJXKyLB3fJXtKkvzyTaxC9aOtJeZg5j4dq9Ztn6JeJgWtSLUSNDY
t1kcIGwshYwQFoIiPIkDX5fS2xmhdRJiugXfjKjehtfOOAyt3NLXzCOvnM/S
5eq47lDP/cpBUrE3wB6z6sOW+rliG5Ar1V6s12euRTt3/ZlM5Cfgx2g24jh2
dyalwb7bcb1lwx6ShbdEZ7esJNpYaPX8Qb+H2arnFW7tnhlWqZTAg1WbPEW5
eg/rtkV9t/aqJdRmwNpXzmE8C7qqs8GWkTWTvUcNaWTZCch+LIzrMauaZFYX
2b+T+YA5irWjrh/jR/aBvRBFDJ5h70C5fMfVGJ/SjvOWyvEaV4/RnbS7jVzO
HWN+Cwc1q2XzJFsX2A0L+W3tAjszAZ3ZncNzNfKu8wCrHZ7A1fCDNovXS2ap
4S9rlpTnWVrrWm5Hbltjs64e9PaR6/f0XumHadcofoxNJtlnZI4sflB9BqRA
A20CLuZ8FlZze6bCPkkyNetjBg4HZgD7HRaoXk8RP7W3J8QLVeaQDndtLKLj
GV8tCfUbpq5KKNvn0cizPUtBQ+waqK37uhR5zBB0p0iAiJFrZAaak2Vb7w+a
MQnTRrsiRxhWLuKQzGM76ECwrjMX6Pm7ncF5AAfUCIBrohqWhsKcLFzrK9iy
EjYN+MHmb5actt0E/GGNj3Fx7Eo+ZM/10ZzurHusq5F/yDnmB+YFPpPVeAX5
OtBdvqb/w3kwbSKONN/Pf8PaqyaIe6LLzSArdsvTrs628vCY460UgRcsSQHe
QcZ4eyREWwMnt+bYN9af22DV1HrQWejPNWJm69zUCKmVn1TpB07ddq6DPSOU
My9aFmCovzL6wXnpJkqaSQ68votr2Ufvdog5A1OHmJIrEsGyANUb55gln14p
aOuq5y/wL7Zhh6Zh75GjrAHnvuu1hHo/Ca9mjnczBTcFr83rSyxGKWLv+yq7
AHJsyNYHWluiwJcFf5V7ohua4BvGWL+1D7pT2rhfokRqaJUtNDIPtBrs+4PS
claEVpFjqb3ER+vi7ZUriGD086NZDZ8tHdV7vZWE+y84FgD01hmStetjhHZe
7ar8bedMWdvODdrCWu52qx+owZkP3QMzEz8yc6fGGPUpCZW6letPEvJ6s2RS
iRXQ4iTsjrP8ecjNXXOVraiu7mkrvtzzBrJ07CmhR45QrVy9m5klW+/jY7Wr
p9pjRxR17XqJBT/naqhxl6SNVqU97gpJvoBr5BxzJ9Z22KqBPXvZSg9pc/U8
OX7QaCJXpQVrmM5adr19q9ZgWiYd4TTPSWlNPeJEL8/JsKcylTlZ+DY+qTEO
WDTF7s6Xd121+ksc6VJuOMqcoLmGsYn6zMuZHULNGX1E6FMkjGmw0G6LV3L5
A1c9/S3Wbv4SI1VTlg4X5UYL3T3Ieu0KLkgIxsU2BXiaM9XLLYi0/JazpSTH
uthO1JNc1/LqNXZ4vK7SG9mglSAMI+8JQj7LwhwKfQUOtXaUnt5bup6Wha+y
zPRcqD+OOANob6Hf9lXTqU8+zQgPmKvMHC08DT5PMWxP9VFVK3a8zPOu0S+W
Sjxf8cTrRTov2+fE7vr+lJfGNXvdcpDR2/Eha5URNbfvRZtlPoJFrVZSkwMh
L0VtfczFeebiR0zM4SyQFVsG3bA/t9iYvkpMxBWYn4uk2hqTWQysMO6Bw7x1
5wF4v2++PlKQqsx93PfP5OEed32VfSeul0ATwmNmsCUvAP5WRJifoSjOhJA2
J69d8oWAMKFUffpk6b1FiV0Hvgu/94WKnreHsJfom0jhIczQnFpfTwWM5a/m
pxNmlsdPg5IuI4R+YSUiFsXcM8708jmSbfjAy10GJLdZCt9wpGF0ZUzP+F7O
nrtk+ey4ITlrYbHuKlVNOPWSzSC8sId7rihnXqZlhFxF8Y0yPuIGR/eLED5E
S2HHjjNlLEMxk6g3lL9cuMrjaKcaUTPMqZv+jtF4J8iU+SsU7NNMHPgql/n8
tu5xRNUnggRWJ/jOss9U3k8N9Ij+shwBakWbITZPOQz14t0I3amIhbN3RRjp
yNc8dRM5MKKFit6KnfHvYHfjzn/wg0av9JTVtEGW/oV68S6E5jxkkBNBd+/7
qrt0PXjdxr6fW5WiRZtmavhENuw/K1fJPj7aWShUe6ukWGaaAqZjwtJLCwuR
tXYnaGxosYlvT5uo1Lsm7eYGKpyOuCdoQtXg9zxB1KcWrDt6XLpy4LVTZ2tj
WKU7rpfemBfFkaJ1KueKbtkU13t5K5qwJvcdPYnA8EomQn1zcsGI88IzpTue
LtIzsaCRJRLlWokY/mpOemmiK1y7Kb7he3SIa1vnBxAHVtTo0+bQzv+hXppc
X9ac4PVA/6HrJYXmmldedS/nCQLOa+AyK7ebeR8ngFeZh+BzctQdqwVvhSU1
5Gag+sj1ElypXndUYzfjaTZHBtWIpwfq9lzVVXsZxfZ727Zb75ZmgCbGiVDb
a+CTuNAlTPWOibWccNqUYTW0FfJx11yShfAiroZxyMeAZY+7EleXohoZqHY7
VNgDeL4O8l1cB8903SNUiJnYviR2V3Pi0YinbXsqSjsn4E30pvG9PRMZfFpE
vrF7dC/O8lw966rvofvfivsPnXRBdr+PlGEvqoTQPPS+iQdnuHHvLfS7ajpE
sfVplKfyG7J/aeCC3YB8A+QirFzWFepXcuwF3jzTeNChQ/xt7e1zw7h+knkw
rmL1ATLnrIXeXcKhuNUmBZalYBu672sIWzpRp1zGNqQ+7bF43IjAG3WQm2jt
kQPiM8To4TvhhJBSMqV5mTBGM9sElJoVPjJRhi4QwwCPNjZiRoIT7LCndH2g
1ZSvzg1+R6RmQnhEq/vWxoS3rZiOBzEdODsKiVxBwLPra3x119kRrFPXx03D
ZZzh4fyhHaHjPO/ltIcurgvN8BtXeS00RiaaxVWZG/7ecH0maY+oxl+OBoJu
a85PTLpD7VOhW32QaFFnCVDnKKzEDV+9YitTGWg8c/U0maVrNHsPLKxlNct4
bRDLNaI9fA1rYj0tBXmlG1J+09V9Cv6G5mZPA7ZY+K0Z/VOTulD7WGb6kEYP
n8mDd4086qXFrmtjP6HcztieZSoRIwEatn2GUKS7YiwZEPl41jADPeujogs0
BUR7ufd7edRjZ/c7FOvgrK/OvebTr5tUzz09uzt1fpGe3tTFX6bV2+kzfuV3
I/EOpEbY83czrXuul0Mc/UIUS0/48rlWxXhjEGOJVOQNoAhvCaVnNLuPOTuq
q5ENztXpfsZ+1Xz/uUkxfIbqTZIM8lkudbqF6b+f6VaLw8ruAwfiq4iRc95L
6b45SHfPBYV7MwjIteDv3fd1t9o8tJsLjvJ86F0zoZrqaEhWLedMANN9y0SD
OxX7s1U6g5CBz8l4PZq+BHnMHaXViR+QZ09jboguWFiRbWN6b2/Huqw90F7H
ytFqFXxCc1ibJ23cxJ9VI9XW4DtIQeuEk975if2mWYn0fAO14HHGjWR0wdfl
tXyvG8v9yUb9Q8KcPk+Ia3t9tchrbvl6erqHvTRFencr0iJBulgqeK7g8TVS
fMT4Td944asPtHb1BLFSq/LmnkmtPm+jrgCk/ZooDf2GjCbkU5M1MD2IZi1z
/8CIKJbeG2hR/6hJPd7ZwRKJe2G+n9Cst5kGZODhc9hnEEu0qVmV0A+e342s
AaQsU6L69rHB1cAuv8gnIziPGuqzFPK+3sMQ9AKewsEnOdhC5IxhKIdnw1nW
IYrvmxQXi55mSSWOtZ/D12OpV2LZm+6O82rxUyWsu+0sua3Sja16aNXRAMrH
h9eledYe5zs50q/jgu/DDAXJEGyvYIuxDW/FlJSrEVM6lPrYgWx/svfOKB7E
Zlq6JmzpH/rQNePoRVx7uf64a32A8PXAtfkF3hu6dnw3MmjQE4FdZPRKGNWG
4hnCyVmcJbojI5zneojfBK7EeQaWWtZKGlaZwXNPXIlmHsvS6VZOc47d0FgE
tzISWM99lpk5AJIN9nZZhS5KwNQ+aWJSiwicANmz8NX6RtalnCjcNDuldepy
OtJ3H9XC+WhLdsA2te7pf8rEotYAP+0LMwIeQCQV2Djmvu9tC5RlP+Ryu09h
lWFvwT7Hk1unBpbPmVh0F+F0/lxosCzO7h4oV6PUyAp3lrlKyW5/sOwE9/O5
W9zzN/SkiXjSo9tpluwbUa9Ys97yYpR+QKepZ3vD2TkqZHPUn+GcCU44Lly1
8BTj01fCCAsPmTZYvbAHwu+7rmpYXRvwbbnLKq/I3Nc8UKAOmT6O4VhUP2NS
bdkV6IXj1Hx25ZTmD5w1onrNfWqCCvMPfx8okIGATuVZY/qeNVGob1dO1+ff
JdvZWZbQ+2eOYiv5+oLosXQnJCfjQTyHMyrQePWkeaX0OROPxgvxpiGVdUPP
A3jKVStHn+/A73pAvJZPoerTkpTi5yvFU5aKYd5OqW++Y9aK6N0kGtmCgv7s
nwTQPg+EVxK+PLCNB2j//Bbaz3zVeXwqPtQOcSW8ycW6J18tWuUOld+2h4k7
OPh5UUDDKOI7AhrdEei/4Wu8Ta23B7ncuiMT6wVdp9IUVOFOjP7cBvf2QqEt
zNdNQsQyaplpGr4Xu5yJ7SxktAlN+kwvqEX2XHfSF7q5C1+ZTvspH/cj2D5u
y+eALEsXnI18GfxUltrcLr5ppNhdt32N+qL1A6HEPIdH2lul7pyo4LsVuM/0
rH9YE5id0BpPBtHdfItGDtwIPVDz9L1FrVTNL6EqvmEnxR4u0pkkUGXtsaHn
pSBjaOnbJursXGMng2LYljomvGaL8vi+onKn2A3fSitYBFNaSehE6wQVnxu1
vRymmWXiegvnpTdm4M13133r+419tVr6MaFDEOPkkxJKz4J+I7ponahI9Bzn
1T71ra+NM3UYJczdHcxJjpjZp+CUDr6DZmXQEd8SFjUbPCHmXVhj/RrcdKQT
aPVAx1ToWOW50/HjO9XS2w03hSe++hdYn/4uYPi/oda+qzpNx57T2Got7xm0
pHfUnOc1OfatbYQTcYHPEeHSUdVmgqy2bL349r/0bIyLesoNve3kmQ977pBG
07Op8yuO9tU42t082qGMtpuxWecjWFczFRytn12RivQOpUfzDB9QLWAeyl53
J0xpZLZnOIIGDtpGUXoD1uO+es1MEb93h8+3WNYReIDPzGs8D9TgRJxaKy/G
bp/w7WlASMQVjYLzvSvCyhKWY7RXHT2+bzXGJUIh383Imij0dtTtSX7WvkUD
7G34b/A+1Td4KXb6dJ4BnH20ImDMj+oH79KoqwEq2HJTKr4Wu3uWqLBO6QFz
WKela6PKGjcb0eh8v76Vw/16bPZ8XoewWvy+s/38m7UxpBlkJ86UWnehvRx7
f8HXs5lz6R337SpmPBWHTy5rvOqVWP2LmXacIwMHHlDvaonrjPE9ShrhezVW
+1LGAKRTGgXrAa8VJ6hxHpB7S+9//LL0BhmHt6JifmFhFd2Yrfv6idwWjxHf
jR41TCiEfgdfgnM06qHRtBPXnw2Mb2+PkhVP4uhzkxz3gO3FWfxQN1HsBnPB
4ZKe3k5jv+jraVicWrMsX9W3+s6T9N7Sl/Ic8WpBO/C5svBpxRyelN/prbIv
E43IeiAahufXcKsH8vv12Mur1EuwAXDu0Mz30O83YutvELIgv9kK5dqPye/0
1t7XaOwwq+whcG19Iv6bsfUbskZHvsZFuLY+Zf6bsfWbvvIt8pHW+yT1GeDp
Tctv+bqzzlx/Xu+2/H4rtvqOr9FefQ6TntTWu6a+Heu+nanWtx/pXX3pPdjv
5PGUf/Spu9+Ntd/NfSuf6HNP346138t935erep9Uehf5w1xbV1Lv2/lerP1+
psR6gj7/fifW/rGvvncfU6y/0/vfP8i1dX71ZPm7sfZHvvqul9f+Se5bZ0zl
2/dj7Y999VD4qnLgD2Ltv/DVhuerKpXfi7U/9dUG56u6N34Ya/91rm095Y1/
P4y1f+qrNcVXdc//KNb+W18tDr6q8uX9WPvvc+3Lnkf0J7H2z3zVyU2ETH5H
ZeB+7qu80HgM//5xrP0Pue/Y17Rc9Sl292Gs809ezm/nqx/Fq7/YfIZg2J/G
X/8io6Q6n/3y8//80z3y3ye1A3M=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{248, 362},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4139611]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->23034],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["NDSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NDSolve"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NSum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NSum"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Integrate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Integrate"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->28501]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->497],

Cell[TextData[{
 ButtonBox["Advanced Numerical Integration in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateOverview"],
  FontSlant->"Italic"]
}], "Tutorials",
 CellID->206418781],

Cell[TextData[ButtonBox["Numerical Mathematics: Basic Operations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalMathematics-BasicOperations"]], "Tutorials",
 CellID->123709578],

Cell[TextData[ButtonBox["Numerical Integration",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NumericalIntegration"]], "Tutorials",
 CellID->23256],

Cell[TextData[ButtonBox["Numerical Sums, Products and Integrals",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalSumsProductsAndIntegrals"]], "Tutorials",
 CellID->20805],

Cell[TextData[{
 ButtonBox["Numerical Mathematics in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
  FontSlant->"Italic"]
}], "Tutorials",
 CellID->4112],

Cell[TextData[ButtonBox["The Uncertainties of Numerical Mathematics",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/TheUncertaintiesOfNumericalMathematics"]], "Tutorials",
 CellID->3856]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->27730],

Cell[TextData[{
 ButtonBox["Demonstrations with NIntegrate",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=NIntegrate"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#23656"], None}]], "RelatedLinks",
 CellID->11231],

Cell[TextData[ButtonBox["NIntegrate Explorer",
 BaseStyle->"Link",
 ButtonData->"paclet:GUIKit/tutorial/NIntegrateExplorer"]], "RelatedLinks",
 CellID->43772631],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
NIntegrate"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->50717085],

Cell[TextData[ButtonBox["Calculus",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Calculus"]], "MoreAbout",
 CellID->191589083],

Cell[TextData[ButtonBox["Functions of Complex Variables",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsOfComplexVariables"]], "MoreAbout",
 CellID->301030378],

Cell[TextData[ButtonBox["Numerical Evaluation & Precision",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumericalEvaluationAndPrecision"]], "MoreAbout",
 CellID->20599829],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->562518341]
}, Open  ]],

Cell["New in 1  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NIntegrate - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 2, 7.6853232}", "context" -> "System`", 
    "keywords" -> {
     "adaptive Monte Carlo", "adaptive Monte Carlo integration", 
      "adaptive quasi Monte Carlo", "adaptive quasi Monte Carlo integration", 
      "Cartesian product cubature", "cartesian rule", 
      "Clenshaw-Curtis integration", "Clenshaw-Curtis rule", 
      "composite quadrature", "cubature", "double-exponential", 
      "double exponential integration", "Gaussian quadrature", 
      "Gauss integration", "Gauss-Kronrod integration", "Gauss-Kronrod rule", 
      "Genz-Malik algorithm", "Genz-Malik cubature", "global adaptive", 
      "global adaptive integration", "integration", "Kronrod points", 
      "Las Vegas integration", "Lobatto-Kronrod integration", 
      "Lobatto-Kronrod rule", "local adaptive", "local adaptive integration", 
      "Monte Carlo", "Monte Carlo integration", "multidimensional rule", 
      "multi-panel quadrature", "multipanel rule", "Newton-Cotes integration",
       "Newton-Cotes rule", "numerical integration", "quadrature", 
      "quasi Monte Carlo", "quasi Monte Carlo integration", 
      "symbolic processing", "trapezoidal integration", "trapezoidal rule"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "NIntegrate[f, {x, x_min, x_max}] gives a numerical approximation to the \
integral SubsuperscriptBox[\\[Integral], x_min, x_max]\\ f\\ dx. \
NIntegrate[f, {x, x_min, x_max}, {y, y_min, y_max}, ...] gives a numerical \
approximation to the multiple integral SubsuperscriptBox[\\[Integral], x_min, \
x_max]d x SubsuperscriptBox[\\[Integral], y_min, y_max]dy\\ \\ ...\\ f.", 
    "synonyms" -> {}, "title" -> "NIntegrate", "type" -> "Symbol", "uri" -> 
    "ref/NIntegrate"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S0.1651"->{
  Cell[19495, 540, 372, 11, 70, "Notes",
   CellTags->{"S0.1651", "NDSolve"},
   CellID->306876545],
  Cell[19870, 553, 864, 19, 70, "2ColumnTable",
   CellTags->{"NDSolve", "S0.1651"},
   CellID->629005005],
  Cell[20737, 574, 1309, 44, 70, "Notes",
   CellTags->{"S0.1651", "NDSolve"},
   CellID->12228510],
  Cell[22049, 620, 425, 10, 70, "2ColumnTable",
   CellTags->{"NDSolve", "S0.1651"},
   CellID->604156095],
  Cell[22477, 632, 262, 7, 70, "Notes",
   CellTags->{"S0.1651", "NDSolve"},
   CellID->100636363],
  Cell[22742, 641, 965, 21, 70, "2ColumnTable",
   CellTags->{"NDSolve", "S0.1651"},
   CellID->578720793]},
 "NDSolve"->{
  Cell[19495, 540, 372, 11, 70, "Notes",
   CellTags->{"S0.1651", "NDSolve"},
   CellID->306876545],
  Cell[19870, 553, 864, 19, 70, "2ColumnTable",
   CellTags->{"NDSolve", "S0.1651"},
   CellID->629005005],
  Cell[20737, 574, 1309, 44, 70, "Notes",
   CellTags->{"S0.1651", "NDSolve"},
   CellID->12228510],
  Cell[22049, 620, 425, 10, 70, "2ColumnTable",
   CellTags->{"NDSolve", "S0.1651"},
   CellID->604156095],
  Cell[22477, 632, 262, 7, 70, "Notes",
   CellTags->{"S0.1651", "NDSolve"},
   CellID->100636363],
  Cell[22742, 641, 965, 21, 70, "2ColumnTable",
   CellTags->{"NDSolve", "S0.1651"},
   CellID->578720793]},
 "PrimaryExamplesSection"->{
  Cell[26004, 748, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->83853679]},
 "S3.9.10"->{
  Cell[52504, 1788, 622, 19, 70, "Input",
   CellTags->"S3.9.10",
   CellID->369070442],
  Cell[53129, 1809, 2579, 51, 126, "Output",
   CellTags->"S3.9.10",
   CellID->192777593]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S0.1651", 183400, 5315},
 {"NDSolve", 184041, 5334},
 {"PrimaryExamplesSection", 184697, 5353},
 {"S3.9.10", 184831, 5357}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3553, 78, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4176, 105, 4194, 83, 70, "ObjectNameGrid"],
Cell[8373, 190, 2836, 91, 70, "Usage",
 CellID->30088]
}, Open  ]],
Cell[CellGroupData[{
Cell[11246, 286, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[11724, 303, 194, 4, 70, "Notes",
 CellID->3224718],
Cell[11921, 309, 583, 16, 70, "Notes",
 CellID->18461],
Cell[12507, 327, 1289, 45, 70, "Notes",
 CellID->19295],
Cell[13799, 374, 69, 1, 70, "Notes",
 CellID->32652],
Cell[13871, 377, 3296, 86, 70, "3ColumnTableMod",
 CellID->23643],
Cell[17170, 465, 544, 15, 70, "Notes",
 CellID->90],
Cell[17717, 482, 580, 18, 70, "Notes",
 CellID->12747],
Cell[18300, 502, 655, 19, 70, "Notes",
 CellID->536560],
Cell[18958, 523, 534, 15, 70, "Notes",
 CellID->32525],
Cell[19495, 540, 372, 11, 70, "Notes",
 CellTags->{"S0.1651", "NDSolve"},
 CellID->306876545],
Cell[19870, 553, 864, 19, 70, "2ColumnTable",
 CellTags->{"NDSolve", "S0.1651"},
 CellID->629005005],
Cell[20737, 574, 1309, 44, 70, "Notes",
 CellTags->{"S0.1651", "NDSolve"},
 CellID->12228510],
Cell[22049, 620, 425, 10, 70, "2ColumnTable",
 CellTags->{"NDSolve", "S0.1651"},
 CellID->604156095],
Cell[22477, 632, 262, 7, 70, "Notes",
 CellTags->{"S0.1651", "NDSolve"},
 CellID->100636363],
Cell[22742, 641, 965, 21, 70, "2ColumnTable",
 CellTags->{"NDSolve", "S0.1651"},
 CellID->578720793],
Cell[23710, 664, 489, 16, 70, "Notes",
 CellID->94879207],
Cell[24202, 682, 382, 9, 70, "Notes",
 CellID->272103621],
Cell[24587, 693, 517, 18, 70, "Notes",
 CellID->15953],
Cell[25107, 713, 385, 11, 70, "Notes",
 CellID->21792],
Cell[25495, 726, 472, 17, 70, "Notes",
 CellID->12110]
}, Closed]],
Cell[CellGroupData[{
Cell[26004, 748, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->83853679],
Cell[CellGroupData[{
Cell[26388, 762, 148, 5, 70, "ExampleSection",
 CellID->150599599],
Cell[26539, 769, 73, 1, 70, "ExampleText",
 CellID->1875922869],
Cell[CellGroupData[{
Cell[26637, 774, 252, 8, 28, "Input",
 CellID->1692469869],
Cell[26892, 784, 176, 5, 36, "Output",
 CellID->392329368]
}, Open  ]],
Cell[27083, 792, 124, 3, 70, "ExampleDelimiter",
 CellID->48604888],
Cell[27210, 797, 121, 3, 70, "ExampleText",
 CellID->800751445],
Cell[CellGroupData[{
Cell[27356, 804, 586, 19, 70, "Input",
 CellID->888011411],
Cell[27945, 825, 175, 5, 36, "Output",
 CellID->25014712]
}, Open  ]],
Cell[28135, 833, 123, 3, 70, "ExampleDelimiter",
 CellID->2155493],
Cell[28261, 838, 80, 1, 70, "ExampleText",
 CellID->1631390226],
Cell[28344, 841, 632, 21, 70, "Input",
 CellID->1137508336],
Cell[CellGroupData[{
Cell[29001, 866, 178, 6, 70, "Input",
 CellID->800901481],
Cell[29182, 874, 178, 5, 36, "Output",
 CellID->803899748]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29409, 885, 217, 7, 70, "ExampleSection",
 CellID->27236],
Cell[CellGroupData[{
Cell[29651, 896, 264, 7, 70, "ExampleSubsection",
 CellID->99904585],
Cell[29918, 905, 80, 1, 70, "ExampleText",
 CellID->1319942373],
Cell[CellGroupData[{
Cell[30023, 910, 250, 8, 70, "Input",
 CellID->1396450524],
Cell[30276, 920, 175, 5, 36, "Output",
 CellID->146158737]
}, Open  ]],
Cell[30466, 928, 122, 3, 70, "ExampleDelimiter",
 CellID->301226],
Cell[30591, 933, 79, 1, 70, "ExampleText",
 CellID->1381566442],
Cell[CellGroupData[{
Cell[30695, 938, 486, 14, 70, "Input",
 CellID->2056248844],
Cell[31184, 954, 255, 7, 36, "Output",
 CellID->437454782]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[31488, 967, 264, 7, 70, "ExampleSubsection",
 CellID->660056409],
Cell[31755, 976, 78, 1, 70, "ExampleText",
 CellID->987442109],
Cell[CellGroupData[{
Cell[31858, 981, 345, 11, 70, "Input",
 CellID->1449220138],
Cell[32206, 994, 176, 5, 36, "Output",
 CellID->404652130]
}, Open  ]],
Cell[32397, 1002, 125, 3, 70, "ExampleDelimiter",
 CellID->124462083],
Cell[32525, 1007, 97, 1, 70, "ExampleText",
 CellID->1247122900],
Cell[CellGroupData[{
Cell[32647, 1012, 328, 11, 70, "Input",
 CellID->845371828],
Cell[32978, 1025, 176, 5, 36, "Output",
 CellID->777025848]
}, Open  ]],
Cell[33169, 1033, 125, 3, 70, "ExampleDelimiter",
 CellID->328358415],
Cell[33297, 1038, 77, 1, 70, "ExampleText",
 CellID->1575881246],
Cell[CellGroupData[{
Cell[33399, 1043, 294, 9, 70, "Input",
 CellID->2123707557],
Cell[33696, 1054, 256, 7, 36, "Output",
 CellID->870817440]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34001, 1067, 240, 7, 70, "ExampleSubsection",
 CellID->1651179549],
Cell[34244, 1076, 136, 4, 70, "ExampleText",
 CellID->979465677],
Cell[CellGroupData[{
Cell[34405, 1084, 349, 13, 70, "Input",
 CellID->1492018970],
Cell[34757, 1099, 192, 6, 36, "Output",
 CellID->13052430]
}, Open  ]],
Cell[34964, 1108, 124, 3, 70, "ExampleDelimiter",
 CellID->19547005],
Cell[35091, 1113, 120, 3, 70, "ExampleText",
 CellID->1728697943],
Cell[CellGroupData[{
Cell[35236, 1120, 678, 23, 70, "Input",
 CellID->5813051],
Cell[35917, 1145, 175, 5, 36, "Output",
 CellID->205278710]
}, Open  ]],
Cell[36107, 1153, 124, 3, 70, "ExampleDelimiter",
 CellID->10805887],
Cell[36234, 1158, 124, 3, 70, "ExampleText",
 CellID->41213695],
Cell[CellGroupData[{
Cell[36383, 1165, 296, 10, 70, "Input",
 CellID->30135056],
Cell[36682, 1177, 522, 9, 70, "Message",
 CellID->630146756],
Cell[37207, 1188, 619, 10, 70, "Message",
 CellID->219068490],
Cell[37829, 1200, 192, 6, 36, "Output",
 CellID->298620374]
}, Open  ]],
Cell[38036, 1209, 139, 4, 70, "ExampleText",
 CellID->542194155],
Cell[CellGroupData[{
Cell[38200, 1217, 307, 10, 70, "Input",
 CellID->102933999],
Cell[38510, 1229, 193, 6, 36, "Output",
 CellID->348156940]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[38752, 1241, 243, 7, 70, "ExampleSubsection",
 CellID->1055007759],
Cell[38998, 1250, 123, 3, 70, "ExampleText",
 CellID->1881162760],
Cell[CellGroupData[{
Cell[39146, 1257, 285, 10, 70, "Input",
 CellID->346592293],
Cell[39434, 1269, 178, 5, 36, "Output",
 CellID->710761742]
}, Open  ]],
Cell[39627, 1277, 125, 3, 70, "ExampleDelimiter",
 CellID->135153400],
Cell[39755, 1282, 73, 1, 70, "ExampleText",
 CellID->173809221],
Cell[CellGroupData[{
Cell[39853, 1287, 304, 11, 70, "Input",
 CellID->377732425],
Cell[40160, 1300, 194, 6, 36, "Output",
 CellID->42390940]
}, Open  ]],
Cell[40369, 1309, 125, 3, 70, "ExampleDelimiter",
 CellID->166759994],
Cell[40497, 1314, 73, 1, 70, "ExampleText",
 CellID->432778268],
Cell[CellGroupData[{
Cell[40595, 1319, 361, 12, 70, "Input",
 CellID->1472879406],
Cell[40959, 1333, 181, 5, 36, "Output",
 CellID->714385086]
}, Open  ]],
Cell[41155, 1341, 126, 3, 70, "ExampleDelimiter",
 CellID->1921328018],
Cell[41284, 1346, 75, 1, 70, "ExampleText",
 CellID->875859325],
Cell[CellGroupData[{
Cell[41384, 1351, 349, 12, 70, "Input",
 CellID->1343633055],
Cell[41736, 1365, 176, 5, 36, "Output",
 CellID->22916172]
}, Open  ]],
Cell[41927, 1373, 126, 3, 70, "ExampleDelimiter",
 CellID->1710898025],
Cell[42056, 1378, 199, 7, 70, "ExampleText",
 CellID->68293982],
Cell[CellGroupData[{
Cell[42280, 1389, 380, 13, 70, "Input",
 CellID->1753504606],
Cell[42663, 1404, 178, 5, 36, "Output",
 CellID->289777912]
}, Open  ]],
Cell[42856, 1412, 124, 3, 70, "ExampleDelimiter",
 CellID->73975753],
Cell[42983, 1417, 74, 1, 70, "ExampleText",
 CellID->902514234],
Cell[CellGroupData[{
Cell[43082, 1422, 1386, 43, 70, "Input",
 CellID->759892428],
Cell[44471, 1467, 176, 5, 36, "Output",
 CellID->106419010]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44696, 1478, 244, 7, 70, "ExampleSubsection",
 CellID->839801663],
Cell[44943, 1487, 124, 3, 70, "ExampleText",
 CellID->796164971],
Cell[45070, 1492, 449, 15, 70, "Input",
 CellID->80021104],
Cell[CellGroupData[{
Cell[45544, 1511, 424, 14, 70, "Input",
 CellID->683155504],
Cell[45971, 1527, 177, 5, 36, "Output",
 CellID->237544071]
}, Open  ]],
Cell[46163, 1535, 125, 3, 70, "ExampleDelimiter",
 CellID->292072618],
Cell[46291, 1540, 91, 1, 70, "ExampleText",
 CellID->12891935],
Cell[CellGroupData[{
Cell[46407, 1545, 999, 30, 70, "Input",
 CellID->188498931],
Cell[47409, 1577, 176, 5, 36, "Output",
 CellID->275956426]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[47646, 1589, 222, 7, 70, "ExampleSection",
 CellID->539005362],
Cell[CellGroupData[{
Cell[47893, 1600, 233, 7, 70, "ExampleSubsection",
 CellID->474483189],
Cell[48129, 1609, 255, 8, 70, "ExampleText",
 CellID->150473182],
Cell[CellGroupData[{
Cell[48409, 1621, 547, 19, 70, "Input",
 CellID->447216101],
Cell[48959, 1642, 216, 7, 51, "Output",
 CellID->830559185]
}, Open  ]],
Cell[49190, 1652, 136, 4, 70, "ExampleText",
 CellID->1979756711],
Cell[CellGroupData[{
Cell[49351, 1660, 603, 21, 70, "Input",
 CellID->963214242],
Cell[49957, 1683, 197, 6, 36, "Output",
 CellID->2885668]
}, Open  ]],
Cell[50169, 1692, 154, 4, 70, "ExampleText",
 CellID->1068336026],
Cell[CellGroupData[{
Cell[50348, 1700, 551, 20, 70, "Input",
 CellID->663400401],
Cell[50902, 1722, 180, 5, 36, "Output",
 CellID->115583411]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[51131, 1733, 238, 7, 70, "ExampleSubsection",
 CellID->328322763],
Cell[51372, 1742, 119, 3, 70, "ExampleText",
 CellID->353531660],
Cell[CellGroupData[{
Cell[51516, 1749, 493, 16, 70, "Input",
 CellID->1406978729],
Cell[52012, 1767, 222, 7, 36, "Output",
 CellID->433772487]
}, Open  ]],
Cell[52249, 1777, 124, 3, 70, "ExampleDelimiter",
 CellID->43543895],
Cell[52376, 1782, 103, 2, 70, "ExampleText",
 CellID->17998260],
Cell[CellGroupData[{
Cell[52504, 1788, 622, 19, 70, "Input",
 CellTags->"S3.9.10",
 CellID->369070442],
Cell[53129, 1809, 2579, 51, 126, "Output",
 CellTags->"S3.9.10",
 CellID->192777593]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[55757, 1866, 230, 7, 70, "ExampleSubsection",
 CellID->96867093],
Cell[55990, 1875, 135, 4, 70, "ExampleText",
 CellID->1830794989],
Cell[CellGroupData[{
Cell[56150, 1883, 630, 22, 70, "Input",
 CellID->1049131592],
Cell[56783, 1907, 272, 7, 36, "Output",
 CellID->84628818]
}, Open  ]],
Cell[57070, 1917, 89, 1, 70, "ExampleText",
 CellID->124341040],
Cell[CellGroupData[{
Cell[57184, 1922, 412, 14, 70, "Input",
 CellID->1067034414],
Cell[57599, 1938, 41244, 708, 70, "Output",
 CellID->88069420]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[98892, 2652, 230, 7, 70, "ExampleSubsection",
 CellID->125944362],
Cell[99125, 2661, 122, 3, 70, "ExampleText",
 CellID->1831979707],
Cell[CellGroupData[{
Cell[99272, 2668, 319, 9, 70, "Input",
 CellID->969050975],
Cell[99594, 2679, 336, 7, 70, "Message",
 CellID->202512233],
Cell[99933, 2688, 176, 5, 36, "Output",
 CellID->405578365]
}, Open  ]],
Cell[CellGroupData[{
Cell[100146, 2698, 756, 21, 70, "Input",
 CellID->751519308],
Cell[100905, 2721, 337, 7, 70, "Message",
 CellID->444265883],
Cell[101245, 2730, 176, 5, 36, "Output",
 CellID->155002268]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[101470, 2741, 232, 7, 70, "ExampleSubsection",
 CellID->68379317],
Cell[101705, 2750, 120, 3, 70, "ExampleText",
 CellID->313888073],
Cell[CellGroupData[{
Cell[101850, 2757, 280, 9, 70, "Input",
 CellID->1988258193],
Cell[102133, 2768, 522, 9, 70, "Message",
 CellID->218108592],
Cell[102658, 2779, 716, 12, 70, "Message",
 CellID->551780816],
Cell[103377, 2793, 272, 7, 36, "Output",
 CellID->244994430]
}, Open  ]],
Cell[103664, 2803, 248, 8, 70, "ExampleText",
 CellID->416954400],
Cell[CellGroupData[{
Cell[103937, 2815, 332, 10, 70, "Input",
 CellID->682978387],
Cell[104272, 2827, 522, 9, 70, "Message",
 CellID->244035193],
Cell[104797, 2838, 273, 7, 36, "Output",
 CellID->477526979]
}, Open  ]],
Cell[105085, 2848, 103, 2, 70, "ExampleText",
 CellID->4837156],
Cell[CellGroupData[{
Cell[105213, 2854, 373, 11, 70, "Input",
 CellID->730714105],
Cell[105589, 2867, 272, 7, 36, "Output",
 CellID->149202581]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[105910, 2880, 226, 7, 70, "ExampleSubsection",
 CellID->20747351],
Cell[106139, 2889, 88, 1, 70, "ExampleText",
 CellID->948959124],
Cell[CellGroupData[{
Cell[106252, 2894, 416, 14, 70, "Input",
 CellID->609781982],
Cell[106671, 2910, 243, 8, 36, "Output",
 CellID->222419039]
}, Open  ]],
Cell[106929, 2921, 97, 1, 70, "ExampleText",
 CellID->1718627431],
Cell[CellGroupData[{
Cell[107051, 2926, 594, 18, 70, "Input",
 CellID->1385782705],
Cell[107648, 2946, 244, 8, 36, "Output",
 CellID->306457727]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[107941, 2960, 233, 7, 70, "ExampleSubsection",
 CellID->250171837],
Cell[108177, 2969, 218, 7, 70, "ExampleText",
 CellID->813625102],
Cell[CellGroupData[{
Cell[108420, 2980, 488, 17, 70, "Input",
 CellID->358756462],
Cell[108911, 2999, 605, 10, 70, "Message",
 CellID->70840692],
Cell[109519, 3011, 160, 5, 36, "Output",
 CellID->644736936]
}, Open  ]],
Cell[109694, 3019, 177, 5, 70, "ExampleText",
 CellID->947812824],
Cell[CellGroupData[{
Cell[109896, 3028, 539, 18, 70, "Input",
 CellID->378241225],
Cell[110438, 3048, 178, 5, 36, "Output",
 CellID->170615515]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[110665, 3059, 234, 7, 70, "ExampleSubsection",
 CellID->585718500],
Cell[110902, 3068, 501, 17, 70, "ExampleText",
 CellID->2038745],
Cell[111406, 3087, 853, 23, 70, "Input",
 CellID->79454938],
Cell[112262, 3112, 265, 8, 70, "ExampleText",
 CellID->184300085],
Cell[CellGroupData[{
Cell[112552, 3124, 225, 7, 70, "Input",
 CellID->772163697],
Cell[112780, 3133, 340, 10, 36, "Output",
 CellID->293817646]
}, Open  ]],
Cell[CellGroupData[{
Cell[113157, 3148, 167, 5, 70, "Input",
 CellID->62454419],
Cell[113327, 3155, 2156, 39, 70, "Output",
 Evaluatable->False,
 CellID->709142616]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[115532, 3200, 237, 7, 70, "ExampleSubsection",
 CellID->364871850],
Cell[115772, 3209, 235, 7, 70, "ExampleText",
 CellID->321720514],
Cell[CellGroupData[{
Cell[116032, 3220, 409, 13, 70, "Input",
 CellID->448363667],
Cell[116444, 3235, 284, 7, 54, "Output",
 CellID->103115510]
}, Open  ]],
Cell[116743, 3245, 364, 13, 70, "ExampleText",
 CellID->395073585],
Cell[CellGroupData[{
Cell[117132, 3262, 463, 14, 70, "Input",
 CellID->105397796],
Cell[117598, 3278, 284, 7, 54, "Output",
 CellID->141116667]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[117943, 3292, 227, 7, 70, "ExampleSection",
 CellID->669772564],
Cell[118173, 3301, 587, 22, 70, "ExampleText",
 CellID->136564390],
Cell[118763, 3325, 524, 15, 70, "Input",
 CellID->957268230],
Cell[CellGroupData[{
Cell[119312, 3344, 260, 9, 70, "Input",
 CellID->2492432],
Cell[119575, 3355, 226, 7, 36, "Output",
 CellID->270059300]
}, Open  ]],
Cell[119816, 3365, 454, 17, 70, "ExampleText",
 CellID->236725317],
Cell[CellGroupData[{
Cell[120295, 3386, 209, 6, 70, "Input",
 CellID->945494830],
Cell[120507, 3394, 176, 5, 36, "Output",
 CellID->474826046]
}, Open  ]],
Cell[120698, 3402, 124, 3, 70, "ExampleDelimiter",
 CellID->14335361],
Cell[120825, 3407, 139, 4, 70, "ExampleText",
 CellID->544988733],
Cell[CellGroupData[{
Cell[120989, 3415, 940, 33, 70, "Input",
 CellID->1392412280],
Cell[121932, 3450, 4466, 77, 70, "Output",
 Evaluatable->False,
 CellID->377873674]
}, Open  ]],
Cell[CellGroupData[{
Cell[126435, 3532, 350, 12, 70, "Input",
 CellID->180897631],
Cell[126788, 3546, 175, 5, 36, "Output",
 CellID->258479703]
}, Open  ]],
Cell[126978, 3554, 125, 3, 70, "ExampleDelimiter",
 CellID->258581796],
Cell[127106, 3559, 233, 8, 70, "ExampleText",
 CellID->128269499],
Cell[CellGroupData[{
Cell[127364, 3571, 557, 20, 70, "Input",
 CellID->271271658],
Cell[127924, 3593, 2624, 47, 70, "Output",
 Evaluatable->False,
 CellID->198431993]
}, Open  ]],
Cell[130563, 3643, 1066, 34, 70, "Input",
 CellID->16513875],
Cell[131632, 3679, 115, 3, 70, "ExampleText",
 CellID->438948190],
Cell[CellGroupData[{
Cell[131772, 3686, 1109, 33, 70, "Input",
 CellID->734908557],
Cell[132884, 3721, 2220, 40, 70, "Output",
 Evaluatable->False,
 CellID->138879792]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[135153, 3767, 237, 7, 70, "ExampleSection",
 CellID->251350421],
Cell[135393, 3776, 380, 13, 70, "ExampleText",
 CellID->1300109756],
Cell[CellGroupData[{
Cell[135798, 3793, 354, 12, 70, "Input",
 CellID->2104185480],
Cell[136155, 3807, 282, 11, 51, "Output",
 CellID->168744022]
}, Open  ]],
Cell[136452, 3821, 230, 8, 70, "ExampleText",
 CellID->1161879132],
Cell[CellGroupData[{
Cell[136707, 3833, 354, 12, 70, "Input",
 CellID->693688742],
Cell[137064, 3847, 195, 6, 36, "Output",
 CellID->309170504]
}, Open  ]],
Cell[137274, 3856, 124, 3, 70, "ExampleDelimiter",
 CellID->39928257],
Cell[137401, 3861, 327, 12, 70, "ExampleText",
 CellID->16875335],
Cell[CellGroupData[{
Cell[137753, 3877, 585, 20, 70, "Input",
 CellID->800886492],
Cell[138341, 3899, 395, 14, 36, "Output",
 CellID->23531646]
}, Open  ]],
Cell[CellGroupData[{
Cell[138773, 3918, 205, 7, 70, "Input",
 CellID->791354346],
Cell[138981, 3927, 195, 6, 36, "Output",
 CellID->35156568]
}, Open  ]],
Cell[139191, 3936, 261, 8, 70, "ExampleText",
 CellID->427358720],
Cell[CellGroupData[{
Cell[139477, 3948, 262, 9, 70, "Input",
 CellID->1299148167],
Cell[139742, 3959, 198, 6, 36, "Output",
 CellID->53526948]
}, Open  ]],
Cell[139955, 3968, 125, 3, 70, "ExampleDelimiter",
 CellID->516429567],
Cell[140083, 3973, 121, 3, 70, "ExampleText",
 CellID->743109189],
Cell[140207, 3978, 201, 7, 70, "Input",
 CellID->1445681194],
Cell[CellGroupData[{
Cell[140433, 3989, 1458, 49, 70, "Input",
 CellID->672681917],
Cell[141894, 4040, 176, 5, 36, "Output",
 CellID->228174824]
}, Open  ]],
Cell[142085, 4048, 136, 4, 70, "ExampleText",
 CellID->59565138],
Cell[CellGroupData[{
Cell[142246, 4056, 272, 10, 70, "Input",
 CellID->1740679851],
Cell[142521, 4068, 198, 6, 36, "Output",
 CellID->275496570]
}, Open  ]],
Cell[142734, 4077, 474, 18, 70, "ExampleText",
 CellID->51850576],
Cell[CellGroupData[{
Cell[143233, 4099, 410, 13, 70, "Input",
 CellID->2001488439],
Cell[143646, 4114, 176, 5, 36, "Output",
 CellID->235034182]
}, Open  ]],
Cell[143837, 4122, 124, 3, 70, "ExampleDelimiter",
 CellID->92174765],
Cell[143964, 4127, 134, 4, 70, "ExampleText",
 CellID->651107854],
Cell[CellGroupData[{
Cell[144123, 4135, 352, 11, 70, "Input",
 CellID->1510519880],
Cell[144478, 4148, 176, 5, 36, "Output",
 CellID->843275400]
}, Open  ]],
Cell[144669, 4156, 191, 7, 70, "ExampleText",
 CellID->568523091],
Cell[CellGroupData[{
Cell[144885, 4167, 195, 6, 70, "Input",
 CellID->1311869428],
Cell[145083, 4175, 174, 5, 36, "Output",
 CellID->5458053]
}, Open  ]],
Cell[145272, 4183, 124, 3, 70, "ExampleDelimiter",
 CellID->96388514],
Cell[145399, 4188, 244, 8, 70, "ExampleText",
 CellID->94267959],
Cell[CellGroupData[{
Cell[145668, 4200, 347, 12, 70, "Input",
 CellID->225328397],
Cell[146018, 4214, 207, 7, 51, "Output",
 CellID->38510876]
}, Open  ]],
Cell[146240, 4224, 294, 10, 70, "ExampleText",
 CellID->781895585],
Cell[CellGroupData[{
Cell[146559, 4238, 479, 17, 70, "Input",
 CellID->67972935],
Cell[147041, 4257, 227, 7, 36, "Output",
 CellID->18582942]
}, Open  ]],
Cell[147283, 4267, 125, 3, 70, "ExampleDelimiter",
 CellID->209142943],
Cell[147411, 4272, 95, 1, 70, "ExampleText",
 CellID->487508788],
Cell[CellGroupData[{
Cell[147531, 4277, 984, 33, 70, "Input",
 CellID->175075343],
Cell[148518, 4312, 274, 9, 36, "Output",
 CellID->449554829]
}, Open  ]],
Cell[CellGroupData[{
Cell[148829, 4326, 335, 11, 70, "Input",
 CellID->201420606],
Cell[149167, 4339, 243, 8, 36, "Output",
 CellID->306056226]
}, Open  ]],
Cell[149425, 4350, 119, 3, 70, "ExampleText",
 CellID->47202774],
Cell[CellGroupData[{
Cell[149569, 4357, 153, 5, 70, "Input",
 CellID->26534081],
Cell[149725, 4364, 284, 7, 54, "Output",
 CellID->370621071]
}, Open  ]],
Cell[CellGroupData[{
Cell[150046, 4376, 440, 14, 70, "Input",
 CellID->58913301],
Cell[150489, 4392, 294, 7, 54, "Output",
 CellID->859292989]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[150832, 4405, 230, 7, 70, "ExampleSection",
 CellID->171909922],
Cell[151065, 4414, 145, 4, 70, "ExampleText",
 CellID->24555619],
Cell[CellGroupData[{
Cell[151235, 4422, 282, 10, 70, "Input",
 CellID->1939490000],
Cell[151520, 4434, 602, 10, 70, "Message",
 CellID->444464910],
Cell[152125, 4446, 160, 5, 36, "Output",
 CellID->196502837]
}, Open  ]],
Cell[152300, 4454, 124, 3, 70, "ExampleText",
 CellID->549328100],
Cell[CellGroupData[{
Cell[152449, 4461, 333, 11, 70, "Input",
 CellID->153548073],
Cell[152785, 4474, 178, 5, 36, "Output",
 CellID->556991540]
}, Open  ]],
Cell[152978, 4482, 125, 3, 70, "ExampleDelimiter",
 CellID->127029480],
Cell[153106, 4487, 148, 4, 70, "ExampleText",
 CellID->273091344],
Cell[CellGroupData[{
Cell[153279, 4495, 430, 14, 70, "Input",
 CellID->1855351150],
Cell[153712, 4511, 522, 9, 70, "Message",
 CellID->284986777],
Cell[154237, 4522, 1009, 15, 70, "Message",
 CellID->86330940],
Cell[155249, 4539, 245, 8, 39, "Output",
 CellID->539713215]
}, Open  ]],
Cell[155509, 4550, 260, 8, 70, "ExampleText",
 CellID->197471965],
Cell[CellGroupData[{
Cell[155794, 4562, 488, 16, 70, "Input",
 CellID->89566389],
Cell[156285, 4580, 247, 8, 39, "Output",
 CellID->180079319]
}, Open  ]],
Cell[156547, 4591, 125, 3, 70, "ExampleDelimiter",
 CellID->111789566],
Cell[156675, 4596, 132, 4, 70, "ExampleText",
 CellID->339226259],
Cell[CellGroupData[{
Cell[156832, 4604, 962, 29, 70, "Input",
 CellID->1691852158],
Cell[157797, 4635, 229, 7, 36, "Output",
 CellID->170810035]
}, Open  ]],
Cell[158041, 4645, 234, 8, 70, "ExampleText",
 CellID->141219965],
Cell[CellGroupData[{
Cell[158300, 4657, 1014, 30, 70, "Input",
 CellID->121416176],
Cell[159317, 4689, 245, 8, 36, "Output",
 CellID->867229128]
}, Open  ]],
Cell[159577, 4700, 262, 8, 70, "ExampleText",
 CellID->586272819],
Cell[CellGroupData[{
Cell[159864, 4712, 1033, 30, 70, "Input",
 CellID->86508208],
Cell[160900, 4744, 245, 8, 36, "Output",
 CellID->72714948]
}, Open  ]],
Cell[161160, 4755, 324, 10, 70, "ExampleText",
 CellID->1689058454],
Cell[CellGroupData[{
Cell[161509, 4769, 1640, 46, 70, "Input",
 CellID->1485004164],
Cell[163152, 4817, 244, 8, 36, "Output",
 CellID->349937733]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[163445, 4831, 228, 7, 70, "ExampleSection",
 CellID->102225208],
Cell[163676, 4840, 124, 3, 70, "ExampleText",
 CellID->17442091],
Cell[163803, 4845, 1733, 50, 70, "Input",
 CellID->277430143],
Cell[CellGroupData[{
Cell[165561, 4899, 142, 4, 70, "Input",
 CellID->36883129],
Cell[165706, 4905, 9877, 166, 70, "Output",
 Evaluatable->False,
 CellID->4139611]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[175644, 5078, 311, 9, 70, "SeeAlsoSection",
 CellID->23034],
Cell[175958, 5089, 688, 26, 70, "SeeAlso",
 CellID->28501]
}, Open  ]],
Cell[CellGroupData[{
Cell[176683, 5120, 312, 9, 70, "TutorialsSection",
 CellID->497],
Cell[176998, 5131, 307, 9, 70, "Tutorials",
 CellID->206418781],
Cell[177308, 5142, 193, 4, 70, "Tutorials",
 CellID->123709578],
Cell[177504, 5148, 152, 3, 70, "Tutorials",
 CellID->23256],
Cell[177659, 5153, 185, 4, 70, "Tutorials",
 CellID->20805],
Cell[177847, 5159, 323, 9, 70, "Tutorials",
 CellID->4112],
Cell[178173, 5170, 193, 4, 70, "Tutorials",
 CellID->3856]
}, Open  ]],
Cell[CellGroupData[{
Cell[178403, 5179, 321, 9, 70, "RelatedLinksSection",
 CellID->27730],
Cell[178727, 5190, 365, 12, 70, "RelatedLinks"],
Cell[179095, 5204, 267, 6, 70, "RelatedLinks",
 CellID->11231],
Cell[179365, 5212, 161, 3, 70, "RelatedLinks",
 CellID->43772631],
Cell[179529, 5217, 388, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[179954, 5235, 318, 9, 70, "MoreAboutSection",
 CellID->50717085],
Cell[180275, 5246, 128, 3, 70, "MoreAbout",
 CellID->191589083],
Cell[180406, 5251, 169, 3, 70, "MoreAbout",
 CellID->301030378],
Cell[180578, 5256, 174, 3, 70, "MoreAbout",
 CellID->20599829],
Cell[180755, 5261, 179, 3, 70, "MoreAbout",
 CellID->562518341]
}, Open  ]],
Cell[180949, 5267, 50, 0, 70, "History"],
Cell[181002, 5269, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

