(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     88094,       3184]
NotebookOptionsPosition[     71436,       2598]
NotebookOutlinePosition[     72981,       2639]
CellTagsIndexPosition[     72895,       2634]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numerical Mathematics: Basic Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalMathematics-BasicOperations"], 
          "Numerical Sums, Products and Integrals" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalSumsProductsAndIntegrals"], 
          "Numerical Mathematics in Mathematica" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalMathematicsInMathematica"], 
          "Numerical Evaluation of Sums and Products" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalEvaluationOfSumsAndProducts"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Numerical Mathematics: Basic Operations\"\>", 
       2->"\<\"Numerical Sums, Products and Integrals\"\>", 
       3->"\<\"Numerical Mathematics in Mathematica\"\>", 
       4->"\<\"Numerical Evaluation of Sums and Products\"\>"}, \
"\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Sum" :> Documentation`HelpLookup["paclet:ref/Sum"], "NProduct" :> 
          Documentation`HelpLookup["paclet:ref/NProduct"], "Total" :> 
          Documentation`HelpLookup["paclet:ref/Total"], "NIntegrate" :> 
          Documentation`HelpLookup["paclet:ref/NIntegrate"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Sum\"\>", 2->"\<\"NProduct\"\>", 
       3->"\<\"Total\"\>", 4->"\<\"NIntegrate\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Calculus" :> Documentation`HelpLookup["paclet:guide/Calculus"],
           "Numerical Evaluation & Precision" :> 
          Documentation`HelpLookup[
           "paclet:guide/NumericalEvaluationAndPrecision"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Calculus\"\>", 
       2->"\<\"Numerical Evaluation & Precision\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["NSum", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NSum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NSum"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["i", "TI"], ",", 
          SubscriptBox[
           StyleBox["i", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["i", "TI"], 
           StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a numerical approximation to the sum ",
     Cell[BoxData[
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{
         StyleBox["i", "TI"], "=", 
         SubscriptBox[
          StyleBox["i", "TI"], 
          StyleBox["min", "TI"]]}], 
        SubscriptBox[
         StyleBox["i", "TI"], 
         StyleBox["max", "TI"]],
        LimitsPositioning->True], " ", 
       StyleBox["f", "TI"]}]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NSum",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NSum"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["i", "TI"], ",", 
          SubscriptBox[
           StyleBox["i", "TI"], 
           StyleBox["min", "TI"]], ",", 
          SubscriptBox[
           StyleBox["i", "TI"], 
           StyleBox["max", "TI"]], ",", 
          StyleBox["di", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]uses a step ",
     Cell[BoxData[
      StyleBox["di", "TI"]], "InlineFormula"],
     " in the sum. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->32720]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " can be used for sums with both finite and infinite limits. "
}], "Notes",
 CellID->28306],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["NSum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NSum"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["i", "TI"], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["j", "TI"], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " can be used to evaluate multidimensional sums. "
}], "Notes",
 CellID->14336],

Cell["The following options can be given: ", "Notes",
 CellID->2566],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Infinity",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Infinity"], Cell[
    "number of digits of final accuracy sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["EvaluationMonitor",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/EvaluationMonitor"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "expression to evaluate whenever ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " is evaluated "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "method to use", "TableText"]},
   {Cell["      ", "TableRowIcon"], "NSumTerms", "15", Cell[
    "number of terms to use before extrapolation ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of digits of final precision sought ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "VerifyConvergence", 
    ButtonBox["True",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/True"], Cell[
    "whether to explicitly test for convergence ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "the precision used in internal computations ", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "3ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->31410],

Cell[TextData[{
 "Possible settings for the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option include:"
}], "Notes",
 CellID->24422],

Cell[BoxData[GridBox[{
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], "\"\<AlternatingSigns\>\"", Cell[
    "method for summands with alternating signs", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<EulerMaclaurin\>\"", Cell["\<\
Euler\[Dash]Maclaurin summation method\
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<WynnEpsilon\>\"", Cell[
    "Wynn epsilon extrapolation method", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True}}}},
 CellID->3159],

Cell[TextData[{
 "With the Euler",
 "\[Dash]",
 "Maclaurin method, the options ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " can be used to specify the accuracy and precision to try and get in the \
final answer. ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " stops when the error estimates it gets imply that either the accuracy or \
precision sought has been reached. "
}], "Notes",
 CellID->14026],

Cell[TextData[{
 "You should realize that with sufficiently pathological summands, the \
algorithms used by ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " can give wrong answers. In most cases, you can test the answer by looking \
at its sensitivity to changes in the setting of options for ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->7244],

Cell[TextData[{
 Cell[BoxData["VerifyConvergence"], "InlineFormula"],
 " is only used for sums with infinite limits. "
}], "Notes",
 CellID->4897],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "[", 
   RowBox[{
    ButtonBox["Sum",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sum"], "[", 
    StyleBox["\[Ellipsis]", "TR"], "]"}], "]"}]], "InlineFormula"],
 " calls ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->30405],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " first localizes the values of all variables, then evaluates ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " with the variables being symbolic, and then repeatedly evaluates the \
result numerically."
}], "Notes",
 CellID->30946],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " has attribute ",
 Cell[BoxData[
  ButtonBox["HoldAll",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HoldAll"]], "InlineFormula"],
 ", and effectively uses ",
 Cell[BoxData[
  ButtonBox["Block",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Block"]], "InlineFormula"],
 " to localize variables."
}], "Notes",
 CellID->6944]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->269800189],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->232714729],

Cell["Approximate an infinite sum numerically:", "ExampleText",
 CellID->226825228],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"-", "5"}], ")"}], "^", "i"}], "/", 
    RowBox[{"i", "!"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"i", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12053745],

Cell[BoxData["0.006737946999081246`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25454835]
}, Open  ]],

Cell[TextData[{
 "The error versus the exact value of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", "5"}]], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->90781056],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Exp", "[", 
   RowBox[{"-", "5"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->39174757],

Cell[BoxData[
 RowBox[{"-", "4.220582217051572`*^-15"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->150339448]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->71768073],

Cell["Approximate the value of a finite sum:", "ExampleText",
 CellID->715570151],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"i", "^", "2"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"i", ",", " ", "100", ",", " ", 
     RowBox[{"10", "^", "6"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->73359430],

Cell[BoxData["0.010049166663833481`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->173194495]
}, Open  ]],

Cell["This is effectively the difference of two infinite sums:", "ExampleText",
 CellID->39454065],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"i", "^", "2"}]}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"i", ",", " ", "100", ",", " ", "\[Infinity]"}], "}"}]}], "]"}], 
  " ", "-", " ", 
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"i", "^", "2"}]}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"i", ",", "  ", 
      RowBox[{"10", "^", "6"}], ",", " ", "\[Infinity]"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->42115105],

Cell[BoxData["0.010049166662833496`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->285213080]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->88535772],

Cell["Approximate a multidimensional sum:", "ExampleText",
 CellID->39868800],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "^", "n"}], 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"2", "/", "n"}], ")"}], "^", "k"}], "/", 
     RowBox[{"k", "^", "2"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "2", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->552226847],

Cell[BoxData["1.1443442096037666`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->278864469]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19975727],

Cell["\<\
Approximate a multidimensional sum with the second index depending on the \
first:\
\>", "ExampleText",
 CellID->923515544],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "^", "n"}], 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"2", "/", "n"}], ")"}], "^", "k"}], "/", 
     RowBox[{"k", "^", "2"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "2", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "1", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->945501495],

Cell[BoxData["0.7701875843536914`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->13582748]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->40547156],

Cell["Complex series summation:", "ExampleText",
 CellID->266585647],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Log", "[", "x", "]"}], "/", 
    RowBox[{"x", "^", 
     RowBox[{"(", 
      RowBox[{"2", "+", 
       RowBox[{"2", "I"}]}], ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->336884800],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "0.18217540238695298`"}], "-", 
  RowBox[{"0.13661831674945685`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{148, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->263118562]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->220715922],

Cell["Sum the even terms of a series:", "ExampleText",
 CellID->144564436],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"2", "^", "i"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "0", ",", "\[Infinity]", ",", "2"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->310085196],

Cell[BoxData["1.3333333333333333`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->568702208]
}, Open  ]],

Cell["An equivalent way of specifying the same sum:", "ExampleText",
 CellID->105472116],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"2", "^", 
     RowBox[{"(", 
      RowBox[{"2", "j"}], ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"j", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->642821719],

Cell[BoxData["1.3333333333333333`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->484750109]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1831],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "AccuracyGoal and PrecisionGoal",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->906276870],

Cell["Approximate a sum with the default tolerances:", "ExampleText",
 CellID->313601664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"n", "^", 
     RowBox[{"(", 
      RowBox[{"11", "/", "10"}], ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->518750079],

Cell[BoxData["10.584448464728068`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->332262193]
}, Open  ]],

Cell[TextData[{
 "Find the error versus the exact value of ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{"\[Zeta]", "(", 
     TagBox[
      FractionBox["11", "10"],
      Zeta,
      Editable->True,
      Selectable->True], ")"}],
    InterpretTemplate[
     Function[BoxForm`e$, 
      Zeta[BoxForm`e$]]],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->482113350],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Zeta", "[", 
   RowBox[{"11", "/", "10"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->129873791],

Cell[BoxData[
 RowBox[{"-", "2.227338313787186`*^-10"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->272793093]
}, Open  ]],

Cell["The error with smaller absolute and relative tolerances:", "ExampleText",
 CellID->76531130],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"n", "^", 
      RowBox[{"(", 
       RowBox[{"11", "/", "10"}], ")"}]}]}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}], ",", " ", 
    RowBox[{"AccuracyGoal", "\[Rule]", "4"}], ",", " ", 
    RowBox[{"PrecisionGoal", "\[Rule]", "4"}]}], "]"}], " ", "-", " ", 
  RowBox[{"Zeta", "[", 
   RowBox[{"11", "/", "10"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->105238400],

Cell[BoxData["1.1091653817629776`*^-8"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->165415353]
}, Open  ]],

Cell["The error with larger absolute and relative tolerances:", "ExampleText",
 CellID->361447028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"n", "^", 
      RowBox[{"(", 
       RowBox[{"11", "/", "10"}], ")"}]}]}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}], ",", " ", 
    RowBox[{"AccuracyGoal", "\[Rule]", "10"}], ",", " ", 
    RowBox[{"PrecisionGoal", "\[Rule]", "10"}]}], "]"}], " ", "-", " ", 
  RowBox[{"Zeta", "[", 
   RowBox[{"11", "/", "10"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->394312397],

Cell[BoxData["9.36140054363932`*^-13"], "Output",
 ImageSize->{87, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->7136769]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "EvaluationMonitor",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->103639796],

Cell["\<\
Get the number of evaluation points used in a numerical summation:\
\>", "ExampleText",
 CellID->353531660],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"k", "=", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"NSum", "[", 
      RowBox[{
       FractionBox["1", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"n", "+", "1"}], ")"}], "2"]], ",", 
       RowBox[{"{", 
        RowBox[{"n", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"k", "++"}]}]}], "]"}], ",", "k"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->711990419],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.6449340667600105`", ",", "114"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5061741]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->264789900],

Cell["\<\
The evaluation points used in a summation approximated by an integration \
method:\
\>", "ExampleText",
 CellID->17998260],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Reap", "[", 
    RowBox[{"NSum", "[", 
     RowBox[{
      FractionBox["1", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"n", "+", "1"}], ")"}], "2"]], ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", "n", "]"}]}]}], "]"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->650141528],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlH9MlHUcx8/jEBok9hMKnYiK5ZzeA3LIL+/NrzuUO+R+/yI2bmbO4XIG
WFTuQctpRbS2GqjTrVwp5Gqu/CNjHaZNi61kXC5cvzYrPVzeTJRlbSnP++PG
82y3Z+/3+3nfPs/r+32+C6PPup82GgyGbXd+d+/albQaZl6Y6adjpj8XM/2H
dX6Ozp+n8/N0fr7OX6zzC3T+Ezp/mc5frvNX6HwzOqfSQu0drdQKlm3Z9feh
I1XUhdjz2qyWE/kW6iL8/Nfe+WFfIfUqVEzGM7782kxdfO9/tbsF3eYPCsrz
VlKXYNM261Bfv8yxGkq+fThmF12KRbcPlhY9JroMmWWzQ7km0eXavO9spa6A
Z3z2n1aXg7oSWY+kfNFWt4Z6DYZ+eqj80wGZ34oXEoPZO3qKqHHvPadvKjiv
Qr+K8/J91CrOK+9XzXklr+a8ktfg4q4bb2cYOb9ag73XOxY5MuV9apF48qXJ
3Qslr8Wpf06mnq2WvA5z9rckOrZKXofBgbuX5Db0vWwc/iEpuQ0V3xZ8ZIDw
tmOp8lTxlV5q1Y6L0cHz80clr8dIT0p34pbk9RrfJSrztRrf80Hma8nXznwd
uk4u7dzkszJfh+tjgZX/Pl7KvIF38lcbYGx1ODuxir4D313a0aUUcT1UB6K1
b+7s7pX95YTtZuVzGwtlfZzIac88bkmXvJF8hX8j+Qr/9eQr+XrylbyJfKnR
RL7yfBP5UseayFf2h4t8qeEiX2rVRb7UMRcmklnHL4/3su/W+D7wDPtuje8Z
D/tu8q1n302+1ezzOQP3Ozwwnerbd1UtY9+Dg+f+uxHduJp9D07vmZeasqCE
PS/euPriHyMbuD7wwnw5Z8OWubJeXpxIu7Ik20TeMS8e/H3C1pcj/H0Y80wl
7q+hhg+vTG7PeOtVWS8fci1DYxO/St8H3/Ql358fym3TzmO/UcOP5991Bmep
3B+qH43RzzePLuC5EvPjszP+98sPyzkTQH9umjF+jRoBje97h9gPaHyVTvYD
Gt/UKPtBjW/cy34Ql45+3DJV7GSf+97A/R4Lkm8N/RD58rxEiHx5rqghZP94
a6T/HNcnFkLr98rZ8Tw5n8JoyxptWJEu/MNY/M2FHrMi/MM48mEynrubOhZG
2sS1R/MyZP0isLR1Hb0wTI0IprfvANdbjSB68z5T5Ct+j7EIksv3f2K8c45q
/WYcON0eP2ytYL8Zm18/ZtteX8l+MwpL5pgGfqnE/6aDmIs=
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 114.}, {0., 64.10498721159531}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61104135]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->251879244],

Cell["\<\
The evaluation points used in a summation approximated by the sequence \
extrapolation method:\
\>", "ExampleText",
 CellID->606763501],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Reap", "[", 
    RowBox[{"NSum", "[", 
     RowBox[{
      FractionBox[
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], "n"], 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"n", "+", "1"}], ")"}], "2"]], ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", "n", "]"}]}]}], "]"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"2", ",", "1"}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->302137647],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], 
    PointBox[{{1., 15.}, {2., 16.}, {3., 17.}, {4., 18.}, {5., 19.}, {6., 
     20.}, {7., 21.}, {8., 22.}, {9., 23.}, {10., 24.}, {11., 25.}, {12., 
     26.}, {13., 0.}, {14., 1.}, {15., 2.}, {16., 3.}, {17., 4.}, {18., 5.}, {
     19., 6.}, {20., 7.}, {21., 8.}, {22., 9.}, {23., 10.}, {24., 11.}, {25., 
     12.}, {26., 13.}, {27., 14.}}]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 27.}, {0., 26.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->175323937]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Method",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->4900274],

Cell["\<\
Use a method for alternating series to get a very precise sum approximation:\
\>", "ExampleText",
 CellID->385344373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "^", "x"}], "/", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"x", "-", "12"}], ")"}], "^", "2"}]}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<AlternatingSigns\>\""}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "30"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->488498118],

Cell[BoxData["0.275193859413388147454815603683435635703403959717`29.\
44391026679884"], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23098839]
}, Open  ]],

Cell[TextData[{
 "The error from the exact result computed by ",
 Cell[BoxData[
  ButtonBox["Sum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sum"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->138635920],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], "^", "x"}], "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "12"}], ")"}], "^", "2"}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->81223570],

Cell[BoxData["2.318916656366212551729`4.369556667877067*^-26"], "Output",
 ImageSize->{87, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->176477554]
}, Open  ]],

Cell["The error using the default method:", "ExampleText",
 CellID->203270288],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], "^", "x"}], "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "12"}], ")"}], "^", "2"}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"WorkingPrecision", "\[Rule]", "30"}]}], "]"}], "-", " ", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], "^", "x"}], "/", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "-", "12"}], ")"}], "^", "2"}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->321726234],

Cell[BoxData["7.88844920109451095`0.02490010873590122*^-12"], "Output",
 ImageSize->{59, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->402269232]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "NSumTerms",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->670697113],

Cell[TextData[{
 "By default ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " uses 15 terms at the beginning before approximating the tail:"
}], "ExampleText",
 CellID->14797858],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{"k", "-", "20"}], ")"}], "^", "2"}], "+", "1"}], ")"}]}], ",",
    
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->536168708],

Cell[BoxData["3.0928604127980863`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->185088991]
}, Open  ]],

Cell["\<\
The error for this example is large because the summand peaks at 20:\
\>", "ExampleText",
 CellID->337030487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"k", "-", "20"}], ")"}], "^", "2"}], "+", "1"}], ")"}]}], 
    ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->478088083],

Cell[BoxData[
 RowBox[{"-", "0.011755449923412264`"}]], "Output",
 ImageSize->{75, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->647777028]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"k", "-", "20"}], ")"}], "^", "2"}], "+", "1"}], ")"}]}], 
     ",", " ", 
     RowBox[{"{", 
      RowBox[{"k", ",", "1", ",", "40"}], "}"}]}], "]"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->687972278],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWM1OFEEQ7p2e2T1tYkxWvSwsROUB4OBPCCYbXRJjOBDjeUNM4IAQxLfQ
BzDxIkFfQAIP4EVPXjnzIkNXz2x3T3X1UD24K4mbzHR/1V3VX/1MzWRfjQ+3
3+6OD3e2xoPRwXh/e2fr/eDF3oESyZYQ4kRdH+4JNc/VTF8nea5G976s7lIt
JWpM1DhQY+rg+XJdlngRtGBDPvkl5aoQK9W10oqQVjIHt6zA5u7tmndwcN07
B1xuwcjkQluJ4dEvYyUy4467k8FhwDghmcgrlkhu/ll/gcMcK6s0Sy4H8FO0
3R3NMnIVk+Xc1iuMCwiDfW61F8/PLcTgazPpQyP9dvxnbfXzp48/pz3fhHOX
ZnacOxep9rwLMpMbHYfHJg6/f13svftxdnpezJXczDnyqLkyoog9H3558/p7
MVHC0Nx60Ub5zK6JdVSeuBHj8DHkrxOBBhEu5ClZ2ffhdscocBiQIb6N7Opz
7oYiFOW0eIBsS9IT6BiiQxItLHRIBuKpsQYd45FnxVaR7n0d1ydrZY3kpG1L
y6OP8LBGy9ruI0xr0Z1L85LWhx7Co4r3zxC/HsKjmjMsvx7CtFaz/gssj2r2
lMWhf9LZobG6Ulqr8gbKkB5HB3cYifAQxb2L8EvCZgfZgO4LcIiy0kWYa8vn
Z3PWRZiyqePiS2pz0i5ziDsqP8OzqaN/VK2Z+wUjgnGKqz1KBzKVMmvb3TPh
BDD07ITzG7XXiQ2uXtrDaWdb9zHnmc0Q3qict55Xn8oM4Y2aM+xzlyFMa822
l8KUm2N2j5iaVqhSJ2vcSvW+/66s2CTIe3Z96Yb2uRteSW5nqqufKfKJemf+
V7WU2sj4/yuE44R3SmYm4vX0t5LzXtiMsAMbCn37pojR93nYd0dTO7EV6f93
KVqXe/k3uQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->639587523]
}, Open  ]],

Cell[TextData[{
 "Increasing ",
 Cell[BoxData["NSumTerms"], "InlineFormula"],
 " to include this feature improves the approximation:"
}], "ExampleText",
 CellID->486325644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{"k", "-", "20"}], ")"}], "^", "2"}], "+", "1"}], ")"}]}], ",",
    
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "\[Infinity]"}], "}"}], ",", " ", 
   RowBox[{"NSumTerms", "\[Rule]", "30"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->6020023],

Cell[BoxData["3.1046158605811454`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->614476665]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"k", "-", "20"}], ")"}], "^", "2"}], "+", "1"}], ")"}]}], 
    ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->136830002],

Cell[BoxData[
 RowBox[{"-", "2.1403532279862247`*^-9"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->93030638]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "VerifyConvergence",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->69133550],

Cell["By default the summand's convergence is verified:", "ExampleText",
 CellID->67628807],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], "^", "k"}], "/", 
     RowBox[{"(", 
      RowBox[{"k", "+", 
       RowBox[{"Log", "[", "k", "]"}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1660533051],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.0699999999999933`", ",", 
   RowBox[{"-", "0.7694021502807998`"}]}], "}"}]], "Output",
 ImageSize->{121, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->551166243]
}, Open  ]],

Cell["\<\
Generally, if the convergence is not verified the summation is faster:\
\>", "ExampleText",
 CellID->901421839],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], "^", "k"}], "/", 
     RowBox[{"(", 
      RowBox[{"k", "+", 
       RowBox[{"Log", "[", "k", "]"}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"VerifyConvergence", "\[Rule]", "False"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1691205057],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.009999999999994763`", ",", 
   RowBox[{"-", "0.7694021502807998`"}]}], "}"}]], "Output",
 ImageSize->{121, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->27396263]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26671386],

Cell["Approximate the formal sum without verifying convergence:", \
"ExampleText",
 CellID->26940473],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   SuperscriptBox["2", "i"], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "0", ",", "Infinity"}], "}"}], ",", 
   RowBox[{"VerifyConvergence", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->601334110],

Cell[BoxData[
 RowBox[{
  RowBox[{"SequenceLimit", "::", "\<\"seqlim\"\>"}], ":", 
  " ", "\<\"The general form of the sequence could not be determined, and the \
result may be incorrect. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/SequenceLimit/seqlim\\\", ButtonNote -> \\\
\"SequenceLimit::seqlim\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->322955711],

Cell[BoxData[
 RowBox[{"-", "1.`"}]], "Output",
 ImageSize->{26, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8175735]
}, Open  ]],

Cell["This is roughly equivalent to the formal symbolic result:", \
"ExampleText",
 CellID->86484621],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sum", "[", 
  RowBox[{
   SuperscriptBox["a", "i"], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "0", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->58709585],

Cell[BoxData[
 FractionBox["1", 
  RowBox[{"1", "-", "a"}]]], "Output",
 ImageSize->{35, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->398481566]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "/.", 
  RowBox[{"a", "\[Rule]", "2"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->124874435],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->271177695]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "WorkingPrecision",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->169653459],

Cell["Use higher precision to get a better approximation:", "ExampleText",
 CellID->557111647],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"n", "^", 
     RowBox[{"(", 
      RowBox[{"51", "/", "50"}], ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "40"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->370760645],

Cell[BoxData["50.578670041015603217613302532825882674053350832312`39.\
999997299590724"], "Output",
 ImageSize->{291, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->130604562]
}, Open  ]],

Cell[TextData[{
 "Find the error versus the exact value of ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{"\[Zeta]", "(", 
     TagBox[
      FractionBox["51", "50"],
      Zeta,
      Editable->True,
      Selectable->True], ")"}],
    InterpretTemplate[
     Function[BoxForm`e$, 
      Zeta[BoxForm`e$]]],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->228103738],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Zeta", "[", 
   RowBox[{"51", "/", "50"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->484886420],

Cell[BoxData[
 RowBox[{"-", "2.581533075642050449458`11.707907587919056*^-27"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->46847941]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->564807618],

Cell[TextData[{
 "Approximate the sum of the reciprocals of the ",
 Cell[BoxData[
  ButtonBox["Fibonacci",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Fibonacci"]], "InlineFormula"],
 " numbers:"
}], "ExampleText",
 CellID->115421685],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Fibonacci", "[", "i", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"i", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->88926106],

Cell[BoxData["3.359885666243177`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->776208071]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->472824333],

Cell["\<\
Find the approximate limit of a slowly converging sequence by summing the \
difference:\
\>", "ExampleText",
 CellID->2048715782],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "n_", "]"}], ":=", 
   RowBox[{"Exp", "[", 
    RowBox[{
     RowBox[{"-", "1"}], "/", 
     RowBox[{"(", 
      RowBox[{"n", "+", "1"}], ")"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->143904722],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{"n", "\[Equal]", "1"}], ",", 
     RowBox[{"f", "[", "1", "]"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"f", "[", "n", "]"}], "-", 
      RowBox[{"f", "[", 
       RowBox[{"n", "-", "1"}], "]"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<WynnEpsilon\>\""}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->78647131],

Cell[BoxData["1.0000827580499787`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->3183375]
}, Open  ]],

Cell["Compare to the exact result:", "ExampleText",
 CellID->416054316],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{"f", "[", "n", "]"}], ",", 
   RowBox[{"n", "\[Rule]", "Infinity"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->23243127],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->649267865]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->692005127],

Cell[TextData[{
 "When the closed-form sum is available, ",
 Cell[BoxData[
  ButtonBox["Sum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sum"]], "InlineFormula"],
 " can be used instead of ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1888242911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Sum", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", "5"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1440448629],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Csch", "[", 
    RowBox[{
     SqrtBox["5"], " ", "\[Pi]"}], "]"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"5", " ", "\[Pi]", " ", 
      RowBox[{"Cosh", "[", 
       RowBox[{
        SqrtBox["5"], " ", "\[Pi]"}], "]"}]}], "+", 
     RowBox[{
      SqrtBox["5"], " ", 
      RowBox[{"Sinh", "[", 
       RowBox[{
        SqrtBox["5"], " ", "\[Pi]"}], "]"}]}]}], ")"}]}], 
  RowBox[{"10", " ", 
   SqrtBox["5"]}]]], "Output",
 ImageSize->{317, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->447059442]
}, Open  ]],

Cell[TextData[{
 "The results of ",
 Cell[BoxData[
  ButtonBox["Sum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sum"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " agree closely:"
}], "ExampleText",
 CellID->1447376270],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", "5"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}], "-", 
  "exact"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->233626939],

Cell[BoxData[
 RowBox[{"-", "1.125365356458019`*^-10"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->230885853]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->372651144],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  ButtonBox["Log",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Log"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Exp",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exp"]], "InlineFormula"],
 " to estimate a product:"
}], "ExampleText",
 CellID->1802176334],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Exp", "[", 
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{"Log", "[", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"4", 
        RowBox[{"n", "^", "2"}]}], ")"}], "/", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"4", 
         RowBox[{"n", "^", "2"}]}], "-", "1"}], ")"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->855895642],

Cell[BoxData["1.5707963267612874`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->685344016]
}, Open  ]],

Cell["The estimate compares well with the exact result:", "ExampleText",
 CellID->545777778],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "-", 
  RowBox[{"Product", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"4", 
       RowBox[{"n", "^", "2"}]}], ")"}], "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"4", 
        RowBox[{"n", "^", "2"}]}], "-", "1"}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1441356320],

Cell[BoxData[
 RowBox[{"-", "3.360911549066259`*^-11"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1424307]
}, Open  ]],

Cell[TextData[{
 "The approximation can be done directly with ",
 Cell[BoxData[
  ButtonBox["NProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NProduct"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->248649223],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NProduct", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"4", 
      RowBox[{"n", "^", "2"}]}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"4", 
       RowBox[{"n", "^", "2"}]}], "-", "1"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->911511977],

Cell[BoxData["1.5707963267612879`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->823472382]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->414333075],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " can be used to calculate oscillatory integrals over ",
 Cell[BoxData[
  RowBox[{"(", 
   RowBox[{"0", ",", "\[Infinity]"}], ")"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1669048706],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", 
   RowBox[{"i_", "?", "NumericQ"}], "]"}], ":=", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"BesselJ", "[", 
      RowBox[{"3", ",", "x"}], "]"}], "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"BesselJZero", "[", 
       RowBox[{"3", ",", 
        RowBox[{"i", "-", "1"}]}], "]"}], ",", 
      RowBox[{"BesselJZero", "[", 
       RowBox[{"3", ",", "i"}], "]"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1811408440],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"BesselJ", "[", 
      RowBox[{"3", ",", "x"}], "]"}], "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", 
      RowBox[{"BesselJZero", "[", 
       RowBox[{"3", ",", "1"}], "]"}]}], "}"}]}], "]"}], "+", 
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{"f", "[", "i", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "2", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1073091552],

Cell[BoxData["0.5735865569063188`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->83195366]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " gives the same result:"
}], "ExampleText",
 CellID->242578820],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"BesselJ", "[", 
     RowBox[{"3", ",", "x"}], "]"}], "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1110373118],

Cell[BoxData["0.5735865569895529`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->20524085]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->538282700],

Cell["\<\
Wynn's extrapolation algorithm loses precision for series that are not \
alternating: \
\>", "ExampleText",
 CellID->1071416675],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NSum", "[", 
   RowBox[{
    FractionBox["1", 
     SuperscriptBox["n", "3"]], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<WynnEpsilon\>\""}], ",", 
    RowBox[{"WorkingPrecision", "\[Rule]", "30"}]}], "]"}], "//", 
  "Precision"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1574416049],

Cell[BoxData["7.277794966681813`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->38906377]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->224789135],

Cell[TextData[{
 "Uses of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " by ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " may lead to convergence messages:"
}], "ExampleText",
 CellID->651233852],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"n", "^", 
     RowBox[{"(", 
      RowBox[{"1001", "/", "1000"}], ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->368490818],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->119944853],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(n\\) near \\!\\({n}\\) = \
\\!\\({8.169069784662288`*^224}\\). NIntegrate obtained \
\\!\\(997.8607733254651`\\) and \\!\\(5.458904190527407`\\) for the integral \
and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->623954726],

Cell[BoxData["1001.2068105760987`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->88206679]
}, Open  ]],

Cell[TextData[{
 "The error compared to the exact value of ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{"\[Zeta]", "(", 
     TagBox[
      FractionBox["1001", "1000"],
      Zeta,
      Editable->True,
      Selectable->True], ")"}],
    InterpretTemplate[
     Function[BoxForm`e$, 
      Zeta[BoxForm`e$]]],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 " is large:"
}], "ExampleText",
 CellID->789177875],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Zeta", "[", 
   RowBox[{"1001", "/", "1000"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->103264420],

Cell[BoxData["0.6295221000870015`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1530817]
}, Open  ]],

Cell[TextData[{
 "You can increase the appropriate parameters for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " using ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " options:"
}], "ExampleText",
 CellID->172089297],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"n", "^", 
     RowBox[{"(", 
      RowBox[{"1001", "/", "1000"}], ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "\[Infinity]"}], "}"}], ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<EulerMaclaurin\>\"", ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<NIntegrate\>\"", ",", " ", 
         RowBox[{"\"\<MaxRecursion\>\"", "\[Rule]", "20"}], ",", " ", 
         RowBox[{"Method", "\[Rule]", "\"\<DoubleExponential\>\""}]}], 
        "}"}]}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->60115228],

Cell[BoxData["1000.5772886183862`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->24235575]
}, Open  ]],

Cell["The error is much smaller:", "ExampleText",
 CellID->170557027],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Zeta", "[", 
   RowBox[{"1001", "/", "1000"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->132738848],

Cell[BoxData["1.423744606654509`*^-7"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->85757905]
}, Open  ]],

Cell[TextData[{
 "Find the error versus the exact value of ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{"\[Zeta]", "(", 
     TagBox[
      FractionBox["11", "10"],
      Zeta,
      Editable->True,
      Selectable->True], ")"}],
    InterpretTemplate[
     Function[BoxForm`e$, 
      Zeta[BoxForm`e$]]],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->40448060],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Zeta", "[", 
   RowBox[{"11", "/", "10"}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->511042746],

Cell[BoxData[
 RowBox[{"-", "10.58444832257634`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->258857717]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1252758],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " may not detect divergence for some infinite sums:"
}], "ExampleText",
 CellID->156580366],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{"1", "/", "k"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1783999730],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->419001297],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(k\\) near \\!\\({k}\\) = \
\\!\\({8.169069784662288`*^224}\\). NIntegrate obtained \
\\!\\(191609.51762979227`\\) and \\!\\(160378.5178102709`\\) for the integral \
and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->533800738],

Cell[BoxData["191612.86743430633`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->290896976]
}, Open  ]],

Cell["\<\
Convergence verification is based on a ratio test that is inconclusive when \
equal to 1:\
\>", "ExampleText",
 CellID->622907249],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"k", "+", "1"}], ")"}]}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{"1", "/", "k"}], ")"}]}], ",", 
   RowBox[{"k", "\[Rule]", "\[Infinity]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1833087983],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->175487704]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->29057],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Sum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sum"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NProduct",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NProduct"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Total",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Total"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NIntegrate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NIntegrate"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->605]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->27536],

Cell[TextData[ButtonBox["Numerical Mathematics: Basic Operations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalMathematics-BasicOperations"]], "Tutorials",
 CellID->123709578],

Cell[TextData[ButtonBox["Numerical Sums, Products and Integrals",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalSumsProductsAndIntegrals"]], "Tutorials",
 CellID->499],

Cell[TextData[{
 ButtonBox["Numerical Mathematics in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NumericalMathematicsInMathematica"],
  FontSlant->"Italic"]
}], "Tutorials",
 CellID->24609],

Cell[TextData[ButtonBox["Numerical Evaluation of Sums and Products",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalEvaluationOfSumsAndProducts"]], "Tutorials",
 CellID->30789]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->5054],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#5064"], None}]], "RelatedLinks",
 CellID->25044]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->100614098],

Cell[TextData[ButtonBox["Calculus",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Calculus"]], "MoreAbout",
 CellID->745092781],

Cell[TextData[ButtonBox["Numerical Evaluation & Precision",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumericalEvaluationAndPrecision"]], "MoreAbout",
 CellID->944037321]
}, Open  ]],

Cell["New in 1  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NSum - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 4, 0.7819684}", "context" -> "System`", 
    "keywords" -> {
     "alternating series summation", "alternating signs", "Euler-Maclaurin", 
      "Euler-Maclaurin summation", "numerical summation", 
      "sequence acceleration", "sums", "Wynn epsilon "}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "NSum[f, {i, i_min, i_max}] gives a numerical approximation to the sum \
UnderoverscriptBox[\"\\[Sum]\", RowBox[{\"i\", \"=\", \"i_min\"}], \"i_max\", \
LimitsPositioning->True]\\ f. NSum[f, {i, i_min, i_max, di}] uses a step di \
in the sum. ", "synonyms" -> {}, "title" -> "NSum", "type" -> "Symbol", "uri" -> 
    "ref/NSum"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[17038, 490, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->269800189]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 72751, 2627}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2871, 64, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3494, 91, 4188, 83, 70, "ObjectNameGrid"],
Cell[7685, 176, 1869, 62, 70, "Usage",
 CellID->32720]
}, Open  ]],
Cell[CellGroupData[{
Cell[9591, 243, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[10069, 260, 219, 7, 70, "Notes",
 CellID->28306],
Cell[10291, 269, 572, 20, 70, "Notes",
 CellID->14336],
Cell[10866, 291, 68, 1, 70, "Notes",
 CellID->2566],
Cell[10937, 294, 2386, 63, 70, "3ColumnTableMod",
 CellID->31410],
Cell[13326, 359, 210, 8, 70, "Notes",
 CellID->24422],
Cell[13539, 369, 816, 17, 70, "2ColumnTableMod",
 CellID->3159],
Cell[14358, 388, 696, 22, 70, "Notes",
 CellID->14026],
Cell[15057, 412, 508, 15, 70, "Notes",
 CellID->7244],
Cell[15568, 429, 146, 4, 70, "Notes",
 CellID->4897],
Cell[15717, 435, 446, 18, 70, "Notes",
 CellID->30405],
Cell[16166, 455, 373, 11, 70, "Notes",
 CellID->30946],
Cell[16542, 468, 459, 17, 70, "Notes",
 CellID->6944]
}, Closed]],
Cell[CellGroupData[{
Cell[17038, 490, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->269800189],
Cell[CellGroupData[{
Cell[17423, 504, 148, 5, 70, "ExampleSection",
 CellID->232714729],
Cell[17574, 511, 83, 1, 70, "ExampleText",
 CellID->226825228],
Cell[CellGroupData[{
Cell[17682, 516, 326, 12, 28, "Input",
 CellID->12053745],
Cell[18011, 530, 177, 5, 36, "Output",
 CellID->25454835]
}, Open  ]],
Cell[18203, 538, 222, 8, 70, "ExampleText",
 CellID->90781056],
Cell[CellGroupData[{
Cell[18450, 550, 150, 5, 28, "Input",
 CellID->39174757],
Cell[18603, 557, 198, 6, 36, "Output",
 CellID->150339448]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[18850, 569, 219, 7, 70, "ExampleSection",
 CellID->71768073],
Cell[19072, 578, 81, 1, 70, "ExampleText",
 CellID->715570151],
Cell[CellGroupData[{
Cell[19178, 583, 274, 9, 70, "Input",
 CellID->73359430],
Cell[19455, 594, 178, 5, 36, "Output",
 CellID->173194495]
}, Open  ]],
Cell[19648, 602, 98, 1, 70, "ExampleText",
 CellID->39454065],
Cell[CellGroupData[{
Cell[19771, 607, 523, 18, 70, "Input",
 CellID->42115105],
Cell[20297, 627, 178, 5, 36, "Output",
 CellID->285213080]
}, Open  ]],
Cell[20490, 635, 124, 3, 70, "ExampleDelimiter",
 CellID->88535772],
Cell[20617, 640, 77, 1, 70, "ExampleText",
 CellID->39868800],
Cell[CellGroupData[{
Cell[20719, 645, 495, 17, 70, "Input",
 CellID->552226847],
Cell[21217, 664, 176, 5, 36, "Output",
 CellID->278864469]
}, Open  ]],
Cell[21408, 672, 124, 3, 70, "ExampleDelimiter",
 CellID->19975727],
Cell[21535, 677, 133, 4, 70, "ExampleText",
 CellID->923515544],
Cell[CellGroupData[{
Cell[21693, 685, 485, 17, 70, "Input",
 CellID->945501495],
Cell[22181, 704, 175, 5, 36, "Output",
 CellID->13582748]
}, Open  ]],
Cell[22371, 712, 124, 3, 70, "ExampleDelimiter",
 CellID->40547156],
Cell[22498, 717, 68, 1, 70, "ExampleText",
 CellID->266585647],
Cell[CellGroupData[{
Cell[22591, 722, 349, 12, 70, "Input",
 CellID->336884800],
Cell[22943, 736, 272, 8, 36, "Output",
 CellID->263118562]
}, Open  ]],
Cell[23230, 747, 125, 3, 70, "ExampleDelimiter",
 CellID->220715922],
Cell[23358, 752, 74, 1, 70, "ExampleText",
 CellID->144564436],
Cell[CellGroupData[{
Cell[23457, 757, 254, 9, 70, "Input",
 CellID->310085196],
Cell[23714, 768, 176, 5, 36, "Output",
 CellID->568702208]
}, Open  ]],
Cell[23905, 776, 88, 1, 70, "ExampleText",
 CellID->105472116],
Cell[CellGroupData[{
Cell[24018, 781, 289, 10, 70, "Input",
 CellID->642821719],
Cell[24310, 793, 176, 5, 36, "Output",
 CellID->484750109]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24535, 804, 217, 7, 70, "ExampleSection",
 CellID->1831],
Cell[CellGroupData[{
Cell[24777, 815, 251, 7, 70, "ExampleSubsection",
 CellID->906276870],
Cell[25031, 824, 89, 1, 70, "ExampleText",
 CellID->313601664],
Cell[CellGroupData[{
Cell[25145, 829, 296, 10, 70, "Input",
 CellID->518750079],
Cell[25444, 841, 176, 5, 36, "Output",
 CellID->332262193]
}, Open  ]],
Cell[25635, 849, 436, 18, 70, "ExampleText",
 CellID->482113350],
Cell[CellGroupData[{
Cell[26096, 871, 159, 5, 70, "Input",
 CellID->129873791],
Cell[26258, 878, 198, 6, 36, "Output",
 CellID->272793093]
}, Open  ]],
Cell[26471, 887, 98, 1, 70, "ExampleText",
 CellID->76531130],
Cell[CellGroupData[{
Cell[26594, 892, 508, 15, 70, "Input",
 CellID->105238400],
Cell[27105, 909, 180, 5, 36, "Output",
 CellID->165415353]
}, Open  ]],
Cell[27300, 917, 98, 1, 70, "ExampleText",
 CellID->361447028],
Cell[CellGroupData[{
Cell[27423, 922, 510, 15, 70, "Input",
 CellID->394312397],
Cell[27936, 939, 177, 5, 36, "Output",
 CellID->7136769]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[28162, 950, 238, 7, 70, "ExampleSubsection",
 CellID->103639796],
Cell[28403, 959, 117, 3, 70, "ExampleText",
 CellID->353531660],
Cell[CellGroupData[{
Cell[28545, 966, 569, 18, 70, "Input",
 CellID->711990419],
Cell[29117, 986, 222, 7, 36, "Output",
 CellID->5061741]
}, Open  ]],
Cell[29354, 996, 125, 3, 70, "ExampleDelimiter",
 CellID->264789900],
Cell[29482, 1001, 132, 4, 70, "ExampleText",
 CellID->17998260],
Cell[CellGroupData[{
Cell[29639, 1009, 561, 17, 70, "Input",
 CellID->650141528],
Cell[30203, 1028, 1749, 37, 140, "Output",
 CellID->61104135]
}, Open  ]],
Cell[31967, 1068, 125, 3, 70, "ExampleDelimiter",
 CellID->251879244],
Cell[32095, 1073, 145, 4, 70, "ExampleText",
 CellID->606763501],
Cell[CellGroupData[{
Cell[32265, 1081, 644, 20, 70, "Input",
 CellID->302137647],
Cell[32912, 1103, 774, 20, 140, "Output",
 CellID->175323937]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[33735, 1129, 225, 7, 70, "ExampleSubsection",
 CellID->4900274],
Cell[33963, 1138, 127, 3, 70, "ExampleText",
 CellID->385344373],
Cell[CellGroupData[{
Cell[34115, 1145, 549, 17, 70, "Input",
 CellID->488498118],
Cell[34667, 1164, 227, 6, 36, "Output",
 CellID->23098839]
}, Open  ]],
Cell[34909, 1173, 217, 8, 70, "ExampleText",
 CellID->138635920],
Cell[CellGroupData[{
Cell[35151, 1185, 468, 16, 70, "Input",
 CellID->81223570],
Cell[35622, 1203, 203, 5, 36, "Output",
 CellID->176477554]
}, Open  ]],
Cell[35840, 1211, 78, 1, 70, "ExampleText",
 CellID->203270288],
Cell[CellGroupData[{
Cell[35943, 1216, 889, 30, 70, "Input",
 CellID->321726234],
Cell[36835, 1248, 201, 5, 36, "Output",
 CellID->402269232]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[37085, 1259, 230, 7, 70, "ExampleSubsection",
 CellID->670697113],
Cell[37318, 1268, 246, 8, 70, "ExampleText",
 CellID->14797858],
Cell[CellGroupData[{
Cell[37589, 1280, 356, 13, 70, "Input",
 CellID->536168708],
Cell[37948, 1295, 176, 5, 36, "Output",
 CellID->185088991]
}, Open  ]],
Cell[38139, 1303, 119, 3, 70, "ExampleText",
 CellID->337030487],
Cell[CellGroupData[{
Cell[38283, 1310, 398, 14, 70, "Input",
 CellID->478088083],
Cell[38684, 1326, 195, 6, 36, "Output",
 CellID->647777028]
}, Open  ]],
Cell[CellGroupData[{
Cell[38916, 1337, 477, 16, 70, "Input",
 CellID->687972278],
Cell[39396, 1355, 1140, 23, 70, "Output",
 Evaluatable->False,
 CellID->639587523]
}, Open  ]],
Cell[40551, 1381, 172, 5, 70, "ExampleText",
 CellID->486325644],
Cell[CellGroupData[{
Cell[40748, 1390, 408, 14, 70, "Input",
 CellID->6020023],
Cell[41159, 1406, 176, 5, 36, "Output",
 CellID->614476665]
}, Open  ]],
Cell[CellGroupData[{
Cell[41372, 1416, 398, 14, 70, "Input",
 CellID->136830002],
Cell[41773, 1432, 196, 6, 36, "Output",
 CellID->93030638]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[42018, 1444, 237, 7, 70, "ExampleSubsection",
 CellID->69133550],
Cell[42258, 1453, 91, 1, 70, "ExampleText",
 CellID->67628807],
Cell[CellGroupData[{
Cell[42374, 1458, 417, 15, 70, "Input",
 CellID->1660533051],
Cell[42794, 1475, 259, 8, 36, "Output",
 CellID->551166243]
}, Open  ]],
Cell[43068, 1486, 121, 3, 70, "ExampleText",
 CellID->901421839],
Cell[CellGroupData[{
Cell[43214, 1493, 478, 16, 70, "Input",
 CellID->1691205057],
Cell[43695, 1511, 260, 8, 36, "Output",
 CellID->27396263]
}, Open  ]],
Cell[43970, 1522, 124, 3, 70, "ExampleDelimiter",
 CellID->26671386],
Cell[44097, 1527, 101, 2, 70, "ExampleText",
 CellID->26940473],
Cell[CellGroupData[{
Cell[44223, 1533, 274, 8, 70, "Input",
 CellID->601334110],
Cell[44500, 1543, 440, 8, 70, "Message",
 CellID->322955711],
Cell[44943, 1553, 175, 6, 36, "Output",
 CellID->8175735]
}, Open  ]],
Cell[45133, 1562, 101, 2, 70, "ExampleText",
 CellID->86484621],
Cell[CellGroupData[{
Cell[45259, 1568, 212, 7, 70, "Input",
 CellID->58709585],
Cell[45474, 1577, 201, 7, 51, "Output",
 CellID->398481566]
}, Open  ]],
Cell[CellGroupData[{
Cell[45712, 1589, 122, 4, 70, "Input",
 CellID->124874435],
Cell[45837, 1595, 175, 6, 36, "Output",
 CellID->271177695]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[46061, 1607, 237, 7, 70, "ExampleSubsection",
 CellID->169653459],
Cell[46301, 1616, 94, 1, 70, "ExampleText",
 CellID->557111647],
Cell[CellGroupData[{
Cell[46420, 1621, 352, 11, 70, "Input",
 CellID->370760645],
Cell[46775, 1634, 230, 6, 36, "Output",
 CellID->130604562]
}, Open  ]],
Cell[47020, 1643, 436, 18, 70, "ExampleText",
 CellID->228103738],
Cell[CellGroupData[{
Cell[47481, 1665, 159, 5, 70, "Input",
 CellID->484886420],
Cell[47643, 1672, 221, 6, 36, "Output",
 CellID->46847941]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[47925, 1685, 227, 7, 70, "ExampleSection",
 CellID->564807618],
Cell[48155, 1694, 239, 8, 70, "ExampleText",
 CellID->115421685],
Cell[CellGroupData[{
Cell[48419, 1706, 248, 8, 70, "Input",
 CellID->88926106],
Cell[48670, 1716, 175, 5, 36, "Output",
 CellID->776208071]
}, Open  ]],
Cell[48860, 1724, 125, 3, 70, "ExampleDelimiter",
 CellID->472824333],
Cell[48988, 1729, 139, 4, 70, "ExampleText",
 CellID->2048715782],
Cell[49130, 1735, 271, 10, 70, "Input",
 CellID->143904722],
Cell[CellGroupData[{
Cell[49426, 1749, 524, 15, 70, "Input",
 CellID->78647131],
Cell[49953, 1766, 174, 5, 36, "Output",
 CellID->3183375]
}, Open  ]],
Cell[50142, 1774, 71, 1, 70, "ExampleText",
 CellID->416054316],
Cell[CellGroupData[{
Cell[50238, 1779, 189, 6, 70, "Input",
 CellID->23243127],
Cell[50430, 1787, 158, 5, 36, "Output",
 CellID->649267865]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[50637, 1798, 237, 7, 70, "ExampleSection",
 CellID->692005127],
Cell[50877, 1807, 353, 13, 70, "ExampleText",
 CellID->1888242911],
Cell[CellGroupData[{
Cell[51255, 1824, 299, 10, 70, "Input",
 CellID->1440448629],
Cell[51557, 1836, 616, 23, 64, "Output",
 CellID->447059442]
}, Open  ]],
Cell[52188, 1862, 324, 13, 70, "ExampleText",
 CellID->1447376270],
Cell[CellGroupData[{
Cell[52537, 1879, 302, 11, 70, "Input",
 CellID->233626939],
Cell[52842, 1892, 198, 6, 36, "Output",
 CellID->230885853]
}, Open  ]],
Cell[53055, 1901, 125, 3, 70, "ExampleDelimiter",
 CellID->372651144],
Cell[53183, 1906, 321, 13, 70, "ExampleText",
 CellID->1802176334],
Cell[CellGroupData[{
Cell[53529, 1923, 474, 17, 70, "Input",
 CellID->855895642],
Cell[54006, 1942, 176, 5, 36, "Output",
 CellID->685344016]
}, Open  ]],
Cell[54197, 1950, 92, 1, 70, "ExampleText",
 CellID->545777778],
Cell[CellGroupData[{
Cell[54314, 1955, 428, 15, 70, "Input",
 CellID->1441356320],
Cell[54745, 1972, 196, 6, 36, "Output",
 CellID->1424307]
}, Open  ]],
Cell[54956, 1981, 227, 8, 70, "ExampleText",
 CellID->248649223],
Cell[CellGroupData[{
Cell[55208, 1993, 394, 14, 70, "Input",
 CellID->911511977],
Cell[55605, 2009, 176, 5, 36, "Output",
 CellID->823472382]
}, Open  ]],
Cell[55796, 2017, 125, 3, 70, "ExampleDelimiter",
 CellID->414333075],
Cell[55924, 2022, 324, 11, 70, "ExampleText",
 CellID->1669048706],
Cell[56251, 2035, 568, 18, 70, "Input",
 CellID->1811408440],
Cell[CellGroupData[{
Cell[56844, 2057, 564, 18, 70, "Input",
 CellID->1073091552],
Cell[57411, 2077, 175, 5, 36, "Output",
 CellID->83195366]
}, Open  ]],
Cell[57601, 2085, 204, 7, 70, "ExampleText",
 CellID->242578820],
Cell[CellGroupData[{
Cell[57830, 2096, 318, 10, 70, "Input",
 CellID->1110373118],
Cell[58151, 2108, 175, 5, 36, "Output",
 CellID->20524085]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[58375, 2119, 230, 7, 70, "ExampleSection",
 CellID->538282700],
Cell[58608, 2128, 138, 4, 70, "ExampleText",
 CellID->1071416675],
Cell[CellGroupData[{
Cell[58771, 2136, 402, 12, 70, "Input",
 CellID->1574416049],
Cell[59176, 2150, 174, 5, 36, "Output",
 CellID->38906377]
}, Open  ]],
Cell[59365, 2158, 125, 3, 70, "ExampleDelimiter",
 CellID->224789135],
Cell[59493, 2163, 348, 13, 70, "ExampleText",
 CellID->651233852],
Cell[CellGroupData[{
Cell[59866, 2180, 300, 10, 70, "Input",
 CellID->368490818],
Cell[60169, 2192, 522, 9, 70, "Message",
 CellID->119944853],
Cell[60694, 2203, 619, 11, 70, "Message",
 CellID->623954726],
Cell[61316, 2216, 175, 5, 36, "Output",
 CellID->88206679]
}, Open  ]],
Cell[61506, 2224, 449, 18, 70, "ExampleText",
 CellID->789177875],
Cell[CellGroupData[{
Cell[61980, 2246, 163, 5, 70, "Input",
 CellID->103264420],
Cell[62146, 2253, 174, 5, 36, "Output",
 CellID->1530817]
}, Open  ]],
Cell[62335, 2261, 370, 13, 70, "ExampleText",
 CellID->172089297],
Cell[CellGroupData[{
Cell[62730, 2278, 688, 19, 70, "Input",
 CellID->60115228],
Cell[63421, 2299, 175, 5, 36, "Output",
 CellID->24235575]
}, Open  ]],
Cell[63611, 2307, 69, 1, 70, "ExampleText",
 CellID->170557027],
Cell[CellGroupData[{
Cell[63705, 2312, 163, 5, 70, "Input",
 CellID->132738848],
Cell[63871, 2319, 178, 5, 36, "Output",
 CellID->85757905]
}, Open  ]],
Cell[64064, 2327, 435, 18, 70, "ExampleText",
 CellID->40448060],
Cell[CellGroupData[{
Cell[64524, 2349, 159, 5, 70, "Input",
 CellID->511042746],
Cell[64686, 2356, 192, 6, 36, "Output",
 CellID->258857717]
}, Open  ]],
Cell[64893, 2365, 123, 3, 70, "ExampleDelimiter",
 CellID->1252758],
Cell[65019, 2370, 219, 7, 70, "ExampleText",
 CellID->156580366],
Cell[CellGroupData[{
Cell[65263, 2381, 217, 7, 70, "Input",
 CellID->1783999730],
Cell[65483, 2390, 522, 9, 70, "Message",
 CellID->419001297],
Cell[66008, 2401, 620, 11, 70, "Message",
 CellID->533800738],
Cell[66631, 2414, 176, 5, 36, "Output",
 CellID->290896976]
}, Open  ]],
Cell[66822, 2422, 140, 4, 70, "ExampleText",
 CellID->622907249],
Cell[CellGroupData[{
Cell[66987, 2430, 346, 12, 70, "Input",
 CellID->1833087983],
Cell[67336, 2444, 158, 5, 36, "Output",
 CellID->175487704]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[67555, 2456, 311, 9, 70, "SeeAlsoSection",
 CellID->29057],
Cell[67869, 2467, 927, 35, 70, "SeeAlso",
 CellID->605]
}, Open  ]],
Cell[CellGroupData[{
Cell[68833, 2507, 314, 9, 70, "TutorialsSection",
 CellID->27536],
Cell[69150, 2518, 193, 4, 70, "Tutorials",
 CellID->123709578],
Cell[69346, 2524, 183, 4, 70, "Tutorials",
 CellID->499],
Cell[69532, 2530, 324, 9, 70, "Tutorials",
 CellID->24609],
Cell[69859, 2541, 191, 4, 70, "Tutorials",
 CellID->30789]
}, Open  ]],
Cell[CellGroupData[{
Cell[70087, 2550, 320, 9, 70, "RelatedLinksSection",
 CellID->5054],
Cell[70410, 2561, 266, 6, 70, "RelatedLinks",
 CellID->25044]
}, Open  ]],
Cell[CellGroupData[{
Cell[70713, 2572, 319, 9, 70, "MoreAboutSection",
 CellID->100614098],
Cell[71035, 2583, 128, 3, 70, "MoreAbout",
 CellID->745092781],
Cell[71166, 2588, 175, 3, 70, "MoreAbout",
 CellID->944037321]
}, Open  ]],
Cell[71356, 2594, 50, 0, 70, "History"],
Cell[71409, 2596, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

