(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     45704,       1639]
NotebookOptionsPosition[     37470,       1345]
NotebookOutlinePosition[     38872,       1384]
CellTagsIndexPosition[     38786,       1379]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DiscreteDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Discrete Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BinomialDistribution"], 
          "GeometricDistribution" :> 
          Documentation`HelpLookup["paclet:ref/GeometricDistribution"], 
          "BernoulliDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BernoulliDistribution"], 
          "BetaNegativeBinomialDistribution" :> 
          Documentation`HelpLookup[
           "paclet:ref/BetaNegativeBinomialDistribution"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"BinomialDistribution\"\>", 
       2->"\<\"GeometricDistribution\"\>", 3->"\<\"BernoulliDistribution\"\>",
        4->"\<\"BetaNegativeBinomialDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/DiscreteStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Discrete Statistical Distributions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["NegativeBinomialDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NegativeBinomialDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NegativeBinomialDistribution"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a negative binomial distribution with \
parameters ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->7242]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability for value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula",
  ShowStringCharacters->False],
 " in a negative binomial distribution is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     TagBox[
      RowBox[{"(", GridBox[{
         {
          TagBox[
           RowBox[{"n", "+", "x", "-", "1"}],
           Identity,
           Editable->True,
           Selectable->True]},
         {
          TagBox[
           RowBox[{"n", "-", "1"}],
           Identity,
           Editable->True,
           Selectable->True]}
        }], ")"}],
      InterpretTemplate[Binomial[#, #2]& ],
      Editable->False,
      Selectable->False],
     TraditionalForm], 
    SuperscriptBox[
     RowBox[{
      SuperscriptBox["p", "n"], "(", 
      RowBox[{"1", "-", "p"}], ")"}], "x"]}], TraditionalForm]], 
  "InlineMath"],
 " for non-negative integers, and is zero otherwise.",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->4092,
  ButtonNote->"4092"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is a positive integer, ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["NegativeBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NegativeBinomialDistribution"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
 " gives the distribution of the number of failures in a sequence of trials \
with success probability ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " before ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " successes occur."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->23225],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " to be any positive real number."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->132305407],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomInteger"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->17290,
  ButtonNote->"17290"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->108634339],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->435718686],

Cell["Mean and variance of a negative binomial distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->17290],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31219],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", 
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}]}], "p"]], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->True,
 ImageSize->{59, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->823480873]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1993],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", 
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}]}], 
  SuperscriptBox["p", "2"]]], "Output",
 ImageSize->{59, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->57194694]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10756],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->4092],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7693],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", 
  SuperscriptBox["p", "n"], " ", 
  RowBox[{"Binomial", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "+", "k", "+", "n"}], ",", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "n"}]}], "]"}]}]], "Output",
 ImageSize->{240, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->69688363]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"PDF", "[", 
      RowBox[{
       RowBox[{"NegativeBinomialDistribution", "[", 
        RowBox[{"10", ",", 
         RowBox[{"1", "/", "3"}]}], "]"}], ",", "k"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "50"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "->", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17032],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 142},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->70785285]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26343],

Cell["\<\
Generate a set of pseudorandom numbers that have the negative binomial \
distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->7933],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomInteger", "[", 
  RowBox[{
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"6", ",", 
     RowBox[{"1", "/", "3"}]}], "]"}], ",", "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20444],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "10", ",", "21", ",", "7", ",", "9", ",", "15", ",", "10", ",", "15", ",", 
   "5", ",", "4", ",", "7"}], "}"}]], "Output",
 ImageSize->{224, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->107614136]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14486],

Cell["Properties based on higher-order moments:", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18025],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", "-", "p"}], 
  SqrtBox[
   RowBox[{"n", " ", 
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}]}]]]], "Output",
 ImageSize->{76, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->980767119]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->22332],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox[
   RowBox[{"6", "-", 
    RowBox[{"6", " ", "p"}], "+", 
    SuperscriptBox["p", "2"]}], 
   RowBox[{"n", " ", 
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}]}]]}]], "Output",
 ImageSize->{90, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->189923683]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9145],

Cell["Second moment of a negative binomial distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->30053],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"k", "^", "2"}], ",", 
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17709],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "p"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "-", "n", "+", 
     RowBox[{"n", " ", "p"}]}], ")"}]}], 
  SuperscriptBox["p", "2"]]], "Output",
 ImageSize->{148, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->48128311]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox["0.8", TraditionalForm]], "InlineMath"],
 " quantile of a negative binomial distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "50"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"p", "=", 
    FractionBox["1", "3"]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->48521834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"50", ",", 
     RowBox[{"1", "/", "3"}]}], "]"}], ",", ".8"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4591],

Cell[BoxData["114"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->67615007]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9319],

Cell[TextData[{
 "The probability of at least 5 failures before 10 successes for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"p", "=", "0.9"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->32514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"NegativeBinomialDistribution", "[", 
     RowBox[{"10", ",", ".9"}], "]"}], ",", "4"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10642],

Cell[BoxData["0.009230212455849984`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->671266247]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->86062964],

Cell["\<\
Plot the cumulative distribution function of a negative binomial \
distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"NegativeBinomialDistribution", "[", 
      RowBox[{"10", ",", 
       RowBox[{"3", "/", "4"}]}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26124],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwPATiqRIMQPZ/IAuM
H/7/D6QxSTMgyQJUoAbiMYNUggGDKpBggoqDaBAfJM0E5TODdZsj6WLAYooK
iGBB8BmwiKhiiIDsAHmDEUSDhSzQzFWGuhpTL3nmW6KZr4TNXximMmGVIWy6
MpJeZqwuJMbNZjjdpE4wFogx3xyHacgpAplmRkkZyGlMAM2Ec3hEeYDkFag7
MNWwk8jnoFD/YLOPUvOZyQpVIRr7arCZjz2UsJWQlMhjtwVf3sAtCsp3wvPn
nWFg/vz5J1R2J6oeoCxIgSBY1dIl53/+/AOS2kVFV+B223ma20J7u6ERBgbM
SCrAfCBmwa4LpZRlRdNHjB42NDXMaHxBMP/wofv5eZuBiEEIRID4UPmlWMxE
T/Gg0h/EhZgFTEPnzz0FIgYhEIGUpog1i5CfwP7GFMEb5mzQ+GTFqnKkpsrB
5yKMfMIKVwGVQI9BSMl49coLaKoTBKe6588/QeUn/ic9V0BKX2DKLS3ZhsgV
wMIPj5mgNMgCpUFG4Uy5SDkYe2rEzPW4Uy5JapHCFmutNZoKB4HduGoKEBNb
3FqB+b09h6EluCA4rVZX7YTK1/7HkQ7xlqAQUx8+fI+Sq27fekOyqXjzAjGl
OFYTic0NyGoI1xLs0LjErnKkpkjiRP3A/O3bbqC0JBApJgPFbf5g1ahthWwq
uYzUHETd1Em6LuRSm/q5ZeBbSslgWWBKsLedCUQMQiCCefKkY1DVIVSKd1rn
kXIwH71MBPKh8g54dDU27EEpn4F8vLrI91EPWPbt268orgS2j6CqdbCa0Q3m
A3t2KLkXaApUXhuPzZBEBQPoLXhsaWk+LEVA+5VC4H4l0KFQeXGwgegmgRMp
3tQMMRfY9kMxd+f2mzhijJlMezb//4+lfQisg8kwF5HjSXcHZfqw5fprYCHM
kV4GRgCEKgR6\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 149},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->47439975]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8113],

Cell["\<\
The probability of getting any real number except a non-negative integer is \
zero:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->7633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"n", ",", "p"}], "]"}], ",", "\[Pi]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30290],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->51517808]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->636],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ExpectedValue", "[", 
   RowBox[{
    RowBox[{"k", "^", "4"}], ",", 
    RowBox[{"NegativeBinomialDistribution", "[", 
     RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17995],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "n"}], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "1"}], "+", "p"}], ")"}], " ", 
  SuperscriptBox["p", "n"], " ", 
  RowBox[{"HypergeometricPFQ", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"2", ",", "2", ",", "2", ",", 
      RowBox[{"1", "+", "n"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
    RowBox[{"1", "-", "p"}]}], "]"}]}]], "Output",
 ImageSize->{424, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->440741880]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", "=", 
  RowBox[{"CharacteristicFunction", "[", 
   RowBox[{
    RowBox[{"NegativeBinomialDistribution", "[", 
     RowBox[{"n", ",", "p"}], "]"}], ",", "t"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->527296452],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"(", 
   FractionBox["p", 
    RowBox[{"1", "-", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"\[ImaginaryI]", " ", "t"}]], " ", 
      RowBox[{"(", 
       RowBox[{"1", "-", "p"}], ")"}]}]}]], ")"}], "n"]], "Output",
 ImageSize->{111, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->360148328]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{"cf", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "4"}], "}"}]}], "]"}], "/.", 
   RowBox[{"{", 
    RowBox[{"t", "->", "0"}], "}"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->19463],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "p"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "6"}], "-", 
     RowBox[{"6", " ", 
      SuperscriptBox["n", "2"], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "1"}], "+", "p"}], ")"}], "2"]}], "+", 
     RowBox[{
      SuperscriptBox["n", "3"], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "1"}], "+", "p"}], ")"}], "3"]}], "+", 
     RowBox[{"6", " ", "p"}], "-", 
     SuperscriptBox["p", "2"], "+", 
     RowBox[{"n", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "11"}], "+", 
        RowBox[{"15", " ", "p"}], "-", 
        RowBox[{"4", " ", 
         SuperscriptBox["p", "2"]}]}], ")"}]}]}], ")"}]}], 
  SuperscriptBox["p", "4"]]], "Output",
 ImageSize->{457, 38},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->144684636]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11620],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "1"}], TraditionalForm]], "InlineMath"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["GeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->8056],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"1", ",", "p"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6133],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "x"], " ", "p"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->76027097]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"GeometricDistribution", "[", "p", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15955],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "x"], " ", "p"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->71885263]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->932691717],

Cell[TextData[{
 "Negative binomial is a mixture of ",
 Cell[BoxData[
  ButtonBox["PoissonDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PoissonDistribution"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["GammaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GammaDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->217493462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"poispdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"PoissonDistribution", "[", "\[Mu]", "]"}], ",", "k"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->364101735],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", "\[Mu]"}]], " ", 
   SuperscriptBox["\[Mu]", "k"]}], 
  RowBox[{"k", "!"}]]], "Output",
 ImageSize->{43, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->474676647]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"gdist", "=", 
   RowBox[{"GammaDistribution", "[", 
    RowBox[{"n", ",", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "p"}], ")"}], "/", "p"}]}], "]"}]}], ";"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->437860854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ev", "=", 
  RowBox[{"ExpectedValue", "[", 
   RowBox[{"poispdf", ",", "gdist", ",", "\[Mu]", ",", 
    RowBox[{"Assumptions", "\[Rule]", 
     RowBox[{"k", ">", "0"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->221897965],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}], "k"], " ", 
   SuperscriptBox["p", "n"], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"k", "+", "n"}], "]"}]}], 
  RowBox[{
   RowBox[{"k", "!"}], " ", 
   RowBox[{"Gamma", "[", "n", "]"}]}]]], "Output",
 ImageSize->{150, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->807830191]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"NegativeBinomialDistribution", "[", 
      RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}], "\[Equal]", "ev"}], "//",
   "FullSimplify"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->15690664],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->360811085]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25421],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is non-positive:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->20461],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22501],

Cell[BoxData[
 RowBox[{
  RowBox[{"NegativeBinomialDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-1\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->27092585],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "p"}], "]"}], "]"}]], "Output",
 ImageSize->{289, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->602841778]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " is not between zero and one:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3163],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{"n", ",", "3.14159"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->28738],

Cell[BoxData[
 RowBox[{
  RowBox[{"NegativeBinomialDistribution", "::", "\<\"probparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(3.14159`\\) is expected to be a nonzero \
probability between 0 and 1.\"\>"}]], "Message", "MSG",
 CellID->642963],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NegativeBinomialDistribution", "[", 
   RowBox[{"n", ",", "3.14159`"}], "]"}], "]"}]], "Output",
 ImageSize->{324, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->317265355]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17334],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"NegativeBinomialDistribution", "[", 
    RowBox[{"n", ",", "p"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"n", "->", "1"}], ",", 
    RowBox[{"p", "->", 
     RowBox[{"3", "/", "2"}]}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29499],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox["1", "3"]}]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180220542]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17252],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GeometricDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BernoulliDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BernoulliDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaNegativeBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaNegativeBinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->15857]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->30881],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Discrete Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteStatisticalDistributions"]], "MoreAbout",
 CellID->746109424]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NegativeBinomialDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 1, 53.4349584}", "context" -> "System`", 
    "keywords" -> {
     "Pascal distribution", "Polya distribution", "Po'lya distribution"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "NegativeBinomialDistribution[n, p] represents a negative binomial \
distribution with parameters n and p.", 
    "synonyms" -> {"negative binomial distribution"}, "title" -> 
    "NegativeBinomialDistribution", "type" -> "Symbol", "uri" -> 
    "ref/NegativeBinomialDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8889, 286, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->108634339]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 38643, 1372}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2343, 57, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2966, 84, 1593, 37, 70, "ObjectNameGrid"],
Cell[4562, 123, 752, 24, 70, "Usage",
 CellID->7242]
}, Open  ]],
Cell[CellGroupData[{
Cell[5351, 152, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5829, 169, 1051, 40, 70, "Notes",
 CellID->429991538],
Cell[6883, 211, 787, 26, 70, "Notes",
 CellID->23225],
Cell[7673, 239, 452, 14, 70, "Notes",
 CellID->132305407],
Cell[8128, 255, 724, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[8889, 286, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->108634339],
Cell[CellGroupData[{
Cell[9274, 300, 148, 5, 70, "ExampleSection",
 CellID->435718686],
Cell[9425, 307, 278, 6, 70, "ExampleText",
 CellID->17290],
Cell[CellGroupData[{
Cell[9728, 317, 175, 5, 28, "Input",
 CellID->31219],
Cell[9906, 324, 299, 11, 53, "Output",
 CellID->823480873]
}, Open  ]],
Cell[CellGroupData[{
Cell[10242, 340, 178, 5, 28, "Input",
 CellID->1993],
Cell[10423, 347, 273, 10, 54, "Output",
 CellID->57194694]
}, Open  ]],
Cell[10711, 360, 121, 3, 70, "ExampleDelimiter",
 CellID->10756],
Cell[10835, 365, 252, 6, 70, "ExampleText",
 CellID->4092],
Cell[CellGroupData[{
Cell[11112, 375, 198, 6, 70, "Input",
 CellID->7693],
Cell[11313, 383, 448, 16, 36, "Output",
 CellID->69688363]
}, Open  ]],
Cell[CellGroupData[{
Cell[11798, 404, 459, 14, 47, "Input",
 CellID->17032],
Cell[12260, 420, 2887, 51, 163, "Output",
 Evaluatable->False,
 CellID->70785285]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[15196, 477, 216, 7, 70, "ExampleSection",
 CellID->26343],
Cell[15415, 486, 269, 8, 70, "ExampleText",
 CellID->7933],
Cell[CellGroupData[{
Cell[15709, 498, 236, 7, 70, "Input",
 CellID->20444],
Cell[15948, 507, 296, 9, 36, "Output",
 CellID->107614136]
}, Open  ]],
Cell[16259, 519, 121, 3, 70, "ExampleDelimiter",
 CellID->14486],
Cell[16383, 524, 83, 1, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[16491, 529, 179, 5, 70, "Input",
 CellID->18025],
Cell[16673, 536, 288, 11, 56, "Output",
 CellID->980767119]
}, Open  ]],
Cell[CellGroupData[{
Cell[16998, 552, 179, 5, 70, "Input",
 CellID->22332],
Cell[17180, 559, 361, 13, 56, "Output",
 CellID->189923683]
}, Open  ]],
Cell[17556, 575, 120, 3, 70, "ExampleDelimiter",
 CellID->9145],
Cell[17679, 580, 226, 5, 70, "ExampleText",
 CellID->30053],
Cell[CellGroupData[{
Cell[17930, 589, 243, 7, 70, "Input",
 CellID->17709],
Cell[18176, 598, 411, 15, 54, "Output",
 CellID->48128311]
}, Open  ]],
Cell[18602, 616, 119, 3, 70, "ExampleDelimiter",
 CellID->113],
Cell[18724, 621, 536, 19, 70, "ExampleText",
 CellID->48521834],
Cell[CellGroupData[{
Cell[19285, 644, 231, 7, 70, "Input",
 CellID->4591],
Cell[19519, 653, 159, 5, 36, "Output",
 CellID->67615007]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19727, 664, 222, 7, 70, "ExampleSection",
 CellID->9319],
Cell[19952, 673, 350, 11, 70, "ExampleText",
 CellID->32514],
Cell[CellGroupData[{
Cell[20327, 688, 227, 7, 70, "Input",
 CellID->10642],
Cell[20557, 697, 178, 5, 36, "Output",
 CellID->671266247]
}, Open  ]],
Cell[20750, 705, 124, 3, 70, "ExampleDelimiter",
 CellID->86062964],
Cell[20877, 710, 264, 8, 70, "ExampleText",
 CellID->13370],
Cell[CellGroupData[{
Cell[21166, 722, 349, 11, 70, "Input",
 CellID->26124],
Cell[21518, 735, 1200, 24, 70, "Output",
 Evaluatable->False,
 CellID->47439975]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22767, 765, 232, 7, 70, "ExampleSection",
 CellID->8113],
Cell[23002, 774, 266, 8, 70, "ExampleText",
 CellID->7633],
Cell[CellGroupData[{
Cell[23293, 786, 203, 6, 70, "Input",
 CellID->30290],
Cell[23499, 794, 157, 5, 36, "Output",
 CellID->51517808]
}, Open  ]],
Cell[23671, 802, 119, 3, 70, "ExampleDelimiter",
 CellID->636],
Cell[23793, 807, 234, 6, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[24052, 817, 281, 9, 70, "Input",
 CellID->17995],
Cell[24336, 828, 572, 19, 36, "Output",
 CellID->440741880]
}, Open  ]],
Cell[CellGroupData[{
Cell[24945, 852, 249, 7, 70, "Input",
 CellID->527296452],
Cell[25197, 861, 415, 14, 57, "Output",
 CellID->360148328]
}, Open  ]],
Cell[CellGroupData[{
Cell[25649, 880, 278, 10, 70, "Input",
 CellID->19463],
Cell[25930, 892, 1013, 35, 59, "Output",
 CellID->144684636]
}, Open  ]],
Cell[26958, 930, 121, 3, 70, "ExampleDelimiter",
 CellID->11620],
Cell[27082, 935, 617, 20, 70, "ExampleText",
 CellID->8056],
Cell[CellGroupData[{
Cell[27724, 959, 198, 6, 70, "Input",
 CellID->6133],
Cell[27925, 967, 252, 9, 36, "Output",
 CellID->76027097]
}, Open  ]],
Cell[CellGroupData[{
Cell[28214, 981, 170, 6, 70, "Input",
 CellID->15955],
Cell[28387, 989, 252, 9, 36, "Output",
 CellID->71885263]
}, Open  ]],
Cell[28654, 1001, 125, 3, 70, "ExampleDelimiter",
 CellID->932691717],
Cell[28782, 1006, 386, 13, 70, "ExampleText",
 CellID->217493462],
Cell[CellGroupData[{
Cell[29193, 1023, 208, 7, 70, "Input",
 CellID->364101735],
Cell[29404, 1032, 311, 11, 54, "Output",
 CellID->474676647]
}, Open  ]],
Cell[29730, 1046, 267, 10, 70, "Input",
 CellID->437860854],
Cell[CellGroupData[{
Cell[30022, 1060, 258, 7, 70, "Input",
 CellID->221897965],
Cell[30283, 1069, 446, 16, 56, "Output",
 CellID->807830191]
}, Open  ]],
Cell[CellGroupData[{
Cell[30766, 1090, 278, 9, 70, "Input",
 CellID->15690664],
Cell[31047, 1101, 161, 5, 36, "Output",
 CellID->360811085]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[31257, 1112, 327, 9, 70, "ExampleSection",
 CellID->25421],
Cell[31587, 1123, 450, 14, 70, "ExampleText",
 CellID->20461],
Cell[CellGroupData[{
Cell[32062, 1141, 195, 6, 70, "Input",
 CellID->22501],
Cell[32260, 1149, 208, 5, 70, "Message",
 CellID->27092585],
Cell[32471, 1156, 288, 9, 36, "Output",
 CellID->602841778]
}, Open  ]],
Cell[32774, 1168, 461, 14, 70, "ExampleText",
 CellID->3163],
Cell[CellGroupData[{
Cell[33260, 1186, 181, 5, 70, "Input",
 CellID->28738],
Cell[33444, 1193, 242, 5, 70, "Message",
 CellID->642963],
Cell[33689, 1200, 275, 8, 36, "Output",
 CellID->317265355]
}, Open  ]],
Cell[33979, 1211, 121, 3, 70, "ExampleDelimiter",
 CellID->17334],
Cell[34103, 1216, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[34409, 1228, 326, 11, 70, "Input",
 CellID->29499],
Cell[34738, 1241, 196, 7, 51, "Output",
 CellID->180220542]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[34995, 1255, 311, 9, 70, "SeeAlsoSection",
 CellID->17252],
Cell[35309, 1266, 1065, 35, 70, "SeeAlso",
 CellID->15857]
}, Open  ]],
Cell[CellGroupData[{
Cell[36411, 1306, 314, 9, 70, "TutorialsSection",
 CellID->30881],
Cell[36728, 1317, 153, 3, 70, "Tutorials",
 CellID->4738]
}, Open  ]],
Cell[CellGroupData[{
Cell[36918, 1325, 299, 8, 70, "MoreAboutSection"],
Cell[37220, 1335, 178, 3, 70, "MoreAbout",
 CellID->746109424]
}, Open  ]],
Cell[37413, 1341, 27, 0, 70, "History"],
Cell[37443, 1343, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

