(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     38668,       1330]
NotebookOptionsPosition[     31581,       1079]
NotebookOutlinePosition[     33126,       1121]
CellTagsIndexPosition[     33040,       1116]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ChiSquareDistribution" :> 
          Documentation`HelpLookup["paclet:ref/ChiSquareDistribution"], 
          "NoncentralFRatioDistribution" :> 
          Documentation`HelpLookup["paclet:ref/NoncentralFRatioDistribution"],
           "NoncentralStudentTDistribution" :> 
          Documentation`HelpLookup[
           "paclet:ref/NoncentralStudentTDistribution"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"ChiSquareDistribution\"\>", 
       2->"\<\"NoncentralFRatioDistribution\"\>", 
       3->"\<\"NoncentralStudentTDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/ContinuousStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Continuous Statistical Distributions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["NoncentralChiSquareDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NoncentralChiSquareDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NoncentralChiSquareDistribution"], "[", 
       RowBox[{
        StyleBox["\[Nu]", "TR"], ",", 
        StyleBox["\[Lambda]", "TR"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a noncentral ",
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
     " distribution with ",
     Cell[BoxData[
      FormBox["\[Nu]", TraditionalForm]], "InlineMath"],
     " degrees of freedom and noncentrality parameter ",
     Cell[BoxData[
      FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->32125]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NoncentralChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NoncentralChiSquareDistribution"]], 
  "InlineFormula"],
 " allows ",
 Cell[BoxData[
  FormBox["\[Nu]", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
 " to be any positive real numbers."
}], "Notes",
 CellID->11544],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NoncentralChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NoncentralChiSquareDistribution"]], 
  "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->14062,
  ButtonNote->"14062"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->926866570],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->23953916],

Cell[TextData[{
 "The mean and variance of a noncentral ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->14062],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10426],

Cell[BoxData[
 RowBox[{"\[Lambda]", "+", "\[Nu]"}]], "Output",
 ImageSize->{31, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12775619]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->10677],

Cell[BoxData[
 RowBox[{
  RowBox[{"4", " ", "\[Lambda]"}], "+", 
  RowBox[{"2", " ", "\[Nu]"}]}]], "Output",
 ImageSize->{51, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->652149428]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4047],

Cell["Probability density function:", "ExampleText",
 CellID->13785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"NoncentralChiSquareDistribution", "[", 
    RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23231],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["2", 
   RowBox[{
    RowBox[{"-", "\[Nu]"}], "/", "2"}]], " ", 
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "x"}], "-", "\[Lambda]"}], ")"}]}]], " ", 
  SuperscriptBox["x", 
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    FractionBox["\[Nu]", "2"]}]], " ", 
  RowBox[{"Hypergeometric0F1Regularized", "[", 
   RowBox[{
    FractionBox["\[Nu]", "2"], ",", 
    FractionBox[
     RowBox[{"x", " ", "\[Lambda]"}], "4"]}], "]"}]}]], "Output",
 ImageSize->{372, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23067664]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"NoncentralChiSquareDistribution", "[", 
      RowBox[{"3", ",", "4"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->8684],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->63788617]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14709],

Cell[TextData[{
 "Generate a set of pseudorandom numbers that have the noncentral ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution:"
}], "ExampleText",
 CellID->27956],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomReal", "[", 
  RowBox[{
   RowBox[{"NoncentralChiSquareDistribution", "[", 
    RowBox[{"5", ",", "10"}], "]"}], ",", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22421],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "10.537907623167781`", ",", "24.3876077744719`", ",", "31.83993730358695`", 
   ",", "24.19770271677156`", ",", "13.374099415176373`"}], "}"}]], "Output",
 ImageSize->{308, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->575406511]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13465],

Cell["Properties based on higher-order moments:", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16016],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   SqrtBox["2"], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"3", " ", "\[Lambda]"}], "+", "\[Nu]"}], ")"}]}], 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"2", " ", "\[Lambda]"}], "+", "\[Nu]"}], ")"}], 
   RowBox[{"3", "/", "2"}]]]], "Output",
 ImageSize->{96, 38},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->323521562]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18335],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox[
   RowBox[{"12", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"4", " ", "\[Lambda]"}], "+", "\[Nu]"}], ")"}]}], 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", " ", "\[Lambda]"}], "+", "\[Nu]"}], ")"}], 
    "2"]]}]], "Output",
 ImageSize->{96, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->60307909]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5677],

Cell[TextData[{
 "The 0.75 quantile of a noncentral ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution:"
}], "ExampleText",
 CellID->28049],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"NoncentralChiSquareDistribution", "[", 
    RowBox[{"5", ",", "10"}], "]"}], ",", "0.75"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21005],

Cell[BoxData["19.15881098492775`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->303980827]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->13143],

Cell["\<\
Plot the cumulative distribution function of the random variable:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->19271],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"NoncentralChiSquareDistribution", "[", 
      RowBox[{"10", ",", "2"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "25"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12368],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWX9o1VUU/+6973erSCiw3JrJigyz5ly2bOpyNjdda84fLbOl9lyzrdLZ
nFbLSqPSP6o/iiBBEikjqz+MxAgMIiESIiESooIwCPsjCZL+GATfPufce9/3
+z3f87b3XvhHj+d993zv53zuueeec+79zu7c6ODAltzoUH+ubtlIbtvgUP/2
uo7hETzKVniedwz/xqo99EP0+N+xMMRvuq1HmwWg1v5m8JvBb7WVPZ9h9kNP
fYHyU6gsC3OpJYqQP14NNX4kG5Y44mohZ7XxwMj5dkIWZxo5pYJ+L5Jl0yZl
mKYwXGyrqoUnpH61kDMpm+bYfZ5enC0RwurVlmpBPnrmiJHa4iwI4hbQjOzF
yriern3VpPwsew12ZdfY32wYZULGRltcduM1YtywxTPxCjHfoQmeTkF7OM+S
x5z+9jd8vSw3kexTcxnkg29/g++rr5wc3vrJ2t738GRo8Oj+t06d/PIXuVpf
yDK2pFwq/r/q/y/xqd28kprKM9//fvjd08+MfdbedgDt5yd+rhS46UK+tsRx
ffapPPv4+D/Y/z27T6zqeQfRcf7833tVNFewAMMAIXgQNhcujHsv5bEuKzwf
jFgPMB9+8N2LKpfJZui/8fpXCMGzZ/98QcWZdcA//Zs++vjoGW+PiprhZkVk
w3+gfV7F1bGMxZopvecStl/Po3iO0VNf/+o9q3IYFKbYMXr80+M/eLtU1A0s
7335C/iBumMqaibLcCQ8St2nVdSNZB8Q0ShZO8s+fUrVcaNPqqM32dGdPGqD
gz/xqGXZVqy49my3NiyfuqMGm6idgeCRHDezDBfC0dTdrnDIHCDbSDS62G3s
008//qHqVgldWd1uYRmxDI5z5/7yRsrgqI/FC2XCEwoHrzv9JMZizjm4Ac6g
7rbUnlTaHQ+EZgPLSDHkEHWHE/HRyKNICKQNdbeqkTDX+RKz0yGhoxqFt7ao
qFtZxo4ABUbvcRU1j2WUOBQc6j6WsLqJR7EgLIu6j6octzmrcYyhNhRANTmr
UTvQekMq6naWsXZwUXdQRc0Xvn4kYXUzj6LyoLRQd7PKcYezGvZQ9dFRzS6q
gCIvDqTigbupe46MjgUswyCYRd2Hw+IyLC4vFLvRr3BQPPv2l1QXir0ppBOv
DtL2RWLnNikcQahnRYvYz1wJuneKXX4o5XtZE6THFotI2JiIk1YeRa2w5080
GudoFRGwQUUtEfu7XkXdxTJOGHsOPaii2oTVfQmrlwqr+1QOg0J1gNVUDR9Q
Ue1ud1BvaG3rVFQHyyhcKF/UvV9FLQuT0bk2YXUnj0YnzH0qx3K3NkQLnSU6
qtOtDR6geOxNxYWMAy267mY5qo/3hgUyY8Lzoovl6LxYUzQLkdzD2ji4cXxT
d3VZNnSH5v5QSDurrH5FmIzElYqu1JHz9rAc3Tx6UvtQFUZvWXHNlUJzRSJW
VoW2ZiEF6OYXjcY5JKpbRa2OoZBOxuMp1JoYim6RXSqq11ltz/DOhNUc73TC
29vjcpXDoMxbX+iyJoXiDKOMs3nZoaLWudizedCuojjzKc5RCegEWJqwmisQ
1XZkHN3o21QORtkbPb7oLFFxG5zdtja1piKimMzc6CwGC24mpoqXnBV8WtA7
DW422DN0WormIRo+5Sg31/cdIb8tUrQLnVx8yvrmhgZ/Ye4FZa1hwPkTNlCM
N5fFstlZg2gyhXV+al8K3Wr5/kV/sED0YCUgaUrED98XfViYfwWdp0YG44L8
mygSgE6lRhXL99kA4+bNFkcKRXeDih12WJw6wGKjaa/qVSy/EdD7NTYVG4Mv
OlCdnVjRDmYEC7jwCo6WbuKzVMadMUZ4FnAYC//OVNH8xliFtcBWuNOQoyiR
J65TNXY5DfCjNHV1HoTR8ApZNEPV2E3NJbAeKFi0uGU/WvQRPZimJrFS/lvB
peDCOEID7AgzTGPglL9T1Tn2UXM5xmG7UYUeZoJtRhvPwQq3TIkFpPuk/o6n
RN1rvG6swswAVqyive0AJvEy6HoBPIcBOBITkaXMkOLOTpobb5IvyJ0BXAxG
7B/4UTLKZjxUgh4BisW/XxJOy+0j/Cj9fw5exb+HiYsz\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->85593266]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31310],

Cell["The probability density function integrates to unity:", "ExampleText",
 CellID->8454],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"NoncentralChiSquareDistribution", "[", 
      RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Nu]", ">", "0"}], ",", 
      RowBox[{"\[Lambda]", ">", "0"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->304],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->382360815]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8968],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["NoncentralChiSquareDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NoncentralChiSquareDistribution"], "[", 
   RowBox[{
    StyleBox["\[Nu]", "TR"], ",", "0"}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ChiSquareDistribution",
    BaseStyle->"FunctionLink",
    ButtonData->"paclet:ref/ChiSquareDistribution"], "[", 
   StyleBox["\[Nu]", "TR"], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1710],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"NoncentralChiSquareDistribution", "[", 
    RowBox[{"\[Nu]", ",", "0"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21968],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{
     RowBox[{"-", "\[Nu]"}], "/", "2"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "x"}], "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["\[Nu]", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{100, 47},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->532505338]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2641],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{
     RowBox[{"-", "\[Nu]"}], "/", "2"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "x"}], "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["\[Nu]", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{100, 47},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->318393330]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->15012],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "2"}], ",", 
   RowBox[{"NoncentralChiSquareDistribution", "[", 
    RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30792],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[Lambda]", "2"], "+", 
  RowBox[{"2", " ", "\[Lambda]", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Nu]"}], ")"}]}], "+", 
  RowBox[{"\[Nu]", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Nu]"}], ")"}]}]}]], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->True,
 ImageSize->{153, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->4094319]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", "=", 
  RowBox[{"CharacteristicFunction", "[", 
   RowBox[{
    RowBox[{"NoncentralChiSquareDistribution", "[", 
     RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], ",", "t"}], "]"}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->183651900],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     RowBox[{"t", " ", "\[Lambda]"}], 
     RowBox[{"\[ImaginaryI]", "+", 
      RowBox[{"2", " ", "t"}]}]]}]], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", 
     RowBox[{"2", " ", "\[ImaginaryI]", " ", "t"}]}], ")"}], 
   RowBox[{
    RowBox[{"-", "\[Nu]"}], "/", "2"}]]}]], "Output",
 ImageSize->{127, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->174296336]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{"cf", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "2"}], "}"}]}], "]"}], "/", 
    RowBox[{"I", "^", "2"}]}], "/.", 
   RowBox[{"t", "\[Rule]", "0"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->5820],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[Lambda]", "2"], "+", 
  RowBox[{"2", " ", "\[Lambda]", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Nu]"}], ")"}]}], "+", 
  RowBox[{"\[Nu]", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Nu]"}], ")"}]}]}]], "Output",
 ImageSize->{153, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->4933293]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(3)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12437],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NoncentralChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NoncentralChiSquareDistribution"]], 
  "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " is not a positive real number:"
}], "ExampleText",
 CellID->11684],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "\[Lambda]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32499],

Cell[BoxData[
 RowBox[{
  RowBox[{"NoncentralChiSquareDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-1\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->741117607],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "\[Lambda]"}], "]"}], "]"}]], "Output",
 ImageSize->{310, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->560166050]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->28583],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NoncentralChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NoncentralChiSquareDistribution"]], 
  "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Lambda]", "TR"]], "InlineFormula"],
 " is not a positive real number:"
}], "ExampleText",
 CellID->29256],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{"\[Nu]", ",", 
    RowBox[{"-", "5"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19692],

Cell[BoxData[
 RowBox[{
  RowBox[{"NoncentralChiSquareDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-5\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->183499132],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"NoncentralChiSquareDistribution", "[", 
   RowBox[{"\[Nu]", ",", 
    RowBox[{"-", "5"}]}], "]"}], "]"}]], "Output",
 ImageSize->{310, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->173404696]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19046],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"NoncentralChiSquareDistribution", "[", 
    RowBox[{"\[Nu]", ",", "\[Lambda]"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"\[Nu]", "\[Rule]", 
     RowBox[{"3", "+", "I"}]}], ",", 
    RowBox[{"\[Lambda]", "\[Rule]", "2"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25838],

Cell[BoxData[
 RowBox[{"5", "+", "\[ImaginaryI]"}]], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->19014855]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->12021],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChiSquareDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChiSquareDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NoncentralFRatioDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NoncentralFRatioDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NoncentralStudentTDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NoncentralStudentTDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->22586]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->12826],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Continuous Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ContinuousStatisticalDistributions"]], "MoreAbout",\

 CellID->237213586]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NoncentralChiSquareDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 2, 22.9669644}", "context" -> "System`", 
    "keywords" -> {
     "noncentral chi squared distribution", 
      "noncentral chi-square distribution", 
      "noncentral chi-squared distribution"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "NoncentralChiSquareDistribution[\\[Nu], \\[Lambda]] represents a \
noncentral \\[Chi]^2 distribution with \\[Nu] degrees of freedom and \
noncentrality parameter \\[Lambda].", 
    "synonyms" -> {"noncentral chi square distribution"}, "title" -> 
    "NoncentralChiSquareDistribution", "type" -> "Symbol", "uri" -> 
    "ref/NoncentralChiSquareDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7072, 218, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->926866570]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 32897, 1109}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2220, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2843, 82, 1596, 37, 70, "ObjectNameGrid"],
Cell[4442, 121, 936, 27, 70, "Usage",
 CellID->32125]
}, Open  ]],
Cell[CellGroupData[{
Cell[5415, 153, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5893, 170, 412, 14, 70, "Notes",
 CellID->11544],
Cell[6308, 186, 727, 27, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7072, 218, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->926866570],
Cell[CellGroupData[{
Cell[7457, 232, 147, 5, 70, "ExampleSection",
 CellID->23953916],
Cell[7607, 239, 253, 8, 70, "ExampleText",
 CellID->14062],
Cell[CellGroupData[{
Cell[7885, 251, 190, 5, 28, "Input",
 CellID->10426],
Cell[8078, 258, 191, 6, 36, "Output",
 CellID->12775619]
}, Open  ]],
Cell[CellGroupData[{
Cell[8306, 269, 194, 5, 28, "Input",
 CellID->10677],
Cell[8503, 276, 238, 8, 36, "Output",
 CellID->652149428]
}, Open  ]],
Cell[8756, 287, 120, 3, 70, "ExampleDelimiter",
 CellID->4047],
Cell[8879, 292, 68, 1, 70, "ExampleText",
 CellID->13785],
Cell[CellGroupData[{
Cell[8972, 297, 214, 6, 70, "Input",
 CellID->23231],
Cell[9189, 305, 696, 24, 52, "Output",
 CellID->23067664]
}, Open  ]],
Cell[CellGroupData[{
Cell[9922, 334, 322, 10, 28, "Input",
 CellID->8684],
Cell[10247, 346, 3111, 55, 166, "Output",
 Evaluatable->False,
 CellID->63788617]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13407, 407, 216, 7, 70, "ExampleSection",
 CellID->14709],
Cell[13626, 416, 231, 7, 70, "ExampleText",
 CellID->27956],
Cell[CellGroupData[{
Cell[13882, 427, 210, 6, 70, "Input",
 CellID->22421],
Cell[14095, 435, 327, 9, 36, "Output",
 CellID->575406511]
}, Open  ]],
Cell[14437, 447, 121, 3, 70, "ExampleDelimiter",
 CellID->13465],
Cell[14561, 452, 83, 1, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[14669, 457, 194, 5, 70, "Input",
 CellID->16016],
Cell[14866, 464, 446, 16, 59, "Output",
 CellID->323521562]
}, Open  ]],
Cell[CellGroupData[{
Cell[15349, 485, 194, 5, 70, "Input",
 CellID->18335],
Cell[15546, 492, 435, 16, 54, "Output",
 CellID->60307909]
}, Open  ]],
Cell[15996, 511, 120, 3, 70, "ExampleDelimiter",
 CellID->5677],
Cell[16119, 516, 201, 7, 70, "ExampleText",
 CellID->28049],
Cell[CellGroupData[{
Cell[16345, 527, 211, 6, 70, "Input",
 CellID->21005],
Cell[16559, 535, 175, 5, 36, "Output",
 CellID->303980827]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16783, 546, 223, 7, 70, "ExampleSection",
 CellID->13143],
Cell[17009, 555, 249, 7, 70, "ExampleText",
 CellID->19271],
Cell[CellGroupData[{
Cell[17283, 566, 324, 10, 70, "Input",
 CellID->12368],
Cell[17610, 578, 2330, 42, 70, "Output",
 Evaluatable->False,
 CellID->85593266]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19989, 626, 233, 7, 70, "ExampleSection",
 CellID->31310],
Cell[20225, 635, 91, 1, 70, "ExampleText",
 CellID->8454],
Cell[CellGroupData[{
Cell[20341, 640, 514, 15, 70, "Input",
 CellID->304],
Cell[20858, 657, 158, 5, 36, "Output",
 CellID->382360815]
}, Open  ]],
Cell[21031, 665, 120, 3, 70, "ExampleDelimiter",
 CellID->8968],
Cell[21154, 670, 530, 17, 70, "ExampleText",
 CellID->1710],
Cell[CellGroupData[{
Cell[21709, 691, 206, 6, 70, "Input",
 CellID->21968],
Cell[21918, 699, 529, 19, 68, "Output",
 CellID->532505338]
}, Open  ]],
Cell[CellGroupData[{
Cell[22484, 723, 173, 6, 70, "Input",
 CellID->2641],
Cell[22660, 731, 529, 19, 68, "Output",
 CellID->318393330]
}, Open  ]],
Cell[23204, 753, 121, 3, 70, "ExampleDelimiter",
 CellID->15012],
Cell[23328, 758, 97, 2, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[23450, 764, 258, 7, 70, "Input",
 CellID->30792],
Cell[23711, 773, 444, 15, 36, "Output",
 CellID->4094319]
}, Open  ]],
Cell[CellGroupData[{
Cell[24192, 793, 266, 8, 70, "Input",
 CellID->183651900],
Cell[24461, 803, 529, 18, 45, "Output",
 CellID->174296336]
}, Open  ]],
Cell[CellGroupData[{
Cell[25027, 826, 314, 11, 70, "Input",
 CellID->5820],
Cell[25344, 839, 395, 13, 36, "Output",
 CellID->4933293]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25788, 858, 327, 9, 70, "ExampleSection",
 CellID->12437],
Cell[26118, 869, 340, 11, 70, "ExampleText",
 CellID->11684],
Cell[CellGroupData[{
Cell[26483, 884, 206, 6, 70, "Input",
 CellID->32499],
Cell[26692, 892, 212, 5, 70, "Message",
 CellID->741117607],
Cell[26907, 899, 299, 9, 36, "Output",
 CellID->560166050]
}, Open  ]],
Cell[27221, 911, 121, 3, 70, "ExampleDelimiter",
 CellID->28583],
Cell[27345, 916, 344, 11, 70, "ExampleText",
 CellID->29256],
Cell[CellGroupData[{
Cell[27714, 931, 202, 6, 70, "Input",
 CellID->19692],
Cell[27919, 939, 212, 5, 70, "Message",
 CellID->183499132],
Cell[28134, 946, 295, 9, 36, "Output",
 CellID->173404696]
}, Open  ]],
Cell[28444, 958, 121, 3, 70, "ExampleDelimiter",
 CellID->19046],
Cell[28568, 963, 144, 4, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[28737, 971, 363, 11, 70, "Input",
 CellID->25838],
Cell[29103, 984, 191, 6, 36, "Output",
 CellID->19014855]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29355, 997, 311, 9, 70, "SeeAlsoSection",
 CellID->12021],
Cell[29669, 1008, 806, 26, 70, "SeeAlso",
 CellID->22586]
}, Open  ]],
Cell[CellGroupData[{
Cell[30512, 1039, 314, 9, 70, "TutorialsSection",
 CellID->12826],
Cell[30829, 1050, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[31023, 1058, 299, 8, 70, "MoreAboutSection"],
Cell[31325, 1068, 184, 4, 70, "MoreAbout",
 CellID->237213586]
}, Open  ]],
Cell[31524, 1075, 27, 0, 70, "History"],
Cell[31554, 1077, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

