(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     42923,       1639]
NotebookOptionsPosition[     34508,       1348]
NotebookOutlinePosition[     35986,       1387]
CellTagsIndexPosition[     35900,       1382]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Fields" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AlgebraicNumberFields"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Algebraic Number Fields\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"AlgebraicNumberNorm" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumberNorm"], 
          "NumberFieldFundamentalUnits" :> 
          Documentation`HelpLookup["paclet:ref/NumberFieldFundamentalUnits"], 
          "AlgebraicIntegerQ" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicIntegerQ"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"AlgebraicNumberNorm\"\>", 
       2->"\<\"NumberFieldFundamentalUnits\"\>", 
       3->"\<\"AlgebraicIntegerQ\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumberTheory"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Algebraic Number Theory\"\>", 
       2->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["NumberFieldNormRepresentatives", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["NumberFieldNormRepresentatives",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/NumberFieldNormRepresentatives"], "[", 
       RowBox[{
        StyleBox["a", "TI"], ",", 
        StyleBox["m", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives a list of representatives of classes of \
algebraic integers of norm ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[PlusMinus]", "m"}], TraditionalForm]], "InlineMath"],
     " in the field ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[DoubleStruckCapitalQ]", "[", "a", "]"}], TraditionalForm]], 
      "InlineMath"],
     " generated by the algebraic number ",
     Cell[BoxData[
      FormBox["a", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "Algebraic integers are considered to be in the same class if their quotient \
is a unit in the field ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", "a", "]"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 "All elements of the number field",
 " ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", "a", "]"}], TraditionalForm]], 
  "InlineMath"],
 " with norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", "m"}], TraditionalForm]], "InlineMath"],
 " can be obtained from the representatives by multiplication by units in the \
field."
}], "Notes",
 CellID->841903923]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->115037004],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->285640943],

Cell[TextData[{
 "Find the representatives of classes of algebraic integers of norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", "7"}], TraditionalForm]], "InlineMath"],
 " in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", 
    SqrtBox["2"], "]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1319811665],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumberFieldNormRepresentatives", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], ",", "7"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15751104],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     SqrtBox["2"], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "3"}], ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "]"}], ",", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     SqrtBox["2"], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "3"}], ",", "1"}], "}"}]}], "]"}]}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{439, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->385928452]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1179987606],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"7", ",", "7"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->898878824]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->810193784],

Cell["Radical expressions:", "ExampleText",
 CellID->42307234],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumberFieldNormRepresentatives", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], "+", 
    RowBox[{"Sqrt", "[", "3", "]"}]}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->381716645],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"1", "-", 
        RowBox[{"10", " ", 
         SuperscriptBox["#1", "2"]}], "+", 
        SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["9", "4"]}], ",", 
      FractionBox["9", "4"], ",", 
      FractionBox["1", "4"], ",", 
      RowBox[{"-", 
       FractionBox["1", "4"]}]}], "}"}]}], "]"}], "}"}]], "Output",
 ImageSize->{415, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2056421970]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1143389269],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->1259770727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumberFieldNormRepresentatives", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"2", "+", 
       RowBox[{"3", 
        RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "2"}], "]"}], ",", "3"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->925332916],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"18", "+", 
        SuperscriptBox["#1", "3"]}], "&"}], ",", "2"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "3"}], ",", "1", ",", 
      RowBox[{"-", 
       FractionBox["1", "3"]}]}], "}"}]}], "]"}], "}"}]], "Output",
 ImageSize->{342, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->682234972]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->794481278],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->1941005947],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumberFieldNormRepresentatives", "[", 
  RowBox[{
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"2", "+", 
         RowBox[{"#1", "^", "3"}]}], "&"}], ",", "1"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", 
       RowBox[{"-", "1"}], ",", "2"}], "}"}]}], "]"}], ",", "12"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->62869652],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"-", "45"}], "-", 
        RowBox[{"9", " ", "#1"}], "+", 
        RowBox[{"3", " ", 
         SuperscriptBox["#1", "2"]}], "+", 
        SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", 
      FractionBox["2", "5"], ",", 
      RowBox[{"-", 
       FractionBox["1", "15"]}]}], "}"}]}], "]"}], "}"}]], "Output",
 ImageSize->{432, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->907983071]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1664567884],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NumberFieldNormRepresentatives",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NumberFieldNormRepresentatives"]], 
  "InlineFormula"],
 " automatically threads over lists:"
}], "ExampleText",
 CellID->1642223539],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumberFieldNormRepresentatives", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sqrt", "[", "3", "]"}], ",", " ", 
     RowBox[{"E", "^", 
      RowBox[{"(", 
       RowBox[{"Pi", "*", 
        RowBox[{"I", "/", "4"}]}], ")"}]}]}], "}"}], ",", "2"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1315736145],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"AlgebraicNumber", "[", 
     RowBox[{
      SqrtBox["3"], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", 
        RowBox[{"-", "1"}]}], "}"}]}], "]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"AlgebraicNumber", "[", 
     RowBox[{
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"1", "+", 
          SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1"}], ",", 
        RowBox[{"-", "1"}], ",", "0", ",", "0"}], "}"}]}], "]"}], "}"}]}], 
  "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{363, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->452678715]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1736499],

Cell[TextData[{
 "Representatives of norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", "5"}], TraditionalForm]], "InlineMath"],
 " in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", 
    SqrtBox["5"], "]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->283144896],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", "a", "}"}], "=", 
  RowBox[{"NumberFieldNormRepresentatives", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "5", "]"}], ",", "5"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->328703506],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    SqrtBox["5"], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "5"}], ",", 
      RowBox[{"-", "2"}]}], "}"}]}], "]"}], "}"}]], "Output",
 ImageSize->{227, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1784085331]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", "a", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1229514136],

Cell[BoxData["5"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1955478703]
}, Open  ]],

Cell[TextData[{
 "The number ",
 Cell[BoxData[
  FormBox[
   SqrtBox["5"], TraditionalForm]], "InlineMath"],
 "has norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "5"}], TraditionalForm]], "InlineMath"],
 ": "
}], "ExampleText",
 CellID->1226827479],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{"Sqrt", "[", "5", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1995105097],

Cell[BoxData[
 RowBox[{"-", "5"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->640827633]
}, Open  ]],

Cell[TextData[{
 "It can be represented in terms of the representative ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " by multiplying by a unit:"
}], "ExampleText",
 CellID->330449609],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootReduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "5", "]"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "2"}], "-", 
      RowBox[{"Sqrt", "[", "5", "]"}]}], ")"}]}], "\[Equal]", "a"}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->63620644],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->759538126]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicUnitQ", "[", 
  RowBox[{
   RowBox[{"-", "2"}], "-", 
   RowBox[{"Sqrt", "[", "5", "]"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->92994129],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->914485914]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2139988967],

Cell[TextData[{
 "Obtain all elements of norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", "7"}], TraditionalForm]], "InlineMath"],
 " in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", 
    SqrtBox["2"], "]"}], TraditionalForm]], "InlineMath"],
 " by multiplying representatives with units:"
}], "ExampleText",
 CellID->240796388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"a", ",", " ", "b"}], "}"}], " ", "=", 
  RowBox[{"NumberFieldNormRepresentatives", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", "7"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->194696687],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     SqrtBox["2"], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "3"}], ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "]"}], ",", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     SqrtBox["2"], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "3"}], ",", "1"}], "}"}]}], "]"}]}], "}"}]], "Output",
 ImageSize->{439, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->224295019]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", "u", "}"}], " ", "=", " ", 
  RowBox[{"NumberFieldFundamentalUnits", "[", 
   RowBox[{"Sqrt", "[", "2", "]"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->44918340],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    SqrtBox["2"], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1"}], "}"}]}], "]"}], "}"}]], "Output",
 ImageSize->{212, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->892583848]
}, Open  ]],

Cell[TextData[{
 "Elements generated by ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->878254457],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"RootReduce", "[", 
    RowBox[{"a", " ", 
     RowBox[{"u", "^", "n"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->381411379],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "5"}], "-", 
    RowBox[{"4", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "13"}], "-", 
    RowBox[{"9", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "31"}], "-", 
    RowBox[{"22", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "75"}], "-", 
    RowBox[{"53", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "181"}], "-", 
    RowBox[{"128", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "437"}], "-", 
    RowBox[{"309", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "1055"}], "-", 
    RowBox[{"746", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "2547"}], "-", 
    RowBox[{"1801", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "6149"}], "-", 
    RowBox[{"4348", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "14845"}], "-", 
    RowBox[{"10497", " ", 
     SqrtBox["2"]}]}]}], "}"}]], "Output",
 ImageSize->{533, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1483459041]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", " ", "/@", " ", "%"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1932490251],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7"}], "}"}]], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->210960417]
}, Open  ]],

Cell[TextData[{
 "Elements generated by ",
 Cell[BoxData[
  FormBox["b", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->2031362948],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"RootReduce", "[", 
    RowBox[{"b", " ", 
     RowBox[{"u", "^", "n"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->422047030],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "1"}], "-", 
    RowBox[{"2", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "5"}], "-", 
    RowBox[{"3", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "11"}], "-", 
    RowBox[{"8", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "27"}], "-", 
    RowBox[{"19", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "65"}], "-", 
    RowBox[{"46", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "157"}], "-", 
    RowBox[{"111", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "379"}], "-", 
    RowBox[{"268", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "915"}], "-", 
    RowBox[{"647", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "2209"}], "-", 
    RowBox[{"1562", " ", 
     SqrtBox["2"]}]}], ",", 
   RowBox[{
    RowBox[{"-", "5333"}], "-", 
    RowBox[{"3771", " ", 
     SqrtBox["2"]}]}]}], "}"}]], "Output",
 ImageSize->{505, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->116227322]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", " ", "/@", " ", "%"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->1757914066],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7", ",", 
   RowBox[{"-", "7"}], ",", "7"}], "}"}]], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->1369965731]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->603415282],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FindInstance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindInstance"]], "InlineFormula"],
 " gives all Gaussian integers of norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", "5"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1182863440],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"gi1", "=", 
  RowBox[{
   RowBox[{"a", "+", 
    RowBox[{"I", " ", "b"}]}], "/.", 
   RowBox[{"FindInstance", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"a", "^", "2"}], "+", 
       RowBox[{"b", "^", "2"}]}], "\[Equal]", " ", "5"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "b"}], "}"}], ",", " ", "Integers", ",", "100"}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->548236987],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "2"}], "-", "\[ImaginaryI]"}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], "+", "\[ImaginaryI]"}], ",", 
   RowBox[{
    RowBox[{"-", "1"}], "-", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"1", "-", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"1", "+", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"2", "-", "\[ImaginaryI]"}], ",", 
   RowBox[{"2", "+", "\[ImaginaryI]"}]}], "}"}]], "Output",
 ImageSize->{389, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1841861430]
}, Open  ]],

Cell["Check the result:", "ExampleText",
 CellID->1023782415],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumberFieldNormRepresentatives", "[", 
  RowBox[{"I", ",", "5"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1725342136],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{"\[ImaginaryI]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "2"}]}], "}"}]}], "]"}], ",", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{"\[ImaginaryI]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}]}], "]"}]}], "}"}]], "Output",
 ImageSize->{392, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->188282505]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"gi2", "=", 
  RowBox[{"Flatten", " ", "@", " ", 
   RowBox[{"Outer", "[", 
    RowBox[{"Times", ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "1", ",", 
       RowBox[{"-", "I"}], ",", "I"}], "}"}], ",", 
     RowBox[{"RootReduce", "[", "%", "]"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->436987237],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "1"}], "-", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"1", "-", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"1", "+", 
    RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], "-", "\[ImaginaryI]"}], ",", 
   RowBox[{"2", "-", "\[ImaginaryI]"}], ",", 
   RowBox[{"2", "+", "\[ImaginaryI]"}], ",", 
   RowBox[{
    RowBox[{"-", "2"}], "+", "\[ImaginaryI]"}]}], "}"}]], "Output",
 ImageSize->{389, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1220649504]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Sort", "[", "gi1", "]"}], "\[Equal]", 
  RowBox[{"Sort", "[", "gi2", "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->244741605],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->544933937]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1167552968],

Cell[TextData[{
 "Find an instance of a quadratic equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["a", "2"], "-", 
     RowBox[{"2", " ", 
      SuperscriptBox["b", "2"]}]}], "=", "8"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1504650537],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"First", " ", "@", " ", 
  RowBox[{"RootReduce", " ", "@", 
   RowBox[{"NumberFieldNormRepresentatives", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], ",", "8"}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2041631934],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "4"}], "-", 
  RowBox[{"2", " ", 
   SqrtBox["2"]}]}]], "Output",
 ImageSize->{66, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1834408351]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"a", ",", "b"}], "}"}], "=", 
  RowBox[{"%", "/.", 
   RowBox[{
    RowBox[{"a_", "+", 
     RowBox[{
      RowBox[{"Sqrt", "[", "2", "]"}], "b_"}]}], "\[RuleDelayed]", 
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}]}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->755619488],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4"}], ",", 
   RowBox[{"-", "2"}]}], "}"}]], "Output",
 ImageSize->{59, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1625918925]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "^", "2"}], "-", 
  RowBox[{"2", 
   RowBox[{"b", "^", "2"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->233332902],

Cell[BoxData["8"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->89964400]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1520291085],

Cell[TextData[{
 "Find the representatives of classes of algebraic integers of norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", "2"}], TraditionalForm]], "InlineMath"],
 " in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", 
    RowBox[{
     SqrtBox["2"], ",", 
     SqrtBox["3"]}], "]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1007153122],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"primitive", "=", 
  RowBox[{
   RowBox[{"ToNumberField", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Sqrt", "[", "2", "]"}], ",", " ", 
       RowBox[{"Sqrt", "[", "3", "]"}]}], "}"}], ",", " ", "All"}], "]"}], 
   "[", 
   RowBox[{"[", 
    RowBox[{"1", ",", "1"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->483123609],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"1", "-", 
     RowBox[{"10", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]], "Output",
 ImageSize->{163, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->203183312]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NumberFieldNormRepresentatives", "[", 
   RowBox[{"primitive", ",", "2"}], "]"}], " "}]], "Input",
 CellLabel->"In[2]:=",
 CellID->312289525],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"1", "-", 
        RowBox[{"10", " ", 
         SuperscriptBox["#1", "2"]}], "+", 
        SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["9", "4"]}], ",", 
      FractionBox["9", "4"], ",", 
      FractionBox["1", "4"], ",", 
      RowBox[{"-", 
       FractionBox["1", "4"]}]}], "}"}]}], "]"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{415, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->402926054]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumberNorm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumberNorm"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NumberFieldFundamentalUnits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NumberFieldFundamentalUnits"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicIntegerQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicIntegerQ"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Algebraic Number Fields",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AlgebraicNumberFields"]], "Tutorials",
 CellID->341631938]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->522916940],

Cell[TextData[ButtonBox["Algebraic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumberTheory"]], "MoreAbout",
 CellID->263031764],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->627481082]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NumberFieldNormRepresentatives - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 4, 10.1103322}", "context" -> "System`", 
    "keywords" -> {"norm representatives", "number field"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "NumberFieldNormRepresentatives[a, m] gives a list of representatives of \
classes of algebraic integers of norm \\[PlusMinus]m in the field \
\\[DoubleStruckCapitalQ][a] generated by the algebraic number a.", 
    "synonyms" -> {"number field norm representatives"}, "title" -> 
    "NumberFieldNormRepresentatives", "type" -> "Symbol", "uri" -> 
    "ref/NumberFieldNormRepresentatives"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6801, 204, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->115037004]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 35757, 1375}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2331, 56, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2954, 83, 1595, 37, 70, "ObjectNameGrid"],
Cell[4552, 122, 1009, 30, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5598, 157, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6076, 174, 278, 9, 70, "Notes",
 CellID->1067943069],
Cell[6357, 185, 407, 14, 70, "Notes",
 CellID->841903923]
}, Closed]],
Cell[CellGroupData[{
Cell[6801, 204, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->115037004],
Cell[CellGroupData[{
Cell[7186, 218, 148, 5, 70, "ExampleSection",
 CellID->285640943],
Cell[7337, 225, 363, 12, 70, "ExampleText",
 CellID->1319811665],
Cell[CellGroupData[{
Cell[7725, 241, 180, 5, 28, "Input",
 CellID->15751104],
Cell[7908, 248, 593, 22, 43, "Output",
 CellID->385928452]
}, Open  ]],
Cell[CellGroupData[{
Cell[8538, 275, 116, 3, 28, "Input",
 CellID->1179987606],
Cell[8657, 280, 251, 9, 36, "Output",
 CellID->898878824]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8957, 295, 220, 7, 70, "ExampleSection",
 CellID->810193784],
Cell[9180, 304, 62, 1, 70, "ExampleText",
 CellID->42307234],
Cell[CellGroupData[{
Cell[9267, 309, 239, 7, 70, "Input",
 CellID->381716645],
Cell[9509, 318, 677, 23, 51, "Output",
 CellID->2056421970]
}, Open  ]],
Cell[10201, 344, 126, 3, 70, "ExampleDelimiter",
 CellID->1143389269],
Cell[10330, 349, 179, 7, 70, "ExampleText",
 CellID->1259770727],
Cell[CellGroupData[{
Cell[10534, 360, 310, 11, 70, "Input",
 CellID->925332916],
Cell[10847, 373, 519, 18, 51, "Output",
 CellID->682234972]
}, Open  ]],
Cell[11381, 394, 125, 3, 70, "ExampleDelimiter",
 CellID->794481278],
Cell[11509, 399, 201, 7, 70, "ExampleText",
 CellID->1941005947],
Cell[CellGroupData[{
Cell[11735, 410, 473, 16, 70, "Input",
 CellID->62869652],
Cell[12211, 428, 658, 22, 51, "Output",
 CellID->907983071]
}, Open  ]],
Cell[12884, 453, 126, 3, 70, "ExampleDelimiter",
 CellID->1664567884],
Cell[13013, 458, 259, 8, 70, "ExampleText",
 CellID->1642223539],
Cell[CellGroupData[{
Cell[13297, 470, 352, 12, 70, "Input",
 CellID->1315736145],
Cell[13652, 484, 811, 29, 65, "Output",
 CellID->452678715]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14512, 519, 235, 7, 70, "ExampleSection",
 CellID->1736499],
Cell[14750, 528, 320, 12, 70, "ExampleText",
 CellID->283144896],
Cell[CellGroupData[{
Cell[15095, 544, 229, 7, 70, "Input",
 CellID->328703506],
Cell[15327, 553, 357, 13, 43, "Output",
 CellID->1784085331]
}, Open  ]],
Cell[CellGroupData[{
Cell[15721, 571, 116, 3, 70, "Input",
 CellID->1229514136],
Cell[15840, 576, 159, 5, 36, "Output",
 CellID->1955478703]
}, Open  ]],
Cell[16014, 584, 251, 11, 70, "ExampleText",
 CellID->1226827479],
Cell[CellGroupData[{
Cell[16290, 599, 147, 4, 70, "Input",
 CellID->1995105097],
Cell[16440, 605, 175, 6, 36, "Output",
 CellID->640827633]
}, Open  ]],
Cell[16630, 614, 206, 6, 70, "ExampleText",
 CellID->330449609],
Cell[CellGroupData[{
Cell[16861, 624, 298, 11, 70, "Input",
 CellID->63620644],
Cell[17162, 637, 161, 5, 36, "Output",
 CellID->759538126]
}, Open  ]],
Cell[CellGroupData[{
Cell[17360, 647, 183, 6, 70, "Input",
 CellID->92994129],
Cell[17546, 655, 161, 5, 36, "Output",
 CellID->914485914]
}, Open  ]],
Cell[17722, 663, 126, 3, 70, "ExampleDelimiter",
 CellID->2139988967],
Cell[17851, 668, 366, 12, 70, "ExampleText",
 CellID->240796388],
Cell[CellGroupData[{
Cell[18242, 684, 263, 8, 70, "Input",
 CellID->194696687],
Cell[18508, 694, 545, 20, 43, "Output",
 CellID->224295019]
}, Open  ]],
Cell[CellGroupData[{
Cell[19090, 719, 210, 6, 70, "Input",
 CellID->44918340],
Cell[19303, 727, 312, 11, 43, "Output",
 CellID->892583848]
}, Open  ]],
Cell[19630, 741, 150, 6, 70, "ExampleText",
 CellID->878254457],
Cell[CellGroupData[{
Cell[19805, 751, 263, 9, 70, "Input",
 CellID->381411379],
Cell[20071, 762, 1163, 47, 65, "Output",
 CellID->1483459041]
}, Open  ]],
Cell[CellGroupData[{
Cell[21271, 814, 122, 3, 70, "Input",
 CellID->1932490251],
Cell[21396, 819, 379, 12, 36, "Output",
 CellID->210960417]
}, Open  ]],
Cell[21790, 834, 151, 6, 70, "ExampleText",
 CellID->2031362948],
Cell[CellGroupData[{
Cell[21966, 844, 263, 9, 70, "Input",
 CellID->422047030],
Cell[22232, 855, 1153, 47, 65, "Output",
 CellID->116227322]
}, Open  ]],
Cell[CellGroupData[{
Cell[23422, 907, 122, 3, 70, "Input",
 CellID->1757914066],
Cell[23547, 912, 380, 12, 36, "Output",
 CellID->1369965731]
}, Open  ]],
Cell[23942, 927, 125, 3, 70, "ExampleDelimiter",
 CellID->603415282],
Cell[24070, 932, 323, 11, 70, "ExampleText",
 CellID->1182863440],
Cell[CellGroupData[{
Cell[24418, 947, 445, 15, 70, "Input",
 CellID->548236987],
Cell[24866, 964, 733, 23, 36, "Output",
 CellID->1841861430]
}, Open  ]],
Cell[25614, 990, 61, 1, 70, "ExampleText",
 CellID->1023782415],
Cell[CellGroupData[{
Cell[25700, 995, 150, 4, 70, "Input",
 CellID->1725342136],
Cell[25853, 1001, 493, 16, 36, "Output",
 CellID->188282505]
}, Open  ]],
Cell[CellGroupData[{
Cell[26383, 1022, 369, 11, 70, "Input",
 CellID->436987237],
Cell[26755, 1035, 733, 23, 36, "Output",
 CellID->1220649504]
}, Open  ]],
Cell[CellGroupData[{
Cell[27525, 1063, 165, 5, 70, "Input",
 CellID->244741605],
Cell[27693, 1070, 161, 5, 36, "Output",
 CellID->544933937]
}, Open  ]],
Cell[27869, 1078, 126, 3, 70, "ExampleDelimiter",
 CellID->1167552968],
Cell[27998, 1083, 303, 12, 70, "ExampleText",
 CellID->1504650537],
Cell[CellGroupData[{
Cell[28326, 1099, 261, 7, 70, "Input",
 CellID->2041631934],
Cell[28590, 1108, 235, 9, 38, "Output",
 CellID->1834408351]
}, Open  ]],
Cell[CellGroupData[{
Cell[28862, 1122, 334, 12, 70, "Input",
 CellID->755619488],
Cell[29199, 1136, 242, 9, 36, "Output",
 CellID->1625918925]
}, Open  ]],
Cell[CellGroupData[{
Cell[29478, 1150, 157, 6, 70, "Input",
 CellID->233332902],
Cell[29638, 1158, 157, 5, 36, "Output",
 CellID->89964400]
}, Open  ]],
Cell[29810, 1166, 126, 3, 70, "ExampleDelimiter",
 CellID->1520291085],
Cell[29939, 1171, 404, 14, 70, "ExampleText",
 CellID->1007153122],
Cell[CellGroupData[{
Cell[30368, 1189, 384, 13, 70, "Input",
 CellID->483123609],
Cell[30755, 1204, 350, 12, 39, "Output",
 CellID->203183312]
}, Open  ]],
Cell[CellGroupData[{
Cell[31142, 1221, 176, 5, 70, "Input",
 CellID->312289525],
Cell[31321, 1228, 724, 25, 51, "Output",
 CellID->402926054]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[32106, 1260, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[32425, 1271, 778, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[33240, 1302, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[33561, 1313, 159, 3, 70, "Tutorials",
 CellID->341631938]
}, Open  ]],
Cell[CellGroupData[{
Cell[33757, 1321, 319, 9, 70, "MoreAboutSection",
 CellID->522916940],
Cell[34079, 1332, 156, 3, 70, "MoreAbout",
 CellID->263031764],
Cell[34238, 1337, 198, 4, 70, "MoreAbout",
 CellID->627481082]
}, Open  ]],
Cell[34451, 1344, 27, 0, 70, "History"],
Cell[34481, 1346, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

