(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     41371,       1513]
NotebookOptionsPosition[     33232,       1232]
NotebookOutlinePosition[     34611,       1271]
CellTagsIndexPosition[     34525,       1266]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Vector Operations" :> 
          Documentation`HelpLookup["paclet:tutorial/VectorOperations"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Vector Operations\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Orthogonalize" :> 
          Documentation`HelpLookup["paclet:ref/Orthogonalize"], "Dot" :> 
          Documentation`HelpLookup["paclet:ref/Dot"], "Inner" :> 
          Documentation`HelpLookup["paclet:ref/Inner"], "Normalize" :> 
          Documentation`HelpLookup["paclet:ref/Normalize"], "VectorAngle" :> 
          Documentation`HelpLookup["paclet:ref/VectorAngle"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Orthogonalize\"\>", 
       2->"\<\"Dot\"\>", 3->"\<\"Inner\"\>", 4->"\<\"Normalize\"\>", 
       5->"\<\"VectorAngle\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Math & Counting Operations on Lists" :> 
          Documentation`HelpLookup[
           "paclet:guide/MathematicalAndCountingOperationsOnLists"], 
          "Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "Operations on Vectors" :> 
          Documentation`HelpLookup["paclet:guide/OperationsOnVectors"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Math & Counting Operations on Lists\"\>", 
       2->"\<\"Matrices and Linear Algebra\"\>", 
       3->"\<\"Operations on Vectors\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       5->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Projection", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Projection",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Projection"], "[", 
       RowBox[{
        StyleBox["u", "TI"], ",", 
        StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]finds the projection of the vector ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " onto the vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Projection",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Projection"], "[", 
       RowBox[{
        StyleBox["u", "TI"], ",", 
        StyleBox["v", "TI"], ",", 
        StyleBox["f", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds projections with respect to the inner product \
function ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "For ordinary vectors ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 ", the projection is taken to be ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox[
     RowBox[{
      OverscriptBox["u", "_"], ".", "v"}], 
     RowBox[{
      OverscriptBox["v", "_"], ".", "v"}]], "v"}], TraditionalForm]], 
  "InlineMath"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->287878945,
  ButtonNote->"287878945"]
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 "In ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Projection",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Projection"], "[", 
   RowBox[{
    StyleBox["u", "TI"], ",", 
    StyleBox["v", "TI"], ",", 
    StyleBox["f", "TI"]}], "]"}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 " can be any expressions or lists of expressions for which the inner product \
function ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " applied to pairs yields real results.",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->36518885,
  ButtonNote->"36518885"]
}], "Notes",
 CellID->898168979]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->641444757],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->215527230],

Cell[TextData[{
 "Project the vector (5, 6, 7) onto the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis:"
}], "ExampleText",
 CellID->797932462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"5", ",", "6", ",", "7"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->630307210],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"5", ",", "0", ",", "0"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->88711008]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->69810695],

Cell["Project onto another vector:", "ExampleText",
 CellID->69587241],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"5", ",", "6", ",", "7"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->319771646],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6", ",", "6", ",", "6"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->147376449]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Use symbolic vectors:", "ExampleText",
 CellID->169182784],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->267085409],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"a", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"a", " ", 
        RowBox[{"Conjugate", "[", "x", "]"}]}], "+", 
       RowBox[{"b", " ", 
        RowBox[{"Conjugate", "[", "y", "]"}]}]}], ")"}]}], 
    RowBox[{
     RowBox[{"a", " ", 
      RowBox[{"Conjugate", "[", "a", "]"}]}], "+", 
     RowBox[{"b", " ", 
      RowBox[{"Conjugate", "[", "b", "]"}]}]}]], ",", 
   FractionBox[
    RowBox[{"b", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"a", " ", 
        RowBox[{"Conjugate", "[", "x", "]"}]}], "+", 
       RowBox[{"b", " ", 
        RowBox[{"Conjugate", "[", "y", "]"}]}]}], ")"}]}], 
    RowBox[{
     RowBox[{"a", " ", 
      RowBox[{"Conjugate", "[", "a", "]"}]}], "+", 
     RowBox[{"b", " ", 
      RowBox[{"Conjugate", "[", "b", "]"}]}]}]]}], "}"}]], "Output",
 ImageSize->{487, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->47259877]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->97966957],

Cell["Assume all variables are real:", "ExampleText",
 CellID->292691989],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}], ",", "Dot"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->52789822],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"a", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"a", " ", "x"}], "+", 
       RowBox[{"b", " ", "y"}]}], ")"}]}], 
    RowBox[{
     SuperscriptBox["a", "2"], "+", 
     SuperscriptBox["b", "2"]}]], ",", 
   FractionBox[
    RowBox[{"b", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"a", " ", "x"}], "+", 
       RowBox[{"b", " ", "y"}]}], ")"}]}], 
    RowBox[{
     SuperscriptBox["a", "2"], "+", 
     SuperscriptBox["b", "2"]}]]}], "}"}]], "Output",
 ImageSize->{179, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->78050625]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->67985664],

Cell[TextData[{
 "Use exact arithmetic to find the projection of ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " onto ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->296240525],

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"1", "/", "2"}], ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "4"}], ",", 
     RowBox[{"1", "/", "5"}], ",", 
     RowBox[{"1", "/", "6"}], ",", 
     RowBox[{"1", "/", "7"}], ",", 
     RowBox[{"1", "/", "8"}], ",", 
     RowBox[{"1", "/", "9"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", "7", ",", "8",
      ",", "9"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->103256713],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{"u", ",", "v"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->789244481],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["3", "95"], ",", 
   FractionBox["6", "95"], ",", 
   FractionBox["9", "95"], ",", 
   FractionBox["12", "95"], ",", 
   FractionBox["3", "19"], ",", 
   FractionBox["18", "95"], ",", 
   FractionBox["21", "95"], ",", 
   FractionBox["24", "95"], ",", 
   FractionBox["27", "95"]}], "}"}]], "Output",
 ImageSize->{271, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->481536718]
}, Open  ]],

Cell["Use machine arithmetic:", "ExampleText",
 CellID->92629629],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"N", "[", "u", "]"}], ",", 
   RowBox[{"N", "[", "v", "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->388363577],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.031578947368421054`", ",", "0.06315789473684211`", ",", 
   "0.09473684210526316`", ",", "0.12631578947368421`", ",", 
   "0.15789473684210525`", ",", "0.18947368421052632`", ",", 
   "0.2210526315789474`", ",", "0.25263157894736843`", ",", 
   "0.28421052631578947`"}], "}"}]], "Output",
 ImageSize->{343, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->315487543]
}, Open  ]],

Cell["\<\
Use 20\[Hyphen]digit precision arithmetic:\
\>", "ExampleText",
 CellID->370934222],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{"u", ",", "20"}], "]"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{"v", ",", "20"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->113850259],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.03157894736842105263157894736833247877`19.30102999566398", ",", 
   "0.06315789473684210526315789473666495754`19.30102999566398", ",", 
   "0.09473684210526315789473684210499743631`19.30102999566398", ",", 
   "0.12631578947368421052631578947332991508`19.30102999566398", ",", 
   "0.15789473684210526315789473684166239385`19.30102999566398", ",", 
   "0.18947368421052631578947368420999487263`19.30102999566398", ",", 
   "0.2210526315789473684210526315783273514`19.30102999566398", ",", 
   "0.25263157894736842105263157894665983017`19.30102999566398", ",", 
   "0.28421052631578947368421052631499230894`19.30102999566398"}], 
  "}"}]], "Output",
 ImageSize->{503, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->474193471]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->206666799],

Cell["Projection of a complex vector onto another:", "ExampleText",
 CellID->111920534],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", "+", "I"}], ",", 
     RowBox[{"{", "4", "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "I", ",", 
     RowBox[{"-", "1"}], ",", "I"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->33671990],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.06893244220633193`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.23101349232888335`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.23101349232888335`"}], "+", 
    RowBox[{"0.06893244220633193`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.06893244220633193`"}], "-", 
    RowBox[{"0.23101349232888335`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.23101349232888335`"}], "+", 
    RowBox[{"0.06893244220633193`", " ", "\[ImaginaryI]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{330, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->360481]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->183831262],

Cell["Use a different inner product:", "ExampleText",
 CellID->36518885],

Cell[BoxData[
 RowBox[{
  RowBox[{"ip", "[", 
   RowBox[{"p1_", ",", "p2_"}], "]"}], ":=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"p1", " ", "p2"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->613995077],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Projection", "[", 
  RowBox[{
   RowBox[{"x", "^", "2"}], ",", 
   RowBox[{"LegendreP", "[", 
    RowBox[{"2", ",", "x"}], "]"}], ",", "ip"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->20266041],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "3"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    RowBox[{"3", " ", 
     SuperscriptBox["x", "2"]}]}], ")"}]}]], "Output",
 ImageSize->{82, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->24240001]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->369487751],

Cell["Find parallel and orthogonal components of a vector:", "ExampleText",
 CellID->105366430],

Cell[BoxData[{
 RowBox[{
  RowBox[{"u", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", "5", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", "5", "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->216138475],

Cell[BoxData[{
 RowBox[{
  RowBox[{"uparallel", "=", 
   RowBox[{"Projection", "[", 
    RowBox[{"u", ",", "v"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"uorthogonal", " ", "=", " ", 
   RowBox[{"u", " ", "-", " ", "uparallel"}]}], ";"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->728748908],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"uorthogonal", ".", "v"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->197607136],

Cell[BoxData["1.942890293094024`*^-16"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->221818832]
}, Open  ]],

Cell["u is the sum of the parallel and orthogonal components:", "ExampleText",
 CellID->480765910],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"u", "\[Equal]", 
  RowBox[{"uparallel", "+", "uorthogonal"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->20294537],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->316335907]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->114890593],

Cell[TextData[{
 "Unnormalized Gram\[Dash]Schmidt algorithm (use ",
 Cell[BoxData[
  ButtonBox["Orthogonalize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Orthogonalize"]], "InlineFormula"],
 " for a better implementation):"
}], "ExampleText",
 CellID->50206794],

Cell[BoxData[
 RowBox[{
  RowBox[{"gs", "[", 
   RowBox[{"vecs_", ",", "ip___"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"ovecs", "=", "vecs"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"ovecs", "[", 
         RowBox[{"[", "i", "]"}], "]"}], "-=", 
        RowBox[{"Projection", "[", 
         RowBox[{
          RowBox[{"ovecs", "[", 
           RowBox[{"[", "i", "]"}], "]"}], ",", 
          RowBox[{"ovecs", "[", 
           RowBox[{"[", "j", "]"}], "]"}], ",", "ip"}], "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "2", ",", 
         RowBox[{"Length", "[", "vecs", "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", 
         RowBox[{"i", "-", "1"}]}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", "ovecs"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7542679],

Cell["\<\
Do Gram\[Dash]Schmidt on a random set of 3 vectors:\
\>", "ExampleText",
 CellID->803365520],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", 
  RowBox[{"gs", "[", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "3"}], "}"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7826000],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.7261240646968354`", ",", "0.2590199141040861`", ",", 
     "0.7455492990271864`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.16863278892367006`"}], ",", 
     RowBox[{"-", "0.021795633487970964`"}], ",", "0.171811346902824`"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.12142414035493643`"}], ",", "0.5006292571563539`", ",", 
     RowBox[{"-", "0.05566896366738058`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{476, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->193559187]
}, Open  ]],

Cell["Verify orthogonality:", "ExampleText",
 CellID->120244920],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{"Outer", "[", 
   RowBox[{"Dot", ",", "b", ",", "b", ",", "1"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->42080091],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1.1501912305142712`", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0.058431206063879554`", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0.26847250849765814`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{360, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->313230950]
}, Open  ]],

Cell["Generate some orthogonal polynomials:", "ExampleText",
 CellID->984904983],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"gs", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "x", ",", 
     RowBox[{"x", "^", "2"}], ",", 
     RowBox[{"x", "^", "3"}]}], "}"}], ",", 
   RowBox[{"Function", "[", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"-", "1"}], "1"], 
     RowBox[{
      FractionBox[
       RowBox[{"Times", "[", "##", "]"}], 
       SqrtBox[
        RowBox[{"1", "-", 
         SuperscriptBox["x", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}]}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->371029249],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "x", ",", 
   RowBox[{
    RowBox[{"-", 
     FractionBox["1", "2"]}], "+", 
    SuperscriptBox["x", "2"]}], ",", 
   RowBox[{
    RowBox[{"-", 
     FractionBox[
      RowBox[{"3", " ", "x"}], "4"]}], "+", 
    SuperscriptBox["x", "3"]}]}], "}"}]], "Output",
 ImageSize->{164, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->790612322]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->930483733],

Cell[TextData[{
 "The projection of ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " onto ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 " is in the direction of ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->92534390],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"u", ",", "v"}], "}"}], "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "6"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->421616479],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"VectorAngle", "[", 
  RowBox[{
   RowBox[{"Projection", "[", 
    RowBox[{"u", ",", "v"}], "]"}], ",", "v"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->168193392],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->427225329]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->133213566],

Cell[TextData[{
 "For ordinary vectors ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 ", the projection is taken to be ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox[
     RowBox[{
      OverscriptBox["u", "_"], ".", "v"}], 
     RowBox[{
      OverscriptBox["v", "_"], ".", "v"}]], "v"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->287878945],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"u", ",", "v"}], "}"}], "=", 
   RowBox[{"RandomComplex", "[", 
    RowBox[{
     RowBox[{"1", "+", "I"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "4"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->387531351],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", " ", "=", " ", 
  RowBox[{
   FractionBox[
    RowBox[{
     RowBox[{"Conjugate", "[", "u", "]"}], ".", "v"}], 
    RowBox[{
     RowBox[{"Conjugate", "[", "v", "]"}], ".", "v"}]], "v"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2272006],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.1689343395346245`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.6896128221602754`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.22932992995079146`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.4282028212099592`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{
    RowBox[{"-", "0.26218208456378794`"}], "+", 
    RowBox[{"0.5919720528384405`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"0.10515601954736731`", "\[InvisibleSpace]", "+", 
    RowBox[{"0.19984249108822424`", " ", "\[ImaginaryI]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{309, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->453835259]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "\[Equal]", 
  RowBox[{"Projection", "[", 
   RowBox[{"u", ",", "v"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->160993646],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->85267049]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->58504913],

Cell[TextData[{
 "The projection of ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " onto ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 " is equivalent to multiplication by an outer product matrix:"
}], "ExampleText",
 CellID->461730486],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"u", ",", "v"}], "}"}], "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "100"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->482098463],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"v", ",", "v"}], "]"}], "/", 
    RowBox[{"v", ".", "v"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->651418463],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"m", ".", "u"}], "\[Equal]", 
  RowBox[{"Projection", "[", 
   RowBox[{"u", ",", "v"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->130083146],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->56383522]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Orthogonalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Orthogonalize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Dot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Dot"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Inner",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Inner"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Normalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Normalize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["VectorAngle",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/VectorAngle"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Vector Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/VectorOperations"]], "Tutorials",
 CellID->513408734]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->179341033],

Cell[TextData[ButtonBox["Math & Counting Operations on Lists",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/MathematicalAndCountingOperationsOnLists"]], "MoreAbout",
 CellID->8404222],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->413146238],

Cell[TextData[ButtonBox["Operations on Vectors",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/OperationsOnVectors"]], "MoreAbout",
 CellID->37694998],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->348007176],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->29615986]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Projection - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 14.2180110}", "context" -> "System`", 
    "keywords" -> {
     "project vector onto vector", "vector projection", 
      "vector space projection"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "Projection[u, v] finds the projection of the vector u onto the vector v. \
Projection[u, v, f] finds projections with respect to the inner product \
function f.", "synonyms" -> {}, "title" -> "Projection", "type" -> "Symbol", 
    "uri" -> "ref/Projection"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8222, 251, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->641444757]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 34382, 1259}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2960, 67, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3583, 94, 1575, 37, 70, "ObjectNameGrid"],
Cell[5161, 133, 1195, 39, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[6393, 177, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6871, 194, 570, 23, 70, "Notes",
 CellID->1067943069],
Cell[7444, 219, 741, 27, 70, "Notes",
 CellID->898168979]
}, Closed]],
Cell[CellGroupData[{
Cell[8222, 251, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->641444757],
Cell[CellGroupData[{
Cell[8607, 265, 148, 5, 70, "ExampleSection",
 CellID->215527230],
Cell[8758, 272, 164, 6, 70, "ExampleText",
 CellID->797932462],
Cell[CellGroupData[{
Cell[8947, 282, 247, 8, 28, "Input",
 CellID->630307210],
Cell[9197, 292, 212, 7, 36, "Output",
 CellID->88711008]
}, Open  ]],
Cell[9424, 302, 124, 3, 70, "ExampleDelimiter",
 CellID->69810695],
Cell[9551, 307, 70, 1, 70, "ExampleText",
 CellID->69587241],
Cell[CellGroupData[{
Cell[9646, 312, 247, 8, 70, "Input",
 CellID->319771646],
Cell[9896, 322, 213, 7, 36, "Output",
 CellID->147376449]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10158, 335, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[10382, 344, 64, 1, 70, "ExampleText",
 CellID->169182784],
Cell[CellGroupData[{
Cell[10471, 349, 227, 8, 70, "Input",
 CellID->267085409],
Cell[10701, 359, 997, 33, 53, "Output",
 CellID->47259877]
}, Open  ]],
Cell[11713, 395, 124, 3, 70, "ExampleDelimiter",
 CellID->97966957],
Cell[11840, 400, 73, 1, 70, "ExampleText",
 CellID->292691989],
Cell[CellGroupData[{
Cell[11938, 405, 238, 8, 70, "Input",
 CellID->52789822],
Cell[12179, 415, 673, 25, 52, "Output",
 CellID->78050625]
}, Open  ]],
Cell[12867, 443, 124, 3, 70, "ExampleDelimiter",
 CellID->67985664],
Cell[12994, 448, 236, 9, 70, "ExampleText",
 CellID->296240525],
Cell[13233, 459, 620, 20, 70, "Input",
 CellID->103256713],
Cell[CellGroupData[{
Cell[13878, 483, 129, 4, 70, "Input",
 CellID->789244481],
Cell[14010, 489, 486, 16, 51, "Output",
 CellID->481536718]
}, Open  ]],
Cell[14511, 508, 65, 1, 70, "ExampleText",
 CellID->92629629],
Cell[CellGroupData[{
Cell[14601, 513, 187, 6, 70, "Input",
 CellID->388363577],
Cell[14791, 521, 464, 12, 54, "Output",
 CellID->315487543]
}, Open  ]],
Cell[15270, 536, 93, 3, 70, "ExampleText",
 CellID->370934222],
Cell[CellGroupData[{
Cell[15388, 543, 239, 8, 70, "Input",
 CellID->113850259],
Cell[15630, 553, 824, 17, 71, "Output",
 CellID->474193471]
}, Open  ]],
Cell[16469, 573, 125, 3, 70, "ExampleDelimiter",
 CellID->206666799],
Cell[16597, 578, 87, 1, 70, "ExampleText",
 CellID->111920534],
Cell[CellGroupData[{
Cell[16709, 583, 336, 11, 70, "Input",
 CellID->33671990],
Cell[17048, 596, 701, 19, 54, "Output",
 CellID->360481]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17798, 621, 243, 7, 70, "ExampleSection",
 CellID->183831262],
Cell[18044, 630, 72, 1, 70, "ExampleText",
 CellID->36518885],
Cell[18119, 633, 320, 11, 70, "Input",
 CellID->613995077],
Cell[CellGroupData[{
Cell[18464, 648, 225, 7, 70, "Input",
 CellID->20266041],
Cell[18692, 657, 319, 12, 51, "Output",
 CellID->24240001]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19060, 675, 227, 7, 70, "ExampleSection",
 CellID->369487751],
Cell[19290, 684, 95, 1, 70, "ExampleText",
 CellID->105366430],
Cell[19388, 687, 346, 12, 70, "Input",
 CellID->216138475],
Cell[19737, 701, 296, 9, 70, "Input",
 CellID->728748908],
Cell[CellGroupData[{
Cell[20058, 714, 102, 3, 70, "Input",
 CellID->197607136],
Cell[20163, 719, 180, 5, 36, "Output",
 CellID->221818832]
}, Open  ]],
Cell[20358, 727, 98, 1, 70, "ExampleText",
 CellID->480765910],
Cell[CellGroupData[{
Cell[20481, 732, 139, 4, 70, "Input",
 CellID->20294537],
Cell[20623, 738, 161, 5, 36, "Output",
 CellID->316335907]
}, Open  ]],
Cell[20799, 746, 125, 3, 70, "ExampleDelimiter",
 CellID->114890593],
Cell[20927, 751, 268, 8, 70, "ExampleText",
 CellID->50206794],
Cell[21198, 761, 949, 28, 70, "Input",
 CellID->7542679],
Cell[22150, 791, 102, 3, 70, "ExampleText",
 CellID->803365520],
Cell[CellGroupData[{
Cell[22277, 798, 235, 8, 70, "Input",
 CellID->7826000],
Cell[22515, 808, 657, 20, 54, "Output",
 CellID->193559187]
}, Open  ]],
Cell[23187, 831, 64, 1, 70, "ExampleText",
 CellID->120244920],
Cell[CellGroupData[{
Cell[23276, 836, 177, 5, 70, "Input",
 CellID->42080091],
Cell[23456, 843, 451, 14, 36, "Output",
 CellID->313230950]
}, Open  ]],
Cell[23922, 860, 80, 1, 70, "ExampleText",
 CellID->984904983],
Cell[CellGroupData[{
Cell[24027, 865, 563, 19, 70, "Input",
 CellID->371029249],
Cell[24593, 886, 439, 16, 51, "Output",
 CellID->790612322]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25081, 908, 237, 7, 70, "ExampleSection",
 CellID->930483733],
Cell[25321, 917, 292, 12, 70, "ExampleText",
 CellID->92534390],
Cell[25616, 931, 275, 10, 70, "Input",
 CellID->421616479],
Cell[CellGroupData[{
Cell[25916, 945, 193, 6, 70, "Input",
 CellID->168193392],
Cell[26112, 953, 160, 5, 36, "Output",
 CellID->427225329]
}, Open  ]],
Cell[26287, 961, 125, 3, 70, "ExampleDelimiter",
 CellID->133213566],
Cell[26415, 966, 461, 19, 70, "ExampleText",
 CellID->287878945],
Cell[26879, 987, 304, 11, 70, "Input",
 CellID->387531351],
Cell[CellGroupData[{
Cell[27208, 1002, 268, 9, 70, "Input",
 CellID->2272006],
Cell[27479, 1013, 702, 17, 54, "Output",
 CellID->453835259]
}, Open  ]],
Cell[CellGroupData[{
Cell[28218, 1035, 160, 5, 70, "Input",
 CellID->160993646],
Cell[28381, 1042, 160, 5, 36, "Output",
 CellID->85267049]
}, Open  ]],
Cell[28556, 1050, 124, 3, 70, "ExampleDelimiter",
 CellID->58504913],
Cell[28683, 1055, 266, 9, 70, "ExampleText",
 CellID->461730486],
Cell[28952, 1066, 277, 10, 70, "Input",
 CellID->482098463],
Cell[29232, 1078, 230, 8, 70, "Input",
 CellID->651418463],
Cell[CellGroupData[{
Cell[29487, 1090, 183, 6, 70, "Input",
 CellID->130083146],
Cell[29673, 1098, 160, 5, 36, "Output",
 CellID->56383522]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29894, 1110, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[30213, 1121, 1192, 44, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[31442, 1170, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[31763, 1181, 148, 3, 70, "Tutorials",
 CellID->513408734]
}, Open  ]],
Cell[CellGroupData[{
Cell[31948, 1189, 319, 9, 70, "MoreAboutSection",
 CellID->179341033],
Cell[32270, 1200, 188, 4, 70, "MoreAbout",
 CellID->8404222],
Cell[32461, 1206, 163, 3, 70, "MoreAbout",
 CellID->413146238],
Cell[32627, 1211, 151, 3, 70, "MoreAbout",
 CellID->37694998],
Cell[32781, 1216, 179, 3, 70, "MoreAbout",
 CellID->348007176],
Cell[32963, 1221, 197, 4, 70, "MoreAbout",
 CellID->29615986]
}, Open  ]],
Cell[33175, 1228, 27, 0, 70, "History"],
Cell[33205, 1230, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

