(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     34628,       1282]
NotebookOptionsPosition[     27092,       1012]
NotebookOutlinePosition[     28461,       1051]
CellTagsIndexPosition[     28375,       1046]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DescriptiveStatistics"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Descriptive Statistics\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"InterquartileRange" :> 
          Documentation`HelpLookup["paclet:ref/InterquartileRange"], 
          "Quartiles" :> Documentation`HelpLookup["paclet:ref/Quartiles"], 
          "Quantile" :> Documentation`HelpLookup["paclet:ref/Quantile"], 
          "QuartileSkewness" :> 
          Documentation`HelpLookup["paclet:ref/QuartileSkewness"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"InterquartileRange\"\>", 
       2->"\<\"Quartiles\"\>", 3->"\<\"Quantile\"\>", 
       4->"\<\"QuartileSkewness\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["QuartileDeviation", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["QuartileDeviation",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/QuartileDeviation"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the quartile deviation, or semi-interquartile \
range of the elements in ",
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->10000]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["QuartileDeviation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QuartileDeviation"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["InterquartileRange",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/InterquartileRange"], "[", 
    StyleBox["list", "TI"], "]"}], "/", "2"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->14560],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileDeviation"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->11686772]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->148540606],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->386697678],

Cell["Quartile deviation for a list of exact numbers:", "ExampleText",
 CellID->31273586],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", "3", ",", "4", ",", "2", ",", "5", ",", "6"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21028],

Cell[BoxData[
 FractionBox["3", "2"]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->402520826]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27110],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileDeviation"]], "InlineFormula"],
 " works with any real numeric quantities:"
}], "ExampleText",
 CellID->13924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", "E", ",", "Pi", ",", 
    RowBox[{"Pi", "^", "2"}], ",", "1", ",", "2", ",", "3"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19854],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", 
  RowBox[{"(", 
   RowBox[{"3", "-", 
    SqrtBox["2"], "+", 
    RowBox[{
     FractionBox["1", "4"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "2"}], "+", 
       SqrtBox["2"]}], ")"}]}], "+", 
    RowBox[{
     FractionBox["3", "4"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "3"}], "+", "\[Pi]"}], ")"}]}]}], ")"}]}]], "Output",
 ImageSize->{244, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->82802447]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->139342064],

Cell[BoxData["0.7727671592062617`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->138592112]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->64612447],

Cell["Quartile deviations for columns of a matrix:", "ExampleText",
 CellID->281606070],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"3", ",", "5"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "8"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "6"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "8"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "4"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->421309609],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["15", "8"], ",", 
   FractionBox["13", "8"]}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{66, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61034302]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158391064],

Cell["Obtain results at any precision: ", "ExampleText",
 CellID->160553898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"5", ",", "10", ",", "4", ",", "25", ",", "2", ",", "1"}], "}"}],
     ",", "30"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190699768],

Cell[BoxData["4.`29.82390874094432"], "Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->42233366]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->650909219],

Cell["Compute results for a large vector or matrix:", "ExampleText",
 CellID->285474028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"10", "^", "5"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->315224713],

Cell[BoxData["0.25046638933340454`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->679252444]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"10", "^", "4"}], ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->594479966],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.250244979552245`", ",", "0.24965356296577462`", ",", 
   "0.24751392091835878`", ",", "0.25046321386802806`", ",", 
   "0.24618810761501464`"}], "}"}]], "Output",
 ImageSize->{343, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->554198698]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24417],

Cell[TextData[{
 "Compute results for a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->625271644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sp", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], ":>", "i"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "j_"}], "}"}], "/;", 
        RowBox[{"j", "<", "i"}]}], "\[RuleDelayed]", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"i", "+", "j"}], ")"}], "^", "2"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19878761],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "955", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 955, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2024905]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", "sp", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->235263230],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2575", ",", "2625", ",", "2675", ",", "2725", ",", "2775", ",", "2825", 
   ",", "2875", ",", "2925", ",", "2975", ",", "3025"}], "}"}]], "Output",
 ImageSize->{399, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->85453067]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11410],

Cell["\<\
Obtain a robust estimate of dispersion when extreme values are present:\
\>", "ExampleText",
 CellID->546359973],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"QuartileDeviation", "[", 
   RowBox[{"{", 
    RowBox[{"5", ",", "3", ",", 
     RowBox[{"10", "^", "6"}], ",", "20", ",", "15", ",", "6"}], "}"}], "]"}],
   "//", "N"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->120903845],

Cell[BoxData["7.5`"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->302166961]
}, Open  ]],

Cell[TextData[{
 "Measures based on the ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 " are heavily influenced by extreme values:"
}], "ExampleText",
 CellID->155964878],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"StandardDeviation", "[", 
   RowBox[{"{", 
    RowBox[{"5", ",", "3", ",", 
     RowBox[{"10", "^", "6"}], ",", "20", ",", "15", ",", "6"}], "}"}], "]"}],
   "//", "N"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->48975516],

Cell[BoxData["408244.289683232`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->749870]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MeanDeviation", "[", 
   RowBox[{"{", 
    RowBox[{"5", ",", "3", ",", 
     RowBox[{"10", "^", "6"}], ",", "20", ",", "15", ",", "6"}], "}"}], "]"}],
   "//", "N"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->932944264],

Cell[BoxData["277775.05555555556`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->272495740]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28463],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileDeviation"]], "InlineFormula"],
 " is half the difference of linearly interpolated ",
 Cell[BoxData[
  ButtonBox["Quantile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quantile"]], "InlineFormula"],
 " values:"
}], "ExampleText",
 CellID->38722882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->133352675],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "6.56508160400551`", ",", "7.220444468286491`", ",", "1.5461838757418178`", 
   ",", "2.174700683637565`", ",", "4.03449773361209`", ",", 
   "0.9731630491201493`", ",", "0.9095265520295737`", ",", 
   "5.816479151713565`", ",", "5.378184524939014`", ",", "8.80374322589332`", 
   ",", "8.771919156292308`", ",", "0.8267822662809543`", ",", 
   "4.582958253856213`", ",", "2.1321177397497815`", ",", 
   "4.719631237486186`", ",", "5.37493482908835`", ",", "4.077385817047208`", 
   ",", "8.11224924179935`", ",", "2.0827916579761885`", ",", 
   "2.9530240315785705`"}], "}"}]], "Output",
 ImageSize->{443, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2119174]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", "data", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->632052870],

Cell[BoxData["2.0416628394982763`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->575212638]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Apply", "[", 
   RowBox[{"Subtract", ",", 
    RowBox[{"Quantile", "[", 
     RowBox[{"data", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"3", "/", "4"}], ",", 
        RowBox[{"1", "/", "4"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", "/", "2"}], ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], "]"}]}], "]"}], "/", 
  "2"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->155558425],

Cell[BoxData["2.0416628394982763`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->448598308]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->103804862],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["InterquartileRange",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterquartileRange"]], "InlineFormula"],
 " is twice ",
 Cell[BoxData[
  ButtonBox["QuartileDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileDeviation"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->15299726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->118537781],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "3.743998815707028`", ",", "2.302764299818081`", ",", "3.522070982148382`", 
   ",", "2.78318630961637`", ",", "6.993134329632609`", ",", 
   "7.319157682866311`", ",", "8.779470373363354`", ",", "7.939850574622216`",
    ",", "6.601641527521085`", ",", "1.741289937064392`", ",", 
   "9.365253313361954`", ",", "0.2940708611589682`", ",", 
   "1.1677022951746419`", ",", "4.194083624917598`", ",", 
   "0.6885471202220961`", ",", "4.707900471436373`", ",", 
   "9.953262265180216`", ",", "7.811032421205515`", ",", "5.415602447344718`",
    ",", "1.6365124406193643`"}], "}"}]], "Output",
 ImageSize->{429, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->484664625]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"2", 
  RowBox[{"QuartileDeviation", "[", "data", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->481300380],

Cell[BoxData["5.543067933594676`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->76556608]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterquartileRange", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->23051440],

Cell[BoxData["5.543067933594676`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->24969510]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->631682428],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileDeviation"]], "InlineFormula"],
 " is half the difference between the first and third quartiles:"
}], "ExampleText",
 CellID->533304287],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->239649738],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.426086875417385`", ",", "5.722129469302695`", ",", "9.020222385547774`", 
   ",", "6.254840348712365`", ",", "2.3813086537953376`", ",", 
   "7.9913103491165955`", ",", "4.754391043683903`", ",", 
   "1.660693573597265`", ",", "3.370314653049624`", ",", 
   "2.3508796432772505`", ",", "2.6225695387973036`", ",", 
   "9.345071814575455`", ",", "2.0122505147982594`", ",", 
   "8.172930716737126`", ",", "8.943963244040658`", ",", 
   "1.7453525423662009`", ",", "3.6617135410005575`", ",", 
   "8.29919041242978`", ",", "5.876917723095887`", ",", 
   "7.797819091015001`"}], "}"}]], "Output",
 ImageSize->{429, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8121273]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"2", 
  RowBox[{"QuartileDeviation", "[", "data", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->333901997],

Cell[BoxData["5.716026384390568`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->105374881]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Apply", "[", 
  RowBox[{"Subtract", ",", 
   RowBox[{
    RowBox[{"Quartiles", "[", "data", "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}], "]"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->501255430],

Cell[BoxData["5.716026384390568`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->59264054]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32048],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileDeviation"]], "InlineFormula"],
 " requires numeric values:"
}], "ExampleText",
 CellID->9090],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9034],

Cell[BoxData[
 RowBox[{
  RowBox[{"QuartileDeviation", "::", "\<\"rectn\"\>"}], ":", 
  " ", "\<\"Rectangular array of real numbers is expected at position \\!\\(1\
\\) in \\!\\(QuartileDeviation[\\(\\({a, b, c}\\)\\)]\\).\"\>"}]], "Message", \
"MSG",
 CellID->594579633],

Cell[BoxData[
 RowBox[{"QuartileDeviation", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Output",
 ImageSize->{195, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->539239422]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->7665],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["InterquartileRange",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InterquartileRange"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Quartiles",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quartiles"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Quantile",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quantile"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QuartileSkewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QuartileSkewness"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->6833]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->1364],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DescriptiveStatistics"]], "Tutorials",
 CellID->4063544]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->196002941],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->54421017]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"QuartileDeviation - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 30.1402936}", "context" -> "System`", 
    "keywords" -> {
     "hinges", "lower quartile", "semi-interquartile range", 
      "upper quartile"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "QuartileDeviation[list] gives the quartile deviation, or \
semi-interquartile range of the elements in list.", 
    "synonyms" -> {"quartile deviation"}, "title" -> "QuartileDeviation", 
    "type" -> "Symbol", "uri" -> "ref/QuartileDeviation"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6480, 197, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->148540606]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 28232, 1039}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2273, 54, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2896, 81, 1582, 37, 70, "ObjectNameGrid"],
Cell[4481, 120, 620, 19, 70, "Usage",
 CellID->10000]
}, Open  ]],
Cell[CellGroupData[{
Cell[5138, 144, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5616, 161, 484, 17, 70, "Notes",
 CellID->14560],
Cell[6103, 180, 340, 12, 70, "Notes",
 CellID->11686772]
}, Closed]],
Cell[CellGroupData[{
Cell[6480, 197, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->148540606],
Cell[CellGroupData[{
Cell[6865, 211, 148, 5, 70, "ExampleSection",
 CellID->386697678],
Cell[7016, 218, 89, 1, 70, "ExampleText",
 CellID->31273586],
Cell[CellGroupData[{
Cell[7130, 223, 199, 6, 28, "Input",
 CellID->21028],
Cell[7332, 231, 226, 8, 51, "Output",
 CellID->402520826]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[7607, 245, 216, 7, 70, "ExampleSection",
 CellID->27110],
Cell[7826, 254, 231, 7, 70, "ExampleText",
 CellID->13924],
Cell[CellGroupData[{
Cell[8082, 265, 269, 8, 70, "Input",
 CellID->19854],
Cell[8354, 275, 553, 21, 51, "Output",
 CellID->82802447]
}, Open  ]],
Cell[CellGroupData[{
Cell[8944, 301, 97, 3, 70, "Input",
 CellID->139342064],
Cell[9044, 306, 176, 5, 36, "Output",
 CellID->138592112]
}, Open  ]],
Cell[9235, 314, 124, 3, 70, "ExampleDelimiter",
 CellID->64612447],
Cell[9362, 319, 87, 1, 70, "ExampleText",
 CellID->281606070],
Cell[CellGroupData[{
Cell[9474, 324, 445, 15, 70, "Input",
 CellID->421309609],
Cell[9922, 341, 296, 11, 51, "Output",
 CellID->61034302]
}, Open  ]],
Cell[10233, 355, 125, 3, 70, "ExampleDelimiter",
 CellID->158391064],
Cell[10361, 360, 76, 1, 70, "ExampleText",
 CellID->160553898],
Cell[CellGroupData[{
Cell[10462, 365, 264, 8, 70, "Input",
 CellID->190699768],
Cell[10729, 375, 177, 5, 36, "Output",
 CellID->42233366]
}, Open  ]],
Cell[10921, 383, 125, 3, 70, "ExampleDelimiter",
 CellID->650909219],
Cell[11049, 388, 88, 1, 70, "ExampleText",
 CellID->285474028],
Cell[CellGroupData[{
Cell[11162, 393, 200, 6, 70, "Input",
 CellID->315224713],
Cell[11365, 401, 177, 5, 36, "Output",
 CellID->679252444]
}, Open  ]],
Cell[CellGroupData[{
Cell[11579, 411, 253, 8, 70, "Input",
 CellID->594479966],
Cell[11835, 421, 338, 10, 36, "Output",
 CellID->554198698]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12222, 437, 239, 7, 70, "ExampleSection",
 CellID->24417],
Cell[12464, 446, 211, 8, 70, "ExampleText",
 CellID->625271644],
Cell[CellGroupData[{
Cell[12700, 458, 592, 20, 70, "Input",
 CellID->19878761],
Cell[13295, 480, 484, 17, 36, "Output",
 CellID->2024905]
}, Open  ]],
Cell[CellGroupData[{
Cell[13816, 502, 114, 3, 70, "Input",
 CellID->235263230],
Cell[13933, 507, 320, 9, 36, "Output",
 CellID->85453067]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14302, 522, 223, 7, 70, "ExampleSection",
 CellID->11410],
Cell[14528, 531, 122, 3, 70, "ExampleText",
 CellID->546359973],
Cell[CellGroupData[{
Cell[14675, 538, 258, 8, 70, "Input",
 CellID->120903845],
Cell[14936, 548, 161, 5, 36, "Output",
 CellID->302166961]
}, Open  ]],
Cell[15112, 556, 238, 8, 70, "ExampleText",
 CellID->155964878],
Cell[CellGroupData[{
Cell[15375, 568, 257, 8, 70, "Input",
 CellID->48975516],
Cell[15635, 578, 171, 5, 36, "Output",
 CellID->749870]
}, Open  ]],
Cell[CellGroupData[{
Cell[15843, 588, 254, 8, 70, "Input",
 CellID->932944264],
Cell[16100, 598, 176, 5, 36, "Output",
 CellID->272495740]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16325, 609, 233, 7, 70, "ExampleSection",
 CellID->28463],
Cell[16561, 618, 375, 12, 70, "ExampleText",
 CellID->38722882],
Cell[CellGroupData[{
Cell[16961, 634, 158, 5, 70, "Input",
 CellID->133352675],
Cell[17122, 641, 759, 16, 71, "Output",
 CellID->2119174]
}, Open  ]],
Cell[CellGroupData[{
Cell[17918, 662, 116, 3, 70, "Input",
 CellID->632052870],
Cell[18037, 667, 176, 5, 36, "Output",
 CellID->575212638]
}, Open  ]],
Cell[CellGroupData[{
Cell[18250, 677, 551, 19, 70, "Input",
 CellID->155558425],
Cell[18804, 698, 176, 5, 36, "Output",
 CellID->448598308]
}, Open  ]],
Cell[18995, 706, 125, 3, 70, "ExampleDelimiter",
 CellID->103804862],
Cell[19123, 711, 349, 12, 70, "ExampleText",
 CellID->15299726],
Cell[CellGroupData[{
Cell[19497, 727, 158, 5, 70, "Input",
 CellID->118537781],
Cell[19658, 734, 762, 16, 71, "Output",
 CellID->484664625]
}, Open  ]],
Cell[CellGroupData[{
Cell[20457, 755, 134, 4, 70, "Input",
 CellID->481300380],
Cell[20594, 761, 174, 5, 36, "Output",
 CellID->76556608]
}, Open  ]],
Cell[CellGroupData[{
Cell[20805, 771, 116, 3, 70, "Input",
 CellID->23051440],
Cell[20924, 776, 174, 5, 36, "Output",
 CellID->24969510]
}, Open  ]],
Cell[21113, 784, 125, 3, 70, "ExampleDelimiter",
 CellID->631682428],
Cell[21241, 789, 257, 7, 70, "ExampleText",
 CellID->533304287],
Cell[CellGroupData[{
Cell[21523, 800, 158, 5, 70, "Input",
 CellID->239649738],
Cell[21684, 807, 767, 17, 71, "Output",
 CellID->8121273]
}, Open  ]],
Cell[CellGroupData[{
Cell[22488, 829, 134, 4, 70, "Input",
 CellID->333901997],
Cell[22625, 835, 175, 5, 36, "Output",
 CellID->105374881]
}, Open  ]],
Cell[CellGroupData[{
Cell[22837, 845, 279, 9, 70, "Input",
 CellID->501255430],
Cell[23119, 856, 174, 5, 36, "Output",
 CellID->59264054]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[23342, 867, 226, 7, 70, "ExampleSection",
 CellID->32048],
Cell[23571, 876, 215, 7, 70, "ExampleText",
 CellID->9090],
Cell[CellGroupData[{
Cell[23811, 887, 165, 5, 70, "Input",
 CellID->9034],
Cell[23979, 894, 271, 6, 70, "Message",
 CellID->594579633],
Cell[24253, 902, 259, 8, 36, "Output",
 CellID->539239422]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[24573, 917, 310, 9, 70, "SeeAlsoSection",
 CellID->7665],
Cell[24886, 928, 978, 35, 70, "SeeAlso",
 CellID->6833]
}, Open  ]],
Cell[CellGroupData[{
Cell[25901, 968, 313, 9, 70, "TutorialsSection",
 CellID->1364],
Cell[26217, 979, 156, 3, 70, "Tutorials",
 CellID->4063544]
}, Open  ]],
Cell[CellGroupData[{
Cell[26410, 987, 299, 8, 70, "MoreAboutSection"],
Cell[26712, 997, 155, 3, 70, "MoreAbout",
 CellID->196002941],
Cell[26870, 1002, 150, 3, 70, "MoreAbout",
 CellID->54421017]
}, Open  ]],
Cell[27035, 1008, 27, 0, 70, "History"],
Cell[27065, 1010, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

