(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     38276,       1426]
NotebookOptionsPosition[     30322,       1140]
NotebookOutlinePosition[     31677,       1179]
CellTagsIndexPosition[     31591,       1174]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DescriptiveStatistics"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Descriptive Statistics\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Quartiles" :> Documentation`HelpLookup["paclet:ref/Quartiles"],
           "InterquartileRange" :> 
          Documentation`HelpLookup["paclet:ref/InterquartileRange"], 
          "QuartileDeviation" :> 
          Documentation`HelpLookup["paclet:ref/QuartileDeviation"], 
          "Quantile" :> Documentation`HelpLookup["paclet:ref/Quantile"], 
          "Skewness" :> Documentation`HelpLookup["paclet:ref/Skewness"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Quartiles\"\>", 
       2->"\<\"InterquartileRange\"\>", 3->"\<\"QuartileDeviation\"\>", 
       4->"\<\"Quantile\"\>", 5->"\<\"Skewness\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["QuartileSkewness", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["QuartileSkewness",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/QuartileSkewness"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the coefficient of quartile skewness for the \
elements in ",
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->30837]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "With ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SubscriptBox[
      StyleBox["q", "TI"], 
      StyleBox["1", "TR"]], ",", 
     SubscriptBox[
      StyleBox["q", "TI"], 
      StyleBox["2", "TR"]], ",", 
     SubscriptBox[
      StyleBox["q", "TI"], 
      StyleBox["3", "TR"]]}], "}"}], "=", 
   RowBox[{
    ButtonBox["Quartiles",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Quartiles"], "[", 
    StyleBox["list", "TI"], "]"}]}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["QuartileSkewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QuartileSkewness"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["q", "1"], "-", 
      RowBox[{"2", " ", 
       SubscriptBox["q", "2"]}], "+", 
      SubscriptBox["q", "3"]}], ")"}], "/", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["q", "3"], "-", 
      SubscriptBox["q", "1"]}], ")"}]}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->25896],

Cell[TextData[{
 "A positive value of quartile skewness indicates the median is closer to the \
\nlower quartile ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["q", "1"], TraditionalForm]], "InlineMath"],
 " than the upper quartile ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["q", "3"], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->27427],

Cell["\<\
A negative value of quartile skewness indicates the median is closer to the 
upper quartile.\
\>", "Notes",
 CellID->626699334],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileSkewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileSkewness"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->11686772]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->230053238],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->51477163],

Cell["Quartile skewness for a list of exact numbers:", "ExampleText",
 CellID->31273586],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", "3", ",", "3", ",", "2", ",", "5", ",", "6"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30565],

Cell[BoxData[
 FractionBox["1", "3"]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->243150461]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->3083],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileSkewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileSkewness"]], "InlineFormula"],
 " works with any real numeric quantities:"
}], "ExampleText",
 CellID->13924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", "E", ",", "Pi", ",", 
    RowBox[{"Pi", "^", "2"}], ",", "1", ",", "2", ",", "3"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19854],

Cell[BoxData[
 FractionBox[
  RowBox[{"3", "+", 
   SqrtBox["2"], "+", 
   RowBox[{
    FractionBox["1", "4"], " ", 
    RowBox[{"(", 
     RowBox[{"2", "-", 
      SqrtBox["2"]}], ")"}]}], "-", 
   RowBox[{"2", " ", "\[ExponentialE]"}], "+", 
   RowBox[{
    FractionBox["3", "4"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "3"}], "+", "\[Pi]"}], ")"}]}]}], 
  RowBox[{"3", "-", 
   SqrtBox["2"], "+", 
   RowBox[{
    FractionBox["1", "4"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "2"}], "+", 
      SqrtBox["2"]}], ")"}]}], "+", 
   RowBox[{
    FractionBox["3", "4"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "3"}], "+", "\[Pi]"}], ")"}]}]}]]], "Output",
 ImageSize->{240, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->35588803]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->139342064],

Cell[BoxData[
 RowBox[{"-", "0.49802128996819783`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->277531936]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->64612447],

Cell["Quartile skewness values for columns of a matrix:", "ExampleText",
 CellID->281606070],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"3", ",", "5"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "8"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"5", ",", "6"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "8"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "4"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->421309609],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", "3"], ",", 
   FractionBox["3", "13"]}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{59, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->486838780]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158391064],

Cell["Obtain results at any precision: ", "ExampleText",
 CellID->160553898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"5", ",", "10", ",", "4", ",", "25", ",", "2", ",", "1"}], "}"}],
     ",", "30"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190699768],

Cell[BoxData["0.375`28.952076447682817"], "Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->7655473]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->650909219],

Cell["Compute results for a large vector or matrix:", "ExampleText",
 CellID->285474028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"10", "^", "5"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->315224713],

Cell[BoxData["0.0016346392517131042`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->142256365]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"10", "^", "4"}], ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->594479966],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.019296285350157757`"}], ",", 
   RowBox[{"-", "0.007931711571173278`"}], ",", "0.0063557219345381655`", ",",
    "0.013298516177523537`", ",", 
   RowBox[{"-", "0.007498081717897673`"}]}], "}"}]], "Output",
 ImageSize->{422, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->472208766]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29719],

Cell[TextData[{
 "Compute results for a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->625271644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sp", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], ":>", "i"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "j_"}], "}"}], "/;", 
        RowBox[{"j", "<", "i"}]}], "\[RuleDelayed]", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"i", "+", "j"}], ")"}], "^", "2"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19878761],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "955", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 955, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->82269802]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", "sp", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->235263230],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["25", "103"], ",", 
   FractionBox["5", "21"], ",", 
   FractionBox["25", "107"], ",", 
   FractionBox["25", "109"], ",", 
   FractionBox["25", "111"], ",", 
   FractionBox["25", "113"], ",", 
   FractionBox["5", "23"], ",", 
   FractionBox["25", "117"], ",", 
   FractionBox["25", "119"], ",", 
   FractionBox["25", "121"]}], "}"}]], "Output",
 ImageSize->{356, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->25123916]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12238],

Cell["\<\
Obtain a robust estimate of asymmetry when extreme values are present:\
\>", "ExampleText",
 CellID->546359973],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"QuartileSkewness", "[", 
   RowBox[{"{", 
    RowBox[{"5", ",", "3", ",", 
     RowBox[{"10", "^", "6"}], ",", "2", ",", "1", ",", "4", ",", "8"}], 
    "}"}], "]"}], "//", "N"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->120903845],

Cell[BoxData["0.3`"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63280599]
}, Open  ]],

Cell[TextData[{
 "Measures based on the ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 " are heavily influenced by extreme values:"
}], "ExampleText",
 CellID->155964878],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Skewness", "[", 
   RowBox[{"{", 
    RowBox[{"5", ",", "3", ",", 
     RowBox[{"10", "^", "6"}], ",", "2", ",", "1", ",", "4", ",", "8"}], 
    "}"}], "]"}], "//", "N"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->48975516],

Cell[BoxData["2.0412414521651154`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1577004]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5241],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileSkewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileSkewness"]], "InlineFormula"],
 " is a function of linearly interpolated ",
 Cell[BoxData[
  ButtonBox["Quantile",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Quantile"]], "InlineFormula"],
 " values:"
}], "ExampleText",
 CellID->38722882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->133352675],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.7776354174457967`", ",", "0.35744039765118085`", ",", 
   "6.171392060714885`", ",", "7.479793777062069`", ",", "0.991573279635698`",
    ",", "2.6874328087720323`", ",", "6.279454898040111`", ",", 
   "0.9274631779575593`", ",", "8.187166961447435`", ",", 
   "1.4247371014797428`", ",", "3.9660754829082467`", ",", 
   "7.14022847273845`", ",", "1.4662608575224545`", ",", "6.451286247634636`",
    ",", "1.4654507123602545`", ",", "1.1125492560450057`", ",", 
   "9.318696398135266`", ",", "7.967009533830787`", ",", "9.071561746949381`",
    ",", "0.32083368144913305`"}], "}"}]], "Output",
 ImageSize->{436, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->224083965]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", "data", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->632052870],

Cell[BoxData["0.2730229771868203`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->114544416]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"q1", ",", "q2", ",", "q3"}], "}"}], "=", 
  RowBox[{"Quantile", "[", 
   RowBox[{"data", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"1", "/", "4"}], ",", 
      RowBox[{"1", "/", "2"}], ",", 
      RowBox[{"3", "/", "4"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"1", "/", "2"}], ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->155558425],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.0520612678403518`", ",", "3.3267541458401393`", ",", 
   "7.310011124900259`"}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->8123272]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"q1", "-", 
    RowBox[{"2", "q2"}], "+", "q3"}], ")"}], "/", 
  RowBox[{"(", 
   RowBox[{"q3", "-", "q1"}], ")"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->83068852],

Cell[BoxData["0.2730229771868203`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->14533881]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->249638483],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileSkewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileSkewness"]], "InlineFormula"],
 " is a function of quartiles:"
}], "ExampleText",
 CellID->686687598],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->782914789],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.7778852831963645`", ",", "7.772670437659473`", ",", "8.825633995916027`",
    ",", "1.3624643811975568`", ",", "6.268320353051829`", ",", 
   "1.2327506435665803`", ",", "5.081696889967753`", ",", 
   "9.550831673630562`", ",", "1.0673292840852966`", ",", 
   "8.957895203039264`", ",", "1.1611011751816136`", ",", 
   "2.7185046093822907`", ",", "8.207602082585034`", ",", 
   "6.007017132068942`", ",", "7.7422319750077095`", ",", 
   "8.635984957060538`", ",", "2.338653869675138`", ",", 
   "0.9975870253288011`", ",", "3.997020819663697`", ",", 
   "2.4067234526156422`"}], "}"}]], "Output",
 ImageSize->{436, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->148764698]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", "data", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->42522034],

Cell[BoxData["0.031232747852364802`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->115320658]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"q1", ",", "q2", ",", "q3"}], "}"}], "=", 
  RowBox[{"Quartiles", "[", "data", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->52908346],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.2976075123820685`", ",", "4.539358854815725`", ",", 
   "7.990136260122253`"}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->106671745]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"q1", "-", 
    RowBox[{"2", "q2"}], "+", "q3"}], ")"}], "/", 
  RowBox[{"(", 
   RowBox[{"q3", "-", "q1"}], ")"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->3941109],

Cell[BoxData["0.031232747852364802`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->84856059]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->103804862],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileSkewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileSkewness"]], "InlineFormula"],
 " is a function of the median, first quartile and a dispersion measure:"
}], "ExampleText",
 CellID->15299726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->118537781],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "4.512601687514419`", ",", "8.796670696669104`", ",", "8.243485850578828`", 
   ",", "6.671347038338622`", ",", "6.967514371657765`", ",", 
   "6.107219846202585`", ",", "4.587892015490032`", ",", "8.397522289936292`",
    ",", "4.619426480835241`", ",", "9.532045518950643`", ",", 
   "5.2672860127816445`", ",", "8.324891194540955`", ",", 
   "2.263324083752254`", ",", "0.10882504830358908`", ",", 
   "6.1027558794118075`", ",", "4.734380492335153`", ",", 
   "6.114707125854464`", ",", "6.368913530744398`", ",", 
   "3.8061352743194954`", ",", "5.267053050083357`"}], "}"}]], "Output",
 ImageSize->{436, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->359571928]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", "data", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->790664229],

Cell[BoxData[
 RowBox[{"-", "0.0002719552337145692`"}]], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->263227845]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"2", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Median", "[", "data", "]"}], "-", 
      RowBox[{"First", "[", 
       RowBox[{"Quartiles", "[", "data", "]"}], "]"}]}], ")"}], "/", 
    RowBox[{"InterquartileRange", "[", "data", "]"}]}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->266056878],

Cell[BoxData[
 RowBox[{"-", "0.0002719552337144826`"}]], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->65506036]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Median", "[", "data", "]"}], "-", 
     RowBox[{"First", "[", 
      RowBox[{"Quartiles", "[", "data", "]"}], "]"}]}], ")"}], "/", 
   RowBox[{"QuartileDeviation", "[", "data", "]"}]}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->714388045],

Cell[BoxData[
 RowBox[{"-", "0.0002719552337144826`"}]], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->836691162]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->20803],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["QuartileSkewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/QuartileSkewness"]], "InlineFormula"],
 " requires numeric values:"
}], "ExampleText",
 CellID->9090],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9034],

Cell[BoxData[
 RowBox[{
  RowBox[{"QuartileSkewness", "::", "\<\"rectn\"\>"}], ":", 
  " ", "\<\"Rectangular array of real numbers is expected at position \\!\\(1\
\\) in \\!\\(QuartileSkewness[\\(\\({a, b, c}\\)\\)]\\).\"\>"}]], "Message", \
"MSG",
 CellID->386262796],

Cell[BoxData[
 RowBox[{"QuartileSkewness", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->312897685]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->7701],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Quartiles",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quartiles"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["InterquartileRange",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InterquartileRange"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QuartileDeviation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QuartileDeviation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Quantile",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quantile"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Skewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Skewness"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->18497]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->5859],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DescriptiveStatistics"]], "Tutorials",
 CellID->4063544]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->40848186],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->49306022]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"QuartileSkewness - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 30.7184334}", "context" -> "System`", 
    "keywords" -> {
     "asymmetry", "hinges", "lower quartile", "skew", "skewness", 
      "upper quartile"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "QuartileSkewness[list] gives the coefficient of quartile skewness for \
the elements in list.", "synonyms" -> {"quartile skewness"}, "title" -> 
    "QuartileSkewness", "type" -> "Symbol", "uri" -> 
    "ref/QuartileSkewness"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7686, 244, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->230053238]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 31448, 1167}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2374, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2997, 82, 1581, 37, 70, "ObjectNameGrid"],
Cell[4581, 121, 604, 19, 70, "Usage",
 CellID->30837]
}, Open  ]],
Cell[CellGroupData[{
Cell[5222, 145, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5700, 162, 1111, 43, 70, "Notes",
 CellID->25896],
Cell[6814, 207, 354, 12, 70, "Notes",
 CellID->27427],
Cell[7171, 221, 137, 4, 70, "Notes",
 CellID->626699334],
Cell[7311, 227, 338, 12, 70, "Notes",
 CellID->11686772]
}, Closed]],
Cell[CellGroupData[{
Cell[7686, 244, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->230053238],
Cell[CellGroupData[{
Cell[8071, 258, 147, 5, 70, "ExampleSection",
 CellID->51477163],
Cell[8221, 265, 88, 1, 70, "ExampleText",
 CellID->31273586],
Cell[CellGroupData[{
Cell[8334, 270, 198, 6, 28, "Input",
 CellID->30565],
Cell[8535, 278, 226, 8, 51, "Output",
 CellID->243150461]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8810, 292, 215, 7, 70, "ExampleSection",
 CellID->3083],
Cell[9028, 301, 229, 7, 70, "ExampleText",
 CellID->13924],
Cell[CellGroupData[{
Cell[9282, 312, 268, 8, 70, "Input",
 CellID->19854],
Cell[9553, 322, 836, 32, 72, "Output",
 CellID->35588803]
}, Open  ]],
Cell[CellGroupData[{
Cell[10426, 359, 97, 3, 70, "Input",
 CellID->139342064],
Cell[10526, 364, 194, 6, 36, "Output",
 CellID->277531936]
}, Open  ]],
Cell[10735, 373, 124, 3, 70, "ExampleDelimiter",
 CellID->64612447],
Cell[10862, 378, 92, 1, 70, "ExampleText",
 CellID->281606070],
Cell[CellGroupData[{
Cell[10979, 383, 444, 15, 70, "Input",
 CellID->421309609],
Cell[11426, 400, 296, 11, 51, "Output",
 CellID->486838780]
}, Open  ]],
Cell[11737, 414, 125, 3, 70, "ExampleDelimiter",
 CellID->158391064],
Cell[11865, 419, 76, 1, 70, "ExampleText",
 CellID->160553898],
Cell[CellGroupData[{
Cell[11966, 424, 263, 8, 70, "Input",
 CellID->190699768],
Cell[12232, 434, 180, 5, 36, "Output",
 CellID->7655473]
}, Open  ]],
Cell[12427, 442, 125, 3, 70, "ExampleDelimiter",
 CellID->650909219],
Cell[12555, 447, 88, 1, 70, "ExampleText",
 CellID->285474028],
Cell[CellGroupData[{
Cell[12668, 452, 199, 6, 70, "Input",
 CellID->315224713],
Cell[12870, 460, 179, 5, 36, "Output",
 CellID->142256365]
}, Open  ]],
Cell[CellGroupData[{
Cell[13086, 470, 252, 8, 70, "Input",
 CellID->594479966],
Cell[13341, 480, 396, 11, 36, "Output",
 CellID->472208766]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13786, 497, 239, 7, 70, "ExampleSection",
 CellID->29719],
Cell[14028, 506, 211, 8, 70, "ExampleText",
 CellID->625271644],
Cell[CellGroupData[{
Cell[14264, 518, 592, 20, 70, "Input",
 CellID->19878761],
Cell[14859, 540, 485, 17, 36, "Output",
 CellID->82269802]
}, Open  ]],
Cell[CellGroupData[{
Cell[15381, 562, 113, 3, 70, "Input",
 CellID->235263230],
Cell[15497, 567, 529, 17, 51, "Output",
 CellID->25123916]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16075, 590, 223, 7, 70, "ExampleSection",
 CellID->12238],
Cell[16301, 599, 121, 3, 70, "ExampleText",
 CellID->546359973],
Cell[CellGroupData[{
Cell[16447, 606, 267, 8, 70, "Input",
 CellID->120903845],
Cell[16717, 616, 160, 5, 36, "Output",
 CellID->63280599]
}, Open  ]],
Cell[16892, 624, 238, 8, 70, "ExampleText",
 CellID->155964878],
Cell[CellGroupData[{
Cell[17155, 636, 258, 8, 70, "Input",
 CellID->48975516],
Cell[17416, 646, 174, 5, 36, "Output",
 CellID->1577004]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17639, 657, 232, 7, 70, "ExampleSection",
 CellID->5241],
Cell[17874, 666, 364, 12, 70, "ExampleText",
 CellID->38722882],
Cell[CellGroupData[{
Cell[18263, 682, 158, 5, 70, "Input",
 CellID->133352675],
Cell[18424, 689, 770, 16, 71, "Output",
 CellID->224083965]
}, Open  ]],
Cell[CellGroupData[{
Cell[19231, 710, 115, 3, 70, "Input",
 CellID->632052870],
Cell[19349, 715, 176, 5, 36, "Output",
 CellID->114544416]
}, Open  ]],
Cell[CellGroupData[{
Cell[19562, 725, 556, 19, 70, "Input",
 CellID->155558425],
Cell[20121, 746, 272, 9, 36, "Output",
 CellID->8123272]
}, Open  ]],
Cell[CellGroupData[{
Cell[20430, 760, 220, 8, 70, "Input",
 CellID->83068852],
Cell[20653, 770, 175, 5, 36, "Output",
 CellID->14533881]
}, Open  ]],
Cell[20843, 778, 125, 3, 70, "ExampleDelimiter",
 CellID->249638483],
Cell[20971, 783, 221, 7, 70, "ExampleText",
 CellID->686687598],
Cell[CellGroupData[{
Cell[21217, 794, 158, 5, 70, "Input",
 CellID->782914789],
Cell[21378, 801, 772, 17, 71, "Output",
 CellID->148764698]
}, Open  ]],
Cell[CellGroupData[{
Cell[22187, 823, 114, 3, 70, "Input",
 CellID->42522034],
Cell[22304, 828, 178, 5, 36, "Output",
 CellID->115320658]
}, Open  ]],
Cell[CellGroupData[{
Cell[22519, 838, 190, 6, 70, "Input",
 CellID->52908346],
Cell[22712, 846, 273, 9, 36, "Output",
 CellID->106671745]
}, Open  ]],
Cell[CellGroupData[{
Cell[23022, 860, 219, 8, 70, "Input",
 CellID->3941109],
Cell[23244, 870, 177, 5, 36, "Output",
 CellID->84856059]
}, Open  ]],
Cell[23436, 878, 125, 3, 70, "ExampleDelimiter",
 CellID->103804862],
Cell[23564, 883, 262, 7, 70, "ExampleText",
 CellID->15299726],
Cell[CellGroupData[{
Cell[23851, 894, 158, 5, 70, "Input",
 CellID->118537781],
Cell[24012, 901, 764, 16, 71, "Output",
 CellID->359571928]
}, Open  ]],
Cell[CellGroupData[{
Cell[24813, 922, 115, 3, 70, "Input",
 CellID->790664229],
Cell[24931, 927, 196, 6, 36, "Output",
 CellID->263227845]
}, Open  ]],
Cell[CellGroupData[{
Cell[25164, 938, 355, 11, 70, "Input",
 CellID->266056878],
Cell[25522, 951, 195, 6, 36, "Output",
 CellID->65506036]
}, Open  ]],
Cell[CellGroupData[{
Cell[25754, 962, 329, 10, 70, "Input",
 CellID->714388045],
Cell[26086, 974, 196, 6, 36, "Output",
 CellID->836691162]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26331, 986, 226, 7, 70, "ExampleSection",
 CellID->20803],
Cell[26560, 995, 213, 7, 70, "ExampleText",
 CellID->9090],
Cell[CellGroupData[{
Cell[26798, 1006, 164, 5, 70, "Input",
 CellID->9034],
Cell[26965, 1013, 269, 6, 70, "Message",
 CellID->386262796],
Cell[27237, 1021, 258, 8, 36, "Output",
 CellID->312897685]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[27556, 1036, 310, 9, 70, "SeeAlsoSection",
 CellID->7701],
Cell[27869, 1047, 1226, 44, 70, "SeeAlso",
 CellID->18497]
}, Open  ]],
Cell[CellGroupData[{
Cell[29132, 1096, 313, 9, 70, "TutorialsSection",
 CellID->5859],
Cell[29448, 1107, 156, 3, 70, "Tutorials",
 CellID->4063544]
}, Open  ]],
Cell[CellGroupData[{
Cell[29641, 1115, 299, 8, 70, "MoreAboutSection"],
Cell[29943, 1125, 154, 3, 70, "MoreAbout",
 CellID->40848186],
Cell[30100, 1130, 150, 3, 70, "MoreAbout",
 CellID->49306022]
}, Open  ]],
Cell[30265, 1136, 27, 0, 70, "History"],
Cell[30295, 1138, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

