(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     57702,       1910]
NotebookOptionsPosition[     46772,       1537]
NotebookOutlinePosition[     48963,       1599]
CellTagsIndexPosition[     48739,       1591]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RamanujanTauL" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauL"], 
          "RamanujanTauZ" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauZ"], 
          "RamanujanTauTheta" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauTheta"], 
          "BernoulliB" :> 
          Documentation`HelpLookup["paclet:ref/BernoulliB"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"RamanujanTauL\"\>", 
       2->"\<\"RamanujanTauZ\"\>", 3->"\<\"RamanujanTauTheta\"\>", 
       4->"\<\"BernoulliB\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Number Theoretic Functions\"\>", 
       2->"\<\"New in 6.0: Mathematical Functions\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       4->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RamanujanTau", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RamanujanTau",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RamanujanTau"], "[", 
       StyleBox["n", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Ramanujan ",
     Cell[BoxData[
      FormBox["\[Tau]", TraditionalForm]], "InlineMath"],
     " function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[Tau]", "(", "n", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["Integer mathematical function.", "Notes",
 CellID->126685801],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Tau]", "(", "n", ")"}], TraditionalForm]], "InlineMath"],
 " gives the coefficient of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["z", "n"], TraditionalForm]], "InlineMath"],
 " in the series expansion of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", 
    RowBox[{
     UnderoverscriptBox["\[Product]", 
      RowBox[{"k", "=", "1"}], "\[Infinity]"], 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", 
        SuperscriptBox["z", "k"]}], ")"}], "24"]}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->555347927]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->16166981],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->20365677],

Cell[TextData[{
 "The first 10 values of ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->7556],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"RamanujanTau", "[", "n", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17487],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"-", "24"}], ",", "252", ",", 
   RowBox[{"-", "1472"}], ",", "4830", ",", 
   RowBox[{"-", "6048"}], ",", 
   RowBox[{"-", "16744"}], ",", "84480", ",", 
   RowBox[{"-", "113643"}], ",", 
   RowBox[{"-", "115920"}]}], "}"}]], "Output",
 ImageSize->{454, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14725]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->466653277],

Cell[TextData[{
 "Evaluate ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 " for large arguments:"
}], "ExampleText",
 CellID->1078869648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTau", "[", 
  RowBox[{"101", "^", "55"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1357742315],

Cell[BoxData[\
"56294085057983105942879157079131682261829903645738433744600030807451003542423\
511234346471478894274087744886409255649855955824831945431793192557968103848199\
118487740714977461806976567499652695339017043095737937692139441826900553156789\
944668425357411945843251413662448511484838846028701113769514149246795721280152\
165812315690276568281214451574562404624883785167977515984656237304006193116376\
495796593506970219364720410082529664984271609872354657202439730811412204081114\
084739638728517785974526526637716840046128048578723268864263856608317593612423\
3854575275972929117277704420073054506441598633243806877974056"], "Output",
 ImageSize->{573, 152},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1878280095]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1484262483],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 " threads over lists: "
}], "ExampleText",
 CellID->829508229],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTau", "[", 
  RowBox[{"{", 
   RowBox[{"2", ",", "3316799", ",", "87539319"}], "}"}], "]"}]], "Input",
 CellTags->"Ex--21915105625151646187,8",
 CellLabel->"In[1]:=",
 CellID->80968479],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "24"}], ",", "919646364275870165595828632195541709", ",", 
   "7362787806306654495754575408695673551585280"}], "}"}]], "Output",
 ImageSize->{349, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21915105625151646187,8",
 CellLabel->"Out[1]=",
 CellID->1025549816]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32639480],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 " is zero for all non-positive integers:"
}], "ExampleText",
 CellID->963410375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"RamanujanTau", "[", 
    RowBox[{"-", "n"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11474245],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
   ",", "0", ",", "0"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1721591132]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->589287344],

Cell["Traditional form:", "ExampleText",
 CellID->2105294291],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTau", "[", "a", "]"}], " ", "//", " ", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1740580865],

Cell[BoxData[
 FormBox[
  RowBox[{
   TagBox["\[Tau]",
    RamanujanTau], "(", "a", ")"}], TraditionalForm]], "Output",
 ImageSize->{24, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->920358621]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->396801805],

Cell[TextData[{
 "Logarithmic plot of ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->823694888],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Log", "[", 
   RowBox[{"Abs", "[", 
    RowBox[{"RamanujanTau", "[", 
     RowBox[{"Range", "[", "2000", "]"}], "]"}], "]"}], "]"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->1160394054],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 105},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->247261268]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2110916249],

Cell[TextData[{
 "The first prime value of ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1739548503],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTau", "[", "63001", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->749319663],

Cell[BoxData[
 RowBox[{"-", "80561663527802406257321747"}]], "Output",
 ImageSize->{211, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->84872911]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimeQ", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->178676490],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->294169218]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->243945765],

Cell[TextData[{
 "The first ",
 "20,000",
 " values are nonzero, satisfying Lehmer's conjecture [",
 ButtonBox["more info",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://mathworld.wolfram.com/TauFunction.html"], None}],
 "]:"
}], "ExampleText",
 CellID->450190232],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"And", " ", "@@", " ", 
  RowBox[{"Thread", "[", 
   RowBox[{
    RowBox[{"RamanujanTau", "[", 
     RowBox[{"Range", "[", "20000", "]"}], "]"}], "!=", " ", "0"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->762340843],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->24047955]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->306554766],

Cell[TextData[{
 "Plot of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     RowBox[{
      TagBox["\[Tau]",
       RamanujanTau], "(", "p", ")"}],
     TraditionalForm], "/", 
    RowBox[{"(", 
     RowBox[{"2", " ", 
      SuperscriptBox["p", 
       RowBox[{"11", "/", "2"}]]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 " at primes ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->452683009],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"RamanujanTau", "[", "#", "]"}], "/", 
      RowBox[{"(", 
       RowBox[{"2", " ", 
        RowBox[{"#", "^", 
         RowBox[{"(", 
          RowBox[{"11", "/", "2"}], ")"}]}]}], ")"}]}], " ", "&"}], " ", "/@",
     " ", 
    RowBox[{"Prime", "[", 
     RowBox[{"Range", "[", "50", "]"}], "]"}]}], ",", 
   RowBox[{"Mesh", "\[Rule]", " ", "All"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1917605138],

Cell[BoxData[
 GraphicsBox[
  GraphicsComplexBox[{{1., -0.265165042944955}, {2., 0.299366806246473}, {3., 
   0.345606666602368}, {4., -0.188273848279482}, {5., 0.500436454748519}, {
   6., -0.215780651646255}, {7., -0.589825249708127}, {8., 
   0.493901393563481}, {9., 0.30198755234041}, {10., 0.581257358669316}, {
   11., -0.165756105528446}, {12., -0.215993473628517}, {13., 
   0.207672649916771}, {14., -0.00888263943429987}, {15., 
   0.854586002652588}, {16., -0.262120365284279}, {17., -0.472481341530272}, {
   18., 0.52728494201767}, {19., -0.700455897839543}, {20., 
   0.322030985718202}, {21., 0.0413212491457406}, {22., 0.696847836620241}, {
   23., -0.408722046636399}, {24., -0.237214873818356}, {25., 
   0.443471068321085}, {26., 0.386948175883595}, {27., -0.959407024119546}, {
   28., 0.311002850592191}, {29., 0.228722679943879}, {
   30., -0.21737380902429}, {31., -0.352807693976015}, {32., 
   0.715106970623704}, {33., -0.263059419807908}, {34., 0.487767369893441}, {
   35., -0.622141624277016}, {36., -0.42732636952114}, {37., 
   0.550156472513261}, {38., -0.12179186414355}, {39., 0.820587374042845}, {
   40., -0.233140181208128}, {41., 0.341936374303999}, {
   42., -0.19069300385056}, {43., 0.393163847646396}, {44., 
   0.731466229072334}, {45., -0.345309446771792}, {46., 0.0827261655770481}, {
   47., -0.559099216195819}, {48., 0.445333396867367}, {
   49., -0.0748713153664579}, {50., -0.620375126048287}}, {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50}]}}, {
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 
       36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50}]}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  PlotRange->{{0., 50.}, {-0.959407024119546, 0.854586002652588}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 98},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->148098767]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->850104482],

Cell[TextData[{
 "The summatory ",
 Cell[BoxData[
  FormBox["\[Tau]", TraditionalForm]], "InlineMath"],
 "-function [",
 ButtonBox["more info",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://mathworld.wolfram.com/TauFunction.html"], None}],
 "]:"
}], "ExampleText",
 CellID->497325483],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{"RamanujanTau", "[", "n", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", "1", ",", 
        RowBox[{"x", "-", "1"}]}], "}"}]}], "]"}], "+", 
    RowBox[{"Piecewise", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"RamanujanTau", "[", "x", "]"}], "/", "2"}], ",", 
         RowBox[{"Element", "[", 
          RowBox[{"x", ",", "Integers"}], "]"}]}], "}"}], "}"}], ",", " ", 
      RowBox[{"RamanujanTau", "[", 
       RowBox[{"Floor", "[", "x", "]"}], "]"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "30"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", " ", "All"}], ",", 
   RowBox[{"Filling", "\[Rule]", " ", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->543038657],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWstLVUEYH8/j6i1IN+pNXVgtgv6HCCKSFkUE0aIgbqZoUIYa4R9QLbJF
FkEEtkh7SiiCtQm0yDaRFBVWELQIo3JTtBCC0zeP85jvzHl6n+Tl3jn3m5nv
PfPN7zruzQ/2dJ3MD/Z25ts7+vOne3o7B9p39/VDl15DCPkOn6Ecge8WfGOf
bZZlEZ1+Yy/yhDZkNb1jjF5Z+bvw6iu8iUEb/ffvFTFe73BpwDTOZo+PLRzY
f6vr+EOis+bQwduXhp/RoYYQHTdHXw5ffPph8QfRWQP0hfOz5eHKQEuZdEqZ
DjcbE72Gt5fO1UQM1qfSmEVcpoemOk1Zp9Dn5ahLpZc7w1/UegNJySDaKJId
5fK/VmStVpnpAG+lnlqR//j6vFbWx9SQ3LM64dkG8dTFE2cY0/Gkb6kgKZt8
ETNTydksMpmOW66BZa5mZwZn3Gq9vPxHzM8mkDI3+xmqt1vF80cnOo9N0Knr
YknJMinzz79A+S8flyojlWHZGlc583SO0cG4pjGV1I0WryCtlowJcoLmBc5+
NQNheGbxMzAYPZliviwlG2KpfarbtV9XWt2m8CL8rG1CtI5oFj3TOb2FuVhK
BlmOa54Xezhuh/DjHGmIn9IUONfQZ4ki0RQpISoK+PTHGqohCs0KG5LEAGO7
KH5fDJS4SnO818Sepc+2eN5K+J/ytRTax6C9bTp+eTUFeOaV2ILo1nie+lBx
jrnj5VNzR3kcF9MXcm3bWW6xZFScs1x0rIna7B3HmVDVKi9/mOXxqnI3o+FE
inlCYYuaImKnPsG6ba6AMwhLLbDWUBy7tPRLzHey0XdqGt5EZw2ouHb1ReGs
2sfod2+/iQzoCE3XK60GqSd6J93YHTl8F6xztFBTwcrpqffEgMZElvI5j2YW
oQvebIhhAoAqllNZozzZCu2cvw5IfgTTRsR4Ujq1PCkGCnkiD3qQfzu2X4c3
0VkD9L07r2ncddZw+awEobzzqgKrDbpoeA0WYyCk30NAf/r4U9Iv2QM5xPzw
dNcJ0HxBu/xglzQOIhLZ59XvW83YyoaI8ZLNF3vYYHuYh9Ryar/CS5xlqSZA
yGDDweYgBm0SR33o7GMR9WyqKGPsFEU32lqg3qq08BoDmx9KgLsXrozM4/mS
VxDKKK8C9Km96ldywbYK9QKIVViVqRZL9nSMuuutHJYMMPrB/TdSveRiim8J
RTAjJfP1clxNIJwlhDVQEoqkCYqQiLrBou7is7qCaVLl73+RFGd1VZ9XfkmA
KnftvEGBisHQChQVvpQSSJos+UyanakKtq9aZlZyHKW7UOnXOkZ0mhV+Exqs
A9/+eREv/bWc8emOuv2Lrwvf9iS/aSy2X5QM1mHfYDLK97eaAG9C7xfDdfnQ
acGkp72xDJYYdb9YDM7o28hgXu8NZCXXhGqauRbHtehUZnQA4AHMczEfv15n
Q0qu1f83F+8lNf8AMI9jZw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{234, 121},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->183884189]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->632133759],

Cell[TextData[{
 "The first 10 values of ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 " using ",
 Cell[BoxData[
  ButtonBox["Product",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Product"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1737602355],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CoefficientList", "[", 
  RowBox[{
   RowBox[{"Take", "[", 
    RowBox[{
     RowBox[{"Expand", "[", 
      RowBox[{"Product", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"1", "-", 
           RowBox[{"x", "^", "k"}]}], ")"}], "^", "24"}], ",", 
        RowBox[{"{", 
         RowBox[{"k", ",", "1", ",", "10"}], "}"}]}], "]"}], "]"}], ",", 
     "10"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->636994126],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"-", "24"}], ",", "252", ",", 
   RowBox[{"-", "1472"}], ",", "4830", ",", 
   RowBox[{"-", "6048"}], ",", 
   RowBox[{"-", "16744"}], ",", "84480", ",", 
   RowBox[{"-", "113643"}], ",", 
   RowBox[{"-", "115920"}]}], "}"}]], "Output",
 ImageSize->{454, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->41739155]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTau", "[", 
  RowBox[{"Range", "[", "10", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1744988501],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"-", "24"}], ",", "252", ",", 
   RowBox[{"-", "1472"}], ",", "4830", ",", 
   RowBox[{"-", "6048"}], ",", 
   RowBox[{"-", "16744"}], ",", "84480", ",", 
   RowBox[{"-", "113643"}], ",", 
   RowBox[{"-", "115920"}]}], "}"}]], "Output",
 ImageSize->{454, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->53459583]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1267379885],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 " is multiplicative for coprime integers:"
}], "ExampleText",
 CellID->186321937],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTau", "[", 
  RowBox[{"7", " ", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1949258147],

Cell[BoxData[
 RowBox[{"-", "80873520"}]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63564010]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTau", "[", "7", "]"}], " ", 
  RowBox[{"RamanujanTau", "[", "5", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->378873668],

Cell[BoxData[
 RowBox[{"-", "80873520"}]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->42555892]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1434562648],

Cell[TextData[{
 "For prime ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->895520067],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"p", "=", "3"}], ",", 
     RowBox[{"n", "=", "6"}]}], "}"}], ",", " ", 
   RowBox[{
    RowBox[{"RamanujanTau", "[", 
     RowBox[{"p", "^", 
      RowBox[{"(", 
       RowBox[{"n", "+", "1"}], ")"}]}], "]"}], "\[Equal]", " ", 
    RowBox[{
     RowBox[{
      RowBox[{"RamanujanTau", "[", "p", "]"}], 
      RowBox[{"RamanujanTau", "[", 
       RowBox[{"p", "^", "n"}], "]"}]}], "-", 
     RowBox[{
      RowBox[{"p", "^", "11"}], " ", 
      RowBox[{"RamanujanTau", "[", 
       RowBox[{"p", "^", 
        RowBox[{"(", 
         RowBox[{"n", "-", "1"}], ")"}]}], "]"}]}]}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1440367893],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->572776399]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1422572156],

Cell["Congruence relations:", "ExampleText",
 CellID->1234783841],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"And", " ", "@@", " ", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Mod", "[", 
      RowBox[{
       RowBox[{"RamanujanTau", "[", 
        RowBox[{"5", " ", "n"}], "]"}], ",", "5"}], "]"}], " ", "==", " ", 
     "0"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"n", " ", ",", "200"}], "}"}]}], "]"}]}]], "Input",
 CellTags->"Ex--21932518796324839719,1",
 CellLabel->"In[1]:=",
 CellID->1877534726],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21932518796324839719,1",
 CellLabel->"Out[1]=",
 CellID->13265601]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"And", "@@", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Mod", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"RamanujanTau", "[", 
         RowBox[{"4", "n"}], "]"}], "-", 
        RowBox[{"RamanujanTau", "[", "n", "]"}]}], ",", "3"}], "]"}], 
     "\[Equal]", "0"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "200"}], "}"}]}], "]"}]}]], "Input",
 CellTags->"Ex--21933845794896247830,2",
 CellLabel->"In[2]:=",
 CellID->43004819],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21933845794896247830,2",
 CellLabel->"Out[2]=",
 CellID->376579128]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"And", " ", "@@", " ", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Mod", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"RamanujanTau", "[", "n", "]"}], "-", 
        RowBox[{"DivisorSigma", "[", 
         RowBox[{"11", ",", "n"}], "]"}]}], ",", "691"}], "]"}], " ", 
     "\[Equal]", " ", "0"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "200"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1627046875],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->183748851]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1625213980],

Cell["Representation of an integer as sum of 24 squares:", "ExampleText",
 CellID->1126855667],

Cell[BoxData[
 RowBox[{
  RowBox[{"r", "[", "n_", "]"}], ":=", 
  RowBox[{
   RowBox[{
    RowBox[{"16", "/", "691"}], " ", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], "^", 
        RowBox[{"(", 
         RowBox[{"n", "+", "d"}], ")"}]}], " ", 
       RowBox[{"d", "^", "11"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"d", ",", 
        RowBox[{"Divisors", "[", "n", "]"}]}], "}"}]}], "]"}]}], "+", 
   RowBox[{
    RowBox[{"128", "/", "691"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"259", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], "^", 
        RowBox[{"(", 
         RowBox[{"n", "-", "1"}], ")"}]}], 
       RowBox[{"RamanujanTau", "[", "n", "]"}]}], "-", 
      RowBox[{"512", " ", 
       RowBox[{"RamanujanTau", "[", 
        RowBox[{"n", "/", "2"}], "]"}]}]}], ")"}]}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->404912376],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "[", "8", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2134109736],

Cell[BoxData["195082320"], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->286017672]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaresR", "[", 
  RowBox[{"24", ",", "8"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->704882386],

Cell[BoxData["195082320"], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->359808861]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->968106421],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " is the Dirichlet ",
 Cell[BoxData[
  FormBox["L", TraditionalForm]], "InlineMath"],
 "-function associated with ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1628508075],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"RamanujanTau", "[", "n", "]"}], "/", 
    RowBox[{"n", "^", 
     RowBox[{"(", 
      RowBox[{"20", "+", "I"}], ")"}]}]}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->656135510],

Cell[BoxData[
 RowBox[{"0.9999824261436988`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.000014561565620212934`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->588754558]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"RamanujanTauL", "[", 
   RowBox[{"20", "+", "I"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->838735361],

Cell[BoxData[
 RowBox[{"0.9999824261436988`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.000014561565620212942`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->9802004]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->7485930],

Cell["Large prime numbers can take a long time:", "ExampleText",
 CellID->1917272193],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTau", "[", "214928639999", "]"}], " ", "//", " ", 
  "Timing"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->601624516],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "29.73854`", ",", 
   "336969318523447435477041542002543549208693346637507838687265285"}], 
  "}"}]], "Output",
 ImageSize->{562, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1150230336]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->772385240],

Cell[TextData[{
 "Successive differences of ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 " modulo 3:"
}], "ExampleText",
 CellID->2083303949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Mod", "[", 
   RowBox[{
    RowBox[{"NestList", "[", 
     RowBox[{"Differences", ",", 
      RowBox[{"RamanujanTau", "[", 
       RowBox[{"Range", "[", "100", "]"}], "]"}], ",", "100"}], "]"}], ",", 
    "3"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->602961573],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 181},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->460493542]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->101495224],

Cell[TextData[{
 "A representation of zero in terms of ",
 Cell[BoxData[
  ButtonBox["RamanujanTau",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTau"]], "InlineFormula"],
 ": "
}], "ExampleText",
 CellID->728214050],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTau", "[", "6", "]"}], "+", 
  RowBox[{"RamanujanTau", "[", "14", "]"}], "+", 
  RowBox[{"RamanujanTau", "[", "29", "]"}], "+", 
  RowBox[{"RamanujanTau", "[", "41", "]"}], "+", 
  RowBox[{"RamanujanTau", "[", "42", "]"}], "+", 
  RowBox[{"RamanujanTau", "[", "44", "]"}], "+", 
  RowBox[{"RamanujanTau", "[", "48", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->584234097],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->68624313]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->429214243],

Cell[TextData[{
 "Find digit counts for",
 Cell[BoxData[
  RowBox[{" ", 
   RowBox[{"RamanujanTau", "[", 
    RowBox[{"10", "^", "12"}], "]"}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->406211828],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DigitCount", "[", 
  RowBox[{"RamanujanTau", "[", 
   RowBox[{"10", "^", "12"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->709697294],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "8", ",", "3", ",", "4", ",", "9", ",", "11", ",", "5", ",", "4", ",", "3", 
   ",", "4", ",", "15"}], "}"}]], "Output",
 ImageSize->{203, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->254567651]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauL",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauL"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauZ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauZ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauTheta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauTheta"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BernoulliB",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BernoulliB"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->23022]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->275655809],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->94319784],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->623199840],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->86343189],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->808685117]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RamanujanTau - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 41.9687214}", "context" -> "System`", 
    "keywords" -> {
     "Ramanujan tau", "Ramanujan tau-Dirichlet series", 
      "Ramanujan tau function"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "RamanujanTau[n] gives the Ramanujan \\[Tau] function \\[Tau](n).", 
    "synonyms" -> {"ramanujan tau"}, "title" -> "RamanujanTau", "type" -> 
    "Symbol", "uri" -> "ref/RamanujanTau"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7115, 214, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->16166981]},
 "Ex--21915105625151646187,8"->{
  Cell[10333, 334, 218, 6, 70, "Input",
   CellTags->"Ex--21915105625151646187,8",
   CellID->80968479],
  Cell[10554, 342, 357, 10, 54, "Output",
   CellTags->"Ex--21915105625151646187,8",
   CellID->1025549816]},
 "Ex--21932518796324839719,1"->{
  Cell[31380, 998, 441, 14, 70, "Input",
   CellTags->"Ex--21932518796324839719,1",
   CellID->1877534726],
  Cell[31824, 1014, 201, 6, 36, "Output",
   CellTags->"Ex--21932518796324839719,1",
   CellID->13265601]},
 "Ex--21933845794896247830,2"->{
  Cell[32062, 1025, 485, 16, 70, "Input",
   CellTags->"Ex--21933845794896247830,2",
   CellID->43004819],
  Cell[32550, 1043, 202, 6, 36, "Output",
   CellTags->"Ex--21933845794896247830,2",
   CellID->376579128]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 47851, 1563},
 {"Ex--21915105625151646187,8", 48003, 1567},
 {"Ex--21932518796324839719,1", 48251, 1574},
 {"Ex--21933845794896247830,2", 48500, 1581}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2777, 65, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3400, 92, 1577, 37, 70, "ObjectNameGrid"],
Cell[4980, 131, 691, 22, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5708, 158, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6186, 175, 67, 1, 70, "Notes",
 CellID->126685801],
Cell[6256, 178, 606, 22, 70, "Notes",
 CellID->1067943069],
Cell[6865, 202, 213, 7, 70, "Notes",
 CellID->555347927]
}, Closed]],
Cell[CellGroupData[{
Cell[7115, 214, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->16166981],
Cell[CellGroupData[{
Cell[7499, 228, 147, 5, 70, "ExampleSection",
 CellID->20365677],
Cell[7649, 235, 209, 8, 70, "ExampleText",
 CellID->7556],
Cell[CellGroupData[{
Cell[7883, 247, 210, 7, 28, "Input",
 CellID->17487],
Cell[8096, 256, 424, 13, 36, "Output",
 CellID->14725]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8569, 275, 220, 7, 70, "ExampleSection",
 CellID->466653277],
Cell[8792, 284, 221, 8, 70, "ExampleText",
 CellID->1078869648],
Cell[CellGroupData[{
Cell[9038, 296, 135, 4, 70, "Input",
 CellID->1357742315],
Cell[9176, 302, 782, 13, 173, "Output",
 CellID->1878280095]
}, Open  ]],
Cell[9973, 318, 126, 3, 70, "ExampleDelimiter",
 CellID->1484262483],
Cell[10102, 323, 206, 7, 70, "ExampleText",
 CellID->829508229],
Cell[CellGroupData[{
Cell[10333, 334, 218, 6, 70, "Input",
 CellTags->"Ex--21915105625151646187,8",
 CellID->80968479],
Cell[10554, 342, 357, 10, 54, "Output",
 CellTags->"Ex--21915105625151646187,8",
 CellID->1025549816]
}, Open  ]],
Cell[10926, 355, 124, 3, 70, "ExampleDelimiter",
 CellID->32639480],
Cell[11053, 360, 224, 7, 70, "ExampleText",
 CellID->963410375],
Cell[CellGroupData[{
Cell[11302, 371, 243, 8, 70, "Input",
 CellID->11474245],
Cell[11548, 381, 292, 9, 36, "Output",
 CellID->1721591132]
}, Open  ]],
Cell[11855, 393, 125, 3, 70, "ExampleDelimiter",
 CellID->589287344],
Cell[11983, 398, 61, 1, 70, "ExampleText",
 CellID->2105294291],
Cell[CellGroupData[{
Cell[12069, 403, 160, 5, 70, "Input",
 CellID->1740580865],
Cell[12232, 410, 266, 9, 50, "Output",
 CellID->920358621]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12547, 425, 227, 7, 70, "ExampleSection",
 CellID->396801805],
Cell[12777, 434, 211, 8, 70, "ExampleText",
 CellID->823694888],
Cell[CellGroupData[{
Cell[13013, 446, 251, 8, 70, "Input",
 CellID->1160394054],
Cell[13267, 456, 4958, 85, 70, "Output",
 Evaluatable->False,
 CellID->247261268]
}, Open  ]],
Cell[18240, 544, 126, 3, 70, "ExampleDelimiter",
 CellID->2110916249],
Cell[18369, 549, 217, 8, 70, "ExampleText",
 CellID->1739548503],
Cell[CellGroupData[{
Cell[18611, 561, 112, 3, 70, "Input",
 CellID->749319663],
Cell[18726, 566, 200, 6, 36, "Output",
 CellID->84872911]
}, Open  ]],
Cell[CellGroupData[{
Cell[18963, 577, 102, 3, 70, "Input",
 CellID->178676490],
Cell[19068, 582, 161, 5, 36, "Output",
 CellID->294169218]
}, Open  ]],
Cell[19244, 590, 125, 3, 70, "ExampleDelimiter",
 CellID->243945765],
Cell[19372, 595, 276, 10, 70, "ExampleText",
 CellID->450190232],
Cell[CellGroupData[{
Cell[19673, 609, 252, 8, 70, "Input",
 CellID->762340843],
Cell[19928, 619, 160, 5, 36, "Output",
 CellID->24047955]
}, Open  ]],
Cell[20103, 627, 125, 3, 70, "ExampleDelimiter",
 CellID->306554766],
Cell[20231, 632, 459, 20, 70, "ExampleText",
 CellID->452683009],
Cell[CellGroupData[{
Cell[20715, 656, 522, 17, 70, "Input",
 CellID->1917605138],
Cell[21240, 675, 2344, 42, 119, "Output",
 CellID->148098767]
}, Open  ]],
Cell[23599, 720, 125, 3, 70, "ExampleDelimiter",
 CellID->850104482],
Cell[23727, 725, 296, 11, 70, "ExampleText",
 CellID->497325483],
Cell[CellGroupData[{
Cell[24048, 740, 900, 26, 70, "Input",
 CellID->543038657],
Cell[24951, 768, 1701, 32, 70, "Output",
 Evaluatable->False,
 CellID->183884189]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[26701, 806, 237, 7, 70, "ExampleSection",
 CellID->632133759],
Cell[26941, 815, 344, 13, 70, "ExampleText",
 CellID->1737602355],
Cell[CellGroupData[{
Cell[27310, 832, 493, 16, 70, "Input",
 CellID->636994126],
Cell[27806, 850, 427, 13, 36, "Output",
 CellID->41739155]
}, Open  ]],
Cell[CellGroupData[{
Cell[28270, 868, 142, 4, 70, "Input",
 CellID->1744988501],
Cell[28415, 874, 427, 13, 36, "Output",
 CellID->53459583]
}, Open  ]],
Cell[28857, 890, 126, 3, 70, "ExampleDelimiter",
 CellID->1267379885],
Cell[28986, 895, 225, 7, 70, "ExampleText",
 CellID->186321937],
Cell[CellGroupData[{
Cell[29236, 906, 132, 4, 70, "Input",
 CellID->1949258147],
Cell[29371, 912, 181, 6, 36, "Output",
 CellID->63564010]
}, Open  ]],
Cell[CellGroupData[{
Cell[29589, 923, 170, 5, 70, "Input",
 CellID->378873668],
Cell[29762, 930, 181, 6, 36, "Output",
 CellID->42555892]
}, Open  ]],
Cell[29958, 939, 126, 3, 70, "ExampleDelimiter",
 CellID->1434562648],
Cell[30087, 944, 138, 6, 70, "ExampleText",
 CellID->895520067],
Cell[CellGroupData[{
Cell[30250, 954, 732, 24, 70, "Input",
 CellID->1440367893],
Cell[30985, 980, 161, 5, 36, "Output",
 CellID->572776399]
}, Open  ]],
Cell[31161, 988, 126, 3, 70, "ExampleDelimiter",
 CellID->1422572156],
Cell[31290, 993, 65, 1, 70, "ExampleText",
 CellID->1234783841],
Cell[CellGroupData[{
Cell[31380, 998, 441, 14, 70, "Input",
 CellTags->"Ex--21932518796324839719,1",
 CellID->1877534726],
Cell[31824, 1014, 201, 6, 36, "Output",
 CellTags->"Ex--21932518796324839719,1",
 CellID->13265601]
}, Open  ]],
Cell[CellGroupData[{
Cell[32062, 1025, 485, 16, 70, "Input",
 CellTags->"Ex--21933845794896247830,2",
 CellID->43004819],
Cell[32550, 1043, 202, 6, 36, "Output",
 CellTags->"Ex--21933845794896247830,2",
 CellID->376579128]
}, Open  ]],
Cell[CellGroupData[{
Cell[32789, 1054, 474, 15, 70, "Input",
 CellID->1627046875],
Cell[33266, 1071, 161, 5, 36, "Output",
 CellID->183748851]
}, Open  ]],
Cell[33442, 1079, 126, 3, 70, "ExampleDelimiter",
 CellID->1625213980],
Cell[33571, 1084, 94, 1, 70, "ExampleText",
 CellID->1126855667],
Cell[33668, 1087, 963, 33, 70, "Input",
 CellID->404912376],
Cell[CellGroupData[{
Cell[34656, 1124, 98, 3, 70, "Input",
 CellID->2134109736],
Cell[34757, 1129, 166, 5, 36, "Output",
 CellID->286017672]
}, Open  ]],
Cell[CellGroupData[{
Cell[34960, 1139, 128, 4, 70, "Input",
 CellID->704882386],
Cell[35091, 1145, 166, 5, 36, "Output",
 CellID->359808861]
}, Open  ]],
Cell[35272, 1153, 125, 3, 70, "ExampleDelimiter",
 CellID->968106421],
Cell[35400, 1158, 434, 15, 70, "ExampleText",
 CellID->1628508075],
Cell[CellGroupData[{
Cell[35859, 1177, 333, 11, 70, "Input",
 CellID->656135510],
Cell[36195, 1190, 278, 7, 36, "Output",
 CellID->588754558]
}, Open  ]],
Cell[CellGroupData[{
Cell[36510, 1202, 162, 5, 70, "Input",
 CellID->838735361],
Cell[36675, 1209, 276, 7, 36, "Output",
 CellID->9802004]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[37000, 1222, 228, 7, 70, "ExampleSection",
 CellID->7485930],
Cell[37231, 1231, 85, 1, 70, "ExampleText",
 CellID->1917272193],
Cell[CellGroupData[{
Cell[37341, 1236, 161, 5, 70, "Input",
 CellID->601624516],
Cell[37505, 1243, 285, 10, 36, "Output",
 CellID->1150230336]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[37839, 1259, 228, 7, 70, "ExampleSection",
 CellID->772385240],
Cell[38070, 1268, 227, 8, 70, "ExampleText",
 CellID->2083303949],
Cell[CellGroupData[{
Cell[38322, 1280, 331, 10, 70, "Input",
 CellID->602961573],
Cell[38656, 1292, 3275, 58, 70, "Output",
 Evaluatable->False,
 CellID->460493542]
}, Open  ]],
Cell[41946, 1353, 125, 3, 70, "ExampleDelimiter",
 CellID->101495224],
Cell[42074, 1358, 229, 8, 70, "ExampleText",
 CellID->728214050],
Cell[CellGroupData[{
Cell[42328, 1370, 421, 10, 70, "Input",
 CellID->584234097],
Cell[42752, 1382, 157, 5, 36, "Output",
 CellID->68624313]
}, Open  ]],
Cell[42924, 1390, 125, 3, 70, "ExampleDelimiter",
 CellID->429214243],
Cell[43052, 1395, 208, 8, 70, "ExampleText",
 CellID->406211828],
Cell[CellGroupData[{
Cell[43285, 1407, 171, 5, 70, "Input",
 CellID->709697294],
Cell[43459, 1414, 293, 9, 36, "Output",
 CellID->254567651]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[43813, 1430, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[44132, 1441, 987, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[45156, 1481, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[45477, 1492, 144, 3, 70, "Tutorials",
 CellID->23022]
}, Open  ]],
Cell[CellGroupData[{
Cell[45658, 1500, 319, 9, 70, "MoreAboutSection",
 CellID->275655809],
Cell[45980, 1511, 161, 3, 70, "MoreAbout",
 CellID->94319784],
Cell[46144, 1516, 174, 3, 70, "MoreAbout",
 CellID->623199840],
Cell[46321, 1521, 178, 3, 70, "MoreAbout",
 CellID->86343189],
Cell[46502, 1526, 198, 4, 70, "MoreAbout",
 CellID->808685117]
}, Open  ]],
Cell[46715, 1533, 27, 0, 70, "History"],
Cell[46745, 1535, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

