(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    130762,       2809]
NotebookOptionsPosition[    123820,       2569]
NotebookOutlinePosition[    125126,       2606]
CellTagsIndexPosition[    125039,       2601]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RamanujanTauZ" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauZ"], 
          "RamanujanTauL" :> 
          Documentation`HelpLookup["paclet:ref/RamanujanTauL"], 
          "RiemannSiegelTheta" :> 
          Documentation`HelpLookup["paclet:ref/RiemannSiegelTheta"]}, #, 2]& ,
         Evaluator -> Automatic], {1->"\<\"RamanujanTauZ\"\>", 
       2->"\<\"RamanujanTauL\"\>", 
       3->"\<\"RiemannSiegelTheta\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Number Theoretic Functions\"\>", 
       2->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RamanujanTauTheta", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RamanujanTauTheta",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RamanujanTauTheta"], "[", 
       StyleBox["t", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the Ramanujan tau theta function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[Theta]", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "Apart from a sign, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Theta]", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
 " gives the phase of the Ramanujan L-function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"L", 
    RowBox[{"(", 
     RowBox[{"6", "+", 
      RowBox[{"i", "\[InvisibleSpace]", "t"}]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Notes",
 CellID->36373499],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->989328282],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->381534625],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->181045676],

Cell["Evaluate numerically:", "ExampleText",
 CellID->2111517477],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauTheta", "[", "9.22", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1106969264],

Cell[BoxData["1.4037204336632279`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1559155195]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2117552439],

Cell["Plot on the real line:", "ExampleText",
 CellID->1604844189],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"RamanujanTauTheta", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->221678203],

Cell[GraphicsData["CompressedBitmap", "\<\
eJytWU1oXFUUfsm8zDTWatKfNNPYNO0ktfbPmqZN/InTWFSkFnGhXRQXMQjJ
Qltq3OhGN7rRjW4siF34A/5BxVIRKogFsSAWxIKoIBWkLiyCxUVAeH7nnHvn
3Dlz500yaUhm3rv33O9+95zvnHvfy+Hp+dmnnp6en5uZHnrwxPTx2bmZZ4ce
OHYCTYWOJEnO4O94f4LrDFf8dybL8B1+HqXPAvXxT/I4t67j1s/P/uRa9+KD
jDqNdScuizkIz82fvfDt73Q5ZqwK9JHqfZJ05TN55eWv6HIXPmhYYVF4ZBDi
XLu28Ogj7yws/Nc2nz5ufenFc1+e+9V7pn2c81//9sLzXywfBys68tj7V6/+
m4zi3keleXTWc+sbr3/z0Yc/0OWUsz4S9F768c+52dO+txGjn1ufOPrB5ct/
0+WBHKv33r148s0LrazAH9FpblXm1mPPfHbx+z/oslrHumw8Ws3BUH/FrTYQ
8qm3v2veG7K9N2o1wK0gA0p0OVnHVnqRHEgR39uIcYvx8T05VgglAtrKSjMg
brXRKOPuOtbS+8vPf808+bHvbcQYNCvf7jDCnE2omjTV56CJ5W2ZzYaSGWfr
geBsMv6rZFI9JJkCuCXhqae3LYqXViEZD+8jBohEZHxXlpe3QyY6t0bHh+Pi
VUNwoGBkAWlhm5u15L7jozab2cdzrBR7vE5BWwzG/ijGFoMRt6oYrH05VsCC
Zinj41bDJrJjdayHfdxc1bC7x1p2NfpRmeQXOSI5nYxk9Vq2lb6Hx0KgKJIo
BoceOnX/wbe4zCVFUK5OnoR+X3v1PBAwwQRjljLdj4ml1E9YY1q67Hb9W81c
q+ijBOVhoVgKgFGWUYRIi6y4Inh8evoSqIIHJr1y5Z/UYY2YOTlDkhUNCrRn
BNl5tdLdaPor5n4l+wOMEAlwxCdICHPyMNiBNlqJs9VpJcqGDkMd9C1NQjzx
Fahxp7UrsHs+9aeRLKk4z9zguSKiWAF8TVx7eR60iu/hkLQN9q393ef17nJn
ZQt/CyLoiBpAdZ1HQMTQ0lALrw9POWVoxq1w3rOaZV2mYANRghDEwHt4CudS
jvDP1uvCyGaRreUj5l7iD89hCLLT7bBIfphQlVgMK19xC+5bzgW6w3qv2LkL
Xk3wHryC6Qb9SKftYYMsst/OVshs5BFd7g57C8h+nPkQE1fF4Wz4mPJvVzAa
s4IdbHl0Uqqxwg/8AARiUfF8gED638kIO3wrrEidjFusIWA1CDPMR4wdjxOM
QDM7vYexfiHOO2pQbZmxzBKcHHd4b2IdOOe6dUHrEBVd8nmD4/uJ14nox+oy
Z6e191IVav4QQTRo7KYWKJ0+7ihHcAI6elqMYMTuMEaIJ9bdPgdR0u2Zqxvw
NEXI7ZK0kwHdlUr87DYR2RxF28OXGIV1ZbUqhec+hJWUZnNVIoqHQnlIlVxL
7uBWBNE9LfI5J8VKcY9fXNhaJjiqaMEZDXBobZJbUAqsMGHtlMY+oIxHTDNf
IdwOL9sojR7wMYPAsBy432pHcDQr2ZPulCH1hbiV/Wy4B7cGRQoKFolKTvVH
UOS8I1Eh7rw/pCABdqCNjm6Ds8dEjGMhZzDnS1QcjFvNLZgJxLHZYIDdayQe
+iTA57pkwisHPAEGJFZcipXhFkdjeGGV8/BowJ58KQh3sdKwGiBI3pN/WMdF
WAFaKGF9MLrZsBJMPTeyIuV5KEWM0QFYHMK6aH/pAh9cY8VgRZWwN4qm+62g
Vb2u5Dgo2wJZuVFrDMperzj3nMzK5bcz5IUxj+bOn9obYoiVVsdNOVaKFbfa
Z7AGc6wUS62I9bjB2BjF2G8w4lYWi/fa5OGs+ZOlrVrjZh5BOMRRD/doWw1t
5ZkwTAaWgEO3Ml6fw5cyPry/M6tX8YYmOL5WWX/IeCga/mg1PhxnT62ci0GO
l+si48838VH6jqlcpxx5N6Ensf6oJuTtg9apPCuNe9xKZtQTwPocK/WZWhFr
eQ+k7zz7ohiThnXcSrAgESkZ8hQVtaL3VHkY+p5KMYjtARO3tVGMKt/rvpxn
pe814lYyo+5ya3KsVBlqRazv4159o7M6ijHl/CJjSYv+ydW+vbbqnjIRFvzD
xqrxmdHqW3jqibN3WTj6/nspOIWav7TatMvjoFFLz7JwkNbOLz21CIVvrePv
n+L/qwj/m5F0/A/vDjuh\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->345386757]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->392378955],

Cell["Evaluate for complex arguments:", "ExampleText",
 CellID->1961420917],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauTheta", "[", 
  RowBox[{"1.", "+", 
   RowBox[{"1.4", " ", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1727370569],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "0.15842567123748275`"}], "-", 
  RowBox[{"0.17569347564854754`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{148, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->85729390]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2039814240],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->728990972],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"RamanujanTauTheta", "[", "1", "]"}], ",", "50"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->757060269],

Cell[BoxData[
 RowBox[{"-", 
  "0.1263468371052371496747371925623142739223808193627429834425078117551`50."}\
]], "Output",
 ImageSize->{376, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->591267630]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->538700578],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->149558694],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauTheta", "[", "1.000000000000000000000000", 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1077103068],

Cell[BoxData[
 RowBox[{"-", 
  "0.12634683710523714967473719256232209468`22.614979959397523"}]], "Output",
 ImageSize->{187, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1731603313]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->358497447],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 " threads over lists:"
}], "ExampleText",
 CellID->1574961894],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTauTheta", "[", 
   RowBox[{"{", 
    RowBox[{"1.5", ",", "1.6", ",", "1.8"}], "}"}], "]"}], " "}]], "Input",
 CellLabel->"In[1]:=",
 CellID->501119536],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.17958546330428238`"}], ",", 
   RowBox[{"-", "0.18896749826032`"}], ",", 
   RowBox[{"-", "0.20625341983943768`"}]}], "}"}]], "Output",
 ImageSize->{232, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2032661665]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->416401570],

Cell[TextData[{
 "Contour plot of the absolute value of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->883919168],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"RamanujanTauTheta", "[", 
     RowBox[{"x", "+", 
      RowBox[{"I", " ", "y"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1885070277],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJyNvQd4VcXzPk4CKaTe5AbpSJUuUqQGzgLSe0dBVIogvUgVBAvSi9Kb9N5B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   "], {{{}, 
     {RGBColor[0.343534610422504, 0.144547128242105, 0.599859769947973], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNVtlyVFUUPa19b1/76C9koNMyygyJQYNlFfkAX2JhGhOCBgz0kJQKaJnE
F0tLFAF5ophE86BQln/gRAQSYkmoAgdigAcc/kCNrlVrpfDh1Nl3n332uPY+
t3Vn7ZnqAyGEWawHsZ6OIawshDCchPBTPoSeNISZXAjrshCOgH4VMh/h7CAu
bc2J97X5beAPYb0Bugw9JejZju8BrBewRsC/CNnXcXYe349jfwXrTfB/y4m3
DXsV+6dY90Dvx/mBqO8a1oVEsoeg50usPtz9BvufBfl0Cud7bIv7btPr4f/R
VL7VwRtMZLed9sDvD+LtNn855Keg87mg+zwbxv4oeJvhz4v4PoL9GNYl0P8W
Q1jAKoPuAu/JqDiewr41ynee8V4NevY5V6OgJ7B/nCif50wfCNo/MU2fG1jj
QXq2pMr3aayztIm7pVT3X4LMz3nViTqbsHemsst9i2nGVLdOyrelqnmb5etB
NdhnP5ch9icQy177vte1nkDepxPh5GHkrT2TzxfAvwr+XfCv4u5x3J3C9yTr
APpz0GPYx7G+AP17TrzunGo5Yf8P4+7LUbmYxLqYqC6tiXK/C/QHUfXg95VE
doaC4rlietpn9JN6vjPNvC7avw35GftMzJ4yfo7D582I6y3QFdA7jJlV4K3I
lIMzsP8e+NdywtGg7zK39IM+fJYon8xrj/PNep3F3XcQ53Xw58D7Na+eersg
vdTJOjWCcHgnL1/pD/d50/Pm0/9f8rJNuzex/4i1B/TJRN/EDPFzy7bmfW9/
0E4+6WvA9Q9F2eSdG+bzfM53Wec535+B7PdFxUDZm7Z7CbzLRflBrNE31pf9
zL4fC6pBt2OtQOdsTj3alyg3HTnpu2FfeKfuvKxBHdZmivdYqp4nPjljHsnU
1zyPlmEda1i9WBvA25gJs+RVzadu7tRPfsV1Z09VfM46Pus5SexQx11jjHzi
NYPuh7Dex1lzIvw0uXfJ6/Icqlk/be50XDtsl/J/IX/rEM+SRH1K+6PGBOUH
sBY8c1sS4Yb5o07inTiln++mmnHso69Ar85Uo17HTl1HYeexTP1FjK80zhkP
YyZu6ftJz59hzxPmmbOINWZ9W+0LfZuG/1NFYfOPgmY3e70/kRxlVsDu8ni/
/s8nqjvr328ZxtWbKLZV9DNKrpgp1/RrKXjLouTOeFYSq1X3HzFXdh6Jk3bI
dkb1bs3vDLHIOXzOs6jF78CA8z3sGlX+hxfWpslznvXvMQZGoPtQQb4sOF7m
tuHZQ8x3ux+I6zWQXx2FfcbJXNDm2qj3j2/fYl7oD2cb5wnry7fuW793Q46R
OOHs7/TbQbpsP08YK8TJ4VTz6LrfDsZDWyXTlG8xxkYc46RnKfvsw1QYaEDH
hqgYB/3OMF//oP5/F6WP7/l553ojZNdH6eUsueV5wjqxXvS/A+dLC3rbmesx
439Tph5m77Ff2GOMhXUrGYev4e7BqBk861myWMMWx9LpPqv77SgZ56N8Xwqq
Uxn7pqj3YbFmY36PFox19lC79fMfqNk5HLAMc9JsnPQZSx2uI/E07ti4tjl2
5uO0e4192Yj39ZxwvLWof6/tnutdnuf8V6pG2WJflvz2/AfAcUHH
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlFlvjVEUhveRs893nO0/SFXHU6p0Mg8Jly5cGqLGtCltFakWNdyIqaeH
1pUYQy/QiJ9AaKtnaJULQgg3gj9g9rxZLt7kfb+99pr2Wl/Jzs5NHbOccxtB
HAxEzqWSzmX4eMs7V4VeHpxrh+8H39BZMALvw2Y+/EfKuRL0DlCBbsK+Fd4G
GvE1w/knnJ/Bfz9nMbAh5txDzpfBl4LTzu4oViU+9nmLWQr/zv258M3gIrob
mxvwm+AY3IH1+HuAHk44tw7d7c1nPfGnuFNAj5Lvas6Og1F0p7e7q0CHNx9P
uf+XeGXofme5rglmq5yr4BlsXsK38e05to/Rbdi+ocYn8IPBcuvh2zC8QPwc
ehKsRB8B97zFPAyfQ4692N5FF/BXBB/w9R5Uc/ckNo84OwDS6LXoLm81Bu7m
sX+HbTs+ssT/jS5Vfeg6bBeBFvg8vi2ED2Lzivxb0LO5fx6f17y9+QV4c7C3
uw6WwOvBLvhuZ7GVk3JRDsXIalRtef+/1ynLXT0f4uxysLsT6F54D/gcsx7k
sa8hhxFij6MvoccSNhtfsEknLYZ8y2YS3xn0bfRbYkzDG5L2tkW+/eL8Z8pq
VQ/G4RMpe5vXcZsdzbhmWzO0GF2G3uOtxnORzbRmWT16wd1p8DFuM5xLWQ6K
rZ6rlqPgvreaGoO9gXq/l28NwXqqXirGCvRQwmZLMy7fqkG5K8YhvU9ks3DH
29uoBuWuNzrF+YlgvdFMaBd1R7bayQXwNGjmfLu33cwG6712VL37Gtnuqoc5
+CDnY/At5FQJLwdN3N+qGdXsB9s9zdAAuVcEO5uJ2S7UBIulnZAvvaneUj6n
iF+XtF3QTF/hvDZps6qd0G538e2qtx0XPxuZb33Trj5L2G5rZ1VbbbDZVY3K
tTrYLirnvmD/LP2r9CZ/8F8e2Vtop/8Bay+qtg==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.443771831267514, 0.318832584726318, 0.74075530984392], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmGmQVdUVhe9r673b/W4qqSQVZlSUgAINNN0NNKOCyEwzKIIIKGESASGB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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtllls1FUUxv8zOvOfmesSNQoFChSkRbqvtIhbRaCllFIMbrikLpRFiFGR
TRQXDERqKdAii1uiQdQoiW8+uBZoaRXrk4kafTAk4vYgS+L6+/L58GW+c89y
7//cc86dgo7V7auSURS9AM4HdSGKCuMouiERRd+AqZko2paOogH46xjORT8H
7Md2TSqKpqEvAXPRLWGtDF05uAvdzciTiTUF+Uv860AVtruIdxL+Dj4z0U3F
ZlbCa2dzUbSStRfxHwuqsa8FrdhuId50+AVgB/J7oA7fZuzXYbsB9BC7Ev3b
6IZAPXyP9kd3GLkKXgPuQ57JfheiX4N8UL7o++BJ0ILuBGsjnOcr8G7SZ6hh
v9nB374WXAWfBu6AL+J8s+BPgfeRH5Ge+M9zpuP4vgHy0U0Ei9GVs0c98eYj
b0TeBCqwLweHsH0LlME/wX8t/E1Qgf3ybBSdxncBKM74DpT7Zp0v45wpV/OR
F2N7OT4l2PbrPtmrCLRHPvMS9Hnoy9AdAfcjFyDXwId0X/CVrJ2FLwQh4zPq
bN2gFL6TeIPEOojcge0EfKqxPQ6y6DOgk/2+Rr4bfT76KvggqISvYO0MvA1M
JlYBqITfSsw6fHv5/uGk72Bn2jWk2mkBt+E7ihil2H8GxsPvyXpv7TEO+crw
/15gO/LDqhd8v0v4bLmMuc44Ef29WX+7vuEEd/9FzrlUTU1C/wD64YRztJ/z
LAi+u5fxH8R2IOdv1Z3VwftVb9id0xn5PUWM7+HrsR9CP5xzLrXWhm0D+s3E
exL8HNtHtj+g/wfbf0Fhyj2q2pmOzeMp19B1yNerJlPu6Wvg16rHkLuQa+FT
Ytf+t6AIXh/MG8EAe/WBP+AbONMMxQ/WLSfGp2n3mHpLOdwHbwnuvZeIfx78
puDab8WnhLMWgyb4L2Ab+72G/iT6BuQydKWgGf4bqICXg3nw30EP9k9gfxj7
buUrdk+plzRz1mtvcCjlGbIL3gOakq7JZ9OeYZpdc9T/xK7MuXdUc7qb2pxr
W3dUA6/OuRZV83tZawru9QPsfzGxLgKvRM7RFZznlqxrTz1Wie2f+F8N78Bn
DPo70R9NuMdGI9+eda+pZkdizyjNprakZ+VqsC/lmdnF/q8i/5hyz7bDFwXX
hs7wE/4fYtMI/wD7j+HbwQj8MfUntiXBs0kzeZ2+JbgWlLNC9i7KODezWduI
7kjataOaeQb56eC71Mx5jv3GKR66vaop+Edp++Yjb0E/VvMFvkf5Rx6DvBDe
m/Ld7Ai+e93RJvjRtO9WNatZ3BBca5rJm/WtenMSroFH4d2xe0FvwEPBM0iz
5wD63WnPdM3yUwnXWnfab5FqbmFwz6pX1WOtwTWs2lUPaXarB1T7muENxJqR
8d3ozVsVnGPlVnf0YHBOlAu9Wb3In8eexZrhK4Jzolz0gWXBOVEudoPxyGNj
vwWqkQnIW4l/LOE34jJ053LOrXJ+Gt4Z7DsKlKLvzLq31EP9ac9IzUbNiBL0
S9H/mnAP/oX/3zm/7apRvZ212KxP+Q29NPYbrLdXd1iOfGNwL+pNLENelnWv
qkdHo8sLPotqUGePyVdX0t8wL7iH1DvaQ2/xsbTfVr3J+m8xKXjW6z/GUtVb
7FpWDNVyRXAvqabPcLblwbnMA5fEXhNXjVUH14RqQW98MfrG4FpQDamXtsb+
L6Ke+g+pMxB0
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.544009052112523, 0.493118041210531, 0.881650849739867], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHm0VmUVxs/9zvDd7zuVdgXngVkELvMoIFwmAXMAuSKgS9PKWSvXspVR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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlneMVVUQxu/ue7z77rtWepXe6+7Sll6WJii9gyJIbyqJRmkLC7tLL0tf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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.622952148069196, 0.622572951001141, 0.910846752392722], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuwVlUZx/d533P23mu/IhzO4aAIKXGnUkst5Y7KnSQgqyk0hTBqJk2U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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllllsVVUUhk/vvT29+14KFDqIgFJpSwEVFBRtmSljFaTgDMosmAiKgCAK
FMFiKVPigIQwqUAQkKlAGQqYmAiIiolvmjjx4INPoE8m+v38D3/O+vdae+11
9ln736d0xoK6+YkoivaAFPgyE0WTQxS9k4yiHxjYhb0bXIKfxX82jqJV4EBO
FC1mYgG+DuB37Ar8PzL/Ffh24m8w/2f4T6Aj/rfBhnQULVFO/FPJUYWvJRtF
Y8j5N/4u+LqCG9iV5Ktn/BH4a8RfId8W4k8TP5rxW8QEfBnwK3YZ8eOwa8Fy
4vvCe2P3AfPgxaw3gPmnmD+K+TeZMxLfKLAUfx/iVzKeoMZGfJdZbyC+oWAx
/m7MT2Hngl/wdyd+NfGD4IvwX025tquM7Y9cYwW+XmAO/iLmLyf3A/CX4XfA
N8E3wpvhL8Hfw24CJ+Cz4R9jbwPn4WfIuZn4zfCT8Ln452MvCN6LffgXYr8O
/oEfhxdjl4A/4D3gA7CrwULm3838fdj7wVfwc/gvsj8XQHOO33kCvifACuxq
/PnYbcFv5CuHr+Ad74XPxf81778V3p4a38f/LzmGk+tbxsbiGwca8Q1kbFTK
NY7OeA/07hMZG5vxO+rdJsGfZj93Mj9D7NrcKHoe/iyYg78Bvpy1l4Fp8E+p
YVjGPabe0prj8T0OqiK/w395GPj34O+J+Sf8ROxvuY2xEczfiP8F7FrmP8Va
k8GL8DWsdx5/K5hNrr2M1RE7CaxPOOcUYneTrwN8HfHF8BlgA/YE8GjwN9De
q6dHYLenhmVJ9+Au7AbiP2B+FhRqbdDI3PGgjPg8Ygr17cDg4B5Vb6oH7wvu
efW6vkkJtRWDC5HP4Exy3aL+u/BvJN8n5BqX8V4X4e+FvYr4rvj7MzaN+JuM
lcDXE98z+JvrW6unC5k/kbH6pDVjEnZdcK9o7CD258G9dVnfR/3DnNIc99gx
+FH1N/5rSddWkvVeqcbjOgvBvlZiWrDPgOtJv1NV8BnU2VNPH4IfDl5LOfdi
fxasXerxrdgfgZbIZ+o09qng3Mq5E3tH8FmT5undi7Lee+1BIXtTlPa3UQ71
4tSsv7V6sp++X9raIg3cIH0hx4SUz3gjvCbPZ7s9Mevhu6i/IOEz/0bwmdFZ
2QPfBB8Dr0v5zKsXn8n6LKgnV+j7k286dhtyjFV8sJbd1kDs6cQ35Vpz6uGD
iZ9BfD7x9wdrnrROGrlGeod/Jv62+BtUD3wWvF3CZ2NpcG06I4OkF+Tvm+sz
XYC/hrHHUu7hIuxj+Ctja0ZH+FF4j9iaXQA/Aq+IraknsFtia38vnsfhp2Nr
V0+ezfAzsc9ub2kN85fk+exKs4fAB4Mp8BheknEOzdUdpbVPxtYu1aC1m2Pf
HapBtZ6KfZep5k4Z16RapKGdM65Ba+uOOoBdy9g81uvI2FDsnXleW3ec7rZr
sbVYd5x64fvYd6F6Qnv1XWyt0Z5dJP5S1rmKdaexn+VgZMIaprvrSmyt1x1W
yvxuGd8lB3PcK9dja6l6pjv8HtAJ3yHQA7sCdMb+ApRjl4E7sQ9LE7ErQRfs
I+Bd1tvO+yRZ/0MwC94aWzs25bq3zsXWBvVYhth+wWspZ4A/GFybamgDfyh4
ba0hbX8zWLul8Vn8/YNrUY078D+ZtfaGhM/Kc1lrv87Mba1NuzZpru4ezVGs
7qDL5Esx1pTwHsbYuWn/G+gfJpn2P4b+LaThitU76t00p1r3Vdp3me70fPI9
HLw30sg0vDz4LpAmj4RvCdZS9UQVfJE0O+GerJFeaQ9S1nzdDedja63uCGm9
/rH0byXNH0bs2uCzpzNZGfxPoX8J3UEj8K8L1hKd0b/g36iPI3+zdvDh+l/I
seZWw98K1grdmYPgK4O1QBoyBL46+OxLI17Vv1fwv5ly/g999yDD
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.696649699470148, 0.740984409540678, 0.912526829573102], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmFmQXVUVhs/te/qcs09noEM6A52m6QwUo5byoKIiRCVJJ90kTYAkIpJB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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmdsl1UUxt//6Pv+7x/ZpYxCSwsoOBM/KUaWQaCFAgVkqIiAC5VNC+p3
Tdw4AGWUvVRQVlugFDBiHKBAjCaObyQmblA2+Hvy9MNN7vM+55473nOec8qm
z6menY6iqJGRZbyci6IB+SgaBjgPsSlE0QbG3FQUfZaJokFwi8B7sR2OTSH2
6+Mo+p35cuw/gdvJmI/9cezvZn4X42bs54KPtIqio4xC1vTBfij+1iZRVMX6
UYxX8Hc/uD7j/Zez9l3GbPw18e1N+ErWTMD2Cnz/xGeWbR38BLgVOdtPw6Ze
Z2UsAJ+E/4D5NsY88BfgpdiOxcf3GftbA7eKMQf+CN8e1vlyvs+T+CuHy2Lf
Abxe3+A68e0BbNuA12nOmkex/ZRvg5k3ct8K7nsB/9WJ99ReP8B3hF8IbsXa
7qwZCF6T+G31D1L4mwafhx+BfWtwITY94MaC24Kv4rs1/CrGT3DfgJt579+w
aQ9fxLdS5uOxbwW+DH8DtisZ7eEWtPjvhk0a/nn4JuYV2N8GvpVRhr8nwMXY
L8a+Lfa9sekKN6sFT4LvDF+T83l6igfX5vw25eBOip+c9y8B/8pdaxldmS/k
TbL4a4NNEbgSfzeBm8DHwD05QwKe3vJeI+HbM3+R991fwL+Gf0574a8G7gz+
rjPvzvo+rH8J3I55GptybLvwbWjimNL/2MeavsF7aq+Z4EewX59z/MzCfhfc
Ft7nH+Yn9P66b2JfnRltE59Be9fCD8D+XsYt+FsArmO+kfV/YnsU3Mz6gXpj
+K/41sB8K/xZ5qfgqxWfjGP4+5xxEq6CNePhj7DmEtw5viWc752UbRdzhhuz
XrMEXIJ936zPNB+cwE/H9y/gCeAf846V4+Cluh/85Kxz8EPwZvz/Df4SvB88
FfunwKfBwxLnrPKvAbwZfhP2f2l/8ArpB/gP8CHFB7brWDMGvII1Y/E1hvEM
Zy8GX4H/D/sceBljKv92UeK7dWJNBn5czm8hTXmctbsSa4lyuC6xpkhLFBPX
8NWOM6xOOSZnws1QDCp+09aW94LPJs2QdqwEN7fgKvwdBJdjuxA8BnwoeG0N
eDS4CdwrbZ/vSw+Dc18aIq2sCc59aWYG+7Jg7WjPOM/5rrJmecp33s59H8vb
tzSyFtvFjIuKT9b/y7fLOb+N/vkV5XJw7iunmlm/gT3u5O7VcJf0lsG5rpwp
gOulf5KyhqXApeC1zNsxNoFH5f0vO7PnRdZfZ78VcIGxBL6H9C1jH8PBbwXb
Pg2uBG8Fl6RdM5TrbYK1SueT1uXBVRmfR7Vid3AuSWOlRQXBuS9NUu3ZEhxL
8ictzARrk3hpUQ/wlIw1SFrTGTyxRY+lRR2CtU//X1pVDJ7cYq9a8FFwbKsm
9Gd+T3CtUs4qNwcF565yVFpwIDj2pQkjuO/b4G5px/BI8A5wadr+RoH3SE/S
vl+FYjk41p9NWUv2BdcmaYpy7ePg2qmcUy5tD65Vyqnbmd8RrMXSsMP8iyHB
WvN11rWuMVg7dL5+qr3BWiYNl/ZsDK7l0qAG1tcz6lJ+k4PSSvgZGWvud6pP
wbX1DHietD3vXJR+DGN+gJgbqVgDjwAfBI9SnKZdS9cGa53yoZD5C3BHsq6Z
k7CfmDenmJnCfHLe/1pvuIP3mpa39ioHxzGvzvvtuoPHM9+eeK6cVe09Fbv2
qwZ3ZL/rsWNBGlAE7oJNWdYxUhJUVBz7eoNcYo3U3fQGu7DdnXftVQ6cjf0G
4ofwLU6ssdJW5cQ15hdi54pyehtvcbiVc1E9y3342g+ujJ3TPdk/nVgLlIMP
wT+Yd+woZnqrHsP3y1qzuyvW4HtlHfODpV151w7VvNHMq/LORcXk6/BvBP+L
w5FrS1HO2qsao7OciF07dSb5bohdq7WHerNOOfd26tFUW6Kca6lqjLRcNUS1
Q5quXud4bK1Sz6NeIg6u1eop1CsuC+7l1DOqNqwO7rVUIxQbXYJ7CcXITs5z
Mee3bsy6ljfG7j1U0/Uv62P3Gvqnij3VSNVGxeCr+HotODbVE02XduWt5dLs
n5k3B/di53Q+9QPBvYBqrGL729hvpxhX77U3dq+mHqwDeE/s3qk4695ld+xe
Rj2MauXp2L2jamYpuAmf7Qpck9Wb7YvdC6lHUy+knkK9hHqiELyHfEsz/wdN
i2BD
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.76956071699243, 0.849308875968127, 0.910926492997337], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuwl1UVxt/zXv/ve0DFS0EEZM7Ihz41kxUVklMiVlwE5SqXQNAsQZua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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl1msVEUQhs/MnDln+lxQEBfAICqJPPjkgzG4BI2CoOyI7Fy4bCqyqDFG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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.835746120081963, 0.871379944325969, 0.881275491724125], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFWGtwlsUVfr/v27zv7n620MYbEFDp2LGtdjrVls6ogzO9t44/kIqQWqut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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl2dsllUUx5++b3mfUT+YgAsCERONA4wRFBNB/OI2MQFkOkGGoqCIbFS2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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.901931523171496, 0.89345101268381, 0.851624490450914], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlklsVlUUx6/vG97rY1JXqCzEuFAXSFwYhIQgoCYWNibGTQuUsVKFSlFA
qi1jNLJwZVQIiGNQimEUFRlUJoUIoQO10IJQKC10oGVySPz9PcewON/9v3PP
vWc+9xs6de5zc6IQwjUoA33dL4QLcQilMG+wXsqHsBzckMAHP4LQ3/A7wCvh
1xaEUMR3G/wWZK7AHw4eDC6G/yEyLciUgLvgT4TfB14GfwK4DfwjeDV668En
c/Zdkw2hgvUFZHrhV4MLwefB+8D3gSeBV4Mr0xB+AL/FmSXgaugd8B5425F7
ApkF2HUejJpwGdwMfhR7hmVsxcTQCr8e/sN8b8mar1wRtmXNJ64NFcgcQAZW
mAfek1jcPmfzC2gjG9/C+w4aE5n+HW6DYjQAOhksFgOhpmB+6twI1jUY2IrS
JuIwhHP3JubjTXjXY8vFMXjz0Y1I+CUxm2TPLfb/hFZE5tPWrPklfzaDEQs/
J2a3bFYO5Kd8bEssRopPPRc3QJ3gTvgL4fcH94EXgQeAizi7k7MDwauyhkfC
m8H+zLz5UgnvqZz5PIF1ItSD/Fj26sDt7vss5KdEdm6Wn91ELGtSi/1c1h0Y
uRi8nXUW3xXZ22cmRxaPI4nF5BZU6LqejAxfAl8PprfT13rHJx3LLsVdtv5/
boLfU4W+p3OWs93sNYK7gtXIQ14na7HrFGAv+6+BLxD80/BP5e1b+Ci8GtUh
Z46DN4H7gw+DN4LTjMXtDagxWN2/mbU6aeLu97hnHd+/o/8j1m7433DuUGz2
y661WbNtXGR4lK/rHOu+Lq+98ZHdM5q1Axu2ctdd7C1ETzvfzcicY90MfxD8
Rnx5mb2GYOuxAsMN8F/ku07xBJeB64Plv9J90f6hApPR/pECk3kd3AVuAe9n
/QpdBejqBm8D352xulLOa5EpR/6V1O69iMwWZO7MmB/r3Zf17H8MbeD7E9ZP
oS/B0yLLn+IzPbJcKYaNxO+ZxPaG+1xQvy/nZ2lqNa761exRPSjPzyZ2viq1
upR9q1O7S/coh5p/8r0d+//y3lRfanaqN9Vb7Yn1l2bYu6nNsc9Yd8fWC805
m2ea05ptb/ez753SH1vffQ8+GFvdfoDM2dh8bWZ9P7W4rGE9F5vf6tdV3r+q
X82IvNd5NXQqWL1XOVYMlnkcmvOWM+VL86A7sZkwT32Z2pnTeash1c8udM5P
7d6jyB6G7s9YP7f5PFSvqW/Vp8rjeM/L7MjsvQn+jXPHoQc50+v13eHzqs77
V/1a6P2rmh/nuT7LuXPQY5wtjqwPFDfZ8qvb8xPrfmgIuM974zLrRXiXoMdl
K75sIv7PRzbfpUc2l0QmP9rt1vkTwXIpuTPB7JDdg3z+aB5pBmqGzfA5pnum
uF69wXqXVW/lkem8qj6KLLc3fH5Ocb1TXabTz+jsWdVqzt6fdp/bxe57Lz5d
g0ZnLE69bqPm00upzagy13mHz9Uev1M9Wppan2puzU5tdrUQn5p+ZssV+F3Q
SPgHYnsXFYtd6jNocMbsGuu2TXLblFPNxTmpzUbZoFjJL9VBt8d2aN76QHfs
4/6VqeV1RWq9MdLrvMj91f3jPPdjPG/Kyw33W3mR/iK3QXGd7Lko9vOj/I5J
LqO5+Gpqs1H573P7zsTWh//lKbFYK856V6d6jjQ7NMe73RflKfK6K3G9HYnV
n2pPvkpOMj3cf9X/F2n+LUhtBu5NLL6K7WR/jzUzNZtm+Nuq/NTGt+tliedY
+nW/4tmaWM+oX9TzD+RtVmhtyRlWfS71GlVPNMR23+mczazI67bM/dVa7rr0
/25j9vb7P9P/A9yDzrGJzc0TifW8+v1gYv2p3tQMLvU5fDG2/6yl7ku56+rU
/02oymfHdO/Lf+ANTWz+/hHbTNSe3r1Fqb19qodpPjfasOua/1+97j6oF/Qu
LU7tbdL/1QrvwZ7E6l41v0FvTmr/veSjZEZ4v5a4fJPmp9v5L9NHroo=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlUtsVVUUhnfubXvO2bfl5QSVRCAO1AEaB0YxIVBIUAQHJsaJmogmIDGt
imgMKCCCGh048ZEitRpfIIiKAS1aiikPFRLQ0iftLZSW0iotysNn4vfz36Tr
dq29Hnvvtf/1n2mLa++pyYUQXkLKkK0xhH+TEDawuIOFbdjFNISb8iF8hV2H
3Yf/YfyPIh9jf4jcir4Vf2UhhM6KENqpNbc8hE34ThL/CP5lSAv6Qup9jz4L
eR3/ftZmo79MfoH8DvLbyK8m/xP8H0XX3oaswLeI/HXEkxZO8LMRf1POZ/oU
vR3/DZz3S+Lvx78duwp7CBlDP4fMRK8i/ymdLwvhOPoa4iex/xXIIvZeiBTZ
b1X02fqIeQXfHmrexl6vsnaRWvWsFYmdTmwzvheJfw3fTGIewv6cmHHsN4ps
J+Y+7DX4uqm3hNgW9m9FX07OeGr1ENOLfQc1JxZ8Bu29AHsD8euja9+OvIm+
N7H+LvldxNaythK9g5xd+CazXx3+51jbid2auReKqSo4R7HzqD8O+zh2D/Z8
7J3k7kKqc75zI/nFzHd5nvwJxPcSXyT+TuKPEnsEQQ3XErMHvRmZjM5f+Bn9
l9Q+0sIo9b6WUHs18iD6PPy70Tcix7Qfshi9naJt6HXs2YbegVxF7OzU+Yrv
x1/P/ZtzxtwT0T3Q3VchD6DPJb4x5570psa8sH4jZ2rU3sicnDFZS/x/pXlo
QKbhm4rQgstr89G78HeWG4MLsHuwu8p9hlb0DuTenO8wHf8wF19fqrecvb9N
jL3VyIXEGBZ2hfHfE/dcvRZmatD/FMbQ65F/0O+mZkvONQexT1D/BfSlyMpo
DAk7k7jfs9FvpreaiH2S3M+iZ5t/YRO+71jrxd+DDFLrGPZ1ec/Ub8IG9oy8
OWFZ9MxqViP2Qc1jwW8jzCzFH7HnYGf4fyL3x9RcMxV7n86eGhtT+OlPfSad
5Wbsx6I5RVxSwG5KfUadTZjS7F1ibVbeM7giGsPC7njWnomeKc3SBM0gsWeR
ymAOOIM+hBASbsGuiZ4JzUIl9gHu8w52d7kxsS565jXrmvEL6OdTc4nO8EE0
54nrtpT57u3qYXAP3o/mUHHn5jJjWRi/jG1kGGCNZH4bccAg+unMZ9dMjqKP
ZX5LzdyW6JnSLH1BvW+odWVp3oXxt6M5V1z7Hv7d6FNSz4pm7O/EGG8o4Udv
+2uFZ0FvfD4x5oQ14VFc2Z+5t+LMg+g/ZH573fEU+QOp3049HUYfSd1b9fwQ
+uHUby+OEPfty4wNcaCwtzaaW4XB/dgHUmNDmBlIzNHiZr3JEXKPZn4rfXMO
oR/OjBV9Q8Ql5xLPju6r2T+bmGvUH2F7gPtenzfG27A3R/dSa+LuJ6NnUxz+
Fnpv4l42IE+T+1fi3mkG9e24i5y9OX9DzuC/JnX/NLMXE+e0lGxxxxvUrC8z
h4xUmCP0HuIXzfal0rwrR7HdibleOeIanUlnEeeMJcZAY4lP/0j85rq7elAt
bkM6gzGib+Pj0dyob6S4tLvCueJUcfPVJbzIblLvor9V+uYPJf5G69usb/op
7NOJuUc9aEfvo97anDlQvRiqMJbUk/8BQ2U4WQ==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{1654, 2038, 8}, {1903, 2042, 397}, {2076, 1979, 561}, {
         1755, 2112, 218}, {1979, 2035, 561}, {2040, 1902, 395}, {1902, 2036, 
         395}, {2073, 1977, 559}, {1977, 2032, 559}, {2041, 1654, 8}, {2125, 
         1653, 707}, {2080, 2081, 671}, {2113, 2114, 697}, {2135, 1711, 
         721}, {2045, 1903, 397}, {2112, 2074, 218}},
         VertexColors->None], 
        PolygonBox[{{2035, 2135, 721, 561}, {2114, 2073, 559, 697}, {2074, 
         2076, 561, 218}, {2036, 2125, 707, 395}, {2081, 2045, 397, 671}, {
         2032, 1755, 218, 559}, {2038, 2040, 395, 8}, {2042, 2041, 8, 397}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1653, 2125, 2036, 1902, 2040, 2038, 1654, 2041, 2042, 1903, 
         2045, 2081, 2080}],
        "1.6`"],
       Annotation[#, 1.6, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1711, 2135, 2035, 1979, 2076, 2074, 2112, 1755, 2032, 1977, 
         2073, 2114, 2113}],
        "1.6`"],
       Annotation[#, 1.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1760, 2007, 2990, 2008, 2004, 2989, 2015, 2005, 2970, 2006, 
         2971, 1992, 2231, 2230, 2229, 1983, 2953, 2688, 2689, 2237, 2973, 
         2238, 2236, 2240, 2239, 1650, 2244, 2245, 2243, 2247, 2246, 1651, 
         2250, 2251, 2249, 2253, 2252, 1652, 2256, 2257, 2255, 2259, 2258, 
         2730, 2729, 2262, 1765, 2261, 1766, 2263, 2731, 2037, 1768, 2079, 
         1769, 2039, 2126, 2043, 1770, 2082, 1771, 2044, 2732, 2270, 1772, 
         2269, 1773, 2271, 2733, 2734, 2274, 2275, 2273, 2277, 2276, 1655, 
         2280, 2281, 2279, 2283, 2282, 1656, 2286, 2287, 2285, 2289, 2288, 
         1657, 2292, 2293, 2291, 2295, 2974, 2294, 2691, 2690, 2692, 1984, 
         2232, 2693, 2694, 1985, 2710, 2011, 2709, 2010, 2119, 2009, 2715, 
         2017, 2713, 2016, 2714, 2722, 2721}],
        "1.4`"],
       Annotation[#, 1.4, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1761, 2022, 2991, 2023, 2020, 2123, 2835, 2834, 2021, 2972, 
         1991, 2235, 2234, 2233, 1990, 2704, 1981, 2684, 1759, 2833, 2683, 
         2228, 2544, 2543, 2682, 1758, 2832, 2681, 2227, 2539, 2538, 2680, 
         1757, 2831, 2679, 2226, 2535, 2534, 2678, 1756, 2830, 2677, 2225, 
         2531, 2530, 2676, 1980, 2675, 2674, 2224, 1889, 2527, 1978, 2034, 
         2033, 2116, 1886, 2075, 1976, 2031, 2030, 2115, 1885, 2072, 1975, 
         2673, 2672, 2223, 1884, 2516, 1974, 2671, 1754, 2829, 2670, 2222, 
         2513, 2512, 2669, 1753, 2828, 2668, 2221, 2509, 2508, 2667, 1752, 
         2827, 2666, 2220, 2505, 2504, 2665, 1751, 2826, 2664, 2219, 2501, 
         2500, 2663, 1973, 2687, 1982, 2725, 2727, 2726, 2728, 2024, 2124, 
         1988, 2712, 2014, 2711, 2013, 2122, 2012, 2718, 2019, 2716, 2018, 
         2717, 2724, 2723}],
        "1.4`"],
       Annotation[#, 1.4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1661, 2049, 1658, 2127, 2708, 2707, 1908, 2696, 2697, 2695, 
         1986, 2954, 2572, 1715, 2802, 2571, 2159, 1778, 2296, 1907, 2570, 
         2569, 2157, 1777, 2290, 1906, 2568, 2567, 2155, 1776, 2284, 1905, 
         2566, 2565, 2153, 1775, 2278, 1904, 2564, 2563, 2151, 1774, 2272, 
         2562, 2561, 1714, 2799, 2560, 2149, 2268, 2267, 2559, 1713, 2798, 
         2558, 2147, 2266, 2265, 2556, 2557, 2555, 2554, 2145, 1767, 2264, 
         1901, 2553, 2552, 2143, 1764, 2260, 1900, 2551, 2550, 2141, 1763, 
         2254, 1899, 2549, 2548, 2139, 1762, 2248, 1898, 2547, 2546, 2137, 
         2242, 2241, 2545, 2796, 2795, 2136, 2686, 2685, 1897, 2117, 2118, 
         2078, 3014, 2077, 2046, 2083}],
        "1.2`"],
       Annotation[#, 1.2, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1698, 2067, 2109, 2110, 2121, 2120, 1959, 2699, 2700, 2698, 
         2955, 1987, 3012, 2783, 1707, 2982, 2497, 2981, 2498, 2196, 1875, 
         2499, 2784, 2502, 1876, 2198, 1877, 2503, 2785, 2506, 1878, 2200, 
         1879, 2507, 2786, 2510, 1880, 2202, 1881, 2511, 2787, 2514, 1882, 
         2204, 1883, 2515, 2788, 1708, 2517, 2518, 2206, 2520, 2821, 2519, 
         1709, 2521, 2522, 2208, 2524, 2822, 2523, 1710, 2789, 2525, 1887, 
         2210, 1888, 2526, 2790, 2528, 1890, 2212, 1891, 2529, 2791, 2532, 
         1892, 2214, 1893, 2533, 2792, 2536, 1894, 2216, 1895, 2537, 2793, 
         2540, 1896, 2218, 2542, 2825, 2541, 2983, 1712, 2794, 2988, 1989, 
         2956, 2701, 2703, 2702, 1972, 2719, 2720, 2134, 3015, 1706, 1750, 
         2111}],
        "1.2`"],
       Annotation[#, 1.2, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1914, 3006, 2591, 2984, 2590, 2169, 1798, 2589, 1913, 2132, 
         2748, 2339, 1797, 2160, 1796, 2338, 2747, 2337, 1795, 2158, 1794, 
         2336, 2746, 2335, 1793, 2156, 1792, 2334, 2744, 2745, 2330, 2801, 
         2331, 2154, 2329, 2328, 1660, 2324, 2800, 2325, 2152, 1791, 2323, 
         2743, 2320, 1790, 2150, 1789, 2319, 2742, 2316, 1788, 2148, 1787, 
         2315, 2741, 2312, 1786, 2146, 2311, 2310, 1659, 2306, 2797, 2307, 
         2144, 2305, 2304, 2740, 2739, 2303, 1785, 2142, 1784, 2302, 2738, 
         2301, 1783, 2140, 1782, 2300, 2737, 2299, 1781, 2138, 1780, 2298, 
         2735, 2128, 1909, 2910, 2573, 2161, 1779, 2297, 2736}],
        "1"],
       Annotation[#, 1, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1960, 2450, 1862, 2188, 1863, 2637, 1958, 2636, 1957, 2451, 
         1864, 2195, 2638, 2639, 1961, 2452, 1865, 2197, 2640, 2641, 1962, 
         2453, 1866, 2199, 2642, 2643, 1963, 2644, 2459, 2460, 2201, 2645, 
         2820, 1747, 2646, 2466, 2467, 2203, 2647, 2648, 1964, 2473, 1867, 
         2205, 2649, 2650, 1965, 2479, 1868, 2207, 2651, 2652, 1966, 2485, 
         1869, 2209, 2653, 2823, 1748, 2654, 2491, 2492, 2211, 2655, 2824, 
         1749, 2656, 1967, 2493, 1870, 2213, 2657, 2658, 1968, 2494, 1871, 
         2215, 2659, 2660, 1969, 2495, 1872, 2217, 2661, 2987, 3011, 3010, 
         1970, 2952, 1971, 2662, 1873, 2194, 1874, 2496, 2782}],
        "1"],
       Annotation[#, 1, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1915, 2340, 1799, 2162, 1800, 2341, 2749, 2750, 2343, 2344, 
         2342, 2346, 2345, 1662, 2348, 2349, 2347, 2351, 2350, 1663, 2050, 
         2576, 1910, 2575, 1911, 2025, 1716, 2574, 2308, 2309, 2163, 2577, 
         2992, 1717, 2578, 2313, 2314, 2164, 2579, 2993, 1718, 2580, 2317, 
         2318, 2165, 2581, 2994, 1719, 2582, 2321, 2322, 2166, 2583, 2995, 
         1720, 2584, 2326, 2327, 2167, 2585, 2996, 1721, 2586, 2332, 2333, 
         2168, 2587, 2997, 1722, 2588, 2047, 2048, 1912, 2129, 2130, 2131, 
         2051, 1665, 2360, 2361, 2359, 2363, 2362, 1666, 2365, 2366, 2364, 
         2368, 2367, 2752, 2751, 2369, 1806, 2170, 1807, 2370, 2753}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1956, 3008, 3009, 2986, 2635, 2193, 1861, 2449, 1955, 2634, 
         1746, 3005, 2633, 2192, 2448, 2447, 2632, 1745, 3004, 2631, 2191, 
         2446, 2445, 2630, 1744, 2781, 2780, 2133, 1954, 2071, 2070, 1705, 
         2489, 2490, 2486, 2488, 2487, 1704, 2483, 2484, 2480, 2482, 2481, 
         1703, 2477, 2478, 2474, 2476, 2475, 1702, 2471, 2472, 2468, 2470, 
         2469, 1701, 2464, 2465, 2461, 2463, 2462, 1700, 2457, 2458, 2454, 
         2456, 2455, 1699, 2068, 2069, 1953, 2629, 1952, 2951, 2065, 2628, 
         1743, 3003, 2627, 2190, 2424, 2423, 2626, 1742, 3002, 2625, 2189, 
         2422, 2421, 2624, 1951, 2623, 2622, 2187, 1850, 2420, 2774}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1922, 2372, 1808, 2371, 2374, 2373, 1667, 2376, 2377, 2375, 
         2379, 2378, 1668, 2053, 2054, 1916, 2975, 1664, 1723, 2911, 2836, 
         2837, 2803, 2912, 1724, 2913, 2352, 2353, 2171, 2592, 2593, 1917, 
         2354, 1801, 2172, 2594, 2595, 1918, 2355, 1802, 2173, 2596, 2597, 
         1919, 2356, 1803, 2174, 2598, 2599, 1920, 2357, 1804, 2175, 2600, 
         2601, 1921, 2358, 1805, 2176, 2602, 2998, 1725, 2914, 2838, 2839, 
         2804, 2915, 1726, 2916, 2052, 1993, 2957, 1994, 2085, 2084, 1670, 
         2381, 2382, 2380, 2384, 2383, 1671, 2386, 2387, 2385, 1812, 2388, 
         2754}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1950, 3007, 2621, 2985, 2620, 2186, 2419, 2418, 2619, 1741, 
         3001, 2618, 2185, 2417, 2416, 2617, 1740, 2029, 1949, 2949, 1948, 
         2950, 2066, 1697, 2908, 2909, 2905, 2907, 2906, 1696, 2443, 2444, 
         2441, 1860, 2442, 2779, 2440, 1859, 2438, 1858, 2439, 2778, 2437, 
         1857, 2435, 1856, 2436, 2777, 2434, 1855, 2432, 1854, 2433, 2776, 
         2431, 1853, 2429, 1852, 2430, 2775, 2428, 1851, 2425, 2427, 2426, 
         1695, 2903, 2904, 2900, 2902, 2901, 1694, 2064, 1688, 2980, 1947, 
         2108, 2107, 2706, 1739, 3000, 2705, 2184, 2415, 2414, 2616, 1738, 
         2999, 2615, 2183, 1843, 2413, 2773}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1928, 2390, 1813, 2389, 1814, 2391, 2755, 2393, 1815, 2392, 
         1816, 2394, 2756, 2396, 1817, 2395, 1818, 2397, 1672, 2846, 1819, 
         2845, 2848, 2847, 1673, 2055, 2919, 1923, 2918, 1924, 2026, 1727, 
         2917, 2840, 2841, 2805, 2920, 2921, 1925, 2842, 1809, 2806, 2922, 
         2923, 1926, 2843, 1810, 2807, 2924, 2925, 1927, 2844, 1811, 2808, 
         2958, 1728, 2959, 2086, 2087, 1995, 2976, 1669, 2088, 1674, 2850, 
         2851, 2849, 1820, 2852, 1675, 2399, 1821, 2398, 1822, 2400, 2757, 
         2402, 1823, 2401, 1824, 2403, 2758, 2405, 1825, 2404, 1826, 2406, 
         2759}],
        "0.4`"],
       Annotation[#, 0.4, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1946, 2614, 3013, 2613, 2182, 1842, 2412, 1945, 2612, 2611, 
         2181, 1841, 2411, 1944, 2610, 2609, 2180, 1840, 2410, 1943, 2948, 
         2947, 2819, 2885, 2884, 2946, 1737, 1687, 2979, 1942, 2063, 2062, 
         1693, 2898, 2899, 2896, 1849, 2897, 1692, 2895, 1848, 2893, 1847, 
         2894, 1691, 2892, 1846, 2890, 1845, 2891, 1690, 2889, 1844, 2886, 
         2888, 2887, 1689, 2105, 2106, 2003, 2969, 2002, 2104, 2968, 1736, 
         2967, 2818, 1839, 2883, 1941, 2608, 2607, 2179, 1838, 2409, 1940, 
         2606, 2605, 2178, 1837, 2408, 1939, 2604, 2603, 2177, 1836, 2407, 
         2772}],
        "0.4`"],
       Annotation[#, 0.4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1676, 2089, 1996, 2960, 1997, 2092, 2090, 2091, 1999, 2965, 
         1998, 2966, 2095, 2096, 2059, 2765, 2764, 2763, 1935, 2027, 1732, 
         2028, 1937, 2932, 1936, 2933, 2060, 2103}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{1686, 2102, 2771, 2770, 2769, 2001, 2101, 2099, 2100, 2000, 
         2766, 2767, 2768, 2097, 1682, 1731, 2931, 1933, 2930, 1934, 2058, 
         2056, 2057, 1931, 2760, 2761, 2762, 1730, 2094}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{2977, 1929, 2928, 2809, 1827, 2853, 2927, 2926, 1729, 1677, 
         2864, 2865, 2811, 2812, 2813, 1681, 2870, 1830, 2814, 2815, 1733, 
         1683, 2098, 1684, 2876, 1832, 2875, 1833, 2938, 2939, 2935, 2937, 
         2936, 2877, 2878, 2816, 2940, 1734, 2941, 2879, 2880, 2817, 2942, 
         1735, 1938, 2943, 2945, 2944, 1834, 2881, 1835, 2882, 1685, 2061, 
         2934, 2873, 2874, 2871, 1831, 2872, 1932, 2869, 1829, 2866, 2868, 
         2867, 2929, 2093, 2963, 2964, 2962, 2961, 2810, 1828, 1930, 2978, 
         1680, 2862, 2863, 2859, 2861, 2860, 1679, 2857, 2858, 2854, 2856, 
         2855, 1678, 2977}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  FrameTicks->True,
  Method->{},
  PlotRange->{{-5, 5}, {-5, 5}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2361302]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->380039371],

Cell[TextData[{
 "The first 10 Gram points of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->705342219],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"gp", "=", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"g", " ", "/.", " ", 
     RowBox[{"FindRoot", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"RamanujanTauTheta", "[", "g", "]"}], "-", 
        RowBox[{"n", " ", "Pi"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"g", ",", "0", ",", "7"}], "}"}]}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "0", ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->576943293],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.`", ",", "11.938675914006556`", ",", "15.605462399468909`", ",", 
   "18.59890310677914`", ",", "21.239189157758148`", ",", 
   "23.65226237590345`", ",", "25.903076223957676`", ",", 
   "28.030338836040347`", ",", "30.059330729785664`", ",", 
   "32.00764857066913`", ",", "33.888129963721305`"}], "}"}]], "Output",
 ImageSize->{368, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->236861531]
}, Open  ]],

Cell[TextData[{
 "Plot of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauZ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauZ"]], "InlineFormula"],
 " and Gram points:"
}], "ExampleText",
 CellID->686929402],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"RamanujanTauZ", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "30"}], "}"}], ",", " ", 
   RowBox[{"Epilog", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", ".02", "]"}], ",", "Red", ",", 
      RowBox[{"Point", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"{", 
          RowBox[{"#", ",", "0"}], "}"}], "&"}], " ", "/@", " ", "gp"}], 
       "]"}]}], "}"}]}], ",", " ", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1874864079],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVx388E/gfwPFtTugihGTyIz/KjK2Z+bGtz7uTitLPI5TSsdv3shLp8NWj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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  Epilog->{
    PointSize[0.02], 
    RGBColor[1, 0, 0], 
    PointBox[{{0., 0}, {11.9386759140066, 0}, {15.6054623994689, 0}, {
      18.5989031067791, 0}, {21.2391891577581, 0}, {23.6522623759035, 0}, {
      25.9030762239577, 0}, {28.0303388360403, 0}, {30.0593307297857, 0}, {
      32.0076485706691, 0}, {33.8881299637213, 0}}]},
  Method->{},
  PlotRange->{{0, 30}, {-3.03313932574782, 2.70982230803453}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->587425597]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->59755782],

Cell[TextData[{
 "On the critical line, ",
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 " gives the phase of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " up to a sign:"
}], "ExampleText",
 CellID->660591717],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Arg", "[", 
  RowBox[{"RamanujanTauL", "[", 
   RowBox[{"6", "+", 
    RowBox[{"9.22", "I"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->321237520],

Cell[BoxData[
 RowBox[{"-", "1.403720433652896`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->735737799]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauTheta", "[", "9.22", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1824064296],

Cell[BoxData["1.4037204336632279`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->173532680]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RamanujanTauL",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauL"]], "InlineFormula"],
 " can be expressed in terms of ",
 Cell[BoxData[
  ButtonBox["RamanujanTauTheta",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RamanujanTauTheta"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->1135033868],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauL", "[", 
  RowBox[{"6", "+", 
   RowBox[{"9.22", "I"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->2141585905],

Cell[BoxData[
 RowBox[{"0.0004030903798357173`", "\[InvisibleSpace]", "-", 
  RowBox[{"0.002390127652106469`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->148578817]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RamanujanTauZ", "[", "9.22", "]"}], 
  RowBox[{"Exp", "[", 
   RowBox[{
    RowBox[{"-", "I"}], " ", 
    RowBox[{"RamanujanTauTheta", "[", "9.22", "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->133747314],

Cell[BoxData[
 RowBox[{"0.0004030903798110217`", "\[InvisibleSpace]", "-", 
  RowBox[{"0.002390127652110629`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->264625804]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->283264421],

Cell["Machine-number inputs can give high-precision results:", "ExampleText",
 CellID->209624087],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RamanujanTauTheta", "[", 
  RowBox[{"100", "^", 
   RowBox[{"-", "200."}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2120192167],

Cell[BoxData[
 RowBox[{"-", 
  "1.31759397977539187373498`11.611265558929265*^-401"}]], "Output",
 ImageSize->{149, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->863719814]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MachineNumberQ", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->183924547],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->188766131]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->659341674],

Cell["Density plot of the argument:", "ExampleText",
 CellID->1033694472],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DensityPlot", "[", 
  RowBox[{
   RowBox[{"Arg", "[", 
    RowBox[{"RamanujanTauTheta", "[", 
     RowBox[{"x", "+", 
      RowBox[{"I", " ", "y"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1042162399],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->523966659]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauZ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauZ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RamanujanTauL",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RamanujanTauL"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RiemannSiegelTheta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RiemannSiegelTheta"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->23022]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->260766036],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->205879787]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RamanujanTauTheta - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 13, 42.7343660}", "context" -> "System`", 
    "keywords" -> {"Ramanujan tau theta", "Ramanujan Tau Theta function"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "RamanujanTauTheta[t] gives the Ramanujan tau theta function \
\\[Theta](t).", "synonyms" -> {"ramanujan tau theta"}, "title" -> 
    "RamanujanTauTheta", "type" -> "Symbol", "uri" -> 
    "ref/RamanujanTauTheta"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6900, 216, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->381534625]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 124896, 2594}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2255, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2878, 82, 1582, 37, 70, "ObjectNameGrid"],
Cell[4463, 121, 626, 19, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5126, 145, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5604, 162, 120, 4, 70, "Notes",
 CellID->20640],
Cell[5727, 168, 406, 15, 70, "Notes",
 CellID->36373499],
Cell[6136, 185, 266, 8, 70, "Notes",
 CellID->989328282],
Cell[6405, 195, 236, 7, 70, "Notes",
 CellID->31226],
Cell[6644, 204, 219, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[6900, 216, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->381534625],
Cell[CellGroupData[{
Cell[7285, 230, 148, 5, 70, "ExampleSection",
 CellID->181045676],
Cell[7436, 237, 65, 1, 70, "ExampleText",
 CellID->2111517477],
Cell[CellGroupData[{
Cell[7526, 242, 117, 3, 28, "Input",
 CellID->1106969264],
Cell[7646, 247, 177, 5, 36, "Output",
 CellID->1559155195]
}, Open  ]],
Cell[7838, 255, 126, 3, 70, "ExampleDelimiter",
 CellID->2117552439],
Cell[7967, 260, 66, 1, 70, "ExampleText",
 CellID->1604844189],
Cell[CellGroupData[{
Cell[8058, 265, 250, 8, 28, "Input",
 CellID->221678203],
Cell[8311, 275, 2429, 44, 133, "Output",
 Evaluatable->False,
 CellID->345386757]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10789, 325, 220, 7, 70, "ExampleSection",
 CellID->392378955],
Cell[11012, 334, 75, 1, 70, "ExampleText",
 CellID->1961420917],
Cell[CellGroupData[{
Cell[11112, 339, 164, 5, 70, "Input",
 CellID->1727370569],
Cell[11279, 346, 271, 8, 36, "Output",
 CellID->85729390]
}, Open  ]],
Cell[11565, 357, 126, 3, 70, "ExampleDelimiter",
 CellID->2039814240],
Cell[11694, 362, 70, 1, 70, "ExampleText",
 CellID->728990972],
Cell[CellGroupData[{
Cell[11789, 367, 168, 6, 70, "Input",
 CellID->757060269],
Cell[11960, 375, 253, 8, 36, "Output",
 CellID->591267630]
}, Open  ]],
Cell[12228, 386, 125, 3, 70, "ExampleDelimiter",
 CellID->538700578],
Cell[12356, 391, 113, 3, 70, "ExampleText",
 CellID->149558694],
Cell[CellGroupData[{
Cell[12494, 398, 142, 4, 70, "Input",
 CellID->1077103068],
Cell[12639, 404, 238, 7, 36, "Output",
 CellID->1731603313]
}, Open  ]],
Cell[12892, 414, 125, 3, 70, "ExampleDelimiter",
 CellID->358497447],
Cell[13020, 419, 216, 7, 70, "ExampleText",
 CellID->1574961894],
Cell[CellGroupData[{
Cell[13261, 430, 196, 6, 70, "Input",
 CellID->501119536],
Cell[13460, 438, 326, 10, 36, "Output",
 CellID->2032661665]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13835, 454, 227, 7, 70, "ExampleSection",
 CellID->416401570],
Cell[14065, 463, 239, 8, 70, "ExampleText",
 CellID->883919168],
Cell[CellGroupData[{
Cell[14329, 475, 431, 14, 70, "Input",
 CellID->1885070277],
Cell[14763, 491, 77573, 1322, 70, "Output",
 CellID->2361302]
}, Open  ]],
Cell[92351, 1816, 125, 3, 70, "ExampleDelimiter",
 CellID->380039371],
Cell[92479, 1821, 221, 8, 70, "ExampleText",
 CellID->705342219],
Cell[CellGroupData[{
Cell[92725, 1833, 487, 15, 70, "Input",
 CellID->576943293],
Cell[93215, 1850, 492, 12, 54, "Output",
 CellID->236861531]
}, Open  ]],
Cell[93722, 1865, 217, 8, 70, "ExampleText",
 CellID->686929402],
Cell[CellGroupData[{
Cell[93964, 1877, 600, 18, 70, "Input",
 CellID->1874864079],
Cell[94567, 1897, 2957, 59, 128, "Output",
 CellID->587425597]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[97573, 1962, 236, 7, 70, "ExampleSection",
 CellID->59755782],
Cell[97812, 1971, 390, 13, 70, "ExampleText",
 CellID->660591717],
Cell[CellGroupData[{
Cell[98227, 1988, 186, 6, 70, "Input",
 CellID->321237520],
Cell[98416, 1996, 192, 6, 36, "Output",
 CellID->735737799]
}, Open  ]],
Cell[CellGroupData[{
Cell[98645, 2007, 117, 3, 70, "Input",
 CellID->1824064296],
Cell[98765, 2012, 176, 5, 36, "Output",
 CellID->173532680]
}, Open  ]],
Cell[98956, 2020, 361, 12, 70, "ExampleText",
 CellID->1135033868],
Cell[CellGroupData[{
Cell[99342, 2036, 155, 5, 70, "Input",
 CellID->2141585905],
Cell[99500, 2043, 278, 7, 36, "Output",
 CellID->148578817]
}, Open  ]],
Cell[CellGroupData[{
Cell[99815, 2055, 253, 8, 70, "Input",
 CellID->133747314],
Cell[100071, 2065, 278, 7, 36, "Output",
 CellID->264625804]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[100398, 2078, 230, 7, 70, "ExampleSection",
 CellID->283264421],
Cell[100631, 2087, 97, 1, 70, "ExampleText",
 CellID->209624087],
Cell[CellGroupData[{
Cell[100753, 2092, 161, 5, 70, "Input",
 CellID->2120192167],
Cell[100917, 2099, 228, 7, 36, "Output",
 CellID->863719814]
}, Open  ]],
Cell[CellGroupData[{
Cell[101182, 2111, 110, 3, 70, "Input",
 CellID->183924547],
Cell[101295, 2116, 162, 5, 36, "Output",
 CellID->188766131]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[101506, 2127, 228, 7, 70, "ExampleSection",
 CellID->659341674],
Cell[101737, 2136, 73, 1, 70, "ExampleText",
 CellID->1033694472],
Cell[CellGroupData[{
Cell[101835, 2141, 431, 14, 70, "Input",
 CellID->1042162399],
Cell[102269, 2157, 19155, 318, 70, "Output",
 Evaluatable->False,
 CellID->523966659]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[121485, 2482, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[121804, 2493, 740, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[122581, 2524, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[122902, 2535, 144, 3, 70, "Tutorials",
 CellID->23022]
}, Open  ]],
Cell[CellGroupData[{
Cell[123083, 2543, 299, 8, 70, "MoreAboutSection"],
Cell[123385, 2553, 162, 3, 70, "MoreAbout",
 CellID->260766036],
Cell[123550, 2558, 198, 4, 70, "MoreAbout",
 CellID->205879787]
}, Open  ]],
Cell[123763, 2565, 27, 0, 70, "History"],
Cell[123793, 2567, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

