(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    157756,       3293]
NotebookOptionsPosition[    150966,       3053]
NotebookOutlinePosition[    152280,       3090]
CellTagsIndexPosition[    152193,       3085]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ChiDistribution" :> 
          Documentation`HelpLookup["paclet:ref/ChiDistribution"], 
          "MaxwellDistribution" :> 
          Documentation`HelpLookup["paclet:ref/MaxwellDistribution"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"ChiDistribution\"\>", 
       2->"\<\"MaxwellDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/ContinuousStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Continuous Statistical Distributions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RayleighDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RayleighDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RayleighDistribution"], "[", 
       StyleBox["\[Sigma]", "TR"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents the Rayleigh distribution with scale \
parameter ",
     Cell[BoxData[
      FormBox["\[Sigma]", TraditionalForm]], "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->14122]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability density for value ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " in a Rayleigh distribution is proportional to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "\[InvisibleTimes]", " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{
      RowBox[{"-", 
       SuperscriptBox["x", "2"]}], "/", 
      RowBox[{"(", 
       RowBox[{"2", 
        SuperscriptBox["\[Sigma]", "2"]}], ")"}]}]]}], TraditionalForm]], 
  "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", ">", "0"}], TraditionalForm]], "InlineMath"],
 ", and is zero for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "<", "0"}], TraditionalForm]], "InlineMath"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->23849,
  ButtonNote->"23849"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RayleighDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RayleighDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  FormBox["\[Sigma]", TraditionalForm]], "InlineMath"],
 " to be any positive real number."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->7997],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RayleighDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RayleighDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->29681,
  ButtonNote->"29681"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->242810213],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->838947038],

Cell["The mean and variance of a Rayleigh distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->29681],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4879],

Cell[BoxData[
 RowBox[{
  SqrtBox[
   FractionBox["\[Pi]", "2"]], " ", "\[Sigma]"}]], "Output",
 ImageSize->{46, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->209078]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17207],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"2", "-", 
    FractionBox["\[Pi]", "2"]}], ")"}], " ", 
  SuperscriptBox["\[Sigma]", "2"]}]], "Output",
 ImageSize->{66, 29},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->12101926]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25706],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->23849],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24075],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox["x", "2"], 
      RowBox[{"2", " ", 
       SuperscriptBox["\[Sigma]", "2"]}]]}]], " ", "x"}], 
  SuperscriptBox["\[Sigma]", "2"]]], "Output",
 ImageSize->{52, 48},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->76420828]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"RayleighDistribution", "[", "1", "]"}], ",", "x"}], "]"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->20124],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 147},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->499424010]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2884],

Cell["\<\
Generate a set of random numbers that are Rayleigh distributed:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->20128],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomReal", "[", 
  RowBox[{
   RowBox[{"RayleighDistribution", "[", "1", "]"}], ",", "5"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24546],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.086016984562465`", ",", "0.6222545402431311`", ",", 
   "1.8794494515477156`", ",", "2.4767623983689315`", ",", 
   "1.37969297756327`"}], "}"}]], "Output",
 ImageSize->{315, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->700071536]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->20481],

Cell["Properties based on higher-order moments:", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11714],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "3"}], "+", "\[Pi]"}], ")"}], " ", 
   SqrtBox[
    FractionBox["\[Pi]", "2"]]}], 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"2", "-", 
     FractionBox["\[Pi]", "2"]}], ")"}], 
   RowBox[{"3", "/", "2"}]]]], "Output",
 ImageSize->{90, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->246959435]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13338],

Cell[BoxData[
 FractionBox[
  RowBox[{"32", "-", 
   RowBox[{"3", " ", 
    SuperscriptBox["\[Pi]", "2"]}]}], 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"4", "-", "\[Pi]"}], ")"}], "2"]]], "Output",
 ImageSize->{59, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->596940546]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26793],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  StyleBox["q", "TI"]], "InlineFormula"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " quantile of a Rayleigh distribution:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->23504],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], ",", "q"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1105],

Cell[BoxData[
 RowBox[{"\[Sigma]", " ", 
  SqrtBox[
   RowBox[{"Log", "[", 
    FractionBox["1", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "q"}], ")"}], "2"]], "]"}]]}]], "Output",
 ImageSize->{120, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->237864969]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16641],

Cell["\<\
Plot the cumulative distribution function of the random variable:\
\>", "ExampleText",
 CellID->29591],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"RayleighDistribution", "[", "3", "]"}], ",", "x"}], "]"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25867],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->317246182]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->172724593],

Cell[TextData[{
 "A contour plot as both ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Sigma]", "TR"]], "InlineFormula"],
 " are varied:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->32176],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], ",", "x"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Sigma]", ",", 
     RowBox[{"1", "/", "10"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26701],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJx0XQVUlM/XXtru7lZsxa77YreY2N3d3e3P7u4WuwVFOhUpaZDuWJal64+s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   "], {{{}, 
     {RGBColor[0.33397672886243995`, 0.1279285534584573, 0.5864250111696041], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl1tslWUWhr99aPf+9267i8bEGAlQBwFj1GgEqQGHcNHJCE1EoLaUTDKj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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllElIlWEUhs+d7++drLZZ6qZ2CdHgFSpxIZQVWBA5EJRl5RQpdS3SHJAC
h6CsCIKmhWkJEaQQlS1yVcuWtWgRLWpRJEZR9BzexQvn+9/3O98Z/7LDXfWd
YTPbC6LgUWCWz5mtSJiVhsxepMxegoeIYpybisyGis3uJM32oP+eNvsBlrh8
H77A3V7QwP0Z+A3YFSDJ+Qn80YxZS0a+D8TMsvgbgG+FPwGqeGs1/mP4T4Ab
cGvQdMANgDNwd4kxx/2L8AfRP4Yv8H5bxKwEvpZ4Rnk7Dt8DbyAJNwjO4u8v
95sSinEM+zRYjCuHbrT9YBrtYlS5NhcpN895M743gQ7cVqGJkstb+BnqEyGf
abg4SHL3ORjB1xaPj7dm0V/AHsbng6R8eG5DxNSWUI7lnK9y5xnaTs7dcL+I
bz92PSjz2oE3Yb2RQd/HucDbx+Gvo6/mWygsH+PcHQGtEfXAY9mFZl1CMXns
XlOvpedwLWv2jvjLsbuoSSatnnuvV/LeHHfnwb6EalyH/Qr/NSH5XA4Us8fq
mq/k8g1c5v1SU2/zKfXGe7yA/hZvTEelec15gvvnufuFGCaJpz1QLKdAP3Yd
+mb09ehD5B4GHyOa0UV8TwTqhX/zXk8R/2hYPfdal3BuT6jmjfi6FGh2evC5
Cl+3A82617iAvQ3NTrj3cc3Kes4VUc3MIexZ/PVGNYN/OFtO2hp8LhH/z6xm
xe9cgRsDyyHleCxQjp7bQly9rk4pN+/5JNqbOc2+9/Qc9r9Au+Q5DXOeS6m2
npP32nPyXLznvls+Iz4bvmO74WrRr02qhnm48Zxm02M6iX0ko930mWrEbgIb
Y+pxA/rPGc2Cz8g93hvMaXd9JytzmmGfXf+HbE9rp32XvYY++33wUyHtgMey
lW8lScX0iVq0prSrT6lHHq6yWP8S1/guz6e1m77To4Fq6LXznf2dVg+89v6G
z5bPlM+Sz9gOYv+QlW/P6T9nrJfy
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4150981865873199, 0.26897686037537194`, 0.7004510335088123], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuQl2UVx9/9/Xb3d9/3FzZRKYjCFIyNOuN4A8ZLWiBYWzkSuQqClSsu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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnuQzmUUx59333d3vdffi4amCN2GyYxmRKjJtUFEU0MKLaumdVtp17hl
GdbKYDEsKtVkkalMRQbrkigjl/4pk0qaNFNTE1NYIl0+3/n648w853euz/md
8z1Pu9Ipj5cXhBDehxLQR6kQ9qdD+DYeQjUfavMhPAK/AKWGJiFcy4RwFfob
2XXoGPp9ciG0KQqhHbQnbR+y/QYfL0UhzIFKikPYh/9KzrugjfDj8VnPeQL6
K+CXQz3x/RA0B91B2B/KYpcMYVthCNuhDui3h0Yhe4YY3yFbTY4/kNvKGLkn
bSPdBnSmItuC/w+QlaOzhPNe7LcQayTxj8D3yzn3A/BPw58g/il8f4mPd/H3
LP5GcI6wKUG+CfuVnO9G/wDyWcRYjO+hxKjg/ruwryP2amgtuuugueifRL6P
837oSfha7tgR2xbwT6DbDB/VnFunrHuSb3M4d1FNyGcg8daS6xfwd3CuIOfL
nK9AD6K7GZs49gVQP/ie2ExGVpDyv9A/mYHsGLRD94emcC6HJmI7GP4F8unN
HbbCl5HPVe43DfnwYuc8Fn9joNPkPgmf15Dfk3fuI5BXY7uUetVQrx3ygW0N
OZeS68PI18C/g00rZPPg3+O8AZtPdH/4XtiOgoZiPw+d5nn/M/2rf5AvQn8U
8adxlzJs/oMfnvZ5LPJ58EMy7o3zxIxl3ZO6fwyfJ/DVhm8tOd+OvAL/L2Nz
lPtUcIfp5Dczcm3UM0uRzU37X/ZE50rSPazeVY2OwM/Gx7ImrmHzrGPId0SM
R8llMLQe25H4OIN+Pf7OoFsH3ce5GfbXiVeFzi0Z95h6q0Pcd12cdK115z7I
jhNjN74XFLv3q+A/i3kG1lDrpvh8kfhToVnkuhD7Pfgerfrp30CHY65Br4xz
UOxB+I+n3DPqFWFADn4+Pj4tcIwl2Lbl29cx3+EVZB/jvz2+50Mz8XUK+RvY
b8J+YsozrtnWzGhWJkkn5pkZwPkBvo1PeKYPEftX7nBQ+cedu2ZYs6s7CGuO
4rNtkTHnqbT/of7dq+Q4BNnPWc9uywLPxmVsjhd5Rgag2z9vLOrUxLXsi01D
oWua1+zdwD/VaEJkjBG2DCD/jsguKgf8NcKv0N0iz/5M+HbY34XPVoWeAfX6
KmxOx9zzW9B9O/KsrkK/EV/t1a/kcwyfM9APkbFVGKhZK4VmJ27MHL7OQZfj
rmFnzk3z/nea+eeIPS7rXIVZO/HVH/thCWNuJfp10J64MeoC8apSxpYy9Pci
i2NfUOiZ2YN9Q2Ss2ox9Hbq7I5+FodmMa65a3wmN5PwL9jcljFFdM66ZaiUM
K8+6Z9QrYwuNRcPQKUsYk6ozxmRh8XTinyO/0Sn7Hod+X+q1nm+JmHtA2CQM
EvYIo7rD94iMJap5Pu2Z0Cz8yLfZkTFDWKGd1An+36Rno1jzhO546vlYE++A
heTzlnYK8dfBf55zj6o3N+Ezw/lSZN9VxCtB9w9i/oW8HtrH+fWksUA9JazT
TtEuEeb9hL/n+dag3QltgG+R9uyrhuvgJyeN9ZVQgvothz8bnJP+pXzIVv90
IefqnHMZU2Rs6ZI2dgpjzpHv+YxzexP7Hsi75Y1Nt3HnRmq9Ou3deTHmWkxK
GvtUk7Kkc1IumtFK+Psz3jWasXr4bpGxpRn2K+Fr4WcR+zd8HMZ3QdZYrDuc
xdf3OWO9avAV53Fp/3v14DZ0O5NTa+T18AeRHUo7d/WAZu3DrGWauaUZY6yw
VRg0mNiDoMaYMeogur2hCzG/KUrQ35oydrwGDYcfqP0XjEnC0iM5/2th6kC9
h/LGDr2JtEvaQjcX+s56G2mmNEt6I9WkXQPdXTnMTXrHardqJ3Ql9nbs4wnn
pLdAy8i7S2+CWvTvjdyb2lHL4C+o5xLGjJFZz5RmSTtTs9Q973+pmWqHrCjj
f6EcNyTd0+plYXodvnunvIv1RhG2XcoZi4RxC9Pe6drlwvwa9BdFfivon4SM
d5J2ke5QrLdizr2nnd8D2bLI2KgeUG31ptRbUjU+kfUbQW8DYfBGzrdqBwa/
CbVb1ZPqRe3YPzPGBGGBatAaeY+034rCuN8570z77SSM1CyWJr2LNJPl+Noc
GXsV43+fDnqh
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4962196443121998, 0.4100251672922865, 0.8144770558480204], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuw1VUVx3/3ce55/M45v0PSDUWFS2I6lSNKBFKpkGUk5IOXJoIwWkPg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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmlslVUQhk97b3t773fv95WgyC5VBIxL2EVBQXBBhEYUS8UoWOMSBKoI
KAqUpVCVGpfgGkAjBVxYZDNRAU0V1yDBKILwQwJajGIgARSjxufNy49Jzjnz
znLmzJmZipram6cUhxA+gtLQd3EIqUIIb3O4rjSE68pDGAH1KguhJhXCe/A2
QyeKQlhVEsLMbAhX50MYhfC+Uq9ncfZTqc9qciH8B/5nsD3Q2YF1O+gs9mdD
Daw7QzvRV8e+I/Ip7B3NhDABf3rDq8anXujbjXwKfhoqxpcK+ENZf5CEsBp8
Jfamg++CzYXomgodQLYnZzPBb0RHCt73UQhH2H+AzQ3sP2F/gH0Gfi909YaW
ItuC/nnIjwAToXs8/D/hvcJZV/CHsfka9j/nvn/h2wYwm+Ftgq5g/zT4HPhz
uU8Z8RuJzEp8WQW9jP7Lsb8Y+U3It0XXMuT7JvZBtnOcvQ7/G/iDWG+E/zy6
E/ztkLJPVfBX4N987N3BWRW2xkJXYO9ZzmrRNRl6EVuj0LEVXXXouBPf3gf/
W94x0N2PoW8avmXZP3wmfqNZt3D2Het26NuNfD3yd4Gv5GwitiZzNgR/R2Mz
ha3L0FnLfi+YIayfQsc4ZK/FXi5nG9I9DXoT/CpoAPwl+NeedTtoDNix+PM0
d22EPsT/T6Fh2P4KHc3gY+S/ZP8FdB74XeDPzTkHpL8eOgrvd+gC8K3TjtVA
fO5S5pi1Zr8/dm7MAj8OW9PgT4e/FH4Cf1HBudmCjWblEvwG+H052wL/auQ7
l/rP7EH+ici5pRx7nLvMhibo/TjbTqyWwD8Ndij79cjXEKPH8G1ZsXO1F9St
xDn7Ivwu2MtirwrMl+zbn/k/splB1yLsl4NNoN76K7Fj1Qp+E9gVBf8V5VwD
9qei7yH07eRsY9Y5qNyTza3oK87Zf93npdhvrLfdl7ZsVeS7SMco1ldx1o31
evT9W/Af1t89jL43YueQcmc2/swEX4PNVvB3cTaD2GyH3iI+k9g3xa4ZqhV1
4Neib03Bf0V/MqC7H/zj7MvA/AjvJPKHkH+O+9exngvdzX4X8Gr4txX8d1Qz
zkb2YOS3VM25EGwlOpeDLc747Z/MOjbKgWawS7PONdWI+rxrmGrXS+hsZL83
57s0q8bA2wDtQ/cfRa51d0eupXqzKWA7IFORds5XKJaxa4Vq3j/gJ+Ucu0PQ
UPiX4NP4tP9EM7L7wOyA9yT4Av7Hif/KKPjPgh0T+e9vx//l8JYljo1kbob3
DJit8BaCuSWyjLDqAQvxZUHsXNIf/yLrGqfappydC28OFKdcQ2ahe4/eMOMc
74O+IvY/wPubfRPridynf7HvHFi/Cy1nfw/88ej/A/3H086JReD75By7qRnX
1imJa5dqrGrhloJ7j2piGuzHkXNVOfs1+2tiv41qxOLENU+17hHwgxP3FPUS
1Vz1ngGKYcb58AuyLVB1qWvsXGTPgX+01DnSHl+/hQ6o1kAJ63OUT8i3Yf8h
2Pd152L3pP/UrxL3xuvQNw19p9B/Ou0aqVgVJ66VitkO8J/lXavUYyp4m3zi
t1UNuFXvW+7cVo0vx/YJ+K3SroELuE99wbVcNfNSsMPA3JpyDnUH2yNxrhdB
UeIepd6knj0S7CBkzkN3V+gG1k3EdGLaM4D+eiWY7mX+88PLPSNoNujJ/gGw
KxPHRm+uXBsdObeUcxezvwhaV+SYytaN8M8vs83hkX2Ub6rRj8bOeeX6WPAv
oH+r/E07xivgt41cK+ZA1WCvL7cvtWA6qRdwFmOvTYlnl8bEuaAZZjDYlTn/
Bd1JvbIhce6pZ65lvU79M+OerliMiNxrFZPb2X+Td21Uj8zn3SPVG/Umi7E3
EPnWGc8kV6o/510rdWfxVkSuHcIcQ3Z14l54r/pX3j1a76k/cor1SeivtHOo
ivs/UnAv7an6yv6+rGOlHP497x6n3iYd6sWtyv031ZNVSzch31Timjo2tk7p
0sylXN8N7U875xdG7rHqrWtVA1gfKTj31WN/wveDiXuPYi5fdSfdRT6vgfcO
9GBwTG/KO+eUaxPSri3zY89aqjHqdR0LfjvVz9b43i/yLKEc2JZ4BtTspx7S
udw9U71SM1dH+J0S69YfUK1Sjig3VLPUO04lnuXUQybnXINVezWzzklcs1Sr
1FPmsV7P2asZ32EG2D6JZzXNTPdz119jzyKaif7Nuwbo779FPHaCH5Zz71LP
vivvmUOzhmYAzRbbsp4NNGMo9m0i9069wemse6Z6pf7QBtbvJtYtn1SL9Cf0
F1STKrDVWPBbqSf117xS8OyiN/8fM+CBgA==
         "],
         VertexColors->None], 
        PolygonBox[{{2954, 3030, 1000, 2576, 2953}, {3028, 2954, 2953, 2575, 
         999}},
         VertexColors->None]}]}, 
     {RGBColor[0.5737613661287864, 0.5435370519035628, 0.9097253540741593], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuMldUVhj/OnMucy3fOgalQLbZeagQD6DCWSVOu2j8KiKKImhSqJWmI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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3mMnWMUxt+Ze+/cufe73/fdaW2lRZWgFNMpJVQX/kAxXWY6rcQuEbF1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         "],
         VertexColors->None], 
        PolygonBox[{{3040, 3027, 3026, 3041, 1005}, {3027, 2452, 944, 3028, 
         3026}},
         VertexColors->None]}]}, 
     {RGBColor[0.6334044084039064, 0.6393668249866483, 0.911085031734951], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwtmGuMFeUZx2d3z9ndmTkz56DQgoQ0opSWOwoKaUVArNhod1nuAgYETWtT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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnuQlnMUxx/vu++27z77PM9bic2KSitKqHZrm6nVrojEbkvaaphKchtl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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6930474506790265, 0.7351965980697337, 0.912444709395743], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmH2QVWUdx8/u3nvuy7nnnKtgvqwKCmyiaNYAQjqjM7QwjMqYNE3pNKEu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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmmoVVUUx49673nec+4594pWjpjmnGmJc4GBqUklolGUBc5TpKVpZfgh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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7526904929541466, 0.8310263711528191, 0.9138043870565348], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuw1VUVx3/ce87vnufv3EwCbggoJohQVDNaTlQD8tRG3tNMGHARkEvi
xEsrdbLJRB1ADMgJLjiOlEoGTpMzadNoSHDlDYICySOYEJBUFCkB6fvpu5j+
2LP3WXuvtfZej+9av3Nl810jZ9QlSbKqXZLUa76QT5J7C0nySS5JTlSSZKDG
KK3TUpJsLCfJQR3qrPnZLEn6aD1H6yMaj4n/RtE26FxTmiRrRNsl3s0601Pr
vVrfoPVhzYc0jknmQv0+ovGqeLuI9kedG607bNG8VeOo9t6v88yZ93S367Xu
K9pm7U/PzH9Yv7toDBDvnVXJLybJeJ39Ry1JvqczbQ2SkbP+/jrXTroWlS1z
ltZ7i77/q1q/orEl5zec0PkNoo/QPFq0KeLdr/0vaXxNvw9IxmdKPnNWej/W
OCi9D4jeVWdWi2+v1p+rWtZ9ukuq3+Mke4ruNlXjJt1tVN7zPMnZqrPbNF4W
bZ5oGzRfLr0LxZuJtxM2E9/b0rtX9O9qvjvz+0frbId60z8Rfabow6WrIr5H
tb4jNd/Dmc80ab1UunZKb0XvaUnNnzYmyZW6fy/RxuTNt1Bn/iS+a8r2wR/E
O0u/n5KcXTq3Me65QHfemfe9sfHrVdsZ3yaN9u/iQrxP9Amaf5D6XdD+HPR2
Orta8n+hvZPStVL8JyX/MdEWZeZ5MbUtudt80RZkfsMY2We9fNpXd+ku2hLJ
SlPHVF3F8fTTsv2Ej4hjZu62SftrRd9Sb5sSv8TVB2XbC1u9o/VxjQ/rnQ/M
0HOyw0WtL9P5Nu036Pwh/X63znvQy6lthc3YOxO88J0LXnQ35az/Y8kZovec
F+35nGVhB2JqX9lx1ao3fqdk+9+ieQKxorMzZIPGiu+/U/TBom/SmRmaJxat
a7tstE3jEcnZqLmrxh6duUH3fDLndXfJmCb+3vrdXrwPan1Ob+hY77hZoPv3
05mHRR+q3zdr3ar1OK176p7PS1cXnT+q9UjRt2t9ddU2wgbXinam7Pdfofl3
0vGQ9F+Fr7WX17pZMXNT0bh0t+R3RE47x9oOjTdyvuNLme9JjBK/YAV5T76S
+520/2Xxz9Hb5yBbv/uJ3qPs+IbvWNH4he+JP/aJjTbxbaxYF5i3L3DjnNa9
S/bh+3nrvz1xDqWZ7zFZ+89p3YM801sGiTfR2WbZYWLVeda5aPrT0jlYPr9e
9OOS31T02xPRz0rXS1XHw7B6YzO4NFTrwVqPrYv35Hxn7rsi853xz4bMPsql
nh/UmZk6PzLWRe3/SOc+0H1eEO9E6XtFe9cqhwZVjIN/I7801ipW7te92mt9
MrWNt2keKXnLtbc11jPE+5/Mv2fxW7KXSNcgyb5Be5+XPX4r2TPLrgnndK6p
YDusJA5Tv+vT1Gfv17q5znvQ+4pvneS3prbHydDzFdHXZ+a/v2AMWCq9zaJN
0vhG6ngn7nfX2U/Yjpgh9jdGLpAf5MnpOufJtjh/IO9cva7O9iUOybUDJWM0
GAsNPdQo6sqtWn81cZ3YHryv6ezjJftgv9a9MmPEHNn2XGabntX8b40tqeME
22BD8I/6AQbmNK/V73vqTZsQ9GWS+0LQwXfqBDlLjRwVePuM7P9m0XTyekzw
PiH/DCzbJ29K/1uZdezVvC+zXPD8lxojif2Kc4k8AsvHh/yLqf23KnE9eTFw
G/utT20HMHFd2OdCnMfX0wvW/3TQLoScu3Tff2Xmv6PRPQH9wAnRTmaWRZ9B
XJD7F3W3j8TznPSuksyputvsxHLPhz3pE7Ax+XtaPrkzcw8yTLQhqWPgm3nH
A7Gwo8F70HvqDt8qu1/Az8PjLRMLjlf0El9gCjG2NO88JsehPRR0cHpXyVhN
vpGf5Php5f2HGm+L76+Kk+Ul722MXgH8p0cCg7g/Pnjv0lp8fUvW1Un37KhR
VzAWVhuNh/OJC4327YzL82PdQ3xLqt5r1bxCY4nWxdR1hh6gLWrH7sCS+oIx
CVq3yKNMel7LnLfTCvYj/iVnZ8UbOYfs/+Ee2NBoWeQK7yZOwBlwBZ++obss
q9n+YPnckm3cX7IHpK6V+GpH5O+pgnEI31NX6gNDwE58NDawFDp4iG/hxedD
pOvdkmswscBeP9E3N1gftqImH4y6vLXB54ifSUXXXWpuB80dM/fa6K1Lrbur
aN0y/+6p+e819wvcJRf3oTeHD3/RY+2M2vdJZpzlbRPp0zLH3lU15zD5A46A
y9gO7AaHsXtT1BZy4SmdH1tyH1fQunPmPbALf+NragNxRayeaLBv6KF+LDk3
ViyHXvXe6FfHS8b3M/cUaWAxtY8a/kTZdfxI9Pr0isslY0TN/Q799dyye2z8
CQ2fgs1gNPFGr7Q5+rUn5aOVVefWrzQvrkZcS+bCinn+kvt/39dB+x01Pps3
7u6vOp+ot/QcvP2MaB9Vjff02q9XrIveqqVozOdNm+JdzPwmxjeEf8jN7dH3
sR4qOdtLxmbwHF/A2xJy4P213n151W8HWxfGdxBr8Jbf2Be8BkvvKbtXxj70
BuQTMdNasU2xHTV8Xc24MiLOgHv/FO1UybEMLhCX5MHImJHD2Vlxnn7t5Yp7
NugzQxc9xbA4v6xknGIP3G2oGQNXiPeZovOF3vzqzHGxp+g6RA1qyNwz0ceA
1acCr+k7j0YvR33rFG+fFjanvv9GcoaIvynvb4rFUZvoA49Gb0aeIJNcoJ+i
zwaL+U7dF9+qxMqaiLHV4GTZtiYu2grG9XtjTV/aqeTelHjuL5kDqq77bxX9
nUmvSH/XOXq/B/T2n2m0iv+Q7vvD6H92RL8B5qD/loJrzSDZb2DNmDqb85nr
zCOaJ1Rd14cEJpFLe0TfrbEmdXyhF99hR3KePOU75nR8y2CLY2HblohB8Jl4
bInYnh5r6PT4vRrd55Njh+JbBqyZHmeI72551zL4pkdf93vpv67qPXKtKXp7
+pM+efdOF/XWT2vGmHdkv+MVx9aH4L3GwQZ/Px0MveAQuU1eN4VOfIG/JkUu
3y4ZkzXuy9lvzeG7n5TtY/zbFn7Fv9dmxmJweJPue1vVscR9sSO8x3WXd2rG
+ysy9xnEOXoeD13U59bAg5+Ld1TFPQy4jx7wvDl0wjuV/rTk70Z6E2yM7ybL
3lMaHROT4v6cX1TxNzn4DAbPDf9Sn++LGk28r4qYvxS76D0WOTY7+q5pkUe9
4z8T8gW8IY/mRe90LGrwyqhdyKEO35a5zuKbPvF/BLE5reb4vDVqCfe5RrQv
1Nwb9o14J87HBTbxXr4DzkfvB058sWasIC84uzt6M3iJ+6+XHSvESfeqsaAp
vhEPx39E9CP0GPQa5OhlFe9R64o1xwk1F9yjHxhei9of397M1HH0XfqPim8X
cnhhfBPzbnAMG+AD4pncb4n4p/43RQ/w7bJ7R97cI+KXbzfylp6evp3vZ3KJ
9bO6780l6xkT3+Ho+i/KH6NJ
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnmw1lMYx8+979b7/t7f732HUl1UpEULDTOyFAapxFCqMYOo0KJlKjVZ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         "],
         VertexColors->None], 
        PolygonBox[{{3248, 3184, 3183, 3249, 1216}, {3184, 2464, 1299, 3352, 
         3183}},
         VertexColors->None]}]}, 
     {RGBColor[0.80726725, 0.861883, 0.894034], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuw1VUVx3/33PO455zf73fOmKWWaQqpjArKoE1eHkr3KiZgmq9CUXn5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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3mwlXMYx3/33nPP6Zzzvu85EslYKy1KqVEz3G6JbmUpZN9Kq0lUljRo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         "],
         VertexColors->None], 
        PolygonBox[{{3035, 2580, 1004, 3039, 3034}, {3349, 3035, 3034, 3350, 
         1298}},
         VertexColors->None]}]}, 
     {RGBColor[0.8608307500000001, 0.879745, 0.8700376], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmXmQ1MUVx2dnh9md+c1VETRUygNBkqhhPbBAlEO5okaWaImKuCCCqBSC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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3mwV2MYx89vu8vvnPM7v0EZY2xJllJKZnK5yrRYW2RIWt26qW6pbE0q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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.91439425, 0.8976069999999999, 0.8460412], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHmMllcVxt+Pgfm2d2bAwJTCiNUAZQYQtJWlZWsT6CyMEhgNDEsNtBZo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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllWuIVWUUhr99znHmtPc+e0/Q5U8XihRRx0uKaZSKVwyssH7oeAtNrRG1
sKTMy4x3B9EaSRtJMbEyxSYEb6F5CxKFlFHyAirYzI/QMinQhKLn5f2xYK29
3vWu9a31fWs/NXXu2DmFEMI7SAk5l4QwERlSDqGjGMLAOISGmhA+x54dhVDA
92cewrXqEC4j5yohjACTQPIH+On4zj8QwkqwQ/F3xrefmAv4ynzrgr0Evhb4
TmE3g12F/x76TfLfRp+MXMG+S0wP+Loj2+C/gb0avR8c78P9HrIxhR/5keIb
iq51QGwu1dyOvpZv+9AHR+Zuxj5Rdo5h8A1HnoDrcSTCfwv7Ivol5ADn+4Zv
P8G9sxM1wLeJfL9gX6emXfj+5ltbtTF9sxCOYx/At75kLvVMvRLnSc77YWrf
ePK/BH8C/gT6f8gK7DOJz7q4k3ulHqp36tkx/ENiY+erH9iNyAKwDcg9fAuJ
+RhsKzW0ku8D8q0j39P4x1LPZvAb0GuppwW9JzHN2J8gCyquQbnnI71iY+RT
jLDdOWNW5Zg2+H/P3avl1LQJ+yr136S2BP8oYg9W3Lt/8O+jlun4G4ue2Rpi
m3LPcib+2dQ+D45xzK+OM9SR6wo5L3CWk8R8mnrmmnV95LOeBj8v8pk/is2h
2C/5Vgf3BOQ5+Psjr8J3EczPBd/JHtRWW3Fv9G0i2K2puRXzFthZ8L0O3/bI
+rvkGxX8bS/Ydr0JuI5znkOyyXG4ym8ih7um4l7oTtyJfQc0+z3YfcH3Q54n
/m04u8EVk+OFkr8tw9cG3x74NvKtHvsNMDuxG7HnoL+Z+62NKXvWqkm1aOYz
Er8B3X31vJ3cM2uM3cK3xWAXIb0i97gZ/Ar45sJ3NvLb6Yh91/WGlnKWJRW/
Xd25zsR2QfoUPbPrxB6Kfff0Ru9g30YeLHpntKN3IN2qvSPGwz8t9ewHwz8G
38t689XGNOFbk7oW9bhe5yXmku5+0btqHPagsnfWaXo1hW9f05+vkB+wj2S+
CwPBn8/8hvR21NOusXeA3r7O/K16nXnWvYveFZMS7z7tDPVeOcStGTyEfZ/4
x+h1V/DHMudUrhb8v6aesWb7W8m1Tkh8VtW8O/FMNAv16FmwRzPH6g58l7km
1dJU5V11I/Gu1c56BPyjqXPrTo6s+M3prWmnfgF2G1IXeYd+r9jEu1U9lO+1
3LtAmCexC9RYW/JMP4O7JXWvR+O/BvZq7rurHfBKbox8mlkj+lLk34Lv1MM6
e+q7qh5NBj8l910W5kX0Qch9+E8VvDu007TLtEP+wt6R+O0LcxD9bOJ/k3bw
QrCLKt5V2qHaxXrzeuvaySvJvSo1t3bEM+itmXupM55Bn5V6d+nO/A9CdPyP

         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[
       PolygonBox[{{299, 279, 256}, {249, 250, 30}, {29, 13, 14}, {256, 249, 
        14}, {279, 305, 250}, {279, 250, 249}, {279, 299, 15}, {29, 249, 
        30}, {256, 279, 249}, {305, 279, 15}, {249, 29, 14}},
        VertexColors->None]]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2400, 3812, 3257, 2629, 3536, 2582, 3042, 3908, 3907, 2600, 
         3973, 2962, 3209, 3210, 3045, 3044, 3250, 3192, 2479, 3373, 3372, 
         2775, 3664, 2561, 2998, 2597, 4223, 2598, 2578, 3455, 3458, 3457, 
         3464, 3037, 2455, 3344, 3343, 2753, 3635, 2752, 3193, 2499, 2801, 
         2743, 4061, 4062, 3628, 3627, 2606, 3481, 2607, 3482, 3922, 3923, 
         2976, 2977, 2738, 3311, 3883, 3884, 3019, 3461, 3450, 3813, 3451, 
         3453, 3452, 2571, 3402, 3926, 3519, 2624, 3520, 3795, 3794, 3203, 
         3935, 3934, 3933, 2809, 3202, 3201, 3197, 3198, 2779, 3673, 2778, 
         3674, 2980, 3059, 3902, 3901, 3900, 2766, 3207, 3206, 2506, 4220, 
         2808, 2744, 4232, 2745, 4063, 2612, 3976, 3978, 3977, 2502, 3171, 
         4185, 2907, 4184, 2908, 3024, 3023, 3000, 3223, 2786, 3244, 2787, 
         2789, 4236, 2788, 2918, 4243, 2917, 3253, 3254, 3222, 3221, 3252, 
         3230, 3932, 3931, 3930, 2804, 2986, 2985, 3511, 3512, 3399, 3398, 
         3397, 2619, 2951, 2450, 2616, 4224, 2617, 2505, 3176, 2909, 4242, 
         2910, 3055, 3054, 3002, 4196, 2922, 4195, 2923, 3225, 3224, 3919, 
         3918, 3917, 4069, 4068, 4194, 3255, 2515, 3941, 3940, 2926, 4199, 
         2532, 2987, 2824, 4079, 4081, 4080, 2818, 4076, 2817, 4077, 2514, 
         3179, 3787, 2911, 3786, 2912, 3178, 3177, 3093, 4090, 2836, 3745, 
         2837, 3095, 3094, 2417, 4203, 2664, 2540, 3312, 3757, 3758, 3760, 
         3759, 3108, 4098, 2847, 3761, 2848, 3110, 3109, 2419, 4205, 2683, 
         2543, 3316, 3773, 3774, 3776, 3775, 3122, 4106, 2858, 4105, 2859, 
         3124, 3123, 3299, 3298}],
        "0.9`"],
       Annotation[#, 0.9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2707, 4021, 2708, 4022, 2549, 3439, 3441, 3440, 2427, 3132, 
         2423, 3839, 3838, 4112, 4111, 2695, 3836, 4246, 2865, 4108, 3837, 
         3120, 2684, 3772, 2857, 3771, 4104, 4103, 3111, 3597, 3598, 3596, 
         3595, 3315, 2542, 3823, 4204, 3824, 3594, 4009, 3825, 3826, 3106, 
         3288, 3287, 3286, 2846, 3097, 3096, 3577, 2662, 3576, 3579, 3578, 
         2539, 3180, 2538, 2996, 3285, 2416, 2994, 2411, 3276, 3807, 3806, 
         2649, 4249, 3092, 3396, 3686, 2793, 4237, 3418, 3937, 2511, 3235, 
         2512, 3939, 3938, 2815, 4075, 2920, 4187, 4189, 4188, 2791, 3684, 
         2792, 3685, 2790, 4193, 2921, 4190, 4192, 4191, 3001, 3227, 2443, 
         2942, 3247, 3246, 2794, 3175, 3174, 2491, 3390, 2811, 3408, 3409, 
         3410, 3080, 3233, 3234, 2925, 3411, 3936, 2507, 3216, 3251, 3219, 
         3220, 2785, 4186, 2916, 3913, 3914, 3389, 4233, 2784, 4235, 3388, 
         4234, 3387, 3386, 2783, 3696, 2807, 3695, 2806, 3472, 2595, 3471, 
         3474, 3473, 2517, 3199, 3200, 2826, 3719, 2825, 3720, 3077, 2458, 
         3985, 3986, 3517, 2623, 3518, 3927, 3403, 3952, 3954, 2527, 3953, 
         3521, 3987, 3988, 3406, 3929, 3407, 3436, 3948, 3435, 3381, 3434, 
         3433, 3516, 2486, 3515, 2621, 3513, 2622, 3069, 3068, 2498, 4071, 
         2799, 3690, 2800, 3195, 3194, 3791, 2914, 3790, 3793, 3792, 2587, 
         3456, 3466, 3465, 2481, 3050, 2480, 3375, 3374, 2596, 3475, 2946, 
         2947, 2772, 3658, 2773, 3660, 2767, 3663, 2774, 3654, 3662, 3661, 
         2469, 3480, 2468, 3362, 2973, 2583, 2935, 3258, 2398, 3031, 3032, 
         2580, 3035, 3349}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3248, 3184, 2464, 3033, 3350, 3034, 3039, 3038, 2577, 2934, 
         2933, 3043, 2522, 2972, 3363, 2470, 3656, 2770, 3655, 2771, 3657, 
         2768, 3801, 3802, 3659, 3798, 3797, 2448, 3015, 3048, 3049, 3058, 
         3057, 2586, 3478, 3479, 3476, 2599, 3477, 2471, 4070, 2797, 3688, 
         2798, 3229, 3228, 3921, 3920, 3400, 2796, 3687, 2620, 3067, 2604, 
         3060, 2605, 3063, 3514, 3062, 3912, 3911, 3382, 2608, 3489, 3679, 
         3678, 3214, 3215, 2803, 4197, 4198, 3924, 3925, 2501, 3401, 2500, 
         3021, 3022, 3074, 3073, 2961, 3530, 3531, 3421, 3534, 3535, 2628, 
         3437, 2528, 3422, 2529, 3728, 3729, 3727, 2830, 2940, 2939, 3243, 
         2829, 3241, 2635, 3725, 3726, 3724, 4083, 4082, 2490, 3081, 3172, 
         3173, 3218, 3217, 2739, 3699, 3700, 3697, 2810, 3698, 3681, 3622, 
         3623, 3625, 3624, 4060, 4059, 3626, 3547, 3999, 4225, 4226, 4001, 
         4002, 2646, 3557, 2645, 3682, 3683, 3394, 3395, 3090, 2648, 3703, 
         3274, 3275, 2992, 2415, 3822, 3284, 3575, 3574, 2537, 3245, 2944, 
         2945, 3103, 3102, 2997, 4096, 4097, 3755, 2845, 3756, 2663, 3593, 
         2681, 3592, 2680, 3107, 2665, 3770, 2856, 3289, 3290, 3291, 3119, 
         2421, 3835, 3606, 3834, 4206, 3833, 2545, 3320, 3607, 3608, 3610, 
         3609, 3121, 4109, 4110, 3780, 2866, 3781, 2696, 3618, 2705, 3617, 
         2704, 3133, 2697, 3783, 2873, 3304, 3305, 3306, 3141, 2430, 3861, 
         3860, 4038, 4037, 2550, 3328, 3619, 4245, 3620, 3886, 3885}],
        "0.7`"],
       Annotation[#, 0.7, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2436, 3155, 3878, 3877, 3876, 2886, 4151, 2722, 3874, 4248, 
         2885, 4150, 3875, 3147, 2717, 4138, 2880, 4136, 2879, 4137, 3142, 
         4035, 2715, 4034, 2716, 4036, 2706, 4122, 4123, 4119, 4121, 4120, 
         3131, 3615, 3616, 3614, 3613, 3327, 2548, 3847, 4207, 3848, 3612, 
         3849, 2426, 3130, 2422, 3297, 3296, 2864, 3779, 2682, 3605, 2693, 
         3292, 3831, 3832, 3115, 3116, 2855, 3769, 2854, 4102, 3105, 3590, 
         2677, 3589, 2678, 3591, 2661, 3754, 2844, 3753, 4095, 4094, 2995, 
         3570, 3571, 3569, 3573, 3572, 2536, 3091, 2535, 2991, 3283, 2414, 
         3088, 2410, 3273, 3272, 3742, 3741, 2637, 3556, 2642, 3271, 3818, 
         3819, 2989, 3265, 3264, 3621, 2636, 3680, 2782, 4067, 2781, 3721, 
         3722, 3424, 3945, 2518, 3240, 2519, 3426, 3425, 2828, 3723, 2927, 
         3709, 2822, 3423, 3942, 3943, 3238, 4222, 3239, 2930, 3808, 2929, 
         4200, 3020, 3691, 2802, 3693, 4073, 4072, 3692, 3540, 3995, 3677, 
         4074, 2780, 4066, 3675, 3910, 3909, 2484, 3485, 3486, 3483, 3488, 
         3487, 2602, 3070, 2601, 3061, 3259, 2403, 3694, 2401, 2936, 2795, 
         2975, 3689, 2974, 2472, 3364, 3017, 2567, 3051, 2566, 3016, 3333, 
         3799, 3800, 3366, 3368, 2477, 3208, 2478, 3371, 3047, 2585, 3191, 
         2584, 2968, 2520, 3187, 2521, 2970, 3348, 3642, 2759, 3036, 3351, 
         3352, 3183, 3249, 2913}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2451, 3893, 3334, 3789, 3788, 2746, 3641, 2758, 3638, 2757, 
         3640, 2462, 2969, 3644, 2971, 2749, 2967, 3337, 2966, 3046, 3342, 
         3341, 3339, 2751, 3189, 3188, 3370, 3369, 3367, 2769, 3653, 2742, 
         3018, 2568, 3196, 2959, 2960, 2588, 3430, 2524, 3431, 2525, 3947, 
         3432, 3405, 3928, 3404, 3538, 3994, 2631, 3539, 2630, 3537, 3064, 
         4215, 3065, 2610, 3484, 2609, 3502, 2485, 3701, 2812, 3676, 2813, 
         3702, 2632, 3541, 2633, 3260, 3261, 3262, 3071, 2404, 3809, 2938, 
         2821, 2984, 2983, 2516, 3236, 3237, 2827, 2941, 2405, 3731, 2409, 
         3268, 2988, 2533, 3082, 2534, 3554, 3555, 3551, 3553, 3552, 3083, 
         4088, 4089, 3740, 3744, 3743, 2647, 3567, 3568, 3564, 3566, 3565, 
         2993, 4092, 4093, 3751, 2843, 3752, 2660, 3588, 2675, 3586, 2674, 
         3587, 3104, 4101, 2852, 3767, 2853, 3768, 2679, 3604, 2691, 3602, 
         2690, 3603, 3117, 3118, 2863, 3293, 3294, 3295, 3128, 3846, 3845, 
         3300, 2702, 3611, 2694, 3782, 2872, 3301, 3302, 3303, 3139, 3857, 
         3856, 4130, 2876, 4247, 3855, 2713, 4134, 4135, 3858, 3859, 2429, 
         3145, 2433, 3871, 3870, 4144, 4143, 2720, 4148, 4149, 3872, 3873, 
         2434, 3153, 2435, 3882, 3443, 2555, 3442, 2556, 4052, 4053, 4049, 
         4051, 4050, 3156, 2896, 4163, 4165, 4164, 2727, 4056, 4057, 4055, 
         4230, 4229}],
        "0.5`"],
       Annotation[#, 0.5, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3040, 3027, 2452, 3335, 2955, 2747, 3009, 3639, 3007, 3182, 
         3645, 3646, 3643, 3648, 3647, 2748, 3631, 3632, 3336, 3338, 2454, 
         3025, 2453, 3340, 2958, 2750, 3013, 3012, 2463, 3666, 3667, 3665, 
         3669, 3668, 2754, 3636, 3637, 3345, 3894, 2456, 3470, 2457, 3895, 
         3469, 3346, 3449, 3448, 3629, 3498, 3499, 3384, 3385, 2488, 3504, 
         2487, 4216, 3503, 3383, 3496, 3495, 2603, 3524, 3525, 3412, 3413, 
         2508, 3529, 2509, 4221, 3528, 3414, 3527, 3526, 3391, 2982, 2531, 
         3079, 2530, 2981, 2928, 3706, 2819, 3707, 2820, 4078, 3708, 3543, 
         3996, 3542, 3549, 4000, 2640, 3550, 2639, 3548, 2990, 3738, 2834, 
         3279, 2835, 3739, 2643, 3563, 2657, 3561, 2656, 3562, 3089, 4091, 
         2841, 3749, 2842, 3750, 2659, 3585, 2672, 3583, 2671, 3584, 3101, 
         4100, 2850, 3765, 2851, 3766, 2676, 3601, 2688, 3599, 2687, 3600, 
         3114, 4107, 2861, 3777, 2862, 3778, 2692, 4020, 2701, 4018, 2700, 
         4019, 3129, 4117, 2870, 4116, 2871, 4118, 2703, 4033, 2712, 4031, 
         2711, 4032, 3140, 4132, 2877, 4131, 2878, 4133, 2714, 4042, 2719, 
         4039, 4041, 4040, 3146, 2884, 4145, 4147, 4146, 2721, 4047, 4048, 
         4044, 4046, 4045, 3154, 2895, 4160, 4162, 4161, 2726, 4174, 4175, 
         4171, 4173, 4172, 3162, 2901, 4176, 4178, 4177, 2730, 2904, 4241, 
         2903, 3166, 2731, 4231, 2732, 2459, 2755, 2756, 2473, 2777, 3904, 
         3905, 3906, 3213, 3380, 3379}],
        "0.4`"],
       Annotation[#, 0.4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2442, 4213, 2736, 2560, 3966, 3968, 3967, 4058, 3165, 4180, 
         4181, 4179, 4183, 4182, 2729, 2900, 4240, 2899, 3161, 2897, 4239, 
         2898, 2725, 4159, 2894, 4158, 2893, 3159, 4157, 2891, 4156, 2892, 
         3152, 3151, 3869, 3868, 3867, 2883, 4142, 2718, 4141, 2882, 3864, 
         3865, 3866, 3144, 3854, 3853, 3852, 4129, 4128, 2710, 4126, 4127, 
         3850, 3851, 2428, 3137, 2425, 3844, 3326, 3325, 3324, 2547, 4016, 
         4017, 4015, 2689, 3113, 4011, 4012, 4010, 4014, 4013, 2673, 3763, 
         3764, 3762, 2849, 4099, 3100, 3581, 2668, 3580, 2669, 3582, 2658, 
         3748, 2840, 3747, 2839, 3087, 2644, 3560, 2654, 3280, 3820, 3821, 
         3086, 3270, 3269, 3737, 4087, 4086, 4085, 3546, 3997, 3998, 3544, 
         3545, 3266, 3704, 3816, 2407, 3705, 3815, 3814, 3263, 2831, 3730, 
         2634, 3713, 3714, 3710, 3712, 3711, 2492, 3979, 3980, 3508, 3510, 
         3509, 2510, 3232, 3417, 3416, 3415, 2814, 3076, 3075, 3066, 3491, 
         3492, 3490, 3494, 3493, 2489, 3500, 3501, 3497, 3890, 3889, 3892, 
         3891, 3446, 2569, 3444, 2570, 3447, 3670, 3671, 3672, 3378, 3903, 
         2483, 3211, 2482, 3377, 3376, 3804, 3803, 2579, 3308, 3460, 3459, 
         2565, 3190, 2956, 2957, 2572, 3633, 3634, 3630, 4065, 4064, 3186, 
         3357, 3356, 3354, 2761, 3649, 2741, 3008, 2564, 3185, 2952, 3030, 
         2954, 3028, 3026, 3041, 2581}],
        "0.3`"],
       Annotation[#, 0.3, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2467, 3896, 3358, 3463, 3462, 2575, 2953, 2576, 3029, 2563, 
         3181, 2562, 3004, 3331, 3796, 2915, 3353, 3355, 3204, 2760, 3652, 
         2763, 3650, 2762, 3651, 2460, 3010, 3011, 2737, 3307, 3309, 3310, 
         2399, 3056, 2402, 2937, 2776, 3212, 2526, 2979, 3899, 3467, 2592, 
         3445, 2591, 2949, 2948, 3468, 2593, 3365, 3505, 3506, 2611, 3974, 
         3975, 3949, 3951, 3950, 2503, 3231, 2504, 4219, 2805, 2964, 2494, 
         3523, 2493, 4217, 3522, 3393, 2614, 3507, 2613, 3072, 2626, 3532, 
         2627, 3533, 3944, 3718, 2823, 3715, 3717, 3716, 2513, 3084, 4084, 
         2832, 3734, 2833, 3784, 3785, 3267, 3817, 2408, 3736, 2413, 4202, 
         3735, 3278, 2651, 3559, 2641, 3746, 2838, 3281, 3282, 3098, 2655, 
         4005, 2667, 4003, 2666, 4004, 3099, 2670, 4006, 4008, 4007, 2544, 
         3317, 3318, 3319, 3830, 2420, 3127, 3843, 3842, 3841, 2867, 4113, 
         2698, 3136, 2868, 4114, 2869, 4115, 2699, 4028, 2709, 4025, 4027, 
         4026, 3138, 4029, 4030, 3958, 2553, 3957, 2554, 4139, 4140, 3862, 
         3863, 2432, 3150, 3881, 3880, 3879, 2888, 4155, 2723, 3158, 2889, 
         4238, 2890, 2724, 4169, 4170, 4166, 4168, 4167, 3160, 4054, 3964, 
         3965, 3963, 2559, 2902, 4210, 2439, 3164, 2441, 4212, 2906, 3169, 
         3168, 3170, 2735}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2734, 3167, 2733, 2905, 4211, 2440, 3163, 2438, 2446, 3888, 
         3887, 2558, 3962, 2437, 4209, 2728, 2557, 3959, 3961, 3960, 4043, 
         3157, 4153, 4154, 4152, 2887, 3149, 3148, 2431, 4208, 2881, 2552, 
         3955, 2551, 3956, 4024, 4023, 3143, 4125, 2874, 4124, 2875, 3135, 
         3134, 2424, 3840, 3323, 3322, 3321, 2546, 2686, 4228, 2685, 3125, 
         3126, 2860, 3827, 3828, 3829, 3112, 2418, 2445, 3314, 3313, 2541, 
         2653, 4227, 2652, 3085, 2638, 3558, 2650, 3277, 3732, 4201, 2412, 
         3733, 2406, 2444, 2943, 2931, 3256, 2816, 2932, 2965, 3811, 2919, 
         3810, 3805, 3392, 3420, 3419, 3438, 3993, 2495, 3916, 3915, 2615, 
         2950, 2449, 3982, 2618, 3981, 3984, 3983, 2497, 3226, 2496, 4218, 
         2924, 3053, 3052, 2999, 3972, 2589, 3971, 2590, 3991, 3992, 3989, 
         2625, 3990, 2474, 2963, 2594, 3897, 3898, 2476, 2978, 2475, 3428, 
         3429, 3427, 3946, 2523, 4244, 3454, 2573, 3969, 2574, 3970, 2466, 
         3205, 2465, 4214, 2764, 3332, 2447, 3014, 3330, 3329, 3003, 2740, 
         3006, 3005, 2461, 3347, 2765, 3359, 3360, 3361, 3078, 3242}],
        "0.1`"],
       Annotation[#, 0.1, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  FrameTicks->True,
  ImageMargins->0.,
  ImageSize->{225, Automatic},
  Method->{},
  PlotRange->NCache[{{0, 3}, {
      Rational[1, 10], 3}}, {{0, 3}, {0.1, 3}}],
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 224},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->918060160]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18086],

Cell["The probability density function integrates to unity:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->20514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], ",", "x"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\[Sigma]", ">", "0"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8938],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->545226291]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14990],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], ",", 
   RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28981],

Cell[BoxData[
 RowBox[{"8", " ", 
  SuperscriptBox["\[Sigma]", "4"]}]], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->True,
 ImageSize->{27, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->18769947]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"D", "[", 
   RowBox[{
    RowBox[{"CharacteristicFunction", "[", 
     RowBox[{
      RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], ",", "t"}], 
     "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "4"}], "}"}]}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{"t", "\[Rule]", "0"}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->5242],

Cell[BoxData[
 RowBox[{"8", " ", 
  SuperscriptBox["\[Sigma]", "4"]}]], "Output",
 ImageSize->{27, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->538062242]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13974924],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RayleighDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RayleighDistribution"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Sigma]", "=", "1"}], TraditionalForm]], "InlineMath"],
 " is a special case of ",
 Cell[BoxData[
  ButtonBox["ChiDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->206794640],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"RayleighDistribution", "[", "1", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->380460473],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", "x"}]], "Output",
 ImageSize->{40, 25},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->975565037]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiDistribution", "[", "2", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32069594],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", "x"}]], "Output",
 ImageSize->{40, 25},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->44621144]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16070],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RayleighDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RayleighDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Sigma]", "TR"]], "InlineFormula"],
 " is not a positive real number:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->25631],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"RayleighDistribution", "[", 
   RowBox[{"-", "1"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24061],

Cell[BoxData[
 RowBox[{
  RowBox[{"RayleighDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Parameter \\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(-1\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" is expected to be \
positive.\\\\\\\"\\\", \\\"MSG\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->27219],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"RayleighDistribution", "[", 
   RowBox[{"-", "1"}], "]"}], "]"}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->28781]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19046],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"RayleighDistribution", "[", "\[Sigma]", "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{"\[Sigma]", "\[Rule]", 
    RowBox[{"3", "+", "I"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25838],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"3", "+", "\[ImaginaryI]"}], ")"}], " ", 
  SqrtBox[
   FractionBox["\[Pi]", "2"]]}]], "Output",
 ImageSize->{81, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->562257353]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->16984],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChiDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChiDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MaxwellDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MaxwellDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->23953]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->1899],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Continuous Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ContinuousStatisticalDistributions"]], "MoreAbout",\

 CellID->162818315]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RayleighDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 14, 18.7196622}", "context" -> "System`", 
    "keywords" -> {"Rayleigh distribution"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "RayleighDistribution[\\[Sigma]] represents the Rayleigh distribution \
with scale parameter \\[Sigma].", "synonyms" -> {"rayleigh distribution"}, 
    "title" -> "RayleighDistribution", "type" -> "Symbol", "uri" -> 
    "ref/RayleighDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7360, 237, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->242810213]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 152050, 3078}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1982, 51, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2605, 78, 1585, 37, 70, "ObjectNameGrid"],
Cell[4193, 117, 623, 19, 70, "Usage",
 CellID->14122]
}, Open  ]],
Cell[CellGroupData[{
Cell[4853, 141, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5331, 158, 839, 30, 70, "Notes",
 CellID->429991538],
Cell[6173, 190, 445, 14, 70, "Notes",
 CellID->7997],
Cell[6621, 206, 702, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7360, 237, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->242810213],
Cell[CellGroupData[{
Cell[7745, 251, 148, 5, 70, "ExampleSection",
 CellID->838947038],
Cell[7896, 258, 273, 6, 70, "ExampleText",
 CellID->29681],
Cell[CellGroupData[{
Cell[8194, 268, 149, 4, 28, "Input",
 CellID->4879],
Cell[8346, 274, 222, 8, 60, "Output",
 CellID->209078]
}, Open  ]],
Cell[CellGroupData[{
Cell[8605, 287, 154, 4, 28, "Input",
 CellID->17207],
Cell[8762, 293, 284, 10, 50, "Output",
 CellID->12101926]
}, Open  ]],
Cell[9061, 306, 121, 3, 70, "ExampleDelimiter",
 CellID->25706],
Cell[9185, 311, 253, 6, 70, "ExampleText",
 CellID->23849],
Cell[CellGroupData[{
Cell[9463, 321, 176, 6, 70, "Input",
 CellID->24075],
Cell[9642, 329, 405, 14, 69, "Output",
 CellID->76420828]
}, Open  ]],
Cell[CellGroupData[{
Cell[10084, 348, 288, 10, 28, "Input",
 CellID->20124],
Cell[10375, 360, 3092, 55, 168, "Output",
 Evaluatable->False,
 CellID->499424010]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13516, 421, 215, 7, 70, "ExampleSection",
 CellID->2884],
Cell[13734, 430, 247, 7, 70, "ExampleText",
 CellID->20128],
Cell[CellGroupData[{
Cell[14006, 441, 176, 6, 70, "Input",
 CellID->24546],
Cell[14185, 449, 332, 10, 36, "Output",
 CellID->700071536]
}, Open  ]],
Cell[14532, 462, 121, 3, 70, "ExampleDelimiter",
 CellID->20481],
Cell[14656, 467, 83, 1, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[14764, 472, 154, 4, 70, "Input",
 CellID->11714],
Cell[14921, 478, 440, 17, 71, "Output",
 CellID->246959435]
}, Open  ]],
Cell[CellGroupData[{
Cell[15398, 500, 154, 4, 70, "Input",
 CellID->13338],
Cell[15555, 506, 333, 12, 57, "Output",
 CellID->596940546]
}, Open  ]],
Cell[15903, 521, 121, 3, 70, "ExampleDelimiter",
 CellID->26793],
Cell[16027, 526, 376, 13, 70, "ExampleText",
 CellID->23504],
Cell[CellGroupData[{
Cell[16428, 543, 180, 6, 70, "Input",
 CellID->1105],
Cell[16611, 551, 336, 12, 60, "Output",
 CellID->237864969]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16996, 569, 223, 7, 70, "ExampleSection",
 CellID->16641],
Cell[17222, 578, 112, 3, 70, "ExampleText",
 CellID->29591],
Cell[CellGroupData[{
Cell[17359, 585, 289, 10, 70, "Input",
 CellID->25867],
Cell[17651, 597, 2600, 47, 70, "Output",
 Evaluatable->False,
 CellID->317246182]
}, Open  ]],
Cell[20266, 647, 125, 3, 70, "ExampleDelimiter",
 CellID->172724593],
Cell[20394, 652, 362, 13, 70, "ExampleText",
 CellID->32176],
Cell[CellGroupData[{
Cell[20781, 669, 406, 13, 70, "Input",
 CellID->26701],
Cell[21190, 684, 121007, 2016, 70, "Output",
 CellID->918060160]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[142246, 2706, 233, 7, 70, "ExampleSection",
 CellID->18086],
Cell[142482, 2715, 229, 5, 70, "ExampleText",
 CellID->20514],
Cell[CellGroupData[{
Cell[142736, 2724, 418, 13, 70, "Input",
 CellID->8938],
Cell[143157, 2739, 158, 5, 36, "Output",
 CellID->545226291]
}, Open  ]],
Cell[143330, 2747, 121, 3, 70, "ExampleDelimiter",
 CellID->14990],
Cell[143454, 2752, 234, 6, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[143713, 2762, 220, 7, 70, "Input",
 CellID->28981],
Cell[143936, 2771, 259, 9, 36, "Output",
 CellID->18769947]
}, Open  ]],
Cell[CellGroupData[{
Cell[144232, 2785, 385, 13, 70, "Input",
 CellID->5242],
Cell[144620, 2800, 211, 7, 36, "Output",
 CellID->538062242]
}, Open  ]],
Cell[144846, 2810, 124, 3, 70, "ExampleDelimiter",
 CellID->13974924],
Cell[144973, 2815, 468, 16, 70, "ExampleText",
 CellID->206794640],
Cell[CellGroupData[{
Cell[145466, 2835, 173, 6, 70, "Input",
 CellID->380460473],
Cell[145642, 2843, 287, 10, 46, "Output",
 CellID->975565037]
}, Open  ]],
Cell[CellGroupData[{
Cell[145966, 2858, 164, 5, 70, "Input",
 CellID->32069594],
Cell[146133, 2865, 286, 10, 46, "Output",
 CellID->44621144]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[146468, 2881, 327, 9, 70, "ExampleSection",
 CellID->16070],
Cell[146798, 2892, 455, 14, 70, "ExampleText",
 CellID->25631],
Cell[CellGroupData[{
Cell[147278, 2910, 162, 5, 70, "Input",
 CellID->24061],
Cell[147443, 2917, 317, 6, 70, "Message",
 CellID->27219],
Cell[147763, 2925, 251, 8, 36, "Output",
 CellID->28781]
}, Open  ]],
Cell[148029, 2936, 121, 3, 70, "ExampleDelimiter",
 CellID->19046],
Cell[148153, 2941, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[148459, 2953, 260, 8, 70, "Input",
 CellID->25838],
Cell[148722, 2963, 277, 10, 60, "Output",
 CellID->562257353]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[149060, 2980, 311, 9, 70, "SeeAlsoSection",
 CellID->16984],
Cell[149374, 2991, 487, 17, 70, "SeeAlso",
 CellID->23953]
}, Open  ]],
Cell[CellGroupData[{
Cell[149898, 3013, 313, 9, 70, "TutorialsSection",
 CellID->1899],
Cell[150214, 3024, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[150408, 3032, 299, 8, 70, "MoreAboutSection"],
Cell[150710, 3042, 184, 4, 70, "MoreAbout",
 CellID->162818315]
}, Open  ]],
Cell[150909, 3049, 27, 0, 70, "History"],
Cell[150939, 3051, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

