(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     68623,       2403]
NotebookOptionsPosition[     56349,       1971]
NotebookOutlinePosition[     58017,       2013]
CellTagsIndexPosition[     57931,       2008]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Digits in Numbers" :> 
          Documentation`HelpLookup["paclet:tutorial/DigitsInNumbers"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Digits in Numbers\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"MantissaExponent" :> 
          Documentation`HelpLookup["paclet:ref/MantissaExponent"], 
          "IntegerDigits" :> 
          Documentation`HelpLookup["paclet:ref/IntegerDigits"], "BaseForm" :> 
          Documentation`HelpLookup["paclet:ref/BaseForm"], "FromDigits" :> 
          Documentation`HelpLookup["paclet:ref/FromDigits"], 
          "ContinuedFraction" :> 
          Documentation`HelpLookup["paclet:ref/ContinuedFraction"], 
          "MultiplicativeOrder" :> 
          Documentation`HelpLookup["paclet:ref/MultiplicativeOrder"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"MantissaExponent\"\>", 
       2->"\<\"IntegerDigits\"\>", 3->"\<\"BaseForm\"\>", 
       4->"\<\"FromDigits\"\>", 5->"\<\"ContinuedFraction\"\>", 
       6->"\<\"MultiplicativeOrder\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Atomic Elements of Expressions" :> 
          Documentation`HelpLookup[
           "paclet:guide/AtomicElementsOfExpressions"], "Constructing Lists" :> 
          Documentation`HelpLookup["paclet:guide/ConstructingLists"], 
          "Number Digits" :> 
          Documentation`HelpLookup["paclet:guide/NumberDigits"], 
          "Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheory"], 
          "Numerical Evaluation & Precision" :> 
          Documentation`HelpLookup[
           "paclet:guide/NumericalEvaluationAndPrecision"], 
          "Representation of Numbers" :> 
          Documentation`HelpLookup[
           "paclet:guide/RepresentationOfNumbers"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Atomic Elements of Expressions\"\>", 
       2->"\<\"Constructing Lists\"\>", 3->"\<\"Number Digits\"\>", 
       4->"\<\"Number Theory\"\>", 
       5->"\<\"Numerical Evaluation & Precision\"\>", 
       6->"\<\"Representation of Numbers\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RealDigits", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RealDigits",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RealDigits"], "[", 
       StyleBox["x", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of the digits in the approximate real \
number ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     ", together with the number of digits that are to the left of the \
decimal point. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RealDigits",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RealDigits"], "[", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        StyleBox["b", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of base\[Hyphen]",
     Cell[BoxData[
      StyleBox["b", "TI"]], "InlineFormula"],
     " digits in ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RealDigits",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RealDigits"], "[", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        StyleBox["b", "TI"], ",", 
        StyleBox["len", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of ",
     Cell[BoxData[
      StyleBox["len", "TI"]], "InlineFormula"],
     " digits. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RealDigits",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RealDigits"], "[", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        StyleBox["b", "TI"], ",", 
        StyleBox["len", "TI"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives ",
     Cell[BoxData[
      StyleBox["len", "TI"]], "InlineFormula"],
     " digits starting with the coefficient of ",
     Cell[BoxData[
      SuperscriptBox[
       StyleBox["b", "TI"], 
       StyleBox["n", "TI"]]], "InlineFormula"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->13334]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RealDigits",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RealDigits"]], "InlineFormula"],
 " gives the most significant digits first, as in standard positional \
notation."
}], "Notes",
 CellID->19959147],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " normally returns a list of digits of length ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Round",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Round"], "[", 
   RowBox[{"Precision", "[", 
    StyleBox["x", "TI"], "]"}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->19826],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["b", "TI"]}], "]"}]], "InlineFormula"],
 " normally require that ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " be an approximate real number, returned for example by ",
 Cell[BoxData[
  ButtonBox["N",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/N"]], "InlineFormula"],
 ". ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["b", "TI"], ",", 
    StyleBox["len", "TI"]}], "]"}]], "InlineFormula"],
 " also works on exact numbers. "
}], "Notes",
 CellID->3996],

Cell[TextData[{
 "For integers and rational numbers with terminating digit expansions, ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " returns an ordinary list of digits. For rational numbers with \
non\[Hyphen]terminating digit expansions it yields a list of the form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    SubscriptBox[
     StyleBox["a", "TI"], 
     StyleBox["1", "TR"]], ",", 
    SubscriptBox[
     StyleBox["a", "TI"], 
     StyleBox["2", "TR"]], ",", 
    StyleBox["\[Ellipsis]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["b", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["b", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}]], "InlineFormula"],
 " representing the digit sequence consisting of the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["a", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " followed by infinite cyclic repetitions of the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["b", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 ". ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->8532,
  ButtonNote->"8532"]
}], "Notes",
 CellID->12432],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  StyleBox["len", "TI"]], "InlineFormula"],
 " is larger than ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["Log",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Log"], "[", 
    RowBox[{"10", ",", 
     StyleBox["b", "TI"]}], "]"}], 
   RowBox[{
    ButtonBox["Precision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Precision"], "[", 
    StyleBox["x", "TI"], "]"}]}]], "InlineFormula"],
 ", then remaining digits are filled in as ",
 Cell[BoxData[
  ButtonBox["Indeterminate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Indeterminate"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->8394],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["b", "TI"], ",", 
    StyleBox["len", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " starts with the digit which is the coefficient of ",
 Cell[BoxData[
  SuperscriptBox[
   StyleBox["b", "TI"], 
   StyleBox["n", "TI"]]], "InlineFormula"],
 ", truncating or padding with zeros as necessary. ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->1430,
  ButtonNote->"1430"]
}], "Notes",
 CellID->31371],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["b", "TI"], ",", 
    StyleBox["len", "TI"], ",", 
    RowBox[{"-", "1"}]}], "]"}]], "InlineFormula"],
 " starts with the digit immediately to the right of the base\[Hyphen]",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " decimal point in ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->17127],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["b", "TI"], ",", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " gives as many digits as it can in a fixed precision number."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->849484991],

Cell[TextData[{
 "The base ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " in ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["b", "TI"]}], "]"}]], "InlineFormula"],
 " need not be an integer. For any real ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " such that ",
 Cell[BoxData[
  RowBox[{
   StyleBox["b", "TI"], ">", "1"}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["b", "TI"]}], "]"}]], "InlineFormula"],
 " successively finds the largest integer multiples of powers of ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " that can be removed while leaving a non\[Hyphen]negative remainder. "
}], "Notes",
 CellID->10792],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " discards the sign of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->12249],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RealDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RealDigits"], "[", "0.", "]"}]], 
  "InlineFormula"],
 " gives ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", "0", "}"}], ",", 
    RowBox[{"-", 
     RowBox[{
      ButtonBox["Floor",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Floor"], "[", 
      RowBox[{
       ButtonBox["Accuracy",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Accuracy"], "[", "0.", "]"}], "]"}]}]}], 
   "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->14108],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["FromDigits",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FromDigits"]], "InlineFormula"],
 " can be used as the inverse of ",
 Cell[BoxData[
  ButtonBox["RealDigits",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RealDigits"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->14397]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->24707639],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->618462491],

Cell["Give the list of digits and exponent:", "ExampleText",
 CellID->22760],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", "123.55555", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16613],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "5", ",", "5", ",", "5", ",", "5", ",", "5",
      ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
     ",", "0"}], "}"}], ",", "3"}], "}"}]], "Output",
 ImageSize->{330, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15555]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", "0.000012355555", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2775],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "5", ",", "5", ",", "5", ",", "5", ",", "5",
      ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
     ",", "0"}], "}"}], ",", 
   RowBox[{"-", "4"}]}], "}"}]], "Output",
 ImageSize->{338, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1022]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13588],

Cell["\<\
Give 25 digits of \[Pi] in base 10:\
\>", "ExampleText",
 CellID->14371],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"Pi", ",", "10", ",", "25"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26673],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "3", ",", "1", ",", "4", ",", "1", ",", "5", ",", "9", ",", "2", ",", "6",
      ",", "5", ",", "3", ",", "5", ",", "8", ",", "9", ",", "7", ",", "9", 
     ",", "3", ",", "2", ",", "3", ",", "8", ",", "4", ",", "6", ",", "2", 
     ",", "6", ",", "4", ",", "3"}], "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{494, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->30660]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->18312],

Cell["Give 25 digits of 19/7:", "ExampleText",
 CellID->950],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{
   RowBox[{"19", "/", "7"}], ",", "10", ",", "25"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6128],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "2", ",", "7", ",", "1", ",", "4", ",", "2", ",", "8", ",", "5", ",", "7",
      ",", "1", ",", "4", ",", "2", ",", "8", ",", "5", ",", "7", ",", "1", 
     ",", "4", ",", "2", ",", "8", ",", "5", ",", "7", ",", "1", ",", "4", 
     ",", "2", ",", "8", ",", "5"}], "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{494, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->24420]
}, Open  ]],

Cell["Give an explicit recurring decimal form:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->8532],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"19", "/", "7"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25744],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "1", ",", "4", ",", "2", ",", "8", ",", "5"}], 
      "}"}]}], "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->18918]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8191],

Cell["Base-2 digits:", "ExampleText",
 CellID->7022],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"1.234", ",", "2", ",", "15"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28651],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "0", ",", "1",
      ",", "1", ",", "1", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1"}], 
    "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{312, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->29317]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11573],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"Pi", ",", "10", ",", "20"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11677],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "3", ",", "1", ",", "4", ",", "1", ",", "5", ",", "9", ",", "2", ",", "6",
      ",", "5", ",", "3", ",", "5", ",", "8", ",", "9", ",", "7", ",", "9", 
     ",", "3", ",", "2", ",", "3", ",", "8", ",", "4"}], "}"}], ",", "1"}], 
  "}"}]], "Output",
 ImageSize->{403, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->29485]
}, Open  ]],

Cell[TextData[{
 "20 digits starting with the coefficient of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", 
    RowBox[{"-", "5"}]], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->1430],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"Pi", ",", "10", ",", "20", ",", 
   RowBox[{"-", "5"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->16061],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "9", ",", "2", ",", "6", ",", "5", ",", "3", ",", "5", ",", "8", ",", "9",
      ",", "7", ",", "9", ",", "3", ",", "2", ",", "3", ",", "8", ",", "4", 
     ",", "6", ",", "2", ",", "6", ",", "4", ",", "3"}], "}"}], ",", 
   RowBox[{"-", "4"}]}], "}"}]], "Output",
 ImageSize->{411, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->13591]
}, Open  ]],

Cell[TextData[{
 "20 digits starting with the coefficient of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", "5"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->21448],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"Pi", ",", "10", ",", "20", ",", "5"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->31974],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "3", ",", "1", ",", "4",
      ",", "1", ",", "5", ",", "9", ",", "2", ",", "6", ",", "5", ",", "3", 
     ",", "5", ",", "8", ",", "9", ",", "7", ",", "9"}], "}"}], ",", "6"}], 
  "}"}]], "Output",
 ImageSize->{403, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->4779]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31728],

Cell["Non-integer bases are allowed:", "ExampleText",
 CellID->26263],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"Pi", ",", "GoldenRatio", ",", "20"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->179],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1",
      ",", "0", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", 
     ",", "0", ",", "0", ",", "0", ",", "1", ",", "0"}], "}"}], ",", "3"}], 
  "}"}]], "Output",
 ImageSize->{403, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->31703]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26352],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RealDigits",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RealDigits"]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  ButtonBox["Indeterminate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Indeterminate"]], "InlineFormula"],
 " if more digits than the precision are requested:"
}], "ExampleText",
 CellID->155071403],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"5.635", ",", "10", ",", "20"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->646526852],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "5", ",", "6", ",", "3", ",", "5", ",", "0", ",", "0", ",", "0", ",", "0",
      ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
     ",", "0", ",", "Indeterminate", ",", "Indeterminate", ",", 
     "Indeterminate", ",", "Indeterminate"}], "}"}], ",", "1"}], 
  "}"}]], "Output",
 ImageSize->{448, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->344579450]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->215668641],

Cell[TextData[{
 "Start at the coefficient of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", 
    RowBox[{"-", "15"}]], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->314154073],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"5.635", ",", "10", ",", "20", ",", 
   RowBox[{"-", "15"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->70919702],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "9", ",", "7", ",", "8", ",", "6", ",", "8", ",", "3", ",", "7", ",", "1",
      ",", "7", ",", "9", ",", "2", ",", "7", ",", "1", ",", "9", ",", "2", 
     ",", "Indeterminate", ",", "Indeterminate", ",", "Indeterminate", ",", 
     "Indeterminate", ",", "Indeterminate"}], "}"}], ",", 
   RowBox[{"-", "14"}]}], "}"}]], "Output",
 ImageSize->{463, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->83685711]
}, Open  ]],

Cell["\<\
Include only digits that are determined by the precision available:\
\>", "ExampleText",
 CellID->44108222],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"5.635", ",", "10", ",", "Automatic", ",", 
   RowBox[{"-", "15"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->313215531],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "9", ",", "7", ",", "8", ",", "6", ",", "8", ",", "3", ",", "7", ",", "1",
      ",", "7", ",", "9", ",", "2", ",", "7", ",", "1", ",", "9", ",", "2", 
     ",", "0"}], "}"}], ",", 
   RowBox[{"-", "14"}]}], "}"}]], "Output",
 ImageSize->{345, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->112152703]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27583],

Cell[TextData[{
 "The 10000",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " digit of ",
 Cell[BoxData[
  StyleBox["\[Pi]", "TR"]], "InlineFormula"],
 " is an 8: "
}], "ExampleText",
 CellID->21133],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"Pi", ",", "10", ",", "1", ",", 
   RowBox[{"-", "10000"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29570],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", "8", "}"}], ",", 
   RowBox[{"-", "9999"}]}], "}"}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17591]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22609],

Cell[TextData[{
 "Number of 1s in the first million base-2 digits of ",
 Cell[BoxData[
  StyleBox["\[Pi]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->23942],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Count", "[", 
  RowBox[{
   RowBox[{"First", "[", 
    RowBox[{"RealDigits", "[", 
     RowBox[{"Pi", ",", "2", ",", 
      RowBox[{"10", "^", "6"}]}], "]"}], "]"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19660],

Cell[BoxData["499722"], "Output",
 ImageSize->{49, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1396]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19245580],

Cell[TextData[{
 "Distribution of first 100000 digits of ",
 Cell[BoxData[
  StyleBox["\[Pi]", "TR"]], "InlineFormula"],
 " in base 47:"
}], "ExampleText",
 CellID->29570402],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"BinCounts", "[", 
    RowBox[{
     RowBox[{"First", "[", 
      RowBox[{"RealDigits", "[", 
       RowBox[{"Pi", ",", "47", ",", "100000"}], "]"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "46"}], "}"}]}], "]"}], ",", 
   RowBox[{"DataRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"0", ",", "46"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->175412740],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWEtrFEEQnn2rIRI0j0128yQJgolKXAkJhhAWDRgCehBBJbAsYoLE1SQE
gh5yCCgYDx48+Dh4UP+AQX+AFz159ewfid3TM10939T2zG7Wg4HMpKaqvu/r
nurqnixXtlbvrVe21qqVocWNyqPVterm0NXahniUSjiOcyB+7+cd8feh+Mv9
PTg8FPfwtUdeUzLC/XH6xEWaSXFPinuvuKfBlv6UZxdkUlrne0+7Nba0isEY
D81kPenanz7+mp9783L/u5OVdkb59TWU1WvYnD8fyXuanYNCZJ7Se+vmZ6H3
Svm9k4vMiNLa7c2zk9FD9iK7jqTx548/tYdfvn397RyLUOC/rWQIOc9lBiKa
1Xj0N1hfdbOa3NrNmhHNaY+jzF9BSWNF9YHd6IozNbxt0dNOcf0gBxnyp8FO
RfgbjW813v+uN8G+pYEIezDCz68VviKeuTb16g4dJevwueulzthhwaDexEet
ANOMJYoYbVHEOBNQ/a6B8XeCqgWw5y1ZpHIBbD7rBGDnYBQXwd8GfkKVo2wD
zjmwly0aCHNW2lnNqR189iJwXLdEEccYjOsMm1UDbIqSox0BjCXg4DGHAbPI
opB9nFC88wr5BoGxCNhLYCss/7RlnsL83p8xeCivFFEl4+AfAb/iVa3G387c
vlR3n+T2u6ShjbBLMMYLYJ9z0VishtmliZWDp6AJaee0P6BVaUkZ49D0ii/G
qQDtMRgvrrmpf8WZ1ePSg52FuZjy3rvJ0OR7D+1oWG004ppBpH786sZ7hkUm
JKxjheyv/zR4sZfiqljR2SZnB3COg90FKHmwz1pQcYWS3QUsebB51BuAkgeb
z9oFxT2WKFJAUXKuq4BRAGYecy6iUi5ZsupXAZ81AFxFyLptySKuncCoBwFj
h8XgTxEF0PPUjer2n4pAUkn8ZpR4KqL016eBdQp0mVniqbfqcoD9itWZBawn
wKW+oM09y99Hgmsb1/IesCscZg+qc1aO04n2QLviiNe/4rOUYUZwN+53cfgO
n4nBJc0yzBbuqjYO/w2E91T8tijDfJXA7vfSQvsFy2nn2oZZGjbggpWTNvRz
lbQNc6GQsI63YSwUJTH3QQ3uYVkW8wUw4/7EZ/UA9mNQlgLbhkJcu5Yo3Few
45JdDswKntWwP5dZzvPAif2Zz3oNSpJs1B2ImgZFtiyKmgZFlCVHfRe8l1nM
Vv1nJc4XaNoJrgxzNdTrryoz2E/NEzrfc+L0O4XcmhN7GBd7vg3F1jFbh2ba
7jeuUX/XVF7EGb0ZpnZgwlPPJPj7FG4Tp+p2QFZMVP+T4FdMWKntkLUONmVJ
Daj+AYs5Acy2KGLiozaBcdQSRYyjAdVbwMRjeCsv8RfNRg0Q\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->307561183]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32380],

Cell["Fibonacci representations of integers:", "ExampleText",
 CellID->18199],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"RealDigits", "[", 
    RowBox[{"n", ",", "GoldenRatio", ",", "10"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15392],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWl1OwzAMTtdtjJ/B4InHnYFDwBOHqCak7QGBxm4J5wAegCuUJKVjspLG
zk+3tJm0Nnbc1f5qu8mn3Reb5cNjsVktivndunherhYv89unNVflGWPZK2Ps
5prxcclH8is+b/xbDavRqTjm9WzJhkA+BvKFZ3nmeD301zTfdryu8bnGm4sD
OzFYifkBF2fI82WDHt69SW96Gli/dP5Q/YL3H6DQE1YjLkL9ET8IcQT048D2
E41eZ0/VY/0w1ZYrjlS9DkcqXtTnQcFHiDkCJ3NWqmSR88IbNtrq5W+x7azq
GYW6atI423yt7SzNS5V1GNxdZnXou+Br90RpSMLuPzAgWeW9qYvAKpsqfYLv
TRerc6VWbUvRwkjgWgL2iXMvyFC0EB8KEjhsccic8cNYk2m7clbadM54MwhG
Atc7sOJghtkhE0MG4SIxxW/3Ju5OPglvBohO1Z88kvcd/ssph1JPcqmuWk7d
yFcPT/lEyycTRyOsaq4l//O4Sd7Vq84qr1X6Kwu/4LnmgFR6eJ2qbwk70269
YuDEEHKv9FFia8PGm9jamNjaWj9EoeaLXfTF/rXN0ur87ipLu2+2OyRL2y53
l7haf9GG4GoPFf3qbeaHr/W/Om6LtZ0CeQzkuFjb9lltWRmedrXxZBF15RjX
rrZ9Fkl2od5nkanW+p5FOj47ZU+JqLG+Z0/qQRiUusjS7gulKqNsGNr6rNLH
wNC6+CUrUArujCxmR9nErsTO0H5IFEwc+KEwmJ9W3oZmvLrCHL5bods+B9MF
HszEqtR2VX2a/lkXnnM4hJUIjAoyL3D+q/SRz93Ajvpe+E7YJewSdhFiR8UK
s4p33V2oOJkf6a0YYtbpLPsFB+KJcQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{459, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->460042417]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->173472172],

Cell["Binary representation of a machine number:", "ExampleText",
 CellID->478043062],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", " ", "=", " ", 
  RowBox[{"RandomReal", "[", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->145938584],

Cell[BoxData["0.454416429097374`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->55631345]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"d", ",", "e"}], "}"}], "=", 
  RowBox[{"RealDigits", "[", 
   RowBox[{"r", ",", "2"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->63138852],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "1", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0",
      ",", "1", ",", "0", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", 
     ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1", 
     ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0", ",", "1", 
     ",", "0", ",", "1", ",", "1", ",", "0", ",", "1", ",", "1", ",", "1", 
     ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "0", 
     ",", "0", ",", "0", ",", "1", ",", "0", ",", "1", ",", "1", ",", "0", 
     ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"-", "1"}]}], "}"}]], "Output",
 ImageSize->{524, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->67568005]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   ButtonBox["MachinePrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MachinePrecision"], TraditionalForm]], 
  "InlineFormula"],
 " is equal to the number of bits times ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["log", "10"], "(", "2", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->170452486],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MachinePrecision", "==", 
  RowBox[{
   RowBox[{"Length", "[", "d", "]"}], "*", 
   RowBox[{"Log", "[", 
    RowBox[{"10.", ",", "2"}], "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->278178530],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->73714518]
}, Open  ]],

Cell["Get the next larger machine number:", "ExampleText",
 CellID->128409941],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"d", "[", 
    RowBox[{"[", 
     RowBox[{"-", "1"}], "]"}], "]"}], "+=", "1"}], ";"}], "\n", 
 RowBox[{"rp", "=", 
  RowBox[{"N", "[", 
   RowBox[{"FromDigits", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"d", ",", "e"}], "}"}], ",", "2"}], "]"}], "]"}]}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->245647836],

Cell[BoxData["0.45441642909737406`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->797657275]
}, Open  ]],

Cell[TextData[{
 "The spacing between these numbers is ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{"(", 
     RowBox[{
      StyleBox["e", "TI"], "-", "1"}], ")"}]], " ", 
   ButtonBox["$MachineEpsilon",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/$MachineEpsilon"]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->2326940],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"rp", " ", "-", "r"}], " ", "\[Equal]", " ", 
  RowBox[{
   RowBox[{"2", "^", 
    RowBox[{"(", 
     RowBox[{"e", "-", "1"}], ")"}]}], "*", "$MachineEpsilon"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->10121718],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->308442335]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->99187183],

Cell["Find the error in representing 1/10 with a machine number:", \
"ExampleText",
 CellID->249954367],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{".1", ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->171851818],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWNtugzAMTaGsrNoGfZjWvnU/sz3tI1A1qX2oNnX9f5YQsiHPuXBfwJUo
+GDs4Bwcx2/Z9fh+zq6nQ7Z/vWSfx9Pha//yceFQuGAsiBljz1vGr3N+VRzq
t+bHryilJ/Ef/mixpGM5nZm9tvaXQIbzs8vVrAUcTB3Pm5H0dPob8FZhie8A
vi3eVsQo4iCMzY0GX/E/IUaOuM6ODo97tq+zo/PblX0T7kP8644H6iUaPclC
yFnxtODsrQYXXllUzb+sHJv9rs6mDo8H9GW2Zh5Jt77Md9fezlf9EWLaiZM2
zm24Kt2jtnDUtsI9dGgLfwr30MwWjtpW9f7j5eYRZrK7XGbIFcBDIBdVB7Fi
oqyAOQBWXQGQKUfMiQ1qb+FWBRELzOPylQWUE9rZmhYbHp20JCewHkBYelZn
2Asw3VfPp5b7SUWvqf+qPpYXMVx2ZFRcsB5bNxJW51dluGsYu5Plu72hx1vU
4H8qcTjLggs+d/5sekunKIjVyIeOFHUE5xn/olqysljXufn/HSrqKNJ8NZsv
ucrB3QT8EuD3RPsK87jG31f0H6+CjbVz6hQjQcwh5hBziDlDRoKYQ8wh5gzH
HBtTRAXta5fVxX+7XilbfANJi8w1\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{520, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->39214202]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"err", " ", "=", " ", 
  RowBox[{
   RowBox[{"FromDigits", "[", 
    RowBox[{"%", ",", "2"}], "]"}], "-", 
   RowBox[{"1", "/", "10"}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->519534650],

Cell[BoxData[
 FractionBox["1", "180143985094819840"]], "Output",
 ImageSize->{145, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->779968040]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->649956795],

Cell[BoxData["5.551115123125783`*^-18"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->428888548]
}, Open  ]],

Cell["The next smaller machine number is farther away:", "ExampleText",
 CellID->209404149],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"d", ",", "e"}], "}"}], "=", 
   RowBox[{"RealDigits", "[", 
    RowBox[{".1", ",", "2"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"d", "[", 
    RowBox[{"[", 
     RowBox[{"-", "1"}], "]"}], "]"}], "-=", "1"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"N", "[", 
   RowBox[{"FromDigits", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"d", ",", "e"}], "}"}], ",", "2"}], "]"}], "]"}], "-", 
  RowBox[{"1", "/", "10"}]}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->111814592],

Cell[BoxData[
 RowBox[{"-", "1.3877787807814457`*^-17"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->389493842]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18339],

Cell["\<\
The default number of digits returned is determined by the precision of the \
number:\
\>", "ExampleText",
 CellID->228238042],

Cell[BoxData[
 RowBox[{
  RowBox[{"p", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"{", 
     RowBox[{"20", ",", "100"}], "}"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->48807940],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"d", ",", "e"}], "}"}], "=", 
  RowBox[{"RealDigits", "[", 
   RowBox[{"N", "[", 
    RowBox[{"Pi", ",", "p"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->46858739],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "3", ",", "1", ",", "4", ",", "1", ",", "5", ",", "9", ",", "2", ",", "6",
      ",", "5", ",", "3", ",", "5", ",", "8", ",", "9", ",", "7", ",", "9", 
     ",", "3", ",", "2", ",", "3", ",", "8", ",", "4", ",", "6", ",", "2", 
     ",", "6", ",", "4", ",", "3", ",", "3", ",", "8", ",", "3", ",", "2", 
     ",", "7", ",", "9", ",", "5", ",", "0", ",", "2", ",", "8", ",", "8", 
     ",", "4", ",", "1", ",", "9", ",", "7", ",", "1", ",", "6", ",", "9", 
     ",", "3", ",", "9", ",", "9", ",", "3", ",", "7", ",", "5", ",", "1", 
     ",", "0", ",", "5", ",", "8", ",", "2", ",", "0", ",", "9", ",", "7", 
     ",", "4", ",", "9", ",", "4", ",", "4", ",", "5", ",", "9", ",", "2", 
     ",", "3", ",", "0", ",", "7", ",", "8", ",", "1", ",", "6", ",", "4"}], 
    "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{476, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->534092626]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Length", "[", "d", "]"}], " ", "\[Equal]", " ", 
  RowBox[{"Round", "[", "p", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->247545568],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->180811658]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->10454],

Cell[TextData[{
 "Digits unknown at the available precision are filled in as ",
 Cell[BoxData[
  ButtonBox["Indeterminate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Indeterminate"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->15681],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RealDigits", "[", 
  RowBox[{"1.234", ",", "10", ",", "20"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12991],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "4", ",", "0", ",", "0", ",", "0", ",", "0",
      ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
     ",", "0", ",", "1", ",", "Indeterminate", ",", "Indeterminate", ",", 
     "Indeterminate"}], "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{401, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->22842]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->149330973],

Cell[TextData[StyleBox["For non-binary bases, the digits given may not be \
enough to reconstruct the number exactly:", "ExampleText"]], "ExampleText",
 CellID->93638598],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"x", " ", "=", " ", 
   RowBox[{"1", " ", "+", "$MachineEpsilon"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"d", ",", "e"}], "}"}], "=", 
  RowBox[{"RealDigits", "[", 
   RowBox[{"1", "+", "$MachineEpsilon"}], "]"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->173606953],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0",
      ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", 
     ",", "0"}], "}"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{330, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->904457770]
}, Open  ]],

Cell[TextData[{
 "More than ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Round",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Round"], "[", 
   ButtonBox["MachinePrecision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MachinePrecision"], "]"}]], "InlineFormula"],
 " decimal digits are required to separate ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " from 1:"
}], "ExampleText",
 CellID->678487227],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FromDigits", "[", 
  RowBox[{"{", 
   RowBox[{"d", ",", "e"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->394382648],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->906486664]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["InputForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InputForm"]], "InlineFormula"],
 " uses a sufficient number of digits to uniquely reconstruct the number:"
}], "ExampleText",
 CellID->3268615],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InputForm", "[", "x", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->43949675],

Cell["1.0000000000000002", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//InputForm=",
 CellID->380277792]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "-", "1.0000000000000002"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->142748628],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->126998155]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2783],

Cell[TextData[{
 "A hundred digits of ",
 Cell[BoxData[
  StyleBox["\[Pi]", "TR"]], "InlineFormula"],
 " starting with the millionth digit: "
}], "ExampleText",
 CellID->285236694],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Row", "[", 
  RowBox[{"First", "@", 
   RowBox[{"RealDigits", "[", 
    RowBox[{"Pi", ",", "10", ",", "100", ",", 
     RowBox[{"-", 
      RowBox[{"10", "^", "6"}]}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->931483473],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWttu2zAMVXxLuy/YY/cz28u6fUNQDGgfhg1dP2L7as9RLEemD0VSspNg
K9DKlnUhD0mJOurj4e352/fD28vT4eHT6+Hn88vTr4ePP16Hqnrn3O63c+7D
ezc898OT/z3+/Bl+T4/80/3wtx46NCuV9VC2Q1mNZT3Wd2PpG7rw05BWNRmt
JaOi764O+vZT+0opHde+BvMHLeL54n5I66afa18JcvgPIzYIDyQXxe/4vvPD
3HlIJmnHoat/orYhlqDvuBe1Xw1b+drm3Gqsof3o7NpxNDVUk+i9yx91m3l0
eCbQo+80qiHaoq0THlCMFZQgU6JWhVnJDPbVIMf2UkQ2GWNSrfWW71hkcS1d
GRhpF/3uRR2IlrFsElqL2fZA/25yF18tYsZolughxac0I8Q2/o40UK3dHZCe
8V0TAnAtxHvQfxZXeKVq3HJHTrcOpW53l22q23lUWG6EE9EhqX9AB0UawloV
cRfxEaXllDhIelEZzztd8DGMLc2pA9aVm58tKuKxeXnoZLEZDuZsoDhrQRam
p5QFuosR0bkK5tX8Hpi0d8AcxxVvibw8Cu11sfVNdpOiZTYON3Psl7z3ovmq
0RI7d8r5orxPd1YR11OKHt77ljlRYv80rL6xl03SrJnhy/twZjYZa4LQlpBA
eR/XvgwZbCl5/9XElVYjQQOlxyAPM3h5Jg+BJAzxOEZG3nn5BlpJXpo6J1hx
vWxtiV/Q2LPGth3r6+ucx8nZvWwdTemIl18V7GzNu37JiDduvv9bv1funNd0
/TLLRZmdpoznDQw0yp5pGcu1F+Sg+lDmm+rXknk6Yfz4O+qP9K4iuUK9t6k3
qOb2xf4UeBIfwUC++L1lfJ22RzcWXD29yVi7Pa2vGT1QhKGTCHcjw90I0Rsd
Tr7Q/os3B8yGbob/vYXaHOarROdVTwGbSSl5tXQTcqnbtU24CjOGONaUfO9i
NHoubvE4lhOfsq3i1kJ5xtHpr7ohKJJdz/2KLKXV+xNnSjvPD0+pGXc+0rhf
e4MnL7hSCSfGk83xJrGtK+0K5b5+xdvbPFZcE+OI/1dFUDFPg3ysnAHCq7fM
+OZxrSFXaxg/js8+eO1exl7IAzEnHp88Uhk3ZpQbN8+Zc/4rRefHeTfP0l0C
8qP4NiXYAq5V5tnC6DruhM752CNPXO1uVkT8Lm/cgpwKYSpxygVsTQL7faEm
07t5L5HXRtXtsGhdKocyPy1axemcn3vk4dflVHW852VllLhb3fqSFynb41PC
Qt5CrYQQ/D+kq/gOP/sp6zl+Q2whx0bFWVioT/VLsZmIbeXYUbpbW9lPyo6l
2NpQUj01OHFynM5VR+xVXOruL/wPZbU=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{392, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->433226062]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->6631],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["MantissaExponent",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MantissaExponent"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["IntegerDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/IntegerDigits"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BaseForm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BaseForm"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FromDigits",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FromDigits"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ContinuedFraction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ContinuedFraction"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MultiplicativeOrder",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MultiplicativeOrder"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->31236]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->4172],

Cell[TextData[ButtonBox["Digits in Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DigitsInNumbers"]], "Tutorials",
 CellID->21819]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->16585],

Cell[TextData[{
 ButtonBox["Demonstrations with RealDigits",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=RealDigits"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#23292"], None}]], "RelatedLinks",
 CellID->10752],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
RealDigits"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->84355834],

Cell[TextData[ButtonBox["Atomic Elements of Expressions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AtomicElementsOfExpressions"]], "MoreAbout",
 CellID->142442474],

Cell[TextData[ButtonBox["Constructing Lists",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ConstructingLists"]], "MoreAbout",
 CellID->10162940],

Cell[TextData[ButtonBox["Number Digits",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberDigits"]], "MoreAbout",
 CellID->31627898],

Cell[TextData[ButtonBox["Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheory"]], "MoreAbout",
 CellID->83135831],

Cell[TextData[ButtonBox["Numerical Evaluation & Precision",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumericalEvaluationAndPrecision"]], "MoreAbout",
 CellID->440031049],

Cell[TextData[ButtonBox["Representation of Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/RepresentationOfNumbers"]], "MoreAbout",
 CellID->194194094]
}, Open  ]],

Cell["New in 2  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RealDigits - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 14, 21.9697454}", "context" -> "System`", 
    "keywords" -> {
     "bases", "digit sequences", "digits of real numbers", 
      "recurring decimals", "representation of numbers", 
      "floating point representation", "binary representation"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "RealDigits[x] gives a list of the digits in the approximate real number \
x, together with the number of digits that are to the left of the decimal \
point. RealDigits[x, b] gives a list of base-b digits in x. RealDigits[x, b, \
len] gives a list of len digits. RealDigits[x, b, len, n] gives len digits \
starting with the coefficient of b^n. ", "synonyms" -> {"real digits"}, 
    "title" -> "RealDigits", "type" -> "Symbol", "uri" -> "ref/RealDigits"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[18568, 577, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->24707639]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 57788, 2001}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3169, 71, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3792, 98, 4194, 83, 70, "ObjectNameGrid"],
Cell[7989, 183, 2262, 73, 70, "Usage",
 CellID->13334]
}, Open  ]],
Cell[CellGroupData[{
Cell[10288, 261, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[10766, 278, 253, 8, 70, "Notes",
 CellID->19959147],
Cell[11022, 288, 476, 17, 70, "Notes",
 CellID->19826],
Cell[11501, 307, 992, 36, 70, "Notes",
 CellID->3996],
Cell[12496, 345, 1365, 45, 70, "Notes",
 CellID->12432],
Cell[13864, 392, 661, 25, 70, "Notes",
 CellID->8394],
Cell[14528, 419, 643, 22, 70, "Notes",
 CellID->31371],
Cell[15174, 443, 543, 19, 70, "Notes",
 CellID->17127],
Cell[15720, 464, 592, 18, 70, "Notes",
 CellID->849484991],
Cell[16315, 484, 966, 34, 70, "Notes",
 CellID->10792],
Cell[17284, 520, 308, 12, 70, "Notes",
 CellID->12249],
Cell[17595, 534, 601, 24, 70, "Notes",
 CellID->14108],
Cell[18199, 560, 332, 12, 70, "Notes",
 CellID->14397]
}, Closed]],
Cell[CellGroupData[{
Cell[18568, 577, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->24707639],
Cell[CellGroupData[{
Cell[18952, 591, 148, 5, 70, "ExampleSection",
 CellID->618462491],
Cell[19103, 598, 76, 1, 70, "ExampleText",
 CellID->22760],
Cell[CellGroupData[{
Cell[19204, 603, 110, 3, 28, "Input",
 CellID->16613],
Cell[19317, 608, 406, 12, 36, "Output",
 CellID->15555]
}, Open  ]],
Cell[CellGroupData[{
Cell[19760, 625, 114, 3, 28, "Input",
 CellID->2775],
Cell[19877, 630, 424, 13, 36, "Output",
 CellID->1022]
}, Open  ]],
Cell[20316, 646, 121, 3, 70, "ExampleDelimiter",
 CellID->13588],
Cell[20440, 651, 82, 3, 70, "ExampleText",
 CellID->14371],
Cell[CellGroupData[{
Cell[20547, 658, 138, 4, 70, "Input",
 CellID->26673],
Cell[20688, 664, 502, 13, 36, "Output",
 CellID->30660]
}, Open  ]],
Cell[21205, 680, 121, 3, 70, "ExampleDelimiter",
 CellID->18312],
Cell[21329, 685, 60, 1, 70, "ExampleText",
 CellID->950],
Cell[CellGroupData[{
Cell[21414, 690, 161, 5, 70, "Input",
 CellID->6128],
Cell[21578, 697, 502, 13, 36, "Output",
 CellID->24420]
}, Open  ]],
Cell[22095, 713, 126, 2, 70, "ExampleText",
 CellID->8532],
Cell[CellGroupData[{
Cell[22246, 719, 126, 4, 70, "Input",
 CellID->25744],
Cell[22375, 725, 349, 12, 36, "Output",
 CellID->18918]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[22773, 743, 215, 7, 70, "ExampleSection",
 CellID->8191],
Cell[22991, 752, 52, 1, 70, "ExampleText",
 CellID->7022],
Cell[CellGroupData[{
Cell[23068, 757, 140, 4, 70, "Input",
 CellID->28651],
Cell[23211, 763, 395, 12, 36, "Output",
 CellID->29317]
}, Open  ]],
Cell[23621, 778, 121, 3, 70, "ExampleDelimiter",
 CellID->11573],
Cell[CellGroupData[{
Cell[23767, 785, 138, 4, 70, "Input",
 CellID->11677],
Cell[23908, 791, 449, 13, 36, "Output",
 CellID->29485]
}, Open  ]],
Cell[24372, 807, 260, 9, 70, "ExampleText",
 CellID->1430],
Cell[CellGroupData[{
Cell[24657, 820, 167, 5, 70, "Input",
 CellID->16061],
Cell[24827, 827, 465, 13, 36, "Output",
 CellID->13591]
}, Open  ]],
Cell[25307, 843, 193, 7, 70, "ExampleText",
 CellID->21448],
Cell[CellGroupData[{
Cell[25525, 854, 148, 4, 70, "Input",
 CellID->31974],
Cell[25676, 860, 448, 13, 36, "Output",
 CellID->4779]
}, Open  ]],
Cell[26139, 876, 121, 3, 70, "ExampleDelimiter",
 CellID->31728],
Cell[26263, 881, 69, 1, 70, "ExampleText",
 CellID->26263],
Cell[CellGroupData[{
Cell[26357, 886, 145, 4, 70, "Input",
 CellID->179],
Cell[26505, 892, 449, 13, 36, "Output",
 CellID->31703]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27003, 911, 239, 7, 70, "ExampleSection",
 CellID->26352],
Cell[27245, 920, 371, 12, 70, "ExampleText",
 CellID->155071403],
Cell[CellGroupData[{
Cell[27641, 936, 145, 4, 70, "Input",
 CellID->646526852],
Cell[27789, 942, 507, 14, 54, "Output",
 CellID->344579450]
}, Open  ]],
Cell[28311, 959, 125, 3, 70, "ExampleDelimiter",
 CellID->215668641],
Cell[28439, 964, 203, 8, 70, "ExampleText",
 CellID->314154073],
Cell[CellGroupData[{
Cell[28667, 976, 174, 5, 70, "Input",
 CellID->70919702],
Cell[28844, 983, 535, 14, 54, "Output",
 CellID->83685711]
}, Open  ]],
Cell[29394, 1000, 117, 3, 70, "ExampleText",
 CellID->44108222],
Cell[CellGroupData[{
Cell[29536, 1007, 182, 5, 70, "Input",
 CellID->313215531],
Cell[29721, 1014, 430, 13, 36, "Output",
 CellID->112152703]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[30200, 1033, 223, 7, 70, "ExampleSection",
 CellID->27583],
Cell[30426, 1042, 236, 10, 70, "ExampleText",
 CellID->21133],
Cell[CellGroupData[{
Cell[30687, 1056, 170, 5, 70, "Input",
 CellID->29570],
Cell[30860, 1063, 245, 9, 36, "Output",
 CellID->17591]
}, Open  ]],
Cell[31120, 1075, 121, 3, 70, "ExampleDelimiter",
 CellID->22609],
Cell[31244, 1080, 172, 6, 70, "ExampleText",
 CellID->23942],
Cell[CellGroupData[{
Cell[31441, 1090, 257, 8, 70, "Input",
 CellID->19660],
Cell[31701, 1100, 158, 5, 36, "Output",
 CellID->1396]
}, Open  ]],
Cell[31874, 1108, 124, 3, 70, "ExampleDelimiter",
 CellID->19245580],
Cell[32001, 1113, 174, 6, 70, "ExampleText",
 CellID->29570402],
Cell[CellGroupData[{
Cell[32200, 1123, 454, 14, 70, "Input",
 CellID->175412740],
Cell[32657, 1139, 1603, 30, 70, "Output",
 Evaluatable->False,
 CellID->307561183]
}, Open  ]],
Cell[34275, 1172, 121, 3, 70, "ExampleDelimiter",
 CellID->32380],
Cell[34399, 1177, 77, 1, 70, "ExampleText",
 CellID->18199],
Cell[CellGroupData[{
Cell[34501, 1182, 253, 8, 70, "Input",
 CellID->15392],
Cell[34757, 1192, 1098, 22, 70, "Output",
 Evaluatable->False,
 CellID->460042417]
}, Open  ]],
Cell[35870, 1217, 125, 3, 70, "ExampleDelimiter",
 CellID->173472172],
Cell[35998, 1222, 85, 1, 70, "ExampleText",
 CellID->478043062],
Cell[CellGroupData[{
Cell[36108, 1227, 134, 4, 70, "Input",
 CellID->145938584],
Cell[36245, 1233, 174, 5, 36, "Output",
 CellID->55631345]
}, Open  ]],
Cell[CellGroupData[{
Cell[36456, 1243, 199, 7, 70, "Input",
 CellID->63138852],
Cell[36658, 1252, 828, 18, 54, "Output",
 CellID->67568005]
}, Open  ]],
Cell[37501, 1273, 395, 15, 70, "ExampleText",
 CellID->170452486],
Cell[CellGroupData[{
Cell[37921, 1292, 223, 7, 70, "Input",
 CellID->278178530],
Cell[38147, 1301, 160, 5, 36, "Output",
 CellID->73714518]
}, Open  ]],
Cell[38322, 1309, 78, 1, 70, "ExampleText",
 CellID->128409941],
Cell[CellGroupData[{
Cell[38425, 1314, 366, 13, 70, "Input",
 CellID->245647836],
Cell[38794, 1329, 177, 5, 36, "Output",
 CellID->797657275]
}, Open  ]],
Cell[38986, 1337, 357, 13, 70, "ExampleText",
 CellID->2326940],
Cell[CellGroupData[{
Cell[39368, 1354, 249, 8, 70, "Input",
 CellID->10121718],
Cell[39620, 1364, 161, 5, 36, "Output",
 CellID->308442335]
}, Open  ]],
Cell[39796, 1372, 124, 3, 70, "ExampleDelimiter",
 CellID->99187183],
Cell[39923, 1377, 103, 2, 70, "ExampleText",
 CellID->249954367],
Cell[CellGroupData[{
Cell[40051, 1383, 130, 4, 70, "Input",
 CellID->171851818],
Cell[40184, 1389, 849, 18, 70, "Output",
 Evaluatable->False,
 CellID->39214202]
}, Open  ]],
Cell[CellGroupData[{
Cell[41070, 1412, 215, 7, 70, "Input",
 CellID->519534650],
Cell[41288, 1421, 196, 6, 51, "Output",
 CellID->779968040]
}, Open  ]],
Cell[CellGroupData[{
Cell[41521, 1432, 97, 3, 70, "Input",
 CellID->649956795],
Cell[41621, 1437, 180, 5, 36, "Output",
 CellID->428888548]
}, Open  ]],
Cell[41816, 1445, 91, 1, 70, "ExampleText",
 CellID->209404149],
Cell[CellGroupData[{
Cell[41932, 1450, 553, 20, 70, "Input",
 CellID->111814592],
Cell[42488, 1472, 199, 6, 36, "Output",
 CellID->389493842]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[42736, 1484, 233, 7, 70, "ExampleSection",
 CellID->18339],
Cell[42972, 1493, 136, 4, 70, "ExampleText",
 CellID->228238042],
Cell[43111, 1499, 201, 7, 70, "Input",
 CellID->48807940],
Cell[CellGroupData[{
Cell[43337, 1510, 230, 8, 70, "Input",
 CellID->46858739],
Cell[43570, 1520, 1007, 20, 71, "Output",
 CellID->534092626]
}, Open  ]],
Cell[CellGroupData[{
Cell[44614, 1545, 174, 5, 70, "Input",
 CellID->247545568],
Cell[44791, 1552, 161, 5, 36, "Output",
 CellID->180811658]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[45001, 1563, 226, 7, 70, "ExampleSection",
 CellID->10454],
Cell[45230, 1572, 248, 8, 70, "ExampleText",
 CellID->15681],
Cell[CellGroupData[{
Cell[45503, 1584, 141, 4, 70, "Input",
 CellID->12991],
Cell[45647, 1590, 488, 13, 54, "Output",
 CellID->22842]
}, Open  ]],
Cell[46150, 1606, 125, 3, 70, "ExampleDelimiter",
 CellID->149330973],
Cell[46278, 1611, 170, 2, 70, "ExampleText",
 CellID->93638598],
Cell[CellGroupData[{
Cell[46473, 1617, 320, 10, 70, "Input",
 CellID->173606953],
Cell[46796, 1629, 410, 12, 36, "Output",
 CellID->904457770]
}, Open  ]],
Cell[47221, 1644, 428, 15, 70, "ExampleText",
 CellID->678487227],
Cell[CellGroupData[{
Cell[47674, 1663, 153, 5, 70, "Input",
 CellID->394382648],
Cell[47830, 1670, 158, 5, 36, "Output",
 CellID->906486664]
}, Open  ]],
Cell[48003, 1678, 248, 7, 70, "ExampleText",
 CellID->3268615],
Cell[CellGroupData[{
Cell[48276, 1689, 104, 3, 70, "Input",
 CellID->43949675],
Cell[48383, 1694, 178, 5, 50, "Output",
 CellID->380277792]
}, Open  ]],
Cell[CellGroupData[{
Cell[48598, 1704, 109, 3, 70, "Input",
 CellID->142748628],
Cell[48710, 1709, 160, 5, 36, "Output",
 CellID->126998155]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[48919, 1720, 223, 7, 70, "ExampleSection",
 CellID->2783],
Cell[49145, 1729, 180, 6, 70, "ExampleText",
 CellID->285236694],
Cell[CellGroupData[{
Cell[49350, 1739, 262, 8, 70, "Input",
 CellID->931483473],
Cell[49615, 1749, 1586, 30, 70, "Output",
 Evaluatable->False,
 CellID->433226062]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[51262, 1786, 310, 9, 70, "SeeAlsoSection",
 CellID->6631],
Cell[51575, 1797, 1501, 53, 70, "SeeAlso",
 CellID->31236]
}, Open  ]],
Cell[CellGroupData[{
Cell[53113, 1855, 313, 9, 70, "TutorialsSection",
 CellID->4172],
Cell[53429, 1866, 143, 3, 70, "Tutorials",
 CellID->21819]
}, Open  ]],
Cell[CellGroupData[{
Cell[53609, 1874, 321, 9, 70, "RelatedLinksSection",
 CellID->16585],
Cell[53933, 1885, 365, 12, 70, "RelatedLinks"],
Cell[54301, 1899, 267, 6, 70, "RelatedLinks",
 CellID->10752],
Cell[54571, 1907, 388, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[54996, 1925, 318, 9, 70, "MoreAboutSection",
 CellID->84355834],
Cell[55317, 1936, 169, 3, 70, "MoreAbout",
 CellID->142442474],
Cell[55489, 1941, 146, 3, 70, "MoreAbout",
 CellID->10162940],
Cell[55638, 1946, 136, 3, 70, "MoreAbout",
 CellID->31627898],
Cell[55777, 1951, 136, 3, 70, "MoreAbout",
 CellID->83135831],
Cell[55916, 1956, 175, 3, 70, "MoreAbout",
 CellID->440031049],
Cell[56094, 1961, 160, 3, 70, "MoreAbout",
 CellID->194194094]
}, Open  ]],
Cell[56269, 1967, 50, 0, 70, "History"],
Cell[56322, 1969, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

