(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     59794,       2240]
NotebookOptionsPosition[     48557,       1843]
NotebookOutlinePosition[     50084,       1884]
CellTagsIndexPosition[     49998,       1879]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Rationalize" :> 
          Documentation`HelpLookup["paclet:ref/Rationalize"], "LatticeReduce" :> 
          Documentation`HelpLookup["paclet:ref/LatticeReduce"], "Root" :> 
          Documentation`HelpLookup["paclet:ref/Root"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Rationalize\"\>", 2->"\<\"LatticeReduce\"\>", 
       3->"\<\"Root\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Numbers" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumbers"], 
          "Number Recognition" :> 
          Documentation`HelpLookup["paclet:guide/NumberRecognition"], 
          "Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheory"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Algebraic Numbers\"\>", 
       2->"\<\"Number Recognition\"\>", 3->"\<\"Number Theory\"\>", 
       4->"\<\"New in 6.0: Symbolic Computation\"\>", 
       5->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       6->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RootApproximant", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RootApproximant",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RootApproximant"], "[", 
       StyleBox["x", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]converts the number ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " to one of the \[OpenCurlyDoubleQuote]simplest",
     "\[CloseCurlyDoubleQuote]",
     " algebraic numbers that approximates it well."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RootApproximant",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RootApproximant"], "[", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds an algebraic number of degree at most ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     " that approximates ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->2196]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "For degrees above 2, ",
 Cell[BoxData[
  ButtonBox["RootApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootApproximant"]], "InlineFormula"],
 " generates ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects."
}], "Notes",
 CellID->73490044],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RootApproximant",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RootApproximant"], "[", 
   StyleBox["x", "TI"], "]"}]], "InlineFormula"],
 " effectively tests the total number of bits in the description of ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " by successively higher-degree algebraic numbers, and returns the first \
case for which the number of bits is small."
}], "Notes",
 CellID->130496394],

Cell[TextData[{
 "Results from ",
 Cell[BoxData[
  ButtonBox["RootApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootApproximant"]], "InlineFormula"],
 " may not be unique."
}], "Notes",
 CellID->724233939],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["MinimalPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MinimalPolynomial"]], "InlineFormula"],
 " yields the minimal polynomial for the result of ",
 Cell[BoxData[
  ButtonBox["RootApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootApproximant"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->244302966]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->516142703],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->211697696],

Cell[TextData[{
 "Find a quadratic approximation to ",
 Cell[BoxData[
  StyleBox["\[Pi]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->409570763],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{"Pi", ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17502413],

Cell[BoxData[
 FractionBox[
  RowBox[{"198261", "+", 
   SqrtBox["105043517473"]}], "166274"]], "Output",
 ImageSize->{173, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->138037350]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->72597055],

Cell[BoxData["3.141592653589793`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->146339227]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->58393422],

Cell["Find algebraic approximants of any order:", "ExampleText",
 CellID->4058247],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"Sqrt", "[", "67", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->201488977],

Cell[BoxData["8.18535277187245`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->100921041]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->155374878],

Cell[BoxData[
 SqrtBox["67"]], "Output",
 ImageSize->{35, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->10817897]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", "3.1415926", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->126272381],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], "-", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "2"]}], "-", 
     RowBox[{"4", " ", 
      SuperscriptBox["#1", "4"]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "5"]}], "+", 
     RowBox[{"3", " ", 
      SuperscriptBox["#1", "6"]}], "-", 
     RowBox[{"6", " ", 
      SuperscriptBox["#1", "7"]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "8"]}], "+", 
     SuperscriptBox["#1", "9"], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "10"]}], "-", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "11"]}], "-", 
     SuperscriptBox["#1", "12"], "-", 
     RowBox[{"3", " ", 
      SuperscriptBox["#1", "14"]}], "+", 
     SuperscriptBox["#1", "15"]}], "&"}], ",", "1"}], "]"}]], "Output",
 ImageSize->{562, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->75398071]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11403582],

Cell[TextData[{
 "There is no simple quadratic approximation to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "[", 
   RowBox[{
    SqrtBox["2"], "+", 
    SqrtBox["3"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->251721111],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], "+", 
     RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->116081484],

Cell[BoxData[
 FractionBox[
  RowBox[{"153704", "+", 
   SqrtBox["24924107305"]}], "99031"]], "Output",
 ImageSize->{166, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->330988320]
}, Open  ]],

Cell[TextData[{
 "At degree 4, there is a simple answer, which turns out to be equal to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SqrtBox["2"], "+", 
    SqrtBox["3"]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->103716590],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], "+", 
     RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}], ",", "4"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->632789446],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"1", "-", 
     RowBox[{"10", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]], "Output",
 ImageSize->{163, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->458841895]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootReduce", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], "+", 
   RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->404797463],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"1", "-", 
     RowBox[{"10", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]], "Output",
 ImageSize->{163, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->22176343]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32467272],

Cell[TextData[{
 "Machine precision is not enough to recover the ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " object equal to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SqrtBox["2"], "+", 
    SqrtBox["3"], "+", 
    SqrtBox["5"]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->395823506],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], "+", 
     RowBox[{"Sqrt", "[", "3", "]"}], "+", 
     RowBox[{"Sqrt", "[", "5", "]"}]}], "]"}], ",", "8"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->426890],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"20", "+", 
     RowBox[{"8", " ", "#1"}], "+", 
     RowBox[{"18", " ", 
      SuperscriptBox["#1", "2"]}], "-", 
     RowBox[{"42", " ", 
      SuperscriptBox["#1", "3"]}], "-", 
     RowBox[{"15", " ", 
      SuperscriptBox["#1", "4"]}], "-", 
     RowBox[{"19", " ", 
      SuperscriptBox["#1", "5"]}], "+", 
     RowBox[{"13", " ", 
      SuperscriptBox["#1", "6"]}], "-", 
     RowBox[{"7", " ", 
      SuperscriptBox["#1", "7"]}], "+", 
     SuperscriptBox["#1", "8"]}], "&"}], ",", "2"}], "]"}]], "Output",
 ImageSize->{446, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->386016280]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootReduce", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], "+", 
   RowBox[{"Sqrt", "[", "3", "]"}], "+", 
   RowBox[{"Sqrt", "[", "5", "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->386550742],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"576", "-", 
     RowBox[{"960", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     RowBox[{"352", " ", 
      SuperscriptBox["#1", "4"]}], "-", 
     RowBox[{"40", " ", 
      SuperscriptBox["#1", "6"]}], "+", 
     SuperscriptBox["#1", "8"]}], "&"}], ",", "8"}], "]"}]], "Output",
 ImageSize->{290, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->67767472]
}, Open  ]],

Cell["\<\
30 digits of precision suffice to recover the exact value here:\
\>", "ExampleText",
 CellID->347888997],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Sqrt", "[", "2", "]"}], "+", 
      RowBox[{"Sqrt", "[", "3", "]"}], "+", 
      RowBox[{"Sqrt", "[", "5", "]"}]}], ",", "30"}], "]"}], ",", "8"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->37382807],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"576", "-", 
     RowBox[{"960", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     RowBox[{"352", " ", 
      SuperscriptBox["#1", "4"]}], "-", 
     RowBox[{"40", " ", 
      SuperscriptBox["#1", "6"]}], "+", 
     SuperscriptBox["#1", "8"]}], "&"}], ",", "8"}], "]"}]], "Output",
 ImageSize->{290, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->38802693]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->276891819],

Cell[TextData[{
 "When given exact input, ",
 Cell[BoxData[
  ButtonBox["RootApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootApproximant"]], "InlineFormula"],
 " uses machine precision approximations:"
}], "ExampleText",
 CellID->465827682],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{"E", ",", "3"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->424603927],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"8654", "+", 
     RowBox[{"6252", " ", "#1"}], "-", 
     RowBox[{"3936", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     RowBox[{"171", " ", 
      SuperscriptBox["#1", "3"]}]}], "&"}], ",", "2"}], "]"}]], "Output",
 ImageSize->{280, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->40049514]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", "E", "]"}], ",", "3"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->498565062],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"8654", "+", 
     RowBox[{"6252", " ", "#1"}], "-", 
     RowBox[{"3936", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     RowBox[{"171", " ", 
      SuperscriptBox["#1", "3"]}]}], "&"}], ",", "2"}], "]"}]], "Output",
 ImageSize->{280, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->282541855]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->38705026],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RootApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootApproximant"]], "InlineFormula"],
 " works with complex numbers:"
}], "ExampleText",
 CellID->37297732],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], "+", "I"}], "]"}], ",", "4"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->92189090],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"9", "-", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]], "Output",
 ImageSize->{156, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->418685479]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->151612288],

Cell["Weights can be used to lower the degree of result:", "ExampleText",
 CellID->826300514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RootApproximant", "[", 
    RowBox[{
     RowBox[{"N", "[", 
      RowBox[{"Pi", ",", "20"}], "]"}], ",", "20", ",", "#"}], "]"}], "&"}], "/@",
   
  RowBox[{"{", 
   RowBox[{"0", ",", "1", ",", "2", ",", "3"}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->90453470],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"3", "+", 
       RowBox[{"3", " ", "#1"}], "-", 
       RowBox[{"5", " ", 
        SuperscriptBox["#1", "2"]}], "-", 
       RowBox[{"2", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       RowBox[{"3", " ", 
        SuperscriptBox["#1", "4"]}], "+", 
       RowBox[{"3", " ", 
        SuperscriptBox["#1", "5"]}], "+", 
       RowBox[{"9", " ", 
        SuperscriptBox["#1", "6"]}], "-", 
       RowBox[{"2", " ", 
        SuperscriptBox["#1", "8"]}], "+", 
       RowBox[{"8", " ", 
        SuperscriptBox["#1", "9"]}], "-", 
       SuperscriptBox["#1", "10"], "-", 
       RowBox[{"5", " ", 
        SuperscriptBox["#1", "11"]}], "+", 
       SuperscriptBox["#1", "12"], "-", 
       RowBox[{"3", " ", 
        SuperscriptBox["#1", "13"]}], "+", 
       SuperscriptBox["#1", "14"]}], "&"}], ",", "2"}], "]"}], ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"131", "-", 
       RowBox[{"8", " ", "#1"}], "-", 
       RowBox[{"11", " ", 
        SuperscriptBox["#1", "2"]}], "-", 
       RowBox[{"7", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       RowBox[{"4", " ", 
        SuperscriptBox["#1", "4"]}], "-", 
       SuperscriptBox["#1", "5"], "-", 
       RowBox[{"3", " ", 
        SuperscriptBox["#1", "6"]}], "+", 
       SuperscriptBox["#1", "7"]}], "&"}], ",", "3"}], "]"}], ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "720"}], "-", 
       RowBox[{"45", " ", "#1"}], "+", 
       RowBox[{"2", " ", 
        SuperscriptBox["#1", "2"]}], "-", 
       RowBox[{"11", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       RowBox[{"9", " ", 
        SuperscriptBox["#1", "4"]}], "+", 
       SuperscriptBox["#1", "5"]}], "&"}], ",", "3"}], "]"}], ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"403", "-", 
       RowBox[{"183", " ", "#1"}], "-", 
       RowBox[{"49", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       RowBox[{"18", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]}], 
  "}"}]], "Output",
 ImageSize->{580, 92},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->793011082]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->329232160],

Cell["\<\
Find successive algebraic approximations to \[Pi]:\
\>", "ExampleText",
 CellID->707564049],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"RootApproximant", "[", 
    RowBox[{"N", "[", 
     RowBox[{"Pi", ",", "n"}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->239863183],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3", ",", "3", ",", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      SqrtBox["53"]}], ")"}]}], ",", 
   RowBox[{
    RowBox[{"-", "4"}], "+", 
    SqrtBox["51"]}], ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "-", 
       RowBox[{"6", " ", "#1"}], "-", 
       RowBox[{"11", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       SuperscriptBox["#1", "3"], "+", 
       SuperscriptBox["#1", "4"]}], "&"}], ",", "2"}], "]"}]}], 
  "}"}]], "Output",
 ImageSize->{457, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->39108590]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->455310405],

Cell["A complicated radical expression:", "ExampleText",
 CellID->13887],

Cell[BoxData[
 RowBox[{
  RowBox[{"expr", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       SqrtBox["2"], "-", 
       SqrtBox[
        RowBox[{
         RowBox[{"(", 
          RowBox[{"1", "+", 
           RowBox[{"2", " ", 
            SqrtBox["2"]}], "-", "s"}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "1"}], "+", "s"}], ")"}]}]], "+", "s"}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "-", 
       SqrtBox["2"], "+", 
       RowBox[{
        SqrtBox["2"], " ", "s"}], "-", 
       SqrtBox[
        RowBox[{
         RowBox[{"-", "1"}], "-", 
         RowBox[{"2", " ", 
          SqrtBox["2"]}], "+", 
         RowBox[{"2", " ", 
          RowBox[{"(", 
           RowBox[{"2", "+", 
            SqrtBox["2"], "-", "s"}], ")"}], " ", "s"}]}]]}], ")"}]}], "-", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "-", 
       SqrtBox["2"], "+", 
       RowBox[{
        SqrtBox["2"], " ", "s"}], "+", 
       SqrtBox[
        RowBox[{
         RowBox[{"-", "1"}], "-", 
         RowBox[{"2", " ", 
          SqrtBox["2"]}], "+", 
         RowBox[{"2", " ", 
          RowBox[{"(", 
           RowBox[{"2", "+", 
            SqrtBox["2"], "-", "s"}], ")"}], " ", "s"}]}]]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "2"}], "-", 
       RowBox[{"2", " ", 
        SqrtBox["2"]}], "+", 
       RowBox[{"2", " ", "s"}], "+", 
       RowBox[{"2", " ", 
        SqrtBox[
         RowBox[{"1", "-", 
          RowBox[{
           FractionBox["1", "4"], " ", 
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", "1"}], "-", 
              SqrtBox["2"], "+", 
              SqrtBox[
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"1", "+", 
                  RowBox[{"2", " ", 
                   SqrtBox["2"]}], "-", "s"}], ")"}], " ", 
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{"-", "1"}], "+", "s"}], ")"}]}]], "+", "s"}], ")"}],
             "2"]}]}]]}]}], ")"}]}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15413],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 " to find the real root near 2.5:"
}], "ExampleText",
 CellID->18158],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "=", 
  RowBox[{
   RowBox[{"s", "/.", 
    RowBox[{"FindRoot", "[", 
     RowBox[{
      RowBox[{"expr", "==", "0"}], ",", 
      RowBox[{"{", 
       RowBox[{"s", ",", "2.5"}], "}"}], ",", 
      RowBox[{"WorkingPrecision", "->", "100"}]}], "]"}]}], "//", 
   "Chop"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21928],

Cell[BoxData["2.\
706945837045705738269073202812169790793196895164013983001575249106155514352266\
542032260316431664549796229819474071`100."], "Output",
 ImageSize->{585, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->492206396]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["RootApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootApproximant"]], "InlineFormula"],
 " to find an algebraic number close to the root:"
}], "ExampleText",
 CellID->7644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"a", "=", 
  RowBox[{"RootApproximant", "[", "x", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->17102],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"64", "+", 
     RowBox[{"512", " ", "#1"}], "-", 
     RowBox[{"6112", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     RowBox[{"21024", " ", 
      SuperscriptBox["#1", "3"]}], "-", 
     RowBox[{"32136", " ", 
      SuperscriptBox["#1", "4"]}], "+", 
     RowBox[{"10832", " ", 
      SuperscriptBox["#1", "5"]}], "+", 
     RowBox[{"43568", " ", 
      SuperscriptBox["#1", "6"]}], "-", 
     RowBox[{"86152", " ", 
      SuperscriptBox["#1", "7"]}], "+", 
     RowBox[{"81425", " ", 
      SuperscriptBox["#1", "8"]}], "-", 
     RowBox[{"46080", " ", 
      SuperscriptBox["#1", "9"]}], "+", 
     RowBox[{"15872", " ", 
      SuperscriptBox["#1", "10"]}], "-", 
     RowBox[{"3072", " ", 
      SuperscriptBox["#1", "11"]}], "+", 
     RowBox[{"256", " ", 
      SuperscriptBox["#1", "12"]}]}], "&"}], ",", "8"}], "]"}]], "Output",
 ImageSize->{472, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->284226428]
}, Open  ]],

Cell["\<\
Check whether the result is a root of the original expression:\
\>", "ExampleText",
 CellID->19566],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"expr", "/.", 
   RowBox[{"s", "->", "a"}]}], "//", "RootReduce"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->5069],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->142727796]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->157143038],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RootApproximant",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootApproximant"]], "InlineFormula"],
 " gives a ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " object close to a given algebraic number:"
}], "ExampleText",
 CellID->281510104],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{"Sqrt", "[", 
   RowBox[{"2", "+", 
    RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->68260027],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"1", "-", 
     RowBox[{"4", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]], "Output",
 ImageSize->{156, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->228199081]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " object found may not be exactly equal to the input:"
}], "ExampleText",
 CellID->141227132],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{"Sqrt", "[", 
    RowBox[{"2", "+", 
     RowBox[{"Sqrt", "[", 
      RowBox[{"3", "+", 
       RowBox[{"Sqrt", "[", 
        RowBox[{"5", "+", 
         RowBox[{"Sqrt", "[", "7", "]"}]}], "]"}]}], "]"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"b", "=", 
  RowBox[{"RootApproximant", "[", "a", "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->424297169],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"2", "-", 
     RowBox[{"3", " ", 
      SuperscriptBox["#1", "2"]}], "-", 
     SuperscriptBox["#1", "3"], "-", 
     SuperscriptBox["#1", "6"], "-", 
     SuperscriptBox["#1", "7"], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "8"]}], "-", 
     SuperscriptBox["#1", "9"], "-", 
     SuperscriptBox["#1", "10"], "-", 
     RowBox[{"3", " ", 
      SuperscriptBox["#1", "11"]}], "+", 
     SuperscriptBox["#1", "12"], "-", 
     SuperscriptBox["#1", "13"], "-", 
     RowBox[{"2", " ", 
      SuperscriptBox["#1", "15"]}], "-", 
     SuperscriptBox["#1", "16"], "+", 
     SuperscriptBox["#1", "17"]}], "&"}], ",", "3"}], "]"}]], "Output",
 ImageSize->{575, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->585525470]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"a", "-", "b"}], ",", "20"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->505451262],

Cell[BoxData["2.807453043275776962413438513849254`20.*^-14"], "Output",
 ImageSize->{192, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->105300900]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["RootReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootReduce"]], "InlineFormula"],
 " to find exact ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " object representations of algebraic numbers:"
}], "ExampleText",
 CellID->239915664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootReduce", "[", "a", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->79778005],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"9", "+", 
     RowBox[{"64", " ", 
      SuperscriptBox["#1", "2"]}], "-", 
     RowBox[{"80", " ", 
      SuperscriptBox["#1", "4"]}], "-", 
     RowBox[{"224", " ", 
      SuperscriptBox["#1", "6"]}], "+", 
     RowBox[{"444", " ", 
      SuperscriptBox["#1", "8"]}], "-", 
     RowBox[{"304", " ", 
      SuperscriptBox["#1", "10"]}], "+", 
     RowBox[{"100", " ", 
      SuperscriptBox["#1", "12"]}], "-", 
     RowBox[{"16", " ", 
      SuperscriptBox["#1", "14"]}], "+", 
     SuperscriptBox["#1", "16"]}], "&"}], ",", "12"}], "]"}]], "Output",
 ImageSize->{513, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->517654701]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1053935],

Cell[TextData[{
 "Specifying a degree one polynomial effectively finds a rational \
approximation to ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->10056],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{"Pi", ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9115],

Cell[BoxData[
 FractionBox["80143857", "25510582"]], "Output",
 ImageSize->{69, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->34683373]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Rationalize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Rationalize"]], "InlineFormula"],
 " also gives a rational approximation, but it need not to be the same:"
}], "ExampleText",
 CellID->4935],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Rationalize", "[", 
  RowBox[{
   RowBox[{"N", "[", "\[Pi]", "]"}], ",", "0"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2727],

Cell[BoxData[
 FractionBox["245850922", "78256779"]], "Output",
 ImageSize->{76, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->360584903]
}, Open  ]],

Cell["\<\
Interestingly, the approximations can be found among continued fraction \
convergents:\
\>", "ExampleText",
 CellID->22015951],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FromContinuedFraction", "[", 
  RowBox[{"ContinuedFraction", "[", 
   RowBox[{
    RowBox[{"N", "[", "\[Pi]", "]"}], ",", "15"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->22338],

Cell[BoxData[
 RowBox[{
  RowBox[{"ContinuedFraction", "::", "\<\"incomp\"\>"}], ":", 
  " ", "\<\"Warning: ContinuedFraction terminated before \\!\\(15\\) terms. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/ContinuedFraction/incomp\\\", ButtonNote \
-> \\\"ContinuedFraction::incomp\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->183509562],

Cell[BoxData[
 FractionBox["80143857", "25510582"]], "Output",
 ImageSize->{69, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->466678894]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FromContinuedFraction", "[", 
  RowBox[{"ContinuedFraction", "[", 
   RowBox[{"\[Pi]", ",", "15"}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->11847],

Cell[BoxData[
 FractionBox["245850922", "78256779"]], "Output",
 ImageSize->{76, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->40763290]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113667239],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["LatticeReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LatticeReduce"]], "InlineFormula"],
 " to recognize linear combinations of more general functions:"
}], "ExampleText",
 CellID->489990103],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"ArcTan", "[", "1", "]"}], ",", 
     RowBox[{"ArcTan", "[", 
      RowBox[{"1", "/", "5"}], "]"}], ",", 
     RowBox[{"ArcTan", "[", 
      RowBox[{"1", "/", "239"}], "]"}], ",", "1"}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->401725695],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a0", ",", "a1", ",", "a2", ",", "a3", ",", "a4"}], "}"}], "=", 
   RowBox[{"Round", "[", 
    RowBox[{
     RowBox[{"10", "^", "20"}], " ", "v"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->132690111],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", 
  RowBox[{"LatticeReduce", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", 
       RowBox[{"-", "a0"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", 
       RowBox[{"-", "a1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", 
       RowBox[{"-", "a2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", 
       RowBox[{"-", "a3"}]}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->728458029],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", 
     RowBox[{"-", "4"}], ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "325302"}], ",", "315725", ",", "367312", ",", "1153518", 
     ",", "928458"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "381213"}], ",", "314234", ",", "633857", ",", "2221192", 
     ",", 
     RowBox[{"-", "2330529"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "3210817"}], ",", "4041574", ",", "249764", ",", 
     RowBox[{"-", "3042512"}], ",", "306976"}], "}"}]}], "}"}]], "Output",
 ImageSize->{427, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->2989764]
}, Open  ]],

Cell[TextData[{
 "The final relationship ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["\[Pi]", "4"], "\[LongEqual]", 
    RowBox[{
     RowBox[{"4", 
      RowBox[{
       SuperscriptBox["tan", 
        RowBox[{"-", "1"}]], "(", 
       FractionBox["1", "5"], ")"}]}], "-", 
     RowBox[{
      SuperscriptBox["tan", 
       RowBox[{"-", "1"}]], "(", 
      FractionBox["1", "239"], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->124195550],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"b", "[", 
   RowBox[{"[", "1", "]"}], "]"}], ".", "v"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->475586984],

Cell[BoxData[
 RowBox[{
  FractionBox["\[Pi]", "4"], "+", 
  RowBox[{"ArcTan", "[", 
   FractionBox["1", "239"], "]"}], "-", 
  RowBox[{"4", " ", 
   RowBox[{"ArcTan", "[", 
    FractionBox["1", "5"], "]"}]}]}]], "Output",
 ImageSize->{200, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->17542422]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->255771205],

Cell[BoxData[
 RowBox[{"-", "1.1102230246251565`*^-16"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->156275444]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->205352207],

Cell["\<\
Recognizing an algebraic number may require using higher precision:\
\>", "ExampleText",
 CellID->24251],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{"Sqrt", "[", 
    RowBox[{"5", "+", 
     RowBox[{"2", " ", 
      RowBox[{"6", "^", 
       RowBox[{"(", 
        RowBox[{"1", "/", "3"}], ")"}]}]}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", "a", "]"}], ",", "10"}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->486726481],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"13", "-", 
     RowBox[{"6", " ", "#1"}], "-", 
     RowBox[{"13", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     RowBox[{"4", " ", 
      SuperscriptBox["#1", "4"]}], "-", 
     RowBox[{"12", " ", 
      SuperscriptBox["#1", "5"]}], "-", 
     RowBox[{"3", " ", 
      SuperscriptBox["#1", "6"]}], "-", 
     RowBox[{"4", " ", 
      SuperscriptBox["#1", "7"]}], "-", 
     RowBox[{"12", " ", 
      SuperscriptBox["#1", "8"]}], "-", 
     RowBox[{"4", " ", 
      SuperscriptBox["#1", "9"]}], "+", 
     RowBox[{"3", " ", 
      SuperscriptBox["#1", "10"]}]}], "&"}], ",", "2"}], "]"}]], "Output",
 ImageSize->{489, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->432029094]
}, Open  ]],

Cell[TextData[{
 "The result is not equal to ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->131737539],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootReduce", "[", "a", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->142402820],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "173"}], "+", 
     RowBox[{"75", " ", 
      SuperscriptBox["#1", "2"]}], "-", 
     RowBox[{"15", " ", 
      SuperscriptBox["#1", "4"]}], "+", 
     SuperscriptBox["#1", "6"]}], "&"}], ",", "2"}], "]"}]], "Output",
 ImageSize->{234, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->3933473]
}, Open  ]],

Cell["\<\
Providing the correct degree improves the chances of recognizing an algebraic \
number:\
\>", "ExampleText",
 CellID->57879970],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", "a", "]"}], ",", "6"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->17808408],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "173"}], "+", 
     RowBox[{"75", " ", 
      SuperscriptBox["#1", "2"]}], "-", 
     RowBox[{"15", " ", 
      SuperscriptBox["#1", "4"]}], "+", 
     SuperscriptBox["#1", "6"]}], "&"}], ",", "2"}], "]"}]], "Output",
 ImageSize->{234, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->844435558]
}, Open  ]],

Cell["\<\
A penalty may be used to lower the degree; here it does not help to recognize \
the number:\
\>", "ExampleText",
 CellID->2761],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RootApproximant", "[", 
    RowBox[{
     RowBox[{"N", "[", "a", "]"}], ",", "10", ",", "#"}], "]"}], "&"}], "/@", 
  
  RowBox[{"{", 
   RowBox[{"1", ",", "2", ",", "3"}], "}"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->2357],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"211", "-", 
       RowBox[{"91", " ", "#1"}], "-", 
       RowBox[{"57", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       RowBox[{"17", " ", 
        SuperscriptBox["#1", "3"]}], "-", 
       RowBox[{"10", " ", 
        SuperscriptBox["#1", "4"]}], "+", 
       SuperscriptBox["#1", "5"], "+", 
       SuperscriptBox["#1", "6"]}], "&"}], ",", "4"}], "]"}], ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"219", "+", 
       RowBox[{"106", " ", "#1"}], "-", 
       RowBox[{"53", " ", 
        SuperscriptBox["#1", "2"]}], "-", 
       RowBox[{"31", " ", 
        SuperscriptBox["#1", "3"]}], "-", 
       RowBox[{"2", " ", 
        SuperscriptBox["#1", "4"]}], "+", 
       SuperscriptBox["#1", "5"], "+", 
       SuperscriptBox["#1", "6"]}], "&"}], ",", "2"}], "]"}], ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"384", "-", 
       RowBox[{"91", " ", "#1"}], "-", 
       RowBox[{"132", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       RowBox[{"17", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       RowBox[{"5", " ", 
        SuperscriptBox["#1", "4"]}], "+", 
       SuperscriptBox["#1", "5"]}], "&"}], ",", "3"}], "]"}]}], 
  "}"}]], "Output",
 ImageSize->{365, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->281565247]
}, Open  ]],

Cell["\<\
Using a higher precision approximation allows to recognize the algebraic \
number:\
\>", "ExampleText",
 CellID->27106],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootApproximant", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{"a", ",", "25"}], "]"}], ",", "10"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->21230],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "173"}], "+", 
     RowBox[{"75", " ", 
      SuperscriptBox["#1", "2"]}], "-", 
     RowBox[{"15", " ", 
      SuperscriptBox["#1", "4"]}], "+", 
     SuperscriptBox["#1", "6"]}], "&"}], ",", "2"}], "]"}]], "Output",
 ImageSize->{234, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->29382683]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->29601],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Rationalize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Rationalize"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["LatticeReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LatticeReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Root",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Root"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->6563]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->231891675],

Cell[TextData[ButtonBox["Algebraic Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumbers"]], "MoreAbout",
 CellID->3339353],

Cell[TextData[ButtonBox["Number Recognition",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberRecognition"]], "MoreAbout",
 CellID->25226860],

Cell[TextData[ButtonBox["Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheory"]], "MoreAbout",
 CellID->289015745],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->107131671],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->190585767],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->218177488]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RootApproximant - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 18, 4.3348128}", "context" -> "System`", 
    "keywords" -> {
     "approximate by algebraic number", "guessing algebraic numbers", 
      "guessing minimal polynomials"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "RootApproximant[x] converts the number x to one of the \
\\[OpenCurlyDoubleQuote]simplest\\[CloseCurlyDoubleQuote] algebraic numbers \
that approximates it well. RootApproximant[x, n] finds an algebraic number of \
degree at most n that approximates x.", "synonyms" -> {"root approximant"}, 
    "title" -> "RootApproximant", "type" -> "Symbol", "uri" -> 
    "ref/RootApproximant"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7660, 231, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->516142703]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 49855, 1872}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2247, 49, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2870, 76, 1580, 37, 70, "ObjectNameGrid"],
Cell[4453, 115, 1220, 37, 70, "Usage",
 CellID->2196]
}, Open  ]],
Cell[CellGroupData[{
Cell[5710, 157, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6188, 174, 346, 13, 70, "Notes",
 CellID->73490044],
Cell[6537, 189, 481, 13, 70, "Notes",
 CellID->130496394],
Cell[7021, 204, 222, 8, 70, "Notes",
 CellID->724233939],
Cell[7246, 214, 377, 12, 70, "Notes",
 CellID->244302966]
}, Closed]],
Cell[CellGroupData[{
Cell[7660, 231, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->516142703],
Cell[CellGroupData[{
Cell[8045, 245, 148, 5, 70, "ExampleSection",
 CellID->211697696],
Cell[8196, 252, 159, 6, 70, "ExampleText",
 CellID->409570763],
Cell[CellGroupData[{
Cell[8380, 262, 134, 4, 28, "Input",
 CellID->17502413],
Cell[8517, 268, 236, 8, 56, "Output",
 CellID->138037350]
}, Open  ]],
Cell[CellGroupData[{
Cell[8790, 281, 96, 3, 28, "Input",
 CellID->72597055],
Cell[8889, 286, 175, 5, 36, "Output",
 CellID->146339227]
}, Open  ]],
Cell[9079, 294, 124, 3, 70, "ExampleDelimiter",
 CellID->58393422],
Cell[9206, 299, 82, 1, 70, "ExampleText",
 CellID->4058247],
Cell[CellGroupData[{
Cell[9313, 304, 129, 4, 70, "Input",
 CellID->201488977],
Cell[9445, 310, 174, 5, 36, "Output",
 CellID->100921041]
}, Open  ]],
Cell[CellGroupData[{
Cell[9656, 320, 111, 3, 70, "Input",
 CellID->155374878],
Cell[9770, 325, 169, 6, 38, "Output",
 CellID->10817897]
}, Open  ]],
Cell[CellGroupData[{
Cell[9976, 336, 119, 3, 70, "Input",
 CellID->126272381],
Cell[10098, 341, 970, 31, 58, "Output",
 CellID->75398071]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11117, 378, 219, 7, 70, "ExampleSection",
 CellID->11403582],
Cell[11339, 387, 298, 12, 70, "ExampleText",
 CellID->251721111],
Cell[CellGroupData[{
Cell[11662, 403, 256, 8, 70, "Input",
 CellID->116081484],
Cell[11921, 413, 234, 8, 56, "Output",
 CellID->330988320]
}, Open  ]],
Cell[12170, 424, 250, 9, 70, "ExampleText",
 CellID->103716590],
Cell[CellGroupData[{
Cell[12445, 437, 256, 8, 70, "Input",
 CellID->632789446],
Cell[12704, 447, 350, 12, 39, "Output",
 CellID->458841895]
}, Open  ]],
Cell[CellGroupData[{
Cell[13091, 464, 193, 6, 70, "Input",
 CellID->404797463],
Cell[13287, 472, 349, 12, 39, "Output",
 CellID->22176343]
}, Open  ]],
Cell[13651, 487, 124, 3, 70, "ExampleDelimiter",
 CellID->32467272],
Cell[13778, 492, 384, 15, 70, "ExampleText",
 CellID->395823506],
Cell[CellGroupData[{
Cell[14187, 511, 297, 9, 70, "Input",
 CellID->426890],
Cell[14487, 522, 717, 23, 39, "Output",
 CellID->386016280]
}, Open  ]],
Cell[CellGroupData[{
Cell[15241, 550, 235, 7, 70, "Input",
 CellID->386550742],
Cell[15479, 559, 485, 16, 39, "Output",
 CellID->67767472]
}, Open  ]],
Cell[15979, 578, 114, 3, 70, "ExampleText",
 CellID->347888997],
Cell[CellGroupData[{
Cell[16118, 585, 332, 11, 70, "Input",
 CellID->37382807],
Cell[16453, 598, 485, 16, 39, "Output",
 CellID->38802693]
}, Open  ]],
Cell[16953, 617, 125, 3, 70, "ExampleDelimiter",
 CellID->276891819],
Cell[17081, 622, 259, 8, 70, "ExampleText",
 CellID->465827682],
Cell[CellGroupData[{
Cell[17365, 634, 134, 4, 70, "Input",
 CellID->424603927],
Cell[17502, 640, 423, 14, 39, "Output",
 CellID->40049514]
}, Open  ]],
Cell[CellGroupData[{
Cell[17962, 659, 163, 5, 70, "Input",
 CellID->498565062],
Cell[18128, 666, 424, 14, 39, "Output",
 CellID->282541855]
}, Open  ]],
Cell[18567, 683, 124, 3, 70, "ExampleDelimiter",
 CellID->38705026],
Cell[18694, 688, 218, 7, 70, "ExampleText",
 CellID->37297732],
Cell[CellGroupData[{
Cell[18937, 699, 224, 8, 70, "Input",
 CellID->92189090],
Cell[19164, 709, 349, 12, 39, "Output",
 CellID->418685479]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19562, 727, 243, 7, 70, "ExampleSection",
 CellID->151612288],
Cell[19808, 736, 93, 1, 70, "ExampleText",
 CellID->826300514],
Cell[CellGroupData[{
Cell[19926, 741, 319, 11, 70, "Input",
 CellID->90453470],
Cell[20248, 754, 2326, 72, 113, "Output",
 CellID->793011082]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22623, 832, 227, 7, 70, "ExampleSection",
 CellID->329232160],
Cell[22853, 841, 101, 3, 70, "ExampleText",
 CellID->707564049],
Cell[CellGroupData[{
Cell[22979, 848, 273, 9, 70, "Input",
 CellID->239863183],
Cell[23255, 859, 739, 27, 51, "Output",
 CellID->39108590]
}, Open  ]],
Cell[24009, 889, 125, 3, 70, "ExampleDelimiter",
 CellID->455310405],
Cell[24137, 894, 72, 1, 70, "ExampleText",
 CellID->13887],
Cell[24212, 897, 2221, 76, 70, "Input",
 CellID->15413],
Cell[26436, 975, 214, 8, 70, "ExampleText",
 CellID->18158],
Cell[CellGroupData[{
Cell[26675, 987, 350, 12, 70, "Input",
 CellID->21928],
Cell[27028, 1001, 283, 7, 54, "Output",
 CellID->492206396]
}, Open  ]],
Cell[27326, 1011, 242, 8, 70, "ExampleText",
 CellID->7644],
Cell[CellGroupData[{
Cell[27593, 1023, 130, 4, 70, "Input",
 CellID->17102],
Cell[27726, 1029, 1044, 32, 58, "Output",
 CellID->284226428]
}, Open  ]],
Cell[28785, 1064, 109, 3, 70, "ExampleText",
 CellID->19566],
Cell[CellGroupData[{
Cell[28919, 1071, 149, 5, 70, "Input",
 CellID->5069],
Cell[29071, 1078, 158, 5, 36, "Output",
 CellID->142727796]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29278, 1089, 237, 7, 70, "ExampleSection",
 CellID->157143038],
Cell[29518, 1098, 358, 12, 70, "ExampleText",
 CellID->281510104],
Cell[CellGroupData[{
Cell[29901, 1114, 198, 6, 70, "Input",
 CellID->68260027],
Cell[30102, 1122, 349, 12, 39, "Output",
 CellID->228199081]
}, Open  ]],
Cell[30466, 1137, 230, 8, 70, "ExampleText",
 CellID->141227132],
Cell[CellGroupData[{
Cell[30721, 1149, 411, 14, 70, "Input",
 CellID->424297169],
Cell[31135, 1165, 855, 26, 39, "Output",
 CellID->585525470]
}, Open  ]],
Cell[CellGroupData[{
Cell[32027, 1196, 145, 5, 70, "Input",
 CellID->505451262],
Cell[32175, 1203, 202, 5, 36, "Output",
 CellID->105300900]
}, Open  ]],
Cell[32392, 1211, 366, 13, 70, "ExampleText",
 CellID->239915664],
Cell[CellGroupData[{
Cell[32783, 1228, 105, 3, 70, "Input",
 CellID->79778005],
Cell[32891, 1233, 755, 24, 39, "Output",
 CellID->517654701]
}, Open  ]],
Cell[33661, 1260, 123, 3, 70, "ExampleDelimiter",
 CellID->1053935],
Cell[33787, 1265, 200, 7, 70, "ExampleText",
 CellID->10056],
Cell[CellGroupData[{
Cell[34012, 1276, 130, 4, 70, "Input",
 CellID->9115],
Cell[34145, 1282, 191, 6, 51, "Output",
 CellID->34683373]
}, Open  ]],
Cell[34351, 1291, 247, 7, 70, "ExampleText",
 CellID->4935],
Cell[CellGroupData[{
Cell[34623, 1302, 158, 5, 70, "Input",
 CellID->2727],
Cell[34784, 1309, 193, 6, 51, "Output",
 CellID->360584903]
}, Open  ]],
Cell[34992, 1318, 136, 4, 70, "ExampleText",
 CellID->22015951],
Cell[CellGroupData[{
Cell[35153, 1326, 216, 6, 70, "Input",
 CellID->22338],
Cell[35372, 1334, 425, 8, 70, "Message",
 CellID->183509562],
Cell[35800, 1344, 192, 6, 51, "Output",
 CellID->466678894]
}, Open  ]],
Cell[CellGroupData[{
Cell[36029, 1355, 186, 5, 70, "Input",
 CellID->11847],
Cell[36218, 1362, 192, 6, 51, "Output",
 CellID->40763290]
}, Open  ]],
Cell[36425, 1371, 125, 3, 70, "ExampleDelimiter",
 CellID->113667239],
Cell[36553, 1376, 256, 8, 70, "ExampleText",
 CellID->489990103],
Cell[36812, 1386, 353, 11, 70, "Input",
 CellID->401725695],
Cell[37168, 1399, 280, 9, 70, "Input",
 CellID->132690111],
Cell[CellGroupData[{
Cell[37473, 1412, 629, 18, 70, "Input",
 CellID->728458029],
Cell[38105, 1432, 757, 23, 71, "Output",
 CellID->2989764]
}, Open  ]],
Cell[38877, 1458, 483, 18, 70, "ExampleText",
 CellID->124195550],
Cell[CellGroupData[{
Cell[39385, 1480, 144, 5, 70, "Input",
 CellID->475586984],
Cell[39532, 1487, 352, 12, 51, "Output",
 CellID->17542422]
}, Open  ]],
Cell[CellGroupData[{
Cell[39921, 1504, 97, 3, 70, "Input",
 CellID->255771205],
Cell[40021, 1509, 199, 6, 36, "Output",
 CellID->156275444]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[40269, 1521, 230, 7, 70, "ExampleSection",
 CellID->205352207],
Cell[40502, 1530, 114, 3, 70, "ExampleText",
 CellID->24251],
Cell[CellGroupData[{
Cell[40641, 1537, 384, 13, 70, "Input",
 CellID->486726481],
Cell[41028, 1552, 808, 26, 39, "Output",
 CellID->432029094]
}, Open  ]],
Cell[41851, 1581, 148, 6, 70, "ExampleText",
 CellID->131737539],
Cell[CellGroupData[{
Cell[42024, 1591, 106, 3, 70, "Input",
 CellID->142402820],
Cell[42133, 1596, 437, 15, 39, "Output",
 CellID->3933473]
}, Open  ]],
Cell[42585, 1614, 137, 4, 70, "ExampleText",
 CellID->57879970],
Cell[CellGroupData[{
Cell[42747, 1622, 162, 5, 70, "Input",
 CellID->17808408],
Cell[42912, 1629, 439, 15, 39, "Output",
 CellID->844435558]
}, Open  ]],
Cell[43366, 1647, 137, 4, 70, "ExampleText",
 CellID->2761],
Cell[CellGroupData[{
Cell[43528, 1655, 276, 10, 70, "Input",
 CellID->2357],
Cell[43807, 1667, 1458, 46, 77, "Output",
 CellID->281565247]
}, Open  ]],
Cell[45280, 1716, 129, 4, 70, "ExampleText",
 CellID->27106],
Cell[CellGroupData[{
Cell[45434, 1724, 186, 6, 70, "Input",
 CellID->21230],
Cell[45623, 1732, 438, 15, 39, "Output",
 CellID->29382683]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[46122, 1754, 311, 9, 70, "SeeAlsoSection",
 CellID->29601],
Cell[46436, 1765, 703, 26, 70, "SeeAlso",
 CellID->6563]
}, Open  ]],
Cell[CellGroupData[{
Cell[47176, 1796, 319, 9, 70, "MoreAboutSection",
 CellID->231891675],
Cell[47498, 1807, 143, 3, 70, "MoreAbout",
 CellID->3339353],
Cell[47644, 1812, 146, 3, 70, "MoreAbout",
 CellID->25226860],
Cell[47793, 1817, 137, 3, 70, "MoreAbout",
 CellID->289015745],
Cell[47933, 1822, 169, 3, 70, "MoreAbout",
 CellID->107131671],
Cell[48105, 1827, 179, 3, 70, "MoreAbout",
 CellID->190585767],
Cell[48287, 1832, 198, 4, 70, "MoreAbout",
 CellID->218177488]
}, Open  ]],
Cell[48500, 1839, 27, 0, 70, "History"],
Cell[48530, 1841, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

