(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     82685,       2669]
NotebookOptionsPosition[     70189,       2260]
NotebookOutlinePosition[     73023,       2332]
CellTagsIndexPosition[     72753,       2323]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Constructing Matrices" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ConstructingMatrices"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Constructing Matrices\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RotationTransform" :> 
          Documentation`HelpLookup["paclet:ref/RotationTransform"], "Rotate" :> 
          Documentation`HelpLookup["paclet:ref/Rotate"], "Dot" :> 
          Documentation`HelpLookup["paclet:ref/Dot"], "UnitVector" :> 
          Documentation`HelpLookup["paclet:ref/UnitVector"], "Sin" :> 
          Documentation`HelpLookup["paclet:ref/Sin"], "ReflectionMatrix" :> 
          Documentation`HelpLookup["paclet:ref/ReflectionMatrix"], 
          "ScalingMatrix" :> 
          Documentation`HelpLookup["paclet:ref/ScalingMatrix"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"RotationTransform\"\>", 
       2->"\<\"Rotate\"\>", 3->"\<\"Dot\"\>", 4->"\<\"UnitVector\"\>", 
       5->"\<\"Sin\"\>", 6->"\<\"ReflectionMatrix\"\>", 
       7->"\<\"ScalingMatrix\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Constructing Matrices" :> 
          Documentation`HelpLookup["paclet:guide/ConstructingMatrices"], 
          "Geometric Transforms" :> 
          Documentation`HelpLookup["paclet:guide/GeometricTransforms"], 
          "Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Constructing Matrices\"\>", 
       2->"\<\"Geometric Transforms\"\>", 
       3->"\<\"Matrices and Linear Algebra\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       5->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RotationMatrix", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationMatrix",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationMatrix"], "[", 
       StyleBox["\[Theta]", "TR"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the 2D rotation matrix that rotates 2D vectors \
counterclockwise by \[Theta] radians."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationMatrix",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationMatrix"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the 3D rotation matrix for a counterclockwise \
rotation around the 3D vector ",
     Cell[BoxData[
      StyleBox["w", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationMatrix",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationMatrix"], "[", 
       RowBox[{"{", 
        RowBox[{
         StyleBox["u", "TI"], ",", 
         StyleBox["v", "TI"]}], "}"}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the matrix that rotates the vector ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " to the direction of the vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     " in any dimension."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationMatrix",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationMatrix"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["u", "TI"], ",", 
          StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the matrix that rotates by ",
     Cell[BoxData[
      FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
     " radians in the hyperplane spanned by ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RotationMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotationMatrix"]], "InlineFormula"],
 " gives matrices for rotations of vectors around the origin."
}], "Notes",
 CellID->289688508],

Cell["\<\
Two different conventions for rotation matrices are in common use.\
\>", "Notes",
 CellID->400250111],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RotationMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotationMatrix"]], "InlineFormula"],
 " is set up to use the vector-oriented convention and to give a matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " so that ",
 Cell[BoxData[
  RowBox[{
   StyleBox["m", "TI"], ".", 
   StyleBox["r", "TI"]}]], "InlineFormula"],
 " yields the rotated version of a vector ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->15891062],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Transpose",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Transpose"], "[", 
   RowBox[{"RotationMatrix", "[", 
    StyleBox["\[Ellipsis]", "TR"], "]"}], "]"}]], "InlineFormula"],
 " gives rotation matrices with the alternative coordinate-system-oriented \
convention for which ",
 Cell[BoxData[
  RowBox[{
   StyleBox["r", "TI"], ".", 
   StyleBox["m", "TI"]}]], "InlineFormula"],
 " yields the rotated version of a vector ",
 Cell[BoxData[
  StyleBox["r", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->59924751],

Cell[TextData[{
 "Angles in ",
 Cell[BoxData[
  ButtonBox["RotationMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotationMatrix"]], "InlineFormula"],
 " are in radians. ",
 Cell[BoxData[
  RowBox[{
   StyleBox["\[Theta]", "TR"], 
   StyleBox["  ", "TR"], 
   ButtonBox["Degree",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Degree"]}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 Cell[BoxData["\[Degree]"], "InlineFormula"],
 " specifies an angle in degrees. "
}], "Notes",
 CellID->17745],

Cell[TextData[{
 "Positive ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 "corresponds to going from the direction of ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " towards the direction of ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->21143365],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   StyleBox["\[Theta]", "TR"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->701125705],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", " ", 
    StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", " ", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 ", where ",
 Cell[BoxData[
  RowBox[{
   StyleBox["u", "TI"], "\[UpTee]", 
   StyleBox["w", "TI"]}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   StyleBox["v", "TI"], "\[UpTee]", 
   StyleBox["w", "TI"]}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   StyleBox["u", "TI"], ",", 
   StyleBox["v", "TI"], ",", 
   StyleBox["w", "TI"]}]], "InlineFormula"],
 " form a right-handed coordinate system."
}], "Notes",
 CellID->268040301],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RotationMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotationMatrix"]], "InlineFormula"],
 " gives an orthogonal matrix of determinant 1, that in ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " dimensions can be considered an element of the group ",
 Cell[BoxData[
  FormBox[
   RowBox[{"SO", " ", 
    RowBox[{"(", "n", ")"}]}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->524887511]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->371340850],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(4)", "ExampleCount"]
}], "ExampleSection",
 CellID->365936931],

Cell["\<\
General 2D rotation matrix for rotating a vector about the origin:\
\>", "ExampleText",
 CellID->12117730],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RotationMatrix", "[", "\[Theta]", "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->220197751],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVV19PwyAQZwWcuk3j/JMZfdjik/F76JMfollM1odFM/ewb4/AYMHr0R6V
NbFJC3fA/flxcNf3crv6WJfbalnO3zbl16pafs9fPzeaxQeMsZ1+H2dM95Xu
2XenlG4X+ltocuTaJ0Df2VnSfLlZZR92D+gRoBct4xNAC6tFAC63Q3ua2cdw
pOnIA8sNczcari/0R7C69cYaQwrX8si82PoLzE7UeoNggUit0fJAO8dEnNsg
6xxdhWvIqZeGyLBvJGCE1aSc0C1I9h5G42lG7yXKHZN8NnbxqAyce4XZWvNQ
hpLB7OhJEKGuIXKKaTvZgF6rhkuSdzAKwtuj8MC6GykFW7hrHkUjXyRGAG2f
avGsu1NsZWt0jlFuN1n0GAvRVv68gfu9CBD0OzRx9DQyTrNg/a9nPajf+f0F
0GdkKU35/bllHM0E0fuka562cXiE+oFmfco9lTPz+ejus+5IQKSnnGyyhlnU
Z82RE4UuWXqW0VdcA+12oHn4l4x9rY5RaeD7h/6VZMrXNwhSOWoKWGvG/1By
ZW/oSfeagma7RzuWy29V9zoBtwDGWoVYpH+09+oGP4/7JnA=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{120, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->39380385]
}, Open  ]],

Cell[TextData[{
 "Apply rotation by ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " to a unit vector in the ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " direction:"
}], "ExampleText",
 CellID->78590272],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RotationMatrix", "[", "\[Theta]", "]"}], ".", 
  RowBox[{"{", 
   RowBox[{"1", ",", "0"}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->72197015],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Cos", "[", "\[Theta]", "]"}], ",", 
   RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->763527506]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->174073126],

Cell["\<\
Counterclockwise rotation by 30\[Degree]:\
\>", "ExampleText",
 CellID->80486913],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RotationMatrix", "[", 
  RowBox[{"30", "Degree"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->543998350],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     FractionBox[
      SqrtBox["3"], "2"], ",", 
     RowBox[{"-", 
      FractionBox["1", "2"]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", "2"], ",", 
     FractionBox[
      SqrtBox["3"], "2"]}], "}"}]}], "}"}]], "Output",
 ImageSize->{166, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->199941535]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1328065],

Cell[TextData[{
 "Rotation that transforms the direction of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"1", ",", "1"}], "}"}], TraditionalForm]], "InlineMath"],
 " into the direction of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"-", "1"}]}], "}"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->90229374],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RotationMatrix", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "1"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"0", ",", 
      RowBox[{"-", "1"}]}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->203252145],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", 
       SqrtBox["2"]]}], ",", 
     FractionBox["1", 
      SqrtBox["2"]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", 
       SqrtBox["2"]]}], ",", 
     RowBox[{"-", 
      FractionBox["1", 
       SqrtBox["2"]]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{215, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->324246958]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->150510612],

Cell[TextData[{
 "3D rotation around the ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " axis:"
}], "ExampleText",
 CellID->591239218],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RotationMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->315104190],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$, BoxForm`opts$5506]]]], "Output",
 ImageSize->{139, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->58678563]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 "A 4D rotation matrix, rotating in the ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", ",", "y"}], TraditionalForm]], "InlineMath"],
 " plane:"
}], "ExampleText",
 CellID->85815288],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RotationMatrix", "[", 
   RowBox[{"\[Theta]", ",", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "0", ",", "0", ",", "0"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1", ",", "0", ",", "0"}], "}"}]}], "}"}]}], " ", 
   "]"}], "//", "MatrixForm"}]], "Input",
 CellTags->"Ex--21996610710904046947,4",
 CellLabel->"In[1]:=",
 CellID->182497482],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0", "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$, BoxForm`opts$5512]]]], "Output",
 ImageSize->{154, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21996610710904046947,4",
 CellLabel->"Out[1]//MatrixForm=",
 CellID->68623785]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->112072416],

Cell[TextData[{
 "A general 3D rotation matrix, rotating in the plane given by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"t", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "+", " ", 
    RowBox[{"s", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "2"}], ",", "1"}], "}"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->275011276],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Theta]", ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", " ", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", 
         RowBox[{"-", "2"}], ",", "1"}], "}"}]}], "}"}]}], "]"}], "//", 
   "Simplify"}], "//", "MatrixForm"}]], "Input",
 CellTags->"Ex--21996610710904046947,5",
 CellLabel->"In[1]:=",
 CellID->587616712],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      SuperscriptBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Theta]", "2"], "]"}], "2"], 
      FractionBox[
       RowBox[{"Sin", "[", "\[Theta]", "]"}], 
       SqrtBox["2"]], 
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "1"}], "+", 
         RowBox[{"Cos", "[", "\[Theta]", "]"}]}], ")"}]}]},
     {
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", "\[Theta]", "]"}], 
        SqrtBox["2"]]}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", "\[Theta]", "]"}], 
        SqrtBox["2"]]}]},
     {
      RowBox[{
       FractionBox["1", "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "1"}], "+", 
         RowBox[{"Cos", "[", "\[Theta]", "]"}]}], ")"}]}], 
      FractionBox[
       RowBox[{"Sin", "[", "\[Theta]", "]"}], 
       SqrtBox["2"]], 
      SuperscriptBox[
       RowBox[{"Cos", "[", 
        FractionBox["\[Theta]", "2"], "]"}], "2"]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$, BoxForm`opts$5518]]]], "Output",
 ImageSize->{276, 93},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21996610710904046947,5",
 CellLabel->"Out[1]//MatrixForm=",
 CellID->88488292]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->232259503],

Cell[TextData[{
 "Rotate the vector ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], TraditionalForm]], 
  "InlineMath"],
 " to the vector ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->80048246],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", " ", "=", " ", 
  RowBox[{"RotationMatrix", "[", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->476335851],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->142973736]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", ".", 
  RowBox[{"{", 
   RowBox[{"1", ",", "0", ",", "0"}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->207497571],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "1"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->79580851]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->507506355],

Cell["Transformation applied to a 2D shape:", "ExampleText",
 CellID->479507293],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->419323043],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
    ",", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}], 
      ",", 
      RowBox[{"RotationMatrix", "[", 
       RowBox[{
        RowBox[{"-", "2"}], 
        RowBox[{"Pi", "/", "3"}]}], "]"}]}], "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->150504143],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], RectangleBox[{0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
     {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, GeometricTransformationBox[
    {RGBColor[1, 0, 0], Opacity[0.85], RectangleBox[{0, 0}], 
     {AbsolutePointSize[10], Opacity[1], 
      {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
      {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, 
    NCache[{{Rational[-1, 2], Rational[1, 2] 3^Rational[1, 2]}, {
      Rational[-1, 2] 3^Rational[1, 2], Rational[-1, 2]}}, {{-0.5, 
     0.8660254037844386}, {-0.8660254037844386, -0.5}}]]}]], "Output",
 ImageSize->{143, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->229931776]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->228289321],

Cell["Transformation applied to a 3D shape:", "ExampleText",
 CellID->137131831],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cuboid", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->367829949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
     ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}],
        ",", 
       RowBox[{"RotationMatrix", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "2"}], 
          RowBox[{"Pi", "/", "3"}]}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}], "}"}],
    ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167437831],

Cell[BoxData[
 Graphics3DBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], CuboidBox[{0, 0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], Point3DBox[{0, 0, 0}]}, 
     {RGBColor[0, 1, 0], Point3DBox[{1, 1, 1}]}}}, 
   InterpretationBox[
    {RGBColor[1., 0., 0.], Opacity[0.85], 
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {
      0.8660254037844386, -0.5, 0.}, {0.8660254037844386, -0.5, 
      1.}, {-0.5, -0.8660254037844386, 0.}, {-0.5, -0.8660254037844386, 1.}, {
      0.3660254037844386, -1.3660254037844386`, 0.}, {
      0.3660254037844386, -1.3660254037844386`, 1.}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic], 
     {AbsolutePointSize[10.], Opacity[1.], 
      {RGBColor[1., 0., 1.], Point3DBox[{0., 0., 0.}]}, 
      {RGBColor[0., 1., 0.], 
       Point3DBox[{0.3660254037844386, -1.3660254037844386`, 1.}]}}},
    GeometricTransformation[{
      Opacity[0.85], 
      RGBColor[1, 0, 0], {
       Cuboid[{0, 0, 0}], 
       AbsolutePointSize[10], 
       Opacity[1], {
        RGBColor[1, 0, 1], 
        Point[{0, 0, 0}]}, {
        RGBColor[0, 1, 0], 
        Point[{1, 1, 1}]}}}, {{
       Rational[-1, 2], Rational[1, 2] 3^Rational[1, 2], 0}, {
      Rational[-1, 2] 3^Rational[1, 2], 
       Rational[-1, 2], 0}, {0, 0, 1}}]]},
  Boxed->False]], "Output",
 ImageSize->{184, 175},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->557747747,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->72676750],

Cell["Rotating 3D shapes:", "ExampleText",
 CellID->684513844],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".5", "]"}], ",", 
       RowBox[{"Cuboid", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", ".5"}], ",", 
          RowBox[{"-", ".5"}], ",", 
          RowBox[{"-", ".5"}]}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"Cuboid", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", ".5"}], ",", 
          RowBox[{"-", ".5"}], ",", 
          RowBox[{"-", ".5"}]}], "}"}], "]"}], ",", 
       RowBox[{"RotationMatrix", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}], "]"}]}], "]"}]}], 
    "}"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->39796395],

Cell[BoxData[
 Graphics3DBox[{
   {Opacity[0.5], CuboidBox[{-0.5, -0.5, -0.5}]}, 
   InterpretationBox[
    GraphicsComplex3DBox[{{-0.7886751345948129, -0.21132486540518708`, \
-0.2886751345948129}, {-0.21132486540518708`, -0.7886751345948129, 
     0.2886751345948129}, {-0.5773502691896257, 0.5773502691896257, 
     0.2886751345948129}, {5.551115123125783*^-17, -5.551115123125783*^-17, 
     0.8660254037844388}, {-5.551115123125783*^-17, 
     5.551115123125783*^-17, -0.8660254037844388}, {
     0.5773502691896257, -0.5773502691896257, -0.2886751345948129}, {
     0.21132486540518708`, 0.7886751345948129, -0.2886751345948129}, {
     0.7886751345948129, 0.21132486540518708`, 0.2886751345948129}}, 
     Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {1,
       2, 4, 3}, {5, 7, 8, 6}}],
     VertexColors->Automatic,
     VertexNormals->Automatic],
    GeometricTransformation[
     Cuboid[{-0.5, -0.5, -0.5}], {{
      Rational[1, 2] + Rational[1, 2] 3^Rational[-1, 2], Rational[1, 2] + 
       Rational[-1, 2] 3^Rational[-1, 2], 3^Rational[-1, 2]}, {
      Rational[1, 2] + Rational[-1, 2] 3^Rational[-1, 2], Rational[1, 2] + 
       Rational[1, 2] 3^Rational[-1, 2], -3^Rational[-1, 2]}, {-3^
        Rational[-1, 2], 3^Rational[-1, 2], 3^Rational[-1, 2]}}]]},
  Boxed->False]], "Output",
 ImageSize->{184, 205},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->11908212,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->346888750],

Cell[TextData[{
 "Produce a basis for all rotations in dimension ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->869011885],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SO", "[", "n_", "]"}], " ", ":=", " ", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"RotationMatrix", "[", 
       RowBox[{"\[Theta]", ",", "#"}], "]"}], "&"}], ",", 
     RowBox[{"Subsets", "[", 
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"UnitVector", "[", 
          RowBox[{"n", ",", "i"}], "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"i", ",", "n"}], "}"}]}], "]"}], ",", " ", 
       RowBox[{"{", "2", "}"}]}], "]"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->458087292],

Cell["All rotations in 2D: ", "ExampleText",
 CellID->17247313],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", " ", 
  RowBox[{"MatrixForm", ",", " ", 
   RowBox[{"SO", "[", "2", "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17224114],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9WNtOwzAMTZuUwS4gBkhDgDTEC4j/gCc+opqQtocJNPYN++2SpMnUnTq9
zeukpnEcO85xkp7sK90uv9fpdrVI55+b9He5WvzNP342uklGQoidfh5nQtcz
XbPPLsv0+1WXsRZH7v0C8pPtNTalNFb2J+5BHoH8XKOfUPpSL2kKlcvC/kxL
YirJvsmppdMW7WNdKN8ZRjOicm8Z6BeyvwR5aINRDjHsLV17THgvycledhNU
4dYKX0PSih6Bc1xExs89TxfmB5Ea9I1QQaa9nDWPoAUq1Doro4NezxnRSchW
3OW0rYlTBn3Qrdc1c2uGQFIcGayDO0wVYxkQp0SzlVCBbu0IVyyzp2R/esU+
Me5EbJMbzLpH2fhXLVcQT55LVro6pTzVrv4x2drNF98aLmbLqqGP//b6DPgM
T5w8Deh5Ilw38oKY4Z5EL6iP8ukXUMDTHz2Ys9hQGWMZMcbAg0edl4fskFe9
g3zBNkoV73qr0WMOjjunu/Iruz9PwPt4Ztfm+8DJWPyu75NHMiLWE9cyX3Nj
1CeX7BOlLuxrxogFPQLP6XXMTmvGxG6yUzDIZvmvuh9y8a9bIhIOjoh3j8N7
HXlvZ2JdOKPuXBDnMCRj96iHONZd1p2/0REg/14R3oTI/60R0T/elHDL\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{134, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->79096525]
}, Open  ]],

Cell["All rotations in 3D:", "ExampleText",
 CellID->491968842],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", " ", 
  RowBox[{"MatrixForm", ",", " ", 
   RowBox[{"SO", "[", "3", "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->557954501],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWztvG0cQPh5JUZJlx3okcRAEseE0sZE4sYEkSJMuqVLkJwhGALoIIjgu
ZMCBSpUqVapkqZJIGpUqVapUyVKlysvszix3OZy7vePtHkk5AkTePuf17evj
3q+7b/q//7H75tXL3Ye/vN7d6796+dfDn/98DVntVpK0/kmS5OmDBJ4zeNL/
6u9f+MdHfFKfKRTeoe/nLP0bSz9m6c90Lxvqs31ycqJlZMlbnT44OKD0tk4f
Hx9T+hOdNjplujc3/Y2n/K5O39zcoEjqfzQaDYdD9filp/2jUv1fXFycnp6O
y6nW5eUlSenAh2r0jL6/g+8ufK8Yb0AH6nFLp6ERKKgfdWOUlui/NrVMuuMs
Km5TqatdCh8dU5lZoZIb9G30a+fUz+vnnoke6Yv6Hx0dUbnp905kO+rpv7e3
R+l1J3oWg1qb7rgVKfyVmPs9k4ARBvzt7+9LHlJ9q9GRChpPpaekdfJzC/pa
F1vJubLckNpgDK6urtDhyabxGI1Yudf1hfFbUx6CCebs7GysB34qP5Hf5L7L
obSn04eHh9Cderyf0axGElfn7uV1Vkvue6W8XgUafJBNjlH0xmAwoFlOln2v
aR816g27Qq9lBn3p+fl5V+yzCupgtYOF0voZ/E4oDIk6Wc8NMbecd9XM3c7t
Wc7dNBZS1n2Wjo+sAj80YrFFj15hC0dUFRTZvRb6FOYyKu+6WjOv5a4gHVdq
L5nekdSZpQri5ZWLo9HORWitHZ2yRLNPiuCDBqyV0eOzuQp67G6Do4fPju5O
MzWDIsG/KuOi3MxjsKukdiqOaRx5sIDDLG0t83lNimkwe4NaZvfn4VGxyiTw
NXBqzYTHLZ4npavgYCp3NgnKWynAYLOUf3j8Q1s2uw2JOifTzkCOuMGUOWWr
qKbJdD0ezW42iW1dzPQyLIbBqxkPdzN7ku6QfVK9vCh0nfZ1dPDJ1t6DQ5b1
Xsep7Wr0t8db73IsecekmLUFD/jcllhSxd3s/9IXVvqnWTFr+IjVl09C09py
bXi9J0zOU5ZeC2ZdHXbxc8F6o2We1VL5E48cHwvaC+INd8+HO4jFYAG3qd0a
05fL5+Vm7v6arDB6vKBocDl57euysFuR/Fou6nxEdoWovy9M3ZbYapV5xKzL
5Xasz0rZpn+NIcyV73uefmoaExzNHO0t7MfZf/WYDxtnQRti+MqhVjwD1UTr
bGeSmL5QslSjnqdeTGQqS1pQrBCJqPTNsQ0xpVEYwh0xd9nQN4vlD1jP4aPG
d7HynOfbt0VkUKOyh7djXqvjAbOzDRu5gKeBBWcal21uqm1rhHiEOUnHZlxn
62sp8BHS1iARCIOIuqxtEWP6YSYzAhJz3KbYuWywYQCK2F6f/FltC+Pd95mT
vM3Sm2X9OQvB2V7OuuJtT/t7L9eCt/+Rpdd8Wl5fX6tH+fcjH1vrY0k/NlIK
tC9q/4OnvOezjtLsjgXZ7O4NFollNKwprpoAI7pJm3ePY1b9jZztSHbgrhiU
p5Eg3SRwWWLzK+WLbBLzbvSGwyHdDK5yNlh2pk7etaB/Af78tpjj33l4KH+P
NR+G095a4AjkDJmsuXxD4jnr20UpTDA0ZqUz3W1i6oqQae8tKr+P70Y26A15
/z8fNpOjcDAYkHfqo9AiGlEo3tdfUsZOPp3yXyLGNzqiWyzr81F0P+BuYDQa
4QQ/9VZAHRR9q9vCAKUbo/atjxRW23Jz2OKxcrIHlJ9SmJ3xvSd7g6ocYupY
KesTx3ZEA0Bl4j0kOz64tOpo6ff75DlEC7KMds6RfqFZNCZOHssrzBL0HRgL
U3YmIKWOZeW4t1D2ju/24oJB7/7BEKcbsXVQge8dAtwm3vvbMLlUUR4/8+bf
inBgdd/O4tlSkoOduf9x5Onsu8NsEW+HVY08ZWHklZ6xGD3DKj72yDD1dmrI
+oL1ZU6pP+muVFN8Gzhp/QeWo2mY\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{441, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->486183971]
}, Open  ]],

Cell[TextData[{
 "All rotations in 4D; in general ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"n", "-", "1"}], ")"}], "/", "2"}]}], TraditionalForm]], 
  "InlineMath"],
 " basis elements are needed for dimension ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ": "
}], "ExampleText",
 CellID->269197210],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", " ", 
  RowBox[{"MatrixForm", ",", " ", 
   RowBox[{"SO", "[", "4", "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->528502665],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{486, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->561530916]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell["\<\
A rotation matrix is orthogonal, i.e. the inverse is equal to the transpose:\
\>", "ExampleText",
 CellID->123059499],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "=", 
  RowBox[{"RotationMatrix", "[", "\[Theta]", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->311124526],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cos", "[", "\[Theta]", "]"}], ",", 
     RowBox[{"-", 
      RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sin", "[", "\[Theta]", "]"}], ",", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->74366427]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Inverse", "[", "r", "]"}], "-", 
   RowBox[{"Transpose", "[", "r", "]"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->38942369],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->228514030]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->309251812],

Cell[TextData[{
 "A rotation matrix has determinant ",
 Cell[BoxData[
  FormBox[
   RowBox[{"+", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->204971806],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Det", "[", " ", 
   RowBox[{"RotationMatrix", "[", "\[Theta]", "]"}], "]"}], " ", "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->143796646],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->48150654]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->63639329],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    RowBox[{"-", 
     StyleBox["\[Theta]", "TR"]}], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->306218622],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], "]"}], ".", 
   RowBox[{"RotationMatrix", "[", 
    RowBox[{
     RowBox[{"-", "\[Theta]"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], "]"}]}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->278425834],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->163250961]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25725914],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 " is also given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["v", "TI"], ",", 
      StyleBox["u", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->737162285],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], "]"}], ".", 
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "}"}]}], "]"}]}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->87239060],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->20130881]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->128401371],

Cell[TextData[{
 "In 2D the inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   StyleBox["\[Theta]", "TR"], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{"-", 
    StyleBox["\[Theta]", "TR"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->144814910],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RotationMatrix", "[", "\[Theta]", "]"}], ".", 
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"-", "\[Theta]"}], "]"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->186078799],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->161495434]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->250304132],

Cell[TextData[{
 "In 3D the inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"-", 
     StyleBox["w", "TI"]}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->304136939],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], ".", 
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"-", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}]}], "]"}]}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->262239887],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180053263]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->43020061],

Cell["The composition of rotations is a rotation:", "ExampleText",
 CellID->257069901],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"MatrixPower", "[", 
    RowBox[{
     RowBox[{"RotationMatrix", "[", "a", "]"}], ",", "n"}], "]"}], "\[Equal]",
    
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"n", " ", "a"}], "]"}]}], "//", 
  RowBox[{
   RowBox[{"FullSimplify", "[", 
    RowBox[{"#", ",", 
     RowBox[{"Element", "[", 
      RowBox[{"n", ",", "Integers"}], "]"}]}], "]"}], "&"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->448261893],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->19566708]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["The order in which rotations are performed is important: ", \
"ExampleText",
 CellID->735779720],

Cell[BoxData[{
 RowBox[{
  RowBox[{"rx", " ", "=", " ", 
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"ry", " ", "=", " ", 
   RowBox[{"RotationMatrix", "[", 
    RowBox[{"\[Phi]", ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellTags->"Ex--21996610710904046947,7",
 CellLabel->"In[1]:=",
 CellID->73993471],

Cell[TextData[{
 "Rotating around ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and then ",
 Cell[BoxData[
  StyleBox["y", "TI"]], "InlineFormula"],
 " is not the same as first rotating around ",
 Cell[BoxData[
  StyleBox["y", "TI"]], "InlineFormula"],
 " and then ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ": "
}], "ExampleText",
 CellID->12041738],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"rx", ".", "ry"}], "-", 
  RowBox[{"ry", ".", "rx"}]}]], "Input",
 CellTags->"Ex--21996610710904046947,8",
 CellLabel->"In[2]:=",
 CellID->235883581],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], " ", 
      RowBox[{"Sin", "[", "\[Phi]", "]"}]}], ",", 
     RowBox[{
      RowBox[{"Sin", "[", "\[Phi]", "]"}], "-", 
      RowBox[{
       RowBox[{"Cos", "[", "\[Theta]", "]"}], " ", 
       RowBox[{"Sin", "[", "\[Phi]", "]"}]}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Sin", "[", "\[Theta]", "]"}], " ", 
      RowBox[{"Sin", "[", "\[Phi]", "]"}]}], ",", "0", ",", 
     RowBox[{
      RowBox[{"Sin", "[", "\[Theta]", "]"}], "-", 
      RowBox[{
       RowBox[{"Cos", "[", "\[Phi]", "]"}], " ", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Sin", "[", "\[Phi]", "]"}], "-", 
      RowBox[{
       RowBox[{"Cos", "[", "\[Theta]", "]"}], " ", 
       RowBox[{"Sin", "[", "\[Phi]", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"Sin", "[", "\[Theta]", "]"}], "-", 
      RowBox[{
       RowBox[{"Cos", "[", "\[Phi]", "]"}], " ", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", "0"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{581, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--21996610710904046947,8",
 CellLabel->"Out[2]=",
 CellID->272329252]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->123953590],

Cell["Rotations of a circular sector:", "ExampleText",
 CellID->254150465],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"Disk", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2", ",", "0"}], "}"}], ",", "1", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "7"}], 
            RowBox[{"Pi", "/", "10"}]}], ",", 
           RowBox[{"7", 
            RowBox[{"Pi", "/", "10"}]}]}], "}"}]}], "]"}], ",", 
       RowBox[{"RotationMatrix", "[", "a", "]"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", 
       RowBox[{"2", 
        RowBox[{"Pi", "/", "5"}], 
        RowBox[{"Range", "[", "5", "]"}]}]}], "}"}]}], "]"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "3.1"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7550495],

Cell[BoxData[
 GraphicsBox[{
   GeometricTransformationBox[
    DiskBox[{2, 0}, 1, 
     NCache[{Rational[-7, 10] Pi, Rational[7, 10] Pi}, {-2.199114857512855, 
      2.199114857512855}]], 
    NCache[{{Rational[1, 4] (-1 + 5^Rational[1, 2]), 
       Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^
        Rational[1, 2]}, {
      Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
       Rational[1, 4] (-1 + 5^Rational[1, 2])}}, {{
     0.30901699437494745`, -0.9510565162951535}, {0.9510565162951535, 
     0.30901699437494745`}}]], 
   GeometricTransformationBox[
    DiskBox[{2, 0}, 1, 
     NCache[{Rational[-7, 10] Pi, Rational[7, 10] Pi}, {-2.199114857512855, 
      2.199114857512855}]], 
    NCache[{{Rational[1, 4] (-1 - 5^Rational[1, 2]), 
       Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^
        Rational[1, 2]}, {
      Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
       Rational[1, 4] (-1 - 5^
        Rational[1, 2])}}, {{-0.8090169943749475, -0.5877852522924731}, {
     0.5877852522924731, -0.8090169943749475}}]], 
   GeometricTransformationBox[
    DiskBox[{2, 0}, 1, 
     NCache[{Rational[-7, 10] Pi, Rational[7, 10] Pi}, {-2.199114857512855, 
      2.199114857512855}]], 
    NCache[{{Rational[1, 4] (-1 - 5^Rational[1, 2]), 
       Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^
        Rational[1, 2]}, {
      Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
       Rational[1, 4] (-1 - 5^Rational[1, 2])}}, {{-0.8090169943749475, 
     0.5877852522924731}, {-0.5877852522924731, -0.8090169943749475}}]], 
   GeometricTransformationBox[
    DiskBox[{2, 0}, 1, 
     NCache[{Rational[-7, 10] Pi, Rational[7, 10] Pi}, {-2.199114857512855, 
      2.199114857512855}]], 
    NCache[{{Rational[1, 4] (-1 + 5^Rational[1, 2]), 
       Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^
        Rational[1, 2]}, {
      Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
       Rational[1, 4] (-1 + 5^Rational[1, 2])}}, {{0.30901699437494745`, 
     0.9510565162951535}, {-0.9510565162951535, 0.30901699437494745`}}]], 
   GeometricTransformationBox[
    DiskBox[{2, 0}, 1, 
     NCache[{Rational[-7, 10] Pi, Rational[7, 10] Pi}, {-2.199114857512855, 
      2.199114857512855}]], {{1, 0}, {0, 1}}]},
  PlotRange->3.1]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->210937527]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Rotate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Rotate"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Dot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Dot"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["UnitVector",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/UnitVector"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Sin",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sin"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ReflectionMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ReflectionMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ScalingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ScalingMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Constructing Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ConstructingMatrices"]], "Tutorials",
 CellID->12359]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with RotationMatrix",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=RotationMatrix"]\
, None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->414921466],

Cell[TextData[ButtonBox["Constructing Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ConstructingMatrices"]], "MoreAbout",
 CellID->248388280],

Cell[TextData[ButtonBox["Geometric Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeometricTransforms"]], "MoreAbout",
 CellID->425491522],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "MoreAbout",
 CellID->217873847],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->691987],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->232823296]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RotationMatrix - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 18, 18.9601872}", "context" -> "System`", 
    "keywords" -> {
     "affine transform", "geometric transformation", "isometric mapping", 
      "proper orthogonal transformation"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "RotationMatrix[\\[Theta]] gives the 2D rotation matrix that rotates 2D \
vectors counterclockwise by \\[Theta] radians. RotationMatrix[\\[Theta], w] \
gives the 3D rotation matrix for a counterclockwise rotation around the 3D \
vector w. RotationMatrix[{u, v}] gives the matrix that rotates the vector u \
to the direction of the vector v in any dimension. RotationMatrix[\\[Theta], \
{u, v}] gives the matrix that rotates by \\[Theta] radians in the hyperplane \
spanned by u and v.", "synonyms" -> {"rotation matrix"}, "title" -> 
    "RotationMatrix", "type" -> "Symbol", "uri" -> "ref/RotationMatrix"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[13150, 419, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->371340850]},
 "Ex--21996610710904046947,4"->{
  Cell[19796, 685, 435, 13, 70, "Input",
   CellTags->"Ex--21996610710904046947,4",
   CellID->182497482],
  Cell[20234, 700, 1046, 30, 92, "Output",
   CellTags->"Ex--21996610710904046947,4",
   CellID->68623785]},
 "Ex--21996610710904046947,5"->{
  Cell[21858, 757, 474, 15, 70, "Input",
   CellTags->"Ex--21996610710904046947,5",
   CellID->587616712],
  Cell[22335, 774, 1796, 57, 128, "Output",
   CellTags->"Ex--21996610710904046947,5",
   CellID->88488292]},
 "Ex--21996610710904046947,7"->{
  Cell[59368, 1899, 517, 16, 70, "Input",
   CellTags->"Ex--21996610710904046947,7",
   CellID->73993471]},
 "Ex--21996610710904046947,8"->{
  Cell[60298, 1936, 183, 6, 70, "Input",
   CellTags->"Ex--21996610710904046947,8",
   CellID->235883581],
  Cell[60484, 1944, 1370, 42, 54, "Output",
   CellTags->"Ex--21996610710904046947,8",
   CellID->272329252]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 71718, 2291},
 {"Ex--21996610710904046947,4", 71872, 2295},
 {"Ex--21996610710904046947,5", 72121, 2302},
 {"Ex--21996610710904046947,7", 72371, 2309},
 {"Ex--21996610710904046947,8", 72512, 2313}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3159, 70, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3782, 97, 1579, 37, 70, "ObjectNameGrid"],
Cell[5364, 136, 2374, 72, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[7775, 213, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[8253, 230, 242, 7, 70, "Notes",
 CellID->289688508],
Cell[8498, 239, 111, 3, 70, "Notes",
 CellID->400250111],
Cell[8612, 244, 533, 18, 70, "Notes",
 CellID->15891062],
Cell[9148, 264, 581, 19, 70, "Notes",
 CellID->59924751],
Cell[9732, 285, 556, 20, 70, "Notes",
 CellID->17745],
Cell[10291, 307, 643, 24, 70, "Notes",
 CellID->21143365],
Cell[10937, 333, 650, 23, 70, "Notes",
 CellID->701125705],
Cell[11590, 358, 1047, 39, 70, "Notes",
 CellID->268040301],
Cell[12640, 399, 473, 15, 70, "Notes",
 CellID->524887511]
}, Closed]],
Cell[CellGroupData[{
Cell[13150, 419, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->371340850],
Cell[CellGroupData[{
Cell[13535, 433, 148, 5, 70, "ExampleSection",
 CellID->365936931],
Cell[13686, 440, 116, 3, 70, "ExampleText",
 CellID->12117730],
Cell[CellGroupData[{
Cell[13827, 447, 153, 5, 28, "Input",
 CellID->220197751],
Cell[13983, 454, 820, 17, 63, "Output",
 Evaluatable->False,
 CellID->39380385]
}, Open  ]],
Cell[14818, 474, 256, 9, 70, "ExampleText",
 CellID->78590272],
Cell[CellGroupData[{
Cell[15099, 487, 186, 6, 28, "Input",
 CellID->72197015],
Cell[15288, 495, 280, 9, 36, "Output",
 CellID->763527506]
}, Open  ]],
Cell[15583, 507, 125, 3, 70, "ExampleDelimiter",
 CellID->174073126],
Cell[15711, 512, 91, 3, 70, "ExampleText",
 CellID->80486913],
Cell[CellGroupData[{
Cell[15827, 519, 134, 4, 70, "Input",
 CellID->543998350],
Cell[15964, 525, 452, 18, 56, "Output",
 CellID->199941535]
}, Open  ]],
Cell[16431, 546, 123, 3, 70, "ExampleDelimiter",
 CellID->1328065],
Cell[16557, 551, 380, 14, 70, "ExampleText",
 CellID->90229374],
Cell[CellGroupData[{
Cell[16962, 569, 286, 10, 70, "Input",
 CellID->203252145],
Cell[17251, 581, 531, 22, 55, "Output",
 CellID->324246958]
}, Open  ]],
Cell[17797, 606, 125, 3, 70, "ExampleDelimiter",
 CellID->150510612],
Cell[17925, 611, 149, 6, 70, "ExampleText",
 CellID->591239218],
Cell[CellGroupData[{
Cell[18099, 621, 238, 8, 70, "Input",
 CellID->315104190],
Cell[18340, 631, 963, 28, 78, "Output",
 CellID->58678563]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[19352, 665, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[19576, 674, 195, 7, 70, "ExampleText",
 CellID->85815288],
Cell[CellGroupData[{
Cell[19796, 685, 435, 13, 70, "Input",
 CellTags->"Ex--21996610710904046947,4",
 CellID->182497482],
Cell[20234, 700, 1046, 30, 92, "Output",
 CellTags->"Ex--21996610710904046947,4",
 CellID->68623785]
}, Open  ]],
Cell[21295, 733, 125, 3, 70, "ExampleDelimiter",
 CellID->112072416],
Cell[21423, 738, 410, 15, 70, "ExampleText",
 CellID->275011276],
Cell[CellGroupData[{
Cell[21858, 757, 474, 15, 70, "Input",
 CellTags->"Ex--21996610710904046947,5",
 CellID->587616712],
Cell[22335, 774, 1796, 57, 128, "Output",
 CellTags->"Ex--21996610710904046947,5",
 CellID->88488292]
}, Open  ]],
Cell[24146, 834, 125, 3, 70, "ExampleDelimiter",
 CellID->232259503],
Cell[24274, 839, 353, 15, 70, "ExampleText",
 CellID->80048246],
Cell[CellGroupData[{
Cell[24652, 858, 328, 10, 70, "Input",
 CellID->476335851],
Cell[24983, 870, 412, 14, 36, "Output",
 CellID->142973736]
}, Open  ]],
Cell[CellGroupData[{
Cell[25432, 889, 149, 5, 70, "Input",
 CellID->207497571],
Cell[25584, 896, 212, 7, 36, "Output",
 CellID->79580851]
}, Open  ]],
Cell[25811, 906, 125, 3, 70, "ExampleDelimiter",
 CellID->507506355],
Cell[25939, 911, 80, 1, 70, "ExampleText",
 CellID->479507293],
Cell[26022, 914, 620, 20, 70, "Input",
 CellID->419323043],
Cell[CellGroupData[{
Cell[26667, 938, 578, 19, 70, "Input",
 CellID->150504143],
Cell[27248, 959, 784, 17, 237, "Output",
 CellID->229931776]
}, Open  ]],
Cell[28047, 979, 125, 3, 70, "ExampleDelimiter",
 CellID->228289321],
Cell[28175, 984, 80, 1, 70, "ExampleText",
 CellID->137131831],
Cell[28258, 987, 637, 20, 70, "Input",
 CellID->367829949],
Cell[CellGroupData[{
Cell[28920, 1011, 764, 25, 70, "Input",
 CellID->167437831],
Cell[29687, 1038, 4803, 94, 196, 1440, 34, "CachedBoxData", "BoxData", \
"Output",
 CellID->557747747]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34539, 1138, 226, 7, 70, "ExampleSection",
 CellID->72676750],
Cell[34768, 1147, 62, 1, 70, "ExampleText",
 CellID->684513844],
Cell[CellGroupData[{
Cell[34855, 1152, 1048, 33, 70, "Input",
 CellID->39796395],
Cell[35906, 1187, 4454, 80, 226, 1324, 24, "CachedBoxData", "BoxData", \
"Output",
 CellID->11908212]
}, Open  ]],
Cell[40375, 1270, 125, 3, 70, "ExampleDelimiter",
 CellID->346888750],
Cell[40503, 1275, 175, 6, 70, "ExampleText",
 CellID->869011885],
Cell[40681, 1283, 608, 19, 70, "Input",
 CellID->458087292],
Cell[41292, 1304, 63, 1, 70, "ExampleText",
 CellID->17247313],
Cell[CellGroupData[{
Cell[41380, 1309, 170, 5, 70, "Input",
 CellID->17224114],
Cell[41553, 1316, 877, 18, 70, "Output",
 Evaluatable->False,
 CellID->79096525]
}, Open  ]],
Cell[42445, 1337, 63, 1, 70, "ExampleText",
 CellID->491968842],
Cell[CellGroupData[{
Cell[42533, 1342, 171, 5, 70, "Input",
 CellID->557954501],
Cell[42707, 1349, 2192, 40, 70, "Output",
 Evaluatable->False,
 CellID->486183971]
}, Open  ]],
Cell[44914, 1392, 364, 14, 70, "ExampleText",
 CellID->269197210],
Cell[CellGroupData[{
Cell[45303, 1410, 171, 5, 70, "Input",
 CellID->528502665],
Cell[45477, 1417, 3616, 63, 70, "Output",
 Evaluatable->False,
 CellID->561530916]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[49142, 1486, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[49383, 1495, 127, 3, 70, "ExampleText",
 CellID->123059499],
Cell[CellGroupData[{
Cell[49535, 1502, 140, 4, 70, "Input",
 CellID->311124526],
Cell[49678, 1508, 483, 16, 36, "Output",
 CellID->74366427]
}, Open  ]],
Cell[CellGroupData[{
Cell[50198, 1529, 194, 6, 70, "Input",
 CellID->38942369],
Cell[50395, 1537, 302, 11, 36, "Output",
 CellID->228514030]
}, Open  ]],
Cell[50712, 1551, 125, 3, 70, "ExampleDelimiter",
 CellID->309251812],
Cell[50840, 1556, 181, 7, 70, "ExampleText",
 CellID->204971806],
Cell[CellGroupData[{
Cell[51046, 1567, 192, 6, 70, "Input",
 CellID->143796646],
Cell[51241, 1575, 157, 5, 36, "Output",
 CellID->48150654]
}, Open  ]],
Cell[51413, 1583, 124, 3, 70, "ExampleDelimiter",
 CellID->63639329],
Cell[51540, 1588, 740, 28, 70, "ExampleText",
 CellID->306218622],
Cell[CellGroupData[{
Cell[52305, 1620, 679, 22, 70, "Input",
 CellID->278425834],
Cell[52987, 1644, 391, 13, 36, "Output",
 CellID->163250961]
}, Open  ]],
Cell[53393, 1660, 124, 3, 70, "ExampleDelimiter",
 CellID->25725914],
Cell[53520, 1665, 724, 27, 70, "ExampleText",
 CellID->737162285],
Cell[CellGroupData[{
Cell[54269, 1696, 652, 21, 70, "Input",
 CellID->87239060],
Cell[54924, 1719, 390, 13, 36, "Output",
 CellID->20130881]
}, Open  ]],
Cell[55329, 1735, 125, 3, 70, "ExampleDelimiter",
 CellID->128401371],
Cell[55457, 1740, 501, 18, 70, "ExampleText",
 CellID->144814910],
Cell[CellGroupData[{
Cell[55983, 1762, 241, 7, 70, "Input",
 CellID->186078799],
Cell[56227, 1771, 302, 11, 36, "Output",
 CellID->161495434]
}, Open  ]],
Cell[56544, 1785, 125, 3, 70, "ExampleDelimiter",
 CellID->250304132],
Cell[56672, 1790, 594, 22, 70, "ExampleText",
 CellID->304136939],
Cell[CellGroupData[{
Cell[57291, 1816, 423, 14, 70, "Input",
 CellID->262239887],
Cell[57717, 1832, 391, 13, 36, "Output",
 CellID->180053263]
}, Open  ]],
Cell[58123, 1848, 124, 3, 70, "ExampleDelimiter",
 CellID->43020061],
Cell[58250, 1853, 86, 1, 70, "ExampleText",
 CellID->257069901],
Cell[CellGroupData[{
Cell[58361, 1858, 456, 15, 70, "Input",
 CellID->448261893],
Cell[58820, 1875, 160, 5, 36, "Output",
 CellID->19566708]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[59029, 1886, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[59263, 1895, 102, 2, 70, "ExampleText",
 CellID->735779720],
Cell[59368, 1899, 517, 16, 70, "Input",
 CellTags->"Ex--21996610710904046947,7",
 CellID->73993471],
Cell[59888, 1917, 385, 15, 70, "ExampleText",
 CellID->12041738],
Cell[CellGroupData[{
Cell[60298, 1936, 183, 6, 70, "Input",
 CellTags->"Ex--21996610710904046947,8",
 CellID->235883581],
Cell[60484, 1944, 1370, 42, 54, "Output",
 CellTags->"Ex--21996610710904046947,8",
 CellID->272329252]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[61903, 1992, 228, 7, 70, "ExampleSection",
 CellID->123953590],
Cell[62134, 2001, 74, 1, 70, "ExampleText",
 CellID->254150465],
Cell[CellGroupData[{
Cell[62233, 2006, 847, 26, 70, "Input",
 CellID->7550495],
Cell[63083, 2034, 2513, 53, 201, "Output",
 CellID->210937527]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[65657, 2094, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[65976, 2105, 1704, 62, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[67717, 2172, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[68038, 2183, 152, 3, 70, "Tutorials",
 CellID->12359]
}, Open  ]],
Cell[CellGroupData[{
Cell[68227, 2191, 305, 8, 70, "RelatedLinksSection"],
Cell[68535, 2201, 370, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[68942, 2218, 319, 9, 70, "MoreAboutSection",
 CellID->414921466],
Cell[69264, 2229, 153, 3, 70, "MoreAbout",
 CellID->248388280],
Cell[69420, 2234, 151, 3, 70, "MoreAbout",
 CellID->425491522],
Cell[69574, 2239, 163, 3, 70, "MoreAbout",
 CellID->217873847],
Cell[69740, 2244, 176, 3, 70, "MoreAbout",
 CellID->691987],
Cell[69919, 2249, 198, 4, 70, "MoreAbout",
 CellID->232823296]
}, Open  ]],
Cell[70132, 2256, 27, 0, 70, "History"],
Cell[70162, 2258, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

