(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    200831,       4924]
NotebookOptionsPosition[    188737,       4528]
NotebookOutlinePosition[    190835,       4576]
CellTagsIndexPosition[    190748,       4571]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RotationMatrix" :> 
          Documentation`HelpLookup["paclet:ref/RotationMatrix"], "Rotate" :> 
          Documentation`HelpLookup["paclet:ref/Rotate"], 
          "TransformationMatrix" :> 
          Documentation`HelpLookup["paclet:ref/TransformationMatrix"], 
          "TransformationFunction" :> 
          Documentation`HelpLookup["paclet:ref/TransformationFunction"], 
          "TranslationTransform" :> 
          Documentation`HelpLookup["paclet:ref/TranslationTransform"], 
          "AffineTransform" :> 
          Documentation`HelpLookup["paclet:ref/AffineTransform"], 
          "ReflectionTransform" :> 
          Documentation`HelpLookup["paclet:ref/ReflectionTransform"], 
          "ScalingTransform" :> 
          Documentation`HelpLookup["paclet:ref/ScalingTransform"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"RotationMatrix\"\>", 
       2->"\<\"Rotate\"\>", 3->"\<\"TransformationMatrix\"\>", 
       4->"\<\"TransformationFunction\"\>", 5->"\<\"TranslationTransform\"\>",
        6->"\<\"AffineTransform\"\>", 7->"\<\"ReflectionTransform\"\>", 
       8->"\<\"ScalingTransform\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Computational Geometry" :> 
          Documentation`HelpLookup["paclet:guide/ComputationalGeometry"], 
          "Geometric Transforms" :> 
          Documentation`HelpLookup["paclet:guide/GeometricTransforms"], 
          "Graphics Transformations" :> 
          Documentation`HelpLookup["paclet:guide/GraphicsTransformations"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Computational Geometry\"\>", 
       2->"\<\"Geometric Transforms\"\>", 
       3->"\<\"Graphics Transformations\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["RotationTransform", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationTransform"], "[", 
       StyleBox["\[Theta]", "TR"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives a ",
     Cell[BoxData[
      ButtonBox["TransformationFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
     " that represents a rotation in 2D by ",
     Cell[BoxData[
      FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
     " radians about the origin."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationTransform"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a 2D rotation about the 2D point ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationTransform"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a 3D rotation around the direction of the 3D \
vector ",
     Cell[BoxData[
      StyleBox["w", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationTransform"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        StyleBox["w", "TI"], ",", 
        StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a 3D rotation around the axis ",
     Cell[BoxData[
      StyleBox["w", "TI"]], "InlineFormula"],
     " anchored at the point ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationTransform"], "[", 
       RowBox[{"{", 
        RowBox[{
         StyleBox["u", "TI"], ",", 
         StyleBox["v", "TI"]}], "}"}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a rotation about the origin that transforms the \
vector ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " to the direction of the vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationTransform"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["u", "TI"], ",", 
          StyleBox["v", "TI"]}], "}"}], ",", 
        StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a rotation about the point ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     " that transforms ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " to the direction of ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["RotationTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/RotationTransform"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["u", "TI"], ",", 
          StyleBox["v", "TI"]}], "}"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a rotation by ",
     Cell[BoxData[
      FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
     " radians in the hyperplane spanned by ",
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RotationTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotationTransform"]], "InlineFormula"],
 " gives a ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " which can be applied to vectors."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   StyleBox["\[Theta]", "TR"], 
   StyleBox["  ", "TR"], 
   ButtonBox["Degree",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Degree"]}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 Cell[BoxData["\[Degree]"], "InlineFormula"],
 " specifies an angle in degrees. "
}], "Notes",
 CellID->17745],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}], ",", 
    StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
 " can be used to specify any rotation about any point ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 ", in any number of dimensions. "
}], "Notes",
 CellID->699151593],

Cell[TextData[{
 "Positive ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}], ",", 
    StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
 " corresponds to going from the direction of ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " toward the direction of ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->21143365],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   StyleBox["\[Theta]", "TR"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->701125705],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", " ", 
    StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", " ", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 ", where ",
 Cell[BoxData[
  RowBox[{
   StyleBox["u", "TI"], "\[UpTee]", 
   StyleBox["w", "TI"]}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   StyleBox["v", "TI"], "\[UpTee]", 
   StyleBox["w", "TI"]}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   StyleBox["u", "TI"], ",", 
   StyleBox["v", "TI"], ",", 
   StyleBox["w", "TI"]}]], "InlineFormula"],
 " form a right-handed coordinate system."
}], "Notes",
 CellID->268040301],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 " can effectively specify any element of the ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 "-dimensional rotation group ",
 Cell[BoxData[
  FormBox[
   RowBox[{"SO", " ", 
    RowBox[{"(", "n", ")"}]}], TraditionalForm]], "InlineMath"],
 ". ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}], ",", 
    StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
 " can effectively specify any element of the ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 "-dimensional special Euclidean group. "
}], "Notes",
 CellID->524887511]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->246665734],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->380951877],

Cell[TextData[{
 "A 2D rotation transform by ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " radians: "
}], "ExampleText",
 CellID->281592028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "=", 
  RowBox[{"RotationTransform", "[", "\[Theta]", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->82274509],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{308, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->47614851]
}, Open  ]],

Cell["Rotate a vector:", "ExampleText",
 CellID->643021637],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1813400],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "-", 
    RowBox[{"y", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"y", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "+", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]}], "}"}]], "Output",
 ImageSize->{263, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->671462656]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3169310],

Cell[TextData[{
 "Rotate around the ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " axis:"
}], "ExampleText",
 CellID->228379697],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "=", 
  RowBox[{"RotationTransform", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->59267923],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0", "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{323, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->503705646]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17132837],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "-", 
    RowBox[{"y", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"y", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "+", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", "z"}], "}"}]], "Output",
 ImageSize->{281, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->318570365]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->169950256],

Cell["\<\
Rotate a 2D graphic by 30\[Degree] about the origin:\
\>", "ExampleText",
 CellID->19862764],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"RotationTransform", "[", 
      RowBox[{"30", "Degree"}], "]"}]}], "]"}], ",", " ", 
   RowBox[{"Frame", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->296133109],

Cell[BoxData[
 GraphicsBox[
  GeometricTransformationBox[RectangleBox[{0, 0}], 
   NCache[{{{Rational[1, 2] 3^Rational[1, 2], Rational[-1, 2]}, {
      Rational[1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {0, 0}}, {{{
     0.8660254037844386, -0.5}, {0.5, 0.8660254037844386}}, {0, 0}}]],
  Frame->True]], "Output",
 ImageSize->{154, 149},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->144644423]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(8)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 "Rotation by ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " radians about the point ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["px", "TI"], ",", 
    StyleBox["py", "TI"]}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->133155009],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RotationTransform", "[", 
  RowBox[{"\[Theta]", ",", " ", 
   RowBox[{"{", 
    RowBox[{"px", ",", "py"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->343484228],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], 
      RowBox[{"px", "-", 
       RowBox[{"px", " ", 
        RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "+", 
       RowBox[{"py", " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"py", "-", 
       RowBox[{"py", " ", 
        RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "-", 
       RowBox[{"px", " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{459, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->221335387]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->562451026],

Cell[TextData[{
 "Rotation by ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " radians around the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis about the point ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["px", "TI"], ",", 
    StyleBox["py", "TI"], ",", 
    StyleBox["pz", "TI"]}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->165503528],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RotationTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"px", ",", "py", ",", "pz"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->198428110],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0"},
     {"0", 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], 
      RowBox[{"py", "-", 
       RowBox[{"py", " ", 
        RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "+", 
       RowBox[{"pz", " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]},
     {"0", 
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"pz", "-", 
       RowBox[{"pz", " ", 
        RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "-", 
       RowBox[{"py", " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{474, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->665000003]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->258442260],

Cell[TextData[{
 "A 2D rotation by ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], ",", 
   StyleBox["y", "TI"]}]], "InlineFormula"],
 " plane: "
}], "ExampleText",
 CellID->239391552],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RotationTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->159820115],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{308, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->698394390]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->51855344],

Cell[TextData[{
 "A 3D rotation by ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], ",", 
   StyleBox["y", "TI"]}]], "InlineFormula"],
 " plane: "
}], "ExampleText",
 CellID->119198238],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RotationTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->112570362],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0", "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{323, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->178717500]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->57979753],

Cell[TextData[{
 "A 4D rotation by ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], ",", 
   StyleBox["y", "TI"]}]], "InlineFormula"],
 " plane: "
}], "ExampleText",
 CellID->298712964],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RotationTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0", ",", "0"}], "}"}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->40926884],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0", "0", "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0", "0", "0"},
     {"0", "0", "1", "0", "0"},
     {"0", "0", "0", "1", "0"},
     {"0", "0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{338, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2852178]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2705599],

Cell[TextData[{
 "A 3D rotation by \[Theta] in the plane parametrized by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"s", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "+", 
    RowBox[{"t", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "1"}], "}"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->95861331],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "=", 
  RowBox[{"RotationTransform", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}]}], "]"}]}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->12257005],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{
       FractionBox["1", "2"], "+", 
       FractionBox[
        RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], 
      RowBox[{
       RowBox[{"-", 
        FractionBox[
         RowBox[{"Sin", "[", "\[Theta]", "]"}], 
         RowBox[{"3", " ", 
          SqrtBox["2"]}]]}], "-", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        SqrtBox["2"], " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], "+", 
       FractionBox[
        RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], "0"},
     {
      RowBox[{
       FractionBox[
        RowBox[{"Sin", "[", "\[Theta]", "]"}], 
        RowBox[{"3", " ", 
         SqrtBox["2"]}]], "+", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        SqrtBox["2"], " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{
       FractionBox[
        RowBox[{"Sin", "[", "\[Theta]", "]"}], 
        RowBox[{"3", " ", 
         SqrtBox["2"]}]], "+", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        SqrtBox["2"], " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], "0"},
     {
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], "+", 
       FractionBox[
        RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], 
      RowBox[{
       RowBox[{"-", 
        FractionBox[
         RowBox[{"Sin", "[", "\[Theta]", "]"}], 
         RowBox[{"3", " ", 
          SqrtBox["2"]}]]}], "-", 
       RowBox[{
        FractionBox["1", "3"], " ", 
        SqrtBox["2"], " ", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], 
      RowBox[{
       FractionBox["1", "2"], "+", 
       FractionBox[
        RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{465, 138},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->777084618]
}, Open  ]],

Cell[TextData[{
 "This rotates the vector ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->82662020],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", 
    RowBox[{"-", "1"}], ",", "1"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->432986673],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", "\[Theta]", "]"}], "+", 
    FractionBox[
     RowBox[{"Sin", "[", "\[Theta]", "]"}], 
     RowBox[{"3", " ", 
      SqrtBox["2"]}]], "+", 
    RowBox[{
     FractionBox["1", "3"], " ", 
     SqrtBox["2"], " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "+", 
    RowBox[{
     SqrtBox["2"], " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"Cos", "[", "\[Theta]", "]"}], "+", 
    FractionBox[
     RowBox[{"Sin", "[", "\[Theta]", "]"}], 
     RowBox[{"3", " ", 
      SqrtBox["2"]}]], "+", 
    RowBox[{
     FractionBox["1", "3"], " ", 
     SqrtBox["2"], " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]}], "}"}]], "Output",
 ImageSize->{565, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->211910682]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->58806993],

Cell["Transformation applied to a 2D shape:", "ExampleText",
 CellID->479507293],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->419323043],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
    ",", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}], 
      ",", 
      RowBox[{"RotationTransform", "[", 
       RowBox[{
        RowBox[{"Pi", "/", "6"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}]}], "]"}]}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->150504143],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], RectangleBox[{0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
     {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, GeometricTransformationBox[
    {RGBColor[1, 0, 0], Opacity[0.85], RectangleBox[{0, 0}], 
     {AbsolutePointSize[10], Opacity[1], 
      {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
      {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, 
    NCache[{{{Rational[1, 2] 3^Rational[1, 2], Rational[-1, 2]}, {
       Rational[1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {
      Rational[3, 2] + Rational[-1, 2] 3^Rational[1, 2], Rational[1, 2] + 
       Rational[-1, 2] 3^Rational[1, 2]}}, {{{0.8660254037844386, -0.5}, {0.5,
       0.8660254037844386}}, {
      0.6339745962155614, -0.3660254037844386}}]]}]], "Output",
 ImageSize->{184, 166},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->162972915]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->337851088],

Cell["Transformation applied to a 3D shape:", "ExampleText",
 CellID->137131831],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cuboid", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->367829949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
     ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}],
        ",", 
       RowBox[{"RotationTransform", "[", 
        RowBox[{
         RowBox[{"Pi", "/", "6"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "]"}]}], "]"}]}], "}"}],
    ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167437831],

Cell[BoxData[
 Graphics3DBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], CuboidBox[{0, 0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], Point3DBox[{0, 0, 0}]}, 
     {RGBColor[0, 1, 0], Point3DBox[{1, 1, 1}]}}}, 
   InterpretationBox[
    {RGBColor[1., 0., 0.], Opacity[0.85], 
     GraphicsComplex3DBox[{{0.6339745962155614, -0.3660254037844386, 0.}, {
      0.6339745962155614, -0.3660254037844386, 1.}, {0.1339745962155614, 0.5, 
      0.}, {0.1339745962155614, 0.5, 1.}, {1.5, 0.1339745962155614, 0.}, {1.5,
       0.1339745962155614, 1.}, {1., 1., 0.}, {1., 1., 1.}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic], 
     {AbsolutePointSize[10.], Opacity[1.], 
      {RGBColor[1., 0., 1.], 
       Point3DBox[{0.6339745962155614, -0.3660254037844386, 0.}]}, 
      {RGBColor[0., 1., 0.], Point3DBox[{1., 1., 1.}]}}},
    GeometricTransformation[{
      Opacity[0.85], 
      RGBColor[1, 0, 0], {
       Cuboid[{0, 0, 0}], 
       AbsolutePointSize[10], 
       Opacity[1], {
        RGBColor[1, 0, 1], 
        Point[{0, 0, 0}]}, {
        RGBColor[0, 1, 0], 
        Point[{1, 1, 1}]}}}, {{{Rational[1, 2] 3^Rational[1, 2], 
        Rational[-1, 2], 0}, {
        Rational[1, 2], Rational[1, 2] 3^Rational[1, 2], 0}, {0, 0, 1}}, {
      Rational[3, 2] + Rational[-1, 2] 3^Rational[1, 2], Rational[1, 2] + 
       Rational[-1, 2] 3^Rational[1, 2], 0}}]]},
  Boxed->False]], "Output",
 ImageSize->{184, 168},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->198211124,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Great Circles",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->279776909],

Cell[TextData[{
 "Parametrize a great circle passing through the points ",
 Cell[BoxData[
  StyleBox["u", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 " on a sphere:"
}], "ExampleText",
 CellID->124045928],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"GreatCircle", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"u_", ",", "v_"}], "}"}], ",", "\[Theta]_"}], "]"}], " ", ":=", 
   
   RowBox[{
    RowBox[{"RotationTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{"u", ",", "v"}], "}"}]}], "]"}], "[", 
    RowBox[{"Normalize", "[", "u", "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->130194797],

Cell[TextData[{
 "The great circle passing through ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}], "/", 
    SqrtBox["3"]}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "1", ",", "1"}], "}"}], "/", 
    SqrtBox["3"]}], TraditionalForm]], "InlineMath"],
 ": "
}], "ExampleText",
 CellID->217142144],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"u", ",", "v"}], "}"}], "=", 
   RowBox[{"Normalize", "/@", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}]}]}], ";"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->153233183],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GreatCircle", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"u", ",", "v"}], "}"}], ",", " ", "\[Theta]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->351369032],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "2"]}], "+", 
      FractionBox[
       RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], 
     SqrtBox["3"]], "+", 
    FractionBox[
     RowBox[{
      FractionBox["1", "2"], "+", 
      FractionBox[
       RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], 
     SqrtBox["3"]], "-", 
    FractionBox[
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", "\[Theta]", "]"}], 
        RowBox[{"3", " ", 
         SqrtBox["2"]}]]}], "-", 
      RowBox[{
       FractionBox["1", "3"], " ", 
       SqrtBox["2"], " ", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], 
     SqrtBox["3"]]}], ",", 
   RowBox[{
    RowBox[{"-", 
     FractionBox[
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      SqrtBox["3"]]}], "+", 
    FractionBox[
     RowBox[{"2", " ", 
      RowBox[{"(", 
       RowBox[{
        FractionBox[
         RowBox[{"Sin", "[", "\[Theta]", "]"}], 
         RowBox[{"3", " ", 
          SqrtBox["2"]}]], "+", 
        RowBox[{
         FractionBox["1", "3"], " ", 
         SqrtBox["2"], " ", 
         RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ")"}]}], 
     SqrtBox["3"]]}], ",", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "2"]}], "+", 
      FractionBox[
       RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], 
     SqrtBox["3"]], "+", 
    FractionBox[
     RowBox[{
      FractionBox["1", "2"], "+", 
      FractionBox[
       RowBox[{"Cos", "[", "\[Theta]", "]"}], "2"]}], 
     SqrtBox["3"]], "-", 
    FractionBox[
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Sin", "[", "\[Theta]", "]"}], 
        RowBox[{"3", " ", 
         SqrtBox["2"]}]]}], "-", 
      RowBox[{
       FractionBox["1", "3"], " ", 
       SqrtBox["2"], " ", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], 
     SqrtBox["3"]]}]}], "}"}]], "Output",
 ImageSize->{548, 111},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->14113611]
}, Open  ]],

Cell["This plots the great circle: ", "ExampleText",
 CellID->55616846],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"c", "=", 
  RowBox[{"ParametricPlot3D", "[", " ", 
   RowBox[{
    RowBox[{"Evaluate", "[", 
     RowBox[{"GreatCircle", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"u", ",", "v"}], "}"}], ",", " ", "\[Theta]"}], "]"}], "]"}], 
    ",", " ", 
    RowBox[{"{", 
     RowBox[{"\[Theta]", ",", "0", ",", 
      RowBox[{"2", "\[Pi]"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->2372651],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 205},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->107950185]
}, Open  ]],

Cell["This shows the great circle and points on the unit sphere: ", \
"ExampleText",
 CellID->89852326],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"g", "=", 
  RowBox[{"Show", "[", " ", 
   RowBox[{"c", ",", 
    RowBox[{"Graphics3D", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.02", "]"}], ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"u", ",", "v"}], "}"}], "]"}], ",", " ", 
       RowBox[{"Opacity", "[", "0.5", "]"}], ",", " ", 
       RowBox[{"Sphere", "[", "]"}]}], "}"}], "]"}], ",", 
    RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->209880432],

Cell[BoxData[
 Graphics3DBox[{{{}, {}, 
    {Hue[0.67, 0.6, 0.6], Line3DBox[CompressedData["
1:eJxF2Xk0VV0fB3AJTSoZklBEoolSkWTTQE+hCBmjyVDKlKmIUKkINyRzMoTM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      "]]}}, 
   {PointSize[0.02], 
    Point3DBox[
     NCache[{{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}, {
       3^Rational[-1, 2], 3^Rational[-1, 2], 3^Rational[-1, 2]}}, {{
      0.5773502691896258, -0.5773502691896258, 0.5773502691896258}, {
      0.5773502691896258, 0.5773502691896258, 0.5773502691896258}}]], 
    {Opacity[0.5], SphereBox[{0, 0, 0}]}}},
  Axes->True,
  PlotRange->All,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 206},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->108944870,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  ButtonBox["GeometricTransformation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricTransformation"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->530992566],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Graphics3D", "[", 
    RowBox[{"{", 
     RowBox[{"Red", ",", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"GeometricTransformation", "[", 
         RowBox[{
          RowBox[{"Point", "[", "u", "]"}], ",", 
          RowBox[{"RotationTransform", "[", 
           RowBox[{"\[Theta]", ",", 
            RowBox[{"{", 
             RowBox[{"u", ",", "v"}], "}"}]}], "]"}]}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"\[Theta]", ",", "0", ",", 
          RowBox[{"2", "Pi"}], ",", 
          RowBox[{"2", 
           RowBox[{"Pi", "/", "50."}]}]}], "}"}]}], "]"}]}], "}"}], "]"}], 
   ",", "\[IndentingNewLine]", "g"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->104083723],

Cell[BoxData[
 Graphics3DBox[{
   {RGBColor[1, 0, 0], 
    InterpretationBox[
     Point3DBox[{0.5773502691896258, -0.5773502691896258, 
      0.5773502691896258}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       1., 0., -5.551115123125783*^-17}, {0., 1., 
       0.}, {-5.551115123125783*^-17, 0., 1.}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.6239647682118342, -0.4704635331890284, 
      0.6239647682118342}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9960573506572388, -0.08862397936135696, -0.0039426493427611176`}, {
       0.08862397936135696, 0.9921147013144778, 
       0.08862397936135696}, {-0.0039426493427611176`, -0.08862397936135696, 
       0.9960573506572388}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.6607389701008568, -0.3561573062287479, 
      0.6607389701008568}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9842915805643154, -0.17585030562678616`, -0.015708419435684517`}, {
       0.17585030562678616`, 0.968583161128631, 
       0.17585030562678616`}, {-0.015708419435684517`, -0.17585030562678616`, 
       0.9842915805643154}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.6870929237250604, -0.23623426579117812`, 
      0.6870929237250604}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9648882429441257, -0.26030336752460026`, -0.035111757055874326`}, {
       0.26030336752460026`, 0.9297764858882513, 
       0.26030336752460026`}, {-0.035111757055874326`, -0.26030336752460026`, 
       0.9648882429441257}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.7026110114927022, -0.11258566986257135`, 
      0.7026110114927022}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9381533400219317, -0.3406512898188569, -0.061846659978068264`}, {
       0.3406512898188569, 0.8763066800438635, 
       0.3406512898188569}, {-0.061846659978068264`, -0.3406512898188569, 
       0.9381533400219317}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.7070485038896305, 0.012838469335187265`, 
      0.7070485038896305}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9045084971874737, -0.4156269377774534, -0.09549150281252633}, {
       0.4156269377774534, 0.8090169943749475, 
       0.4156269377774534}, {-0.09549150281252633, -0.4156269377774534, 
       0.9045084971874737}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.7003354190099161, 0.13806013820220034`, 
      0.7003354190099161}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.8644843137107058, -0.48404790064380165`, -0.13551568628929422`}, {
       0.48404790064380165`, 0.7289686274214114, 
       0.48404790064380165`}, {-0.13551568628929422`, -0.48404790064380165`, 
       0.8644843137107058}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.6825776262123147, 0.2611045162166356, 0.6825776262123147}],
     
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.8187119948743448, -0.5448351389607972, -0.18128800512565524`}, {
       0.5448351389607972, 0.6374239897486896, 
       0.5448351389607972}, {-0.18128800512565524`, -0.5448351389607972, 
       0.8187119948743448}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.6540551764972357, 0.3800311200340571, 0.6540551764972357}],
     
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.7679133974894983, -0.597030001667645, -0.23208660251050178`}, {
       0.597030001667645, 0.5358267949789965, 
       0.597030001667645}, {-0.23208660251050178`, -0.597030001667645, 
       0.7679133974894983}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.6152178859351716, 0.4929644060689544, 0.6152178859351716}],
     
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.7128896457825363, -0.6398093445997585, -0.28711035421746367`}, {
       0.6398093445997585, 0.42577929156507255`, 
       0.6398093445997585}, {-0.28711035421746367`, -0.6398093445997585, 
       0.7128896457825363}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.5666782417985585, 0.5981233489374821, 0.5666782417985585}],
     
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.6545084971874737, -0.6724985119639573, -0.3454915028125263}, {
       0.6724985119639573, 0.3090169943749474, 
       0.6724985119639573}, {-0.3454915028125263, -0.6724985119639573, 
       0.6545084971874737}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.5092017432716093, 0.6938495292916964, 0.5092017432716093}],
     
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.5936906572928623, -0.6945819760633463, -0.4063093427071377}, {
       0.6945819760633463, 0.1873813145857245, 
       0.6945819760633463}, {-0.4063093427071377, -0.6945819760633463, 
       0.5936906572928623}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.44369482907088953`, 0.7786332880833625, 
      0.44369482907088953`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.5313952597646567, -0.7057114674770557, -0.4686047402353433}, {
       0.7057114674770557, 0.0627905195293133, 
       0.7057114674770557}, {-0.4686047402353433, -0.7057114674770557, 
       0.5313952597646567}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.3711905823652784, 0.8511375347889736, 0.3711905823652784}],
     
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.46860474023534326`, -0.7057114674770557, -0.5313952597646567}, {
       0.7057114674770557, -0.06279051952931342, 
       0.7057114674770557}, {-0.5313952597646567, -0.7057114674770557, 
       0.46860474023534326`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.29283243843726103`, 0.9102188341260447, 
      0.29283243843726103`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.4063093427071376, -0.6945819760633463, -0.5936906572928624}, {
       0.6945819760633463, -0.18738131458572488`, 
       0.6945819760633463}, {-0.5936906572928624, -0.6945819760633463, 
       0.4063093427071376}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.20985615202546842`, 0.9549454387105721, 
      0.20985615202546842`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.34549150281252616`, -0.6724985119639573, -0.6545084971874738}, {
       0.6724985119639573, -0.30901699437494756`, 
       0.6724985119639573}, {-0.6545084971874738, -0.6724985119639573, 
       0.34549150281252616`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.12357030873424568`, 0.9846119832698801, 
      0.12357030873424568`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.28711035421746367`, -0.6398093445997584, -0.7128896457825363}, {
       0.6398093445997584, -0.4257792915650727, 
       0.6398093445997584}, {-0.7128896457825363, -0.6398093445997584, 
       0.28711035421746367`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.035335687856959175`, 0.9987506086743334, 
      0.035335687856959175`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.23208660251050153`, -0.5970300016676449, -0.7679133974894985}, {
       0.5970300016676449, -0.5358267949789969, 
       0.5970300016676449}, {-0.7679133974894985, -0.5970300016676449, 
       0.23208660251050153`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.053456197926347884`, 
      0.9971383403552982, -0.053456197926347884`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.18128800512565513`, -0.5448351389607972, -0.8187119948743449}, {
       0.5448351389607972, -0.6374239897486897, 
       0.5448351389607972}, {-0.8187119948743449, -0.5448351389607972, 
       0.18128800512565513`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.14140504753517197`, 
      0.9798006047472884, -0.14140504753517197`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.13551568628929417`, -0.48404790064380154`, -0.8644843137107058}, {
       0.48404790064380154`, -0.7289686274214118, 
       0.48404790064380154`}, {-0.8644843137107058, -0.48404790064380154`, 
       0.13551568628929417`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.22712385507308522`, 
      0.9470108282979032, -0.22712385507308522`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.09549150281252636, -0.4156269377774535, -0.9045084971874737}, {
       0.4156269377774535, -0.8090169943749473, 
       0.4156269377774535}, {-0.9045084971874737, -0.4156269377774535, 
       0.09549150281252636}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.3092607837392817, 
      0.8992861253694124, -0.3092607837392817}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.061846659978068264`, -0.34065128981885684`, -0.9381533400219317}, {
       0.34065128981885684`, -0.8763066800438635, 
       0.34065128981885684`}, {-0.9381533400219317, -0.34065128981885684`, 
       0.061846659978068264`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.3865204851024723, 
      0.8373791430363543, -0.3865204851024723}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.0351117570558743, -0.26030336752460015`, -0.9648882429441257}, {
       0.26030336752460015`, -0.9297764858882515, 
       0.26030336752460015`}, {-0.9648882429441257, -0.26030336752460015`, 
       0.0351117570558743}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.4576845275194508, 
      0.7622661913915597, -0.4576845275194508}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.015708419435684517`, -0.17585030562678616`, -0.9842915805643154}, {
       0.17585030562678616`, -0.968583161128631, 
       0.17585030562678616`}, {-0.9842915805643154, -0.17585030562678616`, 
       0.015708419435684517`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.5216306115299638, 
      0.6751318465527694, -0.5216306115299638}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.0039426493427611176`, -0.0886239793613568, -0.9960573506572389}, {
       0.0886239793613568, -0.9921147013144778, 
       0.0886239793613568}, {-0.9960573506572389, -0.0886239793613568, 
       0.0039426493427611176`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.5773502691896258, 
      0.5773502691896255, -0.5773502691896258}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       5.551115123125783*^-17, 
       2.220446049250313*^-16, -0.9999999999999999}, \
{-2.220446049250313*^-16, -0.9999999999999999, -2.220446049250313*^-16}, \
{-0.9999999999999999, 2.220446049250313*^-16, 5.551115123125783*^-17}}, {0, 0,
        0}}]], 
    InterpretationBox[
     Point3DBox[{-0.6239647682118342, 
      0.4704635331890284, -0.6239647682118342}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.0039426493427611176`, 
       0.08862397936135696, -0.9960573506572388}, {-0.08862397936135696, \
-0.9921147013144778, -0.08862397936135696}, {-0.9960573506572388, 
       0.08862397936135696, 0.0039426493427611176`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.6607389701008569, 
      0.3561573062287477, -0.6607389701008569}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.015708419435684517`, 
       0.17585030562678633`, -0.9842915805643155}, {-0.17585030562678633`, \
-0.968583161128631, -0.17585030562678633`}, {-0.9842915805643155, 
       0.17585030562678633`, 0.015708419435684517`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.6870929237250604, 
      0.23623426579117773`, -0.6870929237250604}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.03511175705587444, 
       0.2603033675246005, -0.9648882429441256}, {-0.2603033675246005, \
-0.9297764858882511, -0.2603033675246005}, {-0.9648882429441256, 
       0.2603033675246005, 0.03511175705587444}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.7026110114927022, 
      0.11258566986257128`, -0.7026110114927022}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.061846659978068264`, 
       0.34065128981885695`, -0.9381533400219317}, {-0.34065128981885695`, \
-0.8763066800438635, -0.34065128981885695`}, {-0.9381533400219317, 
       0.34065128981885695`, 0.061846659978068264`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.7070485038896305, -0.012838469335187714`, \
-0.7070485038896305}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.09549150281252639, 
       0.4156269377774536, -0.9045084971874735}, {-0.4156269377774536, \
-0.809016994374947, -0.4156269377774536}, {-0.9045084971874735, 
       0.4156269377774536, 0.09549150281252639}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.7003354190099161, -0.13806013820220034`, \
-0.7003354190099161}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.13551568628929422`, 
       0.48404790064380165`, -0.8644843137107058}, {-0.48404790064380165`, \
-0.7289686274214114, -0.48404790064380165`}, {-0.8644843137107058, 
       0.48404790064380165`, 0.13551568628929422`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.6825776262123147, -0.2611045162166359, \
-0.6825776262123147}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.18128800512565524`, 
       0.5448351389607973, -0.8187119948743448}, {-0.5448351389607973, \
-0.6374239897486894, -0.5448351389607973}, {-0.8187119948743448, 
       0.5448351389607973, 0.18128800512565524`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.6540551764972357, -0.3800311200340573, \
-0.6540551764972357}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.23208660251050184`, 
       0.5970300016676451, -0.7679133974894982}, {-0.5970300016676451, \
-0.5358267949789963, -0.5970300016676451}, {-0.7679133974894982, 
       0.5970300016676451, 0.23208660251050184`}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.6152178859351712, -0.4929644060689548, \
-0.6152178859351712}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.2871103542174639, 
       0.6398093445997586, -0.712889645782536}, {-0.6398093445997586, \
-0.4257792915650721, -0.6398093445997586}, {-0.712889645782536, 
       0.6398093445997586, 0.2871103542174639}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.5666782417985585, -0.598123348937482, \
-0.5666782417985585}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.3454915028125263, 
       0.6724985119639573, -0.6545084971874737}, {-0.6724985119639573, \
-0.30901699437494756`, -0.6724985119639573}, {-0.6545084971874737, 
       0.6724985119639573, 0.3454915028125263}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.5092017432716093, -0.6938495292916963, \
-0.5092017432716093}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.4063093427071377, 
       0.6945819760633463, -0.5936906572928623}, {-0.6945819760633463, \
-0.18738131458572466`, -0.6945819760633463}, {-0.5936906572928623, 
       0.6945819760633463, 0.4063093427071377}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.4436948290708895, -0.7786332880833625, \
-0.4436948290708895}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.4686047402353434, 
       0.7057114674770557, -0.5313952597646566}, {-0.7057114674770557, \
-0.0627905195293132, -0.7057114674770557}, {-0.5313952597646566, 
       0.7057114674770557, 0.4686047402353434}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.3711905823652783, -0.8511375347889737, \
-0.3711905823652783}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.5313952597646568, 
       0.7057114674770557, -0.46860474023534315`}, {-0.7057114674770557, 
       0.06279051952931375, -0.7057114674770557}, {-0.46860474023534315`, 
       0.7057114674770557, 0.5313952597646568}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.2928324384372607, -0.9102188341260445, \
-0.2928324384372607}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.5936906572928626, 
       0.694581976063346, -0.4063093427071374}, {-0.694581976063346, 
       0.18738131458572505`, -0.694581976063346}, {-0.4063093427071374, 
       0.694581976063346, 0.5936906572928626}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.20985615202546887`, -0.9549454387105722, \
-0.20985615202546887`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.6545084971874735, 
       0.6724985119639575, -0.3454915028125264}, {-0.6724985119639575, 
       0.3090169943749473, -0.6724985119639575}, {-0.3454915028125264, 
       0.6724985119639575, 0.6545084971874735}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.12357030873424588`, -0.9846119832698801, \
-0.12357030873424588`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.7128896457825362, 
       0.6398093445997585, -0.2871103542174638}, {-0.6398093445997585, 
       0.42577929156507255`, -0.6398093445997585}, {-0.2871103542174638, 
       0.6398093445997585, 0.7128896457825362}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{-0.03533568785695936, -0.9987506086743334, \
-0.03533568785695936}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.7679133974894984, 
       0.597030001667645, -0.23208660251050164`}, {-0.597030001667645, 
       0.5358267949789968, -0.597030001667645}, {-0.23208660251050164`, 
       0.597030001667645, 0.7679133974894984}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.05345619792634809, -0.9971383403552982, 
      0.05345619792634809}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.818711994874345, 
       0.544835138960797, -0.181288005125655}, {-0.544835138960797, 
       0.63742398974869, -0.544835138960797}, {-0.181288005125655, 
       0.544835138960797, 0.818711994874345}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.14140504753517222`, -0.979800604747288, 
      0.14140504753517222`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.8644843137107059, 
       0.4840479006438013, -0.13551568628929408`}, {-0.4840479006438013, 
       0.7289686274214118, -0.4840479006438013}, {-0.13551568628929408`, 
       0.4840479006438013, 0.8644843137107059}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.22712385507308583`, -0.9470108282979028, 
      0.22712385507308583`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.904508497187474, 
       0.41562693777745296`, -0.09549150281252608}, {-0.41562693777745296`, 
       0.8090169943749478, -0.41562693777745296`}, {-0.09549150281252608, 
       0.41562693777745296`, 0.904508497187474}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.3092607837392817, -0.8992861253694124, 
      0.3092607837392817}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9381533400219317, 
       0.34065128981885684`, -0.061846659978068264`}, {-0.34065128981885684`, 
       0.8763066800438635, -0.34065128981885684`}, {-0.061846659978068264`, 
       0.34065128981885684`, 0.9381533400219317}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.3865204851024723, -0.8373791430363543, 
      0.3865204851024723}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9648882429441257, 
       0.26030336752460015`, -0.03511175705587427}, {-0.26030336752460015`, 
       0.9297764858882515, -0.26030336752460015`}, {-0.03511175705587427, 
       0.26030336752460015`, 0.9648882429441257}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.45768452751945105`, -0.7622661913915596, 
      0.45768452751945105`}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9842915805643155, 
       0.17585030562678594`, -0.015708419435684462`}, {-0.17585030562678594`, 
       0.9685831611286311, -0.17585030562678594`}, {-0.015708419435684462`, 
       0.17585030562678594`, 0.9842915805643155}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.5216306115299638, -0.6751318465527691, 
      0.5216306115299638}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       0.9960573506572388, 
       0.08862397936135663, -0.0039426493427611176`}, {-0.08862397936135663, 
       0.9921147013144778, -0.08862397936135663}, {-0.0039426493427611176`, 
       0.08862397936135663, 0.9960573506572388}}, {0, 0, 0}}]], 
    InterpretationBox[
     Point3DBox[{0.5773502691896261, -0.5773502691896253, 
      0.5773502691896261}],
     GeometricTransformation[
      Point[{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}], {{{
       1., -4.440892098500626*^-16, 0.}, {4.440892098500626*^-16, 1., 
       4.440892098500626*^-16}, {0., -4.440892098500626*^-16, 1.}}, {0, 0, 
       0}}]]}, {{{}, {}, 
     {Hue[0.67, 0.6, 0.6], Line3DBox[CompressedData["
1:eJxF2Xk0VV0fB3AJTSoZklBEoolSkWTTQE+hCBmjyVDKlKmIUKkINyRzMoTM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       "]]}}, 
    {PointSize[0.02], 
     Point3DBox[
      NCache[{{3^Rational[-1, 2], -3^Rational[-1, 2], 3^Rational[-1, 2]}, {
        3^Rational[-1, 2], 3^Rational[-1, 2], 3^Rational[-1, 2]}}, {{
       0.5773502691896258, -0.5773502691896258, 0.5773502691896258}, {
       0.5773502691896258, 0.5773502691896258, 0.5773502691896258}}]], 
     {Opacity[0.5], SphereBox[{0, 0, 0}]}}}}]], "Output",
 ImageSize->{184, 195},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->6160808,
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJyUvHd0a9t953cC3nufnmxLrnKVLMuybLmObMeezNiZlWTNOGsyjseSMyvj
WU4sybKaVZ5ev73xkgSJDhaQIOpBLxu9AyQBoreN3kGw91uf/0q8kt8+AN+T
ca/k97i+2DgHZ+MQ+/P9/X57HwDkn3/xta995VtffO3rX/7iJ//se1/87te+
/uVXP/kfvvM9eGjiv6MoygW32M9RsP3/wRZze+uf/umfqImvfOUr/8T8UP8D
aajXob0ED/5bZu9VaFmwNzG6/zfMo6+M9i5/3xmgHZ3h29CSB1964dF/GD33
w2Tv0ru/HTb+NXP866Nn/+gPOP7V0fM/SvYuv3t89Dv+mOnzlVGfnxgbIWts
/w+Z3l8a/cafel+9/3Z07o+NHZ0Y2/8DpvffjM798//CuYe9/9vo3L/0L5z7
95ne/3XU+xNjRy+N7X+O6f1fRq/kk++r91+Nzv2rY0cvj+3/K6b350e9f23s
6JWx/d9jev/F6JX8+vvq/eejc//m2NGXxvZ/l+n9H0fn/u331ft/HZ3798aO
fmhs/3eY3v9h1PtzY0dfHttnfjf1v4xeyR+8r97/0+jcfzR29MNj+7/F9P53
o97/+n31/pPRK/k3Y0d/ZGyfIUz9m9G5/3Ts6I+O7X+W6f3Ho3P/u/fV+78f
nft/Hjv6Y2P7v8H0/oNR73//L/Rm4oiJRPLgn72v3r87Ovd/HDv6kbH9zzC9
f2t07v/0vnp/dnTuvxg7+tGxfSZbmNdDev/l2NEfH9v/NNP706NX8oX31ftT
o3P/l7Gj43WRyXLqk6Pe/+f76v3x0Sv567GjPzm2/ymm9y+Ozv03Y0fHa+6v
ML1/bnTu//t99f7Y6NxfGjv602P7TOVjHiW9/27s6M+M7f8y0/snRq/k799X
73EXvjq2Pz5nMJWbiQvyir7+vnt//6z6Lz3r4+8965/NpqQ+UT9++5bx3m3T
5G3T1C3z9C0z+5Zl7paFA7pp4d6w8EDXzfxrZsFVs+Btk+At4/i4f3Zs/5fe
+23Pzc1/Sh796O27xrv3TA/umdn3LHP3LJx7Vu49K/+eVQi6axXfRfN30MId
pr2NKNbCLURNLN60Lt2wLl23UizJNSt1WXLVInnbInnTvPSGcfFV3XOZ9UNe
1Q+Zg5na+2O37+rv3jdNTpqnJy2zk1bOpJU/aRVOIvEkmp9EC5NoaRJJ7sMr
W4bm8so9JAXdRaugO/DoKrzoH5PdQrKbjG4g2XUku2aVX7XK37bK37LI37DI
XjdJXzNKvqN5rgL/3Nj+LzKve7zKjM/YzGrgI3fv6+9PGqcemKenLLNTVs6U
lT9lFU4h8RRamEJLU2gZ9ABJHyDWKtzkk4i6pJxEyvtEqnuIpbqLqCuqO0h9
m9EtpL6J1DcQS30dUR9SX0P0VUS/ZaXftNJvWNWvm1WvmpSvGMZns/c3gvFV
BDP/fmxy0jA9ZWJPmWenrZxpK3/aKpxB4hm0MIMWZ9DyNJJOo9VpJJtG8mmk
mELKKaSaQuoHiH6ANJNIO4lY2vuIpbuHqJ/S3UW6O0gPuo30t5DuFtLfRLob
SHcd6a4h/VWkfxvp30L6N636N6y61yzaV83aV4yqb+ueW4WMr9x+4YUjGn8W
M/N+ZHJSOzVlnJk2zc5YODNWPtsqZFvFbLTARktstMxGK2y0ykZyNmIpZxD1
kmoGqWcQPY0000g7hXRTSD8FMWUAv4wwOiOJONM9ZLrL6A4ykQwxQYZcMd1E
phvIeB2ZriHTVWR+G7FMb5HH37SaXreaXrWYvmfWf9eo/bZ+PHp+4UXjfW58
4+snZob+2IMH2ukpA3vGNDdj5rEtZHyz1vlZtDiLJLNoZRZJZ9HqLJLPIuUs
UoPYSMNGWjbSzSD9DDLMwJimEcsMg7xkfoAsoElkhZFaYaQfs95D6C6jOwjd
RugWsoJuIusNZLmBrNeR9RpCVxF6G6G3EHoTWd+wWl+3Wl+1WL5nMn3HaPgH
3XiGv7+xjq/+mHn6Jyfvq2em9bMzxjm2mTNr4c9ahXNIPIcW5pBkDi3PIekc
ks0hxRxSzSF6DmnmkHYWsfRwM7JhfCY2Ms8gCwhGjOBmg1H/uO0Bsk8SOe4j
J+gect4lreMOctxGjlvIDrqJbKAbyA7ZaIds/BH7NWR/G9nfQvY3kP11q/1V
q+1VM3rFZP2WwfRNLam1H39RXj430vHsZWb2n2ZP6+ZmDJxZE2fWzJu7GCYH
LXHQCgetcpCcg5QcpOYgmoM0HKSbQ/o5ZJhDpjnwchaqhXUWWdkIsZGNjewz
yAGaRk4YsxPGPOF6gFyTyA0FiEWal933kfsekecuct9B7tvIfQu5biIXBDbL
eROeBeN2wrhfdl1FrreQ63Xkes3qfM3ieNVk/44RXr7xa2pq/JppOOLx1d94
xWXWBD8xM0XPzeg5bCN31sSbswg4VhEHzXPQIoyYi1a5SM5FCi5ScRHNRVou
0nFIVkIzYeIgC2gO9hE0L9lmkX0WOdjIyUYu0AzyQGKzPNNowkvG7n2AfIwo
lg/G/pL/PvLdI/KC7iLvHeQhs4gHEnuCgLiJPDfIPjQve64jz1XkeQt53kTu
1yzuV80wDue39bZvaJ8b/c++cPTjq3JmRfHRuWktl63nzRr5c2YhxyLkWOe5
aJGLJFwyehkPKXhIxUM0D2l58Fr0XIhmIxeZuMjCRVYORDKQ+JB9DjnmkHMO
uWaRG8RGXtAMmvDNoEv+aahFgSkUHOoBCk6iAFS1ADD4cOA+8l+IwGAw+EC3
ke8WYvlvwtj915HvGvK9hXxvWuD1e181ub9rcH5LZ/2qejjHjK+Anhv7+D7z
rJ+YnVJzZ7S8Wb1gzijkmEUci5hrXeCiJR5a5iEpjwxfyUNqHtLwkI6HDISA
CZoPmXnIykU2LrJzkYODnFzk4iI318bycGzUZe+czTdn888SBdk2ihWctVGX
QrO2EHskVnjGRl0JT9tCM7bwNCKaIi0rROIk9GBEiWIBHuoSoAncQ4G7KHAH
sYIQHS8Hb6LgdRS4ZoUh+d80+183+V41uL+rd36dfq7OvZjIeDVgquEvcabV
PLZOMGcQcoxijlnMtSwAER6S8NAK4OAjBR+p+EjDRzo+0vORkY8AhQVo8BDi
ITvP5uDanFwboHDzbB6u3cu1s3wcOyvAsVOXgxx7aM4eBs2SlmJFoJmIcOxw
v0Y212ah1zrbvsa2jQSYWBFofjoygyIXoEIMnBDoPgrdQ+G7KHwHhW+j8C0E
Awpds4beNgffNAVeM/he0bu/pRkn8jMvJPLCdeSnuDO0YFYnnDOIOMZ5rnme
a1nkWSFAJHwk5RMiSj5SXxAx8JGJbzPzbVa+DfFtdhDP7uTZ3SCGhZfn8IO4
jgDXEeQ6QiCOI8JxAAkHDH/OscYhoi6tcx0TG7DxkY05R3TOsTFnJ5ol7fos
yLZ+wWcIJjKFWBEoLD8amURr91HkHorcITAit6yRG5bINXP4bVPodUPgVb3v
28/x+OkX8hi/nmGo/TSfTQtntSKOXszwWOARHhK+dZnhQcJDgGgB0gqQXmAz
Cmxmgc0iABh2G9/OcvDt1IddfIeb5/DwHD5CwhngOYMgrjMM4jlZEa6TurLO
ca5zXesc1wbXFSVysmJwi/LgGGzEeM53200gFeM6qJdiHEeUY4/OETG8bICJ
YgEo6qPrM2h9Gq1PofUHaH2SgFm/i9ZvW9dvWtZvmNeumcJvGUKv6QKvaFzf
UD23jh3yGa+f49e6zPXpLwrYtGhOK+bo57nGBZ5pkWde4luW+dYVCBYBUgiG
fGxagY3hYzcL7FaBHQnsdoHDIXAQOHyHh+/0gXgEDsUK8VzU5TDPFea71ngg
9zrPTbE2oLkU5bljXE+M597kuVmbfHgENuB+KFacPBIXuOJ8RgInPAI3OOnP
bvIdmzwHgItx7UNsTIDZNti2KGgGRacJpugkit5D0bvW6B3Lxk3z+nXj2tuG
yBu60Pc0nm8qnyP1Uy8kNX7lzryn8DNCthpIzXP0C1zDkJSEb1nhW6UCq1yA
lAKbWmDTCGw6gd0gsJsEdovAgQQOm8ABpJwC4OKG5opX4PIJXH6BOyhwh/ju
MN/Digg81KU1gWdd4N1gxIoJvNTLmwJfnO+LC3wJkBA2vHGhNwGCgyy4py4n
RN4kSOhJijxJoZs86qY+nBC4kkJXQkj4Ab53wW1y7TGobDEoXB+JzdpibFts
hiDbnEKxB9bYPWvsriV6yxS9Ydy4pl97Uxd+lfZ9SzF+dfpiYuPvXjDV+ld4
U3IhWyWe0yxwdABtkWdc4hNoUoFllUCzqQCa0KYR2nVCu0FoNwkdFqEDCR02
odMhdLoELrfA5RW4vUK3X+gJCDxBgTcM4vvW+L51AchPsaJCPysm8kPciPxx
UQCUEAUuJUUBaiIpDiRF/pQYepHmcmqe7KTFvvS8D1pqIiUGiimxJwX8RG5A
CNSAIvj0U3EGGyN7nGuPc+zxWRsMMD5ji0+j+JQ1PmnZvGfevG2K3TREr+s3
rmojr9OB7yqeu57/yRcSG39vgqlmnxKxVfMMrkWufolnlPBNy3zzqsAySkWh
jRbatUK7Xmg3Ch1mocPKsLILXU6h2yX0eIQer8DrE3j9Ql9Q6AsJ/GGBfw0k
DGwIAhvCYFQYjAmDrE1RkJqIi4MJcSg5H6JYSXGImkiLQ6l5ItiE5lJ6IZRZ
CBItBllwN5FZAKaZeX9mAeSjWECR+rm02EsAEobupGgYec6EwJHgOxI8R4Jr
hyEm5mwJti0xg5LT1sQDS+K+OX7XuHnLELuhi17VrL2hDr0iH6f2Ey+kNv5+
EbkmuAT9qF9f5GjhXvOxz10wg/iyKhlmGsLMoRc6GGZOq9BlE7rtQrdT6HEJ
vR6hz0fkDwj8IWEgLAxEhME1QXBdGNoQhqKCcEwYjonCcZA4nBCHk/PhFFEk
PR8BBgsRYLIYySxEoKVY2cUwK7sUpi7BRm6JkSTMyi2FyCOh7FIwBwKe2cUA
9fPZhRHL9Lw3NU8gJi8gJgXOJN8BA01x7SmOLTWLUmyUmrEmp8zJSVPinjF+
2xC/qYtd1ay/qQq/Kn+OzIv5jVMmV8pXoB/rf/yTRepTK3wTbMsEVoXQdhFr
wM1pFLrMQrdV5LaJPHaR1yn0uYU+j9DvEwb8wmBAGAwJQhEhKLwOEoSjQEwY
2RRG4qJIXLyWEK8l59dSoIW1zPwaK7O4Tl3OLq1nF9ehzYEk6xQrJ1mbyEvW
WPnltYnCcoT6ZF4Szi+HSSsBiCFgCOyyRIHM4nvgRgEocqeELhhYmu9I8+xp
ri09Z0vPojTbmpoypx4YU/cMybv6xE3t5lV64y1l8NWVcWI//kJi4zPGlWHY
MaEGWuLQEq52madndn+fYGSRePwp4MeUNqdO5DKK3GaRxyLyIpHPLvI5RX63
KOARBn3CUEAYCgrDIWE4IgivCSPrQoihqGiNurIpWtsUrydE6wnxenIetAGx
tgHsNqhLuaWN3FI0J4nmJVHgFaUuFVY2CsuMVjZYxZUN6uWidAPui9J1UGFl
vbiyVpSukXYFfgPh+wuF5SHfEAMXAjMAQQlY0yOsHhh0RujKCJwZviPLs2U5
tsycNTNryUyb0lPG9H196p4ucUuzeV0dvap4Mc3xK9jxlRzz3tqHhX/2Nwv/
2xdJ7nI1yzzdCl8v5ZukAlL4LmLzZ7RCJ2wbRR6zyGsV+Wwiv10UcIqCblHQ
Kwr5RWG/MBwURcKiSES0tiZaWxetR8Xr0fmNTfHG5vwGVPX5KPVyaiGaWoym
F2PppVgWJInlQMux/PImxcqvbFJXCiubhZVYQUpUlMbwaoyFpTFqAjZKqzEs
jVIsvArEsXQDr24UVwli6LEOPUaIgS9wpT52EbbB7CKUUEKWABV7siJ3VujK
Chw5nj3HBayW7Kw5O2PMTBkyk7r0PW3yFh2/oYpdk48z/egLmY5fUTKfv35E
8GB5ga1Y+k9fWv7f/26Fp13h6aR846rAtCowy4QkwdVCp1boArZMvP6kVRSw
iYIOUdAlCrlFYa847BdH/OK1oGgtLFpfE4M2NsQbUXE0Nh/dXIjGF2KJhRgr
tbgJ08XSZoYoDvVPEofRx1mFlTirKE1QV4qrieJqHINko5ZVksWpKyV5HO5L
ss2yfLMMrSwGj5MbsC2tRoEtg3cdzrJG/WxhJQIxOywHUAigCsB4oXzm5n05
sTcn8uSFzhwfiNpyXEtu1pxjG7PT+uwDXfquJnVHnbil3Li2+lyN/MgLeY6/
O8G8L/fr8zOypTnl8p9/GcJUytOt8g2rAqMMYMLMLHSohC5a6NaJPEaRD04J
VP+PP2Q7RGGXOOwRR7ziiE+85hevB8UbYdFGRBRdE0c35mMxkHhzc2EzvriZ
WIynluIpSSKzlMhIEtnlRG45QWIySV3KS5NFabKwmiyuJllYlqImSvJkSUZE
TZTl0KOsSMJ9RUE0UVEkqF8tK+JleZxpNyvyIeVYWR4FwCXZBoN3WBrWgG0e
2ErCMGSI1vxiIL/gz8/78mJPXuTKCxx5PspzLfk5U37WkJvWZR9oM/fp1F1V
4rZi7erKcyuZF1P9OLPoeXd/flq6NKtY5qhWuLT0L/5+9T9/VcY3AE45wWln
cHq0Iq9e5DeKghZRyCYKO0QRphYw8fpRv3gjKI6GxdGIKLYu2twQb0bn47GF
eHw+EV9IJBcTqaVkWpLMSFJZSQpml2Wgll9JFaTpojRNsYqraSCZJiTTZUYU
q6KA/YoyDfdV2GRVlWnqclWVrqpSNWWqpkrBBjVRVSapT1aViQpIASKAywoA
HCuBZFHM0GUK71p+ZQ1GDGgLklBhKVhYDBQW/IV5b+GCa4FrKXCMhVlDflqb
e6DJTKrT95Xx27LxSP2xFzL9xAVT8dTyIlu2PKdc4ailXHr1L/4e5iK5wESW
QEK7UuhSE5o+vShoFIXMorBVFLaLIk7Rulu8PpyvfgbABsWxkHgzIt5cF8Wj
okRUnNicT8bnk4mFVHIxlV5KpSXpjCSdXU7nVjJ5aaYAWs0UQbIMVEV5lrpc
UmTLQymzFUUWmCqzE1VVFrCpszVVpqaGnqS5wuykiYaA1SkWEKY+AXgBbhXI
KuMAt3JBFsuiUGwL0g0CdHkNqkFBEi4uhYqLwSIwFROmBaGjQJiai8B0Tpef
0eSm1NkHyvSkYvO2dJzpj76QKfPZ6Ge++6U3lmcVKxyllKuW8TQyvl7ON333
b8WKv3xFJXQCTVro1xGaYYZmxCZacwBNUdQjjvrEMThvQLwJSCPixLo4ERUl
o+LU5nwqvpBOLKSTi+m0JJNezmQlmexKNreSzUuzrMJqDiIyxyrJc1ABFbmy
Il9W5ivKPKuqygOzPDVRU+Vr6lxdnaMu1+kcq0ED1zqdrdOZOuEK99AvTX1i
GKpVVRLEhOmoDkCBxbIYASiFhcJ6YXmtuBwpSsJ4iTAskrh0DxkWecDQUJzV
Fth0fkaVnVKkJmWx20vjuT5kOP5NDobhz89PSSRs2et/f1XKUcm4tJynA4aK
v/z2d7+4qPz8qyqGoVYY1ItCJlHEIlqzidbtog2nKOoWxbziTb84HhQnQuJE
RJx8L+9f3lzIJBYzyaVsSpJNL2ezy7nsSi4vBeUhq2V56jKW57GiUFIUyspC
SVmAGFQVSPwVquoiiMRfgfpQjS7UQRrSNkCaQlNTaGgLlxoaIN3Q5JqaXIPO
UayGJkv9cmNEOV1XQcySWlAjaAldQFuSb8LgC7JYYTVaIOuu9eLyGl4mXPFC
ALgWxYRrUXDBdU5TmFXn2crstDw5KWWuqcnH3T/2QppMlv+S6P7C0ox0ZVYu
5Sjf/Op1OU+r4BsVfItSYFP+5av//g9n1EKfRhjQC0NGUcQsWkeiDbso6hTF
3KJNryjuFyeC4mRInAzPp9bm0+sL6eh8hiH6o4mlXAokyWWW85mVfHYln5cW
8rJCUVYsyotYXiwpiqyysgizjApXQGpcVWNWjcYk+nBdQ0SQYepKQ4ubIB3T
aotNXZHV0hapKy1doakdKt/U5KEFrjnqEwzXLEQvCEK3qia1ALiWlQkAgOXx
gmwTuBZXo7CehQUWZoIVL/rxAoFaFAFUa5FrKnL1RQ5dnFPl2YrM9GrivoSs
lQjE8W+j/cgPZvzxxSnJMntVOieXcZRyrubtr90jgPk2lcChErgJ4M9f1X3+
ukEYMYnWLaKoTRRziDZdorhHlPCJkwFxKjSfDi+k1xYy64uZ6EI2tpgjiJkV
7IdSkkJmuZBZKeZWivnVYl6GiyA5ZpUUJeBboi5VVaWKqlRVl2rqMvAtA19N
uQHSlilWE5pLTV25pS23oNWVWC19idXWl6iXYKOlxy0dJq2eUAf25BkFuOXh
WRDKJJqzNTrLqkKp+NmKOl1WpQhiZbKoSBTl8aJsswjlYUgZQlcSxEt+TELX
NaLMNxZ52iJXXeQocrOy9MwKk/0E4vjnyT+E8WckMysrs7LVOQUAlnP1Cp7p
2tenh4BVAi8pCZ+/rv/8TZOQAEaimF0Ud4kSHnHSJ04FxOnQfCa8kIksZteW
chuLudhSfnMpH5cUkpJiSlIEzpkVnFsFlQqyUlEOKrNKSiBZUZUr6kpVXYF5
na6wapoKFFFtpaGtNnTVJkhbBWK6ykRLXwGqlYm2Ae4N8MyOoQyUCWiG9qW2
jkGtK4IIY+qXoGTUNfkanatpslU6W1ZnCFlVCitTxSFfsvb953AXfXjBg+cB
rr0otBQFxiJfU+SpClx5dlaampF8aMR2/FPPF7NlVle/sjQlWWFLV2chftVy
rg7YKnno2jfmVAKXWuChBX4ayu3nbxo/f/v6l/RWUcwmijvESZc45RGnfeJM
QJwNzWfDC7m1xfzGYj62VIhJivHlYnIZp5ZxermUkZbeq8SXsKJSUlZKqmpF
VWVV1FXgWoV5XVOra2oNLSNdramrsZr6GjXRMtTaeiKK1THUWB1jjZCtdYzV
rhHIw/2lzpA4Q/mXAXFbj0ko64pQluvaQk2Tr2ryFToHwy6pMliVLgJfBSgB
hQImNljRYukaXgG+Abw05OssilFRZC4K9UWBpshXFLir2dll5rqT4Bu/2v8h
dD+1PC2RsqWyOZWco1FwDUqeRcW3q/mu69/g0wIfLQhqhGG9cN0kjN74shF9
YdL+V1NOccotznjFWb84FxTnw/P5yEJhfaGwsViMLeG4BCeWS8mVUnqlnJGW
s6uV3GplgrD9MFZWS6paWVWrqGsVul4Daep1kLbe0NZZDV0dsr9BXW4ZGu2h
jPW2oU441lkdU52a6JrqXWMdWorVJay7JmBdY1iT5tNdY6UDMlSY8C63DVBC
Sk09buiKdV2xqi1UNHkCms5hdRarMkUGN1YkGdZRwprEcmjEetGF5+143loU
m4oiXVGoLvLlee5Kmr04Ij3+adWQ9Hh0M6Q/vTS1uDKzLJuVy+doBUen5JpV
PKTmO9V8t1rgu/kNsUYQ1grXDMKoSbhpEcVvfdli/8KMU5xxi3NecS4gzofE
hbC4uDaP1xdwdLG0uVSKS8rJlUpquZKWVrOr1dxqNS+rFRQ1JpZ/BIK6rK5X
1I2qplEDaRt1kK7R0DUb+mZT36RYLUMTeBubbUOzbWRkakIIQ3Opa2r2zERd
U4PVNTeoD/VMjZ6p3jPXoSVOmIB8DVoS59QngXqbqNwylBv6Ul1fqulwVVcs
awsEuSaP6VxRTZBjVRIrE1ixOUIujeCVEF4G5B686MQLdrxgLc4bimJtUaTU
f+MNzVe/B2cY/6Tnh7D+3PLU0ip7VT6rUsxplRyTimtV8xwENN+rFgQ0gtDN
b0p0wg2DMGYWJqyipO0LM44vzLhEWY8o5xMVAqJiSIQj4tLafGljoRxbKm9K
KollAF1NS2vZ1VpWVs/L6wVFvahsYGWjrGqU1U3AXdU0a5pWTdtq6Fp1kL7V
BBlaLUMLWBvbEMVtwGtsd01EELXmFnWpZ2n1zIwsLVYfHrnctzT75hH+nqVB
sXrA/9NgBMFuJuQ7BH4VwLcMlaax0jCU64ZyTV+q6DChrS1gTaEIwOksVqdJ
1Sb1ZAh8DUvDDHAfXnLjJYfhlRnDdx8YvnNX/+3rRZGsIJCkZuffH23mXYXP
LE3Or0wvy9gKJrL1Kq5FzbMxwL1qvh9oawQRrWDjzjelRmEcgCNRyi7KOEXZ
yS87vaKCX1QMikphUTkyX15fqGwsVWKSalxSTSzXUtJ6ZrWelTXy8kZe0Sgq
m1jZLKlarDLdol6ualrAHASvrKFrN/Ttpr7TNHRahg7MasYO0O4AbXOnC7J0
CO3ORM/SYfVh53Lf2u5bGDEb1ESf0Lc0qV9l0EOoNxjUdUDdNtZaxmrTUK0b
KjV9uWooE8g6jHWFIuGcZ8oJw1kZx4oYlm9gGeFseE1seFVgeJWDJXYsseIl
E17UFucVRZE0x1987jO5D/9gyp9afrCwOiOVsyGmdUqOUcVFaq5dzXOreT6a
H9TwAfG6ThDVCzYnv6mwCFNIlLGLck5R3i0qTH3J4/+CICgqh+crkYXq+lI1
KqnFJLX4cj0pbaRWGxlZMydv5hUtVlHZol4qqdplmqiq6VS1nTpI12WuqVlN
QxfYdqnLbVO3AzJ3u+ZuzwxHe5YuxLK127d0+1aiS1vWLjWxhcj2lhWsIM2v
b1kvoFtJwPcszS7UGHOjY260zfWWqd401RrGWt1QrRkqAKGsLxHS2mKRBDVU
kQxWp5gSEje8LTW8uWJ4YwmvhvCqH6948IoTLyO8bMESPV5UF+dlBZEkyRE+
9znxi0kz7zr+9uI94cqURDYjl8/SyjmDimNRc21qrlPN89K8gIYf0vIjOsGG
ThAzCOImQWr6m2qbKOcQ5V2iokdU9H1BEPiCMPRXovBCbW2xtrFUj0rqm8uN
xEozudpMr7ay8lZO0S4o20Vlp6TulOhORdOtgrTdOkjXbRh6TAV/qWnstUy9
Nsjc65h6HXOva+4T0n1qomft9xlRrL61B+GLelsg2GTBPWvA3LoEP/WZLdQB
9n3UAew9kKXVtbQ65mbb0myZG01zo2Gq1021mrFGgBvKJX0Z60pYV8SEec5w
gzZcVxmuKrAijuVRLF/HsgiWBfGqD0tdWGrHK1a8bMQSLV5SFBekWYH4A/D+
Xcl90eq0VD6jVMxqVXMmNcdKIpvroSGyeUxk89d1fIjsuEGQNAvSVkGO/Q9a
h7DgEmKPqOQTVQKi6txXghHgvdTYkDRiy83NlVZitZVabadl7ayik1d2Csou
VndLdLes6VY0vaq2V9P26rpe3dBvgoz9lrHPrAPb5i1W17IFobxFqG71rVvQ
EqpbrC0bPDZAWwNbH0SxoIFbj7oEuOF+YOsCc+jZZQFs6pPAumttdyyttqXV
sjQblmbd3KiZ6kCgYqyWDZWSoWS4YzLcNhhu6gA0E9xJrEpg5SZWRLFiDcvC
WBbAMg9edeFVO5aa8YoeL6uLS7K8ePGDUF6eXJBNy+QzauWsXjVnUXNsao5T
zSUhTfPCGv66lh/V8TcNgoRRkDYLMkiQswsLnH8wuIUlr7DsF1WC4lp4oc79
amT9vy5Gl1uxlXZ8tZ1Y7aRknYyim1V288puQd3DdK9E9yqafkXbr2r7dV2/
bthqgoxbLZBpAIQHzOqkaxl0rUQ9BNqGto8GWyDbYIAGANo2YG3bB9SlbXjA
vkVk24JHwINte3/7XQ961Ke3bL0+FBrUhQLURZ2OtdO2tluWdtPSapibNVMT
cJjvIdM9q/GuBZPwJsUba/NYk8V0mqEex8oYVq5jRRgrgljuw3IPljuxzIql
JryixcvK4qIkI+SP/zXGkPr4d1YZ6p9ZuMOTPpDIpxUKtkY1a1TPITXHoea4
aS7EdojmrWl4G1r+po6fMPCTAN4iyCJhwS7ETiHmfdPkFVb94lpwoRFebEQk
zfW/lkT/28qmtBNf7SZl3ZS8m1H2cspeXtUv0n1M98uaLVZFu0X9SE0/qBuI
msZB07TdApm32+btjmW7a9l+9xO9y32007ftbCFGth2gad+hLg9sO3A/sG9v
gxzbFDDfpi7vOGBnAIYQOQYU48RnGFf6W/b+lq3ft/V6qNe1dTu2bht1W6jT
tHZsD5zoAfnUt2JqlE31krFaMlSwoYz1QwtyIwvoJFZvYlUUK9ewMoyVAazw
YoUby21YZsKreixVFyXS3Lz4A/D/Nck9weoDqXxapWTrVLNm9RxEvYvmeGlu
gOaGgT/Ni2p5wD9p4KdN/KyFn0eCgkOIXcKyR1gVfgsF5uvBxUZ4qbUmaW/8
9bLwm9HN1W5c1k3Ke2lFL6Ps51T9PL1VpLewBpiUtQPqw1X9ds2wXTdsN407
TdNO27TTMu+0zTsdC2iX1bXuMldEUJx3t2xEhPsuBLxjF+63yea2gzTOnR0H
EcUizW8yDoxMAQcG9sGWfQtqUt++1bP3u7Z+x9ZzTnsc0277lLthbdetbQBS
tTQrZkBfK5mq2FjGhmFlz2NtFmvTWJPCdIKgV29gVQSrQljlw0o3VjqwwoLl
BryqKa7ICksLH4D7b67cn5c9kCmmaRXboJq1qucc6jkPzfHT3CDNjdC8DZq3
qeUldPyUgZ8x8XNWft4mwA5B2SWoAHefuC7+jj202ApL2mvLnY2VTkzaFX8r
lpD1k/J+WrmVVW3l1IMCPShqtkva7bJ2u6rfqRp2aoadhnGX1TTtUpfa5t22
ea9jIQLme3AFg/bgFYOm/i/nln1/y7YP7cC+zxo49iDqnXvbjr0dkHNv17kH
zJ17cIMz7Th24R4EhhAnWCQpfnXgAPwD72zAzfa72b6Ord+y9Zuo20CdurXD
gG9VzM2yuVEC9sYqE/MlrC9iXR7rhuwTmI5jOobpNawOYXUAq71Y5cJKhBUm
LNfhVUVxeem51cvLP5j8Z6WTS7IphXJaq2KbyBf+Z130nJfmBGhOmOau09wY
zYtreUkdL2PgZc38vJVftAlKTkHZLax6RHXffD2w2Fx4xUnIr3Q3pL3Yam9T
1l/4Tjz1RUVaNciqtnP0dkGzg7U7Je1ORb9bNezWDLsNIxBrmvaguu+zOpZ9
WHlb97tov4f2YYViO6Au9e0HW/aDmb91DRwHEN8HJL4PdhgR1gfUlV3n/q5r
H9odkAu0xyJGfHbXBYYM+e8GZkP+uZBvNth3bPfsg6590LEP2vatlq3XsPXq
qEvAW9sVS6tsAfZ1Jugr2FjChiLWF7Auh7UZrE0R9poY1mxgTQTTQUz7sdqN
1XassmClAcvVRelKbkn43N/evZg98x2AP1q4xV2dXJFPqZTTehXbombb1bNu
es5Hc4I0J0JzNwC/hpvQ8lKA38jLmXkFwG8XlJ3CqltU887X/QvNwFIzJGkv
vepaJ/j7MdlWXD5IKLZTX1JmvqTKqnfymp2Cdhdrd0u6vYphr2rYrxv3G6b9
pmm/ZT5ogywHHXTQsR500WEPZDvsg+yHW/ZDiPRD6jL7S26YAradh6Ad5yHQ
dx1SE7AF97uweWnXdUBd3nUfEDcYhebWKFaQE6F+bdu5N3Dubjl3+86dnmOn
69juOLbb9kHLttW09RtD/qhdsbbK1iZYUDLV8bsWQOzrL2Jfm8TaONZGsXYd
a8NYE8AaD6adWG3FKiNW0kWZrLgy/0H4/67krlA2uaqYolXTBtWMVc120LMe
ei5Az4VpzhrNidLcuIab1HLTel7OyMubeUXEL9kFFaew5hbVvfMN4B9cgqrT
iSx3l1/3Rlf7MfkgrhgklTupL6mWv5fO0rt5zV5Ru4d1+2XDfsWwXzUe1E0H
DdNB03zYMh+2LYcd6yGrg45gjY6Oerajvu0IMsB+BOyPYKXoPNpm9O57jjuu
ox330a6LaGLXfTSx5z6kfivE2QhxN8LcdfBih2h/27U/cO1tOff6zr2ec7fr
3O04d9qO7ZZ90LT3G7Y+Q75TsbbL1lbJ0iiZ69hcw6Zh8GMS/Poc1mewPol1
cayLYd061kWwLoi1PqxxYdqGaTNWabFCUVxd/AHkx/8Oiak6v7dyd142KVNO
aVXTJvUMUrOd9KyPngsy5DdoTozmAPmUlpvRc3MmbsHCx0hQtgsrTlHNLW54
Fxr+xVZwqRNe7kRWeuurfelb/k1Cfiep2kmrdzPqPenr2bx2v6g7wLqDsuGg
YjysGQ/rpsOG+agFshy1Qdbjju0YwNuOgfgx9dLAcTxwnAycJ9sg16gdXsnu
uI8v7bqPqUt7nuMwNxbmRsO8KGvPfQT1BuzwHO64ibZdBwP3wZZrv+/a77n2
u869jnO37dhpOQdNx6BhJ9Rrtm7F1imjdgnAWxvYUnsPvLGIDQVsyGJDGhuS
2LCJ9VGsX8P6MNb5sM6NtQ6ssWBaj1WqolzyQaj/pvTuonxSoZzSq6bN6hkb
zXbTs356NkTPrdFzQH1Tw0lqOGktN2vgFUw8bOGXkKBiF1ZdoppH3PAttAh1
CVDvRaT99dWtqHwgezuYUO4k1Xtp9V6G3stpD1bfyBV1h/hrurLxqGI8qpmO
6ubjhvmY1bIcU5fb1pO27aRrO+nZT3q2ExLpJ9SlLcfpAOQkmth2ncLq0XU6
nHsh7nc9pxRr1306EeZtsvY8J9TvgAd77mPSeo53PceQDzueo2334cB9uOU6
7LsPeu79rmuv49prOwH9DoN+i0Hfq9q6ZdS5QA8xX2XQl0nME/S5C/RxbIhh
wwY2hLEhgPVerHNgLcIaI1bTRaU0JeE+9337H0L/s9K7y/JJlfKBQTVlUc84
aLaXZgfo2TA9u07Pxei5uIaT0nAyOm6O0Odji6AM9B2iqktc9yw0fYutwFIn
tNwNr/TWVrc2ZIOoYntTsaO4FiL06b2M5gDo53WH3/3PntLX9eVvGKqm45r5
uG4Gyk3LCfVSy3raQacd22nXftqznfbsZ337GVypOs5InTkbuIi2XWesbfcZ
VJYzqOmeM+Af5icivMSeB/rueU/BAfAHNsCJkbzwG8AG6jM7nuNtzxFxgVhw
0HWBwIK9lmun6dxuOAbEAnu/au+VbZ0SSYAmtrzrQhkbMUkAUx6bstiUxsYE
Nm5i4wY2RrAhiA1erHdhvQ1rjVijKarl+VXBB3Dg91bvShWTtOqBUT2F1NMu
egYcCNLsCD27wTiQ0MxBycnquHkDr2jil8ABm6DqENdd8w3PYsu31A5IOqGV
3oUD28QB5W5Ctae8EUnT+1lwQHdY0B8Vv6b/3ud9qqu4ajqpmU/rltOm5bRl
PWuhsw4669rPurZzitWznwP7c8L+fOAi2nadQ515SL286znfdZ+HeUkifnLP
ey78VnDPcw4bFGvfez6x5z2jfhscgft9cIKYcbrrPbngf7TlOep7GAvc+yQL
XLtN507Dsc3w36raehVbb5gFGCywMvzNFWwqYTPG5jw2Z7E5jc0pbIpjUwyb
1rAxhI1+bHBjgx3rzVirxbSyIBeN158PvZA/M9/+ngTWO3dlikmN6oFZPWWn
iQU+mh2i2Wv0bJSehQxIajkZLSer4+QN/KJJULIKKzbh0IKmZ6HtW+oEJF1i
gXRrTba1Id+OKXc2VcSCFL2vvrme1R7mdEfEAsNxyXhSNp3Q18o1y1nDcta0
nres52103rGdd20PCf+HUPodD7dATtgfuGB/4H64434EoiZCvDT10q7n0a7n
4Z6Xke+h6Nsh2AAPfNB3nzx6vuc7J5vn1M/v+cAPxgbvybb3ZOAZ2nDI2HDQ
ce+33bst127DuV1nnKja+5WLTMAIbBhOBeBEmWJhS5H6CLbksSWDLSlsSWBz
DJvXsSmMTX5s9GCjExusWKfDWmVB+dzV1g/x4bclNwWyu3LFfa1qcuiDm572
0zNhmr1Os2MaqERzSe1cRsfJ6bkFAw+b+SWroGIT1Rzihmuh6Vls+yTdwHIv
JO2HV8GHAePDLvGB3k9pDtKaQ83t9Zz+qGA4LhpOwIeK+axqPtPcqDT/7XeJ
D+ghqwMW/GjX/qgHcjzaAjkfbbkeDUDuxwFeJsjLhHiZHc9jqCyPYYnpfbw3
lO8xVKFH5IoYXHhEXQIrgP6+j4gFFlC/vus7u3DgZMt71Pce9TyHXQ/jgGuv
5QQHdi4c2GIc6JZsbWxrMSbUsaWKLRVsLWFrEVvz2JrD1jS2JrFlE1s2sDmM
zQFs8mCTExut2GDAOnVRvfTcTDx0YPx7AcNKtHxTLL+rVNzXqyat6gcOespD
TweIAzPr9OymZjahnUvrmDTQcwpGXsnMr1gFNZuo7hA3XfMtcMBPHOgzDgzW
5Nsbip2Yajeu2mccOExrD7O6I93daMFwgo0nJdMpccByVgf8f/JK609fAQfA
gI79ETgAmeB4TF3Zcj72c9J+LigD/Lc9T1g7nicwB3if7HqekNb7ZM/7BBZA
vifv0v+dA/9DYgCjPaIzQt93uu07HXiB/nHfe0zouw87Hoj/fYj/pmu37mTo
O7Yqjn7ZDvQ7GFLA1sSoga01jCoYlTDCGBUwymGUwSiJrXFsjWLrGrYEscWH
zS5ssmGDEevpomblg9D/g+WbC/K7KuU9Qp9+4KSnvIT+9Bo9E6XZQD+pnU3r
5rJ6KEIcbOSWzbwh/YZjvulaaHuXOv7lbmClH1odRGRAf2dDsRtT7cXVDH3t
YYbQP84bjvX3YPY6vaB/Xv+TV5p/+j39rVrb9qhje9yxP+46HnvmMt65jI+T
2XI9gRrkfkpdHnie7gzlfbrreQpzq/cpNbHre7rHCGZhaC7v+57s+0GP932P
mXfoDvyPSTY82veDiBmsXf85zMm+M8aM05EZ3qOuB8w4ZMzYY8zYYcwYEDMc
vZK9i4fZYGtgVMO2KraVsQ1jWwHbctiWwbYkRnGMohitYWsIW33Y4sZmGzaZ
sEGDtdIP4MXnVm4uye+olfeMqvuIfuCiH3jpqRDxYprxgn3hxVzewGW84Fet
wqEXLTfxoutf7gXf9UKxs6HcjamHXhwSL3THWf1xXn9cMJ4a78dLprOK+bxq
Pa+jhw30qGl7ZLxVc7Ez7tkMGNFzPuk7wYUt11PqClgBRgwujICW+MACG0gK
PN3zj+yY2Pc/ZYET1O8uvbEOHhBHwIHAoz3/QzBg13e+4zvf9p8OfODAKZMO
x8QB92Hbc9By7zfdew0X44ATHNgqO/rEAXsH21vY3iAmgAP2MraXsL2I7Xls
z2J7GtsS2BbDtg2MwhgFsNWNLXZsNmOjDutWP4ADf7hyUyK/QyvvmYgDk276
gZ84MAUOxOjphIad0rIzutmcfq5o4JQuHKjbxA3HQsu92PFKuv6VflC6FZKB
AzvrxIG9mGo/Th8kNODA0cgBwwk4gE2npslExTJ04JF9JuWYSTtnMqbbja7j
Sdf5hDjgegr8t4C++xlr4HkGV7zeZzueZ9Duep9BGjyD1af/2Z7vGWn9z/b9
z6AWBUYeHASeQDIQGwIkEfb8j3YZG3b859u+s5EHPsiC4673iGTBex7sjTxw
DsrOfokkQgc7WiMb7DXsqGJHCTuK2JHHjix2pLA9ge0xbN/Atgi2BTDyYKsd
W8zYrMN62QfKghsS+W1aedekvm9XT3roST/9ADxYp6eIB/RMSsPOaGfzQw9M
vIpZULWKwIOmc6HNeNBjPBiEZdsR+c66cndDtRdTH4w80B1l9MdZw0necEo8
MJ+VzOeWqaR1KmWbSQ2zoG170rE/ufGl6M0vR3uvOfouYLp1wX/bM+K/M+I/
sesfwYda5H8G9MET8AFc2A88AQsOwIoJCn4+uxd4tBt4BPRJEvjOB/7TLR8Y
cHJhwBExwHPQ9Ow13IwBrm3GgC1igAMMYDxwNLCjTgxwlrGziJ0F7MxhRxo7
ktixiR0b2B7B9iC2eTFyYKuVGGCQf5A54Y9WbizLb2uJA/fs6vtDB8L0A8aB
qThxYCajYxf0nKKBO3SgZhU2Rg4sdb3L4MBWcHXkwJriwgHNQUJ7RLJAf0Ic
MIIDZ8QBC3Ggan0EWcA48KRtJw50v2cH/Le+stl3PwP+W55nkAVgAljAAvrU
xI7vnV1GMCv4n00QCyb2AkCeCNahpDkIEg8IfhZpPkI88D/aCTwkSeA/H/jO
GA9Oe76Tru+o4z1qew8ZD/bJf7Wifr7q2qk4t8uurbKzV3J2sbODnS3sbGJn
HTvBgwp2lbCrgF057MxgZxI749gZxY417AhiuxfbnBhZsUWPjfLn3ot+6Ydl
wXWp/JZWecesvucgHtwP0JPgwQb9YJOeStLTac1MTjcLHpCJwMSvWIR1q6hp
n2/9cw+2iQeK3TXl3oZqn8mCQ+IBZIFh6MFZwQQenJcsD4kHiPHA9rhpZzyA
MuR42nM+67meWe+1mSyAKzTwwfPOtvedHe87rB3fP8J61PePu/6h3oEcgFvg
2SVC/9J+8NlB4CnxIPiE4U8+uyQ5QNLgIeHvP9vyn/X9Q/4nwJ9BQf1i3b1X
c++Sj75c25ABZVe/5OwR+q42djWxq45dNez6PvquDHYlsWsTu6LYuY6dQezw
YbsT2xC26rFJ+QPoXx579JeG88C1VfktjfK2RX3Xob4H9IP0ZISeZOg/SNJT
aQ2boQ/TMK9sElQtgiH9tnOhy9DvD+mH3qWv3o/RkAFD+seEvvH0gv7DkvWh
ZTpVRY/rtscj+g6g/7TrBBH6kAHofnvL8w6QZw2870DsA3/fPw7x+/+RugLY
9wKMgu/sgwLvsAA+9TLYcBB8OhTUo/3g470AWSLt+GGtux0gV3n+c7CAQGfI
f6LtPWp5D5ueg7pnHyyoMi6UXYOSu19y9bCri91t7G5hdwO7a9hdwe4Sdhex
O4/dWexOYXecWOACC0LY6cMOF7YjjAzYrPwADvzRyjWZ7KYWHFDdcajveul7
Qfr+GnFgchT/9ExOy8S/ARzggwMNq7hpm287FrtuSc+z3PdJBwHZdki+E1GC
A/sk/ulD4oDuOKU/ITXIeJY3nRXMjAMk/i8csA8deNpxMg64hg68s+V+B022
B2+4Bm+6BkzsEx/87+wyIjWIRD7cgu+QOsTYAJsjH941Ifh0L/AETNgNEh8Y
Cwj3T/X9Zz0/kwS+4/c88O7XPBceuAclVx+7QYwHnhb2NLCnhj0V7CljD8ae
PPZksSeFPQnsjmL3OnYxHjhd2GEjHlhUH8CDP16+Kpfd0Clv/XMP7g9rUIqe
ztAzec1sUTeHDdyykV8zC8GDlm2+Ax643vVglfEAskC1T7JAw3igZTwwnGYN
Qw/OseW9LKgNs8DxtDX0wMV44CZZAJPA1htO9KAz+a3UwPfOhQWEPXVl9yIB
iJgNYD/Mgyv7F/AZ/k/3go8Z6r8Gg/9//+7trcAZg/90hN933PIdNr2HDP49
wF/xkFVpyb2FPX3s6WJPB3tb2NvA3jr2VrG3jL0Ye/PYm8VeBr8nhj1r2B3C
Lj+DH2Eb4Fd/kCr0+8tXFbLrwN+quu1U3/HSd4P0vQjDP06q0JA/u6jllPTc
CuEPOSBq2RY6jiWG/8qWbxX47zD89xj+B8B/U3vE5MBpGviTHDgvWM6LhP8j
87v87U8gB1pQhZxPAH7XTfj3PIQ/a8tLZgHfO7ap9sBPLGCBA2Qd9A4YAKIu
MuDZXpAIZuIgmQtCzwj7INSgJwz739gNPIaJYDv4iJjwlatb/nNigu+06z8d
meA9bIAJUIe8e1UPMaHsIRfL2NvH3h72Mib4mthXx74q9pWxD2NfHvsYE7wJ
7I1h7zr2hLDbj10u7ETYbsRW+v158HEmB1av6RU3LapbTvVtL30nSN+FBIjS
k3GmCGVodp4GA+YYA3iQAHWSAMSA3sgAUoR2QordiPLCAPr7DDASA3JggBkM
gEn4kWkmXbU9GhngGBowLEJPe+6nPc/Tvufplhf0bADyPbNNtXZgEQoOPGPt
BuAGpF96l/we8A4xQR96yvwDTWLFE4h9IA/PCT6iPrMdePT/fOUaFKKtwHk/
cE5S4F36vqOGD1LggKG/B/RL3m3sJe9bY18P+zrYd0HfX8X+MvZj7Gfo+9LY
l8A+hr43jD1+7HZhlw07jAWr+gPQ/+3VazrFDeuQvhro31mj7zH0L8Kf/v7w
FzJTANBffJf+NqEv3w0T+gcb9EFUczSkn9SfkPA3vUuflB/TNNB/XLMzU4Dj
cctJwp/VdT2hrvTcT4YGkArE0GcNyOrfPt3aedu5A4sdwh+yIPhsl8FPsuAZ
9Qd7oWdfufhh2D/ZCzEtTMQhMgfsED2EaWAQfDgIPAQnRpWI2HDS9h8337Nh
v+rdq3h3Sr5t7BtgH/kQB/s72N/G/ib2D22oMDYUsD+H/WnsTxIbfIwN3gD2
uLELYYcpb1U99w1aZh363P/nZP5K5/ekV/WK61blTaf6lld9O0B8uBOl78Xp
+0n6QYaeyoEPmrmS7j0fIAvapAwt95kytB2QkywIq/aJD+oLH/QnScOFD+az
izL0kClDj2r2R3XHyIc2KUNPOu4nXfcTigU2UC/3vUwm+J4OQP6ng7cd4IVj
prUTIO8SwTXwbvApU2cmhjkA0MGT0BMW2ZjYDT1h4D+6yIPgyACYCsj36wJn
XT/QP2n5GAO8R3UfY4Bvr+LbJQb4QRcGBNo40MSBOg7UcKCCAxgHCjiQw4E0
DiSwP4r9a9gXwj7GALcNO01Zi+oD8P+tlbcM8mtIecOpuumlbwXp2xHC/y7D
fzJLT8M0gIf8DfyaSdiwPM9fthsE/sr9NfXBOn3I8D+O6xj+xtMMqUJkGi5a
z7H1HPhXSBV6VLc/ajqBP1SLtusx9aGO+zEY0PMQ9b1P+r4nW74nA98T1gAu
ci9tB55s+5/YZ5o7AabCfx/pJ99H/zFx5DGpQsFRFboCcwDQhxJEqlDwrBc4
Y/5G7pcBCDHAf5EBPjBgt+LfLRP6AxzYwoEe8QAMCDZxsIGDNRws4yDGwQIO
5nCQMSAQIwb4wQA/9g4NMGcsyg+SAf9q+U1wwKq47lTd8NE3g/StCH17g74d
p++mwAHNFDMRz5a/3wG00LGTibjvXtnySrf9sp0gqUT7EfX+yAEd4wAzD2RM
p+BAnpkHMGTATLJie1izPwQHGo5HLcejtvNR2/Wo43rc8TzueR6zet7H1OW+
73Hf+3jgY0Te8iTxfnkn+GRnVFge74aAMQF9BbaG2JnS82gv+AgefwT8H1Iv
bQ9DP3gOIrNAgJQfivmm3aWm75jQ9x3W/PssYE99rBzYLgUGONhnRL61QuiH
GjhUxaEyDmEcAvpZQj/I0A+sE/p+H6HvsRedpvH/D/VDyP+h5A2t7KpVcc2l
uu5V3wzQNyP0rXfJZwj5GZiBy1puVc+vGb+PvHO575YOvKvbfvkFedWIfAxi
Xw9zwGnKALF/CrUnbz4rWs4g9k3TiQo6r9rP6/aHDcfDlvNhG0TIP+q4H3U8
j3og76M+yPeIqS+Xgfx2gFHwsWOmDjF9zcbE9YcA8G6IUZAAJ9hDD3eDD1m7
oYfEp4c7wfPt4PmAgD+jWP3AGfUrEPzdwHsf+JPg9x/W/QdV/34lsFcO7JaD
O6XgEH8Ph7o4RL6p+M/x53Eoi0NpHErgYAwH13EA8Puxj+AvOE3P/X+uH8L/
c0uv0atvIflVp/KaV30jqAb+N6P0rTh9J0XfB/45LYl8qD0j/mZxy/ou/xXg
PyCRr9gbRb76Pf4Jw0nKcAL8s6bTvOW0QPifMfzPqrbzuuO84ThvOs9bLuD/
sO1+2AF5HvY8oyumCTBgy/9oQK5kB4FHcDX7CGrQNeRg10A7o9iG+kOIgyiG
+i+SndA5UfB8h+hsO3g2CJ6Ra4DAWS94SqI/cNIJnLQDJ60A+bIRGFALHFQD
+5XgXjm4Wwru4NAAh/o4BAZ0KBYOt6iPEhPCVRwu4zDG4TwOZ3E4hUNxHGIc
CIZwwIf97qLXnnca36cDzF/O/470Tav8bcDvUV0Pqm9E6Bsb9K1N+m6S4H+Q
I+EP+DkVPb/+Ln7bYs8xLDyrpPAEFHsh5V5EfbBGH2xoD2Pao7juOKE/ThmO
08aTrOkkbz4pWE6x9dQ0Ey/bAP9ZzX7WcJw1Qa7zNsh9PqLe8573fA/7PuDZ
9z+kLm3B0sVPBGuYCSgk1CXA7mBXSXSHINKD59RloL0TYmgT8mekIp1Rv7ET
OmPonw6Cp1uB0z5Bf9INgo47weN28LgVPGwEDsEAQj/I0A/tlEIM/fAWDvdw
uAPscZh8Rx1HqjhSxpEijgD8DIEfBvhRHFpj4PsJfI895zS+mP34X4Yz/133
N1fesMjecimuelXXgqrra+rr77GHZY9mugDstdyKjl8zCJsmUcs637Ytdh2S
nmt5yyPd9gF7OWFPVj70/obmIKY73NTByucoaThKG4+zpuMcYX9StJ6YpjfL
ttOq/bRmPwX2DecZQ/xK233WAXnOugT9ORjA6vuAap+8c3M+nDWhhFBA/5zQ
P7ezKzvXLLvXLCxADBfIwTOCmnCHTeqzO6HTnSARQR842Qqe9AMnvQCB3iHt
USt41AT0wYN68GD4R2KVEIM+vI3DF+gjHRxp4UgTR+oM/RKDvoAjWRwB9Akc
juHwOg4N495T8NiyDsNz/x3m8g9m/xnJa+bVN13yt7zKqwHVtYj6elRN2Cfo
+2n6AWGvYZe0nIqOVzMIGiZR0zLfRksM+5Utz+qQ/S6z6txbU+9taPaj2v1N
3WFCd5g0HKaNRxnTUc50XLAcF63HpulYGZ0w8+elpvO06Tptu09b7jMWYGd1
vWfUpa7vrO8jbxOAWFAsqB+Bmj0IMAqOWqgkNnZZeCf97Ms3CGrgfwrMT6nP
bocI7e3gCWgQBObH/eAxMO8GjzqBo3bwqB04hHBvBg8awf16cH9Y/KEthXcw
YI9s4UgfRxjsa2281sRrNbxG/soLr2G8lh9hj8RxBLCv4VAQB31FvzvvsT33
f45+CPNfXXjFKH3DKXvTq3g7oLoaUd3YUN/cpO8kh8xpYD6L32MuZpgvdu2S
nnNlC2Zbn4yZbZU7YeVuRL27Tu9FtXubuoO4/iCpB+aHGdNhznyUN4/e571U
sR3X7KCTuvOE1XSdUFea7pMWoPecdkDe0+7/X9mZ+MiR3fe91DND7iqStV4r
sC1YVhStHFmKbNm57MRxYDtOEAQIgiAIAgSBkThx5Hi10mp3udzlklyeQw45
5Nz39N0zfd939VV9zfRM313f6rtnei6Su39AgAQL5Peqmytuk5Zk4Ntv+lW9
rn7vU9/3e6+qa6oCHys6wY+5cTom6lGQCH18EPqY+H88dhj+WEGRQ9G/ZOHO
zV/PEd+jyEdHMuJ+5CMF/aHXE+4Lh/yTA/5xL/y4yz/uhIn3oxZPOmvyZw3+
tM6fSJET+XrSXybklRjBPmS8471yvCPDbpbj9QHvcqIqIy+U43vl+IB3cmhz
PiTzdvycvOXnf3xt/W238kJAfZHXEOwPErorKd21rP5Gjk3rJ4vytL7KhtbZ
unn+GdjDqc1hSNXnNf2o9iimO0roj5OG49TWScZ4umM62zWdDRh/vmR7ZJ4U
Ks7HNZLrMdxPJJLnSYPkfaJo+j7ixlv+j9qkwEedwEfE+yNFN/QRRRiKDk9F
kYLGWf4j7iV6NzAyyTlVPmSOftKPPKH482TskH/MxoRHPf5Rl3/UCT9StMNn
3Odb/GmTP21EGGtEj8XoUS16VI2x/6cpxw8VBJt7leFOtMqJZjlRLydQTjDc
5USJEU8Q7p1yPM1wx+LlaKTEh4ohX85r/RvQ/rW1t1yb7wRU7/Lq92LaS4Lu
ckr3YVZ/Pae/mdffLuhl2lsPibZknq9bF1tsJKWIstaTI8pBSHUYVh9GtP2Y
rh/XHwmGo9TWscz4l3Lm033LWcF2VrSfmScTFcejqvNRzfVIdBPvx5LnseR9
TLAbvidNkv9JixR40g4+4Yj2E26iE3rSDbGAQClRp8lN+AlHFifnMtFb/onC
OVXixg8vGYkxGZpS2h+PuNcOw48OeAJ+1g2fdcJn7fBpmz9t8SdEux45lqLH
iB6JsX4t1q+Ss+OHMvNuOUF6BriAslAtCxUCXk7kZeYEPFOOCzLwaIkPFkLe
0TtG/hTavzL/A8f6W37lO7z6YkzznqD9IKW7mtHRtPEG87b+bsnAxk5xe0Yy
zVEskectK205fvf8G72g8iCsPohoDuWo/Ep6+zhrPNkxneQsJ/uW07z1tGg7
NU7GK46zivOMaIuM9iPJ80jyPqp7Hysavsfc+ab/cZP9qPe4FXzcDj7uhB53
go85cje7xjxMqJ/02FsKDtwYMT0IM7Kc7OKxw0vbzqkioeUULHn1gD87CJ8R
5x5/2g2fdogzQ31CqJuR40b0uB4l2keI9cXYYS1+WIlTEDmQj6J+sSy0y0Kz
LDTKglQWxCFqoVgWnqJOpMtx5u1SNFok1EHvc3fnfDFr+RlFX177sW/z7bDq
Qkx9UdC8n9JeyRJo3Y09Zus7Zf29iuFBbWtGMs49HSiX265VmqJ3fRtd2cGv
RLUHcX0/Yegnt47SW0cZ4/GO6ThnPt6znOStJwXbiXEyVrafVpynVddZzXWm
EN1n3DnJcybjflT3PWr4HjX9j8ZaAerzwUfM04+6wceEminMruQPPyLU/CMK
D6QB1ZcOZPcytvyZ/W7BPpWXOZ8qiDI33gufdMMnnfAJcVa0Isfcy83oUSN6
VI8dSbE+4sT5oBo/qCR65GfuKevWkHVSKifFcrJSTpZl1vtlISfjHrBOlKKR
Ah/Y8Tn+Gtajz+CQnzHxiys/8m68FVJeiKneFTTvpbUfZHUf7uqu7+lv0QBZ
oiiin65tPYQcRRrWxSaxHkTirwY2eyFVj1cfEOyY7jChP0wa+qntfmb7aMd0
tGs+2rMc71uOC7bj7cloyX5SdpxUnCdV56noPhU9p/CcSt4zyXdWJ7Efqc8a
gbNW4IxTtIIU7Nuhs3bwjKFXEHruJQq+ndBZjxRmoaHLy94NnxHZM4rt/Gkv
fErpAX9qu7tPe+eSnvuSzPuY1OGP25HjVuSoFT1qRvuNWH/AWxwgT/QqxFtR
Fjrcq4O5SjnZlFUvJ1FO1mTqpXKyUE4y6mVhRwafLMXiRaIeDozeSfXFxOVn
gX1x7c3g5ttR5QVBfTGtuZTVXt3VXdvTUdBmYaSsf3r0P6D8qnO17V7reDc6
fmU3qOyG1b2Iphdj9j4QDIeprcP0dj9r7O+a+jnL0b71KG892r4dUZQcx9zn
y87jquuEJLpPRM8JEYf3tO49lXyndd+pgl0X0AycchOt4GmbFKL0jCAz4qfc
GIWEriyKMGHZvadDzOETRZc/4c6TmemvbWrPdnevQ5TDRx3+iFO0I0eKVrTP
vdSIHdZjh1L8APEDMXFQY5x7RJlItxXlZIt7RUZdZ9ZOoZyqlVPln3BOMs5l
IUMYirF4Icrngt7RZwCN//WcX175YWDjrejmO4Lq3bT6/azmyq72w33irLtd
HJxb+lXTnMRmfUsN50rLvdb2brT9m52gqhNWdyOablTbi+t6Cf1BcusgvX2Y
NR7umA5z5v6epW+4FS7Yjor2o5L9qOw4KruOOUXVfUwzQM+x6DmG94SkkHwn
xPiE5uCBk+ZTkbtPmLsJ90kndMJYU74bYpGBxLCfUDA/5s51edm672tJ1ru7
Hb5PZPvcF1vRwyYpdtiIHdTjB1K8h0SvluhWhW5F6BLecrLD4BLhJvdKOdUY
zL3LKbGcqpZTlXKqWE49JZxkhEvxJPEoRPi03/k3IHxu6XXX2puRjbcTygtp
1XtZzeVd2bNf0d0p02homKbpR804C/NCnRg7lpuu1ZZnveXbaAeU7ZC6w2s6
UU03pusm9L2koZfa7mW2D3aMBzvmw5zlUHczlLf28/a+oujoc+fKTsJ8VCW5
j2ruI6KsgO9YIfmPFfXAMWN83BwoeDzWChK/Vui4PVCYpZ3QsYJMSsc+Idmp
4aN2+IiWsFefm2jzffO9HVL7sroVPeAUzdgBOTjek/l2xUS3JnSqQqfC+Mru
VRBa7kvlVH1wektGXGEOTpUY39SQb0nIFOPszo65cOC5ewS/mO435fzKG/z6
m+wem5x89c73NFf2NdcL2lsl3WRZP1Ux3Kf5RtU4I5rnYV2s25cbztWGZ63p
22j5N1tBVTusbke0nZi2E9d1BUM3udVNG3sZY2/HdLBrPtDdCu7ZDvK2w7z9
sEhy9suufoXk7tc8pCPRewTvkeg7kkj+ozopwNJG4KhJClIfZ6/QEXeuHTxq
h2TJXDuhI5onM6ahPv2VddgKH1Jk5w+5X2hFDpqXVab7meYVZSPWI8T1eFdK
dJHoiEKHUU62K8n2gHCTRsBUo5ySymmSWE7XWMhIV8rpcin9Kee9UnKnKKQL
cWE/GpUt+HO7WHbslzd+nNx8J6t6f1d9eV99raC5WdJOlnV3K4Z71S2y8UzN
NC9aFiTbUt25UnevNbzrTf9mM6hshVUtRUTT5r4c03US+o6w1UltdzPGbtbc
2zH3NDcDe9Ye42w/yDsOio7DkvOw7DysuA4r8gWJNW9f9PYVoq/PjcPfl3x9
yd+v+/vk6b6iEexzY41AvxlkIkeG+mOtEC1rhfuEsxUiokSW4jfDe9DimZoR
5t1oj3rEFaVxOm18kCK8CinR4c5BaNeEdjXZqiRbbHhLNWUPv1pO12W8GBAu
p6uDkCETpmCRL6X2ikQ4QYQTu5HwJ/JjBF8cKeQp3WChPJtjkVbm/s3lH0TW
fihsvJVRXtxVfbCvulpQXy9pbpW1dyq6qSqbZsyI23OiiTAvSo5lyblad6/X
vRsNv7IZVDXD6lZE24rq2nF9WzC0GWljJ2PqZE1dzU3/rrUrk+7l7b2Co1d0
EoWS60BRcR9wY1XPAbEWvcRL9B1yY/Adws/EKeqUjNcDhw1SkGm8GSSizRDD
2wodkJohRjRM0aAZ7jV5pkaE1G1Eu40YqaOoxzvcecPDJD5cF6+t14RWLdmq
JpsVsq9MeIyihExYonwa9BJpdJa9PMScZpiLyVxRyBYS6VwsNjozHj0yfJ72
1z555o6FX5v/fnD1jcT6m+nNC7vK9/dVVwqqa0X1zbJmsqIl2vdr+ofi1ixM
87Askq8lx4rkWpM8G3XfZj2gbITUTV7TjOiaUV0rrm8JW+3kdjstA1ff8HGK
XUuH+4U9W3ff3s3buzLxXonk6lXcvYqnJ1/mdlDzHdTklGL1AXP4gUQKMI3V
A7Rn6sGDBokRpr/k9lCvKYsR73ETjXC3wXcZ60i3Hu3Q/op1uC9I8Y4UbyPB
JF5b080kaskm8a6mGhUWLIh1fYzBnmCezpBEUilTG9zt4jwBL6WLpdQ+AS8I
2f14MhuJPHe+evTs3vPEf21IXL4DwmtLfxVdfSO9/vbO5sU95eWC8mpRdaOs
vl3R3K1p74m6adEwg+05mBZgWZJsy5JjVXKvS74Nya+sB1X1kKbBaxtRbTOm
byYMRLyVNLbSxrbqujdraRPvXWtnz9bZt3fyjo6i4OySv+kgoezulj3dqoeI
Vb097qWar1fz9igVfT2Q/D2JFOiRyXvcK/Vgr0EKdevBbp3S8CDt1PlOI9Kp
k6Kkdj3WJsBSvKVAosWdF4Vm7dpq9fqKZjammYsR4EqqzsmEx8oZSSYMcjYl
ny/JoEuZaildGdi7mNovJHf3hcxuLDF6ZcQLn939Gca/OmQs33n7W8uvC6tv
ZtffzW2+n9+8UlReK6tuVtR3auopUXsf+ocgSxvnYV6EdfmppYnxphRQSiF1
PaypM8a6RszQSGw1k9tNYrx53ZM1t7KW1o61nbO1c/Y2Md53dAokZ6fo6pRc
NGN1dxQVT5cbr3i7VW+35mNSiP4uRVVfF35Zga5EClK0DXS4l6Rgpx7q1Adp
uCPxpLYUaSukKB1CIdZCnIliUqJJ+40YC41qslFJMro04pVuLJVuLileRHmi
lIEMesC6MvD1eCG1t5/czSVS6Uh09ErcUTc/T/pvD6O1fHX11xe+H1v5QXrt
7d31i/mND4qbH5aVNyqq2zXVXVF9T9RO4zNuXmFuluOH5FNKAZUU0kgEO6Kt
R/V1Bnu7kdxubFxzp03NjKUpw6aW52wtxb6jzY3nne2Cq10kudtjA9YdxrpT
Jfk6NV+HWHeIWqADPxOnoLdjCNJnEWojLIsnv0Za3C8g2kKsSRLjpIaYaBDa
mlCvElomidElEV3lfIQbI5r/7/3LpSFdlmXLGNviU7Zj+6m9nJDNxpPP3XNg
9BeY58m+OvSw7PZvLv5lcuVHO2sX9tffL2xcKW9erypv1pR3RNWUqL4v6h7A
MIvteZgWYVmGbQXONRYnvJsyWTXZWAprpYiOsNZjW/XEVn3tQ2fK1EibGxlz
I2tp7tiau7Zmzt7cs7f2HUQ572wpiq6WouQmWiV20Xe77G1XvG1F1ddW1Hzt
sZq/rRADLYUYbHFjYqiFUEsMM2fyTW5cjDTFaLNGijUUtXhDUU3UFRWhTnNg
ciZxLJLYyMaSiWIG//f9ywznrcUByCy4l2lpMSMWM7VCupxPF//PxbcJUTaR
5oPB5+5TNXolw1OeP1ny9P7H8izv20uvZ1be2lu7WFj/oLRxtbJ5o6a8LZJN
VfdEzbSonYF+DtsLMC3BugL7KvGEe0PyKinuMp5BcqpOiuilqKEel2EKxnrK
VE+Z6xlLPWNt7Fgbu7ZGztbYczT3nM08ydUsOJsFV7PoahLVJqPaLHtaZW+r
4m0pKr4WN1H1N6u+JksDzWqQ5gE1Sr5QCzVr4WaNb5CqkUY12qiRYvUqKV4n
sBWZrcTYKogrN0ZQC2kU0+TNQgYKYqvYXOQVxduLiiKRHS9kxHymup8u76eL
A6cmo8KQ0yjZ0avVnif79K7HXx3MIf5CWHpjZ+Wd/Np7xfXLlY3rtY2bonJy
6FTNA1E3A8M8thdhXpbhrsG5DvcmyKx+lRQkuFoGl2dwpZWrjvi2JJiGcDkF
4eXGsrb6LslO+T1HQ7HvpNlS3tUokNz0tuhuKEqeBo3k3kaF5GtU/I1KoEGj
HSUTlWCjEmKqhutVnrZRjdSJfbReicliP3XQ0nKCOVWQSkmpmGROJbIULUFk
mYhrnqx6nt4UskNtLPKrq7nC7UXqP7cWuHP7mepeurybKrz17pvscp7n767/
s+m+PKQrn1H+yvz/Ti+9ubdyobB6qbx2tbp+Q9y4JW7cEZX3RNW0qHkI3ewz
dFefoauEX4WABiGtFNIzuktXHApiy50XjFLSJKUs9TSzbj1LYvcNZXgnco76
nqO+72TKuwh9wU1Li556ialR8tZLPsbKTz07QKsrwXolVK+E2TJCO86uyojK
ipE/40SQsJZIglQgxxYI7ESemKaQl7HmmWf3WQgguPksE7HMEmuCen5tkd+/
tbC6EMrdnN258SB7/f6P3vmh/H8to/ctHv1vn+e5TgxHrm8OXPv6ztJb+ZWL
pdUPqmsf1tZviuu3xY274uZ9UfUAmhno5mBYGHKl+OogrhsyVxX8agS0COkQ
1mHxiiO6JcW2JCKbMDGuSTN5J22RuJezVilrl3bszLg5EqEdcHUSLQb3ZYJb
8NQL3nrRVy/6n6aBeilIJUohIl8KSyVeKkVkRWnLpRjr73GpmGCi/i7QSF9I
Ik9KYT81JErJOGHdk0VvJvYJ7Pm9LGXF3K2F3Vvz88vphVnP3IxT/l+t0WdJ
jvJUPMdTMfTpt1nmGws/3Ft6p7D8fmX1Sm3tWm3tlrg+Ka5PiZvTUD2EZha6
eWwtwLgkj1ercKzLPJXwqODTMJ8GtZi/bOcNiBgQ20bMiISRAgHjmbJIaauU
IdmoxTuUjO06pJxT2nNSniVj+y5p3y3teyif90rcl/I+Ke+XCgFZQakQkgph
qUjipWJEKpCiUiFGQiHOHJcAdy4vYJ+UxB4phVwKY3vEcYII5j4V8ydhVNAb
xW5G5M7tZGrZdDWdKiWv3YtGM0Rp9K4Uo/nPsnx67ua32NLvLLy5v3ihtHyp
unK1tnpdXLstrt0hjviUo544LjKO5hVY12Anjptwbcoc1fBrMfuBnVmTOG4h
us1QxgmlGYIZSQtSVqStyNhI5Expl+SQdp1SziXtuaUc47dHyUt7XmmAMB+Q
8kFpPyjlQ1I+TP2SJ2flI8hHsR9FPkZLCODEfkJmJzB2uSR1590U8UljfJe8
t8u8t5PBTkbMZsRMupZOlQWhEI3vPDeOj94le8CK8fkd9u7r03+Zmf9xfvFi
eemD2vKHtZUb4uqkuHoX6/eeIpqDfkFGtAzz6lNET63m1eDhJTtZLahniPgB
JSPiJsRlRIKF9i9B4j6fZoyQtSPrwI4TOy7supBzY9eDXS8Uez5q5J4fewHs
B1mPCzGnhLHHYy/CRHSIZYyVimMvgRxZTJETiEoSip0UUcymsZMGpVmaTxKe
8R3GKZsFg5QR0+laKlVNCMVoPPfiZwy9wFOffDqfHPJ6fXfureLCe9Wly7Xl
a+LyTXHlNtaI131sPIDy0665NGop4uVW48F7dtY7dYxXiHhtgd9mvGKmp8gY
r+SQV9qOtAMZJzIuxmvHjR0Z1q6PWu2ntuUCyAWRCyEXxm4YOR65CKNCqCZy
MezGZSWww1jtEKssscoQq7FMGuk0KKW3bLQgZBlZg/xEOiOm0jUhVY2zy5/2
X/ykq+d9NbL0eyz57vQbubl3SguXqotXakvXxeVbWJ7E2tRTZjPMY7oFGBax
/anHNuCgbqjCvffsHjV8Wvh1DBsxC8nMIkYZm5kxS1iQsCJpQ9KOlAMpJ9Iu
hizjRtaDrBdZH7J+RiBAzdsJYifEaFGe+t7LOxHsRGXFkI1jJ85oZRPI0EAw
liZeKYpbqTTOpWnGkmJskmlRSIuJVC2WrITj+eee6Dca+wdsmJU+7Xe/Mf3G
7uyF8vyl2sIVcfEGlm5h6Q5WprA6jfWHjIl6wGSJMTGtwrIOGzFRYupdO2Hx
aODVwkdM9AjKTMJbiBAWE6ImGYsFcSsEGwQHBCdSLqq7m+1XD9JepH3I+JEh
HIoMdbaXsiFkwsjwsiLIRJGNIUOxKBNnhkggLTCRM4lHkvU1IYVkeqixJLMN
oUnK4hQCGy0TaTGWqkWESjCeHz0zPDrbGBAaPU/x20NSOzPvVuYu1eY/FBdu
YPEWFu9gaQor01h7iM1ZqOehXYCeSK3AtALzGqzrmLxgJ1hONdwaeLTw6uE3
IGBAcAshI8Im8CZEzIhaEbUhbkfcgYQTCRcEN7XQA+4LSS9SPqT8SAWQCiIV
QjqEVBgpHqkI0qQoMSVGKcYomUBSYKLWs3CdSCKeQoJQKRJpMCJjAvknwca/
eBqxFKJJkReq/ljhuefDjc5xB3TOPeuf7zx8rzJ7WZy7hvmbWJjEwl0s3sPy
NFZlJCpCsgj9MraWYVxlSG69Y7dtwq5iSFxauPUMic8A/xYCRgRNCJkQNoO3
grchYkfUgZgTMRdibsQ9SHggeCH4IPghBJBgEVpgEVoIQ+CRjMhiEVqgCH1O
iCNBSiAhIE4i18TYK4Xz1HjuXCyNaAqRFPikGEpUvdHCr/8MBp+TGYyeBZAd
8o3p9yozV8TZ65i7hflJzE9h8T6WH2B1BhtzjIVmEbplGFawvYbrb9vNm7Cq
YFPDoYFDC5ceHgM8W/AZ4TMhQDIjZEHYhrAdYQciTkRciFD3UUTJGRMxL2I+
xP2ID0nEGYl4GAkeiQhTnJGIkzdi5I3xWAJRgSlCETjC3EEtj8riJqLMEYxE
CuEkQknRn6gOrqT7mTReHjpCjrJ/9/575QdXxZkbmL2NubuYu4eFaSw9xMos
1uexuQjVEjQr0K3h6tv27U2YlLCoYdHApoVDD6cBri24t+ExwWuC3wy/BQEr
gnYEHQg5EXaBp77BMwIRLyI+RPyIBBBlBKJE4HwsjCiPaERWFNEYImQGRSRO
n6IBaCIsgBdYK0k8YTjHMwDhFLUawSQCArzx6uBqlGfaOvrslEHbR4kMY8VV
8cFNPJzEzBTm7mP+AZZmsDyHtQVsLEG5DPUqPnjbbtjAlhJGFUwamLWw6mA3
wL4F5zacRrhNcJvhtcBnhc8GvwMBJwIuBN0IkTwIexH2IexHOEAtY/GTDyES
Bs/LioCPIkyi/R9ic7lQAsEEQgJTkCLEOWqvghrMjQeS8FOrE3BGK6+NtGj0
PMKg3aORU273b059KE7fxINJPJzCDLX7IRZmsTiPlUWsL2NzBRffsms3oNuE
XoUtNYxamHQw62Hdgm0bdiMcJjjNcFngssJjg8cOrwM+J3wu+N3wexDwIuBD
wI9AAKEggqQQgmwUDdEoOhaMIBhl4hQB2unj/jgCcfgTTApqoyLAXkmM0Xtu
3CfAk4ArBnOo+BsjLRrt44N2f/FZr3/7zqXi1E3x/h1M38PDaczMYG4OCwtY
WsLqCt75sX1zHapNaJTQqWHQYEuHbT1MBpi2YDHCaoLNDLsFDiucNjjtcDvg
dsLtgscNjwc+L3w++PzwkwLwB+EPwReGn8RTeyKsDVF4STF4aD974hin3Tjm
pVZSy7gxdxzOGGwRkfvNkfaM3kVs0L7Rebq8X1+7c0OcuiPeu4fpB3g4g5l5
zC5iYQk/etO+toaNDWxuQqWCRg2dFgYdDAZsbcG4DaMRZhMsZlitsNlgt8Pu
gMMJhwsuN1weuLxwk3xw++EJwBOkerMo5g7DzcuKwMXmki4a5Vxsj7ricD+V
wk171cXGPWccjhis1M7v/FztfOXZUezvTE6Kd++JUw9wfwbT83i4iL/6kX1x
FcvrWN3EuhKbaqg0UOug1UNvgH4bBiO2Tdg2w2SByQqLDRY7LA5YnbC5YHfD
7oHDC4cPDj8UDprijDuDcIbgINHxloP2n50aNmaPwhFllefGWTLhkFtip50W
hYXa81svaM/nKMvaMWjL6Lkn2ZtfuXVXnJzG3Rn8+ev2e4t4uIy5VSysY3kT
qyqsqbGhxaYOKgM0W9BsQ2eE3gSDGVtWbNtgtMPogMkJkwsWircWirdjVi+s
PljZHNZKLTpvDcIakhWGhYeVpxqzdRFMWKkLjlPOHIGRF3UB8XsjtTz3XDt+
abhPfleu/40p8c++b5+cw51F3FvGg1XMrmN+E4sqLKuxqsWaDhsGKLeg3Iba
BI0JWgt0VuhsMNhhcGDLiS0Xtt0weqAweonttg9GP4wBWWy8MIYwZgxj3ER7
w8jix1YYhhBYBf7BSH0nnqvv6LGb3MKvDvI3ZnFrAbeXcXcV99fxYBOzKsyr
saTFsg6rBqxvYcOITROUZigtUFmhtkFjh8YBrRM6F3Ru6DzQk7zQ00GpQkfY
x/UB6ALQB5nGqaYKPfWWMW0QKp/Ivv4fj9Rp/Llajx5B/fZIXp7lvXp9ATdX
MLmGqQ3cV+KhGrNazOuwaMDiFpa3sWrCGsmCdSs2bNiwQ+mAQukkVyhdULqh
8kDphcoLpY8qR6ONmuo/pvRjwyvz/f2fWdNfHvphtIby/PyVq0u4TvPJTdxR
4p4a01o81GPWgLltzBuxaMKiGYsWLFmxZMOKHYoVB/FbdmLFhVUX1tyYWGOe
XnFjySnX6J+PfM/YczUaverh74/k5TnCF69s4LoKtzW4o8OUHve2cH8b00Y8
MGHGjBkLZqyYpUNrxZyNajRnx7wsWvovX7AvPluDrwyZjH7zf/8pn+Sem9mP
jgV/LmP9NP/J4OzV4LNfHSn7rZH8/5B33qf5T7/z14c1/XnLj841vzmS/5/P
5H9Su68Pv2W09F8Mlw9KfWNk7ejc7n99htZrw8+OlvrsNkfnC18byf83ShjS
wRnrvzeydvQ4479+MjiLPbib1beG3zRa6j8Ptzl4Ou23R9aOPi/9P37yzG+6
8rjItjla6t8PS8nRjPvuyNrRcwr/dlgDuSfIYxPb5mipfz3cplx/bjTyj8bN
PxmWlnkPzm/9lF9Y/3BYA5k/97vDGoyW+qfDbcqzAe4fjqwdPQv+j4bblCON
nFM85fZMqd8ZblPeGvdPRtaOnon67rD078mlf29k7ei5mW8Na/AHcunfH9Zg
tNRrw23+kVzqn42sHZ2pf31Y+k/l0n8w3OaLnizGvvnfyKX+cGTt6JUEvzLc
5r+TS/+L4TZHS315WOo/yKX+aGTt6NVNXxrW4D/Jpf94ZO3ouby/Ndz2f5FL
/8mwBqOlzg23+WdyqT99Zi2LHePDbcixiPtXlJ6j3PflHPe5/w8E6b6y\
\>"]]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    RowBox[{"-", 
     StyleBox["\[Theta]", "TR"]}], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->126290438],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"RotationTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], "]"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{"RotationTransform", "[", 
     RowBox[{
      RowBox[{"-", "\[Theta]"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], "]"}]}], "]"}], 
  "//", "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->168670754],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0"},
     {"0", "1", "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{245, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->246463887]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->634911246],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["u", "TI"], ",", 
      StyleBox["v", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["v", "TI"], ",", 
      StyleBox["u", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->214740467],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"RotationTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], "]"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{"RotationTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "}"}]}], "]"}]}], "]"}], 
  "//", "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->118128397],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0"},
     {"0", "1", "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{245, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->110300502]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->270647621],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   StyleBox["\[Theta]", "TR"], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{"-", 
    StyleBox["\[Theta]", "TR"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->215816201],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"RotationTransform", "[", "\[Theta]", "]"}], ",", 
    RowBox[{"RotationTransform", "[", 
     RowBox[{"-", "\[Theta]"}], "]"}]}], "]"}], "//", "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->494140064],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->673474510]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->340099844],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    RowBox[{"-", 
     StyleBox["\[Theta]", "TR"]}], ",", 
    StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->98951137],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"RotationTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", " ", 
    RowBox[{"RotationTransform", "[", 
     RowBox[{
      RowBox[{"-", "\[Theta]"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->723784316],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0"},
     {"0", "1", "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{245, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->450668525]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->58582975],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox["w", "TI"]}], "]"}]], "InlineFormula"],
 " is also given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"-", 
     StyleBox["w", "TI"]}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->130053169],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"RotationTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", " ", 
    RowBox[{"RotationTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"-", 
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "1"}], "}"}]}]}], "]"}]}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->300821124],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0"},
     {"0", "1", "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{245, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->765450446]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->638292573],

Cell["The composition of rotations is a rotation:", "ExampleText",
 CellID->257069901],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"RotationTransform", "[", 
     RowBox[{
      RowBox[{"Pi", "/", "2"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "]"}], ",", 
    RowBox[{"RotationTransform", "[", 
     RowBox[{
      RowBox[{"Pi", "/", "2"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}], "\[Equal]",
   "\[IndentingNewLine]", 
  RowBox[{"RotationTransform", "[", 
   RowBox[{
    RowBox[{"2", 
     RowBox[{"Pi", "/", "3"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->522362852],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->240303629]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->38252755],

Cell[TextData[{
 "For graphics transformation use ",
 Cell[BoxData[
  ButtonBox["Rotate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Rotate"]], "InlineFormula"],
 ": "
}], "ExampleText",
 CellID->343852592],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Graphics", "[", 
    RowBox[{"Rotate", "[", 
     RowBox[{
      RowBox[{"Rectangle", "[", "]"}], ",", 
      RowBox[{"Pi", "/", "6"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"Graphics", "[", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"Rectangle", "[", "]"}], ",", 
      RowBox[{"RotationTransform", "[", 
       RowBox[{"Pi", "/", "6"}], "]"}]}], "]"}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->149542733],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   GraphicsBox[
    GeometricTransformationBox[
     RectangleBox[{0, 0}], {{{0.8660254037844386, -0.5}, {0.5, 
      0.8660254037844386}}, {0, 0}}]], ",", 
   GraphicsBox[
    GeometricTransformationBox[RectangleBox[{0, 0}], 
     NCache[{{{Rational[1, 2] 3^Rational[1, 2], Rational[-1, 2]}, {
        Rational[1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {0, 0}}, {{{
       0.8660254037844386, -0.5}, {0.5, 0.8660254037844386}}, {0, 0}}]]]}], 
  "}"}]], "Output",
 ImageSize->{230, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[1]=",
 CellID->435759259]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Graphics3D", "[", 
    RowBox[{"Rotate", "[", 
     RowBox[{
      RowBox[{"Cuboid", "[", "]"}], ",", 
      RowBox[{"Pi", "/", "6"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"Graphics3D", "[", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"Cuboid", "[", "]"}], ",", 
      RowBox[{"RotationTransform", "[", 
       RowBox[{
        RowBox[{"Pi", "/", "6"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}], "]"}]}], 
  "}"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->587218960],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {-0.49999999999999994`,
       0.8660254037844387, 0.}, {-0.49999999999999994`, 0.8660254037844387, 
      1.}, {0.8660254037844387, 0.49999999999999994`, 0.}, {
      0.8660254037844387, 0.49999999999999994`, 1.}, {0.36602540378443876`, 
      1.3660254037844386`, 0.}, {0.36602540378443876`, 1.3660254037844386`, 
      1.}}, Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 
       7}, {1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     Rotate[
      Cuboid[{0, 0, 0}], Rational[1, 6] Pi, {0, 0, 1}, {0, 0, 0}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]], ",", 
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {-0.5, 
      0.8660254037844386, 0.}, {-0.5, 0.8660254037844386, 1.}, {
      0.8660254037844386, 0.5, 0.}, {0.8660254037844386, 0.5, 1.}, {
      0.3660254037844386, 1.3660254037844386`, 0.}, {0.3660254037844386, 
      1.3660254037844386`, 1.}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     GeometricTransformation[
      Cuboid[{0, 0, 0}], {{{Rational[1, 2] 3^Rational[1, 2], 
         Rational[-1, 2], 0}, {
         Rational[1, 2], Rational[1, 2] 3^Rational[1, 2], 0}, {0, 0, 1}}, {0, 
       0, 0}}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]}], "}"}]], "Output",
 ImageSize->{330, 147},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 Graphics3DBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[2]=",
 CellID->190155709]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["The order in which rotations are applied is important: ", "ExampleText",
 CellID->142331819],

Cell[BoxData[{
 RowBox[{
  RowBox[{"rx", " ", "=", " ", 
   RowBox[{"RotationTransform", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ry", " ", "=", " ", 
   RowBox[{"RotationTransform", "[", 
    RowBox[{"\[Phi]", ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->462404956],

Cell["\<\
Compare the result of the two possible orders; the result is not zero:\
\>", "ExampleText",
 CellID->834306151],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Composition", "[", 
    RowBox[{"rx", ",", "ry"}], "]"}], "[", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}], "-", 
  RowBox[{
   RowBox[{"Composition", "[", 
    RowBox[{"ry", ",", "rx"}], "]"}], "[", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->317852842],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"z", " ", 
     RowBox[{"Sin", "[", "\[Phi]", "]"}]}], "-", 
    RowBox[{"z", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}], " ", 
     RowBox[{"Sin", "[", "\[Phi]", "]"}]}], "-", 
    RowBox[{"y", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}], " ", 
     RowBox[{"Sin", "[", "\[Phi]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"z", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "-", 
    RowBox[{"z", " ", 
     RowBox[{"Cos", "[", "\[Phi]", "]"}], " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "+", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}], " ", 
     RowBox[{"Sin", "[", "\[Phi]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"y", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "-", 
    RowBox[{"y", " ", 
     RowBox[{"Cos", "[", "\[Phi]", "]"}], " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "+", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "\[Phi]", "]"}]}], "-", 
    RowBox[{"x", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}], " ", 
     RowBox[{"Sin", "[", "\[Phi]", "]"}]}]}]}], "}"}]], "Output",
 ImageSize->{572, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->493655730]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->72902016],

Cell["Rotating a character:", "ExampleText",
 CellID->48323490],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"Style", "[", 
       RowBox[{
        RowBox[{"Text", "[", "\"\<Q\>\"", "]"}], ",", "87.5"}], "]"}], ",", 
      RowBox[{"RotationTransform", "[", 
       RowBox[{"a", ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", 
      RowBox[{
       RowBox[{"Pi", "/", "5"}], " ", 
       RowBox[{"Range", "[", "10", "]"}]}]}], "}"}]}], "]"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->163047608],

Cell[BoxData[
 GraphicsBox[{GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (1 + 5^Rational[1, 2]), 
        Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
        Rational[1, 4] (1 + 5^Rational[1, 2])}}, {
      Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (-1 - 5^Rational[1, 2])}}, {{{
      0.8090169943749475, -0.5877852522924731}, {0.5877852522924731, 
      0.8090169943749475}}, {0.5877852522924731, 0.19098300562505255`}}]], 
   GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (-1 + 5^Rational[1, 2]), 
        Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
        Rational[1, 4] (-1 + 5^Rational[1, 2])}}, {
      Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (1 - 5^Rational[1, 2])}}, {{{
      0.30901699437494745`, -0.9510565162951535}, {0.9510565162951535, 
      0.30901699437494745`}}, {0.9510565162951535, 0.6909830056250525}}]], 
   GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (1 - 5^Rational[1, 2]), 
        Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
        Rational[1, 4] (1 - 5^Rational[1, 2])}}, {
      Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (-1 + 
         5^Rational[
           1, 2])}}, {{{-0.30901699437494745`, -0.9510565162951535}, {
      0.9510565162951535, -0.30901699437494745`}}, {0.9510565162951535, 
      1.3090169943749475`}}]], GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (-1 - 5^Rational[1, 2]), 
        Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
        Rational[1, 4] (-1 - 5^Rational[1, 2])}}, {
      Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (1 + 
         5^Rational[1, 2])}}, {{{-0.8090169943749475, -0.5877852522924731}, {
      0.5877852522924731, -0.8090169943749475}}, {0.5877852522924731, 
      1.8090169943749475`}}]], GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], {{{-1, 0}, {0, -1}}, {0, 2}}], 
   GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (-1 - 5^Rational[1, 2]), 
        Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2],
         Rational[1, 4] (-1 - 5^Rational[1, 2])}}, {
      Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (1 + 5^Rational[1, 2])}}, {{{-0.8090169943749475, 
      0.5877852522924731}, {-0.5877852522924731, -0.8090169943749475}}, \
{-0.5877852522924731, 1.8090169943749475`}}]], GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (1 - 5^Rational[1, 2]), 
        Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2],
         Rational[1, 4] (1 - 5^Rational[1, 2])}}, {
      Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (-1 + 5^Rational[1, 2])}}, {{{-0.30901699437494745`,
       0.9510565162951535}, {-0.9510565162951535, -0.30901699437494745`}}, \
{-0.9510565162951535, 1.3090169943749475`}}]], GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (-1 + 5^Rational[1, 2]), 
        Rational[1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2],
         Rational[1, 4] (-1 + 5^Rational[1, 2])}}, {
      Rational[-1, 2] (Rational[1, 2] (5 + 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (1 - 5^Rational[1, 2])}}, {{{0.30901699437494745`, 
      0.9510565162951535}, {-0.9510565162951535, 
      0.30901699437494745`}}, {-0.9510565162951535, 0.6909830056250525}}]], 
   GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], 
    NCache[{{{
       Rational[1, 4] (1 + 5^Rational[1, 2]), 
        Rational[1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^
         Rational[1, 2]}, {
       Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2],
         Rational[1, 4] (1 + 5^Rational[1, 2])}}, {
      Rational[-1, 2] (Rational[1, 2] (5 - 5^Rational[1, 2]))^Rational[1, 2], 
       1 + Rational[1, 4] (-1 - 5^Rational[1, 2])}}, {{{0.8090169943749475, 
      0.5877852522924731}, {-0.5877852522924731, 
      0.8090169943749475}}, {-0.5877852522924731, 0.19098300562505255`}}]], 
   GeometricTransformationBox[
    StyleBox[InsetBox["\<\"Q\"\>", {0, 0}],
     StripOnInput->False,
     FontSize->87.5], {{{1, 0}, {0, 1}}, {0, 0}}]}]], "Output",
 ImageSize->{184, 174},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->177592910]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Rotate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Rotate"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TransformationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TransformationMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TransformationFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TransformationFunction"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TranslationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TranslationTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AffineTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AffineTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ReflectionTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ReflectionTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ScalingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ScalingTransform"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with RotationTransform",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
RotationTransform"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->42546333],

Cell[TextData[ButtonBox["Computational Geometry",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ComputationalGeometry"]], "MoreAbout",
 CellID->8219956],

Cell[TextData[ButtonBox["Geometric Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeometricTransforms"]], "MoreAbout",
 CellID->11059321],

Cell[TextData[ButtonBox["Graphics Transformations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GraphicsTransformations"]], "MoreAbout",
 CellID->11332238],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->270393329]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"RotationTransform - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 18, 20.1008414}", "context" -> "System`", 
    "keywords" -> {
     "affine transform", "geometric transformation", "isometric mapping", 
      "proper orthogonal transformation", "rigid motion", "rotating", 
      "spinning"}, "index" -> True, "label" -> "Built-in Mathematica Symbol", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "RotationTransform[\\[Theta]] gives a TransformationFunction that \
represents a rotation in 2D by \\[Theta] radians about the origin. \
RotationTransform[\\[Theta], p] gives a 2D rotation about the 2D point p. \
RotationTransform[\\[Theta], w] gives a 3D rotation around the direction of \
the 3D vector w. RotationTransform[\\[Theta], w, p] gives a 3D rotation \
around the axis w anchored at the point p. RotationTransform[{u, v}] gives a \
rotation about the origin that transforms the vector u to the direction of \
the vector v. RotationTransform[{u, v}, p] gives a rotation about the point p \
that transforms u to the direction of v. RotationTransform[\\[Theta], {u, v}, \
...] gives a rotation by \\[Theta] radians in the hyperplane spanned by u and \
v.", "synonyms" -> {"rotation transform"}, "title" -> "RotationTransform", 
    "type" -> "Symbol", "uri" -> "ref/RotationTransform"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[14681, 468, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->246665734]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 190604, 4564}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2659, 57, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3282, 84, 1582, 37, 70, "ObjectNameGrid"],
Cell[4867, 123, 4397, 137, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[9301, 265, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[9779, 282, 384, 12, 70, "Notes",
 CellID->1067943069],
Cell[10166, 296, 388, 14, 70, "Notes",
 CellID->17745],
Cell[10557, 312, 546, 18, 70, "Notes",
 CellID->699151593],
Cell[11106, 332, 680, 25, 70, "Notes",
 CellID->21143365],
Cell[11789, 359, 662, 23, 70, "Notes",
 CellID->701125705],
Cell[12454, 384, 1059, 39, 70, "Notes",
 CellID->268040301],
Cell[13516, 425, 1128, 38, 70, "Notes",
 CellID->524887511]
}, Closed]],
Cell[CellGroupData[{
Cell[14681, 468, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->246665734],
Cell[CellGroupData[{
Cell[15066, 482, 148, 5, 70, "ExampleSection",
 CellID->380951877],
Cell[15217, 489, 164, 6, 70, "ExampleText",
 CellID->281592028],
Cell[CellGroupData[{
Cell[15406, 499, 142, 4, 28, "Input",
 CellID->82274509],
Cell[15551, 505, 591, 19, 64, "Output",
 CellID->47614851]
}, Open  ]],
Cell[16157, 527, 59, 1, 70, "ExampleText",
 CellID->643021637],
Cell[CellGroupData[{
Cell[16241, 532, 142, 5, 28, "Input",
 CellID->1813400],
Cell[16386, 539, 512, 17, 36, "Output",
 CellID->671462656]
}, Open  ]],
Cell[16913, 559, 123, 3, 70, "ExampleDelimiter",
 CellID->3169310],
Cell[17039, 564, 144, 6, 70, "ExampleText",
 CellID->228379697],
Cell[CellGroupData[{
Cell[17208, 574, 227, 7, 70, "Input",
 CellID->59267923],
Cell[17438, 583, 634, 20, 80, "Output",
 CellID->503705646]
}, Open  ]],
Cell[CellGroupData[{
Cell[18109, 608, 153, 5, 70, "Input",
 CellID->17132837],
Cell[18265, 615, 522, 17, 36, "Output",
 CellID->318570365]
}, Open  ]],
Cell[18802, 635, 125, 3, 70, "ExampleDelimiter",
 CellID->169950256],
Cell[18930, 640, 102, 3, 70, "ExampleText",
 CellID->19862764],
Cell[CellGroupData[{
Cell[19057, 647, 354, 10, 70, "Input",
 CellID->296133109],
Cell[19414, 659, 497, 12, 170, "Output",
 CellID->144644423]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[19960, 677, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[20184, 686, 306, 12, 70, "ExampleText",
 CellID->133155009],
Cell[CellGroupData[{
Cell[20515, 702, 199, 6, 70, "Input",
 CellID->343484228],
Cell[20717, 710, 956, 29, 64, "Output",
 CellID->221335387]
}, Open  ]],
Cell[21688, 742, 125, 3, 70, "ExampleDelimiter",
 CellID->562451026],
Cell[21816, 747, 417, 16, 70, "ExampleText",
 CellID->165503528],
Cell[CellGroupData[{
Cell[22258, 767, 274, 8, 70, "Input",
 CellID->198428110],
Cell[22535, 777, 998, 30, 80, "Output",
 CellID->665000003]
}, Open  ]],
Cell[23548, 810, 125, 3, 70, "ExampleDelimiter",
 CellID->258442260],
Cell[23676, 815, 266, 11, 70, "ExampleText",
 CellID->239391552],
Cell[CellGroupData[{
Cell[23967, 830, 298, 10, 70, "Input",
 CellID->159820115],
Cell[24268, 842, 592, 19, 64, "Output",
 CellID->698394390]
}, Open  ]],
Cell[24875, 864, 124, 3, 70, "ExampleDelimiter",
 CellID->51855344],
Cell[25002, 869, 266, 11, 70, "ExampleText",
 CellID->119198238],
Cell[CellGroupData[{
Cell[25293, 884, 318, 10, 70, "Input",
 CellID->112570362],
Cell[25614, 896, 634, 20, 80, "Output",
 CellID->178717500]
}, Open  ]],
Cell[26263, 919, 124, 3, 70, "ExampleDelimiter",
 CellID->57979753],
Cell[26390, 924, 266, 11, 70, "ExampleText",
 CellID->298712964],
Cell[CellGroupData[{
Cell[26681, 939, 340, 11, 70, "Input",
 CellID->40926884],
Cell[27024, 952, 684, 21, 94, "Output",
 CellID->2852178]
}, Open  ]],
Cell[27723, 976, 123, 3, 70, "ExampleDelimiter",
 CellID->2705599],
Cell[27849, 981, 398, 15, 70, "ExampleText",
 CellID->95861331],
Cell[CellGroupData[{
Cell[28272, 1000, 373, 13, 70, "Input",
 CellID->12257005],
Cell[28648, 1015, 2159, 72, 159, "Output",
 CellID->777084618]
}, Open  ]],
Cell[30822, 1090, 231, 9, 70, "ExampleText",
 CellID->82662020],
Cell[CellGroupData[{
Cell[31078, 1103, 174, 6, 70, "Input",
 CellID->432986673],
Cell[31255, 1111, 962, 33, 55, "Output",
 CellID->211910682]
}, Open  ]],
Cell[32232, 1147, 124, 3, 70, "ExampleDelimiter",
 CellID->58806993],
Cell[32359, 1152, 80, 1, 70, "ExampleText",
 CellID->479507293],
Cell[32442, 1155, 620, 20, 70, "Input",
 CellID->419323043],
Cell[CellGroupData[{
Cell[33087, 1179, 624, 21, 70, "Input",
 CellID->150504143],
Cell[33714, 1202, 955, 20, 187, "Output",
 CellID->162972915]
}, Open  ]],
Cell[34684, 1225, 125, 3, 70, "ExampleDelimiter",
 CellID->337851088],
Cell[34812, 1230, 80, 1, 70, "ExampleText",
 CellID->137131831],
Cell[34895, 1233, 637, 20, 70, "Input",
 CellID->367829949],
Cell[CellGroupData[{
Cell[35557, 1257, 796, 25, 70, "Input",
 CellID->167437831],
Cell[36356, 1284, 4639, 89, 189, 1541, 34, "CachedBoxData", "BoxData", \
"Output",
 CellID->198211124]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[41044, 1379, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[CellGroupData[{
Cell[41296, 1390, 234, 7, 70, "ExampleSubsection",
 CellID->279776909],
Cell[41533, 1399, 254, 9, 70, "ExampleText",
 CellID->124045928],
Cell[41790, 1410, 448, 15, 70, "Input",
 CellID->130194797],
Cell[42241, 1427, 456, 18, 70, "ExampleText",
 CellID->217142144],
Cell[42700, 1447, 404, 15, 70, "Input",
 CellID->153233183],
Cell[CellGroupData[{
Cell[43129, 1466, 191, 6, 70, "Input",
 CellID->351369032],
Cell[43323, 1474, 2097, 77, 132, "Output",
 CellID->14113611]
}, Open  ]],
Cell[45435, 1554, 71, 1, 70, "ExampleText",
 CellID->55616846],
Cell[CellGroupData[{
Cell[45531, 1559, 440, 14, 70, "Input",
 CellID->2372651],
Cell[45974, 1575, 3763, 66, 70, "Output",
 Evaluatable->False,
 CellID->107950185]
}, Open  ]],
Cell[49752, 1644, 103, 2, 70, "ExampleText",
 CellID->89852326],
Cell[CellGroupData[{
Cell[49880, 1650, 539, 15, 70, "Input",
 CellID->209880432],
Cell[50422, 1667, 38492, 643, 227, 8041, 139, "CachedBoxData", "BoxData", \
"Output",
 CellID->108944870]
}, Open  ]],
Cell[88929, 2313, 219, 8, 70, "ExampleText",
 CellID->530992566],
Cell[CellGroupData[{
Cell[89173, 2325, 763, 22, 70, "Input",
 CellID->104083723],
Cell[89939, 2349, 65919, 1165, 216, 31148, 591, "CachedBoxData", "BoxData", \
"Output",
 CellID->6160808]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[155919, 3521, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[156160, 3530, 743, 28, 70, "ExampleText",
 CellID->126290438],
Cell[CellGroupData[{
Cell[156928, 3562, 759, 24, 70, "Input",
 CellID->168670754],
Cell[157690, 3588, 447, 15, 80, "Output",
 CellID->246463887]
}, Open  ]],
Cell[158152, 3606, 125, 3, 70, "ExampleDelimiter",
 CellID->634911246],
Cell[158280, 3611, 722, 27, 70, "ExampleText",
 CellID->214740467],
Cell[CellGroupData[{
Cell[159027, 3642, 737, 23, 70, "Input",
 CellID->118128397],
Cell[159767, 3667, 447, 15, 80, "Output",
 CellID->110300502]
}, Open  ]],
Cell[160229, 3685, 125, 3, 70, "ExampleDelimiter",
 CellID->270647621],
Cell[160357, 3690, 507, 18, 70, "ExampleText",
 CellID->215816201],
Cell[CellGroupData[{
Cell[160889, 3712, 289, 8, 70, "Input",
 CellID->494140064],
Cell[161181, 3722, 405, 14, 64, "Output",
 CellID->673474510]
}, Open  ]],
Cell[161601, 3739, 125, 3, 70, "ExampleDelimiter",
 CellID->340099844],
Cell[161729, 3744, 599, 22, 70, "ExampleText",
 CellID->98951137],
Cell[CellGroupData[{
Cell[162353, 3770, 480, 15, 70, "Input",
 CellID->723784316],
Cell[162836, 3787, 447, 15, 80, "Output",
 CellID->450668525]
}, Open  ]],
Cell[163298, 3805, 124, 3, 70, "ExampleDelimiter",
 CellID->58582975],
Cell[163425, 3810, 605, 22, 70, "ExampleText",
 CellID->130053169],
Cell[CellGroupData[{
Cell[164055, 3836, 482, 15, 70, "Input",
 CellID->300821124],
Cell[164540, 3853, 447, 15, 80, "Output",
 CellID->765450446]
}, Open  ]],
Cell[165002, 3871, 125, 3, 70, "ExampleDelimiter",
 CellID->638292573],
Cell[165130, 3876, 86, 1, 70, "ExampleText",
 CellID->257069901],
Cell[CellGroupData[{
Cell[165241, 3881, 715, 23, 70, "Input",
 CellID->522362852],
Cell[165959, 3906, 161, 5, 36, "Output",
 CellID->240303629]
}, Open  ]],
Cell[166135, 3914, 124, 3, 70, "ExampleDelimiter",
 CellID->38252755],
Cell[166262, 3919, 212, 8, 70, "ExampleText",
 CellID->343852592],
Cell[CellGroupData[{
Cell[166499, 3931, 566, 17, 70, "Input",
 CellID->149542733],
Cell[167068, 3950, 687, 18, 123, "Output",
 CellID->435759259]
}, Open  ]],
Cell[CellGroupData[{
Cell[167792, 3973, 750, 23, 70, "Input",
 CellID->587218960],
Cell[168545, 3998, 6262, 115, 168, "Output",
 CellID->190155709]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[174856, 4119, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[175090, 4128, 98, 1, 70, "ExampleText",
 CellID->142331819],
Cell[175191, 4131, 463, 14, 70, "Input",
 CellID->462404956],
Cell[175657, 4147, 121, 3, 70, "ExampleText",
 CellID->834306151],
Cell[CellGroupData[{
Cell[175803, 4154, 398, 13, 70, "Input",
 CellID->317852842],
Cell[176204, 4169, 1260, 36, 54, "Output",
 CellID->493655730]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[177513, 4211, 227, 7, 70, "ExampleSection",
 CellID->72902016],
Cell[177743, 4220, 63, 1, 70, "ExampleText",
 CellID->48323490],
Cell[CellGroupData[{
Cell[177831, 4225, 632, 20, 70, "Input",
 CellID->163047608],
Cell[178466, 4247, 6030, 124, 195, "Output",
 CellID->177592910]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[184557, 4378, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[184876, 4389, 2061, 71, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[186974, 4465, 305, 8, 70, "RelatedLinksSection"],
Cell[187282, 4475, 376, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[187695, 4492, 318, 9, 70, "MoreAboutSection",
 CellID->42546333],
Cell[188016, 4503, 153, 3, 70, "MoreAbout",
 CellID->8219956],
Cell[188172, 4508, 150, 3, 70, "MoreAbout",
 CellID->11059321],
Cell[188325, 4513, 158, 3, 70, "MoreAbout",
 CellID->11332238],
Cell[188486, 4518, 179, 3, 70, "MoreAbout",
 CellID->270393329]
}, Open  ]],
Cell[188680, 4524, 27, 0, 70, "History"],
Cell[188710, 4526, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

