(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     77775,       2748]
NotebookOptionsPosition[     63190,       2249]
NotebookOutlinePosition[     66402,       2332]
CellTagsIndexPosition[     66081,       2322]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Power Series" :> 
          Documentation`HelpLookup["paclet:tutorial/PowerSeries"], 
          "Making Power Series Expansions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/MakingPowerSeriesExpansions"], 
          "Operations on Power Series" :> 
          Documentation`HelpLookup["paclet:tutorial/OperationsOnPowerSeries"],
           "The Representation of Power Series" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/TheRepresentationOfPowerSeries"], 
          "Converting Power Series to Normal Expressions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ConvertingPowerSeriesToNormalExpressions"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Power Series\"\>", 
       2->"\<\"Making Power Series Expansions\"\>", 
       3->"\<\"Operations on Power Series\"\>", 
       4->"\<\"The Representation of Power Series\"\>", 
       5->"\<\"Converting Power Series to Normal Expressions\"\>"}, \
"\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Coefficient" :> 
          Documentation`HelpLookup["paclet:ref/Coefficient"], "Series" :> 
          Documentation`HelpLookup["paclet:ref/Series"], "Normal" :> 
          Documentation`HelpLookup["paclet:ref/Normal"], "CoefficientList" :> 
          Documentation`HelpLookup["paclet:ref/CoefficientList"], 
          "SeriesData" :> Documentation`HelpLookup["paclet:ref/SeriesData"], 
          "CoefficientArrays" :> 
          Documentation`HelpLookup["paclet:ref/CoefficientArrays"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Coefficient\"\>", 
       2->"\<\"Series\"\>", 3->"\<\"Normal\"\>", 4->"\<\"CoefficientList\"\>",
        5->"\<\"SeriesData\"\>", 
       6->"\<\"CoefficientArrays\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Mathematics" :> 
          Documentation`HelpLookup["paclet:guide/DiscreteMathematics"], 
          "Series Expansions" :> 
          Documentation`HelpLookup["paclet:guide/SeriesExpansions"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Discrete Mathematics\"\>", 
       2->"\<\"Series Expansions\"\>", 
       3->"\<\"New in 6.0: Symbolic Computation\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SeriesCoefficient", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SeriesCoefficient",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SeriesCoefficient"], "[", 
       RowBox[{
        StyleBox["series", "TI"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds the coefficient of the ",
     Cell[BoxData[
      StyleBox[
       RowBox[{
        StyleBox["n", "TI"], "\[Null]"}]]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " order term in a power series in the form generated by ",
     Cell[BoxData[
      ButtonBox["Series",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Series"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SeriesCoefficient",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SeriesCoefficient"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["0", "TR"]], ",", 
          StyleBox["n", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds the coefficient of ",
     Cell[BoxData[
      FormBox[
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"x", "-", 
          SubscriptBox["x", "0"]}], ")"}], "n"], TraditionalForm]], 
      "InlineMath"],
     " in the expansion of ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " about the point ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", "=", 
        SubscriptBox["x", "0"]}], TraditionalForm]], "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SeriesCoefficient",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SeriesCoefficient"], "[", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["0", "TR"]], ",", 
          SubscriptBox[
           StyleBox["n", "TI"], 
           StyleBox["x", "TI"]]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["y", "TI"], ",", 
          SubscriptBox[
           StyleBox["y", "TI"], 
           StyleBox["0", "TR"]], ",", 
          SubscriptBox[
           StyleBox["n", "TI"], 
           StyleBox["y", "TI"]]}], "}"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]finds a coefficient in a multivariate series. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->20440]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "In the form ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SeriesCoefficient",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SeriesCoefficient"], "[", 
   RowBox[{
    StyleBox["f", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]], ",", 
      StyleBox["n", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
 " the order ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " can be symbolic. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->686864027],

Cell[TextData[{
 "For explicit ",
 Cell[BoxData[
  ButtonBox["SeriesData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SeriesData"]], "InlineFormula"],
 " objects, the form ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SeriesCoefficient",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SeriesCoefficient"], "[", 
   RowBox[{
    StyleBox["series", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["n", "TI"], 
       StyleBox["x", "TI"]], ",", 
      SubscriptBox[
       StyleBox["n", "TI"], 
       StyleBox["y", "TI"]], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
 " can also be used."
}], "Notes",
 CellID->32326293]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->665909816],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->13551923],

Cell["Find the coefficient for a term in a series:", "ExampleText",
 CellID->13815],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"Sin", "[", "x", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19099],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "-", 
   FractionBox[
    SuperscriptBox["x", "4"], "8"], "-", 
   FractionBox[
    SuperscriptBox["x", "5"], "15"], "-", 
   FractionBox[
    SuperscriptBox["x", "6"], "240"], "+", 
   FractionBox[
    SuperscriptBox["x", "7"], "90"], "+", 
   FractionBox[
    RowBox[{"31", " ", 
     SuperscriptBox["x", "8"]}], "5760"], "+", 
   FractionBox[
    SuperscriptBox["x", "9"], "5670"], "-", 
   FractionBox[
    RowBox[{"2951", " ", 
     SuperscriptBox["x", "10"]}], "3628800"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "11"],
    SeriesData[$CellContext`x, 0, {}, 0, 11, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, 1, 
    Rational[1, 2], 0, 
    Rational[-1, 8], 
    Rational[-1, 15], 
    Rational[-1, 240], 
    Rational[1, 90], 
    Rational[31, 5760], 
    Rational[1, 5670], 
    Rational[-2951, 3628800]}, 0, 11, 1],
  Editable->False]], "Output",
 ImageSize->{401, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->900003345]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{"%", ",", "8"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->6713],

Cell[BoxData[
 FractionBox["31", "5760"]], "Output",
 ImageSize->{37, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1628893594]
}, Open  ]],

Cell["Find the coefficient of the general term in a series:", "ExampleText",
 CellID->291740222],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->720447568],

Cell[BoxData[
 FractionBox[
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"-", "1"}], ")"}], "n"], 
  RowBox[{"n", "!"}]]], "Output",
 ImageSize->{43, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->693102402]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{"%", ",", " ", 
   RowBox[{"{", 
    RowBox[{"n", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->313199119],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"-", "1"}], ",", 
   FractionBox["1", "2"], ",", 
   RowBox[{"-", 
    FractionBox["1", "6"]}], ",", 
   FractionBox["1", "24"], ",", 
   RowBox[{"-", 
    FractionBox["1", "120"]}]}], "}"}]], "Output",
 ImageSize->{177, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1020769086]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", "x"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", " ", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->293812152],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "-", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "-", 
   FractionBox[
    SuperscriptBox["x", "3"], "6"], "+", 
   FractionBox[
    SuperscriptBox["x", "4"], "24"], "-", 
   FractionBox[
    SuperscriptBox["x", "5"], "120"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "6"],
    SeriesData[$CellContext`x, 0, {}, 0, 6, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, -1, 
    Rational[1, 2], 
    Rational[-1, 6], 
    Rational[1, 24], 
    Rational[-1, 120]}, 0, 6, 1],
  Editable->False]], "Output",
 ImageSize->{204, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1966656166]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->19835],

Cell["Compute a series coefficient:", "ExampleText",
 CellID->747820033],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", 
      RowBox[{
       RowBox[{"3", "/", "2"}], "x"}], "+", "1"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->36643142],

Cell[BoxData[
 RowBox[{"ChebyshevU", "[", 
  RowBox[{"n", ",", 
   FractionBox["3", "4"]}], "]"}]], "Output",
 ImageSize->{118, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1669370985]
}, Open  ]],

Cell["Plot the resulting sequence:", "ExampleText",
 CellID->69356038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{"%", ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "0", ",", "30"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15686472],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJzt0z9LQmEUBvBLQ7gEDndwcLCLiJiI/67/rnrv0hdoDFqMZmmtxcnP0BLc
qdYmhxySpmgsogiaCovAKBQVWvJ9z3MDn69QB14OPw48vHA466321t6KYRg7
i6e61Ke73A1P2hwOwRM4DD/AJjy8kB4Rd6ZwVDwIHBN7r7AF3yEvLj4awwmx
GfwnKe4G/02J/RM4LR59ID+jPRjO4Ky2vzuBczI/f4Tz2tn+C/IK2qYzhYva
q9+Bbe33gze4pL19eIO8svKgtzGGK8p+35rDVeWzVugLril3usc92FFO1m9H
yK8rdzevZnBD2b6+DPbTVE6ctp9gd+HO2v3+M/J+9xyUt8wQOUw2yRFylBwj
W+Q4OUFOklPkNDlDzpJz5Dy5QC6SbXKJXCZXyFVyjeyQ6+QGuUl2yf/3red/
4L5/AOh9tfg=
   "], {{{}, {}, {}, 
     {Hue[0.67, 0.6, 0.6], Opacity[0.2], LineBox[{36, 5}], LineBox[{37, 6}], 
      LineBox[{38, 7}], LineBox[{39, 8}], LineBox[{45, 14}], 
      LineBox[{46, 15}], LineBox[{47, 16}], LineBox[{48, 17}], 
      LineBox[{53, 22}], LineBox[{54, 23}], LineBox[{55, 24}], 
      LineBox[{56, 25}], LineBox[{57, 26}], LineBox[{62, 31}]}, 
     {Hue[0.67, 0.6, 0.6], Opacity[0.2], LineBox[{32, 1}], LineBox[{33, 2}], 
      LineBox[{34, 3}], LineBox[{35, 4}], LineBox[{40, 9}], LineBox[{41, 10}],
       LineBox[{42, 11}], LineBox[{43, 12}], LineBox[{44, 13}], 
      LineBox[{49, 18}], LineBox[{50, 19}], LineBox[{51, 20}], 
      LineBox[{52, 21}], LineBox[{58, 27}], LineBox[{59, 28}], 
      LineBox[{60, 29}], LineBox[{61, 30}]}}, {{}, 
     {Hue[0.67, 0.6, 0.6], 
      PointBox[{63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 
       78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 
       93}]}, {}}}],
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 31.}, {-1.5084797143936157`, 1.5048828125}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->244296840]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->82926861],

Cell["Rational functions:", "ExampleText",
 CellID->18680779],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"x", "/", 
    RowBox[{"(", 
     RowBox[{"x", "-", "5"}], ")"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->283053155],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   SuperscriptBox["5", 
    RowBox[{"-", "n"}]]}], " ", 
  RowBox[{"UnitStep", "[", 
   RowBox[{
    RowBox[{"-", "1"}], "+", "n"}], "]"}]}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2055740463]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "3"}], "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "3"}], "+", "x"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "2"}], "+", "x"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "1"}], "+", "x"}], ")"}]}], ")"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32826465],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", 
     FractionBox["1", "2"]}], "+", 
    SuperscriptBox["2", 
     RowBox[{"2", "-", "n"}]], "-", 
    FractionBox[
     SuperscriptBox["3", 
      RowBox[{"2", "-", "n"}]], "2"]}], ")"}], " ", 
  RowBox[{"UnitStep", "[", 
   RowBox[{
    RowBox[{"-", "1"}], "+", "n"}], "]"}]}]], "Output",
 ImageSize->{219, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->52868011]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158412130],

Cell["Elementary functions:", "ExampleText",
 CellID->93458657],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Exp", "[", "x", "]"}], "/", 
    RowBox[{"(", 
     RowBox[{"1", "-", "x"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->39330682],

Cell[BoxData[
 FractionBox[
  RowBox[{"\[ExponentialE]", " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "n"}], ")"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{
     RowBox[{"1", "+", "n"}], ",", "1"}], "]"}]}], 
  RowBox[{"Gamma", "[", 
   RowBox[{"2", "+", "n"}], "]"}]]], "Output",
 ImageSize->{156, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1072514496]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"Cos", "[", 
    RowBox[{"Sqrt", "[", "x", "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->132956228],

Cell[BoxData[
 FractionBox[
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"-", "1"}], ")"}], "n"], 
  RowBox[{"Gamma", "[", 
   RowBox[{"1", "+", 
    RowBox[{"2", " ", "n"}]}], "]"}]]], "Output",
 ImageSize->{94, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1920885001]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", 
      RowBox[{"1", "+", 
       RowBox[{"5", 
        RowBox[{"x", "^", "2"}]}]}], "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->137715104],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ImaginaryI]", "n"], " ", 
   SuperscriptBox["5", 
    RowBox[{"n", "/", "2"}]], " ", 
   RowBox[{"KroneckerDelta", "[", 
    RowBox[{"Mod", "[", 
     RowBox[{"n", ",", "2"}], "]"}], "]"}], " ", 
   RowBox[{"UnitStep", "[", 
    RowBox[{
     RowBox[{"-", "2"}], "+", "n"}], "]"}]}], "n"]], "Output",
 ImageSize->{330, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1192546456]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->567838784],

Cell["Special functions:", "ExampleText",
 CellID->182178878],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"PolyLog", "[", 
    RowBox[{
     RowBox[{"-", "k"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->168492540],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["n", "k"], " ", 
  RowBox[{"UnitStep", "[", 
   RowBox[{
    RowBox[{"-", "1"}], "+", "n"}], "]"}]}]], "Output",
 ImageSize->{125, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->663408541]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"BesselI", "[", 
     RowBox[{"3", ",", 
      RowBox[{"2", "x"}]}], "]"}], "/", 
    RowBox[{"x", "^", "7"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23220406],

Cell[BoxData[
 FractionBox[
  RowBox[{"KroneckerDelta", "[", 
   RowBox[{"Mod", "[", 
    RowBox[{"n", ",", "2"}], "]"}], "]"}], 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"6", "+", 
     FractionBox["n", "2"]}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    FractionBox[
     RowBox[{"6", "+", "n"}], "2"], "]"}]}]]], "Output",
 ImageSize->{181, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->172253363]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"HypergeometricPFQ", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4", ",", "5"}], "}"}], ",", 
     RowBox[{"6", "x"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->32253551],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{"5", "+", "n"}]], " ", 
   SuperscriptBox["3", 
    RowBox[{"2", "+", "n"}]], " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "+", "n"}], ")"}], "!"}]}], 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"2", "+", "n"}], ")"}], "!"}], " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"3", "+", "n"}], ")"}], "!"}], " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"4", "+", "n"}], ")"}], "!"}]}]]], "Output",
 ImageSize->{162, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->254315785]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26337],

Cell["Find the coefficients in a multivariate series:", "ExampleText",
 CellID->208098727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"3", "x"}], "+", 
      RowBox[{"y", "^", "2"}]}], ")"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "m"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->58757054],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["3", "m"], " ", 
   RowBox[{"KroneckerDelta", "[", 
    RowBox[{"Mod", "[", 
     RowBox[{"n", ",", "2"}], "]"}], "]"}]}], 
  RowBox[{
   RowBox[{"m", "!"}], " ", 
   RowBox[{
    FractionBox["n", "2"], "!"}]}]]], "Output",
 ImageSize->{197, 38},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->792940965]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"m", "\[Rule]", " ", "3"}], ",", " ", 
    RowBox[{"n", "\[Rule]", " ", "2"}]}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->36712913],

Cell[BoxData[
 FractionBox["9", "2"]], "Output",
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1822769120]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"3", "x"}], "+", 
      RowBox[{"y", "^", "2"}]}], ")"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->263874971],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{485, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->307534597]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{"%", ",", " ", 
   RowBox[{"{", 
    RowBox[{"3", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1350547],

Cell[BoxData[
 FractionBox["9", "2"]], "Output",
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->731144234]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31288],

Cell[TextData[{
 "Find the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["11", "th"], TraditionalForm]]],
 " Fibonacci number from its generating function:"
}], "ExampleText",
 CellID->19852],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"1", "-", "t", "-", 
      RowBox[{"t", "^", "2"}]}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5475],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", "t", "+", 
   RowBox[{"2", " ", 
    SuperscriptBox["t", "2"]}], "+", 
   RowBox[{"3", " ", 
    SuperscriptBox["t", "3"]}], "+", 
   RowBox[{"5", " ", 
    SuperscriptBox["t", "4"]}], "+", 
   RowBox[{"8", " ", 
    SuperscriptBox["t", "5"]}], "+", 
   RowBox[{"13", " ", 
    SuperscriptBox["t", "6"]}], "+", 
   RowBox[{"21", " ", 
    SuperscriptBox["t", "7"]}], "+", 
   RowBox[{"34", " ", 
    SuperscriptBox["t", "8"]}], "+", 
   RowBox[{"55", " ", 
    SuperscriptBox["t", "9"]}], "+", 
   RowBox[{"89", " ", 
    SuperscriptBox["t", "10"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "t", "]"}], "11"],
    SeriesData[$CellContext`t, 0, {}, 0, 11, 1],
    Editable->False]}],
  SeriesData[$CellContext`t, 0, {1, 1, 2, 3, 5, 8, 13, 21, 34, 55, 89}, 0, 11,
    1],
  Editable->False]], "Output",
 ImageSize->{438, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->439712704]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{"%", ",", "10"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->587],

Cell[BoxData["89"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->760186801]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Fibonacci", "[", "11", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->364824048],

Cell[BoxData["89"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1079348404]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->18571],

Cell["Find a Chebyshev polynomial from its generating function:", \
"ExampleText",
 CellID->5826],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"1", "-", 
      RowBox[{"2", " ", "t", " ", "x"}], " ", "+", 
      RowBox[{"t", "^", "2"}]}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11798],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWF1v0zAUdeI0W7eu3UaFgKcKIRjq2MOEJpAQdC/wAr8AIVRNSNsDYhr7
SeN3Bju57qyTk7j5KA9Uat3re30/j6+dfFneXv74uby9uljOPt0sry+vLn7P
Pv66MVM6Uiq6U0odPlLmf2b+5V/7+WO+Si3sr3ZzmToF+hnQ7wL8r0Cf51Y+
wOxroJ8C/T7A/wb04j+ycmR+YkPuyjjMraKWM6BnzNca/negXwC9C/RW7oXS
xWA/lpmC1MD8DAyZiO+xjHa1lrGk1aOHsh7nt2W9Br07RJ8lRwE7owp9KJeK
3KAnu7HEN5bR0ZivPViXiJzu6I8fD+rX4ke56ppowaqrwYoWiOSzyb2UZLck
h9owZxSJgBmf1iW7qCGlfoTsYE3KdrZb5qUf/7B2+0DHa3ozZt600NMuKp4z
jE1TKcQpR6/bSwx3MalimvGe5O9d1gPtmHp7le1FxHmdnRj0bXl2+K7qYhWj
SirkWM5dB2NZ5uhp6m0DlCC2Gnc3diYwnCDdtI91qyN2J9olK7VjD92sL6FO
hXxndxKw2w1b6PW/xBhiJ7dSPjt7RFtasrBmB29Q5501bLBuu7pZ9upNCHXs
TtT8ZEx68X2T2DskXvt3SXd+HYAcxuDznXd1d1Kn92GNXqZnUpmLNtZCKGA1
9e2gvpisX8ev6VoVrrpj8MqOqU6MEOsfqqt7VkJ0Yabcs0kdbqy+x9TLkeQL
tTIrVg6jaHf7dbGhzyyzmtQBvXPPUbwSTfcv8qNcj0PWgdjCJ/eh8KcZP/fQ
C+w9DB9WzxOxh6cw7gDks0o4/12/sWz0Yz+7R8Qqpw3supq5+7KLw49fyU4t
v9U5CdQCvUEEvem4/hXQz7PibEE9ezJ/bMbIjBYj0UYietsw/1h3FhH2/Spc
Wv5c6le8glqA9DFI10Vn+Q+Af9Zx/Rz47i0e6pnI/BzmCxRew+wU6M8B/jnQ
RwH5l1mxRwrrKvoLJVmrsA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{420, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->244511750]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{"%", ",", "4"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->22045],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"12", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  RowBox[{"16", " ", 
   SuperscriptBox["x", "4"]}]}]], "Output",
 ImageSize->{96, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1498979249]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ChebyshevU", "[", 
  RowBox[{"4", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->14357],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"12", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  RowBox[{"16", " ", 
   SuperscriptBox["x", "4"]}]}]], "Output",
 ImageSize->{96, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1213574986]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14910646],

Cell["Solve a linear difference equation:", "ExampleText",
 CellID->3797937],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ZTransform", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"y", "[", 
      RowBox[{"n", "+", "1"}], "]"}], "-", 
     RowBox[{"5", 
      RowBox[{"y", "[", "n", "]"}]}]}], "\[Equal]", "1"}], ",", "n", ",", 
   "x"}], "]"}]], "Input",
 CellTags->"Ex--22034138112425334325,28",
 CellLabel->"In[1]:=",
 CellID->121364303],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"-", "x"}], " ", 
    RowBox[{"y", "[", "0", "]"}]}], "-", 
   RowBox[{"5", " ", 
    RowBox[{"ZTransform", "[", 
     RowBox[{
      RowBox[{"y", "[", "n", "]"}], ",", "n", ",", "x"}], "]"}]}], "+", 
   RowBox[{"x", " ", 
    RowBox[{"ZTransform", "[", 
     RowBox[{
      RowBox[{"y", "[", "n", "]"}], ",", "n", ",", "x"}], "]"}]}]}], 
  "\[Equal]", 
  FractionBox["x", 
   RowBox[{
    RowBox[{"-", "1"}], "+", "x"}]]}]], "Output",
 ImageSize->{448, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--22034138112425334325,28",
 CellLabel->"Out[1]=",
 CellID->1902594143]
}, Open  ]],

Cell["\<\
Add the initial value equation and solve the algebraic equation for the \
transform:\
\>", "ExampleText",
 CellID->222969517],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"%", ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "2"}]}], "}"}], ",", 
   RowBox[{"ZTransform", "[", 
    RowBox[{
     RowBox[{"y", "[", "n", "]"}], ",", "n", ",", "x"}], "]"}]}], 
  "]"}]], "Input",
 CellTags->"Ex--22034138112425334325,29",
 CellLabel->"In[2]:=",
 CellID->801988935],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"ZTransform", "[", 
     RowBox[{
      RowBox[{"y", "[", "n", "]"}], ",", "n", ",", "x"}], "]"}], "\[Rule]", 
    FractionBox[
     RowBox[{
      RowBox[{"-", "x"}], "+", 
      RowBox[{"2", " ", 
       SuperscriptBox["x", "2"]}]}], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "5"}], "+", "x"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "1"}], "+", "x"}], ")"}]}]]}], "}"}], "}"}]], "Output",
 ImageSize->{301, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--22034138112425334325,29",
 CellLabel->"Out[2]=",
 CellID->868257571]
}, Open  ]],

Cell[TextData[{
 "Find the expression for ",
 Cell[BoxData[
  RowBox[{"y", "[", "n", "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->304050393],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"ZTransform", "[", 
      RowBox[{
       RowBox[{"y", "[", "n", "]"}], ",", "n", ",", "x"}], "]"}], "/.", 
     RowBox[{"First", "[", "%", "]"}]}], "/.", 
    RowBox[{"{", 
     RowBox[{"x", "\[Rule]", " ", 
      RowBox[{"1", "/", "x"}]}], "}"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellTags->"Ex--22034138112425334325,30",
 CellLabel->"In[3]:=",
 CellID->187986390],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "4"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "1"}], "+", 
    RowBox[{"9", " ", 
     SuperscriptBox["5", "n"]}]}], ")"}]}]], "Output",
 ImageSize->{82, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--22034138112425334325,30",
 CellLabel->"Out[3]=",
 CellID->653614432]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "/.", 
  RowBox[{"{", 
   RowBox[{"n", "\[Rule]", " ", "6"}], "}"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1151751],

Cell[BoxData["35156"], "Output",
 ImageSize->{42, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->918904219]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["RSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RSolve"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->384615724],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "[", 
        RowBox[{"n", "+", "1"}], "]"}], "-", 
       RowBox[{"5", 
        RowBox[{"y", "[", "n", "]"}]}]}], "\[Equal]", "1"}], ",", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "2"}]}], "}"}], ",", 
   RowBox[{"y", "[", "n", "]"}], ",", "n"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->18133952],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "n", "]"}], "\[Rule]", 
    RowBox[{
     FractionBox["1", "4"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"9", " ", 
        SuperscriptBox["5", "n"]}]}], ")"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{154, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1468303156]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"y", "[", "n", "]"}], "/.", 
   RowBox[{"%", "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], "/.", 
  RowBox[{"n", "\[Rule]", " ", "6"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->121098329],

Cell[BoxData["35156"], "Output",
 ImageSize->{42, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->1036555215]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12477],

Cell["\<\
When the general coefficient formula exists, the results are equal to the \
truncated series coefficients:\
\>", "ExampleText",
 CellID->19755808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Coefficient", "[", 
    RowBox[{
     RowBox[{"Series", "[", 
      RowBox[{
       RowBox[{"Log", "[", 
        RowBox[{"1", "/", 
         RowBox[{"Sqrt", "[", 
          RowBox[{"1", "+", 
           RowBox[{"5", 
            RowBox[{"x", "^", "2"}]}]}], "]"}]}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}], ",", "x", ",", 
     "n"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->75096379],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", 
   RowBox[{"-", 
    FractionBox["5", "2"]}], ",", "0", ",", 
   FractionBox["25", "4"]}], "}"}]], "Output",
 ImageSize->{121, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->996154041]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{
     RowBox[{"Log", "[", 
      RowBox[{"1", "/", 
       RowBox[{"Sqrt", "[", 
        RowBox[{"1", "+", 
         RowBox[{"5", 
          RowBox[{"x", "^", "2"}]}]}], "]"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->60689533],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", 
   RowBox[{"-", 
    FractionBox["5", "2"]}], ",", "0", ",", 
   FractionBox["25", "4"]}], "}"}]], "Output",
 ImageSize->{121, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->578455075]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->872803412],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["CoefficientList",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CoefficientList"]], "InlineFormula"],
 " finds all coefficients in a series:"
}], "ExampleText",
 CellID->13018],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"Series", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"(", 
      RowBox[{"1", "-", "t", "-", 
       RowBox[{"t", "^", "2"}]}], ")"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3703],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", "t", "+", 
   RowBox[{"2", " ", 
    SuperscriptBox["t", "2"]}], "+", 
   RowBox[{"3", " ", 
    SuperscriptBox["t", "3"]}], "+", 
   RowBox[{"5", " ", 
    SuperscriptBox["t", "4"]}], "+", 
   RowBox[{"8", " ", 
    SuperscriptBox["t", "5"]}], "+", 
   RowBox[{"13", " ", 
    SuperscriptBox["t", "6"]}], "+", 
   RowBox[{"21", " ", 
    SuperscriptBox["t", "7"]}], "+", 
   RowBox[{"34", " ", 
    SuperscriptBox["t", "8"]}], "+", 
   RowBox[{"55", " ", 
    SuperscriptBox["t", "9"]}], "+", 
   RowBox[{"89", " ", 
    SuperscriptBox["t", "10"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "t", "]"}], "11"],
    SeriesData[$CellContext`t, 0, {}, 0, 11, 1],
    Editable->False]}],
  SeriesData[$CellContext`t, 0, {1, 1, 2, 3, 5, 8, 13, 21, 34, 55, 89}, 0, 11,
    1],
  Editable->False]], "Output",
 ImageSize->{438, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->435975556]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{"s", ",", "i"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->17911],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "1", ",", "2", ",", "3", ",", "5", ",", "8", ",", "13", ",", "21",
    ",", "34", ",", "55", ",", "89"}], "}"}]], "Output",
 ImageSize->{242, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->226966128]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CoefficientList", "[", 
  RowBox[{"s", ",", "t"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->15787],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "1", ",", "2", ",", "3", ",", "5", ",", "8", ",", "13", ",", "21",
    ",", "34", ",", "55", ",", "89"}], "}"}]], "Output",
 ImageSize->{242, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->688756224]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->51225179],

Cell[TextData[{
 "In general ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["InverseZTransform",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/InverseZTransform"], "[", 
    RowBox[{
     RowBox[{"F", "[", 
      StyleBox["x", "TI"], "]"}], ",", 
     StyleBox["x", "TI"], ",", 
     StyleBox["n", "TI"]}], "]"}], "=", 
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{
     RowBox[{"F", "[", 
      RowBox[{"1", "/", 
       StyleBox["x", "TI"]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{
       StyleBox["x", "TI"], ",", "0", ",", "n"}], "}"}]}], "]"}]}]], 
  "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->795503903],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{"-", "x"}], ")"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellTags->"Ex--22034138112425334325,23",
 CellLabel->"In[1]:=",
 CellID->340404677],

Cell[BoxData[
 FractionBox[
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"-", "1"}], ")"}], "n"], 
  RowBox[{"n", "!"}]]], "Output",
 ImageSize->{43, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--22034138112425334325,23",
 CellLabel->"Out[1]=",
 CellID->988999438]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InverseZTransform", "[", 
  RowBox[{
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "/", "x"}], ")"}]}], ",", " ", "x", ",", "n"}], 
  "]"}]], "Input",
 CellTags->"Ex--22034138112425334325,22",
 CellLabel->"In[2]:=",
 CellID->197148709],

Cell[BoxData[
 FractionBox[
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"-", "1"}], ")"}], "n"], 
  RowBox[{"n", "!"}]]], "Output",
 ImageSize->{43, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"Ex--22034138112425334325,22",
 CellLabel->"Out[2]=",
 CellID->590126686]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5444],

Cell["\<\
Series coefficients can be functions of the expansion variable:\
\>", "ExampleText",
 CellID->4221],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"x", "^", "x"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15972],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", 
   RowBox[{
    RowBox[{"Log", "[", "x", "]"}], " ", "x"}], "+", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    SuperscriptBox[
     RowBox[{"Log", "[", "x", "]"}], "2"], " ", 
    SuperscriptBox["x", "2"]}], "+", 
   RowBox[{
    FractionBox["1", "6"], " ", 
    SuperscriptBox[
     RowBox[{"Log", "[", "x", "]"}], "3"], " ", 
    SuperscriptBox["x", "3"]}], "+", 
   RowBox[{
    FractionBox["1", "24"], " ", 
    SuperscriptBox[
     RowBox[{"Log", "[", "x", "]"}], "4"], " ", 
    SuperscriptBox["x", "4"]}], "+", 
   RowBox[{
    FractionBox["1", "120"], " ", 
    SuperscriptBox[
     RowBox[{"Log", "[", "x", "]"}], "5"], " ", 
    SuperscriptBox["x", "5"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "6"],
    SeriesData[$CellContext`x, 0, {}, 0, 6, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, 
    Log[$CellContext`x], Rational[1, 2] Log[$CellContext`x]^2, Rational[1, 6] 
    Log[$CellContext`x]^3, Rational[1, 24] Log[$CellContext`x]^4, 
    Rational[1, 120] Log[$CellContext`x]^5}, 0, 6, 1],
  Editable->False]], "Output",
 ImageSize->{502, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1179729756]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{"%", ",", "4"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->20443],

Cell[BoxData[
 FractionBox[
  SuperscriptBox[
   RowBox[{"Log", "[", "x", "]"}], "4"], "24"]], "Output",
 ImageSize->{56, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1496341]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->505911541],

Cell["General coefficients of series may not be available:", "ExampleText",
 CellID->305764047],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"EllipticK", "[", 
    RowBox[{"Sin", "[", 
     RowBox[{"x", "^", "2"}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->149439397],

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"EllipticK", "[", 
    RowBox[{"Sin", "[", 
     SuperscriptBox["x", "2"], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Output",
 ImageSize->{331, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1526683070]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"EllipticK", "[", 
    RowBox[{"Sin", "[", 
     RowBox[{"x", "^", "2"}], "]"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->142026371],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox["\[Pi]", "2"], "+", 
   FractionBox[
    RowBox[{"\[Pi]", " ", 
     SuperscriptBox["x", "2"]}], "8"], "+", 
   FractionBox[
    RowBox[{"9", " ", "\[Pi]", " ", 
     SuperscriptBox["x", "4"]}], "128"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "6"],
    SeriesData[$CellContext`x, 0, {}, 0, 6, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
   Rational[1, 2] Pi, 0, Rational[1, 8] Pi, 0, Rational[9, 128] Pi}, 0, 6, 1],
  
  Editable->False]], "Output",
 ImageSize->{153, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->827743134]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{"%", ",", " ", "4"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->346891647],

Cell[BoxData[
 FractionBox[
  RowBox[{"9", " ", "\[Pi]"}], "128"]], "Output",
 ImageSize->{30, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->479787872]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->640320264],

Cell["Series coefficient for a hypergeometric function:", "ExampleText",
 CellID->320629407],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SeriesCoefficient", "[", 
  RowBox[{
   RowBox[{"HypergeometricPFQ", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["a", "1"], ",", 
       SubscriptBox["a", "2"]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["b", "1"], ",", 
       SubscriptBox["b", "2"], ",", 
       SubscriptBox["b", "3"]}], "}"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "n"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->85484165],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"n", "+", 
     SubscriptBox["a", "1"]}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"n", "+", 
     SubscriptBox["a", "2"]}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    SubscriptBox["b", "1"], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    SubscriptBox["b", "2"], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    SubscriptBox["b", "3"], "]"}]}], 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", "n"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    SubscriptBox["a", "1"], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    SubscriptBox["a", "2"], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"n", "+", 
     SubscriptBox["b", "1"]}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"n", "+", 
     SubscriptBox["b", "2"]}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"n", "+", 
     SubscriptBox["b", "3"]}], "]"}]}]]], "Output",
 ImageSize->{466, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2141153866]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->5956],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Coefficient",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Coefficient"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Series",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Series"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Normal",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Normal"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CoefficientList",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CoefficientList"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SeriesData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SeriesData"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CoefficientArrays",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CoefficientArrays"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->10097]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->6223],

Cell[TextData[ButtonBox["Power Series",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/PowerSeries"]], "Tutorials",
 CellID->30232],

Cell[TextData[ButtonBox["Making Power Series Expansions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/MakingPowerSeriesExpansions"]], "Tutorials",
 CellID->31271],

Cell[TextData[ButtonBox["Operations on Power Series",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/OperationsOnPowerSeries"]], "Tutorials",
 CellID->720780250],

Cell[TextData[ButtonBox["The Representation of Power Series",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/TheRepresentationOfPowerSeries"]], "Tutorials",
 CellID->31832],

Cell[TextData[ButtonBox["Converting Power Series to Normal Expressions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConvertingPowerSeriesToNormalExpressions"]], "Tutorials",
 CellID->22435]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with SeriesCoefficient",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
SeriesCoefficient"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->110789946],

Cell[TextData[ButtonBox["Discrete Mathematics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteMathematics"]], "MoreAbout",
 CellID->838131410],

Cell[TextData[ButtonBox["Series Expansions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SeriesExpansions"]], "MoreAbout",
 CellID->112694108],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->433273559],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->65207274]
}, Open  ]],

Cell["New in 3  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SeriesCoefficient - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 19, 49.3218754}", "context" -> "System`", 
    "keywords" -> {
     "inverse Z transform", "Maclaurin series expansion", 
      "power series expansion", "series coefficients", "series terms", 
      "Taylor series expansion"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SeriesCoefficient[series, n] finds the coefficient of the \
n\\[Null]\\[Null]^th order term in a power series in the form generated by \
Series. SeriesCoefficient[f, {x, x_0, n}] finds the coefficient of (x - x_0) \
n in the expansion of f about the point x = x_0. SeriesCoefficient[f, {x, \
x_0, n_x}, {y, y_0, n_y}, ...] finds a coefficient in a multivariate series. \
", "synonyms" -> {"series coefficient"}, "title" -> "SeriesCoefficient", 
    "type" -> "Symbol", "uri" -> "ref/SeriesCoefficient"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[13270, 361, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->665909816]},
 "Ex--22034138112425334325,28"->{
  Cell[39331, 1315, 352, 12, 70, "Input",
   CellTags->"Ex--22034138112425334325,28",
   CellID->121364303],
  Cell[39686, 1329, 668, 23, 49, "Output",
   CellTags->"Ex--22034138112425334325,28",
   CellID->1902594143]},
 "Ex--22034138112425334325,29"->{
  Cell[40529, 1363, 387, 13, 70, "Input",
   CellTags->"Ex--22034138112425334325,29",
   CellID->801988935],
  Cell[40919, 1378, 694, 24, 56, "Output",
   CellTags->"Ex--22034138112425334325,29",
   CellID->868257571]},
 "Ex--22034138112425334325,30"->{
  Cell[41807, 1415, 527, 16, 70, "Input",
   CellTags->"Ex--22034138112425334325,30",
   CellID->187986390],
  Cell[42337, 1433, 362, 13, 51, "Output",
   CellTags->"Ex--22034138112425334325,30",
   CellID->653614432]},
 "Ex--22034138112425334325,23"->{
  Cell[50421, 1763, 312, 10, 70, "Input",
   CellTags->"Ex--22034138112425334325,23",
   CellID->340404677],
  Cell[50736, 1775, 306, 11, 52, "Output",
   CellTags->"Ex--22034138112425334325,23",
   CellID->988999438]},
 "Ex--22034138112425334325,22"->{
  Cell[51079, 1791, 293, 10, 70, "Input",
   CellTags->"Ex--22034138112425334325,22",
   CellID->197148709],
  Cell[51375, 1803, 306, 11, 52, "Output",
   CellTags->"Ex--22034138112425334325,22",
   CellID->590126686]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 64666, 2280},
 {"Ex--22034138112425334325,28", 64821, 2284},
 {"Ex--22034138112425334325,29", 65076, 2291},
 {"Ex--22034138112425334325,30", 65330, 2298},
 {"Ex--22034138112425334325,23", 65584, 2305},
 {"Ex--22034138112425334325,22", 65838, 2312}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3612, 79, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4235, 106, 4201, 83, 70, "ObjectNameGrid"],
Cell[8439, 191, 2898, 92, 70, "Usage",
 CellID->20440]
}, Open  ]],
Cell[CellGroupData[{
Cell[11374, 288, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[11852, 305, 684, 24, 70, "Notes",
 CellID->686864027],
Cell[12539, 331, 694, 25, 70, "Notes",
 CellID->32326293]
}, Closed]],
Cell[CellGroupData[{
Cell[13270, 361, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->665909816],
Cell[CellGroupData[{
Cell[13655, 375, 147, 5, 70, "ExampleSection",
 CellID->13551923],
Cell[13805, 382, 83, 1, 70, "ExampleText",
 CellID->13815],
Cell[CellGroupData[{
Cell[13913, 387, 244, 8, 28, "Input",
 CellID->19099],
Cell[14160, 397, 1166, 40, 54, "Output",
 CellID->900003345]
}, Open  ]],
Cell[CellGroupData[{
Cell[15363, 442, 131, 4, 28, "Input",
 CellID->6713],
Cell[15497, 448, 183, 6, 51, "Output",
 CellID->1628893594]
}, Open  ]],
Cell[15695, 457, 96, 1, 70, "ExampleText",
 CellID->291740222],
Cell[CellGroupData[{
Cell[15816, 462, 246, 8, 70, "Input",
 CellID->720447568],
Cell[16065, 472, 264, 10, 52, "Output",
 CellID->693102402]
}, Open  ]],
Cell[CellGroupData[{
Cell[16366, 487, 188, 6, 70, "Input",
 CellID->313199119],
Cell[16557, 495, 395, 14, 51, "Output",
 CellID->1020769086]
}, Open  ]],
Cell[CellGroupData[{
Cell[16989, 514, 245, 8, 70, "Input",
 CellID->293812152],
Cell[17237, 524, 757, 26, 54, "Output",
 CellID->1966656166]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[18043, 556, 216, 7, 70, "ExampleSection",
 CellID->19835],
Cell[18262, 565, 72, 1, 70, "ExampleText",
 CellID->747820033],
Cell[CellGroupData[{
Cell[18359, 570, 356, 12, 70, "Input",
 CellID->36643142],
Cell[18718, 584, 241, 8, 51, "Output",
 CellID->1669370985]
}, Open  ]],
Cell[18974, 595, 70, 1, 70, "ExampleText",
 CellID->69356038],
Cell[CellGroupData[{
Cell[19069, 600, 287, 9, 70, "Input",
 CellID->15686472],
Cell[19359, 611, 1847, 38, 128, "Output",
 CellID->244296840]
}, Open  ]],
Cell[21221, 652, 124, 3, 70, "ExampleDelimiter",
 CellID->82926861],
Cell[21348, 657, 61, 1, 70, "ExampleText",
 CellID->18680779],
Cell[CellGroupData[{
Cell[21434, 662, 275, 9, 70, "Input",
 CellID->283053155],
Cell[21712, 673, 324, 12, 36, "Output",
 CellID->2055740463]
}, Open  ]],
Cell[CellGroupData[{
Cell[22073, 690, 546, 19, 70, "Input",
 CellID->32826465],
Cell[22622, 711, 489, 18, 54, "Output",
 CellID->52868011]
}, Open  ]],
Cell[23126, 732, 125, 3, 70, "ExampleDelimiter",
 CellID->158412130],
Cell[23254, 737, 63, 1, 70, "ExampleText",
 CellID->93458657],
Cell[CellGroupData[{
Cell[23342, 742, 301, 10, 70, "Input",
 CellID->39330682],
Cell[23646, 754, 416, 14, 53, "Output",
 CellID->1072514496]
}, Open  ]],
Cell[CellGroupData[{
Cell[24099, 773, 259, 8, 70, "Input",
 CellID->132956228],
Cell[24361, 783, 328, 12, 54, "Output",
 CellID->1920885001]
}, Open  ]],
Cell[CellGroupData[{
Cell[24726, 800, 364, 12, 70, "Input",
 CellID->137715104],
Cell[25093, 814, 492, 16, 54, "Output",
 CellID->1192546456]
}, Open  ]],
Cell[25600, 833, 125, 3, 70, "ExampleDelimiter",
 CellID->567838784],
Cell[25728, 838, 61, 1, 70, "ExampleText",
 CellID->182178878],
Cell[CellGroupData[{
Cell[25814, 843, 276, 9, 70, "Input",
 CellID->168492540],
Cell[26093, 854, 284, 10, 36, "Output",
 CellID->663408541]
}, Open  ]],
Cell[CellGroupData[{
Cell[26414, 869, 327, 11, 70, "Input",
 CellID->23220406],
Cell[26744, 882, 463, 16, 60, "Output",
 CellID->172253363]
}, Open  ]],
Cell[CellGroupData[{
Cell[27244, 903, 411, 13, 70, "Input",
 CellID->32253551],
Cell[27658, 918, 625, 24, 56, "Output",
 CellID->254315785]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[28332, 948, 239, 7, 70, "ExampleSection",
 CellID->26337],
Cell[28574, 957, 90, 1, 70, "ExampleText",
 CellID->208098727],
Cell[CellGroupData[{
Cell[28689, 962, 397, 13, 70, "Input",
 CellID->58757054],
Cell[29089, 977, 411, 15, 59, "Output",
 CellID->792940965]
}, Open  ]],
Cell[CellGroupData[{
Cell[29537, 997, 216, 7, 70, "Input",
 CellID->36712913],
Cell[29756, 1006, 179, 6, 51, "Output",
 CellID->1822769120]
}, Open  ]],
Cell[CellGroupData[{
Cell[29972, 1017, 387, 13, 70, "Input",
 CellID->263874971],
Cell[30362, 1032, 3156, 56, 70, "Output",
 Evaluatable->False,
 CellID->307534597]
}, Open  ]],
Cell[CellGroupData[{
Cell[33555, 1093, 188, 6, 70, "Input",
 CellID->1350547],
Cell[33746, 1101, 178, 6, 51, "Output",
 CellID->731144234]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[33973, 1113, 223, 7, 70, "ExampleSection",
 CellID->31288],
Cell[34199, 1122, 192, 7, 70, "ExampleText",
 CellID->19852],
Cell[CellGroupData[{
Cell[34416, 1133, 292, 10, 70, "Input",
 CellID->5475],
Cell[34711, 1145, 1016, 33, 36, "Output",
 CellID->439712704]
}, Open  ]],
Cell[CellGroupData[{
Cell[35764, 1183, 131, 4, 70, "Input",
 CellID->587],
Cell[35898, 1189, 159, 5, 36, "Output",
 CellID->760186801]
}, Open  ]],
Cell[CellGroupData[{
Cell[36094, 1199, 106, 3, 70, "Input",
 CellID->364824048],
Cell[36203, 1204, 160, 5, 36, "Output",
 CellID->1079348404]
}, Open  ]],
Cell[36378, 1212, 121, 3, 70, "ExampleDelimiter",
 CellID->18571],
Cell[36502, 1217, 97, 2, 70, "ExampleText",
 CellID->5826],
Cell[CellGroupData[{
Cell[36624, 1223, 334, 11, 70, "Input",
 CellID->11798],
Cell[36961, 1236, 1212, 24, 70, "Output",
 Evaluatable->False,
 CellID->244511750]
}, Open  ]],
Cell[CellGroupData[{
Cell[38210, 1265, 132, 4, 70, "Input",
 CellID->22045],
Cell[38345, 1271, 289, 10, 36, "Output",
 CellID->1498979249]
}, Open  ]],
Cell[CellGroupData[{
Cell[38671, 1286, 125, 4, 70, "Input",
 CellID->14357],
Cell[38799, 1292, 289, 10, 36, "Output",
 CellID->1213574986]
}, Open  ]],
Cell[39103, 1305, 124, 3, 70, "ExampleDelimiter",
 CellID->14910646],
Cell[39230, 1310, 76, 1, 70, "ExampleText",
 CellID->3797937],
Cell[CellGroupData[{
Cell[39331, 1315, 352, 12, 70, "Input",
 CellTags->"Ex--22034138112425334325,28",
 CellID->121364303],
Cell[39686, 1329, 668, 23, 49, "Output",
 CellTags->"Ex--22034138112425334325,28",
 CellID->1902594143]
}, Open  ]],
Cell[40369, 1355, 135, 4, 70, "ExampleText",
 CellID->222969517],
Cell[CellGroupData[{
Cell[40529, 1363, 387, 13, 70, "Input",
 CellTags->"Ex--22034138112425334325,29",
 CellID->801988935],
Cell[40919, 1378, 694, 24, 56, "Output",
 CellTags->"Ex--22034138112425334325,29",
 CellID->868257571]
}, Open  ]],
Cell[41628, 1405, 154, 6, 70, "ExampleText",
 CellID->304050393],
Cell[CellGroupData[{
Cell[41807, 1415, 527, 16, 70, "Input",
 CellTags->"Ex--22034138112425334325,30",
 CellID->187986390],
Cell[42337, 1433, 362, 13, 51, "Output",
 CellTags->"Ex--22034138112425334325,30",
 CellID->653614432]
}, Open  ]],
Cell[CellGroupData[{
Cell[42736, 1451, 149, 5, 70, "Input",
 CellID->1151751],
Cell[42888, 1458, 162, 5, 36, "Output",
 CellID->918904219]
}, Open  ]],
Cell[43065, 1466, 183, 8, 70, "ExampleText",
 CellID->384615724],
Cell[CellGroupData[{
Cell[43273, 1478, 456, 15, 70, "Input",
 CellID->18133952],
Cell[43732, 1495, 448, 16, 51, "Output",
 CellID->1468303156]
}, Open  ]],
Cell[CellGroupData[{
Cell[44217, 1516, 234, 8, 70, "Input",
 CellID->121098329],
Cell[44454, 1526, 163, 5, 36, "Output",
 CellID->1036555215]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44666, 1537, 233, 7, 70, "ExampleSection",
 CellID->12477],
Cell[44902, 1546, 156, 4, 70, "ExampleText",
 CellID->19755808],
Cell[CellGroupData[{
Cell[45083, 1554, 585, 19, 70, "Input",
 CellID->75096379],
Cell[45671, 1575, 299, 10, 51, "Output",
 CellID->996154041]
}, Open  ]],
Cell[CellGroupData[{
Cell[46007, 1590, 496, 16, 70, "Input",
 CellID->60689533],
Cell[46506, 1608, 299, 10, 51, "Output",
 CellID->578455075]
}, Open  ]],
Cell[46820, 1621, 125, 3, 70, "ExampleDelimiter",
 CellID->872803412],
Cell[46948, 1626, 223, 7, 70, "ExampleText",
 CellID->13018],
Cell[CellGroupData[{
Cell[47196, 1637, 322, 11, 70, "Input",
 CellID->3703],
Cell[47521, 1650, 1016, 33, 36, "Output",
 CellID->435975556]
}, Open  ]],
Cell[CellGroupData[{
Cell[48574, 1688, 250, 8, 70, "Input",
 CellID->17911],
Cell[48827, 1698, 306, 9, 36, "Output",
 CellID->226966128]
}, Open  ]],
Cell[CellGroupData[{
Cell[49170, 1712, 130, 4, 70, "Input",
 CellID->15787],
Cell[49303, 1718, 306, 9, 36, "Output",
 CellID->688756224]
}, Open  ]],
Cell[49624, 1730, 124, 3, 70, "ExampleDelimiter",
 CellID->51225179],
Cell[49751, 1735, 645, 24, 70, "ExampleText",
 CellID->795503903],
Cell[CellGroupData[{
Cell[50421, 1763, 312, 10, 70, "Input",
 CellTags->"Ex--22034138112425334325,23",
 CellID->340404677],
Cell[50736, 1775, 306, 11, 52, "Output",
 CellTags->"Ex--22034138112425334325,23",
 CellID->988999438]
}, Open  ]],
Cell[CellGroupData[{
Cell[51079, 1791, 293, 10, 70, "Input",
 CellTags->"Ex--22034138112425334325,22",
 CellID->197148709],
Cell[51375, 1803, 306, 11, 52, "Output",
 CellTags->"Ex--22034138112425334325,22",
 CellID->590126686]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[51730, 1820, 225, 7, 70, "ExampleSection",
 CellID->5444],
Cell[51958, 1829, 109, 3, 70, "ExampleText",
 CellID->4221],
Cell[CellGroupData[{
Cell[52092, 1836, 204, 7, 70, "Input",
 CellID->15972],
Cell[52299, 1845, 1287, 39, 51, "Output",
 CellID->1179729756]
}, Open  ]],
Cell[CellGroupData[{
Cell[53623, 1889, 132, 4, 70, "Input",
 CellID->20443],
Cell[53758, 1895, 232, 8, 54, "Output",
 CellID->1496341]
}, Open  ]],
Cell[54005, 1906, 125, 3, 70, "ExampleDelimiter",
 CellID->505911541],
Cell[54133, 1911, 95, 1, 70, "ExampleText",
 CellID->305764047],
Cell[CellGroupData[{
Cell[54253, 1916, 290, 9, 70, "Input",
 CellID->149439397],
Cell[54546, 1927, 381, 12, 39, "Output",
 CellID->1526683070]
}, Open  ]],
Cell[CellGroupData[{
Cell[54964, 1944, 284, 9, 70, "Input",
 CellID->142026371],
Cell[55251, 1955, 701, 23, 54, "Output",
 CellID->827743134]
}, Open  ]],
Cell[CellGroupData[{
Cell[55989, 1983, 141, 4, 70, "Input",
 CellID->346891647],
Cell[56133, 1989, 207, 7, 51, "Output",
 CellID->479787872]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[56389, 2002, 228, 7, 70, "ExampleSection",
 CellID->640320264],
Cell[56620, 2011, 92, 1, 70, "ExampleText",
 CellID->320629407],
Cell[CellGroupData[{
Cell[56737, 2016, 525, 17, 70, "Input",
 CellID->85484165],
Cell[57265, 2035, 1054, 35, 53, "Output",
 CellID->2141153866]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[58380, 2077, 310, 9, 70, "SeeAlsoSection",
 CellID->5956],
Cell[58693, 2088, 1465, 53, 70, "SeeAlso",
 CellID->10097]
}, Open  ]],
Cell[CellGroupData[{
Cell[60195, 2146, 313, 9, 70, "TutorialsSection",
 CellID->6223],
Cell[60511, 2157, 134, 3, 70, "Tutorials",
 CellID->30232],
Cell[60648, 2162, 168, 3, 70, "Tutorials",
 CellID->31271],
Cell[60819, 2167, 164, 3, 70, "Tutorials",
 CellID->720780250],
Cell[60986, 2172, 175, 3, 70, "Tutorials",
 CellID->31832],
Cell[61164, 2177, 199, 4, 70, "Tutorials",
 CellID->22435]
}, Open  ]],
Cell[CellGroupData[{
Cell[61400, 2186, 305, 8, 70, "RelatedLinksSection"],
Cell[61708, 2196, 376, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[62121, 2213, 319, 9, 70, "MoreAboutSection",
 CellID->110789946],
Cell[62443, 2224, 151, 3, 70, "MoreAbout",
 CellID->838131410],
Cell[62597, 2229, 145, 3, 70, "MoreAbout",
 CellID->112694108],
Cell[62745, 2234, 169, 3, 70, "MoreAbout",
 CellID->433273559],
Cell[62917, 2239, 178, 3, 70, "MoreAbout",
 CellID->65207274]
}, Open  ]],
Cell[63110, 2245, 50, 0, 70, "History"],
Cell[63163, 2247, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

