(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    122345,       3060]
NotebookOptionsPosition[    112891,       2737]
NotebookOutlinePosition[    114413,       2777]
CellTagsIndexPosition[    114326,       2772]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Constructing Matrices" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ConstructingMatrices"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Constructing Matrices\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ShearingTransform" :> 
          Documentation`HelpLookup["paclet:ref/ShearingTransform"], 
          "ScalingMatrix" :> 
          Documentation`HelpLookup["paclet:ref/ScalingMatrix"], 
          "RotationMatrix" :> 
          Documentation`HelpLookup["paclet:ref/RotationMatrix"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"ShearingTransform\"\>", 
       2->"\<\"ScalingMatrix\"\>", 
       3->"\<\"RotationMatrix\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Constructing Matrices" :> 
          Documentation`HelpLookup["paclet:guide/ConstructingMatrices"], 
          "Geometric Transforms" :> 
          Documentation`HelpLookup["paclet:guide/GeometricTransforms"], 
          "New in 6.0: Matrix & Linear Algebra Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Constructing Matrices\"\>", 
       2->"\<\"Geometric Transforms\"\>", 
       3->"\<\"New in 6.0: Matrix & Linear Algebra Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["ShearingMatrix", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ShearingMatrix",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ShearingMatrix"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        StyleBox["v", "TI"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the matrix corresponding to shearing by ",
     Cell[BoxData[
      StyleBox["\[Theta]", "TR"]], "InlineFormula"],
     " radians along the direction of the vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     ", and normal to the vector ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ShearingMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingMatrix"]], "InlineFormula"],
 " gives matrices corresponding to shearing with the origin kept fixed."
}], "Notes",
 CellID->798539111],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ShearingMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingMatrix"]], "InlineFormula"],
 " gives matrices with determinant 1, corresponding to area- or \
volume-preserving transformations."
}], "Notes",
 CellID->25828702],

Cell[TextData[{
 "In 2D, ",
 Cell[BoxData[
  ButtonBox["ShearingMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingMatrix"]], "InlineFormula"],
 " turns rectangles into parallelograms. ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}]}], "]"}]], "InlineFormula"],
 " effectively slants by angle ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " to the right."
}], "Notes",
 CellID->3268604],

Cell[TextData[{
 "In 3D, ",
 Cell[BoxData[
  ButtonBox["ShearingMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingMatrix"]], "InlineFormula"],
 " does the analog of shearing a deck of cards by angle ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in the direction ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 ", with the cards being oriented so as to have normal vector ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->1067943069]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->177797755],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->172718204],

Cell[TextData[{
 "A shearing by ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " radians along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis:"
}], "ExampleText",
 CellID->259330806],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}]}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->435464200],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlVjsOwjAMNfnwUZEQGwMDKxVnYGCAqYeIKqQyIFDp/UMcAhTjqgGJATVS
6rxX23Ve3U9mqmJ/NNUhN4tdac7FIb8stqfSUbIHAJmb8xm4tXUrPzNrnb0d
NR4lsn7AmmDlvRTDIkyc1c4uyXnJRnlWP7D3ARgTr9/nktweW3OlBIvO5sKe
kSFaOLuKitr8tVca9poEO4qOquOE4EH0vdPh2kgjHkJYwH3QKMyl4P0Jj7tm
n2DaKT7qrZ94lu89+kSLTteJ/urDyjTLxnXm97qJe5XUW9W9B0xH8kpRfWm/
ckrJWi3hpQ7NivDskMnbXs3odaeMFhNul63KxlWj7PMthApPA0ZeBSlEY7a4
b/3tvwB6V4Kpbgw=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{77, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->66711372]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->797551938],

Cell[TextData[{
 "Apply a 30\[Degree] shear along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis to a square:"
}], "ExampleText",
 CellID->328163910],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"ShearingMatrix", "[", 
      RowBox[{
       RowBox[{"30", " ", "Degree"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}], ",", " ", 
   RowBox[{"Frame", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->417290019],

Cell[BoxData[
 GraphicsBox[
  GeometricTransformationBox[RectangleBox[{0, 0}], 
   NCache[{{1, 3^Rational[-1, 2]}, {0, 1}}, {{1, 0.5773502691896258}, {0, 
     1}}]],
  Frame->True]], "Output",
 ImageSize->{184, 124},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->477541236]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 "Shearing along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis: "
}], "ExampleText",
 CellID->172695089],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}]}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->91127034],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlVjsOwjAMNfnwUZEQGwMDKxVnYGCAqYeIKqQyIFDp/UMcAhTjqgGJATVS
6rxX23Ve3U9mqmJ/NNUhN4tdac7FIb8stqfSUbIHAJmb8xm4tXUrPzNrnb0d
NR4lsn7AmmDlvRTDIkyc1c4uyXnJRnlWP7D3ARgTr9/nktweW3OlBIvO5sKe
kSFaOLuKitr8tVca9poEO4qOquOE4EH0vdPh2kgjHkJYwH3QKMyl4P0Jj7tm
n2DaKT7qrZ94lu89+kSLTteJ/urDyjTLxnXm97qJe5XUW9W9B0xH8kpRfWm/
ckrJWi3hpQ7NivDskMnbXs3odaeMFhNul63KxlWj7PMthApPA0ZeBSlEY7a4
b/3tvwB6V4Kpbgw=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{77, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->108488351]
}, Open  ]],

Cell[TextData[{
 "Shearing along the ",
 Cell[BoxData[
  StyleBox["y", "TI"]], "InlineFormula"],
 " axis: "
}], "ExampleText",
 CellID->130502387],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}]}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->12557635],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztVksKwjAQHZtU6x/BhYKCHkDxBC5c6KqHKCLoQpTq/eNMGqWOU6rVjWKg
Sd508vL6Mi0No/N2s4/Ou3U0WsXRcbtbn0bLQ4whVQKAEK9BD3BucGav0Bgc
x9h7COtuHNqoT72iLNugz3Cd4S7D2rJoIUqQxgDcBK5NsWwPOw2PWmrY+QjL
LK7EPflqzwZv2AmQozo7yhlTuMHwd+vkpyJl6ReV+WK0xZgrkv5cPdm+eVeV
PFunsytCJXKniimj+yqlxRkM2Y580id1/6SCF22JJ9fZ4mrI6Y7zgupEO0sI
N91YfYpt8c9yUfKRjmhqkloLnlrFK3VipJMuVvW/wJX3rZa5in333+EiTG/N
zCSnz58mWcV3nAss+MOQ7Fy6ACUebgw=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{77, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->10251281]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->365787824],

Cell[TextData[{
 "Shearing in the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], ",", 
   StyleBox["y", "TI"]}]], "InlineFormula"],
 " plane along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis: "
}], "ExampleText",
 CellID->307503498],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->104664616],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", "0", 
      RowBox[{"Tan", "[", "\[Theta]", "]"}]},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.28], {
         Offset[0.7]}, 
        Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{96, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->102174814]
}, Open  ]],

Cell[TextData[{
 "Shearing the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], ",", 
   StyleBox["z", "TI"]}]], "InlineFormula"],
 " plane along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis: "
}], "ExampleText",
 CellID->330765079],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3655249],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", 
      RowBox[{"Tan", "[", "\[Theta]", "]"}], "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.28], {
         Offset[0.7]}, 
        Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{96, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->227445149]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->368966718],

Cell[TextData[{
 "A shearing by angle ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " in the ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"1", ",", "1"}], "}"}], TraditionalForm]], "InlineMath"],
 " direction in the line ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}], ".", "p"}], "==", "0"}], 
   TraditionalForm]], "InlineMath"],
 ": "
}], "ExampleText",
 CellID->686240600],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"-", "1"}]}], "}"}]}], "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29672303],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"1", "+", 
       FractionBox[
        RowBox[{"Tan", "[", "\[Theta]", "]"}], 
        SqrtBox["2"]]}], 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"Tan", "[", "\[Theta]", "]"}], 
        SqrtBox["2"]]}]},
     {
      FractionBox[
       RowBox[{"Tan", "[", "\[Theta]", "]"}], 
       SqrtBox["2"]], 
      RowBox[{"1", "-", 
       FractionBox[
        RowBox[{"Tan", "[", "\[Theta]", "]"}], 
        SqrtBox["2"]]}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.28], {
         Offset[0.7]}, 
        Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{140, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//MatrixForm=",
 CellID->346680294]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->734046779],

Cell["Transformation applied to a 2D shape:", "ExampleText",
 CellID->479507293],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->419323043],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
    ",", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}], 
      ",", 
      RowBox[{"ShearingMatrix", "[", 
       RowBox[{
        RowBox[{"Pi", "/", "6"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->150504143],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], RectangleBox[{0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
     {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, GeometricTransformationBox[
    {RGBColor[1, 0, 0], Opacity[0.85], RectangleBox[{0, 0}], 
     {AbsolutePointSize[10], Opacity[1], 
      {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
      {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, 
    NCache[{{1, 3^Rational[-1, 2]}, {0, 1}}, {{1, 0.5773502691896258}, {0, 
      1}}]]}]], "Output",
 ImageSize->{184, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->282362206]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30645500],

Cell["Transformation applied to a 3D shape:", "ExampleText",
 CellID->137131831],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cuboid", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->367829949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
     ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}],
        ",", 
       RowBox[{"ShearingMatrix", "[", 
        RowBox[{
         RowBox[{"Pi", "/", "4"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "1"}], ",", "1", ",", "0"}], "}"}]}], "]"}]}], 
      "]"}]}], "}"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167437831],

Cell[BoxData[
 Graphics3DBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], CuboidBox[{0, 0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], Point3DBox[{0, 0, 0}]}, 
     {RGBColor[0, 1, 0], Point3DBox[{1, 1, 1}]}}}, 
   InterpretationBox[
    {RGBColor[1., 0., 0.], Opacity[0.85], 
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {0.5, 1.5, 0.}, {0.5, 
      1.5, 1.}, {0.5, -0.5, 0.}, {0.5, -0.5, 1.}, {1., 1., 0.}, {1., 1., 1.}},
       Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic], 
     {AbsolutePointSize[10.], Opacity[1.], 
      {RGBColor[1., 0., 1.], Point3DBox[{0., 0., 0.}]}, 
      {RGBColor[0., 1., 0.], Point3DBox[{1., 1., 1.}]}}},
    GeometricTransformation[{
      Opacity[0.85], 
      RGBColor[1, 0, 0], {
       Cuboid[{0, 0, 0}], 
       AbsolutePointSize[10], 
       Opacity[1], {
        RGBColor[1, 0, 1], 
        Point[{0, 0, 0}]}, {
        RGBColor[0, 1, 0], 
        Point[{1, 1, 1}]}}}, {{
       Rational[1, 2], 
       Rational[1, 2], 0}, {
       Rational[-1, 2], 
       Rational[3, 2], 0}, {0, 0, 1}}]]},
  Boxed->False]], "Output",
 ImageSize->{184, 200},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->188616850,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->347222277],

Cell["Applying the transformation to a surface:", "ExampleText",
 CellID->70266796],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Plot3D", "[", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"Sin", "[", 
      RowBox[{"x", " ", "y"}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "Pi"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "Pi"}], "}"}], ",", 
    RowBox[{"BoxRatios", "\[Rule]", "Automatic"}], ",", 
    RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], "/.", 
  RowBox[{"gc_GraphicsComplex", "\[RuleDelayed]", 
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{"gc", ",", 
     RowBox[{"ShearingMatrix", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "Pi"}], "/", "4"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->252614726],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->178342705]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->244509174],

Cell[TextData[{
 "Generate all simple (directions parallel to axes) shearing matrices for \
dimension ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->563715391],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Shearings", "[", 
    RowBox[{"\[Theta]_", ",", "n_"}], "]"}], " ", ":=", "\[IndentingNewLine]",
    
   RowBox[{"With", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"pairs", "=", 
       RowBox[{"Subsets", "[", 
        RowBox[{
         RowBox[{"Table", "[", " ", 
          RowBox[{
           RowBox[{"UnitVector", "[", 
            RowBox[{"n", ",", "j"}], "]"}], ",", " ", 
           RowBox[{"{", 
            RowBox[{"j", ",", " ", "n"}], "}"}]}], "]"}], ",", " ", 
         RowBox[{"{", "2", "}"}]}], "]"}]}], "}"}], ",", 
     RowBox[{"Map", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"ShearingMatrix", "[", 
         RowBox[{"\[Theta]", ",", 
          RowBox[{"#", "[", 
           RowBox[{"[", "1", "]"}], "]"}], ",", 
          RowBox[{"#", "[", 
           RowBox[{"[", "2", "]"}], "]"}]}], "]"}], "&"}], ",", 
       RowBox[{"Join", "[", 
        RowBox[{"pairs", ",", " ", 
         RowBox[{"Reverse", "/@", "pairs"}]}], "]"}]}], "]"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->344720790],

Cell["All shearings in 2D:", "ExampleText",
 CellID->659888018],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", " ", 
  RowBox[{"MatrixForm", ",", " ", 
   RowBox[{"Shearings", "[", 
    RowBox[{"\[Theta]", ",", "2"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->112862591],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWMtOwzAQdB6mT0AVHBDi0AMXKkB8AUggwYmPiCqk9oBApR/AP/UDgx1v
IJ1OaxKaUgkiNcmu3Z2dHdfd5DGZjp6ek+l4mPQfJsnraDx869+/TIwrCpRS
M/M5OVLmPjV32WeWpub6bs6hMTtyPc28XXuO7KzsUNdgH4B9DHYH7EM2vjAr
NqdIctHmegbjPbC1zLffayq5UfkRwezQnGJjavC3BW2H+F3IkHwri64/bQHG
uvkYIGZIo3JvvNyLEQs2ZogZREW+CzX8K1VAniGtBqLwXBBrAPY+2A1S47gk
f029ex4kWrVfYr18DYR5LXB2XJzdIDsA5rTN/C1qVGDs0nAHV3f7NceoLRb1
S0WiM1aNroMN1SD/9+mKQueErfW3SRy7Onsybu1YBLb2rlxpdUpneesZb4J9
9y1UrDvulwyVcUI0jBM4BWXtM+0QCVcIItid3jZENnJQiYsPgXMoq5MPpdpq
GKTz3R7qUR/qqu4QNa0nC2tr4f7zrtGXsxa0S0HD31k1Dtin4I5bf+fk443j
FxRlPTvwfzXKVGN9faRv19gm1nX0kZtUfdnzJ9pM7XX0j7hv+Z7dONfVT5n4
f4DcWvOqEN2q9YX1cGtTTrmO1r+q+/PlpGXeVer0xVXk0Mu+v7kB272fcW+I
VPABucbeew==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{179, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->472073084]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Framed", "@", 
     RowBox[{"Graphics", "[", 
      RowBox[{
       RowBox[{"GeometricTransformation", "[", 
        RowBox[{
         RowBox[{"Rectangle", "[", "]"}], ",", "#"}], "]"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0", ",", "1.6"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1.6"}], "}"}]}], "}"}]}]}], "]"}]}], "&"}], ",",
    
   RowBox[{"Shearings", "[", 
    RowBox[{
     RowBox[{"Pi", "/", "6"}], ",", "2"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->73998204],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FrameBox[
    GraphicsBox[
     GeometricTransformationBox[RectangleBox[{0, 0}], 
      NCache[{{1, 3^Rational[-1, 2]}, {0, 1}}, {{1, 0.5773502691896258}, {0, 
        1}}]],
     PlotRange->{{0, 1.6}, {0, 1.6}}],
    StripOnInput->False], ",", 
   FrameBox[
    GraphicsBox[
     GeometricTransformationBox[RectangleBox[{0, 0}], 
      NCache[{{1, 0}, {3^Rational[-1, 2], 1}}, {{1, 0}, {
        0.5773502691896258, 1}}]],
     PlotRange->{{0, 1.6}, {0, 1.6}}],
    StripOnInput->False]}], "}"}]], "Output",
 ImageSize->{250, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[3]=",
 CellID->27517904]
}, Open  ]],

Cell["All shearings in 3D:", "ExampleText",
 CellID->99252900],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", " ", 
  RowBox[{"MatrixForm", ",", " ", 
   RowBox[{"Shearings", "[", 
    RowBox[{"\[Theta]", ",", "3"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->558245758],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{312, 87},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->543224402]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Graphics3D", "[", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"Cuboid", "[", "]"}], ",", "#"}], "]"}], "]"}], "&"}], ",", 
   RowBox[{"Shearings", "[", 
    RowBox[{
     RowBox[{"Pi", "/", "6"}], ",", "3"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->131239154],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {0.5773502691896258, 
      1., 0.}, {0.5773502691896258, 1., 1.}, {1., 0., 0.}, {1., 0., 1.}, {
      1.5773502691896257`, 1., 0.}, {1.5773502691896257`, 1., 1.}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     GeometricTransformation[
      Cuboid[{0, 0, 0}], {{1, 3^Rational[-1, 2], 0}, {0, 1, 0}, {0, 0, 1}}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\
eJzdmNtu00AQhp3Yzjlp0jRtShQ1UFVtQUUVohLiIA4qVAKBBJfcVSVSe4GK
CtcgXoVH4nmIYGf2t+P+XhMnFwiImq3H+8+3s7MHx/vy+NPp+P3xp7OT49HR
xfGH07OTj6Nn5xfmll/wPO+d+b7te+b6p7nS7/PJZOL54/F4oh9vTQrvyJSB
ubmu1lNTFo3l439f7z4xpTiGCYIpQXiI2oqz9gFIdbGCuHVzsar1d+HdFCuM
68HoqeYOGJ0Uw9YfgNF1MlZUcwuaVcpCkWxtw9tHi+u51HtgD6jWJ7ut6htg
W5YZGdRyW27vHbRlq+0nNN8+qQOyl1S+FXkHU2/P88/3vmgkP75+Nn96yXkK
yW4pYDPRl6LxKjHJFHJ/JRdt5IyuQszQFGJ2iFEiW2eUN3Qy68SsIM4lYpTJ
bihkQL1uEa0GWjMXre+MsE3MBnpdI0aFbF1nOnrJCLtEW0KE1Rk0bc3rOiPs
EbMDJvfSzWw7mX1idtFrns0ct9o6i5K9HhBtFRHyKnbT6s4Ih8Tsg5nqIWdA
qcUUb0S8K7psRIaZ6ppDOtd5F9jMIsF2jbelSSmOW7MI6CuPqK686Zo2CgFt
z6AFGFtet+FCtDJi45liacm7kqudGbQqaLwHBwvR6tG8M7dKCxGa0bY8sc+0
fL0aEKXx71Aw6otRfIpl949Q4jGCS/A3UNCj/56C+cKUNVMEuk+aCwvLIBUp
nutEWpHdStavY23PT+uAVs2aiXD1c9FaoDWzYpuLVgetQ7Qa9TQfrYwR6Dl2
2PlpAWLj/bGSnxbENLCG87Ew02LWOPpcJU7ZydEJEoe0k6LIc2KTSKUcJHl2
S57Db9/1Wc7RhDkY1wxDXBqGIXFsZMUBj6KDsQFGB3FwdjkOF2OQysoaaLwr
BDlo/RRtCNo60fwctF6Ktgnaaj5aYj52MWa7IKwQoTiT0Ea29zFiXZ14MUGs
33g3Un05QCS69kvTSCJImlFNMe6B0brMsBgno5RiPAajkRFHIcXwkcsjeOpu
FVrPaQ4cfubyFXx05Qd2jSfbsb+0zU2jM3/ea2S7lKHnEwPRvkEbQTTGFJE1
x/j1IcrX8LAuUXIKqafnBL9yIx9pTVSFWM2/iSuI6YVRS9qLl9TCik5shHmI
OEQVxOPHb3mR+gGp7e9dfmNoQX1AavtuyG8pbahvktqeGtQmidOhxMhKD7fR
w2R+RbusmoYjKj6BkfZGiXal2p5c8ds92yV4r1PU9qQteldlryq8lslrqF5t
UvObZhPeNfK+qt58YpKVZz8ja1tKWUbk7N2FN3vtqtesE6Aecrun6h7V8gzu
oF/7quazKl4dTahvq3ptMj3j5GyI6o6qdJQunTWWUX9f668kvMd4nkrtI62V
EyLZfA7V8gq/AHFAtXA=\
\>"]], ",", 
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0.5773502691896258, 0., 1.}, {0., 
      1., 0.}, {0.5773502691896258, 1., 1.}, {1., 0., 0.}, {
      1.5773502691896257`, 0., 1.}, {1., 1., 0.}, {1.5773502691896257`, 1., 
      1.}}, Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 
       7}, {1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     GeometricTransformation[
      Cuboid[{0, 0, 0}], {{1, 0, 3^Rational[-1, 2]}, {0, 1, 0}, {0, 0, 1}}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\
eJydmOtuHDUUgCc7M7ubzd432VzaRikQCJQVLW1IL6oECKiExAvwL6pWan+g
otL/vAuPxPuMwD4+M+v97MQDI8UZj8/59lw8x2P/cv3hzfq36w9vX1+fvXp/
/fubt6//OPvp3XvzKN/Jsmxt/n49ysz9P+ZO/n6uqirL1+t1JVd2ZJvslWlL
8/BYej+atmN6uf53T783rVUsPYJplfCtyvZtr2j45uZQxl+q9h603ehzHR0H
2ksZf6r02Q3jl6q/sL2yGdffOBCZx8pYIgId9Oci/VCJR62kVyp9B6M5+mJ/
9kAtcSyTFR09hnSB/lS0L/S3sq509Xq3+tO0J60I50qwv98xWqVV9uw4hFaJ
/kQoH3teWOU+KIxzF33JdXZWe1P43gzBOmjFOvUssn5NQdmHVg/9kVDuRC1a
JFi0yLGOoqzlf7RrKMpLeHcCygJaffTlzROp0KJ7YM1bsWZq0TbrPlgz6O6i
PxDlcW1XvmGdgzSNaZIstoUeXoA1acXaRcy/BGUciwvjJnMjtOgrsEaxrHNW
SGuVHkOb0nHtss5Y7kqGhTwBiT7Qrm5DqiuxBWbZN+AwommOi84VOJwvjLkQ
gkrX1Fb18jJBHWqOulFanqDx/bD9oqnwqMdJOiPJelD6tibpRcJ20jtKZ4Ti
9DJB922088TOPs7NdmRGpVA7+SbHad0Erak9TiWoD+2ozdxVCtd/zl1HpVT/
JqqXoWA95nwKqLsxatlQNT+s1u1YT21TNKz6Sy3JGcQ46ml7a/ZuomgWuG60
o8hM7jqKu9x9O9oQtEemsRUh/+vvyvcwnjtmeAzayjR2uA8a51c72ud19hpP
hzVXI0huJ8qdgHsecKfgck2Ic6fg3g+4C8Th/3FPNUeHoLFexWkz0E40R3fh
M9fXOG0O2kHg8xm4XG/bcecB9xNwuQ44rl8rD8AcB8zPEFHaupNk7mluHiSs
C0lLkHqal4cgcQ0PSYcg5Up6AhJX1NtJ9lvQCl8hRu0oVuAFNPldEGruV5sd
oGgagoU8hyfc+YUk+w7ZvHQTJK6cNWnzZFT5JwQb0jN4Z23aMTeWEFIGlXeS
kLuivWWPUopbKT2lcP15Aa/yWyn1F4rs7zbfuOYip3M7Z62nKLHvW40SZ09I
tMrx8wFQyobisbYtIqeMcYrGv3qmNTPHUsngDLryYrSxguss49JLUFwsOBt5
VtEH5WvT2K7sAzWqm90UtXehvfK05RtAtPl+c/8+AOUCFLejY9Xizt2uXefQ
dLv3em9LjRF+955pimw7d1bPnQJxNeP3zBi0Y88WKyZeByssvzcmoOzDI3cK
OFKPqD2F9gTaMhMDy+nZDJQBKKdCSZ1RzEEpb4juR0JLnZ0sQIuRPhXSXGND
gq1NHS8fX4h06nRrX31eiTRP57hq1GvEI5Guz7EoNVSpS5GSc9fSr5nWnz2V
eSYycl66dRrc0/GXMn6iPtV1+Dt5erdyZ90/SC/b+RfjLfRq\
\>"]], ",", 
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0.5773502691896258, 1.}, {0., 
      1., 0.}, {0., 1.5773502691896257`, 1.}, {1., 0., 0.}, {1., 
      0.5773502691896258, 1.}, {1., 1., 0.}, {1., 1.5773502691896257`, 1.}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     GeometricTransformation[
      Cuboid[{0, 0, 0}], {{1, 0, 0}, {0, 1, 3^Rational[-1, 2]}, {0, 0, 1}}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\
eJy9mdtS40YQhgcdfbaXg9cYDIbFCzYQIISqJJXN5iq5ykNQlKt2L1KbIvsq
eYDNI+UJ8iKuRN3T0ki/BksyJq7C2FL/39/dM5Y00q/3nz/Mf7v//PHhfvzL
4/3vHz4+/DH++dNjtMndUErNo79PAxV9/jf6xH/zxWKh1LfRuzOfz8/52130
7kbflBt9mPNLzXjPrcR59I1CdICa8t6b1F4ngucjvhIygd0owPnz4R/l0X6l
dOAZx10Ayac4n2hfHpNQdcqxU2CGcSwFfnnMxJ4Ct0GxQRKboN9y9AmQWxQd
GnI6lwkrjoHfJYVr+A69SVdJcMKqQ/B5FavoP7URVG9YNQKvbavKZKhVQ/Dq
g8oD1TGrBuC1G49b9F8LPcjxiHU74Lb3hM74ad0W+B3kdD7oxqzrgd84pwti
neR5yLoO+L2J55DWSW8CqPGAtU3wnCzRmny1tg6+Z1ZtCNoRawPwnVm1Nch5
P9FyXxbxi9wvU3OBPvKv0zWTHXtuWFQDEa4KCNiFPZONHCso5+sCSghjOGSd
n62pCkeyKctxCzi7huOne6wsHUJWDViDZ7DqMPavn80yeVVheQWsvmG5pu8O
f4+ihHJRQG3ArNhZL1Vy1VQPqG5FahOo2ytR/aeo0oGttVBbMIs2rVQPqOel
qKYD66G2gfpqJWpgpZoO9NZINbnaqT5QZ2WonqZqcJe5LnCDAm5oG68X4FK+
yGitjaz8JGPpctvKDoE9LcN2E7YcVQyvsZJLbclv5OVcimqpv4gL8mr/o4u1
wpTLGbjULS6BLdNC8mn0RrsbBXnjUSAoRT8RulOxS3iO9Eu5HYlbUPG3Us6t
AW6jTOeWu9WA5ZVyGIpDp8JxO55j3IdSLq/FZbPCmSxG4mx2Szlui2MfHPGa
JHGc80vhDNdu6fNgG5x64jQs4YRjVExvC31Uto4n5luxU0OcjsAJr4mf7xSK
0wSccKXyfCdPnPAIWuRkvXrMOHXAiYJ5JeknLvQtenVXYlPADZdgsl6FFa/T
07l+neoGZdizHNljUvYsYn6PQKuUpZNj+8LGMx9mWp3sCBmPD7cxWU+CqOdV
yKSj1T/Ot29iqsyrjqWzmkTqHqjvQF10Xo9zMqPdtVRJX9NUvL6yU/H4ivP9
SsjuX38vktVXaTpuxT7SKoIuu0OmZ3O3rm15vGwjciZ5toDUhDyxXk3FscNZ
NBF6D+iNUvSwgD6WLmxBj5GO3dxgOp5r8EpmJLkPCuiYu6bXYWsN6EOh7wO9
XorekNFEal86MgZqrRS1BVtxbbUpOU9gNImOxwNNbMNWXLd2hTgFYmhdpelj
bgeYuMJuSg8ugRksYXaBicfwmuR5A31dxuzJGCHLk/zuLCz7TOW7IH6ylbfp
pyQk/x44fm5NrvPZwhqjkB9B61nr0b9GvhuVuQP6E3TYXaLeKaWmTDeiTVS3
rr0vXYxPuCR6D0onV7H2HKQqpo69h2qf0u2mclX84j75iZJ0Sp8DstkOc1Wy
MgRlpjN7dk1g8owFOn7f0pF3nB+EC/8AusB0T9JJcQ9zVb+LIxOkrnKcy/gH
jlTxPcYk8igX+V0USYFOEnNsqedWouKzlo48gUquoyhXorxknk9ydcyE5kqc
XpW/zeU2gbgmbz615HcIkfqu1xTy24Mofed1BlF9iNL30M9zVfQgTj8luchV
0YQ4/Szr0lKFD5H6WeBV1hli9BPTa6gh3qufFNMTX5qMV3rvxn8uEXTA\
\>"]], ",", 
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {0., 1., 0.}, {0., 1., 
      1.}, {1., 0.5773502691896258, 0.}, {1., 0.5773502691896258, 1.}, {1., 
      1.5773502691896257`, 0.}, {1., 1.5773502691896257`, 1.}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     GeometricTransformation[
      Cuboid[{0, 0, 0}], {{1, 0, 0}, {3^Rational[-1, 2], 1, 0}, {0, 0, 1}}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\
eJytmPtO3EYUxof1Zb1XloUFlkC5hEBISgpJKuWPtKFNlEqV+hAoQUr+qBKR
vEGfpH2XvkDfx2rnXOwZPs9iy3QFXo/P9/1mzsz6dn67/Prh6vfLrx/fXe79
cn35+cPHd1/23n66toeiJWPMe/v/adPY/X/tHv+/z/PcmBd227m6unrEre/t
NrItE9kd+00HTzjyTHWxfsvRM9Wn1IrEQPaHHH2i2h61YomK4Jjjj9U9WBA/
Uf+YWkkZ1z6OWHOkjIk3AtrtQPs+qw+VuNpIva/sGUQjaB+welfZmxCNob3P
6m1lb0E0gfYeq7eUvdNIvaHsXYim0Oa4mSn7AKJdaH/D6qmyDxupJ8o+hmgG
bc7LjJR9AtEetLdZPVD2t43UmbKfQLQP7XusTpV9DtEBtHn1eBVI/RSiQ2jP
S3Vx1J6HZtP+z9s5//71jznt8CkrnyG38/LsDTvPrZPb9pt2B3bTAedmY6eJ
bSD/89pubaB5jyYRH/3JOd2sz6Hnpa14N5p707Lf0r7Oej5X3ZWo3l8O3fN7
vT8Nubu+241/1pwQufF3aBM5ylorSlRQ1LLaihILqvwFTe9AcRm1oyRAWWlF
SSEjvsvI9f/Gr6QZx43G43ijeVZD6QJluRUlg7Uet6L0YGZGd6C4jNpR+kAZ
tqIMIKPBHShuLELpVH4xdZwhcPotOSNY7Z7jeFk9r6GMYW6yRpTRorFoTmEK
ZUR30+XgGFwm3aD7zLqo2anJaBkySoO0U6WlobtXheYyC9MeKa0PtD7QJkBL
xJ+Uo1XesfLGwOsBbwVmLl7AO1TeFHhZM17sZtOYA2XN/gfWrrLmwOq2YG0r
awdYaQvWXFn7wEoWsXQ9Q6x1ZT0AVtyCtaqsE2BFDVn+j2KirFNgdVqwRso6
q2FN4dyMxO6dSQMl8RUnKUnUsp+1Wn+m/hc8xW4khdete9WbqPell4Vp5Oyo
86JwSsr2aes2J+noevimcN2YJXSR8jmNBtS4Pk5ddPfXP7k5t5s1U/R9q9O7
ttHuW7J7LryGRe5ueuM38YZ9msVCX/W5POxza87jkyvnrT68D4iveP/z5+Y1
OPF+JE7/zdaU/bl5wfuxuHqVcb4G3zDo69f6BkHfoOL7GbLD5xjxDQPz8hM4
8WlMnGOYF+wPnwTFtVwZJ/aWBX2TBT43Lxms+xLLVmp93aBvGpiXC3CmsBLi
XIN5uYD8wq5ZZZyvwJcEfeu1vjjo2wjk9yM4I1gJcc4hv1cwK/j+Kq6tkKtb
unL/vVkc97zMDH94fKnrqTCIfrsyEz+wvswnL97sRb8TmIGXtEmcwx/PLmTA
9Fi0shXdHuhO6Z7rzawj7ntK2h1Bm0Y4MXJNFwdW9LCqRLWwpVJ9XzNEFVWG
aaLkzHoAUay2HalanoCPIIp1vwNVy9sNVglDNUVSyxvrQ4hidXNL1VI/KGrJ
qFpXldRvHkMUK71TVUvdDyuPeE8cq1pqoae0Tcpo7t5VSSM1ZK5O3qiEZxqX
avd3uVQxYz0qVXuqUhKY3cYs/QdzYzq5\
\>"]], ",", 
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {0., 1., 0.}, {0., 1., 
      1.}, {1., 0., 0.5773502691896258}, {1., 0., 1.5773502691896257`}, {1., 
      1., 0.5773502691896258}, {1., 1., 1.5773502691896257`}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     GeometricTransformation[
      Cuboid[{0, 0, 0}], {{1, 0, 0}, {0, 1, 0}, {3^Rational[-1, 2], 0, 1}}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\
eJzVmV1v2zYUhhlZ8rfjb8V2msSJ0zpwkXRL16VZi11uV/0RQWGgvSg6ZL0f
sNv9iv1QYeMhX1vya7qiBazAAFv2Ec/78FAkDyXx3cOXD8tPD18+vn+Y/vr4
8NuHj+9/n/7y+VGfKh0opZb6+9dI6f//6H/mu0ySRKmf9TFcLpczY73Vx0Bb
AX7t2Xt9LGkrwtkLc/YOVlUsKda/+qjOTekP0NTFCm2pdZia8luoWzvKX0Df
Fital6OOM+NzDUYvE4H8Dch+YrwXIA68vK/APqLSEtnHxvsp2GMqDcmeGO8Z
2E+8vKdgn1JpRLapW52CfU6lZbJHxvsY7JmX9xjsZ1RaIdtcMxWDfUWlVbJj
4z0A+7mXdxfsayqtkT003m2wv6PSOtlmZKgm2Lde3nWwX1Fpg+y+8eb4WNUk
24xscw2kjjtv79VZPc9VWX+55S2yu27l5+s/avp3Tt6HZHec6jbUPG7bLjtV
a4UWqpE+VJUdRfurp6ib29lxtWRj/ErkC6h5PnZdV3FL/VKrQ/mzt/JOQkcr
1v3J/b1DJTnenuqTqpGrEnsPVYi6/n506Oq7ddFaZ6Vb2lp++7Qw0IfiSmPv
6JXqJgVjiSmlHIrRb2RQieU1UUKiyEgU95qXOtKHFhS8fvnVH1H9lSRd4/en
lYkmM4fvDfxpFaKJK+fTshepqg8SB2csf3U2Ds4mfpQaUThr+1HqRMm5Hhi7
xSgRxXJPlAZReGX0ozSJwmujH6VFFB5x35Jy6JhT+ZSfVlkfiTGPEnlR2kTh
O7pvSekQhXOMH6XryDFfoWAGMKWnD+5WhFT/G4dyIysVovSJwtnbjzIwKXxN
SexTUeUr2nSUDvVhn9g3lWb1X1fpV2O8pSt76Y4y10ra939RlUj1llQjexth
JOnV/y9Van136FYF6/nmpYzSkaPUuz8T07k/FtGj/lBDhHG7d/TpHLKhSDQR
aDe7aOgtN036vA7CohBBzrZAeFaIUKYWCU2m+7kXjXNKlWhdxHZSiFYj2gC0
cSFanWhHoA2JNqbRHzppDaJNQOsWojWJdgJay0lLR6mb1iLaFLRaIZo89z4F
oUyrA88aN0Geha9ACAoRutQioUkmcZIwCtyknosk7kZpSZxrLSnbi0OizKR1
uiwQY/Vc7UWKiXSmDwKorEiIKc4lHblGpC5rEGmYSxoRKUZMHScp1ZW2SGNX
PtBlAyINckkTR54SwJhI/VySvAttQH1C6h6Nxm31CcVRQovOidTNJZ1mSNIP
5n0nUTq5FGmN1L8gZZt62d03IggdM2LXU0KWlM6pIWLg7M+kVi6ph5g4g81p
XuWTDhETZ445xdTMJTUQk5nzYXono7ZYjVxWFawRZSXJUiK8dBLTPgu2iCHa
eUzECMQLItbdxHRfBBGeEU/WYsm4U+LVnDzBXRChiYh4zlVpjFuCzIZLIrRB
OCZCxU0gNe9T9NEiziBl6kVL4/umKdkxYouJFjlp/ETJV3uC2DhPhk6a9HXg
6LMTUHpEKTkp/PTPe0ZTtLBNtMBJ4/caPD4vEVuTZrabVkMLmTJHTDVnTOmo
PDAUfgfFs3ABmll7y2uafAzJxsJv1Xjf7gYti8wDQkrJtqeVuPf8vkcE5j4i
tFo7rq2O9yR4h/EV6l5d0bRG3o/g99qvM7ao7BXrIk72vsd568WlXJfsJpXW
beCYuU0vE5vR7F38kEr5/eoLeNu32THiYq/n8LK7IdzvPC7mSWbtULzTyu8g
Z/C2O4ATRMBestrLYLB7pzy3eO5NwDT3CLjnSEv5rVIMb7tbvtqXZa8evMye
6lb+4vy2Wj8Xxptzp+vZSrxvjPdFku7t85ok18Dsd6psdl+tlsIwe49mn1t8
3xhLHfwL4Jby3A==\
\>"]], ",", 
   Graphics3DBox[
    InterpretationBox[
     GraphicsComplex3DBox[{{0., 0., 0.}, {0., 0., 1.}, {0., 1., 
      0.5773502691896258}, {0., 1., 1.5773502691896257`}, {1., 0., 0.}, {1., 
      0., 1.}, {1., 1., 0.5773502691896258}, {1., 1., 1.5773502691896257`}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic],
     GeometricTransformation[
      Cuboid[{0, 0, 0}], {{1, 0, 0}, {0, 1, 0}, {0, 3^Rational[-1, 2], 1}}]],
    ImageCache->GraphicsData["CompressedBitmap", "\<\
eJzFmttS21YUhoVOlk8IGxObGjM2EFKYEBqSQEgIvWuu+hBMxjPJRScdmvs+
QJ+id5npM3ravdZeksXvbe1t1U01IFvS/r/1a3vtgw4/3335OP3l7sunD3fj
9/d3v3789OG38U+f79WuYMPzvKn6/2Pgqe9/q2/8P53NZp53q9bhdDo95K0b
tfbVViCfB7z3Wq0DtRXSVkA0Xjx99ErK0tFQHfSoqPpOJSZc4mVB71OBz2e/
e5HmzHihsmMue1GgUaiopOw5cBMqK/7IiM9+c6E3YtUZRGiAKgDVHqtOIVab
VKH4+vPeEE3rvodo6YIO4w1Z9xjidR/q1F8Iuu9YdwjxesX6Yx2dI2p3WTuG
mP1CzZDQN0TVyn2IumtQ8raosK4GTBlC/D0LBWuuz5Rd8LIPlMDiRVMegZcD
pswr0eblEVN64OUIKOgFKTtM6YCXJ5UoKXg5AUqoVjEoe6xsQfynBmUxPubI
NlMaEP8ZUCJDfK2sw15ffDw3EMp8dJmWiA99OFteWFgRsDo5K9tLlEugxBZH
1SjoZcuJUrNQ0hLKvPWsh5KoVQLKzYdKVZhEJmUxPmZMe42UGpzFFVDqFkqr
EqUGlKYTpWHxUo2CXhpOlKZaNUBZd1YW42OOJE6UliG+u7Isfq0ShX4LKg51
ILmJ6rahN6f+Cscp7SUGL69tNLUf22zsRNrMSFIEc2NFypIMc6OkaoWjgruy
rJ18S8qWpS81U66hX0cv1SjoBUcqoEjuVqNE4OUNUDqGmYC7six+ZKTQ70Lz
iSKpa/BgVr9Sagrpf/1rhoSyWZmZdiG0KKMRoDLtXGgJ0LYtc2Az7anQWhZv
brQToaUmb1FOm+krvpqRcSyMbsagT7WswjgUxg4weophqWVpBWMhDAwEvmbL
LWgHITjYF/2QdKHWa8S2I2EohP0FwqKH2EgYCGFCsignFOvTgbIjlCMjpVfI
E6o9M6MrjCeF2qSvun3pJMG2vpyWCu10LbSW0M6AFhhpGjXPG6TVhfbDWmix
0F4ALQRa3keX0gKhXVaicbsJc5qwroEVASt1Yanl5l9zNMqZIznhyomBs7mc
E+UcObu31VlhkUXj2Rtg1YDVXs6Cq20uJZTXa6EGQL0CagLU1n9CbUPLCo3U
EKjYJupGr0g1joorU+dZbabGQH0F1AZQm/8jtQbUl9+E2gRqAzLLTE2Air3s
eqg2r3Unah2oz9WKDrfXQqOxj7qawOI1caI3gH4iXpOv5blQhzYW6P6zsKcF
5GMhN518l5HbQJ5IjaQWz3byJpBH4rlnIdegZdjJu+J5AGTsdezkFMg74nlk
IccrkzvieWLp1WPIvEXyFpDb4vnYMratTq6LZ5x/IjmykjtAjsTzOZBxxrA6
2Rcy9nFIDq1kuuNOZ4/jMM60ImgZvpDm/QTlAl9yFXza5pOhldoUfzjCIxln
vXZyIn6xh7d5DqBVLJJD8Yz9m83zEvKDWSPJMYNxFolc38glbM9CwuscH/JJ
k+jeZ9+FVMtJxEEKjDvoDefdvpWIs0QjMcqJtMWNw3SO/GTNUGM3Roa5nvBu
VdfEenBvwMzBO7LYO7xbwpm3hg3mZM/Z0Me7Qk17S9V4/x7HglvukHIK/URM
0OfQdFFTvWqpTl45/xZoceS8LfwmHi9zbVvOuSzevL71maYQD9v1j1l950Kt
27LoKP/wfQyt7IhLVFyJIpLjuvQ2xMEZ24Wosh5Pq3oW1ZmoKNuirAfmJ9UU
2TQrzEpTVegnKX2IgX3tEaj0U70BqPB6YQwq/Yw2e6sAYwyhtH7WPoQYeK3X
B5V+i2LPouqKKpY6G7FqBCocy2huE4mKYum3drI3NjBGAs6OufQYYuA9gRBU
p6yalKg4MQuKZ6w4mMGzLp3zufcLLnVU4GbzDqqPSz76WLbe8pa38Q/mOpET
\
\>"]]}], "}"}]], "Output",
 ImageSize->{341, 262},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 Graphics3DBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[5]=",
 CellID->123075144]
}, Open  ]],

Cell["All shearings in 4D:", "ExampleText",
 CellID->195264725],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", " ", 
  RowBox[{"MatrixForm", ",", " ", 
   RowBox[{"Shearings", "[", 
    RowBox[{"\[Theta]", ",", "4"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->64404309],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{475, 177},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->36932514]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "The determinant of a shearing matrix is ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 "; hence it preserves areas and volumes: "
}], "ExampleText",
 CellID->4680996],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", " ", 
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->206509648],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->66851060]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->48090674],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox["v", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingMatrix"], "[", 
   RowBox[{
    RowBox[{"-", 
     StyleBox["\[Theta]", "TR"]}], ",", 
    StyleBox["v", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->879284386],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], ".", 
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{
    RowBox[{"-", "\[Theta]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->199982771],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->438585023]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->444088356],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox["v", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " is also given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingMatrix"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox[
     RowBox[{"-", "v"}], "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->191977483],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], ".", 
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"-", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->246196084],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->53162828]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12973788],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " ",
 "power of a shearing matrix is again a shearing matrix with the same ",
 Cell[BoxData[
  FormBox["v", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->224705175],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"MatrixPower", "[", 
   RowBox[{
    RowBox[{"ShearingMatrix", "[", 
     RowBox[{"a", ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "]"}], ",", "n"}], "]"}], "\[Equal]",
   
  RowBox[{"ShearingMatrix", "[", 
   RowBox[{
    RowBox[{"ArcTan", "[", 
     RowBox[{"n", " ", 
      RowBox[{"Tan", "[", "a", "]"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->118419526],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->33352095]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["The order in which shearings are applied is significant: ", \
"ExampleText",
 CellID->71053513],

Cell[BoxData[{
 RowBox[{
  RowBox[{"t1", "=", 
   RowBox[{"ShearingMatrix", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"t2", "=", 
   RowBox[{"ShearingMatrix", "[", 
    RowBox[{"\[Phi]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->61504335],

Cell["Here the two different orders do not yield the same matrix: ", \
"ExampleText",
 CellID->491676387],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"t1", ".", "t2"}], "-", 
   RowBox[{"t2", ".", "t1"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->271203192],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     FractionBox[
      RowBox[{
       RowBox[{"Tan", "[", "\[Theta]", "]"}], " ", 
       RowBox[{"Tan", "[", "\[Phi]", "]"}]}], 
      RowBox[{"2", " ", 
       SqrtBox["2"]}]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     FractionBox[
      RowBox[{
       RowBox[{"Tan", "[", "\[Theta]", "]"}], " ", 
       RowBox[{"Tan", "[", "\[Phi]", "]"}]}], 
      RowBox[{"2", " ", 
       SqrtBox["2"]}]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{381, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->101781964]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->552109279],

Cell[TextData[{
 "The transformation is not defined for angles ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"cos", " ", "a"}], "\[LongEqual]", "0"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->28036276],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{"ShearingMatrix", "[", 
    RowBox[{"a", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", 
   RowBox[{"a", "\[Rule]", 
    RowBox[{"Pi", "/", "2"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->95280454],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", 
     RowBox[{"-", "\[Infinity]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{222, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->249980791]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->496401621],

Cell["The transformation applied to a sphere:", "ExampleText",
 CellID->83677190],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"Sphere", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}], ",", 
       RowBox[{"ShearingMatrix", "[", 
        RowBox[{
         RowBox[{"Pi", "/", "3"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Cos", "[", "u", "]"}], ",", "0", ",", 
           RowBox[{"Sin", "[", "u", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"u", ",", 
       RowBox[{"2", 
        RowBox[{"Pi", "/", "5"}], 
        RowBox[{"Range", "[", "5", "]"}]}]}], "}"}]}], "]"}], ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->112139715],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 206},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->141557106]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ShearingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ScalingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ScalingMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RotationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Constructing Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ConstructingMatrices"]], "Tutorials",
 CellID->12359]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->117052731],

Cell[TextData[ButtonBox["Constructing Matrices",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ConstructingMatrices"]], "MoreAbout",
 CellID->42719829],

Cell[TextData[ButtonBox["Geometric Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeometricTransforms"]], "MoreAbout",
 CellID->6610238],

Cell[TextData[ButtonBox["New in 6.0: Matrix & Linear Algebra Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60MatrixAndLinearAlgebraFunctions"]], "MoreAbout",
 CellID->312982407]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ShearingMatrix - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 20, 14.2912646}", "context" -> "System`", 
    "keywords" -> {
     "affine transform", "area preserving transform", 
      "geometric transformation", "shearing", "slanting transform", 
      "tilting transform", "transvection", "volume preserving transform"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "ShearingMatrix[\\[Theta], v, n] gives the matrix corresponding to \
shearing by \\[Theta] radians along the direction of the vector v, and normal \
to the vector n. ", "synonyms" -> {"shearing matrix"}, "title" -> 
    "ShearingMatrix", "type" -> "Symbol", "uri" -> "ref/ShearingMatrix"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7814, 240, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->177797755]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 114183, 2765}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2417, 59, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3040, 86, 1579, 37, 70, "ObjectNameGrid"],
Cell[4622, 125, 902, 27, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5561, 157, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6039, 174, 252, 7, 70, "Notes",
 CellID->798539111],
Cell[6294, 183, 280, 8, 70, "Notes",
 CellID->25828702],
Cell[6577, 193, 675, 23, 70, "Notes",
 CellID->3268604],
Cell[7255, 218, 522, 17, 70, "Notes",
 CellID->1067943069]
}, Closed]],
Cell[CellGroupData[{
Cell[7814, 240, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->177797755],
Cell[CellGroupData[{
Cell[8199, 254, 148, 5, 70, "ExampleSection",
 CellID->172718204],
Cell[8350, 261, 228, 9, 70, "ExampleText",
 CellID->259330806],
Cell[CellGroupData[{
Cell[8603, 274, 307, 10, 28, "Input",
 CellID->435464200],
Cell[8913, 286, 604, 14, 63, "Output",
 Evaluatable->False,
 CellID->66711372]
}, Open  ]],
Cell[9532, 303, 125, 3, 70, "ExampleDelimiter",
 CellID->797551938],
Cell[9660, 308, 174, 6, 70, "ExampleText",
 CellID->328163910],
Cell[CellGroupData[{
Cell[9859, 318, 508, 15, 70, "Input",
 CellID->417290019],
Cell[10370, 335, 325, 10, 145, "Output",
 CellID->477541236]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10744, 351, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[10968, 360, 146, 6, 70, "ExampleText",
 CellID->172695089],
Cell[CellGroupData[{
Cell[11139, 370, 285, 9, 70, "Input",
 CellID->91127034],
Cell[11427, 381, 605, 14, 70, "Output",
 Evaluatable->False,
 CellID->108488351]
}, Open  ]],
Cell[12047, 398, 146, 6, 70, "ExampleText",
 CellID->130502387],
Cell[CellGroupData[{
Cell[12218, 408, 285, 9, 70, "Input",
 CellID->12557635],
Cell[12506, 419, 620, 14, 70, "Output",
 Evaluatable->False,
 CellID->10251281]
}, Open  ]],
Cell[13141, 436, 125, 3, 70, "ExampleDelimiter",
 CellID->365787824],
Cell[13269, 441, 266, 11, 70, "ExampleText",
 CellID->307503498],
Cell[CellGroupData[{
Cell[13560, 456, 309, 10, 70, "Input",
 CellID->104664616],
Cell[13872, 468, 766, 24, 78, "Output",
 CellID->102174814]
}, Open  ]],
Cell[14653, 495, 263, 11, 70, "ExampleText",
 CellID->330765079],
Cell[CellGroupData[{
Cell[14941, 510, 307, 10, 70, "Input",
 CellID->3655249],
Cell[15251, 522, 766, 24, 78, "Output",
 CellID->227445149]
}, Open  ]],
Cell[16032, 549, 125, 3, 70, "ExampleDelimiter",
 CellID->368966718],
Cell[16160, 554, 504, 20, 70, "ExampleText",
 CellID->686240600],
Cell[CellGroupData[{
Cell[16689, 578, 307, 10, 70, "Input",
 CellID->29672303],
Cell[16999, 590, 1115, 37, 96, "Output",
 CellID->346680294]
}, Open  ]],
Cell[18129, 630, 125, 3, 70, "ExampleDelimiter",
 CellID->734046779],
Cell[18257, 635, 80, 1, 70, "ExampleText",
 CellID->479507293],
Cell[18340, 638, 620, 20, 70, "Input",
 CellID->419323043],
Cell[CellGroupData[{
Cell[18985, 662, 690, 23, 70, "Input",
 CellID->150504143],
Cell[19678, 687, 681, 16, 139, "Output",
 CellID->282362206]
}, Open  ]],
Cell[20374, 706, 124, 3, 70, "ExampleDelimiter",
 CellID->30645500],
Cell[20501, 711, 80, 1, 70, "ExampleText",
 CellID->137131831],
Cell[20584, 714, 637, 20, 70, "Input",
 CellID->367829949],
Cell[CellGroupData[{
Cell[21246, 738, 823, 26, 70, "Input",
 CellID->167437831],
Cell[22072, 766, 5756, 110, 221, 1218, 31, "CachedBoxData", "BoxData", \
"Output",
 CellID->188616850]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27877, 882, 227, 7, 70, "ExampleSection",
 CellID->347222277],
Cell[28107, 891, 83, 1, 70, "ExampleText",
 CellID->70266796],
Cell[CellGroupData[{
Cell[28215, 896, 875, 26, 70, "Input",
 CellID->252614726],
Cell[29093, 924, 11453, 192, 70, "Output",
 Evaluatable->False,
 CellID->178342705]
}, Open  ]],
Cell[40561, 1119, 125, 3, 70, "ExampleDelimiter",
 CellID->244509174],
Cell[40689, 1124, 205, 7, 70, "ExampleText",
 CellID->563715391],
Cell[40897, 1133, 1091, 33, 70, "Input",
 CellID->344720790],
Cell[41991, 1168, 63, 1, 70, "ExampleText",
 CellID->659888018],
Cell[CellGroupData[{
Cell[42079, 1173, 210, 6, 70, "Input",
 CellID->112862591],
Cell[42292, 1181, 895, 19, 70, "Output",
 Evaluatable->False,
 CellID->472073084]
}, Open  ]],
Cell[CellGroupData[{
Cell[43224, 1205, 681, 22, 70, "Input",
 CellID->73998204],
Cell[43908, 1229, 734, 22, 133, "Output",
 CellID->27517904]
}, Open  ]],
Cell[44657, 1254, 62, 1, 70, "ExampleText",
 CellID->99252900],
Cell[CellGroupData[{
Cell[44744, 1259, 210, 6, 70, "Input",
 CellID->558245758],
Cell[44957, 1267, 2489, 45, 70, "Output",
 Evaluatable->False,
 CellID->543224402]
}, Open  ]],
Cell[CellGroupData[{
Cell[47483, 1317, 382, 12, 70, "Input",
 CellID->131239154],
Cell[47868, 1331, 14108, 256, 283, "Output",
 CellID->123075144]
}, Open  ]],
Cell[61991, 1590, 63, 1, 70, "ExampleText",
 CellID->195264725],
Cell[CellGroupData[{
Cell[62079, 1595, 209, 6, 70, "Input",
 CellID->64404309],
Cell[62291, 1603, 4486, 78, 70, "Output",
 Evaluatable->False,
 CellID->36932514]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[66826, 1687, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[67067, 1696, 205, 6, 70, "ExampleText",
 CellID->4680996],
Cell[CellGroupData[{
Cell[67297, 1706, 313, 9, 70, "Input",
 CellID->206509648],
Cell[67613, 1717, 157, 5, 36, "Output",
 CellID->66851060]
}, Open  ]],
Cell[67785, 1725, 124, 3, 70, "ExampleDelimiter",
 CellID->48090674],
Cell[67912, 1730, 650, 24, 70, "ExampleText",
 CellID->879284386],
Cell[CellGroupData[{
Cell[68587, 1758, 520, 16, 70, "Input",
 CellID->199982771],
Cell[69110, 1776, 391, 13, 36, "Output",
 CellID->438585023]
}, Open  ]],
Cell[69516, 1792, 125, 3, 70, "ExampleDelimiter",
 CellID->444088356],
Cell[69644, 1797, 655, 24, 70, "ExampleText",
 CellID->191977483],
Cell[CellGroupData[{
Cell[70324, 1825, 522, 16, 70, "Input",
 CellID->246196084],
Cell[70849, 1843, 390, 13, 36, "Output",
 CellID->53162828]
}, Open  ]],
Cell[71254, 1859, 124, 3, 70, "ExampleDelimiter",
 CellID->12973788],
Cell[71381, 1864, 431, 16, 70, "ExampleText",
 CellID->224705175],
Cell[CellGroupData[{
Cell[71837, 1884, 625, 21, 70, "Input",
 CellID->118419526],
Cell[72465, 1907, 160, 5, 36, "Output",
 CellID->33352095]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[72674, 1918, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[72908, 1927, 101, 2, 70, "ExampleText",
 CellID->71053513],
Cell[73012, 1931, 612, 19, 70, "Input",
 CellID->61504335],
Cell[73627, 1952, 105, 2, 70, "ExampleText",
 CellID->491676387],
Cell[CellGroupData[{
Cell[73757, 1958, 175, 6, 70, "Input",
 CellID->271203192],
Cell[73935, 1966, 745, 25, 55, "Output",
 CellID->101781964]
}, Open  ]],
Cell[74695, 1994, 125, 3, 70, "ExampleDelimiter",
 CellID->552109279],
Cell[74823, 1999, 317, 12, 70, "ExampleText",
 CellID->28036276],
Cell[CellGroupData[{
Cell[75165, 2015, 382, 12, 70, "Input",
 CellID->95280454],
Cell[75550, 2029, 422, 14, 36, "Output",
 CellID->249980791]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[76021, 2049, 228, 7, 70, "ExampleSection",
 CellID->496401621],
Cell[76252, 2058, 81, 1, 70, "ExampleText",
 CellID->83677190],
Cell[CellGroupData[{
Cell[76358, 2063, 897, 26, 70, "Input",
 CellID->112139715],
Cell[77258, 2091, 33067, 546, 70, "Output",
 Evaluatable->False,
 CellID->141557106]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[110386, 2644, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[110705, 2655, 740, 26, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[111482, 2686, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[111803, 2697, 152, 3, 70, "Tutorials",
 CellID->12359]
}, Open  ]],
Cell[CellGroupData[{
Cell[111992, 2705, 319, 9, 70, "MoreAboutSection",
 CellID->117052731],
Cell[112314, 2716, 152, 3, 70, "MoreAbout",
 CellID->42719829],
Cell[112469, 2721, 149, 3, 70, "MoreAbout",
 CellID->6610238],
Cell[112621, 2726, 198, 4, 70, "MoreAbout",
 CellID->312982407]
}, Open  ]],
Cell[112834, 2733, 27, 0, 70, "History"],
Cell[112864, 2735, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

