(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     67294,       1989]
NotebookOptionsPosition[     59095,       1714]
NotebookOutlinePosition[     60715,       1756]
CellTagsIndexPosition[     60629,       1751]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ShearingMatrix" :> 
          Documentation`HelpLookup["paclet:ref/ShearingMatrix"], 
          "GeometricTransformation" :> 
          Documentation`HelpLookup["paclet:ref/GeometricTransformation"], 
          "TransformationFunction" :> 
          Documentation`HelpLookup["paclet:ref/TransformationFunction"], 
          "AffineTransform" :> 
          Documentation`HelpLookup["paclet:ref/AffineTransform"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"ShearingMatrix\"\>", 
       2->"\<\"GeometricTransformation\"\>", 
       3->"\<\"TransformationFunction\"\>", 
       4->"\<\"AffineTransform\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Geometric Transforms" :> 
          Documentation`HelpLookup["paclet:guide/GeometricTransforms"], 
          "Graphics Transformations" :> 
          Documentation`HelpLookup[
           "paclet:guide/GraphicsTransformations"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Geometric Transforms\"\>", 
       2->"\<\"Graphics Transformations\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["ShearingTransform", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ShearingTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ShearingTransform"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        StyleBox["v", "TI"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives a ",
     Cell[BoxData[
      ButtonBox["TransformationFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
     " that represents a shear by ",
     Cell[BoxData[
      StyleBox["\[Theta]", "TR"]], "InlineFormula"],
     " radians along the direction of the vector ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     ", normal to the vector ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     ", and keeping the origin fixed."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ShearingTransform",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ShearingTransform"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        StyleBox["v", "TI"], ",", 
        StyleBox["n", "TI"], ",", 
        StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a shear that keeps the point ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     " fixed, rather than the origin."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ShearingTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingTransform"]], "InlineFormula"],
 " gives a ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " which can be applied to vectors."
}], "Notes",
 CellID->108862120],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ShearingTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingTransform"]], "InlineFormula"],
 " works in any number of dimensions, and always gives area- or \
volume-preserving transformations."
}], "Notes",
 CellID->25828702],

Cell[TextData[{
 "In 2D, ",
 Cell[BoxData[
  ButtonBox["ShearingTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingTransform"]], "InlineFormula"],
 " turns rectangles into parallelograms. ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}]}], "]"}]], "InlineFormula"],
 " effectively represents slanting by angle ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " to the right."
}], "Notes",
 CellID->3268604],

Cell[TextData[{
 "In 3D, ",
 Cell[BoxData[
  ButtonBox["ShearingTransform",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ShearingTransform"]], "InlineFormula"],
 " does the analog of shearing a deck of cards by angle ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in the direction ",
 Cell[BoxData[
  StyleBox["v", "TI"]], "InlineFormula"],
 ", with the cards oriented so as to have normal vector ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ", and the card that goes through the point ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " kept fixed."
}], "Notes",
 CellID->1067943069]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->248578196],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->18107572],

Cell[TextData[{
 "Shearing by ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " radians along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis:"
}], "ExampleText",
 CellID->551152158],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ShearingTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->88087489],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", 
      RowBox[{"Tan", "[", "\[Theta]", "]"}], "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{265, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->159331614]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->114322766],

Cell[TextData[{
 "Apply a 30\[Degree] shearing along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axes to the unit rectangle:"
}], "ExampleText",
 CellID->386741356],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"ShearingTransform", "[", 
      RowBox[{
       RowBox[{"30", "Degree"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}], ",", " ", 
   RowBox[{"Frame", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9065949],

Cell[BoxData[
 GraphicsBox[
  GeometricTransformationBox[RectangleBox[{0, 0}], 
   NCache[{{{1, 3^Rational[-1, 2]}, {0, 1}}, {0, 
     0}}, {{{1, 0.577350269189626}, {0, 1}}, {0, 0}}]],
  Frame->True]], "Output",
 ImageSize->{184, 124},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->127883766]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[TextData[{
 "Simple shearing along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis: "
}], "ExampleText",
 CellID->168693960],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ShearingTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->38104388],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0"},
     {
      RowBox[{"Tan", "[", "\[Theta]", "]"}], "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{265, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17029958]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->180011273],

Cell[TextData[{
 "Simple shearing along the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " axis in the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], ",", 
   StyleBox["y", "TI"]}]], "InlineFormula"],
 " plane: "
}], "ExampleText",
 CellID->327422431],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ShearingTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->429813330],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", 
      RowBox[{"Tan", "[", "\[Theta]", "]"}], "0"},
     {"0", "1", "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{280, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->246819892]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->499986326],

Cell[TextData[{
 "Shearing along the ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " axis in the plane ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", "\[LongEqual]", "1"}], TraditionalForm]], "InlineMath"],
 ": "
}], "ExampleText",
 CellID->211418613],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"ShearingTransform", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->627926031],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", 
      RowBox[{"Tan", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Tan", "[", "\[Theta]", "]"}]}]},
     {"0", "1", "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{323, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2933133]
}, Open  ]],

Cell["Points in the shearing plane are not changed: ", "ExampleText",
 CellID->680822188],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "[", 
  RowBox[{"{", 
   RowBox[{"0", ",", "0", ",", "1"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->92155531],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "1"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->16529001]
}, Open  ]],

Cell["\<\
Points outside the shearing plane are moved in the shearing direction: \
\>", "ExampleText",
 CellID->12738024],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "[", 
  RowBox[{"{", 
   RowBox[{"0", ",", "0", ",", "2"}], "}"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->179092538],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Tan", "[", "\[Theta]", "]"}], ",", "0", ",", "2"}], 
  "}"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->363114554]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->20264459],

Cell["Transformation applied to a 2D shape:", "ExampleText",
 CellID->479507293],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rectangle", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->419323043],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
    ",", 
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}], 
      ",", 
      RowBox[{"ShearingTransform", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "Pi"}], "/", "6"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->150504143],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], RectangleBox[{0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
     {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, GeometricTransformationBox[
    {RGBColor[1, 0, 0], Opacity[0.85], RectangleBox[{0, 0}], 
     {AbsolutePointSize[10], Opacity[1], 
      {RGBColor[1, 0, 1], PointBox[{0, 0}]}, 
      {RGBColor[0, 1, 0], PointBox[{1, 1}]}}}, 
    NCache[{{{1, -3^Rational[-1, 2]}, {0, 1}}, {
      3^Rational[-1, 2], 0}}, {{{1, -0.5773502691896258}, {0, 1}}, {
      0.5773502691896258, 0}}]]}]], "Output",
 ImageSize->{184, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->90886973]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->645421091],

Cell["Transformation applied to a 3D shape:", "ExampleText",
 CellID->137131831],

Cell[BoxData[
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Cuboid", "[", "]"}], ",", 
     RowBox[{"AbsolutePointSize", "[", "10", "]"}], ",", 
     RowBox[{"Opacity", "[", "1", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"Magenta", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"Green", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], "]"}]}], "}"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->367829949],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Opacity", "[", ".35", "]"}], ",", "Blue", ",", "gr"}], "}"}], 
     ",", 
     RowBox[{"GeometricTransformation", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Opacity", "[", ".85", "]"}], ",", "Red", ",", "gr"}], "}"}],
        ",", 
       RowBox[{"ShearingTransform", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "Pi"}], "/", "6"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}], "}"}],
    ",", 
   RowBox[{"Boxed", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->167437831],

Cell[BoxData[
 Graphics3DBox[{
   {RGBColor[0, 0, 1], Opacity[0.35], CuboidBox[{0, 0, 0}], 
    {AbsolutePointSize[10], Opacity[1], 
     {RGBColor[1, 0, 1], Point3DBox[{0, 0, 0}]}, 
     {RGBColor[0, 1, 0], Point3DBox[{1, 1, 1}]}}}, 
   InterpretationBox[
    {RGBColor[1., 0., 0.], Opacity[0.85], 
     GraphicsComplex3DBox[{{0.5773502691896258, 0., 0.}, {0., 0., 1.}, {
      0.5773502691896258, 1., 0.}, {0., 1., 1.}, {1.5773502691896257`, 0., 
      0.}, {1., 0., 1.}, {1.5773502691896257`, 1., 0.}, {1., 1., 1.}}, 
      Polygon3DBox[{{1, 3, 7, 5}, {1, 5, 6, 2}, {2, 6, 8, 4}, {3, 4, 8, 7}, {
       1, 2, 4, 3}, {5, 7, 8, 6}}],
      VertexColors->Automatic,
      VertexNormals->Automatic], 
     {AbsolutePointSize[10.], Opacity[1.], 
      {RGBColor[1., 0., 1.], Point3DBox[{0.5773502691896258, 0., 0.}]}, 
      {RGBColor[0., 1., 0.], Point3DBox[{1., 1., 1.}]}}},
    GeometricTransformation[{
      Opacity[0.85], 
      RGBColor[1, 0, 0], {
       Cuboid[{0, 0, 0}], 
       AbsolutePointSize[10], 
       Opacity[1], {
        RGBColor[1, 0, 1], 
        Point[{0, 0, 0}]}, {
        RGBColor[0, 1, 0], 
        Point[{1, 1, 1}]}}}, {{{1, 0, -3^Rational[-1, 2]}, {0, 1, 0}, {0, 0, 
       1}}, {3^Rational[-1, 2], 0, 0}}]]},
  Boxed->False,
  ImageSize->{180., 162.38829285282466`},
  ViewPoint->{0.7623693356633026, -2.7814789538918876`, 1.7697931588456273`},
  ViewVertical->{-0.025250357647173052`, -0.14610050047836595`, 
   1.5700643914948431`}]], "Output",
 ImageSize->{184, 162},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->108677996,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->452853803],

Cell[TextData[{
 "Transforming the output of ",
 Cell[BoxData[
  ButtonBox["Plot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Plot"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->106838819],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Plot", "[", 
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{
       RowBox[{"-", "2"}], "Pi"}], ",", 
      RowBox[{"2", "Pi"}]}], "}"}]}], "]"}], "/.", 
  RowBox[{"L_Line", "\[RuleDelayed]", 
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{"L", ",", 
     RowBox[{"ShearingTransform", "[", 
      RowBox[{
       RowBox[{"Pi", "/", "4"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->230274799],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->312496101]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->457300817],

Cell["Construct a slanted font from an upright font by shearing: ", \
"ExampleText",
 CellID->1161840],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{
     RowBox[{"Style", "[", 
      RowBox[{
       RowBox[{"Text", "[", 
        RowBox[{"StringJoin", "@", 
         RowBox[{"CharacterRange", "[", 
          RowBox[{
          "\"\<\[DoubleStruckCapitalA]\>\"", ",", 
           "\"\<\[DoubleStruckCapitalO]\>\""}], "]"}]}], "]"}], ",", "50"}], 
      "]"}], ",", 
     RowBox[{"ShearingTransform", "[", 
      RowBox[{
       RowBox[{"25", " ", "Degree"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}], ",", 
   RowBox[{"ImageSize", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"All", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "50"}], "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->50444804],

Cell[BoxData[
 GraphicsBox[GeometricTransformationBox[
   StyleBox[
    InsetBox["\<\"\[DoubleStruckCapitalA]\[DoubleStruckCapitalB]\
\[DoubleStruckCapitalC]\[DoubleStruckCapitalD]\[DoubleStruckCapitalE]\
\[DoubleStruckCapitalF]\[DoubleStruckCapitalG]\[DoubleStruckCapitalH]\
\[DoubleStruckCapitalI]\[DoubleStruckCapitalJ]\[DoubleStruckCapitalK]\
\[DoubleStruckCapitalL]\[DoubleStruckCapitalM]\[DoubleStruckCapitalN]\
\[DoubleStruckCapitalO]\"\>", {0, 0}],
    StripOnInput->False,
    FontSize->50], 
   NCache[{{{1, Tan[25 Degree]}, {0, 1}}, {0, 
     0}}, {{{1, 0.4663076581549986}, {0, 1}}, {0, 0}}]],
  ImageSize->{All, {0, 50}}]], "Output",
 ImageSize->{519, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->243900799]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox["v", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 "is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingTransform"], "[", 
   RowBox[{
    RowBox[{"-", 
     StyleBox["\[Theta]", "TR"]}], ",", 
    StyleBox["v", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->118934125],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"ShearingTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "]"}], ",", 
    RowBox[{"ShearingTransform", "[", 
     RowBox[{
      RowBox[{"-", "\[Theta]"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->407662742],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->263206571]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->219256165],

Cell[TextData[{
 "The inverse of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    StyleBox["v", "TI"], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 " is given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ShearingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingTransform"], "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"-", 
     StyleBox["v", "TI"]}], ",", 
    StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->479816723],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{
    RowBox[{"ShearingTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "]"}], ",", 
    RowBox[{"ShearingTransform", "[", 
     RowBox[{"\[Theta]", ",", 
      RowBox[{"-", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0"}], "}"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], "]"}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->793759033],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}}], ")"}],
   "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->29651460]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->781155394],

Cell["\<\
Performing the shearing transform multiple times corresponds to a single \
shearing transform: \
\>", "ExampleText",
 CellID->666049620],

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{"ShearingTransform", "[", 
    RowBox[{
     RowBox[{"Pi", "/", "4"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->65505422],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{"t", ",", "t"}], "]"}], "\[Equal]", 
  RowBox[{"ShearingTransform", "[", 
   RowBox[{
    RowBox[{"ArcTan", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->887831827],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->310996527]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["The order in which shearings are applied is significant: ", \
"ExampleText",
 CellID->3326907],

Cell[BoxData[{
 RowBox[{
  RowBox[{"t1", "=", 
   RowBox[{"ShearingTransform", "[", 
    RowBox[{"\[Theta]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"t2", "=", 
   RowBox[{"ShearingTransform", "[", 
    RowBox[{"\[Phi]", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->172289097],

Cell["\<\
Applying the two shearings in different orders is not equivalent: \
\>", "ExampleText",
 CellID->58391003],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"Composition", "[", 
     RowBox[{"t1", ",", "t2"}], "]"}], "[", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], "]"}], "-", 
   RowBox[{
    RowBox[{"Composition", "[", 
     RowBox[{"t2", ",", "t1"}], "]"}], "[", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], "]"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13135961],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "-", 
       RowBox[{"2", " ", "y"}]}], ")"}], " ", 
     RowBox[{"Tan", "[", "\[Theta]", "]"}], " ", 
     RowBox[{"Tan", "[", "\[Phi]", "]"}]}], 
    SqrtBox["2"]], ",", 
   RowBox[{"-", 
    FractionBox[
     RowBox[{"y", " ", 
      RowBox[{"Tan", "[", "\[Theta]", "]"}], " ", 
      RowBox[{"Tan", "[", "\[Phi]", "]"}]}], 
     SqrtBox["2"]]}]}], "}"}]], "Output",
 ImageSize->{283, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->557172428]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25754592],

Cell[TextData[{
 "The transformation is not defined for angles ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"cos", " ", "a"}], "\[LongEqual]", "0"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->28036276],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"TransformationMatrix", "[", 
   RowBox[{"ShearingTransform", "[", 
    RowBox[{"a", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], "]"}], "]"}], "/.", 
  RowBox[{"a", "\[Rule]", 
   RowBox[{"Pi", "/", "2"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->524424269],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "ComplexInfinity", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{311, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->178354127]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->713270274],

Cell["Transforming text:", "ExampleText",
 CellID->283882469],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"GeometricTransformation", "[", 
     RowBox[{
      RowBox[{"Style", "[", 
       RowBox[{
        RowBox[{"Text", "[", "\"\<Mathematica\>\"", "]"}], ",", "35"}], "]"}],
       ",", 
      RowBox[{"ShearingTransform", "[", 
       RowBox[{
        RowBox[{"2", 
         RowBox[{"Pi", "/", "5"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Cos", "[", "u", "]"}], ",", 
          RowBox[{"Sin", "[", "u", "]"}]}], "}"}]}], "]"}]}], "]"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{"u", ",", 
      RowBox[{
       RowBox[{"Pi", "/", "5"}], 
       RowBox[{"Range", "[", "10", "]"}]}]}], "}"}]}], "]"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->680045736],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->228706433]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ShearingMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ShearingMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GeometricTransformation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricTransformation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TransformationFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TransformationFunction"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AffineTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AffineTransform"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with ShearingTransform",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
ShearingTransform"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->65781407],

Cell[TextData[ButtonBox["Geometric Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeometricTransforms"]], "MoreAbout",
 CellID->43718440],

Cell[TextData[ButtonBox["Graphics Transformations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GraphicsTransformations"]], "MoreAbout",
 CellID->559902897]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ShearingTransform - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 20, 16.5256968}", "context" -> "System`", 
    "keywords" -> {
     "affine transform", "geometric transformation", "slanting", "tilting", 
      "transvection", "volume-preserving transform"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "ShearingTransform[\\[Theta], v, n] gives a TransformationFunction that \
represents a shear by \\[Theta] radians along the direction of the vector v, \
normal to the vector n, and keeping the origin fixed. \
ShearingTransform[\\[Theta], v, n, p] gives a shear that keeps the point p \
fixed, rather than the origin.", "synonyms" -> {"shearing transform"}, 
    "title" -> "ShearingTransform", "type" -> "Symbol", "uri" -> 
    "ref/ShearingTransform"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8212, 252, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->248578196]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 60486, 1744}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1794, 42, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2417, 69, 1582, 37, 70, "ObjectNameGrid"],
Cell[4002, 108, 1642, 48, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5681, 161, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6159, 178, 383, 12, 70, "Notes",
 CellID->108862120],
Cell[6545, 192, 286, 8, 70, "Notes",
 CellID->25828702],
Cell[6834, 202, 700, 23, 70, "Notes",
 CellID->3268604],
Cell[7537, 227, 638, 20, 70, "Notes",
 CellID->1067943069]
}, Closed]],
Cell[CellGroupData[{
Cell[8212, 252, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->248578196],
Cell[CellGroupData[{
Cell[8597, 266, 147, 5, 70, "ExampleSection",
 CellID->18107572],
Cell[8747, 273, 226, 9, 70, "ExampleText",
 CellID->551152158],
Cell[CellGroupData[{
Cell[8998, 286, 250, 8, 28, "Input",
 CellID->88087489],
Cell[9251, 296, 446, 15, 64, "Output",
 CellID->159331614]
}, Open  ]],
Cell[9712, 314, 125, 3, 70, "ExampleDelimiter",
 CellID->114322766],
Cell[9840, 319, 187, 6, 70, "ExampleText",
 CellID->386741356],
Cell[CellGroupData[{
Cell[10052, 329, 504, 15, 70, "Input",
 CellID->9065949],
Cell[10559, 346, 344, 10, 145, "Output",
 CellID->127883766]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10952, 362, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[11176, 371, 153, 6, 70, "ExampleText",
 CellID->168693960],
Cell[CellGroupData[{
Cell[11354, 381, 250, 8, 70, "Input",
 CellID->38104388],
Cell[11607, 391, 445, 15, 64, "Output",
 CellID->17029958]
}, Open  ]],
Cell[12067, 409, 125, 3, 70, "ExampleDelimiter",
 CellID->180011273],
Cell[12195, 414, 273, 11, 70, "ExampleText",
 CellID->327422431],
Cell[CellGroupData[{
Cell[12493, 429, 271, 8, 70, "Input",
 CellID->429813330],
Cell[12767, 439, 488, 16, 80, "Output",
 CellID->246819892]
}, Open  ]],
Cell[13270, 458, 125, 3, 70, "ExampleDelimiter",
 CellID->499986326],
Cell[13398, 463, 271, 10, 70, "ExampleText",
 CellID->211418613],
Cell[CellGroupData[{
Cell[13694, 477, 370, 11, 70, "Input",
 CellID->627926031],
Cell[14067, 490, 550, 18, 80, "Output",
 CellID->2933133]
}, Open  ]],
Cell[14632, 511, 89, 1, 70, "ExampleText",
 CellID->680822188],
Cell[CellGroupData[{
Cell[14746, 516, 153, 5, 70, "Input",
 CellID->92155531],
Cell[14902, 523, 212, 7, 36, "Output",
 CellID->16529001]
}, Open  ]],
Cell[15129, 533, 121, 3, 70, "ExampleText",
 CellID->12738024],
Cell[CellGroupData[{
Cell[15275, 540, 154, 5, 70, "Input",
 CellID->179092538],
Cell[15432, 547, 254, 9, 36, "Output",
 CellID->363114554]
}, Open  ]],
Cell[15701, 559, 124, 3, 70, "ExampleDelimiter",
 CellID->20264459],
Cell[15828, 564, 80, 1, 70, "ExampleText",
 CellID->479507293],
Cell[15911, 567, 620, 20, 70, "Input",
 CellID->419323043],
Cell[CellGroupData[{
Cell[16556, 591, 787, 26, 70, "Input",
 CellID->150504143],
Cell[17346, 619, 742, 17, 140, "Output",
 CellID->90886973]
}, Open  ]],
Cell[18103, 639, 125, 3, 70, "ExampleDelimiter",
 CellID->645421091],
Cell[18231, 644, 80, 1, 70, "ExampleText",
 CellID->137131831],
Cell[18314, 647, 637, 20, 70, "Input",
 CellID->367829949],
Cell[CellGroupData[{
Cell[18976, 671, 903, 28, 70, "Input",
 CellID->167437831],
Cell[19882, 701, 5099, 97, 183, 1476, 33, "CachedBoxData", "BoxData", \
"Output",
 CellID->108677996]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25030, 804, 227, 7, 70, "ExampleSection",
 CellID->452853803],
Cell[25260, 813, 202, 8, 70, "ExampleText",
 CellID->106838819],
Cell[CellGroupData[{
Cell[25487, 825, 654, 21, 70, "Input",
 CellID->230274799],
Cell[26144, 848, 3319, 58, 70, "Output",
 Evaluatable->False,
 CellID->312496101]
}, Open  ]],
Cell[29478, 909, 125, 3, 70, "ExampleDelimiter",
 CellID->457300817],
Cell[29606, 914, 102, 2, 70, "ExampleText",
 CellID->1161840],
Cell[CellGroupData[{
Cell[29733, 920, 887, 27, 70, "Input",
 CellID->50444804],
Cell[30623, 949, 777, 18, 71, "Output",
 CellID->243900799]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[31449, 973, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[31690, 982, 661, 24, 70, "ExampleText",
 CellID->118934125],
Cell[CellGroupData[{
Cell[32376, 1010, 585, 19, 70, "Input",
 CellID->407662742],
Cell[32964, 1031, 405, 14, 64, "Output",
 CellID->263206571]
}, Open  ]],
Cell[33384, 1048, 125, 3, 70, "ExampleDelimiter",
 CellID->219256165],
Cell[33512, 1053, 662, 24, 70, "ExampleText",
 CellID->479816723],
Cell[CellGroupData[{
Cell[34199, 1081, 587, 19, 70, "Input",
 CellID->793759033],
Cell[34789, 1102, 404, 14, 64, "Output",
 CellID->29651460]
}, Open  ]],
Cell[35208, 1119, 125, 3, 70, "ExampleDelimiter",
 CellID->781155394],
Cell[35336, 1124, 146, 4, 70, "ExampleText",
 CellID->666049620],
Cell[35485, 1130, 322, 11, 70, "Input",
 CellID->65505422],
Cell[CellGroupData[{
Cell[35832, 1145, 376, 12, 70, "Input",
 CellID->887831827],
Cell[36211, 1159, 161, 5, 36, "Output",
 CellID->310996527]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36421, 1170, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[36655, 1179, 100, 2, 70, "ExampleText",
 CellID->3326907],
Cell[36758, 1183, 592, 20, 70, "Input",
 CellID->172289097],
Cell[37353, 1205, 116, 3, 70, "ExampleText",
 CellID->58391003],
Cell[CellGroupData[{
Cell[37494, 1212, 418, 14, 70, "Input",
 CellID->13135961],
Cell[37915, 1228, 607, 21, 55, "Output",
 CellID->557172428]
}, Open  ]],
Cell[38537, 1252, 124, 3, 70, "ExampleDelimiter",
 CellID->25754592],
Cell[38664, 1257, 317, 12, 70, "ExampleText",
 CellID->28036276],
Cell[CellGroupData[{
Cell[39006, 1273, 380, 12, 70, "Input",
 CellID->524424269],
Cell[39389, 1287, 405, 13, 36, "Output",
 CellID->178354127]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[39843, 1306, 228, 7, 70, "ExampleSection",
 CellID->713270274],
Cell[40074, 1315, 61, 1, 70, "ExampleText",
 CellID->283882469],
Cell[CellGroupData[{
Cell[40160, 1320, 870, 28, 70, "Input",
 CellID->680045736],
Cell[41033, 1350, 15190, 253, 70, "Output",
 Evaluatable->False,
 CellID->228706433]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[56284, 1610, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[56603, 1621, 1029, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[57669, 1661, 305, 8, 70, "RelatedLinksSection"],
Cell[57977, 1671, 376, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[58390, 1688, 318, 9, 70, "MoreAboutSection",
 CellID->65781407],
Cell[58711, 1699, 150, 3, 70, "MoreAbout",
 CellID->43718440],
Cell[58864, 1704, 159, 3, 70, "MoreAbout",
 CellID->559902897]
}, Open  ]],
Cell[59038, 1710, 27, 0, 70, "History"],
Cell[59068, 1712, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

