(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     47207,       1596]
NotebookOptionsPosition[     39034,       1304]
NotebookOutlinePosition[     40181,       1339]
CellTagsIndexPosition[     40095,       1334]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Elementary Transcendental Functions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ElementaryTranscendentalFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Elementary Transcendental Functions\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Sin" :> Documentation`HelpLookup["paclet:ref/Sin"], "Degree" :> 
          Documentation`HelpLookup["paclet:ref/Degree"], "FourierTransform" :> 
          Documentation`HelpLookup["paclet:ref/FourierTransform"], 
          "DiracDelta" :> 
          Documentation`HelpLookup["paclet:ref/DiracDelta"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Sin\"\>", 2->"\<\"Degree\"\>", 
       3->"\<\"FourierTransform\"\>", 
       4->"\<\"DiracDelta\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Elementary Functions" :> 
          Documentation`HelpLookup["paclet:guide/ElementaryFunctions"], 
          "Functions Used in Optics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInOptics"], 
          "Mathematical Functions" :> 
          Documentation`HelpLookup["paclet:guide/MathematicalFunctions"], 
          "Signal Processing" :> 
          Documentation`HelpLookup["paclet:guide/SignalProcessing"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Elementary Functions\"\>", 
       2->"\<\"Functions Used in Optics\"\>", 
       3->"\<\"Mathematical Functions\"\>", 4->"\<\"Signal Processing\"\>", 
       5->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Sinc", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Sinc",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Sinc"], "[", 
       StyleBox["z", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives ",
     Cell[BoxData[
      FormBox[
       RowBox[{"sinc", "(", "z", ")"}], TraditionalForm]], "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->8849]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "The argument of ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " is assumed to be in radians. (Multiply by ",
 Cell[BoxData[
  ButtonBox["Degree",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Degree"]], "InlineFormula"],
 " to convert from degrees.) "
}], "Notes",
 CellID->15465],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Sinc",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sinc"], "[", 
   StyleBox["z", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["Sin",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Sin"], "[", 
    StyleBox["z", "TI"], "]"}], "/", 
   StyleBox["z", "TI"]}]], "InlineFormula"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", "\[NotEqual]", "0"}], TraditionalForm]], "InlineMath"],
 ", but is 1 for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", "=", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->6282],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->9487],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->24854116],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->802138822],

Cell["The argument is given in radians:", "ExampleText",
 CellID->27923],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  RowBox[{"Pi", "/", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24640],

Cell[BoxData[
 FractionBox["2", "\[Pi]"]], "Output",
 ImageSize->{17, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->47794422]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7158],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sinc", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->31327670],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Sinc", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7191],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{150, 96},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->417245017]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30829],

Cell[TextData[{
 "Find Fourier transform of ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->26227],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FourierTransform", "[", 
  RowBox[{
   RowBox[{"Sinc", "[", "t", "]"}], ",", "t", ",", "\[Omega]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2870],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", 
  SqrtBox[
   FractionBox["\[Pi]", "2"]], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Sign", "[", 
     RowBox[{"1", "-", "\[Omega]"}], "]"}], "+", 
    RowBox[{"Sign", "[", 
     RowBox[{"1", "+", "\[Omega]"}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{217, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->609875075]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14954],

Cell["Evaluate numerically:", "ExampleText",
 CellID->446135865],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", "3.5", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->47247012],

Cell[BoxData[
 RowBox[{"-", "0.10022377933989138`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->536205926]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->345740074],

Cell["Evaluate for complex numbers:", "ExampleText",
 CellID->516838547],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  RowBox[{"1", "+", 
   RowBox[{"3.5", " ", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->300139532],

Cell[BoxData[
 RowBox[{"3.4134807499770266`", "\[InvisibleSpace]", "-", 
  RowBox[{"3.009162956293194`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->174985525]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->311993860],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->167082427],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Pi", "/", "2"}], ",", 
    RowBox[{"Pi", "/", "3"}], ",", 
    RowBox[{"Pi", "/", "4"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->134462272],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["2", "\[Pi]"], ",", 
   FractionBox[
    RowBox[{"3", " ", 
     SqrtBox["3"]}], 
    RowBox[{"2", " ", "\[Pi]"}]], ",", 
   FractionBox[
    RowBox[{"2", " ", 
     SqrtBox["2"]}], "\[Pi]"]}], "}"}]], "Output",
 ImageSize->{129, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->86615680]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->200013847],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->114303999],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Sinc", "[", "x", "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->631443142],

Cell[BoxData[
 FormBox[
  RowBox[{"sinc", "(", "x", ")"}], TraditionalForm]], "Output",
 ImageSize->{36, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->576395932]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6564],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " can be applied to a power series:"
}], "ExampleText",
 CellID->299715845],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", " ", 
  RowBox[{
   RowBox[{"ArcSin", "[", "x", "]"}], "+", 
   RowBox[{
    RowBox[{"O", "[", "x", "]"}], "^", "10"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31646390],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "-", 
   FractionBox[
    SuperscriptBox["x", "2"], "6"], "-", 
   FractionBox[
    RowBox[{"17", " ", 
     SuperscriptBox["x", "4"]}], "360"], "-", 
   FractionBox[
    RowBox[{"367", " ", 
     SuperscriptBox["x", "6"]}], "15120"], "-", 
   FractionBox[
    RowBox[{"27859", " ", 
     SuperscriptBox["x", "8"]}], "1814400"], "-", 
   FractionBox[
    RowBox[{"589", " ", 
     SuperscriptBox["x", "10"]}], "54432"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "11"],
    SeriesData[$CellContext`x, 0, {}, 0, 11, 1],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {1, 0, 
    Rational[-1, 6], 0, 
    Rational[-17, 360], 0, 
    Rational[-367, 15120], 0, 
    Rational[-27859, 1814400], 0, 
    Rational[-589, 54432]}, 0, 11, 1],
  Editable->False]], "Output",
 ImageSize->{320, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->347660352]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->473565681],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " can be applied to real-valued intervals:"
}], "ExampleText",
 CellID->469239826],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  RowBox[{"Interval", "[", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"Pi", "/", "3"}]}], "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->326401872],

Cell[BoxData[
 RowBox[{"Interval", "[", 
  RowBox[{"{", 
   RowBox[{
    FractionBox[
     RowBox[{"3", " ", 
      SqrtBox["3"]}], 
     RowBox[{"2", " ", "\[Pi]"}]], ",", "1"}], "}"}], "]"}]], "Output",
 ImageSize->{145, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->752883501]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->22064],

Cell[TextData[{
 "Single-slit diffraction pattern for a 4",
 Cell[BoxData[
  StyleBox["\[Lambda]", "TR"]], "InlineFormula"],
 " slit:"
}], "ExampleText",
 CellID->83401973],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   SuperscriptBox[
    RowBox[{"Sinc", "[", 
     FractionBox[
      RowBox[{"4", " ", "\[Pi]", " ", "x"}], 
      SqrtBox[
       RowBox[{"1", "+", 
        SuperscriptBox["x", "2"]}]]], "]"}], "2"], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->601437043],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 120},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->364392469]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->105496228],

Cell["Sinc-filtered Cauchy distribution:", "ExampleText",
 CellID->199848290],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", " ", 
  RowBox[{
   RowBox[{
    FractionBox["1", "Pi"], 
    FractionBox[
     RowBox[{"Sinc", "[", 
      RowBox[{"x", "-", "z"}], "]"}], 
     RowBox[{"1", "+", 
      RowBox[{"x", "^", "2"}]}]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}], ",", 
   RowBox[{"GenerateConditions", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->69593199],

Cell[BoxData[
 FractionBox[
  RowBox[{"\[ExponentialE]", "-", 
   RowBox[{"Cos", "[", "z", "]"}], "+", 
   RowBox[{"z", " ", 
    RowBox[{"Sin", "[", "z", "]"}]}]}], 
  RowBox[{"\[ExponentialE]", "+", 
   RowBox[{"\[ExponentialE]", " ", 
    SuperscriptBox["z", "2"]}]}]]], "Output",
 ImageSize->{135, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->270323230]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Plot", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{"1", "+", 
      SuperscriptBox["x", "2"]}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}], "\[Rule]", 
  RowBox[{"Plot", "[", 
   RowBox[{"%", ",", 
    RowBox[{"{", 
     RowBox[{"z", ",", 
      RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->62032767],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{376, 123},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->70841646]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->534031997],

Cell["A sinc signal is unaltered by sinc filter:", "ExampleText",
 CellID->517609410],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", " ", 
  RowBox[{
   RowBox[{
    FractionBox["1", "Pi"], 
    RowBox[{"Sinc", "[", 
     RowBox[{"x", "-", "z"}], "]"}], 
    RowBox[{"Sinc", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}], ",", 
   RowBox[{"GenerateConditions", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->338369136],

Cell[BoxData[
 RowBox[{"Sinc", "[", "z", "]"}]], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->623480563]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21038],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand expressions involving ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->724013790],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{
   RowBox[{"Sinc", "[", "x", "]"}], ",", 
   RowBox[{"x", "\[NotEqual]", "0"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->333721681],

Cell[BoxData[
 FractionBox[
  RowBox[{"Sin", "[", "x", "]"}], "x"]], "Output",
 ImageSize->{51, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->476713714]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  RowBox[{"Pi", "/", "8"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->126183710],

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  FractionBox["\[Pi]", "8"], "]"}]], "Output",
 ImageSize->{59, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->544121065]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", "%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->166676986],

Cell[BoxData[
 FractionBox[
  RowBox[{"4", " ", 
   SqrtBox[
    RowBox[{"2", "-", 
     SqrtBox["2"]}]]}], "\[Pi]"]], "Output",
 ImageSize->{81, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->485848938]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->63947820],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FullSimplify",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FullSimplify"]], "InlineFormula"],
 " to simplify expressions involving ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->138132544],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{
   RowBox[{"x", "^", "2"}], " ", 
   RowBox[{"Sinc", "[", 
    RowBox[{"x", "/", "2"}], "]"}], 
   RowBox[{"Cos", "[", 
    RowBox[{"x", "/", "2"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->986735780],

Cell[BoxData[
 RowBox[{"x", " ", 
  RowBox[{"Sin", "[", "x", "]"}]}]], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->234515477]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21367],

Cell[TextData[{
 "Non\[Hyphen]trivial minima and maxima of ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " do not have ordinary closed forms:"
}], "ExampleText",
 CellID->289128097],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  RowBox[{"Interval", "[", 
   RowBox[{"{", 
    RowBox[{"Pi", ",", " ", 
     RowBox[{"3", "Pi"}]}], "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->330595212],

Cell[BoxData[
 RowBox[{"Sinc", "[", 
  RowBox[{"Interval", "[", 
   RowBox[{"{", 
    RowBox[{"\[Pi]", ",", 
     RowBox[{"3", " ", "\[Pi]"}]}], "}"}], "]"}], "]"}]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->873522286]
}, Open  ]],

Cell["Find numerical approximations:", "ExampleText",
 CellID->5856819],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"%", ",", "30"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->88635586],

Cell[BoxData[
 RowBox[{"Interval", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "0.217233628211221657408279325562865164675655421338`30."}], 
    ",", "0.128374553525899136693030778234066717387023689018`30."}], "}"}], 
  "]"}]], "Output",
 ImageSize->{555, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->551018776]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->20116],

Cell["A surprising sequence:", "ExampleText",
 CellID->2014848090],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"Product", "[", 
      RowBox[{
       RowBox[{"2", 
        RowBox[{"Sinc", "[", 
         RowBox[{"x", "/", "k"}], "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"k", ",", "1", ",", 
         RowBox[{
          RowBox[{"2", "n"}], "+", "1"}], ",", "2"}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "Infinity"}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "0", ",", "8"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1357454253],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\[Pi]", ",", 
   RowBox[{"2", " ", "\[Pi]"}], ",", 
   RowBox[{"4", " ", "\[Pi]"}], ",", 
   RowBox[{"8", " ", "\[Pi]"}], ",", 
   RowBox[{"16", " ", "\[Pi]"}], ",", 
   RowBox[{"32", " ", "\[Pi]"}], ",", 
   RowBox[{"64", " ", "\[Pi]"}], ",", 
   FractionBox[
    RowBox[{"7484926795415051819144605831504", " ", "\[Pi]"}], 
    "58475990590040056706907609375"], ",", 
   FractionBox[
    RowBox[{"17708695183056190642497315530628422295569865119", " ", "\[Pi]"}],
     "69174591383400772060348345678764669795000000"]}], "}"}]], "Output",
 ImageSize->{487, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->236029607]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->19009],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Sin",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sin"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Degree",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Degree"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FourierTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FourierTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["DiracDelta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/DiracDelta"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->9216]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->24989],

Cell[TextData[ButtonBox["Elementary Transcendental Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ElementaryTranscendentalFunctions"]], "Tutorials",
 CellID->559]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->222913574],

Cell[TextData[ButtonBox["Elementary Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ElementaryFunctions"]], "MoreAbout",
 CellID->353894626],

Cell[TextData[ButtonBox["Functions Used in Optics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInOptics"]], "MoreAbout",
 CellID->191452403],

Cell[TextData[ButtonBox["Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MathematicalFunctions"]], "MoreAbout",
 CellID->323285001],

Cell[TextData[ButtonBox["Signal Processing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SignalProcessing"]], "MoreAbout",
 CellID->618165005],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->276783917]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Sinc - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 20, 49.7765480}", "context" -> "System`", 
    "keywords" -> {"square wave"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> "Sinc[z] gives sinc(z). ",
     "synonyms" -> {}, "title" -> "Sinc", "type" -> "Symbol", "uri" -> 
    "ref/Sinc"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7872, 251, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->24854116]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 39953, 1327}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2766, 64, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3389, 91, 1569, 37, 70, "ObjectNameGrid"],
Cell[4961, 130, 558, 19, 70, "Usage",
 CellID->8849]
}, Open  ]],
Cell[CellGroupData[{
Cell[5556, 154, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6034, 171, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6157, 177, 370, 13, 70, "Notes",
 CellID->15465],
Cell[6530, 192, 658, 26, 70, "Notes",
 CellID->6282],
Cell[7191, 220, 235, 8, 70, "Notes",
 CellID->9487],
Cell[7429, 230, 210, 7, 70, "Notes",
 CellID->31226],
Cell[7642, 239, 193, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[7872, 251, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->24854116],
Cell[CellGroupData[{
Cell[8256, 265, 148, 5, 70, "ExampleSection",
 CellID->802138822],
Cell[8407, 272, 72, 1, 70, "ExampleText",
 CellID->27923],
Cell[CellGroupData[{
Cell[8504, 277, 120, 4, 28, "Input",
 CellID->24640],
Cell[8627, 283, 181, 6, 51, "Output",
 CellID->47794422]
}, Open  ]],
Cell[8823, 292, 120, 3, 70, "ExampleDelimiter",
 CellID->7158],
Cell[8946, 297, 164, 7, 70, "ExampleText",
 CellID->31327670],
Cell[CellGroupData[{
Cell[9135, 308, 232, 8, 28, "Input",
 CellID->7191],
Cell[9370, 318, 2773, 50, 117, "Output",
 Evaluatable->False,
 CellID->417245017]
}, Open  ]],
Cell[12158, 371, 121, 3, 70, "ExampleDelimiter",
 CellID->30829],
Cell[12282, 376, 197, 8, 70, "ExampleText",
 CellID->26227],
Cell[CellGroupData[{
Cell[12504, 388, 182, 6, 70, "Input",
 CellID->2870],
Cell[12689, 396, 429, 15, 60, "Output",
 CellID->609875075]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13167, 417, 216, 7, 70, "ExampleSection",
 CellID->14954],
Cell[13386, 426, 64, 1, 70, "ExampleText",
 CellID->446135865],
Cell[CellGroupData[{
Cell[13475, 431, 101, 3, 70, "Input",
 CellID->47247012],
Cell[13579, 436, 194, 6, 36, "Output",
 CellID->536205926]
}, Open  ]],
Cell[13788, 445, 125, 3, 70, "ExampleDelimiter",
 CellID->345740074],
Cell[13916, 450, 72, 1, 70, "ExampleText",
 CellID->516838547],
Cell[CellGroupData[{
Cell[14013, 455, 149, 5, 70, "Input",
 CellID->300139532],
Cell[14165, 462, 272, 7, 36, "Output",
 CellID->174985525]
}, Open  ]],
Cell[14452, 472, 125, 3, 70, "ExampleDelimiter",
 CellID->311993860],
Cell[14580, 477, 202, 7, 70, "ExampleText",
 CellID->167082427],
Cell[CellGroupData[{
Cell[14807, 488, 235, 8, 70, "Input",
 CellID->134462272],
Cell[15045, 498, 396, 15, 57, "Output",
 CellID->86615680]
}, Open  ]],
Cell[15456, 516, 125, 3, 70, "ExampleDelimiter",
 CellID->200013847],
Cell[15584, 521, 203, 7, 70, "ExampleText",
 CellID->114303999],
Cell[CellGroupData[{
Cell[15812, 532, 138, 4, 70, "Input",
 CellID->631443142],
Cell[15953, 538, 234, 7, 50, "Output",
 CellID->576395932]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16236, 551, 238, 7, 70, "ExampleSection",
 CellID->6564],
Cell[16477, 560, 203, 7, 70, "ExampleText",
 CellID->299715845],
Cell[CellGroupData[{
Cell[16705, 571, 216, 7, 70, "Input",
 CellID->31646390],
Cell[16924, 580, 991, 33, 54, "Output",
 CellID->347660352]
}, Open  ]],
Cell[17930, 616, 125, 3, 70, "ExampleDelimiter",
 CellID->473565681],
Cell[18058, 621, 210, 7, 70, "ExampleText",
 CellID->469239826],
Cell[CellGroupData[{
Cell[18293, 632, 211, 7, 70, "Input",
 CellID->326401872],
Cell[18507, 641, 335, 12, 57, "Output",
 CellID->752883501]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18891, 659, 223, 7, 70, "ExampleSection",
 CellID->22064],
Cell[19117, 668, 172, 6, 70, "ExampleText",
 CellID->83401973],
Cell[CellGroupData[{
Cell[19314, 678, 450, 15, 70, "Input",
 CellID->601437043],
Cell[19767, 695, 2616, 47, 70, "Output",
 Evaluatable->False,
 CellID->364392469]
}, Open  ]],
Cell[22398, 745, 125, 3, 70, "ExampleDelimiter",
 CellID->105496228],
Cell[22526, 750, 77, 1, 70, "ExampleText",
 CellID->199848290],
Cell[CellGroupData[{
Cell[22628, 755, 467, 15, 70, "Input",
 CellID->69593199],
Cell[23098, 772, 414, 13, 53, "Output",
 CellID->270323230]
}, Open  ]],
Cell[CellGroupData[{
Cell[23549, 790, 454, 16, 70, "Input",
 CellID->62032767],
Cell[24006, 808, 4860, 84, 70, "Output",
 Evaluatable->False,
 CellID->70841646]
}, Open  ]],
Cell[28881, 895, 125, 3, 70, "ExampleDelimiter",
 CellID->534031997],
Cell[29009, 900, 85, 1, 70, "ExampleText",
 CellID->517609410],
Cell[CellGroupData[{
Cell[29119, 905, 428, 13, 70, "Input",
 CellID->338369136],
Cell[29550, 920, 188, 6, 36, "Output",
 CellID->623480563]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29787, 932, 233, 7, 70, "ExampleSection",
 CellID->21038],
Cell[30023, 941, 348, 13, 70, "ExampleText",
 CellID->724013790],
Cell[CellGroupData[{
Cell[30396, 958, 199, 6, 70, "Input",
 CellID->333721681],
Cell[30598, 966, 208, 7, 51, "Output",
 CellID->476713714]
}, Open  ]],
Cell[CellGroupData[{
Cell[30843, 978, 124, 4, 70, "Input",
 CellID->126183710],
Cell[30970, 984, 213, 7, 49, "Output",
 CellID->544121065]
}, Open  ]],
Cell[CellGroupData[{
Cell[31220, 996, 110, 3, 70, "Input",
 CellID->166676986],
Cell[31333, 1001, 258, 10, 62, "Output",
 CellID->485848938]
}, Open  ]],
Cell[31606, 1014, 124, 3, 70, "ExampleDelimiter",
 CellID->63947820],
Cell[31733, 1019, 346, 13, 70, "ExampleText",
 CellID->138132544],
Cell[CellGroupData[{
Cell[32104, 1036, 273, 9, 70, "Input",
 CellID->986735780],
Cell[32380, 1047, 210, 7, 36, "Output",
 CellID->234515477]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[32639, 1060, 226, 7, 70, "ExampleSection",
 CellID->21367],
Cell[32868, 1069, 250, 8, 70, "ExampleText",
 CellID->289128097],
Cell[CellGroupData[{
Cell[33143, 1081, 212, 7, 70, "Input",
 CellID->330595212],
Cell[33358, 1090, 307, 10, 36, "Output",
 CellID->873522286]
}, Open  ]],
Cell[33680, 1103, 71, 1, 70, "ExampleText",
 CellID->5856819],
Cell[CellGroupData[{
Cell[33776, 1108, 120, 4, 70, "Input",
 CellID->88635586],
Cell[33899, 1114, 374, 11, 36, "Output",
 CellID->551018776]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34322, 1131, 224, 7, 70, "ExampleSection",
 CellID->20116],
Cell[34549, 1140, 66, 1, 70, "ExampleText",
 CellID->2014848090],
Cell[CellGroupData[{
Cell[34640, 1145, 603, 19, 70, "Input",
 CellID->1357454253],
Cell[35246, 1166, 708, 19, 81, "Output",
 CellID->236029607]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[36015, 1192, 311, 9, 70, "SeeAlsoSection",
 CellID->19009],
Cell[36329, 1203, 946, 35, 70, "SeeAlso",
 CellID->9216]
}, Open  ]],
Cell[CellGroupData[{
Cell[37312, 1243, 314, 9, 70, "TutorialsSection",
 CellID->24989],
Cell[37629, 1254, 180, 4, 70, "Tutorials",
 CellID->559]
}, Open  ]],
Cell[CellGroupData[{
Cell[37846, 1263, 319, 9, 70, "MoreAboutSection",
 CellID->222913574],
Cell[38168, 1274, 151, 3, 70, "MoreAbout",
 CellID->353894626],
Cell[38322, 1279, 157, 3, 70, "MoreAbout",
 CellID->191452403],
Cell[38482, 1284, 155, 3, 70, "MoreAbout",
 CellID->323285001],
Cell[38640, 1289, 145, 3, 70, "MoreAbout",
 CellID->618165005],
Cell[38788, 1294, 174, 3, 70, "MoreAbout",
 CellID->276783917]
}, Open  ]],
Cell[38977, 1300, 27, 0, 70, "History"],
Cell[39007, 1302, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

