(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     50348,       1826]
NotebookOptionsPosition[     40599,       1486]
NotebookOutlinePosition[     42142,       1526]
CellTagsIndexPosition[     42056,       1521]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Advanced Matrix Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AdvancedMatrixOperations"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Advanced Matrix Operations\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SingularValueDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/SingularValueDecomposition"], 
          "Norm" :> Documentation`HelpLookup["paclet:ref/Norm"], 
          "PseudoInverse" :> 
          Documentation`HelpLookup["paclet:ref/PseudoInverse"], "Eigenvalues" :> 
          Documentation`HelpLookup["paclet:ref/Eigenvalues"], 
          "QRDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/QRDecomposition"], 
          "SchurDecomposition" :> 
          Documentation`HelpLookup["paclet:ref/SchurDecomposition"], 
          "NullSpace" :> Documentation`HelpLookup["paclet:ref/NullSpace"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"SingularValueDecomposition\"\>", 2->"\<\"Norm\"\>", 
       3->"\<\"PseudoInverse\"\>", 4->"\<\"Eigenvalues\"\>", 
       5->"\<\"QRDecomposition\"\>", 6->"\<\"SchurDecomposition\"\>", 
       7->"\<\"NullSpace\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Sparse Arrays" :> 
          Documentation`HelpLookup["paclet:guide/SparseArrays"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Sparse Arrays\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SingularValueList", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SingularValueList",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SingularValueList"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives a list of the nonzero singular values of a matrix \
",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SingularValueList",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SingularValueList"], "[", 
       RowBox[{"{", 
        RowBox[{
         StyleBox["m", "TI"], ",", 
         StyleBox["a", "TI"]}], "}"}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the generalized singular values of ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     " with respect to ",
     Cell[BoxData[
      StyleBox["a", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SingularValueList",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SingularValueList"], "[", 
       RowBox[{
        StyleBox["m", "TI"], ",", 
        StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     " largest singular values of ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     ". "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SingularValueList",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SingularValueList"], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          StyleBox["m", "TI"], ",", 
          StyleBox["a", "TI"]}], "}"}], ",", 
        StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     " largest generalized singular values of ",
     Cell[BoxData[
      StyleBox["m", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->4967]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["Singular values are sorted from largest to smallest. ", "Notes",
 CellID->29250],

Cell["\<\
Repeated singular values appear with their appropriate multiplicity. \
\>", "Notes",
 CellID->13484],

Cell[TextData[{
 "By default, singular values are kept only when they are larger than 100 \
times ",
 Cell[BoxData[
  SuperscriptBox["10", 
   RowBox[{"-", 
    StyleBox["p", "TI"]}]]], "InlineFormula"],
 ", where ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Precision",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Precision"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->13161],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SingularValueList",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SingularValueList"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    RowBox[{
     ButtonBox["Tolerance",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Tolerance"], "->", 
     StyleBox["t", "TI"]}]}], "]"}]], "InlineFormula"],
 " keeps only singular values that are at least ",
 Cell[BoxData[
  StyleBox["t", "TI"]], "InlineFormula"],
 " times the largest singular value. "
}], "Notes",
 CellID->31854],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SingularValueList",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SingularValueList"], "[", 
   RowBox[{
    StyleBox["m", "TI"], ",", 
    RowBox[{
     ButtonBox["Tolerance",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Tolerance"], "->", "0"}]}], "]"}]], 
  "InlineFormula"],
 " returns all singular values. "
}], "Notes",
 CellID->19484],

Cell[TextData[{
 "The matrix ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " can be rectangular; the total number of singular values is always ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Min",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Min"], "[", 
   RowBox[{
    ButtonBox["Dimensions",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Dimensions"], "[", 
    StyleBox["m", "TI"], "]"}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->11661],

Cell["\<\
Exact and symbolic matrices can be used, with zero tolerance assumed by \
default.\
\>", "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->26043],

Cell[TextData[{
 "The singular values can be obtained from ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Sqrt",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Sqrt"], "[", 
   RowBox[{
    ButtonBox["Eigenvalues",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Eigenvalues"], "[", 
    RowBox[{
     RowBox[{
      ButtonBox["Conjugate",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Conjugate"], "[", 
      RowBox[{
       ButtonBox["Transpose",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Transpose"], "[", 
       StyleBox["m", "TI"], "]"}], "]"}], ".", 
     StyleBox["m", "TI"]}], "]"}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->32226]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->47255416],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->23696085],

Cell["\<\
There are only two nonzero singular values since the matrix is singular:\
\>", "ExampleText",
 CellID->16569366],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"7", ",", "8", ",", "9"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->558052859],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   SqrtBox[
    RowBox[{
     FractionBox["3", "2"], " ", 
     RowBox[{"(", 
      RowBox[{"95", "+", 
       SqrtBox["8881"]}], ")"}]}]], ",", 
   SqrtBox[
    RowBox[{
     FractionBox["3", "2"], " ", 
     RowBox[{"(", 
      RowBox[{"95", "-", 
       SqrtBox["8881"]}], ")"}]}]]}], "}"}]], "Output",
 ImageSize->{269, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26459047]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24634],

Cell["Exact input:", "ExampleText",
 CellID->28936],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", "4"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32298],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   SqrtBox[
    RowBox[{"15", "+", 
     SqrtBox["221"]}]], ",", 
   SqrtBox[
    RowBox[{"15", "-", 
     SqrtBox["221"]}]]}], "}"}]], "Output",
 ImageSize->{196, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->201772299]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->532492295],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is a 4\[Times]3 matrix:"
}], "ExampleText",
 CellID->245604209],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "8", ",", "9"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"10", ",", "11", ",", "12"}], "}"}]}], "}"}]}], ";"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->9187314],

Cell["Singular values computed using exact arithmetic:", "ExampleText",
 CellID->71273579],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->156393296],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   SqrtBox[
    RowBox[{"325", "+", 
     SqrtBox["104545"]}]], ",", 
   RowBox[{"6", " ", 
    SqrtBox[
     FractionBox["30", 
      RowBox[{"325", "+", 
       SqrtBox["104545"]}]]]}]}], "}"}]], "Output",
 ImageSize->{272, 38},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->168578432]
}, Open  ]],

Cell["Singular values computed using machine arithmetic:", "ExampleText",
 CellID->216451651],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{"N", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->5196375],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"25.46240743603639`", ",", "1.2906616757612326`"}], "}"}]], "Output",\

 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->577635580]
}, Open  ]],

Cell["\<\
Singular values computed using 30-digit precision arithmetic:\
\>", "ExampleText",
 CellID->804896551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{"N", "[", 
   RowBox[{"m", ",", "30"}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->453294499],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "25.46240743603638924893382674583501311377`30.", ",", 
   "1.290661675761231448641624883605056813707814844698`30."}], 
  "}"}]], "Output",
 ImageSize->{464, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->565412164]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->445976199],

Cell["\<\
Singular values for a random complex valued 2\[Times]5 matrix:\
\>", "ExampleText",
 CellID->189409873],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{"RandomComplex", "[", 
   RowBox[{
    RowBox[{"1", "+", "I"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->76792886],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.419503926952924`", ",", "0.962262874968443`"}], "}"}]], "Output",\

 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->480089831]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2275],

Cell["\<\
Find the largest 3 singular values for a large sparse matrix:\
\>", "ExampleText",
 CellID->100340494],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "j_"}], "}"}], "/;", 
        RowBox[{
         RowBox[{"Abs", "[", 
          RowBox[{"i", "-", "j"}], "]"}], "\[Equal]", "1"}]}], "\[Rule]", 
       RowBox[{"N", "[", 
        RowBox[{"j", "-", "i"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"250", ",", "250"}], "}"}]}], "]"}], ",", "3"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->396577924],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.99984334404621`", ",", "1.9993733932742999`", ",", "1.99858780212936`"}],
   "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->98329924]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24740],

Cell["Symbolic input:", "ExampleText",
 CellID->4104],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Assuming", "[", 
  RowBox[{
   RowBox[{"z", "\[Element]", "Reals"}], ",", 
   RowBox[{"Refine", "@", 
    RowBox[{"SingularValueList", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"3", ",", "z"}], "}"}]}], "}"}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17080],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    SqrtBox[
     RowBox[{"14", "+", 
      SuperscriptBox["z", "2"], "-", 
      SqrtBox[
       RowBox[{"52", "+", 
        RowBox[{"48", " ", "z"}], "+", 
        RowBox[{"24", " ", 
         SuperscriptBox["z", "2"]}], "+", 
        SuperscriptBox["z", "4"]}]]}]], 
    SqrtBox["2"]], ",", 
   FractionBox[
    SqrtBox[
     RowBox[{"14", "+", 
      SuperscriptBox["z", "2"], "+", 
      SqrtBox[
       RowBox[{"52", "+", 
        RowBox[{"48", " ", "z"}], "+", 
        RowBox[{"24", " ", 
         SuperscriptBox["z", "2"]}], "+", 
        SuperscriptBox["z", "4"]}]]}]], 
    SqrtBox["2"]]}], "}"}]], "Output",
 ImageSize->{455, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->107870907]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->221902594],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Tolerance",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->40],

Cell[TextData[{
 "Compute the singular values larger than ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", "10"], TraditionalForm]], "InlineMath"],
 " of the largest singular value:"
}], "ExampleText",
 CellID->68392051],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"N", "[", 
    RowBox[{"HilbertMatrix", "[", "10", "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->48239645],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    SubscriptBox["\[Sigma]", "1"], "}"}], "=", 
   RowBox[{"SingularValueList", "[", 
    RowBox[{"m", ",", "1"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->892635470],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Select", "[", 
  RowBox[{
   RowBox[{"SingularValueList", "[", "m", "]"}], ",", 
   RowBox[{
    RowBox[{"#", ">", 
     RowBox[{"0.1", " ", 
      SubscriptBox["\[Sigma]", "1"]}]}], "&"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->383846863],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.7519196702651776`", ",", "0.34292954848350893`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->67610554]
}, Open  ]],

Cell[TextData[{
 "Setting ",
 Cell[BoxData[
  ButtonBox["Tolerance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Tolerance"]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", "10"], TraditionalForm]], "InlineMath"],
 " will directly compute the same set of singular values:"
}], "ExampleText",
 CellID->39394174],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{"m", ",", 
   RowBox[{"Tolerance", "\[Rule]", "0.1"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->91952948],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.7519196702651776`", ",", "0.34292954848350893`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->116985142]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->434916979],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is a 16\[Times]16 Hilbert matrix:"
}], "ExampleText",
 CellID->417180813],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"HilbertMatrix", "[", "16", "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->84761113],

Cell["\<\
The matrix is positive definite so with exact arithmetic there are 16 nonzero \
singular values:\
\>", "ExampleText",
 CellID->88807062],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", 
  RowBox[{"SingularValueList", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15186916],

Cell[BoxData["16"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->126762906]
}, Open  ]],

Cell["\<\
Many of the singular values are too small to show up at machine precision:\
\>", "ExampleText",
 CellID->291479351],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", 
  RowBox[{"SingularValueList", "[", 
   RowBox[{"N", "[", "m", "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->595927103],

Cell[BoxData["11"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->82022989]
}, Open  ]],

Cell["Setting the tolerance to zero will make them all show up:", \
"ExampleText",
 CellID->694732401],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{
   RowBox[{"N", "[", "m", "]"}], ",", 
   RowBox[{"Tolerance", "\[Rule]", "0"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->63108948],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.8600364427433271`", ",", "0.44013143636233815`", ",", 
   "0.061114262916307095`", ",", "0.006297815269922941`", ",", 
   "0.0005146753583779205`", ",", "0.0000341275697165957`", ",", 
   "1.8529753628436258`*^-6", ",", "8.253135278318192`*^-8", ",", 
   "3.004576164931264`*^-9", ",", "8.861708123711805`*^-11", ",", 
   "2.0859386978046283`*^-12", ",", "3.8281107400193036`*^-14", ",", 
   "5.398334365678472`*^-16", ",", "1.0436266522035359`*^-17", ",", 
   "7.602763519749519`*^-18", ",", "4.261253418034127`*^-18"}], 
  "}"}]], "Output",
 ImageSize->{512, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->520470723]
}, Open  ]],

Cell["\<\
Because of numerical roundoff, the values are not computed accurately:\
\>", "ExampleText",
 CellID->125638374],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"SingularValueList", "[", "m", "]"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->11351078],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.8600364427433271`", ",", "0.44013143636233815`", ",", 
   "0.06111426291630709`", ",", "0.006297815269922938`", ",", 
   "0.0005146753583779354`", ",", "0.00003412756971658616`", ",", 
   "1.8529753628386086`*^-6", ",", "8.253135278123955`*^-8", ",", 
   "3.0045761593828615`*^-9", ",", "8.861708844367059`*^-11", ",", 
   "2.085937152775827`*^-12", ",", "3.827669089575262`*^-14", ",", 
   "5.277456405490515`*^-16", ",", "5.14339953206079`*^-18", ",", 
   "3.159250701855795`*^-20", ",", "9.197419820651453`*^-23"}], 
  "}"}]], "Output",
 ImageSize->{505, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->131861358]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->13631],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is a random 10",
 "\[Times]",
 "10 matrix:"
}], "ExampleText",
 CellID->625752925],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "8"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->76329063],

Cell[TextData[{
 "Find the singular values of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->336140930],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sv", "=", 
  RowBox[{"SingularValueList", "[", "m", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->39840616],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "3.7751241139072147`", ",", "1.182071942159695`", ",", 
   "1.0233642816596826`", ",", "0.7595231312543521`", ",", 
   "0.645460879573184`", ",", "0.45127494142671903`", ",", 
   "0.19385092422545228`", ",", "0.11585719871036759`"}], "}"}]], "Output",
 ImageSize->{524, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->59873183]
}, Open  ]],

Cell["\<\
The 2-norm of a matrix is equal to the largest singular value:\
\>", "ExampleText",
 CellID->193711179],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{"m", ",", "2"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->74630292],

Cell[BoxData["3.7751241139072147`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->426264366]
}, Open  ]],

Cell["\<\
The 2-norm of the inverse is equal to the reciprocal of the smallest singular \
value:\
\>", "ExampleText",
 CellID->168206742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{
   RowBox[{"Inverse", "[", "m", "]"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->75487185],

Cell[BoxData["8.631315197771254`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->205786210]
}, Open  ]],

Cell["\<\
The condition number of the matrix is equal to the ratio of largest to \
smallest singular values:\
\>", "ExampleText",
 CellID->83972264],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"sv", "[", 
   RowBox[{"[", "1", "]"}], "]"}], "/", 
  RowBox[{"sv", "[", 
   RowBox[{"[", 
    RowBox[{"-", "1"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->247725463],

Cell[BoxData["32.58428613784008`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->227873625]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27377],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 " is a matrix of random size having random entries:"
}], "ExampleText",
 CellID->83763948],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"RandomInteger", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "10"}], "}"}], ",", "2"}], "]"}]}], "]"}]}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->329128238],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.25951932456330784`", ",", "0.3874624668867219`", ",", 
     "0.03137230087646392`", ",", "0.8051457352546965`", ",", 
     "0.6029394008277595`", ",", "0.1734896923185969`", ",", 
     "0.2442078096096465`", ",", "0.8477627208646783`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.22246072372637848`", ",", "0.692744350116014`", ",", 
     "0.25768588169813733`", ",", "0.873085846868517`", ",", 
     "0.6956834204524016`", ",", "0.30993342548597846`", ",", 
     "0.15881625836498547`", ",", "0.31364167325892467`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.30250601355479056`", ",", "0.11533041815742484`", ",", 
     "0.8255053110550976`", ",", "0.6595836076116335`", ",", 
     "0.5918069482716128`", ",", "0.36889278118119906`", ",", 
     "0.9845428008444581`", ",", "0.26818580041761053`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.3158918412157794`", ",", "0.11744356784036536`", ",", 
     "0.40668768722906457`", ",", "0.2807925872613799`", ",", 
     "0.12975494209961957`", ",", "0.15716516290580196`", ",", 
     "0.5777518345797297`", ",", "0.2509779600982731`"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{559, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->784643636]
}, Open  ]],

Cell[TextData[{
 "Find the singular values of ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->120265410],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", "m", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->43097117],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.503334189655865`", ",", "1.0407714008011903`", ",", 
   "0.4975111588827567`", ",", "0.21911900797657205`"}], "}"}]], "Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->347618263]
}, Open  ]],

Cell[TextData[{
 "These are equal to the square roots of the nonzero eigenvalues of ",
 Cell[BoxData[
  RowBox[{
   StyleBox["m", "TI"], ".", 
   RowBox[{
    ButtonBox["Transpose",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Transpose"], "[", 
    StyleBox["m", "TI"], "]"}]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->85856295],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sqrt", "[", 
  RowBox[{"Eigenvalues", "[", 
   RowBox[{"m", ".", 
    RowBox[{"Transpose", "[", "m", "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->70723659],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.5033341896558645`", ",", "1.0407714008011901`", ",", 
   "0.49751115888275665`", ",", "0.21911900797657122`"}], "}"}]], "Output",
 ImageSize->{262, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->8024904]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12092],

Cell[TextData[{
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " is a large sparse matrix:"
}], "ExampleText",
 CellID->10269307],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"s", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], "\[Rule]", 
       RowBox[{"1.", "/", "i"}]}], ",", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "j_"}], "}"}], "\[Rule]", "1."}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1000", ",", "1000"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9302928],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "1999", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 1999, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"1000", ",", "1000"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{230, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->416964663]
}, Open  ]],

Cell["\<\
Computing all of the singular values uses dense linear algebra, which may be \
prohibitive:\
\>", "ExampleText",
 CellID->489408578],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Short", "[", 
  RowBox[{"SingularValueList", "[", "s", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->216993603],

Cell[BoxData[
 RowBox[{
  RowBox[{"SingularValueList", "::", "\<\"arh\"\>"}], ":", 
  " ", "\<\"Because finding \\!\\(1000\\) out of the \\!\\(1000\\) singular \
values and/or singular vectors is likely to be faster with dense matrix \
methods, the sparse input matrix will be converted.  If fewer singular values \
and/or singular vectors would be sufficient, consider restricting this number \
using the second argument to \\!\\(SingularValueList\\).\"\>"}]], "Message", \
"MSG",
 CellID->181120498],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"31.622786784476993`", ",", "0.4997503949813081`", ",", 
    RowBox[{"\[LeftSkeleton]", "997", "\[RightSkeleton]"}], ",", 
    "0.0000546820251086143`"}], "}"}],
  Short]], "Output",
 ImageSize->{276, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//Short=",
 CellID->174961344]
}, Open  ]],

Cell["Computing just a few will typically be much faster:", "ExampleText",
 CellID->10572515],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularValueList", "[", 
  RowBox[{"s", ",", "2"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->165870020],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"31.62278678447699`", ",", "0.49975039498130797`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->360109265]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->31315],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SingularValueDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SingularValueDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Norm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Norm"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PseudoInverse",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PseudoInverse"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Eigenvalues",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Eigenvalues"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QRDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QRDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SchurDecomposition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SchurDecomposition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NullSpace",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NullSpace"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->8055]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->12491],

Cell[TextData[ButtonBox["Advanced Matrix Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AdvancedMatrixOperations"]], "Tutorials",
 CellID->11685]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->143431254],

Cell[TextData[ButtonBox["Sparse Arrays",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SparseArrays"]], "MoreAbout",
 CellID->286805059]
}, Open  ]],

Cell["New in 5  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SingularValueList - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 20, 53.1672598}", "context" -> "System`", 
    "keywords" -> {"condition number of matrices", "matrices"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "SingularValueList[m] gives a list of the nonzero singular values of a \
matrix m. SingularValueList[{m, a}] gives the generalized singular values of \
m with respect to a. SingularValueList[m, k] gives the k largest singular \
values of m. SingularValueList[{m, a}, k] gives the k largest generalized \
singular values of m.", "synonyms" -> {"singular value list"}, "title" -> 
    "SingularValueList", "type" -> "Symbol", "uri" -> 
    "ref/SingularValueList"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[13256, 390, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->47255416]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 41913, 1514}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2502, 59, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3125, 86, 4201, 83, 70, "ObjectNameGrid"],
Cell[7329, 171, 2297, 74, 70, "Usage",
 CellID->4967]
}, Open  ]],
Cell[CellGroupData[{
Cell[9663, 250, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[10141, 267, 86, 1, 70, "Notes",
 CellID->29250],
Cell[10230, 270, 110, 3, 70, "Notes",
 CellID->13484],
Cell[10343, 275, 489, 19, 70, "Notes",
 CellID->13161],
Cell[10835, 296, 544, 18, 70, "Notes",
 CellID->31854],
Cell[11382, 316, 412, 15, 70, "Notes",
 CellID->19484],
Cell[11797, 333, 476, 17, 70, "Notes",
 CellID->11661],
Cell[12276, 352, 253, 7, 70, "Notes",
 CellID->26043],
Cell[12532, 361, 687, 24, 70, "Notes",
 CellID->32226]
}, Closed]],
Cell[CellGroupData[{
Cell[13256, 390, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->47255416],
Cell[CellGroupData[{
Cell[13640, 404, 147, 5, 70, "ExampleSection",
 CellID->23696085],
Cell[13790, 411, 122, 3, 70, "ExampleText",
 CellID->16569366],
Cell[CellGroupData[{
Cell[13937, 418, 353, 11, 28, "Input",
 CellID->558052859],
Cell[14293, 431, 475, 19, 60, "Output",
 CellID->26459047]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14817, 456, 216, 7, 70, "ExampleSection",
 CellID->24634],
Cell[15036, 465, 51, 1, 70, "ExampleText",
 CellID->28936],
Cell[CellGroupData[{
Cell[15112, 470, 258, 9, 70, "Input",
 CellID->32298],
Cell[15373, 481, 316, 13, 49, "Output",
 CellID->201772299]
}, Open  ]],
Cell[15704, 497, 125, 3, 70, "ExampleDelimiter",
 CellID->532492295],
Cell[15832, 502, 139, 5, 70, "ExampleText",
 CellID->245604209],
Cell[15974, 509, 434, 15, 70, "Input",
 CellID->9187314],
Cell[16411, 526, 90, 1, 70, "ExampleText",
 CellID->71273579],
Cell[CellGroupData[{
Cell[16526, 531, 113, 3, 70, "Input",
 CellID->156393296],
Cell[16642, 536, 378, 15, 59, "Output",
 CellID->168578432]
}, Open  ]],
Cell[17035, 554, 93, 1, 70, "ExampleText",
 CellID->216451651],
Cell[CellGroupData[{
Cell[17153, 559, 139, 4, 70, "Input",
 CellID->5196375],
Cell[17295, 565, 241, 8, 36, "Output",
 CellID->577635580]
}, Open  ]],
Cell[17551, 576, 112, 3, 70, "ExampleText",
 CellID->804896551],
Cell[CellGroupData[{
Cell[17688, 583, 166, 5, 70, "Input",
 CellID->453294499],
Cell[17857, 590, 311, 10, 36, "Output",
 CellID->565412164]
}, Open  ]],
Cell[18183, 603, 125, 3, 70, "ExampleDelimiter",
 CellID->445976199],
Cell[18311, 608, 113, 3, 70, "ExampleText",
 CellID->189409873],
Cell[CellGroupData[{
Cell[18449, 615, 252, 8, 70, "Input",
 CellID->76792886],
Cell[18704, 625, 240, 8, 36, "Output",
 CellID->480089831]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18993, 639, 238, 7, 70, "ExampleSection",
 CellID->2275],
Cell[19234, 648, 112, 3, 70, "ExampleText",
 CellID->100340494],
Cell[CellGroupData[{
Cell[19371, 655, 574, 19, 70, "Input",
 CellID->396577924],
Cell[19948, 676, 269, 9, 36, "Output",
 CellID->98329924]
}, Open  ]],
Cell[20232, 688, 121, 3, 70, "ExampleDelimiter",
 CellID->24740],
Cell[20356, 693, 53, 1, 70, "ExampleText",
 CellID->4104],
Cell[CellGroupData[{
Cell[20434, 698, 402, 13, 70, "Input",
 CellID->17080],
Cell[20839, 713, 806, 29, 80, "Output",
 CellID->107870907]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21694, 748, 222, 7, 70, "ExampleSection",
 CellID->221902594],
Cell[CellGroupData[{
Cell[21941, 759, 223, 7, 70, "ExampleSubsection",
 CellID->40],
Cell[22167, 768, 220, 7, 70, "ExampleText",
 CellID->68392051],
Cell[22390, 777, 181, 6, 70, "Input",
 CellID->48239645],
Cell[22574, 785, 235, 8, 70, "Input",
 CellID->892635470],
Cell[CellGroupData[{
Cell[22834, 797, 275, 9, 70, "Input",
 CellID->383846863],
Cell[23112, 808, 243, 8, 36, "Output",
 CellID->67610554]
}, Open  ]],
Cell[23370, 819, 342, 12, 70, "ExampleText",
 CellID->39394174],
Cell[CellGroupData[{
Cell[23737, 835, 175, 5, 70, "Input",
 CellID->91952948],
Cell[23915, 842, 244, 8, 36, "Output",
 CellID->116985142]
}, Open  ]],
Cell[24174, 853, 125, 3, 70, "ExampleDelimiter",
 CellID->434916979],
Cell[24302, 858, 149, 5, 70, "ExampleText",
 CellID->417180813],
Cell[24454, 865, 151, 5, 70, "Input",
 CellID->84761113],
Cell[24608, 872, 146, 4, 70, "ExampleText",
 CellID->88807062],
Cell[CellGroupData[{
Cell[24779, 880, 145, 4, 70, "Input",
 CellID->15186916],
Cell[24927, 886, 159, 5, 36, "Output",
 CellID->126762906]
}, Open  ]],
Cell[25101, 894, 125, 3, 70, "ExampleText",
 CellID->291479351],
Cell[CellGroupData[{
Cell[25251, 901, 175, 5, 70, "Input",
 CellID->595927103],
Cell[25429, 908, 158, 5, 36, "Output",
 CellID->82022989]
}, Open  ]],
Cell[25602, 916, 102, 2, 70, "ExampleText",
 CellID->694732401],
Cell[CellGroupData[{
Cell[25729, 922, 202, 6, 70, "Input",
 CellID->63108948],
Cell[25934, 930, 718, 16, 75, "Output",
 CellID->520470723]
}, Open  ]],
Cell[26667, 949, 121, 3, 70, "ExampleText",
 CellID->125638374],
Cell[CellGroupData[{
Cell[26813, 956, 140, 4, 70, "Input",
 CellID->11351078],
Cell[26956, 962, 715, 16, 75, "Output",
 CellID->131861358]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[27732, 985, 223, 7, 70, "ExampleSection",
 CellID->13631],
Cell[27958, 994, 158, 7, 70, "ExampleText",
 CellID->625752925],
Cell[28119, 1003, 225, 8, 70, "Input",
 CellID->76329063],
Cell[28347, 1013, 149, 6, 70, "ExampleText",
 CellID->336140930],
Cell[CellGroupData[{
Cell[28521, 1023, 136, 4, 70, "Input",
 CellID->39840616],
Cell[28660, 1029, 423, 11, 36, "Output",
 CellID->59873183]
}, Open  ]],
Cell[29098, 1043, 113, 3, 70, "ExampleText",
 CellID->193711179],
Cell[CellGroupData[{
Cell[29236, 1050, 122, 4, 70, "Input",
 CellID->74630292],
Cell[29361, 1056, 176, 5, 36, "Output",
 CellID->426264366]
}, Open  ]],
Cell[29552, 1064, 137, 4, 70, "ExampleText",
 CellID->168206742],
Cell[CellGroupData[{
Cell[29714, 1072, 157, 5, 70, "Input",
 CellID->75487185],
Cell[29874, 1079, 175, 5, 36, "Output",
 CellID->205786210]
}, Open  ]],
Cell[30064, 1087, 148, 4, 70, "ExampleText",
 CellID->83972264],
Cell[CellGroupData[{
Cell[30237, 1095, 218, 8, 70, "Input",
 CellID->247725463],
Cell[30458, 1105, 175, 5, 36, "Output",
 CellID->227873625]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[30682, 1116, 233, 7, 70, "ExampleSection",
 CellID->27377],
Cell[30918, 1125, 164, 5, 70, "ExampleText",
 CellID->83763948],
Cell[CellGroupData[{
Cell[31107, 1134, 279, 10, 70, "Input",
 CellID->329128238],
Cell[31389, 1146, 1344, 32, 88, "Output",
 CellID->784643636]
}, Open  ]],
Cell[32748, 1181, 149, 6, 70, "ExampleText",
 CellID->120265410],
Cell[CellGroupData[{
Cell[32922, 1191, 112, 3, 70, "Input",
 CellID->43097117],
Cell[33037, 1196, 303, 9, 36, "Output",
 CellID->347618263]
}, Open  ]],
Cell[33355, 1208, 349, 12, 70, "ExampleText",
 CellID->85856295],
Cell[CellGroupData[{
Cell[33729, 1224, 199, 6, 70, "Input",
 CellID->70723659],
Cell[33931, 1232, 303, 9, 36, "Output",
 CellID->8024904]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34283, 1247, 226, 7, 70, "ExampleSection",
 CellID->12092],
Cell[34512, 1256, 140, 5, 70, "ExampleText",
 CellID->10269307],
Cell[CellGroupData[{
Cell[34677, 1265, 473, 16, 70, "Input",
 CellID->9302928],
Cell[35153, 1283, 491, 17, 36, "Output",
 CellID->416964663]
}, Open  ]],
Cell[35659, 1303, 142, 4, 70, "ExampleText",
 CellID->489408578],
Cell[CellGroupData[{
Cell[35826, 1311, 145, 4, 70, "Input",
 CellID->216993603],
Cell[35974, 1317, 501, 9, 70, "Message",
 CellID->181120498],
Cell[36478, 1328, 370, 11, 36, "Output",
 CellID->174961344]
}, Open  ]],
Cell[36863, 1342, 93, 1, 70, "ExampleText",
 CellID->10572515],
Cell[CellGroupData[{
Cell[36981, 1347, 136, 4, 70, "Input",
 CellID->165870020],
Cell[37120, 1353, 243, 8, 36, "Output",
 CellID->360109265]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[37424, 1368, 311, 9, 70, "SeeAlsoSection",
 CellID->31315],
Cell[37738, 1379, 1755, 62, 70, "SeeAlso",
 CellID->8055]
}, Open  ]],
Cell[CellGroupData[{
Cell[39530, 1446, 314, 9, 70, "TutorialsSection",
 CellID->12491],
Cell[39847, 1457, 161, 3, 70, "Tutorials",
 CellID->11685]
}, Open  ]],
Cell[CellGroupData[{
Cell[40045, 1465, 319, 9, 70, "MoreAboutSection",
 CellID->143431254],
Cell[40367, 1476, 137, 3, 70, "MoreAbout",
 CellID->286805059]
}, Open  ]],
Cell[40519, 1482, 50, 0, 70, "History"],
Cell[40572, 1484, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

