(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     46871,       1648]
NotebookOptionsPosition[     38365,       1350]
NotebookOutlinePosition[     39700,       1387]
CellTagsIndexPosition[     39614,       1382]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:tutorial/DescriptiveStatistics"], 
          "Discrete Distributions" :> 
          Documentation`HelpLookup["paclet:tutorial/DiscreteDistributions"], 
          "Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"Discrete Distributions\"\>", 
       3->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Kurtosis" :> Documentation`HelpLookup["paclet:ref/Kurtosis"], 
          "QuartileSkewness" :> 
          Documentation`HelpLookup["paclet:ref/QuartileSkewness"], 
          "CentralMoment" :> 
          Documentation`HelpLookup["paclet:ref/CentralMoment"], 
          "ExpectedValue" :> 
          Documentation`HelpLookup["paclet:ref/ExpectedValue"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Kurtosis\"\>", 
       2->"\<\"QuartileSkewness\"\>", 3->"\<\"CentralMoment\"\>", 
       4->"\<\"ExpectedValue\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "Statistical Distributions" :> 
          Documentation`HelpLookup["paclet:guide/StatisticalDistributions"], 
          "New in 6.0: Data Handling & Data Sources" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60DataHandlingAndDataSources"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Numerical Data Handling" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumericalDataHandling"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"Statistical Distributions\"\>", 
       3->"\<\"New in 6.0: Data Handling & Data Sources\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       5->"\<\"New in 6.0: Numerical Data Handling\"\>", 
       6->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["Skewness", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Skewness",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Skewness"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the coefficient of skewness for the elements in ",
     
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["Skewness",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Skewness"], "[", 
       StyleBox["dist", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the coefficient of skewness for the symbolic \
distribution ",
     Cell[BoxData[
      StyleBox["dist", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->6122]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " measures the amount of asymmetry in ",
 Cell[BoxData[
  StyleBox["list", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->12782],

Cell["\<\
A positive skewness indicates a distribution with a long right tail. A \
negative skewness indicates a distribution with a long left tail.\
\>", "Notes",
 CellID->20760],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " handles both numerical and symbolic data. "
}], "Notes",
 CellID->28487],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Skewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Skewness"], "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["1", "TR"]], ",", 
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["1", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["2", "TR"]], ",", 
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["2", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Skewness", "[", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["1", "TR"]], ",", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["2", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}], ",", 
    RowBox[{"Skewness", "[", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["1", "TR"]], ",", 
       SubscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["2", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"]}], "}"}], "]"}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->26555],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->22482],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["Skewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Skewness"], "[", 
   StyleBox["list", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    ButtonBox["CentralMoment",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/CentralMoment"], "[", 
    RowBox[{
     StyleBox["list", "TI"], ",", "3"}], "]"}], "/", 
   SuperscriptBox[
    RowBox[{"CentralMoment", "[", 
     RowBox[{
      StyleBox["list", "TI"], ",", "2"}], "]"}], 
    RowBox[{"3", "/", "2"}]]}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->17065]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->402592909],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->219712037],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " for a list of values:"
}], "ExampleText",
 CellID->196404163],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6426],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox["3"], " ", 
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"a", "+", 
        RowBox[{
         FractionBox["1", "3"], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], ")"}], "3"], 
     "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"b", "+", 
        RowBox[{
         FractionBox["1", "3"], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], ")"}], "3"], 
     "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         FractionBox["1", "3"], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}], "+", "c"}], 
       ")"}], "3"]}], ")"}]}], 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"a", "+", 
        RowBox[{
         FractionBox["1", "3"], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], ")"}], "2"], 
     "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"b", "+", 
        RowBox[{
         FractionBox["1", "3"], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}]}], ")"}], "2"], 
     "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         FractionBox["1", "3"], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "a"}], "-", "b", "-", "c"}], ")"}]}], "+", "c"}], 
       ")"}], "2"]}], ")"}], 
   RowBox[{"3", "/", "2"}]]]], "Output",
 ImageSize->{435, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->547088145]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->448710962],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " for a symbolic distribution:"
}], "ExampleText",
 CellID->25999718],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"ChiSquareDistribution", "[", "10", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->302726116],

Cell[BoxData[
 FractionBox["2", 
  SqrtBox["5"]]], "Output",
 ImageSize->{32, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->566955566]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8378],

Cell["Compute results at machine precision:", "ExampleText",
 CellID->379986608],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", "20"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->216003675],

Cell[BoxData["0.5800128540227802`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->552936037]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158391064],

Cell["Obtain results at any precision: ", "ExampleText",
 CellID->160553898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "4", ",", "5", ",", "15", ",", "10", ",", "8"}], "}"}],
     ",", "30"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190699768],

Cell[BoxData["0.40943853239194976677597129829021373107`28.459844780126563"], \
"Output",
 ImageSize->{214, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->631515806]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->183534688],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " of a matrix will compute column-wise:"
}], "ExampleText",
 CellID->111413134],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "3"}], ",", 
      RowBox[{"-", "4"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "5"}], ",", "6"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->48341596],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"5", " ", 
     SqrtBox[
      FractionBox["2", "7"]]}], "7"], ",", 
   RowBox[{"-", 
    FractionBox[
     RowBox[{"14", " ", 
      SqrtBox[
       FractionBox["2", "19"]]}], "19"]}]}], "}"}]], "Output",
 ImageSize->{137, 55},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->498316190]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->650909219],

Cell["Compute results for a large vector or matrix:", "ExampleText",
 CellID->285474028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"10", "^", "7"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->315224713],

Cell[BoxData["0.00048403350906302896`"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->404729511]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"10", "^", "6"}], ",", "10"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->594479966],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.0006808093280225294`", ",", 
   RowBox[{"-", "0.002129324671648851`"}], ",", 
   RowBox[{"-", "0.000304092885985329`"}], ",", 
   RowBox[{"-", "0.0011911359569329425`"}], ",", "0.0006704139872469002`", 
   ",", "0.003103693911952698`", ",", 
   RowBox[{"-", "0.001559500094720365`"}], ",", "0.0008648100617925361`", ",",
    
   RowBox[{"-", "0.0015241158457608972`"}], ",", "0.000875339181842164`"}], 
  "}"}]], "Output",
 ImageSize->{457, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->276539571]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->373178873],

Cell["Obtain results for continuous and discrete distributions:", \
"ExampleText",
 CellID->17058508],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"GammaDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->108321647],

Cell[BoxData[
 FractionBox["2", 
  SqrtBox["\[Alpha]"]]], "Output",
 ImageSize->{32, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->491235075]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"HypergeometricDistribution", "[", 
   RowBox[{"n", ",", "ns", ",", "nt"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->130372047],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox[
    RowBox[{
     RowBox[{"-", "1"}], "+", "nt"}]], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], " ", "n"}], "+", "nt"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], " ", "ns"}], "+", "nt"}], ")"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "2"}], "+", "nt"}], ")"}], " ", 
   SqrtBox[
    RowBox[{"n", " ", "ns", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "n"}], "+", "nt"}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "ns"}], "+", "nt"}], ")"}]}]]}]]], "Output",
 ImageSize->{231, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->96096667]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14419],

Cell[TextData[{
 "Compute results for a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->625271644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sp", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], ":>", "i"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "j_"}], "}"}], "/;", 
        RowBox[{"j", "\[Equal]", 
         RowBox[{"i", "+", "1"}]}]}], "\[RuleDelayed]", 
       RowBox[{"i", "-", "1"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "5"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19878761],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "8", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 8, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "5"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->139208492]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", "sp", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->235263230],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["98", 
    RowBox[{"3", " ", 
     SqrtBox["11"]}]], ",", 
   FractionBox["98", 
    RowBox[{"3", " ", 
     SqrtBox["11"]}]], ",", 
   FractionBox[
    RowBox[{"931", " ", 
     SqrtBox[
      FractionBox["6", "41"]]}], "41"], ",", 
   FractionBox["85554", 
    RowBox[{"491", " ", 
     SqrtBox["491"]}]], ",", 
   FractionBox[
    RowBox[{"4998", " ", 
     SqrtBox[
      FractionBox["6", "139"]]}], "139"]}], "}"}]], "Output",
 ImageSize->{355, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->61729602]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21168],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " computed from ",
 Cell[BoxData[
  ButtonBox["CentralMoment",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CentralMoment"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->209192808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"10", ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->388198497],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.1444304748387757`", ",", "3.951077164615633`", ",", 
   "3.2672541159838064`", ",", "1.1803531995095429`", ",", 
   "6.558972658550706`", ",", "3.3766183931907667`", ",", 
   "3.782436781150922`", ",", "9.952684900997713`", ",", "7.779405234515917`",
    ",", "8.548716921997524`", ",", "8.37949852170388`", ",", 
   "4.832074058279401`", ",", "3.43995179018171`", ",", "4.350057565342196`", 
   ",", "6.0478651501271425`", ",", "5.837873979021633`", ",", 
   "2.860565030671496`", ",", "5.99914362210148`", ",", "9.108754179481643`", 
   ",", "9.949198984605356`"}], "}"}]], "Output",
 ImageSize->{422, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->188757594]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", "data", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->934643318],

Cell[BoxData["0.1570329707395831`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->620029837]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CentralMoment", "[", 
   RowBox[{"data", ",", "3"}], "]"}], "/", 
  RowBox[{
   RowBox[{"CentralMoment", "[", 
    RowBox[{"data", ",", "2"}], "]"}], "^", 
   RowBox[{"(", 
    RowBox[{"3", "/", "2"}], ")"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->199964197],

Cell[BoxData["0.1570329707395831`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->262134308]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->203852233],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Skewness",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Skewness"]], "InlineFormula"],
 " for a distribution can be computed from ",
 Cell[BoxData[
  ButtonBox["ExpectedValue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ExpectedValue"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["StandardDeviation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/StandardDeviation"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->212776658],

Cell[BoxData[
 RowBox[{
  RowBox[{"dist", "=", 
   RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->107789233],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ev", "=", 
  RowBox[{
   RowBox[{"ExpectedValue", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", 
        RowBox[{"Mean", "[", "dist", "]"}]}], ")"}], "^", "3"}], ",", "dist", 
     ",", "x"}], "]"}], "/", 
   RowBox[{
    RowBox[{"StandardDeviation", "[", "dist", "]"}], "^", "3"}]}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->273951061],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   SqrtBox["2"]}], 
  SqrtBox["\[Nu]"]]], "Output",
 ImageSize->{42, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->6974295]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"skew", "=", 
  RowBox[{"Skewness", "[", "dist", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->370222015],

Cell[BoxData[
 RowBox[{"2", " ", 
  SqrtBox["2"], " ", 
  SqrtBox[
   FractionBox["1", "\[Nu]"]]}]], "Output",
 ImageSize->{72, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->308856532]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"skew", "==", "ev"}], ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{"\[Nu]", ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->3806373],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->189638888]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->75092913],

Cell["Symmetric distributions have 0 skewness:", "ExampleText",
 CellID->659817507],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"NormalDistribution", "[", 
   RowBox[{"\[Mu]", ",", "\[Sigma]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->590782898],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->123525745]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{"n", ",", 
    RowBox[{"1", "/", "2"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->929317708],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->496723811]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->52659688],

Cell["\<\
Distributions with longer tails to the right have positive skewness:\
\>", "ExampleText",
 CellID->28110729],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"ChiSquareDistribution", "[", "5", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->230688773],

Cell[BoxData[
 RowBox[{"2", " ", 
  SqrtBox[
   FractionBox["2", "5"]]}]], "Output",
 ImageSize->{45, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->214801854]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ChiSquareDistribution", "[", "5", "]"}], ",", "x"}], "]"}], ",",
    
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->55753558],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->134342993]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->426881912],

Cell["\<\
Distributions with longer tails to the left have negative skewness:\
\>", "ExampleText",
 CellID->350183472],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"GumbelDistribution", "[", 
   RowBox[{"3", ",", "2"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->213832539],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{"12", " ", 
    SqrtBox["6"], " ", 
    RowBox[{"Zeta", "[", "3", "]"}]}], 
   SuperscriptBox["\[Pi]", "3"]]}]], "Output",
 ImageSize->{109, 36},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->139182243]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"GumbelDistribution", "[", 
      RowBox[{"3", ",", "2"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "10"}], ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->119583769],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9WE1oXUUUnrz38l6TNDZN05o0jUlq2iS+pv9afxpKrbbWGsWftmpwEYOQ
LLRS40Ld6EY3utGNBbELf0DtomKpCBXEglgQC2JBVJAKUhcWweIiIFy/c87M
PfedN7mpJPhI7r0zc77zf87MvWMTM1NPPDkxMz050bfv6MTTU9OTz/TtPXIU
U8UG59wp/D/X6fCc4In/TyUJ7vXXCq5FEBRwL+A+4Mdj/r7OrBcZtYSuJeLL
P7/uijozYMZjZnx9lAMp30B3mRJh/Etl50rtxoUIGhdB+v9j4/xS1nrP3+vv
6xeNa57u665CSiXRnAiaZTUdMOsSzvocHDeyD6kNjbOz/5z/9jf5u3Jllnln
aW/hsVB9evoH13H87W9c8aMPv/PrIrOM64ngBeGc4SHjC9//LmJcgS4lYuQc
rq+8/EXhyFOfuOL9972za/TYgf3HHY9Lb7z+FZY/P/Mz6FsMT5uBq/7j+jXz
0EstwCPQDsqc/fKXa2MUro+uraAT2+AYqPzSi2eAgS2HD77/7MxpzAB/6dJf
bqPhsZ3HQIOcvNsVlTFE1ybgz339K3i98Pxn4Atn4QHyLl78091gcNtC1CCd
nM1auhEOxuXLf0Mb+Pax8Q/AB/bBwyBdH+WBZShGj8JjC13L0AWzMBPBAgX4
hVwUFGTCInrsZxTbWYYQoET99949D7P7jcytPJ58/MRPP/6h6B0cIMg89uY5
mP3aq2fJl70Gy5pRZsK0JFSg5C9l+ccnL0AssLDf9USxKlewO3kWQHgK+ro1
BrWZxzAeJiWhH7nRkDnQFkCk/eoobnrqJNYUt4tnEU0sUG10eY8KNRSDekno
im43z4IO1OQNm52beAwVoLiibk98JYIX5YWtCkHBx3CSovaELEU2UZauNKiN
Jubc29wdIQeRIGRPRwRVwALnndsb7PHRa/fWi0aIHIxU3vtCrMGbHpcb3iPB
lwgAPYqMu3gWFnhUWxQFh8FtitpvPLksioLpcICi7uZZVAiKK4n0mw3BO6gg
XN1gBqV1G0ehDWCdHgV1IMQHUaVcaPW+GzGaC/U9IQegMkleGpWhtSCoseBx
Hx/bj6tBB9hDj0NpnNKdyVOjWPi0QXtFMXKWqJpsGtL8qNkvS2HGW+1nMpzs
nlz1GTc8H0c4B2U1B0caVk3M59URd7Seq9TRFe/c8xZCMy/nsszQvu2clGwh
4lHZHbSD5Npfyc4gf+WBd/h6npoPwzXxLvs7s0SuImP9jLVf+GidCJ8HeBZR
Rf1Sjsala1cQ1IM8C4XkeMJ+IJlCDf+gsaVoT41ZeIX2BOs15pnp7oJ6yNht
7Rn2OVbNUMMCqEoyYtSuCAEQozIOJrV9qhhFodAQeEXxqS5Tt+GNQqi1MqsZ
arXO5iHnXabTCOowuzWV4XND/Kt9M0ubSqihVYuF9mF2pe47ri7CWTr1frAO
RCClxw1Kl/E7+WHIWCyUj2Ti6VJ+akuWSrq1UqkVQvVoyD9xo5eoNmSp6NQm
VIMmPty19czuJcjpTLuiUpFtgyYrDQ9RwfPQWjNU3glChcKDikoV6+ThMWQa
VxB1FBiDWMJdA8FvaGj0yCeFSOcp183YfCxkKORMKrylVS+cNw35TObn1hqP
LlDzxnQsxSNnvszpetH4+ygKf+0GC+RfTscoJ6QTQtofYutPMJtq8iTsAOHd
tLGWC36Sh8JF9ybhMl5HLZkv1DiRYzOtpdbc1Mzt5TFIATC863Zgrfhe47k8
lO4p1xlvbI6iRMN0Q/TvIWp9HgrO8ttxj/GConSHhQ7QjE4GPcYLcRklQ7XG
eCEPpV2p23hhSxQlO632xdXGC3OjaGftNtbnydB9QWRovSmKfFbhVT3PdRnr
4zLkVKL7dCePtZNvzUGpjzqND+OoitFf3ve0P+WhlLegdN9SFHmhycRF3g71
nBGXscTHZZWJ4rY5qbNxWWmiKai83UaSLvzsqa3JRLLD+E34z9EH62pu7nO6
yNGa7DC+W1w5epJZYfy3fQFyaNjM83pmbk9qczqXv84k8TePZpMT7SYaudwb
s9xj7yDyDqrfUZabGAh3uyOVkvibSIuxvM14+sYkls8tIT6+mgWldakoktnK
q9pl5VuC7gdxGUuNF+WrQHrIcjdFUa2Gt4w1Y/NQekqWsXaMOEo00n5pvbIj
B6WyWoyGiiLftRkZzca+uIxlRobUk34ZujmKajOZJV1LqyQPBfUAJKul+6YH
JvkWWYeSrNWvNGWTI3kocEWrg5K+8vQ0fGuN71YETbBKp89iyG2MqavcFpXB
qIpQQR/5+uy/tuyMIrgL+q+t8uGbTiCjUdpDObOu4V+Bw744\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 121},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->816491292]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->4486],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Kurtosis",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Kurtosis"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["QuartileSkewness",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/QuartileSkewness"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CentralMoment",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CentralMoment"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ExpectedValue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ExpectedValue"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->28908]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->31930],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DescriptiveStatistics"]], "Tutorials",
 CellID->4063544],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->642398046],

Cell[TextData[ButtonBox["Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/StatisticalDistributions"]], "MoreAbout",
 CellID->361033257],

Cell[TextData[ButtonBox["New in 6.0: Data Handling & Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60DataHandlingAndDataSources"]], "MoreAbout",
 CellID->670520744],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->139847363],

Cell[TextData[ButtonBox["New in 6.0: Numerical Data Handling",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60NumericalDataHandling"]], "MoreAbout",
 CellID->44022296],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->828964]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Skewness - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 21, 11.6521080}", "context" -> "System`", 
    "keywords" -> {"asymmetry", "shape", "shape statistic", "skew"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "Skewness[list] gives the coefficient of skewness for the elements in \
list. Skewness[dist] gives the coefficient of skewness for the symbolic \
distribution dist.", "synonyms" -> {}, "title" -> "Skewness", "type" -> 
    "Symbol", "uri" -> "ref/Skewness"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[10267, 320, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->402592909]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 39470, 1375}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3403, 75, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4026, 102, 1573, 37, 70, "ObjectNameGrid"],
Cell[5602, 141, 987, 32, 70, "Usage",
 CellID->6122]
}, Open  ]],
Cell[CellGroupData[{
Cell[6626, 178, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7104, 195, 270, 10, 70, "Notes",
 CellID->12782],
Cell[7377, 207, 179, 4, 70, "Notes",
 CellID->20760],
Cell[7559, 213, 210, 7, 70, "Notes",
 CellID->28487],
Cell[7772, 222, 1507, 54, 70, "Notes",
 CellID->26555],
Cell[9282, 278, 319, 12, 70, "Notes",
 CellID->22482],
Cell[9604, 292, 626, 23, 70, "Notes",
 CellID->17065]
}, Closed]],
Cell[CellGroupData[{
Cell[10267, 320, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->402592909],
Cell[CellGroupData[{
Cell[10652, 334, 148, 5, 70, "ExampleSection",
 CellID->219712037],
Cell[10803, 341, 199, 7, 70, "ExampleText",
 CellID->196404163],
Cell[CellGroupData[{
Cell[11027, 352, 156, 5, 28, "Input",
 CellID->6426],
Cell[11186, 359, 1843, 68, 72, "Output",
 CellID->547088145]
}, Open  ]],
Cell[13044, 430, 125, 3, 70, "ExampleDelimiter",
 CellID->448710962],
Cell[13172, 435, 205, 7, 70, "ExampleText",
 CellID->25999718],
Cell[CellGroupData[{
Cell[13402, 446, 153, 4, 70, "Input",
 CellID->302726116],
Cell[13558, 452, 190, 7, 55, "Output",
 CellID->566955566]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13797, 465, 215, 7, 70, "ExampleSection",
 CellID->8378],
Cell[14015, 474, 80, 1, 70, "ExampleText",
 CellID->379986608],
Cell[CellGroupData[{
Cell[14120, 479, 166, 5, 70, "Input",
 CellID->216003675],
Cell[14289, 486, 176, 5, 36, "Output",
 CellID->552936037]
}, Open  ]],
Cell[14480, 494, 125, 3, 70, "ExampleDelimiter",
 CellID->158391064],
Cell[14608, 499, 76, 1, 70, "ExampleText",
 CellID->160553898],
Cell[CellGroupData[{
Cell[14709, 504, 255, 8, 70, "Input",
 CellID->190699768],
Cell[14967, 514, 219, 6, 36, "Output",
 CellID->631515806]
}, Open  ]],
Cell[15201, 523, 125, 3, 70, "ExampleDelimiter",
 CellID->183534688],
Cell[15329, 528, 215, 7, 70, "ExampleText",
 CellID->111413134],
Cell[CellGroupData[{
Cell[15569, 539, 379, 14, 70, "Input",
 CellID->48341596],
Cell[15951, 555, 404, 16, 76, "Output",
 CellID->498316190]
}, Open  ]],
Cell[16370, 574, 125, 3, 70, "ExampleDelimiter",
 CellID->650909219],
Cell[16498, 579, 88, 1, 70, "ExampleText",
 CellID->285474028],
Cell[CellGroupData[{
Cell[16611, 584, 191, 6, 70, "Input",
 CellID->315224713],
Cell[16805, 592, 180, 5, 36, "Output",
 CellID->404729511]
}, Open  ]],
Cell[CellGroupData[{
Cell[17022, 602, 245, 8, 70, "Input",
 CellID->594479966],
Cell[17270, 612, 595, 15, 54, "Output",
 CellID->276539571]
}, Open  ]],
Cell[17880, 630, 125, 3, 70, "ExampleDelimiter",
 CellID->373178873],
Cell[18008, 635, 101, 2, 70, "ExampleText",
 CellID->17058508],
Cell[CellGroupData[{
Cell[18134, 641, 185, 5, 70, "Input",
 CellID->108321647],
Cell[18322, 648, 197, 7, 55, "Output",
 CellID->491235075]
}, Open  ]],
Cell[CellGroupData[{
Cell[18556, 660, 193, 5, 70, "Input",
 CellID->130372047],
Cell[18752, 667, 776, 30, 61, "Output",
 CellID->96096667]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19577, 703, 239, 7, 70, "ExampleSection",
 CellID->14419],
Cell[19819, 712, 211, 8, 70, "ExampleText",
 CellID->625271644],
Cell[CellGroupData[{
Cell[20055, 724, 569, 19, 70, "Input",
 CellID->19878761],
Cell[20627, 745, 481, 17, 36, "Output",
 CellID->139208492]
}, Open  ]],
Cell[CellGroupData[{
Cell[21145, 767, 105, 3, 70, "Input",
 CellID->235263230],
Cell[21253, 772, 616, 24, 80, "Output",
 CellID->61729602]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21918, 802, 233, 7, 70, "ExampleSection",
 CellID->21168],
Cell[22154, 811, 327, 12, 70, "ExampleText",
 CellID->209192808],
Cell[CellGroupData[{
Cell[22506, 827, 158, 5, 70, "Input",
 CellID->388198497],
Cell[22667, 834, 761, 16, 71, "Output",
 CellID->188757594]
}, Open  ]],
Cell[CellGroupData[{
Cell[23465, 855, 107, 3, 70, "Input",
 CellID->934643318],
Cell[23575, 860, 176, 5, 36, "Output",
 CellID->620029837]
}, Open  ]],
Cell[CellGroupData[{
Cell[23788, 870, 300, 10, 70, "Input",
 CellID->199964197],
Cell[24091, 882, 176, 5, 36, "Output",
 CellID->262134308]
}, Open  ]],
Cell[24282, 890, 125, 3, 70, "ExampleDelimiter",
 CellID->203852233],
Cell[24410, 895, 500, 17, 70, "ExampleText",
 CellID->212776658],
Cell[24913, 914, 166, 5, 70, "Input",
 CellID->107789233],
Cell[CellGroupData[{
Cell[25104, 923, 397, 14, 70, "Input",
 CellID->273951061],
Cell[25504, 939, 228, 9, 60, "Output",
 CellID->6974295]
}, Open  ]],
Cell[CellGroupData[{
Cell[25769, 953, 133, 4, 70, "Input",
 CellID->370222015],
Cell[25905, 959, 240, 9, 60, "Output",
 CellID->308856532]
}, Open  ]],
Cell[CellGroupData[{
Cell[26182, 973, 223, 7, 70, "Input",
 CellID->3806373],
Cell[26408, 982, 161, 5, 36, "Output",
 CellID->189638888]
}, Open  ]],
Cell[26584, 990, 124, 3, 70, "ExampleDelimiter",
 CellID->75092913],
Cell[26711, 995, 83, 1, 70, "ExampleText",
 CellID->659817507],
Cell[CellGroupData[{
Cell[26819, 1000, 184, 5, 70, "Input",
 CellID->590782898],
Cell[27006, 1007, 158, 5, 36, "Output",
 CellID->123525745]
}, Open  ]],
Cell[CellGroupData[{
Cell[27201, 1017, 200, 6, 70, "Input",
 CellID->929317708],
Cell[27404, 1025, 158, 5, 36, "Output",
 CellID->496723811]
}, Open  ]],
Cell[27577, 1033, 124, 3, 70, "ExampleDelimiter",
 CellID->52659688],
Cell[27704, 1038, 118, 3, 70, "ExampleText",
 CellID->28110729],
Cell[CellGroupData[{
Cell[27847, 1045, 152, 4, 70, "Input",
 CellID->230688773],
Cell[28002, 1051, 214, 8, 60, "Output",
 CellID->214801854]
}, Open  ]],
Cell[CellGroupData[{
Cell[28253, 1064, 293, 10, 70, "Input",
 CellID->55753558],
Cell[28549, 1076, 2486, 45, 70, "Output",
 Evaluatable->False,
 CellID->134342993]
}, Open  ]],
Cell[31050, 1124, 125, 3, 70, "ExampleDelimiter",
 CellID->426881912],
Cell[31178, 1129, 118, 3, 70, "ExampleText",
 CellID->350183472],
Cell[CellGroupData[{
Cell[31321, 1136, 173, 5, 70, "Input",
 CellID->213832539],
Cell[31497, 1143, 308, 11, 57, "Output",
 CellID->139182243]
}, Open  ]],
Cell[CellGroupData[{
Cell[31842, 1159, 336, 11, 70, "Input",
 CellID->119583769],
Cell[32181, 1172, 2579, 46, 70, "Output",
 Evaluatable->False,
 CellID->816491292]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[34821, 1225, 310, 9, 70, "SeeAlsoSection",
 CellID->4486],
Cell[35134, 1236, 977, 35, 70, "SeeAlso",
 CellID->28908]
}, Open  ]],
Cell[CellGroupData[{
Cell[36148, 1276, 314, 9, 70, "TutorialsSection",
 CellID->31930],
Cell[36465, 1287, 156, 3, 70, "Tutorials",
 CellID->4063544],
Cell[36624, 1292, 153, 3, 70, "Tutorials",
 CellID->4738],
Cell[36780, 1297, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[36974, 1305, 299, 8, 70, "MoreAboutSection"],
Cell[37276, 1315, 155, 3, 70, "MoreAbout",
 CellID->642398046],
Cell[37434, 1320, 161, 3, 70, "MoreAbout",
 CellID->361033257],
Cell[37598, 1325, 185, 3, 70, "MoreAbout",
 CellID->670520744],
Cell[37786, 1330, 179, 3, 70, "MoreAbout",
 CellID->139847363],
Cell[37968, 1335, 174, 3, 70, "MoreAbout",
 CellID->44022296],
Cell[38145, 1340, 148, 3, 70, "MoreAbout",
 CellID->828964]
}, Open  ]],
Cell[38308, 1346, 27, 0, 70, "History"],
Cell[38338, 1348, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

