(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     31939,       1089]
NotebookOptionsPosition[     25866,        879]
NotebookOutlinePosition[     27175,        916]
CellTagsIndexPosition[     27090,        911]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BesselJ" :> Documentation`HelpLookup["paclet:ref/BesselJ"], 
          "SphericalBesselY" :> 
          Documentation`HelpLookup["paclet:ref/SphericalBesselY"], 
          "SphericalHankelH1" :> 
          Documentation`HelpLookup["paclet:ref/SphericalHankelH1"], 
          "SphericalHankelH2" :> 
          Documentation`HelpLookup["paclet:ref/SphericalHankelH2"], 
          "SpheroidalS1" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalS1"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"BesselJ\"\>", 
       2->"\<\"SphericalBesselY\"\>", 3->"\<\"SphericalHankelH1\"\>", 
       4->"\<\"SphericalHankelH2\"\>", 
       5->"\<\"SpheroidalS1\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "Functions for Separable Coordinate Systems" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsForSeparableCoordinateSystems"], 
          "Functions Used in Optics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInOptics"], 
          "Functions Used in Quantum Mechanics" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsUsedInQuantumMechanics"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"Functions for Separable Coordinate Systems\"\>", 
       3->"\<\"Functions Used in Optics\"\>", 
       4->"\<\"Functions Used in Quantum Mechanics\"\>", 
       5->"\<\"Special Functions\"\>", 
       6->"\<\"New in 6.0: Mathematical Functions\"\>", 
       7->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SphericalBesselJ", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SphericalBesselJ",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SphericalBesselJ"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the spherical Bessel function of the first kind ",
     
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["j", "n"], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->29493]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselJ"]], "InlineFormula"],
 " is given in terms of ordinary Bessel functions by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["j", "n"], "(", "z", ")"}], "=", 
    RowBox[{
     RowBox[{
      SqrtBox[
       RowBox[{"\[Pi]", "/", "2"}]], "/", 
      SqrtBox["z"]}], 
     RowBox[{
      SubscriptBox["J", 
       RowBox[{"n", "+", 
        FractionBox["1", "2"]}]], "(", "z", ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Notes",
 CellID->8993],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SphericalBesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalBesselJ"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity for non\[Hyphen]integer ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " in the complex ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->1299],

Cell[TextData[{
 "Explicit symbolic forms for integer ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " can be obtained using ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->17856],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["SphericalBesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselJ"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->643501970],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselJ"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselJ"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->107609304],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->3685133],

Cell["Evaluate numerically:", "ExampleText",
 CellID->310104354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselJ", "[", 
  RowBox[{"1", ",", "5.2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->693268908],

Cell[BoxData[
 RowBox[{"-", "0.12277149950007797`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->491687479]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->141994335],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["j", "1"], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->49178],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SphericalBesselJ", "[", 
    RowBox[{"1", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->241358785],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 127},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->157453368]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5982],

Cell[TextData[{
 "Find explicit symbolic forms using ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->620889108],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"SphericalBesselJ", "[", 
   RowBox[{"1", ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->39824096],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox[
    RowBox[{"Cos", "[", "x", "]"}], "x"]}], "+", 
  FractionBox[
   RowBox[{"Sin", "[", "x", "]"}], 
   SuperscriptBox["x", "2"]]}]], "Output",
 ImageSize->{117, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->11507957]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"SphericalBesselJ", "[", 
   RowBox[{
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->86659175],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox[
    FractionBox["\[Pi]", "2"]], " ", 
   RowBox[{"BesselJ", "[", 
    RowBox[{"1", ",", "x"}], "]"}]}], 
  SqrtBox["x"]]], "Output",
 ImageSize->{129, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->74913498]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->82719092],

Cell["Evaluate for complex arguments and orders:", "ExampleText",
 CellID->168756706],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselJ", "[", 
  RowBox[{
   RowBox[{"3.2", "+", 
    RowBox[{"1.9", " ", "I"}]}], ",", " ", 
   RowBox[{"4.9", "-", 
    RowBox[{"0.2", " ", "I"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->593547515],

Cell[BoxData[
 RowBox[{"0.48230951466975575`", "\[InvisibleSpace]", "-", 
  RowBox[{"0.2946503001461194`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->367509413]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->133153095],

Cell["Evaluate numerically to high precision:", "ExampleText",
 CellID->339175458],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", " ", 
  RowBox[{
   RowBox[{"SphericalBesselJ", "[", 
    RowBox[{"1", ",", " ", "3"}], "]"}], ",", " ", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->89131405],

Cell[BoxData["0.\
3456774997623559548794959096668771318919967282332256209475540639925`50."], \
"Output",
 ImageSize->{368, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->21750291]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->282337929],

Cell["The precision of the output tracks precision of the input:", \
"ExampleText",
 CellID->175906109],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselJ", "[", 
  RowBox[{"1", ",", "3.000000000000000000"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->272242306],

Cell[BoxData["0.34567749976235595488175401525`17.36957739418189"], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->875180872]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->716662293],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SphericalBesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SphericalBesselJ"]], "InlineFormula"],
 " threads element-wise over lists:"
}], "ExampleText",
 CellID->17043502],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SphericalBesselJ", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", " ", "4.5"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->552427445],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.001429581245757399`"}], ",", "0.21627586087284995`", ",", 
   "0.2417360933267018`"}], "}"}]], "Output",
 ImageSize->{231, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->54092867]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->714138040],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->318883205],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SphericalBesselJ", "[", 
   RowBox[{"n", ",", "r"}], "]"}], "//", "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->58815609],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["j", "n"], "(", "r", ")"}], TraditionalForm]], "Output",
 ImageSize->{29, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->76353723]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->10945],

Cell["Solve the radial part of the Laplace operator in 3D:", "ExampleText",
 CellID->1979980135],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"f", "''"}], "[", "r", "]"}], "+", 
     RowBox[{
      FractionBox[
       RowBox[{"2", " "}], "r"], " ", 
      RowBox[{
       RowBox[{"f", "'"}], "[", "r", "]"}]}], "+", 
     RowBox[{"f", "[", "r", "]"}], "-", 
     RowBox[{
      FractionBox[
       RowBox[{"n", 
        RowBox[{"(", 
         RowBox[{"n", "+", "1"}], ")"}]}], 
       SuperscriptBox["r", "2"]], 
      RowBox[{"f", "[", "r", "]"}]}]}], "\[Equal]", "0"}], ",", 
   RowBox[{"f", "[", "r", "]"}], ",", "r"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->64238625],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f", "[", "r", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"SphericalBesselJ", "[", 
       RowBox[{"n", ",", "r"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"SphericalBesselY", "[", 
       RowBox[{"n", ",", "r"}], "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{453, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5973011]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->6497],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   SuperscriptBox[
    RowBox[{"SphericalBesselJ", "[", 
     RowBox[{"1", ",", "r"}], "]"}], "2"], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->110856366],

Cell[BoxData[
 FractionBox["\[Pi]", "6"]], "Output",
 ImageSize->{17, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->613596028]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->2530],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalBesselY",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalBesselY"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalHankelH1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalHankelH1"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalHankelH2",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalHankelH2"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalS1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalS1"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->17939]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->29980],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->25617]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->231017706],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->433710294],

Cell[TextData[ButtonBox["Functions for Separable Coordinate Systems",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/FunctionsForSeparableCoordinateSystems"]], "MoreAbout",
 CellID->41397],

Cell[TextData[ButtonBox["Functions Used in Optics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInOptics"]], "MoreAbout",
 CellID->650115486],

Cell[TextData[ButtonBox["Functions Used in Quantum Mechanics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInQuantumMechanics"]], "MoreAbout",
 CellID->208499873],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->1485248],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->5444862],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->404395631]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SphericalBesselJ - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 22, 22.4820462}", "context" -> "System`", 
    "keywords" -> {"spherical Bessel function", "spherical Bessel J"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "SphericalBesselJ[n, z] gives the spherical Bessel function of the first \
kind j_n (z). ", "synonyms" -> {"spherical bessel J"}, "title" -> 
    "SphericalBesselJ", "type" -> "Symbol", "uri" -> 
    "ref/SphericalBesselJ"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9340, 290, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->107609304]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 26947, 904}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3408, 77, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4031, 104, 1581, 37, 70, "ObjectNameGrid"],
Cell[5615, 143, 723, 24, 70, "Usage",
 CellID->29493]
}, Open  ]],
Cell[CellGroupData[{
Cell[6375, 172, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6853, 189, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6976, 195, 608, 23, 70, "Notes",
 CellID->8993],
Cell[7587, 220, 682, 24, 70, "Notes",
 CellID->1299],
Cell[8272, 246, 307, 11, 70, "Notes",
 CellID->17856],
Cell[8582, 259, 264, 8, 70, "Notes",
 CellID->643501970],
Cell[8849, 269, 234, 7, 70, "Notes",
 CellID->31226],
Cell[9086, 278, 217, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[9340, 290, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->107609304],
Cell[CellGroupData[{
Cell[9725, 304, 146, 5, 70, "ExampleSection",
 CellID->3685133],
Cell[9874, 311, 64, 1, 70, "ExampleText",
 CellID->310104354],
Cell[CellGroupData[{
Cell[9963, 316, 137, 4, 28, "Input",
 CellID->693268908],
Cell[10103, 322, 194, 6, 36, "Output",
 CellID->491687479]
}, Open  ]],
Cell[10312, 331, 125, 3, 70, "ExampleDelimiter",
 CellID->141994335],
Cell[10440, 336, 185, 9, 70, "ExampleText",
 CellID->49178],
Cell[CellGroupData[{
Cell[10650, 349, 252, 8, 28, "Input",
 CellID->241358785],
Cell[10905, 359, 3462, 61, 148, "Output",
 Evaluatable->False,
 CellID->157453368]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14416, 426, 215, 7, 70, "ExampleSection",
 CellID->5982],
Cell[14634, 435, 230, 8, 70, "ExampleText",
 CellID->620889108],
Cell[CellGroupData[{
Cell[14889, 447, 176, 5, 70, "Input",
 CellID->39824096],
Cell[15068, 454, 329, 12, 52, "Output",
 CellID->11507957]
}, Open  ]],
Cell[CellGroupData[{
Cell[15434, 471, 201, 6, 70, "Input",
 CellID->86659175],
Cell[15638, 479, 313, 12, 65, "Output",
 CellID->74913498]
}, Open  ]],
Cell[15966, 494, 124, 3, 70, "ExampleDelimiter",
 CellID->82719092],
Cell[16093, 499, 85, 1, 70, "ExampleText",
 CellID->168756706],
Cell[CellGroupData[{
Cell[16203, 504, 246, 8, 70, "Input",
 CellID->593547515],
Cell[16452, 514, 274, 7, 36, "Output",
 CellID->367509413]
}, Open  ]],
Cell[16741, 524, 125, 3, 70, "ExampleDelimiter",
 CellID->133153095],
Cell[16869, 529, 82, 1, 70, "ExampleText",
 CellID->339175458],
Cell[CellGroupData[{
Cell[16976, 534, 204, 6, 70, "Input",
 CellID->89131405],
Cell[17183, 542, 234, 7, 36, "Output",
 CellID->21750291]
}, Open  ]],
Cell[17432, 552, 125, 3, 70, "ExampleDelimiter",
 CellID->282337929],
Cell[17560, 557, 103, 2, 70, "ExampleText",
 CellID->175906109],
Cell[CellGroupData[{
Cell[17688, 563, 154, 4, 70, "Input",
 CellID->272242306],
Cell[17845, 569, 207, 5, 36, "Output",
 CellID->875180872]
}, Open  ]],
Cell[18067, 577, 125, 3, 70, "ExampleDelimiter",
 CellID->716662293],
Cell[18195, 582, 225, 7, 70, "ExampleText",
 CellID->17043502],
Cell[CellGroupData[{
Cell[18445, 593, 204, 7, 70, "Input",
 CellID->552427445],
Cell[18652, 602, 293, 9, 36, "Output",
 CellID->54092867]
}, Open  ]],
Cell[18960, 614, 125, 3, 70, "ExampleDelimiter",
 CellID->714138040],
Cell[19088, 619, 203, 7, 70, "ExampleText",
 CellID->318883205],
Cell[CellGroupData[{
Cell[19316, 630, 173, 5, 70, "Input",
 CellID->58815609],
Cell[19492, 637, 253, 8, 50, "Output",
 CellID->76353723]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19794, 651, 223, 7, 70, "ExampleSection",
 CellID->10945],
Cell[20020, 660, 96, 1, 70, "ExampleText",
 CellID->1979980135],
Cell[CellGroupData[{
Cell[20141, 665, 642, 22, 70, "Input",
 CellID->64238625],
Cell[20786, 689, 553, 18, 36, "Output",
 CellID->5973011]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21388, 713, 232, 7, 70, "ExampleSection",
 CellID->6497],
Cell[CellGroupData[{
Cell[21645, 724, 293, 9, 70, "Input",
 CellID->110856366],
Cell[21941, 735, 182, 6, 49, "Output",
 CellID->613596028]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[22184, 748, 310, 9, 70, "SeeAlsoSection",
 CellID->2530],
Cell[22497, 759, 1244, 44, 70, "SeeAlso",
 CellID->17939]
}, Open  ]],
Cell[CellGroupData[{
Cell[23778, 808, 314, 9, 70, "TutorialsSection",
 CellID->29980],
Cell[24095, 819, 144, 3, 70, "Tutorials",
 CellID->25617]
}, Open  ]],
Cell[CellGroupData[{
Cell[24276, 827, 319, 9, 70, "MoreAboutSection",
 CellID->231017706],
Cell[24598, 838, 158, 3, 70, "MoreAbout",
 CellID->433710294],
Cell[24759, 843, 191, 4, 70, "MoreAbout",
 CellID->41397],
Cell[24953, 849, 157, 3, 70, "MoreAbout",
 CellID->650115486],
Cell[25113, 854, 178, 3, 70, "MoreAbout",
 CellID->208499873],
Cell[25294, 859, 143, 3, 70, "MoreAbout",
 CellID->1485248],
Cell[25440, 864, 172, 3, 70, "MoreAbout",
 CellID->5444862],
Cell[25615, 869, 179, 3, 70, "MoreAbout",
 CellID->404395631]
}, Open  ]],
Cell[25809, 875, 27, 0, 70, "History"],
Cell[25839, 877, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

