(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     48215,       1589]
NotebookOptionsPosition[     40650,       1326]
NotebookOutlinePosition[     41994,       1364]
CellTagsIndexPosition[     41908,       1359]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpheroidalFunctions"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"Spheroidal Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"LegendreP" :> Documentation`HelpLookup["paclet:ref/LegendreP"],
           "SpheroidalQS" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalQS"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"LegendreP\"\>", 
       2->"\<\"SpheroidalQS\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Functions for Separable Coordinate Systems" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsForSeparableCoordinateSystems"], 
          "Functions Used in Optics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInOptics"], 
          "Functions Used in Quantum Mechanics" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsUsedInQuantumMechanics"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "Spheroidal and Related Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/SpheroidalAndRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Functions for Separable Coordinate Systems\"\>", 
       2->"\<\"Functions Used in Optics\"\>", 
       3->"\<\"Functions Used in Quantum Mechanics\"\>", 
       4->"\<\"Special Functions\"\>", 
       5->"\<\"Spheroidal and Related Functions\"\>", 
       6->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SpheroidalPS", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SpheroidalPS",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SpheroidalPS"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["\[Gamma]", "TR"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the angular spheroidal function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox[
         StyleBox["PS",
          FontSlant->"Italic"], 
         RowBox[{"n", ",", "m"}]], "(", 
        RowBox[{"\[Gamma]", ",", "z"}], ")"}], TraditionalForm]], 
      "InlineMath"],
     " of the first kind."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->27879]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "The angular spheroidal functions satisfy the differential equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        SuperscriptBox["z", "2"]}], ")"}], 
      SuperscriptBox["y", "\[DoublePrime]"]}], "-", 
     RowBox[{"2", "z", " ", 
      SuperscriptBox["y", "\[Prime]"]}], " ", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"\[Lambda]", "+", 
        RowBox[{
         SuperscriptBox["\[Gamma]", "2"], "(", 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}], ")"}], "-", 
        FractionBox[
         SuperscriptBox["m", "2"], 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}]]}], ")"}], "y"}]}], "\[Equal]", "0"}], 
   TraditionalForm]], "InlineMath"],
 " with the spheroidal eigenvalue ",
 Cell[BoxData[
  FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
 " given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalEigenvalue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalEigenvalue"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["\[Gamma]", "TR"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->14006],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalPS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalPS"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", "0", ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["LegendreP",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LegendreP"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->1220],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalPS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalPS"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["a", "TI"], ",", 
    StyleBox["\[Gamma]", "TR"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula", "TemplateInclusion"],
 " gives spheroidal functions of type ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ". The types are specified as for ",
 Cell[BoxData[
  ButtonBox["LegendreP",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LegendreP"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->15719],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->334798374],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->234550160],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->92188727],

Cell["Evaluate numerically:", "ExampleText",
 CellID->102308445],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{"3", ",", "2", ",", "1", ",", "0.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->43173829],

Cell[BoxData["5.65036805385163054344582606782`15.954589770191"], "Output",
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->66556847]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->835947008],

Cell["Expansion about the spherical case: ", "ExampleText",
 CellID->802528156],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"SpheroidalPS", "[", 
    RowBox[{"n", ",", "m", ",", "\[Gamma]", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Gamma]", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->148566906],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   RowBox[{"LegendreP", "[", 
    RowBox[{"n", ",", "m", ",", "2", ",", "x"}], "]"}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "1"}], "+", "m", "+", "n"}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{"m", "+", "n"}], ")"}], " ", 
         RowBox[{"LegendreP", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "2"}], "+", "n"}], ",", "m", ",", "2", ",", "x"}], 
          "]"}]}], 
        RowBox[{"2", " ", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", "1"}], "+", 
            RowBox[{"2", " ", "n"}]}], ")"}], "2"], " ", 
         RowBox[{"(", 
          RowBox[{"1", "+", 
           RowBox[{"2", " ", "n"}]}], ")"}]}]]}], "+", 
      FractionBox[
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", "m", "+", "n"}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{"2", "-", "m", "+", "n"}], ")"}], " ", 
        RowBox[{"LegendreP", "[", 
         RowBox[{
          RowBox[{"2", "+", "n"}], ",", "m", ",", "2", ",", "x"}], "]"}]}], 
       RowBox[{"2", " ", 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          RowBox[{"2", " ", "n"}]}], ")"}], " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"3", "+", 
           RowBox[{"2", " ", "n"}]}], ")"}], "2"]}]]}], ")"}], " ", 
    SuperscriptBox["\[Gamma]", "2"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[Gamma]", "]"}], "3"],
    SeriesData[$CellContext`\[Gamma], 0, {}, 0, 3, 1],
    Editable->False]}],
  SeriesData[$CellContext`\[Gamma], 0, {
    LegendreP[$CellContext`n, $CellContext`m, 2, $CellContext`x], 0, 
    Rational[-1, 
       2] (-1 + $CellContext`m + $CellContext`n) ($CellContext`m + \
$CellContext`n) (-1 + 2 $CellContext`n)^(-2) (1 + 2 $CellContext`n)^(-1) 
     LegendreP[-2 + $CellContext`n, $CellContext`m, 2, $CellContext`x] + 
    Rational[1, 2] (1 - $CellContext`m + $CellContext`n) (
      2 - $CellContext`m + $CellContext`n) (1 + 2 $CellContext`n)^(-1) (3 + 
       2 $CellContext`n)^(-2) 
     LegendreP[2 + $CellContext`n, $CellContext`m, 2, $CellContext`x]}, 0, 3, 
   1],
  Editable->False]], "Output",
 ImageSize->{465, 68},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->188485107]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->434247032],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{
     SubscriptBox[
      StyleBox["PS", "IT"], 
      RowBox[{
       TagBox["2",
        SpheroidalPS,
        Editable->True,
        Selectable->True], ",", 
       TagBox["0",
        SpheroidalPS,
        Editable->True,
        Selectable->True]}]], "(", 
     RowBox[{
      TagBox["1",
       SpheroidalPS,
       Editable->True,
       Selectable->True], ",", 
      TagBox["x",
       SpheroidalPS,
       Editable->True,
       Selectable->True]}], ")"}],
    InterpretTemplate[SpheroidalPS[
      SlotSequence[1]]& ],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->6381167],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalPS", "[", 
    RowBox[{"2", ",", "0", ",", "1", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->213350775],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 110},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->59628066]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->30326],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->94393633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{"3", ",", "2", ",", 
   RowBox[{"1", "-", "I"}], ",", " ", "3.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->314925903],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "188.97371397982096`"}], "-", 
  RowBox[{"652.1234181136421`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{134, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->550536579]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217504807],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->283569038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"SpheroidalPS", "[", 
    RowBox[{"2", ",", "1", ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "2"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->852018633],

Cell[BoxData[
 RowBox[{"-", 
  "1.3008757335723961523287538480278301026325462004955832132803868289439`50."}\
]], "Output",
 ImageSize->{369, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25082042]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3535261],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->173158339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{"2", ",", "1", ",", 
   RowBox[{"1", "/", "3"}], ",", "0.50000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->429954945],

Cell[BoxData[
 RowBox[{"-", 
  "1.30087573357239615232875384802783010295`22.698970004336022"}]], "Output",
 ImageSize->{180, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->652582499]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->165479942],

Cell[TextData[{
 "For certain parameters ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " evaluates exactly:"
}], "ExampleText",
 CellID->160938561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", "=", "3"}], "}"}], ",", 
   RowBox[{"SpheroidalPS", "[", 
    RowBox[{"n", ",", "1", ",", 
     RowBox[{"n", " ", 
      RowBox[{"Pi", "/", "2"}]}], ",", " ", "x"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10306910],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   RowBox[{"Cos", "[", 
    FractionBox[
     RowBox[{"3", " ", "\[Pi]", " ", "x"}], "2"], "]"}], " ", 
   SqrtBox[
    FractionBox["3", 
     RowBox[{"7", " ", 
      RowBox[{"(", 
       RowBox[{"EulerGamma", "-", 
        RowBox[{"CosIntegral", "[", 
         RowBox[{"6", " ", "\[Pi]"}], "]"}], "+", 
        RowBox[{"Log", "[", 
         RowBox[{"6", " ", "\[Pi]"}], "]"}]}], ")"}]}]]]}], 
  SqrtBox[
   RowBox[{"1", "-", 
    SuperscriptBox["x", "2"]}]]]], "Output",
 ImageSize->{298, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->329034363]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", 
   RowBox[{"1", "/", "2"}], ",", "c", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->283753173],

Cell[BoxData[
 FractionBox[
  RowBox[{"MathieuC", "[", 
   RowBox[{
    RowBox[{"MathieuCharacteristicA", "[", 
     RowBox[{"2", ",", 
      FractionBox[
       SuperscriptBox["c", "2"], "4"]}], "]"}], ",", 
    FractionBox[
     SuperscriptBox["c", "2"], "4"], ",", 
    RowBox[{"ArcCos", "[", "x", "]"}]}], "]"}], 
  RowBox[{
   SqrtBox["\[Pi]"], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", 
      SuperscriptBox["x", "2"]}], ")"}], 
    RowBox[{"1", "/", "4"}]]}]]], "Output",
 ImageSize->{379, 49},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->14559301]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->129130650],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->54774772],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SpheroidalPS", "[", 
   RowBox[{"n", ",", "m", ",", "c", ",", "\[Eta]"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->187276340],

Cell[GraphicsData["CompressedBitmap", "\<\
eJyNVNsOgjAMLWyIJireHnzkK/gEffIjiDHBB6NB/h9XtknX1cuStfRk7WnP
Fk5111xudXc91+WxrR/N9fwsD/fWQCoBgMrs9R7Md2++hl31vfHWbtEqRIcF
5YAu0GYWhWH5Y6AshDY1XrN8xeIcjR5jl2EZ1MigQ0aKkmoZiyck603gWlQ+
X9M51F+8adi1WalD0WfiTCh2gt5CRCdN8gOtXGYWMwqnP88aMkcsYR65u+Au
XY/ueoD0EHQ1j5DfHUxYvJSrRrOlYmUtvIsCTc7yXZUZVT9W2FqRmzKsWOz1
mnrdnJZf7kTmCDoqGMsuqqTkKWhnQp0Ni2M1+XvgFaT5/T+EWkhe5u0xMA==
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{54, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->407258497]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21058],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " of different types have different branch cut structures:"
}], "ExampleText",
 CellID->277064462],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{"n", ",", "m", ",", "1", ",", "0", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->135549909],

Cell[BoxData[
 RowBox[{"LegendreP", "[", 
  RowBox[{"n", ",", "m", ",", "2", ",", "x"}], "]"}]], "Output",
 ImageSize->{143, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->132610523]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{"n", ",", "m", ",", "2", ",", "0", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->591864808],

Cell[BoxData[
 RowBox[{"LegendreP", "[", 
  RowBox[{"n", ",", "m", ",", "3", ",", "x"}], "]"}]], "Output",
 ImageSize->{143, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->234870836]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->3759],

Cell[TextData[{
 "Solve the spheroidal differential equation in terms of ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->640948785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"x", "^", "2"}]}], ")"}], 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], "-", 
     RowBox[{"2", "x", " ", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"SpheroidalEigenvalue", "[", 
         RowBox[{"n", ",", "m", ",", "c"}], "]"}], "+", 
        RowBox[{
         RowBox[{"c", "^", "2"}], 
         RowBox[{"(", 
          RowBox[{"1", "-", 
           RowBox[{"x", "^", "2"}]}], ")"}]}], " ", "-", " ", 
        FractionBox[
         RowBox[{"m", "^", "2"}], 
         RowBox[{"1", "-", 
          RowBox[{"x", "^", "2"}]}]]}], ")"}], 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", "y", ",", 
   "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->297005226],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"C", "[", "1", "]"}], " ", 
        RowBox[{"SpheroidalPS", "[", 
         RowBox[{"n", ",", "m", ",", "c", ",", "x"}], "]"}]}], "+", 
       RowBox[{
        RowBox[{"C", "[", "2", "]"}], " ", 
        RowBox[{"SpheroidalQS", "[", 
         RowBox[{"n", ",", "m", ",", "c", ",", "x"}], "]"}]}]}]}], "]"}]}], 
   "}"}], "}"}]], "Output",
 ImageSize->{551, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->187021462]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->377793566],

Cell["\<\
Plot prolate and oblate versions of the same angular function:\
\>", "ExampleText",
 CellID->470010545],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"SpheroidalPS", "[", 
      RowBox[{"2", ",", "1", ",", "3", ",", "x"}], "]"}], ",", 
     RowBox[{"SpheroidalPS", "[", 
      RowBox[{"2", ",", "1", ",", 
       RowBox[{"3", "I"}], ",", "x"}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "0"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "30"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->663030826],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{-0.999999931034483, 0.00075364970798942}, {-0.933668662146553, 
     0.73977952042601}, {-0.861757965711209, 
     1.05329097021701}, {-0.794612669314658, 
     1.24921097186168}, {-0.728784497094833, 
     1.37770765627842}, {-0.657376897327593, 
     1.45562261801857}, {-0.590734697599148, 
     1.47500070603088}, {-0.518513070323288, 
     1.44139752377392}, {-0.447608567224154, 
     1.35680053273108}, {-0.381469464163813, 
     1.23564301321831}, {-0.309750933556058, 
     1.06325817156557}, {-0.242797802987097, 
     0.869342071260823}, {-0.17716179659486, 
     0.654211759662027}, {-0.10594636265521, 
     0.400046474087089}, {-0.0394963287543533, 0.15072413286858}, {
     0.0325331326939167, -0.124219598086428}, {
     0.103245469965462, -0.390090222004138}, {
     0.169192407198214, -0.626700527953346}, {
     0.24071877197838, -0.862875813586141}, {
     0.307479736719752, -1.05716806359138}, {
     0.379820129008537, -1.23213691034954}, {
     0.450843397120598, -1.36171420079305}, {
     0.517101265193866, -1.44019870634847}, {
     0.588938560814547, -1.47483975504499}, {
     0.656010456396434, -1.45652536560805}, {
     0.721765227801597, -1.38810383277488}, {
     0.793099426754174, -1.25282374845805}, {
     0.859668225667957, -1.06054542256181}, {
     0.931816452129153, -0.749915872518492}, {
     0.999999931034483, -0.00075364970798942}}]}, 
   {Hue[0.90606797749979, 0.6, 0.6], 
    LineBox[{{-0.999999931034483, 0.00156061946261031}, {-0.933668662146553, 
     1.29977190197995}, {-0.861757965711209, 
     1.56913453249012}, {-0.794612669314658, 
     1.61476938801719}, {-0.728784497094833, 
     1.56703899218712}, {-0.657376897327593, 
     1.4592389219693}, {-0.590734697599148, 
     1.32967867975563}, {-0.518513070323288, 
     1.17290646525512}, {-0.447608567224154, 
     1.01137295410638}, {-0.381469464163813, 
     0.858535216640846}, {-0.309750933556058, 
     0.69320709107585}, {-0.242797802987097, 
     0.54045555809303}, {-0.17716179659486, 
     0.392537537291886}, {-0.10594636265521, 
     0.233880984998059}, {-0.0394963287543533, 0.0870277797095849}, {
     0.0325331326939167, -0.0716777647247298}, {
     0.103245469965462, -0.227894240641446}, {
     0.169192407198214, -0.374696233884837}, {
     0.24071877197838, -0.53574235971505}, {
     0.307479736719752, -0.687996242165503}, {
     0.379820129008537, -0.85472114882501}, {
     0.450843397120598, -1.01882192752506}, {
     0.517101265193866, -1.16974108433613}, {
     0.588938560814547, -1.32593960564655}, {
     0.656010456396434, -1.45680330698187}, {
     0.721765227801597, -1.55849086356188}, {
     0.793099426754174, -1.61448364132491}, {
     0.859668225667957, -1.57270223940684}, {
     0.931816452129153, -1.31177026304983}, {
     0.999999931034483, -0.00156061946261031}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  ImageMargins->0.,
  ImageSize->Small,
  Method->{},
  PlotRange->{{-1, 1}, {-1.61448364132491, 1.61476938801719}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->166648803]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->33867524],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " are band-limited functions with bandwidth proportional to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "/", "c"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->459211663],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{"Fourier", "[", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"SpheroidalPS", "[", 
        RowBox[{"2", ",", "0", ",", "5.`20", ",", "x"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "10"}], ",", "10", ",", 
         RowBox[{"1", "/", "10"}]}], "}"}]}], "]"}], "]"}], "]"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->197802376],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxdlwtQVHUUxq9krqKBooQI6gVUlFB5CSggRxAQeS3sssubu+8FDUkMRy25
poERvgMfY7aJD/AdoxAWcJ1MNKJaRUHR2kgElFHM0HFktJo9Np3dmZ2d3373
2733/zjn+7soVyRprBiGufLP+99P82sgNP5k1/TlZR2h+AW82lPQUXvrBvJI
CHm34czpq23IYyG2fH9He8jPyBOg6pN+f6XX98gTYU32KdnNa0eRnaEnpYV1
yTuBzEJUSMNvsZ3NyK6w5qmw5LHTVeRpEOxYsLu76CbyDMh92Ne0fscd5JkA
f4jc10l/R/aAMdUd8yuqXrMn3B/b0Hj3dhfyHEhfsPN2xZuvdS9o+cvRZmXL
a/YG6dT6+0c+aEX2gQihY3JOySFkXxgvHzfsnQ3lyH5gtUz3pExehjwPSuU9
Zw5d2oTsD1lW1spabx45AKb5ufQ7R65FDoTB7c/61j14H3k+xE888OOfx1ci
L4Bn++0UxbPykYMg43BC/jD5cuRgsF2910PmlIMcAu77S2dUr9AiL4QRBUfY
Ah81cii0yyfVNcUokQGMm57LX1VzZuYBhndus7Xjs1BfBKm7Nzp8IcpAfRG4
W9WNHNWZinoYGAobgq0eylEPA0WxvfvX/jLUw2GjPOJ60Ekp6uFQvnJO/fgY
CeqLoeeowue+dRLqi+F2ksNqca8Y9QhoGdEmYZsSUI8A5+3OEtvOONQjQfbZ
wcPOU2NRjwT+xeb8huKlqEdB0ZjkRmubaNSjYNNc473DX0WhvgS6vfolB/Mi
UV8CZTWXCwvCIlCPhtPNU0b96rEY9WhobD7b/2B6OOpL4cobW127vMNQXwrX
LiW+Fxi3CPUYaNsxOmdnIaAeA43nxaKC1FDUY0HaWpvpdScE9VgIjGzwXFYY
jHoctLwY76R2C0I9DoxF9i9KuuajHg/Rg8zHuTWBqMeD3/XClgs7AlBPAH8Q
xSmL/FFPgH1O9aVBa+ehLoa69dtskzf4mRnEwDos62jZ5YvXi+HzL7Ntfjnj
Y2ZBDHFPjaLKdm/0J8IJ79LichEyJIJtU67HCPBCfyKktBW5Ccxc9CdCSY84
zHf7bPQnQd+ugU7tbE/0J4GtoNmS1+6B/iQYPavmeN6WWehPgvryPXX74mei
XwLj3n7y3HeSO/olcG5iU/jmR9PRL4EDp6taB1qnoV8CQ09yfI1n3dAvhfO9
P5RUVLqiXwoZx1atr93rgn4pfNfsWn2qgkW/FHoH7MpEflPRnwx9teOCPoqc
jP5kuCsauvVI5YT+ZHgrWHfhw08d0Z8MRx8Hu6q+dUC/DNxS89xNg/bol8GN
2IuOIQET0C+Dl1vv7blSZId+GRwb2W3q/Gks+uXg55AfYIiyQb8cajQH19oO
WaNfDo65/UZjvQj9cui6uKrhm6Lh6E+BE34jTFWxw9CfAlsrs3oHFw4tNPtT
oM4/cLf3w0EzCykQvroy/Vz0IzMzqdC5wd6470KXmSEVPF9OmSd9dRn9FixY
XM+k0d+DNPp/fBq9HyGN3i+TTp8H0unz8ul0PIR0Ol5MBh1PyKDjzWfQ+RAy
6HwxmXQ+IZPON58J3f9fD0ImXS9MFl1PkEXXG59F16OQRdcrk03XM2TT9c5n
0/0gZNP9wnB0P7Ec3W/A0f3IcXS/8hzdzwbOYr9ztB6YOFovGAWtJ6yC1htQ
0HrEKWi94hW0nhkUtN4JCloPTQpaLxklraesktZbUNJ6zCkt6rWS1nODktZ7
QUn7gUlJ+wWjov2EVdF+AyrajziVRb9S0X5mUNF+J6hoPzSpaL9k1LSfsmra
b0FN+zGntujXatrPDWra7wU1zQMmNc0LjIbmCVZD8wZoaB7hNBZ5RUPzjEFD
846goXnIpLHIS1qap1gtzVugpXmM09K8xmtpnjNoad4TtDQPmrQWeVFH8ySr
o3kTdDSPcjqaV3kdzbMGHc27go7mYZPOIi/raZ5m9TRvg57mcU5P8zqvp3ne
oKd5X9DT84BJb3FeyKHnCTbnv/PG33DljCE=
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  ImageMargins->0.,
  ImageSize->Small,
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{184, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->213374272]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->28854],

Cell[TextData[{
 "Spheroidal angular harmonics are eigenfunctions of ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " transform on the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], ")"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->663722045],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Sinc", "[", 
      RowBox[{"c", " ", 
       RowBox[{"(", 
        RowBox[{"t", "-", "x"}], ")"}]}], "]"}], 
     RowBox[{"SpheroidalPS", "[", 
      RowBox[{"2", ",", "0", ",", "c", ",", "x"}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->183918391],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1VsFOwzAMdZu0Y4VNuzAhDaQJ+BKQxomPqCak7TCBxv4/JGlDrRc3bYWY
1DR24ufn2F7zXl8OH6f6ctzX27dz/XU47r+3u8+zVamMiE72Wd+RnRs788/J
GPtuxp0bldP6Hz177Qa0j3bQViTtl6lZye3ryStue9cLQFIgz0CumOxg/H7d
rQ8jVpIHWg/iIg76STF10zxC1AOIiFB6s6LlogKC7ljHmLFX0LR5yFk+8hBt
yXdBzgpYa3AjrQL8kjPH2Cm8Onvcx/GiOGVOqejt5Cp9YtP9jWCBmRVrK/Ip
VnZcUWNPgSHNpehQLjkCw5lyCgzzetQZoBbroWTylAwgm0q0w57mHaIjzMWo
iKSK1jGeEfpxFvejYzkXuZfgZ5lAGFePEvPhPC/xDNuIXN8pYf9QxnFdqtX8
P2JK5noh2hWRndjTkd1qFPeenCf95X3cWQ9h5Tib8OVuVFLeUL4xrE+irxi7
CbQeQw94b+rXW5JtwAOGAUDOB3CUv1k6REgvvXs4Er8R0UTbe5CzCbYPdnDX
OGfzN7tXY+AugRndsB2mqy46GPYf1d4bKfsBWypeNw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{109, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->33344786]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24316],

Cell[TextData[{
 "Spheroidal functions do not evaluate for half-integer values of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " and generic values of ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->348355882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", "1", ",", "1", ",", "0.5"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->175415402],

Cell[BoxData[
 RowBox[{"SpheroidalPS", "[", 
  RowBox[{
   FractionBox["3", "2"], ",", "1", ",", "1", ",", "0.5`"}], "]"}]], "Output",\

 ImageSize->{182, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->257237684]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->19414],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["LegendreP",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LegendreP"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalQS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalQS"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->11006]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->30668],

Cell[TextData[ButtonBox["Spheroidal Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpheroidalFunctions"]], "Tutorials",
 CellID->12303]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->105420433],

Cell[TextData[ButtonBox["Functions for Separable Coordinate Systems",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/FunctionsForSeparableCoordinateSystems"]], "MoreAbout",
 CellID->341487676],

Cell[TextData[ButtonBox["Functions Used in Optics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInOptics"]], "MoreAbout",
 CellID->389277252],

Cell[TextData[ButtonBox["Functions Used in Quantum Mechanics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInQuantumMechanics"]], "MoreAbout",
 CellID->322934864],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->128215086],

Cell[TextData[ButtonBox["Spheroidal and Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpheroidalAndRelatedFunctions"]], "MoreAbout",
 CellID->522106707],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->130313028]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SpheroidalPS - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 23, 32.1713302}", "context" -> "System`", 
    "keywords" -> {
     "band\\[Hyphen]limited functions", "spheroidal functions", 
      "spheroidal harmonics"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SpheroidalPS[n, m, \\[Gamma], z] gives the angular spheroidal function \
PS n, m (\\[Gamma], z) of the first kind.", "synonyms" -> {}, "title" -> 
    "SpheroidalPS", "type" -> "Symbol", "uri" -> "ref/SpheroidalPS"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9793, 310, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->234550160]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 41765, 1352}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2837, 66, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3460, 93, 1577, 37, 70, "ObjectNameGrid"],
Cell[5040, 132, 906, 29, 70, "Usage",
 CellID->27879]
}, Open  ]],
Cell[CellGroupData[{
Cell[5983, 166, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6461, 183, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6584, 189, 1223, 40, 70, "Notes",
 CellID->14006],
Cell[7810, 231, 585, 22, 70, "Notes",
 CellID->1220],
Cell[8398, 255, 658, 22, 70, "Notes",
 CellID->15719],
Cell[9059, 279, 256, 8, 70, "Notes",
 CellID->334798374],
Cell[9318, 289, 226, 7, 70, "Notes",
 CellID->31226],
Cell[9547, 298, 209, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[9793, 310, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->234550160],
Cell[CellGroupData[{
Cell[10178, 324, 147, 5, 70, "ExampleSection",
 CellID->92188727],
Cell[10328, 331, 64, 1, 70, "ExampleText",
 CellID->102308445],
Cell[CellGroupData[{
Cell[10417, 336, 152, 4, 28, "Input",
 CellID->43173829],
Cell[10572, 342, 204, 5, 36, "Output",
 CellID->66556847]
}, Open  ]],
Cell[10791, 350, 125, 3, 70, "ExampleDelimiter",
 CellID->835947008],
Cell[10919, 355, 79, 1, 70, "ExampleText",
 CellID->802528156],
Cell[CellGroupData[{
Cell[11023, 360, 283, 8, 70, "Input",
 CellID->148566906],
Cell[11309, 370, 2457, 69, 89, "Output",
 CellID->188485107]
}, Open  ]],
Cell[13781, 442, 125, 3, 70, "ExampleDelimiter",
 CellID->434247032],
Cell[13909, 447, 726, 32, 70, "ExampleText",
 CellID->6381167],
Cell[CellGroupData[{
Cell[14660, 483, 288, 9, 28, "Input",
 CellID->213350775],
Cell[14951, 494, 2883, 51, 131, "Output",
 Evaluatable->False,
 CellID->59628066]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[17883, 551, 216, 7, 70, "ExampleSection",
 CellID->30326],
Cell[18102, 560, 88, 1, 70, "ExampleText",
 CellID->94393633],
Cell[CellGroupData[{
Cell[18215, 565, 182, 5, 70, "Input",
 CellID->314925903],
Cell[18400, 572, 269, 8, 36, "Output",
 CellID->550536579]
}, Open  ]],
Cell[18684, 583, 125, 3, 70, "ExampleDelimiter",
 CellID->217504807],
Cell[18812, 588, 70, 1, 70, "ExampleText",
 CellID->283569038],
Cell[CellGroupData[{
Cell[18907, 593, 258, 8, 70, "Input",
 CellID->852018633],
Cell[19168, 603, 252, 8, 36, "Output",
 CellID->25082042]
}, Open  ]],
Cell[19435, 614, 123, 3, 70, "ExampleDelimiter",
 CellID->3535261],
Cell[19561, 619, 113, 3, 70, "ExampleText",
 CellID->173158339],
Cell[CellGroupData[{
Cell[19699, 626, 202, 6, 70, "Input",
 CellID->429954945],
Cell[19904, 634, 237, 7, 36, "Output",
 CellID->652582499]
}, Open  ]],
Cell[20156, 644, 125, 3, 70, "ExampleDelimiter",
 CellID->165479942],
Cell[20284, 649, 232, 8, 70, "ExampleText",
 CellID->160938561],
Cell[CellGroupData[{
Cell[20541, 661, 315, 10, 70, "Input",
 CellID->10306910],
Cell[20859, 673, 652, 22, 87, "Output",
 CellID->329034363]
}, Open  ]],
Cell[CellGroupData[{
Cell[21548, 700, 199, 6, 70, "Input",
 CellID->283753173],
Cell[21750, 708, 632, 22, 70, "Output",
 CellID->14559301]
}, Open  ]],
Cell[22397, 733, 125, 3, 70, "ExampleDelimiter",
 CellID->129130650],
Cell[22525, 738, 202, 7, 70, "ExampleText",
 CellID->54774772],
Cell[CellGroupData[{
Cell[22752, 749, 198, 6, 70, "Input",
 CellID->187276340],
Cell[22953, 757, 594, 14, 70, "Output",
 Evaluatable->False,
 CellID->407258497]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[23596, 777, 239, 7, 70, "ExampleSection",
 CellID->21058],
Cell[23838, 786, 242, 7, 70, "ExampleText",
 CellID->277064462],
Cell[CellGroupData[{
Cell[24105, 797, 161, 4, 70, "Input",
 CellID->135549909],
Cell[24269, 803, 237, 7, 36, "Output",
 CellID->132610523]
}, Open  ]],
Cell[CellGroupData[{
Cell[24543, 815, 161, 4, 70, "Input",
 CellID->591864808],
Cell[24707, 821, 237, 7, 36, "Output",
 CellID->234870836]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[24993, 834, 222, 7, 70, "ExampleSection",
 CellID->3759],
Cell[25218, 843, 246, 8, 70, "ExampleText",
 CellID->640948785],
Cell[CellGroupData[{
Cell[25489, 855, 938, 31, 70, "Input",
 CellID->297005226],
Cell[26430, 888, 669, 21, 36, "Output",
 CellID->187021462]
}, Open  ]],
Cell[27114, 912, 125, 3, 70, "ExampleDelimiter",
 CellID->377793566],
Cell[27242, 917, 113, 3, 70, "ExampleText",
 CellID->470010545],
Cell[CellGroupData[{
Cell[27380, 924, 554, 16, 70, "Input",
 CellID->663030826],
Cell[27937, 942, 3319, 76, 127, "Output",
 CellID->166648803]
}, Open  ]],
Cell[31271, 1021, 124, 3, 70, "ExampleDelimiter",
 CellID->33867524],
Cell[31398, 1026, 338, 11, 70, "ExampleText",
 CellID->459211663],
Cell[CellGroupData[{
Cell[31761, 1041, 532, 15, 70, "Input",
 CellID->197802376],
Cell[32296, 1058, 2504, 49, 139, "Output",
 CellID->213374272]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[34849, 1113, 233, 7, 70, "ExampleSection",
 CellID->28854],
Cell[35085, 1122, 392, 14, 70, "ExampleText",
 CellID->663722045],
Cell[CellGroupData[{
Cell[35502, 1140, 480, 16, 70, "Input",
 CellID->183918391],
Cell[35985, 1158, 821, 17, 70, "Output",
 Evaluatable->False,
 CellID->33344786]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36855, 1181, 226, 7, 70, "ExampleSection",
 CellID->24316],
Cell[37084, 1190, 284, 9, 70, "ExampleText",
 CellID->348355882],
Cell[CellGroupData[{
Cell[37393, 1203, 179, 6, 70, "Input",
 CellID->175415402],
Cell[37575, 1211, 267, 9, 51, "Output",
 CellID->257237684]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[37903, 1227, 311, 9, 70, "SeeAlsoSection",
 CellID->19414],
Cell[38217, 1238, 461, 17, 70, "SeeAlso",
 CellID->11006]
}, Open  ]],
Cell[CellGroupData[{
Cell[38715, 1260, 314, 9, 70, "TutorialsSection",
 CellID->30668],
Cell[39032, 1271, 150, 3, 70, "Tutorials",
 CellID->12303]
}, Open  ]],
Cell[CellGroupData[{
Cell[39219, 1279, 319, 9, 70, "MoreAboutSection",
 CellID->105420433],
Cell[39541, 1290, 195, 4, 70, "MoreAbout",
 CellID->341487676],
Cell[39739, 1296, 157, 3, 70, "MoreAbout",
 CellID->389277252],
Cell[39899, 1301, 178, 3, 70, "MoreAbout",
 CellID->322934864],
Cell[40080, 1306, 145, 3, 70, "MoreAbout",
 CellID->128215086],
Cell[40228, 1311, 173, 3, 70, "MoreAbout",
 CellID->522106707],
Cell[40404, 1316, 174, 3, 70, "MoreAbout",
 CellID->130313028]
}, Open  ]],
Cell[40593, 1322, 27, 0, 70, "History"],
Cell[40623, 1324, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

