(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     36177,       1161]
NotebookOptionsPosition[     30204,        955]
NotebookOutlinePosition[     31552,        992]
CellTagsIndexPosition[     31467,        987]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpheroidalFunctions"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"Spheroidal Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SpheroidalQS" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalQS"], 
          "SpheroidalPSPrime" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalPSPrime"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"SpheroidalQS\"\>", 
       2->"\<\"SpheroidalPSPrime\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal and Related Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/SpheroidalAndRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Spheroidal and Related Functions\"\>", 
       2->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SpheroidalQSPrime", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SpheroidalQSPrime",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SpheroidalQSPrime"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["\[Gamma]", "TR"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the derivative with respect to ",
     Cell[BoxData[
      FormBox["z", TraditionalForm]], "InlineMath"],
     " of the angular spheroidal function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox[
         StyleBox["QS",
          FontSlant->"Italic"], 
         RowBox[{"n", ",", "m"}]], "(", 
        RowBox[{"\[Gamma]", ",", "z"}], ")"}], TraditionalForm]], 
      "InlineMath"],
     " of the second kind."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->26853]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalQSPrime",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalQSPrime"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["a", "TI"], ",", 
    StyleBox["\[Gamma]", "TR"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula", "TemplateInclusion"],
 " uses spheroidal functions of type ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ". The types are specified as for ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->15500],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["SpheroidalQSPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQSPrime"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->183280062],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalQSPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQSPrime"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalQSPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalQSPrime"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->455763567],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->869300684],

Cell["Evaluate numerically:", "ExampleText",
 CellID->48231665],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQSPrime", "[", 
  RowBox[{"3", ",", "2", ",", "1", ",", "0.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18458256],

Cell[BoxData["28.58842457475968222184263248363`15.954589770191"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{123, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63010717]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->290329948],

Cell["Expansion about the spherical case:", "ExampleText",
 CellID->543845061],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"SpheroidalQSPrime", "[", 
    RowBox[{"2", ",", "0", ",", "c", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"c", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->656162328],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{534, 86},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->558913205]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->132275320],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   TagBox[
    RowBox[{
     SuperscriptBox[
      SubscriptBox[
       StyleBox["QS", "IT"], 
       RowBox[{
        TagBox["2",
         SpheroidalQSPrime,
         Editable->True,
         Selectable->True], ",", 
        TagBox["0",
         SpheroidalQSPrime,
         Editable->True,
         Selectable->True]}]], "\[Prime]"], "(", 
     RowBox[{
      TagBox["10",
       SpheroidalQSPrime,
       Editable->True,
       Selectable->True], ",", 
      TagBox["x",
       SpheroidalQSPrime,
       Editable->True,
       Selectable->True]}], ")"}],
    InterpretTemplate[SpheroidalQSPrime[
      SlotSequence[1]]& ],
    Editable->False,
    Selectable->False], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->193592132],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalQSPrime", "[", 
    RowBox[{"2", ",", "0", ",", "10", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->457603330],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJw1lPk7FAgch+V4LClHT1n3OpKE3CKzX+YU4z7L1egSyrE5MpJsi3Qg2tVq
Ry0qZwqJsl+5kzbHuK9xzyiq7TANatsf9ofP8/4D7/tRDwp3OyIsJCTk9m3/
0dkkz9Ff/DVW6Dsq9bvHAW+r4qaChFf4OdWzz6UoEmpCW8SkxxdxOTVU0eK3
4yC5Joj/sHsRv08oXZLvDIKDl4zeDifzUMkxw22+zA/KMrye0Ke4eMuDmmWw
yRv4V5lpCFxkXvflrWi7w9/dej2lmQsYl6LZs4vkAgWbJxRyp+bxsL2/R3A5
HW6WibfVmM2jz8CofraFPdywN4rquzCHDOF52+pyO8jw3RedyJtFHfd6iXds
GiSHMeJ27pvFbRWdgmQGDb7UivEd82awZLGRJ2VIA6ZISWzU0jSW2v71zEGP
BstPrZ20HKbRddiAWmlJA0bSS62Bu1O4bsCSiyXTYJU1ElAnMYVNn0p/ukOg
wbWGudwbIRxMZHkvnlGgQZS+gPUPcxINV1XvZbVSoX5SR6pMYwI1uv/gB1tS
YTnWdEcSZwwf5lLtnM9QQFPGhujBGkW70lvhxb+QIUTlsJeS7QgKKOvWYSYk
eKCbFjI9M4T1zlfQKNcW7Pg3KFKXB9G9057blggw2Xr/B3PzAXwd3u91gL4X
gk4XFMd/YmNN9JzaON8c5vSuGePDPox/e9gwm24MsonF5gYberCjxVibKKsC
euZv9Ired2EKP4soiVL4M3mpuy6uHQ9d35+i7GKA9K4XlRIdLai8ILOTlGCK
Wz0qMvfLN2Efs2n2WKQlnnTOym3Yidi8YLRCKCLgg0+t0h1a9egZwj90vcIG
iVF+oayqagx9J1AYDiJi0ejcZd75CmQ/dzVQe0xCXWFBVahvIeagwYmNBWT0
d1WuHj2Ug/fGxXq2GFIwia3DVmomoFbDsE9mIAXrJJn9TdHZsFRbd9HJlILP
qgLJ9y8UguFF6npVPhnNpjeonvO5B++1GHWMQhLmRwS0EGqrobH/hDubQUTN
t48/WzXVgcfTQFV6tw0eyKu1D7NEWOXuPtabS8B9BcMvnHyaIF/yamTMlz1I
40dz1cVbIcGqUndA2BQpjnIiH2va4WTpkJ+alz5mBA4mGdZ2gbhL5p6YYBls
d1LZ5CLXA139Ljx9E1VYUt+a2Pt7HxAlk0j+HCMwcrj5pnKIDXGNRhPvLplD
9KldjIxtA5AeIe4rPWMFt3VYw/Hhg3BFMMOR3AEwOCbjdrRjCGyk9wu3nbIF
5dLbyhaSI5A6tDSeu0KEg6etF8Tpo+BPZy6nW5Pham/do7m7Y/CoPXJkwYsC
zbss0pvFJsCnxEswoUuF7/IHy1McJiGfIUJPqqGCtsQTbqEdB74OMgWrUjT4
c3qLvCeXA8OacwFje2ig8iSUIpY2BZ26l9YWvvm/2ddiZa/ZNLzsJcuX7KWB
kEC4OGp6Go7eSd24/K2fte0cEeXzM9DYkq6YpU+DFdeGgNbtszDkIWtm7EaD
46TNhI/sWVBXa++gPaXBqGmg0vbzczDimZY8VGAH2m0Wv2oazgMnU/TDxR32
EOEtI6sxNg8ZTV0kTg4dWqIFptkJC/AhxuTI2a/OoJAz6yOqwoUs1vOsGaYb
zMYShdY6uTCvePaH+jAvqPC9eed9HA9eyv5Ynkz2g+Quab6/0iIsma735ukE
gTfhrF1HwyLYM6+xPdSDQdQpxmyD/yvwkxsTHTwXAfcDTmhYrb2CyrsbuyTi
Y6HPklJeLf4a/v/DfwEyNYrT
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  Method->{},
  PlotRange->{{-1, 1}, {-0.000652261239975891, 0.00593932385238409}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{184, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->280245940]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->20278],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->94393633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQSPrime", "[", 
  RowBox[{"3", ",", "2", ",", 
   RowBox[{"1", "-", "I"}], ",", " ", "3.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->314925903],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1302.3886967481694`"}], "-", 
  RowBox[{"419.26782868276103`", " ", "\[ImaginaryI]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{134, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->174534776]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217504807],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->283569038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"SpheroidalQSPrime", "[", 
    RowBox[{"2", ",", "0", ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "2"}]}], "]"}], ",", "30"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->852018633],

Cell[BoxData[
 RowBox[{"-", 
  "0.833256550936083945349467100383634963210530474741`30."}]], "Output",
 ImageSize->{236, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->753513418]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3535261],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->173158339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQSPrime", "[", 
  RowBox[{"2", ",", "0", ",", 
   RowBox[{"1", "/", "3"}], ",", "0.50000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->429954945],

Cell[BoxData[
 RowBox[{"-", 
  "0.83325655093608394534946710038363526579`22.698970004336022"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{180, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->54664126]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->165479942],

Cell[TextData[{
 "For certain parameters ",
 Cell[BoxData[
  ButtonBox["SpheroidalPS",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalPS"]], "InlineFormula"],
 " evaluates exactly:"
}], "ExampleText",
 CellID->160938561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", "=", "3"}], "}"}], ",", 
   RowBox[{"SpheroidalQSPrime", "[", 
    RowBox[{"n", ",", "1", ",", 
     RowBox[{"n", " ", 
      RowBox[{"Pi", "/", "2"}]}], ",", " ", "x"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10306910],

Cell[BoxData[
 RowBox[{
  FractionBox[
   RowBox[{"3", " ", "\[Pi]", " ", 
    RowBox[{"Cos", "[", 
     FractionBox[
      RowBox[{"3", " ", "\[Pi]", " ", "x"}], "2"], "]"}], " ", 
    SqrtBox[
     FractionBox["3", 
      RowBox[{"7", " ", 
       RowBox[{"(", 
        RowBox[{"EulerGamma", "-", 
         RowBox[{"CosIntegral", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}], "+", 
         RowBox[{"Log", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}]}], ")"}]}]]]}], 
   SqrtBox[
    RowBox[{"1", "-", 
     SuperscriptBox["x", "2"]}]]], "+", 
  FractionBox[
   RowBox[{"2", " ", "x", " ", 
    SqrtBox[
     FractionBox["3", 
      RowBox[{"7", " ", 
       RowBox[{"(", 
        RowBox[{"EulerGamma", "-", 
         RowBox[{"CosIntegral", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}], "+", 
         RowBox[{"Log", "[", 
          RowBox[{"6", " ", "\[Pi]"}], "]"}]}], ")"}]}]]], " ", 
    RowBox[{"Sin", "[", 
     FractionBox[
      RowBox[{"3", " ", "\[Pi]", " ", "x"}], "2"], "]"}]}], 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", 
      SuperscriptBox["x", "2"]}], ")"}], 
    RowBox[{"3", "/", "2"}]]]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{319, 131},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->800412820]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQSPrime", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", 
   RowBox[{"1", "/", "2"}], ",", "c", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->283753173],

Cell[BoxData["0"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->3172091]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->129130650],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->54774772],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SpheroidalQSPrime", "[", 
   RowBox[{"n", ",", "m", ",", "c", ",", "\[Eta]"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->187276340],

Cell[GraphicsData["CompressedBitmap", "\<\
eJyVVMEKwjAMzdZuU1QQUREPKvgZ3vXkRwwR9CCK+v+zadrapnVqYWnzlr6k
ed329fN0vNTP86Fe7e717XQ+PFbb611BIgOAjXpGM1DrRq30s2kaNZOdoBWI
6gFrjVYK6OCqoDeghw0FQRDaXM1Lcj1UMtbC85FAhtwG7bJdgvmVpn77NmXO
4no/ZataasRl6e1yCSmXttLml36PEBEmJ24uw5rVyA0LzkV0ItyHsmVBT13T
vfosD5RcI4rN3dEtWkbZeOfSZxbJmpzOrYyjKKKfVLGFv+VGfGBPqpbOUCTj
9VdRMbR531LX+yA3sfMOiChuypU0jB2rGlM9+4ObR/QTuXx/HPBHdyzJjzUO
flDG8hI3ainNlfRPifYb25D5xLhg6Fyj/t8Nshe51HOE\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{59, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->373465301]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8726],

Cell["\<\
Plot prolate and oblate versions of the same angular function:\
\>", "ExampleText",
 CellID->470010545],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalQSPrime", "[", 
    RowBox[{"2", ",", "1", ",", "5", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "25"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->663030826],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->280756361]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalQSPrime", "[", 
    RowBox[{"2", ",", "1", ",", 
     RowBox[{"5", "I"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "25"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->201523365],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{-0.937817560104829, 189.564278559443}, {-0.919849633427084, 
     102.324927545826}, {-0.876403587664064, 
     16.3302815036201}, {-0.832957541901044, -25.6230983179819}, \
{-0.792390591994795, -45.7682015754807}, {-0.751823642088545, \
-55.6851049062158}, {-0.712052454705734, -59.459512610409}, \
{-0.672281267322923, -59.628019184848}, {-0.629139175796882, \
-57.3500836033582}, {-0.585997084270842, -53.599137082142}, \
{-0.545734088601572, -49.3979100530423}, {-0.505471092932303, \
-44.8980389464178}, {-0.418203293307307, -35.1155014461328}, \
{-0.332527018729186, -26.2661084217207}, {-0.252608935864607, \
-18.9337649975097}, {-0.16594904471357, -11.8903652677483}, \
{-0.0850473452760752, -5.93230054582563}, {-0.00573717088545587, \
-0.396284054787221}, {0.0803148117916213, 5.59627103205959}, {
     0.160608602755156, 11.4817390178418}, {0.247644202005149, 
     18.5066851510699}, {0.333088276208267, 26.3207763121087}, {
     0.412774158697842, 34.5266048871414}, {0.499201849473876, 
     44.1870325274859}, {0.539536599005121, 48.7175832168425}, {
     0.579871348536367, 52.9918730275799}, {0.623577002210842, 
     56.9309276060561}, {0.667282655885316, 59.4709229978427}, {
     0.710192547036353, 59.5343314773076}, {0.75310243818739, 
     55.4817140722074}, {0.793133233481656, 45.5077700225961}, {
     0.833164028775922, 25.4835104236487}, {
     0.876565728213417, -16.5468772440101}, {
     0.919967427650911, -102.664361003096}, {
     0.937343462600955, -187.219951380341}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  Method->{},
  PlotRange->{{-1, 1}, {-187.219951380341, 189.564278559443}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->476063727]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->4377],

Cell[TextData[{
 "Spheroidal functions do not generically evaluate for half-integer values of \
",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->348355882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalQSPrime", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", "1", ",", "1", ",", "0.5"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->175415402],

Cell[BoxData[
 RowBox[{"SpheroidalQSPrime", "[", 
  RowBox[{
   FractionBox["3", "2"], ",", "1", ",", "1", ",", "0.5`"}], "]"}]], "Output",\

 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{217, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->196429238]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->3526],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalQS",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalQS"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalPSPrime",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalPSPrime"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->28461]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->30642],

Cell[TextData[ButtonBox["Spheroidal Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpheroidalFunctions"]], "Tutorials",
 CellID->12303]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->260335956],

Cell[TextData[ButtonBox["Spheroidal and Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpheroidalAndRelatedFunctions"]], "MoreAbout",
 CellID->141333024],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->630853071]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SpheroidalQSPrime - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 23, 33.8588734}", "context" -> "System`", 
    "keywords" -> {"spheroidal functions"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SpheroidalQSPrime[n, m, \\[Gamma], z] gives the derivative with respect \
to z of the angular spheroidal function QS n, m (\\[Gamma], z) of the second \
kind.", "synonyms" -> {"spheroidal prime"}, "title" -> "SpheroidalQSPrime", 
    "type" -> "Symbol", "uri" -> "ref/SpheroidalQSPrime"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7449, 234, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->455763567]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 31324, 980}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2130, 53, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2753, 80, 1582, 37, 70, "ObjectNameGrid"],
Cell[4338, 119, 1033, 32, 70, "Usage",
 CellID->26853]
}, Open  ]],
Cell[CellGroupData[{
Cell[5408, 156, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5886, 173, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6009, 179, 673, 22, 70, "Notes",
 CellID->15500],
Cell[6685, 203, 266, 8, 70, "Notes",
 CellID->183280062],
Cell[6954, 213, 236, 7, 70, "Notes",
 CellID->31226],
Cell[7193, 222, 219, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[7449, 234, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->455763567],
Cell[CellGroupData[{
Cell[7834, 248, 148, 5, 70, "ExampleSection",
 CellID->869300684],
Cell[7985, 255, 63, 1, 70, "ExampleText",
 CellID->48231665],
Cell[CellGroupData[{
Cell[8073, 260, 157, 4, 28, "Input",
 CellID->18458256],
Cell[8233, 266, 253, 7, 36, "Output",
 CellID->63010717]
}, Open  ]],
Cell[8501, 276, 125, 3, 70, "ExampleDelimiter",
 CellID->290329948],
Cell[8629, 281, 78, 1, 70, "ExampleText",
 CellID->543845061],
Cell[CellGroupData[{
Cell[8732, 286, 274, 8, 28, "Input",
 CellID->656162328],
Cell[9009, 296, 2717, 49, 107, "Output",
 Evaluatable->False,
 CellID->558913205]
}, Open  ]],
Cell[11741, 348, 125, 3, 70, "ExampleDelimiter",
 CellID->132275320],
Cell[11869, 353, 799, 33, 70, "ExampleText",
 CellID->193592132],
Cell[CellGroupData[{
Cell[12693, 390, 345, 10, 70, "Input",
 CellID->457603330],
Cell[13041, 402, 2331, 48, 134, "Output",
 CellID->280245940]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[15421, 456, 216, 7, 70, "ExampleSection",
 CellID->20278],
Cell[15640, 465, 88, 1, 70, "ExampleText",
 CellID->94393633],
Cell[CellGroupData[{
Cell[15753, 470, 187, 5, 70, "Input",
 CellID->314925903],
Cell[15943, 477, 318, 10, 36, "Output",
 CellID->174534776]
}, Open  ]],
Cell[16276, 490, 125, 3, 70, "ExampleDelimiter",
 CellID->217504807],
Cell[16404, 495, 70, 1, 70, "ExampleText",
 CellID->283569038],
Cell[CellGroupData[{
Cell[16499, 500, 263, 8, 70, "Input",
 CellID->852018633],
Cell[16765, 510, 232, 7, 36, "Output",
 CellID->753513418]
}, Open  ]],
Cell[17012, 520, 123, 3, 70, "ExampleDelimiter",
 CellID->3535261],
Cell[17138, 525, 113, 3, 70, "ExampleText",
 CellID->173158339],
Cell[CellGroupData[{
Cell[17276, 532, 207, 6, 70, "Input",
 CellID->429954945],
Cell[17486, 540, 284, 9, 36, "Output",
 CellID->54664126]
}, Open  ]],
Cell[17785, 552, 125, 3, 70, "ExampleDelimiter",
 CellID->165479942],
Cell[17913, 557, 232, 8, 70, "ExampleText",
 CellID->160938561],
Cell[CellGroupData[{
Cell[18170, 569, 320, 10, 70, "Input",
 CellID->10306910],
Cell[18493, 581, 1338, 44, 152, "Output",
 CellID->800412820]
}, Open  ]],
Cell[CellGroupData[{
Cell[19868, 630, 204, 6, 70, "Input",
 CellID->283753173],
Cell[20075, 638, 204, 7, 36, "Output",
 CellID->3172091]
}, Open  ]],
Cell[20294, 648, 125, 3, 70, "ExampleDelimiter",
 CellID->129130650],
Cell[20422, 653, 202, 7, 70, "ExampleText",
 CellID->54774772],
Cell[CellGroupData[{
Cell[20649, 664, 203, 6, 70, "Input",
 CellID->187276340],
Cell[20855, 672, 686, 16, 70, "Output",
 Evaluatable->False,
 CellID->373465301]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21590, 694, 222, 7, 70, "ExampleSection",
 CellID->8726],
Cell[21815, 703, 113, 3, 70, "ExampleText",
 CellID->470010545],
Cell[CellGroupData[{
Cell[21953, 710, 394, 11, 70, "Input",
 CellID->663030826],
Cell[22350, 723, 2282, 43, 70, "Output",
 Evaluatable->False,
 CellID->280756361]
}, Open  ]],
Cell[CellGroupData[{
Cell[24669, 771, 415, 12, 70, "Input",
 CellID->201523365],
Cell[25087, 785, 1966, 40, 127, "Output",
 CellID->476063727]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[27102, 831, 225, 7, 70, "ExampleSection",
 CellID->4377],
Cell[27330, 840, 206, 7, 70, "ExampleText",
 CellID->348355882],
Cell[CellGroupData[{
Cell[27561, 851, 184, 6, 70, "Input",
 CellID->175415402],
Cell[27748, 859, 320, 11, 51, "Output",
 CellID->196429238]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28129, 877, 310, 9, 70, "SeeAlsoSection",
 CellID->3526],
Cell[28442, 888, 477, 17, 70, "SeeAlso",
 CellID->28461]
}, Open  ]],
Cell[CellGroupData[{
Cell[28956, 910, 314, 9, 70, "TutorialsSection",
 CellID->30642],
Cell[29273, 921, 150, 3, 70, "Tutorials",
 CellID->12303]
}, Open  ]],
Cell[CellGroupData[{
Cell[29460, 929, 319, 9, 70, "MoreAboutSection",
 CellID->260335956],
Cell[29782, 940, 173, 3, 70, "MoreAbout",
 CellID->141333024],
Cell[29958, 945, 174, 3, 70, "MoreAbout",
 CellID->630853071]
}, Open  ]],
Cell[30147, 951, 27, 0, 70, "History"],
Cell[30177, 953, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

