(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     42984,       1349]
NotebookOptionsPosition[     36262,       1117]
NotebookOutlinePosition[     37599,       1154]
CellTagsIndexPosition[     37513,       1149]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Spheroidal Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpheroidalFunctions"]}, #,
          2]& , Evaluator -> Automatic], {
       1->"\<\"Spheroidal Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SpheroidalS2" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalS2"], 
          "SphericalBesselJ" :> 
          Documentation`HelpLookup["paclet:ref/SphericalBesselJ"], 
          "SpheroidalRadialFactor" :> 
          Documentation`HelpLookup["paclet:ref/SpheroidalRadialFactor"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"SpheroidalS2\"\>", 
       2->"\<\"SphericalBesselJ\"\>", 
       3->"\<\"SpheroidalRadialFactor\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Functions for Separable Coordinate Systems" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsForSeparableCoordinateSystems"], 
          "Functions Used in Quantum Mechanics" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsUsedInQuantumMechanics"], 
          "Mathematical Functions" :> 
          Documentation`HelpLookup["paclet:guide/MathematicalFunctions"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "Spheroidal and Related Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/SpheroidalAndRelatedFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Functions for Separable Coordinate Systems\"\>", 
       2->"\<\"Functions Used in Quantum Mechanics\"\>", 
       3->"\<\"Mathematical Functions\"\>", 4->"\<\"Special Functions\"\>", 
       5->"\<\"Spheroidal and Related Functions\"\>", 
       6->"\<\"New in 6.0: Mathematical Functions\"\>", 
       7->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SpheroidalS1", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SpheroidalS1",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SpheroidalS1"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["\[Gamma]", "TR"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the radial spheroidal function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubsuperscriptBox["S", 
         RowBox[{"n", ",", "m"}], 
         RowBox[{"(", "1", ")"}]], "(", 
        RowBox[{"\[Gamma]", ",", "z"}], ")"}], TraditionalForm]], 
      "InlineMath"],
     " of the first kind."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->24244]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 "The radial spheroidal functions satisfy the differential equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        SuperscriptBox["z", "2"]}], ")"}], 
      SuperscriptBox["y", "\[DoublePrime]"]}], "-", 
     RowBox[{"2", "z", " ", 
      SuperscriptBox["y", "\[Prime]"]}], " ", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"\[Lambda]", "+", 
        RowBox[{
         SuperscriptBox["\[Gamma]", "2"], "(", 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}], ")"}], "-", 
        FractionBox[
         SuperscriptBox["m", "2"], 
         RowBox[{"1", "-", 
          SuperscriptBox["z", "2"]}]]}], ")"}], "y"}]}], "\[Equal]", "0"}], 
   TraditionalForm]], "InlineMath"],
 " with the spheroidal eigenvalue ",
 Cell[BoxData[
  FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
 " given by ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SpheroidalEigenvalue",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalEigenvalue"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["\[Gamma]", "TR"]}], "]"}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->24975],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["S", 
     RowBox[{"n", ",", "m"}], 
     RowBox[{"(", "1", ")"}]], "(", 
    RowBox[{"\[Gamma]", ",", "z"}], ")"}], TraditionalForm]], "InlineMath"],
 " are normalized according to the Meixner\[Dash]Sch\[ADoubleDot]fke scheme."
}], "Notes",
 CellID->24009],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalS1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalS1"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SpheroidalS1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalS1"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->94747042],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->568157090],

Cell["Evaluate numerically:", "ExampleText",
 CellID->45503221],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalS1", "[", 
  RowBox[{"1", ",", "0", ",", "2.5", ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->416581269],

Cell[BoxData[
 RowBox[{"0.58450727300361118486532766188`15.722329082685128", "+", 
  RowBox[{"0``16.338314848580424", " ", "\[ImaginaryI]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->437014000]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31487],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->94393633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalS1", "[", 
  RowBox[{"3", ",", "2", ",", 
   RowBox[{"1", "-", "I"}], ",", " ", "3.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->314925903],

Cell[BoxData[
 RowBox[{"0.54758549189102379006754972579`15.757901181178655", "-", 
  RowBox[{
  "1.0879937352192758925895077482`16.056075642719538", " ", 
   "\[ImaginaryI]"}]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{280, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->334224769]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->217504807],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->283569038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"2", ",", "1", ",", 
     RowBox[{"1", "/", "3"}], ",", "5"}], "]"}], ",", "50"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->852018633],

Cell[BoxData["0.\
1488125919166766155193877932289028597714554042705407590800852668479`50."], \
"Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{368, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->160403821]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3535261],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->173158339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalS1", "[", 
  RowBox[{"2", ",", "1", ",", 
   RowBox[{"1", "/", "3"}], ",", "5.000000000000000000000000000"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->429954945],

Cell[BoxData["0.14881259191667661551938779322890285977`27.69897000433602"], \
"Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{214, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->185714807]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->165479942],

Cell[TextData[{
 "For certain parameters ",
 Cell[BoxData[
  ButtonBox["SpheroidalS1",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SpheroidalS1"]], "InlineFormula"],
 " evaluates exactly:"
}], "ExampleText",
 CellID->160938561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", "=", "3"}], "}"}], ",", 
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"n", ",", "1", ",", 
     RowBox[{"n", " ", 
      RowBox[{"Pi", "/", "2"}]}], ",", " ", "x"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10306910],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   RowBox[{"Sin", "[", 
    RowBox[{
     FractionBox["3", "2"], " ", "\[Pi]", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "x"}], ")"}]}], "]"}]}], 
  RowBox[{"3", " ", "\[Pi]", " ", 
   SqrtBox[
    RowBox[{"1", "-", 
     FractionBox["1", 
      SuperscriptBox["x", "2"]]}]], " ", "x"}]]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{126, 64},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->444820917]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->728648831],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " typesetting:"
}], "ExampleText",
 CellID->558624730],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SpheroidalS1", "[", 
   RowBox[{"n", ",", "m", ",", "c", ",", "\[Eta]"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->261657998],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1VM0OgjAMLiuiJMYQw8GDJrwBb6EnH4IYEzkYjfr+uO4HRjsDF5fQrR/d
17Uf49x8btd782kvTXV6Nc9be3lXx8dLQ5gAQK2fcgd63emVeequ07O1a7JI
qBlQGTQju7AomOHDAC1EVul5P8TjEJ+O9zt0wXIh85dk0sF3PBafZOdsPFtY
U5/Al5OGtRKGLg9tUONz6aEcqnxV4tTU+oRmA/l4yHhPbTbVl+VRJVjV7HrG
ue0BAyIoBPdaIJInZfk3cRZxIiWYuFKz/CXjdex5oEakX6JrU18Jj/cqr7xK
2MsX6dG0ahjBuF/GecMd/CZGO89r2zJ/UEN+fxhlzBhDEavE9OrHTQti93+N
zQMlrFq8hwfzxv4FIfkCl76Q4g==\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{47, 20},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->318546879]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17150],

Cell[TextData[{
 "Spheroidal angular harmonics are eigenfunctions of the ",
 Cell[BoxData[
  ButtonBox["Sinc",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sinc"]], "InlineFormula"],
 " transform on the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], ")"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->663722045],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sinc", "[", 
     RowBox[{"c", " ", 
      RowBox[{"(", 
       RowBox[{"t", "-", "x"}], ")"}]}], "]"}], 
    RowBox[{"SpheroidalPS", "[", 
     RowBox[{"2", ",", "0", ",", "c", ",", "x"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->183918391],

Cell[BoxData[
 RowBox[{"2", " ", 
  RowBox[{"SpheroidalPS", "[", 
   RowBox[{"2", ",", "0", ",", "c", ",", "t"}], "]"}], " ", 
  SuperscriptBox[
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"2", ",", "0", ",", "c", ",", "1"}], "]"}], "2"]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{342, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->373992725]
}, Open  ]],

Cell["Plot the eigenvalue:", "ExampleText",
 CellID->241018390],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"SpheroidalS1", "[", 
     RowBox[{"2", ",", "0", ",", "c", ",", "1"}], "]"}], "^", "2"}], ",", 
   RowBox[{"{", 
    RowBox[{"c", ",", "1", ",", "10"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->338650785],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVzH081AccwPHromXI4c6h83B3YWe15jmvmu835mkkL1QardSEF2M46YHk
4Tx2Iy+EuMhZlC7HvVaLepndecq5n1RbSxTCoZIlD9tl7Y/P6/3fh3k0LiCc
TCKRPD72v3++3WNDIs3DbKTRHPnJYeAVxt+9H/UWSClhW5PwONhuK/W+OLQA
hu22X37lFgvWnqmpgyHvwD9g7ZtXF7jADAsXa04swqVnb2KW/c5A7LLkhWfA
EsxI4Q0/KR3aCtX0sjqWIZmVWite4kGzH491wWoVkr8o8T3DLACh9gY7wU//
QIivBmc+vwjGvXxmjagq+Fr6mtwAJcDKKqorrfgAfjTXdaunyiFUYNqURyZh
tFLeFamsgp9Xudlet0m4ZQWfn52pAYPSGxKX1HU4lXeCs/K0DnjbJ8fsXcmY
/Px3KpN+FbTO55otSNdjpEP1L8dABMrw4mDxNnXcVcbfvDTdApT2I86jkg1o
dWwjZyrmFvALywx0WBsx/so7w5G5NmjJVw3p5GrghmEPQZtLBzDJHOc0LU1U
o/sHPmrvhMn3bZXzJ7QQ68Ma1dfLYDFCX0isaeO8/dqp6IPd0ESq4qen6yA7
xztHkNgLoZpdzpVcCoqeuFev2NyHhe121azHFGwWBJ+982k/5AbVkK476aJW
dq9JCdEP3GCr8dZiXUwr4vjeSpbD4VCRrP2VLvJHa1XN2gNwKM0yIMxfD1W6
K3F9VQMQckUwot6qh7aXZcETdAW46bDrRvT18VsffC9OVcDnKVcjbnH18Xrj
lNJmUAGRPe7XKQp9bLjY9LT6EwLqqROvozhUbBIN/LjfjIDFtMTh7wqoWDDo
wXKzIcB9ltwX+IqKnDx+TsVOAmYcmhx7vWiofCcJnnQhoOjcvjoXIQ337f01
/fwuAsxoml7TKhpGxVy77eRAgKixY+7CAQN04s3PhHAI8D8UsfOP2wZY/9sc
a4xOwAJlUz7DmI47xPK/WeSP/7qBA2NcOqqEWbovHyrghWOhRYOCjg5DNH73
ZQXMTmO1hq0hNka8H6s8ooDFygVqdIkhJpwcrD3BUMCz4lhu2T4jNLoUc9in
YADivE0fdGoYo8HE1ZdbV+Vg5dKcaSk3RkMDyWzsfjnkv02yK8jcjB0J+u3N
d/qhY0dQYLAjAxNFAdR8Zj8w7lG7Q5cZaJHiNyxOuQ9ssobz0Rsm+GjSKSH2
ZR/szRr9IIg3xU1xM8Kd7n1gKVN2Dn9mhuesPU2I1l4gKrSWMobNsEby5t8M
Ti/4P86F5r3meLzam55a3gOboD7j8jVzNM2ksyZpPTDS1RJnrc3Epid3xXx+
N5SX95cof2BiT/H2kw+1uqHBm+TM6WFiW1bFaceKLugzsshRs2ZhrVw9osC8
C1x7jwUd4bHQ0bfQx1wiA2Gn6OD8CAspHrHfC/fIwPDpzEYGsnG8ONuRNyaF
sQfo1FPJxtGpGkv7DCncU+boUFRszOu4SS41kUIn7aZa7gc2Xkw5fXIzQwqy
3Y9X1tbYWLX+ry1XjKXwHz2OBmo=
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  Method->{},
  PlotRange->{{1, 10}, {0., 0.253839098575396}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{184, 126},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->774116641]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->72029627],

Cell["\<\
Find resonant frequencies for the Dirichlet problem in the prolate spheroidal \
cavity:\
\>", "ExampleText",
 CellID->24395564],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"k", "/", 
      RowBox[{"(", 
       RowBox[{"2", " ", 
        SuperscriptBox["3", 
         RowBox[{"1", "/", "3"}]]}], ")"}]}], " ", ",", "3"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "20"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "1"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->146527133],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{2.14532604838907, 0.391583437048237}, {2.62243462743417, 
     0.18941641255863}, {3.1348592578289, 0.0028359889849412}, {
     3.63723215108546, -0.129737415439241}, {
     4.13960504434203, -0.202033016739114}, {
     4.6845577794078, -0.213040504653665}, {
     5.22951051447357, -0.167066920639627}, {
     5.7380957229275, -0.092506148031908}, {
     6.24668093138143, -0.00899870936886602}, {6.79784598164457, 
     0.0690234297826636}, {7.3490110319077, 0.116709032326798}, {
     7.89012434503268, 0.126392935971268}, {8.43123765815765, 
     0.101818906526428}, {8.93598344467078, 0.0572704611310032}, {
     9.44072923118391, 0.00436417035864975}, {
     9.98805485950625, -0.0476600236430146}, {
     10.5353804878286, -0.081412532801507}, {
     11.0463385895391, -0.0901215434553999}, {
     11.5572966912496, -0.0767230952577767}, {
     12.0582030558219, -0.0469972684206377}, {
     12.5591094203942, -0.00892570622659614}, {13.1025956267758, 
     0.0310822490284045}, {13.6460818331573, 0.0595646425772845}, {
     14.153200512927, 0.0698254173180446}, {14.6603191926967, 
     0.0629761621234417}, {15.2100177142756, 0.0395251917860161}, {
     15.7597162358545, 0.00686392117262573}, {
     16.2993630202953, -0.0249383420641475}, {
     16.839009804736, -0.0480693111737254}, {
     17.3422890625649, -0.0568808611314536}, {
     17.8455683203938, -0.05196745348159}, {
     18.3914274200319, -0.0334055474798714}, {
     18.93728651967, -0.00682301309250519}, {19.4467780926963, 
     0.0184713090925603}, {19.9562696657225, 0.0380599014907156}, {
     19.9781343065455, 0.0386978065555938}, {19.9999989473684, 
     0.039316506807541}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  Method->{},
  PlotRange->{{0, 20}, {-0.213040504653665, 0.391583437048237}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{184, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5058216]
}, Open  ]],

Cell["Determine the first few frequencies:", "ExampleText",
 CellID->585505651],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"k", "/", 
      RowBox[{"(", 
       RowBox[{"2", " ", 
        SuperscriptBox["3", 
         RowBox[{"1", "/", "3"}]]}], ")"}]}], " ", ",", "3"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "2", ",", "5"}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "50"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->79022032],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "k", "\[Rule]", 
   "3.1437802146054629629408013281435665842587906544536552160495429542127`50.\
"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{398, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->95248846]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"k", "/", 
      RowBox[{"(", 
       RowBox[{"2", " ", 
        SuperscriptBox["3", 
         RowBox[{"1", "/", "3"}]]}], ")"}]}], " ", ",", "3"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "5", ",", "8"}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "50"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->117813954],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "k", "\[Rule]", 
   "6.3035457796055201155748386658594768210285475812410642447617049000652`50.\
"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{398, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->28223054]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22083289],

Cell["Plot the prolate and oblate functions:", "ExampleText",
 CellID->38223527],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"2", ",", "0", ",", "10", ",", "r"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "1", ",", "5"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "2"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->548499913],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->466776655]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"SpheroidalS1", "[", 
    RowBox[{"2", ",", "0", ",", 
     RowBox[{
      RowBox[{"-", "10"}], " ", "I"}], ",", 
     RowBox[{"I", " ", "r"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "0", ",", "5"}], "}"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "2"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->246912353],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 98},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->276032687]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18109],

Cell[TextData[{
 "Spheroidal functions do not evaluate for half-integer values of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " and generic values of ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->348355882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SpheroidalS1", "[", 
  RowBox[{
   RowBox[{"3", "/", "2"}], ",", "1", ",", "1", ",", "0.5"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->175415402],

Cell[BoxData[
 RowBox[{"SpheroidalS1", "[", 
  RowBox[{
   FractionBox["3", "2"], ",", "1", ",", "1", ",", "0.5`"}], "]"}]], "Output",\

 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{182, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->237539177]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->7821],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalS2",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalS2"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalBesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalBesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SpheroidalRadialFactor",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SpheroidalRadialFactor"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->7904]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->29353],

Cell[TextData[ButtonBox["Spheroidal Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpheroidalFunctions"]], "Tutorials",
 CellID->12303]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->33213885],

Cell[TextData[ButtonBox["Functions for Separable Coordinate Systems",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/FunctionsForSeparableCoordinateSystems"]], "MoreAbout",
 CellID->1216108],

Cell[TextData[ButtonBox["Functions Used in Quantum Mechanics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInQuantumMechanics"]], "MoreAbout",
 CellID->566496829],

Cell[TextData[ButtonBox["Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MathematicalFunctions"]], "MoreAbout",
 CellID->420821180],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->870412709],

Cell[TextData[ButtonBox["Spheroidal and Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpheroidalAndRelatedFunctions"]], "MoreAbout",
 CellID->126427805],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->128757756],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->273605228]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SpheroidalS1 - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 23, 34.8588990}", "context" -> "System`", 
    "keywords" -> {"spheroidal functions", "spheroidal harmonics"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "SpheroidalS1[n, m, \\[Gamma], z] gives the radial spheroidal function \
SubsuperscriptBox[S, n, m, (1)](\\[Gamma], z) of the first kind.", 
    "synonyms" -> {"spheroidal s1"}, "title" -> "SpheroidalS1", "type" -> 
    "Symbol", "uri" -> "ref/SpheroidalS1"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9006, 272, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->94747042]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 37371, 1142}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3227, 74, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3850, 101, 1577, 37, 70, "ObjectNameGrid"],
Cell[5430, 140, 893, 28, 70, "Usage",
 CellID->24244]
}, Open  ]],
Cell[CellGroupData[{
Cell[6360, 173, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6838, 190, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6961, 196, 1222, 40, 70, "Notes",
 CellID->24975],
Cell[8186, 238, 342, 11, 70, "Notes",
 CellID->24009],
Cell[8531, 251, 226, 7, 70, "Notes",
 CellID->31226],
Cell[8760, 260, 209, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[9006, 272, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->94747042],
Cell[CellGroupData[{
Cell[9390, 286, 148, 5, 70, "ExampleSection",
 CellID->568157090],
Cell[9541, 293, 63, 1, 70, "ExampleText",
 CellID->45503221],
Cell[CellGroupData[{
Cell[9629, 298, 153, 4, 28, "Input",
 CellID->416581269],
Cell[9785, 304, 333, 9, 36, "Output",
 CellID->437014000]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10167, 319, 216, 7, 70, "ExampleSection",
 CellID->31487],
Cell[10386, 328, 88, 1, 70, "ExampleText",
 CellID->94393633],
Cell[CellGroupData[{
Cell[10499, 333, 182, 5, 70, "Input",
 CellID->314925903],
Cell[10684, 340, 368, 11, 36, "Output",
 CellID->334224769]
}, Open  ]],
Cell[11067, 354, 125, 3, 70, "ExampleDelimiter",
 CellID->217504807],
Cell[11195, 359, 70, 1, 70, "ExampleText",
 CellID->283569038],
Cell[CellGroupData[{
Cell[11290, 364, 234, 8, 70, "Input",
 CellID->852018633],
Cell[11527, 374, 283, 9, 36, "Output",
 CellID->160403821]
}, Open  ]],
Cell[11825, 386, 123, 3, 70, "ExampleDelimiter",
 CellID->3535261],
Cell[11951, 391, 113, 3, 70, "ExampleText",
 CellID->173158339],
Cell[CellGroupData[{
Cell[12089, 398, 206, 6, 70, "Input",
 CellID->429954945],
Cell[12298, 406, 266, 8, 36, "Output",
 CellID->185714807]
}, Open  ]],
Cell[12579, 417, 125, 3, 70, "ExampleDelimiter",
 CellID->165479942],
Cell[12707, 422, 232, 8, 70, "ExampleText",
 CellID->160938561],
Cell[CellGroupData[{
Cell[12964, 434, 315, 10, 70, "Input",
 CellID->10306910],
Cell[13282, 446, 557, 20, 85, "Output",
 CellID->444820917]
}, Open  ]],
Cell[13854, 469, 125, 3, 70, "ExampleDelimiter",
 CellID->728648831],
Cell[13982, 474, 204, 7, 70, "ExampleText",
 CellID->558624730],
Cell[CellGroupData[{
Cell[14211, 485, 198, 6, 70, "Input",
 CellID->261657998],
Cell[14412, 493, 670, 16, 70, "Output",
 Evaluatable->False,
 CellID->318546879]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15131, 515, 223, 7, 70, "ExampleSection",
 CellID->17150],
Cell[15357, 524, 396, 14, 70, "ExampleText",
 CellID->663722045],
Cell[CellGroupData[{
Cell[15778, 542, 428, 14, 70, "Input",
 CellID->183918391],
Cell[16209, 558, 431, 13, 36, "Output",
 CellID->373992725]
}, Open  ]],
Cell[16655, 574, 63, 1, 70, "ExampleText",
 CellID->241018390],
Cell[CellGroupData[{
Cell[16743, 579, 345, 10, 70, "Input",
 CellID->338650785],
Cell[17091, 591, 2070, 44, 147, "Output",
 CellID->774116641]
}, Open  ]],
Cell[19176, 638, 124, 3, 70, "ExampleDelimiter",
 CellID->72029627],
Cell[19303, 643, 137, 4, 70, "ExampleText",
 CellID->24395564],
Cell[CellGroupData[{
Cell[19465, 651, 515, 15, 70, "Input",
 CellID->146527133],
Cell[19983, 668, 2169, 48, 127, "Output",
 CellID->5058216]
}, Open  ]],
Cell[22167, 719, 79, 1, 70, "ExampleText",
 CellID->585505651],
Cell[CellGroupData[{
Cell[22271, 724, 472, 14, 70, "Input",
 CellID->79022032],
Cell[22746, 740, 338, 12, 36, "Output",
 CellID->95248846]
}, Open  ]],
Cell[CellGroupData[{
Cell[23121, 757, 473, 14, 70, "Input",
 CellID->117813954],
Cell[23597, 773, 338, 12, 36, "Output",
 CellID->28223054]
}, Open  ]],
Cell[23950, 788, 124, 3, 70, "ExampleDelimiter",
 CellID->22083289],
Cell[24077, 793, 80, 1, 70, "ExampleText",
 CellID->38223527],
Cell[CellGroupData[{
Cell[24182, 798, 319, 9, 70, "Input",
 CellID->548499913],
Cell[24504, 809, 3380, 59, 70, "Output",
 Evaluatable->False,
 CellID->466776655]
}, Open  ]],
Cell[CellGroupData[{
Cell[27921, 873, 393, 12, 70, "Input",
 CellID->246912353],
Cell[28317, 887, 3591, 63, 70, "Output",
 Evaluatable->False,
 CellID->276032687]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[31957, 956, 226, 7, 70, "ExampleSection",
 CellID->18109],
Cell[32186, 965, 284, 9, 70, "ExampleText",
 CellID->348355882],
Cell[CellGroupData[{
Cell[32495, 978, 179, 6, 70, "Input",
 CellID->175415402],
Cell[32677, 986, 315, 11, 51, "Output",
 CellID->237539177]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[33053, 1004, 310, 9, 70, "SeeAlsoSection",
 CellID->7821],
Cell[33366, 1015, 747, 26, 70, "SeeAlso",
 CellID->7904]
}, Open  ]],
Cell[CellGroupData[{
Cell[34150, 1046, 314, 9, 70, "TutorialsSection",
 CellID->29353],
Cell[34467, 1057, 150, 3, 70, "Tutorials",
 CellID->12303]
}, Open  ]],
Cell[CellGroupData[{
Cell[34654, 1065, 318, 9, 70, "MoreAboutSection",
 CellID->33213885],
Cell[34975, 1076, 193, 4, 70, "MoreAbout",
 CellID->1216108],
Cell[35171, 1082, 178, 3, 70, "MoreAbout",
 CellID->566496829],
Cell[35352, 1087, 155, 3, 70, "MoreAbout",
 CellID->420821180],
Cell[35510, 1092, 145, 3, 70, "MoreAbout",
 CellID->870412709],
Cell[35658, 1097, 173, 3, 70, "MoreAbout",
 CellID->126427805],
Cell[35834, 1102, 174, 3, 70, "MoreAbout",
 CellID->128757756],
Cell[36011, 1107, 179, 3, 70, "MoreAbout",
 CellID->273605228]
}, Open  ]],
Cell[36205, 1113, 27, 0, 70, "History"],
Cell[36235, 1115, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

