(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     31607,       1189]
NotebookOptionsPosition[     25094,        961]
NotebookOutlinePosition[     26446,        998]
CellTagsIndexPosition[     26361,        993]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"EuclideanDistance" :> 
          Documentation`HelpLookup["paclet:ref/EuclideanDistance"], 
          "ManhattanDistance" :> 
          Documentation`HelpLookup["paclet:ref/ManhattanDistance"], 
          "ChebyshevDistance" :> 
          Documentation`HelpLookup["paclet:ref/ChebyshevDistance"], 
          "BrayCurtisDistance" :> 
          Documentation`HelpLookup["paclet:ref/BrayCurtisDistance"], 
          "CanberraDistance" :> 
          Documentation`HelpLookup["paclet:ref/CanberraDistance"], "Norm" :> 
          Documentation`HelpLookup["paclet:ref/Norm"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"EuclideanDistance\"\>", 
       2->"\<\"ManhattanDistance\"\>", 3->"\<\"ChebyshevDistance\"\>", 
       4->"\<\"BrayCurtisDistance\"\>", 5->"\<\"CanberraDistance\"\>", 
       6->"\<\"Norm\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Distance and Dissimilarity Measures" :> 
          Documentation`HelpLookup[
           "paclet:guide/DistanceAndDissimilarityMeasures"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Distance and Dissimilarity Measures\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SquaredEuclideanDistance", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SquaredEuclideanDistance",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SquaredEuclideanDistance"], "[", 
       RowBox[{
        StyleBox["u", "TI"], ",", 
        StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator] gives the squared Euclidean distance between vectors ",
     
     Cell[BoxData[
      StyleBox["u", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["v", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->19629]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["SquaredEuclideanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SquaredEuclideanDistance"], "[", 
   RowBox[{
    StyleBox["u", "TI"], ",", 
    StyleBox["v", "TI"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  SuperscriptBox[
   RowBox[{
    ButtonBox["Norm",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Norm"], "[", 
    RowBox[{
     StyleBox["u", "TI"], "-", 
     StyleBox["v", "TI"]}], "]"}], "2"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->214729278,
  ButtonNote->"214729278"]
}], "Notes",
 CellID->24221]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->300742276],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->399032052],

Cell["Squared Euclidean distance between two vectors:", "ExampleText",
 CellID->235641551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaredEuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->196650888],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"c", "-", "z"}], "]"}], "2"]}]], "Output",
 ImageSize->{231, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->35363037]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->125461358],

Cell["Squared Euclidean distance between numeric vectors:", "ExampleText",
 CellID->319587507],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaredEuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "4", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190454384],

Cell[BoxData["14"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->564145223]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Compute distance between any vectors of equal length:", "ExampleText",
 CellID->86273121],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaredEuclideanDistance", "[", 
  RowBox[{
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", "100"}], "]"}], ",", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"5", ",", "100"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->488208944],

Cell[BoxData["443.22714611414955`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->148103270]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->204991544],

Cell["Compute distance between vectors of any precision:", "ExampleText",
 CellID->14044010],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaredEuclideanDistance", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "5", ",", "2", ",", "3", ",", "10"}], "}"}], ",", 
     "50"}], "]"}], ",", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"4", ",", "15", ",", "20", ",", "5", ",", "5"}], "}"}], ",", 
     "50"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->323525768],

Cell[BoxData["461.9999999999999999999999999999999999999999999999999999999973`\
49.096910013008056"], "Output",
 ImageSize->{354, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->30297098]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Cluster data using squared Euclidean distance:", "ExampleText",
 CellID->245478961],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindClusters", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"DistanceFunction", "\[Rule]", "SquaredEuclideanDistance"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->108189764],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"5", ",", "10"}], "}"}], "}"}]}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->77925108]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->124651963],

Cell["Demonstrate the triangle inequality:", "ExampleText",
 CellID->162085005],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d1", "=", 
  RowBox[{"SquaredEuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"a", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->395493032],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"Abs", "[", 
   RowBox[{"b", "-", "c"}], "]"}], "2"]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->238371785]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d2", "=", 
  RowBox[{"SquaredEuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "c"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"d", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->13424126],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"Abs", "[", 
   RowBox[{"a", "-", "d"}], "]"}], "2"]], "Output",
 ImageSize->{72, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->6375793]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d3", "=", 
  RowBox[{"SquaredEuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"d", ",", "c"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->51893016],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "c"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "d"}], "]"}], "2"]}]], "Output",
 ImageSize->{151, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->576183787]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"d3", "<=", 
   RowBox[{"d1", "+", "d2"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->175054461],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->412928133]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SquaredEuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SquaredEuclideanDistance"]], "InlineFormula"],
 " is equivalent to the squared ",
 Cell[BoxData[
  ButtonBox["Norm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Norm"]], "InlineFormula"],
 " of a difference:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->214729278],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaredEuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->218507111],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"c", "-", "z"}], "]"}], "2"]}]], "Output",
 ImageSize->{231, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->290158681]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Norm", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], "-", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}], "^", "2"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21346996],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{"Abs", "[", 
    RowBox[{"c", "-", "z"}], "]"}], "2"]}]], "Output",
 ImageSize->{231, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->381506332]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->286900690],

Cell[TextData[{
 "The square root of ",
 Cell[BoxData[
  ButtonBox["SquaredEuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SquaredEuclideanDistance"]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  ButtonBox["EuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/EuclideanDistance"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->555367459],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SquaredEuclideanDistance", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}], "//", 
  "Sqrt"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->283254267],

Cell[BoxData[
 SqrtBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"c", "-", "z"}], "]"}], "2"]}]]], "Output",
 ImageSize->{250, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->76933099]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->325280469],

Cell[BoxData[
 SqrtBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"a", "-", "x"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"b", "-", "y"}], "]"}], "2"], "+", 
   SuperscriptBox[
    RowBox[{"Abs", "[", 
     RowBox[{"c", "-", "z"}], "]"}], "2"]}]]], "Output",
 ImageSize->{250, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->8400892]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->70099947],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Variance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Variance"]], "InlineFormula"],
 " as a ",
 Cell[BoxData[
  ButtonBox["SquaredEuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SquaredEuclideanDistance"]], "InlineFormula"],
 " from the ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->394865487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"data", "=", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b", ",", "c"}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->463545505],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"a", ",", "b", ",", "c"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->16095882]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mean", "=", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"Mean", "[", "data", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"Length", "[", "data", "]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->132468290],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "3"], " ", 
    RowBox[{"(", 
     RowBox[{"a", "+", "b", "+", "c"}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{264, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->459189310]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->917053855],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "6"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"2", " ", "a"}], "-", "b", "-", "c"}], ")"}], " ", 
     RowBox[{"Conjugate", "[", "a", "]"}]}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "a"}], "+", 
       RowBox[{"2", " ", "b"}], "-", "c"}], ")"}], " ", 
     RowBox[{"Conjugate", "[", "b", "]"}]}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "a"}], "-", "b", "+", 
       RowBox[{"2", " ", "c"}]}], ")"}], " ", 
     RowBox[{"Conjugate", "[", "c", "]"}]}]}], ")"}]}]], "Output",
 ImageSize->{545, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->245794952]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"%", "==", 
   RowBox[{
    RowBox[{"SquaredEuclideanDistance", "[", 
     RowBox[{"data", ",", "mean"}], "]"}], "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Length", "[", "data", "]"}], "-", "1"}], ")"}]}]}], "//", 
  "FullSimplify"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->42727618],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->224037976]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->59709880],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["SquaredEuclideanDistance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SquaredEuclideanDistance"]], "InlineFormula"],
 " computed from ",
 Cell[BoxData[
  ButtonBox["RootMeanSquare",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootMeanSquare"]], "InlineFormula"],
 " of a difference:"
}], "ExampleText",
 CellID->149042657],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SquaredEuclideanDistance", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "4", ",", "6"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->290459063],

Cell[BoxData["14"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->273170206]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Length", "[", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], "]"}], 
  RowBox[{
   RowBox[{"RootMeanSquare", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], "-", 
     RowBox[{"{", 
      RowBox[{"2", ",", "4", ",", "6"}], "}"}]}], "]"}], "^", "2", 
   " "}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->159972835],

Cell[BoxData["14"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->296945475]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->10116],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["EuclideanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/EuclideanDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ManhattanDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ManhattanDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChebyshevDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChebyshevDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BrayCurtisDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BrayCurtisDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["CanberraDistance",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CanberraDistance"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Norm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Norm"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->32525]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Distance and Dissimilarity Measures",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DistanceAndDissimilarityMeasures"]], "MoreAbout",
 CellID->456310730]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SquaredEuclideanDistance - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 23, 42.7809768}", "context" -> "System`", 
    "keywords" -> {"squared Euclidean distance", "squared Euclidean norm"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "SquaredEuclideanDistance[u, v] gives the squared Euclidean distance \
between vectors u and v.", "synonyms" -> {"squared euclidean distance"}, 
    "title" -> "SquaredEuclideanDistance", "type" -> "Symbol", "uri" -> 
    "ref/SquaredEuclideanDistance"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6066, 186, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->300742276]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 26218, 986}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1884, 44, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2507, 71, 1589, 37, 70, "ObjectNameGrid"],
Cell[4099, 110, 742, 24, 70, "Usage",
 CellID->19629]
}, Open  ]],
Cell[CellGroupData[{
Cell[4878, 139, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5356, 156, 673, 25, 70, "Notes",
 CellID->24221]
}, Closed]],
Cell[CellGroupData[{
Cell[6066, 186, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->300742276],
Cell[CellGroupData[{
Cell[6451, 200, 148, 5, 70, "ExampleSection",
 CellID->399032052],
Cell[6602, 207, 90, 1, 70, "ExampleText",
 CellID->235641551],
Cell[CellGroupData[{
Cell[6717, 212, 261, 8, 28, "Input",
 CellID->196650888],
Cell[6981, 222, 430, 15, 36, "Output",
 CellID->35363037]
}, Open  ]],
Cell[7426, 240, 125, 3, 70, "ExampleDelimiter",
 CellID->125461358],
Cell[7554, 245, 94, 1, 70, "ExampleText",
 CellID->319587507],
Cell[CellGroupData[{
Cell[7673, 250, 261, 8, 70, "Input",
 CellID->190454384],
Cell[7937, 260, 159, 5, 36, "Output",
 CellID->564145223]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8145, 271, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[8369, 280, 95, 1, 70, "ExampleText",
 CellID->86273121],
Cell[CellGroupData[{
Cell[8489, 285, 273, 8, 70, "Input",
 CellID->488208944],
Cell[8765, 295, 176, 5, 36, "Output",
 CellID->148103270]
}, Open  ]],
Cell[8956, 303, 125, 3, 70, "ExampleDelimiter",
 CellID->204991544],
Cell[9084, 308, 92, 1, 70, "ExampleText",
 CellID->14044010],
Cell[CellGroupData[{
Cell[9201, 313, 434, 14, 70, "Input",
 CellID->323525768],
Cell[9638, 329, 240, 6, 36, "Output",
 CellID->30297098]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[9927, 341, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[10157, 350, 89, 1, 70, "ExampleText",
 CellID->245478961],
Cell[CellGroupData[{
Cell[10271, 355, 484, 16, 70, "Input",
 CellID->108189764],
Cell[10758, 373, 497, 18, 36, "Output",
 CellID->77925108]
}, Open  ]],
Cell[11270, 394, 125, 3, 70, "ExampleDelimiter",
 CellID->124651963],
Cell[11398, 399, 79, 1, 70, "ExampleText",
 CellID->162085005],
Cell[CellGroupData[{
Cell[11502, 404, 270, 9, 70, "Input",
 CellID->395493032],
Cell[11775, 415, 235, 8, 36, "Output",
 CellID->238371785]
}, Open  ]],
Cell[CellGroupData[{
Cell[12047, 428, 269, 9, 70, "Input",
 CellID->13424126],
Cell[12319, 439, 233, 8, 36, "Output",
 CellID->6375793]
}, Open  ]],
Cell[CellGroupData[{
Cell[12589, 452, 269, 9, 70, "Input",
 CellID->51893016],
Cell[12861, 463, 341, 12, 36, "Output",
 CellID->576183787]
}, Open  ]],
Cell[CellGroupData[{
Cell[13239, 480, 155, 5, 70, "Input",
 CellID->175054461],
Cell[13397, 487, 161, 5, 36, "Output",
 CellID->412928133]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13607, 498, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[13848, 507, 420, 13, 70, "ExampleText",
 CellID->214729278],
Cell[CellGroupData[{
Cell[14293, 524, 261, 8, 70, "Input",
 CellID->218507111],
Cell[14557, 534, 431, 15, 36, "Output",
 CellID->290158681]
}, Open  ]],
Cell[CellGroupData[{
Cell[15025, 554, 268, 9, 70, "Input",
 CellID->21346996],
Cell[15296, 565, 431, 15, 36, "Output",
 CellID->381506332]
}, Open  ]],
Cell[15742, 583, 125, 3, 70, "ExampleDelimiter",
 CellID->286900690],
Cell[15870, 588, 380, 13, 70, "ExampleText",
 CellID->555367459],
Cell[CellGroupData[{
Cell[16275, 605, 296, 10, 70, "Input",
 CellID->283254267],
Cell[16574, 617, 451, 16, 45, "Output",
 CellID->76933099]
}, Open  ]],
Cell[CellGroupData[{
Cell[17062, 638, 254, 8, 70, "Input",
 CellID->325280469],
Cell[17319, 648, 450, 16, 45, "Output",
 CellID->8400892]
}, Open  ]],
Cell[17784, 667, 124, 3, 70, "ExampleDelimiter",
 CellID->70099947],
Cell[17911, 672, 466, 17, 70, "ExampleText",
 CellID->394865487],
Cell[CellGroupData[{
Cell[18402, 693, 152, 5, 70, "Input",
 CellID->463545505],
Cell[18557, 700, 212, 7, 36, "Output",
 CellID->16095882]
}, Open  ]],
Cell[CellGroupData[{
Cell[18806, 712, 251, 8, 70, "Input",
 CellID->132468290],
Cell[19060, 722, 538, 19, 51, "Output",
 CellID->459189310]
}, Open  ]],
Cell[CellGroupData[{
Cell[19635, 746, 107, 3, 70, "Input",
 CellID->917053855],
Cell[19745, 751, 770, 26, 51, "Output",
 CellID->245794952]
}, Open  ]],
Cell[CellGroupData[{
Cell[20552, 782, 330, 11, 70, "Input",
 CellID->42727618],
Cell[20885, 795, 161, 5, 36, "Output",
 CellID->224037976]
}, Open  ]],
Cell[21061, 803, 124, 3, 70, "ExampleDelimiter",
 CellID->59709880],
Cell[21188, 808, 377, 12, 70, "ExampleText",
 CellID->149042657],
Cell[CellGroupData[{
Cell[21590, 824, 261, 8, 70, "Input",
 CellID->290459063],
Cell[21854, 834, 159, 5, 36, "Output",
 CellID->273170206]
}, Open  ]],
Cell[CellGroupData[{
Cell[22050, 844, 404, 14, 70, "Input",
 CellID->159972835],
Cell[22457, 860, 159, 5, 36, "Output",
 CellID->296945475]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[22677, 872, 311, 9, 70, "SeeAlsoSection",
 CellID->10116],
Cell[22991, 883, 1513, 53, 70, "SeeAlso",
 CellID->32525]
}, Open  ]],
Cell[CellGroupData[{
Cell[24541, 941, 299, 8, 70, "MoreAboutSection"],
Cell[24843, 951, 179, 3, 70, "MoreAbout",
 CellID->456310730]
}, Open  ]],
Cell[25037, 957, 27, 0, 70, "History"],
Cell[25067, 959, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

