(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     42506,       1484]
NotebookOptionsPosition[     35983,       1250]
NotebookOutlinePosition[     37295,       1287]
CellTagsIndexPosition[     37209,       1282]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Symmetric Polynomials" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/SymmetricPolynomials"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Symmetric Polynomials\"\>"}, "\<\"tutorials\"\>",
        
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"SymmetricReduction" :> 
          Documentation`HelpLookup["paclet:ref/SymmetricReduction"], 
          "PolynomialReduce" :> 
          Documentation`HelpLookup["paclet:ref/PolynomialReduce"], "Subsets" :> 
          Documentation`HelpLookup["paclet:ref/Subsets"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"SymmetricReduction\"\>", 
       2->"\<\"PolynomialReduce\"\>", 
       3->"\<\"Subsets\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Polynomial Algebra" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialAlgebra"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Polynomial Algebra\"\>", 
       2->"\<\"New in 6.0: Symbolic Computation\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["SymmetricPolynomial", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["SymmetricPolynomial",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/SymmetricPolynomial"], "[", 
       RowBox[{
        StyleBox["k", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"], ",", 
          SubscriptBox[
           StyleBox["x", "TI"], 
           StyleBox["n", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the",
     " ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " ",
     "elementary symmetric polynomial in the variables ",
     Cell[BoxData[
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["1", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"], ",", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["n", "TI"]]}]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->25402]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "A symmetric polynomial of ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " variables ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    SubscriptBox[
     StyleBox["x", "TI"], 
     StyleBox["1", "TR"]], ",", 
    StyleBox["\[Ellipsis]", "TR"], ",", 
    SubscriptBox[
     StyleBox["x", "TI"], 
     StyleBox["n", "TI"]]}], "}"}]], "InlineFormula"],
 " is invariant under any permutation of its variables. The ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " elementary symmetric polynomial is the sum of all square free monomials of \
degree ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->32676],

Cell[TextData[{
 "The degree ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 " must satisfy ",
 Cell[BoxData[
  RowBox[{"0", "\[LessEqual]", 
   StyleBox["k", "TI"], "\[LessEqual]", 
   StyleBox["n", "TI"]}]], "InlineFormula"],
 "."
}], "Notes",
 CellID->13543],

Cell["\<\
The elementary symmetric polynomials form a basis for the symmetric \
polynomials. \
\>", "Notes",
 CellID->7373],

Cell[TextData[{
 "Expressing a general symmetric polynomial in terms of elementary symmetric \
polynomials is accomplished by using ",
 Cell[BoxData[
  ButtonBox["SymmetricReduction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SymmetricReduction"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->6266]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->106932869],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->114168625],

Cell[TextData[{
 "The elementary symmetric polynomial of degree 3 in variables ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["1", "TR"]], ",", 
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["2", "TR"]], ",", 
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["3", "TR"]], ",", 
   SubscriptBox[
    StyleBox["x", "TI"], 
    StyleBox["4", "TR"]]}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->27892],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricPolynomial", "[", 
  RowBox[{"3", ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     SubscriptBox["x", "2"], ",", 
     SubscriptBox["x", "3"], ",", 
     SubscriptBox["x", "4"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2451],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdVckOgjAQLUvBheXgEo9+jZ78CEJM4GA06qfyP3VaSlKmrYhiXEjoYx6v
s7R02GXXYn/IrmWerbfn7FSU+WW9OZ6B8hxCSAV3uiLwzOBJ3BVjgCGMFMwp
oAcYA7qAS8QnEucKz3Wp5BcSI8Q7IsqYj9xk4hKzVTtF9uJFfdxhz5GdGPSO
zL6ugNfkyto8pPYkb3vvd7wPeur7xuPoA/5bnHpnCK1BiuHyDfOJr0qwgmoK
alLcnTFEVLVyXkk4QFScua74jqiuwGaPTevtoshNriae66n8Fp/xRxV8hx+s
t/HYT3uVmpMRWrKzZY1PaNDTH55v0w3tz7b69/y2+7jeLcS/pfVlRhqD/x/6
ifg1LyMBalcds3aXnSB7huxP60WXItUDI3FuNLFLrQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->33743922]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->22570],

Cell["\<\
The zeroth elementary symmetric polynomial is defined to be 1:\
\>", "ExampleText",
 CellID->13519],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricPolynomial", "[", 
  RowBox[{"0", ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     SubscriptBox["x", "2"], ",", 
     SubscriptBox["x", "3"], ",", 
     SubscriptBox["x", "4"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19472],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->700973310]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14262],

Cell["\<\
The 2 \[Times] 3 matrices with entries 0 or 1:\
\>", "ExampleText",
 CellID->21409],

Cell[BoxData[
 RowBox[{
  RowBox[{"tabs", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Partition", "[", 
      RowBox[{
       RowBox[{"IntegerDigits", "[", 
        RowBox[{"n", ",", "2", ",", "6"}], "]"}], ",", "3"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "64"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28607],

Cell["\<\
Select matrices whose column sums are 1, 1, 1 and whose row sums are 2, 1:\
\>", "ExampleText",
 CellID->18038],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"Cases", "[", 
   RowBox[{"tabs", ",", 
    RowBox[{"a_", "/;", 
     RowBox[{
      RowBox[{
       RowBox[{"Total", "[", 
        RowBox[{"a", ",", 
         RowBox[{"{", "1", "}"}]}], "]"}], "==", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "\[And]", 
      RowBox[{
       RowBox[{"Total", "[", 
        RowBox[{"a", ",", 
         RowBox[{"{", "2", "}"}]}], "]"}], "==", 
       RowBox[{"{", 
        RowBox[{"2", ",", "1"}], "}"}]}]}]}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4360],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmN1OgzAUxwsFp4AxxhsvvNi1JvoWeuVDkMVkuzCabe+5R8K2MFP+nNn2
gItxXbKWnvZ88KPQ9rzW2+Xbe71dLer5y7r+XK4Wm/nzx1qJZCKE2Kn/3a1Q
1426Mv9d06i6LStdSi01P3EP7RtoPwT2l8ZLCVKpily0ci2uunoG466hnXfj
StD31atA/wzGFSbaTJWpEuaElcHo/LttdMUBKXrCCDNSC5/OlLZpW+gRbUub
jWn9RCgSGzJISXKZw0oIOXwbrqiImOR4tv3IoW1kEslFcv+fHPq6+EWSPF/H
JKsj0OvNZdNfwc+JO0n3scGT2evZdjj6th0kxbu7R0c/Rvk08fg2ypRg7oqS
YkB58xmH/hITVQvPCAdzCSPCLwJ6cPXrp6w3y9pzwmIyNgKaQej8GRvFNLP4
lKPAE0rR1T5rmH2iKQP1CtDHLwHvbvz22K4vP8aM/bz1xGWVtyJGTpFT5PRX
OB3KxEgWH9ep328XP9YqL0PRz7tQ+bTT4lGQHPR8kaK/29c1roZUfhBPGaF6
tj75ph0p19rmdUXyBbhkB6w=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{207, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->199970885]
}, Open  ]],

Cell[TextData[{
 "We can also count how many such matrices there are by using ",
 Cell[BoxData[
  ButtonBox["SymmetricPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SymmetricPolynomial"]], "InlineFormula"],
 ". The generating function of 2 \[Times] 3 matrices whose row sums are 2, 1 \
is given by:"
}], "ExampleText",
 CellID->11171],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"genfun", "=", 
  RowBox[{
   RowBox[{"SymmetricPolynomial", "[", 
    RowBox[{"2", ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["x", "1"], ",", 
       SubscriptBox["x", "2"], ",", 
       SubscriptBox["x", "3"]}], "}"}]}], "]"}], 
   RowBox[{"SymmetricPolynomial", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["x", "1"], ",", 
       SubscriptBox["x", "2"], ",", 
       SubscriptBox["x", "3"]}], "}"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->20654],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    SubscriptBox["x", "1"], "+", 
    SubscriptBox["x", "2"], "+", 
    SubscriptBox["x", "3"]}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     SubscriptBox["x", "1"], " ", 
     SubscriptBox["x", "2"]}], "+", 
    RowBox[{
     SubscriptBox["x", "1"], " ", 
     SubscriptBox["x", "3"]}], "+", 
    RowBox[{
     SubscriptBox["x", "2"], " ", 
     SubscriptBox["x", "3"]}]}], ")"}]}]], "Output",
 ImageSize->{207, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->328448441]
}, Open  ]],

Cell[TextData[{
 "The coefficient of ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox[
    SubscriptBox["x", "1"], "1"], 
   SuperscriptBox[
    SubscriptBox["x", "2"], "1"], 
   SuperscriptBox[
    SubscriptBox["x", "3"], "1"]}]], "InlineFormula"],
 " counts how many of these matrices have column sums 1, 1, 1:"
}], "ExampleText",
 CellID->11514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Coefficient", "[", 
  RowBox[{"genfun", ",", 
   RowBox[{"{", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["x", "1"], "1"], 
     SuperscriptBox[
      SubscriptBox["x", "2"], "1"], 
     SuperscriptBox[
      SubscriptBox["x", "3"], "1"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->14523],

Cell[BoxData[
 RowBox[{"{", "3", "}"}]], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->339851050]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->489],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plus", "@@", 
  RowBox[{"Subsets", "[", 
   RowBox[{
    RowBox[{"Times", "[", 
     RowBox[{
      SubscriptBox["x", "1"], ",", 
      SubscriptBox["x", "2"], ",", 
      SubscriptBox["x", "3"], ",", 
      SubscriptBox["x", "4"]}], "]"}], ",", 
    RowBox[{"{", "2", "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6766],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["x", "1"], " ", 
   SubscriptBox["x", "2"]}], "+", 
  RowBox[{
   SubscriptBox["x", "1"], " ", 
   SubscriptBox["x", "3"]}], "+", 
  RowBox[{
   SubscriptBox["x", "2"], " ", 
   SubscriptBox["x", "3"]}], "+", 
  RowBox[{
   SubscriptBox["x", "1"], " ", 
   SubscriptBox["x", "4"]}], "+", 
  RowBox[{
   SubscriptBox["x", "2"], " ", 
   SubscriptBox["x", "4"]}], "+", 
  RowBox[{
   SubscriptBox["x", "3"], " ", 
   SubscriptBox["x", "4"]}]}]], "Output",
 ImageSize->{236, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->85729613]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SymmetricPolynomial", "[", 
  RowBox[{"2", ",", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     SubscriptBox["x", "2"], ",", 
     SubscriptBox["x", "3"], ",", 
     SubscriptBox["x", "4"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->29756],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["x", "1"], " ", 
   SubscriptBox["x", "2"]}], "+", 
  RowBox[{
   SubscriptBox["x", "1"], " ", 
   SubscriptBox["x", "3"]}], "+", 
  RowBox[{
   SubscriptBox["x", "2"], " ", 
   SubscriptBox["x", "3"]}], "+", 
  RowBox[{
   SubscriptBox["x", "1"], " ", 
   SubscriptBox["x", "4"]}], "+", 
  RowBox[{
   SubscriptBox["x", "2"], " ", 
   SubscriptBox["x", "4"]}], "+", 
  RowBox[{
   SubscriptBox["x", "3"], " ", 
   SubscriptBox["x", "4"]}]}]], "Output",
 ImageSize->{236, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->171605805]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7470],

Cell[TextData[{
 "The generating function for the symmetric polynomials in ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " variables is given by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    UnderoverscriptBox["\[Product]", 
     RowBox[{"i", "=", "1"}], "n"], 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["x", "i"], "+", "1"}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->25739],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CoefficientList", "[", 
   RowBox[{
    RowBox[{
     UnderoverscriptBox["\[Product]", 
      RowBox[{"i", "=", "1"}], "4"], 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{
        SubscriptBox["x", "i"], "t"}]}], ")"}]}], ",", "t"}], "]"}], "//", 
  "TableForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20666],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdV0tOwzAQTdMkLfTDAvqRYMFpQCw4RIWQYIFAhYMAG7gEsIELIRa9hXEc
G5xnuxOn7gci1dOZPM/Yjj3zfDq5uzi/mtxdnk0Oj6eTm4vLs9vDo+spNzUb
URTN+O9kHPH/jP8TvxljXBZti7cpN3a4bHLZ4zLmsi/1V4HayttcZeIRaF3f
Ab0L+ovwkvuNpf8mIJrS7nqfEO+xfy6Tn/FHaSEkmD+JxX+U6BBEpAYitYyh
jNBHmUd9K9ZBjhH7p9KeQK+Ww55pfmLDu1qxliNaRowis35DcyU7xqy7hsXE
fBg7Ygu+4Dbo+P6d/e7iKq1rpw/Brnb+nmbPcTvSPpCya7H7n5TBgvgeoe+B
3q+CX+pJxX3le7KpeOrk/7c4xf5aTSaz6nN91omK2bEVICqO3ERsRtShkKEq
QY5Po4jM6S5/qSaX4cdVuSg/5VWqW9HwhGae/rC/Cxfan2v1q/otcnm9io11
wTwVf83LgRA+jGMX9HXj98UM6jOegdQVcxmxMpNRTKjHykynr+n63SAWUSjG
MwIdGQjFUJDxUP2RMQ1Bx7tJw9gXvhXblwGsmiFhhVsV01kWY1J+iv0XghFZ
M/IGcCiqkq2HdYXw6X93Fro3Y6IYjYuJ1GUwmexH1eqqTKkus1L48qpVYSjz
mE3bs19VBmPbL7a4dZlTKOaF/YvaUYVh0RwmDO/B6mhmiDpxwoxNMIW5DAwZ
kS9j8u3v678tZuDPwBSTGko/Y6k/CBQyKGQ8Y9CR0dwb67ooowiNVxW7mC9d
sf3rMd3Dvz5jBXpkei613fnnZWYqc9sqVWxE1TPiInffqji03zu+IJ0PQiCe
Ge50PLGU/sT8TrDeymIqnnyYn0LFs7AOq5rdlz7axjeNNZkQ\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{232, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TableForm=",
 CellID->255801993]
}, Open  ]],

Cell["Check:", "ExampleText",
 CellID->7283],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SymmetricPolynomial", "[", 
     RowBox[{"i", ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["x", "1"], ",", 
        SubscriptBox["x", "2"], ",", 
        SubscriptBox["x", "3"], ",", 
        SubscriptBox["x", "4"]}], "}"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "0", ",", "4"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->8160],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdV0tOwzAQTdMkLfTDAvqRYMFpQCw4RIWQYIFAhYMAG7gEsIELIRa9hXEc
G5xnuxOn7gci1dOZPM/Yjj3zfDq5uzi/mtxdnk0Oj6eTm4vLs9vDo+spNzUb
URTN+O9kHPH/jP8TvxljXBZti7cpN3a4bHLZ4zLmsi/1V4HayttcZeIRaF3f
Ab0L+ovwkvuNpf8mIJrS7nqfEO+xfy6Tn/FHaSEkmD+JxX+U6BBEpAYitYyh
jNBHmUd9K9ZBjhH7p9KeQK+Ww55pfmLDu1qxliNaRowis35DcyU7xqy7hsXE
fBg7Ygu+4Dbo+P6d/e7iKq1rpw/Brnb+nmbPcTvSPpCya7H7n5TBgvgeoe+B
3q+CX+pJxX3le7KpeOrk/7c4xf5aTSaz6nN91omK2bEVICqO3ERsRtShkKEq
QY5Po4jM6S5/qSaX4cdVuSg/5VWqW9HwhGae/rC/Cxfan2v1q/otcnm9io11
wTwVf83LgRA+jGMX9HXj98UM6jOegdQVcxmxMpNRTKjHykynr+n63SAWUSjG
MwIdGQjFUJDxUP2RMQ1Bx7tJw9gXvhXblwGsmiFhhVsV01kWY1J+iv0XghFZ
M/IGcCiqkq2HdYXw6X93Fro3Y6IYjYuJ1GUwmexH1eqqTKkus1L48qpVYSjz
mE3bs19VBmPbL7a4dZlTKOaF/YvaUYVh0RwmDO/B6mhmiDpxwoxNMIW5DAwZ
kS9j8u3v678tZuDPwBSTGko/Y6k/CBQyKGQ8Y9CR0dwb67ooowiNVxW7mC9d
sf3rMd3Dvz5jBXpkei613fnnZWYqc9sqVWxE1TPiInffqji03zu+IJ0PQiCe
Ge50PLGU/sT8TrDeymIqnnyYn0LFs7AOq5rdlz7axjeNNZkQ\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{232, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//TableForm=",
 CellID->540192]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24705],

Cell[TextData[{
 "The monic polynomial with roots ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Alpha]", "i"], TraditionalForm]], "InlineMath"],
 " has coefficients that are elementary symmetric polynomials of the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Alpha]", "i"], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->27091],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SolveAlways", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "4"], "+", 
       RowBox[{"a", " ", 
        SuperscriptBox["x", "3"]}], "+", 
       RowBox[{"b", " ", 
        SuperscriptBox["x", "2"]}], "+", 
       RowBox[{"c", " ", "x"}], "+", "d"}], "==", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"x", "-", 
         SubscriptBox["\[Alpha]", "1"]}], ")"}], 
       RowBox[{"(", 
        RowBox[{"x", "-", 
         SubscriptBox["\[Alpha]", "2"]}], ")"}], 
       RowBox[{"(", 
        RowBox[{"x", "-", 
         SubscriptBox["\[Alpha]", "3"]}], ")"}], 
       RowBox[{"(", 
        RowBox[{"x", "-", 
         SubscriptBox["\[Alpha]", "4"]}], ")"}]}]}], ",", "x"}], "]"}], 
   "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], "//", 
  "TableForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1259],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWc2O0zAQdpOmpd0mZaHtLhxQL4hn4AXgxENUK6TdAwIt+xC7HBAvAhwQ
z4aQQMY/cZV+mdgTJ91QaeOMPT/2eDz+Mvtmd3P59t3u5upit319vftweXXx
cfvq/bXqSkdCCKn+Xp4L9S5LSuqfEM/UM1PkiWpT1eaqTVRblPQ3w7XVz9RJ
ScNdpZdAL4D+arQkpa0ZjGalzcQQ+34jI8TEOzrly1YsjvcrE6lt9BO5cJ4p
0FrzOIIHac2TYR/BU9WtV/c9uAJNz9wMat4h517jQqsZi6vZYsKaOel70mrS
OCfS75FewP2xEW2FXaeeyZxlc072nrTg/SndiUr27reePJX25Fb7ZkTfnMn3
w7zi+n+ZXrefvw0lvc/n0p7+RWljUcpuoD8v21Wlv5qRNiC/rMgJ8QLmiZmo
AHrdkR/HUT7En1PyZl9TFxEw2ksurI0+7EszZNniSHqXHdY69ei1UdQmM+GJ
52Z8zs2Bpw6jq09bmMExUo95kyF+iPUPe10HO6zjYVZqyyD7UTdqOL/zbkPk
mpT26RuId5u1u4eHs9in35pvf3dvHO4pRjZ9wzpsMWH5hufB/GiWyExxL5bo
NcX5h95vcm3eU/oAaBpZtcFbq/9Aw7oFL63XzMF4S991T2Q8Knw8kOyKyUfp
e2ReJfuJ38ZreYhIzySNZPMKv26LCl1FsiZqg9/WZ0Bj9loFxkMIFOXxLt4A
jRhkZFbh/NTXdz2eYL8uP45sp8s/L0R/Yy+3D2lO4lfYSZMPVduIHKImgmvg
oFgO+kS6LySO842t+yCWmfbkhxi9Mci9DJUBalDNKDIOt/p1DofoXYZpNz8/
Ju5jrU5XOAKOUcsj80VHvMvTGYds8fz5fUDPl5XbesLYcbp5XwwWMbiI0b8Q
fm9TRcWKGi1L42OeLG0XcRXtiy52+5/zTNbOrmopfN4Fx9+Hvi52pzUv/DGv
2hd/zZv0Pl1l2+F9jZi1rPu+ujNcWJlGXH4ONOL+WzsnSf8PzV8Z9ld3/aOn
shkv3tXjB7hiqqUY0VwesjIRwC+fgitw+CWmOuhkeKjGjxaauP23s12f++LH
LzR/5WvZqLl9/acguXDHboO7wavs0LUWupeuqnzZ+00beyp52YXqoyoWnyWd
ScToHxp3GWU=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{255, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TableForm=",
 CellID->120513657]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"SymmetricPolynomial", "[", 
     RowBox[{"#", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", 
         SubscriptBox["\[Alpha]", "1"]}], ",", 
        RowBox[{"-", 
         SubscriptBox["\[Alpha]", "2"]}], ",", 
        RowBox[{"-", 
         SubscriptBox["\[Alpha]", "3"]}], ",", 
        RowBox[{"-", 
         SubscriptBox["\[Alpha]", "4"]}]}], "}"}]}], "]"}], "&"}], "/@", 
   RowBox[{"Range", "[", "4", "]"}]}], "//", "TableForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->31687],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWFFu2zAMVWI7adbY6bok7f56jV1g+9ohgmJA+zFs6HqIth/FLtL2Y9id
cgtNkuVUeaFkWlaWBohlUhRF0hL5pK+r26tv31e315eriy83q59X15e/Lj7/
uFGsbCCEWKv/p3Oh3qV6M/+1lKodq2ehyGPVZqotVTtUbWXpZyM10U9NSvMz
0i49A3oK9JPRIrK60b+hfi02ErZjZGene8fB3q2xztz5xguXqyVzkMyApmSQ
1jIF8ggZV7e29qVud7wgbdqRQm0FSyonpXIbN59Fme2n5yDjE+kVxvHJY9E7
cvQxyaVl/9btSG7WYxMkLZ9tgldzKd6E4P0xr2v2U+8xbdrU6praSC+BX9p2
7vDdPbqE8TNnnBAnEFXcmxXQi57y2I/j2+RLarw4kokyw07vSSrNkHOqPemd
9fB1HNBbr5Yx9HBzICeX4v7GlZFyLsx9uMr2mduxGsbGh+3XVmXFrLRjMeV1
ZO1AKb36ioYfWSu6Va3DzZgybv7a2uR5uzO91XnE8p8XpXJvM5EZ4L/MRPsU
Fx/6m5K+eb5aF6wyfwMaFh1kab3GBlPFY9HVhwONnTPlKH2n5pWPCPE0tpDb
iO9M0kixdOR1Wzm0ixRN7m89zZ0BjbVz3tLfhvBwPNbLJdCIEwZ1gHueJo9A
axGUDuO1brrCduHXyYPSIUQ3ivewl6YQeh3ab/fKjT2Jo00cpMhBeEinQrto
b+xtA2IHROyxcYjRG4OO7bZNePPhR2Nx+C+s83DIuMkE3ewLY8sUvja6fF+2
2w0Sua97YkSezjg0iPvEv2799rJyUCJcGqebh7JtZWYiRbzpoS2h8SRvLD0v
4hHatz7zprd5UrfMG0su3qXk+uDn1POavRVxo9rgYI0k9a77aOl7I4U3oohX
z4FGPHxntIRvJMO3iuHe99KPn2oPOLd0uNq4MuTJuaWmP9Rtj1uopg7xqn64
mvqkw9Xr4TWy3uw982roftdQkVIY/buAVfS5nubSJ/jfslnJnB1K8ajT8aOk
d6cY/AMds2LE\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{232, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//TableForm=",
 CellID->123157151]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->42],

Cell[TextData[{
 "The elementary symmetric polynomials ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["e", "TI"], 
    StyleBox["k", "TI"]], "=", 
   RowBox[{"SymmetricPolynomial", "[", 
    RowBox[{
     StyleBox["k", "TI"], ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["1", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"], ",", 
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["n", "TI"]]}], "}"}]}], "]"}]}]], "InlineFormula"],
 "are related to the power sum polynomials ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["s", "p"], "=", 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"k", "=", "1"}], "n"], 
     SubsuperscriptBox["x", "k", "p"]}]}], TraditionalForm]], "InlineMath"],
 " through the Newton-Girard identities:"
}], "ExampleText",
 CellID->28317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   UnderoverscriptBox["\[Sum]", 
    RowBox[{"k", "=", "1"}], "n"], 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "k"], 
    SubscriptBox["s", 
     RowBox[{"n", "-", "k"}]], 
    SubscriptBox["e", "k"]}]}], "==", "0"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22045],

Cell[BoxData[
 RowBox[{
  RowBox[{
   UnderoverscriptBox["\[Sum]", 
    RowBox[{"k", "=", "1"}], "n"], 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "k"], " ", 
    SubscriptBox["e", "k"], " ", 
    SubscriptBox["s", 
     RowBox[{
      RowBox[{"-", "k"}], "+", "n"}]]}]}], "\[Equal]", "0"}]], "Output",
 ImageSize->{130, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->28999115]
}, Open  ]],

Cell[TextData[{
 "For example, with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "4"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->13388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["s", "4"], 
     SubscriptBox["e", "0"]}], "-", 
    RowBox[{
     SubscriptBox["s", "3"], 
     SubscriptBox["e", "1"]}], "+", 
    RowBox[{
     SubscriptBox["s", "2"], 
     SubscriptBox["e", "2"]}], "-", 
    RowBox[{
     SubscriptBox["s", "1"], 
     SubscriptBox["e", "3"]}], "+", 
    RowBox[{
     SubscriptBox["s", "0"], 
     SubscriptBox["e", "4"]}]}], "/.", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["s", "n_"], ":>", 
      RowBox[{"Sum", "[", 
       RowBox[{
        SuperscriptBox[
         SubscriptBox["x", "k"], "n"], ",", 
        RowBox[{"{", 
         RowBox[{"k", ",", "4"}], "}"}]}], "]"}]}], ",", 
     RowBox[{
      SubscriptBox["e", "n_"], ":>", 
      RowBox[{"SymmetricPolynomial", "[", 
       RowBox[{"n", ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox["x", "1"], ",", 
          SubscriptBox["x", "2"], ",", 
          SubscriptBox["x", "3"], ",", 
          SubscriptBox["x", "4"]}], "}"}]}], "]"}]}]}], "}"}]}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25452],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4992949]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->27760],

Cell[TextData[{
 "Find integers ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["a", "1"], ",", 
    SubscriptBox["a", "2"], ",", 
    SubscriptBox["a", "3"]}], TraditionalForm]], "InlineMath"],
 " such that the roots of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "3"], "+", 
     RowBox[{
      SubscriptBox["a", "1"], " ", 
      SuperscriptBox["x", "2"]}], "+", 
     RowBox[{
      SubscriptBox["a", "2"], " ", "x"}], "+", 
     SubscriptBox["a", "3"]}], "=", "0"}], TraditionalForm]], "InlineMath"],
 " are ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["a", "1"], ",", 
    SubscriptBox["a", "2"], ",", 
    SubscriptBox["a", "3"]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->26919],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["x", "3"], "+", 
   RowBox[{
    SubscriptBox["a", "1"], 
    SuperscriptBox["x", "2"]}], "+", 
   RowBox[{
    SubscriptBox["a", "2"], "x"}], "+", 
   SubscriptBox["a", "3"]}], "/.", 
  RowBox[{"{", 
   RowBox[{"ToRules", "[", 
    RowBox[{"Reduce", "[", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"SymmetricPolynomial", "[", 
          RowBox[{"i", ",", 
           RowBox[{"{", 
            RowBox[{
             SubscriptBox["a", "1"], ",", 
             SubscriptBox["a", "2"], ",", 
             SubscriptBox["a", "3"]}], "}"}]}], "]"}], "==", 
         RowBox[{
          SubscriptBox["a", "i"], 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"-", "1"}], ")"}], "i"]}]}], ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "3"}], "}"}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["a", "1"], ",", 
        SubscriptBox["a", "2"], ",", 
        SubscriptBox["a", "3"]}], "}"}], ",", "\[IndentingNewLine]", 
      "Integers"}], "]"}], "]"}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25631],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   SuperscriptBox["x", "3"], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", "x"}], "+", 
    SuperscriptBox["x", "2"], "+", 
    SuperscriptBox["x", "3"]}], ",", 
   RowBox[{
    RowBox[{"-", "1"}], "-", "x", "+", 
    SuperscriptBox["x", "2"], "+", 
    SuperscriptBox["x", "3"]}]}], "}"}]], "Output",
 ImageSize->{213, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->388502724]
}, Open  ]],

Cell["Check:", "ExampleText",
 CellID->21267],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Solve", "[", 
    RowBox[{
     RowBox[{"#", "==", "0"}], ",", "x"}], "]"}], "&"}], "/@", "%"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->25776],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdWMtOwzAQdF6F0kIF4gACoQAXxB31D+DER0QVUntAoNL/D3YhtNqutbuJ
7TpUihUnu+OZsb2x+lqt5m/v1Woxq8qXZfU5X8y+yuePpX6UJUqlT0qpyYXS
97W+W1/Nb6ovpXLTZn/PyX5JvD/yHE/lw/cP/5yva32wn5nmpxVk3SEoqe5C
Nm3jMJZYHox77Dmvtvyp+Upr9izn2zXEjJ1aOAx0Y7oFgXrYI1R0FLZDIxA9
1k2h6Bq8D4cg15zJNUaHwqC2dUiw92DWhpu+GWMZxTZ7ZWFn1Ax7jM8Zj/bv
GESMotHHw4f8pbXMnXOw706TP+Q2bu3u4rSl56ZmD5uavbMmIDfMhX2hQEXc
GTr4HcVWNQdBNTds4Ki+1EJcKU9XOFK9sJ+sfZZ+u5pTK+oeWREnEaNR6DYn
TsBzal2F8QCykq6O0Op94bj5NnQ/4cETJ17zKJWx40pXEX3SwuY5DocgU1iP
/XsD13p3De4Rpa74+HfjrGbPeAQYrHViHeMG8y+oynNCTzd91xw0T9lSZW6+
JZf8OQmUJa1ktwEVXAndpbjfC/Ol8VK++JqSZhlvM8TNU8dxVB62UlzzMjuX
OgVyeGE4MI8bx+WFnbo3G2Iq7KnkG+1hZVQ=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{565, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->617238]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->10761],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["SymmetricReduction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SymmetricReduction"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PolynomialReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PolynomialReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Subsets",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Subsets"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->5469]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->27651],

Cell[TextData[ButtonBox["Symmetric Polynomials",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SymmetricPolynomials"]], "Tutorials",
 CellID->15275]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->645515751],

Cell[TextData[ButtonBox["Polynomial Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialAlgebra"]], "MoreAbout",
 CellID->122702785],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->586048872],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->358618156]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"SymmetricPolynomial - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 25, 37.4870382}", "context" -> "System`", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "SymmetricPolynomial[k, {x_1, ..., x_n}] gives the k\\[Null]^th \
elementary symmetric polynomial in the variables x_1, ..., x_n.", 
    "synonyms" -> {"symmetric polynomial"}, "title" -> "SymmetricPolynomial", 
    "type" -> "Symbol", "uri" -> "ref/SymmetricPolynomial"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7926, 251, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->106932869]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 37066, 1275}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2393, 58, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3016, 85, 1584, 37, 70, "ObjectNameGrid"],
Cell[4603, 124, 1295, 43, 70, "Usage",
 CellID->25402]
}, Open  ]],
Cell[CellGroupData[{
Cell[5935, 172, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6413, 189, 764, 27, 70, "Notes",
 CellID->32676],
Cell[7180, 218, 274, 11, 70, "Notes",
 CellID->13543],
Cell[7457, 231, 123, 4, 70, "Notes",
 CellID->7373],
Cell[7583, 237, 306, 9, 70, "Notes",
 CellID->6266]
}, Closed]],
Cell[CellGroupData[{
Cell[7926, 251, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->106932869],
Cell[CellGroupData[{
Cell[8311, 265, 148, 5, 70, "ExampleSection",
 CellID->114168625],
Cell[8462, 272, 462, 18, 70, "ExampleText",
 CellID->27892],
Cell[CellGroupData[{
Cell[8949, 294, 302, 10, 28, "Input",
 CellID->2451],
Cell[9254, 306, 621, 14, 36, "Output",
 Evaluatable->False,
 CellID->33743922]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9924, 326, 216, 7, 70, "ExampleSection",
 CellID->22570],
Cell[10143, 335, 109, 3, 70, "ExampleText",
 CellID->13519],
Cell[CellGroupData[{
Cell[10277, 342, 303, 10, 70, "Input",
 CellID->19472],
Cell[10583, 354, 158, 5, 36, "Output",
 CellID->700973310]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[10790, 365, 223, 7, 70, "ExampleSection",
 CellID->14262],
Cell[11016, 374, 93, 3, 70, "ExampleText",
 CellID->21409],
Cell[11112, 379, 371, 12, 70, "Input",
 CellID->28607],
Cell[11486, 393, 121, 3, 70, "ExampleText",
 CellID->18038],
Cell[CellGroupData[{
Cell[11632, 400, 586, 19, 70, "Input",
 CellID->4360],
Cell[12221, 421, 785, 17, 70, "Output",
 Evaluatable->False,
 CellID->199970885]
}, Open  ]],
Cell[13021, 441, 349, 9, 70, "ExampleText",
 CellID->11171],
Cell[CellGroupData[{
Cell[13395, 454, 543, 18, 70, "Input",
 CellID->20654],
Cell[13941, 474, 592, 22, 36, "Output",
 CellID->328448441]
}, Open  ]],
Cell[14548, 499, 346, 12, 70, "ExampleText",
 CellID->11514],
Cell[CellGroupData[{
Cell[14919, 515, 339, 12, 70, "Input",
 CellID->14523],
Cell[15261, 529, 180, 6, 36, "Output",
 CellID->339851050]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15490, 541, 231, 7, 70, "ExampleSection",
 CellID->489],
Cell[CellGroupData[{
Cell[15746, 552, 358, 12, 70, "Input",
 CellID->6766],
Cell[16107, 566, 634, 24, 36, "Output",
 CellID->85729613]
}, Open  ]],
Cell[CellGroupData[{
Cell[16778, 595, 303, 10, 70, "Input",
 CellID->29756],
Cell[17084, 607, 635, 24, 36, "Output",
 CellID->171605805]
}, Open  ]],
Cell[17734, 634, 120, 3, 70, "ExampleDelimiter",
 CellID->7470],
Cell[17857, 639, 442, 16, 70, "ExampleText",
 CellID->25739],
Cell[CellGroupData[{
Cell[18324, 659, 362, 13, 70, "Input",
 CellID->20666],
Cell[18689, 674, 1125, 22, 70, "Output",
 Evaluatable->False,
 CellID->255801993]
}, Open  ]],
Cell[19829, 699, 44, 1, 70, "ExampleText",
 CellID->7283],
Cell[CellGroupData[{
Cell[19898, 704, 478, 15, 70, "Input",
 CellID->8160],
Cell[20379, 721, 1122, 22, 70, "Output",
 Evaluatable->False,
 CellID->540192]
}, Open  ]],
Cell[21516, 746, 121, 3, 70, "ExampleDelimiter",
 CellID->24705],
Cell[21640, 751, 352, 11, 70, "ExampleText",
 CellID->27091],
Cell[CellGroupData[{
Cell[22017, 766, 895, 29, 70, "Input",
 CellID->1259],
Cell[22915, 797, 1394, 27, 70, "Output",
 Evaluatable->False,
 CellID->120513657]
}, Open  ]],
Cell[CellGroupData[{
Cell[24346, 829, 562, 18, 70, "Input",
 CellID->31687],
Cell[24911, 849, 1272, 25, 70, "Output",
 Evaluatable->False,
 CellID->123157151]
}, Open  ]],
Cell[26198, 877, 118, 3, 70, "ExampleDelimiter",
 CellID->42],
Cell[26319, 882, 886, 30, 70, "ExampleText",
 CellID->28317],
Cell[CellGroupData[{
Cell[27230, 916, 343, 13, 70, "Input",
 CellID->22045],
Cell[27576, 931, 473, 17, 58, "Output",
 CellID->28999115]
}, Open  ]],
Cell[28064, 951, 166, 7, 70, "ExampleText",
 CellID->13388],
Cell[CellGroupData[{
Cell[28255, 962, 1164, 41, 70, "Input",
 CellID->25452],
Cell[29422, 1005, 156, 5, 36, "Output",
 CellID->4992949]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29627, 1016, 224, 7, 70, "ExampleSection",
 CellID->27760],
Cell[29854, 1025, 770, 29, 70, "ExampleText",
 CellID->26919],
Cell[CellGroupData[{
Cell[30649, 1058, 1165, 38, 70, "Input",
 CellID->25631],
Cell[31817, 1098, 487, 17, 39, "Output",
 CellID->388502724]
}, Open  ]],
Cell[32319, 1118, 45, 1, 70, "ExampleText",
 CellID->21267],
Cell[CellGroupData[{
Cell[32389, 1123, 194, 8, 70, "Input",
 CellID->25776],
Cell[32586, 1133, 855, 18, 70, "Output",
 Evaluatable->False,
 CellID->617238]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[33502, 1158, 311, 9, 70, "SeeAlsoSection",
 CellID->10761],
Cell[33816, 1169, 729, 26, 70, "SeeAlso",
 CellID->5469]
}, Open  ]],
Cell[CellGroupData[{
Cell[34582, 1200, 314, 9, 70, "TutorialsSection",
 CellID->27651],
Cell[34899, 1211, 152, 3, 70, "Tutorials",
 CellID->15275]
}, Open  ]],
Cell[CellGroupData[{
Cell[35088, 1219, 319, 9, 70, "MoreAboutSection",
 CellID->645515751],
Cell[35410, 1230, 147, 3, 70, "MoreAbout",
 CellID->122702785],
Cell[35560, 1235, 169, 3, 70, "MoreAbout",
 CellID->586048872],
Cell[35732, 1240, 179, 3, 70, "MoreAbout",
 CellID->358618156]
}, Open  ]],
Cell[35926, 1246, 27, 0, 70, "History"],
Cell[35956, 1248, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

